/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: lpc.c
 * lpc - control LPR queues
 ***************************************************************************/

#include "lp.h"
#include "library/errormsg.h"
#include "library/readconf.h"
#include "patchlevel.h"

/***************************************************************************
 * SYNOPSIS
 *      lpc [ -Dn ] [ command [ argument ... ] ]
 * DESCRIPTION
 *      Lpc  is  used  by  the  system  administrator to control the
 *      operation  of the line printer system. For each line printer
 *      configured in /etc/printcap, lpc may be used to:
 *      + disable or enable a printer,
 *      + disable or enable a printer's spooling queue,
 *      + rearrange the order of jobs in a spooling queue,
 *      + find the status of printers, and their associated spooling
 *        queues and printer dameons.
 *      Without any arguments, lpc will prompt for commands from the
 *      standard  input.   If arguments are supplied, lpc interprets
 *      the  first argument as a command and the remaining arguments
 *      as  parameters  to  the  command.  The standard input may be
 *      redirected causing lpc to read commands from file.  Commands
 *      may  be  abreviated; the following is the list of recognized
 *      commands.
 *      ? [ command ... ]
 *      abort { all | printer ... }
 *      attach  Printer destPrinter
 *      chprio printer [ jobnum ... ] [ user ... ]
 *      clean { all | printer ... }
 *      down {all | printer ... }
 *      downq printer [ jobnum ... ] [ user ... ]
 *      disable { all | printer ... }
 *      enable { all | printer ... }
 *      exit
 *      help [ command ... ]
 *      hold printer jobnum
 *      kill { all | printer ... }
 *      lpd
 *      lpq [options]
 *      lprm [options]
 *      quit
 *      release printer jobnum 
 *      remote { abort | clean | enable | disable | restart | up | down } printer
 *      requeue printer jobnum 
 *      restart { all | printer ... }
 *      start { all | printer ... }
 *      status [ terse ] [ all ] [ printer ... ]
 *      stop { all | printer ... }
 *      topq printer [ jobnum ... ] [ user ... ]
 *      unattach printer
 *      up { all | printer ... }
 *
 *      -Dn  Enables  display  of  debugging  information.   The  -D
 *           selects  level  1;  -Dn  selects level n (n is a single
 *           digit).
 ***************************************************************************/

static void Usemsg(void)
{
    (void) fprintf (stderr,"Usage: lpc [-Ddebugopts ] [ command [ argument ... ] ]\n");
}

/***************************************************************************
 * 1. pick off the options
 * 2. set up the parameters
 ***************************************************************************/

static void Lpc_parms(int argc, char *argv[])
{
    int option;
    int i;

    while ((option = Getopt (argc, argv, "D:")) != EOF) {
	switch (option) {
	case 'D':
	    if (Debug) {
		Diemsg ("Duplicate -D option");
	    }
	    Parse_debug (Optarg);
	    break;
	case '?':
	    Usemsg ();
	    exit (1);
	    break;
	default:
	    break;
	}
    }

    /*
     * set up the Parms[] array
     */
    allocParmsIfNeeded ();
    for (; Optind < argc; ++Optind) {
	Parms[Parmcount].str = argv[Optind];
	i = atoi (Parms[Parmcount].str);
	if (i > 0) {
	    Parms[Parmcount].num = i;
	} else {
	    Parms[Parmcount].num = -1;
	}
	++Parmcount;
	growParmsIfNeeded ();
    }
}

/***************************************************************************
 * Repeat:
 * 1. Get a line.
 * 2. if EOF return
 * 3. Split it up and put into Parms[]
 * 4. Call Control_ops()
 ***************************************************************************/

static void Get_line(void)
{
    char buf[BUFSIZ];		/* get a line */
    int i;			/* ACME Integers, Inc. */
    char *cp;			/* ACME Pointers, Inc. */

#ifdef READLINE
    char *prompt, *inputline;
    prompt = (char *) malloc(81 * sizeof (char *));
    inputline = (char *) malloc(81 * sizeof (char *));
#endif /* READLINE */

    while (1) {
#ifndef READLINE
	(void) fprintf (stdout, "LPC (%s): ", From);
	(void) fflush (stdout);
	if (fgets (buf, sizeof (buf), stdin) == 0)
	    return;
	buf[strlen (buf) - 1] = 0;	/* clobber the last char */
#else /* IS READLINE */
	(void) strcpy(prompt, "LPC ("); /* Generating the prompt */
	(void) strcat(prompt, From);
	(void) strcat(prompt, "): ");
	/*
	 * Get the typed commands...
	 */
	if ((inputline = (char *) readline(prompt)) == NULL)
	    {
		return;
	    }
	/*
	 * ...and put them in the history
	 */
	add_history(inputline);
	/*
	 * copy the contents of the command line to buf
	 */
	(void) strcpy(buf, inputline);
#endif /* IS NOT READLINE */

	/*
	 * now set up the parameters
	 */
	Parmcount = 0;
	for (cp = buf; *cp;) {
	    /* skip leading blanks */
	    Parms[Parmcount].str = 0;
	    Parms[Parmcount].num = -1;
	    while (*cp && isspace (*cp))
		++cp;
	    if (*cp) {
		Parms[Parmcount].str = cp;
	    }
	    while (*cp && !isspace (*cp))
		++cp;
	    if (*cp) {
		*cp = 0;
		++cp;
	    }
	    if (Parms[Parmcount].str != 0) {
		Parms[Parmcount].num = -1;
		i = atoi (Parms[Parmcount].str);
		if (i > 0) {
		    Parms[Parmcount].num = i;
		}
		++Parmcount;
	    }
	}
	if (Parmcount) {
	    if (Debug > 3) {
		for (i = 0; i < Parmcount; ++i) {
		    (void) fprintf (stderr, "Parms[%d]: '%s', %d\n", i,
				    Parms[i].str, Parms[i].num);
		}
	    }
	    (void) Control_ops ();
	}
    }
    /* NOTREACHED */
}

plp_signal_t cleanup(int signal) { exit (1); }

/***************************************************************************
 * 1. get the host computer name and user name
 * 2. set up the Host and Person information
 * 3. get the parameters or set up a read from STDIN
 * 4. convert parameters and call the server routine
 ****************************************************************************/

int main(int argc, char *argv[], char *envp[])
{
    struct passwd *pw_ent;	/* user entry in /etc/passwd */

    /*
     * set umask to avoid problems with user umask
     */
    (void) umask (0);
    set_user_uid (0);
    root_to_user ();

    /*
     * get the user name
     */
    if ((pw_ent = getpwuid (UserUID)) == 0) {
	logerr_die (XLOG_INFO, "getpwuid failed on uid %d", UserUID);
    }
    (void) strcpy (LOGNAME, pw_ent->pw_name);
    Person = LOGNAME;
    if (UserUID == 0) {
	/* we are being invoked by root */
	Is_root = 1;
    }
    /*
     * setup parameters
     */
    Lpc_parms (argc, argv);
    Std_environ (argc, argv, envp);
    Readconf ();

    FQDN = Host;
    From = ShortHost;

    /*
     * set signals
     */
    (void) plp_signal (SIGPIPE, (plp_sigfunc_t)SIG_IGN);
    (void) plp_signal (SIGHUP, cleanup);
    (void) plp_signal (SIGINT, cleanup);
    (void) plp_signal (SIGQUIT, cleanup);
    (void) plp_signal (SIGTERM, cleanup);

    if (Parmcount != 0) {
	(void) Control_ops ();
    } else {
        (void) printf ("PLP Line Printer Control program, version %s\n\n", PLP_PATCHLEVEL);
	Get_line ();
    }
    return 0;
}
