/*
 * util.c
 *
 * Copyright (c) 1990, 1991, John W. Eaton.
 *
 * You may distribute under the terms of the GNU General Public
 * License as specified in the file COPYING that comes with the man
 * distribution.  
 *
 * John W. Eaton
 * jwe@che.utexas.edu
 * Department of Chemical Engineering
 * The University of Texas at Austin
 * Austin, Texas  78712
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <unistd.h>

#ifdef STDC_HEADERS
#include <stdlib.h>
#else
extern int tolower ();
#endif

extern void fatal (), gripe ();
extern pid_t fork (), wait ();
extern char *strdup ();
extern int system ();
extern char *progname;

#include "gripes.h"

/*
 * Extract last element of a name like /foo/bar/baz.
 */
char *
mkprogname (s)
     register char *s;
{
  char *t;

  t = strrchr (s, '/');
  if (t == (char *)NULL)
    t = s;
  else
    t++;

  return strdup (t);
}

/*
 * Is file a nonempty and newer than file b?
 *
 * case:
 *
 *   a newer than b              returns    1
 *   a older than b              returns    0
 *   stat on a fails or a empty  returns   -1
 *   stat on b fails or b empty  returns   -2
 *   both fail or empty  	 returns   -3
 */
int
is_newer (fa, fb)
  register char *fa;
  register char *fb;
{
  struct stat fa_sb;
  struct stat fb_sb;
  register int fa_stat;
  register int fb_stat;
  register int status = 0;

  fa_stat = stat (fa, &fa_sb);
  if (fa_stat != 0 || fa_sb.st_size == 0)
    status = 1;

  fb_stat = stat (fb, &fb_sb);
  if (fb_stat != 0 || fb_sb.st_size == 0)
    status |= 2;

  if (status != 0)
    return -status;

  return (fa_sb.st_mtime > fb_sb.st_mtime);
}

/*
 * Is path a directory?
 */
int
is_directory (path)
     char *path;
{
  struct stat sb;
  register int status;

  status = stat (path, &sb);

  if (status != 0)
    return -1;

  return ((sb.st_mode & S_IFDIR) == S_IFDIR);

}

extern uid_t getuid (), geteuid ();
extern gid_t getgid (), getegid ();
extern int setuid ();

int ruid, rgid, euid, egid, suid;

void
get_permissions (void) {
  ruid = getuid();
  euid = geteuid();
  rgid = getgid();
  egid = getegid();
  suid = (ruid != euid || rgid != egid);
}

static int
my_system (command)
     char *command;
{
    int pid, pid2, status, stat;

    if (!suid)
      return system (command);

#ifdef _POSIX_SAVED_IDS
    /* we need not fork */
    setuid(ruid);
    setgid(rgid);
    status = system(command);
    setuid(euid);
    setgid(egid);
    return (WIFEXITED(status) ? WEXITSTATUS(status) : 127);
#endif

    fflush(stdout); fflush(stderr);
    pid = fork();
    if (pid == -1) {
	perror(progname);
	fatal (CANNOT_FORK, command);
    }
    if (pid == 0) {
	setuid(ruid);
	setgid(rgid);
	status = system (command);
	exit(WIFEXITED(status) ? WEXITSTATUS(status) : 127);
    }
    pid2 = wait (&stat);
    if (pid2 == -1) {
	perror(progname);
	fatal (WAIT_FAILED, command); 	/* interrupted? */
    }
    if (pid2 != pid)
	fatal (GOT_WRONG_PID);
    if (WIFEXITED(stat) && WEXITSTATUS(stat) != 127)
      return WEXITSTATUS(stat);
    fatal (CHILD_TERMINATED_ABNORMALLY, command);
    return -1;			/* not reached */
}

/*
 * Attempt a system () call.
 */
int
do_system_command (command, silent)
     char *command;
     int silent;
{
  int status = 0;
  extern int debug;

  /*
   * If we're debugging, don't really execute the command -- you never
   * know what might be in that mangled string :-O.
   */
  if (debug == 1) 
    gripe (NO_EXEC, command);
  else
    status = my_system (command);

  if (status && !silent) {
      if (debug)
	fprintf(stderr, "_%s_ (%d)\n", command, status);
      gripe (SYSTEM_FAILED, status);
  }
  return status;
}
