------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS               --
--                                                                          --
--                 I N T E R F A C E S . C . S T H R E A D S                --
--                                                                          --
--                                  S p e c                                 --
--                                                                          --
--                             $Revision: 1.6 $                             --
--                                                                          --
--   Copyright (C) 1991,1992,1993,1994,1995,1996 Florida State University   --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
-- GNARL was developed by the GNARL team at Florida State University. It is --
-- now maintained by Ada Core Technologies Inc. in cooperation with Florida --
-- State University (http://www.gnat.com).                                  --
--                                                                          --
------------------------------------------------------------------------------

--  This is a Solaris version of this package.

--  This package interfaces with Solaris specific thread calls. It is not a
--  complete interface; it only includes what is needed to implement the
--  Ada runtime. The changes made to this Solaris specific version involve
--  the absorbtion of the various attribute structures, and minor interface
--  differences.

with System;

with Interfaces.C.System_Constants;
--  Used for: mutex_t_size
--            cond_t_size
--            thread_key_t_size

with Interfaces.C.POSIX_RTE;
--  Used for: Signal,
--            Signal_Set

with Interfaces.C.POSIX_Error;
--  Used for: Return_Code

with Interfaces.C.POSIX_Timers;
--  Used for: timespec

package Interfaces.C.Sthreads is

   package ICPE renames Interfaces.C.POSIX_Error;
   package ICPR renames Interfaces.C.POSIX_RTE;
   package ICPT renames Interfaces.C.POSIX_Timers;

   Alignment : constant := Natural'Min (Standard'Maximum_Alignment, 8);

   Threads_Error : exception;

   type Priority_Type is new int;

   type thread_t     is private;
   type mutex_t      is private;
   type cond_t       is private;
   type thread_key_t is private;

   procedure thr_create
     (stack_base    : System.Address;
      stack_size    : size_t;
      start_routine : System.Address;
      arg           : System.Address;
      flags         : Integer;
      thread        : out thread_t;
      result        : out ICPE.Return_Code);
   pragma Inline (thr_create);

   function thr_self return thread_t;
   pragma Inline (thr_self);

   procedure mutex_init
     (mutex      : out mutex_t;
      mtype      : Integer;
      arg        : System.Address;
      result     : out ICPE.Return_Code);
   pragma Inline (mutex_init);

   procedure mutex_destroy
     (mutex  : in out mutex_t;
      result : out ICPE.Return_Code);
   pragma Inline (mutex_destroy);

   procedure mutex_lock
     (mutex  : in out mutex_t;
      result : out ICPE.Return_Code);
   pragma Inline (mutex_lock);

   procedure mutex_unlock
     (mutex  : in out mutex_t;
      result : out ICPE.Return_Code);
   pragma Inline (mutex_unlock);

   procedure cond_init
     (condition  : out cond_t;
      ctype      : Integer;
      arg        : Integer;
      result     : out ICPE.Return_Code);
   pragma Inline (cond_init);

   procedure cond_wait
     (condition : in out cond_t;
      mutex     : in out mutex_t;
      result    : out ICPE.Return_Code);
   pragma Inline (cond_wait);

   procedure cond_timedwait
     (condition     : in out cond_t;
      mutex         : in out mutex_t;
      absolute_time : ICPT.timespec;
      result        : out ICPE.Return_Code);
   pragma Inline (cond_timedwait);

   procedure cond_signal
     (condition : in out cond_t;
      result    : out ICPE.Return_Code);
   pragma Inline (cond_signal);

   procedure cond_broadcast
     (condition : in out cond_t;
      result    : out ICPE.Return_Code);
   pragma Inline (cond_broadcast);

   procedure cond_destroy
     (condition : in out cond_t;
      result    : out ICPE.Return_Code);
   pragma Inline (cond_destroy);

   procedure thr_setspecific
     (key    : thread_key_t;
      value  : System.Address;
      result : out ICPE.Return_Code);
   pragma Inline (thr_setspecific);

   procedure thr_getspecific
     (key    : thread_key_t;
      value  : out System.Address;
      result : out ICPE.Return_Code);
   pragma Inline (thr_getspecific);

   procedure thr_keycreate
     (key        : in out thread_key_t;
      destructor : System.Address;
      result     : out ICPE.Return_Code);
   pragma Inline (thr_keycreate);

   procedure thr_setprio
     (thread     : in out thread_t;
      priority : Priority_Type;
      result   : out ICPE.Return_Code);
   pragma Inline (thr_setprio);

   procedure thr_getprio
     (thread     : thread_t;
      priority : out Priority_Type;
      result   : out ICPE.Return_Code);
   pragma Inline (thr_getprio);

   procedure thr_exit (status : System.Address);
   pragma Interface (C, thr_exit);
   pragma Interface_Name (thr_exit, "thr_exit");

   procedure thr_setconcurrency
     (new_level : Integer;
      result   : out ICPE.Return_Code);
   pragma Inline (thr_setconcurrency);

   function thr_getconcurrency return Integer;
   pragma Inline (thr_getconcurrency);

   procedure sigwait
     (set    : ICPR.Signal_Set;
      sig    : out ICPR.Signal;
      result : out ICPE.Return_Code);
   pragma Inline (sigwait);

   procedure thr_kill
     (thread : thread_t; sig : ICPR.Signal;
      result : out ICPE.Return_Code);
   pragma Inline (thr_kill);

   procedure thr_sigsetmask
     (how    : int;
      set    : access ICPR.Signal_Set;
      oset   : access ICPR.Signal_Set;
      Result : out ICPE.Return_Code);
   pragma Inline (thr_sigsetmask);

private

   --  The use of longword alignment for the following C types is
   --  a stopgap measure which is not generally portable.  A portable
   --  solution will require some means of getting alignment information
   --  from the C compiler.

   type mutex_t is
     array (1 .. System_Constants.mutex_t_size) of unsigned;
   for mutex_t'Alignment use Alignment;

   type cond_t is
     array (1 .. System_Constants.cond_t_size) of unsigned;
   for cond_t'Alignment use Alignment;

   type thread_t is new unsigned;
   --  ??? The use of an array here (of one element, usually) seems to cause
   --  problems in the compiler.

   type thread_key_t is new unsigned;
   --  This type is passed as a scaler into a C function. It must
   --  be declared as a scaler, not an array.  This being so, an unsigned
   --  should work.

end Interfaces.C.Sthreads;
