------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             G N A T 1 D R V                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.46 $                             --
--                                                                          --
--   Copyright (C) 1992,1993,1994,1995,1996 Free Software Foundation, Inc.  --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- It is now maintained by Ada Core Technologies Inc (http://www.gnat.com). --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Comperr;
with Csets;    use Csets;
with Back_End;
with Errout;   use Errout;
with Features;
with Frontend;
with Gnatvsn;  use Gnatvsn;
with Lib;      use Lib;
with Lib.Writ; use Lib.Writ;
with Namet;    use Namet;
with Opt;      use Opt;
with Osint;    use Osint;
with Output;   use Output;
with Par;
with Sinfo;    use Sinfo;
with Snames;
with Sprint;   use Sprint;
with Stringt;
with System.Assertions;
with Tree_Gen;
with Treepr;   use Treepr;
with Types;    use Types;
with Uintp;
with Uname;    use Uname;
with Urealp;
with Usage;

procedure Gnat1drv is
   Main_Unit_Node : Node_Id;
   --  Compilation unit node for main unit

   Main_Kind : Node_Kind;
   --  Kind of main compilation unit node.

   Original_Operating_Mode : Operating_Mode_Type;
   --  Save operating type specified by options

   Call_Back_End : Boolean;
   --  Flag indicating whether we call the backend to generate code

begin
   --  This inner block is set up to catch assertion errors and constraint
   --  errors. Since the code for handling these errors can cause another
   --  exception to be raised (namely Unrecoverable_Error), we need two
   --  nested blocks, so that the outer one handles unrecoverable error.

   begin
      Osint.Initialize (Compiler);
      Csets.Initialize;
      Uintp.Initialize;
      Urealp.Initialize;
      Errout.Initialize;
      Namet.Initialize;
      Snames.Initialize;
      Stringt.Initialize;
      Features.Initialize;

      if Verbose_Mode or Full_List then
         Write_Eol;
         Write_Str ("GNAT Compiler Version ");
         Write_Str (Gnat_Version_String);
         Write_Str (" Copyright 1995 Free Software Foundation, Inc.");
         Write_Eol;
      end if;

      Original_Operating_Mode := Operating_Mode;
      Frontend;

      if Errors_Detected /= 0 then
         Errout.Finalize;
         Namet.Finalize;
         Features.Finalize;
         Exit_Program (E_Errors);
      end if;

      --  Case of no code being generated, exit indicating no error

      if Original_Operating_Mode /= Generate_Code then
         Errout.Finalize;
         Tree_Gen;
         Namet.Finalize;
         Features.Finalize;
         return;
      end if;

      --  All remaining cases are cases in which the user requested that code
      --  be generated (i.e. no -gnatc or -gnats switch was used). Check if
      --  we can in fact satisfy this request.

      Main_Unit_Node := Cunit (Main_Unit);
      Main_Kind := Nkind (Unit (Main_Unit_Node));

      --  Cannot generate code if someone has turned off code generation
      --  for any reason at all. We will try to figure out a reason below.

      if Operating_Mode /= Generate_Code then
         Call_Back_End := False;

      --  Cannot generate code if in stub generation mode

      elsif Distribution_Stub_Mode = Generate_Receiver_Stub_Body
           or else
         Distribution_Stub_Mode = Generate_Caller_Stub_Body
      then
         Call_Back_End := False;

      --  We can genreate code for a subprogram body unless its corresponding
      --  subprogram spec is a generic delaration. Note that the check for
      --  No (Library_Unit) here is a defensive check that should not be
      --  necessary, since the Library_Unit field should be set properly.

      elsif Main_Kind = N_Subprogram_Body
        and then not Subunits_Missing
        and then (No (Library_Unit (Main_Unit_Node))
                   or else Nkind (Unit (Library_Unit (Main_Unit_Node))) /=
                                          N_Generic_Subprogram_Declaration)
      then
         Call_Back_End := True;

      --  We can generate code for a package body unless its corresponding
      --  package spec is a generic declaration. As described above, the
      --  check for No (LIbrary_Unit) is a defensive check.

      elsif Main_Kind = N_Package_Body
        and then not Subunits_Missing
        and then (No (Library_Unit (Main_Unit_Node))
           or else Nkind (Unit (Library_Unit (Main_Unit_Node))) /=
                      N_Generic_Package_Declaration)
      then
         Call_Back_End := True;

      --  We can generate code for a package declaration only if it
      --  does not require a body.

      elsif Main_Kind = N_Package_Declaration
        and then not Body_Required (Main_Unit_Node)
      then
         Call_Back_End := True;

      --  Compilation units that are renamings do not require bodies,
      --  so we can generate code for them.

      elsif Main_Kind = N_Package_Renaming_Declaration
        or else Main_Kind = N_Subprogram_Renaming_Declaration
      then
         Call_Back_End := True;

      --  In all other cases (specs which have bodies, generics, and bodies
      --  where subunits are missing), we cannot generate code and we generate
      --  a warning message. Note that generic instantiations are gone at this
      --  stage since they have been replaced by their instances.

      else
         Call_Back_End := False;
      end if;

      --  At this stage Call_Back_End is set to indicate if the backend
      --  should be called to generate code. If it is not set, then code
      --  generation has been turned off, even though code was requested
      --  by the original command. This is not an error from the user
      --  point of view, but it is an error from the point of view of
      --  the gcc driver, so we must exit with an error status.

      --  We generate an informative message (from the gcc point of view,
      --  it is an error message, but from the users point of view this
      --  is not an error, just a consequence of compiling something that
      --  cannot generate code.

      if not Call_Back_End then
         Write_Str ("No code generated for ");
         Write_Str ("file ");
         Write_Name (Unit_File_Name (Main_Unit));

         if Subunits_Missing then
            Write_Str (" (missing subunits)");

         elsif Main_Kind = N_Subunit then
            Write_Str (" (subunit)");

         elsif Main_Kind = N_Package_Body
           or else Main_Kind = N_Subprogram_Body
         then
            Write_Str (" (generic unit)");

         --  Only other case is a package spec

         else
            Write_Str (" (package spec)");
         end if;

         Write_Eol;
         Errout.Finalize;
         Tree_Gen;
         Namet.Finalize;

         --  Exit program with error indication, to kill object file

         Exit_Program (E_Errors);
      end if;

      --  Here we will call the backend to generate code

      Set_Generate_Code (Main_Unit);

      --  If we have a corresponding spec, then we need object
      --  code for the spec unit as well

      if Nkind (Unit (Main_Unit_Node)) in N_Unit_Body
        and then not Acts_As_Spec (Main_Unit_Node)
      then
         Set_Generate_Code
           (Get_Cunit_Unit_Number (Library_Unit (Main_Unit_Node)));
      end if;

      --  Generate back end tables and library information

      Back_End;
      Errout.Finalize;
      Tree_Gen;
      Features.Finalize;

      --  Only write the library if the backend did not generate any error
      --  messages. Otherwise signal errors to the driver program so that
      --  there will be no attempt to generate an object file.

      if Errors_Detected /= 0 then
         Exit_Program (E_Errors);
      end if;

      Lib.Writ.Write_Library_Info;
      Namet.Finalize;

   exception
      --  Handle fatal internal compiler errors

      when System.Assertions.Assert_Failure =>
         Comperr.Compiler_Abort ("Assert_Failure");

      when Constraint_Error =>
         Comperr.Compiler_Abort ("Constraint_Error");

      when Program_Error =>
         Comperr.Compiler_Abort ("Program_Error");

      when Storage_Error =>
         Set_Standard_Error;
         Write_Str ("insufficient memory for compiler");
         Write_Eol;
         raise Unrecoverable_Error;
   end;

--  The outer exception handles an unrecoverable error

exception
   when Unrecoverable_Error =>
      Errout.Finalize;
      Set_Standard_Error;
      Write_Str ("compilation abandoned");
      Write_Eol;
      Set_Standard_Output;

      Tree_Dump;
      Source_Dump;
      Exit_Program (E_Errors);

end Gnat1drv;
