------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             S E M _ C H 1 3                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.210 $                            --
--                                                                          --
--   Copyright (C) 1992,1993,1994,1995,1996 Free Software Foundation, Inc.  --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- It is now maintained by Ada Core Technologies Inc (http://www.gnat.com). --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Einfo;    use Einfo;
with Errout;   use Errout;
with Features; use Features;
with Freeze;   use Freeze;
with Lib;      use Lib;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Output;   use Output;
with Rtsfind;  use Rtsfind;
with Sem;      use Sem;
with Sem_Ch3;  use Sem_Ch3;
with Sem_Ch8;  use Sem_Ch8;
with Sem_Eval; use Sem_Eval;
with Sem_Res;  use Sem_Res;
with Sem_Util; use Sem_Util;
with Stand;    use Stand;
with Sinfo;    use Sinfo;
with Sinput;   use Sinput;
with Snames;   use Snames;
with Ttypes;   use Ttypes;
with Uintp;    use Uintp;
with Urealp;   use Urealp;

package body Sem_Ch13 is

   -----------------------
   -- Local Subprograms --
   -----------------------

   function Already_Frozen (T : Entity_Id; N : Node_Id) return Boolean;
   --  Called at the start of processing a representation clause. Used to
   --  check that type T, referenced by representation clause N, is not
   --  already frozen. If the type is not frozen, then False is returned,
   --  and the caller can proceed. If the type is frozen, then an error
   --  message is issued and True is returned (which is a signal to the
   --  caller to abandon processing of the too late rep clause).

   procedure Check_Component_Overlap (C1_Ent, C2_Ent : Entity_Id);
   --  Given two entities for record components or discriminants, checks
   --  if they hav overlapping component clauses and issues errors if so.

   procedure Check_Size (N : Node_Id; T : Entity_Id; Siz : Uint);
   --  Called when size S is specified for subtype T. This subprogram checks
   --  that the size is appropriate, posting errors on node N as required.
   --  For non-elementary types, a check is only made if an explicit size
   --  has been given for the type (and the specified size must match)

   function Defined_Before (E1, E2 : Entity_Id) return Boolean;
   --  Determine if entity E1 is defined before E2 (returns True if so)

   --------------------
   -- Already_Frozen --
   --------------------

   function Already_Frozen (T : Entity_Id; N : Node_Id) return Boolean is
      S : Entity_Id;

   begin
      if Is_Frozen (T) then
         Error_Msg_N  ("rep clause appears too late", N);

         S := First_Subtype (T);

         if Present (Freeze_Node (S)) then
            Error_Msg_NE
              ("?no more rep clauses for }", Freeze_Node (S), S);
         end if;

         return True;

      else
         return False;
      end if;
   end Already_Frozen;

   -----------------------
   -- Analyze_At_Clause --
   -----------------------

   --  An at clause is replaced by the corresponding Address attribute
   --  definition clause that is the preferred approach in Ada 95.

   procedure Analyze_At_Clause (N : Node_Id) is
   begin
      Rewrite_Substitute_Tree (N,
        Make_Attribute_Definition_Clause (Sloc (N),
          Name  => Identifier (N),
          Chars => Name_Address,
          Expression => Expression (N)));
      Analyze (N);
   end Analyze_At_Clause;

   -----------------------------------------
   -- Analyze_Attribute_Definition_Clause --
   -----------------------------------------

   procedure Analyze_Attribute_Definition_Clause (N : Node_Id) is
      Nam  : constant Node_Id := Name (N);
      Attr : constant Name_Id := Chars (N);
      Expr : constant Node_Id := Expression (N);
      Id   : constant Attribute_Id := Get_Attribute_Id (Attr);
      Ent  : Entity_Id;

   begin
      Analyze (Nam);
      Ent := Entity (Nam);

      --  Rep clause applies to full view of incomplete type or private type
      --  if we have one (if not, this is a premature use of the type).

      Ent := Underlying_Type (Ent);

      if No (Ent) then
         Error_Msg_N ("premature reference to incomplete/private type", Nam);
         return;
      end if;

      --  Complete other routine error checks

      if Etype (Nam) = Any_Type then
         return;

      elsif Scope (Ent) /= Current_Scope then
         Error_Msg_N ("entity must be declared in this scope", Nam);
         return;

      elsif Is_Type (Ent) and then not Is_First_Subtype (Ent) then
         Error_Msg_N ("cannot specify attribute for subtype", Nam);
         return;

      end if;

      --  Switch on particular attribute

      case Id is

         -------------
         -- Address --
         -------------

         --  Address attribute definition clause

         when Attribute_Address => Address : begin
            Note_Feature (New_Representation_Clauses, Sloc (N));

            if Present (Address_Clause (Ent)) then
               Error_Msg_N ("address already given for &", Nam);

            elsif Ekind (Ent) not in Subprogram_Kind
              and then Ekind (Ent) /= E_Variable
              and then Ekind (Ent) /= E_Constant
              and then
                (Ekind (Ent) /= E_Entry
                  or else not Is_Task_Type (Scope (Ent)))

            then
               Error_Msg_N ("address cannot be given for &", Nam);

            --  Following check should not be necessary; this case
            --  should be caught by the test above that the task
            --  entry has been frozen.  For some reason task entries
            --  are not getting frozen ???

            elsif Ekind (Ent) = E_Entry
              and then Nkind (Parent (N)) = N_Task_Body
            then
               Error_Msg_N
                 ("entry address must be specified in task spec", Nam);

            else
               Note_Possible_Modification (Nam);
               Analyze_And_Resolve (Expr, RTE (RE_Address));

               --  Prior defined constant is permitted (RM 13.1(22))

               if Is_Entity_Name (Expr)
                 and then
                   (Ekind (Entity (Expr)) = E_Constant
                     or else
                    Ekind (Entity (Expr)) = E_In_Parameter)
                 and then
                   Defined_Before (Entity (Expr), Ent)
               then
                  Set_Address_Clause (Ent, N);
                  Kill_Size_Check_Code (Ent);

               --  We also permit X'address where x is defined before entity

               elsif Nkind (Expr) = N_Attribute_Reference
                 and then Attribute_Name (Expr) = Name_Address
                 and then Is_Entity_Name (Prefix (Expr))
               then
                  if Defined_Before (Entity (Prefix (Expr)), Ent) then
                     Set_Address_Clause (Ent, N);
                     Kill_Size_Check_Code (Ent);

                  else
                     Error_Msg_NE ("invalid address clause for &!", Expr, Ent);
                     Error_Msg_Name_1 := Chars (Entity (Prefix (Expr)));
                     Error_Msg_Name_2 := Chars (Ent);
                     Error_Msg_N ("% must be defined before %!", Expr);
                  end if;

               --  Here if not one of the permitted forms

               else
                  Error_Msg_NE ("invalid address clause for &!", Expr, Ent);
                  Error_Msg_NE
                    ("must be constant defined before& ('R'M 13.1(22))!",
                     Expr, Ent);
               end if;
            end if;
         end Address;

         ---------------
         -- Alignment --
         ---------------

         --  Alignment attribute definition clause

         when Attribute_Alignment => Alignment : declare
            Align : Uint := Static_Integer (Expr);

         begin
            Note_Feature (New_Representation_Clauses, Sloc (N));

            if not Is_Type (Ent)
              and then Ekind (Ent) /= E_Variable
              and then Ekind (Ent) /= E_Constant
            then
               Error_Msg_N ("alignment cannot be given for &", Nam);

            elsif Has_Alignment_Clause (Ent) then
               Error_Msg_Sloc := Sloc (Alignment_Clause (Ent));
               Error_Msg_N ("alignment clause previously given#", N);

            elsif Align /= No_Uint then
               if Align < 0 then
                  Error_Msg_N ("negative alignment not allowed", Expr);

               elsif Align > Maximum_Alignment then
                  Error_Msg_Uint_1 := UI_From_Int (Maximum_Alignment);
                  Error_Msg_N
                    ("?alignment exceeds ^ (maximum allowed for target)", N);

               else
                  Set_Alignment_Clause (Ent, N);
                  Set_Has_Alignment_Clause (Ent);
               end if;
            end if;
         end Alignment;

         ---------------
         -- Bit_Order --
         ---------------

         --  Bit_Order attribute definition clause

         when Attribute_Bit_Order => Bit_Order : declare
         begin
            Note_Feature (New_Representation_Clauses, Sloc (N));

            if not Is_Record_Type (Ent) then
               Error_Msg_N ("& definition requires record type", Nam);

            else
               Analyze_And_Resolve (Expr, RTE (RE_Bit_Order));

               if Etype (Expr) = Any_Type then
                  return;

               elsif not Is_Static_Expression (Expr) then
                  Error_Msg_N ("& requires static expression", Expr);

               else
                  if (Expr_Value (Expr) = 0) /= Bytes_Big_Endian then
                     Error_Msg_N ("unsupported value for & attribute", Expr);
                  end if;
               end if;
            end if;
         end Bit_Order;

         --------------------
         -- Component_Size --
         --------------------

         --  Component_Size attribute definition clause

         when Attribute_Component_Size => Component_Size_Case : declare
            Csize : constant Uint      := Static_Integer (Expr);
            Btype : Entity_Id;

         begin
            Note_Feature (New_Representation_Clauses, Sloc (N));

            if not Is_Array_Type (Ent) then
               Error_Msg_N ("component size requires array type", Nam);
               return;
            else
               Btype := Base_Type (Ent);
            end if;

            if Has_Component_Size_Clause (Btype) then
               Error_Msg_N
                 ("component size clase for& previously given", Nam);

            elsif Csize /= No_Uint then
               Check_Size (Expr, Component_Type (Btype), Csize);

               --  Note that Gigi is in charge of checking that the size we
               --  are assigning is acceptable, and will generate the error
               --  message if the size is inappropriate.

               Set_Component_Size (Btype, Csize);
               Set_Has_Component_Size_Clause (Btype);
               Set_Has_Non_Standard_Rep (Btype);

               --  If the component size is 4 or less, set the packed flag
               --  since we need to activate the front end packing circuit.

               if Csize <= 4 then
                  Set_Is_Packed (Btype);
               end if;
            end if;
         end Component_Size_Case;

         ------------------
         -- External_Tag --
         ------------------

         when Attribute_External_Tag => External_Tag :
         begin
            if not Is_Tagged_Type (Ent) then
               Error_Msg_N ("should be a tagged type", Nam);
            end if;

            Analyze_And_Resolve (Expr, Standard_String);

            if not Is_Static_Expression (Expr) then
               Error_Msg_N ("must be a static string", Nam);
            end if;

            Set_Has_External_Tag_Rep_Clause (Ent);
         end External_Tag;

         -----------
         -- Input --
         -----------

         when Attribute_Input => Input : declare
            Subp  : Entity_Id;

            function Has_Good_Profile (Subp : Entity_Id) return Boolean;
            --  Return true if the entity is a function with the good
            --  profile for the input attribute.

            function Has_Good_Profile (Subp : Entity_Id) return Boolean is
               F  : Entity_Id;
               Ok : Boolean := False;

            begin
               if Ekind (Subp) = E_Function then
                  F := First_Formal (Subp);

                  if Present (F) and then No (Next_Formal (F)) then
                     if Ekind (Etype (F)) = E_Anonymous_Access_Type
                       and then Designated_Type (Etype (F)) =
                         Class_Wide_Type (RTE (RE_Root_Stream_Type))
                     then
                        Ok := Base_Type (Etype (Subp)) = Base_Type (Ent);
                     end if;
                  end if;
               end if;
               return Ok;
            end Has_Good_Profile;

         --  Start of processing for Input

         begin
            Note_Feature (New_Representation_Clauses, Sloc (N));

            if not Is_Type (Ent) then
               Error_Msg_N ("local name must be a subtype", Nam);
               return;
            end if;

            Subp := Current_Entity (Expr);   -- beginning of homonym chain.

            while Present (Subp) loop
               exit when Has_Good_Profile (Subp);
               Subp := Homonym (Subp);
            end loop;

            if Present (Subp) then
               Set_Entity (Expr, Subp);
               Set_Etype (Expr, Etype (Subp));
            else
               Error_Msg_N ("incorrect expression for input attribute", Expr);
               return;
            end if;
         end Input;

         -------------------
         -- Machine_Radix --
         -------------------

         --  Machine radix attribute definition clause

         when Attribute_Machine_Radix => Machine_Radix : declare
            Radix : constant Uint := Static_Integer (Expr);

         begin
            Note_Feature (New_Representation_Clauses, Sloc (N));

            if not Is_Decimal_Fixed_Point_Type (Ent) then
               Error_Msg_N ("decimal fixed-point type expected for &", Nam);

            elsif Has_Machine_Radix_Clause (Ent) then
               Error_Msg_Sloc := Sloc (Alignment_Clause (Ent));
               Error_Msg_N ("machine radix clause previously given#", N);

            elsif Radix /= No_Uint then
               Set_Has_Machine_Radix_Clause (Ent);
               Set_Has_Non_Standard_Rep (Base_Type (Ent));

               if Radix = 2 then
                  null;
               elsif Radix = 10 then
                  Set_Machine_Radix_10 (Ent);
               else
                  Error_Msg_N ("machine radix value must be 2 or 10", Expr);
               end if;
            end if;
         end Machine_Radix;

         ------------
         -- Output --
         ------------

         when Attribute_Output => Output : declare
            Subp        : Entity_Id;

            function Has_Good_Profile (Subp : Entity_Id) return Boolean;
            --  return true if the entity is a procedure with the good
            --  profile for the output attribute.

            function Has_Good_Profile (Subp : Entity_Id) return Boolean is
               F  : Entity_Id;
               Ok : Boolean := False;

            begin
               if Ekind (Subp) = E_Procedure then
                  F := First_Formal (Subp);

                  if Present (F) then
                     if Ekind (Etype (F)) = E_Anonymous_Access_Type
                       and then Designated_Type (Etype (F)) =
                         Class_Wide_Type (RTE (RE_Root_Stream_Type))
                     then
                        F := Next_Formal (F);
                        Ok :=  Present (F)
                          and then Parameter_Mode (F) = E_In_Parameter
                          and then Base_Type (Etype (F)) = Base_Type (Ent)
                          and then No (Next_Formal (F));
                     end if;
                  end if;
               end if;
               return Ok;
            end Has_Good_Profile;

         begin
            Note_Feature (New_Representation_Clauses, Sloc (N));

            if not Is_Type (Ent) then
               Error_Msg_N ("local name must be a subtype", Nam);
               return;
            end if;

            Subp := Current_Entity (Expr);   -- beginning of homonym chain.

            while Present (Subp) loop
               exit when Has_Good_Profile (Subp);
               Subp := Homonym (Subp);
            end loop;

            if Present (Subp) then
               Set_Entity (Expr, Subp);
               Set_Etype (Expr, Etype (Subp));
            else
               Error_Msg_N ("incorrect expression for read attribute", Expr);
               return;
            end if;
         end Output;

         ----------
         -- Read --
         ----------

         when Attribute_Read => Read : declare
            Subp        : Entity_Id;

            function Has_Good_Profile (Subp : Entity_Id) return Boolean;
            --  return true if the entity is a procedure with the good
            --  profile for the read attribute.

            function Has_Good_Profile (Subp : Entity_Id) return Boolean is
               F     : Entity_Id;
               Ok    : Boolean := False;

            begin
               if Ekind (Subp) = E_Procedure then
                  F := First_Formal (Subp);

                  if Present (F) then
                     if Ekind (Etype (F)) = E_Anonymous_Access_Type
                       and then Designated_Type (Etype (F)) =
                        Class_Wide_Type (RTE (RE_Root_Stream_Type))
                     then
                        F := Next_Formal (F);
                        Ok :=  Present (F)
                          and then Parameter_Mode (F) = E_Out_Parameter
                          and then Base_Type (Etype (F)) = Base_Type (Ent)
                          and then No (Next_Formal (F));
                     end if;
                  end if;
               end if;
               return Ok;
            end Has_Good_Profile;

         begin
            Note_Feature (New_Representation_Clauses, Sloc (N));

            if not Is_Type (Ent) then
               Error_Msg_N ("local name must be a subtype", Nam);
               return;
            end if;

            Subp := Current_Entity (Expr);   -- beginning of homonym chain.

            while Present (Subp) loop
               exit when Has_Good_Profile (Subp);
               Subp := Homonym (Subp);
            end loop;

            if Present (Subp) then
               Set_Entity (Expr, Subp);
               Set_Etype (Expr, Etype (Subp));
            else
               Error_Msg_N ("incorrect expression for read attribute", Expr);
               return;
            end if;
         end Read;

         ----------
         -- Size --
         ----------

         --  Size attribute definition clause

         when Attribute_Size => Size : declare
            Size : constant Uint := Static_Integer (Expr);
            Etyp  : Entity_Id;

         begin
            if Has_Size_Clause (Ent) then
               Error_Msg_N ("size already given for &", Nam);

            elsif not Is_Type (Ent)
              and then Ekind (Ent) /= E_Variable
              and then Ekind (Ent) /= E_Constant
            then
               Error_Msg_N ("size cannot be given for &", Nam);

            elsif Size /= No_Uint then

               if Is_Type (Ent) then
                  Etyp := Ent;
               else
                  Etyp := Etype (Ent);
               end if;

               --  Check size, note that Gigi is in charge of checking
               --  that the size of an array or record type is OK. Also
               --  we do not check the size in the ordinary fixed-point
               --  case, since it is too early to do so (there may be
               --  a subsequent small clause that affects the size). We
               --  can check the size if a small clause has already been
               --  given.

               if not Is_Ordinary_Fixed_Point_Type (Ent)
                 or else Has_Small_Clause (Ent)
               then
                  Check_Size (Expr, Etyp, Size);
               end if;

               Set_Esize (Ent, Size);
               Set_Has_Size_Clause (Ent);
            end if;
         end Size;

         -----------
         -- Small --
         -----------

         --  Small attribute definition clause

         when Attribute_Small => Small : declare
            Int_Type      : Entity_Id;
            Implicit_Base : constant Entity_Id := Base_Type (Ent);
            Small         : Ureal;
            Size_Min      : Nat;

         begin
            Analyze_And_Resolve (Expr, Any_Real);

            if Etype (Expr) = Any_Type then
               return;

            elsif not Is_Static_Expression (Expr) then
               Error_Msg_N ("small requires static expression", Expr);
               return;

            else
               Small := Expr_Value_R (Expr);
            end if;

            if not Is_Ordinary_Fixed_Point_Type (Ent) then
               Error_Msg_N
                 ("small requires an ordinary fixed point type", Nam);

            elsif Has_Small_Clause (Ent) then
               Error_Msg_N ("small already given for &", Nam);

            elsif Small > Delta_Value (Ent) then
               Error_Msg_N
                 ("small value must not be greater then delta value", Nam);

            else
               Set_Small_Value (Ent, Small);
               Set_Small_Value (Implicit_Base, Small);
               Set_Has_Small_Clause (Ent);
               Set_Has_Small_Clause (Implicit_Base);
               Set_Has_Non_Standard_Rep (Implicit_Base);
            end if;
         end Small;

         ------------------
         -- Storage_Size --
         ------------------

         --  Storage_Size attribute definition clause

         when Attribute_Storage_Size => Storage_Size : declare
            Btype : constant Entity_Id := Base_Type (Ent);

         begin
            if not Is_Access_Type (Ent)
              and then Ekind (Ent) /= E_Task_Type
            then
               Error_Msg_N ("storage size cannot be given for &", Nam);

            elsif Is_Access_Type (Ent) and Is_Derived_Type (Ent) then
               Error_Msg_N
                 ("storage size cannot be given for a derived access type",
                  Nam);

            elsif Has_Storage_Size_Clause (Btype) then
               Error_Msg_N ("storage size already given for &", Nam);

            else
               Analyze_And_Resolve (Expr, Any_Integer);

               if Is_Access_Type (Ent)
                 and then Present (Associated_Storage_Pool (Ent))
               then
                  Error_Msg_N ("storage pool already given for &", Nam);
                  return;
               else
                  Set_Has_Storage_Size_Clause (Btype);
               end if;
            end if;
         end Storage_Size;

         ------------------
         -- Storage_Pool --
         ------------------

         --  Storage_Pool attribute definition clause

         when Attribute_Storage_Pool => Storage_Pool : declare
            Pool : Entity_Id;

         begin
            Note_Feature (New_Representation_Clauses, Sloc (N));
            Note_Feature (User_Defined_Storage_Pools, Sloc (N));

            if Ekind (Ent) /= E_Access_Type
              and then Ekind (Ent) /= E_General_Access_Type
            then
               Error_Msg_N (
                 "storage pool can only be given for access types", Nam);
               return;

            elsif Is_Derived_Type (Ent) then
               Error_Msg_N
                 ("storage pool cannot be given for a derived access type",
                  Nam);

            elsif Has_Storage_Size_Clause (Ent) then
               Error_Msg_N ("storage size already given for &", Nam);
               return;

            elsif Present (Associated_Storage_Pool (Ent)) then
               Error_Msg_N ("storage pool already given for &", Nam);
               return;
            end if;

            Analyze_And_Resolve
              (Expr, Class_Wide_Type (RTE (RE_Root_Storage_Pool)));

            if Is_Entity_Name (Expr) then
               Pool := Entity (Expr);

               if Present (Etype (Pool))
                 and then Etype (Pool) /= RTE (RE_Stack_Bounded_Pool)
                 and then Etype (Pool) /= RTE (RE_Unbounded_Reclaim_Pool)
               then
                  Set_Associated_Storage_Pool (Ent, Pool);
               else
                  Error_Msg_N ("Non sharable GNAT Pool", Expr);
               end if;

            else
               Error_Msg_N ("incorrect reference to a Storage Pool", Expr);
               return;
            end if;
         end Storage_Pool;

         -----------
         -- Write --
         -----------

         --  Write attribute definition clause
         --  check for class-wide case will be performed later

         when Attribute_Write => Write : declare
            Subp        : Entity_Id;

            function Has_Good_Profile (Subp : Entity_Id) return Boolean;
            --  return true if the entity is a procedure with the good
            --  profile for the write attribute.

            function Has_Good_Profile (Subp : Entity_Id) return Boolean is
               F     : Entity_Id;
               Ok    : Boolean := False;

            begin
               if Ekind (Subp) = E_Procedure then
                  F := First_Formal (Subp);

                  if Present (F) then
                     if Ekind (Etype (F)) = E_Anonymous_Access_Type
                       and then Designated_Type (Etype (F)) =
                         Class_Wide_Type (RTE (RE_Root_Stream_Type))
                     then
                        F := Next_Formal (F);
                        Ok :=  Present (F)
                          and then Parameter_Mode (F) = E_In_Parameter
                          and then Base_Type (Etype (F)) = Base_Type (Ent)
                          and then No (Next_Formal (F));
                     end if;
                  end if;
               end if;
               return Ok;
            end Has_Good_Profile;

         begin
            Note_Feature (New_Representation_Clauses, Sloc (N));

            if not Is_Type (Ent) then
               Error_Msg_N ("local name must be a subtype", Nam);
               return;
            end if;

            Subp := Current_Entity (Expr);   -- beginning of homonym chain.

            while Present (Subp) loop
               exit when Has_Good_Profile (Subp);
               Subp := Homonym (Subp);
            end loop;

            if Present (Subp) then
               Set_Entity (Expr, Subp);
               Set_Etype (Expr, Etype (Subp));
            else
               Error_Msg_N ("incorrect expression for write attribute", Expr);
               return;
            end if;
         end Write;

         --  All other attributes cannot be set

         when others =>
            Error_Msg_N
              ("attribute& cannot be set with definition clause", N);

      end case;

      --  The test for the type being frozen must be performed after
      --  any expression the clause has been analyzed since the expression
      --  itself might cause freezing that makes the clause illegal.

      if Already_Frozen (Ent, Nam) then
         return;
      end if;
   end Analyze_Attribute_Definition_Clause;

   ----------------------------
   -- Analyze_Code_Statement --
   ----------------------------

   procedure Analyze_Code_Statement (N : Node_Id) is
   begin
      Unimplemented (N, "code statement");
   end Analyze_Code_Statement;

   -----------------------------------------------
   -- Analyze_Enumeration_Representation_Clause --
   -----------------------------------------------

   procedure Analyze_Enumeration_Representation_Clause (N : Node_Id) is
      Loc      : constant Source_Ptr := Sloc (N);
      Ident    : constant Node_Id    := Identifier (N);
      Aggr     : constant Node_Id    := Array_Aggregate (N);
      Enumtype : Entity_Id;
      Elit     : Entity_Id;
      Expr     : Node_Id;
      Assoc    : Node_Id;
      Choice   : Node_Id;
      Val      : Uint;
      Err      : Boolean := False;

      Lo  : constant Uint := Expr_Value (Type_Low_Bound (Universal_Integer));
      Hi  : constant Uint := Expr_Value (Type_High_Bound (Universal_Integer));
      Min : Uint;
      Max : Uint;

   begin
      --  First some basic error checks

      Find_Type (Ident);
      Enumtype := Entity (Ident);

      if not Is_Enumeration_Type (Enumtype) then
         Error_Msg_NE
           ("enumeration type required, found}",
            Ident, First_Subtype (Enumtype));
         return;
      end if;

      if Scope (Enumtype) /= Current_Scope then
         Error_Msg_N ("type must be declared in this scope", Ident);
         return;

      elsif not Is_First_Subtype (Enumtype) then
         Error_Msg_N ("cannot give enumeration rep clause for subtype", Ident);
         return;

      elsif Has_Enumeration_Rep_Clause (Enumtype) then
         Error_Msg_N ("duplicate enumeration rep clause ignored", N);
         return;

      elsif Already_Frozen (Enumtype, Ident) then
         return;

      elsif Root_Type (Enumtype) = Standard_Character
        or else Root_Type (Enumtype) = Standard_Wide_Character
        or else Root_Type (Enumtype) = Standard_Boolean
      then
         Error_Msg_N ("enumeration rep clause not allowed for this type", N);

      else
         Set_Has_Enumeration_Rep_Clause (Enumtype);
         Set_Has_Enumeration_Rep_Clause (Base_Type (Enumtype));
         Set_Has_Non_Standard_Rep (Base_Type (Enumtype));
      end if;

      --  Now we process the aggregate. Note that we don't use the normal
      --  aggregate code for this purpose, because we don't want any of the
      --  normal expansion activities, and a number of special semantic
      --  rules apply (including the component type being any integer type)

      --  Badent signals that we found some incorrect entries processing
      --  the list. The final checks for completeness and ordering are
      --  skipped in this case.

      Elit := First_Literal (Enumtype);

      --  First the positional entries if any

      if Present (Expressions (Aggr)) then
         Expr := First (Expressions (Aggr));
         while Present (Expr) loop

            if No (Elit) then
               Error_Msg_N ("too many entries in aggregate", Expr);
               return;
            end if;

            Val := Static_Integer (Expr);

            if Val = No_Uint then
               Err := True;

            elsif Val < Lo or else Hi < Val then
               Error_Msg_N ("value outside permitted range", Expr);
               Err := True;
            end if;

            Set_Enumeration_Rep (Elit, Val);
            Set_Enumeration_Rep_Expr (Elit, Expr);
            Expr := Next (Expr);
            Elit := Next (Elit);
         end loop;
      end if;

      --  Now process the named entries if present

      if Present (Component_Associations (Aggr)) then
         Assoc := First (Component_Associations (Aggr));
         while Present (Assoc) loop
            Choice := First (Choices (Assoc));

            if Present (Next (Choice)) then
               Error_Msg_N
                 ("multiple choice not allowed here", Next (Choice));
               Err := True;
            end if;

            if Nkind (Choice) = N_Others_Choice then
               Error_Msg_N ("others choice not allowed here", Choice);
               Err := True;

            elsif Nkind (Choice) = N_Range then
               --  ??? should allow zero/one element range here
               Error_Msg_N ("range not allowed here", Choice);
               Err := True;

            else
               Analyze_And_Resolve (Choice, Enumtype);

               if Is_Entity_Name (Choice)
                 and then Is_Type (Entity (Choice))
               then
                  Error_Msg_N ("subtype name not allowed here", Choice);
                  Err := True;
                  --  ??? should allow static subtype with zero/one entry

               elsif Etype (Choice) = Base_Type (Enumtype) then
                  if not Is_Static_Expression (Choice) then
                     Error_Msg_N
                       ("non-static expression used for choice", Choice);
                     Err := True;
                  else
                     Elit := Expr_Value_E (Choice);

                     if Present (Enumeration_Rep_Expr (Elit)) then
                        Error_Msg_Sloc := Sloc (Enumeration_Rep_Expr (Elit));
                        Error_Msg_NE
                          ("representation for& previously given#",
                           Choice, Elit);
                        Err := True;
                     end if;

                     Set_Enumeration_Rep_Expr (Elit, Choice);

                     Val := Static_Integer (Expression (Assoc));

                     if Val = No_Uint then
                        Err := True;
                     elsif Val < Lo or else Hi < Val then
                        Error_Msg_N ("value outside permitted range", Expr);
                        Err := True;
                     end if;

                     Set_Enumeration_Rep (Elit, Val);
                  end if;
               end if;
            end if;

            Assoc := Next (Assoc);
         end loop;
      end if;

      --  Aggregate is fully processed. Now we check that a full set of
      --  representations was given, and that they are in range and in order.
      --  These checks are only done if no other errors occurred.

      if not Err then
         Min  := No_Uint;
         Max  := No_Uint;

         Elit := First_Literal (Enumtype);
         while Present (Elit) loop
            if No (Enumeration_Rep_Expr (Elit)) then
               Error_Msg_NE ("missing representation for&!", N, Elit);

            else
               Val := Enumeration_Rep (Elit);

               if Min = No_Uint then
                  Min := Val;
               end if;

               if Val /= No_Uint then
                  if Max /= No_Uint and then Val <= Max then
                     Error_Msg_NE
                       ("enumeration value for& not ordered!",
                                       Enumeration_Rep_Expr (Elit), Elit);
                  end if;

                  Max := Val;
               end if;

            end if;

            Elit := Next (Elit);
         end loop;
      end if;

      if Has_Size_Clause (Enumtype) then
         if Esize (Enumtype) >= Minimum_Size (Enumtype) then
            return;
         else
            Error_Msg_N ("previously given size is too small", N);
         end if;
      end if;

      --  If we don't have a given size, or if the size given was too
      --  small, then compute an appropriate size for the values given.

      Determine_Enum_Representation (Enumtype);

   end Analyze_Enumeration_Representation_Clause;

   ----------------------------
   -- Analyze_Free_Statement --
   ----------------------------

   procedure Analyze_Free_Statement (N : Node_Id) is
   begin
      Analyze (Expression (N));
   end Analyze_Free_Statement;

   ------------------------------------------
   -- Analyze_Record_Representation_Clause --
   ------------------------------------------

   procedure Analyze_Record_Representation_Clause (N : Node_Id) is
      Loc     : constant Source_Ptr := Sloc (N);
      Ident   : constant Node_Id    := Identifier (N);
      Rectype : Entity_Id;
      Mod_Val : Uint;
      CC      : Node_Id;
      Posit   : Uint;
      Fbit    : Uint;
      Lbit    : Uint;
      Hbit    : Uint := Uint_0;
      Comp    : Entity_Id;
      Ocomp   : Entity_Id;

   begin
      Find_Type (Ident);
      Rectype := Entity (Ident);

      --  First some basic error checks

      if not Is_Record_Type (Rectype) then
         Error_Msg_NE
           ("record type required, found}", Ident, First_Subtype (Rectype));
         return;

      elsif Is_Tagged_Type (Rectype) then
         Error_Msg_N ("record rep clause not allowed for tagged type", Ident);
         return;

      elsif Scope (Rectype) /= Current_Scope then
         Error_Msg_N ("type must be declared in this scope", Ident);
         return;

      elsif not Is_First_Subtype (Rectype) then
         Error_Msg_N ("cannot give record rep clause for subtype", Ident);
         return;

      elsif Has_Record_Rep_Clause (Rectype) then
         Error_Msg_N ("duplicate record rep clause ignored", N);
         return;

      elsif Already_Frozen (Rectype, Ident) then
         return;
      end if;

      --  OK, looks like all is well, so process the rep clause

      Set_Has_Record_Rep_Clause (Rectype);
      Set_Has_Specified_Layout  (Rectype);

      --  A representation like this applies to the base type as well

      Set_Has_Record_Rep_Clause (Base_Type (Rectype));
      Set_Has_Non_Standard_Rep  (Base_Type (Rectype));
      Set_Has_Specified_Layout  (Base_Type (Rectype));

      if Present (Mod_Clause (N)) then
         declare
            M : constant Node_Id := Mod_Clause (N);
            P : constant List_Id := Pragmas_Before (M);

         begin
            if Present (P) then
               Analyze_List (P);
            end if;

            Mod_Val := Static_Integer (Expression (M));
         end;
      end if;

      --  Clear any existing component clauses for the type (this happens
      --  with derived types, where we are now overriding the original)

      Comp := First_Entity (Rectype);
      while Present (Comp) loop
         if Ekind (Comp) = E_Component
           or else Ekind (Comp) = E_Discriminant
         then
            Set_Component_Clause (Comp, Empty);
         end if;

         Comp := Next_Entity (Comp);
      end loop;

      --  Process the component clauses

      CC := First (Component_Clauses (N));

      while Present (CC) loop

         --  If pragma, just analyze it

         if Nkind (CC) = N_Pragma then
            Analyze (CC);

         --  Processing for real component clause

         else
            Posit := Static_Integer (Position  (CC));
            Fbit  := Static_Integer (First_Bit (CC));
            Lbit  := Static_Integer (Last_Bit  (CC));

            if Posit /= No_Uint
              and then Fbit /= No_Uint
              and then Lbit /= No_Uint
            then
               if Posit < 0 then
                  Error_Msg_N
                    ("position cannot be negative", Position (CC));

               elsif Fbit < 0 then
                  Error_Msg_N
                    ("first bit cannot be negative", First_Bit (CC));

               --  Values look OK, so find the corresponding record component

               else
                  Comp := First_Entity (Rectype);
                  while Present (Comp) loop
                     exit when Chars (Comp) = Chars (Component_Name (CC));
                     Comp := Next_Entity (Comp);
                  end loop;

                  if No (Comp) then
                     Error_Msg_N
                       ("component clause is for non-existent field", N);

                  elsif Present (Component_Clause (Comp)) then
                     Error_Msg_Sloc := Sloc (Component_Clause (Comp));
                     Error_Msg_N ("component clause previously given#", CC);

                  else
                     --  Update Fbit and Lbit to the actual bit number.

                     Fbit := Fbit + UI_From_Int (System_Storage_Unit) * Posit;
                     Lbit := Lbit + UI_From_Int (System_Storage_Unit) * Posit;

                     if Has_Size_Clause (Rectype)
                       and then Esize (Rectype) <= Lbit
                     then
                        Error_Msg_N
                          ("bit number out of range of specified size",
                           Last_Bit (CC));
                     else
                        Set_Component_Clause    (Comp, CC);
                        Set_Component_First_Bit (Comp, Fbit);
                        Set_Esize               (Comp, 1 + (Lbit - Fbit));

                        --  This information is also set in the corresponding
                        --  component of the base type, found by accessing the
                        --  Original_Record_Component link if it is present.

                        Ocomp := Original_Record_Component (Comp);

                        if Present (Ocomp) then
                           Set_Component_Clause    (Ocomp, CC);
                           Set_Component_First_Bit (Ocomp, Fbit);
                           Set_Esize               (Ocomp, 1 + (Lbit - Fbit));
                        end if;

                        if Hbit < Lbit then
                           Hbit := Lbit;
                        end if;

                        Check_Size (Component_Name (CC),
                          Etype (Comp), Esize (Comp));

                        if Esize (Comp) < 0 then
                           Error_Msg_N ("component size is negative", CC);
                        end if;
                     end if;
                  end if;
               end if;
            end if;
         end if;

         CC := Next (CC);
      end loop;

      --  Now that we have processed all the component clauses, check for
      --  overlap. We have to leave this till last, since the components
      --  can appear in any arbitrary order in the representation clause.

      Overlap_Check : declare
         C1_Ent, C2_Ent : Entity_Id;
         --  Entities of components being checked for overlap

         Clist : Node_Id;
         --  Component_List node whose Component_Items are being checked

         Citem : Node_Id;
         --  Component declaration for component being checked

      begin
         C1_Ent := First_Entity (Rectype);

         --  Loop through all components in record. For each component check
         --  for overlap with any of the preceding elements on the component
         --  list containing the component, and also, if the component is in
         --  a variant, check against components outside the case structure.
         --  This latter test is repeated recursively up the variant tree.

         Main_Component_Loop : while Present (C1_Ent) loop
            if Ekind (C1_Ent) /= E_Component
              and then Ekind (C1_Ent) /= E_Discriminant
            then
               goto Continue_Main_Component_Loop;
            end if;

            --  Skip overlap check if entity has no declaration node. This
            --  happens with discriminants in constrained derived types.
            --  Probably we are missing some checks as a result, but that
            --  does not seem terribly serious ???

            if No (Declaration_Node (C1_Ent)) then
               goto Continue_Main_Component_Loop;
            end if;

            Clist := Parent (List_Containing (Declaration_Node (C1_Ent)));

            --  Loop through component lists that need checking. We check the
            --  current component list and all lists in variants above us.

            Component_List_Loop : loop

               --  If at outer level, check discriminants if there are any

               if Nkind (Clist) = N_Full_Type_Declaration
                 or else Nkind (Clist) = N_Private_Type_Declaration
               then
                  if Has_Discriminants (Defining_Identifier (Clist)) then
                     C2_Ent :=
                       First_Discriminant (Defining_Identifier (Clist));

                     while Present (C2_Ent) loop
                        exit when C1_Ent = C2_Ent;
                        Check_Component_Overlap (C1_Ent, C2_Ent);
                        C2_Ent := Next_Discriminant (C2_Ent);
                     end loop;
                  end if;

               --  Otherwise check one component list

               else
                  Citem := First (Component_Items (Clist));

                  while Present (Citem) loop
                     if Nkind (Citem) = N_Component_Declaration then
                        C2_Ent := Defining_Identifier (Citem);
                        exit when C1_Ent = C2_Ent;
                        Check_Component_Overlap (C1_Ent, C2_Ent);
                     end if;

                     Citem := Next (Citem);
                  end loop;
               end if;

               --  Check for variants above us (the parent of the Clist can be
               --  a variant, in which case its parent is a variant part, and
               --  the parent of the variant part is a component list whose
               --  components must all be checked against the current component
               --  for overlap.

               if Nkind (Parent (Clist)) = N_Variant then
                  Clist := Parent (Parent (Parent (Clist)));

               --  Check for possible discriminant part in record, this is
               --  treated essentially as another level in the recursion. For
               --  this case we have the parent of the component list is the
               --  record definition, and its parent is the full type
               --  declaration which contains the discriminant specifications.

               elsif Nkind (Parent (Clist)) = N_Record_Definition then
                  Clist := Parent (Parent ((Clist)));

               --  If neither of these two cases, we are at the top of the tree

               else
                  exit Component_List_Loop;
               end if;
            end loop Component_List_Loop;

            <<Continue_Main_Component_Loop>>
               C1_Ent := Next_Entity (C1_Ent);

         end loop Main_Component_Loop;

      end Overlap_Check;

      Set_Esize (Rectype, Hbit + 1);

   end Analyze_Record_Representation_Clause;

   -----------------------------
   -- Check_Component_Overlap --
   -----------------------------

   procedure Check_Component_Overlap (C1_Ent, C2_Ent : Entity_Id) is
   begin
      if Present (Component_Clause (C1_Ent))
        and then Present (Component_Clause (C2_Ent))
      then
         declare
            S1 : constant Uint := Component_First_Bit (C1_Ent);
            S2 : constant Uint := Component_First_Bit (C2_Ent);
            E1 : constant Uint := S1 + Esize (C1_Ent);
            E2 : constant Uint := S2 + Esize (C2_Ent);

         begin
            if E2 <= S1 or else E1 <= S2 then
               null;
            else
               Error_Msg_Node_2 :=
                 Component_Name (Component_Clause (C2_Ent));
               Error_Msg_Sloc := Sloc (Error_Msg_Node_2);
               Error_Msg_Node_1 :=
                 Component_Name (Component_Clause (C1_Ent));
               Error_Msg_N
                 ("component& overlaps & #",
                  Component_Name (Component_Clause (C1_Ent)));
            end if;
         end;
      end if;
   end Check_Component_Overlap;

   ----------------
   -- Check_Size --
   ----------------

   procedure Check_Size (N : Node_Id; T : Entity_Id; Siz : Uint) is
      UT : constant Entity_Id := Underlying_Type (T);
      M  : Uint;

   begin
      --  Immediate return if size is same as standard size or if composite
      --  item with no size available (i.e. none was given explicitly) or
      --  generic type, or type with previous errors.

      if No (UT) or else Esize (UT) = 0 or else Siz = Esize (UT) then
         return;

      --  If type has record representation clause, the saved size is
      --  the mimimum size.

      elsif Is_Record_Type (UT) and then Has_Record_Rep_Clause (UT) then
         if Siz < Esize (UT) then
            Error_Msg_Uint_1 := Esize (UT);
            Error_Msg_NE ("size for& too small, minimum allowed is ^", N, T);
         end if;

      --  If the type is a fat pointer, then allow specifying a thin
      --  pointer.

      elsif Is_Access_Type (UT)
         and then Esize (UT) = System_Address_Size * 2
         and then Siz = System_Address_Size
      then
         null;

      --  If the type is a thin pointer to a non-Taft-amendment type that
      --  is an unconstrained array, allow it to be a fat pointer.

      elsif Is_Access_Type (UT)
         and then not Has_Completion_In_Body (Designated_Type (UT))
         and then Is_Array_Type (Designated_Type (UT))
         and then not Is_Constrained (Designated_Type (UT))
         and then Esize (UT) = System_Address_Size
         and then Siz = System_Address_Size * 2
      then
         null;

      --  Types for which the only permitted size is the standard size

      elsif Is_Floating_Point_Type (UT)
        or else (Is_Access_Type (UT)
                  and then Esize (UT) = System_Address_Size)
        or else Is_Composite_Type (UT)
      then
         Error_Msg_Uint_1 := Esize (UT);
         Error_Msg_NE ("incorrect size for&, must be exactly ^", N, T);

      --  Fat-pointer types, which allow one of two sizes

      elsif Is_Access_Type (UT) then
         Error_Msg_Uint_1 := UI_From_Int (System_Address_Size);
         Error_Msg_Uint_2 := UI_From_Int (2 * System_Address_Size);
         Error_Msg_NE ("incorrect size for&, must be exactly ^ or ^", N, T);

      --  For remaining types, maximum size is Long_Long_Integer size

      elsif Siz > Standard_Long_Long_Integer_Size then
         Error_Msg_Uint_1 := UI_From_Int (Standard_Long_Long_Integer_Size);
         Error_Msg_NE ("size for& too large, maximum allowed is ^", N, T);

      --  Cases for which a minimum check is required

      else
         M := UI_From_Int (Minimum_Size (UT));

         if Siz < M then
            Error_Msg_Uint_1 := M;
            Error_Msg_NE ("size for& too small, minimum allowed is ^", N, T);
         end if;
      end if;
   end Check_Size;

   --------------------
   -- Defined_Before --
   --------------------

   function Defined_Before (E1, E2 : Entity_Id) return Boolean is
      Loc1 : constant Source_Ptr := Sloc (E1);
      Loc2 : constant Source_Ptr := Sloc (E2);

   begin
      --  If E1 is in a different unit from E2, then clearly is is already
      --  defined, since the unit containing E1 is already processed.

      if Get_Sloc_Unit_Number (Loc1) /= Get_Sloc_Unit_Number (Loc2) then
         return True;

      --  Otherwise location of E1 must be less than location of E2

      else
         return Loc1 < Loc2;
      end if;
   end Defined_Before;

   ------------------
   -- Minimum_Size --
   ------------------

   function Minimum_Size (T : Entity_Id) return Nat is
      R_Type   : constant Entity_Id := Root_Type (T);
      Lo, Hi   : Uint;
      LoR, HiR : Ureal;
      B        : Uint;
      S        : Nat;

      function Get_Enum_Rep (N : Node_Id) return Uint;
      --  N is an expression representing a bound of an enumeration type.
      --  This function returns the corresponding enumeration representation,

      function Get_Enum_Rep (N : Node_Id) return Uint is
         Ent  : constant Entity_Id := Entity (N);

      begin
         if Ekind (Ent) = E_Enumeration_Literal then
            return Enumeration_Rep (Ent);
         else
            pragma Assert (Ekind (Ent) = E_Constant);
            return Get_Enum_Rep (Constant_Value (Ent));
         end if;
      end Get_Enum_Rep;

   --  Start of processing for Minimum_Size

   begin
      --  Integer types, and also types derived from [Wide_]Character. The
      --  latter are treated like integer types since the representation is
      --  known to be the same as the Pos value, which is what Expr_Value
      --  returns (they cannot be treated the same as other Enumeration
      --  types, since the enumeration literals do not exist)

      if Is_Integer_Type (T)
        or else R_Type = Standard_Character
        or else R_Type = Standard_Wide_Character
      then
         if Compile_Time_Known_Value (Type_Low_Bound (T)) then
            Lo := Expr_Value (Type_Low_Bound (T));
         else
            Lo := Expr_Value (Type_Low_Bound (Base_Type (T)));
         end if;

         if Compile_Time_Known_Value (Type_High_Bound (T)) then
            Hi := Expr_Value (Type_High_Bound (T));
         else
            Hi := Expr_Value (Type_High_Bound (Base_Type (T)));
         end if;

      --  Enumeration types (other than those derived from [Wide_]Character

      elsif Is_Enumeration_Type (T) then
         if Compile_Time_Known_Value (Type_Low_Bound (T)) then
            Lo := Get_Enum_Rep (Type_Low_Bound (T));
         else
            Lo := Get_Enum_Rep (Type_Low_Bound (Base_Type (T)));
         end if;

         if Compile_Time_Known_Value (Type_High_Bound (T)) then
            Hi := Get_Enum_Rep (Type_High_Bound (T));
         else
            Hi := Get_Enum_Rep (Type_High_Bound (Base_Type (T)));
         end if;

      --  Fixed-point types. We can't simply use Expr_Value to get the
      --  Corresponding_Integer_Value values of the bounds, since these
      --  do not get set till the type is frozen, and this routine can
      --  be called before the type is frozen. Similarly the test for
      --  bounds being static needs to include the case where we have
      --  unanalyzed real literals for the same reason.

      elsif Is_Fixed_Point_Type (T) then
         if Nkind (Type_Low_Bound (T)) = N_Real_Literal
           or else Compile_Time_Known_Value (Type_Low_Bound (T))
         then
            LoR := Expr_Value_R (Type_Low_Bound (T));
         else
            LoR := Expr_Value_R (Type_Low_Bound (Base_Type (T)));
         end if;

         if Nkind (Type_High_Bound (T)) = N_Real_Literal
           or else Compile_Time_Known_Value (Type_High_Bound (T))
         then
            HiR := Expr_Value_R (Type_High_Bound (T));
         else
            HiR := Expr_Value_R (Type_High_Bound (Base_Type (T)));
         end if;

         Lo := UR_To_Uint (LoR / Small_Value (T));
         Hi := UR_To_Uint (HiR / Small_Value (T));

      --  No other types allowed

      else
         pragma Assert (False);
         null;
      end if;

      --  Signed case

      if Lo < 0 then
         S := 1;
         B := Uint_1;

         --  S = size, B = 2 ** (size - 1) (can accomodate -B .. +(B - 1))

         while Lo < -B or else Hi > (B - 1) loop
            S := S + 1;
            B := B + B;
         end loop;

      --  Unsigned case

      else
         S := 0;
         B := Uint_1;

         --  S = size, B = 2 ** size (can accomodate 0 .. (B - 1))

         while Hi > (B - 1) loop
            S := S + 1;
            B := B + B;
         end loop;
      end if;

      return S;
   end Minimum_Size;

   -------------------------
   -- Same_Representation --
   -------------------------

   function Same_Representation (T1, T2 : Entity_Id) return Boolean is
   begin
      --  A quick check, if base types are the same, then we definitely have
      --  the same representation, because the subtype specific representation
      --  attributes (Size and Alignment) do not affect representation from
      --  the point of view of this test.

      if Base_Type (T1) = Base_Type (T2) then
         return True;
      end if;

      --  Representations are definitely different if conventions differ

      if Convention (T1) /= Convention (T2) then
         return False;
      end if;

      --  Types definitely have same representation if neither has non-standard
      --  representation since default representations are always consistent.
      --  If only one has non-standard representation, and the other does not,
      --  then we consider that they do not have the same representation. They
      --  might, but there is no way of telling early enough.

      if Has_Non_Standard_Rep (T1) then
         if not Has_Non_Standard_Rep (T2) then
            return False;
         end if;
      else
         return not Has_Non_Standard_Rep (T2);
      end if;

      --  Here the two types both have non-standard representation, and we
      --  need to determine if they have the same non-standard representation

      --  For arrays, we simply need to test if the component sizes are the
      --  same. Pragma Pack is reflected in modified component sizes, so this
      --  check also deals with pragma Pack.

      if Is_Array_Type (T1) then
         return Component_Size (T1) = Component_Size (T2);

      --  Tagged types always have the same representation, because it is not
      --  possible to specify different representations for common fields.

      elsif Is_Tagged_Type (T1) then
         return True;

      --  Case of record types

      elsif Is_Record_Type (T1) then

         --  Packed status must conform

         if Is_Packed (T1) /= Is_Packed (T2) then
            return False;

         --  Otherwise we must check components

         else
            Record_Case : declare
               CD1, CD2 : Entity_Id;

               function Same_Rep return Boolean;
               --  CD1 and CD2 are either components or discriminants. This
               --  function tests whether the two have the same representation

               function Same_Rep return Boolean is
               begin
                  if No (Component_Clause (CD1)) then
                     return No (Component_Clause (CD2));

                  else
                     return
                        Present (Component_Clause (CD2))
                          and then
                        Component_First_Bit (CD1) = Component_First_Bit (CD2)
                          and then
                        Esize (CD1) = Esize (CD2);
                  end if;
               end Same_Rep;

            --  Start processing for Record_Case

            begin
               if Has_Discriminants (T1) then
                  CD1 := First_Discriminant (T1);
                  CD2 := First_Discriminant (T2);

                  while Present (CD1) loop
                     if not Same_Rep then
                        return False;
                     else
                        CD1 := Next_Discriminant (CD1);
                        CD2 := Next_Discriminant (CD2);
                     end if;
                  end loop;
               end if;

               CD1 := First_Component (T1);
               CD2 := First_Component (T2);

               while Present (CD1) loop
                  if not Same_Rep then
                     return False;
                  else
                     CD1 := Next_Component (CD1);
                     CD2 := Next_Component (CD2);
                  end if;
               end loop;

               return True;
            end Record_Case;
         end if;

      --  For enumeration types, we must check each literal to see if the
      --  representation is the same. Note that we do not permit enumeration
      --  reprsentation clauses for Character and Wide_Character, so these
      --  cases were already dealt with.

      elsif Is_Enumeration_Type (T1) then

         Enumeration_Case : declare
            L1, L2 : Entity_Id;

         begin
            L1 := First_Literal (T1);
            L2 := First_Literal (T2);

            while Present (L1) loop
               if Enumeration_Rep (L1) /= Enumeration_Rep (L2) then
                  return False;
               else
                  L1 := Next_Literal (L1);
                  L2 := Next_Literal (L2);
               end if;
            end loop;

            return True;

         end Enumeration_Case;

      --  Any other types have the same representation for these purposes

      else
         return True;
      end if;

   end Same_Representation;

   -----------------------------------
   -- Validate_Unchecked_Conversion --
   -----------------------------------

   procedure Validate_Unchecked_Conversion
     (N        : Node_Id;
      Act_Unit : Entity_Id)
   is
      Source : Entity_Id;
      Target : Entity_Id;

      procedure No_Unconstrained_Type (T : Node_Id);
      --  Issue error if type T is an unconstrained type

      procedure No_Unconstrained_Type (T : Node_Id) is
      begin
         if Is_Indefinite_Subtype (T) then
            Error_Msg_NE
              ("unconstrained } not allowed in unchecked conversion",
               N, First_Subtype (T));
         end if;
      end No_Unconstrained_Type;

   --  Start of processing for Validate_Unchecked_Conversion

   begin
      --  If we are dealing with private types, then do the check on their
      --  fully declared counterparts if the full declarations have been
      --  encountered (they don't have to be visible, but they must exist!)

      Source := Etype (First_Formal (Act_Unit));

      if Is_Private_Type (Source)
        and then Present (Underlying_Type (Source))
      then
         Source := Underlying_Type (Source);
      end if;

      Target := Etype (Act_Unit);

      if Is_Private_Type (Target)
        and then Present (Underlying_Type (Target))
      then
         Target := Underlying_Type (Target);
      end if;

      No_Unconstrained_Type (Source);
      No_Unconstrained_Type (Target);

      if Esize (Source) /= 0
        and then Esize (Target) /= 0
        and then Esize (Source) /= Esize (Target)
      then
         Error_Msg_N
           ("types for unchecked conversion have different sizes?", N);
      end if;
   end Validate_Unchecked_Conversion;

end Sem_Ch13;
