;; frames.jl -- handle window framing
;; $Id: frames.jl,v 1.56 2000/06/12 15:11:09 john Exp $

;; Copyright (C) 1999 John Harper <john@dcs.warwick.ac.uk>

;; This file is part of sawmill.

;; sawmill is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; sawmill is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with sawmill; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

(require 'timers)
(provide 'frames)

;; Commentary:

;; This sets the following window properties:

;;	frame-style		If set, the _user-chosen_ frame style
;;	current-frame-style	The current frame style
;;	type			The notional window type, defining
;;				 which parts of the frame are included

;; The different window types are:

;;	default			title bar and border
;;	transient		border only
;;	shaped			title-bar only
;;	shaped-transient	border-like title-bar only
;;	unframed		no frame at all


;; custom support

(defun custom-set-frame-style (symbol value &rest args)
  (if (eq symbol 'default-frame-style)
      (set-frame-style value)
    (apply custom-set-variable symbol value args)))

(defun custom-make-frame-style-widget (symbol value doc)
  (let
      ((styles (find-all-frame-styles t)))
    `(frame-style ,styles
		  :variable ,symbol
		  :value ,value
		  :doc ,doc
		  :theme-path ,theme-load-path)))

(defun after-setting-frame-option ()
  (when always-update-frames
    (reframe-all-windows)))

(put 'frame-style 'custom-set custom-set-frame-style)
(put 'frame-style 'custom-widget custom-make-frame-style-widget)
(setq custom-set-alist (cons (cons custom-set-frame-style
				   'custom-set-frame-style) custom-set-alist))


;; variables etc

(defvar frame-part-classes
  '((title . ((cursor . hand2)
	      (keymap . title-keymap)))
    (menu-button . ((keymap . menu-button-keymap)))
    (close-button . ((keymap . close-button-keymap)
		     (cursor . dotbox)))
    (iconify-button . ((keymap . iconify-button-keymap)
		       (cursor . sb_down_arrow)))
    (maximize-button . ((keymap . maximize-button-keymap)
			(cursor . sb_v_double_arrow)))
    (top-border . ((cursor . top_side)
		   (keymap . border-keymap)))
    (left-border . ((cursor . left_side)
		    (keymap . border-keymap)))
    (right-border . ((cursor . right_side)
		     (keymap . border-keymap)))
    (bottom-border . ((cursor . bottom_side)
		      (keymap . border-keymap)))
    (top-left-corner . ((cursor . top_left_corner)
			(keymap . border-keymap)))
    (top-right-corner . ((cursor . top_right_corner)
			 (keymap . border-keymap)))
    (bottom-left-corner . ((cursor . bottom_left_corner)
			   (keymap . border-keymap)))
    (bottom-right-corner . ((cursor . bottom_right_corner)
			    (keymap . border-keymap))))
  "Alist of (CLASS . ALIST) associating classes of frame parts with state
they inherit.")

(defvar override-frame-part-classes nil
  "Alist of (CLASS . ALIST) associating classes of frame parts with state
that overrides settings set elsewhere.")

(defvar frame-type-fallback-alist '((transient . default)
				    (shaped . default)
				    (shaped-transient . shaped)
				    (icon . shaped-transient)
				    (dock . icon))
  "Alist associated frame types with type to try if the style doesn't offer a
frame of the requested type. If no entry, then the `unframed' style is used.")

(defvar nil-frame nil
  "Frame definition used for unframed windows.")

(defcustom default-frame-style nil
  "Default frame style (theme)."
  :type frame-style
  :group appearance)

(defcustom always-update-frames t
  "Update all windows when the default frame style is changed."
  :type boolean
  :group misc)

(defcustom decorate-transients nil
  "Decorate transient windows similarly to top-level windows."
  :type boolean
  :group appearance
  :after-set after-setting-frame-option)

(defcustom reload-themes-when-changed t
  "Automatically reload themes when they are updated."
  :type boolean
  :group misc)

(defvar theme-update-interval 60
  "Number of seconds between checking if theme files have been modified.")

(defvar user-theme-directory "~/.sawfish/themes"
  "Directory containing user-local themes.")

(defvar system-theme-directory (expand-file-name
				"../themes" sawfish-lisp-lib-directory)
  "Directory containing themes from the current sawfish version.")

(defvar site-theme-directory (expand-file-name
			      "../../themes" sawfish-lisp-lib-directory)
  "Directory containing system-wide themes.")

(defvar theme-load-path (list user-theme-directory
			      site-theme-directory
			      system-theme-directory)
  "List of directories from which themes may be loaded.")

(defvar frame-styles nil
  "List of (NAME . FUNCTION) defining all loaded frame styles.")

;; List of (NAME FILENAME MODTIME) mapping loaded frame styles to the
;; files they were loaded from; used to check if the theme needs reloading
(defvar frame-style-files nil)

;; List of styles that can be edited using sawfish-themer
(defvar editable-frame-styles nil)

;; used when decorate-transients is non-nil, map transient window
;; types to type to pass to frame style function
(defvar transient-normal-frame-alist '((transient . default)
				       (shaped-transient . shaped)))

;; list of (REGEXP DIR-EXPAND NAME-EXPAND)
(defvar theme-suffix-regexps
  '(("^(.*)/(.*)\\.tar(\\.gz|\\.Z|\\.bz2|)$" "\\0#tar/\\2" "\\2")))

(defvar theme-suffixes '("" ".tar" ".tar.gz" ".tar.Z" ".tar.bz2"))

(defvar themes-are-gaolled t
  "When non-nil themes are assumed to be malicious.")

(defvar sawfish-themer-program "sawfish-themer")


;; managing frame styles

(defun add-frame-style (name function)
  (let
      ((cell (assq name frame-styles)))
    (if cell
	(rplacd cell function)
      (setq frame-styles (cons (cons name function) frame-styles)))
    (when load-filename
      (let
	  ;; if we're loading from a tar-file, then check the
	  ;; tar file itself, not its contents (for efficiency)
	  ((file (if (string-match "#tar/" load-filename)
		     (substring load-filename 0 (match-start))
		   load-filename)))
	(setq cell (assq name frame-style-files))
	(if cell
	    (rplacd cell (list file (file-modtime file)))
	  (setq frame-style-files (cons (list name file (file-modtime file))
					frame-style-files)))))
    (unless default-frame-style
      (setq default-frame-style name))))

(defun check-frame-availability (name)
  (unless (assq name frame-styles)
    (load-frame-style name)
    (or (assq name frame-styles) (error "No such frame style: %s" name))))

(defun reload-frame-style (name)
  (when (assq name frame-styles)
    (load-frame-style name)
    (reframe-windows-with-style name)))

(defun set-frame-style (name)
  (check-frame-availability name)
  (setq default-frame-style name)
  (when always-update-frames
    (reframe-all-windows)))

(defun find-frame-definition (w style type)
  (letrec ((iter
	    (lambda (type seen)
	      (cond
	       ((eq type 'unframed) nil-frame)
	       ((style w type))
	       (t (let ((next (or (cdr (assq type frame-type-fallback-alist))
				  'unframed)))
		    (if (memq next seen)
			;; been here before..
			nil-frame
		      (iter next (cons next seen)))))))))
    (iter type (list type))))

(defun set-window-frame-style (w style &optional type from-user)
  (check-frame-availability style)
  (if type
      (progn
	(window-put w 'type type)
	(call-window-hook 'window-state-change-hook w (list '(type))))
    (setq type (window-type w)))
  (window-put w 'current-frame-style style)
  (when from-user
    (window-put w 'frame-style style)
    (call-window-hook 'window-state-change-hook w (list '(frame-style))))
  (let ((style-fun (cdr (assq style frame-styles))))
    (set-window-frame w (if style-fun
			    (find-frame-definition w style-fun type)
			  nil-frame))))

(defun set-frame-for-window (w &optional override type)
  (when (or override (not (or (window-frame w) (window-get w 'ignored))))
    (let*
	((style (window-get w 'frame-style))
	 fun tem)
      (unless style
	(setq style default-frame-style))
      (unless (assq style frame-styles)
	(load-frame-style style))
      (set-window-frame-style w style type))))

(defun reframe-one-window (w)
  (when (and (windowp w) (not (window-get w 'ignored)))
    (set-frame-for-window w t (window-get w 'type))))

(defun rebuild-frames-with-style (style)
  (mapc (lambda (w)
	  (when (eq (window-get w 'current-frame-style) style)
	    (rebuild-frame w)))
	(managed-windows)))

(defun reframe-windows-with-style (style)
  (mapc (lambda (w)
	  (when (eq (window-get w 'current-frame-style) style)
	    (reframe-one-window w)))
	(managed-windows)))

(defun reframe-all-windows ()
  (mapc reframe-one-window (managed-windows)))

;; called periodically from a timer
(defun frames-on-idle (timer)
  (set-timer timer theme-update-interval)
  (when reload-themes-when-changed
    (mapc (lambda (cell)
	    (let
		((style (nth 0 cell))
		 (file (nth 1 cell))
		 (modtime (nth 2 cell)))
	      (when (time-later-p (file-modtime file) modtime)
		(reload-frame-style style))))
	  frame-style-files)))

(defun mark-frame-style-editable (style)
  (unless (memq style editable-frame-styles)
    (setq editable-frame-styles (cons style editable-frame-styles))))

(defun frame-style-editable-p (style)
  (memq style editable-frame-styles))

(defun edit-frame-style (style)
  (interactive (list default-frame-style))
  (if (not (memq style editable-frame-styles))
      (error "Frame style isn't editable")
    (let
	((dir (find-frame-style style)))
      (when dir
	(system (format nil "%s %s &" sawfish-themer-program dir))))))


;; kludge different window decors by modifying the assumed window type

(defun window-type (w)
  (or (window-get w 'type)
      (let
	  ((type (if (window-transient-p w)
		     (if (window-shaped-p w)
			 'shaped-transient
		       'transient)
		   (if (window-shaped-p w)
		       'shaped
		     'default)))
	   tem)
	(when (and decorate-transients (window-transient-p w)
		   (setq tem (cdr (assq type transient-normal-frame-alist))))
	  (setq type tem))
	type)))

(defun window-type-remove-title (type)
  (cond ((eq type 'default)
	 'transient)
	((memq type '(shaped shaped-transient))
	 'unframed)
	(t
	 type)))

(defun window-type-remove-border (type)
  (cond ((eq type 'default)
	 'shaped)
	((memq type '(transient shaped-transient))
	 'unframed)
	(t
	 type)))

(defun window-type-add-title (type)
  (cond ((eq type 'transient)
	 'default)
	((eq type 'unframed)
	 'shaped)
	(t
	 type)))

(defun window-type-add-border (type)
  (cond ((eq type 'shaped)
	 'default)
	((eq type 'unframed)
	 'transient)
	(t
	 type)))

;; create some commands for setting the window type
(mapc (lambda (type)
	(define-value (intern (concat "set-frame:" (symbol-name type)))
		      (lambda (w)
			(interactive "%W")
			(set-frame-for-window w t type))))
      '(default transient shaped shaped-transient unframed))


;; loading ``themes'' (currently just frame styles)

(defun frame-style-directory (dir &optional get-name)
  (if (and (file-directory-p dir)
	   (or (file-exists-p (expand-file-name "theme.jl" dir))
	       (file-exists-p (expand-file-name "theme.jlc" dir))))
      (if get-name
	  (file-name-nondirectory dir)
	dir)
    ;; try the list of suffixes
    (catch 'out
      (mapc (lambda (cell)
	      (when (string-match (car cell) dir)
		(throw 'out (expand-last-match (if get-name
						   (nth 2 cell)
						 (nth 1 cell))))))
	    theme-suffix-regexps)
      nil)))

(defun find-frame-style (name)
  (catch 'out
    (mapc (lambda (dir)
	    (mapc (lambda (suf)
		    (let*
			((t-dir (expand-file-name
				 (concat (symbol-name name) suf) dir))
			 tem)
		      (when (file-exists-p t-dir)
			(setq tem (frame-style-directory t-dir))
			(when tem
			  (throw 'out tem)))))
		  theme-suffixes))
	  theme-load-path)
    nil))

(defun load-frame-style (name)
  (let
      ((dir (find-frame-style name)))
    (when dir
      (let
	  ((image-load-path (cons dir image-load-path)))
	(if themes-are-gaolled
	    (gaol-load (expand-file-name "theme.jl" dir) nil t t)
	  (load (expand-file-name "theme" dir) nil t))))))

(defun find-all-frame-styles (&optional sorted)
  (let
      (lst tem)
    (mapc (lambda (dir)
	    (when (file-directory-p dir)
	      (mapc (lambda (t-dir)
		      (when (setq tem (frame-style-directory
				       (expand-file-name t-dir dir) t))
			(unless (member tem lst)
			  (setq lst (cons tem lst)))))
		    (directory-files dir))))
	  theme-load-path)
    (when sorted
      (setq lst (sort lst string-lessp)))
    (mapcar intern lst)))

(defun frame-style-menu ()
  (let
      ((styles (find-all-frame-styles t)))
    (nconc (mapcar (lambda (s)
		     (list (symbol-name s)
			   `(set-window-frame-style
			     (current-event-window) ',s nil t)))
		   styles)
	   `(() (,(_ "Default")
		 (let
		     ((w (current-event-window)))
		   (window-put w 'frame-style nil)
		   (set-frame-for-window w t)
		   (call-window-hook
		    'window-state-change-hook w (list '(frame-style)))))))))


;; removing frame parts

(defun remove-frame-class (w class)
  (window-put w 'removed-classes
	      (cons class (delq class (window-get w 'removed-classes))))
  (when (window-framed-p w)
    (rebuild-frame w)))

(defun add-frame-class (w class)
  (window-put w 'removed-classes (delq class (window-get w 'removed-classes)))
  (when (window-framed-p w)
    (rebuild-frame w)))


;; manipulating the frame part classes variables

(defun set-frame-part-value (class key value &optional override)
  (let*
      ((var (if override 'override-frame-part-classes 'frame-part-classes))
       (item (assq class (symbol-value var)))
       tem)
    (if item
	(if (setq tem (assq key (cdr item)))
	    (rplacd tem value)
	  (rplacd item (cons (cons key value) (cdr item))))
      (set var (cons (cons class (list (cons key value)))
		     (symbol-value var))))))

;; (def-frame-class shade-button '((cursor . foo) ...)
;;   (bind-keys shade-button-keymap
;;     "Button1-Off" 'toggle-window-shaded))
;; 
;; the idea being that it will only create the frame part if it doesn't
;; exist, it will add all properties from the second argument unless
;; they're already set, then create and initialise the keymap from the
;; third argument (unless a keymap is already defined)

(defmacro def-frame-class (class alist &rest keymap-forms)
  (if keymap-forms
      `(when (define-frame-class ',class ,alist t)
	 ,@keymap-forms)
    `(define-frame-class ',class ,alist)))

(defun define-frame-class (class alist &optional with-keymap)
  (let
      ((cell (assq class frame-part-classes))
       (ok-to-bind nil))
    (if (not cell)
	(progn
	  (setq cell (cons class alist))
	  (setq frame-part-classes (cons cell frame-part-classes)))
      (mapc (lambda (attr)
	      (let
		  ((tem (assq (car attr) (cdr cell))))
		(unless tem
		  (rplacd cell (cons attr (cdr cell))))))
	    alist))
    (when with-keymap
      (let
	  ((map-name (intern (concat (symbol-name class) "-keymap"))))
	(unless (boundp map-name)
	  (make-variable-special map-name)
	  (set map-name (make-keymap))
	  (setq ok-to-bind t)
	  ;; so the theme can bind to the keymap..
	  (gaol-add-special map-name))
	(set-frame-part-value class 'keymap map-name)))
    ok-to-bind))


;; initialisation

(add-hook 'add-window-hook set-frame-for-window t)
(add-hook 'shape-notify-hook reframe-one-window t)

(make-timer frames-on-idle theme-update-interval)

(sm-add-saved-properties 'type 'ignored 'frame-style)
(add-swapped-properties 'frame-active-color 'frame-inactive-color)
