/*
    parted - a frontend to libparted
    Copyright (C) 1999-2000  Andrew Clausen, Lennert Buytenhek and Red Hat, Inc.

    Andrew Clausen		<clausen@gnu.org>
    Lennert Buytenhek		<buytenh@gnu.org>
    Matt Wilson, Red Hat Inc.	<msw@redhat.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "../config.h"
#include "command.h"
#include "ui.h"

#include <libintl.h>
#include <locale.h>
#define N_(String) String
#if ENABLE_NLS
#  define _(String) gettext (String)
#else
#  define _(String) (String)
#endif /* ENABLE_NLS */

#include <parted/parted.h>

#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#ifdef HAVE_GETOPT_H
#include <getopt.h>

static struct option	options[] = {
	/* name, has-arg, string-return-val, char-return-val */
	{"help",	0, NULL, 'h'},
	{"interactive",	0, NULL, 'i'},
	{"script",	0, NULL, 's'},
	{"version",	0, NULL, 'v'},
	{NULL,		0, NULL, 0}
};
#endif

static char*	options_help [][2] = {
	{"help",	N_("displays this help message")},
	{"interactive",	N_("where necessary, prompts for user intervention")},
	{"script",	N_("never prompts for user intervention")},
	{"version",	N_("displays the version")},
	{NULL,		NULL}
};

int	opt_script_mode;

static char* minor_msg = N_(
"MINOR is the partition number used by Linux.  The primary partitions number "
"from 1-4, and logical partitions are 5 onwards.\n");

static char* label_type_msg_start = N_("LABEL-TYPE is one of: ");
static char* part_type_msg =	N_("PART-TYPE is one of: primary, logical, "
			           "extended\n");
static char* fs_type_msg_start = N_("FS-TYPE is one of: ");
static char* start_end_msg =	N_("START and END are in megabytes\n");
static char* device_msg =	N_("DEVICE is usually /dev/hda or /dev/sda\n");

static char* label_type_msg;
static char* fs_type_msg;

static Command*	commands [256] = {NULL};

static int do_boot (PedDevice* dev);
static int do_check (PedDevice* dev);
static int do_cp (PedDevice* dev);
int do_help (PedDevice* dev);
static int do_hide (PedDevice* dev);
static int do_mklabel (PedDevice* dev);
static int do_mkfs (PedDevice* dev);
static int do_mkpart (PedDevice* dev);
static int do_mkpartfs (PedDevice* dev);
static int do_print (PedDevice* dev);
static int do_quit (PedDevice* dev);
static int do_resize (PedDevice* dev);
static int do_rm (PedDevice* dev);
static int do_unhide (PedDevice* dev);

void
help_on (char* topic)
{
	Command*	cmd;

	cmd = command_get (commands, topic);
	if (!cmd) return;

	command_print_help (cmd);
}

static int
do_boot (PedDevice* dev)
{
	PedDisk*	disk;
	PedPartition*	part;

	disk = ped_disk_open (dev);
	if (!disk)
		goto error;

	if (!is_integer ()) {
		help_on ("boot");
		goto error_close_disk;
	}
	part = ped_disk_get_partition (disk, get_integer ());
	if (!part) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Partition doesn't exist."));
		goto error_close_disk;
	}

	if (!ped_disk_set_boot_partition (disk, part))
		goto error_close_disk;
	if (!ped_disk_write (disk))
		goto error_close_disk;
	ped_disk_close (disk);
	return 1;

error_close_disk:
	ped_disk_close (disk);
error:
	return 0;
}

static int
do_check (PedDevice* dev)
{
	PedDisk*	disk;
	PedFileSystem*	fs;
	PedPartition*	part;

	disk = ped_disk_open (dev);
	if (!disk)
		goto error;

	if (!is_integer ()) {
		help_on (_("check"));
		goto error_close_disk;
	}
	part = ped_disk_get_partition (disk, get_integer ());
	if (!part) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Partition doesn't exist."));
		goto error_close_disk;
	}
	fs = ped_file_system_open (&part->geom);
	if (!fs)
		goto error_close_disk;
	ped_file_system_check (fs);
	ped_file_system_close (fs);
	ped_disk_close (disk);
	return 1;

error_close_disk:
	ped_disk_close (disk);
error:
	return 0;
}

static int
do_cp (PedDevice* dev)
{
	PedDisk*		src_disk = NULL;
	PedDisk*		dst_disk = NULL;
	PedDevice*		dst_device;
	PedPartition*		src = NULL;
	PedPartition*		dst = NULL;
	PedFileSystem*		src_fs = NULL;

	src_disk = ped_disk_open (dev);
	if (!src_disk)
		goto error;

/* figure out source partition */
	if (!is_integer ()) {
		help_on (_("cp"));
		goto error_close_disk;
	}
	src = ped_disk_get_partition (src_disk, get_integer ());
	if (!src) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Source partition doesn't exist."));
		goto error_close_disk;
	}
	if (src->type == PED_PARTITION_EXTENDED) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Can't copy extended partitions."));
		goto error_close_disk;
	}

/* figure out target partition */
	if (!is_integer ()) {
		dst_device = ped_device_get (get_word ());
		if (!dst_device)
			goto error_close_disk;
		dst_disk = ped_disk_open (dst_device);
	} else {
		dst_disk = src_disk;
	}
	if (!is_integer ()) {
		help_on (_("cp"));
		goto error_close_disk;
	}
	dst = ped_disk_get_partition (dst_disk, get_integer ());
	if (!dst) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Destination partition doesn't exist."));
		goto error_close_disk;
	}
	if (ped_partition_is_busy (dst)) {
		if (ped_exception_throw (
			PED_EXCEPTION_WARNING,
			PED_EXCEPTION_IGNORE_CANCEL,
			_("Destination partition is being used."))
				!= PED_EXCEPTION_IGNORE)
			goto error_close_disk;
	}
	dst->type = src->type;

/* do the copy */
	src_fs = ped_file_system_open (&src->geom);
	if (!src_fs)
		goto error_close_disk;
	if (!ped_file_system_copy (src_fs, &dst->geom))
		goto error_close_fs;
	ped_file_system_close (src_fs);
	if (dst_disk && src_disk != dst_disk)
		ped_disk_close (dst_disk);
	ped_disk_close (src_disk);
	return 1;

error_close_fs:
	ped_file_system_close (src_fs);
error_close_disk:
	if (dst_disk && src_disk != dst_disk)
		ped_disk_close (dst_disk);
	ped_disk_close (src_disk);
error:
	return 0;
}

void
print_commands_help ()
{
	int		i;

	for (i=0; commands [i]; i++)
		command_print_summary (commands [i]);
}

void
print_options_help ()
{
	int		i;

	for (i=0; options_help [i][0]; i++) {
		printf ("  -%c, --%-23.23s %s\n",
			options_help [i][0][0],
			options_help [i][0],
			_(options_help [i][1]));
	}
}

int
do_help (PedDevice* dev)
{
	if (peek_word ())
		help_on (get_word ());
	else
		print_commands_help();
	return 1;
}

static int
do_hide (PedDevice* dev)
{
	PedDisk*	disk;
	PedPartition*	part;

	disk = ped_disk_open (dev);
	if (!disk)
		goto error;

	if (!is_integer ()) {
		help_on (_("hide"));
		goto error_close_disk;
	}
	part = ped_disk_get_partition (disk, get_integer ());
	if (!part) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Partition doesn't exist."));
		goto error_close_disk;
	}

	if (!ped_partition_set_hide_state (part, 1))
		goto error_close_disk;
	if (!ped_disk_write (disk))
		goto error_close_disk;
	ped_disk_close (disk);
	return 1;

error_close_disk:
	ped_disk_close (disk);
error:
	return 0;
}

static int
do_mklabel (PedDevice* dev)
{
	PedDisk*	disk;
	PedDiskType*	type;

	ped_exception_fetch_all ();
	disk = ped_disk_open (dev);
	if (!disk) ped_exception_catch ();
	ped_exception_leave_all ();

	if (disk) {
		if (ped_disk_is_busy (disk)) {
			if (ped_exception_throw (
				PED_EXCEPTION_WARNING,
				PED_EXCEPTION_IGNORE_CANCEL,
				_("Partition(s) on %s are being used."),
				disk->dev->path)
					!= PED_EXCEPTION_IGNORE) {
				ped_disk_close (disk);
				return 0;
			}
		}
		ped_disk_close (disk);
	}

	if (!peek_word ())
		goto error_syntax;

	type = ped_disk_type_get (get_word ());
	if (!type)
		goto error_syntax;

	ped_disk_create (dev, type);
	return 1;

error_syntax:
	help_on (_("mklabel"));
	return 0;
}

static int
do_mkfs (PedDevice* dev)
{
	PedDisk*		disk;
	PedPartition*		part;
	PedFileSystemType*	type;
	PedFileSystem*		fs;

	disk = ped_disk_open (dev);
	if (!disk)
		goto error;

	if (!is_integer ())
		goto error_syntax;
	part = ped_disk_get_partition (disk, get_integer ());
	if (!part) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Partition doesn't exist."));
		goto error_close_disk;
	}
	if (ped_partition_is_busy (part)) {
		if (ped_exception_throw (
			PED_EXCEPTION_WARNING,
			PED_EXCEPTION_IGNORE_CANCEL,
			_("Partition is being used."))
				!= PED_EXCEPTION_IGNORE)
			goto error_close_disk;
	}

	if (!peek_word ())
		goto error_syntax;
	type = ped_file_system_type_get (get_word ());
	if (!type) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Unknown filesystem type."));
		goto error_syntax;
	}

	fs = ped_file_system_create (&part->geom, type);
	if (!fs)
		goto error_close_disk;
	ped_file_system_close (fs);

	if (!ped_partition_set_system (part, type))
		goto error_close_disk;
	if (!ped_disk_write (disk))
		goto error_close_disk;
	ped_disk_close (disk);
	return 1;

error_syntax:
	help_on (_("mkfs"));
error_close_disk:
	ped_disk_close (disk);
error:
	return 0;
}

static int
do_mkpart (PedDevice* dev)
{
	PedDisk*			disk;
	PedPartition*			part;
	PedPartitionType		part_type;
	PedFileSystemType*		fs_type;
	PedSector			start, end;

	disk = ped_disk_open (dev);
	if (!disk)
		goto error;

	if (count_words () < 2) {
		help_on (_("mkpart"));
		goto error_close_disk;
	}
	switch (get_word() [0]) {
		case 'p': part_type = PED_PARTITION_PRIMARY; break;
		case 'e': part_type = PED_PARTITION_EXTENDED; break;
		case 'l': part_type = PED_PARTITION_LOGICAL; break;
		default: help_on (_("mkpart")); goto error_close_disk;
	}

	if (part_type == PED_PARTITION_EXTENDED) {
		fs_type = NULL;
	} else {
		if (!peek_word ()) {
			help_on (_("mkpart"));
			goto error_close_disk;
		}

		fs_type = ped_file_system_type_get (get_word ());
		if (!fs_type) {
			ped_exception_throw (PED_EXCEPTION_ERROR,
				PED_EXCEPTION_CANCEL,
				_("Unknown file system type."));
			goto error_close_disk;
		}
	}

	if (!is_sector ()) {
		help_on (_("mkpart"));
		goto error_close_disk;
	}
	start = get_sector ();
	if (!is_sector ()) {
		help_on (_("mkpart"));
		goto error_close_disk;
	}
	end = get_sector ();
	part = ped_partition_new (disk, part_type, fs_type, start, end);
	if (!part)
		goto error_close_disk;
	if (!ped_disk_add_partition (disk, part))
		goto error_destroy_part;

	/* FIXME:  ped_partition_new() doesn't probe the file system for
	 * what type of partition should be created.  So we do it here.
	 * 	Question: SHOULD we probe the file system on
	 * ped_partition_new()?  It only makes sense if someone accidently
	 * delete their partition...
	 */
       	ped_partition_set_system (part, fs_type);

	ped_disk_write (disk);
	ped_disk_close (disk);
	return 1;

error_destroy_part:
	ped_partition_destroy (part);
error_close_disk:
	ped_disk_close (disk);
error:
	return 0;
}

/* FIXME - this is a mess */
static int
do_mkpartfs (PedDevice* dev)
{
	PedDisk*			disk;
	PedPartition*			part;
	PedPartitionType		part_type;
	PedFileSystemType*		fs_type;
	PedFileSystem*			fs;
	PedSector			start, end;

	disk = ped_disk_open (dev);
	if (!disk)
		goto error;

	if (count_words () < 2) {
		help_on (_("mkpartfs"));
		goto error_close_disk;
	}
	switch (get_word() [0]) {
		case 'p': part_type = PED_PARTITION_PRIMARY; break;
		case 'l': part_type = PED_PARTITION_LOGICAL; break;
		case 'e':
		default: help_on (_("mkpartfs")); goto error_close_disk;
	}

	if (part_type == PED_PARTITION_EXTENDED) {
		fs_type = NULL;
	} else {
		if (!peek_word ()) {
			help_on (_("mkpartfs"));
			goto error_close_disk;
		}

		fs_type = ped_file_system_type_get (get_word ());
		if (!fs_type) {
			ped_exception_throw (PED_EXCEPTION_ERROR,
				PED_EXCEPTION_CANCEL,
				_("Unknown file system type."));
			goto error_close_disk;
		}
	}

	if (!is_sector ()) {
		help_on (_("mkpartfs"));
		goto error_close_disk;
	}
	start = get_sector ();
	if (!is_sector ()) {
		help_on (_("mkpartfs"));
		goto error_close_disk;
	}
	end = get_sector ();
	part = ped_partition_new (disk, part_type, fs_type, start, end);
	if (!part)
		goto error_close_disk;
	if (!ped_disk_add_partition (disk, part))
		goto error_destroy_part;

	fs = ped_file_system_create (&part->geom, fs_type);
	if (!fs) 
		goto error_remove_part;
	ped_file_system_close (fs);

/* set the system AGAIN, because if it's FAT16 or FAT32 (or whatever) will
 * be known now.
 */
	ped_partition_set_system (part, fs_type);

	ped_disk_write (disk);
	ped_disk_close (disk);
	return 1;

error_remove_part:
	ped_disk_delete_partition (disk, part);
	goto error_close_disk;
error_destroy_part:
	ped_partition_destroy (part);
error_close_disk:
	ped_disk_close (disk);
error:
	return 0;
}

static int
do_print (PedDevice* dev)
{
	PedDisk*	disk;
	PedPartition*	part;
	int		first_flag;

	disk = ped_disk_open (dev);
	if (!disk)
		goto error;

	printf (_("Disk geometry: 0-%.1f megabytes\n"),
		disk->dev->length * 1.0 / MEGABYTE_SECTORS);

	printf (
_("Minor   Start     End    Type            Filesystem     Flags\n"));

	for (part = ped_disk_next_partition (disk, NULL); part;
	     part = ped_disk_next_partition (disk, part)) {
		if (part->type == PED_PARTITION_PRIMARY
		    || part->type == PED_PARTITION_LOGICAL
		    || part->type == PED_PARTITION_EXTENDED) {
			printf ("%-5d %8.1f %8.1f  %-15s %-15s",
				(int) part->num,
				(int) part->geom.start * 1.0 / MEGABYTE_SECTORS,
				(int) part->geom.end * 1.0 / MEGABYTE_SECTORS,
				ped_partition_type_get_name (part->type),
				part->fs_type ? part->fs_type->name : "");

			first_flag = 1;
			if (part->bootable) {
				printf (_("boot"));
				first_flag = 0;
			}
			if (part->hidden) {
				if (!first_flag)
					printf (", ");
				printf (_("hidden"));
				first_flag = 0;
			}
			printf ("\n");
		}
	}

	ped_disk_close (disk);
	return 1;

error:
	return 0;
}

static int
do_quit (PedDevice* dev)
{
	ped_device_close (dev);
	ped_done ();
	done_ui ();
	exit (0);
}

static int
do_resize (PedDevice* dev)
{
	PedDisk*		disk;
	PedPartition*		part;
	PedFileSystem*		fs;
	PedFileSystemType*	fs_type;
	PedSector		start, end;

	disk = ped_disk_open (dev);
	if (!disk)
		goto error;

	if (!is_integer ()) {
		help_on (_("resize"));
		goto error_close_disk;
	}
	part = ped_disk_get_partition (disk, get_integer ());
	if (!part) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Partition doesn't exist."));
		goto error_close_disk;
	}
	if (ped_partition_is_busy (part)) {
		if (ped_exception_throw (
			PED_EXCEPTION_WARNING,
			PED_EXCEPTION_IGNORE_CANCEL,
			_("Partition is being used."))
				!= PED_EXCEPTION_IGNORE)
			goto error_close_disk;
	}

	if (!is_sector ()) {
		help_on (_("resize"));
		goto error_close_disk;
	}
	start = get_sector ();
	if (!is_sector ()) {
		help_on (_("resize"));
		goto error_close_disk;
	}

	/* NOTE: this gets aligned by ped_disk_set_partition_geom() */
	end = get_sector ();

	if (part->type == PED_PARTITION_EXTENDED) {
		if (!ped_disk_set_partition_geom (disk, part, start, end))
			goto error_close_disk;
		ped_partition_set_system (part, NULL);
	} else {
		fs = ped_file_system_open (&part->geom);
		if (!fs)
			goto error_close_disk;
		fs_type = fs->type;
		if (!ped_disk_set_partition_geom (disk, part, start, end))
			goto error_close_disk;
		if (!ped_file_system_resize (fs, &part->geom))
			goto error_close_fs;
		ped_file_system_close (fs);
		ped_partition_set_system (part, fs_type);
	}

	ped_disk_write (disk);
	ped_disk_close (disk);
	return 1;

error_close_fs:
	ped_file_system_close (fs);
error_close_disk:
	ped_disk_close (disk);
error:
	return 0;
}

static int
do_rm (PedDevice* dev)
{
	PedDisk*		disk;
	PedPartition*		part;

	disk = ped_disk_open (dev);
	if (!disk)
		goto error;

	if (!peek_word ())
		goto error_syntax;

	part = ped_disk_get_partition (disk, get_integer ());
	if (!part) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Partition doesn't exist."));
		goto error_close_disk;
	}
	if (ped_partition_is_busy (part)) {
		if (ped_exception_throw (
			PED_EXCEPTION_WARNING,
			PED_EXCEPTION_IGNORE_CANCEL,
			_("Partition is being used."))
				!= PED_EXCEPTION_IGNORE)
			goto error_close_disk;
	}

	ped_disk_delete_partition (disk, part);
	ped_disk_write (disk);
	ped_disk_close (disk);
	return 1;

error_syntax:
	help_on (_("rm"));
error_close_disk:
	ped_disk_close (disk);
error:
	return 0;
}

static int
do_unhide (PedDevice* dev)
{
	PedDisk*	disk;
	PedPartition*	part;

	disk = ped_disk_open (dev);
	if (!disk)
		goto error;

	if (!is_integer ()) {
		help_on (_("unhide"));
		goto error_close_disk;
	}
	part = ped_disk_get_partition (disk, get_integer ());
	if (!part) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Partition doesn't exist."));
		goto error_close_disk;
	}

	if (!ped_partition_set_hide_state (part, 0))
		goto error_close_disk;
	if (!ped_disk_write (disk))
		goto error_close_disk;
	ped_disk_close (disk);
	return 1;

error_close_disk:
	ped_disk_close (disk);
error:
	return 0;
}

static void
_init_messages ()
{
	StrList*		list;
	int			first;
	PedFileSystemType*	fs_type;
	PedDiskType*		disk_type;

/* disk type */
	list = str_list_create (_(label_type_msg_start), NULL);

	first = 1;
	for (disk_type = ped_disk_type_get_next (NULL);
	     disk_type; disk_type = ped_disk_type_get_next (disk_type)) {
		if (first)
			first = 0;
		else
			str_list_append (list, ", ");
		str_list_append (list, disk_type->name);
	}
	str_list_append (list, "\n");

	label_type_msg = str_list_convert (list);
	str_list_destroy (list);

/* file system type */
	list = str_list_create (_(fs_type_msg_start), NULL);

	first = 1;
	for (fs_type = ped_file_system_type_get_next (NULL);
	     fs_type; fs_type = ped_file_system_type_get_next (fs_type)) {
		if (first)
			first = 0;
		else
			str_list_append (list, ", ");
		str_list_append (list, fs_type->name);
	}
	str_list_append (list, "\n");

	fs_type_msg = str_list_convert (list);
	str_list_destroy (list);
}

static void
_init_commands ()
{
	command_register (commands, command_create (
		str_list_create_unique ("boot", _("boot"), NULL),
		do_boot,
		str_list_create (
_("boot MINOR                    set the boot partition to be MINOR"),
NULL),
		str_list_create (_(minor_msg), NULL)));

	command_register (commands, command_create (
		str_list_create_unique ("check", _("check"), NULL),
		do_check,
		str_list_create (
_("check MINOR                   do a simple check on the filesystem"),
NULL),
		str_list_create (_(minor_msg), NULL)));

	command_register (commands, command_create (
		str_list_create_unique ("cp", _("cp"), NULL),
		do_cp,
		str_list_create (
_("cp MINOR [DEVICE] MINOR       copy filesystem to another partition"),
NULL),
		str_list_create (_(minor_msg), _(device_msg), NULL)));

	command_register (commands, command_create (
		str_list_create_unique ("help", _("help"), NULL),
		do_help,
		str_list_create (
_("help [COMMAND]                prints general help, or help on COMMAND"),
NULL),
		NULL));

	command_register (commands, command_create (
		str_list_create_unique ("hide", _("hide"), NULL),
		do_hide,
		str_list_create (
_("hide MINOR                    hides partition MINOR"),
NULL),
		str_list_create (_(minor_msg), NULL)));

	command_register (commands, command_create (
		str_list_create_unique ("mklabel", _("mklabel"), NULL),
		do_mklabel,
		str_list_create (
_("mklabel LABEL-TYPE            create a new disklabel (partition table)"),
NULL),
		str_list_create (label_type_msg, NULL)));

	command_register (commands, command_create (
		str_list_create_unique ("mkfs", _("mkfs"), NULL),
		do_mkfs,
		str_list_create (
_("mkfs MINOR FS-TYPE            make a filesystem FS-TYPE on partititon "
  "MINOR"),
NULL),
		str_list_create (_(minor_msg), _(fs_type_msg), NULL)));

	command_register (commands, command_create (
		str_list_create_unique ("mkpart", _("mkpart"), NULL),
		do_mkpart,
		str_list_create (
_("mkpart PART-TYPE [FS-TYPE] START END      make a partition"),
NULL),
		str_list_create (_(part_type_msg),
				 _(fs_type_msg),
				 _(start_end_msg),
				 "\n",
_(
"mkpart makes a partition without creating a new file system on the "
"partition.  FS-TYPE must be specified for data partitions (as opposed to "
"extended partitions).  This command is useful if you accidently deleted a "
"partition.\n"),
NULL)));

	command_register (commands, command_create (
		str_list_create_unique ("mkpartfs", _("mkpartfs"), NULL),
		do_mkpartfs,
		str_list_create (
_("mkpartfs PART-TYPE FS-TYPE START END      make a partition with a "
  "filesystem"),
NULL),
		str_list_create (_(part_type_msg), _(start_end_msg), NULL)));

	command_register (commands, command_create (
		str_list_create_unique ("print", _("print"), NULL),
		do_print,
		str_list_create (
_("print                         display the partition table"),
NULL),
		NULL));

	command_register (commands, command_create (
		str_list_create_unique ("quit", _("quit"), NULL),
		do_quit,
		str_list_create (
_("quit                          exit program"),
NULL),
		NULL));

	command_register (commands, command_create (
		str_list_create_unique ("resize", _("resize"), NULL),
		do_resize,
		str_list_create (
_("resize MINOR START END        resize filesystem on partition MINOR"),
NULL),
		str_list_create (_(minor_msg), _(start_end_msg), NULL)));

	command_register (commands, command_create (
		str_list_create_unique ("rm", _("rm"), NULL),
		do_rm,
		str_list_create (
_("rm MINOR                      delete partition MINOR"),
NULL),
		str_list_create (_(minor_msg), NULL)));

	command_register (commands, command_create (
		str_list_create_unique ("unhide", _("unhide"), NULL),
		do_unhide,
		str_list_create (
_("unhide MINOR                  unhides partition MINOR"),
NULL),
		str_list_create (_(minor_msg), NULL)));

}

static void
_init_i18n ()
{
/* intialize i18n */
#ifdef ENABLE_NLS
	language = getenv ("LANG");	/* FIXME */
	setlocale(LC_ALL, "");
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);
#endif /* ENABLE_NLS */
}

void
_version ()
{
	printf (prog_name);
	exit (0);
}

static int
_parse_options (int* argc_ptr, char*** argv_ptr)
{
	int	opt;

	while (1)
	{
#ifdef HAVE_GETOPT_H
		opt = getopt_long (*argc_ptr, *argv_ptr, "hisv",
				   options, NULL);
#else
		opt = getopt (*argc_ptr, *argv_ptr, "hisv");
#endif
		if (opt == -1)
			break;

		switch (opt) {
			case 'h': help_msg (); break;
			case 'i': opt_script_mode = 0; break;
			case 's': opt_script_mode = 1; break;
			case 'v': _version (); break;
		}
	}

	*argc_ptr -= optind;
	*argv_ptr += optind;
	return 1;

error:
	return 0;
}

static PedDevice*
_choose_device (int* argc_ptr, char*** argv_ptr)
{
	PedDevice*	dev;

	/* specified on comand line? */
	if (*argc_ptr) {
		dev = ped_device_get ((*argv_ptr) [0]);
		if (!dev)
			return NULL;
		(*argc_ptr)--;
		(*argv_ptr)++;
	} else {
	retry:
		ped_device_probe_all ();
		dev = ped_device_get_next (NULL);
		if (!dev) {
			if (ped_exception_throw (PED_EXCEPTION_ERROR,
				PED_EXCEPTION_RETRY_CANCEL,
				_("No device found"))
					== PED_EXCEPTION_RETRY)
				goto retry;
			else
				return NULL;
		}
	}

	if (!ped_device_open (dev))
		return NULL;
	return dev;	
}

static PedDevice*
_init (int* argc_ptr, char*** argv_ptr)
{
	PedDevice*	dev;

	_init_i18n ();
	if (!init_ui ())
		goto error;
	if (!ped_init ())
		goto error;
	_init_messages ();
	_init_commands ();

	if (!_parse_options (argc_ptr, argv_ptr))
		goto error;
	dev = _choose_device (argc_ptr, argv_ptr);
	if (!dev)
		goto error;

	return dev;

error_done_ui:
	done_ui ();
error:
	return NULL;
}

static void
_done (PedDevice *dev)
{
	ped_device_close (dev);
	ped_done ();
	done_ui ();
}

int
main (int argc, char** argv)
{
	PedDevice*	dev;
	int		status;

	dev = _init (&argc, &argv);
	if (!dev)
		return 0;

	if (argc)
		status = non_interactive_mode (dev, commands, argc, argv);
	else
		status = interactive_mode (dev, commands);

	_done (dev);

	return !status;
}

