/* copy.c - copy one file to another.

   Copyright (C) 1998 Tom Tromey

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.  */

#include <config.h>

#include <assert.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>

#include "sync.h"



/* Copy some data from the input file's buffer to the output file.  */
static void
write_chunk (gpointer client_data, gint fd, GdkInputCondition cond)
{
  struct reader *reader;
  struct file *input;
  int ret;

  assert (cond == GDK_INPUT_WRITE);

  reader = (struct reader *) client_data;
  assert (fd == reader->local->fd || fd == reader->remote->fd);

  input = (reader->local->fd == fd) ? reader->remote : reader->local;

  ret = write (fd, &input->buffer[input->slot], input->size - input->slot);
  if (ret == -1)
    {
      char *u;
      struct file *output;

      /* Certain errors are ok.  */
      if (errno == EINTR || errno == EAGAIN)
	return;

      output = (reader->local->fd == fd) ? reader->local : reader->remote;
      if (input == reader->local)
	u = _("Error while writing to remote file");
      else
	u = _("Error while writing to local file");
      display_posix_error (reader->database, reader->local->name,
			   reader->remote->name, u);
      free_reader (reader);

      return;
    }

  input->slot += ret;

  if (input->slot == input->size)
    {
      /* Done writing.  */

      if (update_entry (reader->database->database, reader->local->name,
			reader->remote->name, input->checksum) == -1)
	db_posix_error (_("Error while writing to database file"),
			reader->database->filename);

      display_message (reader->database, reader->local->name,
		       reader->remote->name, "");
      free_reader (reader);
    }
}

/* Copy one file to the other.  We have a separate DIRECTION argument
   so that we can report errors and preserve the information about
   which file is "local". */
void
copy (struct reader *reader, int direction)
{
  struct file *from, *to;

  from = (direction == COPY_FROM_LOCAL) ? reader->local : reader->remote;
  to = (direction == COPY_FROM_LOCAL) ? reader->remote : reader->local;

  display_message (reader->database, reader->local->name,
		   reader->remote->name,
		   (direction == COPY_FROM_LOCAL)
		   ? _("Copying from local to remote")
		   : _("Copying from remote to local"));

  /* FIXME: make backup of TO here.
     Really should read into temp file and then rename at the end.  */

  to->fd = open (to->name, O_WRONLY | O_CREAT | O_TRUNC, from->mode);
  if (to->fd == -1)
    {
      display_posix_error (reader->database, reader->local->name,
			   reader->remote->name,
			   _("Couldn't open file for writing: "));
      free_reader (reader);
      return;
    }

  /* Set this separately because we want open() to block until ready.
     Don't check failure here -- it isn't fatal if this fails.  */
  fcntl (to->fd, F_SETFL, O_NONBLOCK);

  if ((from->flags & FLAG_FULL))
    {
      /* We have the full input file already.  So just start writing
	 immediately.  If we're writing to a local file, this will
	 just grind away without giving time to the UI -- but that
	 doesn't really matter.  */
      if (! (from->flags & FLAG_MAPPED))
	{
	  from->size = from->slot;
	  from->slot = 0;
	}
      to->state = STATE_WRITING;
      to->tag = gdk_input_add (to->fd, GDK_INPUT_WRITE, write_chunk,
			       (gpointer) reader);
    }
  else
    {
      /* FIXME: must open input file for reading, and write results to
	 output.  This requires a bit of synchronization: we don't
	 want the input to overrun the output, so we must disable the
	 input when the output is blocked and the input buffer is too
	 full.  */
      assert (0);
    }
}
