/*
 * gnome-print.c: Main GnomePrint API
 *
 * Authors:
 *   Raph Levien (raph@acm.org)
 *   Miguel de Icaza (miguel@kernel.org)
 *   Lauris Kaplinski <lauris@ariman.ee>
 */
#include <gtk/gtk.h>
#include <stdio.h>
#include <libart_lgpl/art_affine.h>
#include <libart_lgpl/art_vpath.h>
#include <libart_lgpl/art_bpath.h>
#include <libgnomeprint/gnome-print.h>
#include <libgnomeprint/gnome-print-ps.h>
#include <libgnomeprint/gnome-print-pixbuf.h>
#include <string.h>
#include <locale.h>

static void gnome_print_context_class_init (GnomePrintContextClass *klass);

static void gnome_print_context_init (GnomePrintContext *pc);

static void gnome_print_context_finalize (GtkObject *object);

static GtkObjectClass *parent_class = NULL;

GtkType
gnome_print_context_get_type (void)
{
	static GtkType pc_type = 0;
	
	if (!pc_type)
	{
		GtkTypeInfo pc_info =
		{
			"GnomePrintContext",
			sizeof (GnomePrintContext),
			sizeof (GnomePrintContextClass),
			(GtkClassInitFunc) gnome_print_context_class_init,
			(GtkObjectInitFunc) gnome_print_context_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};
		
		pc_type = gtk_type_unique (gtk_object_get_type (), &pc_info);
	}
	
	return pc_type;
}

static void
gnome_print_context_class_init (GnomePrintContextClass *class)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass*) class;

	parent_class = gtk_type_class (gtk_object_get_type ());

	object_class->finalize = gnome_print_context_finalize;

	class->newpath = NULL;
	class->moveto = NULL;
	class->lineto = NULL;
	class->curveto = NULL;
	class->closepath = NULL;
	class->setrgbcolor = NULL;
	class->fill = NULL;
	class->eofill = NULL;
	class->setlinewidth = NULL;
	class->setmiterlimit = NULL;
	class->setlinejoin = NULL;
	class->setlinecap = NULL;
	class->setdash = NULL;
	class->strokepath = NULL;
	class->stroke = NULL;
	class->setfont = NULL;
	class->show = NULL;
	class->concat = NULL;
	class->setmatrix = NULL;
	class->gsave = NULL;
	class->grestore = NULL;
	class->clip = NULL;
	class->eoclip = NULL;
	class->grayimage = NULL;
	class->rgbimage = NULL;
	class->textline = NULL;
	class->beginpage = NULL;
	class->showpage = NULL;
	class->close = NULL;
	class->setopacity = NULL;
	class->rgbaimage = NULL;
}

static void
gnome_print_context_init (GnomePrintContext *pc)
{
	pc->f = NULL;
}

GnomePrintContext *
gnome_print_context_new_with_paper_size (GnomePrinter *printer, const char *paper_size)
{
	g_return_val_if_fail (printer != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_PRINTER (printer), NULL);
	g_return_val_if_fail (paper_size != NULL, NULL);

	if (strcmp (printer->driver, "gnome-print-ps") == 0) {
		GnomePrintPs *ps = gnome_print_ps_new (printer);
		return ps ? GNOME_PRINT_CONTEXT (ps) : NULL;
	}

	if (strncmp (printer->driver, "gnome-print-uni", strlen ("gnome-print-uni")) == 0)
		return gnome_print_pixbuf_new (printer, paper_size, 75);

	return NULL;
}

/*
 * This dispatch is by cases rather than being object oriented. It may
 * be desirable to make it more OO. Basically, the way that would be
 * done would be to have a gnome_printer_make_new_context () method
 * on GnomePrinter objects.
 * 
 * However, this will do for now, and saves making extra classes.
*/
GnomePrintContext *
gnome_print_context_new (GnomePrinter *printer)
{
	g_return_val_if_fail (printer != NULL, NULL);

	return gnome_print_context_new_with_paper_size (printer, "US-Letter");
}

static void
gnome_print_context_finalize (GtkObject *object)
{
	GnomePrintContext *pc;

	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_IS_PRINT_CONTEXT (object));

	pc = GNOME_PRINT_CONTEXT (object);

	(* GTK_OBJECT_CLASS (parent_class)->finalize) (object);
}

/* methods */

#define PRINT_CLASS(pc) GNOME_PRINT_CONTEXT_CLASS(GTK_OBJECT(pc)->klass)

int
gnome_print_newpath (GnomePrintContext *pc)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);

	return PRINT_CLASS(pc)->newpath (pc);
}

int
gnome_print_moveto (GnomePrintContext *pc, double x, double y)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);

	return PRINT_CLASS(pc)->moveto (pc, x, y);
}

int
gnome_print_lineto (GnomePrintContext *pc, double x, double y)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);

	return PRINT_CLASS(pc)->lineto (pc, x, y);
}

int
gnome_print_curveto (GnomePrintContext *pc, double x1, double y1, double x2, double y2, double x3, double y3)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);

	return PRINT_CLASS(pc)->curveto (pc, x1, y1, x2, y2, x3, y3);
}

int
gnome_print_closepath (GnomePrintContext *pc)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);

	return PRINT_CLASS(pc)->closepath (pc);
}

int
gnome_print_setrgbcolor (GnomePrintContext *pc, double r, double g, double b)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);

	return PRINT_CLASS(pc)->setrgbcolor (pc, r, g, b);
}

int
gnome_print_fill (GnomePrintContext *pc)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);

	return PRINT_CLASS(pc)->fill (pc);
}

int
gnome_print_eofill (GnomePrintContext *pc)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);

	return PRINT_CLASS(pc)->eofill (pc);
}

int
gnome_print_setlinewidth (GnomePrintContext *pc, double width)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);

	return PRINT_CLASS(pc)->setlinewidth (pc, width);
}

int
gnome_print_setmiterlimit (GnomePrintContext *pc, double limit)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);

	return PRINT_CLASS(pc)->setmiterlimit (pc, limit);
}

int
gnome_print_setlinejoin (GnomePrintContext *pc, int jointype)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);

	return PRINT_CLASS(pc)->setlinejoin (pc, jointype);
}

int
gnome_print_setlinecap (GnomePrintContext *pc, int captype)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);

	return PRINT_CLASS(pc)->setlinecap (pc, captype);
}

int
gnome_print_setdash (GnomePrintContext *pc, int n_values, const double *values, double offset)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);

	if (n_values != 0)
		g_return_val_if_fail (values != NULL, -1);

	return PRINT_CLASS(pc)->setdash (pc, n_values, values, offset);
}

int
gnome_print_strokepath (GnomePrintContext *pc)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);

	return PRINT_CLASS(pc)->strokepath (pc);
}

int
gnome_print_stroke (GnomePrintContext *pc)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);

	return PRINT_CLASS(pc)->stroke (pc);
}

int
gnome_print_setfont (GnomePrintContext *pc, GnomeFont *font)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);
	g_return_val_if_fail (font != NULL, -1);

	return PRINT_CLASS(pc)->setfont (pc, font);
}

int
gnome_print_show (GnomePrintContext *pc, char const *text)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);
	g_return_val_if_fail (text != NULL, -1);

	return PRINT_CLASS(pc)->show (pc, text);
}

int
gnome_print_concat (GnomePrintContext *pc, const double matrix[6])
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);
	g_return_val_if_fail (matrix != NULL, -1);

	return PRINT_CLASS(pc)->concat (pc, matrix);
}

int
gnome_print_setmatrix (GnomePrintContext *pc, const double matrix[6])
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);
	g_return_val_if_fail (matrix != NULL, -1);

	return PRINT_CLASS(pc)->setmatrix (pc, matrix);
}

int
gnome_print_gsave (GnomePrintContext *pc)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);

	return PRINT_CLASS(pc)->gsave (pc);
}

int
gnome_print_grestore (GnomePrintContext *pc)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);

	return PRINT_CLASS(pc)->grestore (pc);
}

int
gnome_print_clip (GnomePrintContext *pc)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);

	return PRINT_CLASS(pc)->clip (pc);
}

int
gnome_print_eoclip (GnomePrintContext *pc)
{
	return PRINT_CLASS(pc)->eoclip (pc);
}

int
gnome_print_grayimage (GnomePrintContext *pc, const char *data, int width, int height, int rowstride)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);
	g_return_val_if_fail (data != NULL, -1);

	return PRINT_CLASS(pc)->grayimage (pc, data, width, height, rowstride);
}

int
gnome_print_rgbimage (GnomePrintContext *pc, const char *data, int width, int height, int rowstride)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);
	g_return_val_if_fail (data != NULL, -1);

	return PRINT_CLASS(pc)->rgbimage (pc, data, width, height, rowstride);
}

int
gnome_print_rgbaimage (GnomePrintContext *pc, const char *data, int width, int height, int rowstride)
{
	guchar * b, * d;
	const guchar * s;
	gint x, y, alpha, tmp, ret;

	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);
	g_return_val_if_fail (data != NULL, -1);

	if (PRINT_CLASS (pc)->rgbaimage != NULL) {

		/* We have REAL rgba method */

		return PRINT_CLASS (pc)->rgbaimage (pc, data, width, height, rowstride);
	}

	if (PRINT_CLASS (pc)->rgbimage == NULL) return 0;

	/* Print RGB - better than nothing */

	b = g_new (gchar, width * height * 3);
	g_return_val_if_fail (b != NULL, -1);

	for (y = 0; y < height; y++) {
		s = data + y * rowstride;
		d = b + y * width * 3;
		for (x = 0; x < width; x++) {
			alpha = s[3];
			tmp = (s[0] - 0xff) * alpha;
			d[0] = 0xff + ((tmp + 0x80) >> 8);
			tmp = (s[1] - 0xff) * alpha;
			d[1] = 0xff + ((tmp + 0x80) >> 8);
			tmp = (s[2] - 0xff) * alpha;
			d[2] = 0xff + ((tmp + 0x80) >> 8);
			s += 4;
			d += 3;
		}
	}

	ret = PRINT_CLASS (pc)->rgbimage (pc, b, width, height, width * 3);

	g_free (b);

	return ret;
}

int
gnome_print_textline (GnomePrintContext *pc, GnomeTextLine *line)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);
	g_return_val_if_fail (line != NULL, -1);

	return PRINT_CLASS(pc)->textline (pc, line);
}

int
gnome_print_showpage (GnomePrintContext *pc)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);

	return PRINT_CLASS(pc)->showpage (pc);
}

int
gnome_print_beginpage (GnomePrintContext *pc, const char *name_of_this_page)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);

	return PRINT_CLASS(pc)->beginpage (pc, name_of_this_page);
}

int
gnome_print_setopacity (GnomePrintContext *pc, double opacity)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);

	return PRINT_CLASS(pc)->setopacity (pc, opacity);
}

int
gnome_print_context_close (GnomePrintContext *pc)
{
	return PRINT_CLASS(pc)->close (pc);
}



/*
 * These functions are wrapper functions around capabilities that may
 * or may not be deconvolved in back ends.
 */

int
gnome_print_scale (GnomePrintContext *pc, double sx, double sy)
{
	double dst[6];

	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);

	art_affine_scale (dst, sx, sy);
	return gnome_print_concat (pc, dst);
}

int
gnome_print_rotate (GnomePrintContext *pc, double theta)
{
	double dst[6];

	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);

	art_affine_rotate (dst, theta);
	return gnome_print_concat (pc, dst);
}

int
gnome_print_translate (GnomePrintContext *pc, double x, double y)
{
	double dst[6];

	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);

	art_affine_translate (dst, x, y);
	return gnome_print_concat (pc, dst);
}

/*
 * These functions provide a common interface for writing bytes to the
 *
 * printer.
 */

/* Return 0 on success */
int
gnome_print_context_open_file (GnomePrintContext *pc, const char *filename)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);
	g_return_val_if_fail (filename != NULL, -1);

	if (filename[0] == '|')
	{
		pc->f = popen (filename + 1, "w");
		pc->is_pipe = TRUE;
	}
	else
	{
		pc->f = fopen (filename, "w");
		pc->is_pipe = FALSE;
	}
	return pc->f != NULL;
}

/* Return number of bytes written */
int
gnome_print_context_write_file (GnomePrintContext *pc, char *buf, size_t size)
{
	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), 0);
	g_return_val_if_fail (pc->f != NULL, 0);

	return fwrite (buf, sizeof(char), size, pc->f);
}

/* Return number of bytes written, or -1 if error */
int
gnome_print_context_fprintf (GnomePrintContext *pc, const char *fmt, ...)
{
	va_list ap;
	int n_bytes;
	char *oldlocale;

	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);
	g_return_val_if_fail (pc->f != NULL, -1);

	oldlocale = g_strdup (setlocale (LC_NUMERIC, NULL));
	setlocale (LC_NUMERIC, "C");

	va_start (ap, fmt);
	n_bytes = vfprintf (pc->f, fmt, ap);
	va_end (ap);

	setlocale (LC_NUMERIC, oldlocale);
	g_free (oldlocale);

	return n_bytes;
}

/* Return 0 on success */
int
gnome_print_context_close_file (GnomePrintContext *pc)
{
	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);

	if (pc->f != NULL)
	{
		if (pc->is_pipe)
			return pclose (pc->f);
		else
			return fclose (pc->f);
		pc->f = NULL;
	}
	else
		return 0;
}

void
gnome_print_bpath (GnomePrintContext * gpc, ArtBpath * bpath, gboolean append)
{
	ArtBpath * p;
	gboolean closed;

	g_return_if_fail (gpc != NULL);
	g_return_if_fail (GNOME_IS_PRINT_CONTEXT (gpc));
	g_return_if_fail (bpath != NULL);

	if (bpath->code == ART_END) return;

	g_return_if_fail ((bpath->code == ART_MOVETO) || (bpath->code == ART_MOVETO_OPEN));

	closed = (bpath->code == ART_MOVETO);

	if (!append)
		gnome_print_newpath (gpc);

	gnome_print_moveto (gpc,
			    bpath->x3,
			    bpath->y3);

	for (p = bpath + 1; p->code != ART_END; p++) {
		switch (p->code) {
		case ART_MOVETO:
		case ART_MOVETO_OPEN:
			if (closed)
				gnome_print_closepath (gpc);

			closed = (p->code == ART_MOVETO);

			gnome_print_moveto (gpc,
					    p->x3,
					    p->y3);
			break;
		case ART_LINETO:
			gnome_print_lineto (gpc,
					    p->x3,
					    p->y3);
			break;
		case ART_CURVETO:
			gnome_print_curveto (gpc,
					     p->x1,
					     p->y1,
					     p->x2,
					     p->y2,
					     p->x3,
					     p->y3);
			break;
		default:
			g_log (G_LOG_DOMAIN,
			       G_LOG_LEVEL_CRITICAL,
			       "Invalid Bpath element");
			return;
			break;
		}
	}

	if (closed)
		gnome_print_closepath (gpc);

}

void
gnome_print_vpath (GnomePrintContext * gpc, ArtVpath * vpath, gboolean append)
{
	ArtVpath * p;
	gboolean closed;

	g_return_if_fail (gpc != NULL);
	g_return_if_fail (GNOME_IS_PRINT_CONTEXT (gpc));
	g_return_if_fail (vpath != NULL);

	if (vpath->code == ART_END) return;

	g_return_if_fail ((vpath->code == ART_MOVETO) || (vpath->code == ART_MOVETO_OPEN));

	closed = (vpath->code == ART_MOVETO);

	if (!append)
		gnome_print_newpath (gpc);

	gnome_print_moveto (gpc,
			    vpath->x,
			    vpath->y);

	for (p = vpath + 1; p->code != ART_END; p++) {
		switch (p->code) {
		case ART_MOVETO:
		case ART_MOVETO_OPEN:
			if (closed)
				gnome_print_closepath (gpc);

			closed = (p->code == ART_MOVETO);

			gnome_print_moveto (gpc,
					    p->x,
					    p->y);
			break;
		case ART_LINETO:
			gnome_print_lineto (gpc,
					    p->x,
					    p->y);
			break;
		default:
			g_log (G_LOG_DOMAIN,
			       G_LOG_LEVEL_CRITICAL,
			       "Invalid Vpath element");
			return;
			break;
		}
	}

	if (closed)
		gnome_print_closepath (gpc);

}
