/* vi:set ts=8 sts=0 sw=8:
 * $Id: win.c,v 1.78 2000/04/21 17:01:26 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include "main.h"
#ifdef HAVE_SYS_UTSNAME_H
#include <sys/utsname.h>
#endif
#ifdef USE_GNOME
#include <gnome.h>
#endif
#include "win.h"
#include "doc.h"
#include "undo.h"
#include "file.h"
#include "menu.h"
#include "toolbar.h"
#include "msgbar.h"
#include "msgbox.h"
#include "prefs.h"
#include "recent.h"
#include "randomtips.h"
#include "misc.h"
#ifdef APP_GNP
#include "prjbar.h"
#include "print.h"
#include "clipboard.h"
#include "search.h"
#include "htmltags.h"
#include "dialog.h"
#endif

#include "gnpintl.h"


/*** global variables ***/
#ifdef USE_TOOLBARS
toolbar_data_t main_tbdata[] = {
	{
		N_(" New "),
		N_("New File"),
		MAIN_TB_STR_NEW,
		"tb_new.xpm",
		GTK_SIGNAL_FUNC(doc_new_cb),
		NULL,
		MAIN_TB_STR_NEW
	},
	{
		N_(" Open "),
		N_("Open File"),
		MAIN_TB_STR_OPEN,
		"tb_open.xpm",
		GTK_SIGNAL_FUNC(doc_open_cb),
		NULL,
		MAIN_TB_STR_OPEN
	},
	{
		N_(" Close "),
		N_("Close File"),
		MAIN_TB_STR_CLOSE,
		"tb_cancel.xpm",
		GTK_SIGNAL_FUNC(doc_close_cb),
		NULL,
		MAIN_TB_STR_CLOSE
	},
#ifdef APP_GNP
	{
		N_(" Save "),
		N_("Save File"),
		MAIN_TB_STR_SAVE,
		"tb_save.xpm",
		GTK_SIGNAL_FUNC(doc_save_cb),
		NULL,
		MAIN_TB_STR_SAVE
	},
	{
		N_(" Print "),
		N_("Print File"),
		MAIN_TB_STR_PRINT,
		"tb_print.xpm",
		GTK_SIGNAL_FUNC(print_cb),
		NULL,
		MAIN_TB_STR_PRINT
	},
#ifdef USE_CLIPBOARD
	{
		" SPACE "
	},
	{
		N_(" Cut "),
		N_("Cut Text"),
		MAIN_TB_STR_CUT,
		"tb_cut.xpm",
		GTK_SIGNAL_FUNC(cut_cb),
		NULL,
		MAIN_TB_STR_CUT
	},
	{
		N_(" Copy "),
		N_("Copy Text"),
		MAIN_TB_STR_COPY,
		"tb_copy.xpm",
		GTK_SIGNAL_FUNC(copy_cb),
		NULL,
		MAIN_TB_STR_COPY
	},
	{
		N_(" Paste "),
		N_("Paste Text"),
		MAIN_TB_STR_PASTE,
		"tb_paste.xpm",
		GTK_SIGNAL_FUNC(paste_cb),
		NULL,
		MAIN_TB_STR_PASTE
	},
#endif	/* USE_CLIPBOARD */
#ifdef USE_UNDOREDO
#ifdef GTK_HAVE_FEATURES_1_1_0
	{
		" SPACE "
	},
	{
		N_(" Undo "),
		N_("Undo"),
		MAIN_TB_STR_UNDO,
		"tb_undo.xpm",
		GTK_SIGNAL_FUNC(undo_cb),
		NULL,
		MAIN_TB_STR_UNDO
	},
	{
		N_(" Redo "),
		N_("Redo"),
		MAIN_TB_STR_REDO,
		"tb_redo.xpm",
		GTK_SIGNAL_FUNC(redo_cb),
		NULL,
		MAIN_TB_STR_REDO
	},
	{
		" SPACE "
	},
#endif
#endif	/* USE_UNDOREDO */
#ifdef USE_SEARCH
	{
		N_(" Find "),
		N_("Find Text"),
		MAIN_TB_STR_FIND,
		"tb_find.xpm",
		GTK_SIGNAL_FUNC(search_search_cb),
		NULL,
		MAIN_TB_STR_FIND
	},
	{
		N_(" Find Next "),
		N_("Find Next"),
		MAIN_TB_STR_FNEXT,
		"tb_find_next.xpm",
		GTK_SIGNAL_FUNC(search_again_cb),
		NULL,
		MAIN_TB_STR_FNEXT
	},
	{
		N_(" Find/Repl "),
		N_("Find/Repl"),
		MAIN_TB_STR_REPLACE,
		"tb_replace.xpm",
		GTK_SIGNAL_FUNC(search_replace_cb),
		NULL,
		MAIN_TB_STR_REPLACE
	},
#endif	/* USE_SEARCH */
#endif	/* APP_GNP */
	{
		" SPACE "
	},
	{
		N_(" New Win "),
		N_("New Win"),
		MAIN_TB_STR_WINNEW,
		"tb_winnew.xpm",
		GTK_SIGNAL_FUNC(win_new_with_doc),
		NULL,
		MAIN_TB_STR_WINNEW
	},
	{
		N_(" Close Win "),
		N_("Close Win"),
		MAIN_TB_STR_WINCLOSE,
		"tb_winclose.xpm",
		GTK_SIGNAL_FUNC(win_close_cb),
		NULL,
		MAIN_TB_STR_WINCLOSE
	},
	{
		" SPACE "
	},
	{
		N_(" Prefs "),
		N_("Preferences"),
		MAIN_TB_STR_PREFS,
		"tb_prefs.xpm",
		GTK_SIGNAL_FUNC(prefs_cb),
		NULL,
		MAIN_TB_STR_PREFS
	},
	{
		N_(" Exit "),
		N_("Exit App"),
		MAIN_TB_STR_EXIT,
		"tb_exit.xpm",
		GTK_SIGNAL_FUNC(win_close_all_cb),
		NULL,
		MAIN_TB_STR_EXIT
	},
	{ NULL }
}; /* main_tbdata[] */
#endif	/* USE_TOOLBARS */


/*** external declarations ***/
extern void	win_set_title(doc_t *d);

#if defined(ENABLE_NLS) && defined(GTK_HAVE_FEATURES_1_1_0)
static char G_GNUC_UNUSED *dummyMsg[] = {
	N_("Autosaving changed files..."),
	N_("Document tabs OFF"),
	N_("Document tabs ON"),
	N_("Document tabs on the TOP"),
	N_("Document tabs on the BOTTOM"),
	N_("Document tabs on the LEFT"),
	N_("Document tabs on the RIGHT"),
	N_("Wordwrap OFF"),
	N_("Wordwrap ON") };
#endif

/*** local declarations ***/
#define MSG_AUTOSAVE		gettext("Autosaving changed files...")
#define MSG_DOC_TABS_OFF	gettext("Document tabs OFF")
#define MSG_DOC_TABS_ON		gettext("Document tabs ON")
#define MSG_DOC_TABS_TOP	gettext("Document tabs on the TOP")
#define MSG_DOC_TABS_BOTTOM	gettext("Document tabs on the BOTTOM")
#define MSG_DOC_TABS_LEFT	gettext("Document tabs on the LEFT")
#define MSG_DOC_TABS_RIGHT	gettext("Document tabs on the RIGHT")
#define MSG_DOC_WWRAP_OFF	gettext("Wordwrap OFF")
#define MSG_DOC_WWRAP_ON	gettext("Wordwrap ON")


/*** local variables ***/
static GSList *winlist = NULL;		/* list of win_t's */
#ifdef APP_DEBUG
static unsigned nextid = 10000;		/* internal window id for each win_t */
#endif

#ifdef USE_WINLIST
static GtkWidget *wlw = NULL;		/* these three are used for the */
static GtkWidget *wlw_data = NULL;	/* winlist popup window */
static unsigned curwinrow = 0;		/* current window rownum in win list */
#endif	/* USE_WINLIST */
#ifdef USE_AUTOSAVE
static guint autosave_id = 0;		/* doc autosave timeout id */
#endif	/* USE_AUTOSAVE */


/*** local function prototypes ***/
static void	notebook_init(win_t *w);
static int	win_delete_event_cb(GtkWidget *wgt, GdkEvent *, gpointer);
static bool_t	win_close_common(win_t *w, bool_t exiting);
#ifdef USE_WINLIST
static void	win_list_destroy(GtkWidget *wgt, gpointer cbdata);
static void	win_list_remove(win_t *w);
static void	win_list_add(win_t *w, char *title);
static int	win_list_select(GtkWidget *wgt, unsigned row, int column,
				GdkEventButton *event, gpointer cbdata);
static void	win_list_close_selected(GtkWidget *wgt, gpointer cbdata);
#else
# define win_list_destroy(wgt, cbdata)
# define win_list_remove(w)
# define win_list_add(w, title)
# define win_list_select(wgt, row, col, event, cbdata)
# define win_list_close_selected(wgt, cbdata)
#endif	/* USE_WINLIST */
static void	win_doc_tab_settings_update(win_t *w);
#if 0
static gboolean	win_key_press(GtkWidget *txt, GdkEventKey *event, win_t *w);
#endif
#ifdef GTK_HAVE_FEATURES_1_1_0
static void	win_drag_data_recv(GtkWidget *wgt, GdkDragContext *context,
			int x, int y, GtkSelectionData *seldata, guint info,
			guint time, gpointer cbdata);
#endif
#if !defined(USE_GNOME) && defined(GTK_HAVE_FEATURES_1_1_0)
static GList *	gnome_uri_list_extract_filenames(const gchar* uri_list);
static GList *	gnome_uri_list_extract_uris(const gchar* uri_list);
static void	gnome_uri_list_free_strings(GList *list);
#endif
#ifdef WANT_SESSION
static void	win_session_file_write(void);
#else
# define	win_session_file_write()
#endif


/*** global function definitions ***/
#ifdef USE_AUTOSAVE
/*
 * PUBLIC: win_autosave
 *
 * create a new window with a document.  currently, only creates an "Untitled"
 * document.
 */
int
win_autosave(gpointer cbdata)
{
	GSList *wlp, *dlp;
	win_t *w;
	doc_t *d;
	int num = 0;

	if (prefs.autosave < 1) {	/* this should not happen */
		g_warning("win_autosave() entered with value %d\n",
			  prefs.autosave);
		return 1;
	}

	msgbox_printf("Autosave in progress...");
	wlp = winlist;
	while (wlp) {
		w = (win_t *)(wlp->data);
		msgbar_printf(w, MSG_AUTOSAVE);
		dlp = (GSList *)(w->doclist);
		while (dlp) {
			d = (doc_t *)(dlp->data);
			if (d->changed && d->fname) {
				(void)file_save_execute(d, FALSE);
				num++;
			}
			dlp = dlp->next;
		}
		wlp = wlp->next;
	}
	msgbox_printf("Autosave completed (%d files saved)", num);

	return 1;	/* for GTK, must return 1 */
} /* win_new_with_new_doc */


void
win_autosave_reset(void *data)
{
	if (prefs.autosave > 0)
		gtk_timeout_remove(autosave_id);

	win_autosave_init();
} /* win_autosave_reset */


/*
 * PUBLIC: win_autosave_init
 */
void
win_autosave_init(void)
{
	if (prefs.autosave < 1)
		return;

	autosave_id = gtk_timeout_add(prefs.autosave * (60000),
				      (GtkFunction)win_autosave, NULL);
} /* app_autosave_init */
#endif	/* USE_AUTOSAVE */


/*
 * PUBLIC: win_foreach
 *
 * traver window list and call a function on each window.
 */
void
win_foreach(void (*func)(void *))
{
	GSList *wlp;
	win_t *wp;

	wlp = winlist;
	while (wlp) {
		wp = (win_t *)(wlp->data);
		func(wp);
		win_set_title(DOC_CURRENT(wp));
		wlp = wlp->next;
	}
} /* win_foreach */


/*
 * PUBLIC: win_doc_foreach
 *
 * traverse window list, and traverse all docs in each window.
 */
void
win_doc_foreach(void (*func)(void *))
{
	GSList *wlp;
	win_t *wp;

	for (wlp = winlist; wlp; wlp = wlp->next) {
		wp = (win_t *)(wlp->data);
		doc_foreach(wp, func);
	}
} /* win_doc_foreach */


/*
 * PUBLIC: win_new_with_doc
 *
 * create a new window with a document.  currently, only creates an "Untitled"
 * document.
 */
void
win_new_with_doc(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w;

	w = win_new();
	doc_new(w, NULL, DocText, (UPDATE_MSGBAR | DO_LOAD | UPDATE_TITLE));
	gtk_widget_show(w->toplev);
	(void)gtk_signal_connect_after(GTK_OBJECT(w->nb), "switch_page",
				       GTK_SIGNAL_FUNC(doc_switch_page), w);
} /* win_new_with_new_doc */


/*
 * PUBLIC: win_new
 *
 * creates a new window and adds it to the winlist.
 */
win_t *
win_new(void)
{
#ifdef GTK_HAVE_FEATURES_1_1_0
	GtkTargetEntry dragtypes[] = { { "text/uri-list", 0, 0 } };
#endif
#ifdef USE_HTMLTAGS
	GtkWidget *tb;
#endif
	win_t *w;
	GdkPixmap *pixmap;
	GdkBitmap *mask;

	w	= g_new0(win_t, 1);
#ifdef APP_DEBUG
	w->id	= nextid++;
#endif
#ifdef USE_HTMLTAGS
	w->tc_row = -1;
#endif

	/* toplevel */
#ifdef USE_GNOME
	w->toplev = gnome_app_new(APP_NAME, APP_NAME);
#else
	w->toplev = gtk_window_new(GTK_WINDOW_TOPLEVEL);
#endif
#if 0
	gtk_signal_connect(GTK_OBJECT(w->toplev), "key_press_event",
			   GTK_SIGNAL_FUNC(win_key_press), w);
#endif
	gtk_widget_set_name(w->toplev, "gnp+");
	gtk_window_set_wmclass(GTK_WINDOW(w->toplev), "gnp+", "gnp+");
	gtk_window_set_title(GTK_WINDOW(w->toplev), APP_NAME);
	gtk_window_set_policy(GTK_WINDOW(w->toplev), TRUE, TRUE, FALSE);
	gtk_container_border_width(GTK_CONTAINER(w->toplev), 0);
	(void)gtk_signal_connect(GTK_OBJECT(w->toplev), "delete_event",
				 GTK_SIGNAL_FUNC(win_delete_event_cb), w);

	/* window geometry */
# ifdef GTK_HAVE_FEATURES_1_1_0
	if (IS_SAVE_WIN_WIDTH() || IS_SAVE_WIN_HEIGHT())
		gtk_window_set_default_size(GTK_WINDOW(w->toplev),
					    prefs.win_width, prefs.win_height);
# else
	if (IS_SAVE_WIN_WIDTH() && IS_SAVE_WIN_HEIGHT()) {
		gtk_widget_set_usize(GTK_WIDGET(w->toplev),
				     prefs.win_width, prefs.win_height);
	} else {
		if (IS_SAVE_WIN_WIDTH())
			gtk_widget_set_usize(GTK_WIDGET(w->toplev),
					     prefs.win_width, -1);
		if (IS_SAVE_WIN_HEIGHT())
			gtk_widget_set_usize(GTK_WIDGET(w->toplev),
					     -1, prefs.win_height);
	}
# endif
	if (IS_SAVE_WIN_POS())
		gtk_widget_set_uposition(w->toplev,
					 prefs.win_xpos, prefs.win_ypos);

	/* mainbox */
	w->mainbox = gtk_vbox_new(FALSE, 0);
#ifdef USE_GNOME
	gnome_app_set_contents(GNOME_APP(w->toplev), w->mainbox);
#else
	gtk_container_add(GTK_CONTAINER(w->toplev), w->mainbox);
	gtk_widget_show(w->mainbox);
#endif

	/* msgbox can be initialized anywhere before a file is opened */
	msgbox_init();

	/* these need to be in order */
#ifdef USE_GNOME
	/*
	 * since gnome provides each gnomeapp with its own statusbar, we need
	 * to get it done here, before the call to menu_main_init() because the
	 * statusbar is needed for gnome_app_install_menu_hints().
	 */
	w->msgbar = gnome_appbar_new(FALSE, TRUE, GNOME_PREFERENCES_USER);
	gnome_app_set_statusbar(GNOME_APP(w->toplev), GTK_WIDGET(w->msgbar));
	doc_info_init(w, w->msgbar);
#endif	/* USE_GNOME */

	menu_main_init(w);
	recent_list_init(w);
	gtk_widget_realize(w->toplev);

	/* the window's icon after the win is realized */
	(void)misc_new_pixmap("gnp48x48.xpm", w->toplev, &pixmap, &mask);
	gdk_window_set_icon(w->toplev->window, NULL, pixmap, mask);

#ifdef USE_TOOLBARS
	w->main_tb = toolbar_create_start(w, main_tbdata);
	(void)toolbar_create_finish(w, w->mainbox, w->main_tb, "MainToolbar");
	tb_item_enable(w->main_tb, MAIN_TB_STR_UNDO, FALSE);
	tb_item_enable(w->main_tb, MAIN_TB_STR_REDO, FALSE);
# ifdef USE_HTMLTAGS
	tb = html_tb_create(w);
	w->html_tb_h = toolbar_create_finish(w, w->mainbox, tb, "HtmlToolbar");
	if (IS_SHOW_HTML_TOOLBAR())
		gtk_widget_show(w->html_tb_h);
	else
		gtk_widget_hide(w->html_tb_h);
# endif
#endif	/* USE_TOOLBARS */

	notebook_init(w);
	prjbar_init(w);

#ifndef USE_GNOME
	/*
	 * msgbar and docinfo button go together.  note that for gnome, this
	 * is done up above, before the menus.  we do it here because we pack
	 * widgets in the order they appear (vertically).
	 */
	w->hbox_bot = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(w->mainbox), w->hbox_bot, FALSE, FALSE, 0);
	gtk_widget_show(w->hbox_bot);
	msgbar_init(w, w->hbox_bot);
	doc_info_init(w, w->hbox_bot);
#endif	/* no gnome */

	quickmenu_init(w);

#ifdef GTK_HAVE_FEATURES_1_1_0
	gtk_drag_dest_set(w->toplev, GTK_DEST_DEFAULT_ALL, dragtypes,
			  sizeof(dragtypes) / sizeof(dragtypes[0]),
			  GDK_ACTION_COPY);
	(void)gtk_signal_connect(GTK_OBJECT(w->toplev), "drag_data_received",
				 GTK_SIGNAL_FUNC(win_drag_data_recv), w);
#endif

	/* add to winlist and winlist popup */
	winlist = g_slist_append(winlist, (gpointer)w);
	win_list_add(w, GTK_WINDOW(w->toplev)->title);

	/* initialize random tips */
	random_tips_init(w);

	return w;
} /* win_new */


#ifdef GTK_HAVE_FEATURES_1_1_0
static void
win_drag_data_recv(GtkWidget *wgt, GdkDragContext *context, int x, int y,
		   GtkSelectionData *seldata, guint info, guint time,
		   gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	GList *fnames, *fnp;
	guint count;

	fnames = gnome_uri_list_extract_filenames((char *)seldata->data);
	count = g_list_length(fnames);
	if (count > 0) {
		msgbar_printf(w, _("Opening files from drag and drop..."));
		gtk_events_flush();
		/*
		 * we temporarily disable the switch_page signal to avoid
		 * loading files that are going to be closed anyway.
		 */
		gtk_signal_disconnect_by_func(GTK_OBJECT(w->nb),
					GTK_SIGNAL_FUNC(doc_switch_page), w);

		for (fnp = fnames; fnp; fnp = fnp->next, count--) {
			doc_new(w, (char *)(fnp->data), DocText,
					(count - 1 == 0) ? DO_LOAD : 0);
#ifdef USE_RECENT
			if (count <= prefs.maxrecent)
				recent_list_add(w, (char *)(fnp->data));
#endif	/* USE_RECENT */
		}

		/*
		 * now that we're done closing whatever we could, re-establish
		 * the switch_page signal on the notebook.
		 */
		(void)gtk_signal_connect_after(GTK_OBJECT(w->nb), "switch_page",
					GTK_SIGNAL_FUNC(doc_switch_page), w);
	}
	gnome_uri_list_free_strings(fnames);

} /* win_drag_data_recv */
#endif	/* GTK_HAVE_FEATURES_1_1_0 */

#if !defined(USE_GNOME) && defined(GTK_HAVE_FEATURES_1_1_0)

/*** the next three routines are taken straight from gnome-libs so that the
     gtk-only version can receive drag and drops as well ***/
/**
 * gnome_uri_list_free_strings:
 * @list: A GList returned by gnome_uri_list_extract_uris() or gnome_uri_list_extract_filenames()
 *
 * Releases all of the resources allocated by @list.
 */
static void          
gnome_uri_list_free_strings      (GList *list)
{
	g_list_foreach (list, (GFunc)g_free, NULL);
	g_list_free (list);
}


/**
 * gnome_uri_list_extract_uris:
 * @uri_list: an uri-list in the standard format.
 *
 * Returns a GList containing strings allocated with g_malloc
 * that have been splitted from @uri-list.
 */
static GList*        
gnome_uri_list_extract_uris (const gchar* uri_list)
{
	const gchar *p, *q;
	gchar *retval;
	GList *result = NULL;
	
	g_return_val_if_fail(uri_list != NULL, NULL);

	p = uri_list;
	
	/* We don't actually try to validate the URI according to RFC
	 * 2396, or even check for allowed characters - we just ignore
	 * comments and trim whitespace off the ends.  We also
	 * allow LF delimination as well as the specified CRLF.
	 */
	while (p) {
		if (*p != '#') {
			while (isspace((int)(*p)))
				p++;
			
			q = p;
			while (*q && (*q != '\n') && (*q != '\r'))
				q++;
		  
			if (q > p) {
			        q--;
				while (q > p && isspace((int)(*q)))
					q--;
				
				retval = (char *)g_malloc(q - p + 2);
				strncpy (retval, p, q - p + 1);
				retval[q - p + 1] = '\0';
				
				result = g_list_prepend(result, retval);
			}
		}
		p = strchr(p, '\n');
		if (p)
			p++;
	}
	
	return g_list_reverse(result);
}


/**
 * gnome_uri_list_extract_filenames:
 * @uri_list: an uri-list in the standard format
 *
 * Returns a GList containing strings allocated with g_malloc
 * that contain the filenames in the uri-list.
 *
 * Note that unlike gnome_uri_list_extract_uris() function, this
 * will discard any non-file uri from the result value.
 */
static GList*        
gnome_uri_list_extract_filenames (const gchar* uri_list)
{
	GList *tmp_list, *node, *result;
	
	g_return_val_if_fail (uri_list != NULL, NULL);
	
	result = gnome_uri_list_extract_uris (uri_list);

	tmp_list = result;
	while (tmp_list) {
		gchar *s = (char *)tmp_list->data;
		
		node = tmp_list;
		tmp_list = tmp_list->next;

		if (!strncmp (s, "file:", 5)) {
			node->data = g_strdup (s+5);
		} else {
			result = g_list_remove_link(result, node);
			g_list_free_1 (node);
		}
		g_free (s);
	}
	return result;
}
#endif	/* !defined(USE_GNOME) && defined(GTK_HAVE_FEATURES_1_1_0) */


/*
 * PUBLIC: win_close_cb
 *
 * window close callback
 */
void
win_close_cb(GtkWidget *wgt, gpointer cbdata)
{
	(void)win_close_common((win_t *)cbdata, FALSE);
	/* NOT REACHED */
} /* win_close_cb */


/*
 * PUBLIC: win_close_all_cb
 *
 * close all windows (e.g., quit) callback
 */
void
win_close_all_cb(GtkWidget *wgt, gpointer cbdata)
{
	GSList *wlp;
	win_t *w;
	bool_t changed = FALSE;

	msgbox_close();
#ifdef USE_WINLIST
	if (wlw)
		gtk_widget_hide(wlw);
#endif	/* USE_WINLIST */

	for (wlp = winlist; wlp; wlp = wlp->next) {
		w = (win_t *)(wlp->data);
		if (doc_check_if_any_changed(w)) {
			changed = TRUE;
			break;
		}
	}

	if (!changed) {
		for (wlp = winlist; wlp; wlp = wlp->next) {
			w = (win_t *)(wlp->data);
			gtk_widget_hide(w->toplev);
			if (w->saveas)
				gtk_widget_hide(w->saveas);
#ifdef USE_SEARCH
			if (w->srg && w->srg->toplev)
				gtk_widget_hide(w->srg->toplev);
#endif	/* USE_SEARCH */
			if (w->filesel)
				gtk_widget_hide(w->filesel);
#ifdef USE_DOCLIST
			if (w->dlw)
				gtk_widget_hide(w->dlw);
#endif	/* USE_DOCLIST */
		}
	}

	win_session_file_write();
	while (winlist) {
		if (win_close_common((win_t *)(winlist->data), TRUE) == FALSE)
			break;
	}

	g_assert(winlist != NULL);
} /* win_close_all_cb */


/*
 * PUBLIC: win_dtab_toggle
 *
 * callback: toggles doc tabs
 */
void
win_dtab_toggle(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	if (IS_SHOW_TABS()) {
		CLEAR_SHOW_TABS();
		msgbar_printf(w, MSG_DOC_TABS_OFF);
	} else {
		SET_SHOW_TABS();
		msgbar_printf(w, MSG_DOC_TABS_ON);
	}
	win_foreach(win_redraw_doc_tab);
} /* win_dtab_toggle */


/*
 * PUBLIC: win_dtab_top
 *
 * callback: sets doc tab location to top
 */
void
win_dtab_top(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	prefs.tabpos = GTK_POS_TOP;
	win_foreach(win_redraw_doc_tab);
	msgbar_printf(w, MSG_DOC_TABS_TOP);
} /* win_dtab_top */


/*
 * PUBLIC: win_dtab_bot
 *
 * callback: sets doc tab location to bottom
 */
void
win_dtab_bot(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	prefs.tabpos = GTK_POS_BOTTOM;
	win_foreach(win_redraw_doc_tab);
	msgbar_printf(w, MSG_DOC_TABS_BOTTOM);
} /* win_dtab_bot */


/*
 * PUBLIC: win_dtab_left
 *
 * callback: sets doc tab location to left
 */
void
win_dtab_left(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	prefs.tabpos = GTK_POS_LEFT;
	win_foreach(win_redraw_doc_tab);
	msgbar_printf(w, MSG_DOC_TABS_LEFT);
} /* win_dtab_left */


/*
 * PUBLIC: win_dtab_right
 *
 * callback: sets doc tab location to right
 */
void
win_dtab_right(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	prefs.tabpos = GTK_POS_RIGHT;
	win_foreach(win_redraw_doc_tab);
	msgbar_printf(w, MSG_DOC_TABS_RIGHT);
} /* win_dtab_right */


/*
 * PUBLIC: win_redraw_doc_tab
 *
 * redraw the document tabs for a specified window.  called from:
 *	- prefs_save, when the preferences may have changed.
 */
void
win_redraw_doc_tab(void *data)
{
	win_t *w = (win_t *)data;

	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(w->nb), prefs.tabpos);
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(w->nb), IS_SHOW_TABS());
	win_doc_tab_settings_update(w);
} /* win_redraw_doc_tab */


/*
 * PRIVATE: win_doc_tab_settings_update
 *
 * updates the various bit fields/options for doc tabs.  needed when updating
 * the toggle/radio buttons in the menu items.
 */
static void
win_doc_tab_settings_update(win_t *w)
{
	switch (prefs.tabpos) {
	case GTK_POS_LEFT:
		SET_DOC_TABS_LEFT();
		CLEAR_DOC_TABS_TOP();
		CLEAR_DOC_TABS_BOTTOM();
		CLEAR_DOC_TABS_RIGHT();
		break;
	case GTK_POS_RIGHT:
		SET_DOC_TABS_RIGHT();
		CLEAR_DOC_TABS_TOP();
		CLEAR_DOC_TABS_BOTTOM();
		CLEAR_DOC_TABS_LEFT();
		break;
	case GTK_POS_TOP:
		SET_DOC_TABS_TOP();
		CLEAR_DOC_TABS_BOTTOM();
		CLEAR_DOC_TABS_LEFT();
		CLEAR_DOC_TABS_RIGHT();
		break;
	case GTK_POS_BOTTOM:
		SET_DOC_TABS_BOTTOM();
		CLEAR_DOC_TABS_TOP();
		CLEAR_DOC_TABS_RIGHT();
		CLEAR_DOC_TABS_LEFT();
		break;
	default:
		g_warning("Unknown tab position type = %d\n", prefs.tabpos);
	}

	/* need to update the toggle items in the menu */
	menu_main_state_set(w);
} /* win_doc_tab_settings_update */


/*
 * PUBLIC: win_toggle_wwrap
 *
 * callback: toggles word wrap
 */
void
win_wwrap_toggle(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	doc_t *d;

	g_assert(w != NULL);
	d = DOC_CURRENT(w);
	g_assert(d != NULL);
	if (IS_USE_WORDWRAP()) {
		gtk_text_set_word_wrap(GTK_TEXT(d->data), FALSE);
		CLEAR_USE_WORDWRAP();
		msgbar_printf(w, MSG_DOC_WWRAP_OFF);
	} else {
		gtk_text_set_word_wrap(GTK_TEXT(d->data), TRUE);
		SET_USE_WORDWRAP();
		msgbar_printf(w, MSG_DOC_WWRAP_ON);
	}
} /* win_wwrap_toggle */


/*** local function definitions ***/
static int
win_delete_event_cb(GtkWidget *wgt, GdkEvent *event, gpointer cbdata)
{
	(void)win_close_common((win_t *)cbdata, FALSE);
	return TRUE;	/* NOT REACHED */
} /* win_delete_event_cb */


/*
 * PRIVATE: win_close_common
 *
 * common routine for actually closing a window.  called from win_close_cb()
 * and win_close_all_cb().
 */
static bool_t
win_close_common(win_t *w, bool_t exiting)
{
#if defined(USE_TOOLBARS) && defined(USE_HTMLTAGS)
	GSList *tblp;
	tblist_data_t *tbldp;
#endif
	GSList *dlp, *wlp;

	doc_list_destroy(NULL, w);
	win_list_destroy(NULL, NULL);

#ifdef WANT_SESSION
	/*
	 * Exiting is TRUE only from win_close_all_cb, in which case, the
	 * session has already been saved.
	 */
	if (!exiting && (g_slist_length(winlist) < 2))
		win_session_file_write();
#endif

	dlp = (GSList *)w->doclist;
	if (dlp == NULL || w->numdoc < 1)
		return TRUE;

	if (doc_close_all_common(w) == FALSE)
		return FALSE;

	dlp = (GSList *)w->doclist;
	g_assert(dlp == NULL && w->numdoc == 0);

	prj_free_all(w);

	/* update preferences */
	if (IS_SAVE_WIN_POS()) {
		gdk_window_get_position(w->toplev->window,
					&prefs.win_xpos, &prefs.win_ypos);
		GNPDBG_WIN(("win_close_common: xpos = %d, ypos = %d\n",
			    prefs.win_xpos, prefs.win_ypos));
	}

	gdk_window_get_size(w->toplev->window,
			    &prefs.win_width, &prefs.win_height);
	GNPDBG_WIN(("win_close_common: width = %d, height = %d\n",
		    prefs.win_width, prefs.win_height));

	/* remove from the win list popup window */
	win_list_remove(w);

	/***** point of no return *****/

	/* remove from the linked list of win_t's */
	wlp = g_slist_find(winlist, w);
	winlist = g_slist_remove_link(winlist, wlp);
	g_slist_free_1(wlp);

#ifdef USE_TOOLBARS
	g_slist_free(w->tblist);
	g_slist_free(w->tbhlist);
# ifdef USE_HTMLTAGS
	while (w->html_tblist) {
		tblp = w->html_tblist;
		w->html_tblist = g_slist_remove_link(w->html_tblist, tblp);
		tbldp = (tblist_data_t *)tblp->data;
		g_free(tbldp->name);
		g_free(tbldp->private_name);
		g_free(tbldp);
		g_slist_free_1(tblp);
	}
# endif
#endif	/* USE_TOOLBARS */

	recent_list_free(w);

	g_free(w->lastmsg);
	if (w->timeout_id)
		gtk_timeout_remove(w->timeout_id);
#ifdef USE_RANDOMTIPS
	if (w->rtips_timeout)
		gtk_timeout_remove(w->rtips_timeout);
#endif	/* USE_RANDOMTIPS */

	gtk_widget_destroy(w->toplev);
	g_free(w);

	/* save preferences */
	if (IS_SAVE_WIN_WIDTH() || IS_SAVE_WIN_HEIGHT() || IS_SAVE_WIN_POS())
		prefs_save();

	if (winlist == NULL)
		gtk_exit(0);

	return TRUE;
} /* win_close_common */


/*
 * PRIVATE: notebook_init
 *
 * creates the notebook to contain all the documents.
 */
static void
notebook_init(win_t *w)
{
	w->nb = gtk_notebook_new();
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(w->nb), TRUE);
	gtk_notebook_popup_enable(GTK_NOTEBOOK(w->nb));
	gtk_box_pack_start(GTK_BOX(w->mainbox), w->nb, TRUE, TRUE, 0);

	/* set according to preferences */
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(w->nb), prefs.tabpos);
	switch (prefs.tabpos) {
	case GTK_POS_LEFT:
		SET_DOC_TABS_LEFT();
		break;
	case GTK_POS_RIGHT:
		SET_DOC_TABS_RIGHT();
		break;
	case GTK_POS_TOP:
		SET_DOC_TABS_TOP();
		break;
	case GTK_POS_BOTTOM:
		SET_DOC_TABS_BOTTOM();
		break;
	default:
		g_warning("Unknown tab position type = %d\n", prefs.tabpos);
	}
	if (IS_SHOW_TABS())
		gtk_notebook_set_show_tabs(GTK_NOTEBOOK(w->nb), TRUE);
	else
		gtk_notebook_set_show_tabs(GTK_NOTEBOOK(w->nb), FALSE);

	gtk_widget_show(w->nb);
} /* notebook_init */


/*
 * PUBLIC: win_set_title
 *
 * sets the window title.
 */
#define MAX_HOST_LEN	256
void
win_set_title(doc_t *d)
{
#ifdef HAVE_SYS_UTSNAME_H
	struct utsname un;
#else
	char hname[MAX_HOST_LEN];
#endif
	char *title;
	char *fullpath = NULL, *user = NULL, *host = NULL;
	int len = strlen(APP_NAME);

	if (IS_USER_NAME_IN_TITLE() && (user = getenv("USER")))
		len += strlen(user);

	if (IS_HOST_NAME_IN_TITLE()) {
#ifdef HAVE_SYS_UTSNAME_H
		if (!uname(&un))
			host = un.nodename;
#else
# ifdef HAVE_SYS_SYSTEMINFO_H
#  define gethostname(nam, len) sysinfo(SI_HOSTNAME, nam, len)
# endif
		gethostname(hname, MAX_HOST_LEN);
		host = hname;
#endif
		len += strlen(host) + 1;
	}

	if ((IS_USER_NAME_IN_TITLE() && user) ||
	    (IS_HOST_NAME_IN_TITLE() && host))
		len += 4;

	if (d && (fullpath = file_full_pathname_make(d->fname)))
		len += strlen(fullpath) + 5;
	else
		len += 4;

	title = g_new(char, len);
	title[0] = '\0';

	if ((IS_USER_NAME_IN_TITLE() && user) ||
	    (IS_HOST_NAME_IN_TITLE() && host))
		strcat(title, "(");
	if (IS_USER_NAME_IN_TITLE() && user)
		strcat(title, user);
	if (IS_HOST_NAME_IN_TITLE() && host) {
		strcat(title, "@");
		strcat(title, host);
	}

	if ((IS_USER_NAME_IN_TITLE() && user) ||
	    (IS_HOST_NAME_IN_TITLE() && host))
		strcat(title, ") ");

	if (d && fullpath) {
		strcat(title, APP_NAME);
		strcat(title, " - ");
		strcat(title, fullpath);
		g_free(fullpath);
	} else
		strcat(title, APP_NAME);

	gtk_window_set_title(GTK_WINDOW(d->w->toplev), title);
	g_free(title);
} /* win_set_title */


#ifdef USE_WINLIST
/*
 * PUBLIC: win_list_set_curdoc
 *
 * set the current document column/field for the window specified.
 */
void
win_list_set_curdoc(win_t *w)
{
	GtkCList *clist;
	doc_t *d;

	if (wlw_data == NULL)
		return;

	d = DOC_CURRENT(w);
	clist = GTK_CLIST(wlw_data);
	gtk_clist_freeze(clist);
	gtk_clist_set_text(clist, w->rownum, FnameCol, doc_basefname(d));
	gtk_clist_thaw(clist);
} /* win_list_set_curdoc */


/*
 * PUBLIC: win_list_set_numdoc
 *
 * set the # of docs column in the window list popup.
 */
void
win_list_set_numdoc(win_t *w)
{
	GtkCList *clist;
	char buf[8];

	if (wlw_data == NULL)
		return;
	g_assert(w != NULL);

	clist = GTK_CLIST(wlw_data);
	gtk_clist_freeze(clist);
	g_snprintf(buf, 16, "%d", w->numdoc);
	gtk_clist_set_text(clist, w->rownum, FsizeCol, buf);
	gtk_clist_thaw(clist);
} /* win_list_set_numdoc */


/*
 * PUBLIC: win_list_show
 *
 * creates a new popup window containing a list of open files/documents
 * for the window from which it was invoked.
 */
#define WIN_LIST_NUM_COLUMNS	3
void
win_list_show(GtkWidget *wgt, gpointer cbdata)
{
#ifdef USE_LIBGLADE
	GladeXML *xml;
	GtkWidget *tmp;
#else
	char *titles[]= { N_(" # "), N_(" # Docs "), N_(" Current Document ") };
	int i;
#endif
	GSList *wlp;
	win_t *wp;

	if (misc_show_and_raise(wlw))
		return;

#ifdef USE_LIBGLADE
	xml = my_glade_xml_get("Win List");
	wlw = my_glade_widget_get(xml, "Win List", GLADE_POS_MOUSE);
	wlw_data = my_glade_widget_get(xml, "winlistclist", 0);
	(void)gtk_signal_connect(GTK_OBJECT(wlw), "destroy",
				 GTK_SIGNAL_FUNC(win_list_destroy), NULL);
	(void)gtk_signal_connect(GTK_OBJECT(wlw_data), "select_row",
				 GTK_SIGNAL_FUNC(win_list_select), NULL);
	tmp = my_glade_widget_get(xml, "closeselected", 0);
	(void)gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
				 GTK_SIGNAL_FUNC(win_list_close_selected),NULL);
	tmp = my_glade_widget_get(xml, "closeall", 0);
	(void)gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
				 GTK_SIGNAL_FUNC(win_close_all_cb), NULL);
	tmp = my_glade_widget_get(xml, "exitwinlist", 0);
	(void)gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
				 GTK_SIGNAL_FUNC(win_list_destroy), NULL);
	gtk_window_set_default_size(GTK_WINDOW(wlw),
				    prefs.winlist_w, prefs.winlist_h);
	gtk_object_destroy(GTK_OBJECT(xml));
#else
	for (i = 0; i < sizeof(titles)/sizeof(char*); i++)
		titles[i] = gettext(titles[i]);

	do_dialog_clist(&wlw, N_("Window List"),
			prefs.winlist_h, prefs.winlist_w,
			&wlw_data, WIN_LIST_NUM_COLUMNS, titles,
			win_list_destroy, NULL,
			win_list_select, NULL,
			win_list_close_selected, NULL,
			win_close_all_cb, NULL);
#endif

	/* for each window, add info into the clist */
	gtk_clist_freeze(GTK_CLIST(wlw_data));
	wlp = winlist;
	while (wlp) {
		wp = (win_t *)(wlp->data);
		win_list_add(wp, doc_basefname(DOC_CURRENT(wp)));
		wlp = wlp->next;
	}
	gtk_clist_thaw(GTK_CLIST(wlw_data));

	gtk_widget_show_all(wlw);
} /* win_list_show */


/*
 * PUBLIC: win_list_destroy
 *
 * zap!
 */
static void
win_list_destroy(GtkWidget *wgt, gpointer cbdata)
{
	if (wlw) {
		if (wlw->window) {
			gdk_window_get_size(wlw->window,
					&prefs.winlist_w, &prefs.winlist_h);
			GNPDBG_HTMLVIEW(("win_list_destroy: w = %d, h = %d\n",
					prefs.winlist_w, prefs.winlist_h));
		}
		gtk_widget_destroy(wlw);
		wlw = NULL;
		wlw_data = NULL;
	}
} /* win_list_destroy */


/*
 * PRIVATE: win_list_close_selected
 *
 * close the window that is highlighted in the win list popup.
 */
static void
win_list_close_selected(GtkWidget *wgt, gpointer cbdata)
{
	GSList *wlp;
	win_t *w;

	if (wlw == NULL)
		return;

	wlp = winlist;
	while (wlp) {
		w = (win_t *)(wlp->data);
		if (w->rownum == curwinrow) {
			if (win_close_common(w, FALSE) == TRUE)
				break;
		}
		wlp = wlp->next;
	}
} /* win_list_close_selected */


/*
 * PUBLIC: win_list_remove
 *
 * removes an entry from the files list window.
 */
static void
win_list_remove(win_t *w)
{
	char buf[4];
	int i;
	GtkCList *clist;
	GSList *wlp;
	win_t *wp;
	
	if (wlw == NULL)
		return;
	g_assert(w != NULL);

	clist = GTK_CLIST(wlw_data);
	g_assert(clist != NULL);
	gtk_clist_freeze(clist);
	gtk_clist_remove(clist, w->rownum);

	/* renumber any entries starting with the row just removed */
	if (clist->rows > 1) {
		for (i = w->rownum; i < clist->rows; i++) {
			g_snprintf(buf, 4, "%d", i + 1);
			gtk_clist_set_text(clist, i, FnumCol, buf);
		}
	}

	/* renumber rownum field for remaining windows */
	wlp = winlist;
	while (wlp) {
		wp = (win_t *)(wlp->data);
		if (wp->rownum > w->rownum)
			wp->rownum--;
		wlp = wlp->next;
	}

	/* pick "next most recent" row to highlight */
	if (w->rownum == (unsigned)(clist->rows))
		gtk_clist_select_row(clist, w->rownum - 1, FnumCol);
	else
		gtk_clist_select_row(clist, w->rownum, FnumCol);
	gtk_clist_thaw(clist);
} /* win_list_remove */


/*
 * PRIVATE: win_list_add
 *
 * appends an entry from the files list window.
 */
static void
win_list_add(win_t *w, char *title)
{
	char numstr[4], numdocstr[8];
	char *rownfo[WIN_LIST_NUM_COLUMNS];
	GtkCList *clist;

	
	if (wlw == NULL)
		return;
	g_assert(w != NULL);

	clist = GTK_CLIST(wlw_data);

	g_snprintf(numstr, 4, "%d", clist->rows + 1);
	rownfo[0] = numstr;
	g_snprintf(numdocstr, 8, "%4d", w->numdoc);
	rownfo[1] = numdocstr;
	rownfo[2] = title;
	gtk_clist_freeze(clist);
	gtk_clist_append(clist, rownfo);
	gtk_clist_select_row(clist, clist->rows - 1, FnumCol);
	gtk_clist_thaw(clist);
	w->rownum = clist->rows - 1;
} /* win_list_add */


/*
 * PRIVATE: win_list_select
 *
 * callback routine when selecting a window in the winlist popup.  this does
 * two things:
 *
 * 1. sets the curwinrow variable to the row that was selected.  this is used
 * for the "close" button on the winlist popup, so we know which window to
 * close.  we need this because the winlist itself is our own creation, and
 * there is no association at all with the close button's context.
 *
 * 2. raises the window which was selected.
 */
static int
win_list_select(GtkWidget *wgt, unsigned row, int column,
	GdkEventButton *event, gpointer cbdata)
{
	GSList *wlp;
	win_t *w;

	curwinrow = row;
	wlp = winlist;
	while (wlp) {
		w = (win_t *)(wlp->data);
		if (w->rownum == row) {
			(void)misc_show_and_raise(w->toplev);
			break;
		}
		wlp = wlp->next;
	}

	return TRUE;
} /* win_list_select */
#endif	/* USE_WINLIST */


#if 0
#include <gdk/gdkkeysyms.h>
/*
 * see gtk_text_key_press() on how to handle control + char
 */
static gboolean
win_key_press(GtkWidget *txt, GdkEventKey *event, win_t *w)
{
	switch (event->keyval) {
	case GDK_Control_L:
		printf("left control\n");
		return TRUE;;
	case GDK_Control_R:
		printf("right control\n");
		return TRUE;;
	case GDK_Multi_key:
		printf("multi key\n");
		return TRUE;
	case GDK_f:
		if (event->state & GDK_CONTROL_MASK)
			printf("control + lower case f\n");
		else
			printf("lower case f\n");
		return TRUE;
	} /* switch (event->keyval) */

	return FALSE;
} /* win_key_press */
#endif


#ifdef WANT_SESSION
static void
win_session_file_write(void)
{
	FILE *fp;
	win_t *w;
	doc_t *d;
	char *fpath;
	GSList *wp, *dp;
	int xpos, ypos, width, height;

	fp = file_open_fp(prefs.sessionrc, "w", OPEN_EXCL,
			  "win_session_file_write");
	if (fp == NULL)
		return;

	for (wp = winlist; wp; wp = wp->next) {
		w = (win_t *)wp->data;
		gdk_window_get_position(w->toplev->window, &xpos, &ypos);
		gdk_window_get_size(w->toplev->window, &width, &height);
		fprintf(fp, "\t\t%d %d %d %d %d\n",
			w->numdoc, xpos, ypos, width, height);

		for (dp = w->doclist; dp; dp = dp->next) {
			d = (doc_t *)dp->data;

			gtk_events_flush();
			if (!d->fname) {
				fprintf(fp, "\n");
				continue;
			}

			if (my_path_is_absolute(d->fname))
				fprintf(fp, "%s\n", d->fname);
			else {
				fpath = file_full_pathname_make(d->fname);
				fprintf(fp, "%s\n", fpath);
				g_free(fpath);
			}
		}
	}

	file_close_fp(prefs.sessionrc, fp);
} /* win_session_file_write */


#ifdef USE_GNOME
/*
 * Callback for "save_yourself" signal, for GNOME session management.
 */
int
win_session_save(GnomeClient *client, int phase, GnomeSaveStyle save_style,
		 int is_shutdown, GnomeInteractStyle interact_style,
		 int is_fast, gpointer client_data)
{
	char **argv;
	guint argc;

	argc = 2;
	argv = g_malloc(sizeof(char *) * 3);
	argv[0] = client_data;
	argv[1] = "--ldsession";
	argv[2] = NULL;

	gnome_client_set_clone_command(client, argc, argv);
	gnome_client_set_restart_command(client, argc, argv);

	g_free(argv);

	win_session_file_write();

	return TRUE;
} /* win_session_save */


void
win_session_die(GnomeClient *client, gpointer client_data)
{
	gtk_main_quit();
}
#endif	/* USE_GNOME */


static void
win_session_load_complete(void *wp)
{
	win_t *w = (win_t *)wp;

	doc_load(DOC_CURRENT(w));
	win_set_title(DOC_CURRENT(w));

	/* see app_init() as to why we connect this signal here */
	(void)gtk_signal_connect_after(GTK_OBJECT(w->nb), "switch_page",
				       GTK_SIGNAL_FUNC(doc_switch_page), w);
	gtk_widget_show(w->toplev);
} /* win_session_load_complete */


/*
 * returns TRUE upon successful loading, FALSE if not.
 */
bool_t
win_session_load(void)
{
	char buf[MAXPATH + 1];
	int added, num, xpos, ypos, width, height;
	FILE *fp;
	win_t *w;

	if (!prefs.sessionrc || !file_exist(prefs.sessionrc))
		return FALSE;

	fp = file_open_fp(prefs.sessionrc, "r", 0, "win_session_load");
	if (fp == NULL)
		return FALSE;

	w = NULL;
	num = 0;
	added = 0;
	while (fgets(buf, MAXPATH + 1, fp)) {
		if (buf[0] == '\t' && buf[1] == '\t') {
			w = win_new();
			sscanf(buf, "%d %d %d %d %d",
			       &num, &xpos, &ypos, &width, &height);
			gtk_widget_set_uposition(w->toplev, xpos, ypos);
#ifdef GTK_HAVE_FEATURES_1_1_0
			gtk_window_set_default_size(GTK_WINDOW(w->toplev),
						    width, height);
#else
			gtk_widget_set_usize(GTK_WIDGET(w->toplev),
					     width, height);
#endif
		} else if (buf[0] != '\n') {
			g_assert(w);
			g_assert(num >= 0);
			num--;
			if (buf[strlen(buf) - 1] == '\n')
				buf[strlen(buf) - 1] = '\0';

			if (!file_exist(buf))
				continue;

			added++;
			doc_new(w, buf, DocText, num == 0 ? DO_LOAD : 0);
		} else if (buf[0] == '\n') {	/* empty line (no document) */
			g_assert(w);
			g_assert(num >= 0);
			num--;
			added++;
			doc_new(w, NULL, DocText, 0);
		}
	}

	file_close_fp(prefs.sessionrc, fp);

	win_foreach(win_session_load_complete);

	return TRUE;
} /* win_session_load */
#endif	/* WANT_SESSION */


/* the end */
