#include <stdlib.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <string.h>
#include <gtk/gtk.h>

#ifdef HAVE_NETAX25_AX25_H
#include <netax25/ax25.h>
#else
#include <netax25/kernel_ax25.h>
#endif
#ifdef HAVE_NETROM_NETROM_H
#include <netrom/netrom.h>
#else
#include <netax25/kernel_netrom.h>
#endif
#ifdef HAVE_NETROSE_ROSE_H
#include <netrose/rose.h>
#else
#include <netax25/kernel_rose.h>
#endif

#include <netax25/axlib.h>
#include <netax25/axconfig.h>

#include "config.h"
#include "user_io.h"
#include "support.h"

gint s, addrlen = 0, ports = -1, lax25;
gboolean connected;
gchar *callsign;
gchar *port = NULL;
gchar buffer[256];
gint rxmonitor;
struct full_sockaddr_ax25 axconnect;


void write_text (gchar *msg, gint type) {
  GtkWidget *maintext;
  extern GtkWidget *window1;
  GdkFont *textfont;
  extern GdkColor yellow, white, red, green;
  
  maintext = lookup_widget(window1, "text1");
  textfont = gdk_font_load("8x13bold");
  
  switch(type) {
    case RXMESSAGE:
      gtk_text_insert(GTK_TEXT(maintext), textfont, &white, NULL, msg, -1);
      break;
    case TXMESSAGE:
      gtk_text_insert(GTK_TEXT(maintext), textfont, &yellow, NULL, msg, -1);
      break;
    case STATUSMESSAGE:
      gtk_text_insert(GTK_TEXT(maintext), textfont, &green, NULL, msg, -1);
      break;
    case ERRORMESSAGE:
      gtk_text_insert(GTK_TEXT(maintext), textfont, &red, NULL, msg, -1);
      break;
  }
}


void rx(gpointer data, gint source, GdkInputCondition condition) {
  gchar rxbuf[8192];
  gint numbytes;
  extern gint s;

  memset(rxbuf, 0, 8192*sizeof(gchar));
  numbytes = read(s, rxbuf, 4096);
  if (numbytes <=0) disconn();
  g_strdelimit(rxbuf, "\r", '\n');
  write_text(rxbuf, RXMESSAGE);
}


void tx(GString *message) {
  extern gint s;
  GString *sendtext, *maintext;

  sendtext = g_string_new(message->str);
  maintext = g_string_new(message->str);
  g_string_append(sendtext, "\r");
  g_string_append(maintext, "\n");
  write(s, sendtext->str, sendtext->len);
  write_text(maintext->str, TXMESSAGE);
  g_string_free(sendtext, TRUE);
  g_string_free(maintext, TRUE);
}


void alarm_handler(gint sig) {
  if (!connected) disconn();
}

void connectto(GString *where) {
  GString *tmp, *address, *message;
  size_t len;
  gchar **list = NULL;

  if (ports == -1) {
    if ((ports = ax25_config_load_ports()) == 0) {
      message = g_string_new(_("ERROR: Problem with axports file\n"));
      write_text(message->str, ERRORMESSAGE);
      return;
    }
  }

  if ((s = socket(AF_AX25, SOCK_SEQPACKET, PF_AX25)) < 0) {
    message = g_string_new(_("ERROR: Cannot open AX.25 socket; "));
    g_string_append(message, strerror(errno));
    g_string_append(message, "\n");
    write_text(message->str, ERRORMESSAGE);
    return;
  }
  
  tmp = g_string_new(g_strchug(where->str));
  list = g_strsplit(tmp->str, " ", 0);
  
  if (!list[1] || where->len <= 1) {
    message = g_string_new(_("ERROR: Invalid number of arguments\n"));
    write_text(message->str, ERRORMESSAGE);
    g_strfreev(list);
    return;
  }
  
  port = g_strdup(list[0]);
  g_strfreev(list);
  
  if (!callsign) {
    if ((callsign = ax25_config_get_addr(port)) == NULL) {
      message = g_string_new(_("ERROR: Invalid port\n"));
      write_text(message->str, ERRORMESSAGE);
      return;
    }
  }
  
  len = strlen(port);
  address = g_string_erase(tmp, 0, len + 1);
  list = g_strsplit(address->str, " ", 0);

  if ((lax25 = ax25_aton(callsign, &axconnect)) == -1) {
    message = g_string_new(_("ERROR: Cannot convert callsign to network address\n"));
    write_text(message->str, ERRORMESSAGE);
    return; 
  }
  
  axconnect.fsa_ax25.sax25_family = AF_AX25;
  addrlen = sizeof(struct full_sockaddr_ax25);

  if (bind(s, (struct sockaddr *) &axconnect, addrlen) != 0) {
    message = g_string_new(_("ERROR: Cannot bind AX.25 socket; "));
    g_string_append(message, strerror(errno));
    g_string_append(message, "\n");
    write_text(message->str, ERRORMESSAGE);
    return;
  }
  
  if (ax25_aton_arglist((const char**)list, &axconnect) == -1) {
    message = g_string_new(_("ERROR: Invalid destination callsign or digipeater\n"));
    write_text(message->str, ERRORMESSAGE);
    return;
  }

  message = g_string_new(_("--> Connecting...\n"));
  write_text(message->str, STATUSMESSAGE);

  alarm(30);
  signal(SIGALRM, alarm_handler);

  if (connect(s, (struct sockaddr *)&axconnect, addrlen) != 0) {
    switch (errno) {
      case ECONNREFUSED:
        message = g_string_new(_("*** Connection refused - aborting\n"));
        break;
      case ENETUNREACH:
        message = g_string_new(_("*** No known route - aborting\n"));
        break;
      case EINTR:
        message = g_string_new(_("*** Connection timed out - aborting\n"));
        break;
      default:
        message = g_string_new(_("*** Cannot connect to "));
        g_string_append(message, list[0]);
        g_string_append(message, "\n");
        break;
    }
  write_text(message->str, ERRORMESSAGE);
  g_strfreev(list);
  return;
  }

  fcntl(s, F_SETFL, O_NONBLOCK);
  rxmonitor = gdk_input_add(s, GDK_INPUT_READ, GTK_SIGNAL_FUNC(rx), NULL);

  message = g_string_new(_("--> Connected to "));
  g_string_append(message, list[0]);
  g_string_append(message, "\n");
  write_text(message->str, STATUSMESSAGE);
  connected = TRUE;
  g_strfreev(list);
  g_string_free(message,TRUE);
}


void disconn() {
  GString *message;
  
  close(s);
  message = g_string_new(_("--> Disconnected\n"));
  write_text(message->str, STATUSMESSAGE);
  connected = FALSE;
  gdk_input_remove(rxmonitor);
  g_string_free(message, TRUE);
}