/*
Donut Bump Mapping Demo
This demo shows how to use a bump mapping technique using Glide(tm)
Copyright (C) 1999  3Dfx Interactive, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "basics.h"
#include "util.h"
#include "mathutil.h"
#include "texcache.h"
#include "crystal.h"
#include "bspline.h"
#include "lighting.h"
#include "camera.h"
#include "Printerr.h"
#include "clip.h"
#include "tlib.h"
#include "scrncptr.h"
#include "linux_utils.h"

#define MOUSE_SENSITIVITY (0.5f)

// fog stuff
#define FOG_DENSITY  (1.6f)
#define FOG_BEGIN  40
#define FOG_END  47

// mouse state
FxU8 gMouseState;
POINT gMousePos1, gMousePos2;
#define LEFT_MOUSE_BUTTON_DOWN   0x1
#define MIDDLE_MOUSE_BUTOTN_DOWN 0x2
#define RIGHT_MOUSE_BUTTON_DOWN  0x4

// keyboard state
#define VK_MINUS  0xbd // -
#define VK_EQUAL  0xbb // =
FxU32 gKeyboardState;
#define SHIFT_KEY_DOWN       0x1
#define CONTROL_KEY_DOWN     0x2
#define LEFT_ARROW_KEY_DOWN  0x4
#define RIGHT_ARROW_KEY_DOWN 0x8
#define DOWN_ARROW_KEY_DOWN  0x10
#define UP_ARROW_KEY_DOWN    0x20

// global variables
int SCREEN_RES, SCREEN_REFRESH;
float SCREEN_RES_X, SCREEN_RES_Y;
int HI_SCREEN_RES, HI_SCREEN_RES_X, HI_SCREEN_RES_Y;
int LO_SCREEN_RES, LO_SCREEN_RES_X, LO_SCREEN_RES_Y;

#ifdef USE_GLIDE3
GrContext_t gContext;
#endif // USE_GLIDE3

int gNumTMUsUsed;
int gSubdivs, gDynamicLod, gWireframe, gSpecular, gEnvmap, gEditing, gDrawNormals, gVsynch, gFillinGaps, gPause, gFog, gUseCtrlNorms, gForwardDifferencing;
Matrix gProjMat;
CrystalBall gObjectCrystalBall, gLightCrystalBall, *gCurrCrystalBall;
FxBool gHelpMenu;
Vector gLightDir, gCameraDir;
Vector *gSelectedCtrlPt;
Camera gCamera;
float gNearPlane, gFarPlane;
int gNumScreenShots, gNumScreensToCapture, gScreenCaptureEnabled;
Vector gWorldScale;
float gCameraSpeed;
static int g_bExitApp = FALSE;

// lighting
float gLightAmbientFactor =   32.0f; // (ambient + diffuse) must be <= 255
float gLightDiffuseFactor =  192.0f;
float gLightSpecularFactor = 128.0f;

// local function prototypes
FxBool WinOpenHighestResolution(int *resolution, int *res_x, int *res_y);
void Render( void );
void InitGlideState( void );
void SetupCrystalBallRotation(CrystalBall *crystal_ball, POINT pos1, POINT pos2);
void linux_UIMain(XEvent report);

int main(int argc, char *argv[])
{
	int i;
	FxBool set_resolution;
	char str[256];

	set_resolution = FXFALSE;

	gNumScreenShots = 0;
	gNumScreensToCapture = 0;

	SCREEN_REFRESH = GR_REFRESH_NONE;
	gScreenCaptureEnabled = 0;

	gVsynch = 1;
	gHelpMenu = FXFALSE;
	gSubdivs = 6;
	gDynamicLod = 1;
	gWireframe = 0;
	gSpecular = 0;
	gEnvmap = 0;
	gDrawNormals = 0;
	gEditing = 0;
	gFillinGaps = 1;
	gPause = 0;
	gFog = 1;
	gUseCtrlNorms = 1;
	gForwardDifferencing = 0;

	gCamera.u[0] = 1.0f;
	gCamera.u[1] = 0.0f;
	gCamera.u[2] = 0.0f;

	gCamera.v[0] = 0.0f;
	gCamera.v[1] = 0.0f;
	gCamera.v[2] = 1.0f;

	gCamera.n[0] = 0.0f;
	gCamera.n[1] = -1.0f;
	gCamera.n[2] = 0.0f;

	gCamera.pos[X] = 0.0f;
	gCamera.pos[Y] = 0.0f;
	gCamera.pos[Z] = 10.0f;

	gCameraSpeed = 1.0f;

	gLightDir[X] = 1.0f;
	gLightDir[Y] = 1.0f;
	gLightDir[Z] = 1.0f;
	gLightDir[W] = 0.0f;
	gCameraDir[X] = 0.0f;
	gCameraDir[Y] = 0.0f;
	gCameraDir[Z] = 1.0f;
	gCameraDir[W] = 0.0f;

	gNearPlane = 1.0f;
	gFarPlane = 1000.0f;

	gWorldScale[X] = 20.0f;
	gWorldScale[Y] = 20.0f;
	gWorldScale[Z] = 10.0f;
		// parse the arguments
	for (i=1; i<argc; i++)
	{
		if (strcasecmp(argv[i], "-res") == 0)
		{
			if (argc <= i+1)
			{
				printf("resolution not specified");
				return 0;
			}

			i++;

			set_resolution = FXTRUE;

			if (strcmp(argv[i], "1600x1200") == 0)
			{
				HI_SCREEN_RES = GR_RESOLUTION_1600x1200;
				HI_SCREEN_RES_X = 1600;
				HI_SCREEN_RES_Y = 1200;
			}
			else if (strcmp(argv[i], "1280x1024") == 0)
			{
				HI_SCREEN_RES = GR_RESOLUTION_1280x1024;
				HI_SCREEN_RES_X = 1280;
				HI_SCREEN_RES_Y = 1024;
			}
			else if (strcmp(argv[i], "1024x768") == 0)
			{
				HI_SCREEN_RES = GR_RESOLUTION_1024x768;
				HI_SCREEN_RES_X = 1024;
				HI_SCREEN_RES_Y = 768;
			}
			else if (strcmp(argv[i], "800x600") == 0)
			{
				HI_SCREEN_RES = GR_RESOLUTION_800x600;
				HI_SCREEN_RES_X = 800;
				HI_SCREEN_RES_Y = 600;
			}
			else if (strcmp(argv[i], "640x480") == 0)
			{
				HI_SCREEN_RES = GR_RESOLUTION_640x480;
				HI_SCREEN_RES_X = 640;
				HI_SCREEN_RES_Y = 480;
			}
			else if (strcmp(argv[i], "512x384") == 0)
			{
				HI_SCREEN_RES = GR_RESOLUTION_512x384;
				HI_SCREEN_RES_X = 512;
				HI_SCREEN_RES_Y = 384;
			}
			else
			{
				sprintf(str, "%s is not a valid resolution.\nValid options:\n512x384\n640x480\n800x600\n1024x768\n1280x1024\n1600x1200\n", argv[i]);
				printf( str );
				return 0;
			}
		}
		else if (strcasecmp(argv[i], "-ref") == 0)
		{
			if (argc <= i+1)
			{
				printf("refresh rate not specified");
				return 0;
			}

			i++;

			if (strcmp(argv[i], "60") == 0)
			{
				SCREEN_REFRESH = GR_REFRESH_60Hz;
			}
			else if (strcmp(argv[i], "70") == 0)
			{
				SCREEN_REFRESH = GR_REFRESH_70Hz;
			}
			else if (strcmp(argv[i], "72") == 0)
			{
				SCREEN_REFRESH = GR_REFRESH_72Hz;
			}
			else if (strcmp(argv[i], "75") == 0)
			{
				SCREEN_REFRESH = GR_REFRESH_75Hz;
			}
			else if (strcmp(argv[i], "80") == 0)
			{
				SCREEN_REFRESH = GR_REFRESH_80Hz;
			}
			else if (strcmp(argv[i], "90") == 0)
			{
				SCREEN_REFRESH = GR_REFRESH_90Hz;
			}
			else if (strcmp(argv[i], "100") == 0)
			{
				SCREEN_REFRESH = GR_REFRESH_100Hz;
			}
			else if (strcmp(argv[i], "85") == 0)
			{
				SCREEN_REFRESH = GR_REFRESH_85Hz;
			}
			else if (strcmp(argv[i], "120") == 0)
			{
				SCREEN_REFRESH = GR_REFRESH_120Hz;
			}
			else
			{
				sprintf( str, "%s is not a valid refresh rate.\nValid options:\n60\n70\n72\n75\n80\n85\n90\n100\n120\n", argv[i] );
				printf( str );
				return 0;
			}
		}
		else if (strcasecmp(argv[i], "-screencapture") == 0)
		{
			if (argc <= i+1)
			{
				printf( "Number of screen captures not specified.\n  -screencapture #\nWhere # is a positive integer.\n");
				return 0;
			}
			i++;
			gScreenCaptureEnabled = atoi(argv[i]);
		}
	}

	if (!GlideInitialize())
	{
		printf( "Could not initialize Glide.  Application Aborted.\n" );
		return -1;
	}

	gNumTMUsUsed = GetNumTMUs();

	// HACK: some fuckin' Voodoo Graphics cards don't support
	// GR_REFRESH_NONE, and just hang, so just make it 60Hz
	if (GetNumTMUs() == 1 && SCREEN_REFRESH == GR_REFRESH_NONE)
	{
		SCREEN_REFRESH = GR_REFRESH_60Hz;
	}

	// this is the lowest resolution I'll support
	LO_SCREEN_RES = GR_RESOLUTION_640x480;
	LO_SCREEN_RES_X = 640;
	LO_SCREEN_RES_Y = 480;

	if (set_resolution)
	{
#ifdef USE_GLIDE3
		// try to get a triple-buffer with depth-buffer frame buffer
		gContext = grSstWinOpen(0, HI_SCREEN_RES, SCREEN_REFRESH, GR_COLORFORMAT_ABGR, GR_ORIGIN_UPPER_LEFT, GetNumColorBuffers(), 1);
		if (!gContext)
		{
			// now try to get a double-buffer with depth-buffer frame buffer
			gContext = grSstWinOpen(0, HI_SCREEN_RES, SCREEN_REFRESH, GR_COLORFORMAT_ABGR, GR_ORIGIN_UPPER_LEFT, 2, 1);
			if (!gContext)
#else // USE_GLIDE3

		// try to get a triple-buffer with depth-buffer frame buffer
		if (!grSstWinOpen(0, HI_SCREEN_RES, SCREEN_REFRESH, GR_COLORFORMAT_ABGR, GR_ORIGIN_UPPER_LEFT, GetNumColorBuffers(), 1))
		{
			// now try to get a double-buffer with depth-buffer frame buffer
			if (!grSstWinOpen(0, HI_SCREEN_RES, SCREEN_REFRESH, GR_COLORFORMAT_ABGR, GR_ORIGIN_UPPER_LEFT, 2, 1))
#endif // USE_GLIDE3

			{
				GlideCleanup();

				sprintf(str, "Could not open display (%dx%d)  Application Aborted.\n", (int)HI_SCREEN_RES_X, (int)HI_SCREEN_RES_Y);
				printf( str );
				return -1;
			}
		}
	}
	// find out what's the highest resolution I can get (at least 640x480)
	else if (!WinOpenHighestResolution(&HI_SCREEN_RES, &HI_SCREEN_RES_X, &HI_SCREEN_RES_Y))
	{
		GlideCleanup();
		printf( "Could not open display (640x480)  Application Aborted.\n" );
		return -1;
	}
   
	SCREEN_RES = HI_SCREEN_RES;
	SCREEN_RES_X = (float)HI_SCREEN_RES_X;
	SCREEN_RES_Y = (float)HI_SCREEN_RES_Y;

	linux_InitializeXWindow(SCREEN_RES_X, SCREEN_RES_Y);
	tlConSet(0.0f, 0.0f, 1.0f, 1.0f, SCREEN_RES_X, SCREEN_RES_Y, 80, 40, 0Xff00007f);

	if (gScreenCaptureEnabled)
	{
		if (!SetScreenCaptureRect(0, 0, (int)SCREEN_RES_X, (int)SCREEN_RES_Y))
		{
#ifdef USE_GLIDE3
			grSstWinClose(gContext);
#else
			grSstWinClose();
#endif // USE_GLIDE3
			GlideCleanup();
			printf( "Could not setup screen capture rectangle.  Application Aborted.\n" );
			return -1;
		}
	}

	// initialize random number generator //
	srand(linux_timeGetTime()); 

	// set up glide state
	InitGlideState();


	// We've gotten this far, which means we were able in intialize Glide
	// and set the proper resolution
	// Let's start the demo fun ! ! !

	// LINUX: need to make UI change here
	SetCurrUIFunction(linux_UIMain);

	/*
	grTexClampMode(GR_TMU0, GR_TEXTURECLAMP_CLAMP, GR_TEXTURECLAMP_CLAMP);
	if (GetNumTMUs() >= 2)
	{
		grTexClampMode(GR_TMU1, GR_TEXTURECLAMP_CLAMP, GR_TEXTURECLAMP_CLAMP);
	}
	*/
	gMouseState = 0;
	gKeyboardState = 0;

	InitCrystalBall(&gObjectCrystalBall);
//	gObjectCrystalBall.rotx = DEG_TO_RAD(30.0f);
//	gObjectCrystalBall.roty = DEG_TO_RAD(10.0f);
//	gObjectCrystalBall.rot_axis = X;
	InitCrystalBall(&gLightCrystalBall);
	gCurrCrystalBall = &gObjectCrystalBall;

	if (!InitSurface()) {
#ifdef USE_GLIDE3
		grSstWinClose(gContext);
#else
		grSstWinClose();
#endif // USE_GLIDE3
		GlideCleanup();
		linux_ShutdownXWindow();
		printf( "InitSurface() failed.  Application Aborted.\n" );
		return -1;
	}

	// ** main event and rendering loop **
	while (!g_bExitApp) {

		linux_EventLoop();

		Render();

	}


#ifdef USE_GLIDE3
	grSstWinClose(gContext);
#else
	grSstWinClose();
#endif // USE_GLIDE3


	if (gScreenCaptureEnabled)
	{
		CleanupScreenCapture();
	}

	CleanupSurface();

	GlideCleanup();

	linux_ShutdownXWindow();

	return 0;
}

FxBool WinOpenHighestResolution(int *resolution, int *res_x, int *res_y)
{
	struct
	{
		int resolution;
		int res_x;
		int res_y;
	} res_info[3] = {
		{GR_RESOLUTION_1024x768,  1024,  768},
		{GR_RESOLUTION_800x600,    800,  600},
		{GR_RESOLUTION_640x480,    640,  480}
	};
	int i;

	// HACK: some Voodoo1s hang if I call grSstWinOpen with a resolution of 1024x768
	// so I'll just skip it for all Voodoo1s
	if (GetNumTMUs() == 1)
	{
		i = 1;
	}
	else
	{
		i = 0;
	}
	for (; i<3; i++)
	{
		// try to get a triple-buffer with depth-buffer frame buffer
		if (grSstWinOpen(0, res_info[i].resolution, SCREEN_REFRESH, GR_COLORFORMAT_ABGR, GR_ORIGIN_UPPER_LEFT, GetNumColorBuffers(), 1))
		{
			*resolution = res_info[i].resolution;
			*res_x = res_info[i].res_x;
			*res_y = res_info[i].res_y;
			return FXTRUE;
		}
		// now try to get a double-buffer with depth-buffer frame buffer
		else if (grSstWinOpen(0, res_info[i].resolution, SCREEN_REFRESH, GR_COLORFORMAT_ABGR, GR_ORIGIN_UPPER_LEFT, 2, 1))
		{
			*resolution = res_info[i].resolution;
			*res_x = res_info[i].res_x;
			*res_y = res_info[i].res_y;
			return FXTRUE;
		}
	}

	return FXFALSE;
}

void Render()
{
	/*
	static float fog_density = FOG_DENSITY;
	static int fog_begin = FOG_BEGIN;
	static int fog_end = FOG_END;
	FxU8 fog_table[64];
	int i;

	if ((GetAsyncKeyState(VK_F2) & 0x8001) == 0x8001)
	{
		if (gKeyboardState & SHIFT_KEY_DOWN)
		{
			fog_begin = MAX(0, fog_begin-1);
		}
		else
		{
			fog_begin = MIN(255, fog_begin+1);
		}
	}
	if ((GetAsyncKeyState(VK_F3) & 0x8001) == 0x8001)
	{
		if (gKeyboardState & SHIFT_KEY_DOWN)
		{
			fog_end = MAX(0, fog_end-1);
		}
		else
		{
			fog_end = MIN(63, fog_end+1);
		}
	}
	if ((GetAsyncKeyState(VK_F4) & 0x8001) == 0x8001)
	{
		if (gKeyboardState & SHIFT_KEY_DOWN)
		{
			fog_density = MAX(0.01f, fog_density-0.01f);
		}
		else
		{
			fog_density = MIN(10.0f, fog_density+0.01f);
		}
	}

	for (i=0; i<64; i++)
	{
		if (i <= fog_begin)
		{
			fog_table[i] = 0;
		}
		else if (i >= fog_end)
		{
			fog_table[i] = 255;
		}
		else
		{
			fog_table[i] = (FxU8)(255.0*pow((float)(i - fog_begin)/(float)(fog_end - fog_begin), fog_density));
		}
	}
	grFogTable(fog_table);
	*/

	static float fps = 0.0f;
	static int frames = 0;
	static int total_time = 0;
	int frame_time;
	int num_polys;
	Matrix camera_mat, viewport_mat, frustum_mat, scale_mat;
	Vector light_dir;
	float dx, dy, mag;

	frame_time = linux_timeGetTime();

	// clears (color-bgr, alpha, depth)
	grBufferClear(0, 0, 0xffff);

	/*
	if (gEditing)
	{
		if (gSelectedCtrlPt)
		{
			if (gMouseState & LEFT_MOUSE_BUTTON_DOWN)
			{
				(*gSelectedCtrlPt)[X] -= 0.01f*(gMousePos2.x - gMousePos1.x);
				(*gSelectedCtrlPt)[Y] += 0.01f*(gMousePos2.y - gMousePos1.y);
			}
			else if (gMouseState & RIGHT_MOUSE_BUTTON_DOWN)
			{
				(*gSelectedCtrlPt)[Z] += 0.01f*(gMousePos2.y - gMousePos1.y);
			}
		}
	}
	else
	{
		if (gMouseState & LEFT_MOUSE_BUTTON_DOWN)
		{
			SetupCrystalBallRotation(gCurrCrystalBall, gMousePos1, gMousePos2);
		}
		if (gMouseState & RIGHT_MOUSE_BUTTON_DOWN)
		{
			if (gCurrCrystalBall == &gObjectCrystalBall)
			{
				gObjectCrystalBall.scale = CLAMP(gObjectCrystalBall.scale*(1.0f+0.025f*(gMousePos2.y - gMousePos1.y)), 0.025f, 2.0f);
			}
			else
			{
				gLightSpecularFactor = CLAMP(gLightSpecularFactor + 0.5f*(gMousePos2.y - gMousePos1.y), 0.0f, 255.0f);
				gLightDiffuseFactor = MIN(255.0f-gLightAmbientFactor, 1.5f*gLightSpecularFactor);
			}
		}
	}
	*/

	if (gKeyboardState & LEFT_ARROW_KEY_DOWN)
	{
		dx = gCamera.u[X];
		dy = gCamera.u[Y];
		mag = fsqrt(dx*dx + dy*dy);
		mag = (mag != 0) ? 1.0f/mag : 0.0f;
		dx *= mag;
		dy *= mag;
		gCamera.pos[X] -= gCameraSpeed*dx;
		gCamera.pos[Y] -= gCameraSpeed*dy;
//		gCamera.pos[Z] -= gCameraSpeed*gCamera.u[Z];
	}
	if (gKeyboardState & RIGHT_ARROW_KEY_DOWN)
	{
		dx = gCamera.u[X];
		dy = gCamera.u[Y];
		mag = fsqrt(dx*dx + dy*dy);
		mag = (mag != 0) ? 1.0f/mag : 0.0f;
		dx *= mag;
		dy *= mag;
		gCamera.pos[X] += gCameraSpeed*dx;
		gCamera.pos[Y] += gCameraSpeed*dy;
//		gCamera.pos[Z] += gCameraSpeed*gCamera.u[Z];
	}
	if (gKeyboardState & DOWN_ARROW_KEY_DOWN)
	{
		if (gKeyboardState & SHIFT_KEY_DOWN)
		{
//			gCamera.pos[X] -= gCameraSpeed*gCamera.v[X];
//			gCamera.pos[Y] -= gCameraSpeed*gCamera.v[Y];
			gCamera.pos[Z] -= gCameraSpeed;//*gCamera.v[Z];
		}
		else
		{
			dx = gCamera.n[X];
			dy = gCamera.n[Y];
			mag = fsqrt(dx*dx + dy*dy);
			mag = (mag != 0) ? 1.0f/mag : 0.0f;
			dx *= mag;
			dy *= mag;
			gCamera.pos[X] += gCameraSpeed*dx;
			gCamera.pos[Y] += gCameraSpeed*dy;
//			gCamera.pos[Z] += gCameraSpeed*gCamera.n[Z];
		}
	}
	if (gKeyboardState & UP_ARROW_KEY_DOWN)
	{
		if (gKeyboardState & SHIFT_KEY_DOWN)
		{
//			gCamera.pos[X] += gCameraSpeed*gCamera.v[X];
//			gCamera.pos[Y] += gCameraSpeed*gCamera.v[Y];
			gCamera.pos[Z] += gCameraSpeed;//*gCamera.v[Z];
		}
		else
		{
			dx = gCamera.n[X];
			dy = gCamera.n[Y];
			mag = fsqrt(dx*dx + dy*dy);
			mag = (mag != 0) ? 1.0f/mag : 0.0f;
			dx *= mag;
			dy *= mag;
			gCamera.pos[X] -= gCameraSpeed*dx;
			gCamera.pos[Y] -= gCameraSpeed*dy;
//			gCamera.pos[Z] -= gCameraSpeed*gCamera.n[Z];
		}
	}

	gMousePos2.x = gMousePos1.x;
	gMousePos2.y = gMousePos1.y;
	gMousePos1.x = (int)(0.5f*SCREEN_RES_X);
	gMousePos1.y = (int)(0.5f*SCREEN_RES_Y);
	linux_SetCursorPos(gMousePos1.x, gMousePos1.y);

	Vector rot_axis = {0.0f, 0.0f, 1.0f, 0.0f};
	float angle;

	// yaw
	angle = DEG_TO_RAD(0.1f*(gMousePos1.x - gMousePos2.x));
	RotateByAxis(gCamera.u, gCamera.u, angle, rot_axis);
	RotateByAxis(gCamera.v, gCamera.v, angle, rot_axis);
	RotateByAxis(gCamera.n, gCamera.n, angle, rot_axis);

	// pitch
	angle = DEG_TO_RAD(0.1f*(gMousePos1.y - gMousePos2.y));
	RotateByAxis(gCamera.v, gCamera.v, angle, gCamera.u);
	RotateByAxis(gCamera.n, gCamera.n, angle, gCamera.u);

	gCameraDir[X] = gCamera.n[X];
	gCameraDir[Y] = gCamera.n[Y];
	gCameraDir[Z] = gCamera.n[Z];

	// set up the crystal ball matrix
	UpdateCrystalBall(&gObjectCrystalBall);

	UpdateCrystalBall(&gLightCrystalBall);
	MatMultVec3x4_3(light_dir, gLightCrystalBall.rot_mat, gLightDir);

	// transform the directional light by applying
	// the opposite of the overall object rotation
	DirLightXforms(gObjectCrystalBall.rot_mat, light_dir, gCameraDir);

	// set up the projection matrix
//	float cam_height = HeightAt(&gSurfaces[0], gCamera.pos[X], gCamera.pos[Y]);
//	gCamera.pos[Z] += cam_height;
	CameraMat(camera_mat, &gCamera);
//	gCamera.pos[Z] -= cam_height;

	ViewportMat(viewport_mat, 0.0f, SCREEN_RES_X, 0.0f, SCREEN_RES_Y, 1.0f, 65535.0f);
	SetClipVolume(0.0f, SCREEN_RES_X, 0.0f, SCREEN_RES_Y, gNearPlane, gFarPlane);

	PerspectiveMat(frustum_mat, DEG_TO_RAD(60.0f), SCREEN_RES_X/SCREEN_RES_Y, gNearPlane, gFarPlane);
	MatMultMat4x4(gProjMat, viewport_mat, frustum_mat);

	MatMultMat4x4(gProjMat, gProjMat, camera_mat);

	// put the transformations and projection into one matrix - gProjMat
//	MatMultMat4x4(gProjMat, gProjMat, gObjectCrystalBall.rot_mat);
//	ScaleMat(scale_mat, gObjectCrystalBall.scale, gObjectCrystalBall.scale, gObjectCrystalBall.scale);
	ScaleMat(scale_mat, gWorldScale[X], gWorldScale[Y], gWorldScale[Z]);
	MatMultMat4x4(gProjMat, gProjMat, scale_mat);

	num_polys = RenderBSplineSurface();

	// capture screen before rendering console stuff
	if (gNumScreensToCapture)
	{
		char str[64];

		gNumScreensToCapture--;

		sprintf(str, "scrn%d.tga", gNumScreenShots);
		if (ScreenCaptureTGA(str))
		{
			tlConOutput("\nscreen captured...%d more to go...\n", gNumScreensToCapture);
			gNumScreenShots++;
		}
	}

	// output the console stuff
	tlConClear();
//	tlConOutput("clip: %.2f - %.2f\n", gNearPlane, gFarPlane);
//	tlConSetColor(0xff00007f);
	tlConOutput("fps: %.2f, ppf: %6d, pps: %6d [%d x %d vsynch: %s]\n",
							fps, num_polys, (int)(fps*num_polys),
							(int)SCREEN_RES_X, (int)SCREEN_RES_Y, gVsynch ? " on" : "off");
//	tlConOutput("fog_density: %.2f, fog_begin: %d, fog_end: %d\n", fog_density, fog_begin, fog_end);
	tlConRender();
	if (gHelpMenu)
	{
		tlConClear();
		tlConOutput("\n\n\n"
								"mouse movement: rotate camera\n"
								"arrow keys: move camera\n"
								"escape: quit program\n"
								"+/-: increase/decrease static patch subdivs [%d]\n"
								"C: toggle normals' source [%s]\n"
								"D: toggle control points\n"
								"E: toggle environment mapping\n"
								"G: gap fill mode [%s]\n"
								"H: toggle fog [%s]\n"
								"I: toggle forward differencing [%s]\n"
								"M: toggle resolution\n"
								"N: toggle drawing of normals\n"
								"P: pause/unpause waves\n"
								"T: toggle surface type [%s]\n"
								"V: toggle vsynch\n"
								"W: toggle wireframe\n"
								"X: toggle dynamic lod [%s]\n"
								"Y: toggle specular lighting [%s]\n",
								gSubdivs,
								gUseCtrlNorms ? "control normals" : "poly normals",
								(gFillinGaps == 0) ? "no fill" : ((gFillinGaps == 1) ? "fill" : "fill green"),
								gFog ? "on" : "off",
								gForwardDifferencing ? "on" : "off",
								gSurfaceType ? "b-spline" : "bezier",
								gDynamicLod ? "on" : "off",
								gSpecular ? "on" : "off");

		//tlConSetColor(0xff007f00);
		tlConRender();
	}

//	DrawCursor(gMousePos1.x, gMousePos1.y);
	grBufferSwap(gVsynch);

	total_time += linux_timeGetTime() - frame_time;
	frames++;

	// compute frame rate
	if (total_time >= 1000)
	{
		fps = 1000.0f*(float)frames/(float)total_time;
		frames = 0;
		total_time = 0;
	}
}

void InitGlideState()
{
	FxU8 fog_table[64];
	int i;

	grTexFilterMode(GR_TMU0, GR_TEXTUREFILTER_BILINEAR, GR_TEXTUREFILTER_BILINEAR);
	grTexLodBiasValue(GR_TMU0, 0.5f);
	grTexMipMapMode(GR_TMU0, GR_MIPMAP_NEAREST, FXFALSE);
	grTexClampMode(GR_TMU0, GR_TEXTURECLAMP_WRAP, GR_TEXTURECLAMP_WRAP);
	if (GetNumTMUs() >= 2)
	{
		grTexFilterMode(GR_TMU1, GR_TEXTUREFILTER_BILINEAR, GR_TEXTUREFILTER_BILINEAR);
		grTexLodBiasValue(GR_TMU1, 0.5f);
		grTexMipMapMode(GR_TMU1, GR_MIPMAP_NEAREST, FXFALSE);
		grTexClampMode(GR_TMU1, GR_TEXTURECLAMP_WRAP, GR_TEXTURECLAMP_WRAP);
	}
	grAlphaTestReferenceValue(0xff);
	grCullMode(GR_CULL_NEGATIVE);
	grDepthBufferMode(GR_DEPTHBUFFER_WBUFFER);
	grDepthBufferFunction(GR_CMP_LEQUAL);
	grDepthMask(FXTRUE);
	grAlphaTestFunction(GR_CMP_ALWAYS);
	grClipWindow(0, 0, (int)SCREEN_RES_X, (int)SCREEN_RES_Y);

	// let glide know that we will be using different S and Ts for the 2 TMUs
#ifdef USE_GLIDE3
	grCoordinateSpace(GR_WINDOW_COORDS);
	grVertexLayout(GR_PARAM_XY, offsetof(GrVertex, x), GR_PARAM_ENABLE);
	grVertexLayout(GR_PARAM_Q, offsetof(GrVertex, oow), GR_PARAM_ENABLE);
	grVertexLayout(GR_PARAM_RGB, offsetof(GrVertex, r), GR_PARAM_ENABLE);
	grVertexLayout(GR_PARAM_A, offsetof(GrVertex, a), GR_PARAM_ENABLE);
	grVertexLayout(GR_PARAM_ST0, offsetof(GrVertex, tmuvtx[0]), GR_PARAM_ENABLE);
#endif // USE_GLIDE3

	for (i=0; i<64; i++)
	{
		if (i <= FOG_BEGIN)
		{
			fog_table[i] = 0;
		}
		else if (i >= FOG_END)
		{
			fog_table[i] = 255;
		}
		else
		{
			fog_table[i] = (FxU8)(255.0*pow((float)(i - FOG_BEGIN)/(float)(FOG_END - FOG_BEGIN), FOG_DENSITY));
		}
	}

	grFogColorValue(0);
#ifdef USE_GLIDE3
	grFogMode(GR_FOG_WITH_TABLE_ON_W);
#else
	grFogMode(GR_FOG_WITH_TABLE);
#endif // USE_GLIDE3
	grFogTable(fog_table);
}

void SetupCrystalBallRotation(CrystalBall *crystal_ball, POINT pos1, POINT pos2)
{
	float crossz, mag_inv;

	if (gKeyboardState & SHIFT_KEY_DOWN)
	{
		// find the rotation of the mouse cursor about the z-axis
		// by computing the cross product of the mouse positions (treating
		// the center of the screen as the origin)
		crossz = ((pos1.x-(0.5f*SCREEN_RES_X))*(pos2.y-(0.5f*SCREEN_RES_Y)) -
							(pos1.y-(0.5f*SCREEN_RES_Y))*(pos2.x-(0.5f*SCREEN_RES_X)));
		// normalize the cross product
		mag_inv = fsqrt_inv(SQR(pos1.x-(0.5f*SCREEN_RES_X)) + SQR(pos1.y-(0.5f*SCREEN_RES_Y)));
		mag_inv *= fsqrt_inv(SQR(pos2.x-(0.5f*SCREEN_RES_X)) + SQR(pos2.y-(0.5f*SCREEN_RES_Y)));
		// apply the rotation of the mouse about the z-axis to the object
		crystal_ball->rotz = MOUSE_SENSITIVITY*100.0f*crossz*mag_inv;
		crystal_ball->rot_axis = Z;
	}
	else
	{
		crystal_ball->rotx = MOUSE_SENSITIVITY*(-0.5f)*(pos2.y - pos1.y);
		crystal_ball->roty = MOUSE_SENSITIVITY*(-0.5f)*(pos2.x - pos1.x);
		crystal_ball->rot_axis = X;
	}
}

void linux_UIMain(XEvent report)
{
	KeySym key;
	int i;
	TextureCache *tex_cache;

	switch(report.type)	{

		case KeyPress:
			key = XLookupKeysym(&report.xkey, 0);

			switch(key)	{
				case XK_Shift_L:
				case XK_Shift_R:
					gKeyboardState |= SHIFT_KEY_DOWN;
				break;

				case XK_Control_L:
				case XK_Control_R:
					gKeyboardState |= CONTROL_KEY_DOWN;
				break;

				case XK_Escape:
					g_bExitApp = TRUE;
				break;

				case XK_equal:
					gSubdivs = MIN(MAX_SUBDIVS, gSubdivs+1);
				break;

				case XK_minus:
					gSubdivs = MAX(1, gSubdivs-1);
				break;

				case XK_Left:
				case XK_KP_4:
				case XK_KP_Left:
					gKeyboardState |= LEFT_ARROW_KEY_DOWN;
				break;

				case XK_Right:
				case XK_KP_6:
				case XK_KP_Right:
					gKeyboardState |= RIGHT_ARROW_KEY_DOWN;
				break;

				case XK_KP_2:
				case XK_Down:
				case XK_KP_Down:
					gKeyboardState |= DOWN_ARROW_KEY_DOWN;
				break;

				case XK_KP_8:
				case XK_Up:
				case XK_KP_Up:
					gKeyboardState |= UP_ARROW_KEY_DOWN;
				break;
				case XK_Home:
					gCameraSpeed = MIN(1000.0f, gCameraSpeed*1.1f);
				break;

				case XK_End:
					gCameraSpeed = MAX(0.001f, gCameraSpeed*0.9f);
				break;


				case XK_B:
				case XK_b:
					if (gKeyboardState & SHIFT_KEY_DOWN) {
						gFarPlane = MAX(1.0f, gFarPlane - 100.0f);
					}
					else {
						gFarPlane = MIN(65535.0f, gFarPlane + 100.0f);
					}
				break;

				case XK_C:
				case XK_c:
					gUseCtrlNorms ^= 1;
				break;

				case XK_D:
				case XK_d:
					gEditing ^= 1;
				break;

				case XK_E:
				case XK_e:
					if (GetNumTMUs() > 1) {
						gEnvmap ^= 1;
					}
				break;

				case XK_F:
				case XK_f:
					if (gKeyboardState & SHIFT_KEY_DOWN) {
						gNearPlane = MAX(1.0f, gNearPlane - 1.0f);
					}
					else {
						gNearPlane = MIN(65535.0f, gNearPlane + 1.0f);
					}
					break;

				case XK_G:
				case XK_g:
					gFillinGaps = (gFillinGaps+1)%3;
				break;

				case XK_H:
				case XK_h:
					gFog ^= 1;
					if (gFog) {
#ifdef USE_GLIDE3
						grFogMode(GR_FOG_WITH_TABLE_ON_W);
#else
						grFogMode(GR_FOG_WITH_TABLE);
#endif // USE_GLUDE3
					}
					else {
						grFogMode(GR_FOG_DISABLE);
					}
				break;

				case XK_F1:
					gHelpMenu ^= 1;
				break;

				case XK_I:
				case XK_i:
					gForwardDifferencing ^= 1;
				break;

				case XK_L:
				case XK_l:
					LoadSurfaces("data/surf.txt");
				break;


				case XK_M:  /* resolution mode change */
				case XK_m:

					if (SCREEN_RES == LO_SCREEN_RES)
					{
						SCREEN_RES = HI_SCREEN_RES;
						SCREEN_RES_X = (float)HI_SCREEN_RES_X;
						SCREEN_RES_Y = (float)HI_SCREEN_RES_Y;
					}
					else
					{
						SCREEN_RES = LO_SCREEN_RES;
						SCREEN_RES_X = (float)LO_SCREEN_RES_X;
						SCREEN_RES_Y = (float)LO_SCREEN_RES_Y;
					}

					if (gScreenCaptureEnabled)
					{
						if (!SetScreenCaptureRect(0, 0, (int)SCREEN_RES_X, (int)SCREEN_RES_Y))
						{
#ifdef USE_GLIDE3
							grSstWinClose(gContext);
#else
							grSstWinClose();
#endif // USE_GLIDE3
							GlideCleanup();

							linux_ShutdownXWindow();
							exit(0xdeadbeef);
						}
					}

					// close glide window before reopening it
#ifdef USE_GLIDE3
					grSstWinClose(gContext);
#else
					grSstWinClose();
#endif // USE_GLIDE3
					// try to get a triple-buffer with depth-buffer frame buffer
					if (!grSstWinOpen(0, SCREEN_RES, SCREEN_REFRESH, GR_COLORFORMAT_ABGR, GR_ORIGIN_UPPER_LEFT, GetNumColorBuffers(), 1))
					{
						// now try to get a double-buffer with depth-buffer frame buffer
						if (!grSstWinOpen(0, SCREEN_RES, SCREEN_REFRESH, GR_COLORFORMAT_ABGR, GR_ORIGIN_UPPER_LEFT, 2, 1))
						{
							GlideCleanup();
							linux_ShutdownXWindow();
							exit(0xdeadbeef);
						}
					}

					InitGlideState();

					// InitGlideState will enable fogging, so
					// turn it off if it was off before
					if (!gFog)
					{
						grFogMode(GR_FOG_DISABLE);
					}

					// reset the console
					tlConSet(0.0f, 0.0f, 1.0f, 1.0f, SCREEN_RES_X, SCREEN_RES_Y, 80, 40, 0xff00007f);
					linux_ResizeXWindow(SCREEN_RES_X, SCREEN_RES_Y);
					ReloadBsplineTextures();
				break;

				case XK_N:
				case XK_n:
					gDrawNormals ^= 1;
				break;

				case XK_P:
				case XK_p:
					gPause ^= 1;				
				break;


				case XK_S:
				case XK_s:
					SaveSurfaces("data/surf.txt");
				break;

				case XK_T:
				case XK_t:
					gSurfaceType ^= 1;
				break;

				case XK_V:
				case XK_v:
					gVsynch ^= 1;
				break;

				case XK_W:
				case XK_w:
					gWireframe ^= 1;
				break;

				case XK_X:
				case XK_x:
					gDynamicLod ^= 1;
				break;

				case XK_Y:
				case XK_y:
					gSpecular ^= 1;
				break;

				case XK_Z:
				case XK_z:
					if (gKeyboardState & SHIFT_KEY_DOWN) {
						gWorldScale[X] *= 0.9f;
						gWorldScale[Y] *= 0.9f;
						gWorldScale[Z] *= 0.9f;
					}
					else {
						gWorldScale[X] *= 1.1f;
						gWorldScale[Y] *= 1.1f;
						gWorldScale[Z] *= 1.1f;
					}
				break;

				default: break;
			} // switch on key pressed //

		break;

		case KeyRelease:
			key = XLookupKeysym(&report.xkey, 0);

			switch(key)	{
				case XK_Return:
				case XK_KP_Enter:
					gNumScreensToCapture = gScreenCaptureEnabled;
				break;

				case XK_Shift_L:
				case XK_Shift_R:
					gKeyboardState &= ~SHIFT_KEY_DOWN;
				break;

				case XK_Control_L:
				case XK_Control_R:
					gKeyboardState &= ~CONTROL_KEY_DOWN;
				break;

				case XK_Left:
				case XK_KP_4:
				case XK_KP_Left:
					gKeyboardState &= ~LEFT_ARROW_KEY_DOWN;
				break;

				case XK_Right:
				case XK_KP_6:
				case XK_KP_Right:
					gKeyboardState &= ~RIGHT_ARROW_KEY_DOWN;
				break;

				case XK_KP_2:
				case XK_Down:
				case XK_KP_Down:
					gKeyboardState &= ~DOWN_ARROW_KEY_DOWN;
				break;

				case XK_KP_8:
				case XK_Up:
				case XK_KP_Up:
					gKeyboardState &= ~UP_ARROW_KEY_DOWN;
				break;


			} /* switch */

		break;

		case ButtonPress:

			switch(report.xbutton.button) {
				case (1): /* MOUSE_LEFT */
					gMouseState |= LEFT_MOUSE_BUTTON_DOWN;
					gMousePos1.x = report.xbutton.x;
					gMousePos1.y = report.xbutton.y;

					gMousePos2.x = gMousePos1.x;
					gMousePos2.y = gMousePos1.y;

					if (gEditing)
					{
						gSelectedCtrlPt = SelectControlPoint(gMousePos1.x, gMousePos1.y);
					}
				break;

				case (2): /* MOUSE_MIDDLE */
				break;

				case (3): /* MOUSE_RIGHT */ 
					gMouseState |= RIGHT_MOUSE_BUTTON_DOWN;
					gMousePos1.x = report.xbutton.x;
					gMousePos1.y = report.xbutton.y;

					gMousePos2.x = gMousePos1.x;
					gMousePos2.y = gMousePos1.y;

					if (gEditing)
					{
						gSelectedCtrlPt = SelectControlPoint(gMousePos1.x, gMousePos1.y);
					}
				break;

				default: break;
			} /* switch on which mouse button was pressed */

		break;

		case ButtonRelease:

			switch(report.xbutton.button) {

				case (1): /* MOUSE_LEFT */

					gMouseState &= ~LEFT_MOUSE_BUTTON_DOWN;
					gMousePos1.x = report.xbutton.x;
					gMousePos1.y = report.xbutton.y;

					if (ABS(gMousePos1.x - gMousePos2.x) + ABS(gMousePos1.y - gMousePos2.y) == 0)
					{
						gCurrCrystalBall->rotx = gCurrCrystalBall->roty = gCurrCrystalBall->rotz = 0.0f;
						gCurrCrystalBall->rot_axis = W;
					}
					else
					{
						SetupCrystalBallRotation(gCurrCrystalBall, gMousePos1, gMousePos2);
					}

				break;

				case (2): /* MOUSE_MIDDLE */
				break;

				case (3): /* MOUSE_RIGHT */

					gMouseState &= ~RIGHT_MOUSE_BUTTON_DOWN;
					gMousePos1.x = report.xbutton.x;
					gMousePos1.y = report.xbutton.y;

				break;
			}

		break;

		case MotionNotify:
			// handle movement of mouse//
			gMousePos1.x = report.xmotion.x;
			gMousePos1.y = report.xmotion.y;

		break;

		default: break;
	}
}

//////////////////////////////////////////////////////////////////////
