/*
 *  Copyright (C) 2002 Daniel Erat
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include "galeon.h"
#include "gestures.h"
	
static GHashTable *gesture_hash = NULL;

/**
 * gestures_init: initializes the hash table of gestures
 */
void
gestures_init ()
{
	gint i;
	/* gestures shamelessly pinched from
	 * http://optimoz.mozdev.org/gestures/ */
	static const struct
	{
		char *string;
		GaleonGestureAction action;
	}
	gestures[] =
	{
		/* down */
		{ "258"		, NEW_TAB	},

		/* up */
		{ "852"		, NEW_WINDOW	},

		/* up-down */
		{ "85258"	, RELOAD	},
		{ "8525"	, RELOAD	},
		{ "5258"	, RELOAD	},

		/* up-down-up */
		{ "8525852"	, RELOAD_BYPASS	},
		{ "852585"	, RELOAD_BYPASS	},
		{ "85252"	, RELOAD_BYPASS	},
		{ "85852"	, RELOAD_BYPASS	},
		{ "525852"	, RELOAD_BYPASS	},
		{ "52585"	, RELOAD_BYPASS	},

		/* up-left-down */
		{ "9632147"	, HOMEPAGE	},
		{ "963214"	, HOMEPAGE	},
		{ "632147"	, HOMEPAGE	},
		{ "962147"	, HOMEPAGE	},
		{ "963147"	, HOMEPAGE	},

		/* down-up */
		{ "25852"	, CLONE_WINDOW	},
		{ "2585"	, CLONE_WINDOW	},
		{ "5852"	, CLONE_WINDOW	},

		/* down-up-down */
		{ "2585258"	, CLONE_TAB	},
		{ "258525"	, CLONE_TAB	},
		{ "25858"	, CLONE_TAB	},
		{ "25258"	, CLONE_TAB	},
		{ "585258"	, CLONE_TAB	},
		{ "58525"	, CLONE_TAB	},

		/* up-left-up */
		{ "96541"	, UP		},
		{ "9651"	, UP		},
		{ "9541"	, UP		},

		/* right-left-right */
		{ "4565456"	, CLOSE		},
		{ "456545"	, CLOSE		},
		{ "45656"	, CLOSE		},
		{ "45456"	, CLOSE		},
		{ "565456"	, CLOSE		},
		{ "56545"	, CLOSE		},

		/* down-right */
		{ "14789"	, CLOSE		},
		{ "1489"	, CLOSE		},

		/* left */
		{ "654"		, BACK		},

		/* right */
		{ "456"		, FORWARD	},

		/* down-left */
		{ "36987"	, FULLSCREEN	},
		{ "3687"	, FULLSCREEN	},

		/* up-right */
		{ "74123"	, NEXT_TAB	},
		{ "7423"	, NEXT_TAB	},

		/* up-left */
		{ "96321"	, PREV_TAB	},
		{ "9621"	, PREV_TAB	},

		/* s */
		{ "321456987"	, VIEW_SOURCE	},
		{ "21456987"	, VIEW_SOURCE	},
		{ "32145698"	, VIEW_SOURCE	},
		{ "3256987"	, VIEW_SOURCE	},
		{ "3214587"	, VIEW_SOURCE	},
		{ "32145987"	, VIEW_SOURCE	},
		{ "32156987"	, VIEW_SOURCE	},

		/* left-up */
		{ "98741"	, STOP		},
		{ "9841"	, STOP		},

		/* left-down */
		{ "32147"	, PREV_LINK		},
		{ "3247"	, PREV_LINK		},

		/* right-down */
		{ "12369"	, NEXT_LINK		},
		{ "1269"	, NEXT_LINK		},

		/* up-right-up */
		{ "74563"	, CONTENTS_LINK		},
		{ "7453"	, CONTENTS_LINK		},
		{ "7563"	, CONTENTS_LINK		},

		/* terminator */
		{ NULL		, UNRECOGNIZED	}
	};

	/* create hash table */
	if (!gesture_hash)
		gesture_hash = g_hash_table_new (g_str_hash, g_str_equal);

	/* add strings */
	for (i = 0; gestures[i].string != NULL; i++)
	{
		gchar *action_string;

		/* skip if this string is already in the table */
		if (g_hash_table_lookup (gesture_hash, gestures[i].string))
		{
			g_warning ("gesture %s declared multiple times\n",
				   gestures[i].string);
			continue;
		}

		/* add the string to the table */
		action_string = 
			g_strdup_printf ("%d", gestures[i].action);
		g_hash_table_insert (gesture_hash, gestures[i].string,
				     action_string);
	}
}

/**
 * gestures_lookup: looks up the action for a given gesture string
 */
GaleonGestureAction
gestures_lookup (char *string)
{
	gchar *action_string;
	GaleonGestureAction action = UNRECOGNIZED;

	/* make sure table has been created */
	if (!gesture_hash)
	{
		g_warning ("gestures hash table uninitialized\n");
		return UNRECOGNIZED;
	}
	
	/* get action */
	action_string = g_hash_table_lookup (gesture_hash, string);
	if (action_string) action = atoi (action_string);

	return action;
}
