/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Pan - A Newsreader for X
 * Copyright (C) 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <libgnomeui/libgnomeui.h>

#include <pan/base/article.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>

#include <pan/filters/filter.h>
#include <pan/filters/filter-aggregate.h>
#include <pan/filters/filter-phrase.h>
#include <pan/filters/filter-top.h>
#include <pan/filters/filter-manager.h>

#include <pan/article-filter-ui.h>
#include <pan/util.h>

typedef struct
{
	GtkWidget * dialog;

	GtkWidget * bozo_author_tb;
	GtkWidget * bozo_author_entry;
	GtkWidget * bozo_author_regexp_tb;

	GtkWidget * spam_author_tb;
	GtkWidget * spam_author_entry;
	GtkWidget * spam_author_regexp_tb;

	GtkWidget * spam_subject_tb;
	GtkWidget * spam_subject_entry;
	GtkWidget * spam_subject_regexp_tb;
}
ArticleFilterDialog;



static void
add_to_filters (ArticleFilterDialog * d)
{
	gint i;
	Filter * spam;
	Filter * bozo;
	GPtrArray * filters;
	gboolean bozoize_author = FALSE;
	gboolean spamize_author = FALSE;
	gboolean spamize_subject = FALSE;

	/* get a copy of the filters, and find Spam and Bozo */	
	bozo = NULL;
	spam = NULL;
	filters = g_ptr_array_new ();
	filter_manager_get_filters (filters);
	for (i=0; i<filters->len; ++i) {
		FilterTop * f = FILTER_TOP(g_ptr_array_index(filters,i));
		if (!pan_strcmp (f->name, bozo_filter_name)) bozo = FILTER(f);
		if (!pan_strcmp (f->name, spam_filter_name)) spam = FILTER(f);
	}

	/* will we be making any changes? */
	bozoize_author = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(d->bozo_author_tb));
	spamize_author = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(d->spam_author_tb));
	spamize_subject = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(d->spam_subject_tb));

	/* ensure there's a bozo filter if we need one */
	if (bozo==NULL && bozoize_author) {
		bozo = filter_top_new ();
		filter_top_set_visible (FILTER_TOP(bozo), TRUE);
		filter_top_set_name (FILTER_TOP(bozo), bozo_filter_name);
		filter_aggregate_set_type (FILTER_AGGREGATE(bozo), AGGREGATE_TYPE_OR);
		g_ptr_array_add (filters, bozo);
	}

	/* ensure there's a spam filter if we need one */
	if (spam==NULL && (spamize_author || spamize_subject)) {
		spam = filter_top_new ();
		filter_top_set_visible (FILTER_TOP(spam), TRUE);
		filter_top_set_name (FILTER_TOP(spam), spam_filter_name);
		filter_aggregate_set_type (FILTER_AGGREGATE(spam), AGGREGATE_TYPE_OR);
		g_ptr_array_add (filters, spam);
	}

	/* update the bozo filter if necessary */
	if (bozoize_author)
	{
		Filter * filter = filter_phrase_new ();
		gchar * str = gtk_editable_get_chars (GTK_EDITABLE(d->bozo_author_entry), 0, -1);
		gboolean is_regexp = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(d->bozo_author_regexp_tb));
		g_strstrip (str);
		filter_phrase_set (FILTER_PHRASE(filter), FILTER_PHRASE_AUTHOR, str, is_regexp);
		filter_aggregate_add (FILTER_AGGREGATE(bozo), &filter, 1);
		g_free (str);
		pan_object_unref (PAN_OBJECT(filter));
	}

	/* update the spam filter if necessary */
	if (spamize_author)
	{
		Filter * filter = filter_phrase_new ();
		gchar * str = gtk_editable_get_chars (GTK_EDITABLE(d->spam_author_entry), 0, -1);
		gboolean is_regexp = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(d->spam_author_regexp_tb));
		g_strstrip (str);
		filter_phrase_set (FILTER_PHRASE(filter), FILTER_PHRASE_AUTHOR, str, is_regexp);
		filter_aggregate_add (FILTER_AGGREGATE(spam), &filter, 1);
		g_free (str);
		pan_object_unref (PAN_OBJECT(filter));
	}

	/* update the spam filter if necessary */
	if (spamize_subject)
	{
		Filter * filter = filter_phrase_new ();
		gchar * str = gtk_editable_get_chars (GTK_EDITABLE(d->spam_subject_entry), 0, -1);
		gboolean is_regexp = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(d->spam_subject_regexp_tb));
		g_strstrip (str);
		filter_phrase_set (FILTER_PHRASE(filter), FILTER_PHRASE_SUBJECT, str, is_regexp);
		filter_aggregate_add (FILTER_AGGREGATE(spam), &filter, 1);
		g_free (str);
		pan_object_unref (PAN_OBJECT(filter));
	}

	/* if we've made a change, update the filter manager */
	if (bozoize_author || spamize_author || spamize_subject)
		filter_manager_set_filters (filters);

	/* cleanup */
	pan_g_ptr_array_foreach (filters, (GFunc)pan_object_unref, NULL);
}


static gint
dialog_close_cb (GnomeDialog * dialog, gpointer data)
{
	g_free (data);
	return FALSE;
}

static void
dialog_clicked_cb (GnomeDialog * dialog, gint index, gpointer data)
{
	ArticleFilterDialog * d = (ArticleFilterDialog*) data;

	if (index==0)
		add_to_filters (d);
	if (index==0 || index==1)
		gnome_dialog_close (dialog);
	if (index==2) {
		GtkWidget * l = gnome_message_box_new (_(
"Addresses and Subjects can be changed before being added\n"
"to the Spam/Bozo filters.\n"
"\n"
"For example, a subject of `-->MAKE MONEY FAST<--' will\n"
"likey find more spam if shortened to just `MAKE MONEY FAST'.\n"
"\n"
"Likewise, \"Dudley Dursley\" <ddursley@spambot.com>' could\n"
"be shortened to `<ddurseley@spambot.com>' or even `<*@spambot.com>'.\n"
"\n"
"Wildcards are supported and comparisons are case-insensitive.\n"
"For more complex pattern matching, enable the regular expression\n"
"button.  http://pan.rebelbase.com/regexp.txt for a tutorial.\n"),
			GNOME_MESSAGE_BOX_INFO, GNOME_STOCK_BUTTON_OK, NULL);
		gnome_dialog_set_parent (GNOME_DIALOG(l), GTK_WINDOW(d->dialog));
	        gtk_widget_show_all (l);
	}
}

static void
activate_me_cb (GtkWidget * w, gpointer user_data)
{
	GtkToggleButton * tb = GTK_TOGGLE_BUTTON(user_data);
	gtk_toggle_button_set_active (tb, TRUE);
}

GtkWidget*
article_filter_dialog_new (const Article * a)
{
	gchar * pch;
	gint row = 0;
	GtkWidget * w;
	GtkWidget * t;
	GtkWidget * tb;
	ArticleFilterDialog * d;
	gchar author[256];
	const gint atch = GTK_EXPAND | GTK_SHRINK | GTK_FILL;
	const int wide_enough = 250;

	/* dialog */
	d = g_new0 (ArticleFilterDialog, 1);
	w = d->dialog = gnome_dialog_new (_("Pan: Filter Out Articles"),
	                                  GNOME_STOCK_BUTTON_OK,
	                                  GNOME_STOCK_BUTTON_CLOSE,
	                                  GNOME_STOCK_BUTTON_HELP,
	                                  NULL);
	gtk_window_set_policy (GTK_WINDOW(w), TRUE, TRUE, TRUE);
	gtk_signal_connect (GTK_OBJECT(w), "close", GTK_SIGNAL_FUNC(dialog_close_cb), d);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", GTK_SIGNAL_FUNC(dialog_clicked_cb), d);

	/* main table */
	t = gtk_table_new (3, 3, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(t), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(t), GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER(t), GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX(GNOME_DIALOG(w)->vbox), t, TRUE, TRUE, 0);

	/* bozo author */
	d->bozo_author_tb = tb = w = gtk_check_button_new_with_label (_("Add Author to Bozo Filter"));
	gtk_table_attach (GTK_TABLE(t), w, 0, 1, row, row+1, atch, atch, 0, 0);
	w = gtk_entry_new ();
	gtk_widget_set_usize (w, wide_enough, -1);
	d->bozo_author_entry = w;
	article_get_author_str (a, author, sizeof(author));
	gtk_entry_set_text (GTK_ENTRY(w), author);
	gtk_signal_connect (GTK_OBJECT(w), "changed", activate_me_cb, tb);
	gtk_table_attach (GTK_TABLE(t), w, 1, 2, row, row+1, atch, atch, 0, 0);
	w = gtk_toggle_button_new_with_label (_("regexp"));
	d->bozo_author_regexp_tb = w;
	gtk_signal_connect (GTK_OBJECT(w), "clicked", activate_me_cb, tb);
	gtk_table_attach (GTK_TABLE(t), w, 2, 3, row, row+1, atch, atch, 0, 0);
	++row;

	/* spam author */
	d->spam_author_tb = tb = w = gtk_check_button_new_with_label (_("Add Author to Spam Filter"));
	gtk_table_attach (GTK_TABLE(t), w, 0, 1, row, row+1, atch, atch, 0, 0);
	w = gtk_entry_new ();
	gtk_widget_set_usize (w, wide_enough, -1);
	d->spam_author_entry = w;
	article_get_author_str (a, author, sizeof(author));
	gtk_entry_set_text (GTK_ENTRY(w), author);
	gtk_signal_connect (GTK_OBJECT(w), "changed", activate_me_cb, tb);
	gtk_table_attach (GTK_TABLE(t), w, 1, 2, row, row+1, atch, atch, 0, 0);
	w = gtk_toggle_button_new_with_label (_("regexp"));
	d->spam_author_regexp_tb = w;
	gtk_signal_connect (GTK_OBJECT(w), "clicked", activate_me_cb, tb);
	gtk_table_attach (GTK_TABLE(t), w, 2, 3, row, row+1, atch, atch, 0, 0);
	++row;

	/* spam subject */
	d->spam_subject_tb = w = tb = gtk_check_button_new_with_label (_("Add Subject to Spam Filter"));
	gtk_table_attach (GTK_TABLE(t), w, 0, 1, row, row+1, atch, atch, 0, 0);
	w = gtk_entry_new ();
	gtk_widget_set_usize (w, wide_enough, -1);
	d->spam_subject_entry = w;
	pch = g_strdup (article_get_subject(a));
	g_strstrip (pch);
	gtk_entry_set_text (GTK_ENTRY(w), pch);
	g_free (pch);
	gtk_signal_connect (GTK_OBJECT(w), "changed", activate_me_cb, tb);
	gtk_table_attach (GTK_TABLE(t), w, 1, 2, row, row+1, atch, atch, 0, 0);
	w = gtk_toggle_button_new_with_label (_("regexp"));
	d->spam_subject_regexp_tb = w;
	gtk_signal_connect (GTK_OBJECT(w), "clicked", activate_me_cb, tb);
	gtk_table_attach (GTK_TABLE(t), w, 2, 3, row, row+1, atch, atch, 0, 0);

	return d->dialog;
}
