/*
 * Pan - A Newsreader for X
 * Copyright (C) 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <glib.h>
#include <libgnomeui/libgnomeui.h>

#include <pan/base/pan-i18n.h>

#include <pan/message-check-ui.h>
#include <pan/globals.h>
#include <pan/gui.h>
#include <pan/util.h>

typedef struct
{
	GtkWidget * window;

	Server * server;
	Article * article;

	GoodnessLevel goodness;
	GPtrArray * errors;

	ArticleCheckedFunc checked_func;
	gpointer checked_func_user_data;
}
CheckedCallbackStruct;

static void
cleanup (CheckedCallbackStruct * cbs)
{
	group_unref_articles (cbs->article->group, NULL);
	pan_g_ptr_array_foreach (cbs->errors, (GFunc)g_free, NULL);
	g_ptr_array_free (cbs->errors, TRUE);
	g_free (cbs);
}

static void
error_response_cb (GnomeDialog * dialog, int button, gpointer user_data)
{
	CheckedCallbackStruct * cbs = (CheckedCallbackStruct*) user_data;

	(cbs->checked_func)(cbs->server, cbs->article, cbs->goodness, cbs->checked_func_user_data);

	cleanup (cbs);
}

static void
warning_response_cb (GnomeDialog * dialog, int response, gpointer user_data)
{
	CheckedCallbackStruct * cbs = (CheckedCallbackStruct*) user_data;

	if (response==0) /* yes, post anyway */
		cbs->goodness = OKAY;

	(cbs->checked_func)(cbs->server, cbs->article, cbs->goodness, cbs->checked_func_user_data);

	cleanup (cbs);
}

static gint
dialog_cb (gpointer user_data)
{
	guint i;
	GString * str;
	CheckedCallbackStruct * cbs = (CheckedCallbackStruct*) user_data;

	/* build the error message */
	str = g_string_new (NULL);
	for (i=0; i!=cbs->errors->len; ++i) {
		char * e = (char*)g_ptr_array_index(cbs->errors,i);
		g_string_append (str, e);
		g_string_append_c (str, '\n');
	}

	/* create the dialog */
	if (cbs->goodness == WARN)
	{
		GtkWidget * w;

		/* make sure the user wants to proceed. */
		const gchar *ques = cbs->errors->len > 1
			? _("Post anyway, despite these problems?")
			: _("Post anyway, despite this problem?");
		g_string_append (str, "\n");
		g_string_append (str, ques);

		pan_lock ();
		w = gnome_question_dialog_parented  (str->str, NULL, cbs, GTK_WINDOW(cbs->window));
		gtk_signal_connect (GTK_OBJECT(w), "clicked", warning_response_cb, cbs);
		gtk_widget_show (w);
		pan_unlock ();
	}
	else
	{
		GtkWidget * w;

		pan_lock ();
		w = gnome_error_dialog_parented (str->str, GTK_WINDOW(cbs->window));
		gtk_signal_connect (GTK_OBJECT(w), "clicked", error_response_cb, cbs);
		gtk_widget_show (w);
		pan_unlock ();
	}

	/* cleanup */
	g_string_free (str, TRUE);
	return 0;
}

void
check_article_and_prompt (GtkWidget           * window,
                          Server              * server,
                          Article             * article,
                          ArticleCheckedFunc    checked_func,
                          gpointer              checked_func_user_data)
{
	GoodnessLevel goodness;
	GPtrArray * errors;

	/* sanity clause */
	g_return_if_fail (GTK_IS_WINDOW(window));
	g_return_if_fail (server_is_valid (server));
	g_return_if_fail (article!=NULL); /* not article_is_valid(); let check_article check for subject */

	/* check the article */
	goodness = OKAY;
       	errors = g_ptr_array_new ();
	check_article (article, server, errors, &goodness);

	if (goodness == OKAY)
	{
		(*checked_func)(server, article, goodness, checked_func_user_data);
		g_ptr_array_free (errors, TRUE);
	}
	else
	{
		CheckedCallbackStruct * cbs;
		cbs = g_new0 (CheckedCallbackStruct, 1);
		cbs->window = window;
		cbs->server = server;
		cbs->article = article;
		cbs->goodness = goodness;
		cbs->errors = errors;
		cbs->checked_func = checked_func;
		cbs->checked_func_user_data = checked_func_user_data;
		gui_queue_add (dialog_cb, cbs);

		group_ref_articles (article->group, NULL);
	}
}
