/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <string.h>

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <unistd.h>

#include <libgnomeui/libgnomeui.h>

#include <pan/base/argset.h>
#include <pan/base/debug.h>
#include <pan/base/gnksa.h>
#include <pan/base/log.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/serverlist.h>
#include <pan/base/util-file.h>
#include <pan/base/util-wrap.h>
#include <pan/base/pan-callback.h>

#include <pan/identities/identity-manager.h>

#include <pan/articlelist.h>
#include <pan/globals.h>
#include <pan/grouplist.h>
#include <pan/gui.h>
#include <pan/gui-headers.h>
#include <pan/message-check-ui.h>
#include <pan/message-send.h>
#include <pan/message-window.h>
#include <pan/prefs.h>
#include <pan/print.h>
#include <pan/queue.h>
#include <pan/task-body.h>
#include <pan/text.h> /* for colors of fg, bg, and quoted text */
#include <pan/util.h>

/* #define ATTACHMENTS */

typedef struct
{
	GtkWidget * window;

	GtkWidget * read_info_pane;

	GtkWidget * from_om;
	GtkWidget * custom_headers_text;
	GtkWidget * user_agent_tb;
	GtkWidget * organization;
	GtkWidget * newsgroups;
	GtkWidget * users;
	GtkWidget * subject;
        GtkWidget * followup_to;
        GtkWidget * reply_to;
	GtkWidget * body;

#ifdef ATTACHMENTS
	/* Attachment */
	GtkWidget * file_clist;
	GtkWidget * file_add_button;
	GtkWidget * file_remove_button;
	GtkWidget * attachment_lines_per_part_sb;
	GtkWidget * single_part_rb;
	GtkWidget * multi_part_rb;
	GtkWidget * encoding_mime_tb;
	GtkWidget * encoding_uuenc_tb;
	GtkWidget * encoding_yenc_tb;
#endif

	Article * article;
	ComposeType type;

	/* external editor */
	gchar * exteditor_file;
	pid_t   exteditor_pid;
	gint    exteditor_readdes;
	gint    exteditor_tag;

	gchar * identity_name;
	gchar * last_attribution_text;
}
Compose;

extern GtkTooltips *ttips;

static void rot13_cb                       (GtkWidget*, Compose*);
static void wrap_cb                        (GtkWidget*, Compose*);
static void save_cb                        (GtkWidget*, Compose*);
static void send_now_cb                    (GtkWidget*, Compose*);
static void send_later_cb                  (GtkWidget*, Compose*);
static void message_window_save_cb         (GtkWidget*, Compose*);
static void message_window_cut_cb          (GtkWidget*, Compose*);
static void message_window_copy_cb         (GtkWidget*, Compose*);
static void message_window_paste_cb        (GtkWidget*, Compose*);
static void message_window_close           (GtkWidget*, Compose*);
static void compose_ext_editor_cb          (GtkWidget*, Compose*);
static void identities_changed_cb          (gpointer, gpointer, gpointer);

static GtkWidget * extra_headers_page      (Compose*);
static GtkWidget * create_body_pane_nolock (Compose*);

/***
****  SHUTDOWN
***/

static gint
window_delete_event_cb (GtkWidget * w, GdkEvent * e, gpointer data)
{
	Compose * mw = (Compose*) data;
	gui_save_window_size (mw->window, "message_window");
	return FALSE;
}
static void
window_destroy_cb (GtkWidget * window, Compose * mw)
{
	pan_callback_remove (identity_manager_get_identities_changed_callback(),
		identities_changed_cb, mw);

	if (mw->article != NULL)
		group_unref_articles (mw->article->group, NULL);
	g_free (mw->identity_name);
	g_free (mw->last_attribution_text);
	g_free (mw);
}

/***
****  EXPERIMENTAL WRAP CODE
***/

static gboolean experimental_toggle_active = TRUE;

static void
wrap_tb_toggled_cb (GtkToggleButton * togglebutton, gpointer user_data)
{
	experimental_toggle_active = gtk_toggle_button_get_active (togglebutton);
}


static void
text_inserted_cb (GtkEditable * editable,
                  gchar * new_text,
                  gint new_text_length,
                  gint * position,
                  gpointer user_data)
{
	static gboolean dampen_feedback = FALSE;

	if (!dampen_feedback && experimental_toggle_active)
	{
		gchar * text = gtk_editable_get_chars (editable, 0, *position);
		gint text_len = text ? strlen(text) : 0;
		gchar * linefeed = strrchr (text, '\n');
		gboolean need_break;

		/* do we need to break the line up? */
		need_break = linefeed==NULL
			? text_len >= wrap_column
			: &text[text_len-1] - linefeed >= wrap_column;

		/* break the line up... */
		if (need_break)
		{
			gchar * pch;
			gchar * del_end_pos = NULL;

			/* try to find a place to break... */
			for (pch=text+text_len; pch!=text; --pch) {
				if (isspace((int)pch[-1])) {
					del_end_pos = pch;
					break;
				}
			}

			/* break... */
			if (del_end_pos != NULL)
			{
				gint del_start_index;
				gint del_end_index;
				gint tmp;

				/* find the beginning of the whitespace */
				for (pch=del_end_pos; pch!=text; --pch)
					if (!isspace((int)pch[-1]))
						break;
				del_end_index = del_end_pos - text;
				del_start_index = pch - text;

				/* replace the space(s) with a linefeed */
				dampen_feedback = TRUE;
				tmp = del_start_index;
				gtk_editable_delete_text (editable, tmp, del_end_index);
				gtk_editable_insert_text (editable, "\n", 1, &tmp);
				dampen_feedback = FALSE;

				/* update the position of the original text */
				*position += 1; /* for '\n' */
				*position -= (del_end_index - del_start_index);
			}
		}

		g_free (text);
	}
}

static void
experimental_wrap_handler (GtkText * text, GtkToggleButton * tb)
{
	pan_lock ();
	gtk_toggle_button_set_active (tb, experimental_toggle_active);
	gtk_signal_connect (GTK_OBJECT(text), "insert-text", text_inserted_cb, NULL);
	gtk_signal_connect (GTK_OBJECT(tb), "toggled", wrap_tb_toggled_cb, NULL);
	pan_unlock ();
}

/***
****
***/

static GnomeUIInfo message_edit_file_menu[] =
{
	GNOMEUIINFO_ITEM_STOCK (N_("Save Changes"), N_("Save Changes"), 
				save_cb, GNOME_STOCK_MENU_SAVE),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_SAVE_AS_ITEM (message_window_save_cb, NULL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_CLOSE_ITEM (message_window_close, NULL),	
	GNOMEUIINFO_END
};

static GnomeUIInfo message_post_file_menu[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("Send Now"),
		N_("Send Now"),
		send_now_cb, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_MAIL_SND,
		GDK_Return, GDK_CONTROL_MASK, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Send Later"),
		N_("Send Later"),
		send_later_cb, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_TIMER,
		'D', GDK_CONTROL_MASK, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_SAVE_AS_ITEM (message_window_save_cb, NULL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_CLOSE_ITEM (message_window_close, NULL),	
	GNOMEUIINFO_END
};

static GnomeUIInfo message_edit_menu[] =
{
	GNOMEUIINFO_MENU_CUT_ITEM (message_window_cut_cb, NULL),
	GNOMEUIINFO_MENU_COPY_ITEM (message_window_copy_cb, NULL),
	GNOMEUIINFO_MENU_PASTE_ITEM (message_window_paste_cb, NULL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_NONE (N_("_Rot13 Selected Text"),
	                       N_("Rot13 Selected Text"),
	                       &rot13_cb),
	GNOMEUIINFO_SEPARATOR, {
		GNOME_APP_UI_ITEM,
		N_("Edit with E_xternal Editor"),
		N_("Edit with External Editor"),
		compose_ext_editor_cb, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PROP,
		'D', GDK_CONTROL_MASK, NULL
	},
	GNOMEUIINFO_END
};

static GnomeUIInfo message_post_main_menu[] =
{
	GNOMEUIINFO_MENU_FILE_TREE (message_post_file_menu),
	GNOMEUIINFO_MENU_EDIT_TREE (message_edit_menu),
	GNOMEUIINFO_MENU_HELP_TREE (help_menu),
	GNOMEUIINFO_END
};

static GnomeUIInfo message_edit_main_menu[] =
{
	GNOMEUIINFO_MENU_FILE_TREE (message_edit_file_menu),
	GNOMEUIINFO_MENU_EDIT_TREE (message_edit_menu),
	GNOMEUIINFO_MENU_HELP_TREE (help_menu),
	GNOMEUIINFO_END
};


static GnomeUIInfo message_post_toolbar[] =
{
	GNOMEUIINFO_ITEM_STOCK (N_("Send Now"),
	                        N_("Send this Message Now"), 
	                        send_now_cb,
	                        GNOME_STOCK_PIXMAP_MAIL_SND),
	GNOMEUIINFO_ITEM_STOCK (N_("Send Later"),
	                        N_("Send this Message Later"), 
	                        send_later_cb,
	                        GNOME_STOCK_PIXMAP_TIMER),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK (N_("Rewrap"),
	                        N_("Rewrap all the text in the Message Body."),
	                        wrap_cb,
	                        GNOME_STOCK_PIXMAP_ALIGN_JUSTIFY),
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Wrap Text"),
		N_("Turn line wrap on/off in the Message Body"),
		NULL,
		NULL, NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_ALIGN_JUSTIFY,
		'\0', 0, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK (N_("External Editor"),
	                        N_("Edit with External Editor"),
	                        compose_ext_editor_cb,
	                        GNOME_STOCK_PIXMAP_PROPERTIES),
	GNOMEUIINFO_END
};


/**
 *  1. If the user selected an area, we use that.
 *  2. If the user didn't select an area, use the whole body.
 *  3. Quote fresh text with "> "
 *  4. Requote older quoted text with ">"
 *  5. Strip the signature if user didn't select it.
 *  6. Wrap everything.
 */
static gchar*
create_reply_body (Article * article)
{
	gboolean marked;
	GString * out = g_string_new (NULL);
	gchar * pch;

	/* body */
	pch = g_strdup (article_get_extra_header (article, PAN_REPLY_PORTION));
	marked = pch != NULL;
	if (pch == NULL)
		pch = article_get_body (article);
	g_strchomp (pch);
	if (is_nonempty_string(pch)) {
		gchar ** sd = g_strsplit (pch, "\n", -1);
		gint i;
		for (i=0; sd!=NULL && sd[i]!=NULL; ++i) {
			const gchar * l = sd[i];
			if (!marked && (!strcmp(l,"-- ") || !strcmp(l,"-- \r")))
				break;
			g_string_sprintfa (out,
				"%s%s\n", (*l=='>' ? ">" : "> "), l);
		}
		g_strfreev (sd);
	}
	g_free (pch);

	if (wrap_body_on_reply)
	{
		/* wrap what we've got so far */
		pch = fill_body (out->str, wrap_column);
		g_string_assign (out, pch);
		g_free (pch);
	}

	g_string_append (out, "\n \n");


	/* return the result */
	pch = out->str;
	g_string_free (out, FALSE);
	return pch;
}

/***
****
***/

static gboolean
is_posting (ComposeType type)
{
	return type==NNTP_POST
		|| type==NNTP_REPLY
		|| type==EMAIL_AND_POST_REPLY;
}

static gchar*
make_reply_string (const gchar* string)
{
	gchar * retval = NULL;

	if (string != NULL)
	{
		if (g_strncasecmp ("Re: ", string, 4))
			retval = g_strconcat ("Re: ", string, NULL);
		else
			retval = g_strdup (string);
	}

	return retval;
}

static void
rot13_cb (GtkWidget * widget, Compose * mw)
{
	GtkEditable * e = GTK_EDITABLE(mw->body);
	gint a = e->selection_start_pos;
	gint b = e->selection_end_pos;
	gint start = MIN (a, b);
	gint end = MAX (a, b);

	if (end > start)
	{
		gchar * str = gtk_editable_get_chars (e, start, end);
		rot13_inplace (str);
		gtk_editable_delete_text (e, start, end);
		gtk_editable_insert_text (e, str, strlen(str), &start);
	}
}

static void
wrap_cb (GtkWidget *widget, Compose * mw)
{
	gchar * body;
	gchar * new_body;
	gboolean b;
	debug_enter ("wrap_cb");

	/* get the current body */
	pan_lock();
       	body = gtk_editable_get_chars (GTK_EDITABLE(mw->body), 0, -1);
	pan_unlock ();

	new_body = fill_body (body, wrap_column);

	/* turn off our own wrapping while we fill the body pane */
	b = experimental_toggle_active;
	experimental_toggle_active = FALSE;
	update_body_pane (mw->body, new_body, FALSE);
	experimental_toggle_active = b;

	/* cleanup */
	g_free (body);
	g_free (new_body);
	debug_exit ("wrap_cb");
}

/**
***
**/

static void
message_window_cut_cb (GtkWidget * widget, Compose * mw)
{
	g_return_if_fail (mw->body);
	pan_lock ();
	gtk_editable_cut_clipboard (GTK_EDITABLE(mw->body));
	pan_unlock ();
}


static void
message_window_copy_cb (GtkWidget * widget, Compose * mw)
{
	g_return_if_fail (mw->body);
	pan_lock ();
	gtk_editable_copy_clipboard (GTK_EDITABLE(mw->body));
	pan_unlock ();
}


static void
message_window_paste_cb (GtkWidget *widget, Compose * mw)
{
	pan_lock();
	gtk_editable_paste_clipboard (GTK_EDITABLE(mw->body));
	pan_unlock();
}


static void
message_window_destroy (Compose *mw)
{
	pan_lock();
/*	gtk_container_foreach (GTK_CONTAINER (mw->window), GTK_SIGNAL_FUNC (gtk_widget_destroy), NULL); */
	gtk_widget_destroy (mw->window);
	pan_unlock();
}

static void
message_window_close (GtkWidget * widget, Compose * mw)
{
	message_window_destroy (mw);
}


static void
update_title_with_subject (GtkWidget * widget, Compose * mw)
{
	gtk_window_set_title (GTK_WINDOW (mw->window),
	                      gtk_entry_get_text (GTK_ENTRY (widget)));
}

#ifdef ATTACHMENTS

/***
****
****   ATTACHMENTS
****
***/

typedef enum
{
	ENCODING_BASE64,
	ENCODING_UUENC,
	ENCODING_YENC
}
PanEncoding;

#if 0
static PanEncoding
get_encoding_type (Compose * mw)
{
	PanEncoding retval;

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(mw->encoding_mime_tb)))
		retval = ENCODING_BASE64;
	else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(mw->encoding_uuenc_tb)))
		retval = ENCODING_UUENC;
	else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(mw->encoding_yenc_tb)))
		retval = ENCODING_YENC;
	else
		pan_warn_if_reached ();

	return retval;
}
#endif

static gulong
get_estimated_encoded_size (gulong raw_size, PanEncoding encoding)
{
	switch (encoding) {
		case ENCODING_BASE64: raw_size *= 1.33; break;
		case ENCODING_UUENC: raw_size *= 1.35; break;
		case ENCODING_YENC: raw_size *= 1.03; break;
	}
				    
	return raw_size;
}

static void
refresh_attachment_page (Compose * mw)
{
	size_t raw_size = 0;
	size_t lines_per_part = 0;
	GString * gstr = g_string_new (NULL);

	/* get lines-per-part */
	lines_per_part = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(mw->attachment_lines_per_part_sb));

	/* get the unencoded size */
	if (1) {
		gint i;
		GtkCList * clist = GTK_CLIST(mw->file_clist);
		for (i=0; i<clist->rows; ++i) {
			const gchar * filename = gtk_clist_get_row_data (clist, i);
			raw_size += get_filesize (filename);
		}
	}

	/* update mime label */
	if (1) {
		gulong encoded_size = get_estimated_encoded_size (raw_size, ENCODING_BASE64);
		gulong lines = encoded_size / 45;
		gulong parts = (lines/lines_per_part) + 1;

		g_string_assign (gstr, _("Mime BASE64 Encoding - Single Articles Only "));
		if (lines!=0 && parts==1)
			g_string_sprintfa (gstr, _("(%lu lines in 1 article)"), lines);
		else if (lines!=0)
			g_string_sprintfa (gstr, _("(%lu lines in %lu articles)"), lines, parts);
		gtk_label_set_text (GTK_LABEL(GTK_BIN(mw->encoding_mime_tb)->child), gstr->str);

		if (parts>1 && gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(mw->encoding_mime_tb)))
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(mw->encoding_uuenc_tb), TRUE);
		gtk_widget_set_sensitive (mw->encoding_mime_tb, parts<2);
	}

	/* update uuenc label */
	if (1) {
		gulong encoded_size = get_estimated_encoded_size (raw_size, ENCODING_UUENC);
		gulong lines = encoded_size / 45;
		gulong parts = (lines/lines_per_part) + 1;

		g_string_assign (gstr, _("UUEncoding - Universally Accepted "));
		if (lines!=0 && parts==1)
			g_string_sprintfa (gstr, _("(%lu lines in 1 article)"), lines);
		else if (lines!=0)
			g_string_sprintfa (gstr, _("(%lu lines in %lu articles)"), lines, parts);

		gtk_label_set_text (GTK_LABEL(GTK_BIN(mw->encoding_uuenc_tb)->child), gstr->str);
	}

	/* update yenc label */
	if (1) {
		gulong encoded_size = get_estimated_encoded_size (raw_size, ENCODING_YENC);
		gulong lines = encoded_size / 45;
		gulong parts = (lines/lines_per_part) + 1;

		g_string_assign (gstr, _("Yenc - Smaller files, less universal "));
		if (lines!=0 && parts==1)
			g_string_sprintfa (gstr, _("(%lu lines in 1 article)"), lines);
		else if (lines!=0)
			g_string_sprintfa (gstr, _("(%lu lines in %lu articles)"), lines, parts);
		gtk_label_set_text (GTK_LABEL(GTK_BIN(mw->encoding_yenc_tb)->child), gstr->str);
	}
}

static void
attachments_lines_per_part_changed (GtkWidget * w, Compose * mw)
{
	refresh_attachment_page (mw);
}

static void
attachments_page_file_selected (GtkWidget * w, GtkWidget * f)
{
	gint row;
	char * text[2];
	char line_buf[32];
	char * selected_filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION(f));
	Compose * mw = gtk_object_get_user_data (GTK_OBJECT(f));
	GtkCList * clist = GTK_CLIST(mw->file_clist);

	/* insert the file */
	g_snprintf (line_buf, sizeof(line_buf), "%lu", (gulong)(get_filesize(selected_filename)/1024));
	text[0] = selected_filename;
	text[1] = line_buf;
	row = gtk_clist_insert (clist, -1, text);
	gtk_clist_set_row_data (clist, row, g_strdup(selected_filename));

	/* update the line counts */
	refresh_attachment_page (mw);
}

static void
attachment_add_button_clicked_cb (GtkButton * button,
                                  gpointer user_data)
{
	Compose * mw = (Compose*) user_data;
	GtkWidget * w;

	w = gtk_file_selection_new(_("Select the file to attach."));
   	gtk_object_set_user_data (GTK_OBJECT(w), mw);
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION(w)->ok_button),
	                    "clicked", GTK_SIGNAL_FUNC (attachments_page_file_selected), w);
    	gtk_signal_connect_object (GTK_OBJECT(GTK_FILE_SELECTION(w)->ok_button),
	                           "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT(w));
	gtk_signal_connect_object (GTK_OBJECT(GTK_FILE_SELECTION(w)->cancel_button),
	                           "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT(w));
    	gtk_widget_show (w);
}

static void
attachment_remove_button_clicked_cb (GtkButton * button,
                                     gpointer user_data)
{
	GList * l;
	Compose * mw = (Compose*) user_data;
	GtkCList * clist = GTK_CLIST (mw->file_clist);

	l = clist->selection;
	if (l != NULL)
	{
		/* remove the file */
		gint row = GPOINTER_TO_INT (l->data);
		gchar * filename = gtk_clist_get_row_data (clist, row);
		g_free (filename);
		gtk_clist_remove (clist, row);

		/* update the line counts */
		refresh_attachment_page (mw);
	}
}

static void
attachment_clist_selection_changed_cb (GtkCList * clist,
                                       gint row,
                                       gint column,
                                       GdkEventButton * event,
                                       gpointer user_data)
{
	Compose * mw = (Compose*) user_data;
	gboolean has_selection = clist->selection != NULL;
	gtk_widget_set_sensitive (mw->file_remove_button, has_selection);
}

static GtkWidget *
attachment_page (Compose *mw)
{
	char * titles[2];
	GtkAdjustment * a;
	GtkWidget * w;
	GtkWidget * h;
	GtkWidget * v;
	GtkWidget * p;
	GtkWidget * frame;
	GtkWidget * top;

	top = gtk_vbox_new (FALSE, GNOME_PAD);


	/**
	***  Files Frame
	**/

	frame = gtk_frame_new (_("Files to Attach"));
	gtk_container_set_border_width (GTK_CONTAINER(frame), GNOME_PAD_SMALL);

	v = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER(v), GNOME_PAD);
	gtk_container_add (GTK_CONTAINER(frame), v);
	gtk_box_pack_start (GTK_BOX(top), frame, TRUE, TRUE, 0);

	/* Files clist */
	titles[0] = _("Filename");
	titles[1] = _("Kilobytes");
	mw->file_clist = w = gtk_clist_new_with_titles (2, titles);
	gtk_widget_set_usize (GTK_WIDGET(w), -1, 160);
	gtk_clist_set_column_width (GTK_CLIST(w), 0, 400);
	gtk_clist_set_column_width (GTK_CLIST(w), 1, 80);
	gtk_signal_connect (GTK_OBJECT(w), "select_row",
	                    GTK_SIGNAL_FUNC (attachment_clist_selection_changed_cb), mw);
	gtk_signal_connect (GTK_OBJECT(w), "unselect_row",
	                    GTK_SIGNAL_FUNC (attachment_clist_selection_changed_cb), mw);
        w = gtk_scrolled_window_new (NULL, NULL);
        gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(w),
                GTK_POLICY_AUTOMATIC,
                GTK_POLICY_AUTOMATIC);
        gtk_container_add (GTK_CONTAINER(w), mw->file_clist);
	gtk_box_pack_start (GTK_BOX(v), w, TRUE, TRUE, GNOME_PAD_SMALL);

	h = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX(v), h, FALSE, FALSE, 0);

	/* "Add" button */
	mw->file_add_button = w = gtk_button_new ();
	gtk_button_set_relief (GTK_BUTTON(w), GTK_RELIEF_NONE);
	p = gnome_stock_pixmap_widget_new (mw->window, GNOME_STOCK_PIXMAP_ATTACH);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", attachment_add_button_clicked_cb, mw);
	gtk_container_add (GTK_CONTAINER(w), p);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w,
		_("Add a File to the Attachment List"), "");
	gtk_box_pack_start (GTK_BOX(h), w, FALSE, FALSE, 0);

	/* "Remove" button */
	mw->file_remove_button = w = gtk_button_new ();
	gtk_widget_set_sensitive (w, FALSE);
	gtk_button_set_relief (GTK_BUTTON(w), GTK_RELIEF_NONE);
	p = gnome_stock_pixmap_widget_new (mw->window, GNOME_STOCK_PIXMAP_CLEAR);
	gtk_signal_connect (GTK_OBJECT(w), "clicked",
	                    attachment_remove_button_clicked_cb, mw);
	gtk_container_add (GTK_CONTAINER(w), p);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w,
		_("Remove a File from the Attachment List"), "");
	gtk_box_pack_end (GTK_BOX(h), w, FALSE, FALSE, 0);

	/**
	***  Parts Frame
	**/

	frame = gtk_frame_new (_("Parts"));
	gtk_container_set_border_width (GTK_CONTAINER(frame), GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX(top), frame, FALSE, FALSE, 0);

	h = gtk_hbox_new (FALSE, GNOME_PAD);
	gtk_container_set_border_width (GTK_CONTAINER(h), GNOME_PAD_BIG);
	gtk_container_add (GTK_CONTAINER(frame), h);

	w = gtk_label_new (_("Lines Per Article:"));
	gtk_box_pack_start (GTK_BOX(h), w, FALSE, FALSE, 0);
	a = GTK_ADJUSTMENT(gtk_adjustment_new (5000, 100, 15000, 100, 10, 10));
	mw->attachment_lines_per_part_sb = w = gtk_spin_button_new (a, 0, 0);
 	gtk_signal_connect (GTK_OBJECT(a), "value_changed",
	                    GTK_SIGNAL_FUNC (attachments_lines_per_part_changed), mw);
	gtk_widget_set_usize (GTK_WIDGET(w), 70, 0);
	gtk_box_pack_start (GTK_BOX(h), w, FALSE, FALSE, 0);


	/**
	***  Encoding Frame
	**/

	frame = gtk_frame_new (_("Encoding"));
	gtk_container_set_border_width (GTK_CONTAINER(frame), GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX(top), frame, FALSE, FALSE, 0);

	v = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER(v), GNOME_PAD_BIG);
	gtk_container_add (GTK_CONTAINER(frame), v);

	w = gtk_radio_button_new_with_label (NULL, _("UUEncoded (universally accepted)"));
	mw->encoding_uuenc_tb = w;
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);
	w = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(w), _("Yenc (30 percent smaller than UUEnc but less universal)"));
	mw->encoding_yenc_tb = w;
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);
	w = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(w), _("Mime (single-part posts only)"));
	mw->encoding_mime_tb = w;
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);


	refresh_attachment_page (mw);
	return top;
}

#endif

/***
****
****   EXTRA HEADERS
****
***/

static GtkWidget *
extra_headers_page (Compose * mw)
{
	GtkWidget * table;
	GtkWidget * scroll;
	GtkWidget * frame;
	GtkWidget * vbox;
	GtkWidget * w;
	GtkWidget * eventbox; 
	int row = 0;

	/**
	***  The top table: common headers
	**/

	table = gtk_table_new (4, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);

	w = gtk_label_new ( _("Followup-To:") );
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	eventbox = gtk_event_box_new ();
	gtk_container_add (GTK_CONTAINER(eventbox), w);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), eventbox,
		_("The newsgroup or newsgroups where replies to "
		  "your posted message should go.  This is only "
		  "needed if it differs from the \"Post To Groups\" "
		  "header. "
		  "\nTo direct all replies to your email address, "
		  "use \"Followup-To: poster\""), "");
	gtk_table_attach (GTK_TABLE(table), eventbox, 0, 1, row, row+1,
			  GTK_FILL, GTK_FILL, GNOME_PAD, 0);
	mw->followup_to = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE(table), mw->followup_to,
			  1, 2, row, row+1,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, GNOME_PAD, 0);

	++row;

	w = gtk_label_new ( _("Reply-To:") );
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	eventbox = gtk_event_box_new ();
	gtk_container_add (GTK_CONTAINER(eventbox), w);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), eventbox,
		_("The e-mail account where mail replies to "
		  "your posted message should go.  This is only "
		  "needed if it differs from the \"From\" "
		  "header."), "");
	gtk_table_attach (GTK_TABLE(table), eventbox,
			  0, 1, row, row+1,
			  GTK_FILL, GTK_FILL, GNOME_PAD, 0);
	mw->reply_to = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE(table), mw->reply_to,
			  1, 2, row, row+1,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, GNOME_PAD, 0);

	++row;

	w = gtk_label_new ( _("Organization:") );
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	eventbox = gtk_event_box_new ();
	gtk_container_add (GTK_CONTAINER(eventbox), w);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), eventbox,
		_("The organization you're associated with."), "");
	gtk_table_attach (GTK_TABLE(table), eventbox,
			  0, 1, row, row+1,
			  GTK_FILL, GTK_FILL, GNOME_PAD, 0);
	mw->organization = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE(table), mw->organization,
			  1, 2, row, row+1,
			  GTK_FILL|GTK_EXPAND, GTK_FILL, GNOME_PAD, 0);

	++row;

	/**
	***  The extra headers edit field
	**/

	w = gtk_text_new (NULL, NULL);
	gtk_text_set_word_wrap (GTK_TEXT(w), FALSE);
	gtk_text_set_line_wrap (GTK_TEXT(w), FALSE);
	gtk_text_set_editable (GTK_TEXT(w), TRUE);
	mw->custom_headers_text = w;
	scroll = gtk_scrolled_window_new (NULL, NULL);
        gtk_container_set_border_width (GTK_CONTAINER(scroll), GNOME_PAD_SMALL);
	gtk_container_add (GTK_CONTAINER(scroll), w);
        gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
                                        GTK_POLICY_AUTOMATIC,
                                        GTK_POLICY_AUTOMATIC);

	w = gtk_check_button_new_with_label (_("Don't add the \"User-Agent\" identification header"));
	mw->user_agent_tb = w;

	w = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER(w), GNOME_PAD);
	gtk_box_pack_start (GTK_BOX(w), scroll, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX(w), mw->user_agent_tb, FALSE, TRUE, 0);

	frame = gtk_frame_new (_("Custom Headers"));
	gtk_container_set_border_width (GTK_CONTAINER(frame), GNOME_PAD);
	gtk_container_add (GTK_CONTAINER(frame), w);

	/**
	***  Tie the top and bottom together
	**/

	vbox = gtk_vbox_new (FALSE, GNOME_PAD);
	gtk_container_set_border_width (GTK_CONTAINER(vbox), GNOME_PAD);
	gtk_box_pack_start (GTK_BOX(vbox), table, FALSE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX(vbox), frame, TRUE, TRUE, 0);
	return vbox;
}

/*--------------------------------------------------------------------
 * Newsgroups:
 * CC:
 * Subject:
 *--------------------------------------------------------------------*/
static GtkWidget*
create_post_info_pane (Compose *mw)
{
	GtkWidget * w;
	GtkWidget * main_page;
	GtkWidget * headers;
	GtkWidget * notebook;

	pan_lock();
	
	/*-----
	 * Headers Page
	 * ---- */
	
	headers = gtk_table_new (4, 2, FALSE);
	gtk_container_set_border_width (GTK_CONTAINER(headers), GNOME_PAD);
	gtk_table_set_row_spacings (GTK_TABLE(headers), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(headers), GNOME_PAD_SMALL);

	/* set the From: field */
	w = gtk_label_new ( _("From:") );
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(headers), w, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, GNOME_PAD, 0);
	mw->from_om = gtk_option_menu_new ();
	gtk_table_attach (GTK_TABLE(headers), mw->from_om, 1, 2, 0, 1,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, GNOME_PAD, 0);

	/* set the subject of the new message */
	w = gtk_label_new ( _("Subject:") );
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(headers), w, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, GNOME_PAD, 0);
	mw->subject = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE(headers), mw->subject,
	                  1, 2, 1, 2,
			  GTK_FILL|GTK_EXPAND, GTK_FILL, GNOME_PAD, 0);
	gtk_signal_connect (GTK_OBJECT (mw->subject),
	                    "changed",
	                    update_title_with_subject,
	                    mw);

	/* set the Post To Groups: */
	w = gtk_label_new ( _("Post To Groups:") );
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(headers), w,
	                  0, 1, 2, 3,
			  GTK_FILL, GTK_FILL, GNOME_PAD, 0);
	mw->newsgroups = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE(headers), mw->newsgroups,
	                  1, 2, 2, 3,
			  GTK_FILL|GTK_EXPAND, GTK_FILL, GNOME_PAD, 0);


	/* set the Mail To: */
	w = gtk_label_new ( _("Mail To:") );
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(headers), w, 0, 1, 3, 4,
			  GTK_FILL, GTK_FILL, GNOME_PAD, 0);
	mw->users = gtk_entry_new();
	gtk_table_attach (GTK_TABLE(headers), mw->users,
	                  1, 2, 3, 4,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, GNOME_PAD, 0);


	/* create the main hbox */
	main_page = gtk_vbox_new (FALSE, GNOME_PAD);
	gtk_container_set_border_width (GTK_CONTAINER(main_page), GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX(main_page), headers, FALSE, TRUE, 0);
	w = create_body_pane_nolock (mw);
	gtk_box_pack_start (GTK_BOX(main_page), w, TRUE, TRUE, 0);

	
	/*-----
	 * Fill the notebook
	 * ---- */
	
	notebook = gtk_notebook_new ();

        gtk_notebook_append_page (GTK_NOTEBOOK (notebook), main_page,
	                          gtk_label_new (_("Message")));
        gtk_notebook_append_page (GTK_NOTEBOOK (notebook), extra_headers_page(mw),
	                          gtk_label_new (_("More Headers")));
#ifdef ATTACHMENTS
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), attachment_page(mw),
	                          gtk_label_new (_("Attachments")));
#endif

	pan_unlock();

	/* listen for changes to the identity manager */
	pan_callback_add (identity_manager_get_identities_changed_callback(),
		identities_changed_cb, mw);

	return notebook;
}

/***
****
**** IDENTITIES
****
***/

/*
 * Quick helper functions for controlling the info pane
 */

static void
clear_gtk_entry_if_same (GtkEntry * entry, const char * string)
{
	const char * field = gtk_entry_get_text (entry);

	if (!pan_strcmp (field, string))
		gtk_entry_set_text (entry, "");
}

static void
set_gtk_entry_if_blank (GtkEntry * entry, const char * string)
{
	const char * field = gtk_entry_get_text (entry);

	if (is_nonempty_string(string) && !is_nonempty_string(field))
		gtk_entry_set_text (entry, string);
}

/***
**** ATTRIBUTION LINE
***/

/*
 * Generates an attribution line
 */

static gchar*
article_gen_attribution_string (const Article * a, const gchar * attribution_line)
{
	gchar author[128];
	gchar * pch = NULL;
	gchar * retval = NULL;
	const gchar * cpch = NULL;
	GString * gstr = NULL;

	/* get the unsubstituted string */
	gstr = g_string_new (attribution_line);

	/* substitutions: message-id */
	cpch = article_get_message_id (a);
	pan_g_string_replace (gstr, "%i", cpch);

	/* substitutions: date */
	pch = a==NULL ? g_strdup("") : rfc822_date_generate(a->date);
	pan_g_string_replace (gstr, "%d", pch);
	g_free (pch);

	/* substitutions: author */
	if (1) {
		gchar author[512] = { '\0' };
		article_get_author_str (a, author, sizeof(author));
		pan_g_string_replace (gstr, "%a", author);
	}

	/* substitutions: author name */
	*author = '\0';
	if (a != NULL)
		article_get_short_author_str (a, author, sizeof(author));
	pan_g_string_replace (gstr, "%n", author);

	/* cleanup & return the result */
	retval = gstr->str;
	g_string_free (gstr, FALSE);

	return retval;
}

/*
 * Remove the attribution line 
 */

static void
remove_attribution_line (Compose * mw)
{
	g_return_if_fail (mw!=NULL);

	if (is_nonempty_string(mw->last_attribution_text))
	{
		gchar * text;
		gchar * index;

		text = gtk_editable_get_chars (GTK_EDITABLE (mw->body), 0, -1);
		index = pan_strstr(text, mw->last_attribution_text);

		if (index)
		{
			index += 2+strlen (mw->last_attribution_text);
			update_body_pane (mw->body, index, FALSE);
		}

		replace_gstr (&mw->last_attribution_text, NULL);
		g_free (text);
	}
}

/*
 * Add the attribution line
 */

static void
prepend_attribution_line (Compose     * mw,
                          const gchar * attribution_line)
{
	g_return_if_fail (mw!=NULL);
	g_return_if_fail (attribution_line!=NULL);

	if (mw->article!=NULL)
	{
		gchar * pch;
		gchar * text;

		text = gtk_editable_get_chars (GTK_EDITABLE (mw->body), 0, -1);
		pch = article_gen_attribution_string (mw->article, 
		                                      attribution_line);

		if (is_nonempty_string (pch))
		{
			gchar * out = g_strdup_printf ("%s\n\n%s", pch, text);

			if (wrap_body_on_reply)
			{
				/* wrap what we've got so far */
				gchar *copy = fill_body (out, wrap_column);
				replace_gstr (&out, copy);
			}

			update_body_pane (mw->body, out, FALSE);

			replace_gstr (&mw->last_attribution_text, pch);
			g_free (out);
		}
		g_free (text);
	}
}

/***
**** SIGNATURE
***/

static void
remove_signature (GtkWidget * body)
{
	gchar * text;
	gchar * index;

	g_return_if_fail (body!=NULL);

	text = gtk_editable_get_chars (GTK_EDITABLE (body), 0, -1);

	if (!strncmp (text, "-- \r\n", 5) ||
	    !strncmp (text, "-- \n", 4))
		index = text;
	else
	{
		index = pan_strstr (text, "\n-- \n");
		if (index == NULL)
			pan_strstr (text, "\n-- \r"); 
	}	

	if (index)
	{
		*index = 0;
		update_body_pane (body, text, FALSE);
	}

	g_free (text);
}

static void
append_signature (GtkWidget * body, const gchar *sig_file)
{
	gchar * sig = NULL;
	gchar * pch;

	g_return_if_fail (body!=NULL);
	g_return_if_fail (sig_file!=NULL);

	/* since we're applying a signature, remove any signature anyway */
	remove_signature (body);

        /* load the signature */	
	pch = g_strdup (sig_file);
	g_strstrip (pch);
	if (is_nonempty_string (pch))
	{
		GArray * array;

		if (*pch=='~')
			replace_gstr (&pch, g_strconcat (g_get_home_dir(), pch+1, NULL));

		array = read_file (pch);
		if (array != NULL) {
			sig = array->data;
			g_array_free (array, FALSE);
		}
	}
	g_free (pch);

	/* append the signature */
	if (sig!=NULL)
	{
		gchar * out;
		gchar * text = gtk_editable_get_chars (GTK_EDITABLE (body), 0, -1);

		/* we insert "-- " if it's not already there */
		gboolean delimiter_exists = (
			    (!strncmp (sig, "-- \r\n", 5))
			 || (!strncmp (sig, "-- \n", 4))
			 || (pan_strstr(sig, "\n-- \r")!=NULL)
			 || (pan_strstr(sig, "\n-- \n")!=NULL));

		out = g_strdup_printf ("%s%s%s", text, 
			delimiter_exists ? "" : "\n-- \n", sig);

		update_body_pane (body, out, FALSE);

		g_free (out);
		g_free (sig);
		g_free (text);
	}
}

/***
**** CUSTOM HEADERS
****
****/

static void
remove_custom_headers (GtkWidget * header_text, Identity * id)
{
	gchar    * text;
	gint       i, pos = 0;
	gchar   ** lines;
	GString  * new_body = g_string_new (NULL);

	g_return_if_fail (header_text!=NULL);
	g_return_if_fail (id!=NULL);
	g_return_if_fail (id->custom_headers!=NULL);

	text = gtk_editable_get_chars (GTK_EDITABLE (header_text), 0, -1);
	lines = g_strsplit (text, "\n", -1);
	for (i=0; lines != NULL && lines[i] != NULL; i++)
	{
		gchar * delimit;

		g_strstrip (lines[i]);
		delimit = strchr (lines[i], ':');

		if (delimit != NULL)
		{
			gint      h;
			gboolean  match;
			gchar   * n =  g_strndup (lines[i], delimit - lines[i]);
			gchar   * v = g_strdup (delimit + 1);

			g_strstrip (n);
			g_strstrip (v);

			/*
			 * If it's a custom header from this ID, 
			 * don't add it again.
			 */

			match = FALSE;
			for (h = 0; h < id->custom_headers->len; h++)
			{
				Header * header = g_ptr_array_index (id->custom_headers, h);

				if (!pan_strcmp (n, header->name))
				{
					match = TRUE;
					break;
				}
			}
			if (!match)
				g_string_sprintfa (new_body, "%s: %s\n", 
					n, v);

			g_free (v);
			g_free (n);
		}
	}
	g_strfreev (lines);
	g_free (text);

	gtk_editable_delete_text (GTK_EDITABLE (header_text), 0, -1);
	gtk_editable_insert_text (GTK_EDITABLE (header_text), new_body->str, 
	                         new_body->len, &pos);

	g_string_free (new_body, TRUE);
}

static void
add_custom_headers (GtkWidget * header_text, Identity * id)
{
	gint      i, pos=0;
	gchar   * text;
	GString * new_body;

	g_return_if_fail (header_text!=NULL);
	g_return_if_fail (id!=NULL);
	g_return_if_fail (id->custom_headers!=NULL);

	/* To avoid duplicates, delete them first */
	remove_custom_headers (header_text, id);

	text = gtk_editable_get_chars (GTK_EDITABLE (header_text), 0, -1);
	new_body = g_string_new (text);

	for (i=0; i<id->custom_headers->len; i++)
	{
		Header * h = g_ptr_array_index (id->custom_headers, i);

		g_string_sprintfa (new_body, "%s: %s\n", h->name, h->value);
	}

	gtk_editable_delete_text (GTK_EDITABLE (header_text), 0, -1);
	gtk_editable_insert_text (GTK_EDITABLE (header_text), new_body->str,
	                          new_body->len, &pos);

	g_string_free (new_body, TRUE);
	g_free (text);
}

/***
**** SETTING IDENTITIES 
***/

static void
revoke_identity (Compose * mw, Identity * id)
{
	clear_gtk_entry_if_same (GTK_ENTRY(mw->organization), id->organization);

	clear_gtk_entry_if_same (GTK_ENTRY(mw->reply_to), id->reply_to);

	remove_attribution_line (mw);

	if (is_nonempty_string (id->signature))
		remove_signature (mw->body);

	remove_custom_headers (mw->custom_headers_text, id);
}

static void
apply_identity (Compose * mw, Identity * id)
{
	set_gtk_entry_if_blank (GTK_ENTRY(mw->organization), id->organization);

	set_gtk_entry_if_blank (GTK_ENTRY(mw->reply_to), id->reply_to);

	if (is_nonempty_string (id->attribution) && 
	    mw->type != NNTP_POST && mw->type != EDIT_ORIGINAL)
		prepend_attribution_line (mw, id->attribution);

	if (is_nonempty_string (id->signature))
		append_signature (mw->body, id->signature);

	add_custom_headers (mw->custom_headers_text, id);
}


static void
set_identity_cb (GtkMenuItem * item, gpointer data)
{
	Compose * mw = (Compose *) data;
	gchar         * new_id_name;
	Identity      * old_id;
	Identity      * new_id;

	g_return_if_fail (item);
	g_return_if_fail (data);

	new_id_name = (gchar *) gtk_object_get_data (GTK_OBJECT(item),
		"identity_name");

	new_id = identity_manager_get_identity (new_id_name);
	old_id = identity_manager_get_identity (mw->identity_name);

	if (old_id!=NULL)
	{
		revoke_identity (mw, old_id);
		pan_object_unref(PAN_OBJECT(old_id));
	}

	if (new_id!=NULL)
	{
		apply_identity (mw, new_id);
		replace_gstr(&mw->identity_name, g_strdup(new_id_name));
		pan_object_unref(PAN_OBJECT(new_id));
	}
}

static gchar *
determine_default_identity (Compose *mw)
{
	gchar       * name = NULL;
	Identity    * id = NULL;

	g_return_val_if_fail (mw!=NULL, NULL);

	/* If we're replying/forwarding via email, use that identity */
	if (mw->type == EMAIL_REPLY ||
	    mw->type == EMAIL_FORWARD)
	{
		id = identity_manager_get_default (ID_MAIL_DEFAULT);
	}

	if (id == NULL)
	{
		/* Check we're already posted with an identity */
		Group * g = mw->article ? mw->article->group :
			                  articlelist_get_group ();

		if (g != NULL && is_nonempty_string(g->identity_name))
		{
			id = identity_manager_get_identity (g->identity_name);
		}
	}

	/*
	 * If it's a new message, see if there's already an author specified.
	 * If so, use the identity with the same name/address.
	 *
	 * This is mainly here to find the right identity for feedback email.
	 *
	 * Note: this isn't bullet proof: several ids could have a matching
	 * author and email addresses. Would be better if we could 
	 * distinguish whether we have are writing an email or an article.
	 */

	if (id == NULL && mw->type == EDIT_ORIGINAL && mw->article != NULL)
	{
		id = identity_manager_get_identity_by_author (mw->article->author_real, mw->article->author_addr);
	}

	/* Still nothing. Just use the default identity for news */
	if (id == NULL)
	{
		id = identity_manager_get_default (ID_NEWS_DEFAULT);
	}

	if (id != NULL)
	{
		name = g_strdup (id->name);
		pan_object_unref (PAN_OBJECT(id));
	}

	return name;
}

static void
populate_identity_menu (Compose * mw, const char * id_name)
{
	gint        i;
	gint        index = 0;
	GtkWidget * menu = gtk_menu_new ();
	GPtrArray * identities = g_ptr_array_new ();
	const Identity  * id = NULL;

	/* get all identities */
	identity_manager_get_identities (identities);

	for (i=0; i<identities->len; ++i)
	{
		GtkWidget * item;
		gchar buf[512];
		gboolean have_addr;
		gboolean have_name;

		id = IDENTITY(g_ptr_array_index(identities,i));

		/* get default id */
		if (!pan_strcmp (id->name, id_name) || i == 0)
			index = i;

		have_addr = is_nonempty_string (id->author_addr);
		have_name = is_nonempty_string (id->author_real);

	        if (have_addr && have_name)
			g_snprintf (buf, sizeof(buf), "\"%s\" <%s>", id->author_real, id->author_addr);
		else if (have_addr)
			g_snprintf (buf, sizeof(buf), "%s", id->author_addr);
		else if (have_name)
			g_snprintf (buf, sizeof(buf), "%s", id->author_real);


		item = gtk_menu_item_new_with_label (buf);
		gtk_object_set_data_full (GTK_OBJECT(item), "identity_name", g_strdup(id->name), g_free);
		gtk_signal_connect (GTK_OBJECT(item), "activate", (GtkSignalFunc)set_identity_cb, mw);
		gtk_widget_show (item);
		gtk_menu_append (GTK_MENU(menu), item);
	}

	gtk_option_menu_set_menu (GTK_OPTION_MENU(mw->from_om), menu);
	gtk_option_menu_set_history (GTK_OPTION_MENU(mw->from_om), index);
	gtk_widget_show_all (GTK_WIDGET(mw->from_om));

	/* remember the id name */
	id = IDENTITY(g_ptr_array_index(identities, index));
	replace_gstr (&mw->identity_name, g_strdup(id->name));

	pan_g_ptr_array_foreach (identities, (GFunc)pan_object_unref, NULL);
	g_ptr_array_free (identities, TRUE);
}


static void 
identities_changed_cb (gpointer a, gpointer b, gpointer c)
{
	Compose * mw = (Compose *) c;

	populate_identity_menu (mw, mw->identity_name);

}

static void
populate_post_info_pane (Compose *mw)
{
	gchar * p;
	const gchar * cp;
	Article * a;
	char * default_id_name;

	g_return_if_fail (mw != NULL);

	a = mw->article;

	/**
	***  Populate the extra headers
	**/

	{
		GString * gstr = g_string_new (NULL);

		/* start with "X-Comment-To:" header for Fido users. */
		if (1) {
			gchar author[512] = { '\0' };
			article_get_author_str (a, author, sizeof(author));
			g_string_sprintfa (gstr, "X-Comment-To: %s\n", *author ? author : "ALL");
		}

		/* add all the extra headers. */
		if (a!=NULL && group_is_folder(a->group)) {
			guint i;
			GPtrArray * h = article_get_all_headers (a);
			for (i=0; i!=h->len; i+=2) {
				char* key = (char*)g_ptr_array_index(h,i);
				char* val = (char*)g_ptr_array_index(h,i+1);
				if (article_header_is_extra (key))
					g_string_sprintfa (gstr, "%s: %s\n", key, val);
			}
			g_ptr_array_free (h, TRUE);
		}

		/* add the headers */
		if (1) {
			gint foo = 0;
			GtkText * text = GTK_TEXT(mw->custom_headers_text);
			guint len = gtk_text_get_length (text);
			gtk_text_set_point (text, len);
			gtk_editable_insert_text (GTK_EDITABLE(text), gstr->str, gstr->len, &foo);
			if (len)
				gtk_editable_delete_text (GTK_EDITABLE(text), 0, len);
		}

		/* cleanup */
		g_string_free (gstr, TRUE);
	}

	/**
	***  Subject
	**/

	cp = a != NULL ? a->subject : NULL;
	p = NULL;
	if (cp != NULL) {
		if (mw->type == EDIT_ORIGINAL)
			p = g_strdup (cp);
		else if (mw->type == EMAIL_FORWARD)
			p = g_strdup_printf ("[%s] %s", a->group->name, cp);
		else
			p = make_reply_string (cp);
	}
	if (is_nonempty_string(p))
		gtk_entry_set_text (GTK_ENTRY(mw->subject), p);
	g_free (p);


	/**
	***  Populate the other pieces
	**/

	if (mw->type == EDIT_ORIGINAL)
	{
		gchar * pch;
		const gchar * header;

		/* newsgroups */
		g_assert (mw->newsgroups!=NULL);
		header = article_get_extra_header (a, HEADER_NEWSGROUPS);
		if (is_nonempty_string(header))
			gtk_entry_set_text (GTK_ENTRY(mw->newsgroups), header);

		/* mail-to */
		g_assert (mw->users!=NULL);
		header = article_get_extra_header (a, PAN_MAIL_TO);
		if (is_nonempty_string(header))
			gtk_entry_set_text (GTK_ENTRY(mw->users), header);

		/* followup-to */
		g_assert (mw->followup_to!=NULL);
		header = article_get_extra_header (a, HEADER_FOLLOWUP_TO);
		if (is_nonempty_string(header))
			gtk_entry_set_text (GTK_ENTRY(mw->followup_to), header);

		/* editor */
		g_assert (mw->body!=NULL);
		pch = article_get_body (a);
		update_body_pane (mw->body, pch, FALSE);
		g_free (pch);

	}
	else /* a reply */
	{
		gchar * p;
		gboolean default_newsgroups = TRUE;

		/**
		***  BODY
		**/

		g_assert (mw->body!=NULL);
		if (a != NULL) {
			p = create_reply_body (a);
			update_body_pane (mw->body, p, FALSE);
			g_free (p);
		} 

		/**
		***   NEWSGROUPS
		***   MAILTO
		**/

		p = NULL;

		/* are we posting an article? */
		if (a!=NULL && is_posting(mw->type))
		{
			const gchar * followup = article_get_extra_header (a, HEADER_FOLLOWUP_TO);
			const gchar * newsgroups = article_get_extra_header (a, HEADER_NEWSGROUPS);

			/* if user set followup-to, use that;
			 * otherwise, use newsgroups;
			 * otherwise, use the currently-selected group */
			if (is_nonempty_string(followup))
			{
				default_newsgroups = FALSE;

				/* Followup-To: poster */
				if (!g_strcasecmp ("poster", followup)) /* explicit reply by mail */ {
					const gchar * replyto = article_get_extra_header (a, HEADER_REPLY_TO);

					gnome_ok_dialog_parented (
						_("``Followup-To: poster'': sending email to author."),
						GTK_WINDOW(mw->window));

					if (is_nonempty_string(replyto))
						gtk_entry_set_text (GTK_ENTRY(mw->users), replyto);
					else {
						gchar author[512];
						article_get_author_str (a, author, sizeof(author));
						gtk_entry_set_text (GTK_ENTRY(mw->users), author);
					}
				}
				else
				/* Does Followup-To: contain a valid email address ?    */
				/* Son-of-1036 says this is invalid, but what the hell. */
				if (gnksa_check_from (followup, FALSE) == GNKSA_OK) {

					gnome_ok_dialog_parented (
						_("``Followup-To:`` contains an email address: sending email to author."),
						GTK_WINDOW(mw->window));

					gtk_entry_set_text (GTK_ENTRY(mw->users), followup);	
				}
				else /* explicit followup groups */
				{
					gtk_entry_set_text (GTK_ENTRY(mw->newsgroups), followup);
				}
			}
			else if (is_nonempty_string(newsgroups)) /* explicit groups */
			{
				gtk_entry_set_text (GTK_ENTRY(mw->newsgroups), newsgroups);
				default_newsgroups = FALSE;
			}
		}

		/* are we sending mail? */
		if (mw->type==EMAIL_REPLY || mw->type==EMAIL_AND_POST_REPLY)
		{
			/* figure out who to send mail to by checking reply-to: and from: */
			if (a!=NULL)
			{
				const gchar * replyto = article_get_extra_header (a, HEADER_REPLY_TO);
				if (is_nonempty_string(replyto))
					gtk_entry_set_text (GTK_ENTRY(mw->users), replyto);
				else {
					gchar author[512];
					article_get_author_str (a, author, sizeof(author));
					gtk_entry_set_text (GTK_ENTRY(mw->users), author);
				}
			}
		}

		/* no explicit newsgroup specified for this post,
		 * so let's guess where the user might want to post */
		if (default_newsgroups && is_posting(mw->type))
		{
			GString * str = g_string_new (NULL);

			if (a!=NULL && a->group!=NULL)
				g_string_sprintfa (str, "%s,", a->group->name);
			else {
				gint i;
				Group * thread_group = articlelist_get_group ();
				GPtrArray * ggroups = grouplist_get_selected_groups ();
				if (ggroups->len<2 && thread_group!=NULL)
					g_string_sprintfa (str, "%s,", thread_group->name);
				else for (i=0; i<ggroups->len; ++i) {
					Group * g = GROUP(g_ptr_array_index(ggroups,i));
					if (g!=NULL && !group_is_folder(g))
						g_string_sprintfa (str, "%s,", g->name);
				}
				g_ptr_array_free (ggroups, TRUE);
			}

			if (str->len != 0) {
				g_string_truncate (str, str->len-1); /* zotz last comma */
				gtk_entry_set_text (GTK_ENTRY(mw->newsgroups), str->str);
			}

			g_string_free (str, TRUE);
		}
	}

	/* populate the identity menu, setting a default id */
	default_id_name = determine_default_identity (mw);
	populate_identity_menu (mw, default_id_name);
	g_free (default_id_name);

	if (is_nonempty_string (mw->identity_name))
	{
		Identity * id = identity_manager_get_identity (mw->identity_name);

		if (id)
		{
			apply_identity (mw, id);
			pan_object_unref (PAN_OBJECT (id));
		}
	}
}


/* ctrl+return is GtkText's activate accelerator and Agent/Gravity's shortcut for "send now" */
static void
body_activate_cb (GtkWidget * widget, gpointer data) {
		send_now_cb (NULL, data);
}

static gboolean
ignore_activation_keys (GtkWidget   * widget,
                        GdkEventKey * event,
                        gpointer      data)
{
	return TRUE;
}

static GtkWidget*
create_body_pane_nolock (Compose * mw)
{
	GtkWidget * scrolled_window;
	GtkStyle * style;

	g_return_val_if_fail (mw!=NULL, NULL);

	mw->body = gtk_text_new (NULL, NULL);
	gtk_signal_connect (GTK_OBJECT(mw->body), "activate",
	                    GTK_SIGNAL_FUNC (body_activate_cb), mw);

	/* http://bugzilla.gnome.org/show_bug.cgi?id=72337 */
	gtk_signal_connect_after (GTK_OBJECT (mw->body), "key_press_event",
	                          GTK_SIGNAL_FUNC (ignore_activation_keys), mw);

	/* set the text widget's properties */
	pan_widget_set_font (GTK_WIDGET(mw->body), message_body_font_fixed);
	style = gtk_widget_get_style(mw->body);
	style->text[0] = text_fg_color;
	style->base[0] = text_bg_color;
	gtk_text_set_word_wrap (GTK_TEXT (mw->body), FALSE);
	gtk_text_set_line_wrap (GTK_TEXT (mw->body), FALSE);
	gtk_text_set_editable (GTK_TEXT (mw->body), TRUE);
	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
					GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (scrolled_window), mw->body);

	return scrolled_window;
}


void
message_post_window (void)
{
	message_window_new (NULL, NNTP_POST);
}

static void
message_post_window_create (Compose *mw)
{
	GtkWidget * w;

	pan_lock();
	mw->window = gnome_app_new ("Post", "New Message");
	gnome_app_create_menus_with_data (GNOME_APP (mw->window),
	                                  message_post_main_menu, mw);
	gnome_app_create_toolbar_with_data (GNOME_APP (mw->window),
	                                    message_post_toolbar, mw);
	pan_unlock();

	w = create_post_info_pane (mw);
	populate_post_info_pane (mw);
	experimental_wrap_handler (GTK_TEXT(mw->body),
	                           GTK_TOGGLE_BUTTON(message_post_toolbar[4].widget));

	pan_lock ();
	gnome_app_set_contents (GNOME_APP(mw->window), w);
	pan_unlock();
}


static void
make_reply_window (Article * article, ComposeType type)
{
	gchar * body;

	g_return_if_fail (article_is_valid(article));
	g_return_if_fail (type==EMAIL_REPLY
		|| type==NNTP_REPLY
		|| type==EMAIL_AND_POST_REPLY
		|| type==EMAIL_FORWARD);

	/* if this isn't the same article in the article reading window,
	 * then fire up a task to retreive the body from the local cache
	 * or on the net.  FIXME task-body really needs to be decoupled
	 * from viewing.
	 */
	if (article != get_current_article()) {
		text_set_from_article (article);
		return;
	}

	/* open up a populated reply window. */
	body = text_get_message_to_reply_to ();
	article_set_header (article, PAN_REPLY_PORTION, body, DO_CHUNK);
	message_window_new (article, type);
	g_free (body);
}

void
message_forward_window (void)
{
	Article * a = get_current_article();
	if (a != NULL)
		make_reply_window (a, EMAIL_FORWARD);
}

void
message_edit_window (Article * a)
{
	if (a == NULL)
		a = get_current_article();
	if (a != NULL)
		message_window_new (a, EDIT_ORIGINAL);
}

void
message_reply_window (void)
{
	Article * a = get_current_article();
	if (a != NULL)
		make_reply_window (a, EMAIL_REPLY);
}

void
message_followup_reply_window (void)
{
	Article * a = get_current_article();
	if (a != NULL)
		make_reply_window (a, EMAIL_AND_POST_REPLY);
}


void
message_followup_window (void)
{
	Article * a = get_current_article();
	if (a != NULL)
		make_reply_window (a, NNTP_REPLY);
}

static void
message_edit_window_create (Compose *mw)
{
	GtkWidget * w;

	pan_lock ();
	mw->window = gnome_app_new ("Edit", article_get_subject(mw->article));
	gnome_app_create_menus_with_data (GNOME_APP(mw->window),
	                                  message_edit_main_menu,
	                                  mw);
	gnome_app_create_toolbar_with_data (GNOME_APP(mw->window),
	                                    message_post_toolbar,
	                                    mw);
	pan_unlock ();

	w = create_post_info_pane (mw);
	populate_post_info_pane (mw);
	experimental_wrap_handler (GTK_TEXT(mw->body),
	                           GTK_TOGGLE_BUTTON(message_post_toolbar[4].widget));

	pan_lock ();
	gnome_app_set_contents (GNOME_APP(mw->window), w);
	gtk_text_set_editable (GTK_TEXT(mw->body), TRUE);
	pan_unlock();
}
static void
message_reply_window_create (Compose * mw)
{
	GtkWidget * w;
	gchar * title = make_reply_string (article_get_subject(mw->article));

	pan_lock();
	mw->window = gnome_app_new ("Reply", title);
	gnome_app_create_menus_with_data (GNOME_APP(mw->window),
	                                  message_post_main_menu, mw);
	gnome_app_create_toolbar_with_data (GNOME_APP(mw->window),
	                                    message_post_toolbar, mw);
	pan_unlock();

	w = create_post_info_pane (mw);
	populate_post_info_pane (mw);
	experimental_wrap_handler (GTK_TEXT(mw->body),
	                           GTK_TOGGLE_BUTTON(message_post_toolbar[4].widget));

	pan_lock ();
	gnome_app_set_contents (GNOME_APP (mw->window), w);
	pan_unlock();

	/* cleanup */
	g_free (title);
}

void
message_window_new (Article * article, ComposeType type)
{
	Compose *mw;

	if (article != NULL)
		group_ref_articles (article->group, NULL);

	mw = g_new0 (Compose, 1);
	mw->article = article;
	mw->type = type;

	switch (type)
	{
		case NNTP_POST:
			message_post_window_create (mw);
			break;

		case NNTP_REPLY:
		case EMAIL_REPLY:
		case EMAIL_AND_POST_REPLY:
		case EMAIL_FORWARD:
			message_reply_window_create (mw);
			break;

		case EDIT_ORIGINAL:
			message_edit_window_create (mw);
			break;

		default:
			pan_warn_if_reached ();
			break;
	}

	if (mw->window != NULL)
	{
		pan_lock ();
		gtk_window_set_default_size (GTK_WINDOW (mw->window), 550, 410);
		gtk_signal_connect (GTK_OBJECT(mw->window), "delete_event",
		                    GTK_SIGNAL_FUNC(window_delete_event_cb), mw);
		gtk_signal_connect (GTK_OBJECT(mw->window), "destroy",
		                    GTK_SIGNAL_FUNC(window_destroy_cb), mw);
		gtk_window_set_policy (GTK_WINDOW(mw->window), TRUE, TRUE, TRUE);
		gtk_widget_grab_focus (mw->subject);
		pan_unlock ();

		gtk_window_set_default_size (GTK_WINDOW(mw->window), 600, 575);
		gui_restore_window_size (mw->window, "message_window");

		pan_lock ();
		gtk_widget_show_all (mw->window);
		pan_unlock ();
	}
}

/***
****
****  SENDING THE MESSAGE
****
***/

static gchar*
get_text_from_editable (GtkWidget* w)
{
	gchar * pch;
	g_return_val_if_fail (GTK_IS_EDITABLE(w), NULL);
	pch = gtk_editable_get_chars (GTK_EDITABLE(w), 0, -1);
	g_strstrip (pch);
	return pch;
}

static gchar*
get_text_from_maybe_editable (GtkWidget * w)
{
	gchar * pch = NULL;

	if (w!=NULL && GTK_IS_EDITABLE(w))
	{
		pch = gtk_editable_get_chars (GTK_EDITABLE(w), 0, -1);
		g_strstrip (pch);
	}

	return pch;
}

#if 0
static gchar*
message_window_get_lines_per_part (const Compose* mw)
{
	int lpp; 

	g_return_val_if_fail (mw!=NULL, NULL);
	
	lpp = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(mw->lines_per_part));

	return g_strdup_printf ("%d", lpp);
}
#endif

#if 0
static gchar*
message_window_get_filename (const Compose* mw)
{
	gchar * pch;

	g_return_val_if_fail (mw!=NULL, NULL);

	pch = get_text_from_maybe_editable (mw->filename);

	return pch;
}
#endif

static gchar*
message_window_get_reply_to (const Compose* mw)
{
	gchar * pch;

	g_return_val_if_fail (mw!=NULL, NULL);

	pch = get_text_from_maybe_editable (mw->reply_to);
	if (pch == NULL)
		pch = g_strdup ("");

	return pch;
}

static gchar*
message_window_get_group_str (GtkWidget * e)
{
	gchar * pch;

	g_return_val_if_fail (e!=NULL, NULL);

	pch = get_text_from_maybe_editable (e);
	if (pch == NULL)
	{
		pch = g_strdup ("");
	}
	else
	{
		gchar * tmp;

		/* ensure all the delimiters are commas */
		g_strdelimit (pch, ":; ", ',');

		/* remove leading/trailing commas */
		for (tmp=pch; *tmp==','; ++tmp);
		g_memmove (pch, tmp, strlen(tmp)+1);
		for (tmp=pch+strlen(pch)-1; tmp>pch && *tmp==','; --tmp);
		tmp[1] = '\0';

		/* remove empty entries */
		while ((tmp=pan_strstr(pch,",,")) != NULL)
			g_memmove (tmp, tmp+1, strlen(tmp+1)+1);
	}

	return pch;
}

static gchar*
message_window_get_organization (const Compose* mw)
{
	gchar * pch = NULL;
	g_return_val_if_fail (mw!=NULL, NULL);
	pch = get_text_from_maybe_editable (mw->organization);
	if (pch == NULL)
		pch = g_strdup ("");
	if (pch != NULL)
		g_strstrip (pch);
	return pch;
}

static gchar*
message_window_get_reverse_path (const Compose* mw)
{
	Identity * id;
	gchar    * from;
	gchar    * retval = NULL;
	int        status;
	gchar      address[512];
	gchar      realname[512];

	g_return_val_if_fail (mw!=NULL, NULL);
	g_return_val_if_fail (is_nonempty_string (mw->identity_name), NULL);

	id = identity_manager_get_identity (mw->identity_name);
	from = g_strdup_printf ("\"%s\" <%s>", id->author_real, id->author_addr);
	pan_object_unref (PAN_OBJECT(id));

	status = gnksa_do_check_from (from,
	                              address, sizeof(address),
	                              realname, sizeof(realname),
				      FALSE);

	if (status == GNKSA_OK)
		retval = g_strdup (address);

	g_free (from);

	return retval;
}


static void
populate_article_from_mw (Article * article, const Compose * mw)
{
	gint i;
	gchar ** sd;
	gchar * pch;
	Identity * id;

	g_return_if_fail (mw!=NULL);
	g_return_if_fail (is_nonempty_string (mw->identity_name));

	/* identity */
	id = identity_manager_get_identity (mw->identity_name);

	/* date */
	article->date = time(0);

	/* group's article number */
	article->number = 1;

	/* author */
	article->author_addr = is_nonempty_string (id->author_addr)
		? group_chunk_string (article->group, id->author_addr, TRUE)
		: NULL;
	article->author_real = is_nonempty_string (id->author_real)
		? group_chunk_string (article->group, id->author_real, TRUE)
		: NULL;

	/* message-id */
	if (is_nonempty_string (id->msg_id_fqdn)) 
		pch = gnksa_generate_message_id (id->msg_id_fqdn);
	else
		pch = gnksa_generate_message_id_from_email_addr (article->author_addr);
	article_init_header (article, HEADER_MESSAGE_ID, pch, DO_CHUNK);
	g_free (pch);

	/* subject */
	pch = get_text_from_editable (mw->subject);
	article_init_header (article, HEADER_SUBJECT, pch, DO_CHUNK_SHARE);
	g_free (pch);

	/* references (for posting articles) */
	if ((mw->type==NNTP_REPLY || mw->type==EMAIL_REPLY || mw->type==EMAIL_AND_POST_REPLY)
	    && (mw->article != NULL))
	{
		const gchar * refs = mw->article->references;
		const gchar * msg_id = article_get_message_id (mw->article);
		pch = gnksa_generate_references (refs, msg_id);
		if (pch != NULL) {
			article_init_header (article, HEADER_REFERENCES, pch, DO_CHUNK_SHARE);
			g_free (pch);
		}
	}

	/* organization */
	pch = message_window_get_organization(mw);
	article_init_header (article, HEADER_ORGANIZATION, pch, DO_CHUNK_SHARE);
	g_free (pch);

	/* followup-to */
	pch = message_window_get_group_str(mw->followup_to);
	article_init_header (article, HEADER_FOLLOWUP_TO, pch, DO_CHUNK_SHARE);
	g_free (pch);

	/* reply-to */
	pch = message_window_get_reply_to (mw);
	article_init_header (article, HEADER_REPLY_TO, pch, DO_CHUNK_SHARE);
	g_free (pch);

	/* pan-body */
	pch = get_text_from_editable (mw->body);
	article_init_header (article, PAN_BODY, pch, DO_CHUNK);
	g_free (pch);

	/* newsgroups (for posting articles) */
	pch = message_window_get_group_str (mw->newsgroups);
	article_init_header (article, HEADER_NEWSGROUPS, pch, DO_CHUNK_SHARE);
	g_free (pch);

	/* pan-mail-to (for sending mail) */
	pch = get_text_from_editable (mw->users);
	article_init_header (article, PAN_MAIL_TO, pch, DO_CHUNK_SHARE);
	g_free (pch);

	/* pan-reverse-path (for sending mail) */
	pch = message_window_get_reverse_path (mw);
	if (is_nonempty_string(pch))
		article_init_header (article, PAN_REVERSE_PATH, pch, DO_CHUNK_SHARE);
	g_free (pch);

	/* attribution header (for message-check) */
	if (mw->article!=NULL && mw->identity_name!=NULL) {
		Identity * id = identity_manager_get_identity (mw->identity_name);
		gchar * attribution = article_gen_attribution_string (mw->article,
			id->attribution);
		if (is_nonempty_string(attribution))
			article_init_header (article, PAN_ATTRIBUTION, attribution, DO_CHUNK);
		g_free (attribution);
		pan_object_unref (PAN_OBJECT (id));
	}

#if 0
	/* attachment: filename */
	pch = message_window_get_filename (mw);
	if (pch != NULL) {
		article_set_header (article, PAN_ATTACH_FILE, pch, DO_CHUNK);
		g_free (pch);
	}

	/* attachment: lines per article */
	pch = message_window_get_lines_per_part (mw);
	if (pch != NULL) {
		article_set_header (article, PAN_LINES_PER_PART, pch, DO_CHUNK);
		g_free (pch);
	}
#endif

	/* custom headers */
	pch = gtk_editable_get_chars (GTK_EDITABLE(mw->custom_headers_text), 0, -1);
	sd = g_strsplit (pch, "\n", -1);
	for (i=0; sd!=NULL && sd[i]!=NULL; ++i) {
		gchar * delimit;
		g_strstrip (sd[i]);
		delimit = strchr (sd[i], ':');
		if (delimit != NULL) {
			gchar * key = g_strndup (sd[i], delimit-sd[i]);
			gchar * val = g_strdup (delimit + 1);
			g_strstrip (key);
			g_strstrip (val);
			article_init_header (article, key, val, DO_CHUNK_SHARE);
			g_free (key);
			g_free (val);
		}
	}
	g_strfreev (sd);
	g_free (pch);

	/* cleanup */
	pan_object_unref(PAN_OBJECT(id));
}

static Article*
build_article_from_window (const Compose * mw)
{
	Group * folder;
	Article * article;
	debug_enter ("build_article_from_window");

	folder  = serverlist_get_named_folder (PAN_SENDLATER);
	article = article_new (folder);
	populate_article_from_mw (article, mw);

	debug_exit ("build_article_from_window");
	return article;
}

static void
save_cb (GtkWidget * button, Compose * mw)
{
	Article * a = mw->article;
	Article * tmp_article;
	guint i;
	GPtrArray * headers;

	/* update the article fields from changes made in the gui */
	tmp_article = article_new (a->group);
	populate_article_from_mw (tmp_article, mw);
	a->author_addr = tmp_article->author_addr;
	a->author_real = tmp_article->author_real;
	a->subject = tmp_article->subject;

	/* replace old headers with new headers */
	headers = article_get_all_headers (a);
	for (i=0; i!=headers->len; i+=2) {
		gchar * key = (gchar*) g_ptr_array_index (headers, i);
		article_remove_header (a, key);
	}
	g_ptr_array_free (headers, TRUE);
	headers = article_get_all_headers (tmp_article);
	for (i=0; i!=headers->len; i+=2) {
		gchar * key = (gchar*) g_ptr_array_index (headers, i);
		gchar * val = (gchar*) g_ptr_array_index (headers, i+1);
		article_set_header (a, key, val, 0);
	}
	g_ptr_array_free (headers, TRUE);

	/* cleanup */
	message_window_destroy (mw);
}

typedef enum
{
	SEND_NOW,
	SEND_LATER
}
SendMode;

static void
post_checked_cb (Server        * server,
                 Article       * article,
                 GoodnessLevel   goodness,
                 gpointer        user_data)
{
	ArgSet * argset = (ArgSet*)user_data;
	Compose * mw = (Compose*) argset_get (argset, 0);
	SendMode mode = GPOINTER_TO_INT(argset_get(argset, 1));

	if (goodness == OKAY)
	{
		const char * pch;
		char * group_name;
		Group * sendlater = serverlist_get_named_folder (PAN_SENDLATER);

		/* gotta ref the articles to make sure that, after sendlater
		 * takes over memory management of article, it stays alive
		 * until after we've called queue_article_for_posting. */
		group_ref_articles (sendlater, NULL);

		/* store which server to post through, in case the article
		 * winds up waiting in pan.sendlater */
		if (!is_nonempty_string(article_get_extra_header (article, PAN_SERVER)))
			article_set_header (article, PAN_SERVER, server->name, DO_CHUNK_SHARE);

		/* add this article to the sendlater group */
		group_add_article (sendlater, article);

		/* try to post right now, if desired... */
		if (mode == SEND_NOW)
			queue_article_for_posting (server, article, FALSE);

		/* remember which identity we used */
		pch = article_get_extra_header (article, HEADER_NEWSGROUPS);
		while ((group_name = get_next_token_str (pch, ',', &pch))!=NULL)
		{
			Group * group;

			g_strstrip (group_name);
			group = server_get_named_group (server, group_name);
			if (group!=NULL && !group_is_folder (group))
				group_set_identity (group, mw->identity_name);

			g_free (group_name);
		}

		/* close this window */
		message_window_destroy (mw);

		group_unref_articles (sendlater, NULL);
	}

	/* cleanup */
	if (goodness!=OKAY && mw->type!=EDIT_ORIGINAL)
		group_remove_article (serverlist_get_named_folder(PAN_SENDLATER), article);

	argset_free (argset);
}

static void
post (Compose * mw, SendMode mode)
{
	Article * article;
	Server * server;

	debug_enter ("post");

       	server = serverlist_get_active_server ();

	/* Get the article pointer, either through repopulating the
	 * current article (if we're editing an existing message) or
	 * by creating a new article (if this is a new message).
	 */
	if (mw->type == EDIT_ORIGINAL) {
		article = mw->article;
		populate_article_from_mw (article, mw);
	} else {
		article = build_article_from_window (mw);
	}

	/* if the article's okay, then send or queue */
	check_article_and_prompt (mw->window,
	                          server,
	                          article,
	                          post_checked_cb,
	                          argset_new2 (mw, GINT_TO_POINTER(mode)));

	debug_exit ("post");
}

static void
send_now_cb (GtkWidget * button, Compose * mw)
{
	post (mw, SEND_NOW);
}
static void
send_later_cb (GtkWidget * button, Compose * mw)
{
	post (mw, SEND_LATER);
}

/**
***  Save as File
**/

static void
message_window_save_ok_clicked (GtkWidget * widget, GtkWidget * file_entry)
{
	const gchar * filename;
	gchar * writeme = NULL;
	const Compose * mw;

	/* get information from the widget */
	pan_lock();
	mw = (const Compose*) gtk_object_get_data (GTK_OBJECT(file_entry), "mw");
	filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION (file_entry));
	pan_unlock();

	/* get the article text to save... */
	if (1) {
		Article * a = build_article_from_window (mw);
		writeme = article_get_message (a);
	}

	/* write the text */
	if (is_nonempty_string(writeme))
	{
		FILE * msg_dump = fopen (filename, "w");
		if (msg_dump != NULL)
		{
			fprintf (msg_dump, "%s\n", writeme);
			fclose (msg_dump);

			pan_lock();
			gtk_widget_destroy (file_entry);
			pan_unlock();
		}
		else
		{
		}
	}

	/* cleanup */
	g_free (writeme);
}

static void
message_window_save_cb (GtkWidget *widget, Compose * mw)
{
	GtkWidget *file_entry = NULL;
	
	pan_lock();
	
	file_entry = gtk_file_selection_new (_("Save message to file"));

	gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (file_entry));

	gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (file_entry)->cancel_button),
				   "clicked",
				   GTK_SIGNAL_FUNC (gtk_widget_destroy),
				   GTK_OBJECT (file_entry));

	/* message_id is const, but we don't alter it here */
	gtk_object_set_data (GTK_OBJECT (file_entry), "mw", (gpointer)mw);

	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (file_entry)->ok_button),
				"clicked",
				GTK_SIGNAL_FUNC (message_window_save_ok_clicked),
				file_entry);

	gtk_widget_show_all (file_entry);
	pan_unlock();
}

/***
****
****  External Editor (mostly from Sylpheed)
****
***/

static void
compose_insert_file (Compose * mw, const gchar * file)
{
	FILE * fp;
	gchar buf[1024];
	GtkText * text = GTK_TEXT(mw->body);
				 
	g_return_if_fail(file != NULL);

	fp = fopen (file, "r");
	if (!fp) {
		log_add_va (LOG_ERROR, _("Couldn't open file `%s'"), file);
		return;
	}
						 
	gtk_text_freeze(text);
	while (fgets(buf, sizeof(buf), fp) != NULL)
		gtk_text_insert(text, NULL, NULL, NULL, buf, -1); 
	gtk_text_thaw(text);
									 
	fclose (fp);
}

static void
compose_input_cb (gpointer data, gint source, GdkInputCondition condition)
{
	gchar buf[3];
	Compose * mw = (Compose*) data;
	gint i = 0;

	gdk_input_remove (mw->exteditor_tag);

	for (;;) {
		if (read(source, &buf[i], 1) < 1) {
			buf[0] = '3';
			break;
		}
		if (buf[i] == '\n') {
			buf[i] = '\0';
			break;
		}
		++i;
		if (i == sizeof(buf) - 1)
			break;
	}

	waitpid (mw->exteditor_pid, NULL, 0);

	if (buf[0] == '0') { /* success */
		GtkText * text = GTK_TEXT(mw->body);

		gtk_text_freeze (text);
		gtk_text_set_point (text, 0);
		gtk_text_forward_delete (text, gtk_text_get_length(text));
		compose_insert_file (mw, mw->exteditor_file);
		gtk_text_thaw (text);

		unlink (mw->exteditor_file);
	}
	else if (buf[0] == '1') { /* failed */
		log_add (LOG_ERROR, _("Couldn't exec external editor"));
		unlink (mw->exteditor_file);
	}
	else if (buf[0] == '2') {
		log_add (LOG_ERROR, _("Couldn't write to file"));
	}
	else if (buf[0] == '3') {
		log_add (LOG_ERROR, _("Pipe read failed"));
	}

	close (source);

	gtk_widget_set_sensitive (mw->window, TRUE);

	replace_gstr (&mw->exteditor_file, NULL);
	mw->exteditor_pid = -1;
	mw->exteditor_tag = -1;
	mw->exteditor_readdes = -1;
}

static gint
compose_write_body_to_file (Compose * mw, const gchar * file)
{
	FILE * fp;
	size_t len;
	gchar * chars;

	fp = fopen (file, "w");
	if (!fp) {
		log_add_va (LOG_ERROR, _("Unable to open file `%s'"), file);
		return -1;
	}

	/* chmod for security */
	if (change_file_mode_rw (fp, file) < 0)
		log_add (LOG_ERROR, _("Unable to change file mode"));

	chars = gtk_editable_get_chars (GTK_EDITABLE(mw->body), 0, -1);

	/* write body */
	len = strlen (chars);
	if (fwrite (chars, sizeof(gchar), len, fp) != len) {
		log_add (LOG_ERROR, _("Unable to write to temp file"));
		g_free (chars);
		fclose (fp);
		unlink (file);
		return -1;
	}

	g_free (chars);

	if (fclose(fp) == EOF) {
		unlink (file);
		return -1;
	}

	return 0;
}

static gint
compose_exec_ext_editor_real (const gchar * file)
{
	gchar * tmp;
	gchar ** cmdline;
	pid_t pid;

	g_return_val_if_fail (is_nonempty_string(file), -1);
	g_return_val_if_fail (is_nonempty_string(external_editor), -1);

	if ((pid = fork()) < 0) {
		perror ("fork");
		return -1;
	}

	if (pid != 0)
		return pid;

	/* grandchild process */
	if (setpgid (0, getppid()))
		perror ("setpgid");

	tmp = pan_substitute (external_editor, "%t", file);
	cmdline = strsplit_command_line_arguments (tmp);
	execvp (cmdline[0], cmdline);

	perror ("execvp");
	g_strfreev (cmdline);
	g_free (tmp);

	_exit (1);
	return 1;
}

static void
compose_exec_ext_editor (Compose * mw)
{
	gchar tmp[64];
	pid_t pid;
	gint pipe_fds[2];

	/* get a temporary filename */
	if (1) {
		gchar * tmp_dir = pan_substitute (g_get_tmp_dir(), "~", g_get_home_dir());
		g_snprintf (tmp, sizeof(tmp), "%s%ctmpmsg.%08x", tmp_dir, G_DIR_SEPARATOR, GPOINTER_TO_INT(mw));
		g_free (tmp_dir);
	}

	if (pipe (pipe_fds) < 0) {
		perror ("pipe");
		return;
	}

	if ((pid = fork()) < 0) {
		perror ("fork");
		return;
	}

	if (pid != 0)
	{
		/* close out the write side of the pipe */
		close (pipe_fds[1]);

		mw->exteditor_file = g_strdup (tmp);
		mw->exteditor_pid = pid;
		mw->exteditor_readdes = pipe_fds[0];

		gtk_widget_set_sensitive (mw->window, FALSE);

		mw->exteditor_tag = gdk_input_add (pipe_fds[0], GDK_INPUT_READ, compose_input_cb, mw);
	}
	else
	{
		pid_t pid_ed;

		if (setpgid(0, 0))
			perror ("setpigid");

		/* close the read side of the pipe */
		close (pipe_fds[0]);

		if (compose_write_body_to_file (mw, tmp) < 0) {
			fd_write (pipe_fds[1], "2\n", 2);
			_exit (1);
		}

                pid_ed = compose_exec_ext_editor_real(tmp);
		if (pid_ed < 0) {
			fd_write(pipe_fds[1], "1\n", 2);
			_exit(1);
		}

		/* wait until editor is terminated */
		waitpid (pid_ed, NULL, 0);

		fd_write (pipe_fds[1], "0\n", 2);

		close (pipe_fds[1]);
		_exit(0);
	}
}

static void
compose_ext_editor_cb (GtkWidget * widget, Compose * mw)
{
	compose_exec_ext_editor (mw);
}
