#include "piece.h"

#include <kglobal.h>


//-----------------------------------------------------------------------------
GPieceInfo::GPieceInfo()
{
	Piece::setPieceInfo(this);
}

QPixmap *GPieceInfo::pixmap(uint blockSize, uint blockType, uint blockMode,
                            bool lighted) const
{
    QPixmap *pixmap = new QPixmap(blockSize, blockSize);
    draw(pixmap, blockType, blockMode, lighted);
    setMask(pixmap, blockMode);
    return pixmap;
}

uint GPieceInfo::maxWidth() const
{
	int min, max;
	uint w = 0;
	for (uint n=0; n<nbForms(); n++) {
        min = max = i(n, 0)[0];
        for (uint k=0; k<nbBlocks(); k++) {
            int tmp = i(n, 0)[k];
            if ( tmp>max ) max = tmp;
            else if ( tmp<min ) min = tmp;
        }
        w = kMax(w, (uint)max-min);
    }
    return w;
}

uint GPieceInfo::maxHeight() const
{
	int min, max;
	uint h = 0;
	for (uint n=0; n<nbForms(); n++) {
        min = max = j(n, 0)[0];
        for (uint k=0; k<nbBlocks(); k++) {
            int tmp = j(n, 0)[k];
            if ( tmp>max ) max = tmp;
            else if ( tmp<min ) min = tmp;
        }
        h = kMax(h, (uint)max-min);
    }
	return h;
}

uint GPieceInfo::generateType(KRandomSequence *random) const
{
    return random->getLong( nbTypes() );
}

uint GPieceInfo::generateGarbageBlockType(KRandomSequence *random) const
{
    return nbNormalBlockTypes() + random->getLong( nbGarbageBlockTypes() );
}

//-----------------------------------------------------------------------------
SequenceArray::SequenceArray()
: _size(0)
{
	const GPieceInfo *pinfo = Piece::info();
	fill(0, pinfo->nbNormalBlockTypes() + pinfo->nbGarbageBlockTypes());
}

void SequenceArray::setBlockSize(uint bsize)
{
	_size = bsize;
	const GPieceInfo *pinfo = Piece::info();
	QPtrList<QPixmap> pixmaps;
	pixmaps.setAutoDelete(TRUE);
	QPtrList<QPoint> points;
	points.setAutoDelete(TRUE);
	uint nm = pinfo->nbBlockModes();
	for (uint i=0; i<size(); i++) {
		for (uint k=0; k<2; k++)
			for (uint j=0; j<nm; j++) {
				QPoint *po = new QPoint(0, 0);
				QPixmap *pi = pinfo->pixmap(bsize, i, j, k==1);
				if ( at(i) ) {
					at(i)->setImage(k*nm + j, new QCanvasPixmap(*pi, *po));
					delete po;
					delete pi;
				} else {
					points.append(po);
					pixmaps.append(pi);
				}
			}
		if ( at(i)==0 ) {
			at(i) = new QCanvasPixmapArray(pixmaps, points);
			pixmaps.clear();
			points.clear();
		}
	}
}

SequenceArray::~SequenceArray()
{
	for (uint i=0; i<size(); i++) delete at(i);
}

//-----------------------------------------------------------------------------
BlockInfo::BlockInfo(const SequenceArray &s)
: _sequences(s)
{}

int BlockInfo::toX(uint col) const
{
	return col * _sequences.blockSize();
}

int BlockInfo::toY(uint line) const
{
	return line * _sequences.blockSize();
}

//-----------------------------------------------------------------------------
Block::Block(uint value)
: _value(value), _sprite(0)
{}

Block::~Block()
{
	delete _sprite;
}

void Block::setValue(uint value, BlockInfo *binfo)
{
	_value = value;
	if (binfo) {
		QCanvasPixmapArray *seq = binfo->sequences()[value];
		if (_sprite) _sprite->setSequence(seq);
		else {
			_sprite = new QCanvasSprite(seq, binfo);
			_sprite->setZ(0);
		}
	}
}

void Block::toggleLight()
{
	const GPieceInfo *pinfo = Piece::info();
	uint f = _sprite->frame() + pinfo->nbBlockModes()
		* (_sprite->frame()>=(int)pinfo->nbBlockModes() ? -1 : 1);
	_sprite->setFrame(f);
}

bool Block::isGarbage() const
{
	return Piece::info()->isGarbage(_value);
}


//-----------------------------------------------------------------------------
const GPieceInfo *Piece::_info = 0;

Piece::Piece()
    : _binfo(0), _i(0), _j(0)
{
    _blocks.setAutoDelete(true);
}

void Piece::rotate(bool left, int x, int y)
{
    if (left) {
        if ( _rotation==0 ) _rotation = 3;
        else _rotation--;
    } else {
        if ( _rotation==3 ) _rotation = 0;
        else _rotation++;
    }

    uint form = _info->form(_type);
    _i = _info->i(form, _rotation);
    _j = _info->j(form, _rotation);
    if (_binfo) move(x, y);
}

int Piece::minX() const
{
    if ( _i==0 ) return 0;
    int min = _i[0];
    for(uint k=1; k<_info->nbBlocks(); k++) min = kMin(min, _i[k]);
    return min;
}

int Piece::maxX() const
{
    if ( _i==0 ) return 0;
    int max = _i[0];
    for(uint k=1; k<_info->nbBlocks(); k++) max = kMax(max, _i[k]);
    return max;
}

int Piece::minY() const
{
    if ( _j==0 ) return 0;
    int min = _j[0];
    for(uint k=1; k<_info->nbBlocks(); k++) min = kMin(min, _j[k]);
    return min;
}

int Piece::maxY() const
{
    if ( _j==0 ) return 0;
    int max = _j[0];
    for(uint k=1; k<_info->nbBlocks(); k++) max = kMax(max, _j[k]);
    return max;
}

void Piece::copy(const Piece *p)
{
    if ( p->_blocks.size()!=0 ) {
        _blocks.resize(p->_blocks.size());
        for (uint k=0; k<_blocks.size(); k++) {
            if ( _blocks[k]==0 ) _blocks.insert(k, new Block);
            _blocks[k]->setValue(p->_blocks[k]->value(), _binfo);
        }
    }
	_type = p->_type;
	_random = p->_random;
    _rotation = p->_rotation;
    _i = p->_i;
    _j = p->_j;
}

void Piece::generateNext(int type)
{
	Q_ASSERT(_binfo); // ie graphic
    if ( _blocks.size()==0 ) {
        _blocks.resize(_info->nbBlocks());
        for (uint k=0; k<_blocks.size(); k++) _blocks.insert(k, new Block);
    }
	_type = (type==-1 ? _info->generateType(_random) : (uint)type );
    _rotation = 0;

    uint form = _info->form(_type);
    _i = _info->i(form, _rotation);
    _j = _info->j(form, _rotation);

	for (uint k=0; k<_blocks.size(); k++)
		_blocks[k]->setValue(_info->value(_type, k), _binfo);
}

void Piece::moveCenter()
{
	uint size = _binfo->sequences().blockSize();
	int x = (_binfo->width()  - width() *size)/2 - minX()*size;
	int y = (_binfo->height() - height()*size)/2 - minY()*size;
	move(x, y);
}

void Piece::move(int x, int y)
{
	for (uint k=0; k<_blocks.size(); k++) moveBlock(k, x, y);
}

void Piece::moveBlock(uint k, int x, int y)
{
	_blocks[k]->sprite()->move(x + _binfo->toX(_i[k]), y + _binfo->toY(_j[k]));
}

Block *Piece::garbageBlock() const
{
	Block *b = new Block;
	b->setValue(_info->generateGarbageBlockType(_random), _binfo);
	return b;
}

Block *Piece::takeBlock(uint k)
{
	Block *b = _blocks.take(k);
	_blocks.insert(k, new Block);
	return b;
}

void Piece::show(bool show)
{
	for (uint k=0; k<_blocks.size(); k++) {
		if (show) _blocks[k]->sprite()->show();
        else _blocks[k]->sprite()->hide();
    }
}
