/*
 * Copyright (c) 2003 Benedikt Meurer (benedikt.meurer@unix-ag.uni-siegen.de)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* !HAVE_CONFIG_H */

#include <sys/stat.h>
#include <sys/types.h>

#include <fcntl.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#ifdef HAVE_MMAP
#include <sys/mman.h>
#endif

#include <glib.h>
#include <libxfce4util/util.h>

#include "xfprinter.h"
#include "xfprinterlist.h"

static void load_printers_from_file(const gchar *, XfPrinterList *);
static void save_printers_to_file(const gchar *, XfPrinterList *);


XfPrinterList *
xfprinterlist_load(void)
{
	XfPrinterList *list;
	gchar *path;

	list = g_new0(XfPrinterList, 1);

	path = xfce_get_userfile("printers.xml", NULL);
	load_printers_from_file(path, list);
	g_free(path);

	/* XXX - if no default printer given, try envvar $PRINTER */

	return(list);
}

void
xfprinterlist_save(XfPrinterList *list)
{
	gchar *path;

	g_return_if_fail(list != NULL);

	path = xfce_get_userfile("printers.xml", NULL);
	save_printers_to_file(path, list);
	g_free(path);
}

void
xfprinterlist_free(XfPrinterList *list)
{
	g_return_if_fail(list != NULL);

	/* XXX */
}

static gint
cmp_pr2str(const XfPrinter *pr, const gchar *alias)
{
	return(g_ascii_strcasecmp(pr->alias, alias));
}

static gint
cmp_pr2pr(const XfPrinter *pr1, const XfPrinter *pr2)
{
	return(g_ascii_strcasecmp(pr1->alias, pr2->alias));
}

void
xfprinterlist_insert(XfPrinterList *list, XfPrinter *pr)
{
	GList *entry;

	g_return_if_fail(list != NULL);
	g_return_if_fail(pr != NULL);

	entry = g_list_find_custom(list->printers, pr->alias,
		(GCompareFunc)cmp_pr2str);

	if (entry != NULL) {
		/* Overwrite existing entries */
		if (list->default_printer == entry->data)
			list->default_printer = NULL;

		xfprinter_free(entry->data);

		entry->data = pr;
	}
	else {
		list->printers = g_list_insert_sorted(list->printers,
			pr, (GCompareFunc)cmp_pr2pr);
	}
}

gboolean
xfprinterlist_is_empty(const XfPrinterList *list)
{
	g_return_val_if_fail(list != NULL, TRUE);

	return(list->printers == NULL);
}

XfPrinter *
xfprinterlist_lookup(XfPrinterList *list, const gchar *alias)
{
	GList *entry;

	g_return_val_if_fail(alias != NULL, NULL);
	g_return_val_if_fail(list != NULL, NULL);

	entry = g_list_find_custom(list->printers, alias,
		(GCompareFunc)cmp_pr2str);

	return(entry != NULL ? (XfPrinter *)entry->data : NULL);
}

XfPrinter *
xfprinterlist_get_default(XfPrinterList *list)
{
	g_return_val_if_fail(list != NULL, NULL);

	return(list->default_printer);
}

void
xfprinterlist_set_default(XfPrinterList *list, XfPrinter *pr)
{
	g_return_if_fail(list != NULL);

	if (g_list_find(list->printers, pr) != NULL)
		list->default_printer = pr;
}

gboolean
xfprinterlist_is_default(XfPrinterList *list, XfPrinter *pr)
{
	g_return_val_if_fail(list != NULL, FALSE);
	g_return_val_if_fail(pr != NULL, FALSE);

	return(list->default_printer == pr);
}

gboolean
xfprinterlist_is_default_by_alias(XfPrinterList *list, const gchar *alias)
{
	g_return_val_if_fail(list != NULL, FALSE);
	g_return_val_if_fail(alias != NULL, FALSE);

	return(list->default_printer && !strcmp(list->default_printer->alias,
			alias));
}

static void
add_alias(XfPrinter *pr, GList **aliases)
{
	*aliases = g_list_append(*aliases, pr->alias);
}

GList *
xfprinterlist_get_aliases(XfPrinterList *list)
{
	GList *aliases;

	g_return_val_if_fail(list != NULL, NULL);

	aliases = NULL;
	g_list_foreach(list->printers, (GFunc)add_alias, &aliases);

	return(aliases);
}

void
xfprinterlist_foreach(XfPrinterList *list, XfPrinterListFunc func,
	gpointer user_data)
{
	g_return_if_fail(func != NULL);
	g_return_if_fail(list != NULL);

	g_list_foreach(list->printers, (GFunc)func, user_data);
}

XfPrinter *
xfprinterlist_first(XfPrinterList *list)
{
	g_return_val_if_fail(list != NULL, NULL);

	list->current = g_list_first(list->printers);

	return(list->current ? (XfPrinter *)list->current->data : NULL);
}

XfPrinter *
xfprinterlist_next(XfPrinterList *list)
{
	g_return_val_if_fail(list != NULL, NULL);

	if (list->current)
		list->current = g_list_next(list->current);

	return(list->current ? (XfPrinter *)list->current->data : NULL);
}

void
xfprinterlist_remove(XfPrinterList *list, XfPrinter *pr)
{
	g_return_if_fail(list != NULL);
	g_return_if_fail(pr != NULL);

	if (list->default_printer == pr)
		list->default_printer = NULL;
	
	list->printers = g_list_remove(list->printers, pr);

	xfprinter_free(pr);
}

/******
  * XML parser
 */
typedef enum _PrinterParserState PrinterParserState;
enum _PrinterParserState
{
	START,
	PRINTERS
};

typedef struct _PrinterParser PrinterParser;
struct _PrinterParser
{
	gchar			*defpr;
	XfPrinterList		*list;
	PrinterParserState	state;
};

static void
start_element_handler(GMarkupParseContext *ctx, const gchar *element_name,
	const gchar **attr_names, const gchar **attr_values,
	gpointer user_data, GError **error)
{
	PrinterParser *parser;
	const gchar *alias;
	const gchar *name;
	const gchar *type;
	XfPrinter *pr;
	gint i;
	
	parser = (PrinterParser *)user_data;

	switch (parser->state) {
	case START:
		if (strcmp(element_name, "printers") != 0)
			break;
			
		parser->state = PRINTERS;

		for (i = 0; attr_names[i] != NULL; i++)
			if (!strcmp(attr_names[i], "default"))
				parser->defpr = g_strdup(attr_values[i]);
		break;

	case PRINTERS:
		if (strcmp(element_name, "printer") != 0)
			break;
			
		alias = NULL;
		name = NULL;
		pr = NULL;
		type = NULL;

		for (i = 0; attr_names[i] != NULL; i++) {
			if (!strcmp(attr_names[i], "alias"))
				alias = attr_values[i];
			else if (!strcmp(attr_names[i], "name"))
				name = attr_values[i];
			else if (!strcmp(attr_names[i], "type"))
				type = attr_values[i];
		}

		if (type != NULL) {
			if (!strcmp(type, "lp"))
				pr = xfprinter_new(alias, name);
			else if (!strcmp(type, "ps"))
				pr = xfprinter_new_ps(alias, name);
		}

		if (pr != NULL)
			xfprinterlist_insert(parser->list, pr);

		break;

	default:
		g_warning("start unknown element \"%s\"", element_name);
		break;
	}
}

static void
end_element_handler(GMarkupParseContext *ctx, const gchar *element_name,
	gpointer user_data, GError **error)
{
	PrinterParser *parser;

	parser = (PrinterParser *)user_data;

	switch (parser->state) {
	case START:
		/* XXX - this shouldn't happen */
		break;

	case PRINTERS:
		if (strcmp(element_name, "printers") != 0)
			break;

		parser->state = START;

		if (parser->defpr != NULL) {
			xfprinterlist_set_default(parser->list,
				xfprinterlist_lookup(parser->list,
					parser->defpr));
			g_free(parser->defpr);
			parser->defpr = NULL;
		}
		break;
		
	default:
		g_warning("end unknown element \"%s\"", element_name);
		break;
	}
}

static void
error_handler(GMarkupParseContext *ctx, GError *error, gpointer user_data)
{
	if (error && error->message)
		g_warning(" %s", error->message);
}

static GMarkupParser markup_parser = {
	start_element_handler,
	end_element_handler,
	NULL,
	NULL,
	error_handler
};

static void
load_printers_from_file(const gchar *filename, XfPrinterList *list)
{
	GMarkupParseContext *ctx;
	PrinterParser parser;
	gchar *contents;
	GError *error;
	struct stat sb;
#ifdef HAVE_MMAP
	void *addr;
#endif
	int fd;

	error = NULL;

	parser.defpr = NULL;
	parser.list = list;
	parser.state = START;

#ifdef O_SHLOCK
	if ((fd = open(filename, O_RDONLY | O_SHLOCK, 0)) < 0)
#else
	if ((fd = open(filename, O_RDONLY, 0)) < 0)
#endif
		return;

	if (fstat(fd, &sb) < 0)
		goto finished;

#ifdef HAVE_MMAP
	/* Try to mmap(2) the config file, as this save us a lot of
	 * kernelspace -> userspace copying
	 */
#ifdef MAP_FILE
	addr = mmap(NULL, sb.st_size, PROT_READ, MAP_FILE | MAP_SHARED, fd, 0);
#else
	addr = mmap(NULL, sb.st_size, PROT_READ, MAP_SHARED, fd, 0);
#endif

	if (addr != NULL) {
		/* nice, mmap did the job */
		contents = addr;
	}
	else {
#endif
		if ((contents = malloc(sb.st_size * sizeof(gchar))) == NULL)
			goto finished;

		/* read(2):
		 * --------
		 * The system guarantees to read the number of bytes requested
		 * if the descriptor references a normal file that has that 
		 * many bytes left before the end-of-file.
		 */
		if (read(fd, contents, sb.st_size) < sb.st_size)
			goto finished2;
#ifdef HAVE_MMAP
	}
#endif

	ctx = g_markup_parse_context_new(&markup_parser, 0, &parser, NULL);

	if (!g_markup_parse_context_parse(ctx, contents, sb.st_size, NULL))
		goto finished3;
	
	if (!g_markup_parse_context_end_parse(ctx, NULL))
		goto finished3;

finished3:
	g_markup_parse_context_free(ctx);

finished2:
#ifdef HAVE_MMAP
	if (addr != NULL)
		munmap(addr, sb.st_size);
	else
#endif
		free(contents);

finished:
	(void)close(fd);
}

static void
save_printer_to_file(XfPrinter *pr, gpointer user_data)
{
	FILE *fp;

	fp = (FILE *)user_data;

	fprintf(fp, "\t<printer alias=\"%s\" name=\"%s\" ", pr->alias,pr->name);

	if (pr->postscript)
		fprintf(fp, "type=\"ps\"");
	else
		fprintf(fp, "type=\"lp\"");

	fprintf(fp, " />\n");
}

static void
save_printers_to_file(const gchar *filename, XfPrinterList *list)
{
	XfPrinter *pr;
	FILE *fp;
	int fd;

#ifdef O_EXLOCK
	if ((fd = open(filename, O_CREAT|O_EXLOCK|O_TRUNC|O_WRONLY, S_IRUSR | S_IWUSR))< 0)
#else
	if ((fd = open(filename, O_CREAT | O_TRUNC | O_WRONLY, S_IRUSR | S_IWUSR)) < 0)
#endif
		return;

	if ((fp = fdopen(fd, "w")) == NULL) {
		(void)close(fd);
		return;
	}

	/* write header */
	fprintf(fp,
		"<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
		"<!DOCTYPE printers SYSTEM \"printers.dtd\">\n"
		"\n");

	if ((pr = xfprinterlist_get_default(list)))
		fprintf(fp, "<printers default=\"%s\">\n", pr->alias);
	else
		fprintf(fp, "<printers>\n");

	/* write list of printers */
	xfprinterlist_foreach(list, save_printer_to_file, fp);

	/* write footer */
	fprintf(fp, "</printers>\n");

	(void)fflush(fp);
	(void)fclose(fp);
}

