/* xoreos-tools - Tools to help with xoreos development
 *
 * xoreos-tools is the legal property of its developers, whose names
 * can be found in the AUTHORS file distributed with this source
 * distribution.
 *
 * xoreos-tools is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * xoreos-tools is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xoreos-tools. If not, see <http://www.gnu.org/licenses/>.
 */

/** @file
 *  Resolve a hash found in hashed Dragon Age: Origins and Dragon Age 2
 *  archives back to the filename.
 */

#include "src/common/util.h"
#include "src/common/binsearch.h"

#include "src/files_dragonage.h"

typedef Common::BinSearchValue<uint64, const char *> DragonAgeFileHash;

/** All currently known Dragon Age: Origins and Dragon Age 2 file names,
 *  together with their FNV64 hashes.
 *
 *  Based on fnvdump-20110527.filelist, courtesy of tazpn, Thought Process,
 *  4Aces, a friendly donor, and Mephales. fnvdump-20110527.filelist can
 *  be found in fnvdump-20110527.zip on the BioWare Social Network page
 *  for pyGFF (<http://social.bioware.com/project/1936/>).
 *
 *  Note: For the binary search to work, this list needs to stay sorted by
 *  hash value!
 */
static const DragonAgeFileHash kDragonAgeFilesHashes[] = {
	{0x000023C542D3A2D7ULL, "145396997.ogg"},
	{0x000058C598998F7AULL, "vxca_caverns1_mines_ikf_0_0.vlm"},
	{0x0000816D7858B6D5ULL, "6032202_m.fxe"},
	{0x00008AD55410A2C2ULL, "253679411.ogg"},
	{0x0000FB05E0D9A812ULL, "vxlt_0_lowtown_h25_0_0.vlm"},
	{0x000154378DC66970ULL, "6153973_m.fxe"},
	{0x000223914E32B0F7ULL, "art\\vfx\\combat\\vfx_r_stun_grena_proj_p.mmh"},
	{0x00026CA3C4FC1740ULL, "w_encounter2_camp_27_0.phy"},
	{0x000293524CE74D78ULL, "211629641.ogg"},
	{0x00029AB70FAA0D5FULL, "art\\characters\\creatures\\rubblegolem\\animation\\stage1light\\chestcast.gad"},
	{0x0002EE258B36F956ULL, "6094605_m.fxe"},
	{0x0002F52B87C0D0B1ULL, "6129133_m.fxe"},
	{0x00032C77163301CEULL, "mag110_vincento6216162.cl"},
	{0x0003546C439AB2B5ULL, "vxlt_foundry_n_xm_0_0.vlm"},
	{0x0003683F1DE7D362ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\0ff_45se.phy"},
	{0x0004C93D6A7C586AULL, "art\\vfx\\cinematic\\vfx_d_aoe_shockwave_p.impa.gad"},
	{0x00055A53B019426FULL, "6040938_f.fxe"},
	{0x0005923A728B292BULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\baseswordshield\\combat\\meleeattackv1.gad"},
	{0x0005C286F484043EULL, "follower_banter6148479.cl"},
	{0x0006C6D7418AF881ULL, "grey_wardens.stg"},
	{0x00076CA1071F84B6ULL, "mer341_earl6054222.cl"},
	{0x0007701A829773ECULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\1wi_galowint.phy"},
	{0x0007D4A0FE6A9980ULL, "991581810.ogg"},
	{0x0007E97BA5508DB9ULL, "hlac_w_sundermount_-131074_0y.dds"},
	{0x00080FCC98BF7103ULL, "vxlt_0_lowtown_a2_n_do0_0_0.vlm"},
	{0x00087195F8744707ULL, "gen00fl_anders6111103.cl"},
	{0x00088BA3C500B7F8ULL, "w_encounter2_camp_27_0.rml"},
	{0x0008943BD1D0C3EFULL, "llac_lt_0_lowtown_a2_-131071_0.dds"},
	{0x00098FB70FB0194CULL, "art\\characters\\creatures\\rubblegolem\\animation\\stage1light\\chestcast.evt"},
	{0x000A20B161C3C7A9ULL, "art\\vfx\\combat\\vfx_c_debili_proj_p.mmh"},
	{0x000A46DD2C4F6ECBULL, "art\\vfx\\cinematic\\vfx_ship_wake_p.dura.evt"},
	{0x000A588D5292AE5DULL, "166862660.ogg"},
	{0x000A891F7A385692ULL, "one100_solivitus6060372.cl"},
	{0x000AAA9B2660F775ULL, "gen_im_arm_glv_med_act2_01.uti"},
	{0x000ACB2763CA1E3EULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\ambient\\am_std_beg_lp.ani"},
	{0x000B0D8BF2AED7EFULL, "qun110_petrice6018526.cl"},
	{0x000B4C8DE698AA43ULL, "364196154.ogg"},
	{0x000B8F3A728F8F90ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\baseswordshield\\combat\\meleeattackv1.evt"},
	{0x000BAC3D6A824501ULL, "art\\vfx\\cinematic\\vfx_d_aoe_shockwave_p.impa.evt"},
	{0x000BBC77F664E751ULL, "747383522.ogg"},
	{0x000C004545999560ULL, "da2_prc_seb_en-us.cre"},
	{0x000CBB4520CC7EFDULL, "28876310.ogg"},
	{0x000D393A73708095ULL, "vxht_0_hightown_a2_b9i_0_0.vlm"},
	{0x000D85464CAE0335ULL, "6103310_f.fxe"},
	{0x000DCF332026338FULL, "static_static_p31.psh.fxo.dx11"},
	{0x000E3889E8BE20A7ULL, "6174227_m.fxe"},
	{0x000E6F9A4581E5B7ULL, "art\\characters\\playercharacter\\humanmale\\arm_lgte.mmh"},
	{0x000E739A4581ECC4ULL, "art\\characters\\playercharacter\\humanmale\\arm_lgte.mao"},
	{0x000E859A45820B29ULL, "art\\characters\\playercharacter\\humanmale\\arm_lgte.msh"},
	{0x000EC6B5DD954981ULL, "vxw_bonepitmining_a2_e4_0_0.vlm"},
	{0x000EDD0496EFA897ULL, "6126758_m.fxe"},
	{0x000F3F46A4BF6171ULL, "t3_arm_slr.tnt"},
	{0x000FB978BCBAA7E3ULL, "art\\characters\\creatures\\ragedemon\\animation\\die_lp_v1.ani"},
	{0x000FC4CD04078490ULL, "gen00fl_aveline6122738.cl"},
	{0x000FCA4BC549B6F5ULL, "vxlt_0_lowtown_a2_ehy_0_0.vlm"},
	{0x001080714944FDCEULL, "6053103_m.fxe"},
	{0x0010BE39535B7A6EULL, "6125185_m.fxe"},
	{0x0010C6D0061874BAULL, "vxlt_0_lowtown_a2_gnv_0_0.vlm"},
	{0x00112E739BBA0E9EULL, "hlac_lt_0_lowtown_a2_n_c_65535_3y.dds"},
	{0x001152F33057DCB6ULL, "vxlt_gallowstemplar_5yb_0_0.vlm"},
	{0x00115FDD2C55B78CULL, "art\\vfx\\cinematic\\vfx_ship_wake_p.dura.gad"},
	{0x001225B0EF71BE05ULL, "6126352_m.fxe"},
	{0x00127E26D28D1E4AULL, "6135698_m.fxe"},
	{0x0012D1EFAA79F394ULL, "cth_nb5.tnt"},
	{0x00130334E047807AULL, "vxlt_0_lowtown_a2_n_e1j_0_0.vlm"},
	{0x0013157FE714D0CDULL, "585520048.ogg"},
	{0x0013A918EBCC12E9ULL, "gen00fl_aveline6053600.cl"},
	{0x0013CF421D07FF84ULL, "lt_0_lowtown_docks_a3_184_0.mmh"},
	{0x0013D3421D0806D7ULL, "lt_0_lowtown_docks_a3_184_0.mao"},
	{0x0013E5421D082526ULL, "lt_0_lowtown_docks_a3_184_0.msh"},
	{0x0014063BCDD43F1CULL, "headgrab_flip.dds"},
	{0x001431B76B6EF550ULL, "vxlt_undercitydungeon_188_0_0.vlm"},
	{0x001492FCC3E5B650ULL, "105956307.ogg"},
	{0x0015059462359891ULL, "62300253.ogg"},
	{0x00151E52E1AA7EB5ULL, "vxlt_gallowstemplar_f_5wt_0_0.vlm"},
	{0x00151E9A8EF1E6BEULL, "llac_ht_playershouse_n_855_18.dds"},
	{0x00152DB9F56C1C12ULL, "vxca_cavern1_haunted_ei8_0_0.vlm"},
	{0x001677B70FBA7AA5ULL, "art\\characters\\creatures\\rubblegolem\\animation\\stage1light\\chestcast.ani"},
	{0x00169440A3E8D8A2ULL, "mag330_walter6045963.cl"},
	{0x0016A0EE214B301CULL, "vxdr_0_deeproads_cvk_0_0.vlm"},
	{0x0016A3BB48E30FF6ULL, "vxlt_gallowstemplar_f_47z_0_0.vlm"},
	{0x001708A8FDED2A42ULL, "vxht_mansion2b_61_0_0.vlm"},
	{0x00172027EC26E3CCULL, "mrl150_merrill6088663.cl"},
	{0x001735E21A4F64E6ULL, "6107303_m.fxe"},
	{0x00173F0E5AA2D86DULL, "gen00fl_sebastian6178347.cl"},
	{0x0017523170273A9CULL, "6080771_f.fxe"},
	{0x00175EDD2C5A218AULL, "art\\vfx\\cinematic\\vfx_ship_wake_p.dura.ani"},
	{0x0017C5E3A27DAA6FULL, "amb_scene_sleeping_snore.bnk"},
	{0x0017E32763D4D10FULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\ambient\\am_std_beg_lp.evt"},
	{0x0017F0839A77765DULL, "vxca_caverns1_mines_ei3_0_0.vlm"},
	{0x0018115DC791A492ULL, "6082691_m.fxe"},
	{0x001853AB659092B9ULL, "rightarrow.dds"},
	{0x00187AA89ADB7CC6ULL, "96555776.ogg"},
	{0x0018CD6178758E85ULL, "6095519_m.fxe"},
	{0x0018D001972FADF2ULL, "ash_01.dds"},
	{0x001927FFEEF5AB38ULL, "dre000_sibling_has_blight6036454.cl"},
	{0x0019827F0005258CULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\0wi_basic_1.mao"},
	{0x0019A43D6A8E7550ULL, "art\\vfx\\cinematic\\vfx_d_aoe_shockwave_p.impa.ani"},
	{0x0019C134E21397E3ULL, "llac_ht_mansion2b_818_16.dds"},
	{0x0019DDA1A7A7F169ULL, "411858169.ogg"},
	{0x001A43F139F053F6ULL, "6107164_f.fxe"},
	{0x001A66DBC144C270ULL, "vxht_0_hightown_n_aut_0_0.vlm"},
	{0x001A77B8A9B3C8E7ULL, "6142789_m.fxe"},
	{0x001B250033CCA938ULL, "6156350_m.fxe"},
	{0x001B33289076A5E3ULL, "art\\levels\\races\\elf\\props\\dalish\\prp_postc.msh"},
	{0x001B41289076BD3DULL, "art\\levels\\races\\elf\\props\\dalish\\prp_postc.mmh"},
	{0x001B45289076C44EULL, "art\\levels\\races\\elf\\props\\dalish\\prp_postc.mao"},
	{0x001B62B0E0069490ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\0ww_ceilingcap.mao"},
	{0x001B66B0E0069BA3ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\0ww_ceilingcap.mmh"},
	{0x001B6F7DE45B7487ULL, "476576503.ogg"},
	{0x001B74B0E006B3FDULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\0ww_ceilingcap.msh"},
	{0x001C41F2756B8BCCULL, "vxht_keep_3g3_0_0.vlm"},
	{0x001C4E08C9E0A178ULL, "6141732_m.fxe"},
	{0x001CC21555DC8275ULL, "qun110_petrice6205991.cl"},
	{0x001DBCD0664F6AB4ULL, "vxdr_primevaltomb_6sk_0_0.vlm"},
	{0x001DFB72FA149F85ULL, "vxdr_primevalentrance_2qj_0_2.vlm"},
	{0x001E027EDAED2642ULL, "mag101_alain6052992.cl"},
	{0x001E28CB7B061CC1ULL, "gen00fl_aveline6139566.cl"},
	{0x001EB81553E2D67CULL, "vxht_mansion2b_1pt_0_0.vlm"},
	{0x001EE42763DAF1B8ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\ambient\\am_std_beg_lp.gad"},
	{0x001F11EF8F4DFC32ULL, "643291818.ogg"},
	{0x001F96FA49E53804ULL, "vxlt_gallowstemplar_5c2_0_0.vlm"},
	{0x001FA1A972DFBC37ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\abomination_sailor.gad"},
	{0x00202E1FDE123E07ULL, "art\\vfx\\combat\\vfx_ha_bd_gthrow_act_c.anb"},
	{0x002030400135DD0CULL, "zz_dae_debug6029804.cl"},
	{0x0020A6F624463F27ULL, "vxlt_0_lowtown_docks_a3_n_aqa_0_0.vlm"},
	{0x0020CBD49A7979E8ULL, "350740614.ogg"},
	{0x0021603C5781AEEAULL, "6107075_m.fxe"},
	{0x0021B162A89704AFULL, "class_abilities_rogue.gda"},
	{0x0021F0847D52F7D9ULL, "203975825.ogg"},
	{0x002216C8AC1D30BFULL, "gen00fl_anders6125214.cl"},
	{0x00222348434A4CF2ULL, "37734295.ogg"},
	{0x00225105228C69A2ULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_nos06b.phy"},
	{0x002277A3B1CFC16BULL, "270529028.ogg"},
	{0x0022A12276ABFD59ULL, "425294863.ogg"},
	{0x00232AB58903105AULL, "460069760.ogg"},
	{0x0023A18C5FB66AE7ULL, "hlac_lt_0_lowtown_n_0_4y.dds"},
	{0x0023D078BCCB77CDULL, "art\\characters\\creatures\\ragedemon\\animation\\die_lp_v1.gad"},
	{0x0023DF1A5429E0FFULL, "w_bonepit_123_0.phy"},
	{0x0023F5A8A2C3D4A5ULL, "35985850.ogg"},
	{0x0024C53F25972C87ULL, "gen00fl_merrill6105618.cl"},
	{0x0024D01A7D5E883CULL, "art\\vfx\\combat\\vfx_r_brd_psong_loop_mdl_c.anb"},
	{0x0024DB7DFE428D5BULL, "follower_banter6161217.cl"},
	{0x00254E5A625AC565ULL, "12127855.ogg"},
	{0x002571A327C86AB3ULL, "231971006.ogg"},
	{0x00257CA41C82AE87ULL, "6163595_m.fxe"},
	{0x0025B6A972E44BC4ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\abomination_sailor.evt"},
	{0x0025C426446AD38DULL, "785649323.ogg"},
	{0x00260E22A2D86169ULL, "539311964.ogg"},
	{0x002618C156235BFBULL, "mer240_jansen6050408.cl"},
	{0x0026348732202654ULL, "dae000_player_generic_find6179052.cl"},
	{0x00264D270259CCFCULL, "vxlt_0_lowtown_a2_n_gnc_0_0.vlm"},
	{0x00264EE2AB43CBFDULL, "mrl000_marethari6129284.cl"},
	{0x0026833AB987BD12ULL, "6085791_m.fxe"},
	{0x002731E864A1D4A5ULL, "vxht_keep_64n_0_0.vlm"},
	{0x002769D4B2DEF4CBULL, "ser270_emeric6149349.cl"},
	{0x002783141D4D1A70ULL, "ring_stolen_shadows.dds"},
	{0x002789A3C51B64D5ULL, "w_encounter2_camp_27_0.msh"},
	{0x00278BA3C51B684BULL, "w_encounter2_camp_27_0.mmh"},
	{0x002797A3C51B7CE8ULL, "w_encounter2_camp_27_0.mao"},
	{0x0027A98543B16D73ULL, "vxht_0_hightown_n_c68_0_0.vlm"},
	{0x0027EF45CEE302B9ULL, "vxlt_0_lowtown_n_fh5_0_0.vlm"},
	{0x00281D5D5A0F7AC4ULL, "art\\characters\\playercharacter\\humanmale\\boo_lgtb.msh"},
	{0x00282F5D5A0F992DULL, "art\\characters\\playercharacter\\humanmale\\boo_lgtb.mao"},
	{0x0028335D5A0FA066ULL, "art\\characters\\playercharacter\\humanmale\\boo_lgtb.mmh"},
	{0x0028433F1E06B016ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\0ff_45se.mao"},
	{0x00284F3F1E06C4B5ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\0ff_45se.mmh"},
	{0x0028513F1E06C82BULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\0ff_45se.msh"},
	{0x0028D69D941BA8DFULL, "mag291_ntbanter6056606.cl"},
	{0x0028D9B65CC05D32ULL, "llac_dr_dungeon_3216_0.dds"},
	{0x0028DCBE06380CD6ULL, "art\\characters\\playercharacter\\humanmale\\textures\\hm_hlh_lgta_0s.dds"},
	{0x0028FA07511EFB6AULL, "6040688_m.fxe"},
	{0x002955EC23DC2AABULL, "385617670.ogg"},
	{0x0029F034AA427EADULL, "608514247.ogg"},
	{0x002A43F92345D39BULL, "vxht_keep_n_6ji_0_0.vlm"},
	{0x002A5AACADE5DCE9ULL, "zz_dae_debug6047641.cl"},
	{0x002A5DDA8B9F46BAULL, "gen_im_trs_krk_junk2_03.uti"},
	{0x002A6F380AD2BC1EULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\r_die_lp_v1.ani"},
	{0x002AE278BB663F5FULL, "6063821_f.fxe"},
	{0x002B50699D0E1EC5ULL, "vxca_caverns1_mines_ikm_0_0.vlm"},
	{0x002C053BD3F9B5B6ULL, "hlac_lt_undercityserialkiller_-1_40z.dds"},
	{0x002C2EAD58458334ULL, "vxdr_primevaltomb_6vh_0_0.vlm"},
	{0x002C3AE74A8B164FULL, "w_bonepit_134_0.phy"},
	{0x002D6AC4E0AC271BULL, "908341707.ogg"},
	{0x002E4DB07FCF0C41ULL, "6019098_m.fxe"},
	{0x002E6A53B49F3802ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\urgent_run_enter.gad"},
	{0x002EC8A206FAE2F4ULL, "hlac_lt_0_lowtown_a2_n_c_131071_7z.dds"},
	{0x002EF7B1AC6FC59AULL, "841232134.ogg"},
	{0x002FD53E9EC6E030ULL, "hlac_ht_chantry_2534_41x.dds"},
	{0x00300C2DA908BAF8ULL, "6071146_f.fxe"},
	{0x00307D97DD7D1651ULL, "vxlt_darktown_4t6_0_1.vlm"},
	{0x00309F019743EC70ULL, "ash_01.mao"},
	{0x0030E89E7D164272ULL, "art\\characters\\weapons\\longswords\\textures\\w_lsw_ls107a_0n.dds"},
	{0x0031192CB30D2B31ULL, "vxlt_gallowstemplar_f_3s6_0_0.vlm"},
	{0x00312EE300415E85ULL, "6109697_f.fxe"},
	{0x0031684FE73956BBULL, "art\\characters\\creatures\\strider\\animation\\groundstompenter.ani"},
	{0x0031E31D496F9C10ULL, "vxht_0_hightown_caq_0_0.vlm"},
	{0x0031EE6CC31C7D33ULL, "6108888_m.fxe"},
	{0x0031FFA4634EEC27ULL, "hlac_lt_0_lowtown_a2_-131071_0x.dds"},
	{0x00322A5D5A17E531ULL, "art\\characters\\playercharacter\\humanmale\\boo_lgtb.phy"},
	{0x003246D9905DAB42ULL, "hlac_w_encounter3_-131074_0z.dds"},
	{0x00331DB161E6C7DEULL, "art\\vfx\\combat\\vfx_c_debili_proj_p.anb"},
	{0x00332F6BED2345DFULL, "465890089.ogg"},
	{0x0033558038859080ULL, "6083597_m.fxe"},
	{0x0033781207307763ULL, "art\\levels\\races\\natural\\kirkcave\\props\\_textures\\prp_rockblend_drelief.dds"},
	{0x0033CEA972F0B1BDULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\abomination_sailor.ani"},
	{0x0033D5EA417CAB38ULL, "6087320_m.fxe"},
	{0x003444511391D53CULL, "gen00fl_varric6097185.cl"},
	{0x003448CB51C1FD6FULL, "hlac_w_encounter1_-131071_0z.dds"},
	{0x0034CAA15281391CULL, "hlac_lt_0_lowtown_docks_65537_0z.dds"},
	{0x0034D43D7E63A75DULL, "vxca_caverns1_mines_ec1_0_0.vlm"},
	{0x0034E01A54385504ULL, "w_bonepit_123_0.mmh"},
	{0x0034E41A54385C57ULL, "w_bonepit_123_0.mao"},
	{0x0034F61A54387AA6ULL, "w_bonepit_123_0.msh"},
	{0x0035653DADF62DBAULL, "6039688_m.fxe"},
	{0x0035776F9BEB2863ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\9uw_tarphangb.mao"},
	{0x00357B6F9BEB2EB0ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\9uw_tarphangb.mmh"},
	{0x0035896F9BEB46CAULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\9uw_tarphangb.msh"},
	{0x00358D53B4A592F9ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\urgent_run_enter.evt"},
	{0x0035A3C55697ED25ULL, "vxht_keep_61m_0_0.vlm"},
	{0x0035DEF54A285DDFULL, "gen00fl_merrill6121256.cl"},
	{0x0035ECE2F84845A3ULL, "6065910_m.fxe"},
	{0x003655017F0E091BULL, "vxlt_0_lowtown_a2_eor_0_0.vlm"},
	{0x0036B057F20896D1ULL, "73253986.ogg"},
	{0x00371275793CA396ULL, "dae000_gen_turn_in_16178395.cl"},
	{0x00378C58017C5323ULL, "gen00fl_fenris6113868.cl"},
	{0x0037995F215677C2ULL, "vxht_0_hightown_a3_bb6_0_0.vlm"},
	{0x00382B0522927834ULL, "follower_banter6171554.cl"},
	{0x0038BC9B587FFDF0ULL, "gen_im_wep_rog_arc_act0_03.uti"},
	{0x0038BF621EA3E6EFULL, "art\\characters\\weapons\\greatswords\\textures\\w_gsw_gs103a_0n.dds"},
	{0x0038DFBD861C552FULL, "dae021_chantry_initiate_mob6145199.cl"},
	{0x0039617FDC3433ADULL, "dr_primevaltomb_658.rml"},
	{0x0039804B1A65E5F7ULL, "6110275_f.fxe"},
	{0x0039C3C329123AAFULL, "vxlt_hangedman_1qv_0_0.vlm"},
	{0x003A987FF637D907ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\idunna_prostitute_action_sit_loop.gad"},
	{0x003AA0B73B59B883ULL, "qun131cr_taskmaster.utc"},
	{0x003ADDF09E8D22CAULL, "vxlt_hangedman_n_5c_0_0.vlm"},
	{0x003B9CE40977B8FFULL, "vxlt_0_lowtown_fge_0_0.vlm"},
	{0x003BAEF9559038E2ULL, "dre000_vampire6034317.cl"},
	{0x003BDE97682E1B95ULL, "vxdr_primevaltomb_3hn_0_0.vlm"},
	{0x003C53CACDA7F309ULL, "art\\vfx\\combat\\vfx_m_wpn_fire_crust_c.mmh"},
	{0x003CA53D9A6655E3ULL, "gen00fl_isabela_init6110275.cl"},
	{0x003CEC1932C40F11ULL, "vxht_brothel_n_5b7_0_0.vlm"},
	{0x003D113E9C145BD9ULL, "vxdr_0_deeproads_ck9_0_0.vlm"},
	{0x003D44EADA8830D4ULL, "art\\vfx\\combat\\vfx_d_aoe_imp_p.anb"},
	{0x003DDDA762E5E295ULL, "follower_barks6153820.cl"},
	{0x003DFAD8E4BC0FC5ULL, "vxlt_0_lowtown_a2_n_e1a_0_0.vlm"},
	{0x003DFFDAAB99CF7FULL, "318769168.ogg"},
	{0x003E704FE743EEB2ULL, "art\\characters\\creatures\\strider\\animation\\groundstompenter.evt"},
	{0x003E88380AE38F98ULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\r_die_lp_v1.gad"},
	{0x003E9ED2FD062390ULL, "vxht_0_hightown_a3_aqv_0_0.vlm"},
	{0x003EB0C2FD1B4609ULL, "dae170ar_gallows.lst"},
	{0x003EDB8AB8AAA0E1ULL, "vxca_caverns1_mines_im7_0_0.vlm"},
	{0x003F69A68C3B720DULL, "follower_banter6173010.cl"},
	{0x003FA0BFF2377705ULL, "6102797_f.fxe"},
	{0x003FB9BAD9C39DA5ULL, "heallight3.dds"},
	{0x003FCD67074CFE0AULL, "34683419.ogg"},
	{0x0040A1C054A8C586ULL, "vxca_caverns1_mines_k2n_0_1.vlm"},
	{0x004146CCAF787990ULL, "art\\characters\\playercharacter\\elffemale\\mt_ant_ef06.mmh"},
	{0x004154CCAF7891AAULL, "art\\characters\\playercharacter\\elffemale\\mt_ant_ef06.msh"},
	{0x0041B0A1BD365856ULL, "341869908.ogg"},
	{0x0041F684F72D799CULL, "vxdr_0_deeproads_cen_0_0.vlm"},
	{0x0042D339F761B03EULL, "vxw_bonepitmining_a2_1lg_0_0.vlm"},
	{0x004352EC3AB3D01BULL, "775212395.ogg"},
	{0x00439553B4B1DDE8ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\urgent_run_enter.ani"},
	{0x0043D265B7DBC76DULL, "6153756_m.fxe"},
	{0x004416E276753B28ULL, "6064165_m.fxe"},
	{0x0044AFBE0634098EULL, "558687102.ogg"},
	{0x0044C829C0BB970AULL, "llac_lt_0_lowtown_docks_a3_131071_0.dds"},
	{0x0044CD4AC08410CAULL, "hlac_ht_keep_n_4519_10z.dds"},
	{0x0044E5F4972DCA79ULL, "vxlt_undercityencounter_wj_0_0.vlm"},
	{0x0044EA1592AE002BULL, "355069094.ogg"},
	{0x004588C2FD212140ULL, "dae170ar_gallows.ncs"},
	{0x00458F4FE74A4245ULL, "art\\characters\\creatures\\strider\\animation\\groundstompenter.gad"},
	{0x0045C39A7447B334ULL, "6150183_m.fxe"},
	{0x00460ED72C76E873ULL, "hlac_lt_0_lowtown_docks_n_-3_6z.dds"},
	{0x0046EFB42AA77BA6ULL, "487737712.ogg"},
	{0x00472F872A501F58ULL, "hlac_lt_0_lowtown_a3_n_c_65535_7x.dds"},
	{0x004794CC37697B5EULL, "6105610_f.fxe"},
	{0x0047C3A7D825B658ULL, "664692917.ogg"},
	{0x0048380522AD0EF5ULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_nos06b.mmh"},
	{0x00483A0522AD126BULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_nos06b.msh"},
	{0x00483B95DCA2FE64ULL, "follower_banter6163562.cl"},
	{0x0048471A82CEC4B7ULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\1wi_galowint.mmh"},
	{0x00484B1A82CECBC4ULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\1wi_galowint.mao"},
	{0x00485D1A82CEEA29ULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\1wi_galowint.msh"},
	{0x0048644079FF4CC8ULL, "vxca_caverns1_mines_im2_0_0.vlm"},
	{0x00492B1FDE353E08ULL, "art\\vfx\\combat\\vfx_ha_bd_gthrow_act_c.mmh"},
	{0x004946BF3873F4A1ULL, "hlac_ca_cavern1_haunted_24426_7z.dds"},
	{0x00499694061BFD81ULL, "6099209_m.fxe"},
	{0x0049D139505C1271ULL, "6084751_m.fxe"},
	{0x004A1AE58D64EA5BULL, "vxlt_gallowstemplar_f_5yx_0_0.vlm"},
	{0x004A23EADA928297ULL, "art\\vfx\\combat\\vfx_d_aoe_imp_p.mmh"},
	{0x004A789F2C9725B4ULL, "mrl150tr_undead_pre_camp.ncs"},
	{0x004AEC911FC78184ULL, "vxlt_gallowstemplar_5r1_0_0.vlm"},
	{0x004B01F98A0A5648ULL, "6175160_m.fxe"},
	{0x004BADA854448F04ULL, "zz_followers_banter6158799.cl"},
	{0x004BE3D9DB10F2A7ULL, "350863035.ogg"},
	{0x004C56DB5143BAF7ULL, "art\\levels\\races\\ferelden\\props\\lowerclass\\prp_scraplight3a.msh"},
	{0x004C68DB5143D95AULL, "art\\levels\\races\\ferelden\\props\\lowerclass\\prp_scraplight3a.mao"},
	{0x004C6CDB5143E069ULL, "art\\levels\\races\\ferelden\\props\\lowerclass\\prp_scraplight3a.mmh"},
	{0x004C776E4670B73FULL, "817862058.ogg"},
	{0x004C899D00441B00ULL, "vxca_caverns1_icy_0_0.vlm"},
	{0x004D4D67AAA0B961ULL, "vxca_caverns1_mines_ij6_0_0.vlm"},
	{0x004D4FC6E09A0A41ULL, "vxdr_0_deeproads_f5f_0_0.vlm"},
	{0x004DB7C787E55088ULL, "6119706_m.fxe"},
	{0x004DE4ACE0B35C66ULL, "6202275_m.fxe"},
	{0x004E455F63411FA9ULL, "lt_0_lowtown_docks_a3_n_32_0_lc.mmh"},
	{0x004ECED115B0F2DDULL, "365834235.ogg"},
	{0x004F58265D1712BEULL, "6134994_m.fxe"},
	{0x004F7511D48BD696ULL, "vxlt_hangedman_n_2dl_0_0.vlm"},
	{0x004F857FF64A154DULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\idunna_prostitute_action_sit_loop.ani"},
	{0x004FA9C2FD29ADAEULL, "dae170ar_gallows.are"},
	{0x0050025975EF2870ULL, "41536900.ogg"},
	{0x0050481970F5AAC9ULL, "vxdr_primevaltomb_85r_0_0.vlm"},
	{0x00508237CE26A505ULL, "art\\characters\\playercharacter\\humanfemale\\textures\\hf_cth_homa_0n.dds"},
	{0x0050A1B3B878AC27ULL, "188256296.ogg"},
	{0x0050B43620E6BAAFULL, "zz_one000_barber6179396.cl"},
	{0x0050F9AFE614EB1EULL, "mer300pt_mine_massacre.ncs"},
	{0x0051231C30AD8D15ULL, "694489519.ogg"},
	{0x00512F2AB1A7F42EULL, "dre000_dougal6131164.cl"},
	{0x0051BCB8E7985AACULL, "hlac_lt_gallowstemplar_f_2391_6x.dds"},
	{0x005201702AAA53F3ULL, "art\\characters\\playercharacter\\humanmale\\textures\\hn_glv_medp_0t.dds"},
	{0x00524442A9D1DFEDULL, "6156456_m.fxe"},
	{0x0052C99C87B2E2B1ULL, "art\\characters\\playercharacter\\humanmale\\textures\\hm_hlf_masg_0t.dds"},
	{0x005357D18923E2F9ULL, "vxlt_undercityencounter_ri_0_0.vlm"},
	{0x00542628D92EEBD7ULL, "zz_isa_debug6175468.cl"},
	{0x005499A8EA5ECDA1ULL, "and111_refugee_barks6111541.cl"},
	{0x0054CC1D2D8BDC23ULL, "vxlt_gallowscourtyard_n_651_0_0.vlm"},
	{0x0054DC0EE9550CAFULL, "hlac_lt_gallowscourtyard_-65536_1z.dds"},
	{0x005555E60260158FULL, "6149909_f.fxe"},
	{0x005581F36B3F9B7FULL, "6056840_m.fxe"},
	{0x005596342A4887CDULL, "1008660080.ogg"},
	{0x00560BC55111BB60ULL, "6170277_m.fxe"},
	{0x00566E8F96A06971ULL, "vxlt_gallowstemplar_n_64z_0_0.vlm"},
	{0x0056C4A916F4BD09ULL, "vxht_bartrand_n_5t9_0_0.vlm"},
	{0x0056D61D6BF56548ULL, "vxca_caverns1_mines_ij3_0_0.vlm"},
	{0x0056F4193B5650F7ULL, "gencr_coterie_archer_f.utc"},
	{0x005769F7233A1142ULL, "vxht_0_hightown_a3_aux_0_0.vlm"},
	{0x00583459F857D40AULL, "6171038_m.fxe"},
	{0x0058716FBF7E1AB2ULL, "vxw_woundedcoast_19q_0_0.vlm"},
	{0x00594BEED57AEA48ULL, "46517586.ogg"},
	{0x005978FFC1A1A1E4ULL, "static_static_p38.psh.fxo.dx11"},
	{0x0059CFD316B04011ULL, "vxht_brothel_n_5h5_0_0.vlm"},
	{0x005A21A3B8516287ULL, "gen00fl_isabela6097012.cl"},
	{0x005A67CCCEBE38F3ULL, "vxw_woundedcoast_ow_0_0.vlm"},
	{0x005A8E034B4316E8ULL, "603570096.ogg"},
	{0x005AB7BA3B1F7BF2ULL, "vxht_0_hightown_akc_0_0.vlm"},
	{0x005B141B42784C7DULL, "pro000_merchant6051845.cl"},
	{0x005B394BB87ADD1CULL, "vxht_chantry_2zi_0_0.vlm"},
	{0x005B6E022D75118CULL, "vxca_caverns1_mines_hm3_0_0.vlm"},
	{0x005B7307D7598323ULL, "hlac_lt_warehouse_1_5z.dds"},
	{0x005C19A21A763AA2ULL, "hlac_ht_keep_3732_9y.dds"},
	{0x005D6E0B5EC4ABC5ULL, "hlac_lt_0_lowtown_a3_n_c_1_3z.dds"},
	{0x005D9C3279C1804EULL, "6146919_m.fxe"},
	{0x005EDCD3384AEE78ULL, "1063123879.ogg"},
	{0x0060164A2721259DULL, "gen00fl_fenris6107188.cl"},
	{0x00605338DE9198FDULL, "6173941_m.fxe"},
	{0x0060D4F2876347E5ULL, "gen00fl_isabela_init6075936.cl"},
	{0x00614B3D6E2DFCF6ULL, "vxht_mansion2_3dh_0_0.vlm"},
	{0x006158732238BB65ULL, "hightown_male_elfservnt_mob6170079.cl"},
	{0x0061F8C2D0F0EF60ULL, "hla_ht_0_hightown_n_-65537_0x.dds"},
	{0x00620407007FD66AULL, "vxlt_hangedman_258_0_0.vlm"},
	{0x0062579F2CAB7FE1ULL, "mrl150tr_undead_pre_camp.utt"},
	{0x006290455D8BF87DULL, "zz_approval_debug6003969.cl"},
	{0x0062C08CDAB9905BULL, "llac_lt_gallowsprison_n_-1_4.dds"},
	{0x0062DD44DD7A5E0BULL, "face_face_p11.psh.fxo.dx11"},
	{0x006312C1C7216939ULL, "mag210_arianni6021205.cl"},
	{0x00635454F0AF57AAULL, "220513863.ogg"},
	{0x0063B001D0FD0D81ULL, "follower_banter6148737.cl"},
	{0x0063C62BAC68501DULL, "6032044_m.fxe"},
	{0x0063D1C13964E8B1ULL, "art\\vfx\\combat\\vfx_w_jug_lineback_loop_c.anb"},
	{0x0063E63ADDE8368EULL, "vxht_keep_5hf_0_0.vlm"},
	{0x0063EBBDED1F9111ULL, "184995067.ogg"},
	{0x00649659C3F1E295ULL, "vxht_brothel_n_4fj_0_0.vlm"},
	{0x0064A78D3DD1D8F9ULL, "mag360_thrask6044272.cl"},
	{0x0064A81F9168A9A3ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\9ff_cartrock.mao"},
	{0x0064AC1F9168B0F0ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\9ff_cartrock.mmh"},
	{0x0064BA1F9168C80AULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\9ff_cartrock.msh"},
	{0x0064C5E24F57D7D9ULL, "vxht_chantry_2y9_0_0.vlm"},
	{0x0064ECD6B213E9D0ULL, "6039692_f.fxe"},
	{0x006550CACDCAF33EULL, "art\\vfx\\combat\\vfx_m_wpn_fire_crust_c.anb"},
	{0x0065B877D8AE4DCAULL, "art\\vfx\\environment\\level_fx\\lightbeams\\vfx_lightbeam_hta.mmh"},
	{0x0065DBD4153029C2ULL, "vxht_0_hightown_a3_ary_0_0.vlm"},
	{0x0065FA741FA74228ULL, "gen00fl_isabela6135692.cl"},
	{0x0065FDD170D80137ULL, "vxdr_primevaltomb_75s_0_0.vlm"},
	{0x00660EC079CDEA22ULL, "61611133.ogg"},
	{0x00669CD6DAAB6D97ULL, "art\\characters\\playercharacter\\humanmale\\textures\\hm_hlh_medh_0n.dds"},
	{0x00669F569A60D0ACULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\po_eyejitter.gad"},
	{0x0066B4A9E5252F87ULL, "241811738.ogg"},
	{0x0067327514E925E0ULL, "vxw_bonepitmining_a2_2ae_0_0.vlm"},
	{0x006792AA493FE3D1ULL, "mer320_hubert6053728.cl"},
	{0x0067A15A76D5D4ECULL, "vxlt_darktown_680_0_0.vlm"},
	{0x0067D04B03B3B284ULL, "vxlt_gallowstemplar_5x7_0_0.vlm"},
	{0x00685E6789F14761ULL, "hlac_w_blightlands_0_8z.dds"},
	{0x00688ED3E3C91180ULL, "one350_variel6088060.cl"},
	{0x0068D9A9C0B45173ULL, "vxw_woundedcoast_dv_0_0.vlm"},
	{0x0068E87C2D072130ULL, "hlac_lt_0_lowtown_docks_a3_-2_1y.dds"},
	{0x00695D58EECEC0BBULL, "783507773.ogg"},
	{0x00696D56E4304C00ULL, "vxca_caverns1_iig_0_0.vlm"},
	{0x0069802899AF7021ULL, "570192197.ogg"},
	{0x0069B5807D6AE7EBULL, "mag291_demon_marethari6166177.cl"},
	{0x0069B597CEA283A7ULL, "follower_banter6163646.cl"},
	{0x0069EC828CA002DDULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_scroll_d.dds"},
	{0x006A450A690DA4D8ULL, "gen00fl_aveline6132937.cl"},
	{0x006A697CFF2B246FULL, "rdr111dg_martin6015087.cl"},
	{0x006B89BEE40BA807ULL, "vxht_0_hightown_a83_0_0.vlm"},
	{0x006CE432D69E6421ULL, "6151471_m.fxe"},
	{0x006D6E982566601FULL, "vxlt_darktown_5nn_0_0.vlm"},
	{0x006D9E569A66EDAAULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\po_eyejitter.ani"},
	{0x006DC90A686C03E9ULL, "vxht_0_hightown_a3_ajv_0_0.vlm"},
	{0x006E083FB639CD72ULL, "730212427.ogg"},
	{0x006E0DE428101A3DULL, "hlac_lt_0_lowtown_docks_n_65534_1x.dds"},
	{0x006E7029E9E71D13ULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\9ff_post.phy"},
	{0x006E76AB30B32109ULL, "474878209.ogg"},
	{0x006E7ADC8CBF22DBULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\2hand_stop.evt"},
	{0x006EEB9EEADE402FULL, "art\\characters\\creatures\\dragon\\animation\\ctrn90r.gad"},
	{0x006EEDFB37CD7CFBULL, "vxht_0_hightown_a3_50n_0_0.vlm"},
	{0x006EF4F94110D12EULL, "800210944.ogg"},
	{0x006F389E05E5209EULL, "zz_vault_debug6203262.cl"},
	{0x007020B763BBAB95ULL, "isa111_isabela6066580.cl"},
	{0x007075E3F275EEEAULL, "vxlt_hangedman_229_0_0.vlm"},
	{0x007092C2BE4035DAULL, "977107726.ogg"},
	{0x0070F2C1396FAB2AULL, "art\\vfx\\combat\\vfx_w_jug_lineback_loop_c.mmh"},
	{0x007170C6E9DD6E04ULL, "zz_pride_demon2_v3.cut"},
	{0x00718FC612EA3021ULL, "art\\characters\\playercharacter\\humanmale\\textures\\hn_bdy_lgtc_0n.dds"},
	{0x007255E70EF40E90ULL, "hlac_ca_cavern1_haunted_24426_14y.dds"},
	{0x007327294FED60B8ULL, "6108578_f.fxe"},
	{0x007337BF414DF059ULL, "vxht_chantry_2r8_0_0.vlm"},
	{0x007383067C9300A1ULL, "static_motionvectors_p0.psh.fxo.dx11.assembly"},
	{0x00738B36F0EA9CAFULL, "vxlt_hangedman_1mr_0_0.vlm"},
	{0x0073D5A18DCE4CD6ULL, "6106195_m.fxe"},
	{0x00740502BBA02E5DULL, "770134267.ogg"},
	{0x0074356370E3BFDAULL, "zz_followers_banter6158991.cl"},
	{0x007441EFF481E0BBULL, "6111210_f.fxe"},
	{0x00746CDB7A8B704BULL, "6180817_m.fxe"},
	{0x0074C0CAE6508451ULL, "vxlt_0_lowtown_a3_n_c_gof_0_0.vlm"},
	{0x0074D120E5706286ULL, "llac_lt_0_lowtown_docks_0_4.dds"},
	{0x0074EBBA22DC4403ULL, "706267296.ogg"},
	{0x007573DC8CC5351CULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\2hand_stop.gad"},
	{0x00758304FE7B9AA6ULL, "6096323_f.fxe"},
	{0x007595192E71CA4FULL, "gen_im_arm_glv_med_act1_02.uti"},
	{0x0075DE0D1707DB67ULL, "6122923_m.fxe"},
	{0x0076009EEAE482DCULL, "art\\characters\\creatures\\dragon\\animation\\ctrn90r.evt"},
	{0x0076957364FF91A7ULL, "449179346.ogg"},
	{0x0076B1AB8538B911ULL, "art\\levels\\races\\design\\prp_moneypoucha.mao"},
	{0x0076DC3B52D19BFEULL, "gen_im_trs_qun_lgt_cht_r2.uti"},
	{0x0076E13AA25DF37FULL, "art\\levels\\ztools\\helpers\\character\\prp_humanlean_3.mao"},
	{0x0076F45F3D383B03ULL, "ico_boots_light_4.dds"},
	{0x00774B86B2AA69F3ULL, "vxw_woundedcoast_ay_0_0.vlm"},
	{0x007786D744097058ULL, "gen00fl_dog6171156.cl"},
	{0x00778B12F8021B48ULL, "scaleform_sf_main_p0.psh.fxo.dx11.assembly"},
	{0x007790F516F8A7C1ULL, "28613663.ogg"},
	{0x0077DCFF367E5004ULL, "6156215_m.fxe"},
	{0x0077DF33D6266480ULL, "vxca_caverns1_ild_0_0.vlm"},
	{0x0077E1D4BA39867AULL, "971430361.ogg"},
	{0x0077F9CBF056B84DULL, "lt_0_lowtown_docks_n_228_0.mmh"},
	{0x0077FDCBF056BE9EULL, "lt_0_lowtown_docks_n_228_0.mao"},
	{0x00780BCBF056D6D3ULL, "lt_0_lowtown_docks_n_228_0.msh"},
	{0x00784C9275FE3720ULL, "6116739_f.fxe"},
	{0x0078690223316BF1ULL, "aip_dalishwarrior2h_base.gda"},
	{0x00786C8F5C547E64ULL, "923631154.ogg"},
	{0x0078AB0C160C8E0EULL, "20275664.ogg"},
	{0x00795B334E0224A6ULL, "hlac_ca_caverns2_5802_39y.dds"},
	{0x007996A044D7BFEDULL, "gen00fl_sebastian6195856.cl"},
	{0x007A54B5533D3769ULL, "931355191.ogg"},
	{0x007A76F7E98E8D63ULL, "287836797.ogg"},
	{0x007AAAFCDC67FE54ULL, "art\\vfx\\combat\\vfx_gib_corm_1.mao"},
	{0x007ADD7E31C57477ULL, "6145212_m.fxe"},
	{0x007AE89ECE8EFE00ULL, "dae000ip_crowd_mf5_drunk1.utp"},
	{0x007B582DF9ED6908ULL, "hlac_lt_0_lowtown_65533_0y.dds"},
	{0x007B6DEB2FC99E24ULL, "zz_followers_banter6159666.cl"},
	{0x007BD7B47D7C3714ULL, "hlac_w_bonepitmining_a2_262149_0z.dds"},
	{0x007C0FE30C4B96F8ULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\trap_gallowpad_6.pwk"},
	{0x007C308AC0C24B53ULL, "art\\vfx\\combat\\vfx_w_gn2_threat_loop_mo_c.impa.evt"},
	{0x007C72DC8CCB521AULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\2hand_stop.ani"},
	{0x007CAF8C2DF50C6FULL, "llac_lt_0_lowtown_a2_-2_1.dds"},
	{0x007CF595A213DA26ULL, "564717246.ogg"},
	{0x007D311766C701F6ULL, "fx_2003_diff.dds"},
	{0x007D58186F12B8F1ULL, "1003059896.ogg"},
	{0x007D678F80951E2FULL, "6201276_f.fxe"},
	{0x007DBFC2C99F27C9ULL, "vxw_bonepitdestroyed_a3_1pz_0_0.vlm"},
	{0x007DFCAC32E95D76ULL, "945667779.ogg"},
	{0x007E048F5739EB76ULL, "art\\characters\\creatures\\desiredemon\\animation\\ctn180r.evt"},
	{0x007FB02AD24CE1BEULL, "hlac_lt_undercityserialkiller_-1_17z.dds"},
	{0x007FD95535039EC1ULL, "llac_lt_0_lowtown_a2_n_c_-262143_0.dds"},
	{0x008027AB4C54C7CFULL, "vxlt_0_lowtown_a3_n_c_fiw_0_0.vlm"},
	{0x00807F9597EBFD7BULL, "zz_prmpt_debug6198272.cl"},
	{0x008109755BA917F5ULL, "6143267_m.fxe"},
	{0x00813F04F58A5D63ULL, "follower_banter6148036.cl"},
	{0x0081DCA3E840C21AULL, "dae000ip_craft_potions3.utp"},
	{0x0081EFDF6A10064FULL, "430871833.ogg"},
	{0x00824963898C07B2ULL, "hlac_ht_brothel_n_3370_0z.dds"},
	{0x00824DB3CFADAE3DULL, "vxca_caverns1_mines_cjp_0_0.vlm"},
	{0x00824DCCAFB01CABULL, "art\\characters\\playercharacter\\elffemale\\mt_ant_ef06.phy"},
	{0x0082C2852638E5D5ULL, "hlac_lt_0_lowtown_n_-196608_2x.dds"},
	{0x0082F679FB4421C8ULL, "bloodsmall5.dds"},
	{0x0083298AC0C85D94ULL, "art\\vfx\\combat\\vfx_w_gn2_threat_loop_mo_c.impa.gad"},
	{0x00836A84DE83E573ULL, "vxlt_backalley_g_1pc_0_1.vlm"},
	{0x008383B7FA146A80ULL, "392929790.ogg"},
	{0x0083FB8F573E4701ULL, "art\\characters\\creatures\\desiredemon\\animation\\ctn180r.gad"},
	{0x0084089EEAF0CDF5ULL, "art\\characters\\creatures\\dragon\\animation\\ctrn90r.ani"},
	{0x008450C07A35FDFAULL, "vxlt_0_lowtown_docks_a3_n_cx9_0_0.vlm"},
	{0x0084F63824C73C29ULL, "vxht_0_hightown_8rr_0_0.vlm"},
	{0x00851CF2C9CE94D7ULL, "lt_0_lowtown_docks_n_172_0.mmh"},
	{0x008520F2C9CE9BE4ULL, "lt_0_lowtown_docks_n_172_0.mao"},
	{0x0085258A67D1D237ULL, "gen_im_arm_cht_noba_t2.uti"},
	{0x008526F70450F8ACULL, "vxlt_0_lowtown_docks_a3_n_8rn_0_0.vlm"},
	{0x008532F2C9CEBA49ULL, "lt_0_lowtown_docks_n_172_0.msh"},
	{0x00857E115EDD209DULL, "6126525_m.fxe"},
	{0x0085966D5F06B379ULL, "6108809_f.fxe"},
	{0x0085C878542797FBULL, "w_bonepitdestroyed_a3_221_0.msh"},
	{0x0085DA785427B646ULL, "w_bonepitdestroyed_a3_221_0.mao"},
	{0x0085E6785427CAE5ULL, "w_bonepitdestroyed_a3_221_0.mmh"},
	{0x00860B61FEB6A68EULL, "vxht_0_hightown_n_ccr_0_0.vlm"},
	{0x008704B29759CA4AULL, "6068545_m.fxe"},
	{0x008758B04EAE2632ULL, "445424801.ogg"},
	{0x0087756683D62C13ULL, "hlac_lt_0_lowtown_docks_n_131073_1x.dds"},
	{0x0087EB6935FAC94BULL, "fx_impact02.dds"},
	{0x0087F1AB2ADE4081ULL, "zz_vrc_debug6200241.cl"},
	{0x0088B9E9963871B3ULL, "p_leather_stretcher_01n.dds"},
	{0x0089B4DD7B0CE205ULL, "6107289_m.fxe"},
	{0x008A388AC0CE9672ULL, "art\\vfx\\combat\\vfx_w_gn2_threat_loop_mo_c.impa.ani"},
	{0x008A52520952911FULL, "vxlt_darktown_5th_0_0.vlm"},
	{0x008AA535D28171B6ULL, "vxht_0_hightown_b5f_0_0.vlm"},
	{0x008ADEF3B7F31885ULL, "40117880.ogg"},
	{0x008AF1451C28D760ULL, "nat320_delilah6065926.cl"},
	{0x008AFC8F574467AFULL, "art\\characters\\creatures\\desiredemon\\animation\\ctn180r.ani"},
	{0x008B79B8B8EA7396ULL, "hlac_w_bonepitdestroyed_a3_327686_0x.dds"},
	{0x008BCD2F746C1077ULL, "hlac_lt_0_lowtown_a3_n_c_-196608_3z.dds"},
	{0x008C06B8A5683B53ULL, "6044195_m.fxe"},
	{0x008C167280DE2AADULL, "734590368.ogg"},
	{0x008C8C3436EC955EULL, "zz_one_debug6078374.cl"},
	{0x008CD8C55BDCC4E3ULL, "936640766.ogg"},
	{0x008DAE37B235F02EULL, "6195814_m.fxe"},
	{0x008DD575178A481BULL, "6033690_m.fxe"},
	{0x008E1CC60FF3AEA7ULL, "hlac_lt_0_lowtown_n_-65537_1y.dds"},
	{0x008F3314EFD7647DULL, "vxht_0_hightown_a3_9aw_0_0.vlm"},
	{0x008F3EC50935429AULL, "vxlt_gallowstemplar_f_3n2_0_0.vlm"},
	{0x008F668B23757B40ULL, "vxdr_primevalentrance_1qb_0_0.vlm"},
	{0x009077FBF2F6D343ULL, "vxlt_gallowstemplar_f_2j6_0_0.vlm"},
	{0x0090FDB7D8BB7E37ULL, "933121158.ogg"},
	{0x009144284C3D7864ULL, "art\\levels\\races\\natural\\highlands\\foliage\\_textures\\plc_flowerwild_01_0_sa.dds"},
	{0x00917787EE4A9621ULL, "zz_mag_debug6024441.cl"},
	{0x0091A73F5F1E0459ULL, "139669065.ogg"},
	{0x0091BF191D832111ULL, "52572992.ogg"},
	{0x0093034733B9AF05ULL, "pro000_captain_wright6004379.cl"},
	{0x00932345FF525584ULL, "804509941.ogg"},
	{0x00933E48313AEF98ULL, "6054492_m.fxe"},
	{0x0093600E645B4808ULL, "vxlt_hangedman_ny_0_0.vlm"},
	{0x0093753245C60492ULL, "zz_one_debug6099937.cl"},
	{0x00944E907988FFF8ULL, "lgt000ip_tombs_seal.utp"},
	{0x009484CF10DD101CULL, "ht_brothel_n_2995.anb"},
	{0x009576A331EA1715ULL, "vxdr_primevaltomb_6y0_0_0.vlm"},
	{0x0095A1144932F469ULL, "6142542_f.fxe"},
	{0x0095A342CB8F88CCULL, "6004201_f.fxe"},
	{0x0095F3E06D1E3DD3ULL, "vxlt_0_lowtown_a3_n_c_e1x_0_0.vlm"},
	{0x009638539567815AULL, "zz_mag_debug6051361.cl"},
	{0x009638893AFCE56FULL, "llac_lt_0_lowtown_docks_131072_1.dds"},
	{0x009641308DAB7B0BULL, "hlac_lt_0_lowtown_a2_-2_2y.dds"},
	{0x0096A72447FC63DAULL, "vxlt_undercitydungeon_1il_0_1.vlm"},
	{0x00970A97DFB98FAFULL, "vxw_encounter2_camp_9h_0_0.vlm"},
	{0x00974C6C6B38A073ULL, "vxw_bonepit_1wq_0_0.vlm"},
	{0x0097D60D8640A026ULL, "879358767.ogg"},
	{0x00980394879376F3ULL, "gen00fl_sebastian6179764.cl"},
	{0x00989BBA11F08109ULL, "vxht_keep_4nf_0_0.vlm"},
	{0x0099022DA1CC18BDULL, "vxca_caverns2_97o_0_0.vlm"},
	{0x00990C63C983D7A0ULL, "vxlt_gallowsprison_n_6nq_0_0.vlm"},
	{0x00995C50CABF5B11ULL, "zz_ser_debug6078557.cl"},
	{0x0099BEA9052357D5ULL, "510828892.ogg"},
	{0x009AC7ADF63E9DCFULL, "gen00fl_anders6114456.cl"},
	{0x009ACCA16703180CULL, "and111_lirene6084381.cl"},
	{0x009AEE93A23A420EULL, "vxlt_gallowstemplar_453_0_0.vlm"},
	{0x009B6D9F036786C4ULL, "6112119_f.fxe"},
	{0x009B7DE632D19657ULL, "gen00fl_isabela6098215.cl"},
	{0x009C229007CC7748ULL, "vxca_caverns1_mines_j2t_0_0.vlm"},
	{0x009CBB3849384CA8ULL, "vxlt_0_lowtown_eoq_0_0.vlm"},
	{0x009D032D2157BE0DULL, "art\\characters\\creatures\\dog\\textures\\c_eye_dogamo_0n.dds"},
	{0x009DF1091831FE74ULL, "ca_caverns1_24426_lc.mmh"},
	{0x009E0DE154F2E3F7ULL, "634622183.ogg"},
	{0x009F02A5AC47D0CEULL, "6093687_f.fxe"},
	{0x009F2B704DC1F73EULL, "art\\vfx\\combat\\vfx_w_bsk_haste_loop_c.mmh"},
	{0x009F6D9B87B49B54ULL, "vxlt_0_lowtown_a3_n_c_eqk_0_0.vlm"},
	{0x009F886042913287ULL, "hm_desp_refugee.mor"},
	{0x009F9DBF04E19265ULL, "vxlt_undercityserialkiller_yx_0_0.vlm"},
	{0x009FB479EC8BF356ULL, "vxlt_0_lowtown_a2_gpd_0_0.vlm"},
	{0x009FFA353C2BBFDDULL, "181858611.ogg"},
	{0x00A0236E80AB1A6BULL, "187655715.ogg"},
	{0x00A0B3AEA86BDE6EULL, "364075432.ogg"},
	{0x00A0B4B734FC9EB7ULL, "vxlt_0_lowtown_n_gc3_0_0.vlm"},
	{0x00A13C9CE3DACF78ULL, "919390249.ogg"},
	{0x00A176EEA3EE30AAULL, "335667790.ogg"},
	{0x00A18F0B3C06388BULL, "6142536_m.fxe"},
	{0x00A27EBCFD1CFA0CULL, "vxht_0_hightown_a3_c8s_0_0.vlm"},
	{0x00A28114776FB76DULL, "face2_ds1.psh.fxo"},
	{0x00A2A11D888CB373ULL, "vxw_woundedcoast_pz_0_0.vlm"},
	{0x00A2E5DAF8F45440ULL, "6150965_m.fxe"},
	{0x00A3A32D974FB699ULL, "hlac_dr_dungeon_3215_2z.dds"},
	{0x00A3DE0344918E20ULL, "vxht_mansion_n_4zp_0_0.vlm"},
	{0x00A3FD4A94B84CFFULL, "gen00fl_aveline6121895.cl"},
	{0x00A403B8992BD8A5ULL, "hlac_lt_gallowstemplar_n_2394_7y.dds"},
	{0x00A42A06AA0F0496ULL, "1002264399.ogg"},
	{0x00A4393FA2CD410AULL, "framebuffereffects_passthrough_p0.psh.fxo"},
	{0x00A475B61D5B739CULL, "6128019_f.fxe"},
	{0x00A4A248E1C9DEA8ULL, "vxca_cave_1cu_0_0.vlm"},
	{0x00A57204D9206010ULL, "vxht_0_hightown_c9y_0_0.vlm"},
	{0x00A57C74D1AFA82FULL, "vxw_encounter2_camp_4k_0_0.vlm"},
	{0x00A5B31F91A0530BULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\9ff_cartrock.phy"},
	{0x00A612B4409566CEULL, "lgt261tr_girl_flee.utt"},
	{0x00A659E533E8D4DCULL, "500646603.ogg"},
	{0x00A668F9B865E199ULL, "zz_tbe100_barber6196160.cl"},
	{0x00A6712BC984BE20ULL, "gen00fl_isabela6015842.cl"},
	{0x00A6881676F0EAA5ULL, "ht_mansion_n_3369.rml"},
	{0x00A688BC2BDA9C57ULL, "vxw_encounter2_camp_po_0_0.vlm"},
	{0x00A6DC4A844293AEULL, "363069456.ogg"},
	{0x00A6F5B735259CE2ULL, "6169750_m.fxe"},
	{0x00A727FDA94A999AULL, "hlac_lt_0_lowtown_docks_a3_n_-65538_6x.dds"},
	{0x00A783CBC31345CEULL, "vxht_0_hightown_avh_0_0.vlm"},
	{0x00A7931CB1B4AC39ULL, "hlac_w_bonepitmining_a2_131072_0y.dds"},
	{0x00A7D0C9ABBAD271ULL, "vxlt_0_lowtown_n_i4f_0_0.vlm"},
	{0x00A8A8A1D57A0F18ULL, "900522910.ogg"},
	{0x00A8CCC73C4CB446ULL, "1056624136.ogg"},
	{0x00A8EF2AEB09AF0AULL, "6027863_m.fxe"},
	{0x00A961411374A758ULL, "llac_lt_0_lowtown_docks_a3_n_65533_2.dds"},
	{0x00A9979C36AB80A3ULL, "clf260_harley6088770.cl"},
	{0x00A9AFD9358C5B3FULL, "gen00fl_aveline6115474.cl"},
	{0x00A9C0932CC4168BULL, "vxlt_0_lowtown_n_e1k_0_0.vlm"},
	{0x00A9DF38AD8023EAULL, "llac_ca_caverns1_24424_14.dds"},
	{0x00AA78FABAC09C74ULL, "lgt000cr_merch_turn_in_4.utc"},
	{0x00AAA1E768368AC8ULL, "hf_mother_h1s2.mor"},
	{0x00AAA76218FF99C7ULL, "one000_ilen6037077.cl"},
	{0x00ACEB2859EFFD6CULL, "llac_lt_gallowstemplar_7198_5.dds"},
	{0x00AD0A704DCDFCFDULL, "art\\vfx\\combat\\vfx_w_bsk_haste_loop_c.anb"},
	{0x00AD13B8979208DFULL, "6036522_f.fxe"},
	{0x00AD2BE36499C489ULL, "hlac_ht_mansion2_818_47z.dds"},
	{0x00AD669B45C9F231ULL, "366095080.ogg"},
	{0x00AD693D43CF15E9ULL, "6149364_m.fxe"},
	{0x00AD7F3636B4EC5DULL, "vxw_encounter2_camp_rg_0_0.vlm"},
	{0x00AD813E955BE0A3ULL, "gen_im_wep_rog_dua_act1_01.uti"},
	{0x00AE2656DE820BD6ULL, "vxlt_0_lowtown_a2_gwe_0_0.vlm"},
	{0x00AE3A0891DD3150ULL, "al_mage_t06.gda"},
	{0x00AF081B9A229492ULL, "896936067.ogg"},
	{0x00AF48647EAA8D45ULL, "vxlt_0_lowtown_docks_a3_n_cx2_0_0.vlm"},
	{0x00AF963ADA4ECE68ULL, "zz_vis_debug6072534.cl"},
	{0x00AFA5CE6190D4C2ULL, "285934508.ogg"},
	{0x00AFB3B1F71C70D3ULL, "471630697.ogg"},
	{0x00AFC168B9629AE7ULL, "766805326.ogg"},
	{0x00AFEB202D999FD8ULL, "qcr000_fake_siege.cnv"},
	{0x00B015297D731688ULL, "lgt000im_ultra_swatch.uti"},
	{0x00B0346DF0C43D8AULL, "ser120_ghyslain6104804.cl"},
	{0x00B03FAAEC17A740ULL, "zz_hawke_cu_v2.cut"},
	{0x00B0A02881567BDFULL, "55122978.ogg"},
	{0x00B0FB5211166477ULL, "6076015_m.fxe"},
	{0x00B1B5125A3C5B04ULL, "267162787.ogg"},
	{0x00B1F80C3DF81F29ULL, "framebuffereffects_radialblur_p1.vsh.fxo.dx11"},
	{0x00B201986350CBB8ULL, "zz_dae_debug6029413.cl"},
	{0x00B22E1D060B260DULL, "mcr310_argument6074338.cl"},
	{0x00B248CAB0DA2EE4ULL, "6062794_m.fxe"},
	{0x00B2CDC3430BEBE0ULL, "t3_wdg_w23.tnt"},
	{0x00B3760026B2F9FBULL, "6078707_m.fxe"},
	{0x00B399B92598B656ULL, "sf_solid.psh.fxo"},
	{0x00B3CD202D9D55E0ULL, "qcr000_fake_siege.bnk"},
	{0x00B44C71FD328F2DULL, "6165563_f.fxe"},
	{0x00B467B41C7EB8B3ULL, "hlac_lt_0_lowtown_a2_131071_3x.dds"},
	{0x00B4A3181F276F89ULL, "dae000_flemeth6000962.cl"},
	{0x00B4FA991DD0B4D7ULL, "vxw_encounter2_camp_kn_0_0.vlm"},
	{0x00B5A5E5B1622E15ULL, "449708548.ogg"},
	{0x00B619711E5F921EULL, "hlac_ca_caverns1_24427_2x.dds"},
	{0x00B633BE4209E103ULL, "21972141.ogg"},
	{0x00B6367E2E9736AEULL, "fx_smokeghost_fb.mao"},
	{0x00B6CFDB2C83FB72ULL, "hlac_lt_0_lowtown_docks_65534_3x.dds"},
	{0x00B6E6B375FD3801ULL, "6128400_m.fxe"},
	{0x00B706FA2A76F508ULL, "hf_qmq_minion6.mor"},
	{0x00B707B3E4FD87D0ULL, "hlac_lt_0_lowtown_n_131069_0z.dds"},
	{0x00B721411598624EULL, "38737293.ogg"},
	{0x00B72812AB6B6373ULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_laundry_n.dds"},
	{0x00B761EF3DCD0B7DULL, "seb221st_flora_drunk.stg"},
	{0x00B7DC4061EE2F60ULL, "978777270.ogg"},
	{0x00B7E4BC9601F82DULL, "llac_dr_0_deeproads_5849_7.dds"},
	{0x00B7F75E445F11A1ULL, "vxw_bonepit_zs_0_0.vlm"},
	{0x00B8FE0C0EEB828AULL, "hlac_ht_0_hightown_a3_-3_0z.dds"},
	{0x00BAB87D21C0C708ULL, "art\\characters\\playercharacter\\shared\\heads\\mt_bas_uh_lashesm1.mao"},
	{0x00BB1FA008F6998DULL, "661855604.ogg"},
	{0x00BB31EAE608F5E2ULL, "gen00fl_aveline_post_patrol.stg"},
	{0x00BB5941F233C494ULL, "6093776_m.fxe"},
	{0x00BB7495421E3AE5ULL, "fxc_bm_drop.mao"},
	{0x00BBA9C31B9A8529ULL, "449793971.ogg"},
	{0x00BBB32D5B1ED2D8ULL, "ser221_gascard6149261.cl"},
	{0x00BBCD80252C5F98ULL, "dae000_oldmangossip6175769.cl"},
	{0x00BBE35A514FCA38ULL, "dae000ip_crowd_lm5_talk5.utp"},
	{0x00BBF74C0D9738B0ULL, "vxlt_0_lowtown_a3_n_c_fep_0_0.vlm"},
	{0x00BC014FCC61F54DULL, "d15_link_smo2.dds"},
	{0x00BC51F0A7FE5639ULL, "6054725_m.fxe"},
	{0x00BC624D0B994B50ULL, "gen_im_trs_rdr_hvy_glv_r0.uti"},
	{0x00BC81F2F918E8F6ULL, "cod_hst_kirk_arch_1a_9.utp"},
	{0x00BCB2946E8F6E83ULL, "vxca_caverns1_mines_e4z_0_0.vlm"},
	{0x00BCB86810BBA9AFULL, "vxca_caverns1_j6j_0_0.vlm"},
	{0x00BD0999AB3F22B3ULL, "hlac_w_encounter2_3_0x.dds"},
	{0x00BD25B3927CE7E3ULL, "hlac_lt_0_lowtown_a3_n_c_1_0y.dds"},
	{0x00BDC26E4E0A8F36ULL, "587554613.ogg"},
	{0x00BE05B440A9E2F7ULL, "lgt261tr_girl_flee.ncs"},
	{0x00BE090CB40E9F4EULL, "fx_dotblur_alpha.mao"},
	{0x00BE9AB8D2C22B21ULL, "hlac_ht_keep_4519_1x.dds"},
	{0x00BF2F5660C98A68ULL, "vxht_brothel_n_5ki_0_0.vlm"},
	{0x00BF6276E1092B0CULL, "vxht_0_hightown_a3_c6q_0_0.vlm"},
	{0x00BF7D2F181687B6ULL, "vxdr_primevaltomb_6iw_0_0.vlm"},
	{0x00BF9D000E6924D7ULL, "and100_anders6111659.cl"},
	{0x00C017EE093F64D5ULL, "hlac_lt_0_lowtown_docks_a3_n_131073_0x.dds"},
	{0x00C031700838BA93ULL, "art\\characters\\weapons\\staff\\textures\\w_stf_001a_0n.dds"},
	{0x00C03512697714C9ULL, "vxca_caverns2prc_90c_0_0.vlm"},
	{0x00C049454793E96EULL, "vxlt_0_lowtown_n_ers_0_0.vlm"},
	{0x00C059FB6FF9951DULL, "gen_aoe_player_detect.ncs"},
	{0x00C0707960B46D0EULL, "348519769.ogg"},
	{0x00C076AB535F4C97ULL, "434639282.ogg"},
	{0x00C0BAA7A2D04764ULL, "6069694_m.fxe"},
	{0x00C0D9DC6A3BB2E8ULL, "lt_0_lowtown_docks_a3_n_185_0.msh"},
	{0x00C0EBDC6A3BD179ULL, "lt_0_lowtown_docks_a3_n_185_0.mao"},
	{0x00C0F294A1CE68A1ULL, "framebuffereffects_composedof_p0.psh.fxo.dx11"},
	{0x00C0F7DC6A3BE512ULL, "lt_0_lowtown_docks_a3_n_185_0.mmh"},
	{0x00C103A8979C3992ULL, "art\\vfx\\materials\\vfx_lightning_add.mao"},
	{0x00C23B44E1B0A897ULL, "6134731_m.fxe"},
	{0x00C29D2511928CAAULL, "6161652_f.fxe"},
	{0x00C36C760FC6CD57ULL, "vxw_encounter2_camp_fi_0_0.vlm"},
	{0x00C389C0C0DE5A64ULL, "vxca_cave_vk_0_0.vlm"},
	{0x00C39E136EF88CAAULL, "hlac_ht_mansion_n_3369_1x.dds"},
	{0x00C46432ECE8D28AULL, "gen00fl_varric6201278.cl"},
	{0x00C53F7F3DE3BE5EULL, "art\\vfx\\combat\\vfx_gui_ground_ring_l_t_p.impa.evt"},
	{0x00C54A98F1B5FD3EULL, "mer220_hubert6059919.cl"},
	{0x00C58324EB25590AULL, "893824633.ogg"},
	{0x00C67141A4C12B36ULL, "llac_lt_undercityserialkiller_-1_44.dds"},
	{0x00C6BE31D202EDEAULL, "seb321_leliana6180969.cl"},
	{0x00C6C2A94FCCCF39ULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_containerb_n.dds"},
	{0x00C71967B0A9059DULL, "mcr310_sibling.bnk"},
	{0x00C827D2420536DFULL, "cod_bks_demons_gift.utp"},
	{0x00C84F871496A1F6ULL, "vxw_encounter3_fg_0_0.vlm"},
	{0x00C85783205CB16DULL, "6030495_m.fxe"},
	{0x00C86C327DC50341ULL, "vrc221st_varric_outsidemans.stg"},
	{0x00C88DF27F341000ULL, "6170840_m.fxe"},
	{0x00C8A99BC31AB527ULL, "vxht_mansion2b_1xk_0_0.vlm"},
	{0x00C90B3E44DC4936ULL, "23351594.ogg"},
	{0x00C91C09E1651B73ULL, "6056615_m.fxe"},
	{0x00C9287788D8E11BULL, "vxht_keep_n_63b_0_0.vlm"},
	{0x00C967C7AC031A79ULL, "gen00fl_isabela_init6078166.cl"},
	{0x00C9B34F13A1930DULL, "dae000_bartrand6064039.cl"},
	{0x00CA31ED107CD797ULL, "lt_warehouse_lc.mmh"},
	{0x00CA400E8CB6064FULL, "one000_ilen6059253.cl"},
	{0x00CA569E67A4FB3BULL, "163572584.ogg"},
	{0x00CA5A4297ED4CFEULL, "32561661.ogg"},
	{0x00CA7E490E789194ULL, "141880707.ogg"},
	{0x00CA803739234357ULL, "llac_lt_foundry_n_320_2.dds"},
	{0x00CA8DFCA93F5224ULL, "w_encounter3_126_0.mmh"},
	{0x00CA91FCA93F5977ULL, "w_encounter3_126_0.mao"},
	{0x00CAA3FCA93F77C6ULL, "w_encounter3_126_0.msh"},
	{0x00CAA6D4807D58FDULL, "zz_followers_banter6159045.cl"},
	{0x00CAC3CD99F46EB9ULL, "6051825_m.fxe"},
	{0x00CAFB67B0ACBBA5ULL, "mcr310_sibling.cnv"},
	{0x00CB0151FD359DB3ULL, "vxca_caverns2_9xp_0_0.vlm"},
	{0x00CB40DF8C0FAD1EULL, "6062183_m.fxe"},
	{0x00CC567F3DEA0389ULL, "art\\vfx\\combat\\vfx_gui_ground_ring_l_t_p.impa.gad"},
	{0x00CCA39804DD065FULL, "llac_lt_0_lowtown_docks_n_262143_0.dds"},
	{0x00CCCAE3FAD8BAF8ULL, "vxca_cavern1_haunted_jlj_0_0.vlm"},
	{0x00CCDBB71BDC9997ULL, "vxdr_0_deeproads_cp6_0_0.vlm"},
	{0x00CD5F6E86484F75ULL, "mcr310_argument6075096.cl"},
	{0x00CD63C76C834CD4ULL, "21050686.ogg"},
	{0x00CDCE29EE9D1135ULL, "254702025.ogg"},
	{0x00CDE9D147695B0DULL, "650341972.ogg"},
	{0x00CE19FFC29A6098ULL, "vxlt_undercityserialkiller_14e_0_0.vlm"},
	{0x00CEA4F3B75BA414ULL, "33780184.ogg"},
	{0x00CEB8274D7AC8CFULL, "783738038.ogg"},
	{0x00CEBB22398A01EEULL, "vxlt_0_lowtown_n_eot_0_0.vlm"},
	{0x00CECB840FE8A3BFULL, "vxlt_gallowstemplar_f_46y_0_1.vlm"},
	{0x00CF299E2374B4B3ULL, "art\\vfx\\combat\\vfx_r_arc_front_close_c.impa.evt"},
	{0x00CF6940292F968DULL, "dae100_viscount6118516.cl"},
	{0x00CFB3AC2C4F2CC8ULL, "hlac_lt_undercityserialkiller_-1_2x.dds"},
	{0x00CFC240D3368D63ULL, "llac_w_woundedcoast_65539_0.dds"},
	{0x00CFD56994056DF5ULL, "vxht_mansion_n_6jc_0_0.vlm"},
	{0x00D04C7748044931ULL, "vxlt_undercityencounter_wr_0_0.vlm"},
	{0x00D05641145A790EULL, "6060282_m.fxe"},
	{0x00D06929EA39DB88ULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\9ff_post.mmh"},
	{0x00D07529EA39F02BULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\9ff_post.mao"},
	{0x00D075415B522ABBULL, "dre000_sandal6028367.cl"},
	{0x00D08729EA3A0EB2ULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\9ff_post.msh"},
	{0x00D0B7A2C1EB6423ULL, "vxht_bartrand_n_64h_0_0.vlm"},
	{0x00D0BCBE4757D070ULL, "hlac_w_bonepit_393217_0y.dds"},
	{0x00D0F5D2D2BCD4A7ULL, "739967534.ogg"},
	{0x00D131641982AACBULL, "253416240.ogg"},
	{0x00D155653B0E77C4ULL, "128504251.ogg"},
	{0x00D23400F5EAFD55ULL, "pro000_wesley6007722.cl"},
	{0x00D2646D7EEC12C2ULL, "451492700.ogg"},
	{0x00D3277F3DEFD2D7ULL, "art\\vfx\\combat\\vfx_gui_ground_ring_l_t_p.impa.ani"},
	{0x00D3334A3EB0DAC8ULL, "hlac_ca_caverns2prc_5802_4x.dds"},
	{0x00D34F7F001D129AULL, "85216000.ogg"},
	{0x00D35BF19EB32229ULL, "943794326.ogg"},
	{0x00D4E5EA685FC594ULL, "51108397.ogg"},
	{0x00D56A73F4923077ULL, "6139457_m.fxe"},
	{0x00D583CF1114A5ACULL, "ht_brothel_n_2995.rml"},
	{0x00D5E17991ADE8E0ULL, "305153607.ogg"},
	{0x00D6229E237AC7F4ULL, "art\\vfx\\combat\\vfx_r_arc_front_close_c.impa.gad"},
	{0x00D632454E3D3A87ULL, "6105623_m.fxe"},
	{0x00D6C164068CBA76ULL, "vxw_encounter3_ch_0_0.vlm"},
	{0x00D71B78B510CDA7ULL, "vxht_mansion2b_1sj_0_0.vlm"},
	{0x00D79B18E2BAD953ULL, "vxlt_0_lowtown_a2_bxp_0_0.vlm"},
	{0x00D7AB60116AF14FULL, "hlac_w_encounter2_camp_3_0z.dds"},
	{0x00D7D40E2FB71EDDULL, "genip_cin_dagger.utp"},
	{0x00D7E2C58ECD0CECULL, "hlac_lt_0_lowtown_docks_0_0x.dds"},
	{0x00D818C2275420A3ULL, "6182028_m.fxe"},
	{0x00D867A0797CD544ULL, "vxlt_0_lowtown_docks_n_bte_0_0.vlm"},
	{0x00D89A6E305B1E4AULL, "6065393_m.fxe"},
	{0x00D8A2B98AD457E8ULL, "gen00fl_anders6097437.cl"},
	{0x00D8AE80BF681676ULL, "lowtown_fem_lowlife_mob6169188.cl"},
	{0x00D8D4CD0C9735DDULL, "vxw_encounter2_camp_cd_0_0.vlm"},
	{0x00D97BEDB4645556ULL, "vxlt_0_lowtown_a2_gdh_0_0.vlm"},
	{0x00D98716771C42CDULL, "ht_mansion_n_3369.anb"},
	{0x00DA339FE5B239CBULL, "gen00fl_varric6105795.cl"},
	{0x00DA51BCA4A7B0E6ULL, "zz_rockwraith2.cut"},
	{0x00DB0790DC82DCF3ULL, "827305660.ogg"},
	{0x00DB3126E5F0B849ULL, "art\\levels\\races\\primeval\\props\\_texture\\prp_lyriumtile_e.dds"},
	{0x00DB4D940DD2B217ULL, "vxdr_0_deeproads_cm1_0_0.vlm"},
	{0x00DB670DF6645011ULL, "hlac_w_sundermount_196613_0x.dds"},
	{0x00DB7DD77257D2C8ULL, "79440042.ogg"},
	{0x00DB8C1C01B3EE6DULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\0wi_basic.mmh"},
	{0x00DB901C01B3F4BEULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\0wi_basic.mao"},
	{0x00DB9E1C01B40CF3ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\0wi_basic.msh"},
	{0x00DBE5E5CE5A6375ULL, "617597084.ogg"},
	{0x00DC5B4F3D34C77EULL, "6113074_m.fxe"},
	{0x00DCE190B31FB391ULL, "zz_vault_debug6203379.cl"},
	{0x00DCFC3E73E11000ULL, "vxca_caverns1_i1o_0_0.vlm"},
	{0x00DD2CFF2B801A6EULL, "vxlt_0_lowtown_n_ehu_0_0.vlm"},
	{0x00DD319E2380FFD2ULL, "art\\vfx\\combat\\vfx_r_arc_front_close_c.impa.ani"},
	{0x00DD34EF61877BE4ULL, "ave200_donnic6112154.cl"},
	{0x00DD823114D8C796ULL, "isa300ip_trail_marker.ncs"},
	{0x00DDB3009C5B406EULL, "combat_deathblow_1.gda"},
	{0x00DE474685FB8675ULL, "vxht_mansion_n_6mb_0_0.vlm"},
	{0x00DEBE5439FA61B1ULL, "vxlt_undercityencounter_rq_0_0.vlm"},
	{0x00DF14871F9D8769ULL, "hlac_ht_0_hightown_n_-196608_2z.dds"},
	{0x00DF2898BC150827ULL, "6068526_m.fxe"},
	{0x00DF861CAFEB9038ULL, "dae170ip_at_templarhall.utp"},
	{0x00DF95D94556B30DULL, "character_character_p53.psh.fxo.dx11"},
	{0x00DF9A8C19B5A054ULL, "vxlt_0_lowtown_docks_b06_0_0.vlm"},
	{0x00DFA897D37485D4ULL, "vxdr_primevaltomb_507_0_2.vlm"},
	{0x00DFD43AF9AC1E6DULL, "hlac_w_bonepitmining_a2_327683_0x.dds"},
	{0x00DFE058FC99EF7DULL, "physics1.psh.fxo"},
	{0x00E02592B5BA0734ULL, "art\\vfx\\combat\\vfx_m_spi_frc_aoe_p.mmh"},
	{0x00E02CC5BE8E97F2ULL, "6156464_m.fxe"},
	{0x00E0B05A238ADD02ULL, "vxht_mansion_n_509_0_0.vlm"},
	{0x00E11D22A642D338ULL, "framebuffereffects_gaussblur.csh.hlsl_p0.csh.fxo.dx11"},
	{0x00E14B3B442F1E60ULL, "64533346.ogg"},
	{0x00E14B3F8AEBA7CEULL, "vxht_0_hightown_ajl_0_0.vlm"},
	{0x00E157EDB60F0618ULL, "dae030cr_lowlife_male.utc"},
	{0x00E188BDE77366F3ULL, "454174884.ogg"},
	{0x00E1BE90F6F4DBC1ULL, "6119695_m.fxe"},
	{0x00E206EE8DF6CCC4ULL, "6110262_m.fxe"},
	{0x00E26C870C25A216ULL, "83078354.ogg"},
	{0x00E27ADBC276A9C7ULL, "lgt000im_ultra_seal.uti"},
	{0x00E30CCB7A75D8EAULL, "240504924.ogg"},
	{0x00E34CF51A415B21ULL, "vxw_bonepit_kp_0_0.vlm"},
	{0x00E3A9E99EA05E54ULL, "115635243.ogg"},
	{0x00E4A28FEDD9E374ULL, "22317869.ogg"},
	{0x00E4C027F5BB3878ULL, "hlac_ca_caverns2prc_5802_49y.dds"},
	{0x00E58D55A706E627ULL, "vxht_mansion2b_1ve_0_0.vlm"},
	{0x00E592963278EC56ULL, "153084807.ogg"},
	{0x00E5973FF86293A2ULL, "vxht_mansion2_1rh_0_0.vlm"},
	{0x00E5982D6B1D0EF1ULL, "dae200_bran6094815.cl"},
	{0x00E5C3667ED32959ULL, "zz_followers_banter6159096.cl"},
	{0x00E5D6DB91FE27B3ULL, "hlac_ca_caverns2prc_5802_38z.dds"},
	{0x00E5D76FC1F49ED3ULL, "vxlt_warehouse_n_137_0_0.vlm"},
	{0x00E60AC1C11C3F73ULL, "dae000ip_crowd_lm5_drunk2.utp"},
	{0x00E6856E3891B424ULL, "vxlt_darktown_5m0_0_0.vlm"},
	{0x00E6904E4112D690ULL, "zz_followers_banter6158742.cl"},
	{0x00E6D38ADD632BEEULL, "6139823_m.fxe"},
	{0x00E739D02828E3EAULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\9ff_cavernflow.mao"},
	{0x00E73DD02828EA39ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\9ff_cavernflow.mmh"},
	{0x00E747D02828FB27ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\9ff_cavernflow.msh"},
	{0x00E74AC9B354FA36ULL, "hlac_w_encounter3_0_6y.dds"},
	{0x00E791161A881B6BULL, "6110258_m.fxe"},
	{0x00E7E50BE121CEB3ULL, "vxca_caverns2_9fr_0_0.vlm"},
	{0x00E7F7487F271FACULL, "6120034_m.fxe"},
	{0x00E80C453A760628ULL, "vxht_mansion_n_6p8_0_0.vlm"},
	{0x00E823682FC02AABULL, "835630958.ogg"},
	{0x00E8BBA6DF3B2110ULL, "330837361.ogg"},
	{0x00E934F2BFB74195ULL, "vxdr_primevaltomb_34r_0_0.vlm"},
	{0x00E975738026DF83ULL, "vxlt_hangedman_25u_0_0.vlm"},
	{0x00E9AFF746D146B0ULL, "hlac_lt_0_lowtown_a2_n_c_65536_5x.dds"},
	{0x00E9BB764CD553A6ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\shield and weapon\\sw_ent_override.evt"},
	{0x00E9BF70FFC8CA97ULL, "vxdr_0_deeproads_cj0_0_0.vlm"},
	{0x00EAA07AAAF5473EULL, "follower_banter6161096.cl"},
	{0x00EAE29276794280ULL, "528518661.ogg"},
	{0x00EB5E5BA7306D9BULL, "w_sundermount_127_0.mmh"},
	{0x00EB6A5BA7308238ULL, "w_sundermount_127_0.mao"},
	{0x00EB7C5BA730A085ULL, "w_sundermount_127_0.msh"},
	{0x00EBCE878211075EULL, "6087281_m.fxe"},
	{0x00EC1FEBD0DA5451ULL, "character_character_p71.psh.fxo.dx11.assembly"},
	{0x00EC4D70BCAEC2EFULL, "194588415.ogg"},
	{0x00EC851C01C2552AULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\0wi_basic.phy"},
	{0x00ECE2EF461E6F0EULL, "vxht_brothel_n_5gk_0_0.vlm"},
	{0x00ECF1F1C4CDD1F1ULL, "610810005.ogg"},
	{0x00ED2E33385F6A9DULL, "hlac_w_sundermount_65532_0y.dds"},
	{0x00ED89EE8BA54C11ULL, "mag170_hugh6080708.cl"},
	{0x00ED98733C5FD856ULL, "mag170_thrask6054044.cl"},
	{0x00EED12AD7A35BB6ULL, "vxlt_undercityserialkiller_wu_0_0.vlm"},
	{0x00EF3579B7A1F2F5ULL, "6129740_m.fxe"},
	{0x00EF582583493D52ULL, "vxlt_warehouse_n_132_0_0.vlm"},
	{0x00EFD2764CD9E6F1ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\shield and weapon\\sw_ent_override.gad"},
	{0x00F02E5A38FD6238ULL, "zz_approval_debug6003553.cl"},
	{0x00F0459079D74E99ULL, "lgt000ip_tombs_seal.ncs"},
	{0x00F0944D9033C450ULL, "6209891_f.fxe"},
	{0x00F12A872D359F3FULL, "601022978.ogg"},
	{0x00F14B6A44FEC78AULL, "437129781.ogg"},
	{0x00F15F62F23269D5ULL, "6119321_m.fxe"},
	{0x00F20DF24F7FDA61ULL, "vxlt_warehouse_hz_0_0.vlm"},
	{0x00F21663825B4BD5ULL, "3032894.ogg"},
	{0x00F3E1A61B7BF02DULL, "hlac_lt_0_lowtown_docks_a3_n_65537_0y.dds"},
	{0x00F4091CEA58AC22ULL, "vxht_mansion2_1ui_0_0.vlm"},
	{0x00F4CD274272A9EDULL, "437099647.ogg"},
	{0x00F4D0F625577F45ULL, "6135823_f.fxe"},
	{0x00F4E9DF86451327ULL, "hlac_lt_0_lowtown_a2_n_65534_3y.dds"},
	{0x00F51A707DD755A3ULL, "t1_skn_h03.tnt"},
	{0x00F54EE5F116E65BULL, "vxlt_gallowstemplar_n_5qx_0_0.vlm"},
	{0x00F56CAAA0734009ULL, "llac_ht_0_hightown_a2_65536_1.dds"},
	{0x00F5800AB1232476ULL, "vxht_keep_6hu_0_0.vlm"},
	{0x00F582A016094124ULL, "lt_0_lowtown_docks_130_0.msh"},
	{0x00F594A016095F8DULL, "lt_0_lowtown_docks_130_0.mao"},
	{0x00F598A0160966C6ULL, "lt_0_lowtown_docks_130_0.mmh"},
	{0x00F5D142694C4E52ULL, "vxht_0_hightown_a2_9b4_0_0.vlm"},
	{0x00F5D1E2AB5F3B9EULL, "art\\characters\\playercharacter\\humanmale\\animation\\ambient\\ambient\\am_weep_lp.evt"},
	{0x00F62CDC4358B701ULL, "6116492_m.fxe"},
	{0x00F6D3764CE0069FULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\shield and weapon\\sw_ent_override.ani"},
	{0x00F70C7177EA564CULL, "vxca_caverns2prc_9jh_0_0.vlm"},
	{0x00F715A88D0BFEFCULL, "llac_w_sundermount_65541_0.dds"},
	{0x00F74D06B90B5D63ULL, "vxlt_gallowscourtyard_4v5_0_0.vlm"},
	{0x00F77140E4EE105BULL, "vxlt_gallowstemplar_f_55t_0_0.vlm"},
	{0x00F7E750721CF803ULL, "vxlt_hangedman_22t_0_0.vlm"},
	{0x00F81DB5F105D31EULL, "art\\vfx\\combat\\vfx_glb_com_disorient_s2_c.cess.ani"},
	{0x00F8791B26920958ULL, "6164180_m.fxe"},
	{0x00F8DDAFCC3E99F2ULL, "6087634_m.fxe"},
	{0x00F8F744712E9165ULL, "qun110_mugger6055638.cl"},
	{0x00F9395681DD7708ULL, "vxht_mansion2_2e8_0_0.vlm"},
	{0x00F94496704AB55FULL, "vxlt_gallowsprison_n_6bf_0_0.vlm"},
	{0x00F9ACD3D9225525ULL, "6135067_f.fxe"},
	{0x00FA19C70D34DCB5ULL, "vxlt_gallowstemplar_n_5cp_0_0.vlm"},
	{0x00FA23F0FA236DDCULL, "gen00fl_fenris6110842.cl"},
	{0x00FA39DE65375AAFULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\r_p.gad"},
	{0x00FAC08757B20BCBULL, "hlac_lt_gallowstemplar_2388_0x.dds"},
	{0x00FAE12D8E5EE448ULL, "143043216.ogg"},
	{0x00FB54CC3814878EULL, "vxht_brothel_n_5jl_0_0.vlm"},
	{0x00FBC2EB665D0988ULL, "ser210_witness6168145.cl"},
	{0x00FBE8E2AB63CEC9ULL, "art\\characters\\playercharacter\\humanmale\\animation\\ambient\\ambient\\am_weep_lp.gad"},
	{0x00FBEAF54FD320AFULL, "llac_ht_chantry_n_2534_47.dds"},
	{0x00FC6060D7A7A23CULL, "llac_lt_warehouse_n_1_3.dds"},
	{0x00FCA7F64780A6ACULL, "20731078.ogg"},
	{0x00FD6CF4376FF93EULL, "vxdr_0_deeproads_bcc_0_0.vlm"},
	{0x00FDFF5F81AC5561ULL, "blank_exterior_mmi.gff"},
	{0x00FE1F4B777C72EEULL, "273181833.ogg"},
	{0x00FE3A103E9D5DF7ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\head_look_up.gad"},
	{0x00FEA24ADAE10CC1ULL, "6113095_m.fxe"},
	{0x00FEDBF5B28FA8A5ULL, "vxca_caverns1_igt_0_0.vlm"},
	{0x010042B62CC68459ULL, "6064029_m.fxe"},
	{0x01007BB52DB0DB27ULL, "126194848.ogg"},
	{0x010089F49F70EE60ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\0ww_grandright_1.mao"},
	{0x0100CF188AE10B2AULL, "art\\characters\\playercharacter\\humanmale\\textures\\hm_cth_noba_0d.dds"},
	{0x01010C3DE73E7201ULL, "787507502.ogg"},
	{0x010116FFC09E035BULL, "vxht_bartrand_n_3yr_0_0.vlm"},
	{0x01014EDE653D9D5CULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\r_p.evt"},
	{0x01016AEF341B616DULL, "crt_cycle_4.dds"},
	{0x0101BB1DBB102CF9ULL, "art\\characters\\creatures\\dragon\\animation\\tal_ovlm_ex.ani"},
	{0x0101C35BA9947B3DULL, "llac_w_encounter1_0_2.dds"},
	{0x01022D02493B455CULL, "hlac_ht_keep_n_4519_17y.dds"},
	{0x0102B9E2AB699D17ULL, "art\\characters\\playercharacter\\humanmale\\animation\\ambient\\ambient\\am_weep_lp.ani"},
	{0x0102D17DA41F91B3ULL, "hlac_dr_dungeon_3217_3y.dds"},
	{0x0102FD8DA26289EFULL, "var_merchant.gda"},
	{0x01037042541B145CULL, "art\\vfx\\combat\\vfx_glb_pois_act_c.impa.ani"},
	{0x01038D3114F9A9A3ULL, "isa300ip_trail_marker.utp"},
	{0x0103C16F6908319EULL, "vxlt_0_lowtown_n_f0f_0_0.vlm"},
	{0x0103DC5535BF10D5ULL, "tools_sound.phy"},
	{0x01044965DB0B970CULL, "840089393.ogg"},
	{0x010457763B36C761ULL, "306192549.ogg"},
	{0x01047016AA64250BULL, "6183957_m.fxe"},
	{0x01048BE67F8AB903ULL, "6208052_m.fxe"},
	{0x0104CE59710B142DULL, "art\\characters\\playercharacter\\humanfemale\\cth_comc.mmh"},
	{0x0104D259710B1B7EULL, "art\\characters\\playercharacter\\humanfemale\\cth_comc.mao"},
	{0x0104D541EF9D93E2ULL, "380246162.ogg"},
	{0x0104E059710B32B3ULL, "art\\characters\\playercharacter\\humanfemale\\cth_comc.msh"},
	{0x01054F103EA39F84ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\head_look_up.evt"},
	{0x01059A30C0243A96ULL, "lowtown_male_alienage1_sta6162776.cl"},
	{0x010635B5F11239EFULL, "art\\vfx\\combat\\vfx_glb_com_disorient_s2_c.cess.evt"},
	{0x010640279D0C7646ULL, "925420644.ogg"},
	{0x01068A1CF3E5FB68ULL, "129277583.ogg"},
	{0x0106B470903BDF02ULL, "6127257_m.fxe"},
	{0x0107600D85D70BA9ULL, "267281836.ogg"},
	{0x01077A09B1D5C1DEULL, "mag110cr_vincento.utc"},
	{0x0107814E4746C9B1ULL, "665893479.ogg"},
	{0x01079F0CAD20B885ULL, "hlac_lt_gallowscourtyard_-2_3y.dds"},
	{0x0108229E552FB27CULL, "gen00fl_aveline6128549.cl"},
	{0x0108961DBB160C3BULL, "art\\characters\\creatures\\dragon\\animation\\tal_ovlm_ex.gad"},
	{0x0108A5646CA6D8D4ULL, "art\\cinematics\\_crit path\\prologue\\pro000cs_victim\\cs_player_victim.gad"},
	{0x0108F8F7A7DC3601ULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\3ww_pillar.phy"},
	{0x0108F9AC415E162FULL, "6101952_f.fxe"},
	{0x01092892B5DD11EBULL, "art\\vfx\\combat\\vfx_m_spi_frc_aoe_p.anb"},
	{0x010942727B270A1FULL, "6106917_f.fxe"},
	{0x01094ED2F608377BULL, "hlac_ht_chantry_2534_51z.dds"},
	{0x010975EE29505833ULL, "388531366.ogg"},
	{0x01097D42541F959EULL, "art\\vfx\\combat\\vfx_glb_pois_act_c.impa.gad"},
	{0x01099AD1672E27CCULL, "mag291ip_chest.utp"},
	{0x0109C6A92A0AA00EULL, "vxht_brothel_n_5mm_0_0.vlm"},
	{0x0109C929D5952D93ULL, "art\\levels\\races\\dwarf\\props\\_textures\\prp_hammer_r.dds"},
	{0x0109EE462A7BED28ULL, "137590041.ogg"},
	{0x010A0621396A0DA7ULL, "vxlt_0_lowtown_a2_n_gbx_0_0.vlm"},
	{0x010A51105F62AA77ULL, "786950035.ogg"},
	{0x010A8A885C1AFE71ULL, "blood_conj_01_l3.mao"},
	{0x010AD1369C12D9E6ULL, "6136003_f.fxe"},
	{0x010AD870E73A54B5ULL, "ico_boots_light_2.dds"},
	{0x010B0B8055703875ULL, "vxlt_undercitydungeon_189_0_0.vlm"},
	{0x010B15704C7B2438ULL, "124445040.ogg"},
	{0x010B4C474B9A6D01ULL, "llac_ht_0_hightown_n_-1_1.dds"},
	{0x010B6A9895001DF3ULL, "985165113.ogg"},
	{0x010BB61C9A7DE02EULL, "art\\vfx\\combat\\vfx_w_2nd_atk_lng_c.mmh"},
	{0x010BFAEFF9BD075CULL, "473932488.ogg"},
	{0x010C04B79577EFCBULL, "vxlt_0_lowtown_a2_n_byx_0_0.vlm"},
	{0x010C25569EEDD472ULL, "hlac_ht_0_hightown_a2_131073_0x.dds"},
	{0x010C26AC0048FC46ULL, "6114466_m.fxe"},
	{0x010C2D4C4F3C08BCULL, "hlac_lt_0_lowtown_docks_-196610_2z.dds"},
	{0x010C31C1AC38156CULL, "vxdr_0_deeproads_fht_0_0.vlm"},
	{0x010C36B5F116A698ULL, "art\\vfx\\combat\\vfx_glb_com_disorient_s2_c.cess.gad"},
	{0x010CF3B9076AB940ULL, "6005304_m.fxe"},
	{0x010CFB6EA9C4CECCULL, "38878772.ogg"},
	{0x010D4DD2A485C125ULL, "vxca_caverns1_ijs_0_0.vlm"},
	{0x010D94A85FFFBF90ULL, "physics2.vsh.fxo"},
	{0x010DBE7B21BB4595ULL, "1071404251.ogg"},
	{0x010E431CD61E9AD5ULL, "6113521_m.fxe"},
	{0x010E5D71676F30A7ULL, "6152722_m.fxe"},
	{0x010ECD928D981722ULL, "qun160_vashothlead6053926.cl"},
	{0x010EF1AC336C0B61ULL, "vxlt_warehouse_vx_0_0.vlm"},
	{0x010F1C93C996487AULL, "323927471.ogg"},
	{0x010F56DE6549E875ULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\r_p.ani"},
	{0x010FB31DBB1C5C20ULL, "art\\characters\\creatures\\dragon\\animation\\tal_ovlm_ex.evt"},
	{0x010FB4646CAD10B2ULL, "art\\cinematics\\_crit path\\prologue\\pro000cs_victim\\cs_player_victim.ani"},
	{0x010FEA3EAE113B74ULL, "vxlt_undercitydungeon_1ke_0_0.vlm"},
	{0x010FF7E027BD4539ULL, "llac_ca_cavern1_haunted_24425_11.dds"},
	{0x01100EB95967626CULL, "art\\characters\\playercharacter\\humanmale\\glv_wic.phy"},
	{0x01105842542575BDULL, "art\\vfx\\combat\\vfx_glb_pois_act_c.impa.evt"},
	{0x0110620B13F7727FULL, "998393354.ogg"},
	{0x0110ECD6CE44DD22ULL, "vxht_mansion2_1ck_0_0.vlm"},
	{0x0110FD4DE5F0AC63ULL, "hlac_ca_caverns1_mines_24424_25z.dds"},
	{0x0111225B246837D3ULL, "zz_followers_banter6158646.cl"},
	{0x011177CEC9E16E7BULL, "zz_qcr_debug6052119.cl"},
	{0x0112329FD503175BULL, "vxlt_gallowstemplar_n_5oz_0_0.vlm"},
	{0x011267103EAE537DULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\head_look_up.ani"},
	{0x0112B3C1E0837664ULL, "6169741_m.fxe"},
	{0x0112D7605290F997ULL, "271606913.ogg"},
	{0x0112F681CDA56E93ULL, "6055613_f.fxe"},
	{0x0113663AE69682F0ULL, "vxca_caverns1_d5z_0_0.vlm"},
	{0x0113984428897F07ULL, "254143813.ogg"},
	{0x0114FE5ECB115437ULL, "6121966_m.fxe"},
	{0x0115C75971197BEAULL, "art\\characters\\playercharacter\\humanfemale\\cth_comc.phy"},
	{0x01160BF628F61B90ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\3ww12_galloalcovlo.mao"},
	{0x01160FF628F622A3ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\3ww12_galloalcovlo.mmh"},
	{0x01161DF628F63AFDULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\3ww12_galloalcovlo.msh"},
	{0x0116820FA6B89848ULL, "6164809_m.fxe"},
	{0x0116E2E0D13866F0ULL, "w_bonepit_56_0.mao"},
	{0x0116E6E0D1386D03ULL, "w_bonepit_56_0.mmh"},
	{0x0116F4E0D138855DULL, "w_bonepit_56_0.msh"},
	{0x0117763EEBBFC653ULL, "6061235_m.fxe"},
	{0x01179A5B873D1AABULL, "llac_lt_0_lowtown_a2_n_c_131071_0.dds"},
	{0x0117BC81CA913247ULL, "llac_lt_gallowstemplar_n_2388_5.dds"},
	{0x0117DC367036B8A1ULL, "vxdr_primevaltomb_5ao_0_0.vlm"},
	{0x0117E8FFFA8C0735ULL, "vxdr_0_deeproads_d36_0_0.vlm"},
	{0x0118CB8CC9ECB419ULL, "art\\characters\\creatures\\ashwraith\\animation\\ras_crnf.ani"},
	{0x0118E114D2E36056ULL, "vxht_0_hightown_5eq_0_0.vlm"},
	{0x01190729F386AD1EULL, "6075288_m.fxe"},
	{0x011938249CB28DD0ULL, "hlac_lt_0_lowtown_a2_-262143_0z.dds"},
	{0x011943138E08F266ULL, "6131103_f.fxe"},
	{0x01197959A32D6F34ULL, "hlac_ht_keep_3732_0x.dds"},
	{0x0119D51C9A8A522DULL, "art\\vfx\\combat\\vfx_w_2nd_atk_lng_c.anb"},
	{0x011A444CECB86AD2ULL, "follower_banter6151284.cl"},
	{0x011A462C7F6D69D7ULL, "dae000_flemeth6000681.cl"},
	{0x011A7EDFB28E3E53ULL, "6119099_m.fxe"},
	{0x011B8D1893046BEBULL, "hlac_lt_warehouse_n_65536_2x.dds"},
	{0x011BBFAF967BD9A5ULL, "vxca_caverns1_imr_0_0.vlm"},
	{0x011BFE9FBB55094CULL, "33440783.ogg"},
	{0x011C058829A5A8C3ULL, "6020423_m.fxe"},
	{0x011C7F72C36848BDULL, "6023954_m.fxe"},
	{0x011C8A2DFED965E1ULL, "6045107_m.fxe"},
	{0x011CE19B1CDF3991ULL, "t1_skn_e04.tnt"},
	{0x011D2041B143EA87ULL, "art\\vfx\\environment\\level_fx\\smoke\\vfx_foundryfire2.dura.gad"},
	{0x011D2FEB24028A28ULL, "vrc221_varric6048576.cl"},
	{0x011D8EEA70D464B1ULL, "6054031_m.fxe"},
	{0x011E51B0273AC2BDULL, "gen_im_trs_chn_shd_wdn_r2.uti"},
	{0x011E5C1BA00753F4ULL, "vxlt_undercitydungeon_1nj_0_0.vlm"},
	{0x011E649C3CFBB5C1ULL, "sib111_mother6098960.cl"},
	{0x011ED719DABDAC4DULL, "vxlt_foundry_n_zo_0_0.vlm"},
	{0x011F219404818A80ULL, "6094769_railing1.cut"},
	{0x011FA68CC9F2945BULL, "art\\characters\\creatures\\ashwraith\\animation\\ras_crnf.gad"},
	{0x0120252EBA1CE904ULL, "614766196.ogg"},
	{0x0120B98FCAED64AAULL, "765117665.ogg"},
	{0x0120C9C726429C29ULL, "vxlt_gallowstemplar_5cc_0_0.vlm"},
	{0x01211F3CF1245794ULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_landscapes_d.dds"},
	{0x01212FDE27846117ULL, "6096260_m.fxe"},
	{0x012146B77500027AULL, "6173047_m.fxe"},
	{0x0121A329FC9E6433ULL, "vxht_0_hightown_aoz_0_0.vlm"},
	{0x012285FDA3F08232ULL, "p_cyc.mao"},
	{0x01229B6CCC772B03ULL, "mag291_demon_marethari6027832.cl"},
	{0x0122E6E877C12A75ULL, "vxdr_primevaltomb_vy_0_0.vlm"},
	{0x012324591D3E721BULL, "vxlt_undercitydungeon_143_0_0.vlm"},
	{0x0123ABEF6D81B168ULL, "lt_0_lowtown_docks_a3_102_0.msh"},
	{0x0123BDEF6D81CFF9ULL, "lt_0_lowtown_docks_a3_102_0.mao"},
	{0x0123BEC874C14776ULL, "art\\characters\\creatures\\strider\\animation\\attackbackleft.evt"},
	{0x0123C9EF6D81E392ULL, "lt_0_lowtown_docks_a3_102_0.mmh"},
	{0x0123D6BEAE12153CULL, "dae000_mother6121048.cl"},
	{0x01240B75F5623F25ULL, "56645124.ogg"},
	{0x01241541B149F694ULL, "art\\vfx\\environment\\level_fx\\smoke\\vfx_foundryfire2.dura.evt"},
	{0x0125A0BEB7E78CF5ULL, "6071690_m.fxe"},
	{0x0125B39C968B9393ULL, "vxlt_hangedman_n_1t6_0_0.vlm"},
	{0x0126353A6B765499ULL, "108928135.ogg"},
	{0x01264F912C85A688ULL, "zz_ave_debug6056781.cl"},
	{0x0126648C7D6255EAULL, "fen110_anso6094958.cl"},
	{0x0126BD1F9545A718ULL, "hlac_w_woundedcoast_131072_0y.dds"},
	{0x0126C38CC9F8E440ULL, "art\\characters\\creatures\\ashwraith\\animation\\ras_crnf.evt"},
	{0x0126D2B58EA776E6ULL, "vxlt_0_lowtown_n_feu_0_0.vlm"},
	{0x01270BEB92F36C78ULL, "vxlt_backalley_g_1li_0_0.vlm"},
	{0x012710F6290497C8ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\3ww12_galloalcovlo.phy"},
	{0x01291217D2904F26ULL, "screentexrect.psh.fxo"},
	{0x0129180F4937AB66ULL, "hlac_lt_0_lowtown_n_65537_5y.dds"},
	{0x012984B887928882ULL, "6060450_f.fxe"},
	{0x0129B5C874C5A301ULL, "art\\characters\\creatures\\strider\\animation\\attackbackleft.gad"},
	{0x012A77651B9BC143ULL, "6027523_m.fxe"},
	{0x012A97B5606AF907ULL, "vxht_0_hightown_a2_amt_0_0.vlm"},
	{0x012BA3F48A00FCD2ULL, "w_encounter1_13_0_ll.mmh"},
	{0x012C11884576479EULL, "689008926.ogg"},
	{0x012C144C62775FECULL, "814973217.ogg"},
	{0x012C5D4318A85157ULL, "t3_wdg_w06.tnt"},
	{0x012C7C3A979D63C1ULL, "6076314_m.fxe"},
	{0x012CD86B16F48E02ULL, "44776808.ogg"},
	{0x012D488C85F17E10ULL, "6135985_f.fxe"},
	{0x012D846A8C48DA96ULL, "743929453.ogg"},
	{0x012DC2A9EC61BBB1ULL, "257990337.ogg"},
	{0x012DD090B2310E22ULL, "vxht_mansion2_1iu_0_0.vlm"},
	{0x012DEC623210291DULL, "162851701.ogg"},
	{0x012E07FB96DC6503ULL, "6135627_f.fxe"},
	{0x012EE10AF231C3A4ULL, "vxlt_0_lowtown_docks_a3_n_d3i_0_0.vlm"},
	{0x012F3ECC86606343ULL, "vxw_bonepitmining_a2_6g_0_0.vlm"},
	{0x012F72DEE00CC0A7ULL, "129556009.ogg"},
	{0x012F8582BE3E9466ULL, "931499584.ogg"},
	{0x012FA4122C223AD0ULL, "vxw_sundermount_25l_0_0.vlm"},
	{0x0130774418722DADULL, "vxht_0_hightown_a3_c86_0_0.vlm"},
	{0x0130A10078CD784DULL, "vxlt_gallowstemplar_f_5qf_0_0.vlm"},
	{0x0130B6C874CBC3AFULL, "art\\characters\\creatures\\strider\\animation\\attackbackleft.ani"},
	{0x0130F3E2C8BB8A83ULL, "mrl150tr_mtn_undead_2.ncs"},
	{0x0131E982EFD59C66ULL, "fx_magma.mao"},
	{0x0131F67E87B42EC0ULL, "hlac_ht_mansion2_818_66z.dds"},
	{0x01320D41B15626CDULL, "art\\vfx\\environment\\level_fx\\smoke\\vfx_foundryfire2.dura.ani"},
	{0x01323EBC024DC36FULL, "347433975.ogg"},
	{0x0132582CF6BB92C9ULL, "22864467.ogg"},
	{0x01327314F7F5EF06ULL, "696922482.ogg"},
	{0x01328371A4AEE3BEULL, "6174774_m.fxe"},
	{0x0132BE3DF07A4E68ULL, "vxht_brothel_n_53a_0_0.vlm"},
	{0x01348E20EB10648DULL, "llac_w_sundermount_-262142_0.dds"},
	{0x0134D85ED2B14B53ULL, "vxlt_0_lowtown_a3_n_c_eje_0_0.vlm"},
	{0x01354BBEDDD6F27FULL, "741108543.ogg"},
	{0x0135803867154B27ULL, "6086852_m.fxe"},
	{0x0135AF1F6174F4E6ULL, "hlac_lt_0_lowtown_a3_n_c_-3_3x.dds"},
	{0x0135E741485C6BAFULL, "nodiffuse.dds"},
	{0x01367970C32F3492ULL, "art\\vfx\\cinematic\\vfx_orsino_swirly_c.impa.gad"},
	{0x01369AB1FF1A209CULL, "vxlt_gallowstemplar_n_687_0_0.vlm"},
	{0x0136B69C77F925B5ULL, "6127027_m.fxe"},
	{0x01372BF5C5D66FECULL, "vxlt_0_lowtown_a2_n_fi7_0_0.vlm"},
	{0x0137BA517C0C76CAULL, "6053929_m.fxe"},
	{0x01381139B6670AC7ULL, "309872747.ogg"},
	{0x0138D596DF2D0C1BULL, "hlac_lt_blackemporium_1055_2y.dds"},
	{0x0138DA73035AEC00ULL, "6208279_m.fxe"},
	{0x01393FBC4F6C8563ULL, "llac_dr_0_deeproads_5842_15.dds"},
	{0x01395E5BA7728F28ULL, "w_sundermount_127_0.rml"},
	{0x013971D880A5993BULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\mage\\mw_rng_atk_v4.evt"},
	{0x013975D2BD66C94FULL, "vxlt_0_lowtown_h2d_0_0.vlm"},
	{0x0139782E93A012BFULL, "mrl210_merrill6117417.cl"},
	{0x0139BDA5A60F96FEULL, "6160645_m.fxe"},
	{0x0139E6708359127FULL, "hlac_w_encounter1_0_0x.dds"},
	{0x013A202C551B2C67ULL, "6037391_m.fxe"},
	{0x013A90A42BB14AD6ULL, "art\\levels\\races\\ferelden\\props\\plotspecifc\\prp_candle.msh"},
	{0x013A9AA42BB15BD4ULL, "art\\levels\\races\\ferelden\\props\\plotspecifc\\prp_candle.mmh"},
	{0x013A9EA42BB162E7ULL, "art\\levels\\races\\ferelden\\props\\plotspecifc\\prp_candle.mao"},
	{0x013AD249563419ECULL, "hlac_w_woundedcoast_-131067_0z.dds"},
	{0x013B386961E62802ULL, "vxw_encounter2_camp_q1_0_0.vlm"},
	{0x013B5596AFED28B1ULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\custom\\cuddle_lying_female_loop.ani"},
	{0x013BC067DF63294CULL, "521157243.ogg"},
	{0x013C1EB8D1D9EAA6ULL, "hlac_w_sundermount_-3_0y.dds"},
	{0x013C3004121031DCULL, "gen00fl_varric6076210.cl"},
	{0x013C30FE5F05C672ULL, "vxca_caverns2prc_7o7_0_0.vlm"},
	{0x013C6AF7691B91CCULL, "art\\levels\\races\\ferelden\\props\\plotspecifc\\prp_whoreflower.phy"},
	{0x013CC0CE9BF11A2DULL, "6167193_m.fxe"},
	{0x013CC50C83C22C59ULL, "hlac_dr_primevalentrance_1726_9x.dds"},
	{0x013CCAA252760A50ULL, "493272999.ogg"},
	{0x013CE099C0D23EB4ULL, "343299574.ogg"},
	{0x013D308FC5883493ULL, "6129333_m.fxe"},
	{0x013D48AA19FCECFDULL, "6098206_m.fxe"},
	{0x013D7C70C33558C9ULL, "art\\vfx\\cinematic\\vfx_orsino_swirly_c.impa.evt"},
	{0x013D9E754324070CULL, "and100_anders6095972.cl"},
	{0x013DF958A812EBC5ULL, "hla_ht_0_hightown_a3_65534_0y.dds"},
	{0x013EFA0404FEDE3FULL, "6089732_m.fxe"},
	{0x013F6AD880A9F97CULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\mage\\mw_rng_atk_v4.gad"},
	{0x013FAD1E5F891E84ULL, "866068617.ogg"},
	{0x01400E7395361879ULL, "866889873.ogg"},
	{0x014029F3C80AD1D5ULL, "mag320_orsino6071237.cl"},
	{0x014031F0B4020620ULL, "mag321_comtesse_de_launcet6064905.cl"},
	{0x01404CB08ABBB3E8ULL, "6173409_m.fxe"},
	{0x0140E4A8649D527BULL, "hlac_lt_0_lowtown_a2_65533_3y.dds"},
	{0x0140E79DE9519F92ULL, "one100dg_lia6064065.cl"},
	{0x0141010CE37BB9B6ULL, "502891069.ogg"},
	{0x0141204AD0571C2EULL, "6081816_m.fxe"},
	{0x01413793AA04C8A3ULL, "art\\characters\\creatures\\gibgolem\\animation\\body\\gibsgolem_walkr.ani"},
	{0x01426096AFF35A53ULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\custom\\cuddle_lying_female_loop.gad"},
	{0x0142724ED0396EC6ULL, "head_flipv2.dds"},
	{0x014287471D0CBAC8ULL, "6122043_m.fxe"},
	{0x0142D330067B50CAULL, "145825638.ogg"},
	{0x0143063D7B021C12ULL, "vxlt_0_lowtown_a2_n_c_hq1_0_0.vlm"},
	{0x0143BD625CA6D3DCULL, "vxca_caverns1_i13_0_0.vlm"},
	{0x0143E2C1298D1582ULL, "6139658_m.fxe"},
	{0x0143F13B387B56FEULL, "172994011.ogg"},
	{0x01442A86476C550DULL, "1005578092.ogg"},
	{0x014482A012A6C1CDULL, "250315779.ogg"},
	{0x01450C8EF110391CULL, "vxlt_gallowstemplar_n_654_0_0.vlm"},
	{0x0145450AEF991EECULL, "gen00fl_dog6167696.cl"},
	{0x014638A8407667C9ULL, "vxw_bonepitmining_a2_1wl_0_1.vlm"},
	{0x014669D880B0167AULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\mage\\mw_rng_atk_v4.ani"},
	{0x0146A1FB7CEC8CFBULL, "zz_one_debug6074012.cl"},
	{0x0146C1998CB491EBULL, "vxlt_undercitydungeon_17j_0_0.vlm"},
	{0x0146C8F5923069A1ULL, "6137202_m.fxe"},
	{0x014759FA6CDB2ED2ULL, "368571035.ogg"},
	{0x0147D629BDC0829AULL, "vxht_0_hightown_7lx_0_0.vlm"},
	{0x014802E14DBEED2CULL, "6036503_f.fxe"},
	{0x014840C89DBC5748ULL, "hlac_w_woundedcoast_5_0x.dds"},
	{0x0148428B34FC7477ULL, "prm000im_evra_belt.uti"},
	{0x01485D96AFF7C0B8ULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\custom\\cuddle_lying_female_loop.evt"},
	{0x0148F34CD24BC17DULL, "gen_im_npc_arm_hum_boo_cotc.uti"},
	{0x014997AA81929C6EULL, "vxw_bonepitmining_a2_n6_0_0.vlm"},
	{0x01499CB31D4534EDULL, "vxw_bonepitmining_a2_gv_0_0.vlm"},
	{0x0149AA4653DC4082ULL, "vxw_encounter2_camp_l2_0_0.vlm"},
	{0x0149BC8C2BDF8395ULL, "zz_one000_crafter6179395.cl"},
	{0x014A7AA3EC6090EFULL, "art\\levels\\races\\ferelden\\props\\lowerclass\\prp_cratesmred_0.pwk"},
	{0x014AC63026A835E2ULL, "6155292_m.fxe"},
	{0x014AC892977F21FEULL, "830992032.ogg"},
	{0x014AEF5535FB33F0ULL, "tools_sound.msh"},
	{0x014AFD5535FB4B0AULL, "tools_sound.mmh"},
	{0x014B34262CF68E7FULL, "isa311_isabela6092510.cl"},
	{0x014B5D256F89C314ULL, "414734373.ogg"},
	{0x014B6470C3416D38ULL, "art\\vfx\\cinematic\\vfx_orsino_swirly_c.impa.ani"},
	{0x014B6589E1604F1DULL, "vxlt_warehouse_11x_0_0.vlm"},
	{0x014B6DEA47F778AFULL, "6140942_f.fxe"},
	{0x014B92EC3E3A9520ULL, "vxca_caverns2_92x_0_1.vlm"},
	{0x014C22866A4C9443ULL, "vxw_bonepitmining_a2_8i_0_0.vlm"},
	{0x014C4F86A526E65FULL, "vxht_mansion_n_4ab_0_0.vlm"},
	{0x014CC8940247A9A3ULL, "mag370_meredith6046874.cl"},
	{0x014D2410E1EBB334ULL, "6151236_m.fxe"},
	{0x014D2A3785788455ULL, "vxca_cavern1_haunted_e4l_0_0.vlm"},
	{0x014D6AE54C825AA7ULL, "zz_vault_debug6177766.cl"},
	{0x014E078F33247033ULL, "llac_ca_caverns2_5802_0.dds"},
	{0x014E337A0A532782ULL, "hlac_ht_bartrand_n_3374_0x.dds"},
	{0x014EB38A9CF0ACAAULL, "zz_otl_debug6060958.cl"},
	{0x014F4F93AA112E9AULL, "art\\characters\\creatures\\gibgolem\\animation\\body\\gibsgolem_walkr.evt"},
	{0x014FE464E8C779C8ULL, "llac_ht_0_hightown_a2_-131074_0.dds"},
	{0x014FF778E015D973ULL, "vxw_woundedcoast_4f_0_0.vlm"},
	{0x0150E5B9599EB337ULL, "art\\characters\\playercharacter\\humanmale\\glv_wic.mmh"},
	{0x0150E9B9599EBA44ULL, "art\\characters\\playercharacter\\humanmale\\glv_wic.mao"},
	{0x0150FBB9599ED8A9ULL, "art\\characters\\playercharacter\\humanmale\\glv_wic.msh"},
	{0x0150FD3C10B690A1ULL, "382643223.ogg"},
	{0x0152502A0B46B08DULL, "vxlt_0_lowtown_a2_n_c_ejf_0_0.vlm"},
	{0x01527981B94CC797ULL, "300814449.ogg"},
	{0x015292F002AC074DULL, "one220_yevhen6115738.cl"},
	{0x0152A666450ED107ULL, "6122817_m.fxe"},
	{0x0152A77A4A8E2442ULL, "vxht_brothel_n_5b8_0_0.vlm"},
	{0x015303F62E2578D0ULL, "mag360_samson6044901.cl"},
	{0x01537B215C0B0B2CULL, "gen_im_npc_arm_hum_glv_cota.uti"},
	{0x01539528E0ACCA95ULL, "335745138.ogg"},
	{0x0153E35FA48885EDULL, "lt_0_lowtown_docks_a3_n_252_0.mmh"},
	{0x0153E75FA4888C3EULL, "lt_0_lowtown_docks_a3_n_252_0.mao"},
	{0x0153F55FA488A473ULL, "lt_0_lowtown_docks_a3_n_252_0.msh"},
	{0x0153FF685A8EA83BULL, "vxw_bonepit_1wh_0_1.vlm"},
	{0x0154416ADAEC5989ULL, "hlac_lt_darktown_-1_84y.dds"},
	{0x0154A6CC49B1A7F4ULL, "llac_lt_backalley_g_65536_1.dds"},
	{0x0154A6FFFD2A70BFULL, "pro000_refugee_angry6052171.cl"},
	{0x0154E82043F10FC9ULL, "vxca_cavern1_haunted_j29_0_0.vlm"},
	{0x015525854AAA9523ULL, "art\\vfx\\combat\\vfx_m_sta_spi_range_f_p.anb"},
	{0x0155637D0F5B1E05ULL, "vxlt_backalley_g_1lp_0_0.vlm"},
	{0x01558E93278468BAULL, "art\\characters\\playercharacter\\humanmale\\textures\\hm_cth_noba_0t.dds"},
	{0x0155E851C3E4D5D2ULL, "6161619_m.fxe"},
	{0x015646F346309F1EULL, "359616565.ogg"},
	{0x01564E93AA174B8DULL, "art\\characters\\creatures\\gibgolem\\animation\\body\\gibsgolem_walkr.gad"},
	{0x01565EAB438EE962ULL, "361616212.ogg"},
	{0x0156ACEFFC96E0DEULL, "follower_banter6170645.cl"},
	{0x0156C0E31A3D43E1ULL, "art\\cinematics\\_crit path\\mage crisis\\gallows arrival\\gallowsarrive_follower2_varric_.gad"},
	{0x01574C0FBAFE4C34ULL, "308458129.ogg"},
	{0x0157DEF065A8C45FULL, "6124741_m.fxe"},
	{0x01580E0BE72558EEULL, "667286197.ogg"},
	{0x01580E900F3B4D6DULL, "vxw_bonepitmining_a2_bu_0_0.vlm"},
	{0x01581C2345D25902ULL, "vxw_encounter2_camp_g3_0_0.vlm"},
	{0x015828D6E6F975BAULL, "vxlt_gallowstemplar_n_6sd_0_0.vlm"},
	{0x0158602D2F2D1E87ULL, "dae000_fenarel6147434.cl"},
	{0x01588EC1508640B9ULL, "vxca_caverns2prc_9fe_0_0.vlm"},
	{0x0158975ABFE2A788ULL, "341775883.ogg"},
	{0x01596A2F065A4E59ULL, "6175998_m.fxe"},
	{0x0159BBC07253907BULL, "6139765_m.fxe"},
	{0x0159C07577EB1727ULL, "zz_mrl_debug6184211.cl"},
	{0x0159C259FA524382ULL, "qun120cr_javaris6044137.cl"},
	{0x0159D766D356679DULL, "vxlt_warehouse_14w_0_0.vlm"},
	{0x015AC163971CFEDFULL, "vxht_mansion_n_4dm_0_0.vlm"},
	{0x015AC46B618D763EULL, "dae000_fem_priest_mob.cnv"},
	{0x015AD8866A0CB8BEULL, "dae000_meredith6092581.cl"},
	{0x015B2F802EF67489ULL, "vxdr_dungeon_1lv_0_0.vlm"},
	{0x015B3CCE77F21AD2ULL, "vxlt_0_lowtown_a3_n_c_epb_0_0.vlm"},
	{0x015B4000AED2E468ULL, "6060508_f.fxe"},
	{0x015B80FF928D3651ULL, "6077843_m.fxe"},
	{0x015B9B790D8EF9FBULL, "6015998_f.fxe"},
	{0x015BDDBDE8EB0F3FULL, "6087532_m.fxe"},
	{0x015C16CF19E60A0EULL, "6166557_m.fxe"},
	{0x015C67A87BD06C26ULL, "art\\characters\\creatures\\spider\\animation\\kd_exit.evt"},
	{0x015C8246C27F39D9ULL, "6112154_f.fxe"},
	{0x015C8362BAC9B730ULL, "6089825_m.fxe"},
	{0x015C905A268648A3ULL, "655447868.ogg"},
	{0x015CC1E31A41B08FULL, "art\\cinematics\\_crit path\\mage crisis\\gallows arrival\\gallowsarrive_follower2_varric_.ani"},
	{0x015CC58A36B8EE0FULL, "303257207.ogg"},
	{0x015D2A64172ADB61ULL, "728034344.ogg"},
	{0x015E49AC778A6946ULL, "lgt000cr_kirk_turn_in_6.utc"},
	{0x015E5725A2A51C93ULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\a_bnt_dblw_lp.ani"},
	{0x015E6955D20BF1F3ULL, "vxw_woundedcoast_1i_0_0.vlm"},
	{0x015E913F0398E003ULL, "gen00fl_fenris6126554.cl"},
	{0x015EA66B61912C4EULL, "dae000_fem_priest_mob.bnk"},
	{0x015F1AC8C7FB4F41ULL, "vxht_bartrand_n_642_0_0.vlm"},
	{0x015F567CEE7DDF99ULL, "ht_0_hightown_a2_16_0.anb"},
	{0x015F6D1FFF80DCD5ULL, "vxlt_hangedman_n_258_0_0.vlm"},
	{0x0160673FAAA43040ULL, "vxlt_backalley_n_1eo_0_0.vlm"},
	{0x016083DF8E752897ULL, "156249868.ogg"},
	{0x0160AF47C7CCF9C6ULL, "art\\levels\\races\\dwarf\\props\\decor\\prp_potc_0.pwk"},
	{0x0160C206FD3CC90DULL, "vxlt_0_lowtown_a2_n_c_eoe_0_0.vlm"},
	{0x0160D35F5ACBCE78ULL, "vxlt_backalley_g_1xe_0_0.vlm"},
	{0x016119573C843CC2ULL, "vxht_brothel_n_5e9_0_0.vlm"},
	{0x01611DFD3C29933BULL, "292046921.ogg"},
	{0x0162350A86D3E685ULL, "231316768.ogg"},
	{0x01627EA87BD4FF71ULL, "art\\characters\\creatures\\spider\\animation\\kd_exit.gad"},
	{0x0162A98FF8EDD571ULL, "vxdr_primevaltomb_535_0_0.vlm"},
	{0x0162BA1C952C2741ULL, "vfx.mat.dx11"},
	{0x0162E700DEE7830DULL, "6075770_f.fxe"},
	{0x016302854AB6974CULL, "art\\vfx\\combat\\vfx_m_sta_spi_range_f_p.mmh"},
	{0x0163654A0994EE9BULL, "vxca_caverns1_ckh_0_0.vlm"},
	{0x01637CF5EFAA18ADULL, "llac_lt_0_lowtown_a2_65533_2.dds"},
	{0x0163BC8EB0254CF1ULL, "6132888_f.fxe"},
	{0x0163BEAEB7598F34ULL, "6103451_m.fxe"},
	{0x0163EBC0AF8EAF3EULL, "vxlt_blackemporium_vs_0_0.vlm"},
	{0x0164093038FC4A80ULL, "or_db_sh.mao"},
	{0x01641BFF90FB9B9AULL, "art\\levels\\races\\ferelden\\props\\armory\\prp_staffc.msh"},
	{0x016429FF90FBB3D3ULL, "art\\levels\\races\\ferelden\\props\\armory\\prp_staffc.mao"},
	{0x01642DFF90FBBAE0ULL, "art\\levels\\races\\ferelden\\props\\armory\\prp_staffc.mmh"},
	{0x0164748835D1EBCFULL, "dae000_flemeth6013252.cl"},
	{0x0164946BBE846A10ULL, "802140597.ogg"},
	{0x0164BA7A42F0E866ULL, "dae000_pc_home_ambients6208169.cl"},
	{0x0164F78A22ECDA7CULL, "hlac_lt_backalley_n_65537_1z.dds"},
	{0x0165096F67A1D36DULL, "vxdr_0_deeproads_cui_0_2.vlm"},
	{0x016526CBE831CB66ULL, "ave121_brennan6105896.cl"},
	{0x01653C2D20505A14ULL, "t3_tra.tnt"},
	{0x01653C6C0ED4E917ULL, "root_tree.mao"},
	{0x01670065185A3B7CULL, "45067022.ogg"},
	{0x016770C27E8B2BF5ULL, "hlac_w_bonepit_327680_0y.dds"},
	{0x0167B3E7116CDA51ULL, "dae000_arishok6067932.cl"},
	{0x01681FE80856A25AULL, "dae000ip_crowd_lm3_talk5.utp"},
	{0x0168BC5FA3D6E214ULL, "6090037_f.fxe"},
	{0x0168C815AB70C561ULL, "vxlt_blackemporium_15x_0_0.vlm"},
	{0x01690317E01116A9ULL, "vxlt_gallowstemplar_5xf_0_0.vlm"},
	{0x01697FA87BDB1F1FULL, "art\\characters\\creatures\\spider\\animation\\kd_exit.ani"},
	{0x01698A186187E7FAULL, "vxlt_house_dr_0_0.vlm"},
	{0x0169F0A12783856AULL, "vxlt_0_lowtown_a2_n_ep4_0_1.vlm"},
	{0x016A767E35B1F4C1ULL, "6165964_m.fxe"},
	{0x016AE2C9204E10FCULL, "6185501_m.fxe"},
	{0x016B01DB296C5B79ULL, "386194802.ogg"},
	{0x016B1503D25218B0ULL, "vxca_caverns2_7k0_0_0.vlm"},
	{0x016B261A3C025D52ULL, "al_mage_t04.gda"},
	{0x016BB729C376BC8FULL, "llac_lt_0_lowtown_docks_a3_n_65532_0.dds"},
	{0x016BFDF7A830BC3DULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\3ww_pillar.mao"},
	{0x016C01F7A830C376ULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\3ww_pillar.mmh"},
	{0x016C0BF7A830D474ULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\3ww_pillar.msh"},
	{0x016C63E2FBDF7288ULL, "gen00fl_varric6075642.cl"},
	{0x016CC10D8B063C8AULL, "vxlt_gallowstemplar_47y_0_0.vlm"},
	{0x016CE6D0BF6A5452ULL, "llac_lt_gallowstemplar_n_2382_2.dds"},
	{0x016CEE1F505D15A5ULL, "hightown_male_noble2_mob6170931.cl"},
	{0x016D40007ACA0E2DULL, "6071715_m.fxe"},
	{0x016E9ECE2EC33ED0ULL, "hlac_lt_0_lowtown_docks_a3_-131074_5y.dds"},
	{0x016EB1C67BA66617ULL, "hlac_lt_0_lowtown_-65537_0y.dds"},
	{0x016EEECECD055145ULL, "398046902.ogg"},
	{0x016F50089BF7F9ABULL, "6124723_m.fxe"},
	{0x016F53337D647BE3ULL, "el_lr_middle_dusk.mmh"},
	{0x016FB68A0D24FF2FULL, "vxw_encounter2_camp_ny_0_0.vlm"},
	{0x01701625ACA080B8ULL, "mag100_cullen6082314.cl"},
	{0x0170A76A544CB240ULL, "442011668.ogg"},
	{0x0170C059C9910CE8ULL, "vxw_sundermount_1zn_0_0.vlm"},
	{0x0170DA050B73880AULL, "525232403.ogg"},
	{0x01710866529B8C73ULL, "art\\vfx\\combat\\vfx_c_rw_vulnerable_c.dura.ani"},
	{0x017123C29D63D195ULL, "hlac_ht_brothel_n_3373_2z.dds"},
	{0x0171C99841C58323ULL, "hlac_lt_0_lowtown_-196609_4z.dds"},
	{0x0171D726FB8B071BULL, "vxca_caverns1_cnk_0_0.vlm"},
	{0x017255CCC9D97F0FULL, "z_dust8_white.dds"},
	{0x01727B35EEEC28AAULL, "vxht_bartrand_n_1pn_0_0.vlm"},
	{0x01727CDE65EB9EC0ULL, "423126312.ogg"},
	{0x01728B95A81D1730ULL, "w_blightlands_60_0.mao"},
	{0x01728F95A81D1E43ULL, "w_blightlands_60_0.mmh"},
	{0x01729D95A81D359DULL, "w_blightlands_60_0.msh"},
	{0x01732E25A2B7333DULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\a_bnt_dblw_lp.gad"},
	{0x017335CB52ABB632ULL, "438398380.ogg"},
	{0x01735808884E5626ULL, "545792343.ogg"},
	{0x0173987F1EA20DCBULL, "vxlt_0_lowtown_a2_ghu_0_0.vlm"},
	{0x0173BF968DDE5021ULL, "vxw_bonepit_9n_0_0.vlm"},
	{0x01742EAF0D34630FULL, "ss_rogue1.cut"},
	{0x017532D860541CD9ULL, "6064589_f.fxe"},
	{0x01756E0A6B72CB2DULL, "dae000_elegant6166890.cl"},
	{0x0175997AB8787D5BULL, "art\\characters\\weapons\\longswords\\textures\\w_lsw_021a_0s.dds"},
	{0x0175DEE1BA4DEBF1ULL, "art\\characters\\playercharacter\\humanmale\\textures\\hm_hlf_masb_0s.dds"},
	{0x0176539F5CC6F271ULL, "vxw_bonepit_p1_0_0.vlm"},
	{0x01766FD5FDD5856DULL, "6097405_m.fxe"},
	{0x0176ABDD0A2C3055ULL, "985918524.ogg"},
	{0x0176E4335BA19667ULL, "vxlt_0_lowtown_a2_n_c_e3c_0_0.vlm"},
	{0x017733D74266EED0ULL, "35235318.ogg"},
	{0x01779AB25B86DC19ULL, "zz_dre_debug6025538.cl"},
	{0x017933259222BDD3ULL, "vxlt_0_lowtown_docks_n_b61_0_1.vlm"},
	{0x01798F29B4D7143AULL, "215721774.ogg"},
	{0x0179D043F0359CE7ULL, "vxca_caverns2_7o7_0_0.vlm"},
	{0x017A0646B11CC6CFULL, "284604047.ogg"},
	{0x017A12D2FD58FCB9ULL, "gen_im_trs_apo_lgt_boo_r2.uti"},
	{0x017A5A813B384639ULL, "719895107.ogg"},
	{0x017A93E5B44E5600ULL, "vxlt_0_lowtown_euw_0_0.vlm"},
	{0x017B18C9027FC4C1ULL, "9312095.ogg"},
	{0x017B589D4D918382ULL, "325686806.ogg"},
	{0x017BA3E19313D603ULL, "6208243_m.fxe"},
	{0x017BD9E2089CE89EULL, "mcr371_idunna6087279.cl"},
	{0x017C0A4F0C248002ULL, "vxlt_0_lowtown_a3_n_c_ghh_0_0.vlm"},
	{0x017C13CC7F9D9E5BULL, "llac_dr_slavers_3214_9.dds"},
	{0x017C288650533DEBULL, "1039700506.ogg"},
	{0x017C41F769512F97ULL, "art\\levels\\races\\ferelden\\props\\plotspecifc\\prp_whoreflower.mmh"},
	{0x017C45F6A607EE48ULL, "855403700.ogg"},
	{0x017C45F7695136A4ULL, "art\\levels\\races\\ferelden\\props\\plotspecifc\\prp_whoreflower.mao"},
	{0x017C57F769515509ULL, "art\\levels\\races\\ferelden\\props\\plotspecifc\\prp_whoreflower.msh"},
	{0x017D11A2CA5C4EF6ULL, "art\\characters\\creatures\\stonegolem\\animation\\blast.evt"},
	{0x017D4AF98E906140ULL, "vxlt_backalley_n_1km_0_0.vlm"},
	{0x017D82F550C6D0C2ULL, "seb201_lady_harimann6177402.cl"},
	{0x017E006652A608AAULL, "art\\vfx\\combat\\vfx_c_rw_vulnerable_c.dura.evt"},
	{0x017E2866FF1B17AFULL, "vxw_encounter2_camp_ix_0_0.vlm"},
	{0x017E5AAF9ABE6C78ULL, "art\\vfx\\combat\\vfx_g_spear_mod_c.anb"},
	{0x017E8E4AF41FF507ULL, "art\\vfx\\combat\\vfx_m_arc_shield_mod_c.impa.gad"},
	{0x017EAEA98C28A492ULL, "plt_cod_crt_pride_demon.plo"},
	{0x017F14914D7B4EFDULL, "440700053.ogg"},
	{0x017F2F0EA206A0EFULL, "lt_0_lowtown_a3_n_c_36_0.anb"},
	{0x017F480178D83C6DULL, "atl_startmenu2_dxt5_dat.xml"},
	{0x017FC8EF6DD0408DULL, "lt_0_lowtown_docks_a3_102_0.phy"},
	{0x017FEFED8AFD005BULL, "6088151_m.fxe"},
	{0x01808259E885290EULL, "6087164_m.fxe"},
	{0x0180F3200C37CA5BULL, "lt_0_lowtown_docks_n_178_0.msh"},
	{0x018105200C37E8A6ULL, "lt_0_lowtown_docks_n_178_0.mao"},
	{0x018111200C37FD45ULL, "lt_0_lowtown_docks_n_178_0.mmh"},
	{0x01816FE7BF146648ULL, "plt_mer110pt_back_to_work.nss"},
	{0x01818B099D1ED7DCULL, "620875888.ogg"},
	{0x01826EA6D697D443ULL, "1054891784.ogg"},
	{0x018308A2CA60AA81ULL, "art\\characters\\creatures\\stonegolem\\animation\\blast.gad"},
	{0x0183855EF7599D4BULL, "middle_dusk_rig.msh"},
	{0x0183B4BAF01C2C34ULL, "880501347.ogg"},
	{0x0183EA1596E4F77FULL, "art\\levels\\races\\design\\prp_largetreasur.msh"},
	{0x0183F01596E50121ULL, "art\\levels\\races\\design\\prp_largetreasur.mmh"},
	{0x0183FC1596E515C2ULL, "art\\levels\\races\\design\\prp_largetreasur.mao"},
	{0x018429A73E2B4E18ULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_potplant_dp.dds"},
	{0x018470AEED6D4E50ULL, "6196685_f.fxe"},
	{0x01848FAE33C343BBULL, "831679755.ogg"},
	{0x0184DF6652ABEF1DULL, "art\\vfx\\combat\\vfx_c_rw_vulnerable_c.dura.gad"},
	{0x0184E4D4DA477040ULL, "765546665.ogg"},
	{0x01851B2FE0FDE728ULL, "fxa_unholy_bt.dds"},
	{0x0185665F6A476BFAULL, "w_bonepitdestroyed_a3_231_0.mmh"},
	{0x01856A5F6A477231ULL, "w_bonepitdestroyed_a3_231_0.mao"},
	{0x0185785F6A478A40ULL, "w_bonepitdestroyed_a3_231_0.msh"},
	{0x0185834AF4260114ULL, "art\\vfx\\combat\\vfx_m_arc_shield_mod_c.impa.evt"},
	{0x0185A16D86075AC8ULL, "hlac_ht_mansion2_818_31z.dds"},
	{0x0185C453DC166D4FULL, "6079050_m.fxe"},
	{0x0185DB24FE08A686ULL, "mcr371_orsino6083353.cl"},
	{0x0185F8E2C903D31AULL, "mrl150tr_mtn_undead_2.utt"},
	{0x018613C558306793ULL, "hlac_lt_hangedman_n_-1_3x.dds"},
	{0x018660D21644436DULL, "gen00fl_merrill6106363.cl"},
	{0x01869084ACF9E8AFULL, "6119679_m.fxe"},
	{0x01872B6B22C2BB5FULL, "547978045.ogg"},
	{0x018731FF8F1AF19CULL, "1004859216.ogg"},
	{0x018785175B328BD5ULL, "490171856.ogg"},
	{0x0187BD53DB7F2ACEULL, "6071037_m.fxe"},
	{0x018859EE0232BCA6ULL, "t3_0000052.tnt"},
	{0x018878E7BF1A945BULL, "plt_mer110pt_back_to_work.plo"},
	{0x0188A892F33A0ED5ULL, "mer240_jansen6050334.cl"},
	{0x0188F2F77EE99675ULL, "and110_refugee6111601.cl"},
	{0x01894804B2E81C4AULL, "vxlt_gallowsprison_n_5qu_0_0.vlm"},
	{0x0189586A448A27E2ULL, "dae000ip_crowd_ff1_talk11.utp"},
	{0x018A09A2CA66CB2FULL, "art\\characters\\creatures\\stonegolem\\animation\\blast.ani"},
	{0x018A80629B3B5802ULL, "zz_dog_debug6127920.cl"},
	{0x018A8F95A831B060ULL, "w_blightlands_60_0.rml"},
	{0x018AF8FD30DCD4B5ULL, "6048824_f.fxe"},
	{0x018B13A756CE6AFFULL, "art\\vfx\\combat\\vfx_r_gn3_paraly_brittle_c.cess.gad"},
	{0x018B4481BA91F138ULL, "art\\characters\\playercharacter\\humanfemale\\textures\\hm_arm_cas_0s.dds"},
	{0x018BE5292F5DEE01ULL, "hlac_w_encounter2_camp_65534_0z.dds"},
	{0x018C3FDF2351AB6CULL, "771313688.ogg"},
	{0x018CB59396951FD3ULL, "art\\characters\\weapons\\longswords\\textures\\w_lsw_011a_0n.dds"},
	{0x018D06BF9F3E6A54ULL, "54311468.ogg"},
	{0x018DE10B6CAD12E8ULL, "lgt311tr_shw_mer_mettin.utt"},
	{0x018DF14772550F90ULL, "hlac_lt_0_lowtown_a2_n_c_-131073_1y.dds"},
	{0x018DF259353D6875ULL, "zz_facefx_test.cnv"},
	{0x018E0EBB17BB1F48ULL, "llac_lt_0_lowtown_a2_n_c_0_0.dds"},
	{0x018E1CF2FE4D7D5EULL, "cs_mcr370cs_epilogue.bnk"},
	{0x018E758F3F000025ULL, "vxw_encounter3_60_0_0.vlm"},
	{0x018E80219F50A151ULL, "art\\characters\\heraldry\\hld_circle_of_mages_1.dds"},
	{0x018ED943AF635BB5ULL, "zz_rogueslate_white2.cut"},
	{0x018F464FECE88CE2ULL, "ht_keep_9215.rml"},
	{0x018F87043EE92DB7ULL, "348179205.ogg"},
	{0x018FF10F9505ED5FULL, "llac_w_encounter2_camp_65538_0.dds"},
	{0x019077CA90682E11ULL, "follower_banter6136471.cl"},
	{0x01907C39028E3ECBULL, "vxlt_0_lowtown_a2_gbw_0_0.vlm"},
	{0x0190E4D99A8CA4BFULL, "6101490_m.fxe"},
	{0x019108A756D2C3ECULL, "art\\vfx\\combat\\vfx_r_gn3_paraly_brittle_c.cess.evt"},
	{0x01914CA1814DEF13ULL, "mag370_meredith6204104.cl"},
	{0x01916D26B016DA86ULL, "art\\characters\\playercharacter\\dwarfmale\\animation\\vrc_cmov_wf.ani"},
	{0x0191731A65F8D95FULL, "zz_idle_test.cut"},
	{0x019199A98E6CA7BAULL, "gen00fl_fenris6126342.cl"},
	{0x019213AC1BE2032CULL, "llac_lt_gallowscourtyard_-131072_1.dds"},
	{0x019253A1E3D9FA9DULL, "one140cr_veld.utc"},
	{0x01926F5A946FF6D8ULL, "vxca_caverns2_907_0_0.vlm"},
	{0x0192955A3EE20117ULL, "follower_banter6165608.cl"},
	{0x0192D19496F024F1ULL, "6173398_m.fxe"},
	{0x0192FC7023F088E7ULL, "273076312.ogg"},
	{0x0193049835CBF643ULL, "follower_banter6161240.cl"},
	{0x01934AF62108D663ULL, "6053592_f.fxe"},
	{0x01937B4AF432314DULL, "art\\vfx\\combat\\vfx_m_arc_shield_mod_c.impa.ani"},
	{0x0193F4D314D66E77ULL, "mcr370_cut_player_survives6171832.cl"},
	{0x0193F89D18E50A3BULL, "art\\characters\\playercharacter\\dwarfmale\\boo_hvyo.phy"},
	{0x01942D3980108A35ULL, "vxht_0_hightown_a3_c91_0_0.vlm"},
	{0x019445B0A4D4154CULL, "144333932.ogg"},
	{0x019470CC8178BC3DULL, "6037106_m.fxe"},
	{0x019481BC65FC7745ULL, "vxlt_house_dy_0_0.vlm"},
	{0x0194D1752BCF5DA6ULL, "vxht_mansion2_3ek_0_0.vlm"},
	{0x0194DE59354364D0ULL, "zz_facefx_test.are"},
	{0x0195B2D41D9F080AULL, "plt_rdr300pt_boss.plo"},
	{0x01963AB1116A60DAULL, "140731671.ogg"},
	{0x019699A42BFFB8CFULL, "art\\levels\\races\\ferelden\\props\\plotspecifc\\prp_candle.phy"},
	{0x0196C3CA7D392D66ULL, "mag170_thrask6016958.cl"},
	{0x0196DFF829C75918ULL, "vxht_bartrand_n_755_0_0.vlm"},
	{0x0196FC5FAD39A3F5ULL, "vfxchunkydd.vsh.fxo"},
	{0x019703A6B55E6B9EULL, "vxht_keep_n_3ba_0_0.vlm"},
	{0x01979F4477FD832CULL, "vxlt_gallowscourtyard_n_3bb_0_0.vlm"},
	{0x0197A4D13258201FULL, "vxht_mansion2_3g0_0_0.vlm"},
	{0x019804111482207AULL, "6102555_m.fxe"},
	{0x01982EABA4695FD3ULL, "102664365.ogg"},
	{0x0198A2D2D7F30393ULL, "6064719_m.fxe"},
	{0x019929CDB213B456ULL, "6125407_f.fxe"},
	{0x0199577CEEAF5999ULL, "ht_0_hightown_a2_16_0.rml"},
	{0x019AA1859AF0647EULL, "132105732.ogg"},
	{0x019AD648FA0486E9ULL, "vxlt_0_lowtown_n_e28_0_0.vlm"},
	{0x019B8DBD88EF1AE5ULL, "rdr111dg_martin6014998.cl"},
	{0x019CC0C96AB4BA80ULL, "zz_qun_debug6045400.cl"},
	{0x019DC0741822B625ULL, "vxca_caverns1_i8y_0_0.vlm"},
	{0x019DC8A0FCC65169ULL, "bur_ho_clu_glo.mao"},
	{0x019E61DE5A262129ULL, "art\\characters\\playercharacter\\elffemale\\mt_ant_ef04.phy"},
	{0x019E922B6209AAE7ULL, "355926805.ogg"},
	{0x019EB180B0CCD63FULL, "hlac_lt_0_lowtown_a2_-2_6y.dds"},
	{0x019EED59354BD2C3ULL, "zz_facefx_test.lst"},
	{0x019EF0A756DED845ULL, "art\\vfx\\combat\\vfx_r_gn3_paraly_brittle_c.cess.ani"},
	{0x019F51BDA8630C57ULL, "one220_yevhen6102813.cl"},
	{0x019FB2B41AE1C4C3ULL, "6174876_m.fxe"},
	{0x01A037FA82DB54F5ULL, "vxlt_0_lowtown_a3_n_c_ho9_0_0.vlm"},
	{0x01A0487ADB18238CULL, "17075002.ogg"},
	{0x01A09CB39B1BA582ULL, "6060641_f.fxe"},
	{0x01A0BD1A250E2917ULL, "hlac_w_bonepitmining_a2_3_0z.dds"},
	{0x01A0E13786660F58ULL, "vxca_caverns2_954_0_0.vlm"},
	{0x01A10B0A0157E607ULL, "evl_dot.mao"},
	{0x01A1397A47C932B3ULL, "hm_mag_controlled_3.mor"},
	{0x01A1F632D57A8860ULL, "934948865.ogg"},
	{0x01A2B923587C865DULL, "303419649.ogg"},
	{0x01A2DC7FA8DC4E7AULL, "6157194_f.fxe"},
	{0x01A2E2A9736ED4BDULL, "6030006_m.fxe"},
	{0x01A310BC28D7F3AFULL, "6053897_f.fxe"},
	{0x01A3DA2B86D60AD4ULL, "art\\vfx\\combat\\vfx_r_var_activation2_c.mmh"},
	{0x01A3E4243450B303ULL, "6134726_m.fxe"},
	{0x01A468EED5468B54ULL, "vxw_woundedcoast_y2_0_0.vlm"},
	{0x01A4825D52C18091ULL, "6143713_m.fxe"},
	{0x01A511BC6091CA9EULL, "one300_nuncio6073802.cl"},
	{0x01A517C8B55BA252ULL, "hlac_w_sundermount_393215_0z.dds"},
	{0x01A536B9E03AB19FULL, "6080189.cut"},
	{0x01A55626B0275C80ULL, "art\\characters\\playercharacter\\dwarfmale\\animation\\vrc_cmov_wf.gad"},
	{0x01A55FF21B0D94BAULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\gt2a_lowtowngate_8.pwk"},
	{0x01A5841188D61BB0ULL, "vxht_chantry_n_2zf_0_0.vlm"},
	{0x01A58C44448942E3ULL, "6048343_f.fxe"},
	{0x01A5DA8D16E56C25ULL, "ser270_templar6164087.cl"},
	{0x01A656DE86A7B2C1ULL, "6175712_m.fxe"},
	{0x01A659AF9ADFBC13ULL, "art\\vfx\\combat\\vfx_g_spear_mod_c.mmh"},
	{0x01A663EB16740FF4ULL, "hlac_lt_0_lowtown_a2_-65535_0z.dds"},
	{0x01A7042BA5CB5D5EULL, "221075995.ogg"},
	{0x01A75FE5E206C982ULL, "vxlt_0_lowtown_a3_n_c_gyk_0_0.vlm"},
	{0x01A78AFC5AD85BD5ULL, "rdr111dg_harbormaster6016142.cl"},
	{0x01A79BAAA409CCD6ULL, "6122507_f.fxe"},
	{0x01A7D263A8F67B39ULL, "gen00fl_fenris6099801.cl"},
	{0x01A8A14D793BEB0FULL, "6052923_m.fxe"},
	{0x01A90D8B8B23C6D5ULL, "vxca_caverns2prc_a7f_0_0.vlm"},
	{0x01A925F36F12BD7DULL, "vxw_woundedcoast_19j_0_0.vlm"},
	{0x01A929AB56EB361FULL, "6143122_m.fxe"},
	{0x01A985D9B6FDE0A9ULL, "hlac_lt_0_lowtown_a3_n_c_1_10x.dds"},
	{0x01A9F1F04FAF6943ULL, "vxlt_0_lowtown_n_ep4_0_0.vlm"},
	{0x01A9FCF93755D4A8ULL, "vxlt_0_lowtown_a3_n_c_hta_0_0.vlm"},
	{0x01AA9AE04A5B1341ULL, "gen00fl_aveline6137717.cl"},
	{0x01AAA2317B744198ULL, "369490588.ogg"},
	{0x01AAAE7FD1D3FE69ULL, "mag131_taraneh6094536.cl"},
	{0x01AAF9E5A5FE42C2ULL, "6079288_m.fxe"},
	{0x01AB0C0EA22B9F93ULL, "lt_0_lowtown_a3_n_c_36_0.rml"},
	{0x01AB581D068BA7F3ULL, "vxw_bonepitmining_a2_1ow_0_0.vlm"},
	{0x01AB6DD17CFEA83FULL, "mag360_alain6044656.cl"},
	{0x01AB762B7868B338ULL, "6126397_m.fxe"},
	{0x01AB88CB984EF531ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\0di_right_1.mao"},
	{0x01AB9EEA1B8431FEULL, "46457837.ogg"},
	{0x01AC8D17D4103989ULL, "6032853_m.fxe"},
	{0x01AC9A284789126BULL, "ht_0_hightown_a3_35_0_ll.mmh"},
	{0x01AC9C7A71BA9D00ULL, "plt_cod_hst_qunari.plo"},
	{0x01ACAE762DCECE1BULL, "vxw_encounter2_nm_0_0.vlm"},
	{0x01AD92A20C2F69E5ULL, "940437419.ogg"},
	{0x01AD9EB44185B843ULL, "art\\characters\\creatures\\ragedemon\\animation\\shuffle.ani"},
	{0x01ADB039AF136844ULL, "vxca_caverns1_mines_cna_0_0.vlm"},
	{0x01ADFE76ED90F803ULL, "ser270_templar6165680.cl"},
	{0x01AE1BE0F94AC596ULL, "261595125.ogg"},
	{0x01AE3E989995F9CAULL, "vxlt_hangedman_1ht_0_0.vlm"},
	{0x01AE4A8CD7B9D972ULL, "361233576.ogg"},
	{0x01AE96A50A611DACULL, "325402529.ogg"},
	{0x01AEE6DBE00245CEULL, "art\\vfx\\combat\\vfx_ha_bd_bdrain_proj_c.dura.ani"},
	{0x01AF41C6939919E8ULL, "lgt110_orz_harrowmont6172607.cl"},
	{0x01AFDF096FA80778ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\0ff_crumble.phy"},
	{0x01B01B13249F5471ULL, "vxw_bonepit_l5_0_0.vlm"},
	{0x01B03098FDB02C44ULL, "follower_banter6153976.cl"},
	{0x01B0987CE43D64EBULL, "vxlt_0_lowtown_a3_n_c_hqi_0_0.vlm"},
	{0x01B0A71419D2608FULL, "zz_followers_banter6157856.cl"},
	{0x01B0A9962331E38EULL, "mcr371_hostile_anders6082118.cl"},
	{0x01B0CA0CDF4526C4ULL, "hlac_lt_0_lowtown_docks_a3_n_-3_8z.dds"},
	{0x01B0F115970BF09EULL, "art\\levels\\races\\design\\prp_largetreasur.phy"},
	{0x01B12D2517B76DF2ULL, "896880102.ogg"},
	{0x01B1C2CFEC005159ULL, "vxlt_shop_t1_0_0.vlm"},
	{0x01B1E13170A2CC7EULL, "docks_scene_dockworkers6176219.cl"},
	{0x01B1F4023E623143ULL, "6149649_m.fxe"},
	{0x01B201A679B4DE43ULL, "vxca_caverns2_981_0_1.vlm"},
	{0x01B20A50D3B75653ULL, "ser121_jethann6106564.cl"},
	{0x01B20DE9CFCDE06CULL, "hlac_ht_mansion2_818_68x.dds"},
	{0x01B2A98542665558ULL, "vxlt_0_lowtown_docks_n_ba7_0_1.vlm"},
	{0x01B2B9408AFF51D3ULL, "art\\levels\\races\\natural\\kirkcoastal\\foliage\\tre_cashstumpsma.phy"},
	{0x01B2DACBC73CA3D4ULL, "vxw_woundedcoast_v3_0_0.vlm"},
	{0x01B2E69D15E3FD4BULL, "dae000ip_crowd_ff8_talk8.utp"},
	{0x01B3E3B77FDC0B13ULL, "hlac_ht_mansion2_818_9x.dds"},
	{0x01B3E649E3630DBAULL, "art\\characters\\playercharacter\\humanmale\\textures\\hn_bdy_lgtb_0n.dds"},
	{0x01B43E8D247D09CBULL, "636924627.ogg"},
	{0x01B48B39A0D27F21ULL, "vxlt_0_lowtown_glo_0_0.vlm"},
	{0x01B4BEE78B37161BULL, "6087200_m.fxe"},
	{0x01B507B73613CB4EULL, "vxdr_primevaltomb_7kh_0_0.vlm"},
	{0x01B607C6AFC85230ULL, "6082214_f.fxe"},
	{0x01B6433111B8C2A9ULL, "6208479_f.fxe"},
	{0x01B6580FE9A98BC6ULL, "hlac_lt_hangedman_-1_28y.dds"},
	{0x01B66B12DCD6F70EULL, "vxht_0_hightown_a2_c8p_0_0.vlm"},
	{0x01B76B454BAE2A69ULL, "hlac_ht_0_hightown_65536_0y.dds"},
	{0x01B7903418681E44ULL, "vxca_caverns1_mines_gk3_0_1.vlm"},
	{0x01B79CB3B1CD4C56ULL, "hlac_lt_0_lowtown_a2_n_c_131071_7x.dds"},
	{0x01B80969F42EA9CCULL, "192986629.ogg"},
	{0x01B811C9BFC3F455ULL, "blood_imp_01.mao"},
	{0x01B82C2D891BE625ULL, "vxlt_0_lowtown_a2_ep1_0_0.vlm"},
	{0x01B83FA5CE58C36DULL, "201017059.ogg"},
	{0x01B84FD60C2A2C27ULL, "isa000ip_bottle_container.utp"},
	{0x01B8686C82ECD32AULL, "842637075.ogg"},
	{0x01B89FAD2B7A76F1ULL, "vxca_caverns2prc_9qx_0_0.vlm"},
	{0x01B967A96654C8A5ULL, "22087330.ogg"},
	{0x01B97FCAA2CC6769ULL, "hlac_ht_0_hightown_a3_131071_2z.dds"},
	{0x01BADAB18F75317CULL, "159883868.ogg"},
	{0x01BAE15A534D56A2ULL, "vxht_0_hightown_n_8pd_0_0.vlm"},
	{0x01BB096C228531DEULL, "vxlt_gallowscourtyard_n_79y_0_0.vlm"},
	{0x01BB20531FC4E69BULL, "vxw_encounter2_il_0_0.vlm"},
	{0x01BB339AA2F2E459ULL, "vxht_mansion2_24z_0_0.vlm"},
	{0x01BB5154530A4EB8ULL, "6172583_m.fxe"},
	{0x01BB600E52FB4BE0ULL, "6133346_m.fxe"},
	{0x01BC48E5157C6D6BULL, "art\\characters\\playercharacter\\dwarfmale\\mt_bas_dm_eyesm1.mao"},
	{0x01BCDEDBE00E751FULL, "art\\vfx\\combat\\vfx_ha_bd_bdrain_proj_c.dura.evt"},
	{0x01BCE8593564C7A4ULL, "zz_facefx_test.utc"},
	{0x01BCFBD4CDAE2F7FULL, "one100dg_kelder6060540.cl"},
	{0x01BCFE196FBA2BAFULL, "zz_ave_debug6064626.cl"},
	{0x01BE7A2A12516178ULL, "ht_0_hightown_n_52_0.rml"},
	{0x01BF0ED5167E5281ULL, "hlac_ca_caverns2prc_5802_23x.dds"},
	{0x01BFD4A99432E3E1ULL, "vxca_caverns2_62n_0_1.vlm"},
	{0x01BFDF9D190A1ADAULL, "art\\characters\\playercharacter\\dwarfmale\\boo_hvyo.msh"},
	{0x01BFE09060C8E7C0ULL, "ave310_donnic6134769.cl"},
	{0x01BFED9D190A3213ULL, "art\\characters\\playercharacter\\dwarfmale\\boo_hvyo.mao"},
	{0x01BFF19D190A3920ULL, "art\\characters\\playercharacter\\dwarfmale\\boo_hvyo.mmh"},
	{0x01C016B39FBF2FD0ULL, "vxw_sundermount_2kr_0_0.vlm"},
	{0x01C02CB2484CF8CFULL, "430105954.ogg"},
	{0x01C0A0D34B6ED2EDULL, "91346626.ogg"},
	{0x01C0E74DFC94DEC2ULL, "11061155.ogg"},
	{0x01C1425D1780AA70ULL, "vxlt_0_lowtown_docks_cvx_0_1.vlm"},
	{0x01C14CA8B932BC54ULL, "vxw_woundedcoast_s0_0_0.vlm"},
	{0x01C16D8ABD44A686ULL, "gen00fl_fenris6113885.cl"},
	{0x01C1B5B44196882DULL, "art\\characters\\creatures\\ragedemon\\animation\\shuffle.gad"},
	{0x01C2FD1692C897A1ULL, "vxlt_0_lowtown_gkn_0_0.vlm"},
	{0x01C37A1B295B7C61ULL, "w_encounter2_39_0.phy"},
	{0x01C3BFDBE0145F08ULL, "art\\vfx\\combat\\vfx_ha_bd_bdrain_proj_c.dura.gad"},
	{0x01C3C3E73E4E78AFULL, "6043724_f.fxe"},
	{0x01C3EC59356AED26ULL, "zz_facefx_test.stg"},
	{0x01C4E1DB41202740ULL, "549434560.ogg"},
	{0x01C555E2A9D326D1ULL, "follower_banter6170572.cl"},
	{0x01C558C8EF5DFAECULL, "6206813_m.fxe"},
	{0x01C64A56F9D54FFBULL, "art\\characters\\playercharacter\\humanfemale\\animation\\exploration\\movement\\runf_injured.evt"},
	{0x01C659AA7E582F5FULL, "hlac_w_sundermount_327677_0z.dds"},
	{0x01C67E7110C30F98ULL, "572360438.ogg"},
	{0x01C6939DD04745C6ULL, "mrl000_marethari6142116.cl"},
	{0x01C6BA492B497A47ULL, "vxw_bonepitmining_a2_8u_0_0.vlm"},
	{0x01C6EF06FA04B371ULL, "llac_ht_chantry_n_2534_49.dds"},
	{0x01C7992C8FC3BECFULL, "art\\vfx\\combat\\vfx_m_pri_petrify_conj_c.mmh"},
	{0x01C7D2C46646C9B2ULL, "6053693_m.fxe"},
	{0x01C7DBC8CBE834DDULL, "llac_lt_0_lowtown_a2_n_c_-65535_2.dds"},
	{0x01C7F0B1B5E98F69ULL, "gen00fl_merrill6115792.cl"},
	{0x01C819CE1E7B9F75ULL, "628918599.ogg"},
	{0x01C85EA0B7A3025FULL, "vxlt_gallowscourtyard_748_0_0.vlm"},
	{0x01C8A36AE9182C95ULL, "w_blightlands_217_0.msh"},
	{0x01C8A56AE918300BULL, "w_blightlands_217_0.mmh"},
	{0x01C8B16AE91844A8ULL, "w_blightlands_217_0.mao"},
	{0x01C8B17499F65627ULL, "zz_approval_debug6003703.cl"},
	{0x01C8E1127CFBA9E3ULL, "vxw_encounter1_e1_0_0.vlm"},
	{0x01C97114320C831BULL, "6077012_m.fxe"},
	{0x01C98E4694F49D57ULL, "dae210_gate_guard6085083.cl"},
	{0x01C9DDBD15885ACBULL, "gen00fl_varric6078675.cl"},
	{0x01C9F3F659A786BEULL, "6111933_m.fxe"},
	{0x01CA4ADE5A4B354EULL, "art\\characters\\playercharacter\\elffemale\\mt_ant_ef04.mmh"},
	{0x01CA54C0A42AD190ULL, "6127620_m.fxe"},
	{0x01CA64DE5A4B613CULL, "art\\characters\\playercharacter\\elffemale\\mt_ant_ef04.msh"},
	{0x01CA831B2961AA7DULL, "w_encounter2_39_0.rml"},
	{0x01CAE661F60E3B53ULL, "dae000_arishok6068531.cl"},
	{0x01CAFDE3A8FA9856ULL, "zz_qun_debug6085864.cl"},
	{0x01CBA4739D33B153ULL, "739936143.ogg"},
	{0x01CC39D7F4A7A40AULL, "mcr310_argument6087315.cl"},
	{0x01CC817A0695615AULL, "crt_pro_cycle.mao"},
	{0x01CCCD2E12B8132CULL, "zz_qcr_debug6074958.cl"},
	{0x01CCDD2B86F9148BULL, "art\\vfx\\combat\\vfx_r_var_activation2_c.anb"},
	{0x01CCF199227E8F72ULL, "490959097.ogg"},
	{0x01CD4356F9DB623CULL, "art\\characters\\playercharacter\\humanfemale\\animation\\exploration\\movement\\runf_injured.gad"},
	{0x01CD518E886244EDULL, "hlac_lt_gallowscourtyard_-1_1z.dds"},
	{0x01CD56C6D43D64A1ULL, "938928808.ogg"},
	{0x01CDD10B214B7572ULL, "hla_ht_0_hightown_a2_131073_1z.dds"},
	{0x01CDD823E3C3F753ULL, "one110_gustav6089601.cl"},
	{0x01CDEEF2FE831364ULL, "cs_mcr370cs_epilogue.wbd"},
	{0x01CE132CC2410420ULL, "6133830_m.fxe"},
	{0x01CE1B16CD6F9B8AULL, "follower_banter6171538.cl"},
	{0x01CE33A4E68D8EB2ULL, "6156006_m.fxe"},
	{0x01CE86C0BB5D8A85ULL, "217899971.ogg"},
	{0x01CE889091B54850ULL, "vxw_sundermount_2ls_0_0.vlm"},
	{0x01CEA1E4FC17764FULL, "pro000cs_act1_crim_mage.cut"},
	{0x01CECDD4ED81CE62ULL, "t3_wdg_w21.tnt"},
	{0x01CECFF85CAEBF76ULL, "art\\vfx\\combat\\vfx_r_arc_pin_shot_act_c.impa.evt"},
	{0x01CF595D0A64DA25ULL, "vxht_0_hightown_n_b27_0_0.vlm"},
	{0x01CF852069068A69ULL, "519941235.ogg"},
	{0x01CFADB71D023A3EULL, "636760602.ogg"},
	{0x01CFEF0298B5F033ULL, "vxw_sundermount_313_0_1.vlm"},
	{0x01D0075E01072065ULL, "mcr371_orsino6083327.cl"},
	{0x01D04F68F67EBF5DULL, "vxlt_0_lowtown_docks_c02_0_0.vlm"},
	{0x01D0AF78AE85B426ULL, "gen00fl_isabela6099417.cl"},
	{0x01D1BDF98864ECCAULL, "vxw_encounter2_camp_4b_0_0.vlm"},
	{0x01D20CBFB5603844ULL, "482685270.ogg"},
	{0x01D2E874EDE79F5AULL, "178305163.ogg"},
	{0x01D2EDBC2287E3C2ULL, "vxht_0_hightown_n_c0g_0_0.vlm"},
	{0x01D3400790900693ULL, "sound_item_materials.gda"},
	{0x01D37ABC68B63262ULL, "ttx_pillar_spire_1.mmh"},
	{0x01D37CBC68B63518ULL, "ttx_pillar_spire_1.msh"},
	{0x01D3B113F3CDBE12ULL, "129696791.ogg"},
	{0x01D40C8057281B9EULL, "dre000_cut_rock_wraith.cnv"},
	{0x01D44256F9E17F3AULL, "art\\characters\\playercharacter\\humanfemale\\animation\\exploration\\movement\\runf_injured.ani"},
	{0x01D46C967FC72789ULL, "character_rendertodepth_p3.psh.fxo.dx11"},
	{0x01D4C6F85CB31B01ULL, "art\\vfx\\combat\\vfx_r_arc_pin_shot_act_c.impa.gad"},
	{0x01D5264D695AC836ULL, "rog_criticalstrike.dds"},
	{0x01D5C33E81C14148ULL, "hlac_ca_caverns2prc_5802_55x.dds"},
	{0x01D6DC8B8DCBA639ULL, "129076433.ogg"},
	{0x01D6EE80572A1EAEULL, "dre000_cut_rock_wraith.bnk"},
	{0x01D7462625D1320CULL, "def_cobble_n.dds"},
	{0x01D752EF6EF1C263ULL, "vxw_encounter1_j0_0_0.vlm"},
	{0x01D77BF86F3AC2BCULL, "w_blightlands_125_0.mao"},
	{0x01D787F86F3AD75FULL, "w_blightlands_125_0.mmh"},
	{0x01D78DF86F3AE101ULL, "w_blightlands_125_0.msh"},
	{0x01D7B00BE9E4F970ULL, "lt_0_lowtown_docks_233_0.msh"},
	{0x01D7BE0BE9E5108AULL, "lt_0_lowtown_docks_233_0.mmh"},
	{0x01D7C20BE9E517C1ULL, "lt_0_lowtown_docks_233_0.mao"},
	{0x01D7C216FB332441ULL, "79106815.ogg"},
	{0x01D7C83798B8C8DDULL, "mag210_arianni6024530.cl"},
	{0x01D8C9C8CF3B59A4ULL, "6178484_m.fxe"},
	{0x01D8DAF53F82E084ULL, "35009841.ogg"},
	{0x01D92BC16EE16261ULL, "vxlt_warehouse_8j_0_0.vlm"},
	{0x01D94F4C4D935BFCULL, "one300_nuncio6075046.cl"},
	{0x01D95A680822BF16ULL, "vxlt_gallowstemplar_6so_0_0.vlm"},
	{0x01DAC3A455D58B94ULL, "mag210_arianni6057320.cl"},
	{0x01DACB140B2F7369ULL, "6137628_m.fxe"},
	{0x01DBB1E152299DA8ULL, "6035905_m.fxe"},
	{0x01DBC7F85CB93BAFULL, "art\\vfx\\combat\\vfx_r_arc_pin_shot_act_c.impa.ani"},
	{0x01DBD25B5520470DULL, "art\\characters\\playercharacter\\dwarfmale\\dm_har_ha11a.mmh"},
	{0x01DBDE068D45A0AFULL, "vxlt_0_lowtown_a3_n_c_gz3_0_0.vlm"},
	{0x01DBE45B55206593ULL, "art\\characters\\playercharacter\\dwarfmale\\dm_har_ha11a.msh"},
	{0x01DC601DE9BC67D9ULL, "609686389.ogg"},
	{0x01DC6F715BBB37E7ULL, "6207707_m.fxe"},
	{0x01DC750F55CC9CDCULL, "vxlt_hangedman_n_1jw_0_0.vlm"},
	{0x01DCFA6D83AB60D0ULL, "vxw_sundermount_2qp_0_0.vlm"},
	{0x01DDBBFB5C703E80ULL, "art\\levels\\races\\proxy\\props\\storage\\prp_zzbarrel.phy"},
	{0x01DDEFDC19FFC3DEULL, "gen_im_arm_glv_hvy_act2_03.uti"},
	{0x01DE4F31208A4155ULL, "vxlt_0_lowtown_n_eny_0_1.vlm"},
	{0x01DE7872A8E8A30AULL, "plt_cod_mgc_possession.plo"},
	{0x01DE9176A15D54FAULL, "art\\vfx\\models\\vfx_cin_rock_02.msh"},
	{0x01DEA376A15D7340ULL, "art\\vfx\\models\\vfx_cin_rock_02.mmh"},
	{0x01DEF6A3E20082D6ULL, "6138805_f.fxe"},
	{0x01DFB448161AFB47ULL, "476205535.ogg"},
	{0x01DFD610F7259C3BULL, "189029176.ogg"},
	{0x01DFE447AFDF4DA0ULL, "mir_rip_sliver.mao"},
	{0x01DFFFE903795315ULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_filecabinet.phy"},
	{0x01E04910B8315ADBULL, "mag100_karras6034708.cl"},
	{0x01E05D4E0BF614CEULL, "vxdr_primevaltomb_7zk_0_0.vlm"},
	{0x01E06548E7FB361BULL, "349712331.ogg"},
	{0x01E114B157B50FF3ULL, "vxht_playershouse_n_1j7_0_0.vlm"},
	{0x01E141D7600B4D00ULL, "735759448.ogg"},
	{0x01E15D673A098BB9ULL, "oip_dust.mao"},
	{0x01E18BCC339D10E3ULL, "188420035.ogg"},
	{0x01E1D6315D82B7B3ULL, "one100ip_st_ilen.utp"},
	{0x01E26224CEAC4D75ULL, "dae000_merchant_access_hf6162977.cl"},
	{0x01E2B47730360E48ULL, "vxlt_warehouse_8o_0_0.vlm"},
	{0x01E33D0B7B247AD4ULL, "6104941_f.fxe"},
	{0x01E37FB89980A04BULL, "sib131_slaver_lord6098067.cl"},
	{0x01E38D02F0D6FAFBULL, "w_encounter3_112_0.msh"},
	{0x01E39F02F0D71946ULL, "w_encounter3_112_0.mao"},
	{0x01E3AB02F0D72DE5ULL, "w_encounter3_112_0.mmh"},
	{0x01E3B7B504618AB8ULL, "vxdr_primevaltomb_81z_0_1.vlm"},
	{0x01E402D8704E693FULL, "art\\vfx\\creatures\\fxc_spider_dppr_c.impa.gad"},
	{0x01E51222BF3FC4BEULL, "125306649.ogg"},
	{0x01E52DDA3819F7DCULL, "270296176.ogg"},
	{0x01E52E924A85706CULL, "art\\vfx\\materials\\vfx_particle_static.mmh"},
	{0x01E534FC3871891AULL, "vxlt_0_lowtown_f0e_0_0.vlm"},
	{0x01E55154849E8463ULL, "qun200_party_comment6090454.cl"},
	{0x01E575D1DBBE2CCDULL, "vxht_0_hightown_a3_ca1_0_0.vlm"},
	{0x01E6566484968BA5ULL, "534968233.ogg"},
	{0x01E73BA5C1317224ULL, "6173584_m.fxe"},
	{0x01E784F789981856ULL, "lowtown_male_beggar_sta6151021.cl"},
	{0x01E7BBFB98AA6CACULL, "fex_spt_pil.msh"},
	{0x01E7C1FB98AA769EULL, "fex_spt_pil.mmh"},
	{0x01E7DF58323CF685ULL, "mrl150_dalish_guard6170402.cl"},
	{0x01E7FB6DB8759C32ULL, "78376993.ogg"},
	{0x01E8B80CDBC6AEE5ULL, "gen00fl_isabela6100004.cl"},
	{0x01E8D00B6CF9A115ULL, "lgt311tr_shw_mer_mettin.ncs"},
	{0x01E914B6C1B3580BULL, "character_character_p80.psh.fxo"},
	{0x01E9E71C1AE6C7B9ULL, "520390239.ogg"},
	{0x01EA4B97D35A13D0ULL, "mer140_bonepit_comments6062992.cl"},
	{0x01EAB659BF527201ULL, "643471853.ogg"},
	{0x01EAF7D87054752CULL, "art\\vfx\\creatures\\fxc_spider_dppr_c.impa.evt"},
	{0x01EB6C4A75A17950ULL, "vxw_sundermount_2zq_0_0.vlm"},
	{0x01EB9136731F74FDULL, "mag160_feynriel6020093.cl"},
	{0x01EBCA11840CBFC7ULL, "isa111_isabela6075421.cl"},
	{0x01EBD45D0DBE3604ULL, "zz_one_debug6025820.cl"},
	{0x01EBDD9EE6451D35ULL, "44718964.ogg"},
	{0x01EC3ABAAC66A5FDULL, "734087125.ogg"},
	{0x01EC3D16EE510B25ULL, "vxht_0_hightown_n_b85_0_0.vlm"},
	{0x01EC5730C87FDBC0ULL, "6073781_m.fxe"},
	{0x01EC5843569BDEF1ULL, "651545614.ogg"},
	{0x01ECA81E1ED83EDAULL, "187830120.ogg"},
	{0x01ECC10E128059D5ULL, "vxlt_0_lowtown_n_ekf_0_1.vlm"},
	{0x01ECCB5B552EAECAULL, "art\\characters\\playercharacter\\dwarfmale\\dm_har_ha11a.phy"},
	{0x01ECDB5D4EE54B5FULL, "art\\effects\\particlelightparams.dat"},
	{0x01ED2C7D7EE92C8CULL, "zz_dae_debug6049998.cl"},
	{0x01EDA595D6A93AE1ULL, "1061589519.ogg"},
	{0x01EE6296C21D7E8CULL, "vxlt_hangedman_28i_0_0.vlm"},
	{0x01EECBAC64CCE5F0ULL, "hlac_w_woundedcoast_65541_0x.dds"},
	{0x01EEF63635BE2E52ULL, "vxca_cavern1_haunted_j4x_0_0.vlm"},
	{0x01EF00119EA81C4EULL, "zz_dae_debug6029918.cl"},
	{0x01EF17276D553AB5ULL, "art\\vfx\\combat\\vfx_r_ibl_fore_loop_c.mmh"},
	{0x01EF2ED9E7616DC2ULL, "dae000ip_crowd_ff2_talk3.utp"},
	{0x01EFDD6AF963F3B8ULL, "vxht_0_hightown_a3_c8g_0_0.vlm"},
	{0x01EFE5633FCD7789ULL, "vxca_caverns1_mines_ehl_0_0.vlm"},
	{0x01F09A2C8FE6C50CULL, "art\\vfx\\combat\\vfx_m_pri_petrify_conj_c.anb"},
	{0x01F0E68F9B836937ULL, "dae020cr_hm_gossip3.utc"},
	{0x01F12FAB647FA073ULL, "6049471_m.fxe"},
	{0x01F1AA934DBDA559ULL, "dae000_meredith6092607.cl"},
	{0x01F1D3EA6074BEDDULL, "40364646.ogg"},
	{0x01F29122E595AFB5ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\1ww_basic.phy"},
	{0x01F2969E8D4E9FEAULL, "and121_karl6093893.cl"},
	{0x01F2FD6140E35CA9ULL, "hlac_ca_caverns2prc_5802_61z.dds"},
	{0x01F3916DC05A2273ULL, "vxw_bonepitmining_a2_1pz_0_0.vlm"},
	{0x01F3A6D92A67A19AULL, "vxlt_0_lowtown_f5f_0_0.vlm"},
	{0x01F3BA7EBC85E2BEULL, "hlac_ht_0_hightown_n_-131074_2y.dds"},
	{0x01F3F22B99185311ULL, "vxw_woundedcoast_39_0_0.vlm"},
	{0x01F43030A6246FDFULL, "hlac_w_bonepitmining_a2_262146_0x.dds"},
	{0x01F4B055E8C886F9ULL, "6161300_m.fxe"},
	{0x01F4BCF1DF097E12ULL, "hlac_ht_chantry_2534_38z.dds"},
	{0x01F4CD35980B40E2ULL, "qcr220cr_nobleman3.utc"},
	{0x01F55214D7F33377ULL, "fx_glow03.dds"},
	{0x01F5F4D677D9FFAEULL, "gen00fl_aveline6004159.cl"},
	{0x01F6102B6ECFE1A0ULL, "6160962_m.fxe"},
	{0x01F6A81CA6DCCC69ULL, "zz_followers_banter6157847.cl"},
	{0x01F6AB37B131AAD9ULL, "250282369.ogg"},
	{0x01F6ACF173B2673BULL, "gen00fl_aveline6132833.cl"},
	{0x01F6B5372D24D43EULL, "vxlt_undercityserialkiller_1t3_0_0.vlm"},
	{0x01F6C1BCD93C2AB4ULL, "6039003_f.fxe"},
	{0x01F6E36B29668D97ULL, "art\\vfx\\models\\vfx_debris_04.mmh"},
	{0x01F6E76B296694A4ULL, "art\\vfx\\models\\vfx_debris_04.mao"},
	{0x01F6F96B2966B309ULL, "art\\vfx\\models\\vfx_debris_04.msh"},
	{0x01F72542E05B91A3ULL, "139941280.ogg"},
	{0x01F78693B3A9708BULL, "character_character_p71.psh.fxo"},
	{0x01F7C5D32F72B1FFULL, "one100dg_kelder6055449.cl"},
	{0x01F81C0385390678ULL, "and230_mage_girl6116260.cl"},
	{0x01F8616067C2AF74ULL, "147824282.ogg"},
	{0x01F8AF0EACB6737AULL, "art\\characters\\playercharacter\\humanfemale\\glv_mede.mmh"},
	{0x01F8B30EACB679B1ULL, "art\\characters\\playercharacter\\humanfemale\\glv_mede.mao"},
	{0x01F8C10EACB691C0ULL, "art\\characters\\playercharacter\\humanfemale\\glv_mede.msh"},
	{0x01F8D1CD9E0BDF6BULL, "vxlt_0_lowtown_a3_n_c_hjl_0_0.vlm"},
	{0x01F8DFD870608985ULL, "art\\vfx\\creatures\\fxc_spider_dppr_c.impa.ani"},
	{0x01F971E86013E254ULL, "art\\vfx\\combat\\vfx_glb_potion_drink_c.impa.ani"},
	{0x01FA3BEDFDCF98DBULL, "vxdr_0_deeproads_du2_0_0.vlm"},
	{0x01FA636E6D4B320CULL, "637429153.ogg"},
	{0x01FAF921B518FF7DULL, "mag121_idunna6084231.cl"},
	{0x01FB0131B2BB4039ULL, "lgt160_ntb_elf_conflict6163368.cl"},
	{0x01FB0E02909236D4ULL, "712523648.ogg"},
	{0x01FB41FF74394BB4ULL, "224939151.ogg"},
	{0x01FB7F99A77317E0ULL, "framebuffereffects_diffusiondof_p11.psh.fxo.dx11"},
	{0x01FBA14D7327376BULL, "vxw_encounter2_r9_0_0.vlm"},
	{0x01FC11EA3CE5C6EAULL, "vxlt_0_lowtown_docks_cp1_0_0.vlm"},
	{0x01FC3A760D82A851ULL, "6088226_f.fxe"},
	{0x01FCA0C6AA8C8D94ULL, "gen00fl_aveline6105463.cl"},
	{0x01FCA83E6932A203ULL, "6104758_m.fxe"},
	{0x01FCC1DBD012CB9FULL, "vxht_mansion2_389_0_0.vlm"},
	{0x01FCD473B413970CULL, "vxlt_hangedman_25n_0_0.vlm"},
	{0x01FCE50B0A7D73BBULL, "290658571.ogg"},
	{0x01FCF5B8E831E27AULL, "follower_banter6155712.cl"},
	{0x01FD14276D617342ULL, "art\\vfx\\combat\\vfx_r_ibl_fore_loop_c.anb"},
	{0x01FD681327B446D2ULL, "vxca_cavern1_haunted_j3y_0_0.vlm"},
	{0x01FD7ABBF41CF9DFULL, "mer320_hubert6053744.cl"},
	{0x01FDE7C475DE8C12ULL, "one350cr_variel.utc"},
	{0x01FE3C28B06B567AULL, "6098902_m.fxe"},
	{0x01FE91BC325444BAULL, "vxca_caverns2prc_9sq_0_0.vlm"},
	{0x01FEA91AD8C0C51FULL, "vxw_encounter3_g1_0_0.vlm"},
	{0x01FEC6C17ABE5160ULL, "vxlt_blackemporium_z9_0_0.vlm"},
	{0x01FECED23654916FULL, "930583148.ogg"},
	{0x01FEEB552E079626ULL, "gen_im_trs_gen_bracelet_03.uti"},
	{0x01FF14B801284C21ULL, "497417416.ogg"},
	{0x01FF198ED1C34459ULL, "vxlt_gallowscourtyard_773_0_0.vlm"},
	{0x01FFB2154FADC7F1ULL, "mag311_keran6062058.cl"},
	{0x01FFDDC6E3D1C40AULL, "gen00fl_bethany6121314.cl"},
	{0x0200403161D83F59ULL, "gen00fl_aveline6116622.cl"},
	{0x020059DD3721E11EULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\lying_injured_exit.ani"},
	{0x02005A475A5C820DULL, "6101027_m.fxe"},
	{0x02006EE86019FBB6ULL, "art\\vfx\\combat\\vfx_glb_potion_drink_c.impa.gad"},
	{0x0200D8FDE548F171ULL, "vxca_caverns2prc_9hs_0_0.vlm"},
	{0x0201AD918EEB91A1ULL, "vxht_chantry_n_36j_0_0.vlm"},
	{0x02020B77AE6B181CULL, "ave100_donnic6058927.cl"},
	{0x02023FC99B3F1D36ULL, "vxlt_0_lowtown_a2_n_c_i18_0_0.vlm"},
	{0x0202D8FB5C8F8C15ULL, "art\\levels\\races\\proxy\\props\\storage\\prp_zzbarrel.msh"},
	{0x0202DAFB5C8F8F8BULL, "art\\levels\\races\\proxy\\props\\storage\\prp_zzbarrel.mmh"},
	{0x0202E6FB5C8FA428ULL, "art\\levels\\races\\proxy\\props\\storage\\prp_zzbarrel.mao"},
	{0x0202F81252463789ULL, "prty_bethany_armor_1.dds"},
	{0x02032232DABE9F79ULL, "6166200_m.fxe"},
	{0x0203470170345992ULL, "989617390.ogg"},
	{0x02034C4ACC3BB18DULL, "llac_w_sundermount_-65538_0.dds"},
	{0x02035FB4FA25328DULL, "vxlt_0_lowtown_a2_ekj_0_2.vlm"},
	{0x0203A2729CCDE70EULL, "lt_0_lowtown_docks_a3_39_0.anb"},
	{0x0203BF19448E1B01ULL, "vxlt_0_lowtown_euz_0_0.vlm"},
	{0x02044B9CD07F75D3ULL, "677141380.ogg"},
	{0x02045C64946646C2ULL, "mag121_viveka6157199.cl"},
	{0x0204820860C5FA20ULL, "6165662_m.fxe"},
	{0x0204B41B9F1A189BULL, "6153541_m.fxe"},
	{0x0205D2E089622C02ULL, "331129245.ogg"},
	{0x0205F870A59F890BULL, "character_character_p22.psh.fxo"},
	{0x02061EAF4BD2A109ULL, "art\\vfx\\creatures\\fxc_gol_sla_c.mmh"},
	{0x020629E261C2FBA3ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\charge_1h_sw.ani"},
	{0x0206381474F1ABA5ULL, "6188720_f.fxe"},
	{0x020689E8601E9515ULL, "art\\vfx\\combat\\vfx_glb_potion_drink_c.impa.evt"},
	{0x0206D4B70F5C8D19ULL, "6036479_m.fxe"},
	{0x020725685F9CEB36ULL, "fen110_anso6095012.cl"},
	{0x02072A5B35612331ULL, "ht_0_hightown_a3_21_0_lc.mmh"},
	{0x02072A97B261BDCDULL, "vxlt_gallowstemplar_n_5jc_0_0.vlm"},
	{0x02076B57D021141BULL, "6166462_m.fxe"},
	{0x0207865662F84123ULL, "art\\characters\\playercharacter\\shared\\heads\\textures\\mt_har_crew_0t.dds"},
	{0x0207AEC938AC8924ULL, "dre000_dougal6129743.cl"},
	{0x020803AB2EE647ACULL, "hlac_dr_primevaltomb_11985_0x.dds"},
	{0x02081FC2A3DDADB4ULL, "hlac_w_bonepitdestroyed_a3_393217_0y.dds"},
	{0x020824017CBE5AC9ULL, "horizontalblur5.vsh.fxo"},
	{0x0208774ADBFA0E7AULL, "gen_im_trs_krk_junk1_02.uti"},
	{0x020898C8B70A4CC1ULL, "zz_followers_banter6158460.cl"},
	{0x0208E4D1AA552E55ULL, "dae000_arishok6072839.cl"},
	{0x02093FEC64169731ULL, "645263111.ogg"},
	{0x020965B502B09D7FULL, "vxlt_0_lowtown_a3_n_c_gjs_0_0.vlm"},
	{0x020A25D419B9ADC0ULL, "follower_banter6163720.cl"},
	{0x020A3A5541ED7731ULL, "atl_startmenu2_dxt1_dat.xml"},
	{0x020A479BC42E1B26ULL, "vxw_encounter3_bk_0_0.vlm"},
	{0x020A6AF8362991B7ULL, "6172531_m.fxe"},
	{0x020A9ACAA3B22F5BULL, "static_staticreliefmap_p0.gsh.fxo.dx11"},
	{0x020ACC096FF4920DULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\0ff_crumble.msh"},
	{0x020ADA096FF4AA40ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\0ff_crumble.mao"},
	{0x020ADE096FF4B093ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\0ff_crumble.mmh"},
	{0x020B3E46717F51D3ULL, "vxlt_darktown_6j1_0_0.vlm"},
	{0x020B428209C90D32ULL, "vxlt_0_lowtown_docks_n_bba_0_0.vlm"},
	{0x020B4650A609AF8CULL, "vxlt_hangedman_22o_0_0.vlm"},
	{0x020BCFD8BA2CBBE6ULL, "6004095_m.fxe"},
	{0x020C0AB6C549229FULL, "6180963_m.fxe"},
	{0x020CC124DD5024B8ULL, "vxht_0_hightown_a3_c6e_0_0.vlm"},
	{0x020D44EFDF4009A8ULL, "6173870_m.fxe"},
	{0x020D6D06B3CE06D8ULL, "one120_athenril6088920.cl"},
	{0x020DA7D3C64A6445ULL, "307771646.ogg"},
	{0x020E0346DDC07662ULL, "vxdr_primevaltomb_4im_0_0.vlm"},
	{0x020E71DD372E47EFULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\lying_injured_exit.evt"},
	{0x020EB68AC54E7B52ULL, "6092258_m.fxe"},
	{0x020ED5DB041E2A84ULL, "6095499_m.fxe"},
	{0x020FCC7679D26F17ULL, "6133629_f.fxe"},
	{0x02101AC2ECB96A55ULL, "vxw_woundedcoast_1fk_0_0.vlm"},
	{0x02101F6E80E1AA21ULL, "vxht_chantry_n_33i_0_0.vlm"},
	{0x02101FB4EA80F1FAULL, "hlac_dr_0_deeproads_5845_14y.dds"},
	{0x02104681F05A4770ULL, "isa210ip_at_foundry.ncs"},
	{0x0210559E5F195828ULL, "6072324_m.fxe"},
	{0x021088A84B7C9566ULL, "447407321.ogg"},
	{0x0210C1617F6FC45FULL, "w_woundedcoast_87_0.rml"},
	{0x0211421C0717D53EULL, "vxlt_blackemporium_2o_0_0.vlm"},
	{0x02121A055A16A6C7ULL, "6153985_m.fxe"},
	{0x02121EDCF5CDAE1BULL, "vxw_blightlands_7i_0_0.vlm"},
	{0x0212ECF18D5EB487ULL, "313346623.ogg"},
	{0x02135BA33750A54AULL, "vxlt_hangedman_15s_0_0.vlm"},
	{0x02136EAD83BFD78DULL, "vxlt_undercityserialkiller_144_0_0.vlm"},
	{0x0213A4920FEE4C7AULL, "846670694.ogg"},
	{0x0213B1004BB83B7EULL, "937310838.ogg"},
	{0x021441E261CF619AULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\charge_1h_sw.evt"},
	{0x021446153DBEC8E3ULL, "art\\levels\\races\\natural\\kirkcoastal\\foliage\\bsh_bushb.msh"},
	{0x021454153DBEE03DULL, "art\\levels\\races\\natural\\kirkcoastal\\foliage\\bsh_bushb.mmh"},
	{0x021472DD3732B498ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\lying_injured_exit.gad"},
	{0x02150C740131EF0DULL, "6155655_m.fxe"},
	{0x0215375A2F9C1B35ULL, "zz_vault_debug.cnv"},
	{0x0215B2408B53C348ULL, "art\\levels\\races\\natural\\kirkcoastal\\foliage\\tre_cashstumpsma.mmh"},
	{0x0215D0408B53F672ULL, "art\\levels\\races\\natural\\kirkcoastal\\foliage\\tre_cashstumpsma.msh"},
	{0x0216D239AD763A26ULL, "vxht_mansion2_30t_0_0.vlm"},
	{0x0216F4B2E2C3E057ULL, "65601744.ogg"},
	{0x02171BD218907065ULL, "enemytrap_ret.mao"},
	{0x0217FFD838128397ULL, "lgt110_gxa_coa_spy6117450.cl"},
	{0x02180BFE242D6575ULL, "vxw_sundermount_20n_0_0.vlm"},
	{0x021813B04C2198FCULL, "gen00fl_anders6125435.cl"},
	{0x0218506CFE2A1775ULL, "455117835.ogg"},
	{0x0218561D7ACD280FULL, "vxlt_0_lowtown_a3_n_c_gx9_0_0.vlm"},
	{0x021884921EDB3423ULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\custom\\scaredmage_trim.ani"},
	{0x021884F2EE723085ULL, "hlac_lt_0_lowtown_a2_65537_1y.dds"},
	{0x02189D8FA95FE952ULL, "gen_im_arm_cht_hvy_act1_04.uti"},
	{0x0219FBFDEF4973C3ULL, "329007565.ogg"},
	{0x021A2418127276CFULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\left_hand_point_forward_2.gad"},
	{0x021A4BCC7365B4BAULL, "hlac_lt_gallowstemplar_2392_0z.dds"},
	{0x021B01E10E703C72ULL, "gen00fl_anders6196732.cl"},
	{0x021B3AFA15AFC109ULL, "vxca_caverns1_mines_egq_0_0.vlm"},
	{0x021B40E261D57E8DULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\charge_1h_sw.gad"},
	{0x021B5BBF4DAB4A9CULL, "vxlt_0_lowtown_a2_n_hcs_0_0.vlm"},
	{0x021BAFD318C689C6ULL, "static1_relief.vsh.fxo"},
	{0x021BB65DB9705C27ULL, "dae000_epic_merchant_3.bnk"},
	{0x021BC7B5F05315D0ULL, "21736382.ogg"},
	{0x021BD3E98B462C04ULL, "llac_ht_0_hightown_n_-65535_1.dds"},
	{0x021D1016CC2ABF54ULL, "953924992.ogg"},
	{0x021D847D96D1EB35ULL, "ser100_party_comments6120593.cl"},
	{0x021DCE0EACD5C385ULL, "art\\characters\\playercharacter\\humanfemale\\glv_mede.phy"},
	{0x021DD3E336EF5688ULL, "425418143.ogg"},
	{0x021E46BB61253996ULL, "711405394.ogg"},
	{0x021E8070DD77A9E6ULL, "character_character_p142.psh.fxo"},
	{0x021E8C9FDEAF82D5ULL, "vxw_woundedcoast_1aj_0_0.vlm"},
	{0x021F3A5A2FA47492ULL, "zz_vault_debug.ncs"},
	{0x021F83D797C59D54ULL, "vxca_caverns1_eb2_0_0.vlm"},
	{0x021FA63E3632CCF8ULL, "vxht_mansion2b_2f9_0_0.vlm"},
	{0x021FD85DB9747E9FULL, "dae000_epic_merchant_3.cnv"},
	{0x022041A66A74531FULL, "6143373_m.fxe"},
	{0x0220539132B85BD3ULL, "6113025_f.fxe"},
	{0x022090B9E7C3C69BULL, "vxw_blightlands_8h_0_0.vlm"},
	{0x0220B44EF3679CC2ULL, "genpt_personality.ncs"},
	{0x0220F2EC952EA49BULL, "mag100_cullen6093555.cl"},
	{0x0220FD04B3A6F406ULL, "crt_sf_dust.mao"},
	{0x02210662F24DDBADULL, "llac_dr_primevaltomb_659_10.dds"},
	{0x022139181278B97CULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\left_hand_point_forward_2.evt"},
	{0x022146CA31057E97ULL, "vxht_mansion_n_4hq_0_0.vlm"},
	{0x02216A07F6B39A5DULL, "mag101_grace6034239.cl"},
	{0x02220C0D9DAF1A86ULL, "fxc_puke_dis.mao"},
	{0x02221DF3D5BB82B4ULL, "vxca_caverns2prc_3nv_0_0.vlm"},
	{0x0223165BADE5E317ULL, "art\\characters\\weapons\\longbows\\w_lbw_dslb.mmh"},
	{0x02231A02D0DD0E6AULL, "313801561.ogg"},
	{0x02231A5BADE5EA24ULL, "art\\characters\\weapons\\longbows\\w_lbw_dslb.mao"},
	{0x02232C5BADE60889ULL, "art\\characters\\weapons\\longbows\\w_lbw_dslb.msh"},
	{0x0223C24584FF7D78ULL, "dae000cr_grand_cleric6147717.cl"},
	{0x0223D18750FAAD40ULL, "356469182.ogg"},
	{0x0223E79AB7338C50ULL, "montage_subs_demohack6217123.cl"},
	{0x02244FA19FF218E0ULL, "509054391.ogg"},
	{0x0224A16B49DAF3D1ULL, "zz_one_debug6099943.cl"},
	{0x0224CE15BC8A146BULL, "6110980_m.fxe"},
	{0x0224CE7897B803C3ULL, "vxw_bonepitmining_a2_mv_0_0.vlm"},
	{0x0224F6C7000B0012ULL, "6208282_m.fxe"},
	{0x0224FFA0317DD02BULL, "6143867_m.fxe"},
	{0x022544169F6C52A6ULL, "vxht_mansion2_37u_0_0.vlm"},
	{0x0225E1FAB4724019ULL, "296174770.ogg"},
	{0x0225F3883D00DB5AULL, "156872740.ogg"},
	{0x022626E07E913D2BULL, "ave200_donnic6110647.cl"},
	{0x022650F4FF41D747ULL, "6110395_m.fxe"},
	{0x0226525556C858BDULL, "static_static_p6.vsh.fxo.dx11"},
	{0x02267DDB16237DF5ULL, "vxw_sundermount_25m_0_0.vlm"},
	{0x02267F1B29B0019DULL, "w_encounter2_39_0.mao"},
	{0x0226831B29B008D6ULL, "w_encounter2_39_0.mmh"},
	{0x02268B87C40DB3BEULL, "626923295.ogg"},
	{0x02268D1B29B019D4ULL, "w_encounter2_39_0.msh"},
	{0x02269C921EE79A1AULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\custom\\scaredmage_trim.evt"},
	{0x022712E903B57630ULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_filecabinet.msh"},
	{0x022720E903B58E4AULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_filecabinet.mmh"},
	{0x022724E903B59481ULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_filecabinet.mao"},
	{0x0227782A7DA59625ULL, "art\\vfx\\materials\\chunky\\vfx_c_fire_tile_add_scl_2.mao"},
	{0x02288362DB2D91C1ULL, "sib131cr_slave_a.utc"},
	{0x0228EB315DBEDE66ULL, "one100ip_st_ilen.ncs"},
	{0x02297069505CAD3AULL, "hlac_lt_0_lowtown_docks_a3_0_5z.dds"},
	{0x0229789D9621920FULL, "fighting60c.mao"},
	{0x02297A90CA00B246ULL, "lgt250_way_dalish_conflict6147155.cl"},
	{0x02297ED8A6821B5BULL, "vxlt_0_lowtown_a2_fff_0_0.vlm"},
	{0x0229C0E22BA8E719ULL, "dae221ip_at_brothel.utp"},
	{0x0229D93B61934B4CULL, "ave151cr_ambush_bandit_f.utc"},
	{0x022A0BF9C052CB0EULL, "255286280.ogg"},
	{0x022A2060EB9A2FD7ULL, "696054786.ogg"},
	{0x022A2FF029B681AFULL, "55116379.ogg"},
	{0x022A41FFFAEFF8DBULL, "narrative_varric6172470.cl"},
	{0x022AEF604BCE4D9FULL, "708385550.ogg"},
	{0x022AF737E6885349ULL, "6117188_f.fxe"},
	{0x022B2065301EC8BEULL, "gen00fl_dog6117796.cl"},
	{0x022B7CDE538D2966ULL, "hlac_lt_undercitydungeon_-1_2y.dds"},
	{0x022B8402F114A092ULL, "w_encounter3_112_0.phy"},
	{0x022BBA10BC3EC603ULL, "w_sundermount_252_0.mao"},
	{0x022BBE10BC3ECD50ULL, "w_sundermount_252_0.mmh"},
	{0x022BCC10BC3EE56AULL, "w_sundermount_252_0.msh"},
	{0x022BD6E29EC0331BULL, "6087011_m.fxe"},
	{0x022BEA2EDABD429BULL, "vxht_brothel_n_6kq_0_0.vlm"},
	{0x022C1A8228640065ULL, "t1_skn_h05.tnt"},
	{0x022C20A9B074545AULL, "44305587.ogg"},
	{0x022C810B09C24F09ULL, "6148129_m.fxe"},
	{0x022CABAF58A3D784ULL, "6027047_m.fxe"},
	{0x022CFE7CD0A59B55ULL, "vxw_woundedcoast_1li_0_0.vlm"},
	{0x022D0FAECE2B6AC9ULL, "vxw_bonepitmining_a2_2a0_0_0.vlm"},
	{0x022D102C9C5C6E4FULL, "art\\characters\\creatures\\mubari\\animation\\pp_rst_p.gad"},
	{0x022D4042C717F6D6ULL, "gen_im_arm_hlm_lgt_act2_01.uti"},
	{0x022D9B921EEDB70DULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\custom\\scaredmage_trim.gad"},
	{0x022DD0EA30F6FB30ULL, "follower_banter6141683.cl"},
	{0x022E0533028631FBULL, "382468831.ogg"},
	{0x022E2549C1DE90BDULL, "lt_undercityencounter_mmap.dds"},
	{0x022E2B14C6109F2CULL, "413602488.ogg"},
	{0x022E3A9BD18729B2ULL, "hlac_ht_0_hightown_a2_131070_0y.dds"},
	{0x022E411812835095ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\left_hand_point_forward_2.ani"},
	{0x022E47AFABA6C426ULL, "vxht_0_hightown_n_bxb_0_0.vlm"},
	{0x022E6F1014FE819CULL, "1030398000.ogg"},
	{0x022F1BAF4BF5A13EULL, "art\\vfx\\creatures\\fxc_gol_sla_c.anb"},
	{0x022F6D2268E51C7CULL, "seb300im_resolution_helm.uti"},
	{0x022F7888F7967737ULL, "art\\characters\\playercharacter\\qunarimale\\eye_com.mao"},
	{0x022FA9CB0228115AULL, "273915825.ogg"},
	{0x022FBB681154635CULL, "genip_node_c2.utp"},
	{0x02304233C216AD2FULL, "6128346_f.fxe"},
	{0x0230AA96B8FC5A00ULL, "412768.ogg"},
	{0x0230F3537AB9D84CULL, "hlac_dr_dungeon_3207_5y.dds"},
	{0x0231226A18485B20ULL, "vxlt_0_lowtown_a2_bxw_0_0.vlm"},
	{0x02312D96A8C36BBCULL, "llac_ht_keep_n_3732_14.dds"},
	{0x02316B69A8B0F7A0ULL, "6155319_m.fxe"},
	{0x02316BAC1577D587ULL, "9311751.ogg"},
	{0x023176DD16F23B17ULL, "one200cr_solivitus.ncs"},
	{0x0231C92927ADFF4FULL, "664892581.ogg"},
	{0x0233405589AE1C43ULL, "vxw_bonepitmining_a2_hy_0_0.vlm"},
	{0x023347B99DC0C2CFULL, "vxlt_0_lowtown_a2_n_c_hje_0_0.vlm"},
	{0x023370D4612154A7ULL, "834426679.ogg"},
	{0x02338D738DCA32DEULL, "art\\characters\\weapons\\arrows\\w_arw_001.dura.evt"},
	{0x0234252C9C62B0FCULL, "art\\characters\\creatures\\mubari\\animation\\pp_rst_p.evt"},
	{0x02346CF2A2973BEDULL, "444449558.ogg"},
	{0x023478FAFC86D99DULL, "150179660.ogg"},
	{0x0234B1625ACE0E8EULL, "w_blightlands_41_0.msh"},
	{0x0234BF625ACE26DFULL, "w_blightlands_41_0.mao"},
	{0x0234CB625ACE3B7CULL, "w_blightlands_41_0.mmh"},
	{0x0234D2C11CC35A27ULL, "636418678.ogg"},
	{0x0234F5579BA063EAULL, "w_bonepit_221_0.mao"},
	{0x0234F9579BA06A39ULL, "w_bonepit_221_0.mmh"},
	{0x023503579BA07B27ULL, "w_bonepit_221_0.msh"},
	{0x02353E3470ACF388ULL, "hlac_lt_0_lowtown_docks_a3_n_1_0y.dds"},
	{0x02358850CE6D8385ULL, "zz_ave_debug6064130.cl"},
	{0x02369818DFAB56B2ULL, "vxlt_0_lowtown_docks_n_bql_0_0.vlm"},
	{0x0236A5729CF94672ULL, "lt_0_lowtown_docks_a3_39_0.rml"},
	{0x0236C8E828DB4925ULL, "919519717.ogg"},
	{0x0236D91C72665AACULL, "art\\vfx\\combat\\vfx_ha_bd_bfrz_frd_lp_c.cess.ani"},
	{0x023711D3D038C902ULL, "6042255_m.fxe"},
	{0x02376D565624CBCDULL, "730904185.ogg"},
	{0x023775C84B66D460ULL, "ip_restrict_class_war_rog.ncs"},
	{0x02379B459957CADBULL, "521084878.ogg"},
	{0x0237B09A45BA0AC4ULL, "hlac_lt_0_lowtown_a3_n_c_65534_1y.dds"},
	{0x02380E91693C8BD1ULL, "6207576_m.fxe"},
	{0x02383A3BB10EEE1BULL, "6121024_f.fxe"},
	{0x0238592FFA2CA6BAULL, "vxca_caverns2prc_9om_0_0.vlm"},
	{0x02386052D3342779ULL, "hlac_ht_keep_3732_18z.dds"},
	{0x023892F15A957DEFULL, "gen_im_trs_gen_junk2_02.uti"},
	{0x023901ACC75552B3ULL, "t1_skn_e06.tnt"},
	{0x02393FD23C6B59CBULL, "llac_ht_0_hightown_n_131073_0.dds"},
	{0x023990F86F8DB0A4ULL, "w_blightlands_125_0.phy"},
	{0x0239EBF74E7D864BULL, "zz_lgt_3_debug6140435.cl"},
	{0x023A2797B625495CULL, "6122432_m.fxe"},
	{0x023A5C0BCCB35B1BULL, "vxht_brothel_n_6tp_0_0.vlm"},
	{0x023AA422E5D385D0ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\1ww_basic.msh"},
	{0x023AA4738DD07809ULL, "art\\characters\\weapons\\arrows\\w_arw_001.dura.gad"},
	{0x023AB222E5D39DEAULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\1ww_basic.mmh"},
	{0x023AB622E5D3A421ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\1ww_basic.mao"},
	{0x023ACEDF28B5683EULL, "narrative_banter6149658.cl"},
	{0x023AEEEB4CA80219ULL, "location_states.gda"},
	{0x023AF13AD977130BULL, "hlac_w_encounter2_camp_65534_0x.dds"},
	{0x023AF2E7FBB86789ULL, "6145229_m.fxe"},
	{0x023B12C99AE3C0A6ULL, "art\\characters\\weapons\\staff\\textures\\w_stf_015a_0d.dds"},
	{0x023B829FE687C7D8ULL, "55378402.ogg"},
	{0x023BC63A69B22AA3ULL, "103297414.ogg"},
	{0x023C51D991B1ABBFULL, "6107082_f.fxe"},
	{0x023C67917BB1CE54ULL, "vxca_caverns1_eh0_0_0.vlm"},
	{0x023C6CAA45CCD789ULL, "gen00fl_isabela6096765.cl"},
	{0x023C6D36CD3DEFDFULL, "552445884.ogg"},
	{0x023CB98C9D9CDCA6ULL, "vxht_0_hightown_n_bcc_0_0.vlm"},
	{0x023CFB00BA82A834ULL, "hlac_w_blightlands_393214_0y.dds"},
	{0x023CFEEFEBB90F24ULL, "209325487.ogg"},
	{0x023D13B3B437A7ACULL, "mage_chainlightning.dds"},
	{0x023D2DFFEEE79C2CULL, "mrl310_party_comments6143069.cl"},
	{0x023D588AC92047B0ULL, "gen00fl_anders6142414.cl"},
	{0x023DE5916805FF12ULL, "vxlt_hangedman_n_2k1_0_0.vlm"},
	{0x023DE61C726C8FEEULL, "art\\vfx\\combat\\vfx_ha_bd_bfrz_frd_lp_c.cess.gad"},
	{0x023DFCAE9062AA8BULL, "vxlt_0_lowtown_h8v_0_0.vlm"},
	{0x023F0C06C4E2F102ULL, "darktown_fem_fereldan_sta6156752.cl"},
	{0x023F5EC0F78220A0ULL, "vxlt_warehouse_n_130_0_0.vlm"},
	{0x023F77C784D65EC2ULL, "isa211_isabela6071610.cl"},
	{0x02405262AC4CEC25ULL, "art\\characters\\playercharacter\\humanmale\\textures\\bdy_arm_maso_0n.dds"},
	{0x0240F0AAB0B0B3E5ULL, "56464989.ogg"},
	{0x024104650A71895CULL, "store_lgt_epic2_1.utm"},
	{0x02412D2C9C6D4815ULL, "art\\characters\\creatures\\mubari\\animation\\pp_rst_p.ani"},
	{0x024175738DD64757ULL, "art\\characters\\weapons\\arrows\\w_arw_001.dura.ani"},
	{0x024198B2C6E9EDE4ULL, "hlac_ht_chantry_n_2534_28x.dds"},
	{0x0241F5666B754204ULL, "vxlt_backalley_n_1ki_0_0.vlm"},
	{0x02423D1EEF273011ULL, "680076734.ogg"},
	{0x02427BECCF24B08CULL, "gen00fl_fenris6105592.cl"},
	{0x02429F267389183EULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\sibling_dieaction1.evt"},
	{0x024332C194FC60EAULL, "hlac_lt_gallowstemplar_2383_4x.dds"},
	{0x02436136C592C0F5ULL, "lgt360_mag_orlanna6156292.cl"},
	{0x0243FD9C8908DC89ULL, "6099414_m.fxe"},
	{0x02444DABB743CFCAULL, "lt_0_lowtown_a2_n_35_0_lc.mmh"},
	{0x024472CEB9466F31ULL, "6180415_m.fxe"},
	{0x0244E11C7272A5CDULL, "art\\vfx\\combat\\vfx_ha_bd_bfrz_frd_lp_c.cess.evt"},
	{0x0244E5756B805EADULL, "94908601.ogg"},
	{0x0245510C17FF0D8CULL, "vxlt_0_lowtown_docks_b50_0_1.vlm"},
	{0x024559EE033AFB3AULL, "349964445.ogg"},
	{0x0245781855D23A11ULL, "vxlt_gamlenshouse_n_az_0_0.vlm"},
	{0x0245929737115EB7ULL, "rdr111dg_martin6015058.cl"},
	{0x024689B9825FDA2AULL, "746612217.ogg"},
	{0x0246C2A424BDCAE3ULL, "6142217_m.fxe"},
	{0x0246C305BCBFB1EAULL, "940825965.ogg"},
	{0x024732FC9C7EE002ULL, "938400262.ogg"},
	{0x02479D5450111B2AULL, "6206561_m.fxe"},
	{0x0247F1F83EC63AC4ULL, "one200ip_ink_chest.utp"},
	{0x02481FB5E8D4762AULL, "277450829.ogg"},
	{0x02483ACF2FAA7EB4ULL, "zz_followers_banter6158940.cl"},
	{0x02486C7E93A43003ULL, "vxht_0_hightown_n_aig_0_0.vlm"},
	{0x0248B63F031AB599ULL, "6059304_m.fxe"},
	{0x0248BC3898D429C1ULL, "104177900.ogg"},
	{0x0248C6A5A56E2ADFULL, "vxdr_primevalentrance_2zj_0_1.vlm"},
	{0x0248DCF7A5C296A9ULL, "6182080_m.fxe"},
	{0x0249781C911DC7B6ULL, "140498356.ogg"},
	{0x0249A98708109D64ULL, "vxht_0_hightown_bc2_0_0.vlm"},
	{0x0249B626738F5E69ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\sibling_dieaction1.gad"},
	{0x0249BF8654013CAFULL, "6128006_m.fxe"},
	{0x0249E236B491CC55ULL, "vxw_woundedcoast_1rw_0_0.vlm"},
	{0x024AE9DE505D4296ULL, "vxlt_0_lowtown_docks_n_arz_0_0.vlm"},
	{0x024B9863716321D5ULL, "mrl150_merrill6085803.cl"},
	{0x024CD78290B08730ULL, "vxca_cave_1o8_0_1.vlm"},
	{0x024D0EE3502FF0E9ULL, "6025978_m.fxe"},
	{0x024D1D7C666FD1BFULL, "6207853_m.fxe"},
	{0x024D358AB812FF52ULL, "6127316_m.fxe"},
	{0x024DB6DC9E96F290ULL, "20033468.ogg"},
	{0x024DBE802F71D5EAULL, "hlac_w_bonepitdestroyed_a3_393219_0y.dds"},
	{0x024E4416ABD5ED61ULL, "hlac_ht_0_hightown_a3_65536_0x.dds"},
	{0x024EAEBA0D5C9BAEULL, "6057349_m.fxe"},
	{0x024EB2A789E42631ULL, "858665221.ogg"},
	{0x024FB13E6C1A6365ULL, "hlac_ht_0_hightown_n_0_0y.dds"},
	{0x024FFF6108FDD31BULL, "art\\characters\\creatures\\desiredemon\\animation\\wkf.ani"},
	{0x025035D740BE56BEULL, "t3_dog_d06.tnt"},
	{0x02503D9CAE99ED09ULL, "vxht_0_hightown_a2_bau_0_0.vlm"},
	{0x025067435D6B5A84ULL, "vxlt_backalley_n_1vn_0_0.vlm"},
	{0x0250872673952CB7ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\sibling_dieaction1.ani"},
	{0x0250CCBAF0E97B23ULL, "gen00fl_varric6133853.cl"},
	{0x02514AF676212CC7ULL, "and111_lirene6084537.cl"},
	{0x0251BD6F6AE47170ULL, "6077312_m.fxe"},
	{0x0251E30F1EB89069ULL, "6137479_m.fxe"},
	{0x0251EE81EE5D82E4ULL, "lt_0_lowtown_docks_a3_20_0.anb"},
	{0x025274477DC543F2ULL, "218628858.ogg"},
	{0x0252C6FB0E76FEF7ULL, "6119532_m.fxe"},
	{0x0252F56283C46734ULL, "41634055.ogg"},
	{0x025307B14F6EEC2EULL, "6019174_m.fxe"},
	{0x02533357F6C6122EULL, "gen00fl_anders6113058.cl"},
	{0x025398AB2A385DE8ULL, "hlac_ht_playershouse_n_855_13y.dds"},
	{0x0253CB625AE8EBCFULL, "w_blightlands_41_0.rml"},
	{0x02540F97CB33E37AULL, "vxht_0_hightown_a82_0_0.vlm"},
	{0x0254560122173FD4ULL, "590983552.ogg"},
	{0x02545A400D40248CULL, "ss_tower.dds"},
	{0x0254D59018073006ULL, "gen_im_arm_glv_lgt_act3_02.uti"},
	{0x0255084CD5EB0D28ULL, "hlac_lt_hangedman_-1_21x.dds"},
	{0x0255491A13189741ULL, "vxlt_0_lowtown_docks_n_9k6_0_0.vlm"},
	{0x025554F761267F1CULL, "follower_banter6169708.cl"},
	{0x02556D8BF96A79F6ULL, "fen110_fenris6100237.cl"},
	{0x0255815440BF1254ULL, "hlac_lt_0_lowtown_a2_0_9z.dds"},
	{0x0255F47C27181A7DULL, "vxht_mansion2b_336_0_0.vlm"},
	{0x02560863D92237D8ULL, "hlac_w_encounter3_-262143_0z.dds"},
	{0x02562763DEB4C348ULL, "dae000_flemeth6013301.cl"},
	{0x025722FC402D01D6ULL, "724129579.ogg"},
	{0x02573FC5B09F8C1BULL, "vxht_brothel_n_6zr_0_0.vlm"},
	{0x0257954D98C78506ULL, "llac_dr_slavers_3216_6.dds"},
	{0x025815FD724FAA29ULL, "6044658_m.fxe"},
	{0x02581B63FA06B5E4ULL, "vxht_0_hightown_bl3_0_0.vlm"},
	{0x025845E5339C224FULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_showcase_n.dds"},
	{0x02584E54CED5015BULL, "vxht_keep_73h_0_0.vlm"},
	{0x02592AAB72D76496ULL, "6037407_f.fxe"},
	{0x02599D4681890DA6ULL, "vxht_0_hightown_n_bie_0_0.vlm"},
	{0x025A18190885F1D9ULL, "vxht_mansion2_2og_0_0.vlm"},
	{0x025A279D1F061B93ULL, "hlac_lt_gallowstemplar_n_2384_5x.dds"},
	{0x025AA5E1FE9DF51BULL, "llac_ca_cavern1_haunted_24423_11.dds"},
	{0x025AF4B7BF523D8CULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\0ff_22base.phy"},
	{0x025B91C9286C25ABULL, "185781131.ogg"},
	{0x025BE91477DF6EC4ULL, "gen00fl_bethany6102892.cl"},
	{0x025C046A5662A4DDULL, "mcr310_cut_chantry_blows_up6087560.cl"},
	{0x025C5324CF28199FULL, "hlac_lt_0_lowtown_n_-131072_1y.dds"},
	{0x025CFECD9D653868ULL, "91242934.ogg"},
	{0x025D076109086B12ULL, "art\\characters\\creatures\\desiredemon\\animation\\wkf.evt"},
	{0x025D2B4299919CD0ULL, "6113606_f.fxe"},
	{0x025D41F9DC63EF03ULL, "hlac_lt_gallowstemplar_7198_0z.dds"},
	{0x025E3D81F09C5979ULL, "isa210ip_at_foundry.utp"},
	{0x025FAFE92D5EDBEBULL, "vxw_sundermount_2zo_0_1.vlm"},
	{0x025FB5617FB3844FULL, "w_woundedcoast_87_0.mao"},
	{0x025FB5F984E94411ULL, "hlac_lt_0_lowtown_docks_n_-65538_8x.dds"},
	{0x025FC1617FB398ECULL, "w_woundedcoast_87_0.mmh"},
	{0x025FC7617FB3A2DEULL, "w_woundedcoast_87_0.msh"},
	{0x025FD9D4E79286D4ULL, "art\\vfx\\combat\\vfx_gib_insect_bld_p.mmh"},
	{0x0260448CC75C9BC4ULL, "art\\levels\\races\\proxy\\props\\armoire\\animation\\opening.gad"},
	{0x02604DBE320AC89DULL, "6095018_m.fxe"},
	{0x02604F05337F1D9AULL, "mag360cr_cullen.utc"},
	{0x026100C5382F63E3ULL, "gst_hands_clasped.gda"},
	{0x026138D8006D1777ULL, "6114232_m.fxe"},
	{0x026159BEC9BAFDF8ULL, "bl_ar_cl.mao"},
	{0x02616FB3778F53E5ULL, "dae011_corff_sta6149647.cl"},
	{0x0261E9CFDFF19FAFULL, "hlac_w_encounter1_65536_0y.dds"},
	{0x0262AC0EB24D5D56ULL, "hlac_lt_0_lowtown_131070_0y.dds"},
	{0x0262AFB9ADCEA02BULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\wander_right.evt"},
	{0x0262F9ED9736A7BBULL, "hlac_lt_0_lowtown_a2_65536_1x.dds"},
	{0x026373D4B6512D51ULL, "vxlt_darktown_2r7_0_0.vlm"},
	{0x0263BAF7050EAFC1ULL, "vxlt_0_lowtown_docks_n_9f5_0_0.vlm"},
	{0x02642661090EBEA5ULL, "art\\characters\\creatures\\desiredemon\\animation\\wkf.gad"},
	{0x026446F9178BE069ULL, "690869382.ogg"},
	{0x02644FAABE544F43ULL, "pro000_colonel_ewald6040584.cl"},
	{0x02646B7FEC6D8EB9ULL, "fen310_fenris6118680.cl"},
	{0x0264731FA137195BULL, "hlac_w_sundermount_196613_0z.dds"},
	{0x02650B17D67936B8ULL, "vxht_keep_4ar_0_0.vlm"},
	{0x026510577E95B338ULL, "6136129_m.fxe"},
	{0x02656087314899C3ULL, "6191297_m.fxe"},
	{0x026579ECD184E77DULL, "vxlt_gallowstemplar_6q6_0_0.vlm"},
	{0x026599F8E706E699ULL, "6053104_m.fxe"},
	{0x0265A917E3455C68ULL, "6119419_f.fxe"},
	{0x0266538CC76120A2ULL, "art\\levels\\races\\proxy\\props\\armoire\\animation\\opening.ani"},
	{0x02666951C57B0D9FULL, "73321348.ogg"},
	{0x0266EFAE9FB18471ULL, "vxlt_0_lowtown_a3_n_c_ewh_0_0.vlm"},
	{0x02672D0D21D03746ULL, "658129838.ogg"},
	{0x0267C20032AD6CB6ULL, "104109959.ogg"},
	{0x0267D8AE3DCD4B0AULL, "105475712.ogg"},
	{0x026852AFB03EC354ULL, "ave260cr_beast.utc"},
	{0x0268A1525AE77B9BULL, "llac_lt_0_lowtown_docks_-3_1.dds"},
	{0x0268F211832C54DCULL, "dae000ip_wall_lean_left_d.utp"},
	{0x026958F5446DC387ULL, "follower_banter6172697.cl"},
	{0x0269BB4CED8ABAC8ULL, "vxht_keep_n_67y_0_0.vlm"},
	{0x0269C8B9ADD4E9ECULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\wander_right.gad"},
	{0x026A14938E848677ULL, "vxlt_gallowscourtyard_63p_0_0.vlm"},
	{0x026A7099EEE93633ULL, "art\\cinematics\\_crit path\\prologue\\pro000cs_flemeth_real\\cs_hur2_flemeth_fake.ani"},
	{0x026B738DA1A427A6ULL, "1041345516.ogg"},
	{0x026B80A850653455ULL, "venge_revival.dds"},
	{0x026B8F9CFD21E396ULL, "6103628_m.fxe"},
	{0x026B9A899F038477ULL, "mag210_arianni6182207.cl"},
	{0x026BF794A0D124F1ULL, "6068986_m.fxe"},
	{0x026C4D611EAA5BBAULL, "dre000_cut2d_ragstoriches6025103.cl"},
	{0x026C60B52DBCA36BULL, "vxlt_0_lowtown_a3_n_c_h2d_0_0.vlm"},
	{0x026C6BEA2C1E48F6ULL, "428323297.ogg"},
	{0x026CA745D32C53E7ULL, "6161618_m.fxe"},
	{0x026CACF2CA4B95ECULL, "6068001_f.fxe"},
	{0x026D0F2D659924CFULL, "vxlt_0_lowtown_a2_n_c_hva_0_0.vlm"},
	{0x026DB27050D3DD96ULL, "qun231_varnell6078097.cl"},
	{0x026DBEF36DEC23BEULL, "284466219.ogg"},
	{0x026DC69A2616B85CULL, "6067663_m.fxe"},
	{0x026DED693E661AE6ULL, "qun120cr_javaris6042809.cl"},
	{0x026E1AE6C097778BULL, "hlac_ca_caverns2prc_5802_23z.dds"},
	{0x026E89833C8C5DA3ULL, "vxht_0_hightown_a3_avh_0_0.vlm"},
	{0x026EA4741E2C2761ULL, "vxht_keep_n_62s_0_0.vlm"},
	{0x026ECE146E6ED207ULL, "fen311ar_hanged_man.ncs"},
	{0x02703FAA7AEBEEA3ULL, "vxht_0_hightown_ccp_0_0.vlm"},
	{0x027055C7CC34D629ULL, "652132592.ogg"},
	{0x0270C7B9ADDB06EAULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\wander_right.ani"},
	{0x02714FA435EF58B4ULL, "6090247_m.fxe"},
	{0x0271E5B1A84745D1ULL, "vxlt_darktown_2u6_0_0.vlm"},
	{0x0271F2015B543116ULL, "vxlt_warehouse_v3_0_0.vlm"},
	{0x0272277EC0A3F129ULL, "vxdr_primevalentrance_1k5_0_0.vlm"},
	{0x027294DE91BC749CULL, "gen00fl_aveline6113990.cl"},
	{0x0272AB38769065F1ULL, "215467420.ogg"},
	{0x0273ABA495DBC0E2ULL, "6170639_m.fxe"},
	{0x0273C9A25DFE7C9FULL, "zz_vault_debug6203355.cl"},
	{0x0273DEC27C1EFB64ULL, "vxht_mansion_n_4ym_0_0.vlm"},
	{0x0273E49D2E460A14ULL, "hlac_lt_undercityencounter_-1_48z.dds"},
	{0x0274A076E86E0E2DULL, "6070882_m.fxe"},
	{0x0274AC6522D9DAC9ULL, "hlac_lt_undercitydungeon_-1_22x.dds"},
	{0x0274F11C6558C1ECULL, "lgt260cr_way_mage2.utc"},
	{0x0274F7511E0E6FCAULL, "6053011_m.fxe"},
	{0x027500347AE34E19ULL, "vxht_0_hightown_aib_0_0.vlm"},
	{0x0275181F42E682C3ULL, "324904485.ogg"},
	{0x02751D2C8B447C1EULL, "6028394_m.fxe"},
	{0x0275A626AF0C6EACULL, "hlac_lt_gallowstemplar_f_2383_8y.dds"},
	{0x0275AE146E74BAB6ULL, "fen311ar_hanged_man.lst"},
	{0x0275B9BC1B56FCB2ULL, "23518858.ogg"},
	{0x0275E59D9E3BA78FULL, "dae000_mother6121009.cl"},
	{0x02761B4308C67276ULL, "6135816_m.fxe"},
	{0x02764D0CE8B7A968ULL, "6120562_m.fxe"},
	{0x02768F474D28930AULL, "vxca_caverns2_78s_0_0.vlm"},
	{0x0276AB0A4CD0C18FULL, "68053612.ogg"},
	{0x0276ABA3C9107094ULL, "art\\characters\\playercharacter\\humanfemale\\animation\\exploration\\runf.evt"},
	{0x0276C541E8CD341BULL, "7123747.ogg"},
	{0x0276CBEA9AED5F4BULL, "vxlt_0_lowtown_docks_b5c_0_1.vlm"},
	{0x02782D29DF80D348ULL, "vxht_keep_n_62v_0_0.vlm"},
	{0x027912A2B1C1B1D9ULL, "6000980_m.fxe"},
	{0x02793056411C1423ULL, "mag130_danzig6016445.cl"},
	{0x027947B0745091D8ULL, "follower_banter6160978.cl"},
	{0x027A1FA4DF5EDB8DULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\tre_timberbeamxb.mmh"},
	{0x027A23A4DF5EE2DEULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\tre_timberbeamxb.mao"},
	{0x027A31A4DF5EFA13ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\tre_timberbeamxb.msh"},
	{0x027ABABC73AC794FULL, "333123730.ogg"},
	{0x027B02A41EAC4A5DULL, "isa311_castillon6094901.cl"},
	{0x027B55109D08D0D7ULL, "vxw_sundermount_35r_0_0.vlm"},
	{0x027B8BA965055653ULL, "gen00fl_aveline6128412.cl"},
	{0x027BEA96B5DF7979ULL, "art\\characters\\playercharacter\\humanfemale\\textures\\hf_hoo_robc_t.dds"},
	{0x027C8B1AE273381DULL, "355035268.ogg"},
	{0x027CFB602E827623ULL, "vxht_0_hightown_a3_ask_0_0.vlm"},
	{0x027D4655F65C3796ULL, "381827300.ogg"},
	{0x027D691B35BADDA3ULL, "lgt310_mag_melson6152119.cl"},
	{0x027D805C551D8E21ULL, "486914434.ogg"},
	{0x027D8B8AFA58576CULL, "hm_gen_templar_captain.mor"},
	{0x027E022F3EA45490ULL, "vxht_keep_n_73t_0_0.vlm"},
	{0x027E2C08B41071D2ULL, "llac_lt_0_lowtown_-1_5.dds"},
	{0x027E7D4037CD79B4ULL, "19407311.ogg"},
	{0x027EC06463B60E96ULL, "vxlt_0_lowtown_a2_n_eqf_0_0.vlm"},
	{0x027F073BCFA872C5ULL, "vxht_0_hightown_a89_0_0.vlm"},
	{0x027F4799EEFB4CDDULL, "art\\cinematics\\_crit path\\prologue\\pro000cs_flemeth_real\\cs_hur2_flemeth_fake.gad"},
	{0x027F525AAC233FE4ULL, "follower_banter6155901.cl"},
	{0x027FBD146E7D28A1ULL, "fen311ar_hanged_man.are"},
	{0x028052829CE9405EULL, "6128054_m.fxe"},
	{0x0280578E9A3D5E51ULL, "vxlt_darktown_2x5_0_0.vlm"},
	{0x0280A9DF5A67A19FULL, "vxw_encounter3_88_0_0.vlm"},
	{0x028127F2AE0BCA12ULL, "hlac_ca_cavern1_haunted_24421_0y.dds"},
	{0x02814BBEA0167552ULL, "6081849_m.fxe"},
	{0x0281570C79FD40BDULL, "393101068.ogg"},
	{0x02815C49515122C0ULL, "561160858.ogg"},
	{0x0281F84E911F847EULL, "one100dg_nabil6060508.cl"},
	{0x028246213483F73DULL, "6064681_m.fxe"},
	{0x028295C7C1C7D903ULL, "art\\characters\\weapons\\maces\\w_mce_mc105.mao"},
	{0x0282B1026295D5DDULL, "6103616_m.fxe"},
	{0x028318F739A8DDE7ULL, "gen00fl_anders6119190.cl"},
	{0x0283C24824A9EE93ULL, "vxdr_primevaltomb_50w_0_1.vlm"},
	{0x028403D91FDC1D77ULL, "lgt150_nat_were_supporter6110995.cl"},
	{0x02840BF9B2CC62C0ULL, "dae000ip_crowd_lm3_talk7.utp"},
	{0x028431D01D2A2FAEULL, "vxht_mansion2b_2ng_0_0.vlm"},
	{0x028488F452C7F3D9ULL, "171003944.ogg"},
	{0x0284EDEA00104D45ULL, "vrc221_varric6044049.cl"},
	{0x02851894803B1593ULL, "1001536621.ogg"},
	{0x028526D2629C0BFAULL, "vxlt_gallowstemplar_3gr_0_0.vlm"},
	{0x0285D0A0C2D7E1F8ULL, "6110657_m.fxe"},
	{0x02865593A6C1818EULL, "hlac_lt_0_lowtown_docks_-131075_7y.dds"},
	{0x02865E2FBC81FE89ULL, "llac_lt_0_lowtown_-65536_8.dds"},
	{0x02868F6403B0B67BULL, "hlac_lt_0_lowtown_a2_65535_0z.dds"},
	{0x02871D0C5F59DC6DULL, "hlac_lt_0_lowtown_a2_n_65535_5z.dds"},
	{0x02875097E3AD0251ULL, "6139396_m.fxe"},
	{0x0287EA650AADDB8BULL, "6117956_m.fxe"},
	{0x02881446918A4F93ULL, "vxca_cavern1_haunted_e64_0_0.vlm"},
	{0x0288DCD4E7B5908BULL, "art\\vfx\\combat\\vfx_gib_insect_bld_p.anb"},
	{0x0288FB2460E4DF09ULL, "75636167.ogg"},
	{0x0289081F20D67496ULL, "art\\characters\\creatures\\desiredemon\\c_desiredemon.msh"},
	{0x0289121F20D68594ULL, "art\\characters\\creatures\\desiredemon\\c_desiredemon.mmh"},
	{0x0289161F20D68CA7ULL, "art\\characters\\creatures\\desiredemon\\c_desiredemon.mao"},
	{0x0289C69925970997ULL, "vxlt_shop_f5_0_0.vlm"},
	{0x028B18A4DF6D434AULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\tre_timberbeamxb.phy"},
	{0x028B2817018DF415ULL, "hlac_lt_0_lowtown_docks_a3_n_-131075_1z.dds"},
	{0x028BFCE74A38EF16ULL, "237798594.ogg"},
	{0x028C2838DD7C3E01ULL, "6110523_f.fxe"},
	{0x028C486045D4F7A6ULL, "vxw_encounter3_7t_0_0.vlm"},
	{0x028CBF19C823A177ULL, "vxlt_0_lowtown_e2e_0_0.vlm"},
	{0x028D178474EF90CCULL, "6162436_m.fxe"},
	{0x028E4844E3B6A422ULL, "vxw_bonepitdestroyed_a3_260_0_0.vlm"},
	{0x028E9573104B5D44ULL, "vxht_0_hightown_apt_0_0.vlm"},
	{0x028ED2D9828BC92FULL, "20564381.ogg"},
	{0x028F9888DE575A14ULL, "328645088.ogg"},
	{0x02902EA9B3EAFC6EULL, "gen00fl_anders6121183.cl"},
	{0x02908FDB0897A939ULL, "hlac_lt_0_lowtown_n_131074_0y.dds"},
	{0x0290A5CF4D72AA83ULL, "vxht_0_hightown_n_avj_0_0.vlm"},
	{0x0290CBEDE72AB4EDULL, "vxlt_0_lowtown_a3_n_c_fhz_0_0.vlm"},
	{0x0290F45E66BED147ULL, "vxw_bonepitmining_a2_fk_0_0.vlm"},
	{0x02912BC480E8FD8BULL, "6131209_f.fxe"},
	{0x02915FD197BB9900ULL, "994519018.ogg"},
	{0x0291E2D7C1DF17E4ULL, "vxht_0_hightown_bx7_0_0.vlm"},
	{0x0291EF32F4D033D6ULL, "w_sundermount_3_0.msh"},
	{0x0291F932F4D044D4ULL, "w_sundermount_3_0.mmh"},
	{0x0291FD32F4D04BE7ULL, "w_sundermount_3_0.mao"},
	{0x029205075923471FULL, "6108509_m.fxe"},
	{0x02920D81EE939B44ULL, "lt_0_lowtown_docks_a3_20_0.rml"},
	{0x02922F76521EB8D8ULL, "vxlt_0_lowtown_f0w_0_0.vlm"},
	{0x02926986AAF3DF9DULL, "vxlt_gallowscourtyard_n_745_0_0.vlm"},
	{0x0292B2DAAE6B3493ULL, "919440501.ogg"},
	{0x0293B012A5B6FFFCULL, "vxlt_undercityserialkiller_1jb_0_1.vlm"},
	{0x0293C4971276B8E9ULL, "6118304_m.fxe"},
	{0x02943AE0A555C5DEULL, "579058972.ogg"},
	{0x02944A8CB229A022ULL, "vxlt_0_lowtown_a2_f52_0_0.vlm"},
	{0x029520FE17A4A2FCULL, "follower_banter6165366.cl"},
	{0x02957930E687D009ULL, "vxlt_0_lowtown_a2_hfz_0_0.vlm"},
	{0x0295918808DCFC26ULL, "pro000_merchant6051637.cl"},
	{0x0295E2393A16998CULL, "121507228.ogg"},
	{0x0296284E4444407DULL, "hlac_ht_keep_n_4519_0x.dds"},
	{0x02965E477E7B2CCFULL, "gen00fl_anders6119389.cl"},
	{0x02968A54F1891329ULL, "genip_trap_leghold.utp"},
	{0x0296B11A05029688ULL, "vxht_0_hightown_n_c41_0_0.vlm"},
	{0x029752EE71353978ULL, "ip_restrict_weapon_style.ncs"},
	{0x02980933716989DFULL, "6014873_m.fxe"},
	{0x02989F66BB7418CFULL, "lava_alpha.dds"},
	{0x0298CBCF93F42E1CULL, "vxca_caverns2prc_9u5_0_0.vlm"},
	{0x029903F1D2900E63ULL, "llac_ca_cavern1_haunted_24425_13.dds"},
	{0x0299152960DE80B0ULL, "109036135.ogg"},
	{0x029927306B611236ULL, "6044239_m.fxe"},
	{0x0299520594A2BD6CULL, "hlac_lt_0_lowtown_a2_n_c_-65537_5z.dds"},
	{0x029990E37071D58BULL, "vxht_chantry_n_2it_0_0.vlm"},
	{0x029A382F236533E6ULL, "6094726_m.fxe"},
	{0x029A572F0B92A813ULL, "vxlt_hangedman_n_27h_0_0.vlm"},
	{0x029A5D54C3643DB9ULL, "t3_wdg_w08.tnt"},
	{0x029A69A9F38BBC1FULL, "llac_dr_0_deeproads_5843_3.dds"},
	{0x029B413A054FDA12ULL, "505111542.ogg"},
	{0x029B6279246AC3F4ULL, "vxw_woundedcoast_122_0_0.vlm"},
	{0x029B9AD560F3802CULL, "art\\characters\\weapons\\longbows\\textures\\w_lbw_lb102a_0n.dds"},
	{0x029BCBB7BF898F57ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\0ff_22base.mmh"},
	{0x029BCFB7BF899664ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\0ff_22base.mao"},
	{0x029BDEBE23F09D14ULL, "6103114_m.fxe"},
	{0x029BE1B7BF89B4C9ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\0ff_22base.msh"},
	{0x029BF80EC3DA43CAULL, "vxw_encounter2_camp_np_0_0.vlm"},
	{0x029C5D4F8ACE9CD1ULL, "zz_ave_debug6174631.cl"},
	{0x029C5F88D3F51109ULL, "art\\levels\\4cinematics\\qunari\\_textures\\prp_qunariamulet_d.dds"},
	{0x029C726C3D0CAD9AULL, "genlock_emissary_def.utc"},
	{0x029CA635A90F0266ULL, "6105820_m.fxe"},
	{0x029CA8D18BACAEE2ULL, "6016158_f.fxe"},
	{0x029D10A5AE9D4244ULL, "vxca_caverns2_9fq_0_0.vlm"},
	{0x029D7E1890832C68ULL, "6078685_m.fxe"},
	{0x029D86201520919FULL, "hlac_ht_playershouse_n_855_26z.dds"},
	{0x029E3B5FC7A805A9ULL, "6206258_m.fxe"},
	{0x029ED15290881224ULL, "vxlt_warehouse_vq_0_0.vlm"},
	{0x029F1B86E8FC204FULL, "vxca_caverns2prc_a9v_0_0.vlm"},
	{0x029F7D576D1CC508ULL, "657507611.ogg"},
	{0x029F7E7288F2AAE4ULL, "hlac_lt_0_lowtown_docks_a3_n_-262144_3y.dds"},
	{0x029FDB70D2DF31D4ULL, "vxlt_backalley_g_1jw_0_0.vlm"},
	{0x02A00EA7E7A29926ULL, "follower_banter6160965.cl"},
	{0x02A06817953304B1ULL, "mag110_vincento6167359.cl"},
	{0x02A0D60ACC55E316ULL, "107044949.ogg"},
	{0x02A1158A011660AEULL, "vxht_mansion2b_2dy_0_0.vlm"},
	{0x02A1AC99751A82C5ULL, "vxlt_gallowstemplar_3jz_0_0.vlm"},
	{0x02A2327BDF3EA03DULL, "6116225_m.fxe"},
	{0x02A24D6B4DFFD896ULL, "hlac_ht_keep_3732_0z.dds"},
	{0x02A2917E696AA31BULL, "vxw_blightlands_ec_0_0.vlm"},
	{0x02A29A795D9C1DE6ULL, "art\\vfx\\combat\\vfx_f_and_fury_c.impa.evt"},
	{0x02A2CDB5D7B84DDCULL, "6125705_m.fxe"},
	{0x02A3162BC37D2441ULL, "dae000_mother6004193.cl"},
	{0x02A356A751197BA4ULL, "w_bonepitmining_a2_33_0.msh"},
	{0x02A35E2A8025D3F5ULL, "dae000_junar6147868.cl"},
	{0x02A368A751199A0DULL, "w_bonepitmining_a2_33_0.mao"},
	{0x02A36BA77502FEE6ULL, "6173779_m.fxe"},
	{0x02A36CA75119A146ULL, "w_bonepitmining_a2_33_0.mmh"},
	{0x02A38A746D2D08F5ULL, "6129029_m.fxe"},
	{0x02A3EB0DD87DE889ULL, "vxlt_0_lowtown_a2_hcy_0_0.vlm"},
	{0x02A4B96086580F00ULL, "vxlt_0_lowtown_a3_n_c_hpe_0_0.vlm"},
	{0x02A5456912F3BA69ULL, "rog_combo.dds"},
	{0x02A605DDFC4BEB59ULL, "hlac_lt_0_lowtown_a2_n_65536_3y.dds"},
	{0x02A67C4789B7AAA7ULL, "vxdr_0_deeproads_9ci_0_0.vlm"},
	{0x02A693299BCAE066ULL, "726154384.ogg"},
	{0x02A6ED64987AAF95ULL, "ser221_quentin6128963.cl"},
	{0x02A802C06267EE0BULL, "vxht_chantry_n_2dw_0_0.vlm"},
	{0x02A82ECBB8DCA85DULL, "vxca_caverns1_gje_0_0.vlm"},
	{0x02A8ECA3026133B1ULL, "6092276_f.fxe"},
	{0x02A8F7A8CAB05D6BULL, "vxw_encounter2_65_0_0.vlm"},
	{0x02A91A866AAE0460ULL, "851993401.ogg"},
	{0x02A9381821D4B551ULL, "art\\characters\\creatures\\mubari\\animation\\blendtree\\mabari_ambient.anb"},
	{0x02A99CF73CDAD747ULL, "art\\levels\\races\\ferelden\\props\\_textures\\plc_hammer_d.dds"},
	{0x02A9B1795DA26331ULL, "art\\vfx\\combat\\vfx_f_and_fury_c.impa.gad"},
	{0x02AA509B15E6B594ULL, "6104014_m.fxe"},
	{0x02AA5C10D22EC4C9ULL, "208473144.ogg"},
	{0x02AA607DA483879FULL, "vxlt_0_lowtown_a2_er9_0_0.vlm"},
	{0x02AA69EBB5D05C4AULL, "vxw_encounter2_camp_iq_0_0.vlm"},
	{0x02AB22B5BC4EE34BULL, "201096058.ogg"},
	{0x02AB460EC832627BULL, "gen00fl_anders6196825.cl"},
	{0x02AB485921239CC2ULL, "mcr370_assignments6081743.cl"},
	{0x02AB926B64C44145ULL, "6129048_f.fxe"},
	{0x02AC1A761BE06725ULL, "6128807_f.fxe"},
	{0x02ACEE1D27DB45A1ULL, "gen00fl_aveline6197229.cl"},
	{0x02AD2C2979BF7CBCULL, "art\\characters\\playercharacter\\dwarfmale\\textures\\w_var\\bianca_n.dds"},
	{0x02AD5AB299F451ECULL, "412519080.ogg"},
	{0x02AD96EBB42CF207ULL, "mag360_alain6044627.cl"},
	{0x02ADABA98E5340F4ULL, "483920033.ogg"},
	{0x02ADD8184AAB0247ULL, "vxw_bonepitmining_a2_he_0_0.vlm"},
	{0x02AE21AE22BCD664ULL, "8107244.ogg"},
	{0x02AE4D4DC4D54A54ULL, "vxlt_backalley_g_1wt_0_0.vlm"},
	{0x02AEE8C13D0F781FULL, "6106709_m.fxe"},
	{0x02AF5CF4A4286809ULL, "1007223374.ogg"},
	{0x02AF68C24650CF63ULL, "6149124_m.fxe"},
	{0x02B01EC7AEDDCEA5ULL, "atl_background_dxt5_00.dds"},
	{0x02B08D4D45F4083FULL, "6072288_m.fxe"},
	{0x02B0A850F662E9E9ULL, "6112104_m.fxe"},
	{0x02B0B2795DA883DFULL, "art\\vfx\\combat\\vfx_f_and_fury_c.impa.ani"},
	{0x02B1035B5B60BB9BULL, "vxw_blightlands_nb_0_0.vlm"},
	{0x02B13BC190343EDDULL, "vxw_blightlands_121_0_0.vlm"},
	{0x02B141F28DB271A7ULL, "mag170_recruit016094519.cl"},
	{0x02B1478052FA1823ULL, "and100_anders6095865.cl"},
	{0x02B14FF5FB25CC5CULL, "vxca_cavern1_haunted_hm3_0_0.vlm"},
	{0x02B15F866F9EFD36ULL, "442642754.ogg"},
	{0x02B1C2E7E3B5876DULL, "hlac_lt_0_lowtown_n_-131072_8x.dds"},
	{0x02B1ED34CAE92769ULL, "6146784.cut"},
	{0x02B1F35C610AB1C8ULL, "6166282_f.fxe"},
	{0x02B21EE3999357BFULL, "vxht_mansion2_28p_0_0.vlm"},
	{0x02B28C3469ECC7F7ULL, "follower_banter6151272.cl"},
	{0x02B2AF732AB13CC0ULL, "6115782_m.fxe"},
	{0x02B34CFAF091BBF3ULL, "art\\characters\\playercharacter\\humanfemale\\textures\\hf_hoo_lgtb_s.dds"},
	{0x02B39894AA042B7CULL, "art\\characters\\playercharacter\\humanfemale\\textures\\hf_arm_urva_0s.dds"},
	{0x02B4280E603BADD3ULL, "hlac_lt_gallowstemplar_2394_3x.dds"},
	{0x02B45C821AB9CA17ULL, "hlac_lt_0_lowtown_-65535_7z.dds"},
	{0x02B46CA75128140DULL, "w_bonepitmining_a2_33_0.rml"},
	{0x02B4EE247BADC327ULL, "vxdr_0_deeproads_9dh_0_0.vlm"},
	{0x02B514BE0BC46318ULL, "6136305_m.fxe"},
	{0x02B5207ABAC23960ULL, "6147198_m.fxe"},
	{0x02B5D6BE406DE493ULL, "6152313_m.fxe"},
	{0x02B5E04E3AB6E654ULL, "nat320_delilah6065917.cl"},
	{0x02B5FEFE8FF0DDA6ULL, "vxlt_0_lowtown_a2_ews_0_0.vlm"},
	{0x02B6085E46C0EAA4ULL, "762393266.ogg"},
	{0x02B614AEA820FBD7ULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_jaw03b.phy"},
	{0x02B651CAF5C50032ULL, "6089947_f.fxe"},
	{0x02B6621EB8A102FEULL, "86297039.ogg"},
	{0x02B6D0E7ADECFE5DULL, "art\\vfx\\environment\\level_fx\\fire\\vfx_firexxxl.mmh"},
	{0x02B6F07C89C072F5ULL, "vxw_sundermount_2ks_0_0.vlm"},
	{0x02B714B4EF0770ADULL, "6033692_m.fxe"},
	{0x02B76985BCA675EBULL, "vxw_encounter2_14_0_0.vlm"},
	{0x02B846330856F4F4ULL, "vxw_woundedcoast_180_0_0.vlm"},
	{0x02B8A6554ABD1C86ULL, "472143021.ogg"},
	{0x02B8A6A7D9AF8962ULL, "6139070_m.fxe"},
	{0x02B8BD69756A0C4DULL, "6140864_m.fxe"},
	{0x02B8D25A9679A01FULL, "vxlt_0_lowtown_a2_eu8_0_0.vlm"},
	{0x02B8D77F9DE18DC9ULL, "w_blightlands_3_0.rml"},
	{0x02B9D3C983A06B52ULL, "qun220cr_javaris6071778.cl"},
	{0x02B9E1253ABBC1D3ULL, "264513211.ogg"},
	{0x02B9F8E0384BBCCCULL, "6166611_f.fxe"},
	{0x02BA0CBC6632CD84ULL, "hlac_w_sundermount_196604_0y.dds"},
	{0x02BA7FA59BA51C84ULL, "zz_delete_me.cut"},
	{0x02BCEA6D563A19AAULL, "598101857.ogg"},
	{0x02BD5A51A477F7DEULL, "art\\vfx\\combat\\vfx_r_gn2_speed_bl_loop_c.impa.ani"},
	{0x02BD6EE6CBC01AC8ULL, "ts_slavers_prog_junk1.gda"},
	{0x02BD83A0A2FA3F28ULL, "355951251.ogg"},
	{0x02BE0D2E349CED62ULL, "vxht_chantry_n_20f_0_0.vlm"},
	{0x02BE1C600D73F9F1ULL, "hlac_lt_0_lowtown_196606_0x.dds"},
	{0x02BE444546437E07ULL, "335400291.ogg"},
	{0x02BE7E34FF8F9A60ULL, "zz_lgt_2_debug6211863.cl"},
	{0x02BEEE462A746DFAULL, "vxlt_gallowstemplar_3sn_0_0.vlm"},
	{0x02BF2DD73FAA79D5ULL, "vxlt_darktown_5q1_0_0.vlm"},
	{0x02BF5889B30C8288ULL, "dae000_arishok6067764.cl"},
	{0x02BF75384D56D41BULL, "vxw_blightlands_se_0_0.vlm"},
	{0x02BF9B49A87FC5D0ULL, "hlac_lt_0_lowtown_a2_n_-65537_1x.dds"},
	{0x02BF9CABB8ABBDDBULL, "vxw_encounter1_fd_0_0.vlm"},
	{0x02C0254AA047CF83ULL, "311608505.ogg"},
	{0x02C049AD7F364732ULL, "6110407_f.fxe"},
	{0x02C0F742544BCC94ULL, "555452862.ogg"},
	{0x02C121501CA75540ULL, "6110882_m.fxe"},
	{0x02C1BEF522F9A6F3ULL, "qun210cr_guard6087765.cl"},
	{0x02C1E77CD398204DULL, "art\\vfx\\models\\vfx_debris_02.mmh"},
	{0x02C1EB7CD398269EULL, "art\\vfx\\models\\vfx_debris_02.mao"},
	{0x02C1F97CD3983ED3ULL, "art\\vfx\\models\\vfx_debris_02.msh"},
	{0x02C224D8FEC11B27ULL, "vxlt_0_lowtown_n_feh_0_0.vlm"},
	{0x02C2530E3795C55AULL, "6169163_m.fxe"},
	{0x02C27228D8011F84ULL, "llac_lt_0_lowtown_docks_n_65533_3.dds"},
	{0x02C2783B4CC922E4ULL, "hlac_lt_0_lowtown_docks_a3_-131073_3z.dds"},
	{0x02C28C07BE7BB920ULL, "llac_lt_gallowstemplar_n_2391_4.dds"},
	{0x02C2CB7F9DE9CDA9ULL, "w_blightlands_3_0.mao"},
	{0x02C2D77F9DE9E242ULL, "w_blightlands_3_0.mmh"},
	{0x02C2D97F9DE9E5F8ULL, "w_blightlands_3_0.msh"},
	{0x02C3397DD5291221ULL, "6033128_m.fxe"},
	{0x02C3A0B89CE4DE41ULL, "fen210_dying_magister6108609.cl"},
	{0x02C3DBBCD07F7A94ULL, "6176006_m.fxe"},
	{0x02C44D6AA39E1869ULL, "1034795020.ogg"},
	{0x02C48D7D6155CA57ULL, "character_character_p127.psh.fxo.dx11.assembly"},
	{0x02C4C43D0E46E0ADULL, "lgt360_mag_orlanna6156289.cl"},
	{0x02C4C62F3D179B62ULL, "zz_store_dae110cr_store5.utm"},
	{0x02C51BF59D64BCE0ULL, "311737374.ogg"},
	{0x02C562597BB68B75ULL, "vxw_sundermount_2lr_0_0.vlm"},
	{0x02C596208828D545ULL, "hm_raider_2.mor"},
	{0x02C5C2880AC7C6B0ULL, "vxlt_0_lowtown_a2_n_c_hxl_0_0.vlm"},
	{0x02C63C032A18E496ULL, "591695867.ogg"},
	{0x02C68BEE51D1C443ULL, "6133741_m.fxe"},
	{0x02C6D1E1253DCEA5ULL, "qun231_varnell6099592.cl"},
	{0x02C7329FDAA2C9AAULL, "ave200_donnic6111052.cl"},
	{0x02C758157BAA17AAULL, "isa211st_walleyedsam.stg"},
	{0x02C762064FAABBBDULL, "vxdr_primevaltomb_7sw_0_0.vlm"},
	{0x02C7A68F2CF68BB1ULL, "286692659.ogg"},
	{0x02C7BE023347C5CDULL, "6054079_m.fxe"},
	{0x02C7D0CC3133AEC4ULL, "6170237_m.fxe"},
	{0x02C8598D0568C1C7ULL, "169017157.ogg"},
	{0x02C88F32A7CC012DULL, "6144936_m.fxe"},
	{0x02C91CD914F62503ULL, "plt_cod_mgc_chantry_founded.plo"},
	{0x02C9CBC35E3BAE9FULL, "6087122_f.fxe"},
	{0x02C9F2EDB73ABAE5ULL, "hlac_w_bonepit_0_3x.dds"},
	{0x02CA09546AD37403ULL, "w_woundedcoast_61_0.rml"},
	{0x02CA7251A482AAAFULL, "art\\vfx\\combat\\vfx_r_gn2_speed_bl_loop_c.impa.evt"},
	{0x02CB1D3F0CE1163CULL, "6201514_f.fxe"},
	{0x02CB2779894D9787ULL, "mag101_grace6053090.cl"},
	{0x02CB5D21F379ED3BULL, "dae000_junar6147853.cl"},
	{0x02CB60B7721F45BCULL, "plt_cod_cha_dao_hero.plo"},
	{0x02CC15F61E8D74BFULL, "229017019.ogg"},
	{0x02CC54AD4FEB766AULL, "968832196.ogg"},
	{0x02CD4BB7C293FECCULL, "6177841_m.fxe"},
	{0x02CDE2F4045461A7ULL, "follower_banter6168027.cl"},
	{0x02CDE7153F4CEC9BULL, "vxw_blightlands_td_0_0.vlm"},
	{0x02CE0C14FEC453DBULL, "w_sundermount_220_0.mmh"},
	{0x02CE0E88AAA1D65BULL, "vxw_encounter1_kg_0_0.vlm"},
	{0x02CE1814FEC46878ULL, "w_sundermount_220_0.mao"},
	{0x02CE2A14FEC486C5ULL, "w_sundermount_220_0.msh"},
	{0x02CE4DE8554EF6E6ULL, "hlac_lt_0_lowtown_n_65536_10x.dds"},
	{0x02CF0955518EAC9BULL, "vxw_sundermount_2gy_0_0.vlm"},
	{0x02CF3FE94F7CE3E3ULL, "gen00fl_anders6125441.cl"},
	{0x02CF56C82A2B1B8BULL, "rob_pl4.tnt"},
	{0x02CF5F617BCE2CA9ULL, "vxlt_warehouse_n_12z_0_0.vlm"},
	{0x02CFA838CD829A69ULL, "hlac_ht_keep_n_3732_12y.dds"},
	{0x02CFF80CBCCFDA47ULL, "vxw_encounter2_q1_0_1.vlm"},
	{0x02D05B7D93A94FDFULL, "zz_personality_debug6004631.cl"},
	{0x02D05C5D2F37A61EULL, "one280_merin6116761.cl"},
	{0x02D0961EAE5B0FC3ULL, "726448729.ogg"},
	{0x02D0ADFA163553DDULL, "77655423.ogg"},
	{0x02D0BB14DB37A956ULL, "6073875_m.fxe"},
	{0x02D17351A488CB58ULL, "art\\vfx\\combat\\vfx_r_gn2_speed_bl_loop_c.impa.gad"},
	{0x02D2DB0E2902C694ULL, "abi_m_haste.bnk"},
	{0x02D38E7F6C8BDA12ULL, "vxlt_0_lowtown_docks_a3_n_d0v_0_0.vlm"},
	{0x02D39E26AFE0AF81ULL, "414171283.ogg"},
	{0x02D3BA05A29FCD81ULL, "vxca_cavern1_haunted_hmb_0_0.vlm"},
	{0x02D3C1E017D80F8AULL, "gen00fl_anders6119221.cl"},
	{0x02D4068E5C93A810ULL, "6175098_m.fxe"},
	{0x02D43464FCBDDF30ULL, "vxlt_0_lowtown_a2_n_c_hsm_0_0.vlm"},
	{0x02D4B3FB8DEDA2A1ULL, "6144284_m.fxe"},
	{0x02D526BB986A6050ULL, "art\\levels\\races\\proxy\\props\\4tools\\vis_blast15_8m_tube.msh"},
	{0x02D534BB986A786AULL, "art\\levels\\races\\proxy\\props\\4tools\\vis_blast15_8m_tube.mmh"},
	{0x02D538BB986A7EA1ULL, "art\\levels\\races\\proxy\\props\\4tools\\vis_blast15_8m_tube.mao"},
	{0x02D56B1989E18FC6ULL, "745940270.ogg"},
	{0x02D57DBA372C1645ULL, "620642287.ogg"},
	{0x02D61405551FB652ULL, "vxca_cavern1_haunted_jdh_0_0.vlm"},
	{0x02D6B2E9293FE213ULL, "gen00fl_merrill6106435.cl"},
	{0x02D6CEC3CA19D7C4ULL, "vxht_0_hightown_aoy_0_0.vlm"},
	{0x02D843C8EB36CB9DULL, "hlac_ht_chantry_n_2534_34x.dds"},
	{0x02D8C7AC1C2D1C77ULL, "6150180_m.fxe"},
	{0x02D8F1DFCDBEB7A0ULL, "6207968_m.fxe"},
	{0x02D91E3AE93C2AC6ULL, "vxca_caverns1_mines_drd_0_0.vlm"},
	{0x02DAC481D39B78FDULL, "zz_mrl_debug6185298.cl"},
	{0x02DAF11F57A0F102ULL, "zz_start_dre6080515.cl"},
	{0x02DBCAB9933BBFF6ULL, "vxca_caverns2prc_90a_0_1.vlm"},
	{0x02DC060BEBF2B13BULL, "mer210_seldrek6033962.cl"},
	{0x02DC1EC9D7478CCDULL, "zz_dae_debug6029111.cl"},
	{0x02DC34B32A3FAED1ULL, "art\\characters\\weapons\\maces\\textures\\w_mce_mc102a_0s.dds"},
	{0x02DCC72D99D76EA2ULL, "6077949_m.fxe"},
	{0x02DCF7BFBF67B393ULL, "gen00fl_anders6125931.cl"},
	{0x02DD41CC9C9F91FBULL, "dae000_arishok6067800.cl"},
	{0x02DD7ADDF4841156ULL, "llac_lt_backalley_g_65536_3.dds"},
	{0x02DD7B324384C51BULL, "vxw_sundermount_2hx_0_0.vlm"},
	{0x02DD83C973DA8490ULL, "499737896.ogg"},
	{0x02DDB92B81CFA9CCULL, "art\\characters\\playercharacter\\humanfemale\\animation\\exploration\\movement\\walkf_injured.ani"},
	{0x02DDD3BCECFCC213ULL, "6185201_m.fxe"},
	{0x02DE80D44E307100ULL, "vxlt_0_lowtown_a3_n_c_hla_0_0.vlm"},
	{0x02DE989E53930909ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\grovel_floor_exit.ani"},
	{0x02DF36C81B81F65AULL, "6163363_m.fxe"},
	{0x02DF6E3463E69F45ULL, "hlac_w_woundedcoast_-262143_0y.dds"},
	{0x02E0CA51FBAA3823ULL, "vxht_keep_5j5_0_0.vlm"},
	{0x02E1230644666A13ULL, "vxht_bartrand_n_4sg_0_0.vlm"},
	{0x02E133C7C01AD66FULL, "mag371_elsa6041246.cl"},
	{0x02E1AADF7717BB75ULL, "6167236_m.fxe"},
	{0x02E1C110A3A14BDDULL, "6035184_m.fxe"},
	{0x02E1CF49353BBB28ULL, "291896924.ogg"},
	{0x02E20A797B00059AULL, "655279093.ogg"},
	{0x02E246135FA2BC75ULL, "vxw_sundermount_2zp_0_0.vlm"},
	{0x02E2F478F682B43EULL, "dae200cr_viscount6087392.cl"},
	{0x02E32AEA06F6CEFAULL, "w_encounter3_120_0.mmh"},
	{0x02E32EEA06F6D531ULL, "w_encounter3_120_0.mao"},
	{0x02E33CEA06F6ED40ULL, "w_encounter3_120_0.msh"},
	{0x02E3D11C5BDA5D21ULL, "genip_trap_firecone.utp"},
	{0x02E47F88980FB5D7ULL, "vxlt_darktown_2qd_0_0.vlm"},
	{0x02E4989754E99218ULL, "pro000_athenril6048726.cl"},
	{0x02E4C62B81D5DE0EULL, "art\\characters\\playercharacter\\humanfemale\\animation\\exploration\\movement\\walkf_injured.gad"},
	{0x02E5AEE9FB8CFAACULL, "6097840_m.fxe"},
	{0x02E5B39E5399558BULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\grovel_floor_exit.gad"},
	{0x02E5D4C7F1954B2CULL, "vxlt_0_lowtown_docks_n_b0q_0_0.vlm"},
	{0x02E621F490122EA8ULL, "hlac_ca_caverns1_mines_24425_8y.dds"},
	{0x02E63008E704FB29ULL, "llac_w_encounter3_-65538_0.dds"},
	{0x02E66AEFE92109A8ULL, "6068486_m.fxe"},
	{0x02E6BC0BEDC7FC7DULL, "art\\characters\\playercharacter\\humanfemale\\mt_cbt_icon.phy"},
	{0x02E6CC23C3DD381DULL, "19595233.ogg"},
	{0x02E6D3AB5DBAB85BULL, "mcr370_cullen6087462.cl"},
	{0x02E6F932F51884D7ULL, "w_sundermount_3_0.rml"},
	{0x02E739890E2334F7ULL, "6157080_m.fxe"},
	{0x02E74671EDB80317ULL, "538180206.ogg"},
	{0x02E76909A8B1074AULL, "912865520.ogg"},
	{0x02E78E24F5176A6AULL, "gen_im_trs_drk_amu_r1.uti"},
	{0x02E791406EAEB2CCULL, "284234708.ogg"},
	{0x02E7D6A250EE8847ULL, "6179497_m.fxe"},
	{0x02E7E7C6509906A3ULL, "pendant_morningfrost.dds"},
	{0x02E8B6759F3EAD5DULL, "vxlt_0_lowtown_docks_a3_d1i_0_0.vlm"},
	{0x02E8C4B4E05215F1ULL, "hlac_dr_0_deeproads_5839_0z.dds"},
	{0x02E8FAE6D3052EB4ULL, "dae000_gen_turn_in_26198699.cl"},
	{0x02E95581991DB24DULL, "qun220cr_javaris6071520.cl"},
	{0x02E957C2D80EE845ULL, "323500196.ogg"},
	{0x02E971C4CBAA6EBBULL, "zz_one_debug6113373.cl"},
	{0x02E9CAE7A34799EBULL, "gen00fl_aveline6114020.cl"},
	{0x02E9E5EA2EE8FD45ULL, "vxlt_gallowstemplar_3se_0_0.vlm"},
	{0x02EA4C4E963D7FC2ULL, "693270863.ogg"},
	{0x02EA8807530DDB1BULL, "119102636.ogg"},
	{0x02EAB952F3BD25CFULL, "lgt200_eau_tome6171791.cl"},
	{0x02EB2952992C40F9ULL, "zz_followers_banter6158679.cl"},
	{0x02EBAD96831A6475ULL, "hlac_ca_caverns2prc_5802_65z.dds"},
	{0x02EBE972259DE9C7ULL, "hlac_lt_0_lowtown_65537_5y.dds"},
	{0x02EC5C2AA1E427C5ULL, "lgt110_gxa_vgk_silver_order6117024.cl"},
	{0x02ECA86698F6C10BULL, "vxlt_0_lowtown_a2_n_c_er8_0_0.vlm"},
	{0x02ECB09E539F6FF0ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\grovel_floor_exit.evt"},
	{0x02ED76B8B4F4F8BFULL, "514394588.ogg"},
	{0x02EE4217C30797D6ULL, "vxlt_gallowscourtyard_7f3_0_0.vlm"},
	{0x02EEDBD2E05982A6ULL, "6179333_m.fxe"},
	{0x02EFBEABF95F69DEULL, "gen00fl_varric6080086.cl"},
	{0x02EFE19662989EFDULL, "6175529_m.fxe"},
	{0x02F0C7FB82337724ULL, "and321tr_act3_opening.utt"},
	{0x02F1ACC47104D346ULL, "hlac_ht_chantry_n_2534_28z.dds"},
	{0x02F1B632BC04AA79ULL, "hlac_w_bonepit_196614_0z.dds"},
	{0x02F1BAD6CD2254BEULL, "422967507.ogg"},
	{0x02F221542D56D548ULL, "qcr220_grey_wardens6135912.cl"},
	{0x02F28B36020E4708ULL, "828167187.ogg"},
	{0x02F3385F579B46E8ULL, "6122854_m.fxe"},
	{0x02F3446E91885C72ULL, "llac_w_bonepit_131076_0.dds"},
	{0x02F35DFF22829048ULL, "hlac_lt_0_lowtown_a3_n_c_-65535_4z.dds"},
	{0x02F3695C85A0F12AULL, "vxht_0_hightown_a3_b8l_0_0.vlm"},
	{0x02F3AEDA95ADF0FCULL, "vxht_keep_n_36w_0_0.vlm"},
	{0x02F408D6D73B4199ULL, "207475233.ogg"},
	{0x02F4A1A215A57E9FULL, "611107694.ogg"},
	{0x02F4ED2FDE9EF855ULL, "638217596.ogg"},
	{0x02F5107B11685DEEULL, "dae000ip_crowd_ff7_talk13.utp"},
	{0x02F53E944F1CF485ULL, "hf_lilley.mor"},
	{0x02F608943A6335B3ULL, "art\\characters\\playercharacter\\humanmale\\textures\\hm_hlf_medh_0d.dds"},
	{0x02F61C06F7F98678ULL, "6111190_f.fxe"},
	{0x02F66E609D724B62ULL, "6118589_m.fxe"},
	{0x02F67E631CF5FF01ULL, "art\\vfx\\combat\\material__43.mao"},
	{0x02F6C8E7B055EF43ULL, "art\\vfx\\textures\\vfx_twirl01.dds"},
	{0x02F7BD0BEDD67082ULL, "art\\characters\\playercharacter\\humanfemale\\mt_cbt_icon.mmh"},
	{0x02F7BF0BEDD67438ULL, "art\\characters\\playercharacter\\humanfemale\\mt_cbt_icon.msh"},
	{0x02F8A7BC088CCC24ULL, "vxlt_blackemporium_15q_0_0.vlm"},
	{0x02F9088CDF27858FULL, "gen00fl_varric6118735.cl"},
	{0x02F99C4C0BFD0718ULL, "vxht_keep_61t_0_0.vlm"},
	{0x02FA5668ECBA77E4ULL, "703619480.ogg"},
	{0x02FA5EEC2770F61BULL, "vxw_sundermount_2vz_0_0.vlm"},
	{0x02FA7E655FF8CBD3ULL, "w_bonepitmining_a2_198_0.phy"},
	{0x02FA917C48485D7BULL, "dae000_quintus6154940.cl"},
	{0x02FAFB082202E820ULL, "lowtown_male_urchin_sta6164213.cl"},
	{0x02FB05ACC0E257ECULL, "zz_trans.ncc"},
	{0x02FB47AE51D0FD21ULL, "6164462_m.fxe"},
	{0x02FCECA9429C48C1ULL, "vxht_0_hightown_a3_av2_0_0.vlm"},
	{0x02FCF7546AFEAF1AULL, "w_woundedcoast_61_0.msh"},
	{0x02FD05546AFEC753ULL, "w_woundedcoast_61_0.mao"},
	{0x02FD09546AFECE60ULL, "w_woundedcoast_61_0.mmh"},
	{0x02FD2B01ECE5A6CAULL, "dae000_hate_follower_react6103217.cl"},
	{0x02FD8C54754D1CF6ULL, "art\\characters\\playercharacter\\shared\\heads\\textures\\mt_esm_nrmi_0n.dds"},
	{0x02FDF5AEA85E7CCCULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_jaw03b.mmh"},
	{0x02FDFBAEA85E86BEULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_jaw03b.msh"},
	{0x02FE384F49BF5826ULL, "vxlt_0_lowtown_a2_enx_0_0.vlm"},
	{0x02FEB0D23DBB2D14ULL, "hlac_lt_gallowstemplar_f_2391_5y.dds"},
	{0x02FECD427A70D93BULL, "dae000ip_crowd_mf6_talk14.utp"},
	{0x02FF4C460013343FULL, "art\\characters\\playercharacter\\humanmale\\textures\\hm_arm_nud_0s.dds"},
	{0x02FF70A3DE753FB7ULL, "1025070583.ogg"},
	{0x02FF7939266DE08BULL, "6153664_m.fxe"},
	{0x02FFAB210D5FA98CULL, "642661986.ogg"},
	{0x02FFEDA8751E276DULL, "vxlt_hangedman_n_1xl_0_0.vlm"},
	{0x03004868E7B1CED3ULL, "769156700.ogg"},
	{0x0300B24BEE88E3A3ULL, "vxht_0_hightown_c5z_0_0.vlm"},
	{0x0300DE3DE112AA54ULL, "vxw_bonepitmining_a2_27k_0_0.vlm"},
	{0x03017F33D00557FAULL, "vfx0_cs_addv.vsh.fxo"},
	{0x0301824AA96F4FDBULL, "lt_0_lowtown_a3_n_c_20_0_lc.mmh"},
	{0x030197FA354D9EBBULL, "6066348_m.fxe"},
	{0x0301D2D98EA5E84BULL, "vxw_woundedcoast_hk_0_1.vlm"},
	{0x0301DB39779709AAULL, "vxht_0_hightown_a3_b7m_0_0.vlm"},
	{0x0302E0096BBDDC4AULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\3ww_basic.mao"},
	{0x0302E4096BBDE299ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\3ww_basic.mmh"},
	{0x0302EE096BBDF387ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\3ww_basic.msh"},
	{0x0303377CF222F66BULL, "6177882_m.fxe"},
	{0x0303A7B3173C40FFULL, "6169171_m.fxe"},
	{0x0303CB1BFF0A12D4ULL, "vxw_bonepitmining_a2_1v3_0_0.vlm"},
	{0x030454E5AABD7BBEULL, "vxlt_backalley_g_1og_0_1.vlm"},
	{0x030461A35AE9B9D7ULL, "gen00fl_fenris6128057.cl"},
	{0x03047D52B352F746ULL, "ht_0_hightown_mmap.dds"},
	{0x0304BA5C34DAB947ULL, "6177697_m.fxe"},
	{0x0304DE64991FB543ULL, "ts_dwarven_prog_junk5.gda"},
	{0x0305301E7B5CB673ULL, "6094486_m.fxe"},
	{0x0305350BAE3F6407ULL, "6087042_m.fxe"},
	{0x03055213766524BBULL, "vxlt_0_lowtown_h95_0_0.vlm"},
	{0x0305AA6EFE110AD5ULL, "vxw_woundedcoast_11z_0_0.vlm"},
	{0x0306ABAF9CE27215ULL, "gen_im_trs_slv_hvy_hlm_r2.uti"},
	{0x0306FDE8DEC83250ULL, "vxlt_gallowstemplar_f_5ao_0_0.vlm"},
	{0x030782584DEF0E97ULL, "vxdr_primevaltomb_56h_0_0.vlm"},
	{0x030897F6522FC6B7ULL, "6094679_m.fxe"},
	{0x0308E899BDD04146ULL, "cr_spe_01.mao"},
	{0x0309680416C611ECULL, "vxht_mansion2_2mt_0_0.vlm"},
	{0x030987F88BEF7627ULL, "vxht_keep_3yf_0_0.vlm"},
	{0x0309A9CED5BFA168ULL, "llac_ht_mansion2b_818_55.dds"},
	{0x0309AD8CB0E26842ULL, "vxw_woundedcoast_36_0_0.vlm"},
	{0x0309B2EB65DC5221ULL, "hlac_lt_0_lowtown_a2_-262144_0z.dds"},
	{0x0309FEF5ED2770B8ULL, "691921873.ogg"},
	{0x030A708DFC14A671ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\ambient\\compressed\\sitting_bench_talk_1.ani"},
	{0x030A747007754EE2ULL, "ser121_jethann6104565.cl"},
	{0x030B22C0B81340A1ULL, "ss_rogue3.cut"},
	{0x030B7A846F1DD2DCULL, "vxlt_0_lowtown_a3_n_c_hp9_0_0.vlm"},
	{0x030BACB6DADA7DB5ULL, "2106169.ogg"},
	{0x030BCB67F02A41B0ULL, "vxlt_warehouse_137_0_0.vlm"},
	{0x030BDC74D2B2367CULL, "ico_1h_axe_5.dds"},
	{0x030C037DC784B862ULL, "follower_barks6153854.cl"},
	{0x030C116CF7603BFFULL, "tbe200im_ponderer_helm.uti"},
	{0x030C14CB1C17AD67ULL, "vxlt_warehouse_109_0_0.vlm"},
	{0x030C36C986E731EAULL, "vxw_bonepit_268_0_0.vlm"},
	{0x030D3BD6CBB7C200ULL, "165630047.ogg"},
	{0x030D5DD4DE152B7CULL, "775278411.ogg"},
	{0x030E993D0311FE39ULL, "436089737.ogg"},
	{0x030F4DCAF236BFF7ULL, "vxca_caverns2_8d0_0_1.vlm"},
	{0x030F64F855DCE8F9ULL, "vxlt_darktown_4p3_0_0.vlm"},
	{0x030FD09D381F8C4EULL, "6102549_f.fxe"},
	{0x031017891A6548E5ULL, "6164435_m.fxe"},
	{0x03104161F063C3BDULL, "hlac_ht_mansion_n_3370_0z.dds"},
	{0x03105CAE3F4B1B25ULL, "180853079.ogg"},
	{0x0310F8B7B2F3F32DULL, "prty_carver_armor_1.dds"},
	{0x03117B8DFC1AD713ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\ambient\\compressed\\sitting_bench_talk_1.gad"},
	{0x0311915F1F8301DEULL, "vxlt_warehouse_eb_0_0.vlm"},
	{0x0311AB3AFF67FFF0ULL, "961115753.ogg"},
	{0x0311CE9F238E7E75ULL, "art\\levels\\races\\dwarf\\common\\int\\_textures\\til_floortile_s.dds"},
	{0x0311D0AB6D73C144ULL, "qun111_ketojan6041043.cl"},
	{0x0311D55E70474C02ULL, "art\\vfx\\combat\\vfx_glb_ele_n_additive_c.impa.gad"},
	{0x03123CF8F1002B54ULL, "vxw_bonepitmining_a2_1s0_0_0.vlm"},
	{0x0312ADB44EB60540ULL, "vxlt_gallowscourtyard_n_6yr_0_0.vlm"},
	{0x0312B61833B2C5E3ULL, "114051186.ogg"},
	{0x0313B9DA0A7B7C9CULL, "vxlt_0_lowtown_docks_bfx_0_0.vlm"},
	{0x0313F50E293A8A46ULL, "abi_m_haste.wbd"},
	{0x031461FB81778369ULL, "vxlt_0_lowtown_docks_a3_cyb_0_0.vlm"},
	{0x03149A9B3C87DC4FULL, "vxw_woundedcoast_1gz_0_0.vlm"},
	{0x03158EE6137F88A7ULL, "zz_ser_debug6091987.cl"},
	{0x0315AFE839262CA1ULL, "cd_dae000_bartrand.bnk"},
	{0x0315C865588FF381ULL, "art\\characters\\playercharacter\\elfmale\\animation\\exploration\\idle\\twitch_v2.ani"},
	{0x0315DEFB8252BAD1ULL, "and321tr_act3_opening.ncs"},
	{0x0316200AE570C452ULL, "543553393.ogg"},
	{0x03162D14FF024110ULL, "w_sundermount_220_0.phy"},
	{0x0316B0CE8652A5BBULL, "vxlt_0_lowtown_eql_0_0.vlm"},
	{0x0316E7755FDA7DBCULL, "hlac_lt_undercityserialkiller_-1_5y.dds"},
	{0x03170F9C548A54C0ULL, "670087323.ogg"},
	{0x03173A95A737BEE6ULL, "hlac_lt_0_lowtown_a2_n_65537_7y.dds"},
	{0x031755ADA5D13121ULL, "hlac_lt_gallowsprison_n_-1_12y.dds"},
	{0x03175CCDD1E98B3BULL, "6093606_f.fxe"},
	{0x0317A12E537644E8ULL, "crack2.dds"},
	{0x0317D24CDA644093ULL, "hlac_w_bonepitdestroyed_a3_65536_0z.dds"},
	{0x0317D9E108BC2A6CULL, "vxht_mansion2_2vu_0_0.vlm"},
	{0x03188D81D8CFBD8BULL, "6147975_m.fxe"},
	{0x0318F85E704DA6F9ULL, "art\\vfx\\combat\\vfx_glb_ele_n_additive_c.impa.evt"},
	{0x03192116CC505F2FULL, "gen00fl_sebastian6175001.cl"},
	{0x03192EB009D8BBFFULL, "llac_lt_0_lowtown_a3_n_c_-3_0.dds"},
	{0x0319344A378FBA8FULL, "hlac_w_woundedcoast_-196607_0x.dds"},
	{0x03197F0F9B773796ULL, "vxlt_house_eq_0_0.vlm"},
	{0x0319A27D215D62D9ULL, "vxht_keep_5zw_0_0.vlm"},
	{0x0319DADD698ED066ULL, "vxca_caverns1_j2a_0_2.vlm"},
	{0x0319E4DABCB3E455ULL, "20361247.ogg"},
	{0x031A3D44E2205A30ULL, "vxlt_warehouse_164_0_0.vlm"},
	{0x031A480DC188A4DEULL, "art\\vfx\\combat\\vfx_m_bld_deathmagic_act_c.mmh"},
	{0x031A6713723C2BBCULL, "hlac_ca_caverns2prc_5802_3y.dds"},
	{0x031AA8A678DD4A6AULL, "vxw_bonepit_219_0_0.vlm"},
	{0x031AB88B307076EFULL, "hlac_ht_keep_n_4519_9y.dds"},
	{0x031ABA0515CE1C00ULL, "gen00fl_fenris6126616.cl"},
	{0x031BD36558947223ULL, "art\\characters\\playercharacter\\elfmale\\animation\\exploration\\idle\\twitch_v2.gad"},
	{0x031CB410B94BE083ULL, "6190441_m.fxe"},
	{0x031CD162590A586DULL, "vxlt_hangedman_n_1rj_0_0.vlm"},
	{0x031D0A6EB2212141ULL, "vxca_cavern1_haunted_gkj_0_1.vlm"},
	{0x031DC1F7C4FEDB54ULL, "vxw_bonepitmining_a2_29i_0_0.vlm"},
	{0x031E0798349BC033ULL, "vxlt_warehouse_n_x5_0_0.vlm"},
	{0x031E36EF4695D52EULL, "mag311st_keran.stg"},
	{0x031E489FBDBE967FULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\rogue\\dual weapons\\dw_attk_d1_v1.gad"},
	{0x031E71098096F07FULL, "mrl310_merrill6143819.cl"},
	{0x031EC4AA3747B2D9ULL, "713192954.ogg"},
	{0x031EEA4CB98D2BD2ULL, "vxht_brothel_n_3qw_0_0.vlm"},
	{0x031F035E96C09EF7ULL, "482505535.ogg"},
	{0x031F3853B6151093ULL, "art\\characters\\playercharacter\\humanmale\\animation\\ambient\\dialogue\\hands_behind_back_enter.evt"},
	{0x031F72D83B76A305ULL, "zz_approval_debug6003118.cl"},
	{0x031F9D1D827FAE5DULL, "fen210_orana6133071.cl"},
	{0x03208B6CFB2871FFULL, "6163371_m.fxe"},
	{0x0320F3494EE415A1ULL, "hlac_w_bonepitdestroyed_a3_196612_0x.dds"},
	{0x03210C4E2552BA92ULL, "6143732_m.fxe"},
	{0x03213178681B92C5ULL, "vxht_0_hightown_a2_awp_0_0.vlm"},
	{0x03213760F980B743ULL, "6118438_m.fxe"},
	{0x03213E7C72FF1934ULL, "vxht_bartrand_n_6kj_0_0.vlm"},
	{0x03215AFB8C669358ULL, "454960271.ogg"},
	{0x0321B9278A9E5BF6ULL, "art\\vfx\\materials\\chunky\\vfx_c_spi_hand_vc.mao"},
	{0x0321E22183B4D04BULL, "6163421_m.fxe"},
	{0x0321E6C6C860654CULL, "717178414.ogg"},
	{0x0322352DFAFB37F0ULL, "vxht_chantry_25x_0_0.vlm"},
	{0x0322782E63B7B50AULL, "vxca_cavern1_haunted_fmw_0_0.vlm"},
	{0x03229E2FB5BA1CA7ULL, "266281053.ogg"},
	{0x0322B065589A5508ULL, "art\\characters\\playercharacter\\elfmale\\animation\\exploration\\idle\\twitch_v2.evt"},
	{0x0322BD6246656345ULL, "ser270st_templar.stg"},
	{0x0322BFECB844F23BULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\additive\\shoulder_r_roll_back.ani"},
	{0x0323164E82BD3D2EULL, "vxht_mansion2b_29r_0_0.vlm"},
	{0x03236D243D461125ULL, "254026138.ogg"},
	{0x03237876CCF29373ULL, "hlac_lt_undercitydungeon_-1_22z.dds"},
	{0x03237E38D6BA0271ULL, "1053829834.ogg"},
	{0x03239622B6920EEDULL, "ts_circleofmagi_prog_junk1.gda"},
	{0x0323A465993BDC5CULL, "vxlt_undercitydungeon_1c5_0_0.vlm"},
	{0x0323C35E527AF2EBULL, "6077229_m.fxe"},
	{0x03243D9FBDC2EF6CULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\rogue\\dual weapons\\dw_attk_d1_v1.evt"},
	{0x03253153B61970D4ULL, "art\\characters\\playercharacter\\humanmale\\animation\\ambient\\dialogue\\hands_behind_back_enter.gad"},
	{0x0326211BF2FFFEFCULL, "vxlt_warehouse_n_13l_0_0.vlm"},
	{0x0326809713A77958ULL, "6084372_m.fxe"},
	{0x03269C002613FB0BULL, "6087554_m.fxe"},
	{0x0326D4572BE5C953ULL, "vxlt_backalley_g_1dn_0_0.vlm"},
	{0x0327005E7059F1E8ULL, "art\\vfx\\combat\\vfx_glb_ele_n_additive_c.impa.ani"},
	{0x0327270DC192F69DULL, "art\\vfx\\combat\\vfx_m_bld_deathmagic_act_c.anb"},
	{0x03279B5C249ECA7BULL, "art\\levels\\ztools\\default\\tilesets\\setint\\_textures\\xx_wood.dds"},
	{0x0327F999BDEB0EE8ULL, "cr_spe_01.dds"},
	{0x0328F3B19432771CULL, "vxht_keep_n_4nd_0_0.vlm"},
	{0x03292601330BAD62ULL, "gen_im_arm_boo_hvy_champion.uti"},
	{0x03292E55A93032E3ULL, "gen00fl_anders6121207.cl"},
	{0x032976A5AF247D44ULL, "131858664.ogg"},
	{0x0329B954FE983938ULL, "hlac_lt_0_lowtown_a2_n_c_65534_1z.dds"},
	{0x0329D201C83627A2ULL, "placeable_core.ncs"},
	{0x0329EF7436B47AA6ULL, "art\\characters\\weapons\\greataxes\\textures\\w_gax_002a_s.dds"},
	{0x032A199E6DF6B897ULL, "zz_ave_debug6174721.cl"},
	{0x032A2493533D6E40ULL, "vxdr_0_deeproads_f5k_0_0.vlm"},
	{0x032A88DAC7E5D611ULL, "vxlt_undercityserialkiller_135_0_0.vlm"},
	{0x032B201996D6F33EULL, "6118145_m.fxe"},
	{0x032B30408D4B62E1ULL, "915557874.ogg"},
	{0x032B41EBE7F1DB6BULL, "art\\vfx\\combat\\vfx_m_bld_mode_c.cess.ani"},
	{0x032BD971DD730E1EULL, "305643762.ogg"},
	{0x032C4053B61FA8B2ULL, "art\\characters\\playercharacter\\humanmale\\animation\\ambient\\dialogue\\hands_behind_back_enter.ani"},
	{0x032C48B239C919F9ULL, "vxlt_darktown_4z1_0_0.vlm"},
	{0x032C9A38D9030FFCULL, "vxht_0_hightown_a2_9bj_0_0.vlm"},
	{0x032CAB13B7DFBCC4ULL, "6078335_m.fxe"},
	{0x032DB0A0A577C334ULL, "hlac_w_woundedcoast_0_11y.dds"},
	{0x032DE4941697D401ULL, "vxlt_0_lowtown_a3_n_c_hph_0_0.vlm"},
	{0x032DE4A9AC93387BULL, "crt_sw_bee_gl.mao"},
	{0x032E7519036F32DEULL, "vxlt_warehouse_sd_0_0.vlm"},
	{0x032ECDC3D4AEA212ULL, "mrl210_merrill6102153.cl"},
	{0x032F02722187B09CULL, "vxw_bonepitdestroyed_a3_1zm_0_0.vlm"},
	{0x032F0481B6C042DBULL, "vxlt_hangedman_1xq_0_0.vlm"},
	{0x032F34D3B0CE2E8CULL, "lt_0_lowtown_docks_a3_212_0.msh"},
	{0x032F37FCC1467956ULL, "hlac_lt_0_lowtown_a2_-65535_0x.dds"},
	{0x032F3AD3B0CE397EULL, "lt_0_lowtown_docks_a3_212_0.mmh"},
	{0x032F46D3B0CE4D15ULL, "lt_0_lowtown_docks_a3_212_0.mao"},
	{0x032F5E434C1AB49BULL, "zz_one_debug6077408.cl"},
	{0x032F9F0FF03682FAULL, "6171535_m.fxe"},
	{0x032FC7ECB84F8A32ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\additive\\shoulder_r_roll_back.evt"},
	{0x032FF0329A650F2BULL, "llac_ca_caverns1_24427_5.dds"},
	{0x0330550CED3A67D2ULL, "vxlt_backalley_g_1dk_0_0.vlm"},
	{0x0330A23BBD605816ULL, "art\\characters\\creatures\\hurlock\\animation\\sword_shield\\hurlock_abilitycharge.ani"},
	{0x0330FD08B003911AULL, "vxlt_0_lowtown_a3_n_c_gjz_0_0.vlm"},
	{0x033149567426ED40ULL, "6165114_m.fxe"},
	{0x0331841A59B5D584ULL, "168460306.ogg"},
	{0x03318573C8A8D0B9ULL, "254109219.ogg"},
	{0x0331DA0C3D546757ULL, "608945896.ogg"},
	{0x0332259FBDCF03C5ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\rogue\\dual weapons\\dw_attk_d1_v1.ani"},
	{0x0332353B44710B6BULL, "6078329_m.fxe"},
	{0x033248236C51AA31ULL, "6153378_m.fxe"},
	{0x0332C3F28022C81AULL, "6068268_m.fxe"},
	{0x033346614B14E009ULL, "415140884.ogg"},
	{0x0333E75BC68C11E7ULL, "6097159_m.fxe"},
	{0x033453404FC9BA54ULL, "dae000cr_grand_cleric6182004.cl"},
	{0x033492F8E4F6177CULL, "vxlt_warehouse_n_18m_0_0.vlm"},
	{0x0334F4D36F9BFAEFULL, "285305121.ogg"},
	{0x0336421CCD410BACULL, "dre000st_varric_lyrium.stg"},
	{0x0336A89E8C83F825ULL, "vxca_cavern1_haunted_huf_0_0.vlm"},
	{0x0336E6ECB855DDC5ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\additive\\shoulder_r_roll_back.gad"},
	{0x03375FD155B780F6ULL, "34100833.ogg"},
	{0x0337FF6F719FBDBFULL, "731050005.ogg"},
	{0x033829EBE7FC3CE2ULL, "art\\vfx\\combat\\vfx_m_bld_mode_c.cess.evt"},
	{0x033832BEFC56CBE4ULL, "326606094.ogg"},
	{0x0338CB385958275FULL, "art\\levels\\races\\proxy\\tilesets\\setint\\8co_basic\\1wi_diagonal.msh"},
	{0x0338D13859583101ULL, "art\\levels\\races\\proxy\\tilesets\\setint\\8co_basic\\1wi_diagonal.mmh"},
	{0x0338DD38595845A2ULL, "art\\levels\\races\\proxy\\tilesets\\setint\\8co_basic\\1wi_diagonal.mao"},
	{0x03395966E74051ADULL, "6068501_f.fxe"},
	{0x033A591FDE00FDB1ULL, "6053603_m.fxe"},
	{0x033A8B01FCF23D56ULL, "vxht_keep_n_4h8_0_0.vlm"},
	{0x033ABA8F2BBF3279ULL, "vxlt_darktown_4a0_0_0.vlm"},
	{0x033BEE460B1E4A86ULL, "vxlt_hangedman_283_0_0.vlm"},
	{0x033BEFF44E96CFCFULL, "505882856.ogg"},
	{0x033CE5B4EFFB1553ULL, "gen00fl_aveline6122137.cl"},
	{0x033CE6F5F5654B5EULL, "vxlt_warehouse_te_0_0.vlm"},
	{0x033CF3419AA3CF4EULL, "art\\characters\\playercharacter\\humanmale\\textures\\hm_cth_nobc_0n.dds"},
	{0x033DBA8B3E04E919ULL, "ts_apostates_03.gda"},
	{0x033E223656EB4A34ULL, "vxht_bartrand_n_6qh_0_0.vlm"},
	{0x033E4FAE023D0693ULL, "vxlt_darktown_56r_0_0.vlm"},
	{0x033E82C528562082ULL, "846999526.ogg"},
	{0x033EAA3BBD6CA307ULL, "art\\characters\\creatures\\hurlock\\animation\\sword_shield\\hurlock_abilitycharge.evt"},
	{0x033ED6009F58A9BAULL, "1000753099.ogg"},
	{0x033EF01DD134B384ULL, "pro000_captain_wright6088112.cl"},
	{0x033EF139F91B52D9ULL, "arc_mit_st.mao"},
	{0x033EFCD1A9F711A0ULL, "vxca_cavern1_haunted_j4z_0_0.vlm"},
	{0x033F28EBE80259F5ULL, "art\\vfx\\combat\\vfx_m_bld_mode_c.cess.gad"},
	{0x033F373A4169DE38ULL, "dae300_bran6143087.cl"},
	{0x033F7A17B1DF9C94ULL, "hlac_lt_0_lowtown_docks_65536_0y.dds"},
	{0x033FB1F0CC0D382CULL, "vxlt_gallowstemplar_n_3k7_0_0.vlm"},
	{0x0340024E56B5ABE0ULL, "gen00fl_fenris6127985.cl"},
	{0x034023C8171555D0ULL, "combat_st_r1_a5.gda"},
	{0x034099C28A4776D6ULL, "vxw_bonepitdestroyed_a3_1x1_0_0.vlm"},
	{0x03409A1F49747387ULL, "vxlt_warehouse_n_rk_0_0.vlm"},
	{0x0341D6EE8F2D2615ULL, "framebuffereffects_diffusiondof_p5.psh.fxo.dx11"},
	{0x034275C580860B50ULL, "vxlt_backalley_n_1n0_0_2.vlm"},
	{0x0342AE2A2A2C3BD3ULL, "hlac_w_blightlands_327678_0y.dds"},
	{0x0343283347D2A629ULL, "vxlt_backalley_n_1k8_0_0.vlm"},
	{0x03434AF658CF43FFULL, "802579644.ogg"},
	{0x0343D5E4D62FB214ULL, "191764040.ogg"},
	{0x0344111AF65EA291ULL, "686657894.ogg"},
	{0x0344A1458C1A6CF4ULL, "6095868_m.fxe"},
	{0x0344A49DF8E575A2ULL, "vxca_caverns1_mines_edo_0_0.vlm"},
	{0x0344F84C90F732B8ULL, "6749365.ogg"},
	{0x0345444076488FFBULL, "vxca_cavern1_haunted_im4_0_1.vlm"},
	{0x0345CB3BBD72FA50ULL, "art\\characters\\creatures\\hurlock\\animation\\sword_shield\\hurlock_abilitycharge.gad"},
	{0x0345FB83CAA2CD27ULL, "dre200st_dougal_ambush.stg"},
	{0x0346407262B2BC8BULL, "vxlt_0_lowtown_a2_n_c_hov_0_0.vlm"},
	{0x03465E60D0AB18A4ULL, "vxlt_house_e3_0_0.vlm"},
	{0x034660FDC026F160ULL, "hlac_ht_keep_n_4519_18x.dds"},
	{0x034681FE87AA7517ULL, "vxw_bonepit_ni_0_0.vlm"},
	{0x03473C3F954F48A6ULL, "hlac_lt_0_lowtown_a3_n_c_-2_0z.dds"},
	{0x0347FDAC44753ED3ULL, "6143250_m.fxe"},
	{0x0348336A3DDEC326ULL, "6216955_m.fxe"},
	{0x0348875211A2DFF7ULL, "vxlt_0_lowtown_eqx_0_0.vlm"},
	{0x03492C6C1DB54AF9ULL, "vxlt_darktown_4l7_0_0.vlm"},
	{0x034949C4A7D16B62ULL, "6165255_m.fxe"},
	{0x034977353D8C7880ULL, "6015864_m.fxe"},
	{0x03497E5529DA48A2ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\humanmale_ambient_cheerwinning_2_darren.gad"},
	{0x034997CBC7A31B5BULL, "sst_ss_livingarmor_1.gda"},
	{0x0349BA98E450C731ULL, "327217579.ogg"},
	{0x034A0563CA114AC1ULL, "6070504_f.fxe"},
	{0x034A0C9A88C194ECULL, "89077315.ogg"},
	{0x034A3321B05A90B8ULL, "vxlt_hangedman_25z_0_0.vlm"},
	{0x034A41447EF4E956ULL, "vxlt_0_lowtown_a3_n_c_enx_0_0.vlm"},
	{0x034A6022FD146306ULL, "vxlt_hangedman_254_0_0.vlm"},
	{0x034B04D1F8CB3593ULL, "vxw_woundedcoast_1d7_0_0.vlm"},
	{0x034B2079ACDDBFFEULL, "ser120_ghyslain6105623.cl"},
	{0x034B3892778B5AF8ULL, "187907354.ogg"},
	{0x034B58D2E75B63DEULL, "vxlt_warehouse_yf_0_0.vlm"},
	{0x034BEEEC1FE9598BULL, "806652237.ogg"},
	{0x034C44FE46FD1134ULL, "6051396_m.fxe"},
	{0x034CC19C7BA636C2ULL, "vxlt_gamlenshouse_n_nt_0_0.vlm"},
	{0x034CD9216DA5708CULL, "6174215_m.fxe"},
	{0x034CFE993A143526ULL, "6067269_m.fxe"},
	{0x034D44AF9721551FULL, "art\\levels\\races\\ferelden\\props\\lowerclass\\prp_cratelg_0.pwk"},
	{0x034E4F5D00491F54ULL, "618982592.ogg"},
	{0x034E778FCCAB27EFULL, "gen00fl_aveline6002587.cl"},
	{0x034E87B1802DDFB3ULL, "6084919_m.fxe"},
	{0x034EBB99562F2CBFULL, "t3_psp_b02.tnt"},
	{0x034F0BFC3B6A8C07ULL, "vxlt_warehouse_n_yj_0_0.vlm"},
	{0x034F3249F7EA261DULL, "414573605.ogg"},
	{0x034F7892644E22DEULL, "6113496_m.fxe"},
	{0x035022EEB6D27FB9ULL, "vxlt_0_lowtown_a2_n_c_csg_0_0.vlm"},
	{0x0350F0DFB9FFCCC3ULL, "art\\characters\\playercharacter\\humanfemale\\hf_har_hb2a.mmh"},
	{0x0350FEDFB9FFE41DULL, "art\\characters\\playercharacter\\humanfemale\\hf_har_hb2a.msh"},
	{0x03513BB49A766ADCULL, "hlac_w_encounter2_196611_0z.dds"},
	{0x0352044C4B99A7C1ULL, "hlac_w_sundermount_0_4x.dds"},
	{0x03522A72C3221461ULL, "vxw_encounter2_mz_0_0.vlm"},
	{0x03526999A2385A8CULL, "gen00fl_merrill6133144.cl"},
	{0x03526FE980F06D7BULL, "gencr_templar_meredith.utc"},
	{0x0352DA465310522DULL, "311475772.ogg"},
	{0x0353B9B09C9DE307ULL, "vxlt_darktown_4ub_0_0.vlm"},
	{0x03547025ED48EA16ULL, "6049434_m.fxe"},
	{0x035528CD38A9AF39ULL, "142073347.ogg"},
	{0x0355851E9D0DEC37ULL, "vxw_bonepitdestroyed_a3_p7_0_0.vlm"},
	{0x0355C13C0A21E72EULL, "vxlt_gallowstemplar_f_2bg_0_0.vlm"},
	{0x0355D39CD1A1D1ABULL, "hlac_lt_0_lowtown_a2_65538_4y.dds"},
	{0x0355FC4ECFCD10DBULL, "578376902.ogg"},
	{0x035601CDA5A30E02ULL, "6099699_m.fxe"},
	{0x03574EC3C1AD2C35ULL, "vxw_encounter2_camp_6s_0_0.vlm"},
	{0x0357E8B74F0CBE10ULL, "vxlt_0_lowtown_a2_n_c_ehy_0_0.vlm"},
	{0x0357FB8237A80385ULL, "6055142_m.fxe"},
	{0x0357FD6E3B1223BEULL, "vxht_keep_674_0_0.vlm"},
	{0x035808A565E562E2ULL, "vxlt_0_lowtown_h6q_0_0.vlm"},
	{0x035871A34AF6F39FULL, "vxlt_gallowstemplar_39e_0_0.vlm"},
	{0x0358C84BD143F1DBULL, "585070178.ogg"},
	{0x0358D1FFEF0A7B86ULL, "vxlt_hangedman_225_0_0.vlm"},
	{0x0359364B5F36C146ULL, "vxht_mansion2b_6u_0_0.vlm"},
	{0x035976AEEAC14E13ULL, "vxw_woundedcoast_1o6_0_0.vlm"},
	{0x035A0B9326490838ULL, "140349955.ogg"},
	{0x035A2E1512A34E21ULL, "461742218.ogg"},
	{0x035A6CD37E264391ULL, "590404810.ogg"},
	{0x035A96B49A8F924AULL, "6083825_m.fxe"},
	{0x035AF58383DCCB62ULL, "vxlt_0_lowtown_ei9_0_0.vlm"},
	{0x035B6F0E56886284ULL, "653549345.ogg"},
	{0x035B9EE5D4EDFF16ULL, "zz_dre_debug6054625.cl"},
	{0x035BF705B95DAFFDULL, "vxlt_0_lowtown_docks_n_bbz_0_0.vlm"},
	{0x035C74B022F6A4B5ULL, "250330734.ogg"},
	{0x035CE4DE0E29D75BULL, "589118319.ogg"},
	{0x035CF1AC9A2FFAE2ULL, "vxht_chantry_n_2yi_0_0.vlm"},
	{0x035D7765604D3D48ULL, "w_bonepitmining_a2_198_0.mmh"},
	{0x035D7D7C6E33A7D6ULL, "vxw_bonepitdestroyed_a3_1f3_0_0.vlm"},
	{0x035D8365604D51EBULL, "w_bonepitmining_a2_198_0.mao"},
	{0x035D90F77D7D7FCCULL, "vxlt_backalley_g_1vs_0_0.vlm"},
	{0x035D9565604D7072ULL, "w_bonepitmining_a2_198_0.msh"},
	{0x035DA95529EB3A88ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\humanmale_ambient_cheerwinning_2_darren.ani"},
	{0x035DBDE83963FA53ULL, "cd_dae000_bartrand.wbd"},
	{0x035DFFCFC3D85F09ULL, "389201594.ogg"},
	{0x035E0328F82B9A8EULL, "qun231_party_comment6077137.cl"},
	{0x035E10ACD79762BBULL, "gen00fl_isabela6097009.cl"},
	{0x035E2F555020138DULL, "21294410.ogg"},
	{0x035EAD9FFE9E210EULL, "vxlt_0_lowtown_a2_n_c_f4x_0_0.vlm"},
	{0x035F3F3D24BED290ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\9wi_baserocks_1.mao"},
	{0x035F57EDD80EE336ULL, "fx_m3c6.mao"},
	{0x036044B572DD1E2EULL, "llac_lt_0_lowtown_a3_n_c_0_7.dds"},
	{0x0360BB9B0FC40AC1ULL, "vxht_keep_5gh_0_0.vlm"},
	{0x03612193CF96F56BULL, "vxlt_0_lowtown_a2_n_cmq_0_0.vlm"},
	{0x03619017311D2A5CULL, "follower_banter6160677.cl"},
	{0x0361D9D34CD8EE65ULL, "6047439_m.fxe"},
	{0x0361F1DFBA0E41E8ULL, "art\\characters\\playercharacter\\humanfemale\\hf_har_hb2a.phy"},
	{0x0361F3C045683B88ULL, "vxht_0_hightown_a2_cf8_0_0.vlm"},
	{0x03622BB0D393DB1EULL, "6113904_m.fxe"},
	{0x03626E08691455EAULL, "gen_im_wep_rog_dua_start.uti"},
	{0x036294FB14ACF004ULL, "vxca_caverns2prc_3p8_0_0.vlm"},
	{0x0362A34322DA0C58ULL, "qun221_petrice6083823.cl"},
	{0x0362BEDB452AD2B3ULL, "623643582.ogg"},
	{0x0362CECA72101FE6ULL, "hlac_lt_0_lowtown_docks_-65538_4x.dds"},
	{0x0362FC8AB226E0B5ULL, "gen00fl_anders6113037.cl"},
	{0x0363ED5F2FE9F133ULL, "gen00fl_dog6115072.cl"},
	{0x03644F7AAC5F617EULL, "505504122.ogg"},
	{0x036479429AEC5628ULL, "llac_lt_backalley_g_1_7.dds"},
	{0x0364C597A70CD3A0ULL, "vxlt_darktown_6j6_0_0.vlm"},
	{0x0364E5096C10AE86ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\3ww_basic.phy"},
	{0x0365C0A0B3A344B5ULL, "vxw_encounter2_camp_1r_0_0.vlm"},
	{0x03666F4B2D083C3EULL, "vxht_keep_605_0_0.vlm"},
	{0x036686EC2EF6E601ULL, "355940205.ogg"},
	{0x0367658A61E1F2A0ULL, "vxca_cavern1_haunted_gf1_0_0.vlm"},
	{0x03678E48FB713A3CULL, "ht_0_hightown_a2_21_0_lc.mmh"},
	{0x0367AC07DE703601ULL, "vxlt_0_lowtown_a3_n_c_hll_0_0.vlm"},
	{0x0367E88BDCB76693ULL, "vxw_woundedcoast_1j9_0_0.vlm"},
	{0x0368963A0936CA57ULL, "gen00fl_carver6108934.cl"},
	{0x0368C9E5E960129CULL, "vxw_bonepitdestroyed_a3_1na_0_0.vlm"},
	{0x0368FBB84BB36E0AULL, "6083710_f.fxe"},
	{0x036909B1D3C035B6ULL, "247101991.ogg"},
	{0x03690B67CBFD6924ULL, "vxlt_warehouse_8c_0_0.vlm"},
	{0x036916776B4AA0A2ULL, "6000741_m.fxe"},
	{0x036932E2C9EC2D0AULL, "vxlt_0_lowtown_a2_n_c_e1h_0_0.vlm"},
	{0x0369924438405536ULL, "art\\characters\\weapons\\staff\\textures\\w_stf_015a_0t.dds"},
	{0x036A304A909BC593ULL, "gen00fl_anders6116119.cl"},
	{0x036A4CDEEAE82CD6ULL, "art\\characters\\playercharacter\\humanmale\\animation\\partypicker\\magestaffidle.ani"},
	{0x036AF58BA36C905FULL, "27831013.ogg"},
	{0x036C018A49AAF34BULL, "lt_gallowstemplar_f_2394.rml"},
	{0x036CB504386FDE18ULL, "vxca_caverns1_mines_j42_0_0.vlm"},
	{0x036CEA7E3D66AE46ULL, "6075496_f.fxe"},
	{0x036D91F0F191728AULL, "gen00fl_fenris6112042.cl"},
	{0x036DA22BE8592401ULL, "6151116_m.fxe"},
	{0x036DDCC8B1810F71ULL, "dae000_gen_turn_in_26178485.cl"},
	{0x036E09107FD28502ULL, "mag160_feynriel6020423.cl"},
	{0x036E163CDF24D05AULL, "309465768.ogg"},
	{0x036E47D2E0A7B1A1ULL, "hlac_lt_0_lowtown_a2_n_131072_3x.dds"},
	{0x036EA4A0EDEA36DBULL, "follower_banter6163655.cl"},
	{0x036EBF898AB34F3CULL, "693183747.ogg"},
	{0x036F5825486A683BULL, "vxht_chantry_n_2c2_0_0.vlm"},
	{0x036FD46D5DF7D10BULL, "132643051.ogg"},
	{0x036FFED0DE598802ULL, "mcr371_hostile_fenris6082255.cl"},
	{0x03701BF6BBBAE67BULL, "hlac_ht_keep_4519_10z.dds"},
	{0x037041677DA2228CULL, "6071890_f.fxe"},
	{0x03705453B9D7E27EULL, "845927087.ogg"},
	{0x0371F672BB4E7E91ULL, "vxdr_0_deeproads_ewg_0_0.vlm"},
	{0x03728C4CF6E40846ULL, "hlac_lt_0_lowtown_docks_a3_-131073_3x.dds"},
	{0x0372E21B7D044978ULL, "hlac_w_blightlands_0_11y.dds"},
	{0x0372E587898F3F02ULL, "6093499_m.fxe"},
	{0x037307A85113FFCEULL, "212942974.ogg"},
	{0x037322A85E20286AULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\0rw_cap.mao"},
	{0x037326A85E202EB9ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\0rw_cap.mmh"},
	{0x037330A85E203FA7ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\0rw_cap.msh"},
	{0x0373AD1A06A943CCULL, "100788062.ogg"},
	{0x0373BA87F3DF0577ULL, "gen00fl_sebastian6180841.cl"},
	{0x0373D199799DF6DEULL, "521205737.ogg"},
	{0x0373F57473176324ULL, "6166238_m.fxe"},
	{0x03755DEA508EC3A9ULL, "800259845.ogg"},
	{0x0375D716E7F0CB79ULL, "152554641.ogg"},
	{0x03761DE4D0664E81ULL, "vxlt_0_lowtown_a3_n_c_hic_0_0.vlm"},
	{0x03764F630AD7550AULL, "vxlt_0_lowtown_h83_0_0.vlm"},
	{0x03768B36E3B8D2BFULL, "hlac_ca_caverns2prc_5802_27z.dds"},
	{0x0376E54007E6BEACULL, "vxlt_0_lowtown_fgj_0_0.vlm"},
	{0x037754DEEAF2C4C7ULL, "art\\characters\\playercharacter\\humanmale\\animation\\partypicker\\magestaffidle.evt"},
	{0x037783D6113BE61EULL, "large_d.dds"},
	{0x0377B0E9BACA1B49ULL, "6164457_m.fxe"},
	{0x0377E0D20F482DD8ULL, "vxca_caverns2prc_7wu_0_0.vlm"},
	{0x03788686E405BD00ULL, "1032962997.ogg"},
	{0x0378D3644A350B47ULL, "mag170_hugh6080801.cl"},
	{0x03792F35CE1B4182ULL, "vxlt_darktown_2r8_0_0.vlm"},
	{0x0379931E9F77BDB5ULL, "gen00fl_merrill6151099.cl"},
	{0x0379DEDC5B7F298EULL, "543078483.ogg"},
	{0x0379FA767A3A0F69ULL, "w_bonepitdestroyed_a3_198_0.phy"},
	{0x037A74B16169B0CCULL, "vxlt_backalley_g_1xq_0_0.vlm"},
	{0x037AB6AAF319B9F2ULL, "ht_brothel_n_3370.anb"},
	{0x037AE2741CA86135ULL, "vxlt_foundry_n_142_0_0.vlm"},
	{0x037AE7D307F69F89ULL, "gen00fl_varric6087600.cl"},
	{0x037B24D12EBE4AA5ULL, "mag210_arianni6021223.cl"},
	{0x037B50583A01A549ULL, "isa321_velasco6094489.cl"},
	{0x037BC554C7E4FE5BULL, "6180097_m.fxe"},
	{0x037C256A723F5AD1ULL, "96182818.ogg"},
	{0x037C27CC05168EFAULL, "86324526.ogg"},
	{0x037CFC93F2AF4635ULL, "vxca_caverns1_j5q_0_0.vlm"},
	{0x037D1801882D3E41ULL, "mag370_meredith6092172.cl"},
	{0x037D4E59A605E556ULL, "6078460_f.fxe"},
	{0x037D68EDD827FDB8ULL, "fx_m3c6.dds"},
	{0x037D85A9641DBF01ULL, "gen00fl_sebastian6195849.cl"},
	{0x037DA77754B7BEA5ULL, "vxlt_undercitydungeon_1jc_0_0.vlm"},
	{0x037E6F033D890141ULL, "784904153.ogg"},
	{0x037E75DEEAF91B10ULL, "art\\characters\\playercharacter\\humanmale\\animation\\partypicker\\magestaffidle.gad"},
	{0x037EF5CB79C5FE17ULL, "vrc321ip_spooky6_obj_6.utp"},
	{0x037FD39971E4F0A5ULL, "6080834_m.fxe"},
	{0x03802555A296DDEBULL, "6094891_m.fxe"},
	{0x0380684FAD449711ULL, "vxdr_0_deeproads_epf_0_0.vlm"},
	{0x0380EBB167F49D4EULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\prp_dooriron1c.msh"},
	{0x0380F9B167F4B49FULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\prp_dooriron1c.mao"},
	{0x038105B167F4C93CULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\prp_dooriron1c.mmh"},
	{0x03815695388B17ECULL, "dae000_mother6028523.cl"},
	{0x03825CEF969E5649ULL, "ico_gloves_heavy_4.dds"},
	{0x03825E433F4005E3ULL, "6106899_m.fxe"},
	{0x038287B4058A220AULL, "dae000_madam_lusine6151391.cl"},
	{0x038386C80EE345CAULL, "305820507.ogg"},
	{0x0383D6DEE4EE2C54ULL, "hlac_lt_0_lowtown_docks_0_3y.dds"},
	{0x0384338C88BBA5D8ULL, "t3_boy_b02.tnt"},
	{0x038438585BA931FCULL, "hlac_dr_slavers_3217_8y.dds"},
	{0x038439E3C2AAC42BULL, "combat_arishock_combo.gda"},
	{0x0384DBEEE12FC14CULL, "6105439_m.fxe"},
	{0x0385571CF9DCD72CULL, "vxlt_0_lowtown_fhk_0_0.vlm"},
	{0x0385970B01E09668ULL, "dre000_vampire6034337.cl"},
	{0x0386497ED977EAE9ULL, "160319597.ogg"},
	{0x0386A9D4C56E0717ULL, "hlac_lt_0_lowtown_0_4x.dds"},
	{0x0386B7F66F897C03ULL, "205614137.ogg"},
	{0x03874C9C8F3FF97DULL, "vxlt_0_lowtown_docks_n_bqw_0_0.vlm"},
	{0x03877155931A6F62ULL, "403055393.ogg"},
	{0x0387A112C0115A02ULL, "vxlt_darktown_2u9_0_0.vlm"},
	{0x038856B320127A62ULL, "426966247.ogg"},
	{0x03887B44BA860D40ULL, "24943529.ogg"},
	{0x0388E6B33ECBC904ULL, "vxca_cavern1_haunted_icq_0_1.vlm"},
	{0x03893C234B71A47DULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_laundry_sa.dds"},
	{0x0389647BE07A165EULL, "vxw_encounter3_8u_0_0.vlm"},
	{0x0389831CCF95EB17ULL, "vxdr_primevaltomb_5cc_0_0.vlm"},
	{0x0389CA80D95C61F4ULL, "125220886.ogg"},
	{0x038A38B7A9C0B4BBULL, "6155802_m.fxe"},
	{0x038B090B71FAB5EDULL, "zz_lgt_2_debug6206587.cl"},
	{0x038B367FF78977CCULL, "478867279.ogg"},
	{0x038B38CF90972ED0ULL, "6033565_m.fxe"},
	{0x038B3A51B7A5F0A9ULL, "605741718.ogg"},
	{0x038C3572DFE28CA3ULL, "vxdr_primevaltomb_1ks_0_0.vlm"},
	{0x038C58E8368B8642ULL, "vxht_playershouse_n_1ld_0_0.vlm"},
	{0x038CBB74945C3D36ULL, "6094307_f.fxe"},
	{0x038CF6262E8661EFULL, "ser221_party_comments6209854.cl"},
	{0x038CFC28D6765150ULL, "213385057.ogg"},
	{0x038D942CABB00BBEULL, "6102919_m.fxe"},
	{0x038E21B60FEF18A9ULL, "652846022.ogg"},
	{0x038E4FC763F0E50FULL, "341866816.ogg"},
	{0x038E6BFA2FB87056ULL, "134042949.ogg"},
	{0x038E99D12969CD54ULL, "w_encounter1_8_0.phy"},
	{0x038F3B4B84E1F1D3ULL, "dae00st_gamlens_house.stg"},
	{0x038F94E32FE63DAAULL, "109549006.ogg"},
	{0x038FB0F4D371EF66ULL, "ht_0_hightown_a3_42_0_lc.mmh"},
	{0x038FE85E11FF2068ULL, "vxlt_undercityserialkiller_10i_0_0.vlm"},
	{0x0390259F07ED5CABULL, "6077420_f.fxe"},
	{0x03903542073422A8ULL, "gen00fl_merrill6108674.cl"},
	{0x03907F72E25E2633ULL, "51354205.ogg"},
	{0x0390D8D2CBA85FB7ULL, "zze_roguevshighdragon_track.cut"},
	{0x0390F4E642E984E4ULL, "vxdr_primevaltomb_6x6_0_0.vlm"},
	{0x03917FC99FA62D75ULL, "art\\vfx\\combat\\vfx_w_2nd_atk_l_dwn_c.mmh"},
	{0x0392049EA8680FC3ULL, "6031307_f.fxe"},
	{0x03925B850A02BFF3ULL, "art\\vfx\\combat\\vfx_r_gn4_smo_g_lar_aoe_p.cess.evt"},
	{0x03927A45C32E7A54ULL, "284694658.ogg"},
	{0x0392FB4CE4769BDBULL, "vxlt_foundry_n_108_0_0.vlm"},
	{0x03933124060DBC14ULL, "6044228_m.fxe"},
	{0x039332FF6318F3FBULL, "vxlt_0_lowtown_n_cmm_0_0.vlm"},
	{0x0393573F36E56397ULL, "one280_iwan6096866.cl"},
	{0x03937AF17AAC606BULL, "zz_vrc_debug6064294.cl"},
	{0x0393A836068E3B03ULL, "vxw_bonepit_26u_0_0.vlm"},
	{0x0393B7190B5A7C3DULL, "zz_mer_debug6017035.cl"},
	{0x039468AA4143DAE9ULL, "vfx_lightnig_dot.dds"},
	{0x03954D7DACBC95F5ULL, "rog_slowcloud.dds"},
	{0x039590D1296FDCFCULL, "w_encounter1_8_0.rml"},
	{0x0395BA9E8C22591CULL, "vxlt_0_lowtown_docks_b5q_0_0.vlm"},
	{0x03969C5CA6BBA982ULL, "dae000_worthy6166568.cl"},
	{0x0397115745C25388ULL, "vxca_cavern1_haunted_flv_0_0.vlm"},
	{0x03973A88F76EBDD3ULL, "vxw_encounter2_camp_qu_0_1.vlm"},
	{0x039759FB96A49617ULL, "6126985_m.fxe"},
	{0x0397D658D2702EDEULL, "vxw_encounter3_5v_0_0.vlm"},
	{0x03984F5C0D09D443ULL, "6118669_m.fxe"},
	{0x039855BB74CEAE6DULL, "vrc321_varric6084444.cl"},
	{0x0398D4E642A698C2ULL, "zz_followers_banter6158292.cl"},
	{0x0399522E935BE1AEULL, "hlac_w_encounter1_-131072_0z.dds"},
	{0x039954850A08D234ULL, "art\\vfx\\combat\\vfx_r_gn4_smo_g_lar_aoe_p.cess.gad"},
	{0x039976D129739951ULL, "w_encounter1_8_0.msh"},
	{0x039984D12973B16CULL, "w_encounter1_8_0.mao"},
	{0x039990D12973C50FULL, "w_encounter1_8_0.mmh"},
	{0x0399974FC66B2A6EULL, "lt_0_lowtown_docks_29_0_ll.mmh"},
	{0x0399D26BA9DEA809ULL, "vxlt_0_lowtown_docks_c6l_0_0.vlm"},
	{0x039A035A35F4E7BDULL, "follower_banter6163204.cl"},
	{0x039A346F988EEF33ULL, "hlac_lt_0_lowtown_a2_n_-131073_3z.dds"},
	{0x039B03AEE19620BCULL, "art\\cinematics\\additional animation\\cs_femnobleiii_cinematicanim.ani"},
	{0x039B61BD0D43E992ULL, "vxht_mansion2b_322_0_0.vlm"},
	{0x039B83BDC8A33CBEULL, "gen00fl_aveline6139149.cl"},
	{0x039BBF69072D43B3ULL, "379748269.ogg"},
	{0x039C475C9D31D481ULL, "1063577718.ogg"},
	{0x039C9085C92F035AULL, "hlac_lt_0_lowtown_65533_1x.dds"},
	{0x039CD0D539384C41ULL, "rdr111dg_martin6054583.cl"},
	{0x039D4C099130C811ULL, "vxdr_0_deeproads_efd_0_0.vlm"},
	{0x039D8573A8A5FD83ULL, "llac_lt_darktown_-1_45.dds"},
	{0x039DE6912EF297BFULL, "6108308_f.fxe"},
	{0x039E15B8B7FDCC03ULL, "art\\vfx\\combat\\vfx_r_gn4_stealth_wpn_c.impa.evt"},
	{0x039E7CC99FB0B202ULL, "art\\vfx\\combat\\vfx_w_2nd_atk_l_dwn_c.anb"},
	{0x039E7DBBB53A1F17ULL, "llac_w_encounter3_-131071_0.dds"},
	{0x039F1AACAF0DECFCULL, "100311459.ogg"},
	{0x039F63850A0D5712ULL, "art\\vfx\\combat\\vfx_r_gn4_smo_g_lar_aoe_p.cess.ani"},
	{0x039F8ED9A845E795ULL, "6098822_m.fxe"},
	{0x03A051CE9D50395AULL, "vxdr_0_deeproads_fbp_0_0.vlm"},
	{0x03A0581ABB155EFDULL, "6121173_m.fxe"},
	{0x03A092AE5447E641ULL, "tint_override.gda"},
	{0x03A098A3A08B143BULL, "vxca_cave_18c_0_0.vlm"},
	{0x03A0FBEC1718A0DDULL, "hlac_ht_bartrand_n_3365_1z.dds"},
	{0x03A134D51B1707A6ULL, "vxlt_0_lowtown_byr_0_0.vlm"},
	{0x03A210AEE19C55FEULL, "art\\cinematics\\additional animation\\cs_femnobleiii_cinematicanim.gad"},
	{0x03A3810A327CEC09ULL, "zz_mer_debug6062750.cl"},
	{0x03A3BE2213DF78B0ULL, "vxlt_hangedman_n_2xn_0_0.vlm"},
	{0x03A3C0E219700977ULL, "3174314.ogg"},
	{0x03A404CF815F620CULL, "vxht_0_hightown_n_avq_0_0.vlm"},
	{0x03A434A4C7415928ULL, "dre100cr_pickpocket.utc"},
	{0x03A46BE5E4734951ULL, "w_bonepitmining_a2_142_0.mmh"},
	{0x03A477E5E4735DF2ULL, "w_bonepitmining_a2_142_0.mao"},
	{0x03A485E5E473750FULL, "w_bonepitmining_a2_142_0.msh"},
	{0x03A4A079D2692737ULL, "smoke_water_add.mao"},
	{0x03A4E21CCF3E3AA1ULL, "vxht_bartrand_n_6r8_0_0.vlm"},
	{0x03A4EEB8B803A884ULL, "art\\vfx\\combat\\vfx_r_gn4_stealth_wpn_c.impa.gad"},
	{0x03A58F19F6A6464CULL, "6111386_m.fxe"},
	{0x03A5E3767A5F228EULL, "w_bonepitdestroyed_a3_198_0.mmh"},
	{0x03A5EF767A5F3725ULL, "w_bonepitdestroyed_a3_198_0.mao"},
	{0x03A5FD767A5F4F7CULL, "w_bonepitdestroyed_a3_198_0.msh"},
	{0x03A78BD3C4BA0A64ULL, "vxw_bonepit_zz_0_0.vlm"},
	{0x03A7A4C2123B4199ULL, "786890948.ogg"},
	{0x03A83350CCEF2559ULL, "gen00fl_isabela6135659.cl"},
	{0x03A83C10BCFDCC73ULL, "gen00fl_sebastian6189006.cl"},
	{0x03A86E1DC1683838ULL, "lgt200ip_eau_tome02.utp"},
	{0x03A8856F6493E6FFULL, "dae100_viscount6101974.cl"},
	{0x03A890CBAA5E1BB0ULL, "146076095.ogg"},
	{0x03A9C8D6851215E6ULL, "gen_im_npc_arm_hum_cht_cota.uti"},
	{0x03AA55A5B692746FULL, "hlac_lt_0_lowtown_n_0_3x.dds"},
	{0x03AB252A3B51D7D0ULL, "6141549_m.fxe"},
	{0x03AB280F74041E2FULL, "1005817762.ogg"},
	{0x03AB8B1717C24C70ULL, "460566625.ogg"},
	{0x03ABD9201B5B295CULL, "hlac_ht_0_hightown_131070_0y.dds"},
	{0x03ABFDB8B809E162ULL, "art\\vfx\\combat\\vfx_r_gn4_stealth_wpn_c.impa.ani"},
	{0x03AC173E9397A18CULL, "1068858671.ogg"},
	{0x03AC8A625F1A207AULL, "6177843_m.fxe"},
	{0x03ACE0349A804578ULL, "vxw_bonepitmining_a2_8f_0_0.vlm"},
	{0x03AD006C4A2971AAULL, "289791914.ogg"},
	{0x03AD57DF752E177CULL, "art\\levels\\races\\proxy\\tilesets\\setint\\8co_basic\\4so_basic_3.mao"},
	{0x03AD64E4C523C283ULL, "109768444.ogg"},
	{0x03ADB9AAF345182EULL, "ht_brothel_n_3370.rml"},
	{0x03AE065B10866593ULL, "ht_0_hightown_a3_13_0.rml"},
	{0x03AE0F3B76F038FCULL, "535324543.ogg"},
	{0x03AE7BE980F950E8ULL, "zz_qun_debug6085732.cl"},
	{0x03AF16B8444E4647ULL, "6083994_m.fxe"},
	{0x03AF5D6E9E60D50CULL, "73393359.ogg"},
	{0x03B051B62DFA5C54ULL, "zz_ser_debug6091928.cl"},
	{0x03B0AFC2E38C7528ULL, "and111_lirene6044961.cl"},
	{0x03B0EC059EFCE810ULL, "ser210_witness6168174.cl"},
	{0x03B104309D912571ULL, "zz_approval_debug6003525.cl"},
	{0x03B13729A981565CULL, "llac_lt_0_lowtown_a2_n_2_0.dds"},
	{0x03B1519951F3C445ULL, "244521764.ogg"},
	{0x03B2069797310756ULL, "vxdr_primevaltomb_8v4_0_0.vlm"},
	{0x03B20E77C705C3C9ULL, "and100_anders6095878.cl"},
	{0x03B25B2096C6F64BULL, "one300_nuncio6084353.cl"},
	{0x03B27DB47EE2BAF6ULL, "gen_im_trs_bst_drag_gland_03.uti"},
	{0x03B2993E8F84B60AULL, "art\\cinematics\\_crit path\\prologue\\additional animations\\cs_hurlock_fdeath1.gad"},
	{0x03B2A6F7518C3B9AULL, "462758899.ogg"},
	{0x03B2CCC762B3E08FULL, "art\\vfx\\combat\\vfx_w_gn2_knbk_aoe_lar_p.impa.gad"},
	{0x03B303A8E48419BCULL, "6125208_m.fxe"},
	{0x03B31B1A2A95FC5EULL, "vxlt_0_lowtown_a2_ert_0_0.vlm"},
	{0x03B32CBB0FFB8B53ULL, "vxw_bonepitmining_a2_26g_0_0.vlm"},
	{0x03B33FE60727A7A6ULL, "6015087_f.fxe"},
	{0x03B34391FF5C3638ULL, "8763680.ogg"},
	{0x03B38FB26BE74A33ULL, "6016670_f.fxe"},
	{0x03B43E2E688BAED6ULL, "art\\characters\\creatures\\mubari\\animation\\eat_twitch.ani"},
	{0x03B458F6961DC645ULL, "gen_im_trs_gen_one_axe_r1.uti"},
	{0x03B4CACE24F7A385ULL, "37098996.ogg"},
	{0x03B57A34D1074759ULL, "6183797_m.fxe"},
	{0x03B58957F5A136B8ULL, "hlac_ht_chantry_2534_42y.dds"},
	{0x03B664875AD73C2DULL, "6117491_m.fxe"},
	{0x03B67B74EEA6DCC3ULL, "hlac_lt_0_lowtown_a2_n_c_-3_3x.dds"},
	{0x03B6D09B3A3999B5ULL, "follower_banter6169862.cl"},
	{0x03B73684F7876254ULL, "562759684.ogg"},
	{0x03B8867F6E9C6B85ULL, "6000802_f.fxe"},
	{0x03B8E1C762B8703CULL, "art\\vfx\\combat\\vfx_w_gn2_knbk_aoe_lar_p.impa.evt"},
	{0x03B92124A63F3822ULL, "w_woundedcoast_73_0.rml"},
	{0x03B92F3DBC265776ULL, "758612263.ogg"},
	{0x03B9350F36F7E138ULL, "6048762_f.fxe"},
	{0x03B9CD050A6A55F8ULL, "6155838_m.fxe"},
	{0x03BA7355D93502F9ULL, "266599275.ogg"},
	{0x03BA75F0C9AFC220ULL, "gen000fl_merrill_house.stg"},
	{0x03BAD111CF2CE3BDULL, "6103353_m.fxe"},
	{0x03BADF442C304E39ULL, "lt_0_lowtown_a2_52_0.anb"},
	{0x03BAF9C75EB73015ULL, "gen00fl_fenris6110812.cl"},
	{0x03BC60F9A8C9CB17ULL, "6137033_m.fxe"},
	{0x03BCAD70D15029D2ULL, "vxw_bonepitmining_a2_26b_0_0.vlm"},
	{0x03BD3588813C6A5AULL, "vxdr_0_deeproads_fhr_0_0.vlm"},
	{0x03BD803D0B6D37F7ULL, "506510954.ogg"},
	{0x03BD84CB4C99E4ECULL, "vxht_0_hightown_c68_0_0.vlm"},
	{0x03BE540C32FBC605ULL, "vxw_bonepitmining_a2_6q_0_0.vlm"},
	{0x03BF105F040828B9ULL, "6143970_f.fxe"},
	{0x03C0040FA63F2B40ULL, "vxlt_gallowscourtyard_n_65f_0_0.vlm"},
	{0x03C0A1DBF7CBA9B0ULL, "vxlt_hangedman_n_2bl_0_0.vlm"},
	{0x03C0CFAD73BF6241ULL, "art\\vfx\\combat\\vfx_glb_acid_proj_p.cess.ani"},
	{0x03C1131D7116E10AULL, "dae150_harshal6148854.cl"},
	{0x03C1462E689646C7ULL, "art\\characters\\creatures\\mubari\\animation\\eat_twitch.evt"},
	{0x03C16C647E06EFA3ULL, "hlac_ht_keep_3732_18x.dds"},
	{0x03C18CF71C8C14DEULL, "vxlt_0_lowtown_a2_euu_0_0.vlm"},
	{0x03C26CE5B010DCDFULL, "one200_gold_followers6120151.cl"},
	{0x03C2895040F8DDA1ULL, "6117107_m.fxe"},
	{0x03C2A8E10D6216C3ULL, "vxlt_0_lowtown_a3_n_c_f4x_0_0.vlm"},
	{0x03C2BBF14B34F0E1ULL, "6033322_m.fxe"},
	{0x03C2D47C67F4C298ULL, "vxca_caverns1_igm_0_0.vlm"},
	{0x03C332EC541E3496ULL, "vxht_keep_n_677_0_0.vlm"},
	{0x03C3DEE53715B513ULL, "6068067_m.fxe"},
	{0x03C3E625799ECD4CULL, "hlac_w_woundedcoast_262147_0y.dds"},
	{0x03C4400A20079B8EULL, "6061335_m.fxe"},
	{0x03C4A3FB34009649ULL, "vxlt_blackemporium_zl_0_0.vlm"},
	{0x03C4D5F1AFD0D93DULL, "489489928.ogg"},
	{0x03C587316176BD01ULL, "392717886.ogg"},
	{0x03C5EC76024AF164ULL, "gen00fl_anders6096618.cl"},
	{0x03C647B634BEE469ULL, "124506230.ogg"},
	{0x03C6743E8F9520F0ULL, "art\\cinematics\\_crit path\\prologue\\additional animations\\cs_hurlock_fdeath1.ani"},
	{0x03C69D149F8C68FCULL, "182625619.ogg"},
	{0x03C6E9C762C4BB55ULL, "art\\vfx\\combat\\vfx_w_gn2_knbk_aoe_lar_p.impa.ani"},
	{0x03C7DAAD73C593E3ULL, "art\\vfx\\combat\\vfx_glb_acid_proj_p.cess.gad"},
	{0x03C84CA3E94741D8ULL, "608203063.ogg"},
	{0x03C85981382CE917ULL, "vxw_bonepitdestroyed_a3_1r2_0_0.vlm"},
	{0x03C8672E689C9D10ULL, "art\\characters\\creatures\\mubari\\animation\\eat_twitch.gad"},
	{0x03C8AB18ED109F3EULL, "and100_anders6095985.cl"},
	{0x03C8CAB3A89719DFULL, "11968675.ogg"},
	{0x03C93F3E2CAEE05BULL, "330019779.ogg"},
	{0x03C97FE11000FF17ULL, "gen00fl_aveline6122069.cl"},
	{0x03C9F920B79257B5ULL, "bl_tri_f.mao"},
	{0x03C9FE151B66226DULL, "mag291_torpor6025981.cl"},
	{0x03CA0B24A64D8677ULL, "w_woundedcoast_73_0.msh"},
	{0x03CA1D24A64DA4DAULL, "w_woundedcoast_73_0.mao"},
	{0x03CA2124A64DABE9ULL, "w_woundedcoast_73_0.mmh"},
	{0x03CA546E80209001ULL, "hlac_w_blightlands_65537_0x.dds"},
	{0x03CA71638B8593D9ULL, "23693978.ogg"},
	{0x03CB3D2850E2A575ULL, "dae000_merchant_hm_mob6174221.cl"},
	{0x03CB99467DBB4F57ULL, "6082812_m.fxe"},
	{0x03CD3BEDDF9D8F1AULL, "fx_c_grad_dis.mao"},
	{0x03CDDF98CF6F13CBULL, "vxlt_0_lowtown_docks_bc8_0_0.vlm"},
	{0x03CE1B30B5B3E8FCULL, "6116800_m.fxe"},
	{0x03CE57BF4DAD4F34ULL, "hlac_ht_0_hightown_a2_65532_0z.dds"},
	{0x03CEB7AD73CB76C8ULL, "art\\vfx\\combat\\vfx_glb_acid_proj_p.cess.evt"},
	{0x03CEE81D70775331ULL, "907264014.ogg"},
	{0x03CF3AD88F4EB869ULL, "hlac_ca_cavern1_haunted_24426_2y.dds"},
	{0x03CF49B9AB60C730ULL, "cod_mgc_mana.utp"},
	{0x03CFFED40E822D5EULL, "vxlt_0_lowtown_a2_exv_0_0.vlm"},
	{0x03D01074F3E7BC53ULL, "vxw_bonepitmining_a2_28i_0_0.vlm"},
	{0x03D04EA432B9876CULL, "340521108.ogg"},
	{0x03D0587AA7603BF9ULL, "mag100_thrask6032048.cl"},
	{0x03D07B40E2B40643ULL, "w_blightlands_30_0.mao"},
	{0x03D07F40E2B40C90ULL, "w_blightlands_30_0.mmh"},
	{0x03D08D40E2B424AAULL, "w_blightlands_30_0.msh"},
	{0x03D128AE9BBCE524ULL, "lgt110_orz_harrowmont6108996.cl"},
	{0x03D1465959EADB18ULL, "vxca_caverns1_ijj_0_0.vlm"},
	{0x03D18C4CAB0CD77BULL, "1941608.ogg"},
	{0x03D199A1781A92BEULL, "vxht_mansion2b_2r2_0_1.vlm"},
	{0x03D2821AF2E72BF5ULL, "vxlt_gallowstemplar_f_5pa_0_0.vlm"},
	{0x03D29F4F3A0A161AULL, "lt_gallowscourtyard_30_0.rml"},
	{0x03D2E16A9A9C53E4ULL, "vxw_bonepit_ky_0_0.vlm"},
	{0x03D3232CCD34EA49ULL, "6029031_m.fxe"},
	{0x03D34B7577875244ULL, "6166014_m.fxe"},
	{0x03D3D6D24302B171ULL, "gen00fl_bethany6098055.cl"},
	{0x03D48BCEC2C71D3AULL, "art\\characters\\playercharacter\\humanmale\\animation\\exploration\\movement\\mov_rf_leanl.gad"},
	{0x03D49238F05A00C2ULL, "vxht_playershouse_n_1wa_0_0.vlm"},
	{0x03D4E4F67E8DCBCCULL, "hlac_lt_0_lowtown_docks_a3_n_-196610_6x.dds"},
	{0x03D527A85E72FAA6ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\0rw_cap.phy"},
	{0x03D5478B556BBFD4ULL, "531333280.ogg"},
	{0x03D5B7F6DD701617ULL, "578295991.ogg"},
	{0x03D5CE0DB8704B24ULL, "938927686.ogg"},
	{0x03D5D0293DB82CE6ULL, "and111_lirene6090210.cl"},
	{0x03D669778517A972ULL, "vxlt_0_lowtown_docks_a3_n_brz_0_0.vlm"},
	{0x03D758212B9AE604ULL, "vxlt_gallowscourtyard_n_6yv_0_0.vlm"},
	{0x03D75A8E57D1744FULL, "mag211_marethari6025148.cl"},
	{0x03D7C866EFE53957ULL, "6040577_m.fxe"},
	{0x03D8197304033E36ULL, "one300_zevran6089708.cl"},
	{0x03D88E9D89BD8C18ULL, "art\\levels\\races\\natural\\highlands\\foliage\\_textures\\plc_flowerwild_01_0_n.dds"},
	{0x03D8D7727976C55BULL, "art\\characters\\playercharacter\\elffemale\\ef_har_he8a.mmh"},
	{0x03D8F5727976F845ULL, "art\\characters\\playercharacter\\elffemale\\ef_har_he8a.msh"},
	{0x03D92C4D5E183B41ULL, "dae000_flemeth6000823.cl"},
	{0x03D957BC870F53A5ULL, "445419301.ogg"},
	{0x03D967AE581FAE50ULL, "6002976_m.fxe"},
	{0x03D987D0CD88FEEFULL, "vxca_caverns1_mines_j1w_0_1.vlm"},
	{0x03D9912AB53C5AD2ULL, "vxw_bonepitmining_a2_28l_0_0.vlm"},
	{0x03D9CA925F17EEBFULL, "vxht_0_hightown_c34_0_0.vlm"},
	{0x03D9CD66A8B31E9AULL, "hlac_lt_0_lowtown_a2_n_c_65534_1x.dds"},
	{0x03D9DC52F1A94432ULL, "one100dg_kelder6056894.cl"},
	{0x03D9FEE8D7AA1CE9ULL, "109555343.ogg"},
	{0x03DA7267CC9DD01FULL, "vxlt_gallowstemplar_3nl_0_0.vlm"},
	{0x03DACC7201E358D4ULL, "6217114_m.fxe"},
	{0x03DB5D5EA5481C26ULL, "132187802.ogg"},
	{0x03DB65B1CB1BA396ULL, "6132794_m.fxe"},
	{0x03DB8ECEC2CD4151ULL, "art\\characters\\playercharacter\\humanmale\\animation\\exploration\\movement\\mov_rf_leanl.evt"},
	{0x03DBC28B679CD356ULL, "171625570.ogg"},
	{0x03DBE93559E1E548ULL, "gen00fl_aveline6137585.cl"},
	{0x03DC1471FB91A2A1ULL, "6147779_m.fxe"},
	{0x03DC24CB4DF38FECULL, "344809040.ogg"},
	{0x03DC71EC8C3F91E6ULL, "isa300ip_warehouse_exit.utp"},
	{0x03DCC948F307D81FULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_generalwood_n.dds"},
	{0x03DD24D95B7B841DULL, "vxlt_0_lowtown_a2_e16_0_0.vlm"},
	{0x03DD598C50691C57ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\baseswordshield\\combat\\combatrun.gad"},
	{0x03DDDC5E80BD768AULL, "hlac_lt_hangedman_-1_21z.dds"},
	{0x03DE540E8483C632ULL, "vxht_0_hightown_apr_0_0.vlm"},
	{0x03DE9B7C9FAED58DULL, "107154232.ogg"},
	{0x03DF87EE859D3B53ULL, "54224187.ogg"},
	{0x03DFB8364BE0F398ULL, "vxca_caverns1_imk_0_0.vlm"},
	{0x03DFC75E4EDA4CD5ULL, "hlac_lt_0_lowtown_a2_n_-131071_2y.dds"},
	{0x03DFE06F731A1742ULL, "527720795.ogg"},
	{0x03E091334FC91A3CULL, "ht_0_hightown_n_34_0.anb"},
	{0x03E09C21EAC0D5E5ULL, "6201480_m.fxe"},
	{0x03E0D9C62F190CA4ULL, "gen00fl_anders6103909.cl"},
	{0x03E165A6DCF7BE76ULL, "vxw_sundermount_28a_0_0.vlm"},
	{0x03E19028C09F7ECAULL, "art\\vfx\\combat\\vfx_m_sta_ele_close_c.anb"},
	{0x03E1B91A068ED380ULL, "125629908.ogg"},
	{0x03E1C6E57CDF1FF1ULL, "vxca_cavern1_haunted_jgb_0_0.vlm"},
	{0x03E233CEB6D32BCBULL, "6161603_m.fxe"},
	{0x03E28570F1B72AB4ULL, "one200ip_flower.ncs"},
	{0x03E3AC9512A42575ULL, "dae000_mother6121014.cl"},
	{0x03E43C1AFFFB2C9EULL, "art\\vfx\\combat\\vfx_m_sta_atk_l_dwn_c.impa.ani"},
	{0x03E46E8C506F5EE4ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\baseswordshield\\combat\\combatrun.evt"},
	{0x03E516AECC56AB4FULL, "hlac_w_woundedcoast_-196604_0y.dds"},
	{0x03E5A3895B93838EULL, "zz_dae_debug6047608.cl"},
	{0x03E5A3D9D86F8946ULL, "6031751_f.fxe"},
	{0x03E5E6B4360056AEULL, "6069439_f.fxe"},
	{0x03E69C624BB3A6B6ULL, "6162773_m.fxe"},
	{0x03E6EE96A380264BULL, "ave200_donnic6110513.cl"},
	{0x03E713BEC4AE1FC9ULL, "6072563_m.fxe"},
	{0x03E7526161917727ULL, "759605591.ogg"},
	{0x03E834C408F2DF60ULL, "6027708_m.fxe"},
	{0x03E83C6F510E073FULL, "vxht_0_hightown_c67_0_0.vlm"},
	{0x03E8A6CEC2D7F420ULL, "art\\characters\\playercharacter\\humanmale\\animation\\exploration\\movement\\mov_rf_leanl.ani"},
	{0x03E8C20B2FBC0D5BULL, "hlac_lt_0_lowtown_docks_n_-65538_8z.dds"},
	{0x03E8CE621C561680ULL, "ht_0_hightown_33_0_ll.mmh"},
	{0x03E90B55E4BBBB75ULL, "6166583_m.fxe"},
	{0x03E91DFF99B4F8D9ULL, "vxht_0_hightown_n_c5o_0_0.vlm"},
	{0x03E976F17A5211D1ULL, "stone_s_rock.mao"},
	{0x03E99D85A0A271F4ULL, "542484696.ogg"},
	{0x03E9C405E2E61B2DULL, "vxht_0_hightown_n_ai9_0_0.vlm"},
	{0x03E9CFF764842FBFULL, "682375734.ogg"},
	{0x03E9D3399DBA0FADULL, "vxdr_primevalentrance_19s_0_0.vlm"},
	{0x03EA38AB4EE513AFULL, "336041127.ogg"},
	{0x03EB7891AE747034ULL, "vxht_bartrand_n_65t_0_0.vlm"},
	{0x03EBEF8EAAC27AF9ULL, "6037599_m.fxe"},
	{0x03EBF482F0454CDFULL, "llac_lt_0_lowtown_a3_n_c_196607_1.dds"},
	{0x03EC00C1A614010FULL, "6174797_m.fxe"},
	{0x03EC7613B6C10510ULL, "998582898.ogg"},
	{0x03EC94F5A0B3B3D3ULL, "6071681_m.fxe"},
	{0x03ECCF4122D5AB32ULL, "vxca_caverns2_9fw_0_0.vlm"},
	{0x03ECE28DF26E5E5EULL, "vxlt_0_lowtown_a2_efh_0_0.vlm"},
	{0x03ED2F71ACA8DC2AULL, "6098726_m.fxe"},
	{0x03ED35AED938001DULL, "vxw_encounter3_e1_0_0.vlm"},
	{0x03EDD766ADC0BA24ULL, "68250502.ogg"},
	{0x03EE8F83CA3B4752ULL, "follower_banter6161067.cl"},
	{0x03EECF0030224F04ULL, "vxdr_0_deeproads_f5o_0_0.vlm"},
	{0x03EEEB026C12DBD5ULL, "llac_w_encounter2_1_0.dds"},
	{0x03EF10684A702E35ULL, "270682253.ogg"},
	{0x03EF9D783252AAAAULL, "6109028_m.fxe"},
	{0x03EFF99209E2DFC9ULL, "vxlt_blackemporium_uq_0_0.vlm"},
	{0x03F0123D895215A4ULL, "vxht_keep_n_67u_0_0.vlm"},
	{0x03F071D10CC8D3EDULL, "vxlt_undercitydungeon_130_0_0.vlm"},
	{0x03F137A940FA66BBULL, "hlac_lt_gallowscourtyard_-131072_4z.dds"},
	{0x03F1ED35BA275DF4ULL, "hlac_ht_0_hightown_a2_65535_1x.dds"},
	{0x03F2541B0007936FULL, "art\\vfx\\combat\\vfx_m_sta_atk_l_dwn_c.impa.evt"},
	{0x03F2868C507BC4DDULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\baseswordshield\\combat\\combatrun.ani"},
	{0x03F2B0540989ABADULL, "6077010_m.fxe"},
	{0x03F30644BBEDC81CULL, "isa311_isabela6092498.cl"},
	{0x03F30EB19032A377ULL, "hlac_lt_0_lowtown_docks_a3_n_-131074_0x.dds"},
	{0x03F35467C616265CULL, "vxw_woundedcoast_1kq_0_0.vlm"},
	{0x03F38A2803BB0BD8ULL, "312160025.ogg"},
	{0x03F3BB5438292C52ULL, "follower_barks6153344.cl"},
	{0x03F449876A49ACBEULL, "263827842.ogg"},
	{0x03F44E5156E2FD7AULL, "lowtown_fem_prostitute_sta6152367.cl"},
	{0x03F4B09E2CB120D9ULL, "215284487.ogg"},
	{0x03F4CA5C2BF85D8FULL, "lt_0_lowtown_docks_31_0.anb"},
	{0x03F4DE42F720EC8FULL, "zz_isa_debug6133930.cl"},
	{0x03F4E0442C61C839ULL, "lt_0_lowtown_a2_52_0.rml"},
	{0x03F50E3F3DA9BF36ULL, "6167473_m.fxe"},
	{0x03F51CB7208B1894ULL, "vxw_encounter1_c4_0_0.vlm"},
	{0x03F5774A0690EA1DULL, "lt_0_lowtown_a3_n_c_44_0_lc.mmh"},
	{0x03F641889FE318ABULL, "matt_warrior_2.utc"},
	{0x03F6B7CD64F5CC65ULL, "vxht_0_hightown_n_7g5_0_0.vlm"},
	{0x03F77EB3175A3698ULL, "vxlt_warehouse_q8_0_0.vlm"},
	{0x03F784D4ECE37097ULL, "gen00fl_fenris6127117.cl"},
	{0x03F78FDC8BAB1159ULL, "vxht_0_hightown_n_c2n_0_0.vlm"},
	{0x03F82DBC2471FE72ULL, "art\\characters\\creatures\\crowds\\humanmale\\textures\\prp_atlas_jimmy_0s.dds"},
	{0x03F8551B000C0018ULL, "art\\vfx\\combat\\vfx_m_sta_atk_l_dwn_c.impa.gad"},
	{0x03F8C0B9CCDCB938ULL, "art\\characters\\playercharacter\\elfmale\\textures\\em_arm_flsa_0d.dds"},
	{0x03F8F700DC0BF690ULL, "520611234.ogg"},
	{0x03F909AAD653292AULL, "vxht_keep_55q_0_0.vlm"},
	{0x03F9529CAE64936EULL, "dae000_mother6099190.cl"},
	{0x03FA1F3C69F16AB3ULL, "49115056.ogg"},
	{0x03FA24D5DF24D875ULL, "lgt110_arl_bann_teagan6115626.cl"},
	{0x03FA3ADDD204223FULL, "vxlt_0_lowtown_docks_aln_0_0.vlm"},
	{0x03FA6470F1CB84E5ULL, "one200ip_flower.utp"},
	{0x03FB64A19CF42475ULL, "llac_w_encounter3_-131068_0.dds"},
	{0x03FC62DD152B7A5AULL, "hlac_lt_0_lowtown_a2_n_c_65535_2z.dds"},
	{0x03FC8B1E74A59A88ULL, "403448814.ogg"},
	{0x03FCBC8EA9775D00ULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_daggerb_n.dds"},
	{0x03FCC64EEC9E9836ULL, "vxht_bartrand_n_6qz_0_0.vlm"},
	{0x03FD49424F90D23AULL, "6119274_m.fxe"},
	{0x03FD844D865117F7ULL, "character_character_p178.psh.fxo.dx11"},
	{0x03FDB53F7A9259DEULL, "vxht_keep_n_6go_0_0.vlm"},
	{0x03FDD7B1C8C97575ULL, "vxlt_gallowstemplar_f_5an_0_0.vlm"},
	{0x03FE841A7B482E24ULL, "vxht_keep_n_62z_0_0.vlm"},
	{0x03FEFD58745622B5ULL, "vxca_caverns1_jfx_0_0.vlm"},
	{0x03FF7B3B8D550223ULL, "124543312.ogg"},
	{0x03FFEC23B95346DEULL, "llac_dr_0_deeproads_17039_2.dds"},
	{0x04001BD964F0E93FULL, "vxlt_undercitydungeon_1ie_0_1.vlm"},
	{0x0400A49E4B03E788ULL, "440696593.ogg"},
	{0x0400E3A3E2984BEBULL, "art\\vfx\\combat\\vfx_f_mer_emove_act_c.impa.ani"},
	{0x0401143B35884834ULL, "vxht_mansion2b_2k2_0_0.vlm"},
	{0x0401E97557F2F84DULL, "art\\characters\\creatures\\qunari\\textures\\qm_hlm_com_0s.dds"},
	{0x0401EDD5CCDE8F76ULL, "w_encounter2_60_0.msh"},
	{0x0401F7D5CCDEA074ULL, "w_encounter2_60_0.mmh"},
	{0x0401FBD5CCDEA687ULL, "w_encounter2_60_0.mao"},
	{0x0402320B58FF2D46ULL, "823997369.ogg"},
	{0x04023911F137C663ULL, "gen00fl_sebastian6179799.cl"},
	{0x040256CE43D4427BULL, "199721422.ogg"},
	{0x040266DF879A1FC1ULL, "ser221ip_door_west.utp"},
	{0x04028C3A7B921441ULL, "nat380_temmerin6090888.cl"},
	{0x0402B24E38248D38ULL, "231802145.ogg"},
	{0x0402EE8582ABE5C2ULL, "6096809_m.fxe"},
	{0x04038E9412813114ULL, "vxw_encounter1_h5_0_0.vlm"},
	{0x04048AD20BE267E4ULL, "33835233.ogg"},
	{0x04048D94542E7D20ULL, "art\\cinematics\\_crit path\\mage crisis\\lowtown\\textures\\windowglowcombined_s.dds"},
	{0x0404C869ECFC5677ULL, "109017843.ogg"},
	{0x0404E674D383EE65ULL, "vxdr_0_deeproads_fiv_0_0.vlm"},
	{0x04050B0A24BB94A4ULL, "llac_dr_dungeon_3207_8.dds"},
	{0x0405349D513510F7ULL, "6104552_m.fxe"},
	{0x040596C6434FA9E9ULL, "55898743.ogg"},
	{0x0405A430A5837EE8ULL, "1009756150.ogg"},
	{0x0405F09009504F18ULL, "vxlt_warehouse_z9_0_0.vlm"},
	{0x0405FE26A956FEE9ULL, "lt_backalley_g_0_0.phy"},
	{0x0406B99E0C924CDBULL, "6110715_m.fxe"},
	{0x0406DCF586DBB1ECULL, "hand2.cur"},
	{0x04072D6652FFA503ULL, "209274311.ogg"},
	{0x040770ABF091B164ULL, "vxdr_primevaltomb_5af_0_0.vlm"},
	{0x040885AFD7BF7647ULL, "59911015.ogg"},
	{0x0408ACBAC3FA3ABFULL, "vxlt_0_lowtown_docks_aqi_0_0.vlm"},
	{0x0408D8C76E3CCC96ULL, "art\\vfx\\combat\\vfx_glb_spi_imp_dbl_c.anb"},
	{0x04090AF42A645B6AULL, "gen_im_trs_dwv_med_glv_r0.uti"},
	{0x04094B9092876887ULL, "dae000_pc_home_ambients6208632.cl"},
	{0x0409C647D65A8F5EULL, "vxlt_0_lowtown_a2_elj_0_0.vlm"},
	{0x040A090C7A411AD7ULL, "art\\vfx\\combat\\vfx_w_gn2_knbk_c.cess.gad"},
	{0x040A7F40E2E58553ULL, "w_blightlands_30_0.rml"},
	{0x040A919A3CCC8461ULL, "hlac_lt_0_lowtown_docks_n_131071_1x.dds"},
	{0x040A9328C0C288BDULL, "art\\vfx\\combat\\vfx_m_sta_ele_close_c.mmh"},
	{0x040AA9D96C98FCA5ULL, "art\\characters\\weapons\\maces\\w_mce_mc107.mao"},
	{0x040C26A3CC95D1E4ULL, "llac_lt_0_lowtown_docks_-4_1.dds"},
	{0x040CDFBD5589FBC0ULL, "zz_mer_debug6017858.cl"},
	{0x040CE88D24F615C2ULL, "llac_lt_0_lowtown_n_-65534_0.dds"},
	{0x040D0726A95D2CE5ULL, "lt_backalley_g_0_0.rml"},
	{0x040DCBA3E2A2AD62ULL, "art\\vfx\\combat\\vfx_f_mer_emove_act_c.impa.evt"},
	{0x040DF2131549EC79ULL, "llac_lt_gallowstemplar_f_2388_3.dds"},
	{0x040E0434C6C5664DULL, "vxw_bonepitdestroyed_a3_278_0_0.vlm"},
	{0x040E52FE27570C75ULL, "936784495.ogg"},
	{0x040E63ABBE44B18EULL, "73206073.ogg"},
	{0x040E66ED2FFA477BULL, "art\\characters\\playercharacter\\elfmale\\animation\\exploration\\movement\\mov_rf_leanr.evt"},
	{0x040E70BF6F343017ULL, "vxdr_primevaltomb_8o2_0_0.vlm"},
	{0x040E77DB1A8FAAD5ULL, "vxlt_0_lowtown_docks_a3_n_ctn_0_0.vlm"},
	{0x040E8C6419E6ECF1ULL, "hlac_lt_0_lowtown_-3_6z.dds"},
	{0x040EB6278F427C39ULL, "494588162.ogg"},
	{0x040FF0E861229613ULL, "gen00fl_sebastian6179889.cl"},
	{0x041030EB4CF00B72ULL, "vxlt_0_lowtown_docks_a3_n_bff_0_0.vlm"},
	{0x04103821AA02575CULL, "vxw_woundedcoast_1qw_0_0.vlm"},
	{0x041040E79C135DDDULL, "combat_lmm_r1_a3.gda"},
	{0x0410A229103B3C5BULL, "vxht_brothel_n_75b_0_0.vlm"},
	{0x0410B216049403ABULL, "gen00fl_isabela6096796.cl"},
	{0x0410B49C30B48E40ULL, "273684510.ogg"},
	{0x0410C6690D0D973FULL, "w_encounter1_32_0.phy"},
	{0x0410E1440FCBC670ULL, "vxlt_0_lowtown_a2_gvd_0_0.vlm"},
	{0x04111E0C7A475D64ULL, "art\\vfx\\combat\\vfx_w_gn2_knbk_c.cess.evt"},
	{0x04112FCF1CEA9050ULL, "vxht_keep_7aa_0_0.vlm"},
	{0x04115A4F66A68B39ULL, "852184223.ogg"},
	{0x04115D0762E74002ULL, "84585901.ogg"},
	{0x04119A4BCC2D6DCAULL, "mer210_coterie_contact6036771.cl"},
	{0x0411EA99B3E84BD9ULL, "art\\characters\\creatures\\ashwraith\\animation\\ras_electr.gad"},
	{0x0411F69F7D66856DULL, "cod_hst_orlais.utp"},
	{0x041204411E4A5A4AULL, "377156371.ogg"},
	{0x041264226BFC9975ULL, "681291877.ogg"},
	{0x041273A2E7DA5DD5ULL, "art\\characters\\playercharacter\\humanmale\\textures\\hm_hlm_medj_0s.dds"},
	{0x0412B76ED1D83426ULL, "6174536_m.fxe"},
	{0x0412DC64D869F209ULL, "6119509_m.fxe"},
	{0x0413974C1F814CABULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\baseswordshield\\combat\\meleeattackv3.ani"},
	{0x0413D8E7F62AE6A7ULL, "6167560_m.fxe"},
	{0x04144860C0BDDBAEULL, "55759269.ogg"},
	{0x0414CAA3E2A8CA75ULL, "art\\vfx\\combat\\vfx_f_mer_emove_act_c.impa.gad"},
	{0x0414CFC10ADDA68DULL, "895667437.ogg"},
	{0x04153B1EDC2F4E59ULL, "48373123.ogg"},
	{0x041553FE6C8CCD87ULL, "plt_gen00pt_generic_actions.plo"},
	{0x04155FED300059BCULL, "art\\characters\\playercharacter\\elfmale\\animation\\exploration\\movement\\mov_rf_leanr.gad"},
	{0x04162E1919445F59ULL, "vxht_brothel_n_59r_0_0.vlm"},
	{0x0416ABD88D2082FBULL, "100849290.ogg"},
	{0x0416AFE2C5F28861ULL, "6048445_m.fxe"},
	{0x0416B6A3C228009CULL, "6095845_f.fxe"},
	{0x0416DBC76E490F01ULL, "art\\vfx\\combat\\vfx_glb_spi_imp_dbl_c.mmh"},
	{0x0418062935E5E9ABULL, "6071898_m.fxe"},
	{0x0418119A031BC7D1ULL, "252762106.ogg"},
	{0x041816E77BFD2697ULL, "gen00fl_anders6125690.cl"},
	{0x041883C52065BC45ULL, "6033717_m.fxe"},
	{0x0418E03CF81411C2ULL, "plt_qun100pt_mage_escort.plo"},
	{0x0418F457F9FAFC8BULL, "705361302.ogg"},
	{0x0418FB99B3EE8727ULL, "art\\characters\\creatures\\ashwraith\\animation\\ras_electr.ani"},
	{0x0419373F5CA6B5A5ULL, "one220_dera6099729.cl"},
	{0x0419761C3F6CBB8AULL, "vxlt_0_lowtown_a2_n_c_h24_0_0.vlm"},
	{0x0419EF1ABD362B22ULL, "zz_dae_debug6029921.cl"},
	{0x041A4BF5473469E9ULL, "mag110_arianni6016826.cl"},
	{0x041A5BB84A800696ULL, "hlac_ht_mansion_n_3373_3y.dds"},
	{0x041B81B38FC3E398ULL, "static_static_p27.psh.fxo.dx11"},
	{0x041B866A1FBB9218ULL, "37455165.ogg"},
	{0x041B9F845C34E0CEULL, "6078474_m.fxe"},
	{0x041BDD3D9DB5B5A0ULL, "tools_charpost.phy"},
	{0x041BE6B85DEED52FULL, "art\\characters\\creatures\\dragon\\textures\\cn_eye_drga_0d.dds"},
	{0x041C5EED300676BAULL, "art\\characters\\playercharacter\\elfmale\\animation\\exploration\\movement\\mov_rf_leanr.ani"},
	{0x041C6487FE976B61ULL, "vxw_encounter2_7t_0_0.vlm"},
	{0x041C900B410CFAAEULL, "vxht_0_hightown_cb6_0_0.vlm"},
	{0x041CF707B54C923CULL, "hlac_ca_caverns2prc_5802_52y.dds"},
	{0x041D23ED5912C581ULL, "6084235_f.fxe"},
	{0x041D48ED94FA0248ULL, "345223630.ogg"},
	{0x041D4F6878BE5663ULL, "lava0.vsh.fxo"},
	{0x041DB65A5FAF4D1DULL, "atl_demosplash_dxt5_00.dds"},
	{0x041DF7F519747934ULL, "vxht_mansion2b_2a0_0_0.vlm"},
	{0x041E0E633188D842ULL, "6032000_f.fxe"},
	{0x041E113BDFB88E4DULL, "follower_banter6161291.cl"},
	{0x041E6151EEA041D7ULL, "6061795_f.fxe"},
	{0x041EAC21016F5563ULL, "6118377_m.fxe"},
	{0x041F360C7A53C35DULL, "art\\vfx\\combat\\vfx_w_gn2_knbk_c.cess.ani"},
	{0x0420275A05AA6E9AULL, "303610733.ogg"},
	{0x04202821F3B4BB1FULL, "6144941_m.fxe"},
	{0x04207F2DA02BACBCULL, "484631275.ogg"},
	{0x042080871E1041A2ULL, "6129596_m.fxe"},
	{0x0420A75C2C1D5D33ULL, "lt_0_lowtown_docks_31_0.rml"},
	{0x0420C18D7FDAE21CULL, "vxw_woundedcoast_hy_0_0.vlm"},
	{0x0420C89413B0AAADULL, "ave121_jeven6105631.cl"},
	{0x0420F87279B4B290ULL, "art\\characters\\playercharacter\\elffemale\\ef_har_he8a.phy"},
	{0x04217F4C1F8D6122ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\baseswordshield\\combat\\meleeattackv3.evt"},
	{0x042190335000AFCCULL, "ht_0_hightown_n_34_0.rml"},
	{0x0421B3C16841600CULL, "w_woundedcoast_254_0.msh"},
	{0x0421B9C168416AFEULL, "w_woundedcoast_254_0.mmh"},
	{0x0421C5C168417E95ULL, "w_woundedcoast_254_0.mao"},
	{0x0421C7690D1C0C44ULL, "w_encounter1_32_0.mmh"},
	{0x0421CB690D1C1297ULL, "w_encounter1_32_0.mao"},
	{0x0421D917DB0A4D94ULL, "6198675_m.fxe"},
	{0x0421DD690D1C31E6ULL, "w_encounter1_32_0.msh"},
	{0x04222C8E94578C5EULL, "6041904_m.fxe"},
	{0x0422314DB1CE6BECULL, "953888856.ogg"},
	{0x0422788697E52552ULL, "face0_s2.psh.fxo"},
	{0x042311B2E122DD41ULL, "162459204.ogg"},
	{0x04235098F4AAD0E3ULL, "w_bonepit_20_0.mao"},
	{0x04235498F4AAD730ULL, "w_bonepit_20_0.mmh"},
	{0x04236003F1EA8456ULL, "vxlt_0_lowtown_df1_0_0.vlm"},
	{0x04236298F4AAEF4AULL, "w_bonepit_20_0.msh"},
	{0x0423D87FEBE51429ULL, "hlac_ht_0_hightown_131072_0z.dds"},
	{0x0423EE04667DA8D9ULL, "vxht_0_hightown_a2_c7z_0_0.vlm"},
	{0x0423F688C6C2DA06ULL, "dae000_worthy6166616.cl"},
	{0x0424036916E5FC7CULL, "1028937705.ogg"},
	{0x042459FC80D12567ULL, "6054640_m.fxe"},
	{0x0424806A61302628ULL, "vxlt_gallowstemplar_f_5r2_0_0.vlm"},
	{0x04249817EAEDCDB5ULL, "mrl000_marethari6129385.cl"},
	{0x0424AC5B6D402B0EULL, "493262192.ogg"},
	{0x0424FBE9F94DE98FULL, "6034325_m.fxe"},
	{0x04250F32B9D17B6BULL, "6144431_m.fxe"},
	{0x04254A792EFA57F1ULL, "6129447_m.fxe"},
	{0x0425880E2E1A75DBULL, "vxca_caverns1_jgu_0_0.vlm"},
	{0x0425ED3DBFEC1748ULL, "vxw_encounter2_7q_0_0.vlm"},
	{0x04262AE99E006336ULL, "and231im_alrik_papers.uti"},
	{0x0426AF3F26E94FA2ULL, "689504468.ogg"},
	{0x0426C82A4C1D2E67ULL, "art\\characters\\creatures\\crowds\\humanfemale\\crd_ferelden.mao"},
	{0x0426D2B88AF85DA9ULL, "230360171.ogg"},
	{0x0426E5388AAE6824ULL, "vrc321ip_spooky5_obj_6.utp"},
	{0x04277ADB7E411A53ULL, "62975446.ogg"},
	{0x042803739B2D2BB8ULL, "fen210st_tevniter_ambusher.stg"},
	{0x042826EB5D805820ULL, "48062705.ogg"},
	{0x042850F48FF06CE4ULL, "729684778.ogg"},
	{0x04286992177D06B2ULL, "aip_circlemage_base.gda"},
	{0x04286FEF8AC68AE8ULL, "6114416_f.fxe"},
	{0x04287E4C1F937E35ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\baseswordshield\\combat\\meleeattackv3.gad"},
	{0x042896EC1E7B0BDCULL, "dae000_adriano6163672.cl"},
	{0x042926A48DF7CA4AULL, "vxw_bonepitdestroyed_a3_73_0_0.vlm"},
	{0x04292BF2DDB7493BULL, "and111st_donation_box.stg"},
	{0x04294D753F503FD1ULL, "mag291cr_crowd_elf_1.utc"},
	{0x042A04A16A1AB94DULL, "hlac_ht_keep_3732_11y.dds"},
	{0x042A3B9B635A45F7ULL, "6111468_m.fxe"},
	{0x042A73338BE1F9CDULL, "hlac_lt_0_lowtown_docks_-4_0x.dds"},
	{0x042A79912CA167BBULL, "vxlt_0_lowtown_a2_n_c_hd7_0_0.vlm"},
	{0x042ACF100F806ACDULL, "hlac_dr_primevaltomb_663_2x.dds"},
	{0x042AD664F08D83E1ULL, "vxw_encounter2_2k_0_0.vlm"},
	{0x042B1189F81A9816ULL, "qun120_arishok6052909.cl"},
	{0x042B5B94FE7BDBD5ULL, "vxlt_0_lowtown_docks_a3_n_czl_0_0.vlm"},
	{0x042B5D5B13D31A69ULL, "204847640.ogg"},
	{0x042B5F72282F2076ULL, "art\\levels\\races\\natural\\proxy\\props\\prp_zzrocklarge.phy"},
	{0x042C56924621B2DFULL, "78104986.ogg"},
	{0x042C600D23925282ULL, "art\\characters\\heraldry\\hld_city_guard.dds"},
	{0x042D1A0214F9714EULL, "6105558_m.fxe"},
	{0x042D1DFDF3656DE3ULL, "6117277_m.fxe"},
	{0x042D693D5C40FDB1ULL, "6019105_f.fxe"},
	{0x042DBB7698504D36ULL, "art\\levels\\races\\ferelden\\props\\mining\\prp_pickaxe.msh"},
	{0x042DC57698505E34ULL, "art\\levels\\races\\ferelden\\props\\mining\\prp_pickaxe.mmh"},
	{0x042DC97698506547ULL, "art\\levels\\races\\ferelden\\props\\mining\\prp_pickaxe.mao"},
	{0x042E8860371BD474ULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\h_crn_rdy.gad"},
	{0x042EB496F2E9AC78ULL, "art\\characters\\weapons\\dagger\\textures\\w_dgr_105a_sa.dds"},
	{0x042EC30EF41E611CULL, "follower_banter6147410.cl"},
	{0x042F1E33D1A79782ULL, "dre000_dougal6131155.cl"},
	{0x042F3D90C9AEAD4FULL, "hlac_lt_blackemporium_1055_6y.dds"},
	{0x04307798F4B59D4FULL, "w_bonepit_20_0.anb"},
	{0x0430BED5F9D58846ULL, "art\\characters\\playercharacter\\qunarimale\\animation\\combat\\arishock\\ari_ability_impale_lp.ani"},
	{0x0430DE287B252EFAULL, "6102660_f.fxe"},
	{0x0430E4F0BF0082F6ULL, "fen110_fenris6097047.cl"},
	{0x043150991C029792ULL, "llac_ca_cavern1_haunted_24424_4.dds"},
	{0x0431698552A81735ULL, "llac_lt_darktown_-1_43.dds"},
	{0x04317269014947FDULL, "6174614_m.fxe"},
	{0x0431E726A97C120EULL, "lt_backalley_g_0_0.mmh"},
	{0x0431F326A97C26A5ULL, "lt_backalley_g_0_0.mao"},
	{0x0431FD569CB495ADULL, "vxht_0_hightown_n_av4_0_0.vlm"},
	{0x04320126A97C3EFCULL, "lt_backalley_g_0_0.msh"},
	{0x0432230200C155F8ULL, "6070212_m.fxe"},
	{0x04325FE15873C159ULL, "vxht_0_hightown_a2_c2y_0_0.vlm"},
	{0x0432959DA9F90F0FULL, "205098535.ogg"},
	{0x0432A5D23A8B828DULL, "gen00fl_merrill6128918.cl"},
	{0x0432D0E24D564B9EULL, "30185661.ogg"},
	{0x0433EFD768F70419ULL, "gen00fl_isabela6102455.cl"},
	{0x04345F1AB1E22FC8ULL, "vxw_encounter2_2n_0_0.vlm"},
	{0x04349A286A7470D7ULL, "215434413.ogg"},
	{0x0434A9A167B92CACULL, "vxht_chantry_n_147_0_0.vlm"},
	{0x0434DA3E31C85178ULL, "6027511_m.fxe"},
	{0x0434F2A4FD934FECULL, "fx_dd_streak.mao"},
	{0x04350F0923109A9BULL, "llac_ht_mansion2_818_1.dds"},
	{0x04351CF2C10EFC13ULL, "hlac_dr_0_deeproads_17038_2x.dds"},
	{0x0435487228374D17ULL, "art\\levels\\races\\natural\\proxy\\props\\prp_zzrocklarge.msh"},
	{0x04355A7228376C7AULL, "art\\levels\\races\\natural\\proxy\\props\\prp_zzrocklarge.mao"},
	{0x04355E7228377289ULL, "art\\levels\\races\\natural\\proxy\\props\\prp_zzrocklarge.mmh"},
	{0x0435727981F5A31FULL, "91770588.ogg"},
	{0x0435976037220C52ULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\h_crn_rdy.ani"},
	{0x0435A54FC86F7B31ULL, "116119439.ogg"},
	{0x0435E0944C0BE6B8ULL, "lt_0_lowtown_36_0_lc.mmh"},
	{0x0435E0D8F780F6DBULL, "w_bonepitdestroyed_a3_126_0.msh"},
	{0x0435F2D8F7811526ULL, "w_bonepitdestroyed_a3_126_0.mao"},
	{0x0435FED8F78129C5ULL, "w_bonepitdestroyed_a3_126_0.mmh"},
	{0x04360E964630BDA9ULL, "art\\vfx\\combat\\vfx_gib_ogre_bld_p.mmh"},
	{0x043625873327FA6AULL, "mrl251ip_loot_d.utp"},
	{0x043742D4CE6861DDULL, "6137245_f.fxe"},
	{0x0437D57BF14DC9E2ULL, "zz_and_debug6176464.cl"},
	{0x043833A182EDBA6DULL, "follower_banter6169879.cl"},
	{0x04392FBE3F80E053ULL, "gen_im_trs_chn_rng_r0.uti"},
	{0x043A1CEC39F19C54ULL, "6116552_m.fxe"},
	{0x043A25D33261D1D1ULL, "t3_wdg_dn1.tnt"},
	{0x043AF3A3C54E7B8AULL, "depth_speedtree.vsh.fxo"},
	{0x043AFEEE0318C69EULL, "art\\characters\\creatures\\shriek\\animation\\blendtree\\shriek_movement.anb"},
	{0x043B7B6BB2C35366ULL, "hlac_lt_darktown_-1_40z.dds"},
	{0x043BAD950FD3DD62ULL, "vxca_caverns2prc_afz_0_0.vlm"},
	{0x043C07ED14F9D171ULL, "vxlt_backalley_g_1eg_0_0.vlm"},
	{0x043C16BF6289C81FULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\0pp22_flared2a.phy"},
	{0x043C3066C5C70D27ULL, "521470572.ogg"},
	{0x043C6538688D59ADULL, "6156643_m.fxe"},
	{0x043C760155BC002EULL, "29562530.ogg"},
	{0x043CF9CE483913B9ULL, "vxw_bonepitmining_a2_21p_0_0.vlm"},
	{0x043DB6F0AD6631C5ULL, "525392193.ogg"},
	{0x043DE161CBA56EFDULL, "6033677_f.fxe"},
	{0x043E9A49EFAF47B5ULL, "vxdr_slavers_148_0_0.vlm"},
	{0x043EA6D5F9E19CB7ULL, "art\\characters\\playercharacter\\qunarimale\\animation\\combat\\arishock\\ari_ability_impale_lp.evt"},
	{0x043EE8296FBB1CF0ULL, "112412601.ogg"},
	{0x043F9CBB78FAE18EULL, "vxlt_undercityserialkiller_1id_0_0.vlm"},
	{0x043FC648E3FDA8C2ULL, "6126561_f.fxe"},
	{0x04400626A988840DULL, "lt_backalley_g_0_0.anb"},
	{0x044067BAE4ED6672ULL, "vxht_bartrand_n_5wg_0_0.vlm"},
	{0x044094DEF2B76E78ULL, "6075112_m.fxe"},
	{0x0440D1BE4A69D9D9ULL, "vxht_0_hightown_a2_c9x_0_0.vlm"},
	{0x0440ED03F5EAE5C7ULL, "zz_dae_debug6142957.cl"},
	{0x0440FA3D9DD50335ULL, "tools_charpost.msh"},
	{0x0440FC3D9DD506ABULL, "tools_charpost.mmh"},
	{0x0441846A069568CEULL, "gen00fl_aveline6137486.cl"},
	{0x0441A1C116C9E383ULL, "6121895_f.fxe"},
	{0x0441ECC6D084C4BFULL, "art\\characters\\playercharacter\\elfmale\\cth_comc.phy"},
	{0x0441FFAD4E4BA08EULL, "367230948.ogg"},
	{0x0442A92F6037ED4FULL, "491895825.ogg"},
	{0x0442D24FD9337958ULL, "6208672_f.fxe"},
	{0x04430683FAC85B73ULL, "6016174_m.fxe"},
	{0x04432453CE1C7CC8ULL, "vxw_blightlands_ax_0_0.vlm"},
	{0x04432E994EE7B453ULL, "co_starline.mao"},
	{0x044368AF544943E2ULL, "vxlt_0_lowtown_a2_n_c_hkw_0_0.vlm"},
	{0x044380A298AD92F2ULL, "art\\vfx\\combat\\vfx_gib_hlk_p.justfeet.imp.gad"},
	{0x0443C831BE42C986ULL, "129993528.ogg"},
	{0x044417F791DEF48DULL, "6127276_m.fxe"},
	{0x044433FB34766377ULL, "815842234.ogg"},
	{0x044462D4D1110CD0ULL, "623890158.ogg"},
	{0x04447FE41E27E7D9ULL, "6081341_m.fxe"},
	{0x0444D540E99B9F18ULL, "vxca_caverns1_i2b_0_0.vlm"},
	{0x0444D9DE4D206A6DULL, "vxlt_ruinedhouse_n_5b_0_0.vlm"},
	{0x0444DB24A3087DF3ULL, "6105638_m.fxe"},
	{0x0444DE7A95B6882DULL, "776518696.ogg"},
	{0x044531B8B1ACD3AAULL, "dae021_sister_alema_mob.cnv"},
	{0x04456A988C30B26DULL, "vxca_cavern1_haunted_e8p_0_0.vlm"},
	{0x044597D5815DFC55ULL, "788055769.ogg"},
	{0x0445A7D5F9E7BD40ULL, "art\\characters\\playercharacter\\qunarimale\\animation\\combat\\arishock\\ari_ability_impale_lp.gad"},
	{0x0445C89323AB20B3ULL, "6103683_m.fxe"},
	{0x0445CCA3A043EB9AULL, "6102335_m.fxe"},
	{0x04467B4E1A4A41D6ULL, "w_encounter3_211_0.mao"},
	{0x044683C079A4DECDULL, "219821552.ogg"},
	{0x0446874E1A4A5675ULL, "w_encounter3_211_0.mmh"},
	{0x0446894E1A4A59EBULL, "w_encounter3_211_0.msh"},
	{0x0446C32AA24C6969ULL, "hlac_w_encounter3_-262144_0z.dds"},
	{0x044785F6C50EF95EULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\rightarm_rainbow.ani"},
	{0x04479CE19544C4EBULL, "lla_ht_0_hightown_a3_-131073_0.dds"},
	{0x0447AF7EE9C5E4C3ULL, "hlac_lt_0_lowtown_docks_a3_n_-4_2z.dds"},
	{0x0447BA1ED479B4E1ULL, "vxw_encounter2_8i_0_0.vlm"},
	{0x0447BA8FAC3B4FD3ULL, "gen00fl_fenris6103326.cl"},
	{0x0447E2B2972CEBAFULL, "dae000_gamlen6149406.cl"},
	{0x0448138EB6E6D65BULL, "vxw_encounter2_camp_ov_0_0.vlm"},
	{0x04484423C4320640ULL, "hlac_ht_chantry_n_2534_4x.dds"},
	{0x0448574D98D8B659ULL, "fen110_anso6095026.cl"},
	{0x04485B69DD412C42ULL, "w_blightlands_31_0.rml"},
	{0x0448A1056BBE04C7ULL, "ser191_emeric6150905.cl"},
	{0x0448FC342A5B760CULL, "art\\levels\\races\\proxy\\props\\4tools\\inv_flat1x1_tool.msh"},
	{0x044902342A5B80FEULL, "art\\levels\\races\\proxy\\props\\4tools\\inv_flat1x1_tool.mmh"},
	{0x04490E342A5B9495ULL, "art\\levels\\races\\proxy\\props\\4tools\\inv_flat1x1_tool.mao"},
	{0x044913B8B1B089D2ULL, "dae021_sister_alema_mob.bnk"},
	{0x0449905C395601F3ULL, "vxca_caverns1_mines_ghl_0_0.vlm"},
	{0x0449BF0B075F129BULL, "364898384.ogg"},
	{0x0449F320B9528E12ULL, "gen_im_trs_rdr_two_2hs_r1.uti"},
	{0x044A4A7206659F40ULL, "6144322_m.fxe"},
	{0x044A7D8C5628ECE5ULL, "875637687.ogg"},
	{0x044AE7E0B0098CC4ULL, "331920600.ogg"},
	{0x044B5F5C4358DC5DULL, "266945119.ogg"},
	{0x044B65773E62DA3EULL, "994343206.ogg"},
	{0x044BAAC54FE2DDD2ULL, "combat_2h_mv_default.gda"},
	{0x044BEB6373C1F1EAULL, "glows.mao"},
	{0x044C2CC34D39468AULL, "dae000_pc_home_ambients6208693.cl"},
	{0x044CFFC922122830ULL, "hlac_lt_0_lowtown_docks_a3_-3_3x.dds"},
	{0x044D17BF62983D24ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\0pp22_flared2a.mmh"},
	{0x044D1BBF62984477ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\0pp22_flared2a.mao"},
	{0x044D2DBF629862C6ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\0pp22_flared2a.msh"},
	{0x044D79FB4A8477B5ULL, "vxht_0_hightown_n_an4_0_0.vlm"},
	{0x044E0E986AF0FA0EULL, "vxlt_undercityserialkiller_1ne_0_0.vlm"},
	{0x044E8D82A85866B5ULL, "projdecal_projdecal_p0.psh.fxo"},
	{0x044EA06FBB5B63CFULL, "6104923_m.fxe"},
	{0x044FD60137126FA8ULL, "vxlt_gallowstemplar_f_5c1_0_0.vlm"},
	{0x0450938AD2B4EBEEULL, "13484574.ogg"},
	{0x0450B55B2118A6CDULL, "845095204.ogg"},
	{0x0451138F78BE5C18ULL, "art\\levels\\4cinematics\\ferelden\\_textures\\prp_bookofinquisition_n.dds"},
	{0x045131456FDB310FULL, "zz_qcr_debug6058852.cl"},
	{0x045199AA1C9ED5B8ULL, "6136701_m.fxe"},
	{0x0451CB979E71CA09ULL, "920692722.ogg"},
	{0x0451CE380ACC615EULL, "532887636.ogg"},
	{0x0451E590329B8307ULL, "llac_ht_0_hightown_a3_-131073_1.dds"},
	{0x0452387826C760A1ULL, "vxht_bartrand_n_664_0_0.vlm"},
	{0x04528F6378E116C9ULL, "915052311.ogg"},
	{0x04529F1462685457ULL, "ts_raiders_prog_junk2.gda"},
	{0x0452EDC6D09339C4ULL, "art\\characters\\playercharacter\\elfmale\\cth_comc.mmh"},
	{0x0452F1C6D0934017ULL, "art\\characters\\playercharacter\\elfmale\\cth_comc.mao"},
	{0x045303C6D0935F66ULL, "art\\characters\\playercharacter\\elfmale\\cth_comc.msh"},
	{0x0453621C99921397ULL, "26864461.ogg"},
	{0x045375DD7CAFF2B4ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\prp_sandpilexa.phy"},
	{0x0453787BC421732FULL, "vxw_encounter1_ip_0_1.vlm"},
	{0x045399874D27FA44ULL, "6046838_f.fxe"},
	{0x045401623B0BA6F1ULL, "qcr221cr_qunari_honor_guard.utc"},
	{0x04549DF6C519AC2FULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\rightarm_rainbow.evt"},
	{0x0454C2565D42FB57ULL, "w_encounter2_camp_mmap.dds"},
	{0x0454DF701F3B4D5CULL, "194085175.ogg"},
	{0x04551622763D5BADULL, "6087013_m.fxe"},
	{0x045564BFC8AD044EULL, "709392334.ogg"},
	{0x0455A2F8DE1662A2ULL, "vxdr_primevaltomb_3ir_0_0.vlm"},
	{0x0455FA5F55FCF2DCULL, "sw_sh.mao"},
	{0x045632933C296A48ULL, "art\\levels\\races\\natural\\kirkcoastal\\foliage\\bsh_highthistle.mmh"},
	{0x04563E933C297EEBULL, "art\\levels\\races\\natural\\kirkcoastal\\foliage\\bsh_highthistle.mao"},
	{0x045650933C299D72ULL, "art\\levels\\races\\natural\\kirkcoastal\\foliage\\bsh_highthistle.msh"},
	{0x04567F57F0B401E1ULL, "vxw_blightlands_os_0_0.vlm"},
	{0x0456856BA8DCEEDBULL, "vxw_encounter2_camp_ji_0_0.vlm"},
	{0x045696D74CF15DE2ULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_pulpit_sa.dds"},
	{0x0456AE825B5B56ADULL, "hightown_fem_cityguard_mob6168528.cl"},
	{0x0456B65859DFC864ULL, "zz_followers_banter6159561.cl"},
	{0x0456C90FCA3A888FULL, "544282309.ogg"},
	{0x0456F7D5CD26E077ULL, "w_encounter2_60_0.rml"},
	{0x0457B5A439E6C1A8ULL, "qun160_vashothlead6053946.cl"},
	{0x0457E97ECD8E7D7BULL, "vxlt_0_lowtown_docks_b6j_0_1.vlm"},
	{0x045802392B4C1A73ULL, "vxca_caverns1_mines_geo_0_0.vlm"},
	{0x045818EF9B2B0616ULL, "6103400_m.fxe"},
	{0x04584CE9891CFA52ULL, "1003989707.ogg"},
	{0x04586BA298BFCA98ULL, "art\\vfx\\combat\\vfx_gib_hlk_p.justfeet.imp.ani"},
	{0x0458B401DAD248D3ULL, "gen00fl_aveline6114011.cl"},
	{0x0458FF8E4AA0ED94ULL, "vxlt_hangedman_21j_0_0.vlm"},
	{0x045940355B54DB21ULL, "270346430.ogg"},
	{0x04594569DD4F7997ULL, "w_blightlands_31_0.msh"},
	{0x04595769DD4F98FAULL, "w_blightlands_31_0.mao"},
	{0x04595B69DD4F9F09ULL, "w_blightlands_31_0.mmh"},
	{0x0459B9B483FE104FULL, "6173811_m.fxe"},
	{0x0459E8B57AA41198ULL, "hlac_w_woundedcoast_327685_0z.dds"},
	{0x045A25A10A1515B9ULL, "vxlt_0_lowtown_a2_n_fe2_0_0.vlm"},
	{0x045A5805EFA73074ULL, "hlac_lt_0_lowtown_docks_n_-196609_1x.dds"},
	{0x045A7F33834A72E9ULL, "568985712.ogg"},
	{0x045AF368CF0AFFAEULL, "103727455.ogg"},
	{0x045B1ECFDE2FF3BBULL, "vxlt_hangedman_n_209_0_0.vlm"},
	{0x045B27FFEBDD4D3AULL, "6088910_m.fxe"},
	{0x045B9EF6C51FCCD8ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\rightarm_rainbow.gad"},
	{0x045BE39A0E185F93ULL, "6188673_m.fxe"},
	{0x045C934D27F49A41ULL, "one220_yevhen6099030.cl"},
	{0x045CF1A890540FD0ULL, "365862714.ogg"},
	{0x045CF6D5CD2B4A6FULL, "w_encounter2_60_0.phy"},
	{0x045D137EA0DCAEFEULL, "6078796_f.fxe"},
	{0x045D52DD7CB80BB1ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\prp_sandpilexa.msh"},
	{0x045D5498F4DC4FF3ULL, "w_bonepit_20_0.rml"},
	{0x045D60DD7CB823CCULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\prp_sandpilexa.mao"},
	{0x045D6CDD7CB8386FULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\prp_sandpilexa.mmh"},
	{0x045DA51B3C103DD8ULL, "art\\levels\\races\\natural\\kirkcoastal\\foliage\\bush.mao"},
	{0x045E523A430C08CBULL, "vxlt_0_lowtown_docks_b52_0_0.vlm"},
	{0x045E8E230AFC5188ULL, "vxht_playershouse_n_1o3_0_0.vlm"},
	{0x045E9BB113B9BC70ULL, "6101826_m.fxe"},
	{0x045EB7C9536CCE1AULL, "811027582.ogg"},
	{0x045EC8FF7B267AD1ULL, "hlac_lt_0_lowtown_docks_196607_2x.dds"},
	{0x045F0B964653BDDEULL, "art\\vfx\\combat\\vfx_gib_ogre_bld_p.anb"},
	{0x045F20DE85C5B18BULL, "hlac_ca_caverns1_mines_24427_3y.dds"},
	{0x045F25B94846B54CULL, "gen00fl_anders6125273.cl"},
	{0x045F88449A035450ULL, "vxlt_gallowstemplar_n_55s_0_0.vlm"},
	{0x045FB7AB30925EFDULL, "6162455_m.fxe"},
	{0x045FC53713F45759ULL, "mag320_nobleman_3.bnk"},
	{0x0460080DB208ADC8ULL, "vxw_blightlands_ov_0_0.vlm"},
	{0x04600B870E94EE38ULL, "6131601_m.fxe"},
	{0x0460A30C9AE79051ULL, "59123258.ogg"},
	{0x0460CBA0931EF756ULL, "vxw_bonepit_1q6_0_0.vlm"},
	{0x0460F4F67E9C8588ULL, "761207995.ogg"},
	{0x0460FDE2422B610CULL, "dre000_bethany6037317.cl"},
	{0x046107231E7DF43DULL, "6018667_f.fxe"},
	{0x04611F58989DD82CULL, "art\\levels\\races\\elf\\props\\_textures\\prp_eluvian_d.dds"},
	{0x04611FDA96094357ULL, "hlac_ht_chantry_n_2534_34z.dds"},
	{0x04612F8A8919AFC0ULL, "6096906_m.fxe"},
	{0x0462690C9778384BULL, "6127519_m.fxe"},
	{0x04628D4FEFC2C5CDULL, "vxdr_primevaltomb_917_0_0.vlm"},
	{0x0462F2639AA6D2D2ULL, "gen_im_trs_gem_ruby_04.uti"},
	{0x04631750C835A224ULL, "37164881.ogg"},
	{0x046395A2B54E5324ULL, "820369356.ogg"},
	{0x0463B55B32869027ULL, "t3_wdg_tb3.tnt"},
	{0x0463E73713F879B9ULL, "mag320_nobleman_3.cnv"},
	{0x0463FBF7B36424EBULL, "gen00fl_anders6125579.cl"},
	{0x04645CE3CD67FCAAULL, "hlac_lt_0_lowtown_a3_n_c_-65537_5y.dds"},
	{0x0464643735CAC369ULL, "243639855.ogg"},
	{0x0464F7489AD3075BULL, "vxw_encounter2_camp_eh_0_0.vlm"},
	{0x0465A15812D4F6CFULL, "vxlt_0_lowtown_docks_a3_n_c3u_0_0.vlm"},
	{0x0465B7C8E2C91209ULL, "6164035_m.fxe"},
	{0x04668BC097F64726ULL, "837076207.ogg"},
	{0x0466F5A2EB3C19C8ULL, "ht_playershouse_n.arl"},
	{0x046729F5BEAFA475ULL, "follower_banter6171579.cl"},
	{0x0467F5E4DB09E5CDULL, "one100dg_kelder6055425.cl"},
	{0x04686923B194F89FULL, "vxht_0_hightown_a2_ava_0_0.vlm"},
	{0x04686D0D2436E6C8ULL, "578012366.ogg"},
	{0x0468BA915647BF6DULL, "vxlt_0_lowtown_a3_n_c_hr2_0_0.vlm"},
	{0x0468CFD20FAAFFFFULL, "zz_ave_debug6175058.cl"},
	{0x046947275BF22397ULL, "6174964_m.fxe"},
	{0x046A046373DB19FCULL, "glows.dds"},
	{0x046A6D915483307DULL, "6034046_m.fxe"},
	{0x046AB1E0EA9EF38BULL, "206651454.ogg"},
	{0x046AB4E7ECF8FFBEULL, "6119511_m.fxe"},
	{0x046AD60854BB2727ULL, "632421234.ogg"},
	{0x046B160595C25A5CULL, "mag360_samson6044923.cl"},
	{0x046B47220D40FC61ULL, "6209945_m.fxe"},
	{0x046CBC1DA4A470B3ULL, "vxht_mansion2_1v1_0_0.vlm"},
	{0x046CC15ED63269CDULL, "hlac_lt_gallowscourtyard_-131073_1x.dds"},
	{0x046E1E816DA5E184ULL, "77426167.ogg"},
	{0x046E291B002CB497ULL, "mag311_keran6061822.cl"},
	{0x046E79EAA3FEC648ULL, "vxw_blightlands_pw_0_0.vlm"},
	{0x046EAF63C146F17CULL, "hlac_lt_0_lowtown_a3_n_c_-65536_5x.dds"},
	{0x046EBFED5C97D7E6ULL, "hlac_ht_0_hightown_0_2y.dds"},
	{0x046EDDC45726CE96ULL, "nat380_nathaniel6066618.cl"},
	{0x046EE33D6463BD96ULL, "vxlt_gallowstemplar_3ru_0_0.vlm"},
	{0x046EE6AFACE4A610ULL, "vxdr_0_deeproads_esf_0_0.vlm"},
	{0x046EF0E82DC60F20ULL, "vxht_brothel_n_3qu_0_0.vlm"},
	{0x046F3B3066DA7D2EULL, "vxlt_0_lowtown_n_exe_0_0.vlm"},
	{0x046F3D7D85150FD6ULL, "vxw_bonepit_1z7_0_0.vlm"},
	{0x046F4B6FCE808441ULL, "dre000_varric6027753.cl"},
	{0x046F64109DAB9D38ULL, "hlac_lt_0_lowtown_a2_n_c_65536_6y.dds"},
	{0x046F855039A9E214ULL, "art\\cinematics\\_crit path\\prologue\\additional animations\\cs_hurlock_fdeath3.gad"},
	{0x046FFCB667E2E161ULL, "qun120_arishok_guard6060604.cl"},
	{0x04701D459ED5E6F8ULL, "6163242_m.fxe"},
	{0x0470C02F6212FBEDULL, "vxca_cavern1_haunted_e7m_0_0.vlm"},
	{0x0470E2541C55135FULL, "6177830_f.fxe"},
	{0x0471510E993F64D9ULL, "6108774_m.fxe"},
	{0x04715234E26FF3C1ULL, "6101073_f.fxe"},
	{0x047159708D9EFC07ULL, "6181757_m.fxe"},
	{0x0471C69A05A61810ULL, "891693116.ogg"},
	{0x0471D8485094B376ULL, "vxw_sundermount_2nk_0_0.vlm"},
	{0x0471DFBFF4D37061ULL, "gen00fl_varric6105421.cl"},
	{0x0472365588D07766ULL, "6026976_m.fxe"},
	{0x0472D17158D025CDULL, "vxw_encounter2_camp_8e_0_0.vlm"},
	{0x04731CB3F372DF39ULL, "6099211_f.fxe"},
	{0x0473210D28765FF8ULL, "643539622.ogg"},
	{0x047342C8A3A4146BULL, "6066707_f.fxe"},
	{0x047351FE2C30680DULL, "6028163_f.fxe"},
	{0x04736311D4A032E1ULL, "vxw_blightlands_uq_0_0.vlm"},
	{0x0473C358AFCE4E25ULL, "det_smo.mao"},
	{0x0475090E978C2DD8ULL, "274053304.ogg"},
	{0x047566ABA40488E0ULL, "art\\levels\\races\\natural\\highlands\\terrainmats\\ter_hflatgrass_da.dds"},
	{0x0475B7C301502C5FULL, "6088663_m.fxe"},
	{0x0475D2CEF93E8E91ULL, "975791807.ogg"},
	{0x04760534C10E3B6DULL, "hlac_dr_slavers_3207_4y.dds"},
	{0x04764302182C367DULL, "132621602.ogg"},
	{0x0476945039B01AF2ULL, "art\\cinematics\\_crit path\\prologue\\additional animations\\cs_hurlock_fdeath3.ani"},
	{0x0476FDC319B008D8ULL, "one220_yevhen6135788.cl"},
	{0x047760A7804C93CFULL, "263196295.ogg"},
	{0x0477802647290DE2ULL, "plt_one000pt_talked_to.nss"},
	{0x0478F4003B6AC1F7ULL, "6178371_m.fxe"},
	{0x047902DAE944A083ULL, "6126992_m.fxe"},
	{0x0479077993F822DDULL, "ts_beasts_prog_liver.gda"},
	{0x04793AF9E4FA6BA2ULL, "hlac_w_encounter3_-262139_0y.dds"},
	{0x047960802A39B50BULL, "hlac_w_blightlands_65537_0z.dds"},
	{0x047967827E4C70FDULL, "docks_male_sailor_mob6176670.cl"},
	{0x0479B71306B7A3D6ULL, "art\\characters\\playercharacter\\humanmale\\animation\\exploration\\movement\\mov_wf.ani"},
	{0x047A710363F240EFULL, "6112339_m.fxe"},
	{0x047AC24466D772A3ULL, "hlac_w_bonepit_196614_0x.dds"},
	{0x047B19821F698E5AULL, "art\\levels\\4cinematics\\qunari\\_textures\\prp_qunarirelic_n.dds"},
	{0x047B2DFA969A8933ULL, "vxht_mansion2_1y0_0_0.vlm"},
	{0x047BABF8D890404FULL, "hlac_lt_0_lowtown_docks_-65537_3y.dds"},
	{0x047BB4D27162C850ULL, "hlac_lt_undercityencounter_-1_44z.dds"},
	{0x047C254BCC15EB0CULL, "gen_im_trs_rdr_med_boo_r0.uti"},
	{0x047D30231C21A5E2ULL, "vxlt_0_lowtown_a2_n_c_hws_0_0.vlm"},
	{0x047DAD0D58D095AEULL, "vxlt_0_lowtown_n_euf_0_0.vlm"},
	{0x047DB61B569927F5ULL, "vxlt_gallowstemplar_n_5xa_0_0.vlm"},
	{0x047DDFBFF481A674ULL, "hlac_ht_chantry_2534_27x.dds"},
	{0x047E667D91716503ULL, "w_encounter2_camp_60_0.mao"},
	{0x047E6A7D91716C50ULL, "w_encounter2_camp_60_0.mmh"},
	{0x047E787D9171846AULL, "w_encounter2_camp_60_0.msh"},
	{0x047EC08275A93292ULL, "art\\characters\\playercharacter\\humanmale\\textures\\hm_hlh_medd_0s.dds"},
	{0x047ED691ADC9844AULL, "665082386.ogg"},
	{0x047ED93CF86AB889ULL, "plt_qun100pt_mage_escort.nss"},
	{0x047F09BC5790DE76ULL, "6083318_f.fxe"},
	{0x047F102B26D8EE83ULL, "530781805.ogg"},
	{0x047F5ED5A4714ADDULL, "follower_banter6164490.cl"},
	{0x0480386049760591ULL, "t1_mue_or3.tnt"},
	{0x04804A25428ACBF6ULL, "vxw_sundermount_2sl_0_0.vlm"},
	{0x04809C3D2AE01F32ULL, "222395906.ogg"},
	{0x0480AF4645086021ULL, "art\\characters\\weapons\\maces\\textures\\w_mce_mc105a_0d.dds"},
	{0x04813CD4BDEBC588ULL, "6120110_f.fxe"},
	{0x0481434E4AC63E4DULL, "vxw_encounter2_camp_3d_0_0.vlm"},
	{0x048146DDF9219DF2ULL, "dae200_bran6094766.cl"},
	{0x048160CFEFD7461BULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\0ff44_galofloor.phy"},
	{0x04819243C447AD73ULL, "6138739_m.fxe"},
	{0x0481EE238D3840B8ULL, "llac_lt_gallowstemplar_f_2395_0.dds"},
	{0x04825869DD729F3EULL, "w_blightlands_31_0.anb"},
	{0x04827FCAF7CDE862ULL, "vxlt_hangedman_n_26b_0_0.vlm"},
	{0x0482F75E2BA32412ULL, "vxw_sundermount_2zr_0_1.vlm"},
	{0x0483518C1FD7D147ULL, "qun211cr_elf_warrior6072200.cl"},
	{0x048381330A38B59DULL, "766552943.ogg"},
	{0x0483BC12B37ED3B1ULL, "6110490_m.fxe"},
	{0x0483C8D691B512F0ULL, "6089577_m.fxe"},
	{0x0484299FF34644DFULL, "6083763_m.fxe"},
	{0x0484AE7C83240C04ULL, "vxlt_gallowscourtyard_n_65b_0_0.vlm"},
	{0x0484C1F861AF3B5AULL, "6207423_m.fxe"},
	{0x04858C26C7EBB4F8ULL, "vxw_bonepitmining_a2_my_0_0.vlm"},
	{0x04859E4B3A33F06DULL, "vxlt_0_lowtown_a3_n_c_hd0_0_0.vlm"},
	{0x0485BAB145AD4208ULL, "759675933.ogg"},
	{0x048618C3158EAD7EULL, "lowtown_male_alienage1_sta6161629.cl"},
	{0x0486BF1306C23BC7ULL, "art\\characters\\playercharacter\\humanmale\\animation\\exploration\\movement\\mov_wf.evt"},
	{0x04875BA299583F27ULL, "79879180.ogg"},
	{0x04876E020FB968B8ULL, "t3_arm_mer.tnt"},
	{0x0487B1D3D31893C2ULL, "vxdr_0_deeproads_ewh_0_0.vlm"},
	{0x048835C7540BE74AULL, "vxca_caverns1_gkq_0_0.vlm"},
	{0x0488A10B9EBBE0F2ULL, "vxht_bartrand_n_5nl_0_0.vlm"},
	{0x0488E39F82185C24ULL, "hlac_lt_0_lowtown_a2_-131072_4z.dds"},
	{0x0488FFE6238D36B1ULL, "mcr371_hostile_aveline6082178.cl"},
	{0x04894977D2D0002CULL, "llac_dr_primevalentrance_1725_3.dds"},
	{0x0489BCF49C72DEB9ULL, "gencr_gate_guardian_statue.utc"},
	{0x0489C31D0566E488ULL, "plt_gen00pt_money.plo"},
	{0x0489D9342A92D199ULL, "art\\levels\\races\\proxy\\props\\4tools\\inv_flat1x1_tool.phy"},
	{0x0489FA496C788A91ULL, "6149760_m.fxe"},
	{0x048A64E2C4854F83ULL, "dae300_petrice6097867.cl"},
	{0x048AC8EA53641FDFULL, "6146811_m.fxe"},
	{0x048AFE4AB22583B5ULL, "6053337_m.fxe"},
	{0x048B4D1B307DD503ULL, "hightown_market_guard_sta6169601.cl"},
	{0x048C14C4E80AD1A7ULL, "vxlt_backalley_n_1y8_0_0.vlm"},
	{0x048C74A935428233ULL, "art\\vfx\\combat\\vfx_r_asn_mdeath_tag_loop_c.dura.evt"},
	{0x048C96CAC4776ABCULL, "mrl250pt_mirrorimg.ncs"},
	{0x048DE01306C89210ULL, "art\\characters\\playercharacter\\humanmale\\animation\\exploration\\movement\\mov_wf.gad"},
	{0x048EBC023480E476ULL, "vxw_sundermount_2tm_0_0.vlm"},
	{0x048EE3911B011E86ULL, "511744329.ogg"},
	{0x048FE58CEB8AFE03ULL, "alphamagicarc.mao"},
	{0x0490D5FD68A030DEULL, "gen00fl_sebastian6180495.cl"},
	{0x0490EA46B1A5D4ACULL, "304442948.ogg"},
	{0x04911001C088B897ULL, "vxca_cavern1_haunted_j5r_0_0.vlm"},
	{0x049182533D3B193BULL, "llac_lt_0_lowtown_a3_n_c_131072_4.dds"},
	{0x04923CAD361E2581ULL, "6142519_f.fxe"},
	{0x04927DE080920ACAULL, "llac_ht_mansion2b_818_57.dds"},
	{0x0492E243C44F86BAULL, "addtalent.ncs"},
	{0x04936DA935489574ULL, "art\\vfx\\combat\\vfx_r_asn_mdeath_tag_loop_c.dura.gad"},
	{0x049377AF0194DBBEULL, "vxlt_0_lowtown_e8b_0_0.vlm"},
	{0x049389A62D6353E9ULL, "qun160_vashothlead6052564.cl"},
	{0x0493A1C9330B038DULL, "vxlt_gallowscourtyard_n_7av_0_0.vlm"},
	{0x0494404CBEE47B5FULL, "vxw_bonepitmining_a2_2af_0_0.vlm"},
	{0x0495694958DB3D64ULL, "ach_kill_combo.dds"},
	{0x04958C02954E57CCULL, "64504191.ogg"},
	{0x0495A4DB479A1EFDULL, "6206867_m.fxe"},
	{0x0495C41C14F87263ULL, "6118504_m.fxe"},
	{0x049623B0C50EAC42ULL, "vxdr_0_deeproads_epi_0_0.vlm"},
	{0x0496246A570B4AE3ULL, "dae300_cullen6142067.cl"},
	{0x0496BE3FFDC73529ULL, "6082160_m.fxe"},
	{0x0496F881A1B6B3F9ULL, "50520841.ogg"},
	{0x049744CE0A572588ULL, "gen00fl_fenris6106706.cl"},
	{0x0497EDCA55BCF013ULL, "6143416_m.fxe"},
	{0x04980DF71F3A1344ULL, "character_character_p30.psh.fxo.dx11.assembly"},
	{0x04986C265E6EA311ULL, "6142660_m.fxe"},
	{0x04986DEAA7F518ECULL, "vxht_0_hightown_n_c7l_0_0.vlm"},
	{0x04987F4189D23FCDULL, "vxca_caverns1_j7c_0_0.vlm"},
	{0x049898F2EB91A676ULL, "332122720.ogg"},
	{0x0498EBC4909266AFULL, "rog_critsneak.dds"},
	{0x0499060459056395ULL, "422984921.ogg"},
	{0x049987A46269BFCFULL, "128332134.ogg"},
	{0x04998C2FFB59D1B2ULL, "hlac_lt_0_lowtown_docks_n_-4_7y.dds"},
	{0x049A50B13F017308ULL, "vxlt_warehouse_1dv_0_0.vlm"},
	{0x049A7CA9354ECD52ULL, "art\\vfx\\combat\\vfx_r_asn_mdeath_tag_loop_c.dura.ani"},
	{0x049AF4C26D2EE3E6ULL, "438809916.ogg"},
	{0x049B13AFF215FC40ULL, "hla_ht_0_hightown_a3_-65536_0y.dds"},
	{0x049B53472468C523ULL, "mag170_thrask6042848.cl"},
	{0x049BA901CEE8AF52ULL, "one100dg_vanard6017496.cl"},
	{0x049BC28E99979EA4ULL, "vxlt_gallowstemplar_3r7_0_0.vlm"},
	{0x049C119330F6B04EULL, "gen00fl_anders6114486.cl"},
	{0x049C2048B50BA91FULL, "hlac_lt_0_lowtown_a2_n_2_0y.dds"},
	{0x049C3C1609765E1CULL, "hlac_lt_hangedman_-1_26y.dds"},
	{0x049C58EFA30CBB34ULL, "635789606.ogg"},
	{0x049C706E83634454ULL, "6056829_m.fxe"},
	{0x049C8A2E31DC09EAULL, "vxlt_0_lowtown_a2_n_c_gwh_0_0.vlm"},
	{0x049CF1B76B7E81B3ULL, "art\\characters\\weapons\\mauls\\w_mal_ml106.mao"},
	{0x049D0D6BDEE00D75ULL, "zz_vault_debug6203285.cl"},
	{0x049D2DDF2676FCF6ULL, "vxw_sundermount_2yn_0_0.vlm"},
	{0x049D5147FA30154BULL, "341609295.ogg"},
	{0x049DC081AB733C87ULL, "vxht_mansion2_1wn_0_0.vlm"},
	{0x049DE12714668D4DULL, "gen00fl_isabela6100075.cl"},
	{0x049DE8AE51051CBCULL, "seb201_lady_harimann6197709.cl"},
	{0x049F3C0273D9445CULL, "dae021_male_worshipper_sta6154417.cl"},
	{0x049F7054C40117B1ULL, "debug.gfx"},
	{0x049F9AAF692F678AULL, "vxlt_0_lowtown_docks_a3_9nn_0_0.vlm"},
	{0x04A046D1F8ACAD03ULL, "w_woundedcoast_106_0.mao"},
	{0x04A04AD1F8ACB450ULL, "w_woundedcoast_106_0.mmh"},
	{0x04A058D1F8ACCC6AULL, "w_woundedcoast_106_0.msh"},
	{0x04A059D512255012ULL, "hm_feynriel_b.mor"},
	{0x04A12921DEE2D9CCULL, "mag211_marethari6025314.cl"},
	{0x04A14DDA97D0FDDAULL, "hlac_ca_caverns1_24427_1y.dds"},
	{0x04A1E98BF38AF43EULL, "vxlt_0_lowtown_e3c_0_0.vlm"},
	{0x04A217B8ECACD475ULL, "seb221_flora_harimann6177694.cl"},
	{0x04A24603C28A3298ULL, "ser221_gascard6165443.cl"},
	{0x04A3492BE7C0C46AULL, "art\\vfx\\combat\\vfx_r_var_kback_act_c.anb"},
	{0x04A435F906EE8AE3ULL, "6117404_m.fxe"},
	{0x04A4B26F8E01DF4FULL, "dae000_templar_recruit_1.bnk"},
	{0x04A4CE961CBCB2C1ULL, "vxht_0_hightown_a2_c4g_0_0.vlm"},
	{0x04A4D5DAA537B723ULL, "6144743_m.fxe"},
	{0x04A4E19BFCDCCE7AULL, "gen00fl_dog6141140.cl"},
	{0x04A5B4F5C97923BAULL, "llac_ca_cavern1_haunted_24421_9.dds"},
	{0x04A5B99C13173E14ULL, "llac_dr_primevaltomb_9621_2.dds"},
	{0x04A5F33375302EC0ULL, "hlac_ca_cavern1_haunted_24424_23y.dds"},
	{0x04A61E531B3CFF24ULL, "6054391_m.fxe"},
	{0x04A6B8EAF9142196ULL, "vxw_encounter2_b0_0_0.vlm"},
	{0x04A6D38D6632C36EULL, "vxlt_warehouse_13y_0_0.vlm"},
	{0x04A6D6F1D1DBB9D5ULL, "979369949.ogg"},
	{0x04A707363A7AF28CULL, "vxw_bonepit_26n_0_0.vlm"},
	{0x04A71F52C0025F26ULL, "6069951_m.fxe"},
	{0x04A76D2647521165ULL, "plt_one000pt_talked_to.plo"},
	{0x04A7A02A1301417BULL, "6068441_m.fxe"},
	{0x04A7D46F8E044EB7ULL, "dae000_templar_recruit_1.cnv"},
	{0x04A84FAEC656966AULL, "6056830_m.fxe"},
	{0x04A87FAB3138BBE7ULL, "hlac_lt_0_lowtown_a2_n_c_-65535_0x.dds"},
	{0x04A8AF4FA5C5091AULL, "vxw_bonepitdestroyed_a3_9i_0_0.vlm"},
	{0x04A932559A91DBCBULL, "art\\vfx\\combat\\vfx_ha_bd_gthrow_proj_p.cess.evt"},
	{0x04A94F0E62174736ULL, "llac_lt_0_lowtown_a2_2_1.dds"},
	{0x04A9FA9ADFDA7A87ULL, "6060705_f.fxe"},
	{0x04AA1CAA4427BE36ULL, "vxht_bartrand_n_65f_0_0.vlm"},
	{0x04AA40FED85C468AULL, "mcr310_argument6127606.cl"},
	{0x04AA83CE1CABE712ULL, "vxht_0_hightown_n_9xe_0_0.vlm"},
	{0x04AAFC0B23D2226AULL, "vxlt_0_lowtown_a2_n_c_gpi_0_0.vlm"},
	{0x04AB6CB8BB02E007ULL, "art\\characters\\weapons\\dagger\\w_dgr_105a.mao"},
	{0x04AB714F1323950BULL, "6054491_m.fxe"},
	{0x04AC71EC38F87438ULL, "gen_im_trs_krk_junk2_01.uti"},
	{0x04AC82C40A409EE9ULL, "vxdr_0_deeproads_eue_0_0.vlm"},
	{0x04AD0F4A8CFACB70ULL, "vxlt_gallowstemplar_f_6sv_0_0.vlm"},
	{0x04AD3237DDDA97DEULL, "mer140_bonepit_comments6077813.cl"},
	{0x04AD4F92F1B5E8BAULL, "vxlt_backalley_g_1vu_0_0.vlm"},
	{0x04AD55F719FBCDA4ULL, "ef_gen_act2.mor"},
	{0x04ADA1CC95AC66BBULL, "vxht_0_hightown_a3_9zt_0_0.vlm"},
	{0x04ADECDF7E2A8EA3ULL, "vxht_keep_n_70j_0_0.vlm"},
	{0x04AE176A37A19E8BULL, "6172395_m.fxe"},
	{0x04AE47CFEFFE09BAULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\0ff44_galofloor.msh"},
	{0x04AE51A91F507C84ULL, "6136662_m.fxe"},
	{0x04AE55CFEFFE21F3ULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\0ff44_galofloor.mao"},
	{0x04AE59CFEFFE2800ULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\0ff44_galofloor.mmh"},
	{0x04AE71AA7D574F42ULL, "365475068.ogg"},
	{0x04AEDCC26BFD87F6ULL, "tbe100_xenon6189674.cl"},
	{0x04AF8BE36F5A8CCAULL, "ave110_ambush_26105930.cl"},
	{0x04AFBAC0E81F74EEULL, "vxca_caverns2prc_a13_0_0.vlm"},
	{0x04AFC28E0D4D122EULL, "art\\vfx\\creatures\\fxc_ashwraith_c.dura.evt"},
	{0x04AFE07E94F11365ULL, "6043365_m.fxe"},
	{0x04AFFF94C2EB6E32ULL, "genip_node_d1.utp"},
	{0x04B04B559A98248CULL, "art\\vfx\\combat\\vfx_ha_bd_gthrow_proj_p.cess.gad"},
	{0x04B0B84ED232DEBDULL, "art\\vfx\\combat\\cn_gut.mao"},
	{0x04B0F0B30CAA8911ULL, "llac_dr_primevaltomb_665_0.dds"},
	{0x04B1537ED7905580ULL, "233086860.ogg"},
	{0x04B1577FD717D30CULL, "vxca_caverns2_2qw_0_0.vlm"},
	{0x04B16305D1900806ULL, "vxdr_primevaltomb_2vo_0_1.vlm"},
	{0x04B1F3575E62DE66ULL, "6030603_m.fxe"},
	{0x04B209B1F995550DULL, "vxca_caverns1_mines_j4c_0_0.vlm"},
	{0x04B2150B6EC46275ULL, "511082057.ogg"},
	{0x04B23A5D926A88F9ULL, "vxht_0_hightown_a3_c5a_0_0.vlm"},
	{0x04B26E3C349FD885ULL, "hlac_lt_0_lowtown_a3_n_c_-1_6z.dds"},
	{0x04B2A7D5F8E4A363ULL, "6112704_m.fxe"},
	{0x04B340730EB2CB41ULL, "vxht_0_hightown_a2_c3f_0_0.vlm"},
	{0x04B354876223DADAULL, "413002776.ogg"},
	{0x04B398DB4CE6A4DEULL, "dae010_alienage_elf_f2.utc"},
	{0x04B3D7D90E0C7D91ULL, "dae000cr_grand_cleric6144096.cl"},
	{0x04B46F66CB45D6B5ULL, "hlac_ht_brothel_n_2995_3z.dds"},
	{0x04B4CE8B1D510361ULL, "hlac_lt_0_lowtown_a2_n_-131072_4z.dds"},
	{0x04B4EBC028B433D7ULL, "914868443.ogg"},
	{0x04B4F7C485091A14ULL, "hlac_dr_primevaltomb_11985_3y.dds"},
	{0x04B5223EB437B537ULL, "gen00fl_merrill6108437.cl"},
	{0x04B52AC7EB0A3A16ULL, "vxw_encounter2_m1_0_0.vlm"},
	{0x04B579132C710B0CULL, "vxw_bonepit_21o_0_0.vlm"},
	{0x04B5FD8BF9573525ULL, "vxw_sundermount_2m5_0_0.vlm"},
	{0x04B64A559A9C8E8AULL, "art\\vfx\\combat\\vfx_ha_bd_gthrow_proj_p.cess.ani"},
	{0x04B67A80B96420D5ULL, "vxht_mansion2b_3ed_0_0.vlm"},
	{0x04B6917ABEEB890DULL, "343108042.ogg"},
	{0x04B6B98E0D5321D9ULL, "art\\vfx\\creatures\\fxc_ashwraith_c.dura.gad"},
	{0x04B6E22CB679B230ULL, "232855076.ogg"},
	{0x04B7200066B32090ULL, "vxdr_0_deeproads_ehc_0_0.vlm"},
	{0x04B73A876FEAEA12ULL, "666294689.ogg"},
	{0x04B73E0C3C129A96ULL, "174706719.ogg"},
	{0x04B755756F98CCC2ULL, "6107458_f.fxe"},
	{0x04B76A7D91A13113ULL, "w_encounter2_camp_60_0.rml"},
	{0x04B7C3EE50E846A5ULL, "and100_anders6173569.cl"},
	{0x04B7D80916ECADE2ULL, "820403571.ogg"},
	{0x04B80AD7B2331960ULL, "547249313.ogg"},
	{0x04B83860F34323EEULL, "81260081.ogg"},
	{0x04B8E2D4AEE8356BULL, "hlac_lt_0_lowtown_n_-131072_5y.dds"},
	{0x04B8EF713CDE4954ULL, "art\\levels\\races\\design\\prp_bodybaga_1.mao"},
	{0x04B9561F1136F2A6ULL, "art\\characters\\creatures\\spider\\textures\\c_corspidra_0d.dds"},
	{0x04B9BA34616F8A17ULL, "ts_circleofmagi_prog_junk3.gda"},
	{0x04B9E5012CE34DB7ULL, "127357187.ogg"},
	{0x04B9E95D87134D2BULL, "hlac_lt_shop_-1_0y.dds"},
	{0x04BA0144AF117774ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\knife_stab_v1_dead_loop.gad"},
	{0x04BA37B2303A08F0ULL, "6057141_m.fxe"},
	{0x04BA43B925875B56ULL, "795014385.ogg"},
	{0x04BA59B3CFAB2A72ULL, "hlac_dr_primevaltomb_661_5x.dds"},
	{0x04BAAA70567CD21CULL, "w_blightlands_51_0.mao"},
	{0x04BAB670567CE6BFULL, "w_blightlands_51_0.mmh"},
	{0x04BABC70567CF161ULL, "w_blightlands_51_0.msh"},
	{0x04BACA7C8B4CCA90ULL, "6102580_m.fxe"},
	{0x04BADFC4D241BA7EULL, "vxdr_primevalentrance_16m_0_0.vlm"},
	{0x04BBF09DF7DAB6D5ULL, "gen00fl_anders6214666.cl"},
	{0x04BC3AB576B0B746ULL, "llac_lt_0_lowtown_docks_-4_3.dds"},
	{0x04BC619E048C59D8ULL, "mer240_jansen6050441.cl"},
	{0x04BCB015FD5E218CULL, "331563872.ogg"},
	{0x04BCD2B71B25BD46ULL, "vxlt_0_lowtown_docks_b4h_0_0.vlm"},
	{0x04BCD69CD2153688ULL, "ch2armtnt_dy.psh.fxo"},
	{0x04BD26E064B0B598ULL, "262200545.ogg"},
	{0x04BDA6BAC6CB6E27ULL, "llac_lt_hangedman_n_-1_27.dds"},
	{0x04BDCA8E0D595D27ULL, "art\\vfx\\creatures\\fxc_ashwraith_c.dura.ani"},
	{0x04BE717D91A75C6BULL, "w_encounter2_camp_60_0.phy"},
	{0x04BEBBEAA0753EAFULL, "vxca_cave_tz_0_0.vlm"},
	{0x04BEFBAF13625B21ULL, "llac_lt_0_lowtown_65533_0.dds"},
	{0x04BEFC8A5A6D4C91ULL, "dae000cr_grand_cleric6187949.cl"},
	{0x04BFE500703BC3DAULL, "736467513.ogg"},
	{0x04C080E4B4E1A2A7ULL, "lgt110_arl_bann_teagan6115602.cl"},
	{0x04C081571304F78EULL, "588873373.ogg"},
	{0x04C0FB2C83B9649FULL, "1067708547.ogg"},
	{0x04C11044AF17AF52ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\knife_stab_v1_dead_loop.ani"},
	{0x04C1FC9358DA7527ULL, "312653586.ogg"},
	{0x04C2B9C63A89808BULL, "6027952_m.fxe"},
	{0x04C2BE62531BB66FULL, "vxht_0_hightown_c8i_0_0.vlm"},
	{0x04C2F7F511975933ULL, "hlac_ht_mansion2_818_47x.dds"},
	{0x04C30220117C5552ULL, "gen00fl_varric6118967.cl"},
	{0x04C3258EAC693B3FULL, "vxht_0_hightown_n_c7c_0_0.vlm"},
	{0x04C39CA4DD005296ULL, "vxw_encounter2_h2_0_0.vlm"},
	{0x04C3BCD978D032CCULL, "character_character_p5.psh.fxo.dx11"},
	{0x04C40A66191CBBE2ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\1ww_basic.phy"},
	{0x04C4C15510211B44ULL, "vxlt_warehouse_n_u1_0_0.vlm"},
	{0x04C541C560F394B1ULL, "112775813.ogg"},
	{0x04C5B7C685BCFAE9ULL, "docks_scene_dockworkers6176504.cl"},
	{0x04C63C794D13C3CEULL, "one300_zevran6089735.cl"},
	{0x04C668C58082FAF9ULL, "vxdr_0_deeproads_dqd_0_0.vlm"},
	{0x04C6755689B1EF66ULL, "vxw_sundermount_36w_0_1.vlm"},
	{0x04C6CDA13F17EE1DULL, "hlac_dr_primevalentrance_1726_3z.dds"},
	{0x04C6FB4034B9924EULL, "vxht_bartrand_n_5vh_0_0.vlm"},
	{0x04C705ABE121DDF8ULL, "plt_prm_cod_parth_staff.plo"},
	{0x04C86050F90F9371ULL, "vxw_bonepitmining_a2_21h_0_0.vlm"},
	{0x04C87A24CEA95FA4ULL, "mrl000_marethari6129203.cl"},
	{0x04C8BAEC31B36D43ULL, "art\\characters\\playercharacter\\humanmale\\hm_har_ha14a.mmh"},
	{0x04C8BD713A6DEC4AULL, "vxlt_0_lowtown_docks_a3_d0u_0_0.vlm"},
	{0x04C8C8EC31B3849DULL, "art\\characters\\playercharacter\\humanmale\\hm_har_ha14a.msh"},
	{0x04C8E7081759A8C6ULL, "lgt160_ntb_elf_conflict6163394.cl"},
	{0x04C91354BCD88A30ULL, "vxht_0_hightown_a3_7g2_0_0.vlm"},
	{0x04C93BA2FDA154E6ULL, "art\\vfx\\combat\\vfx_gui_pp_mage_c.cess.ani"},
	{0x04CA0B74BF11ED50ULL, "vxlt_0_lowtown_docks_n_b67_0_0.vlm"},
	{0x04CA334CD5A219BAULL, "vxlt_backalley_g_1xw_0_0.vlm"},
	{0x04CA900088298461ULL, "hlac_dr_0_deeproads_5842_8x.dds"},
	{0x04CAB82A14B37C3AULL, "6167532_m.fxe"},
	{0x04CB357D72B0A5AEULL, "dae000_gamlen6131099.cl"},
	{0x04CB4B41D6150A9FULL, "vxht_brothel_n_6g9_0_0.vlm"},
	{0x04CB63A55B3A7A43ULL, "6092299_m.fxe"},
	{0x04CBD0AF9B6C5FFCULL, "844188577.ogg"},
	{0x04CC4C2BE7E3CE5DULL, "art\\vfx\\combat\\vfx_r_var_kback_act_c.mmh"},
	{0x04CCA38367BE3E6DULL, "vxca_caverns2prc_a0w_0_0.vlm"},
	{0x04CCD3DB6EB0ACD9ULL, "574609747.ogg"},
	{0x04CD1C06AA79C4F3ULL, "hlac_ca_cavern1_haunted_24424_13y.dds"},
	{0x04CD37DD2C4B94FEULL, "vxht_0_hightown_a2_b8y_0_0.vlm"},
	{0x04CD8239DF962B34ULL, "vxht_0_hightown_a3_a6k_0_0.vlm"},
	{0x04CDEAF3744013A1ULL, "vxdr_primevaltomb_3lw_0_1.vlm"},
	{0x04CE099A4884FCEFULL, "art\\characters\\weapons\\axes\\textures\\w_axe_ax104a_0s.dds"},
	{0x04CE8BF9EE93053EULL, "vxca_caverns1_i99_0_0.vlm"},
	{0x04CF475E9C548F36ULL, "art\\vfx\\combat\\vfx_r_arc_front_close_c.anb"},
	{0x04CF690678EE394BULL, "darktown_male_sickly_sta.bnk"},
	{0x04CF7F3417F6A710ULL, "456273016.ogg"},
	{0x04CFB46AC4E22228ULL, "vxlt_gallowstemplar_n_5z2_0_0.vlm"},
	{0x04D0541898BC9568ULL, "135791899.ogg"},
	{0x04D0A4A54AB32C3FULL, "96253635.ogg"},
	{0x04D11C8D5C857B4AULL, "6125199_f.fxe"},
	{0x04D1923397965D6CULL, "6125338_m.fxe"},
	{0x04D1A2193696B6B8ULL, "vxdr_primevalentrance_2yi_0_0.vlm"},
	{0x04D1D1B2A4428C69ULL, "art\\characters\\playercharacter\\qunarimale\\mt_bas_qm_facem1.mao"},
	{0x04D1DC147CFB53B9ULL, "292693206.ogg"},
	{0x04D20E81CEF66B16ULL, "vxw_encounter2_s3_0_0.vlm"},
	{0x04D245143D38F8F1ULL, "ser270_emeric6113502.cl"},
	{0x04D26B0678F0722BULL, "darktown_male_sickly_sta.cnv"},
	{0x04D2B5D24333A10CULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\twohanded\\right.ani"},
	{0x04D302FF375A51D8ULL, "6144847_m.fxe"},
	{0x04D33332021733C4ULL, "vxlt_warehouse_n_x6_0_0.vlm"},
	{0x04D3C727659D13EAULL, "511724464.ogg"},
	{0x04D3F8BEA8DF5C0FULL, "537220783.ogg"},
	{0x04D41ACBF54E6CECULL, "art\\levels\\races\\dwarf\\props\\decor\\prp_potb.msh"},
	{0x04D420CBF54E76DEULL, "art\\levels\\races\\dwarf\\props\\decor\\prp_potb.mmh"},
	{0x04D42CCBF54E8B75ULL, "art\\levels\\races\\dwarf\\props\\decor\\prp_potb.mao"},
	{0x04D45388F9D9898EULL, "hlac_ht_bartrand_n_3373_0y.dds"},
	{0x04D48B1615E235FDULL, "6112121_f.fxe"},
	{0x04D4B4A1D6851795ULL, "plt_cod_ltr_junk_starkprinc.plo"},
	{0x04D52DA3C30F544CULL, "6040955_f.fxe"},
	{0x04D53A76FD97FC0CULL, "w_blightlands_87_0.msh"},
	{0x04D54076FD9806FEULL, "w_blightlands_87_0.mmh"},
	{0x04D54C76FD981A95ULL, "w_blightlands_87_0.mao"},
	{0x04D56EC0E1ABBDB1ULL, "art\\characters\\weapons\\maces\\textures\\w_mce_mc105a_0t.dds"},
	{0x04D5A31CA9E39771ULL, "294386887.ogg"},
	{0x04D5AC5D0AE7084DULL, "vxdr_primevalentrance_272_0_0.vlm"},
	{0x04D5F83DDDDCCDC5ULL, "945810808.ogg"},
	{0x04D65CB784D82F50ULL, "prty_seb.dds"},
	{0x04D6A668546AE58BULL, "vxlt_darktown_5oa_0_0.vlm"},
	{0x04D6E9C2C18260A6ULL, "vxht_brothel_n_6bs_0_0.vlm"},
	{0x04D723A2FDAD6957ULL, "art\\vfx\\combat\\vfx_gui_pp_mage_c.cess.evt"},
	{0x04D787F5734B9E87ULL, "vxht_mansion2_1kj_0_0.vlm"},
	{0x04D7ADC25C3EEAF8ULL, "vxdr_primevaltomb_7ua_0_1.vlm"},
	{0x04D7B5BC9DC73138ULL, "ht_brothel_n_3372.rml"},
	{0x04D859D884017B39ULL, "vxlt_0_lowtown_docks_a3_n_d39_0_1.vlm"},
	{0x04D8B8EBF27A65ABULL, "404980372.ogg"},
	{0x04D8F1CAC7FBF82AULL, "hlac_lt_0_lowtown_n_-65538_0z.dds"},
	{0x04D91422467EBFF4ULL, "narrative_varric6124292.cl"},
	{0x04D9BBEC31C1E268ULL, "art\\characters\\playercharacter\\humanmale\\hm_har_ha14a.phy"},
	{0x04D9C2D24339D64EULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\twohanded\\right.gad"},
	{0x04D9E4351AF8F0F1ULL, "6175990_m.fxe"},
	{0x04D9ED06DD7199C6ULL, "gen00fl_carver6108887.cl"},
	{0x04DA4630DB65513CULL, "one200pt_fools_gold.ncs"},
	{0x04DB06688F27D9FFULL, "372507193.ogg"},
	{0x04DB8DD76779FD95ULL, "6135921_m.fxe"},
	{0x04DC1AA7E5115755ULL, "506307671.ogg"},
	{0x04DC4A5E9C5F1EA1ULL, "art\\vfx\\combat\\vfx_r_arc_front_close_c.mmh"},
	{0x04DD2343F291F779ULL, "327899676.ogg"},
	{0x04DD7FE8111DBC86ULL, "lt_0_lowtown_a2_n_c_19_0_lc.mmh"},
	{0x04DDBA2C1D1B9B8CULL, "art\\characters\\playercharacter\\humanmale\\textures\\ulrichvon\\hm_arm_urvb_0d.dds"},
	{0x04DDCF59CEE708F6ULL, "lgt260_eau_tmp_angelique6143810.cl"},
	{0x04DE24A2FDB389E0ULL, "art\\vfx\\combat\\vfx_gui_pp_mage_c.cess.gad"},
	{0x04DE5C16E009A2D5ULL, "dae000ip_crowd_mf4_talk18.utp"},
	{0x04DEDB40F2B91CFFULL, "1028048000.ogg"},
	{0x04DF0546383918DAULL, "mag291_ntbanter.cnv"},
	{0x04DF2C6FC42450A1ULL, "6177281_f.fxe"},
	{0x04DF883213665F23ULL, "6069412_f.fxe"},
	{0x04DF8E6A4E7B93CAULL, "6122099_f.fxe"},
	{0x04DF993A9C1F1172ULL, "vxlt_0_lowtown_a2_n_gv5_0_0.vlm"},
	{0x04DF9EE307E78524ULL, "s_ballista_06_l2.mao"},
	{0x04DFAE03299A9F0AULL, "97719090.ogg"},
	{0x04DFF356AD5BB1BCULL, "vxca_cave_2dw_0_0.vlm"},
	{0x04E00410898C75ECULL, "6122238_m.fxe"},
	{0x04E00A0B04ED8239ULL, "6079298_m.fxe"},
	{0x04E01FC971008DA2ULL, "vxlt_0_lowtown_docks_c1n_0_0.vlm"},
	{0x04E051D1F8E2A46BULL, "w_woundedcoast_106_0.phy"},
	{0x04E0BDD2433FEC2DULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\twohanded\\right.evt"},
	{0x04E16278EECF4F6DULL, "6174666_f.fxe"},
	{0x04E20A19203E1B24ULL, "vxw_encounter2_ms_0_0.vlm"},
	{0x04E22746383B8822ULL, "mag291_ntbanter.bnk"},
	{0x04E22E1492443CEEULL, "dre000_vampire6034414.cl"},
	{0x04E24DA928F13893ULL, "vxca_caverns1_mines_ikk_0_0.vlm"},
	{0x04E28A40121B28EEULL, "hlac_ht_mansion2_818_62x.dds"},
	{0x04E2E626F140E2DDULL, "art\\characters\\creatures\\qunari\\textures\\qm_hlm_magb_0s.dds"},
	{0x04E3B770569FEDFCULL, "w_blightlands_51_0.anb"},
	{0x04E51171B0648317ULL, "hlac_dr_primevaltomb_664_0y.dds"},
	{0x04E54033C11B0F5DULL, "vxht_chantry_2v8_0_0.vlm"},
	{0x04E56685FF98CE48ULL, "vxlt_darktown_468_0_0.vlm"},
	{0x04E5835094412A05ULL, "hlac_lt_0_lowtown_a3_n_c_2_1y.dds"},
	{0x04E59BCB97BC2C30ULL, "ser270_emeric6149353.cl"},
	{0x04E5F9D26541B707ULL, "vxht_mansion2_1ne_0_0.vlm"},
	{0x04E634E8C77C2C16ULL, "genip_trap_vert_fire_jet.utp"},
	{0x04E80F172C3980F1ULL, "330757309.ogg"},
	{0x04E8843A7197CC8EULL, "6068880_m.fxe"},
	{0x04E88CFC0763C42DULL, "follower_banter6171544.cl"},
	{0x04E8DF1C47BCACDEULL, "6120951_m.fxe"},
	{0x04E8E566193B9896ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\1ww_basic.mao"},
	{0x04E8F166193BAD35ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\1ww_basic.mmh"},
	{0x04E8F366193BB0ABULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\1ww_basic.msh"},
	{0x04E914549FECF0C7ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\grovel.gad"},
	{0x04E9435207723503ULL, "29839064.ogg"},
	{0x04E95CA557434452ULL, "gen_im_npc_wep_arc_lbwb.uti"},
	{0x04E99EC095361849ULL, "lt_0_lowtown_docks_a3_n_1_0_lc.mmh"},
	{0x04E9CCD1FD9E1503ULL, "379643227.ogg"},
	{0x04EB80D62DC1FC8CULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\pose_e.gad"},
	{0x04EBA1F6CF300610ULL, "follower_banter6171929.cl"},
	{0x04EBA81683E4AE1BULL, "3295196.ogg"},
	{0x04EC1F87AD4477E8ULL, "lt_0_lowtown_docks_109_0.msh"},
	{0x04EC3187AD449679ULL, "lt_0_lowtown_docks_109_0.mao"},
	{0x04EC3D87AD44AA12ULL, "lt_0_lowtown_docks_109_0.mmh"},
	{0x04EC5360370118D9ULL, "857843093.ogg"},
	{0x04EC54B22E6EC59EULL, "hlac_w_sundermount_-196608_0z.dds"},
	{0x04EC7A3E5A5F67E5ULL, "mcr370_assignments6081719.cl"},
	{0x04ECD46D990750A2ULL, "vxdr_slavers_1ja_0_0.vlm"},
	{0x04ED2BF683869DF1ULL, "t3_hey_am3.tnt"},
	{0x04ED33F10E6DACE6ULL, "857451040.ogg"},
	{0x04ED4D5445BEBF8AULL, "llac_lt_backalley_g_1_5.dds"},
	{0x04ED79599F94C17CULL, "94285862.ogg"},
	{0x04EDCAE83D8B193AULL, "hlac_lt_0_lowtown_a3_n_c_131072_5z.dds"},
	{0x04EE3D40A3D53FECULL, "635891422.ogg"},
	{0x04EEE6822F5B7407ULL, "618963231.ogg"},
	{0x04EFDAB32DC1B5FCULL, "gen_im_trs_gen_hvy_boo_r1.uti"},
	{0x04F00172D6B22E15ULL, "944417318.ogg"},
	{0x04F009549FF2FCD4ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\grovel.evt"},
	{0x04F01543C6B4DB4EULL, "art\\vfx\\combat\\vfx_w_wns_shield_defen_l_c.impa.ani"},
	{0x04F01F6402256B7CULL, "llac_ht_chantry_n_2534_13.dds"},
	{0x04F025D5268B5463ULL, "w_bonepitmining_a2_81_0.msh"},
	{0x04F026DAEB1F3352ULL, "hlac_lt_gallowstemplar_2383_3y.dds"},
	{0x04F033D5268B6BBDULL, "w_bonepitmining_a2_81_0.mmh"},
	{0x04F036AB6DB95D65ULL, "hlac_ht_0_hightown_a3_131073_0z.dds"},
	{0x04F037D5268B72CEULL, "w_bonepitmining_a2_81_0.mao"},
	{0x04F07BF6123433A4ULL, "vxw_encounter2_hp_0_0.vlm"},
	{0x04F0E1E5217855CAULL, "framebuffereffects_verticalblurshadows_p0.vsh.fxo"},
	{0x04F142070F468A64ULL, "vxht_chantry_n_36c_0_0.vlm"},
	{0x04F17FD62DC6668AULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\pose_e.ani"},
	{0x04F1C1205F631719ULL, "rdr111dg_martin6055074.cl"},
	{0x04F1CEBF60312D53ULL, "hlac_w_bonepitdestroyed_a3_196610_0x.dds"},
	{0x04F250912713C035ULL, "seb222ar_ruin.are"},
	{0x04F2A1E27FA2515DULL, "983898147.ogg"},
	{0x04F30176B994BB82ULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\h_cwkr.gad"},
	{0x04F3323D936CB894ULL, "867033515.ogg"},
	{0x04F3CD7CA56E91A6ULL, "vxht_brothel_n_6hu_0_0.vlm"},
	{0x04F3D59FBF98DB38ULL, "889324293.ogg"},
	{0x04F409F1BBE34F90ULL, "622366784.ogg"},
	{0x04F43811949485B2ULL, "6001198_f.fxe"},
	{0x04F4BABE1A924780ULL, "hlac_lt_0_lowtown_docks_a3_-131074_4z.dds"},
	{0x04F4E784A51C777EULL, "6095857_m.fxe"},
	{0x04F59D9B10E4804CULL, "qcr220_meredith6054750.cl"},
	{0x04F625372C2946FBULL, "6146943_m.fxe"},
	{0x04F6DA4ADA7DA505ULL, "vxht_chantry_n_2th_0_1.vlm"},
	{0x04F6DBFD41B5329DULL, "face_rendertodepth_p1.vsh.fxo.dx11"},
	{0x04F72775E1698056ULL, "art\\characters\\playercharacter\\qunarimale\\animation\\blendtree\\qm_overlay.anb"},
	{0x04F76D5351D5D8A0ULL, "llac_lt_undercityserialkiller_-1_42.dds"},
	{0x04F79731DE6168EAULL, "6077344_f.fxe"},
	{0x04F7E40CB033FB14ULL, "vxlt_hangedman_2xq_0_0.vlm"},
	{0x04F8598C121EE128ULL, "lgt160_wk_avern_deaddrop.cnv"},
	{0x04F87006E999BB87ULL, "ser110ip_to_foundry.utp"},
	{0x04F967A4E491E003ULL, "6041946_f.fxe"},
	{0x04F977E1C23BC949ULL, "hlac_lt_0_lowtown_docks_a3_n_-131075_5z.dds"},
	{0x04F9AD6850B26810ULL, "vxlt_darktown_587_0_0.vlm"},
	{0x04FA15CFF4E0AD6BULL, "hlac_ht_playershouse_n_855_22z.dds"},
	{0x04FA1C21A2A111D1ULL, "6110619_f.fxe"},
	{0x04FA20DD280207ABULL, "6092178_m.fxe"},
	{0x04FA3FB1E7424E88ULL, "6095914_m.fxe"},
	{0x04FAA7F9641D74EFULL, "hlac_lt_gallowscourtyard_-131072_0z.dds"},
	{0x04FAF21A8FB155C5ULL, "follower_banter6161192.cl"},
	{0x04FB0A019AC46BA8ULL, "vxlt_gallowstemplar_n_5k1_0_0.vlm"},
	{0x04FB79DCA18F7768ULL, "dae000_meredith6085275.cl"},
	{0x04FB7B8C12215090ULL, "lgt160_wk_avern_deaddrop.bnk"},
	{0x04FB88C6138A2255ULL, "hlac_lt_0_lowtown_a2_n_c_-65538_0y.dds"},
	{0x04FB99A2A4C2342FULL, "6055001_m.fxe"},
	{0x04FC003F443D576DULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\4so_straightl.mmh"},
	{0x04FC043F443D5DBEULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\4so_straightl.mao"},
	{0x04FC123F443D75F3ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\4so_straightl.msh"},
	{0x04FC7191271C4C9BULL, "seb222ar_ruin.ncs"},
	{0x04FC8B506160B7B4ULL, "2068354.ogg"},
	{0x04FCF576195DB56DULL, "gen_im_acc_blt_act3_03.uti"},
	{0x04FD01549FFD790DULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\grovel.ani"},
	{0x04FD54ECA7A58FA5ULL, "follower_banter6155667.cl"},
	{0x04FD69790B5FE795ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\3ww_crumble_1.mao"},
	{0x04FD7EFB3B17C5E8ULL, "225279094.ogg"},
	{0x04FDB9540E473A84ULL, "6169819_m.fxe"},
	{0x04FDF5937931EB3FULL, "vxht_0_hightown_a2_c1v_0_0.vlm"},
	{0x04FE0D43C6C10A9FULL, "art\\vfx\\combat\\vfx_w_wns_shield_defen_l_c.impa.evt"},
	{0x04FF29514B92FACCULL, "hlac_lt_0_lowtown_docks_a3_n_131070_0x.dds"},
	{0x04FF956C55764F7AULL, "vxlt_undercitydungeon_1c2_0_1.vlm"},
	{0x04FFBDD9B2951138ULL, "6052087_m.fxe"},
	{0x0500438903DA1388ULL, "6086389_m.fxe"},
	{0x050065F7A6BE95ECULL, "1004779893.ogg"},
	{0x05009232F8DA3971ULL, "lgt260cr_way_merc_3.utc"},
	{0x0500CFA5ED938BB0ULL, "hlac_ht_store_187_3x.dds"},
	{0x05015F7DCFCE2868ULL, "t3_wpn_wht.tnt"},
	{0x050230C5DEAF28BEULL, "vxca_caverns1_mines_j20_0_2.vlm"},
	{0x05023F599764AA26ULL, "vxht_brothel_n_6uv_0_0.vlm"},
	{0x0502899EA954D0ABULL, "6018576_m.fxe"},
	{0x0502AB688C1ABD68ULL, "hlac_lt_0_lowtown_65534_2z.dds"},
	{0x0502EA5FA59376C2ULL, "868071912.ogg"},
	{0x050330317F1B2AF0ULL, "lt_0_lowtown_docks_a3_11_0.rml"},
	{0x050330D0BBBF7A95ULL, "llac_lt_0_lowtown_a3_n_c_65533_2.dds"},
	{0x0503519127223542ULL, "seb222ar_ruin.lst"},
	{0x05036E27330BEC52ULL, "zz_ave_debug6064144.cl"},
	{0x050396BC9DEC37F0ULL, "ht_brothel_n_3372.anb"},
	{0x0503BF6C7C2DE99EULL, "664560915.ogg"},
	{0x050479AB0BD721E3ULL, "art\\levels\\races\\ferelden\\props\\mining\\prp_burn2pil.msh"},
	{0x050487AB0BD7393DULL, "art\\levels\\races\\ferelden\\props\\mining\\prp_burn2pil.mmh"},
	{0x05048BAB0BD7404EULL, "art\\levels\\races\\ferelden\\props\\mining\\prp_burn2pil.mao"},
	{0x0504EE43C6C6F488ULL, "art\\vfx\\combat\\vfx_w_wns_shield_defen_l_c.impa.gad"},
	{0x0504F8815E6178FAULL, "art\\characters\\playercharacter\\qunarimale\\animation\\combat\\arishock\\ari_ability_trip.gad"},
	{0x0505DF8C7AE1B71EULL, "6072986_m.fxe"},
	{0x050613A967E024EEULL, "art\\levels\\races\\natural\\blightlands\\foliage\\tre_ashstumpsma.phy"},
	{0x050622712961134DULL, "gen00fl_carver6103887.cl"},
	{0x050628A1F19224EBULL, "6073785_f.fxe"},
	{0x05067877EF7CBAFAULL, "201366813.ogg"},
	{0x05067A01A8A306A6ULL, "vxw_sundermount_26c_0_0.vlm"},
	{0x0506810780D7EB50ULL, "hlac_lt_gallowstemplar_f_2391_9y.dds"},
	{0x0506974AD45160B4ULL, "sky_lt_0_lowtown_docks_a3.mao"},
	{0x0506B8A0136E4621ULL, "vxht_mansion2b_27o_0_0.vlm"},
	{0x0506E111A76A1BAAULL, "375618774.ogg"},
	{0x0506FF046F58317AULL, "6148089_m.fxe"},
	{0x05079FC988BA0CA9ULL, "36482741.ogg"},
	{0x05082C76B9A76168ULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\h_cwkr.ani"},
	{0x05087DC15EF85680ULL, "zz_mcr_debug6078868.cl"},
	{0x0508DF7E75358AC4ULL, "follower_banter6164071.cl"},
	{0x0509AA0230F6C608ULL, "867661756.ogg"},
	{0x050A0B7F96B84CAFULL, "6052101_m.fxe"},
	{0x050A0EB614FB7E0BULL, "vxca_caverns2_90a_0_1.vlm"},
	{0x050A7DC5E64D1363ULL, "190988183.ogg"},
	{0x050AB6E1CFBD226AULL, "43561290.ogg"},
	{0x050AFB815E65E911ULL, "art\\characters\\playercharacter\\qunarimale\\animation\\combat\\arishock\\ari_ability_trip.evt"},
	{0x050B38BC4D70D4B3ULL, "llac_ht_0_hightown_a2_65536_3.dds"},
	{0x050C67679EB55468ULL, "439717786.ogg"},
	{0x050CB32056BB992DULL, "llac_ht_keep_4519_1.dds"},
	{0x050CF93F444BBE2AULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\4so_straightl.phy"},
	{0x050D5156EFF8EA29ULL, "gen00fl_sebastian6179326.cl"},
	{0x050D6120043BA67BULL, "w_encounter2_mmi.gff"},
	{0x050D8647A987FD1DULL, "lgt320_mis_resp_swordhunter6150271.cl"},
	{0x050E81B79506800BULL, "gen_im_arm_cht_hvy_act1_01.uti"},
	{0x050E862F8EEB1B93ULL, "llac_lt_0_lowtown_docks_a3_n_0_3.dds"},
	{0x050F1EC253022B44ULL, "dae000_bartrand6206913.cl"},
	{0x050F3490EE880CCEULL, "vxht_bartrand_n_5mm_0_0.vlm"},
	{0x050FC48D769D51A1ULL, "gen00fl_bethany6101490.cl"},
	{0x050FCC2E01ED0761ULL, "vxdr_0_deeproads_gg1_0_0.vlm"},
	{0x0510342093E15155ULL, "art\\levels\\races\\proxy\\tilesets\\setext\\8co_basic\\0wo_diagonal.phy"},
	{0x05106496E3C8690FULL, "188606680.ogg"},
	{0x051094ECE5CDC5D5ULL, "vxlt_house_f3_0_0.vlm"},
	{0x0510ABE4C1136643ULL, "vxlt_0_lowtown_a2_n_c_hja_0_0.vlm"},
	{0x05111318A313A47BULL, "lt_0_lowtown_docks_a3_158_0.mmh"},
	{0x05111F18A313B818ULL, "lt_0_lowtown_docks_a3_158_0.mao"},
	{0x05113118A313D765ULL, "lt_0_lowtown_docks_a3_158_0.msh"},
	{0x0512726E95C01EC3ULL, "6093560_m.fxe"},
	{0x05127900B22656AAULL, "vxht_brothel_n_5bp_0_0.vlm"},
	{0x0512A7A1D6BA6372ULL, "plt_cod_ltr_junk_starkprinc.nss"},
	{0x0512E08148F4E9DBULL, "follower_banter6173113.cl"},
	{0x0512EBAEDE1D1419ULL, "hlac_ca_cavern1_haunted_24424_12x.dds"},
	{0x0513B0F1B01E2523ULL, "vrc321_varric6096939.cl"},
	{0x0513F7CBF5841579ULL, "art\\levels\\races\\dwarf\\props\\decor\\prp_potb.phy"},
	{0x0514012E5B59C71FULL, "gen00fl_anders6113084.cl"},
	{0x05140D375E241940ULL, "hlac_lt_0_lowtown_a2_n_c_-131073_0z.dds"},
	{0x05147ACA6ED99ED3ULL, "dae221ar_player_home.ncs"},
	{0x0514CCA761A02869ULL, "gen00fl_merrill6145142.cl"},
	{0x05151E26E81D0024ULL, "vxca_caverns2_657_0_0.vlm"},
	{0x051520E666F2250AULL, "6177331_m.fxe"},
	{0x051544FDB387D761ULL, "art\\characters\\creatures\\crowds\\humanmale\\textures\\prp_lowtownjimmy_t.dds"},
	{0x0515A67E93B793D1ULL, "6084712_f.fxe"},
	{0x05161776FDCF5799ULL, "w_blightlands_87_0.phy"},
	{0x05163C0104A26DAEULL, "darktown_fem_fereldan_sta6152806.cl"},
	{0x05169DD3C775CA9EULL, "hlac_lt_0_lowtown_-262143_0z.dds"},
	{0x051745B9F06E3BC1ULL, "153616720.ogg"},
	{0x051804244F294A5EULL, "6131162_m.fxe"},
	{0x0518A54A893F611BULL, "mag130_danzig6018810.cl"},
	{0x0518E067F74C14F8ULL, "sky_lt_hangedman.mao"},
	{0x0518E8F8CBD55240ULL, "vxht_mansion2_27j_0_0.vlm"},
	{0x051913815E724FE0ULL, "art\\characters\\playercharacter\\qunarimale\\animation\\combat\\arishock\\ari_ability_trip.ani"},
	{0x05195266E56BB72BULL, "vxht_0_hightown_b8l_0_0.vlm"},
	{0x05195BFB92F786D1ULL, "hm_guard_2.mor"},
	{0x0519956DFD49898CULL, "llac_lt_undercityencounter_-1_31.dds"},
	{0x0519EDC3CA4529C8ULL, "dae000_sandal6147765.cl"},
	{0x051ABA07DBF649A9ULL, "6099321_f.fxe"},
	{0x051B085899395386ULL, "6118390_m.fxe"},
	{0x051B1ACFC244924AULL, "6041687_m.fxe"},
	{0x051B43E4639AB8CAULL, "lt_0_lowtown_docks_19_0_lc.mmh"},
	{0x051BAC557760E3C2ULL, "681964974.ogg"},
	{0x051BFD46FA5BE9C2ULL, "dre000_hireling26038687.cl"},
	{0x051C43029FCAA4E1ULL, "vxht_mansion2_1pl_0_1.vlm"},
	{0x051C8F63D87CF60FULL, "mag170_thrask6054021.cl"},
	{0x051DA52181A0B1F9ULL, "w_bonepitdestroyed_a3_201_0.msh"},
	{0x051DAF2181A0C2E7ULL, "w_bonepitdestroyed_a3_201_0.mmh"},
	{0x051DB32181A0C934ULL, "w_bonepitdestroyed_a3_201_0.mao"},
	{0x051DDC274CAFB4A0ULL, "mag101_aombanter6052745.cl"},
	{0x051E4C1B3CA51F61ULL, "llac_w_encounter2_327681_0.dds"},
	{0x051FB3D25AD4AA76ULL, "nat380_temmerin6120948.cl"},
	{0x05201E28A865299CULL, "107776671.ogg"},
	{0x05202B0EFFFE1603ULL, "mcr370_samson6087532.cl"},
	{0x0520B881D3F6F2DBULL, "w_encounter2_camp_52_0.mmh"},
	{0x0520BF8C20961D98ULL, "312186403.ogg"},
	{0x0520C481D3F70778ULL, "w_encounter2_camp_52_0.mao"},
	{0x0520D681D3F725C5ULL, "w_encounter2_camp_52_0.msh"},
	{0x0522423A339E73EDULL, "hlac_lt_0_lowtown_docks_a3_n_262142_0x.dds"},
	{0x05224C9D1D2BD38DULL, "vxht_mansion2_31x_0_0.vlm"},
	{0x05226C462465CC1EULL, "104078808.ogg"},
	{0x0522A038EAC5AE79ULL, "zz_mer_debug6189048.cl"},
	{0x0522A1D403D468C3ULL, "6076554_m.fxe"},
	{0x0522ABCA64EE4980ULL, "6117814_m.fxe"},
	{0x0523B67056D5CF8CULL, "w_blightlands_51_0.rml"},
	{0x0523CCBF1CD2B898ULL, "notifications.gfx"},
	{0x05244D1651744CBCULL, "gen00fl_varric6093624.cl"},
	{0x052456EE82B78C4EULL, "6177853_f.fxe"},
	{0x0524C3CFBD282CEAULL, "6037624_m.fxe"},
	{0x0525EA114090F714ULL, "lowtown_male_urchin_sta6164208.cl"},
	{0x0526AE08F1D3A7F5ULL, "lt_warehouse_3_0.phy"},
	{0x05275AD5BDCB6AC0ULL, "vxht_mansion2_20k_0_0.vlm"},
	{0x0527790C1B46E516ULL, "1003392060.ogg"},
	{0x0528728B3AF0D5DFULL, "6105448_f.fxe"},
	{0x0528C2055F2EF4C5ULL, "vxlt_warehouse_n_ed_0_0.vlm"},
	{0x052997C629753E20ULL, "vxw_bonepitmining_a2_28n_0_0.vlm"},
	{0x0529A6A1DDC1D0B1ULL, "6106270-mrlact1_pt3_fire.cut"},
	{0x052A6177EEE00AF9ULL, "art\\characters\\creatures\\hurlock\\c_uni_hrlk_hed_cb.mao"},
	{0x052A7D27436C02BBULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\reactions\\generic exits\\genericexit_twohanded.evt"},
	{0x052B04AE70D3064DULL, "6083703_m.fxe"},
	{0x052B74B790BC5245ULL, "677076202.ogg"},
	{0x052B9AE9F5BE9E2AULL, "6129725_m.fxe"},
	{0x052BC5BDF7FAA2A8ULL, "vxlt_0_lowtown_docks_n_bdy_0_0.vlm"},
	{0x052C37F4C0321FC4ULL, "mag100_cullen6096070.cl"},
	{0x052C59B11E391175ULL, "isa211_isabela6197506.cl"},
	{0x052CCDC0B307E95EULL, "84352073.ogg"},
	{0x052CEB013E43FF06ULL, "vxlt_blackemporium_o1_0_1.vlm"},
	{0x052D57DCA4E2D666ULL, "hlac_lt_0_lowtown_docks_262142_0y.dds"},
	{0x052DAA7CAAE3F541ULL, "vxca_cavern1_haunted_ckg_0_0.vlm"},
	{0x052DB379F3F74066ULL, "vxlt_0_lowtown_a2_n_fim_0_0.vlm"},
	{0x052DC190BE302824ULL, "lt_0_lowtown_docks_a3_230_0.msh"},
	{0x052DCF08F1D9FE21ULL, "lt_warehouse_3_0.rml"},
	{0x052DD390BE30468DULL, "lt_0_lowtown_docks_a3_230_0.mao"},
	{0x052DD790BE304DC6ULL, "lt_0_lowtown_docks_a3_230_0.mmh"},
	{0x052E1ACF6D69AEEBULL, "dae000_mother6004120.cl"},
	{0x052E5AFB1CBD7253ULL, "plt_lgt000pt_baz_golemite.plo"},
	{0x052E698F6A856826ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\mage\\crushing_prison.evt"},
	{0x052EF178A6AD6000ULL, "vxht_mansion2b_346_0_0.vlm"},
	{0x052F0892341868FDULL, "vxht_0_hightown_api_0_0.vlm"},
	{0x052F2F03514AEF31ULL, "vxht_0_hightown_a3_c0z_0_0.vlm"},
	{0x052F31317F406758ULL, "lt_0_lowtown_docks_a3_11_0.anb"},
	{0x052F5CBA961287AAULL, "vxht_brothel_n_5hr_0_0.vlm"},
	{0x052FB92D739A7509ULL, "952357242.ogg"},
	{0x052FC47455F4BA7FULL, "vxlt_0_lowtown_docks_n_coz_0_0.vlm"},
	{0x05307627437062FCULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\reactions\\generic exits\\genericexit_twohanded.gad"},
	{0x05319279F0930C4BULL, "vxw_woundedcoast_q1_0_0.vlm"},
	{0x053228953F07EAE6ULL, "6072339_m.fxe"},
	{0x053229365E22EB2CULL, "w_bonepit_97_0.msh"},
	{0x05322F365E22F51EULL, "w_bonepit_97_0.mmh"},
	{0x05323B365E2309B5ULL, "w_bonepit_97_0.mao"},
	{0x05328479E6C321DDULL, "6137066_m.fxe"},
	{0x053349478AA30CD1ULL, "vxht_mansion_n_4y4_0_0.vlm"},
	{0x05337E797411E9A1ULL, "hlac_lt_gallowscourtyard_65534_2z.dds"},
	{0x0533B0BAC953E1EDULL, "vxw_woundedcoast_1cl_0_0.vlm"},
	{0x0533B2F31F8F8B6DULL, "gencr_high_dragon.utc"},
	{0x0533BBEE27CD8E9CULL, "6179007_m.fxe"},
	{0x0533D382C842E35AULL, "6102717_m.fxe"},
	{0x05342F8D9B9F7AC9ULL, "250953656.ogg"},
	{0x05346C1CE5DE4076ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\twohanded\\mh_m.mov_rf_leanr.evt"},
	{0x0534808F6A89FB71ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\mage\\crushing_prison.gad"},
	{0x0534A23FB83931F1ULL, "vxht_mansion2b_2o7_0_0.vlm"},
	{0x0534C2828CAD4F5DULL, "vfx_vfx_p2.vsh.fxo"},
	{0x0535A374B72F33BFULL, "592159521.ogg"},
	{0x0535AAB5214C0BCEULL, "w_encounter3_249_0.mmh"},
	{0x0535ADF8FCD6AF22ULL, "art\\characters\\playercharacter\\humanmale\\textures\\hn_arm_medb_0t.dds"},
	{0x0535B6B5214C2065ULL, "w_encounter3_249_0.mao"},
	{0x0535C4B5214C37BCULL, "w_encounter3_249_0.msh"},
	{0x0535D339BDDF2FA9ULL, "6051263_m.fxe"},
	{0x0535F443968DFADDULL, "hlac_ca_cavern1_haunted_24424_18z.dds"},
	{0x0535F5557E54F7A2ULL, "904977543.ogg"},
	{0x0536182B09389167ULL, "fen110_fenris6096425.cl"},
	{0x053621A9E917CC3BULL, "character_character_p51.psh.fxo.dx11.assembly"},
	{0x0536223B9FF1E190ULL, "gen00fl_anders6119119.cl"},
	{0x053650D49CC68A0CULL, "gen00fl_merrill6127224.cl"},
	{0x0536CA2FB93ABF15ULL, "vxht_0_hightown_a3_a71_0_0.vlm"},
	{0x0536CA4BD33B0A0CULL, "6081788_m.fxe"},
	{0x0537752743767FFAULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\reactions\\generic exits\\genericexit_twohanded.ani"},
	{0x053791D52292C77AULL, "vxlt_warehouse_n_yk_0_0.vlm"},
	{0x0537BD9B05869E90ULL, "738421465.ogg"},
	{0x053805818BB3BA55ULL, "hlac_lt_blackemporium_1055_12x.dds"},
	{0x05386EC97B90A0E7ULL, "vxht_mansion2_3ev_0_0.vlm"},
	{0x05391390605C2BDFULL, "hlac_lt_darktown_-1_37y.dds"},
	{0x05397545A24FD42EULL, "vxlt_0_lowtown_n_e2s_0_0.vlm"},
	{0x0539B12A7977850BULL, "seb201_lady_harimann.bnk"},
	{0x0539C2BD21C08E62ULL, "hla_ht_0_hightown_-65537_0y.dds"},
	{0x053A0502005FB8CAULL, "295059736.ogg"},
	{0x053A379AE9F0BB28ULL, "vxlt_0_lowtown_docks_n_bav_0_0.vlm"},
	{0x053A6D8D718CAAF3ULL, "vxlt_gallowscourtyard_6yz_0_0.vlm"},
	{0x053A769B217458AAULL, "vxlt_0_lowtown_docks_b4d_0_0.vlm"},
	{0x053A85119A840098ULL, "6134826_m.fxe"},
	{0x053AA0A1E7A49E15ULL, "hlac_lt_gallowstemplar_f_2388_3x.dds"},
	{0x053AB62F772F8F5DULL, "222187236.ogg"},
	{0x053B6F9D6DFAF881ULL, "vxlt_0_lowtown_docks_a3_n_c42_0_0.vlm"},
	{0x053B818F6A901B1FULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\mage\\crushing_prison.ani"},
	{0x053C241180BEC1B4ULL, "static_billboard_p0.psh.fxo.dx11.assembly"},
	{0x053C3F8983C61E7AULL, "vxlt_darktown_4uc_0_0.vlm"},
	{0x053CB5A157AA6186ULL, "art\\effects\\cloudshadows\\w_encounter2\\cloud1.dds"},
	{0x053CD360D87FAD8DULL, "vxlt_warehouse_qi_0_0.vlm"},
	{0x053CFD418BEAB09AULL, "gen_im_trs_dal_blt_r0.uti"},
	{0x053D7584077C4FCAULL, "1432136.ogg"},
	{0x053D83C4D26A4DFDULL, "vxca_caverns2_9fl_0_0.vlm"},
	{0x053D9BE026178E75ULL, "mag291cr_caress.utc"},
	{0x053DA0E0434107B1ULL, "vxht_0_hightown_a3_c5y_0_0.vlm"},
	{0x053DAB469F34F852ULL, "6102076_m.fxe"},
	{0x053DB32A797B71EBULL, "seb201_lady_harimann.cnv"},
	{0x053DDCD74A9A5E99ULL, "6127889_m.fxe"},
	{0x053E6C3725E9F3C7ULL, "follower_banter6166458.cl"},
	{0x053EBDE847FB6852ULL, "narrative_banter6149661.cl"},
	{0x053EE8496B9D25A8ULL, "hlac_lt_0_lowtown_a2_n_c_65533_2z.dds"},
	{0x053F5DDA28A726FDULL, "t1_mub_bk2.tnt"},
	{0x053F6BEB9EACAF79ULL, "art\\vfx\\combat\\vfx_c_2nd_buff_frd_loop_c.dura.ani"},
	{0x053F9F0B1D816BA5ULL, "561664353.ogg"},
	{0x05405CCAFC452C21ULL, "316572553.ogg"},
	{0x0540F8E87295D83EULL, "vxht_mansion_n_3yh_0_0.vlm"},
	{0x054113CEDB2132E7ULL, "264075058.ogg"},
	{0x0541802354001436ULL, "qun211_braggart6075704.cl"},
	{0x054191940F339DE1ULL, "fx_wavering_dis.mao"},
	{0x0541C51A4372113FULL, "105756324.ogg"},
	{0x05421B633139F94FULL, "dae300_bran6143503.cl"},
	{0x05422297BB49FA6DULL, "vxw_woundedcoast_1nk_0_0.vlm"},
	{0x0542253B0873578AULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\listen_surrounding_male.gad"},
	{0x05429487F767F8C0ULL, "artcharactersplayercharacterhumanmaleanimationblendtreehumanoid_damage_reactions.anb"},
	{0x0542C7BD46BED4B2ULL, "vxlt_0_lowtown_a3_n_c_gik_0_0.vlm"},
	{0x054325F71B256AAEULL, "pro000_varric6170652.cl"},
	{0x05435C5798E059C9ULL, "lt_0_lowtown_docks_probe_bw7.dds"},
	{0x054379EF77A24DA4ULL, "18332247.ogg"},
	{0x0544019165092E33ULL, "vxlt_gallowscourtyard_3t8_0_0.vlm"},
	{0x054478B54A7BAD68ULL, "vxca_caverns1_mines_gkn_0_0.vlm"},
	{0x0544E3FEB4192957ULL, "gen_im_acc_amu_act3_08.uti"},
	{0x0545025AC3ABEA35ULL, "hlac_ca_cavern1_haunted_24427_2z.dds"},
	{0x0545466D6F9F8375ULL, "vxht_bartrand_n_33z_0_0.vlm"},
	{0x0545A5BF431B25C5ULL, "vxlt_warehouse_n_ob_0_0.vlm"},
	{0x0545BA010DF5E4B2ULL, "6112962_f.fxe"},
	{0x0546370D25119506ULL, "36593040.ogg"},
	{0x054646EB9EB28EBBULL, "art\\vfx\\combat\\vfx_c_2nd_buff_frd_loop_c.dura.gad"},
	{0x05466CFECD7571B3ULL, "1058832196.ogg"},
	{0x054709F4FC5F8785ULL, "hlac_lt_0_lowtown_docks_n_-65538_7y.dds"},
	{0x054712A96817B9B1ULL, "art\\levels\\races\\natural\\blightlands\\foliage\\tre_ashstumpsma.mmh"},
	{0x054724A9C98CC492ULL, "6108067_m.fxe"},
	{0x05472CA96817E66FULL, "art\\levels\\races\\natural\\blightlands\\foliage\\tre_ashstumpsma.msh"},
	{0x05477A5CD6F14C2CULL, "crt_pro_02.mao"},
	{0x05478CC10049EEF8ULL, "6078991_m.fxe"},
	{0x05486030D976BFF1ULL, "6091883_m.fxe"},
	{0x0549083B08794521ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\listen_surrounding_male.evt"},
	{0x054921602331816CULL, "6106401_m.fxe"},
	{0x054932D199E31A1CULL, "vxca_caverns1_e7u_0_0.vlm"},
	{0x05493A22B8C60BFAULL, "prm000im_seeker_shield.uti"},
	{0x05495FE2E55A1BA5ULL, "6055164_m.fxe"},
	{0x0549691710576C13ULL, "art\\characters\\creatures\\corpse\\animation\\combat\\walkright.ani"},
	{0x0549982181C5D61CULL, "w_bonepitdestroyed_a3_201_0.phy"},
	{0x0549DA9C906A299EULL, "382166365.ogg"},
	{0x054A499CDED1F9CDULL, "vxca_cavern1_haunted_k2p_0_0.vlm"},
	{0x054A725CE57979BEULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\_textures\\lt_set_wood_n.dds"},
	{0x054A735888EBC843ULL, "vxlt_0_lowtown_a2_n_c_hve_0_0.vlm"},
	{0x054AB0F7AAB1E60DULL, "997244204.ogg"},
	{0x054AFD1EC780D426ULL, "mag291_vincento6027669.cl"},
	{0x054B453DCA75C60DULL, "vxlt_warehouse_zh_0_0.vlm"},
	{0x054B52A1F3C11B64ULL, "hlac_ht_mansion2b_818_46z.dds"},
	{0x054C05C91597B415ULL, "art\\characters\\weapons\\mauls\\w_mal_ml104.mao"},
	{0x054C886171D62300ULL, "gen00fl_aveline6139198.cl"},
	{0x054CF8CB3FF6E274ULL, "hlac_lt_warehouse_65536_3y.dds"},
	{0x054D6302946C8BF0ULL, "vxdr_0_deeproads_cuc_0_1.vlm"},
	{0x054D63EB9EB8DEA0ULL, "art\\vfx\\combat\\vfx_c_2nd_buff_frd_loop_c.dura.evt"},
	{0x054D900B00545B9EULL, "prp_str_dbone_d.dds"},
	{0x054E1F01320CDD9EULL, "177801756.ogg"},
	{0x054E5740219C4E19ULL, "lgt260_eau_tmp_dockworker6143602.cl"},
	{0x054F01BF19DC4089ULL, "art\\characters\\playercharacter\\humanfemale\\hf_har_he4a.phy"},
	{0x054F40E84A4ECA23ULL, "zz_dre_debug6056985.cl"},
	{0x054F897BA5D86AFCULL, "vxlt_darktown_60x_0_0.vlm"},
	{0x054FA401C9CCE7B8ULL, "hlac_lt_0_lowtown_-196608_2x.dds"},
	{0x054FFC0592E3FE29ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\fx_deeproads.mao"},
	{0x055014E5F43547B5ULL, "hf_warden.mor"},
	{0x05509474AD4012EDULL, "vxw_woundedcoast_1ij_0_0.vlm"},
	{0x0550A3851FBF9175ULL, "ht_keep_n.arl"},
	{0x0550DFA252B1C628ULL, "6067779_m.fxe"},
	{0x05514D5CD0632951ULL, "vxht_0_hightown_bxn_0_0.vlm"},
	{0x055151A357A6E737ULL, "dae000cr_commoner_d3.utc"},
	{0x055185F99C2562F1ULL, "vxht_mansion2b_2e5_0_0.vlm"},
	{0x0551C79807D22404ULL, "279932747.ogg"},
	{0x05521198DD9DBAD0ULL, "mrl251ip_loot_f.utp"},
	{0x0552FC1942D09CC4ULL, "577147096.ogg"},
	{0x05533A788CBBAA9AULL, "669509920.ogg"},
	{0x05538DC1A97FAFFBULL, "hlac_lt_0_lowtown_a3_n_c_0_0z.dds"},
	{0x0553AD5636016023ULL, "gen_im_trs_gen_one_mac_r0.uti"},
	{0x0553D095B80A18EDULL, "vxlt_0_lowtown_a3_n_c_eki_0_1.vlm"},
	{0x055412788BCE3B07ULL, "584854807.ogg"},
	{0x0554179C35113E45ULL, "vxlt_warehouse_n_ra_0_0.vlm"},
	{0x055447B92E163B48ULL, "vxlt_warehouse_n_b8_0_0.vlm"},
	{0x0554D311324079C7ULL, "gen_im_cns_cft_pot_05.uti"},
	{0x055502A8B41135B0ULL, "one300_zevran6089113.cl"},
	{0x055539FF95672CFBULL, "and111_lirene6084506.cl"},
	{0x0555BB0D0C5AEBB9ULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\dialogue\\lean_back_wall_exit.ani"},
	{0x0555FB83454BEA9CULL, "vxlt_backalley_g_1u6_0_0.vlm"},
	{0x0556003B0883C270ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\listen_surrounding_male.ani"},
	{0x055645E4DCD7EAF3ULL, "t3_wdg_dn3.tnt"},
	{0x0556620D84049775ULL, "vxlt_gallowstemplar_n_55r_0_0.vlm"},
	{0x05567426712CD664ULL, "vxlt_0_lowtown_a3_n_c_ep5_0_1.vlm"},
	{0x05572BD0F87FB896ULL, "hlac_ht_0_hightown_a2_65532_0x.dds"},
	{0x0557317B021A5298ULL, "vxht_keep_n_635_0_0.vlm"},
	{0x055733026A189AA7ULL, "t3_nbl_n03.tnt"},
	{0x05573EBE7E5610E2ULL, "6068004_m.fxe"},
	{0x05574720941D7470ULL, "art\\levels\\races\\proxy\\tilesets\\setext\\8co_basic\\0wo_diagonal.msh"},
	{0x05575520941D8B8AULL, "art\\levels\\races\\proxy\\tilesets\\setext\\8co_basic\\0wo_diagonal.mmh"},
	{0x055758669A2DCAAAULL, "hlac_dr_slavers_3214_2x.dds"},
	{0x05575920941D92C1ULL, "art\\levels\\races\\proxy\\tilesets\\setext\\8co_basic\\0wo_diagonal.mao"},
	{0x0557611710639C4AULL, "art\\characters\\creatures\\corpse\\animation\\combat\\walkright.evt"},
	{0x05578EF4BF0B1CC1ULL, "llac_lt_gallowscourtyard_131071_2.dds"},
	{0x0557C32BF5DA7D2AULL, "w_bonepitmining_a2_21_0_ll.mmh"},
	{0x0557F2C6771E5C97ULL, "336964561.ogg"},
	{0x0558CB8E73956AEEULL, "character_character_p98.psh.fxo.dx11.assembly"},
	{0x0558E47DEDA88C90ULL, "zz_mcr_debug6071544.cl"},
	{0x055941EAA8E4A37AULL, "hlac_w_bonepitdestroyed_a3_65538_0y.dds"},
	{0x0559EC0CB7D7A54FULL, "zz_approval_debug6133725.cl"},
	{0x0559FBDEAFE070D1ULL, "vxlt_gallowstemplar_5qa_0_0.vlm"},
	{0x055A577E6106E958ULL, "mrl150_dalish_guard6068739.cl"},
	{0x055A64921B5B95DDULL, "art\\vfx\\materials\\chunky\\vfx_c_fire_tile_add.mao"},
	{0x055AA40051FD9FB5ULL, "6167650_m.fxe"},
	{0x055AB22A4BB81440ULL, "185489731.ogg"},
	{0x055C19D602E50B6CULL, "vxca_cavern1_haunted_cny_0_0.vlm"},
	{0x055C451F9F750EABULL, "487173508.ogg"},
	{0x055C72B5DCED6F63ULL, "6057090_m.fxe"},
	{0x055C79D3455FB542ULL, "art\\vfx\\combat\\vfx_m_sta_ele_close_f_c.mmh"},
	{0x055C8FE8BDE5769CULL, "art\\cinematics\\_crit path\\mage crisis\\mcr370cs_attack_begins\\max files\\extraanims_mage2_.ani"},
	{0x055C960D0C60CBFBULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\dialogue\\lean_back_wall_exit.gad"},
	{0x055D235911AAC54EULL, "art\\vfx\\cinematic\\vfx_meredith_power_cin_c.cess.evt"},
	{0x055D3154D2129A83ULL, "6149784_m.fxe"},
	{0x055D3256AE928331ULL, "qun110_petrice6042299.cl"},
	{0x055DCC4F136A855CULL, "hf_ave1_banditf.mor"},
	{0x055E4017106982BDULL, "art\\characters\\creatures\\corpse\\animation\\combat\\walkright.gad"},
	{0x055E41FB1CE66BF0ULL, "plt_lgt000pt_baz_golemite.nss"},
	{0x055E5489C9A7BE30ULL, "ave121_brennan6105814.cl"},
	{0x055F1C8654EE854CULL, "636098286.ogg"},
	{0x055FB4A61FBD0ED1ULL, "hlac_w_encounter2_-131071_0y.dds"},
	{0x05600E8579A3819EULL, "325396927.ogg"},
	{0x05605017DB9E7951ULL, "620009819.ogg"},
	{0x0560F814F6B6C8AFULL, "135094375.ogg"},
	{0x0560FDD42301056BULL, "2381692.ogg"},
	{0x056120BD771CB6E1ULL, "6140244_m.fxe"},
	{0x056137D57E9D0BCAULL, "hlac_lt_gallowstemplar_f_2382_3y.dds"},
	{0x05618DB5EA4D22C2ULL, "34285563.ogg"},
	{0x0561A1C00810A608ULL, "gen00fl_anders6142537.cl"},
	{0x05622BD13CB08DB5ULL, "6088863_m.fxe"},
	{0x0562B30D0C6568E0ULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\dialogue\\lean_back_wall_exit.evt"},
	{0x0562D85925EA6D15ULL, "gen_im_trs_dwv_med_cht_r1.uti"},
	{0x05633C02A12F235EULL, "vxw_encounter2_ea_0_0.vlm"},
	{0x05639AFA694D867FULL, "vxlt_0_lowtown_a2_n_gpn_0_0.vlm"},
	{0x05639CE8BDEBABDEULL, "art\\cinematics\\_crit path\\mage crisis\\mcr370cs_attack_begins\\max files\\extraanims_mage2_.gad"},
	{0x05641A5911B0D4F9ULL, "art\\vfx\\cinematic\\vfx_meredith_power_cin_c.cess.gad"},
	{0x056432051C120CD9ULL, "513439048.ogg"},
	{0x05646D603742031CULL, "vxlt_backalley_g_1r7_0_0.vlm"},
	{0x0564BD5FF2CA7746ULL, "vxw_woundedcoast_cb_0_0.vlm"},
	{0x0564CB31C5719E31ULL, "qun131_party_comment6117262.cl"},
	{0x056509EC7FFDA377ULL, "mrl150_merrill6078328.cl"},
	{0x05659275B35E38EEULL, "art\\levels\\races\\proxy\\props\\foodstuffs\\prp_zzbarrelfood_0.pwk"},
	{0x056631AC3FD34935ULL, "155133054.ogg"},
	{0x0566552066BF3511ULL, "6067187_m.fxe"},
	{0x05666A715B77A768ULL, "563503190.ogg"},
	{0x056691F53EA53606ULL, "6196688_m.fxe"},
	{0x0566C647A8BE1469ULL, "6202218_m.fxe"},
	{0x056703017E369307ULL, "qcr000_arishok6086005.cl"},
	{0x056733BA9F4BCD5EULL, "6173793_m.fxe"},
	{0x05678AECD24583F2ULL, "vxlt_0_lowtown_docks_n_c06_0_0.vlm"},
	{0x056792D3AE0BB0D6ULL, "vxlt_backalley_g_1wf_0_0.vlm"},
	{0x05689AE8D481CED2ULL, "606976967.ogg"},
	{0x0568D981D434E010ULL, "w_encounter2_camp_52_0.phy"},
	{0x056981D16F7FD9DAULL, "702117156.ogg"},
	{0x05698DC9C7597950ULL, "vxlt_0_lowtown_fj1_0_0.vlm"},
	{0x05699D9A172F7062ULL, "vxlt_hangedman_n_2fr_0_0.vlm"},
	{0x056A049040B982FEULL, "289088724.ogg"},
	{0x056A2B5911B55D47ULL, "art\\vfx\\cinematic\\vfx_meredith_power_cin_c.cess.ani"},
	{0x056A77BB055B3EDAULL, "art\\vfx\\combat\\vfx_m_sta_spi_range_f_c.impa.gad"},
	{0x056AFCD07EACC212ULL, "mag210ip_store_arthuris.utp"},
	{0x056B0478502CE7F1ULL, "art\\characters\\creatures\\crowds\\humanmale\\textures\\prp_lowtownjimmy_d.dds"},
	{0x056B23CEFBFEAC6AULL, "hm_one_mekel.mor"},
	{0x056B5C006B895CC1ULL, "6174489_m.fxe"},
	{0x056B89ADF7A41B6DULL, "hlac_lt_0_lowtown_n_65534_6z.dds"},
	{0x056B8F2FA41963B5ULL, "hlac_w_bonepitdestroyed_a3_262144_0x.dds"},
	{0x056BBB1E47981578ULL, "hlac_w_sundermount_-327681_0y.dds"},
	{0x056C123037557571ULL, "qun211_braggart6075680.cl"},
	{0x056C66B8452938B4ULL, "vxht_0_hightown_a3_amv_0_0.vlm"},
	{0x056C783579D92374ULL, "llac_w_bonepit_262149_0.dds"},
	{0x056CEC3931CC073FULL, "gen00fl_anders6108509.cl"},
	{0x056D00275E394102ULL, "art\\characters\\weapons\\shortbows\\textures\\w_sbw_sb107a_0s.dds"},
	{0x056D175DCA76FF4CULL, "166447101.ogg"},
	{0x056DC108F20FCA10ULL, "lt_warehouse_3_0.msh"},
	{0x056DCAB696C6024AULL, "6217088_m.fxe"},
	{0x056DCF08F20FE22AULL, "lt_warehouse_3_0.mmh"},
	{0x056DD308F20FE961ULL, "lt_warehouse_3_0.mao"},
	{0x056E7AE64BD2D11BULL, "vxlt_gallowstemplar_n_51x_0_0.vlm"},
	{0x056E97B0B78C9F0BULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\dialogue\\wall_lean_left_loop.ani"},
	{0x056F0885DA4F521AULL, "6129685_f.fxe"},
	{0x056F0C6A483D70C3ULL, "139298434.ogg"},
	{0x056F6621B744BE92ULL, "gen00fl_aveline6120954.cl"},
	{0x056F701A9AB5B7F0ULL, "6084695_m.fxe"},
	{0x056FB0B7542FE9FEULL, "vxlt_0_lowtown_a3_n_c_cmo_0_0.vlm"},
	{0x056FB46E7B17E59BULL, "hlac_ht_0_hightown_a3_65536_1y.dds"},
	{0x056FB881D43AC768ULL, "w_encounter2_camp_52_0.rml"},
	{0x057084635EE1A54BULL, "follower_banter6165617.cl"},
	{0x0570B44F9BF649EDULL, "vxlt_0_lowtown_a3_n_c_eqg_0_1.vlm"},
	{0x05716622B4E68FD7ULL, "gen00fl_anders6202353.cl"},
	{0x05717ABB056162F1ULL, "art\\vfx\\combat\\vfx_m_sta_spi_range_f_c.impa.evt"},
	{0x0571B1DDB7A8511CULL, "vxw_bonepitmining_a2_1hy_0_0.vlm"},
	{0x05720CD75B439EFFULL, "vxlt_0_lowtown_a2_n_gmi_0_0.vlm"},
	{0x05722BB9DE463EE1ULL, "qun111_ketojan6040955.cl"},
	{0x0572B19DB84B512FULL, "6055276_m.fxe"},
	{0x0572D92C533A42DFULL, "6153884_m.fxe"},
	{0x0572FCA36C6CE288ULL, "vxlt_warehouse_11i_0_0.vlm"},
	{0x0573FFB24F7A503DULL, "art\\characters\\playercharacter\\humanmale\\textures\\hn_arm_masc_0s.dds"},
	{0x0574577B6452EB88ULL, "681305092.ogg"},
	{0x05747AA4F6C91263ULL, "778003310.ogg"},
	{0x0574F5FEC7DB72AEULL, "dae321ar_chantry.ncs"},
	{0x0575154EAF15AB14ULL, "follower_banter6156203.cl"},
	{0x05752827A7E95EFFULL, "6102705_m.fxe"},
	{0x0575567457F8E0DAULL, "ts_coterie_arm_props.gda"},
	{0x057635EDD87C5E12ULL, "vxlt_gallowstemplar_37j_0_0.vlm"},
	{0x0576FDD1D883B0C0ULL, "860479082.ogg"},
	{0x0577DBE1D121BDBBULL, "vxht_0_hightown_a3_90b_0_0.vlm"},
	{0x05780F77092588E2ULL, "vxlt_hangedman_n_2ms_0_0.vlm"},
	{0x05783098FA0A6531ULL, "781775349.ogg"},
	{0x05783300D907E29CULL, "steam_01.mao"},
	{0x0578D617548CE18DULL, "hla_ht_0_hightown_a2_-65537_0z.dds"},
	{0x0578F5A59573FFCAULL, "vxlt_0_lowtown_docks_a3_n_aon_0_0.vlm"},
	{0x057940A53EC212DDULL, "art\\characters\\playercharacter\\humanfemale\\bdy_rob_magb.mao"},
	{0x05794291C9B386DAULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\7rb_basic.mmh"},
	{0x05794691C9B38D11ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\7rb_basic.mao"},
	{0x05795491C9B3A520ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\7rb_basic.msh"},
	{0x05797E18E4E39D11ULL, "148407676.ogg"},
	{0x0579C9AA87C48DABULL, "6217966_m.fxe"},
	{0x0579FB665A34616EULL, "art\\vfx\\environment\\vfx_trap_floor_spike_p.impa.evt"},
	{0x057A070151574445ULL, "gen00fl_fenris6126761.cl"},
	{0x057AC14764F9C756ULL, "hlac_ht_0_hightown_a2_65535_1z.dds"},
	{0x057AD895371F5134ULL, "vxht_0_hightown_a3_ajw_0_0.vlm"},
	{0x057AEABF1A0154AEULL, "art\\characters\\playercharacter\\humanfemale\\hf_har_he4a.mmh"},
	{0x057B04BF1A01809CULL, "art\\characters\\playercharacter\\humanfemale\\hf_har_he4a.msh"},
	{0x057B146A15558868ULL, "llac_dr_primevalentrance_1726_11.dds"},
	{0x057B60029394552BULL, "6095485_m.fxe"},
	{0x057BA0001BABA076ULL, "vxlt_foundry_n_vt_0_0.vlm"},
	{0x057BEDFEC7E18387ULL, "dae321ar_chantry.lst"},
	{0x057C16A25511D115ULL, "vxht_0_hightown_a3_byt_0_0.vlm"},
	{0x057C2D01CA6FFFCAULL, "composedof.psh.fxo"},
	{0x057C7FB0B798B382ULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\dialogue\\wall_lean_left_loop.evt"},
	{0x057C7FB20C4EC3C3ULL, "vxca_caverns2prc_a9r_0_0.vlm"},
	{0x057CCBB7DF930859ULL, "6132713_m.fxe"},
	{0x057CDAD41139B57DULL, "hlac_dr_primevaltomb_658_2z.dds"},
	{0x057D15334464F084ULL, "6133140_m.fxe"},
	{0x057D9136D1718924ULL, "t1_mue_bl1.tnt"},
	{0x057E4A99E2C94595ULL, "168434962.ogg"},
	{0x057E92BB056C15C0ULL, "art\\vfx\\combat\\vfx_m_sta_spi_range_f_c.impa.ani"},
	{0x057E95F0A52F2552ULL, "qcr000_arishok6060183.cl"},
	{0x057EBBBCCE4E8291ULL, "follower_banter6161013.cl"},
	{0x057EFE4F543385D8ULL, "gen00fl_dog6165777.cl"},
	{0x057F303A5B8CB5D4ULL, "one220_yevhen6200154.cl"},
	{0x057FBED894A43EE4ULL, "art\\levels\\races\\elf\\props\\_textures\\prp_hangingb_d.dds"},
	{0x057FF2665A38BD19ULL, "art\\vfx\\environment\\vfx_trap_floor_spike_p.impa.gad"},
	{0x05801FBC851B545EULL, "vxw_encounter2_kc_0_0.vlm"},
	{0x05806E6017A0B201ULL, "dae000_arishok6067730.cl"},
	{0x0580A2A746E8EBA1ULL, "102207188.ogg"},
	{0x05816E805E62FB08ULL, "vxlt_warehouse_14f_0_0.vlm"},
	{0x0581A119D6B6A846ULL, "vxw_woundedcoast_ud_0_0.vlm"},
	{0x05821FDEA367FA3EULL, "6049654_m.fxe"},
	{0x05824A24E65AFD30ULL, "dae000_arishok6072890.cl"},
	{0x0582D32E6C41382AULL, "mag110_arianni6015761.cl"},
	{0x05834F49E3E480C0ULL, "758310862.ogg"},
	{0x05837EB0B79ED095ULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\dialogue\\wall_lean_left_loop.gad"},
	{0x058394ECDCA42931ULL, "art\\vfx\\combat\\vfx_glb_com_disorient_s2_c.anb"},
	{0x05839A0499DF777FULL, "6107805_m.fxe"},
	{0x05839C38EF5D4111ULL, "6166208_m.fxe"},
	{0x05839F03F46A4586ULL, "hightown_male_noble_gossip6171431.cl"},
	{0x05843EAA7BAB2E9CULL, "6054874_m.fxe"},
	{0x058441CF813DABDDULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_nos07b.phy"},
	{0x0584489D108E5CFDULL, "llac_lt_0_lowtown_a3_n_c_-2_3.dds"},
	{0x05844D44B9E88596ULL, "mag200_feynriel6023889.cl"},
	{0x0584768D91F7E1D6ULL, "vxlt_backalley_g_1yh_0_0.vlm"},
	{0x0585054EEFFE747DULL, "dae000cr_ineria.utc"},
	{0x058578D34582B8E9ULL, "art\\vfx\\combat\\vfx_m_sta_ele_close_f_c.anb"},
	{0x0585B34798409E45ULL, "one300_murder_followers6089902.cl"},
	{0x05860F23C1532B8FULL, "l_env_wal_lowtown_int_tavern_bed.wbd"},
	{0x058612901CF52C64ULL, "hf_zmother_h3s2.mor"},
	{0x0586C7BA323BE370ULL, "777137567.ogg"},
	{0x058703665A3EF967ULL, "art\\vfx\\environment\\vfx_trap_floor_spike_p.impa.ani"},
	{0x058770F7BF2E827AULL, "art\\characters\\playercharacter\\humanmale\\textures\\hm_cth_nobd_0s.dds"},
	{0x05887AC62A49E326ULL, "vxw_sundermount_2cl_0_0.vlm"},
	{0x05891EC40FFCEEC0ULL, "6044821_f.fxe"},
	{0x058A39A1815B902EULL, "6139917_m.fxe"},
	{0x058A5AF382C8222FULL, "one140cr_charade6166014.cl"},
	{0x058AE5ECA69A43CBULL, "6140286_m.fxe"},
	{0x058B5345DFA95BABULL, "ht_0_hightown_a3_16_0_lc.mmh"},
	{0x058B7DA9BA3C0289ULL, "gen00fl_aveline6104957.cl"},
	{0x058C265275497936ULL, "vxht_0_hightown_a3_a6y_0_0.vlm"},
	{0x058CEEFEC7EFF8BCULL, "dae321ar_chantry.are"},
	{0x058D07EBD97DF876ULL, "hlac_lt_0_lowtown_docks_a3_n_131071_1z.dds"},
	{0x058D37C9BF609A35ULL, "407091354.ogg"},
	{0x058D60C55B430EBCULL, "art\\levels\\races\\proxy\\tilesets\\setint\\_textures\\ben_booster_d.dds"},
	{0x058E04335A10A16FULL, "637125805.ogg"},
	{0x058E0F2CF5EE9D48ULL, "vxlt_warehouse_n_v4_0_0.vlm"},
	{0x058E5867F852D69DULL, "ts_books_01.gda"},
	{0x058E8FA73BB7C28FULL, "510220805.ogg"},
	{0x058EA9643DF80DAAULL, "follower_banter6155802.cl"},
	{0x058F04DA5A886040ULL, "cod_hst_amell.utp"},
	{0x058F06D0BE6F6633ULL, "626516320.ogg"},
	{0x058F55D8A7F0C824ULL, "01_dust.mao"},
	{0x0590B5ECDCAEEBAAULL, "art\\vfx\\combat\\vfx_glb_com_disorient_s2_c.mmh"},
	{0x0590C7328C5B1AF2ULL, "static_static_p3.psh.fxo.dx11.assembly"},
	{0x0590D94B8ECD771AULL, "art\\characters\\playercharacter\\humanmale\\cth_visa.phy"},
	{0x05912C1EBD2B9BD5ULL, "art\\vfx\\combat\\vfx_w_off_stun_chc_p_p.anb"},
	{0x0591B50905BC8884ULL, "6163808_m.fxe"},
	{0x0591C5AF6C2AAA89ULL, "30422172.ogg"},
	{0x05929A0801BBB7B9ULL, "art\\characters\\playercharacter\\humanfemale\\cth_nobc.msh"},
	{0x0592A40801BBC8A7ULL, "art\\characters\\playercharacter\\humanfemale\\cth_nobc.mmh"},
	{0x0592A80801BBCFF4ULL, "art\\characters\\playercharacter\\humanfemale\\cth_nobc.mao"},
	{0x059435C912FC06C1ULL, "vxlt_0_lowtown_a2_n_c_e84_0_0.vlm"},
	{0x05944295C41D2887ULL, "dae110ip_at_hh_house.utp"},
	{0x059542CF814C20E2ULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_nos07b.mmh"},
	{0x059544CF814C2398ULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_nos07b.msh"},
	{0x059578820C900A5AULL, "mag110_vincento6016322.cl"},
	{0x0595B086E9DB44C4ULL, "221793484.ogg"},
	{0x0595FE266DC1B164ULL, "142796207.ogg"},
	{0x059693125B647492ULL, "461123575.ogg"},
	{0x0596ECA31C3FFBA6ULL, "vxw_sundermount_2dm_0_0.vlm"},
	{0x0596FE24C01CB4A3ULL, "llac_lt_gallowstemplar_f_2388_1.dds"},
	{0x059722E1969F1E3DULL, "gen00fl_fenris6110823.cl"},
	{0x05974D6400EEB683ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\boss\\meredith\\combat\\energyburst.ani"},
	{0x05976C7BE4F5CE0AULL, "hlac_ht_playershouse_n_855_6y.dds"},
	{0x059793E47DFB3A79ULL, "vxca_caverns1_mines_e8l_0_0.vlm"},
	{0x0597AE21E74AA269ULL, "gen00fl_merrill6129881.cl"},
	{0x0597C2C3B58A910DULL, "vxw_encounter3_6x_0_0.vlm"},
	{0x0598742187292003ULL, "vxht_chantry_26v_0_0.vlm"},
	{0x0598BBECA64A7DECULL, "ave331_post_fight6138805.cl"},
	{0x0598C1562640DEFFULL, "95581331.ogg"},
	{0x0598FA5C38FE0215ULL, "vxht_0_hightown_a3_bsr_0_0.vlm"},
	{0x059910EBFBA1BA2DULL, "504325556.ogg"},
	{0x05992E4377A78AF3ULL, "vxht_keep_6y3_0_0.vlm"},
	{0x0599987D8C8CACE0ULL, "288114523.ogg"},
	{0x0599BFFF22C8BE1FULL, "vxw_sundermount_2j2_0_0.vlm"},
	{0x059A2D396DEDC9D3ULL, "6162123_m.fxe"},
	{0x059B0263A9C54066ULL, "llac_w_encounter2_262149_0.dds"},
	{0x059B03FB7C6DC7FBULL, "art\\levels\\races\\ferelden\\props\\lowerclass\\prp_cooker.msh"},
	{0x059B117EFF5DC657ULL, "vxlt_gallowsprison_n_6ga_0_0.vlm"},
	{0x059B15FB7C6DE646ULL, "art\\levels\\races\\ferelden\\props\\lowerclass\\prp_cooker.mao"},
	{0x059B21FB7C6DFAE5ULL, "art\\levels\\races\\ferelden\\props\\lowerclass\\prp_cooker.mmh"},
	{0x059C5871F3EC1EB3ULL, "t1_mue_or1.tnt"},
	{0x059C9BF334AFF696ULL, "vxw_blightlands_a6_0_0.vlm"},
	{0x059CF084ABD7CB58ULL, "and100_anders6095954.cl"},
	{0x059D2578DBD67A2EULL, "6068857_m.fxe"},
	{0x059D656AC57AC3C2ULL, "fen210_orana6109765.cl"},
	{0x059DA3A0AE3F63A4ULL, "6065250_m.fxe"},
	{0x059DBFD8AEC47115ULL, "649703626.ogg"},
	{0x059DC51125BD4A92ULL, "art\\levels\\races\\proxy\\tilesets\\setext\\4co_basic\\1wo_diagonal_1.mao"},
	{0x059DEDA6A46A473CULL, "6000798_m.fxe"},
	{0x059E36C3417C89D3ULL, "vxw_bonepitmining_a2_5n_0_0.vlm"},
	{0x059EB2EAC2C4461DULL, "346517612.ogg"},
	{0x059F0DEE669D6C25ULL, "ser270_emeric6113223.cl"},
	{0x059F39B719684276ULL, "llac_w_bonepitmining_a2_327684_0.dds"},
	{0x059F6191C9D48AE5ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\7rb_basic.phy"},
	{0x05A08DC8145B07A2ULL, "vxw_sundermount_317_0_0.vlm"},
	{0x05A0D224176A84A1ULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_quill_d.dds"},
	{0x05A0D611B953740EULL, "mcr370_negotiation6080170.cl"},
	{0x05A0FB3234D9E3ABULL, "qcr000_arishok6060108.cl"},
	{0x05A13EA1C895999EULL, "summon.ncs"},
	{0x05A177F0533521E8ULL, "hlac_ca_caverns1_mines_24424_19z.dds"},
	{0x05A24A28D3F2E9A0ULL, "gen00fl_anders6111083.cl"},
	{0x05A26584C03E52A4ULL, "6086932_m.fxe"},
	{0x05A29039D63903BAULL, "6078795_m.fxe"},
	{0x05A2DFDFE237C5D4ULL, "art\\levels\\races\\design\\animation\\gasbarrel.inactivetoactive.gad"},
	{0x05A41216699738F3ULL, "hlac_lt_gallowstemplar_f_2395_0y.dds"},
	{0x05A452082FD7F62CULL, "lt_0_lowtown_n_probe_gp0.dds"},
	{0x05A4656400F96A7AULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\boss\\meredith\\combat\\energyburst.evt"},
	{0x05A47DFBE0C03E57ULL, "6016695_m.fxe"},
	{0x05A4BEC57A8774D9ULL, "vxht_keep_63b_0_1.vlm"},
	{0x05A55E800E361426ULL, "vxw_sundermount_2in_0_0.vlm"},
	{0x05A5953316062351ULL, "74674018.ogg"},
	{0x05A6916B31F09C6CULL, "vxlt_0_lowtown_docks_8rn_0_0.vlm"},
	{0x05A6AD3DFD075FD7ULL, "6174810_m.fxe"},
	{0x05A71C536311D4FAULL, "6126698_m.fxe"},
	{0x05A7A020699DA373ULL, "vxht_keep_6b2_0_0.vlm"},
	{0x05A7B77902D12852ULL, "vxw_bonepitmining_a2_5k_0_0.vlm"},
	{0x05A813C46D6218B7ULL, "6137464_m.fxe"},
	{0x05A821F69B9D43A7ULL, "ico_maul_5.dds"},
	{0x05A82F3F062EF89BULL, "zz_qun_debug6028219.cl"},
	{0x05A82FB4C599F171ULL, "6088337_m.fxe"},
	{0x05A843F36C5481F3ULL, "hlac_lt_0_lowtown_docks_a3_n_-131075_5x.dds"},
	{0x05A89F165FE3E253ULL, "6167223_m.fxe"},
	{0x05A90A28D8D8EFDCULL, "406088451.ogg"},
	{0x05A91F7E7A1FA40FULL, "6105989_f.fxe"},
	{0x05A9C82C69056018ULL, "hlac_lt_undercityserialkiller_-1_10y.dds"},
	{0x05A9EEDFE23DFDB2ULL, "art\\levels\\races\\design\\animation\\gasbarrel.inactivetoactive.ani"},
	{0x05AA0BE6B5370932ULL, "562094675.ogg"},
	{0x05AA0DF8196EAF3EULL, "hlac_w_bonepitdestroyed_a3_196608_0x.dds"},
	{0x05AA9834DF443B99ULL, "art\\characters\\creatures\\ragedemon\\animation\\combat\\stealth.ani"},
	{0x05AB0DD026A60F16ULL, "vxw_blightlands_j7_0_0.vlm"},
	{0x05AB4662E4F0C2DBULL, "art\\vfx\\combat\\vfx_m_spi_crh_aoe_p.impa.ani"},
	{0x05AB646400FF876DULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\boss\\meredith\\combat\\energyburst.gad"},
	{0x05ABCAA9F40181D2ULL, "hlac_lt_0_lowtown_docks_a3_n_-131076_1x.dds"},
	{0x05ABDA0F000AD45CULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\death_cradle_action.ani"},
	{0x05AC2808891C2F4FULL, "6206875_f.fxe"},
	{0x05AC3203D23A2969ULL, "6076216_f.fxe"},
	{0x05ACE668626BB4B4ULL, "6039307_m.fxe"},
	{0x05ACF3D071355CEBULL, "797050997.ogg"},
	{0x05AD27F3EEC122A5ULL, "151698944.ogg"},
	{0x05ADBD2201B8E702ULL, "gen_im_trs_crc_lgt_glv_r2.uti"},
	{0x05ADE4F013EF5E67ULL, "vxw_encounter3_bv_0_0.vlm"},
	{0x05AE4115BE26F1EAULL, "667201325.ogg"},
	{0x05AE501FB37E1329ULL, "vxlt_0_lowtown_docks_a3_8pf_0_0.vlm"},
	{0x05AF20AA63AC93A0ULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_eaglebody_n.dds"},
	{0x05AFCDCDE06B2732ULL, "6087432_m.fxe"},
	{0x05AFE22D9D4095FBULL, "578924389.ogg"},
	{0x05B02FDAB4BF9619ULL, "art\\characters\\playercharacter\\shared\\heads\\textures\\mt_hed_scra_0n.dds"},
	{0x05B0625440BF2197ULL, "532048302.ogg"},
	{0x05B0CDCAB711186CULL, "llac_ht_mansion_n_3382_0.dds"},
	{0x05B12D5C967FE398ULL, "vxlt_undercitydungeon_14d_0_0.vlm"},
	{0x05B17334DF4A1BDBULL, "art\\characters\\creatures\\ragedemon\\animation\\combat\\stealth.gad"},
	{0x05B185F4D372F23BULL, "mag360_alain6044616.cl"},
	{0x05B1D485F5D5CD95ULL, "425126052.ogg"},
	{0x05B1D79FA6A5F3A2ULL, "6083146_m.fxe"},
	{0x05B1E70F000F559EULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\death_cradle_action.gad"},
	{0x05B247A41F3C9C39ULL, "art\\cinematics\\_crit path\\prologue\\pro000cs_avalineintro_real\\avalineintro_hurlock2.ani"},
	{0x05B2A74F35559BDFULL, "6032198_m.fxe"},
	{0x05B2D7A89D80C243ULL, "pro000_cut2d_voyage6005291.cl"},
	{0x05B2ECD06D68FA3FULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_paintingsmla.msh"},
	{0x05B2F2D06D6904E1ULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_paintingsmla.mmh"},
	{0x05B2FED06D691882ULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_paintingsmla.mao"},
	{0x05B34BC466F7579CULL, "266367170.ogg"},
	{0x05B393F1350AE539ULL, "plt_cod_bks_thrask_notes.nss"},
	{0x05B3D05D002C2CA6ULL, "vxw_sundermount_2ro_0_0.vlm"},
	{0x05B3D625CA63B8A2ULL, "276213801.ogg"},
	{0x05B4238F852C9865ULL, "712017955.ogg"},
	{0x05B43121A4975B61ULL, "mag291_torpor6026935.cl"},
	{0x05B4DED6B89732D0ULL, "vxht_mansion_n_5mh_0_0.vlm"},
	{0x05B4E3BB6DE56266ULL, "198921679.ogg"},
	{0x05B4F838330E748CULL, "501098935.ogg"},
	{0x05B6A191B79009F1ULL, "6083237_m.fxe"},
	{0x05B6AABD8CE67E10ULL, "vrc221_varric6175232.cl"},
	{0x05B7587408635C48ULL, "art\\vfx\\combat\\vfx_glb_ele_f_deathblow_c.anb"},
	{0x05B76605DFAAF591ULL, "vxht_mansion2_3cz_0_0.vlm"},
	{0x05B7A928A2FDD87FULL, "hlac_lt_0_lowtown_a2_n_c_-2_1y.dds"},
	{0x05B7C5EB4D4AF2C3ULL, "dae200_bran6094897.cl"},
	{0x05B7F538E349F757ULL, "vxlt_gallowsprison_n_6mc_0_0.vlm"},
	{0x05B89034DF506BC0ULL, "art\\characters\\creatures\\ragedemon\\animation\\combat\\stealth.evt"},
	{0x05B8C20F001535BDULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\death_cradle_action.evt"},
	{0x05B922A41F427C7BULL, "art\\cinematics\\_crit path\\prologue\\pro000cs_avalineintro_real\\avalineintro_hurlock2.gad"},
	{0x05B92D1EBD4CEF16ULL, "art\\vfx\\combat\\vfx_w_off_stun_chc_p_p.mmh"},
	{0x05B92D79154E3FB1ULL, "22607826.ogg"},
	{0x05B92DD369271A91ULL, "vxlt_gallowscourtyard_n_7ev_0_0.vlm"},
	{0x05B94E62E4FD0DD2ULL, "art\\vfx\\combat\\vfx_m_spi_crh_aoe_p.impa.evt"},
	{0x05B97FAD189C2796ULL, "vxw_blightlands_o8_0_0.vlm"},
	{0x05B9B646C5DD6757ULL, "689130196.ogg"},
	{0x05BA139F53F6E571ULL, "gen00fl_isabela6101546.cl"},
	{0x05BA339CB95F7A88ULL, "qcr221_cut_isabelas_return6042635.cl"},
	{0x05BA7AF13510D912ULL, "plt_cod_bks_thrask_notes.plo"},
	{0x05BAB60B4DA98AA4ULL, "vxw_encounter2_2b_0_0.vlm"},
	{0x05BAB930B4768141ULL, "333270957.ogg"},
	{0x05BAFA59093292CFULL, "vxht_mansion2b_3bs_0_0.vlm"},
	{0x05BB18B37B756982ULL, "mer220_hubert6049752.cl"},
	{0x05BBFF19A4ADA2B1ULL, "6108020_m.fxe"},
	{0x05BC3010BD954C7BULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\cower_loop.ani"},
	{0x05BC68AC969879ADULL, "6108030_f.fxe"},
	{0x05BCBB749922C39CULL, "hlac_w_encounter3_-262146_0y.dds"},
	{0x05BCD8778C287DFBULL, "spy100pt_among_shadows.ncs"},
	{0x05BD0C8A3187CDB4ULL, "hlac_lt_0_lowtown_a3_n_c_65534_0z.dds"},
	{0x05BD17A7581FB67BULL, "llac_lt_gallowstemplar_f_2391_3.dds"},
	{0x05BDB4C0046AEBE8ULL, "vxw_bonepitmining_a2_e1_0_0.vlm"},
	{0x05BE3CA4DD15E192ULL, "6108236_m.fxe"},
	{0x05BE8D0801E0DCDCULL, "art\\characters\\playercharacter\\humanfemale\\cth_nobc.phy"},
	{0x05BEA038CBD23C89ULL, "hlac_lt_gallowstemplar_2388_5y.dds"},
	{0x05BEA61C1C65E7C5ULL, "art\\vfx\\combat\\vfx_w_jug_dam_redu_loop_c.anb"},
	{0x05BF2955C7255C7FULL, "art\\characters\\weapons\\maces\\textures\\w_mce_mc101a_0n.dds"},
	{0x05BF40687732D5DBULL, "vxw_bonepit_247_0_0.vlm"},
	{0x05BF4D0BF1752E8FULL, "art\\vfx\\environment\\vfx_blt_fire_wall_p.cess.gad"},
	{0x05BF5789F7A17870ULL, "31463962.ogg"},
	{0x05BFE74875D71C41ULL, "204157844.ogg"},
	{0x05C06D62E5036165ULL, "art\\vfx\\combat\\vfx_m_spi_crh_aoe_p.impa.gad"},
	{0x05C071534D283B14ULL, "art\\vfx\\combat\\vfx_r_var_modal_act_c.mmh"},
	{0x05C0D9B17A10B018ULL, "vxca_cavern1_haunted_hic_0_0.vlm"},
	{0x05C0E53DA9133075ULL, "vxlt_0_lowtown_docks_n_b66_0_0.vlm"},
	{0x05C10461F628C105ULL, "hlac_ht_mansion_n_2995_0x.dds"},
	{0x05C14B7F46608485ULL, "vxlt_0_lowtown_a3_n_c_ffx_0_0.vlm"},
	{0x05C1927B973AE374ULL, "llac_lt_0_lowtown_docks_a3_n_196607_1.dds"},
	{0x05C1D51983D04BCFULL, "6199327_m.fxe"},
	{0x05C1E9C4D9ADA932ULL, "t3_wdg_w14.tnt"},
	{0x05C23D7EB5BD33F5ULL, "6188569_m.fxe"},
	{0x05C24F5B8438BC43ULL, "515519104.ogg"},
	{0x05C250B959F64CDDULL, "art\\levels\\races\\dwarf\\props\\_textures\\prp_stand_sa.dds"},
	{0x05C2FA7522ED04ABULL, "isa211_isabela6175166.cl"},
	{0x05C2FCC929B3EA64ULL, "6068015_m.fxe"},
	{0x05C30FC947988D06ULL, "vxdr_primevaltomb_35b_0_0.vlm"},
	{0x05C34A1E7390A435ULL, "6162741_m.fxe"},
	{0x05C38EF569E9107FULL, "art\\levels\\races\\elf\\props\\plotspecifc\\prp_eluvianbroke_0.pwk"},
	{0x05C3BB4F84829346ULL, "6034279_f.fxe"},
	{0x05C4000D46FE05FAULL, "6128498_m.fxe"},
	{0x05C404479733736FULL, "vxht_keep_n_60v_0_1.vlm"},
	{0x05C47A3CC5F43D38ULL, "329491671.ogg"},
	{0x05C483DA4D89D473ULL, "vxht_keep_6h4_0_0.vlm"},
	{0x05C53763D2062A62ULL, "plt_lgt000pt_ultra_1_05.nss"},
	{0x05C53D3E7EC4DF16ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\1ww_gatetopright.mao"},
	{0x05C5493E7EC4F3B5ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\1ww_gatetopright.mmh"},
	{0x05C54B3E7EC4F72BULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\1ww_gatetopright.msh"},
	{0x05C5620BF179BE3CULL, "art\\vfx\\environment\\vfx_blt_fire_wall_p.cess.evt"},
	{0x05C596CBC506C701ULL, "6134975_m.fxe"},
	{0x05C5D7E2D1A10E11ULL, "vxht_mansion2_3fy_0_0.vlm"},
	{0x05C5E4F9615086ADULL, "zz_lgt_debug6109146.cl"},
	{0x05C62923C1893C65ULL, "l_env_wal_lowtown_int_tavern_bed.bnk"},
	{0x05C63BF477967388ULL, "llac_w_sundermount_-65531_0.dds"},
	{0x05C68186A11E27CCULL, "vxw_bonepitmining_a2_8r_0_0.vlm"},
	{0x05C7463E2FC8E882ULL, "art\\characters\\weapons\\mauls\\textures\\w_mal_ml104a_0s.dds"},
	{0x05C79FB05B1D3311ULL, "vxlt_gallowscourtyard_n_7hu_0_0.vlm"},
	{0x05C7A1B54360A4AEULL, "llac_lt_undercityserialkiller_-1_35.dds"},
	{0x05C7B7A31F42DC9CULL, "6177016_m.fxe"},
	{0x05C7D60CE9900601ULL, "mag311_keran6041512.cl"},
	{0x05C7EDC4AAA58B9FULL, "133994738.ogg"},
	{0x05C7EE6343C7A154ULL, "gen00fl_varric6118763.cl"},
	{0x05C7F18A0A924016ULL, "vxw_blightlands_p9_0_0.vlm"},
	{0x05C810DEFB276EB4ULL, "6164481_m.fxe"},
	{0x05C8896B25D957FFULL, "zz_qun_debug6028303.cl"},
	{0x05C8DBFC91CFBD55ULL, "lt_0_lowtown_docks_a3_133_0.phy"},
	{0x05C8DDA7DDC07F5AULL, "vxht_mansion2_2vs_0_0.vlm"},
	{0x05C97B4469E3D7A4ULL, "vxw_blightlands_at_0_0.vlm"},
	{0x05CA3810BDA19772ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\cower_loop.evt"},
	{0x05CA480018FF5709ULL, "zz_dae_debug6049966.cl"},
	{0x05CB142C672EFF37ULL, "hlac_lt_0_lowtown_n_65535_0y.dds"},
	{0x05CBB1A077DB6173ULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_mth01a.phy"},
	{0x05CC47C8B3A77DF3ULL, "qun111_ketojan6040975.cl"},
	{0x05CC5B23EB70FFABULL, "6071151_m.fxe"},
	{0x05CD02BA6962B75BULL, "hlac_ht_keep_4519_0y.dds"},
	{0x05CD0CD3A13CF66CULL, "vxlt_0_lowtown_n_bz2_0_0.vlm"},
	{0x05CD4E082395F18BULL, "vxca_caverns1_mines_ihz_0_0.vlm"},
	{0x05CD5D8FE01A5FDDULL, "vxlt_gallowsprison_n_6hz_0_0.vlm"},
	{0x05CD8F56135D60FDULL, "6041415_m.fxe"},
	{0x05CD9281385B44EEULL, "6059772_m.fxe"},
	{0x05CDA445D091E997ULL, "100065249.ogg"},
	{0x05CDCE33E7E242D9ULL, "420848115.ogg"},
	{0x05CE7202B51989CBULL, "art\\characters\\playercharacter\\elffemale\\textures\\bdy_arm_hvyo_0s.dds"},
	{0x05CF00413B5BBA10ULL, "vxlt_gallowscourtyard_n_79g_0_0.vlm"},
	{0x05CF1B4F4126BCA4ULL, "art\\levels\\races\\ferelden\\props\\higherclass\\prp_eagleheadsml.mmh"},
	{0x05CF1F4F4126C3F7ULL, "art\\levels\\races\\ferelden\\props\\higherclass\\prp_eagleheadsml.mao"},
	{0x05CF314F4126E246ULL, "art\\levels\\races\\ferelden\\props\\higherclass\\prp_eagleheadsml.msh"},
	{0x05CF9298B6DC79C1ULL, "abi_w_taunt.wbd"},
	{0x05CFE26800E1C7CDULL, "one100dg_lia6061030.cl"},
	{0x05D00F8A9A4BA58AULL, "593523212.ogg"},
	{0x05D01B2CFA920590ULL, "76747625.ogg"},
	{0x05D0452DD8423AFCULL, "185396028.ogg"},
	{0x05D05710BDA63705ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\cower_loop.gad"},
	{0x05D08382C927B04BULL, "art\\characters\\playercharacter\\humanmale\\textures\\hm_hat_medc_0t.dds"},
	{0x05D0B416E4185DA6ULL, "vxw_sundermount_2xq_0_0.vlm"},
	{0x05D0C18CACAB0BCFULL, "vxlt_gallowstemplar_f_37l_0_0.vlm"},
	{0x05D0CEC7698CAF2CULL, "819899703.ogg"},
	{0x05D0EEC767FE5478ULL, "6133733_m.fxe"},
	{0x05D0F61375028D2EULL, "hlac_ht_brothel_n_3367_1y.dds"},
	{0x05D1114683D5880DULL, "one300_zevran6075746.cl"},
	{0x05D1A41C7342A55CULL, "hlac_w_woundedcoast_327680_0y.dds"},
	{0x05D1B38221FE05F0ULL, "6120411_f.fxe"},
	{0x05D1BA7F5EB4FBE2ULL, "fen200ip_slaver_chest_3.utp"},
	{0x05D27C69137C301DULL, "vxlt_undercityencounter_up_0_0.vlm"},
	{0x05D36A0BF1860955ULL, "art\\vfx\\environment\\vfx_blt_fire_wall_p.cess.ani"},
	{0x05D37CF83BDEC6C7ULL, "vxlt_0_lowtown_a2_n_c_hqx_0_0.vlm"},
	{0x05D394242AC81F4BULL, "vxw_bonepit_100_0_0.vlm"},
	{0x05D3ABCFD0E4FECDULL, "6078735-offeriscommendable.cut"},
	{0x05D405DE5C277F5EULL, "vxht_brothel_n_6gt_0_0.vlm"},
	{0x05D490E80D186E8CULL, "art\\characters\\playercharacter\\dwarfmale\\dm_brd_bd03a.msh"},
	{0x05D496E80D18797EULL, "art\\characters\\playercharacter\\dwarfmale\\dm_brd_bd03a.mmh"},
	{0x05D55E695B0B4278ULL, "llac_lt_0_lowtown_docks_n_65534_4.dds"},
	{0x05D5D7EF529E278AULL, "ser270_templar6112551.cl"},
	{0x05D5E37D168026F1ULL, "6072691_m.fxe"},
	{0x05D629B7A31AAD14ULL, "vxlt_0_lowtown_docks_cvm_0_0.vlm"},
	{0x05D66163B31C55AEULL, "llac_lt_0_lowtown_docks_a3_-65538_5.dds"},
	{0x05D697C64074F30EULL, "art\\characters\\playercharacter\\humanmale\\textures\\ulrichvon\\hm_arm_urva_0s.dds"},
	{0x05D69D6447F782C8ULL, "6149981_m.fxe"},
	{0x05D6DB2F43F5FB80ULL, "483766983.ogg"},
	{0x05D72204D5CB314AULL, "58779280.ogg"},
	{0x05D7DE17169ED4E1ULL, "6113756_m.fxe"},
	{0x05D8ACE7A891F1FFULL, "254870243.ogg"},
	{0x05D9558F4E5082BDULL, "vxht_bartrand_n_5nw_0_0.vlm"},
	{0x05D9D495694F17FEULL, "art\\characters\\creatures\\gallowsslavestatue\\animation\\enterv1.evt"},
	{0x05D9D49E5334D122ULL, "6050274_m.fxe"},
	{0x05DA49808168F951ULL, "1053426831.ogg"},
	{0x05DA9CF761AB5363ULL, "1006894153.ogg"},
	{0x05DAB1CF02ABF4FAULL, "hlac_dr_0_deeproads_5838_2x.dds"},
	{0x05DB9E450FA37D05ULL, "6208043_m.fxe"},
	{0x05DCBC09F1812C15ULL, "hlac_lt_0_lowtown_docks_131070_2y.dds"},
	{0x05DD3B3B553E07F5ULL, "164432362.ogg"},
	{0x05DD41F71301DEC4ULL, "696384518.ogg"},
	{0x05DD9365A8BA3304ULL, "vxht_mansion2_27n_0_0.vlm"},
	{0x05DDBE99612A3A5EULL, "art\\characters\\playercharacter\\humanmale\\mt_bas_hm_eyesm1.mao"},
	{0x05DEF3D06D8E415EULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_paintingsmla.phy"},
	{0x05DF077A53E88B7AULL, "785837411.ogg"},
	{0x05DF672FE4EAC444ULL, "lt_warehouse_n_0_0.msh"},
	{0x05DF792FE4EAE2ADULL, "lt_warehouse_n_0_0.mao"},
	{0x05DF7D2FE4EAE9E6ULL, "lt_warehouse_n_0_0.mmh"},
	{0x05DF81C1307926D3ULL, "lgt000ip_black_vial.utp"},
	{0x05DF8819965F539AULL, "284042965.ogg"},
	{0x05DF9C3B505AE9E9ULL, "art\\vfx\\combat\\vfx_r_dwp_front_med2_c.impa.ani"},
	{0x05DFC8DDD430CFBFULL, "vxlt_0_lowtown_docks_a3_cto_0_0.vlm"},
	{0x05E00DD2F68D8CE1ULL, "zz_dae_debug6029128.cl"},
	{0x05E031A7C625F266ULL, "274237908.ogg"},
	{0x05E0577408865FE3ULL, "art\\vfx\\combat\\vfx_glb_ele_f_deathblow_c.mmh"},
	{0x05E0A2009E01DC74ULL, "hlac_lt_0_lowtown_a3_n_c_65533_1x.dds"},
	{0x05E0D313A8C036CEULL, "161653378.ogg"},
	{0x05E0E8388E5D612BULL, "isa211_isabela6202284.cl"},
	{0x05E0EB9569555D29ULL, "art\\characters\\creatures\\gallowsslavestatue\\animation\\enterv1.gad"},
	{0x05E10BCFCEE08B27ULL, "54791665.ogg"},
	{0x05E198CBA67CC999ULL, "vxw_bonepitdestroyed_a3_c0_0_0.vlm"},
	{0x05E222EBCB147EA6ULL, "hlac_lt_gallowscourtyard_0_0x.dds"},
	{0x05E24014D2E56319ULL, "vxlt_0_lowtown_docks_a3_n_bqt_0_0.vlm"},
	{0x05E26143F91912AAULL, "vxw_woundedcoast_cn_0_0.vlm"},
	{0x05E2A6BCF13096A4ULL, "hlac_ht_playershouse_n_855_36x.dds"},
	{0x05E2FAFB7CAB6D92ULL, "art\\levels\\races\\ferelden\\props\\lowerclass\\prp_cooker.phy"},
	{0x05E32C179A7999D6ULL, "hlac_lt_0_lowtown_docks_n_-196609_1z.dds"},
	{0x05E3522D83C95771ULL, "gen_im_trs_bst_spd_silk_02.uti"},
	{0x05E371BCC4282687ULL, "art\\vfx\\combat\\vfx_w_wns_front_medium1_c.impa.gad"},
	{0x05E39BD3AE8C1AF9ULL, "hlac_lt_0_lowtown_a2_n_c_0_9x.dds"},
	{0x05E3D6241DA9CFFEULL, "545625298.ogg"},
	{0x05E444A0BD6C45A5ULL, "dae000ip_crowd_em2_talk11.utp"},
	{0x05E456E4DDC72262ULL, "pro000_captain_wright6004402.cl"},
	{0x05E49CDD9731B060ULL, "lt_0_lowtown_docks_a3_n_38_0_ll.mmh"},
	{0x05E58C567731779DULL, "6090810_f.fxe"},
	{0x05E61F7F304233B6ULL, "6060462_m.fxe"},
	{0x05E6A10C4D1E2EDDULL, "vxht_mansion_n_4wz_0_0.vlm"},
	{0x05E6B73B5061366BULL, "art\\vfx\\combat\\vfx_r_dwp_front_med2_c.impa.gad"},
	{0x05E6EF58FE2E1EFBULL, "art\\characters\\playercharacter\\humanfemale\\hf_har_ha2a.phy"},
	{0x05E722ED8EE71DB0ULL, "602527660.ogg"},
	{0x05E75F60EF39A035ULL, "dae000_dalish_apprentices6147556.cl"},
	{0x05E7A71C1C88EE06ULL, "art\\vfx\\combat\\vfx_w_jug_dam_redu_loop_c.mmh"},
	{0x05E7ABC30B309C09ULL, "6166998_m.fxe"},
	{0x05E7B75CAF157431ULL, "w_bonepitmining_a2_227_0.mmh"},
	{0x05E7BC95695B2C77ULL, "art\\characters\\creatures\\gallowsslavestatue\\animation\\enterv1.ani"},
	{0x05E7C35CAF1588D2ULL, "w_bonepitmining_a2_227_0.mao"},
	{0x05E7CB65618FA7C2ULL, "vxlt_gallowscourtyard_n_75i_0_0.vlm"},
	{0x05E7D15CAF15A0EFULL, "w_bonepitmining_a2_227_0.msh"},
	{0x05E7D599AE938736ULL, "art\\characters\\creatures\\spider\\textures\\c_corspidra_0t.dds"},
	{0x05E814DD39B95134ULL, "follower_banter6163759.cl"},
	{0x05E9742FE4F32EB1ULL, "lt_warehouse_n_0_0.phy"},
	{0x05E974534D4B45CBULL, "art\\vfx\\combat\\vfx_r_var_modal_act_c.anb"},
	{0x05E978B3FB13072BULL, "vxlt_warehouse_n_14z_0_0.vlm"},
	{0x05E98421235B7AD4ULL, "631790187.ogg"},
	{0x05E9E1BE9D3A8819ULL, "hlac_lt_gallowstemplar_f_2395_1x.dds"},
	{0x05EA66BCC42E3294ULL, "art\\vfx\\combat\\vfx_w_wns_front_medium1_c.impa.evt"},
	{0x05EA70338542E39AULL, "6129761_m.fxe"},
	{0x05EC0507D8511B6DULL, "61869258.ogg"},
	{0x05EC05429AB04B84ULL, "vxht_mansion2_20o_0_0.vlm"},
	{0x05EC485F77941029ULL, "6171091_m.fxe"},
	{0x05ECD24B8F1BCA43ULL, "art\\characters\\playercharacter\\humanmale\\cth_visa.msh"},
	{0x05ECE04B8F1BE19DULL, "art\\characters\\playercharacter\\humanmale\\cth_visa.mmh"},
	{0x05ECE44B8F1BE8AEULL, "art\\characters\\playercharacter\\humanmale\\cth_visa.mao"},
	{0x05ED6DBB0FB4794FULL, "gen00fl_sebastian6195865.cl"},
	{0x05EDB43B50674FD0ULL, "art\\vfx\\combat\\vfx_r_dwp_front_med2_c.impa.evt"},
	{0x05EECB48C83E3C86ULL, "585188236.ogg"},
	{0x05EFAC5A6574F4F9ULL, "vxlt_hangedman_20h_0_0.vlm"},
	{0x05F00A58A1685A21ULL, "vxht_0_hightown_a3_at4_0_0.vlm"},
	{0x05F00AA89872E219ULL, "vxw_bonepitdestroyed_a3_n7_0_0.vlm"},
	{0x05F012E5C7FFD56FULL, "vxdr_primevaltomb_4h3_0_0.vlm"},
	{0x05F07D2FE4F95CADULL, "lt_warehouse_n_0_0.rml"},
	{0x05F0A3F3915FA7D3ULL, "hlac_lt_0_lowtown_a2_-65536_6z.dds"},
	{0x05F11AEC905825E1ULL, "hlac_w_encounter2_camp_327680_0y.dds"},
	{0x05F17FBD0BAE1CABULL, "390172430.ogg"},
	{0x05F19378107C44A2ULL, "rdr200st_martin.utm"},
	{0x05F202AA24D75324ULL, "gencr_street_thug_2w_f.utc"},
	{0x05F216A79E7F4D2FULL, "follower_banter6161262.cl"},
	{0x05F22DCA47B1A021ULL, "194461907.ogg"},
	{0x05F26D86F59001A9ULL, "art\\cinematics\\_crit path\\prologue\\pro000cs_flemeth_real\\cs_sib_flemeth_real.ani"},
	{0x05F27F2D2AAB7064ULL, "1048112508.ogg"},
	{0x05F298500195F21EULL, "364509069.ogg"},
	{0x05F2FB8FA273FC93ULL, "345424339.ogg"},
	{0x05F30AE8EC0E88F3ULL, "120277321.ogg"},
	{0x05F4314AB87DF35BULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_grainsack.mmh"},
	{0x05F43D4AB87E07F8ULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_grainsack.mao"},
	{0x05F443E30F142CBFULL, "lt_0_lowtown_docks_195_0.msh"},
	{0x05F449E30F143761ULL, "lt_0_lowtown_docks_195_0.mmh"},
	{0x05F44F4AB87E2645ULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_grainsack.msh"},
	{0x05F455E30F144B02ULL, "lt_0_lowtown_docks_195_0.mao"},
	{0x05F4639B73E058B0ULL, "default_additive.mao"},
	{0x05F465E436C1EC38ULL, "vxw_bonepit_26z_0_0.vlm"},
	{0x05F492E5837BBE86ULL, "vxw_bonepit_264_0_0.vlm"},
	{0x05F4B498B6FBCFC3ULL, "abi_w_taunt.bnk"},
	{0x05F4B6D6CF66A04AULL, "vxlt_0_lowtown_docks_n_cs7_0_0.vlm"},
	{0x05F52463D22F2DE5ULL, "plt_lgt000pt_ultra_1_05.plo"},
	{0x05F532CC7B1DE7B0ULL, "727715760.ogg"},
	{0x05F578B8D883934CULL, "vxlt_hangedman_43_0_0.vlm"},
	{0x05F5E89D36134F4DULL, "835544679.ogg"},
	{0x05F6056BE1DAFA68ULL, "778446039.ogg"},
	{0x05F60A574B99F5A3ULL, "lowtown_male_lowlife1_sta6165031.cl"},
	{0x05F6C4831CCEDCCDULL, "hlac_lt_0_lowtown_a2_n_c_131073_2z.dds"},
	{0x05F6F8CEC64896F9ULL, "6152632_m.fxe"},
	{0x05F71774CD4BA9C7ULL, "vxca_caverns2prc_a9n_0_0.vlm"},
	{0x05F73B1431B60707ULL, "151354426.ogg"},
	{0x05F85EBCC43A62CDULL, "art\\vfx\\combat\\vfx_w_wns_front_medium1_c.impa.ani"},
	{0x05F9374E8A986435ULL, "6206913_m.fxe"},
	{0x05F94AE709B1E83BULL, "art\\levels\\races\\proxy\\props\\4tools\\vis_spike_wall_tool.mmh"},
	{0x05F956E709B1FCD8ULL, "art\\levels\\races\\proxy\\props\\4tools\\vis_spike_wall_tool.mao"},
	{0x05F968E709B21B25ULL, "art\\levels\\races\\proxy\\props\\4tools\\vis_spike_wall_tool.msh"},
	{0x05F98886F5964E2BULL, "art\\cinematics\\_crit path\\prologue\\pro000cs_flemeth_real\\cs_sib_flemeth_real.gad"},
	{0x05FA5961FA19A350ULL, "vxlt_hangedman_1ow_0_0.vlm"},
	{0x05FB66B39DDC00C6ULL, "hlac_ht_mansion2b_818_46x.dds"},
	{0x05FB838395B6EAAFULL, "801058010.ogg"},
	{0x05FBA3661C420EB9ULL, "mrl000_marethari6205603.cl"},
	{0x05FC02169ADBE49CULL, "hlac_lt_gallowstemplar_f_2383_9z.dds"},
	{0x05FC15370542AB25ULL, "325374599.ogg"},
	{0x05FC1A0CB9FE7011ULL, "841941646.ogg"},
	{0x05FCAA84433D1D30ULL, "vxca_caverns1_mines_icu_0_1.vlm"},
	{0x05FCC443FC0F6E5AULL, "art\\characters\\playercharacter\\humanfemale\\mt_cbt_hf01.mmh"},
	{0x05FCD643FC0F8CA0ULL, "art\\characters\\playercharacter\\humanfemale\\mt_cbt_hf01.msh"},
	{0x05FDB6EB43D7150AULL, "18587229.ogg"},
	{0x05FE7C858A68FA99ULL, "vxw_bonepitdestroyed_a3_i6_0_0.vlm"},
	{0x05FF44FDDD0543AAULL, "vxw_woundedcoast_uh_0_0.vlm"},
	{0x05FFB81373E7CD1AULL, "hlac_lt_0_lowtown_-196608_2z.dds"},
	{0x05FFC36A2B48A710ULL, "994024761.ogg"},
	{0x06002D64FAC92F60ULL, "vrc321ip_spooky2_obj_1.utp"},
	{0x0600539CB2615264ULL, "6082676_f.fxe"},
	{0x0600A74CA41EB324ULL, "595003969.ogg"},
	{0x0601CCCEF3E0F829ULL, "vxht_bartrand_n_64j_0_0.vlm"},
	{0x060270EE54EF00A3ULL, "hairalpha_rendertodepth_p0.vsh.fxo.dx11"},
	{0x060304C27571D706ULL, "vxw_bonepit_215_0_0.vlm"},
	{0x060342B831BC39A4ULL, "vxw_blightlands_ux_0_0.vlm"},
	{0x0603F141F6BDCE1CULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\lean_on_table_enter.ani"},
	{0x0604324FF1DC2D9EULL, "vxht_mansion2_2k0_0_0.vlm"},
	{0x060447A3B97615C4ULL, "bnn.80.dds"},
	{0x06049470747BE1CFULL, "486514339.ogg"},
	{0x0604C3FA60F6ADC7ULL, "mag360_samson6062122.cl"},
	{0x0604D4D37E78AFB4ULL, "58194906.ogg"},
	{0x0604EB98B8CB3060ULL, "t3_har_wht.tnt"},
	{0x060559E7C16FF923ULL, "1054308189.ogg"},
	{0x06058951BF41C247ULL, "vxca_caverns2prc_a4i_0_0.vlm"},
	{0x06058F9E8BF19ED6ULL, "w_encounter2_camp_47_0.rml"},
	{0x0605A521B6877E22ULL, "hla_ht_0_hightown_-65534_0x.dds"},
	{0x060619AFE5F7097DULL, "character_character_p31.psh.fxo.dx11"},
	{0x060644AB7702351AULL, "vxlt_0_lowtown_a2_n_db6_0_0.vlm"},
	{0x0606695C51A22764ULL, "gen00fl_varric6087481.cl"},
	{0x0606AA9283C9F48FULL, "plt_mrl200pt_n_siblingleft.plo"},
	{0x0606B3D19F540FD6ULL, "hlac_ht_chantry_2534_27z.dds"},
	{0x0606DF81E7F9396AULL, "hlac_ca_caverns2_5802_30z.dds"},
	{0x060705EDCC655615ULL, "6051461_m.fxe"},
	{0x0607502AE786A11FULL, "hlac_ht_keep_n_3732_0x.dds"},
	{0x06079FBC5B4B0339ULL, "hlac_dr_0_deeproads_5849_4y.dds"},
	{0x06085A159B0049BCULL, "lowtown_male_lowlife1_sta6164969.cl"},
	{0x060860CCBBDFE656ULL, "190034454.ogg"},
	{0x06088F684EDB3026ULL, "hlac_lt_0_lowtown_a2_131072_1z.dds"},
	{0x0608CB3EEC0FBBD0ULL, "vxlt_hangedman_1jt_0_0.vlm"},
	{0x0608E101C7FB007AULL, "lt_0_lowtown_docks_217_0.msh"},
	{0x0608EF01C7FB17B3ULL, "lt_0_lowtown_docks_217_0.mao"},
	{0x0608F301C7FB1EC0ULL, "lt_0_lowtown_docks_217_0.mmh"},
	{0x0609B2B2B974F18DULL, "vxlt_0_lowtown_a2_eot_0_0.vlm"},
	{0x0609EA8278A7DA81ULL, "hlac_lt_0_lowtown_docks_a3_131071_3x.dds"},
	{0x0609FC8D3B51E73CULL, "351599264.ogg"},
	{0x0609FDD55704D067ULL, "zz_ruins.cut"},
	{0x060A6E6F02B4EBF3ULL, "vxlt_0_lowtown_e0s_0_0.vlm"},
	{0x060AA471EF1F4FA1ULL, "dae000_mother6121700.cl"},
	{0x060AC24688D6C1EEULL, "llac_w_bonepit_0_0.dds"},
	{0x060ACE07CE1772AEULL, "804999991.ogg"},
	{0x060AFE41F6C4035EULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\lean_on_table_enter.gad"},
	{0x060B009BC81C87B6ULL, "zz_vrc_debug6200222.cl"},
	{0x060B13FB2F4B2947ULL, "vxlt_warehouse_156_0_0.vlm"},
	{0x060B8D4F90DFC8BEULL, "qun100ip_at_safehouse.utp"},
	{0x060BBE01BE6E0633ULL, "vxdr_dungeon_1m3_0_0.vlm"},
	{0x060C889E8BF7B1FAULL, "w_encounter2_camp_47_0.phy"},
	{0x060CA9623140CFA8ULL, "6162704_m.fxe"},
	{0x060CD58AC502492BULL, "vxht_0_hightown_a2_9wf_0_0.vlm"},
	{0x060D43E9906B4B01ULL, "6039947_f.fxe"},
	{0x060DB6DACEFB5C2AULL, "vxw_woundedcoast_ri_0_0.vlm"},
	{0x060E1EF310E36389ULL, "llac_lt_0_lowtown_131070_2.dds"},
	{0x060EFBC7EBB739C4ULL, "gen00fl_sebastian6174025.cl"},
	{0x060FD3D49AD0808DULL, "hlac_lt_0_lowtown_n_65535_1x.dds"},
	{0x060FEEFC920BE070ULL, "lt_0_lowtown_docks_a3_133_0.msh"},
	{0x060FFCFC920BF78AULL, "lt_0_lowtown_docks_a3_133_0.mmh"},
	{0x061000FC920BFEC1ULL, "lt_0_lowtown_docks_a3_133_0.mao"},
	{0x06108E9430080365ULL, "vxht_mansion2_2iu_0_0.vlm"},
	{0x061092927A219362ULL, "zz_dae_debug6047631.cl"},
	{0x0611D941F6C9E37DULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\lean_on_table_enter.evt"},
	{0x0612D658FE532E9AULL, "art\\characters\\playercharacter\\humanfemale\\hf_har_ha2a.msh"},
	{0x0612E38D175C2AA8ULL, "vxlt_0_lowtown_docks_n_b4i_0_0.vlm"},
	{0x0612E858FE534DE0ULL, "art\\characters\\playercharacter\\humanfemale\\hf_har_ha2a.mmh"},
	{0x061349760CF36CF6ULL, "art\\characters\\creatures\\ashwraith\\animation\\ras_cnjure_strt.evt"},
	{0x0613E4F99A6BAA3CULL, "sse_ss_lowclass_thugs_m_1.gda"},
	{0x0613FB2EB137DAC7ULL, "vxca_caverns2prc_a3h_0_0.vlm"},
	{0x06140BE54110C986ULL, "one350_terath6087147.cl"},
	{0x06144B57B2D21D70ULL, "vxca_cavern1_haunted_igx_0_0.vlm"},
	{0x06154BE486216986ULL, "follower_barks6153757.cl"},
	{0x06156DE80D4FCA19ULL, "art\\characters\\playercharacter\\dwarfmale\\dm_brd_bd03a.phy"},
	{0x0615AAAF1230A7BBULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\sitting_bench_lean_forward_look_right.ani"},
	{0x0615B7C1DD0FFA44ULL, "art\\levels\\races\\qunari\\props\\_textures\\prp_qunarifort_sa.dds"},
	{0x0615F3D5EE88F62DULL, "lgt000cr_gifre.utc"},
	{0x0616C83FD930C760ULL, "579091014.ogg"},
	{0x06177E6898C883C3ULL, "mcr371_merebeth_st.stg"},
	{0x0617883A66270A45ULL, "782128283.ogg"},
	{0x06179CB2FD8CD145ULL, "vxlt_0_lowtown_docks_a3_n_b10_0_0.vlm"},
	{0x06189F98A631970CULL, "502211960.ogg"},
	{0x0618A8EC9A5FCABEULL, "art\\vfx\\combat\\vfx_r_dwp_front_long_c.impa.evt"},
	{0x0618F9EAFBE31BF8ULL, "6208179_f.fxe"},
	{0x06190F6F4F05581EULL, "mag121_eau_banter6107383.cl"},
	{0x061940760CF7C881ULL, "art\\characters\\creatures\\ashwraith\\animation\\ras_cnjure_strt.gad"},
	{0x061980AF713EDB59ULL, "558954320.ogg"},
	{0x061A13DBE9C2EC9CULL, "6072316_f.fxe"},
	{0x061B1B3F2336E828ULL, "6167404_m.fxe"},
	{0x061B65660CDA8B63ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\persuade_die.evt"},
	{0x061B66D125898803ULL, "849369558.ogg"},
	{0x061BBBC87408DB94ULL, "con_ico_charm.dds"},
	{0x061C1154E3E4B49FULL, "369161940.ogg"},
	{0x061C3B19222A2C73ULL, "6116384_m.fxe"},
	{0x061C9163AEE6D7EDULL, "llac_dr_dungeon_3214_14.dds"},
	{0x061C9C1495D37004ULL, "follower_banter6170092.cl"},
	{0x061C9E2FE3AF7B73ULL, "126333085.ogg"},
	{0x061CC04F7DC253C3ULL, "6102801_f.fxe"},
	{0x061CDFF26F155D94ULL, "vrc221_bartrand6065316.cl"},
	{0x061CE7F3BAA990EDULL, "isa311_castillon6092077.cl"},
	{0x061DD8C2E8E89650ULL, "vxw_encounter2_camp_7a_0_0.vlm"},
	{0x061EB4CCEBA64B59ULL, "art\\characters\\playercharacter\\dwarfmale\\bdy_lgta.mmh"},
	{0x061EBECCEBA65C47ULL, "art\\characters\\playercharacter\\dwarfmale\\bdy_lgta.msh"},
	{0x061FBFEC9A6610E9ULL, "art\\vfx\\combat\\vfx_r_dwp_front_long_c.impa.gad"},
	{0x061FC0C66056136EULL, "dae000_keep_noblewoman_sta6162045.cl"},
	{0x061FC5C6DDE033F4ULL, "gen00fl_anders6215204.cl"},
	{0x06202EE7076BEB1CULL, "882027049.ogg"},
	{0x062041760CFDE92FULL, "art\\characters\\creatures\\ashwraith\\animation\\ras_cnjure_strt.ani"},
	{0x06208951EC1F33A2ULL, "art\\vfx\\combat\\vfx_w_2nd_atk_l_180_c.impa.gad"},
	{0x0620AE902E142600ULL, "hlac_ht_bartrand_n_3369_2z.dds"},
	{0x062105E5C45F4047ULL, "6170228-pridedrain-h.cut"},
	{0x06211609D5C85E9EULL, "vxht_mansion2_2q2_0_0.vlm"},
	{0x06216923AC1BC0B6ULL, "alphastreaks01.mao"},
	{0x0621C18720FF58F9ULL, "gen00fl_fenris6103319.cl"},
	{0x0621E7B7721FE444ULL, "vxht_mansion2_1v2_0_0.vlm"},
	{0x06223E660CE067E4ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\persuade_die.gad"},
	{0x06224D634734F624ULL, "art\\levels\\races\\ferelden\\props\\mining\\prp_walkrockend.msh"},
	{0x06225F634735148DULL, "art\\levels\\races\\ferelden\\props\\mining\\prp_walkrockend.mao"},
	{0x0622636347351BC6ULL, "art\\levels\\races\\ferelden\\props\\mining\\prp_walkrockend.mmh"},
	{0x0622B2AF123B3FB2ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\sitting_bench_lean_forward_look_right.evt"},
	{0x0622E343FC307265ULL, "art\\characters\\playercharacter\\humanfemale\\mt_cbt_hf01.phy"},
	{0x0622F1CD764F9CB2ULL, "vxlt_warehouse_n_x0_0_0.vlm"},
	{0x0622FA672ED1B931ULL, "vxca_caverns1_mines_e8t_0_0.vlm"},
	{0x0623668F3D95C7F8ULL, "vxlt_0_lowtown_a3_n_c_f4w_0_0.vlm"},
	{0x0623F628EF1F22BCULL, "vxw_bonepitdestroyed_a3_ci_0_0.vlm"},
	{0x06240BCCCF14D9F7ULL, "dae000_gamlen6149477.cl"},
	{0x06247E1467956503ULL, "18845896.ogg"},
	{0x0624ACDD9656AAB2ULL, "601233711.ogg"},
	{0x0624BAE0260C8C79ULL, "6143674_m.fxe"},
	{0x062503B0A0A7C6C7ULL, "352874749.ogg"},
	{0x0625272397D1463EULL, "ht_0_hightown_a2_34_0_lc.mmh"},
	{0x062540F86534613DULL, "553133580.ogg"},
	{0x0625E0E51725C14CULL, "656821536.ogg"},
	{0x062601D69EA44D96ULL, "prm000im_ivo_amulet.uti"},
	{0x062690EC9A6BDF37ULL, "art\\vfx\\combat\\vfx_r_dwp_front_long_c.impa.ani"},
	{0x06270CA40869D6DBULL, "vxlt_0_lowtown_a3_n_c_i0s_0_0.vlm"},
	{0x0627742BB92CA6D5ULL, "hlac_lt_0_lowtown_-1_1y.dds"},
	{0x06279F81D782F3B2ULL, "vxlt_darktown_4t8_0_0.vlm"},
	{0x0627AC51EC258D99ULL, "art\\vfx\\combat\\vfx_w_2nd_atk_l_180_c.impa.evt"},
	{0x06280B481387BC8DULL, "282196207.ogg"},
	{0x06284D6CC5D63AA2ULL, "hlac_lt_0_lowtown_n_65536_7x.dds"},
	{0x062859F591BA3A71ULL, "102215895.ogg"},
	{0x06286EF292037A43ULL, "vxlt_hangedman_n_2zc_0_0.vlm"},
	{0x0628E8BDF379B20CULL, "vxlt_0_lowtown_a2_n_c_gz9_0_0.vlm"},
	{0x0628F08B57ED77EBULL, "6122780.cut"},
	{0x0628F228FEA7D547ULL, "hla_ht_0_hightown_a2_-65537_0x.dds"},
	{0x06293102D55AD0BDULL, "6109557_m.fxe"},
	{0x06294D660CE69FC2ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\persuade_die.ani"},
	{0x06295F0FC05BC5F0ULL, "fx_headcloud.mao"},
	{0x0629C84E3035B052ULL, "and111_lirene6084369.cl"},
	{0x0629D1AF12419345ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\sitting_bench_lean_forward_look_right.gad"},
	{0x062A0C1C82DA5AA2ULL, "vxw_bonepit_2aa_0_0.vlm"},
	{0x062B12C46699DCDAULL, "26083346.ogg"},
	{0x062B5DA5DFF29D4EULL, "hlac_lt_0_lowtown_a2_n_-196609_0x.dds"},
	{0x062B751A0BDF3364ULL, "mag121_viveka6083366.cl"},
	{0x062BAB2006C554A5ULL, "gen00fl_aveline6114844.cl"},
	{0x062C31A6E6E1E9D5ULL, "vxlt_gallowstemplar_3e3_0_0.vlm"},
	{0x062C4A9FDADEAED0ULL, "vxw_encounter2_camp_2n_0_0.vlm"},
	{0x062CDC2C7809B311ULL, "vxw_encounter3_dz_0_0.vlm"},
	{0x062D897BD85EFD67ULL, "vxht_0_hightown_n_9p4_0_0.vlm"},
	{0x062DAAA0782E1FE8ULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_mth01a.mmh"},
	{0x062DC8A0782E5212ULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_mth01a.msh"},
	{0x062DFFEF9CBAA8F0ULL, "dae011_corff_sta6142476.cl"},
	{0x062E83ED0CEC3B8BULL, "plt_vis160pt_traps.plo"},
	{0x062FAC9AF0092742ULL, "dae000_oldmangossip6175756.cl"},
	{0x062FE5B495963AE7ULL, "vxw_encounter3_7i_0_0.vlm"},
	{0x062FFC25A684E237ULL, "art\\characters\\creatures\\rubblegolem\\c_rubblegolem.phy"},
	{0x0630F74D2A4D4712ULL, "6016485_m.fxe"},
	{0x0631226116C5B582ULL, "gen00fl_anders6113061.cl"},
	{0x06315104DA6F8803ULL, "6169885_m.fxe"},
	{0x063161D12C42A27DULL, "ser221_gascard6122830.cl"},
	{0x0631A3E744DE632DULL, "one300_nuncio6075328.cl"},
	{0x0631A8EBE21D2CB9ULL, "6185496_m.fxe"},
	{0x0631AA630256823DULL, "6129480_f.fxe"},
	{0x0631E11941D34C07ULL, "art\\characters\\creatures\\darkspawnogre\\animation\\cindes\\custom\\meleeattackv1_frozen.gad"},
	{0x06326805E1153B3CULL, "vxw_bonepitdestroyed_a3_nn_0_0.vlm"},
	{0x06329FE302FA839FULL, "6092354_m.fxe"},
	{0x063304F6CAC88F8AULL, "zz_approval_debug6003090.cl"},
	{0x06334BF365E9844FULL, "972006821.ogg"},
	{0x06337BB16D816551ULL, "vxdr_primevaltomb_6ni_0_0.vlm"},
	{0x0633CE3EDA1CD0C8ULL, "dae300_bran6143450.cl"},
	{0x0634134978E62C19ULL, "character_character_p95.psh.fxo"},
	{0x063421D151C8DBF7ULL, "lt_0_lowtown_docks_2_0.anb"},
	{0x0634938EF20906C8ULL, "dae000_templar_knight6165163.cl"},
	{0x063495ADEB2F23D3ULL, "6179909_m.fxe"},
	{0x06349A871C75BFD1ULL, "hlac_lt_0_lowtown_131071_1z.dds"},
	{0x0634B451EC302588ULL, "art\\vfx\\combat\\vfx_w_2nd_atk_l_180_c.impa.ani"},
	{0x0634CF2C05B80234ULL, "269185751.ogg"},
	{0x06360DF9B7ECEFAEULL, "art\\vfx\\combat\\vfx_c_mer_thrust_c.mmh"},
	{0x063683815CEEE673ULL, "889917315.ogg"},
	{0x0637112CCE6E9047ULL, "652720674.ogg"},
	{0x06377C53ADA92B4BULL, "art\\characters\\playercharacter\\dwarfmale\\dm_bdy_lgtb.mao"},
	{0x0637A2DFC750E93DULL, "6102457_m.fxe"},
	{0x0637E6940F07D8CBULL, "78783938.ogg"},
	{0x0637E7F9D1F4E798ULL, "vxlt_0_lowtown_a3_n_c_hfg_0_0.vlm"},
	{0x063818E7371D2C56ULL, "759644513.ogg"},
	{0x06389F2781768824ULL, "hlac_lt_undercitydungeon_-1_41z.dds"},
	{0x0638E829D4836A4DULL, "nat380_party_banter6066171.cl"},
	{0x06391C92D3A9D759ULL, "gen00fl_aveline6053327.cl"},
	{0x06399DAC15D1A050ULL, "6172594_m.fxe"},
	{0x0639AF10DE396811ULL, "vxlt_0_lowtown_a2_e8e_0_0.vlm"},
	{0x0639ECEB41AF2FD2ULL, "941997773.ogg"},
	{0x0639FCC4B77D525BULL, "llac_dr_primevaltomb_665_2.dds"},
	{0x063A1B0964651896ULL, "262197999.ogg"},
	{0x063AC04211449276ULL, "vxlt_gallowstemplar_f_51i_0_0.vlm"},
	{0x063AC8E32743926DULL, "351487108.ogg"},
	{0x063AD6C96EB69DB0ULL, "472996533.ogg"},
	{0x063B2EB80AADDD88ULL, "art\\levels\\races\\ferelden\\props\\food\\prp_cheesewheela.msh"},
	{0x063B40B80AADFC19ULL, "art\\levels\\races\\ferelden\\props\\food\\prp_cheesewheela.mao"},
	{0x063B457095B02918ULL, "hlac_w_woundedcoast_-65533_0z.dds"},
	{0x063B4CB80AAE10B2ULL, "art\\levels\\races\\ferelden\\props\\food\\prp_cheesewheela.mmh"},
	{0x063B4E0969FFCB91ULL, "vxw_encounter3_ay_0_0.vlm"},
	{0x063B817C9AE01A3DULL, "mag291st_ambient1.stg"},
	{0x063B8528EA500E80ULL, "6207986_m.fxe"},
	{0x063BB327124AF402ULL, "vxht_0_hightown_bif_0_0.vlm"},
	{0x063C2711512C6911ULL, "vxca_caverns1_in3_0_0.vlm"},
	{0x063C524AB8BBE090ULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_grainsack.phy"},
	{0x063D3708CE3FCA58ULL, "509078959.ogg"},
	{0x063DA78767414EA0ULL, "147531661.ogg"},
	{0x063E58AF0931E74EULL, "art\\levels\\races\\primeval\\props\\_texture\\prp_lyriumbig_d.dds"},
	{0x063EBE7F2FAD5C5AULL, "dae000_bartrand6206887.cl"},
	{0x063F49AFFD14A187ULL, "6137500_f.fxe"},
	{0x063F8307F2482BCFULL, "ts_beasts_spdr_prog_fang.gda"},
	{0x06400EA0B4469C4AULL, "6089470_m.fxe"},
	{0x06401327D41AF0ABULL, "45107947.ogg"},
	{0x06403D2B0B542EAFULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_bottle.mao"},
	{0x0640492B0B54434CULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_bottle.mmh"},
	{0x06404F2B0B544D3EULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_bottle.msh"},
	{0x0640B0E1FC76AD1CULL, "w_encounter3_73_0.mao"},
	{0x0640BCE1FC76C1BFULL, "w_encounter3_73_0.mmh"},
	{0x0640C2E1FC76CC61ULL, "w_encounter3_73_0.msh"},
	{0x064191B45567747DULL, "gen00fl_varric6105537.cl"},
	{0x06419FFE0A1CD898ULL, "vxlt_undercitydungeon_1jz_0_0.vlm"},
	{0x0641E6BA71A21F92ULL, "45738317.ogg"},
	{0x064218550FB76FCCULL, "321974101.ogg"},
	{0x064238E27553D4C5ULL, "1066855077.ogg"},
	{0x06424CB0BD51D169ULL, "t3_arm_pl1.tnt"},
	{0x064285266ADC4499ULL, "character_character_p42.psh.fxo"},
	{0x06434E5F36036FD2ULL, "vxht_0_hightown_c13_0_0.vlm"},
	{0x06437E2E16C06A86ULL, "223278053.ogg"},
	{0x0643C7018CD5C347ULL, "art\\characters\\creatures\\ragedemon\\animation\\rst_ex.gad"},
	{0x0643F9D70B5FD9A2ULL, "art\\characters\\weapons\\dagger\\textures\\w_dgr_dg101a_0s.dds"},
	{0x06440EA4221BECC4ULL, "vxlt_0_lowtown_docks_a3_n_bss_0_0.vlm"},
	{0x06441FEE194217A8ULL, "art\\levels\\ztools\\default\\tilesets\\setext\\3ci_basic_1.mao"},
	{0x06442CF9B7F961ADULL, "art\\vfx\\combat\\vfx_c_mer_thrust_c.anb"},
	{0x06443FE219478C5AULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\blendtree\\pose_lying_injured_dead.anb"},
	{0x064486B4A4B26D5DULL, "632722863.ogg"},
	{0x0644EBC1B2F96061ULL, "vxht_0_hightown_a3_cco_0_0.vlm"},
	{0x06454F62FF5A1C58ULL, "hlac_lt_0_lowtown_a2_n_-65537_2y.dds"},
	{0x06459C668128EDBDULL, "vxht_mansion2b_1v7_0_0.vlm"},
	{0x0645AAD470ECEDE2ULL, "dae150ip_rockslide.utp"},
	{0x0646223C06E10CD0ULL, "459901378.ogg"},
	{0x064630725B5A98E7ULL, "6136179_m.fxe"},
	{0x06468997677B0CA4ULL, "hlac_lt_gallowstemplar_n_2383_10x.dds"},
	{0x06468D6F0125DCCCULL, "153796425.ogg"},
	{0x0646CE1941E5884DULL, "art\\characters\\creatures\\darkspawnogre\\animation\\cindes\\custom\\meleeattackv1_frozen.ani"},
	{0x0646E50525CEE776ULL, "6099589_m.fxe"},
	{0x0647967C7586D58EULL, "dae221st_dougal_pc_home.stg"},
	{0x06481A75F581CD1EULL, "6151036_m.fxe"},
	{0x06483F8666A4E9D3ULL, "vxht_0_hightown_c49_0_0.vlm"},
	{0x06489F1DF0A7BC6CULL, "llac_w_encounter2_camp_262145_0.dds"},
	{0x0649A37C43CC9029ULL, "hlac_w_woundedcoast_327682_0z.dds"},
	{0x0649A4B1FD2848B6ULL, "vxht_0_hightown_cbn_0_0.vlm"},
	{0x0649EAECE40AA2AAULL, "2488493.ogg"},
	{0x064A00DF18E9F967ULL, "hlac_lt_gallowstemplar_n_2391_8y.dds"},
	{0x064ADC72ED7EC9F5ULL, "192745520.ogg"},
	{0x064B61AA0EB013A0ULL, "339075018.ogg"},
	{0x064BA7063BC20DC5ULL, "154452707.ogg"},
	{0x064BDB2F122BFF46ULL, "vxw_woundedcoast_3r_0_0.vlm"},
	{0x064C453BCA38CB3DULL, "6095090_m.fxe"},
	{0x064CBF12D0029200ULL, "dae000ip_crowd_mf5_talk9.utp"},
	{0x064D1EDFCFC96736ULL, "269483132.ogg"},
	{0x064DA87460174A9CULL, "6162304_m.fxe"},
	{0x064E1ED4552C709BULL, "437272125.ogg"},
	{0x064E23B76853C4B2ULL, "dae000_gamlen6048489.cl"},
	{0x064E34029D2A659EULL, "dae000_madam_lusine6151889.cl"},
	{0x064E400818B98250ULL, "6027357_m.fxe"},
	{0x064E4E3B73B62BB5ULL, "vxlt_0_lowtown_docks_a3_d2x_0_0.vlm"},
	{0x064E7477744E0C48ULL, "vxht_0_hightown_a3_ccj_0_0.vlm"},
	{0x064EC3BD0840C799ULL, "ico_locked_sustained.dds"},
	{0x064ED1790AC93157ULL, "gen00fl_merrill6111408.cl"},
	{0x064F18683D7FA376ULL, "497257866.ogg"},
	{0x064FA9B2E9EA65D7ULL, "hlac_dr_primevalentrance_1726_3x.dds"},
	{0x064FB3AC2431D553ULL, "6072117_f.fxe"},
	{0x065021A6EDFA0555ULL, "vxlt_gallowstemplar_n_3sb_0_0.vlm"},
	{0x06505F6B516D9651ULL, "vxdr_primevaltomb_6xo_0_0.vlm"},
	{0x065081C63A1C6FA8ULL, "fx_cloud.dds"},
	{0x06509CC67DCFE1BEULL, "694374863.ogg"},
	{0x0650DC405F825C11ULL, "vxht_0_hightown_n_bzc_0_0.vlm"},
	{0x0650F7035CD25D19ULL, "character_character_p33.psh.fxo"},
	{0x0651BE20B9FD7985ULL, "vxlt_0_lowtown_a3_n_c_f4n_0_0.vlm"},
	{0x0651C992840A1574ULL, "plt_mrl200pt_n_siblingleft.nss"},
	{0x065238EA908944F8ULL, "t3_wdg_g01.tnt"},
	{0x06533010DFF029C0ULL, "hlac_lt_0_lowtown_n_65537_4x.dds"},
	{0x06534B03205930E3ULL, "6172632_m.fxe"},
	{0x06535B9DECC4ACA0ULL, "471460666.ogg"},
	{0x0653F5542FECD575ULL, "361963953.ogg"},
	{0x06542C0A8E1E3814ULL, "hlac_ht_0_hightown_a2_65536_4z.dds"},
	{0x0654B54640B7E1A3ULL, "gen00fl_aveline6121925.cl"},
	{0x0654C68FDF49F083ULL, "hlac_ht_brothel_n_3381_1x.dds"},
	{0x0654F20C95C5493CULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\crouch_pray_loop.ani"},
	{0x0655D9A522E1F58DULL, "llac_ht_0_hightown_-131072_1.dds"},
	{0x0655FE439528E1D5ULL, "llac_lt_0_lowtown_a3_n_c_131070_4.dds"},
	{0x065736580E9553C1ULL, "179219130.ogg"},
	{0x06574D793CB0904FULL, "zz_ave_debug6056757.cl"},
	{0x06575BC80347AE1AULL, "vxlt_0_lowtown_docks_n_cos_0_0.vlm"},
	{0x0657873DBCC43355ULL, "vxlt_gallowstemplar_3t0_0_0.vlm"},
	{0x0657B4018CE64B8DULL, "art\\characters\\creatures\\ragedemon\\animation\\rst_ex.ani"},
	{0x0657C5CE606008B5ULL, "6209890_f.fxe"},
	{0x0658BC188460B7EBULL, "ice_shard_body.dds"},
	{0x06591918A232B3D4ULL, "vxlt_gallowscourtyard_6z0_0_0.vlm"},
	{0x06591F0ADCCAF85EULL, "vxw_blightlands_bg_0_0.vlm"},
	{0x065978867713EE03ULL, "6111103_m.fxe"},
	{0x06599AD045216501ULL, "71737.ogg"},
	{0x06599C52DFB220E7ULL, "vxlt_0_lowtown_a2_ewn_0_0.vlm"},
	{0x0659AC5DA7F0DAB6ULL, "hlac_lt_0_lowtown_a2_n_-65536_4z.dds"},
	{0x0659C79E8CDE4022ULL, "llac_lt_0_lowtown_a2_n_65536_3.dds"},
	{0x0659CE7B26450B22ULL, "dae010cr_joker3.utc"},
	{0x0659F8F936179894ULL, "vxw_encounter2_p9_0_1.vlm"},
	{0x065A21401DD93D1EULL, "vxlt_hangedman_5n_0_0.vlm"},
	{0x065B77C809EA295AULL, "follower_banter6163586.cl"},
	{0x065B81A3526AD303ULL, "gen00fl_sebastian6177894.cl"},
	{0x065BFF0C95CB7E7EULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\crouch_pray_loop.gad"},
	{0x065C2B7FD948C556ULL, "gr_02_glow.mao"},
	{0x065C4DAC2014CBA6ULL, "6071837_m.fxe"},
	{0x065C53DDE0EB1232ULL, "6133142_m.fxe"},
	{0x065D40171C57E3C1ULL, "dae011_gossips6149781.cl"},
	{0x065E488F97F43432ULL, "vxw_woundedcoast_1be_0_1.vlm"},
	{0x065E9E93CE473A5EULL, "64067283.ogg"},
	{0x065F659A24A30C81ULL, "gen_im_acc_rng_act3_06.uti"},
	{0x0660C25D0528F683ULL, "dae000_meredith6080996.cl"},
	{0x06611ED151EFC40BULL, "lt_0_lowtown_docks_2_0.rml"},
	{0x066155DFC963B5D5ULL, "216294867.ogg"},
	{0x066187D3B6B2421FULL, "vxlt_warehouse_n_aj_0_0.vlm"},
	{0x0661CB6FF138F76FULL, "6201543_m.fxe"},
	{0x0662DA0C95D15D9DULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\crouch_pray_loop.evt"},
	{0x06631C7E2DF110FEULL, "vxca_caverns1_k2t_0_2.vlm"},
	{0x066333997E2052C1ULL, "307187591.ogg"},
	{0x06633B5418A5EE12ULL, "633983138.ogg"},
	{0x066423C0012E802EULL, "493957694.ogg"},
	{0x06645AA637DF1496ULL, "vxdr_0_deeproads_gby_0_0.vlm"},
	{0x066473C735E156F0ULL, "6014337_m.fxe"},
	{0x06648D077E87E7A0ULL, "rdr111dg_assistant6055108.cl"},
	{0x0664A3A0AAFFAE4CULL, "hlac_dr_0_deeproads_5845_16y.dds"},
	{0x0665174E148338A8ULL, "6049821_m.fxe"},
	{0x06651B6927675CF8ULL, "gen00fl_sebastian6180444.cl"},
	{0x06653DBBFF5B857CULL, "988915853.ogg"},
	{0x06654ED74BCA3DDEULL, "34813456.ogg"},
	{0x066581AD38443CADULL, "qun110_petrice6036418.cl"},
	{0x06659CA7CD869BDEULL, "hlac_lt_0_lowtown_docks_n_0_2y.dds"},
	{0x0665D1B4FA9EA5FEULL, "vxlt_hangedman_n_b_0_0.vlm"},
	{0x06665C477196D165ULL, "303949269.ogg"},
	{0x0666C0711077F039ULL, "art\\characters\\playercharacter\\humanmale\\hoo_lgta.msh"},
	{0x0666C832DB628F40ULL, "art\\characters\\playercharacter\\humanmale\\textures\\hm_hoo_roba_0s.dds"},
	{0x0666CA7110780127ULL, "art\\characters\\playercharacter\\humanmale\\hoo_lgta.mmh"},
	{0x0666CE7110780874ULL, "art\\characters\\playercharacter\\humanmale\\hoo_lgta.mao"},
	{0x0667053E010B38D7ULL, "vxdr_primevaltomb_5ud_0_1.vlm"},
	{0x06673857C35743C5ULL, "981012385.ogg"},
	{0x0667819E8C445123ULL, "w_encounter2_camp_47_0.msh"},
	{0x06678F9E8C44697DULL, "w_encounter2_camp_47_0.mmh"},
	{0x0667939E8C446F8EULL, "w_encounter2_camp_47_0.mao"},
	{0x0667EF37A19CC2A8ULL, "art\\levels\\races\\proxy\\tilesets\\setint\\8co_basic\\0dd_small_2.mao"},
	{0x06683EDC49299F17ULL, "art\\characters\\creatures\\crowds\\humanmale\\animation\\exodus\\standing_neutral_gestures_2_2_sylvesta.gad"},
	{0x06699485A0AE25C2ULL, "358276984.ogg"},
	{0x0669FC5C12B24AC1ULL, "art\\cinematics\\_crit path\\prologue\\additional animations\\cs_hurlock_charge11.ani"},
	{0x066A5846B795A4FBULL, "vxlt_0_lowtown_docks_a3_bgc_0_0.vlm"},
	{0x066B01E00A83C86CULL, "vxdr_primevaltomb_2um_0_0.vlm"},
	{0x066B2C85DB33E52BULL, "72667407.ogg"},
	{0x066BF00343ED4264ULL, "hlac_w_encounter2_196607_0y.dds"},
	{0x066C64E5FC7B0FDBULL, "503900244.ogg"},
	{0x066C968920D765C7ULL, "vxw_woundedcoast_1gb_0_0.vlm"},
	{0x066CDC59759CADC3ULL, "hlac_w_sundermount_196612_0y.dds"},
	{0x066DA93DE6A12C6FULL, "art\\characters\\playercharacter\\humanfemale\\animation\\partypicker\\magestaffdeselectedexit.gad"},
	{0x066DD2BDB166C9DCULL, "6086987_m.fxe"},
	{0x066E324979CE0AFCULL, "lt_backalley_n_0_0.anb"},
	{0x066E57B71A1BD059ULL, "6021223_f.fxe"},
	{0x066F6BC35A675556ULL, "mag100_cullen6095132.cl"},
	{0x066F88A3CD41DBD4ULL, "6034541_f.fxe"},
	{0x066FB73FC7AA8502ULL, "vxlt_gallowstemplar_5qn_0_0.vlm"},
	{0x067055AC40104B97ULL, "6117677_m.fxe"},
	{0x0670925F13FA99EEULL, "art\\vfx\\combat\\vfx_ha_hd_mblast_act_c.impa.evt"},
	{0x0671075C12B87C63ULL, "art\\cinematics\\_crit path\\prologue\\additional animations\\cs_hurlock_charge11.gad"},
	{0x0671105222260F2EULL, "seb100im_vael_amulet.uti"},
	{0x0671240F1872FA4BULL, "vxw_bonepitmining_a2_25f_0_0.vlm"},
	{0x06719D8B7C78404FULL, "hlac_lt_0_lowtown_65537_2x.dds"},
	{0x0671EF324273A874ULL, "573112652.ogg"},
	{0x06720B5E604B096CULL, "1066940770.ogg"},
	{0x0672655AE545CA76ULL, "vxlt_gallowstemplar_n_5aa_0_0.vlm"},
	{0x06734DAE89526D27ULL, "vxca_cavern1_haunted_k2p_0_2.vlm"},
	{0x0674349339CE7E0CULL, "490157242.ogg"},
	{0x06745895431FC168ULL, "gen_im_trs_gen_med_glv_r1.uti"},
	{0x067461029B4C259BULL, "399227454.ogg"},
	{0x0674BE3DE6A76E1CULL, "art\\characters\\playercharacter\\humanfemale\\animation\\partypicker\\magestaffdeselectedexit.evt"},
	{0x0674D8F12D7A7EDBULL, "vxlt_0_lowtown_docks_a3_d3u_0_0.vlm"},
	{0x0674FC2474726AE3ULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_grogbottlecin_d.dds"},
	{0x0675C2407711BEB3ULL, "vxlt_gallowstemplar_53a_0_0.vlm"},
	{0x0675EBF48C1A63CFULL, "753436621.ogg"},
	{0x06762303F47A463EULL, "284356213.ogg"},
	{0x067671A29D221814ULL, "mcr371ip_vfx_target.ncs"},
	{0x0676895F13FEF599ULL, "art\\vfx\\combat\\vfx_ha_hd_mblast_act_c.impa.gad"},
	{0x0676A83A508E95BCULL, "6085137_m.fxe"},
	{0x0676D125A6C09B8FULL, "art\\characters\\creatures\\rubblegolem\\c_rubblegolem.mao"},
	{0x0676DD25A6C0B02CULL, "art\\characters\\creatures\\rubblegolem\\c_rubblegolem.mmh"},
	{0x0676E325A6C0BA1EULL, "art\\characters\\creatures\\rubblegolem\\c_rubblegolem.msh"},
	{0x0677132CC78532F8ULL, "lgt000_chanter6147199.cl"},
	{0x067817FEBEE1568EULL, "art\\levels\\races\\proxy\\props\\4tools\\inv_box_25x_25_0.pwk"},
	{0x06782562A2EE36A3ULL, "art\\vfx\\creatures\\fxc_drg_brth_c_c.impa.ani"},
	{0x06782654B991384CULL, "art\\characters\\playercharacter\\dwarfmale\\mt_cbt_sand.phy"},
	{0x0678980704C5C881ULL, "6117979_f.fxe"},
	{0x06789CB5CF416389ULL, "vxdr_0_deeproads_cqk_0_0.vlm"},
	{0x06793797C4D74332ULL, "6139742_m.fxe"},
	{0x067ADF33F1AE602EULL, "zz_followers_banter6157613.cl"},
	{0x067AF682C4697705ULL, "6134737_m.fxe"},
	{0x067B12DD164B73B1ULL, "vxlt_hangedman_20p_0_0.vlm"},
	{0x067B69AC1E5014F9ULL, "w_encounter3_136_0.msh"},
	{0x067B73AC1E5025E7ULL, "w_encounter3_136_0.mmh"},
	{0x067B77AC1E502C34ULL, "w_encounter3_136_0.mao"},
	{0x067C072AD92D2871ULL, "art\\vfx\\combat\\vfx_w_wns_front_close4_c.anb"},
	{0x067C6BDC493A949DULL, "art\\characters\\creatures\\crowds\\humanmale\\animation\\exodus\\standing_neutral_gestures_2_2_sylvesta.ani"},
	{0x067CAA4139A745C5ULL, "llac_lt_0_lowtown_docks_a3_n_0_5.dds"},
	{0x067CD06BAED4ABDCULL, "vxht_mansion2b_3dz_0_0.vlm"},
	{0x067CEE27272BBFB8ULL, "one280_emrys6104279.cl"},
	{0x067D0073C9E86A1BULL, "vxw_encounter1_ei_0_0.vlm"},
	{0x067D517D04807244ULL, "49728967.ogg"},
	{0x067D5C822E2094FEULL, "438640351.ogg"},
	{0x067D9A5F140531E7ULL, "art\\vfx\\combat\\vfx_ha_hd_mblast_act_c.impa.ani"},
	{0x067DDA9C71740ED4ULL, "6134743_m.fxe"},
	{0x067E07DC8A8CDA0CULL, "dae000_bartrand6206962.cl"},
	{0x067E87C602D588EBULL, "character_character_p57.psh.fxo"},
	{0x067EC7802D164611ULL, "6117680_m.fxe"},
	{0x067EC78932066417ULL, "6112577_m.fxe"},
	{0x067EF3865858E497ULL, "dae000_mother6156397.cl"},
	{0x067F28FD3170DF7FULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\order_right.gad"},
	{0x067F47F0EDB57310ULL, "95120636.ogg"},
	{0x067F7BC96A2774BEULL, "540402038.ogg"},
	{0x0680344E2BA498D6ULL, "and100_anders6095884.cl"},
	{0x068098A7BA8BAC27ULL, "6126894_f.fxe"},
	{0x0680C947288E3A7CULL, "llac_lt_0_lowtown_131071_2.dds"},
	{0x0680D4200847B11BULL, "follower_banter6143370.cl"},
	{0x0680E0A3F85936B4ULL, "6118024_m.fxe"},
	{0x0681A61D7A5979DBULL, "explosion_v2.cut"},
	{0x0681AEB74C4FC355ULL, "llac_lt_gallowstemplar_n_2383_8.dds"},
	{0x0681B5CCEBFACA46ULL, "art\\characters\\playercharacter\\dwarfmale\\bdy_lgta.phy"},
	{0x0681C63DE6B20635ULL, "art\\characters\\playercharacter\\humanfemale\\animation\\partypicker\\magestaffdeselectedexit.ani"},
	{0x0681C693492586F2ULL, "t3_wdg_t00.tnt"},
	{0x06827B271BC8E11AULL, "vrc221_bartrand6065273.cl"},
	{0x06827B8E8F1B657BULL, "79372845.ogg"},
	{0x068282CA28607B4BULL, "vxw_bonepitmining_a2_1q1_0_0.vlm"},
	{0x0682D4BEA32FF4E4ULL, "6180832_m.fxe"},
	{0x06834ACE1F70975BULL, "vxlt_0_lowtown_docks_a3_d4t_0_0.vlm"},
	{0x06838D4792C9127FULL, "6131186_f.fxe"},
	{0x0683EE09C1C6BB46ULL, "art\\vfx\\creatures\\fxc_arcane_tele_p.cess.ani"},
	{0x0684A2DAB7598268ULL, "vxca_caverns1_icq_0_0.vlm"},
	{0x0684C5CE60E768FDULL, "vxdr_primevaltomb_7rd_0_0.vlm"},
	{0x0685175562FB4DADULL, "fex_ruin_spt_a.mmh"},
	{0x06851DFD3175386CULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\order_right.evt"},
	{0x0685295562FB6C33ULL, "fex_ruin_spt_a.msh"},
	{0x06855AE425BF1D95ULL, "vxlt_0_lowtown_docks_bgb_0_0.vlm"},
	{0x0686098902E33678ULL, "vxlt_0_lowtown_h7l_0_0.vlm"},
	{0x068623C25ED4294EULL, "630762602.ogg"},
	{0x06863D62A2FA9C9AULL, "art\\vfx\\creatures\\fxc_drg_brth_c_c.impa.evt"},
	{0x0686465A929B77FAULL, "vxht_mansion2_1wo_0_0.vlm"},
	{0x068681F1D1F530E1ULL, "754460595.ogg"},
	{0x0686ABD44C0EFCA2ULL, "zz_and_debug6178574.cl"},
	{0x0686E7BC560B6E63ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\twohanded\\runfront.evt"},
	{0x06874129E621993AULL, "341269542.ogg"},
	{0x0687A167AB4C4ABAULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\1ww_grandcrackr.mmh"},
	{0x0687A567AB4C51F1ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\1ww_grandcrackr.mao"},
	{0x0687B367AB4C6900ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\1ww_grandcrackr.msh"},
	{0x06881739975C73A6ULL, "hlac_lt_0_lowtown_a2_131069_0y.dds"},
	{0x06882883C20A023EULL, "mag311_nyssa6041969.cl"},
	{0x06897628676DB0AEULL, "character_character_p34.psh.fxo.dx11"},
	{0x06898535CF146A5BULL, "6071839_f.fxe"},
	{0x0689D9F2FB7B8314ULL, "z_cl_offset.dds"},
	{0x068A062DD32045A3ULL, "1038293607.ogg"},
	{0x068A282AD9399DEAULL, "art\\vfx\\combat\\vfx_w_wns_front_close4_c.mmh"},
	{0x068A60C55C697278ULL, "dae000_oldmangossip6174982.cl"},
	{0x068A60F58D6A35D6ULL, "637483964.ogg"},
	{0x068A979504D79F5AULL, "gen_im_wep_rog_arc_act3_04.uti"},
	{0x068AEDCA83D35E58ULL, "6048435_m.fxe"},
	{0x068B0360A33B1379ULL, "follower_banter6163420.cl"},
	{0x068B324E16A07A1AULL, "vxlt_0_lowtown_a2_n_gpg_0_0.vlm"},
	{0x068B5EFC25E7FBD8ULL, "pro000ip_corpse_2.utp"},
	{0x068B7250BBDE829BULL, "vxw_encounter1_jh_0_0.vlm"},
	{0x068B7F7A36ED26CAULL, "hlac_lt_0_lowtown_65534_2x.dds"},
	{0x068B839B0F0C0E65ULL, "vxlt_0_lowtown_docks_bv5_0_0.vlm"},
	{0x068BCA60257F91F8ULL, "vxca_cavern1_haunted_eib_0_0.vlm"},
	{0x068C29C038623D46ULL, "ht_brothel_n_3364_lc.mmh"},
	{0x068CC1CD1CFADCE2ULL, "ave200_aveline_bar6116201.cl"},
	{0x068CD8458800C1A2ULL, "zz_mag_debug6054483.cl"},
	{0x068CD84F96140684ULL, "113148028.ogg"},
	{0x068D3C62A300B98DULL, "art\\vfx\\creatures\\fxc_drg_brth_c_c.impa.gad"},
	{0x068D9E721D1E2A9AULL, "gen_im_trs_gen_amu_r0.uti"},
	{0x068DC0BC56114AE4ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\twohanded\\runfront.gad"},
	{0x068DE51E14BC3259ULL, "vxlt_blackemporium_t6_0_0.vlm"},
	{0x068DFFB51C335D0FULL, "6059782_m.fxe"},
	{0x068E12A0A9EB6B96ULL, "6062782_f.fxe"},
	{0x068E50A29D367245ULL, "mcr371ip_vfx_target.utp"},
	{0x068EB2191F9BEEDBULL, "gen00fl_isabela6075315.cl"},
	{0x068EEF3EDCEFCDC6ULL, "vxlt_gamlenshouse_n_n0_0_0.vlm"},
	{0x06901609EC155DA6ULL, "art\\vfx\\combat\\vfx_glb_ele_p_additive_c.impa.ani"},
	{0x069061D67A7F2A0DULL, "6164196_m.fxe"},
	{0x0690B2115864BF9EULL, "gen00fl_varric6099408.cl"},
	{0x0690BB9BDC937ECCULL, "vxw_bonepitmining_a2_fl_0_0.vlm"},
	{0x06916FDB7F5FDAC9ULL, "mag200_feynriel6023974.cl"},
	{0x0691D609C1D2CFB7ULL, "art\\vfx\\creatures\\fxc_arcane_tele_p.cess.evt"},
	{0x0691DC321CCF4901ULL, "vxlt_gallowstemplar_685_0_0.vlm"},
	{0x0692B371109D155CULL, "art\\characters\\playercharacter\\humanmale\\hoo_lgta.phy"},
	{0x069305FD31814CC5ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\order_right.ani"},
	{0x069309907C9FC6A4ULL, "fx_darkbolt_add.mao"},
	{0x069363C6A78A82E7ULL, "vxlt_0_lowtown_a2_ekj_0_0.vlm"},
	{0x06936B37C6A9BB09ULL, "453903928.ogg"},
	{0x06939DD4712EF8BFULL, "dae150ip_rockslide.ncs"},
	{0x0693B432910A03C5ULL, "6076541_m.fxe"},
	{0x0694812FED096670ULL, "149231442.ogg"},
	{0x069482FA0B9777B3ULL, "6099590_f.fxe"},
	{0x06949AED0D431658ULL, "plt_vis160pt_traps.nss"},
	{0x0694A36D09986BBBULL, "vxca_caverns1_mines_ii9_0_0.vlm"},
	{0x0694AF3A5085E8F6ULL, "vxlt_0_lowtown_docks_n_ba8_0_0.vlm"},
	{0x0694CFBC561782C2ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\twohanded\\runfront.ani"},
	{0x069589BC56718349ULL, "6040575_m.fxe"},
	{0x0695ADDD8D77EE7BULL, "vxlt_0_lowtown_docks_a3_bvd_0_0.vlm"},
	{0x0695F5AEE2480D84ULL, "6153788_m.fxe"},
	{0x0695FB439E3625CFULL, "434541046.ogg"},
	{0x0696049613020EDCULL, "llac_lt_0_lowtown_a2_65536_2.dds"},
	{0x069639B7FC85DB71ULL, "313122261.ogg"},
	{0x0696D90442A35F57ULL, "486383232.ogg"},
	{0x06970BAE485B2564ULL, "6143402_m.fxe"},
	{0x06974BB80AFC6DADULL, "art\\levels\\races\\ferelden\\props\\food\\prp_cheesewheela.phy"},
	{0x0698CBE414A71C74ULL, "6119135_m.fxe"},
	{0x0698D3B63966A202ULL, "gen00fl_sebastian6173743.cl"},
	{0x0698D709C1D8F040ULL, "art\\vfx\\creatures\\fxc_arcane_tele_p.cess.gad"},
	{0x0698F3AA07DC0795ULL, "face_face_p0.vsh.fxo.dx11.assembly"},
	{0x06991947EFE6F138ULL, "413877842.ogg"},
	{0x069954C0B60B66B4ULL, "hlac_lt_foundry_n_320_8y.dds"},
	{0x06995FA775C976D8ULL, "6043535_m.fxe"},
	{0x0699FEEAB50CCED5ULL, "vxht_0_hightown_asx_0_0.vlm"},
	{0x069A126775C735ECULL, "vxht_playershouse_n_1tb_0_0.vlm"},
	{0x069AAEB9E7D268B6ULL, "388813504.ogg"},
	{0x069AEF3278D0FBC3ULL, "444030646.ogg"},
	{0x069C4E81C9B94829ULL, "zz_ave_debug6056764.cl"},
	{0x069C70701F8FFB46ULL, "484605898.ogg"},
	{0x069C8628C33FC98DULL, "vxht_keep_n_63d_0_0.vlm"},
	{0x069C961E5D60886CULL, "182822612.ogg"},
	{0x069CA715B4814B5AULL, "302364844.ogg"},
	{0x069CA72C277234ACULL, "art\\characters\\playercharacter\\humanfemale\\mt_aet_hf05.phy"},
	{0x069CE7DC40935F16ULL, "6166903_f.fxe"},
	{0x069CFE09EC1FBF17ULL, "art\\vfx\\combat\\vfx_glb_ele_p_additive_c.impa.evt"},
	{0x069D49C6FCABF4E9ULL, "hlac_lt_gallowsprison_n_-1_11x.dds"},
	{0x069D6DF0732C549DULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\_textures\\prp_gate_guardian_d.dds"},
	{0x069E012D686E57E5ULL, "295456976.ogg"},
	{0x069E6BE381ACC531ULL, "mag291_torpor6196575.cl"},
	{0x069F2D78CE89974CULL, "vxw_bonepitmining_a2_mm_0_0.vlm"},
	{0x069FC6A4C3AC79A6ULL, "6156095_m.fxe"},
	{0x06A0009E988ECF74ULL, "872997646.ogg"},
	{0x06A0312CB1F867BEULL, "dae000_pc_home_ambients.cnv"},
	{0x06A04ABBB5927ECEULL, "vxht_0_hightown_bgd_0_0.vlm"},
	{0x06A051598284DF89ULL, "vxht_chantry_2ck_0_0.vlm"},
	{0x06A0744A56FC2B90ULL, "dre000_bodahn6031354.cl"},
	{0x06A0D94431790E7CULL, "lt_0_lowtown_docks_a3_28_0.rml"},
	{0x06A12E2FC255C613ULL, "art\\characters\\creatures\\stonegolem\\animation\\cwkb.evt"},
	{0x06A1441BEEE71491ULL, "vxca_caverns1_i34_0_0.vlm"},
	{0x06A186949B45B368ULL, "vxca_caverns1_iio_0_0.vlm"},
	{0x06A1A7CDAF76C471ULL, "1027632389.ogg"},
	{0x06A265CE65BE7F6BULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_cupa_sa.dds"},
	{0x06A2783B90E211F7ULL, "358956274.ogg"},
	{0x06A27AA61708CEB7ULL, "373591425.ogg"},
	{0x06A2D1DDF4652AD7ULL, "mag330st_walter_1.stg"},
	{0x06A3044F390557F3ULL, "dae200_saemus6091705.cl"},
	{0x06A309143C39B75DULL, "hlac_ht_0_hightown_a2_131071_1z.dds"},
	{0x06A3FF09EC25DFA0ULL, "art\\vfx\\combat\\vfx_glb_ele_p_additive_c.impa.gad"},
	{0x06A40C70574580D8ULL, "vxca_caverns1_gkb_0_1.vlm"},
	{0x06A4132CB1FC1DCEULL, "dae000_pc_home_ambients.bnk"},
	{0x06A42B3CD4F2D96EULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\9ff_thinsmall.msh"},
	{0x06A4393CD4F2F0BFULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\9ff_thinsmall.mao"},
	{0x06A4453CD4F3055CULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\9ff_thinsmall.mmh"},
	{0x06A4E44E968DE465ULL, "622301975.ogg"},
	{0x06A4E5C172DB5A15ULL, "lgt260cr_eau_tmp_conrad.utc"},
	{0x06A536C5F7892B19ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\_textures\\vst_hightown_n.dds"},
	{0x06A5E89369C4E47FULL, "6152119_m.fxe"},
	{0x06A5F26B349EDB84ULL, "hlac_lt_0_lowtown_a2_n_c_-131071_0x.dds"},
	{0x06A631FC016650EBULL, "hlac_dr_primevalentrance_1726_11y.dds"},
	{0x06A636A598AAB53AULL, "6125595_f.fxe"},
	{0x06A6D71369FE5EFDULL, "gen00fl_varric6033690.cl"},
	{0x06A70E37ABEACDEBULL, "474962373.ogg"},
	{0x06A7226046830C09ULL, "vxht_0_hightown_n_7fv_0_0.vlm"},
	{0x06A7272FC25A2654ULL, "art\\characters\\creatures\\stonegolem\\animation\\cwkb.gad"},
	{0x06A75B37301E7DDDULL, "6152197_m.fxe"},
	{0x06A7AC3368C72990ULL, "vxlt_gallowscourtyard_n_7fv_0_0.vlm"},
	{0x06A7E5D3C8B7D434ULL, "vxlt_0_lowtown_n_e29_0_0.vlm"},
	{0x06A812BF5B32C452ULL, "387893583.ogg"},
	{0x06A81607FA8CAB1AULL, "vxlt_0_lowtown_a2_n_gja_0_0.vlm"},
	{0x06A83C4D4A3B8B8FULL, "gen00fl_isabela6079298.cl"},
	{0x06A8A8E264D9B93EULL, "420859301.ogg"},
	{0x06A8B1FBBE9D661EULL, "6178459_m.fxe"},
	{0x06A8C2170C9E9801ULL, "6174795_m.fxe"},
	{0x06A8C870976846B6ULL, "art\\levels\\races\\proxy\\props\\4tools\\inv_ball4_0.pwk"},
	{0x06A8DDA2A4E81434ULL, "6147716_m.fxe"},
	{0x06A8DED24F143070ULL, "zz_isa_debug6175285.cl"},
	{0x06A91233756292FFULL, "nat320_delilah6065949.cl"},
	{0x06A947B198F757DCULL, "gen_im_arm_glv_lgt_act1_02.uti"},
	{0x06A950C89044B542ULL, "6075734_m.fxe"},
	{0x06AA328CA6B9398BULL, "vxlt_0_lowtown_a2_n_ett_0_0.vlm"},
	{0x06AAA8C6A16AC204ULL, "hlac_w_sundermount_-196607_0y.dds"},
	{0x06AB2376917FEEAAULL, "art\\vfx\\environment\\vfx_trap_acid_proj_p.cess.gad"},
	{0x06ABB89FA29875F5ULL, "vxht_mansion_n_5mi_0_0.vlm"},
	{0x06ABD321BB15D78CULL, "vxht_chantry_26m_0_0.vlm"},
	{0x06AC1A6763A9AC9EULL, "vxht_0_hightown_n_ca8_0_0.vlm"},
	{0x06ACAF388ACCA10DULL, "vxlt_0_lowtown_bxz_0_0.vlm"},
	{0x06ACC067AB6B9BC5ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\1ww_grandcrackr.phy"},
	{0x06ACE13CB03BC3F9ULL, "vxdr_primevalentrance_24s_0_0.vlm"},
	{0x06ACFDAE5E8E7BDDULL, "mer341_jansen6082993.cl"},
	{0x06AD0901AFBBBCACULL, "624818676.ogg"},
	{0x06AD146EF8B87274ULL, "6074387_f.fxe"},
	{0x06AD1B8F1D2B92AFULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\reactions\\rct_bldwnd_ent.gad"},
	{0x06AD46D449DFD011ULL, "22523524.ogg"},
	{0x06AD9F55C07FAFCCULL, "vxw_bonepitmining_a2_hb_0_0.vlm"},
	{0x06AE31497A03EC8CULL, "lt_backalley_n_0_0.rml"},
	{0x06AE362FC2605E32ULL, "art\\characters\\creatures\\stonegolem\\animation\\cwkb.ani"},
	{0x06AE70ECE8D9FD7EULL, "gen00fl_merrill6119616.cl"},
	{0x06AE907C6C5EACABULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\blendtree\\pose_blessing.anb"},
	{0x06AEB194382F56DCULL, "llac_lt_warehouse_0_9.dds"},
	{0x06AEDCF0F9B08F91ULL, "vxw_encounter3_9q_0_0.vlm"},
	{0x06AF32805C499AA9ULL, "767256262.ogg"},
	{0x06AFB88509306F2AULL, "vxw_bonepit_11j_0_0.vlm"},
	{0x06AFB9C53B6596C1ULL, "l_isa121ar_chantry.bnk"},
	{0x06AFDB8E71C7A0FDULL, "plt_cod_hst_antiva.plo"},
	{0x06AFEB303CD990CAULL, "45836127.ogg"},
	{0x06B020B5A69DD163ULL, "971961642.ogg"},
	{0x06B028CC58438F27ULL, "382137704.ogg"},
	{0x06B0D4F6879DADE7ULL, "mag100_cullen6095081.cl"},
	{0x06B0EC789AD14B5EULL, "vxw_sundermount_2op_0_0.vlm"},
	{0x06B13DFE97100745ULL, "vxht_mansion2_1wd_0_0.vlm"},
	{0x06B13E760A9C8A75ULL, "mcr310_argument6182303.cl"},
	{0x06B1C9328703BFBDULL, "572169896.ogg"},
	{0x06B206769185DC41ULL, "art\\vfx\\environment\\vfx_trap_acid_proj_p.cess.evt"},
	{0x06B223FABAFD085EULL, "gen00fl_aveline6131097.cl"},
	{0x06B2AC2B99473852ULL, "mag320_nobleman_16060261.cl"},
	{0x06B33C575C5A21CFULL, "ch1armtnt_ds2.psh.fxo"},
	{0x06B3BA69F88E573EULL, "vxht_keep_n_8aw_0_0.vlm"},
	{0x06B4308F1D31D55CULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\reactions\\rct_bldwnd_ent.evt"},
	{0x06B4611A7F4AE805ULL, "vxlt_0_lowtown_h7u_0_0.vlm"},
	{0x06B4B5C60729948EULL, "6082695_m.fxe"},
	{0x06B569C1644A4A9FULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\animation\\hightown_gate.open.gad"},
	{0x06B57F6EB66F52B7ULL, "qun120_arishok6051312.cl"},
	{0x06B6303BC4ECBAF9ULL, "vxlt_gallowstemplar_2p1_0_0.vlm"},
	{0x06B646C98EAF07DDULL, "6026988_m.fxe"},
	{0x06B6582E88453FBFULL, "vxht_playershouse_n_1qb_0_0.vlm"},
	{0x06B6B7DA799DD497ULL, "llac_lt_0_lowtown_a2_n_c_-65535_0.dds"},
	{0x06B6E6889CDECEF8ULL, "620696377.ogg"},
	{0x06B70D78CAFE9245ULL, "6126514_m.fxe"},
	{0x06B72E317CB27779ULL, "1009157208.ogg"},
	{0x06B73E0334EE3DB6ULL, "vxca_caverns1_mines_i1j_0_2.vlm"},
	{0x06B7FD54B9C6D617ULL, "art\\characters\\playercharacter\\dwarfmale\\mt_cbt_sand.mmh"},
	{0x06B81354B9C6FB89ULL, "art\\characters\\playercharacter\\dwarfmale\\mt_cbt_sand.msh"},
	{0x06B83F5F446F6F87ULL, "fen110_anso6095006.cl"},
	{0x06B8F69422C0FF8BULL, "hlac_lt_0_lowtown_docks_a3_131071_3z.dds"},
	{0x06B925531352414CULL, "hlac_w_bonepit_3_0y.dds"},
	{0x06B952A8848F0096ULL, "vxlt_darktown_517_0_1.vlm"},
	{0x06B9D777089AADFBULL, "hlac_w_bonepitmining_a2_0_5x.dds"},
	{0x06B9E9BA5D3C166CULL, "138554042.ogg"},
	{0x06BA2C051D4B5E50ULL, "317738526.ogg"},
	{0x06BA54B1010D9CCAULL, "6030441_m.fxe"},
	{0x06BAE7B2FEA707D6ULL, "688482529.ogg"},
	{0x06BB06786B56C430ULL, "6139469_m.fxe"},
	{0x06BB2D9FC5FE5E96ULL, "isa120st_hayder.stg"},
	{0x06BB52EE0A11983FULL, "6167568_m.fxe"},
	{0x06BB58997076584FULL, "gen00fl_merrill6129928.cl"},
	{0x06BB864BEAAE8AF4ULL, "6079087_f.fxe"},
	{0x06BBD1B48FDFB0DEULL, "mag121_idunna6084191.cl"},
	{0x06BC046823366062ULL, "vxca_caverns1_mines_iob_0_0.vlm"},
	{0x06BC6057F6E11B2CULL, "vxw_bonepitdestroyed_a3_9o_0_0.vlm"},
	{0x06BC99F04062BC3AULL, "w_bonepitdestroyed_a3_2_0_lc.mmh"},
	{0x06BCABD3BFCA8AC4ULL, "hlac_ca_caverns2prc_5802_15x.dds"},
	{0x06BCC3CCBDD1C9A5ULL, "hlac_lt_0_lowtown_a2_n_c_-3_2y.dds"},
	{0x06BD0ABF7B66804BULL, "art\\vfx\\combat\\vfx_glb_fire_imp_maj_c.mmh"},
	{0x06BD273898968D94ULL, "pro000_cut_flemeth_rescue6001230.cl"},
	{0x06BD27C19D221143ULL, "dae020ip_merch_tailor.utp"},
	{0x06BD351366711089ULL, "vxht_chantry_2yi_0_0.vlm"},
	{0x06BD4ECDEBA6A811ULL, "vxw_encounter3_2p_0_0.vlm"},
	{0x06BDEE44D582789EULL, "vxht_0_hightown_a2_50n_0_0.vlm"},
	{0x06BDF8302C2E353FULL, "6110287_m.fxe"},
	{0x06BE679AE7009F55ULL, "vxlt_0_lowtown_n_ded_0_0.vlm"},
	{0x06BE6BE20706C0C2ULL, "6084231_m.fxe"},
	{0x06BEFE7691905890ULL, "art\\vfx\\environment\\vfx_trap_acid_proj_p.cess.ani"},
	{0x06BF3BA753780411ULL, "gen00fl_dog6117713.cl"},
	{0x06BFC17F5B66E013ULL, "art\\characters\\playercharacter\\qunarimale\\animation\\cindes\\custom\\arishok_collapse_loop.evt"},
	{0x06BFEBF6459A49E8ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\3ww_alcovetop.msh"},
	{0x06BFEFE644EB7447ULL, "hlac_lt_0_lowtown_n_65535_1z.dds"},
	{0x06BFFDF6459A6879ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\3ww_alcovetop.mao"},
	{0x06C007A8415E4583ULL, "dae210_gate_guard6209980.cl"},
	{0x06C009F6459A7C12ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\3ww_alcovetop.mmh"},
	{0x06C00DCE5A73D9FAULL, "vxht_mansion2_1kk_0_0.vlm"},
	{0x06C0C8D14C3E1EFCULL, "307723460.ogg"},
	{0x06C1272C2DC99AFFULL, "603666314.ogg"},
	{0x06C135C1231A7C73ULL, "6127432_m.fxe"},
	{0x06C1AE934BAFB07BULL, "319725558.ogg"},
	{0x06C1E221059152AAULL, "131177778.ogg"},
	{0x06C1FE0400B06FE6ULL, "rdr100pt_keepers.ncs"},
	{0x06C2388F1D3E2075ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\reactions\\rct_bldwnd_ent.ani"},
	{0x06C239679814A81BULL, "art\\characters\\playercharacter\\humanmale\\textures\\bdy_arm_masa_0n.dds"},
	{0x06C31694A5A43D71ULL, "zz_followers_banter6159489.cl"},
	{0x06C34B31F65E407FULL, "vxlt_gallowstemplar_5cu_0_0.vlm"},
	{0x06C36E22D7955F0DULL, "6081208_m.fxe"},
	{0x06C39B62D0555210ULL, "6042406_m.fxe"},
	{0x06C3B2A6B4B40B51ULL, "499585223.ogg"},
	{0x06C3F062F650E0D5ULL, "vxca_cavern1_haunted_dpi_0_0.vlm"},
	{0x06C41F8D1C8DFEC6ULL, "vxw_bonepitmining_a2_1xc_0_0.vlm"},
	{0x06C4B2C5834F133AULL, "656171415.ogg"},
	{0x06C4DCF4A311139AULL, "646180225.ogg"},
	{0x06C548C007554D85ULL, "6103645_m.fxe"},
	{0x06C56E8AEFDE1B46ULL, "138118116.ogg"},
	{0x06C5BA7F5B6B4054ULL, "art\\characters\\playercharacter\\qunarimale\\animation\\cindes\\custom\\arishok_collapse_loop.gad"},
	{0x06C636832F3E9FE8ULL, "916578916.ogg"},
	{0x06C66B2495072715ULL, "197047975.ogg"},
	{0x06C6E8A1A5A02DC5ULL, "6172700_m.fxe"},
	{0x06C7BC5B166F8F0AULL, "hlac_lt_0_lowtown_a2_n_c_65533_2x.dds"},
	{0x06C81ADB47B4A68CULL, "493228196.ogg"},
	{0x06C837378D60A793ULL, "vxlt_gallowstemplar_38r_0_0.vlm"},
	{0x06C8AFDD0821E6AFULL, "495003840.ogg"},
	{0x06C8EB5D2636C269ULL, "522751623.ogg"},
	{0x06C8EF09071803CBULL, "art\\characters\\creatures\\mubari\\animation\\p_wkf.ani"},
	{0x06C9059D6410E4A7ULL, "vxlt_0_lowtown_a2_n_fip_0_0.vlm"},
	{0x06C9243CD511E927ULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\9ff_thinsmall.phy"},
	{0x06C9ED13226247C8ULL, "hlac_ca_caverns1_mines_24421_7x.dds"},
	{0x06CA070F880A558BULL, "6071128_m.fxe"},
	{0x06CA46C1645C6BE5ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\animation\\hightown_gate.open.ani"},
	{0x06CA8A0078043E14ULL, "mas.cut"},
	{0x06CABAF9CF1A6E80ULL, "vxlt_gallowstemplar_f_3n8_0_0.vlm"},
	{0x06CAD45D1B7744AEULL, "6086170_m.fxe"},
	{0x06CAD72CB374DF4AULL, "ser221_gascard6133877.cl"},
	{0x06CAEBBF7B7288B0ULL, "art\\vfx\\combat\\vfx_glb_fire_imp_maj_c.anb"},
	{0x06CB0F2A0CC38704ULL, "art\\characters\\creatures\\corpse\\animation\\combat\\walkbackright.ani"},
	{0x06CB3461369828F3ULL, "246794454.ogg"},
	{0x06CB7A5C7843B080ULL, "65821246.ogg"},
	{0x06CBC3279C0B4C3FULL, "hlac_lt_0_lowtown_a2_n_c_1_2x.dds"},
	{0x06CBEDE26A8A3E08ULL, "vxlt_0_lowtown_docks_a3_n_d4p_0_0.vlm"},
	{0x06CC9A220C569F52ULL, "and100_anders6095996.cl"},
	{0x06CCACBF6B21F0F2ULL, "vxht_0_hightown_a2_4n1_0_0.vlm"},
	{0x06CCC97F5B717832ULL, "art\\characters\\playercharacter\\qunarimale\\animation\\cindes\\custom\\arishok_collapse_loop.ani"},
	{0x06CCD3D447821DA5ULL, "gen00fl_aveline6112705.cl"},
	{0x06CCDA44319E4ACCULL, "lt_0_lowtown_docks_a3_28_0.anb"},
	{0x06CD13BD2DB69D01ULL, "1004621212.ogg"},
	{0x06CD2166F7750AC2ULL, "vxht_mansion2_3cu_0_0.vlm"},
	{0x06CD40777161573CULL, "hlac_ht_chantry_n_2534_20x.dds"},
	{0x06CD720310411DC3ULL, "68297658.ogg"},
	{0x06CD76D6AD1EDEBBULL, "hlac_ca_caverns2prc_5802_22y.dds"},
	{0x06CDE32D2B5462C9ULL, "gen00fl_anders6111318.cl"},
	{0x06CDF6AF10411B6CULL, "6173364_m.fxe"},
	{0x06CE7FAB4C69F27AULL, "vxht_mansion2_1nd_0_0.vlm"},
	{0x06CEB02185F977F5ULL, "ser221_gamlen6131640.cl"},
	{0x06CEE69F219FA037ULL, "ttx_roof_int_d_1.mmh"},
	{0x06CEE93480F12FC2ULL, "vxlt_gallowscourtyard_n_7ey_0_0.vlm"},
	{0x06CEFC9F219FC5A9ULL, "ttx_roof_int_d_1.msh"},
	{0x06CF07B84081C90CULL, "480460697.ogg"},
	{0x06CF61FB6952E5A6ULL, "vxlt_gallowstemplar_f_5ya_0_0.vlm"},
	{0x06CF6F75F923609EULL, "vxlt_gallowstemplar_46z_0_0.vlm"},
	{0x06CFA2019DD67671ULL, "400254932.ogg"},
	{0x06CFCF86CF383FA4ULL, "vxw_encounter2_rs_0_1.vlm"},
	{0x06CFDE1CC948E0ADULL, "861527240.ogg"},
	{0x06D08CAAE1DF71A5ULL, "ave200_party_comment6116370.cl"},
	{0x06D092A796BBC5EBULL, "ico_dagger_2.dds"},
	{0x06D0B6541455603CULL, "6116163_m.fxe"},
	{0x06D0FA9B6D4F9AE3ULL, "art\\characters\\playercharacter\\humanmale\\animation\\ambient\\dialogue\\talk_pointing.ani"},
	{0x06D17848D613D507ULL, "vxca_caverns1_mines_ehr_0_0.vlm"},
	{0x06D178B9AC6573FAULL, "492494020.ogg"},
	{0x06D1BD0EE85458FFULL, "vxlt_gallowstemplar_5lt_0_0.vlm"},
	{0x06D1DA5DF4E88DB0ULL, "6150334_m.fxe"},
	{0x06D1F278B4AE0859ULL, "6120783_m.fxe"},
	{0x06D20C2A0CC9A166ULL, "art\\characters\\creatures\\corpse\\animation\\combat\\walkbackright.gad"},
	{0x06D2916A0E841746ULL, "vxw_bonepitmining_a2_1ud_0_0.vlm"},
	{0x06D3119AFDF7B621ULL, "art\\characters\\playercharacter\\humanmale\\animation\\ambient\\dialogue\\thinking_weightshift_gesture_left_hand.ani"},
	{0x06D31AD8B2574BFDULL, "one121_trap6112683.cl"},
	{0x06D3BA9CF94B6605ULL, "6104745_m.fxe"},
	{0x06D41485E531022BULL, "qun220cr_javaris6071953.cl"},
	{0x06D44F71002EB575ULL, "vxlt_hangedman_1eh_0_0.vlm"},
	{0x06D4A34779D02D7FULL, "360481153.ogg"},
	{0x06D4A8843E2A42F3ULL, "vxlt_0_lowtown_eja_0_0.vlm"},
	{0x06D4CF9206249817ULL, "276434610.ogg"},
	{0x06D4D5780A743E53ULL, "hlac_lt_0_lowtown_n_131073_1y.dds"},
	{0x06D537B05935AF12ULL, "hlac_ht_0_hightown_a3_-131073_1y.dds"},
	{0x06D56CFBCB0DCD41ULL, "llac_lt_0_lowtown_docks_a3_n_65534_4.dds"},
	{0x06D5730A8C84118FULL, "hlac_ht_keep_4519_4y.dds"},
	{0x06D59B82F8AB2FEDULL, "406212550.ogg"},
	{0x06D6C88650B00F3BULL, "7732721.ogg"},
	{0x06D6D70907241842ULL, "art\\characters\\creatures\\mubari\\animation\\p_wkf.evt"},
	{0x06D70A7BCD395D00ULL, "vxlt_gallowstemplar_n_395_0_0.vlm"},
	{0x06D73D30CFC119B1ULL, "art\\vfx\\environment\\level_fx\\bugs\\vfx_moths.mmh"},
	{0x06D7A59E08E32644ULL, "617337283.ogg"},
	{0x06D7A84D75DDE1B5ULL, "snow_02_l2.mao"},
	{0x06D7BBCFDFA8B417ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\prp_railend.mmh"},
	{0x06D7BFCFDFA8BB24ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\prp_railend.mao"},
	{0x06D7D1CFDFA8D989ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\prp_railend.msh"},
	{0x06D7F8286FD7CEE3ULL, "6116303_f.fxe"},
	{0x06D7FEF580F39C0FULL, "vxht_0_hightown_by7_0_0.vlm"},
	{0x06D8072A0CCE0445ULL, "art\\characters\\creatures\\corpse\\animation\\combat\\walkbackright.evt"},
	{0x06D80F956A362CEDULL, "seb221_ruxton_harimann6177203.cl"},
	{0x06D8173DA2D49AFDULL, "gen00fl_anders6201156.cl"},
	{0x06D82B7AE452EAC0ULL, "6176288_m.fxe"},
	{0x06D840E1E0E34738ULL, "vxlt_0_lowtown_docks_a3_aos_0_0.vlm"},
	{0x06D87543D883DCD9ULL, "vxlt_undercityserialkiller_1ms_0_0.vlm"},
	{0x06D8938F43CC7591ULL, "vxlt_0_lowtown_a2_eqr_0_0.vlm"},
	{0x06D90D47ACF89D74ULL, "vxlt_0_lowtown_n_gpe_0_0.vlm"},
	{0x06D914EF6A8BFE96ULL, "vxlt_warehouse_n_sy_0_0.vlm"},
	{0x06D9648C7AA91423ULL, "vxlt_undercityserialkiller_1a5_0_0.vlm"},
	{0x06D966444A6F87ADULL, "481016883.ogg"},
	{0x06D98FCA7897F848ULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_brrckbrd_sa.dds"},
	{0x06D9E58AEACFB1FDULL, "mer140ip_at_caverns.utp"},
	{0x06D9E74687B6536FULL, "44933787.ogg"},
	{0x06D9EB1C0A3D2F85ULL, "6096266_m.fxe"},
	{0x06DA1C9AFDFDE7C3ULL, "art\\characters\\playercharacter\\humanmale\\animation\\ambient\\dialogue\\thinking_weightshift_gesture_left_hand.gad"},
	{0x06DB2195B0370BD0ULL, "w_sundermount_228_0.mao"},
	{0x06DB2595B03712E3ULL, "w_sundermount_228_0.mmh"},
	{0x06DB3395B0372A3DULL, "w_sundermount_228_0.msh"},
	{0x06DBB9A1DFA3210BULL, "6131206_m.fxe"},
	{0x06DBE2E7F3C58855ULL, "one100dg_lia6049832.cl"},
	{0x06DCC5275B9E48B6ULL, "vxdr_dungeon_1lt_0_1.vlm"},
	{0x06DCD60907288255ULL, "art\\characters\\creatures\\mubari\\animation\\p_wkf.gad"},
	{0x06DCF25BD9DC5A14ULL, "llac_lt_0_lowtown_docks_-131075_4.dds"},
	{0x06DD5B1172E74842ULL, "vxlt_gallowscourtyard_n_7hz_0_0.vlm"},
	{0x06DD7E2C27A98677ULL, "art\\characters\\playercharacter\\humanfemale\\mt_aet_hf05.mmh"},
	{0x06DD942C27A9ABE9ULL, "art\\characters\\playercharacter\\humanfemale\\mt_aet_hf05.msh"},
	{0x06DE129B6D5A4DDAULL, "art\\characters\\playercharacter\\humanmale\\animation\\ambient\\dialogue\\talk_pointing.evt"},
	{0x06DEE0A1A94DC315ULL, "fx_fl_smoke_add.mao"},
	{0x06DF24E37A0BDCC0ULL, "mag100_cullen6093547.cl"},
	{0x06DF35365CD5E088ULL, "6180443_m.fxe"},
	{0x06DF4E5872219A3CULL, "6082266_m.fxe"},
	{0x06DFF09FA5113FA7ULL, "6167484_f.fxe"},
	{0x06E02EEBDA4A717FULL, "vxlt_gallowstemplar_5iw_0_0.vlm"},
	{0x06E03A0B75D2EC53ULL, "fen210_fenris6108754.cl"},
	{0x06E0F99AFE03CAA8ULL, "art\\characters\\playercharacter\\humanmale\\animation\\ambient\\dialogue\\thinking_weightshift_gesture_left_hand.evt"},
	{0x06E18FFF4B5212B4ULL, "vxht_keep_n_7aj_0_0.vlm"},
	{0x06E1BD3F5FD88081ULL, "6034327_m.fxe"},
	{0x06E23521A3C55992ULL, "vxlt_0_lowtown_a2_n_c_dow_0_0.vlm"},
	{0x06E250AC7AE95D0BULL, "gen00fl_fenris6128066.cl"},
	{0x06E2AC41DA3B7C68ULL, "vxlt_gallowscourtyard_62t_0_0.vlm"},
	{0x06E2F7544FCE717AULL, "glo_vfx_traps.bnk"},
	{0x06E300A72F484E74ULL, "12571673.ogg"},
	{0x06E31AC4A781C230ULL, "vxlt_gallowstemplar_n_38v_0_0.vlm"},
	{0x06E3E8645A254DB3ULL, "vxht_0_hightown_a2_bx0_0_0.vlm"},
	{0x06E3FBBD40E71F08ULL, "35915696.ogg"},
	{0x06E4941A3DA785D8ULL, "vxlt_0_lowtown_docks_a3_d0f_0_1.vlm"},
	{0x06E4A090802F911DULL, "hlac_lt_0_lowtown_docks_-2_4z.dds"},
	{0x06E5119B6D606ACDULL, "art\\characters\\playercharacter\\humanmale\\animation\\ambient\\dialogue\\talk_pointing.gad"},
	{0x06E522FF27769D7BULL, "mag311_huon6042574.cl"},
	{0x06E52C1A67F5131BULL, "lt_0_lowtown_docks_126_0.mmh"},
	{0x06E5381A67F527B8ULL, "lt_0_lowtown_docks_126_0.mao"},
	{0x06E54A1A67F54605ULL, "lt_0_lowtown_docks_126_0.msh"},
	{0x06E57AC046A01A58ULL, "6092230_m.fxe"},
	{0x06E5985E82B67BF2ULL, "840231134.ogg"},
	{0x06E5AE25714138FCULL, "6143284_m.fxe"},
	{0x06E5B874C5633FABULL, "richard_test_st.stg"},
	{0x06E6224298BB6AE6ULL, "llac_lt_0_lowtown_n_1_7.dds"},
	{0x06E6458ECE491DC4ULL, "274953840.ogg"},
	{0x06E6DF88F7959AF9ULL, "root_clu.mao"},
	{0x06E70867D7B9D25DULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\trap_1dwarven_0.pwk"},
	{0x06E722F701C97E76ULL, "vxht_0_hightown_n_5jc_0_0.vlm"},
	{0x06E743D9EEE94A39ULL, "6100042_m.fxe"},
	{0x06E7555F79BD410FULL, "513628503.ogg"},
	{0x06E77A0349A3CE95ULL, "187533396.ogg"},
	{0x06E786CC5C821716ULL, "vxlt_warehouse_n_vz_0_0.vlm"},
	{0x06E812D8245DED50ULL, "vxw_encounter2_camp_qs_0_0.vlm"},
	{0x06E8212A86F9189CULL, "art\\characters\\playercharacter\\humanmale\\textures\\hn_glv_medc_0n.dds"},
	{0x06E85CBFC7434CDDULL, "37159721.ogg"},
	{0x06E892AB47C7D157ULL, "mag160_feynriel6016586.cl"},
	{0x06E8DD38188E04A9ULL, "71683659.ogg"},
	{0x06E91086BBD3AEA4ULL, "vxlt_warehouse_n_i5_0_0.vlm"},
	{0x06E96814BDACEB9DULL, "vxw_sundermount_219_0_0.vlm"},
	{0x06E980280B9ED268ULL, "mag370_meredith6064390.cl"},
	{0x06E9CFBD850E72C2ULL, "6045368_m.fxe"},
	{0x06EA3B810963A1CFULL, "vxlt_0_lowtown_n_d9b_0_0.vlm"},
	{0x06EAFCAD4B10600BULL, "777317973.ogg"},
	{0x06EB1AE4C9EA2B15ULL, "llac_w_encounter2_camp_196612_0.dds"},
	{0x06EB4FFC40A28AC2ULL, "gen00fl_varric6105745.cl"},
	{0x06EB6CC5AA6F82CFULL, "460220665.ogg"},
	{0x06EBEF20FDDCE744ULL, "hlac_w_sundermount_196603_0x.dds"},
	{0x06EC20E3B61BE543ULL, "432429324.ogg"},
	{0x06EC661E54569289ULL, "6025361_m.fxe"},
	{0x06ED49D8B94DF698ULL, "387769264.ogg"},
	{0x06ED5A5E3DFFD47AULL, "6119469_m.fxe"},
	{0x06EDF6BA1FC5C34DULL, "dae300_cullen6140647.cl"},
	{0x06EE35E8EE2FDD70ULL, "t3_dog_d08.tnt"},
	{0x06EE65539F3D6D8AULL, "llac_ht_keep_3733_1.dds"},
	{0x06EF0AFD09AFD6DBULL, "6155631_m.fxe"},
	{0x06EF1DD8C5972033ULL, "vxca_caverns2_8pa_0_0.vlm"},
	{0x06F0811EA8D3F070ULL, "vxht_brothel_n_5k1_0_0.vlm"},
	{0x06F08983F4C7F354ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\blendtree\\pose_leaning_over_table.anb"},
	{0x06F09E1AB8929D00ULL, "w_woundedcoast_215_0.phy"},
	{0x06F12B826BA1D2A2ULL, "one300_zevran6089726.cl"},
	{0x06F1332AE41AE675ULL, "vxlt_hangedman_1ov_0_0.vlm"},
	{0x06F15287DF923A0AULL, "6112988_f.fxe"},
	{0x06F15B324432200CULL, "6083812_m.fxe"},
	{0x06F18B056635ED20ULL, "808370756.ogg"},
	{0x06F1C693FF2867B4ULL, "6107550_m.fxe"},
	{0x06F1F2572E9D15E9ULL, "vxht_mansion2_3eh_0_0.vlm"},
	{0x06F236B883A34000ULL, "mag100_thrask6034812.cl"},
	{0x06F256904DF1E395ULL, "hlac_lt_0_lowtown_-2_3z.dds"},
	{0x06F29A2E1351378BULL, "dre000_varric6034636.cl"},
	{0x06F3656EDFA97909ULL, "766333203.ogg"},
	{0x06F3EC9D389632D8ULL, "6097130_m.fxe"},
	{0x06F488A597DFBCAEULL, "579286869.ogg"},
	{0x06F493274B0D8699ULL, "llac_lt_0_lowtown_65536_5.dds"},
	{0x06F4C073DF123094ULL, "vxw_bonepit_27e_0_0.vlm"},
	{0x06F4C51AA8B3EEE5ULL, "llac_lt_0_lowtown_docks_n_-4_5.dds"},
	{0x06F5370B58404ADAULL, "6180508_m.fxe"},
	{0x06F59C46A4BC39CEULL, "hlac_ht_playershouse_n_855_2y.dds"},
	{0x06F5F10190E4CE74ULL, "vxlt_0_lowtown_n_gjk_0_0.vlm"},
	{0x06F5F970905FDFD2ULL, "6149187_m.fxe"},
	{0x06F633ED7AC0B997ULL, "fex_ruin_arch_b.msh"},
	{0x06F649ED7AC0DF09ULL, "fex_ruin_arch_b.mmh"},
	{0x06F67C874B9B3EB9ULL, "6067855_m.fxe"},
	{0x06F6820A57A55A8DULL, "vxlt_undercitydungeon_145_0_0.vlm"},
	{0x06F684B5165405D0ULL, "vxw_encounter2_camp_lp_0_0.vlm"},
	{0x06F6C7C53BA1B173ULL, "l_isa121ar_chantry.wbd"},
	{0x06F6D43D6C433A7CULL, "6031977_f.fxe"},
	{0x06F6F5E9615711CEULL, "vxht_0_hightown_byz_0_0.vlm"},
	{0x06F7305AD2CE3670ULL, "one100dg_vanard6048252.cl"},
	{0x06F73D09F0EFC683ULL, "6082598_f.fxe"},
	{0x06F78937CE0C151DULL, "mcr310cr_templar_escort.utc"},
	{0x06F7ABBE977D61B4ULL, "vxlt_0_lowtown_docks_a0m_0_0.vlm"},
	{0x06F7BE14770A0BA0ULL, "vxw_bonepitmining_a2_5i_0_0.vlm"},
	{0x06F81DEE335ABDB0ULL, "zz_lgt_3_debug6140427.cl"},
	{0x06F88149F57DF3B0ULL, "351726277.ogg"},
	{0x06F887E76918C709ULL, "70590875.ogg"},
	{0x06F89CC6147BCAADULL, "zz_followers_banter6158958.cl"},
	{0x06F94ECF5A2A75FDULL, "6128072_f.fxe"},
	{0x06F95C68316A5FBAULL, "vxw_bonepitmining_a2_6z_0_0.vlm"},
	{0x06F96010542DC8D3ULL, "hlac_lt_gallowstemplar_n_7198_7x.dds"},
	{0x06F9A1476CC832CFULL, "6147416_m.fxe"},
	{0x06F9A845AD0E6500ULL, "sky_sky2_p1.psh.fxo.dx11.assembly"},
	{0x06FA07E8B20634B0ULL, "zz_pro_debug6007863.cl"},
	{0x06FA37644D8C2363ULL, "gen00fl_isabela6101056.cl"},
	{0x06FA879EE3B17136ULL, "vxlt_0_lowtown_docks_aor_0_0.vlm"},
	{0x06FAB7923F352F26ULL, "vxlt_gallowstemplar_f_5jd_0_0.vlm"},
	{0x06FB02366D6D105DULL, "gen00fl_sebastian6180429.cl"},
	{0x06FB7C403599C80AULL, "6100311_m.fxe"},
	{0x06FBB5225ECDA472ULL, "art\\characters\\playercharacter\\elfmale\\textures\\bdy_arm_hvyn_0d.dds"},
	{0x06FBD5C3DE8EE1D3ULL, "6083626_m.fxe"},
	{0x06FBDD921AF986ADULL, "w_encounter2_61_0.mmh"},
	{0x06FBE1921AF98DFEULL, "w_encounter2_61_0.mao"},
	{0x06FBEF921AF9A533ULL, "w_encounter2_61_0.msh"},
	{0x06FBF3233502A2B6ULL, "seb221_brett_harimann6177249.cl"},
	{0x06FCCDDFABF61E87ULL, "vxca_caverns1_mines_ego_0_0.vlm"},
	{0x06FCE4E50401E9F7ULL, "vxht_mansion2b_389_0_0.vlm"},
	{0x06FD7189E53FA64BULL, "art\\vfx\\combat\\vfx_m_cre_haste_c.mmh"},
	{0x06FD9C017DA1116FULL, "vxlt_0_lowtown_n_byp_0_0.vlm"},
	{0x06FDA10B14CA2E14ULL, "6174242_m.fxe"},
	{0x06FE48D49D09E91BULL, "art\\levels\\races\\natural\\kirkcoastal\\foliage\\bsh_01flwrplnt.msh"},
	{0x06FE4ACD74B862D2ULL, "rand_underground.gda"},
	{0x06FE5AD49D0A0866ULL, "art\\levels\\races\\natural\\kirkcoastal\\foliage\\bsh_01flwrplnt.mao"},
	{0x06FE66D49D0A1C05ULL, "art\\levels\\races\\natural\\kirkcoastal\\foliage\\bsh_01flwrplnt.mmh"},
	{0x06FE90937A84EB61ULL, "vxht_mansion2_1a2_0_0.vlm"},
	{0x06FEF6EC42F9442EULL, "art\\vfx\\combat\\vfx_r_arc_burst_smo_aoe_p.mmh"},
	{0x06FF42FD6684B11BULL, "art\\characters\\playercharacter\\humanmale\\textures\\hm_hat_medc_0d.dds"},
	{0x06FFEE33B3654387ULL, "zz_ser_debug6078155.cl"},
	{0x06FFF229CA7BC6D7ULL, "vxlt_0_lowtown_a3_n_c_e3b_0_0.vlm"},
	{0x070060D82D872721ULL, "40988930.ogg"},
	{0x0700CC1E3E117EB3ULL, "vxht_0_hightown_a2_bf2_0_0.vlm"},
	{0x0700ED03054A0668ULL, "fex_ruin_arch_lrg_e.msh"},
	{0x07010B03054A3892ULL, "fex_ruin_arch_lrg_e.mmh"},
	{0x07012274FEC56507ULL, "vxlt_0_lowtown_docks_9fh_0_0.vlm"},
	{0x0701228A1C5E08C0ULL, "6110011_m.fxe"},
	{0x0701A62D3957506FULL, "vxlt_0_lowtown_a2_n_c_e1a_0_0.vlm"},
	{0x0701CD25773DBAF7ULL, "vxht_store_bt_0_0.vlm"},
	{0x07023AD45F4C4F84ULL, "vxht_keep_5ga_0_0.vlm"},
	{0x0702C6E8109CFA1EULL, "art\\vfx\\combat\\vfx_c_rw_s2_attack2_c.impa.ani"},
	{0x07030DA0737CC0ABULL, "vxca_cavern1_haunted_gjz_0_0.vlm"},
	{0x070320CA36AD9E21ULL, "art\\characters\\creatures\\shriek\\animation\\combat\\devestationcast.ani"},
	{0x07033250D1084914ULL, "vxw_bonepit_22j_0_0.vlm"},
	{0x0703E46430000C0EULL, "vxlt_warehouse_w8_0_0.vlm"},
	{0x0703E87BC027F1CAULL, "llac_dr_primevalentrance_1726_13.dds"},
	{0x070456421CF4DDD8ULL, "vxlt_0_lowtown_docks_n_coa_0_0.vlm"},
	{0x0704AB407DB6ADB3ULL, "6082096_m.fxe"},
	{0x0704C8313ACC6391ULL, "6156452_m.fxe"},
	{0x0704F692084A1E50ULL, "vxw_encounter2_camp_gq_0_0.vlm"},
	{0x070512FB8C6E22E8ULL, "lt_0_lowtown_a2_n_c_52_0_ll.mmh"},
	{0x070585BCBE860B53ULL, "6065248_m.fxe"},
	{0x07058782DE3B4B47ULL, "fen110_fenris6100249.cl"},
	{0x0705DD921B01DB26ULL, "w_encounter2_61_0.rml"},
	{0x0705EA518304AB2DULL, "6046887_m.fxe"},
	{0x07064453E5561D0FULL, "hlac_w_bonepitdestroyed_a3_458754_0x.dds"},
	{0x070662AAC490513EULL, "gen00fl_sebastian6174926.cl"},
	{0x0707C4077C6ADCD5ULL, "one100ip_st_solivitus.utp"},
	{0x070811544FEDB918ULL, "glo_vfx_traps.wbd"},
	{0x070819493BD99748ULL, "vxht_mansion2_1a7_0_0.vlm"},
	{0x0708274561BCE89FULL, "llac_lt_foundry_n_320_11.dds"},
	{0x070829F985F6158BULL, "12089447.ogg"},
	{0x070839C55B870D97ULL, "hlac_lt_0_lowtown_a2_n_c_131072_4y.dds"},
	{0x07087F225C0FA46BULL, "vxw_bonepitdestroyed_a3_232_0_0.vlm"},
	{0x0708CCB85E14FA7FULL, "vxht_playershouse_n_1su_0_1.vlm"},
	{0x07090A02C89F3E5AULL, "369209083.ogg"},
	{0x0709FF07E399BDD8ULL, "153220355.ogg"},
	{0x070A2BCA36B3CFC3ULL, "art\\characters\\creatures\\shriek\\animation\\combat\\devestationcast.gad"},
	{0x070A3D37E389FC52ULL, "hlac_ca_cavern1_haunted_24424_0z.dds"},
	{0x070A47A0D084FA53ULL, "6086926_m.fxe"},
	{0x070ABCCF825FB589ULL, "20331479.ogg"},
	{0x070B28991FA2AFCFULL, "vxca_caverns2_6z9_0_0.vlm"},
	{0x070B5289E54BAEB0ULL, "art\\vfx\\combat\\vfx_m_cre_haste_c.anb"},
	{0x070BBBA933D9E625ULL, "82473490.ogg"},
	{0x070BC0D5725D214CULL, "art\\characters\\playercharacter\\dwarfmale\\textures\\dm_glv_hvyn_0d.dds"},
	{0x070BD19804C8B53CULL, "1002504282.ogg"},
	{0x070C0192A9835133ULL, "vxca_caverns2_8zc_0_0.vlm"},
	{0x070C55FCA9BBA910ULL, "662837619.ogg"},
	{0x070CD6921B07EE6AULL, "w_encounter2_61_0.phy"},
	{0x070CF8B3B853A73CULL, "ave310_donnic6134721.cl"},
	{0x070D02706C7B03E1ULL, "vxht_mansion2_1d1_0_0.vlm"},
	{0x070D15EC4305B62DULL, "art\\vfx\\combat\\vfx_r_arc_burst_smo_aoe_p.anb"},
	{0x070DEC928ACB2E2DULL, "6169366_m.fxe"},
	{0x070E769067AB08FDULL, "214762027.ogg"},
	{0x070EA2371155DA07ULL, "89334860.ogg"},
	{0x070EDF8AAA9DBA2BULL, "vxlt_darktown_6id_0_0.vlm"},
	{0x070F0EF06E6B2A2EULL, "6162052_m.fxe"},
	{0x070F1F1BCF78A52FULL, "vxca_caverns2_805_0_0.vlm"},
	{0x070F21A5991B9A39ULL, "vxw_bonepitmining_a2_do_0_0.vlm"},
	{0x070F29B154B76298ULL, "215561819.ogg"},
	{0x070F44D8E3DE8259ULL, "follower_banter6171070.cl"},
	{0x070FC873F479549CULL, "vxht_chantry_30v_0_0.vlm"},
	{0x070FE259384A9A94ULL, "6149893_m.fxe"},
	{0x0710ACB151426804ULL, "vxht_keep_5jf_0_0.vlm"},
	{0x0710D8D38DC13258ULL, "lt_0_lowtown_docks_a3_n_3_0_ll.mmh"},
	{0x0710DEE810A960EFULL, "art\\vfx\\combat\\vfx_c_rw_s2_attack2_c.impa.evt"},
	{0x071108CA36B9B2A8ULL, "art\\characters\\creatures\\shriek\\animation\\combat\\devestationcast.evt"},
	{0x071187B43D71E3F3ULL, "vxht_mansion_n_6p7_0_0.vlm"},
	{0x071219E05286B19AULL, "w_sundermount_160_0.msh"},
	{0x071227E05286C9D3ULL, "w_sundermount_160_0.mao"},
	{0x07122BE05286D0E0ULL, "w_sundermount_160_0.mmh"},
	{0x07123CB7A05C3ED9ULL, "vxlt_undercityserialkiller_1qw_0_0.vlm"},
	{0x0712564121F6248EULL, "vxlt_warehouse_x9_0_0.vlm"},
	{0x0712A4D00B51A0FDULL, "vxdr_primevaltomb_91e_0_2.vlm"},
	{0x0713DC34F10F5EA3ULL, "477487556.ogg"},
	{0x0714631E84AD6CFBULL, "fx_c_woodgrain.mao"},
	{0x0714661D91B5F824ULL, "vxlt_warehouse_n_v8_0_0.vlm"},
	{0x07149357D7A9777BULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\ambient\\compressed\\sitting_table_clasped_1_exit.evt"},
	{0x0715157C2AF67768ULL, "vxca_caverns1_i1g_0_0.vlm"},
	{0x071566720D9496ECULL, "tow_smoke.mao"},
	{0x0715BB1AB8B1EA95ULL, "w_woundedcoast_215_0.msh"},
	{0x0715BD1AB8B1EE0BULL, "w_woundedcoast_215_0.mmh"},
	{0x0715C91AB8B202A8ULL, "w_woundedcoast_215_0.mao"},
	{0x0715D61FA8EE5710ULL, "mag210_arianni6024568.cl"},
	{0x07161A6140CE569CULL, "art\\characters\\creatures\\hurlock\\animation\\crossbow\\hurlock_cb_die_loop.ani"},
	{0x07164022155690BAULL, "vxw_bonepitmining_a2_8t_0_0.vlm"},
	{0x0716708AD7F8F1A7ULL, "hlac_lt_0_lowtown_docks_-131074_4z.dds"},
	{0x07168B262DCFAFC8ULL, "vxht_mansion2_1d4_0_0.vlm"},
	{0x0716DFE810ADCD98ULL, "art\\vfx\\combat\\vfx_c_rw_s2_attack2_c.impa.gad"},
	{0x0716E7D6B93D5C63ULL, "follower_banter6172716.cl"},
	{0x0717E6EB23876168ULL, "qun111_ketojan6019046.cl"},
	{0x07186727274D67C8ULL, "145548216.ogg"},
	{0x0718A4E7A8FCFB78ULL, "440931876.ogg"},
	{0x0718C8B8E9506B3DULL, "6051641_m.fxe"},
	{0x071926CA8441D50AULL, "6062806_m.fxe"},
	{0x07192C007C1BA687ULL, "vxlt_0_lowtown_n_ci5_0_0.vlm"},
	{0x071931D33E7932E2ULL, "6188622_f.fxe"},
	{0x07194BE00DCF71E8ULL, "6180883_m.fxe"},
	{0x071976EAA2E3547FULL, "852584759.ogg"},
	{0x0719C7DAEDA9CEEAULL, "vxlt_shop_j0_0_0.vlm"},
	{0x0719ED625EBB63DDULL, "al_varric_s01.gda"},
	{0x071B5A8C77A9C210ULL, "vxw_encounter3_gb_0_0.vlm"},
	{0x071B8C57D7AF89BCULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\ambient\\compressed\\sitting_table_clasped_1_exit.gad"},
	{0x071B9E3B25436CA2ULL, "l_ser211ar_foundry.bnk"},
	{0x071BB6835ECDFC59ULL, "follower_barks6173417.cl"},
	{0x071BC34CF5A7D29EULL, "6135896_m.fxe"},
	{0x071C08F645E8D90DULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\3ww_alcovetop.phy"},
	{0x071C5B9EA59A5CB1ULL, "zz_mrl_debug6184194.cl"},
	{0x071CCA4F90251023ULL, "6110786_m.fxe"},
	{0x071D17973300DD21ULL, "vxlt_0_lowtown_n_e0n_0_0.vlm"},
	{0x071D23801873FE40ULL, "mage_firemastery.dds"},
	{0x071D276140D48BDEULL, "art\\characters\\creatures\\hurlock\\animation\\crossbow\\hurlock_cb_die_loop.gad"},
	{0x071D58431DE2279CULL, "648819365.ogg"},
	{0x071D80CD606142AEULL, "6167352_m.fxe"},
	{0x071D93828B11B2B9ULL, "vxw_bonepitmining_a2_cn_0_0.vlm"},
	{0x071DA10A57536A6DULL, "6083053_m.fxe"},
	{0x071DAFD821FDAFB3ULL, "vxht_0_hightown_a2_bl4_0_0.vlm"},
	{0x071DD19164F5499CULL, "56354090.ogg"},
	{0x071DDC79ED2E4091ULL, "6140983_m.fxe"},
	{0x071E54362A40B314ULL, "6142993_m.fxe"},
	{0x071EC0D8971AC8F7ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\vista_channelwater.mmh"},
	{0x071EC4D8971ACF04ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\vista_channelwater.mao"},
	{0x071ED6D8971AEE69ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\vista_channelwater.msh"},
	{0x071EDD5F8653B5D0ULL, "19347031.ogg"},
	{0x071FABA001E97F71ULL, "vxlt_0_lowtown_n_ey1_0_0.vlm"},
	{0x071FDD1FC11A3FBAULL, "gen00fl_fenris6108085.cl"},
	{0x071FF586EB099D7EULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\drunk_drink.evt"},
	{0x07205BB0290B7DE1ULL, "qun120_arishok6051466.cl"},
	{0x0720CCDFFD9AF011ULL, "vxlt_0_lowtown_a2_ehu_0_0.vlm"},
	{0x07213068B53A1AADULL, "316495243.ogg"},
	{0x07213D9347A82173ULL, "411377213.ogg"},
	{0x07228B57D7B5A6BAULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\ambient\\compressed\\sitting_table_clasped_1_exit.ani"},
	{0x0722AFEFBB47D14AULL, "lt_0_lowtown_docks_n_237_0.mao"},
	{0x0722B3EFBB47D799ULL, "lt_0_lowtown_docks_n_237_0.mmh"},
	{0x0722BDEFBB47E887ULL, "lt_0_lowtown_docks_n_237_0.msh"},
	{0x0722EEC1585491C9ULL, "zz_soundset.ncs"},
	{0x0722F0F749EECB2DULL, "603291185.ogg"},
	{0x07237A2C8A297303ULL, "6135707_m.fxe"},
	{0x0723D7F28B4A84C8ULL, "vxlt_gallowscourtyard_n_46q_0_0.vlm"},
	{0x0724026140DA6BFDULL, "art\\characters\\creatures\\hurlock\\animation\\crossbow\\hurlock_cb_die_loop.evt"},
	{0x072511DC1EDD89AFULL, "lt_gallowscourtyard_48_0.rml"},
	{0x07258E3DA77FB253ULL, "water_water_p0.psh.fxo.dx11"},
	{0x07259D894CECF87AULL, "775797314.ogg"},
	{0x0725A708218B70E8ULL, "ht_0_hightown_n_41_0.rml"},
	{0x0727087CC27E6D89ULL, "6174966_m.fxe"},
	{0x07270C86EB0FE2A9ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\drunk_drink.gad"},
	{0x07279B0EB1F6B126ULL, "ico_attack.dds"},
	{0x07279B8016613755ULL, "6048385_f.fxe"},
	{0x072805D54EE98515ULL, "zz_stand_neutral_test.cut"},
	{0x07291A7BBF7AE60BULL, "274920584.ogg"},
	{0x072992495B0F77C2ULL, "genip_cin_crown.utp"},
	{0x0729BC379173B035ULL, "667288383.ogg"},
	{0x0729E70645134C43ULL, "722414801.ogg"},
	{0x072A4C01FE078B4AULL, "hlac_ca_caverns1_mines_24424_19x.dds"},
	{0x072A524724158DD3ULL, "pro000_aveline_intro6109378.cl"},
	{0x072A6C6B15035977ULL, "ca_caverns2prc_5802_lc.mmh"},
	{0x072A8F0A8D5B6603ULL, "dae000cr_grand_cleric6148160.cl"},
	{0x072AD04C6EB75F2DULL, "6163166_m.fxe"},
	{0x072B407F3D622AB3ULL, "6109634_m.fxe"},
	{0x072B5C2E66D156D5ULL, "art\\vfx\\combat\\vfx_c_2nd_deb_cast_aoe_c.mmh"},
	{0x072B6B938FCAF706ULL, "one111ip_ggt_crate_2.utp"},
	{0x072BE054989F67DDULL, "6031816_m.fxe"},
	{0x072BFD64D84E723EULL, "686881272.ogg"},
	{0x072C01F20C0E7C9BULL, "hlac_lt_0_lowtown_docks_n_131071_0y.dds"},
	{0x072C055F7D07CB39ULL, "vxw_bonepitmining_a2_nm_0_0.vlm"},
	{0x072C904D5C3C0AEBULL, "vxlt_0_lowtown_f1o_0_0.vlm"},
	{0x072D1F342CE3D733ULL, "324947462.ogg"},
	{0x072D6D9DEBFE128DULL, "6110612_m.fxe"},
	{0x072D8BC7562A6BC5ULL, "825657466.ogg"},
	{0x072DDD86EB15B1F7ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\drunk_drink.ani"},
	{0x072DECA54C9FCB64ULL, "601123951.ogg"},
	{0x072E0453AE903E07ULL, "gen00fl_fenris6113950.cl"},
	{0x072E0AFBFB9127EEULL, "hlac_w_encounter2_camp_327679_0z.dds"},
	{0x072E57177938D44AULL, "877899518.ogg"},
	{0x072F9847159074A6ULL, "6113632_m.fxe"},
	{0x07304D7FBE7FD7CCULL, "ave310_brennan6136872.cl"},
	{0x07309FA9507D8B15ULL, "ss_qunari_arishok.bnk"},
	{0x0730BBFA1DC86458ULL, "dae000ip_crowd_ff6_talk16.utp"},
	{0x0730C0CE472E5F93ULL, "6016952_m.fxe"},
	{0x0730F425D2659045ULL, "hlac_dr_0_deeproads_5849_3x.dds"},
	{0x07317C1DB7E0325CULL, "vxlt_gallowstemplar_f_3nd_0_0.vlm"},
	{0x07318A78C090FD33ULL, "vxlt_gallowstemplar_5pl_0_0.vlm"},
	{0x0731B344079E94DAULL, "vxca_cave_23a_0_1.vlm"},
	{0x0731C01B0B1ECC03ULL, "vxlt_gallowstemplar_f_6qk_0_0.vlm"},
	{0x07320CC02FC65EA5ULL, "art\\levels\\races\\proxy\\tilesets\\setint\\4co_basic\\1wi_basic_1.mao"},
	{0x07335EC1C5061989ULL, "359329134.ogg"},
	{0x0733B011EFAACB42ULL, "6125616_f.fxe"},
	{0x0733CA5A9B579C9EULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\rogue\\bow\\archer_range_attack_v5.ani"},
	{0x0733D6BF4E7B6296ULL, "vis121_saemus6101339.cl"},
	{0x07340F70834F8EEDULL, "llac_lt_0_lowtown_docks_a3_n_-3_1.dds"},
	{0x07346DE191B8EAE2ULL, "812832990.ogg"},
	{0x07347E71616C22E3ULL, "lgt120cr_bhelen_supp_war.utc"},
	{0x0734DB6E7701A1F0ULL, "hf_amb_keep_servant.mor"},
	{0x0735BF83A5B71610ULL, "416701605.ogg"},
	{0x0735E1E59FC8580AULL, "605264988.ogg"},
	{0x0735FE24D663BBACULL, "175455843.ogg"},
	{0x0736570752E7F990ULL, "vxht_mansion2_32h_0_0.vlm"},
	{0x0736C2FB43FBD70DULL, "vxca_caverns1_ckn_0_0.vlm"},
	{0x0736DE92D449DE80ULL, "vxlt_0_lowtown_n_d46_0_0.vlm"},
	{0x07375013BEACE496ULL, "231099.ogg"},
	{0x07376BBCAF6C2197ULL, "36056791.ogg"},
	{0x073785709B903232ULL, "187668956.ogg"},
	{0x0737E2CD129A4835ULL, "genip_mug.utp"},
	{0x0738592E66DBDC62ULL, "art\\vfx\\combat\\vfx_c_2nd_deb_cast_aoe_c.anb"},
	{0x0738B36640CC0440ULL, "follower_banter6162107.cl"},
	{0x073A5364A7438E91ULL, "841588050.ogg"},
	{0x073A5AE299C667F1ULL, "402073764.ogg"},
	{0x073A605C03EBF375ULL, "vxca_caverns1_j4n_0_0.vlm"},
	{0x073A6665F4FDD415ULL, "6177909_m.fxe"},
	{0x073A84DD4A4A0064ULL, "vxht_mansion2b_2vu_0_0.vlm"},
	{0x073AA76A268935B4ULL, "398275833.ogg"},
	{0x073AB8A0C685603EULL, "663420472.ogg"},
	{0x073AD4A09C850278ULL, "art\\levels\\races\\proxy\\tilesets\\setint\\benstestset\\9pp_setprops.phy"},
	{0x073AE1D060891B59ULL, "588140410.ogg"},
	{0x073B0697747D6E7AULL, "240747354.ogg"},
	{0x073BA2F7A4999A64ULL, "6071435_m.fxe"},
	{0x073C22AAA233B981ULL, "follower_banter6169840.cl"},
	{0x073C48EE87040CA9ULL, "sky_sky2_p0.psh.fxo.dx11.assembly"},
	{0x073C7B5247EB958AULL, "vxca_caverns1_mines_j5c_0_0.vlm"},
	{0x073D1A47FBC67368ULL, "nat380_sibling6207554.cl"},
	{0x073D1AD01C8890E0ULL, "gen_im_trs_dal_hvy_glv_r1.uti"},
	{0x073D97E8A6842C4AULL, "art\\vfx\\combat\\vfx_m_wpn_ele_crust_c.impa.gad"},
	{0x073DA6784380E4BBULL, "461785518.ogg"},
	{0x073DCB9FB5FF20DCULL, "vxlt_gallowstemplar_n_39i_0_0.vlm"},
	{0x073DFF46A68D7C2CULL, "ter_cracked01_s.dds"},
	{0x073E22D96EA58B75ULL, "627804293.ogg"},
	{0x073E4AABCDCCB36AULL, "299208642.ogg"},
	{0x073F0C5ED4A22D18ULL, "vxw_blightlands_n5_0_0.vlm"},
	{0x073F120BD6F6A49FULL, "318458356.ogg"},
	{0x073F204D537CDE99ULL, "313824710.ogg"},
	{0x073F5ABB9A98F322ULL, "zz_qun_debug6049745.cl"},
	{0x073F62D1601DE5BBULL, "vxdr_primevalentrance_392_0_1.vlm"},
	{0x073FC74718E1C0DFULL, "art\\characters\\creatures\\gallowsgateguard\\animation\\spear\\spearwalkfront.gad"},
	{0x07402E2CF2B7CFCBULL, "312233193.ogg"},
	{0x07404C8171D3B6E9ULL, "6096532_m.fxe"},
	{0x074056E647C9EF5DULL, "ave200_donnic6112144.cl"},
	{0x0740840DD9EBC583ULL, "6060540_m.fxe"},
	{0x0741883B256416C0ULL, "l_ser211ar_foundry.wbd"},
	{0x07419ECF1901095DULL, "391320634.ogg"},
	{0x0741A411CFA7141CULL, "art\\vfx\\combat\\vfx_glb_arw_spi_c.dura.ani"},
	{0x0741C37122DC9A2AULL, "vxw_bonepitmining_a2_1xo_0_0.vlm"},
	{0x0741E25A9B64036FULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\rogue\\bow\\archer_range_attack_v5.evt"},
	{0x0742177F366D457FULL, "zz_dae_debug6143033.cl"},
	{0x07422937D211A43FULL, "59713491.ogg"},
	{0x07422D32D055EFCFULL, "fenris_2.cut"},
	{0x07423258597027A5ULL, "132602425.ogg"},
	{0x074269783E6D5B7FULL, "hlac_ht_0_hightown_n_131074_0y.dds"},
	{0x07426BEE1C45BB10ULL, "vxlt_warehouse_sz_0_0.vlm"},
	{0x0742978D8FD3C842ULL, "vxw_encounter3_du_0_0.vlm"},
	{0x0742C70263D871C6ULL, "46807652.ogg"},
	{0x07437095770065C2ULL, "vxca_caverns1_ik3_0_0.vlm"},
	{0x0743AAD9FB76D179ULL, "6136238_m.fxe"},
	{0x0743D7F74E0008FAULL, "hlac_lt_0_lowtown_docks_a3_131068_0x.dds"},
	{0x07443166F782A34CULL, "dae000_pc_home_ambients6208646.cl"},
	{0x07447AE8A68A19E1ULL, "art\\vfx\\combat\\vfx_m_wpn_ele_crust_c.impa.evt"},
	{0x0744D2296AD84AE7ULL, "vxlt_0_lowtown_byo_0_0.vlm"},
	{0x0744E0E011D412DCULL, "460914563.ogg"},
	{0x0744F27BD79A22E4ULL, "1004138122.ogg"},
	{0x07451EB6D2D319D0ULL, "833616524.ogg"},
	{0x07452C851EB3D821ULL, "42800917.ogg"},
	{0x074581BABF46F1CBULL, "419519299.ogg"},
	{0x0745E09BDC5A3716ULL, "hlac_lt_0_lowtown_a3_n_c_65534_0x.dds"},
	{0x0745F963B960FB35ULL, "vxht_0_hightown_a3_8pu_0_0.vlm"},
	{0x074607B275FF61B3ULL, "vxlt_0_lowtown_a2_hg7_0_0.vlm"},
	{0x07462DEC22F30BC3ULL, "gen00fl_varric6076228.cl"},
	{0x07469EA52243C258ULL, "vxlt_0_lowtown_a2_n_gmr_0_0.vlm"},
	{0x0746BC4718E7CCCCULL, "art\\characters\\creatures\\gallowsgateguard\\animation\\spear\\spearwalkfront.evt"},
	{0x07472E13FD00A7B0ULL, "zz_fol_debug6066136.cl"},
	{0x07478EF7C60BB5D3ULL, "hlac_lt_warehouse_1_4y.dds"},
	{0x0747E35A9B687018ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\rogue\\bow\\archer_range_attack_v5.gad"},
	{0x0747E6D2B9AC6375ULL, "mag170cr_thrask.utc"},
	{0x0747E6E4E0D44B57ULL, "vxdr_primevaltomb_5bl_0_0.vlm"},
	{0x074828499DE7269AULL, "6106960_m.fxe"},
	{0x07482925226E82DFULL, "mer320_hubert6083763.cl"},
	{0x074848E77FCD4CF9ULL, "vxht_keep_n_55i_0_0.vlm"},
	{0x0748B111CFAD495EULL, "art\\vfx\\combat\\vfx_glb_arw_spi_c.dura.gad"},
	{0x0748CED930C82BC3ULL, "6013313_m.fxe"},
	{0x0748E2843FB2ABB9ULL, "hlac_lt_gallowscourtyard_-65537_4x.dds"},
	{0x0749432FDE5CDE7CULL, "gen_im_wep_war_two_act1_04.uti"},
	{0x074971B17CF96DDAULL, "vxht_0_hightown_n_c7j_0_0.vlm"},
	{0x07499AC4685464C2ULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\4so_chantryleftb.phy"},
	{0x074A668D420D4CD6ULL, "llac_lt_0_lowtown_docks_a3_n_196607_3.dds"},
	{0x074C3D7CA7F5395CULL, "vxlt_gallowstemplar_n_34n_0_0.vlm"},
	{0x074D0F831EE1A29DULL, "vxw_encounter2_9j_0_0.vlm"},
	{0x074D41FBB844B6D3ULL, "art\\characters\\playercharacter\\humanmale\\animation\\ambient\\dialogue\\talk_wary.evt"},
	{0x074D5AD5499D56D8ULL, "dae000_bartrand6031162.cl"},
	{0x074D666134B2E90EULL, "gen00fl_merrill6151102.cl"},
	{0x074D7E3BC6984598ULL, "vxw_blightlands_s2_0_0.vlm"},
	{0x074DAD47F2754703ULL, "qun120_arishok_guard6054843.cl"},
	{0x074E3B9D20719C98ULL, "670987113.ogg"},
	{0x074E96609316EB6BULL, "vxdr_primevaltomb_902_0_0.vlm"},
	{0x074EF18CDB16536DULL, "vxlt_0_lowtown_docks_a3_n_cym_0_0.vlm"},
	{0x074EF2397FECFBDDULL, "321024058.ogg"},
	{0x074F011C99A09115ULL, "746790619.ogg"},
	{0x074F49EA4EDFE75FULL, "183784624.ogg"},
	{0x074F634D06690554ULL, "6083086_f.fxe"},
	{0x074F6A71F6037D42ULL, "vxlt_0_lowtown_a2_e8j_0_0.vlm"},
	{0x074F83603D56586AULL, "11256007.ogg"},
	{0x074F8C11CFB3297DULL, "art\\vfx\\combat\\vfx_glb_arw_spi_c.dura.evt"},
	{0x074FB23510EB18ECULL, "hlac_lt_warehouse_65537_3z.dds"},
	{0x075018390E987FBCULL, "vrc221st_varric_postmans.stg"},
	{0x0750354E14D2B2AAULL, "vxw_bonepitmining_a2_1uh_0_0.vlm"},
	{0x07506266BFF63304ULL, "vxlt_blackemporium_rt_0_1.vlm"},
	{0x0750A2F2A51922B0ULL, "gen00fl_anders6119524.cl"},
	{0x0750E8649B4B63E1ULL, "vxlt_gallowscourtyard_4uz_0_0.vlm"},
	{0x0751096A81C9E0C2ULL, "vxw_encounter3_av_0_0.vlm"},
	{0x07513C88BEF452CDULL, "vxlt_undercityencounter_10t_0_0.vlm"},
	{0x075182B2EB45D5D4ULL, "vxlt_0_lowtown_haf_0_0.vlm"},
	{0x0751B7127AF385C3ULL, "13880782.ogg"},
	{0x0751FE8B92B05603ULL, "6117664_m.fxe"},
	{0x075272E8A6964930ULL, "art\\vfx\\combat\\vfx_m_wpn_ele_crust_c.impa.ani"},
	{0x07527756B8A351C1ULL, "dre000_sibling_has_blight6037032.cl"},
	{0x0752CD6882200E3EULL, "vxw_woundedcoast_1bx_0_0.vlm"},
	{0x0752D0075CD96110ULL, "and230tr_act2_opening.utt"},
	{0x0753A44718F22E25ULL, "art\\characters\\creatures\\gallowsgateguard\\animation\\spear\\spearwalkfront.ani"},
	{0x0753D7DB62427FCEULL, "hlac_w_bonepitdestroyed_a3_393218_0z.dds"},
	{0x0753DA29CD3A13FCULL, "6129727_m.fxe"},
	{0x0753E62D5F5A3381ULL, "vxlt_gallowstemplar_f_3n5_0_0.vlm"},
	{0x07543AFBB84AC914ULL, "art\\characters\\playercharacter\\humanmale\\animation\\ambient\\dialogue\\talk_wary.gad"},
	{0x075510821439DAD8ULL, "vxlt_0_lowtown_a2_n_gjs_0_0.vlm"},
	{0x075594DFEA45A7C0ULL, "hlac_lt_undercitydungeon_-1_3x.dds"},
	{0x0755BBAF173188D3ULL, "201408721.ogg"},
	{0x07562082EE59CF64ULL, "mcr310_argument6182285.cl"},
	{0x075656A689B45F31ULL, "mcr310_argument6075135.cl"},
	{0x07576BBBEC719A41ULL, "vxht_0_hightown_c3k_0_1.vlm"},
	{0x075835C9E70522C4ULL, "383844884.ogg"},
	{0x07583FD270F1EB5DULL, "vxlt_gallowscourtyard_7c3_0_0.vlm"},
	{0x0758B3ED606640BAULL, "zz_pro_cutscenes6005832.cl"},
	{0x0758D68B864D698BULL, "gen_im_trs_gem_ruby_01.uti"},
	{0x07593975BDD23101ULL, "hurlock_lsword.utc"},
	{0x075955D58EC2E6C1ULL, "vxlt_undercityserialkiller_1lh_0_0.vlm"},
	{0x07596989B91B4CF4ULL, "vxca_caverns2prc_7oa_0_0.vlm"},
	{0x075995D824C7388FULL, "dae200_cullen6140309.cl"},
	{0x0759A18BD1F8CB44ULL, "481034042.ogg"},
	{0x0759DE80440353E0ULL, "vxlt_hangedman_n_27o_0_0.vlm"},
	{0x0759E7698C1953A0ULL, "gen00fl_dog6165706.cl"},
	{0x075A0A8E23546994ULL, "6135656_m.fxe"},
	{0x075A29AEE1F4855FULL, "hlac_ht_mansion2b_818_56x.dds"},
	{0x075A3C08131A1100ULL, "art\\characters\\weapons\\staff\\textures\\w_stf_007a_0s.dds"},
	{0x075A757C0C619152ULL, "6105644_m.fxe"},
	{0x075B1A7C5AE5641FULL, "259813274.ogg"},
	{0x075B491D3A149DC9ULL, "vxht_chantry_n_216_0_0.vlm"},
	{0x075B49FBB85101F2ULL, "art\\characters\\playercharacter\\humanmale\\animation\\ambient\\dialogue\\talk_wary.ani"},
	{0x075BC716F77E5036ULL, "art\\characters\\weapons\\dagger\\textures\\w_dgr_103a_n.dds"},
	{0x075BF018B88E5E18ULL, "vxw_blightlands_t3_0_0.vlm"},
	{0x075CC2705714CB4CULL, "mrl250_pol6101318.cl"},
	{0x075CC5077CB31E24ULL, "one100ip_st_solivitus.ncs"},
	{0x075CC5A950A3069FULL, "ss_qunari_arishok.wbd"},
	{0x075CD33A1A2AC12BULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\shield and weapon\\sw_reaver.evt"},
	{0x075CD5F2AC0DAF92ULL, "316391160.ogg"},
	{0x075CD6966D10BD37ULL, "mag291_demon_marethari6027865.cl"},
	{0x075CF708A69B8463ULL, "zz_dae_debug6029311.cl"},
	{0x075D7CF7852AF7D0ULL, "art\\vfx\\combat\\vfx_r_dlt_duelling_act_c.mmh"},
	{0x075EA72B06C8CB2AULL, "vxw_bonepitmining_a2_1ri_0_0.vlm"},
	{0x075EAC3DF50A19B9ULL, "147438291.ogg"},
	{0x075ECFF029CDA1C5ULL, "art\\levels\\races\\natural\\kirkcoastal\\foliage\\_textures\\coastwood_sa.dds"},
	{0x075F2BAFD4184008ULL, "6135670_m.fxe"},
	{0x075F4FA80031EC10ULL, "vxlt_warehouse_yx_0_0.vlm"},
	{0x076004AE731DFDCEULL, "hlac_lt_darktown_-1_24x.dds"},
	{0x076035AF5D792201ULL, "vxlt_gallowstemplar_n_398_0_0.vlm"},
	{0x076064C4D1EBAE5CULL, "mag320pt_otl_emile.ncs"},
	{0x0760706884A66E83ULL, "6112564_m.fxe"},
	{0x076070B3A8B749AEULL, "128912203.ogg"},
	{0x0760A323567FB6A2ULL, "ser270_emeric6111400.cl"},
	{0x07613F45741626BEULL, "vxw_woundedcoast_1my_0_0.vlm"},
	{0x0761968875139812ULL, "vxlt_0_lowtown_docks_n_b64_0_1.vlm"},
	{0x0761A0F5620411ADULL, "380873923.ogg"},
	{0x0761C8ACC0D88A3AULL, "hlac_ht_playershouse_n_855_12x.dds"},
	{0x0761DD504ADFD2BCULL, "23656240.ogg"},
	{0x0762832C902880AEULL, "vxca_cavern1_haunted_j2c_0_1.vlm"},
	{0x07629E240CDF7D6DULL, "llac_lt_darktown_-1_32.dds"},
	{0x0762E24ADE526F0FULL, "art\\levels\\races\\ferelden\\props\\dead\\prp_04skeleton.mao"},
	{0x0762EE4ADE5283ACULL, "art\\levels\\races\\ferelden\\props\\dead\\prp_04skeleton.mmh"},
	{0x0762F44ADE528D9EULL, "art\\levels\\races\\ferelden\\props\\dead\\prp_04skeleton.msh"},
	{0x076316149AC59B3BULL, "vxlt_0_lowtown_a2_n_ep0_0_0.vlm"},
	{0x07636D171E3C8D85ULL, "vxdr_primevaltomb_8q0_0_2.vlm"},
	{0x07637E1967DFAF89ULL, "6025112_m.fxe"},
	{0x0763A0EA6111255FULL, "430702140.ogg"},
	{0x0763EC3A1A310AECULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\shield and weapon\\sw_reaver.gad"},
	{0x076410063884B470ULL, "vxht_brothel_n_539_0_0.vlm"},
	{0x076443E37403CD34ULL, "gen00pt_skills.ncs"},
	{0x0765B19AF6CC08E5ULL, "darktown_male_sickly_sta6175324.cl"},
	{0x0765BB77A5B50F9DULL, "hlac_lt_0_lowtown_a2_n_-131071_1x.dds"},
	{0x0765DA74202C49E4ULL, "vxht_mansion2b_2ix_0_0.vlm"},
	{0x0765DEF5D0425D63ULL, "vxlt_0_lowtown_docks_n_c4o_0_0.vlm"},
	{0x07667B009A180031ULL, "vxht_0_hightown_n_a4n_0_0.vlm"},
	{0x07667CE2A6A459B8ULL, "6205706_m.fxe"},
	{0x076697A1774C7142ULL, "vxht_0_hightown_n_bzl_0_0.vlm"},
	{0x0766B0DB4132D3A8ULL, "vxw_encounter2_camp_93_0_0.vlm"},
	{0x0766B1AF62E803DDULL, "vxlt_gallowscourtyard_7d2_0_0.vlm"},
	{0x0766BDD632DE1EEDULL, "hlac_ca_caverns1_mines_24426_9z.dds"},
	{0x0767687DC6FBD369ULL, "vxw_encounter3_bh_0_0.vlm"},
	{0x0767C7B280B8FF41ULL, "vxlt_undercityserialkiller_1ao_0_0.vlm"},
	{0x0767E26C4BA1783BULL, "6144350_m.fxe"},
	{0x0767F60ABA6BBEF4ULL, "art\\vfx\\combat\\vfx_r_gn2_accu_loop_c.anb"},
	{0x0768105DF897A8CBULL, "hlac_w_sundermount_0_4z.dds"},
	{0x0768505D35F96C60ULL, "vxlt_hangedman_n_22l_0_0.vlm"},
	{0x076890C4AF5AAF7FULL, "blank_exterior.arl"},
	{0x07689988BAA3287EULL, "ser270_emeric6201574.cl"},
	{0x076912F83637B60CULL, "dae000_gamlen6124239.cl"},
	{0x0769761248D445D6ULL, "hlac_lt_0_lowtown_a3_n_c_65533_1z.dds"},
	{0x0769A9E5600DE8D3ULL, "640346969.ogg"},
	{0x076A56E25A51A36FULL, "vxht_chantry_n_2i8_0_0.vlm"},
	{0x076A6BBA155AF4BDULL, "vxht_0_hightown_bfp_0_0.vlm"},
	{0x076A750FC1B6C13DULL, "vxca_caverns2_8it_0_0.vlm"},
	{0x076A7F95EAD86A32ULL, "358427009.ogg"},
	{0x076A8210A696C5E7ULL, "ser270_templar6127404.cl"},
	{0x076A9FF78535BDEFULL, "art\\vfx\\combat\\vfx_r_dlt_duelling_act_c.anb"},
	{0x076AEB3A1A3727EAULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\shield and weapon\\sw_reaver.ani"},
	{0x076B6EBC15AD346FULL, "306628972.ogg"},
	{0x076C6994BF611C91ULL, "6037591_m.fxe"},
	{0x076CA7E5595EE323ULL, "hlac_lt_0_lowtown_a2_n_c_0_9z.dds"},
	{0x076D534DF2883568ULL, "6074493_f.fxe"},
	{0x076D955CE05E8FEAULL, "vxht_0_hightown_a3_bxl_0_0.vlm"},
	{0x076E0F8E3EA51B51ULL, "6059007_m.fxe"},
	{0x076E75C468734276ULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\4so_chantryleftb.mao"},
	{0x076E81C468735615ULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\4so_chantryleftb.mmh"},
	{0x076E83C46873598BULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\4so_chantryleftb.msh"},
	{0x076F18A3BAB3389DULL, "297301126.ogg"},
	{0x076F6A61DADA6896ULL, "6054551_m.fxe"},
	{0x076F79D0C3D4A8E1ULL, "dae000_bartrand6207005.cl"},
	{0x076FB122660C3F3EULL, "vxw_woundedcoast_1hz_0_0.vlm"},
	{0x0770B75F41E22962ULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\0pp_galowint.msh"},
	{0x0770B95F41E22C18ULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\0pp_galowint.mmh"},
	{0x0770C55F41E240BBULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\0pp_galowint.mao"},
	{0x07715BD512DD1E98ULL, "fen110_fenris6117929.cl"},
	{0x07716992F1471778ULL, "vxlt_0_lowtown_a2_n_c_hjn_0_0.vlm"},
	{0x077187F18CBBB3BBULL, "vxlt_0_lowtown_a2_n_eu3_0_0.vlm"},
	{0x07719A4A763F9E0DULL, "vxw_encounter2_cl_0_0.vlm"},
	{0x0771A652E6584D32ULL, "vxht_mansion2_1v4_0_0.vlm"},
	{0x0771FD0272B03829ULL, "dae000cr_fereldan_f.utc"},
	{0x0772841924A1940CULL, "594841100.ogg"},
	{0x077364C744B37950ULL, "follower_banter6151350.cl"},
	{0x077368423E86E6C9ULL, "558430963.ogg"},
	{0x0773C164EE2BC005ULL, "hlac_lt_0_lowtown_a2_131073_1z.dds"},
	{0x0773C794B8D3E4E5ULL, "332714298.ogg"},
	{0x0773CB87D6A31A3AULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\1ww_oct2window.phy"},
	{0x0773F31BB043F269ULL, "vxht_mansion2_30w_0_0.vlm"},
	{0x07744C5112226264ULL, "vxht_mansion2b_2by_0_0.vlm"},
	{0x077457EE70ADADDAULL, "6205991_m.fxe"},
	{0x07747E80C31B4633ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\prp_chainhead.phy"},
	{0x077484A9BB96FD86ULL, "gen_im_trs_crc_lgt_boo_r1.uti"},
	{0x0774A3A13C9E6BDEULL, "lgt311_shw_mer_mettin6156468.cl"},
	{0x0774C57C8EE7E33FULL, "mag110_vincento6017674.cl"},
	{0x0775097E694289C2ULL, "vxht_0_hightown_n_bem_0_0.vlm"},
	{0x077522B83328EC28ULL, "vxw_encounter2_camp_40_0_0.vlm"},
	{0x0775238C54DE1C5DULL, "vxlt_gallowscourtyard_7i1_0_0.vlm"},
	{0x077554C0A6F4CC60ULL, "vxht_bartrand_n_4de_0_0.vlm"},
	{0x0775D50ABA77C3B7ULL, "art\\vfx\\combat\\vfx_r_gn2_accu_loop_c.mmh"},
	{0x0775E2073C663A22ULL, "661098789.ogg"},
	{0x0776398F72AF17C1ULL, "vxlt_undercityserialkiller_1fn_0_0.vlm"},
	{0x07765CB29522CC71ULL, "vxdr_0_deeproads_cun_0_1.vlm"},
	{0x077669D78B9E76C3ULL, "6123233_f.fxe"},
	{0x0776746D3C7A7E06ULL, "vxlt_0_lowtown_a2_n_c_gzc_0_0.vlm"},
	{0x07770C167CCE2D87ULL, "vxlt_0_lowtown_a2_hfd_0_0.vlm"},
	{0x07775DD6F4EDDB8FULL, "6033477_f.fxe"},
	{0x0777A68350397220ULL, "6071295_m.fxe"},
	{0x0777AA0EA0802536ULL, "art\\characters\\creatures\\desiredemon\\animation\\cwkfr.ani"},
	{0x07785507FC891BFAULL, "6128397_m.fxe"},
	{0x07785F8672CD7897ULL, "vxht_mansion2b_25c_0_0.vlm"},
	{0x07788F833403BB9BULL, "6160943_m.fxe"},
	{0x077896662D0B691FULL, "996266580.ogg"},
	{0x0778AFAB9B4F0F35ULL, "777001793.ogg"},
	{0x0778BF075CFA13CDULL, "and230tr_act2_opening.ncs"},
	{0x07795C809A9974BEULL, "ico_ring_3.dds"},
	{0x07795D0AACAE9617ULL, "6000829_m.fxe"},
	{0x07797A293E774F07ULL, "6143210_m.fxe"},
	{0x0779DA77EF771225ULL, "dae000_madam_lusine6151628.cl"},
	{0x077A1FC2696342DAULL, "w_sundermount_55_0.anb"},
	{0x077A3A07C3E99A42ULL, "llac_lt_gallowstemplar_f_2384_0.dds"},
	{0x077A5C2114391F06ULL, "cage_05.mao"},
	{0x077A95FBCD04E1A6ULL, "vxlt_gallowstemplar_n_5qa_0_0.vlm"},
	{0x077AC05F41EA8CA3ULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\0pp_galowint.phy"},
	{0x077ADB71B1573511ULL, "6030491_m.fxe"},
	{0x077AEF91C2992A15ULL, "art\\levels\\races\\ferelden\\props\\mining\\prp_burn1table.msh"},
	{0x077AF191C2992D8BULL, "art\\levels\\races\\ferelden\\props\\mining\\prp_burn1table.mmh"},
	{0x077AFD91C2994228ULL, "art\\levels\\races\\ferelden\\props\\mining\\prp_burn1table.mao"},
	{0x077B36D8062DE1BBULL, "6059988_m.fxe"},
	{0x077BC52AE47E4DE8ULL, "6079793_f.fxe"},
	{0x077C544DFB296A47ULL, "art\\characters\\creatures\\desiredemon\\animation\\inst.gad"},
	{0x077D3CA7931A5E8DULL, "vxlt_0_lowtown_a3_n_c_hf6_0_0.vlm"},
	{0x077D42F253D1B630ULL, "6208213_m.fxe"},
	{0x077D8CA662F5811BULL, "706450533.ogg"},
	{0x077DC76747AADFA3ULL, "and330st_base_justice_post1.stg"},
	{0x077DEE6B71BC4579ULL, "vxht_0_hightown_a2_b82_0_0.vlm"},
	{0x077DF90DECF1E67FULL, "6042287_m.fxe"},
	{0x077E0FAA6D2E79AAULL, "hf_mother_h3s2.mor"},
	{0x077E1CFF819DFE61ULL, "lav_crust02d.dds"},
	{0x077E7EF7BFDE3165ULL, "307693223.ogg"},
	{0x077F39B987FE9B26ULL, "6115880_f.fxe"},
	{0x077F86F03A409239ULL, "715014997.ogg"},
	{0x077F9A76DF5EC51CULL, "llac_lt_gallowscourtyard_n_-2_0.dds"},
	{0x07800C276835B68DULL, "vxw_encounter2_nk_0_0.vlm"},
	{0x0780182FD84E65B2ULL, "vxht_mansion2_1y5_0_0.vlm"},
	{0x0780784ABEB7E3FBULL, "vxdr_0_deeproads_f0o_0_0.vlm"},
	{0x078093C3EF4814FEULL, "6095743_m.fxe"},
	{0x0781163B2E7857C0ULL, "one280_merin6120920.cl"},
	{0x0781C15427CB8319ULL, "dae000ip_crowd_mf6_talk16.utp"},
	{0x0781C187185F5F67ULL, "1033526843.ogg"},
	{0x0783494DFB2F7654ULL, "art\\characters\\creatures\\desiredemon\\animation\\inst.evt"},
	{0x0784C65991151C1EULL, "vxht_0_hightown_n_c47_0_0.vlm"},
	{0x0784E366F5019BBFULL, "zz_christest.cut"},
	{0x07851BCA95AC4BF4ULL, "240241348.ogg"},
	{0x07854637103FF08AULL, "922015221.ogg"},
	{0x07857DF36EC44607ULL, "vxlt_0_lowtown_a2_hcg_0_0.vlm"},
	{0x07858BA2D07A6D1DULL, "1008466646.ogg"},
	{0x0785B20EA08C7027ULL, "art\\characters\\creatures\\desiredemon\\animation\\cwkfr.evt"},
	{0x0785F3FF1209A95DULL, "llac_w_encounter2_327685_0.dds"},
	{0x0786C15F7A38865EULL, "472225132.ogg"},
	{0x07875FA50D322E44ULL, "t1_skn_004.tnt"},
	{0x0787635987BC2DBCULL, "llac_lt_0_lowtown_docks_n_-65540_0.dds"},
	{0x07880E761638CDFFULL, "6082585_bespecific.cut"},
	{0x0788D787F6352B0BULL, "dae000_arishok6078457.cl"},
	{0x07899495FFEFD8CCULL, "87459977.ogg"},
	{0x0789ABFFEEE17AE8ULL, "gen00fl_fenris6105866.cl"},
	{0x0789C344BF8DA072ULL, "6115797_m.fxe"},
	{0x0789F536AD3CEEEAULL, "plt_cod_hst_sundermount.plo"},
	{0x078A07CC34D4AC92ULL, "lowtown_male_alienage1_sta6161632.cl"},
	{0x078A770FB4638FE2ULL, "vxlt_0_lowtown_a2_n_erk_0_0.vlm"},
	{0x078AA866AD0F2EB5ULL, "zz_followers_banter6157678.cl"},
	{0x078BBFEDEFA75402ULL, "hlac_lt_backalley_n_0_12z.dds"},
	{0x078BD30EA0911470ULL, "art\\characters\\creatures\\desiredemon\\animation\\cwkfr.gad"},
	{0x078D4E242DF26129ULL, "6103339_m.fxe"},
	{0x078DAC1C93D3A945ULL, "vxht_keep_4nr_0_0.vlm"},
	{0x078DB352B3B8D6F7ULL, "70402972.ogg"},
	{0x078DC20B4D4AD418ULL, "lgt220pt_shp_kill_qun.ncs"},
	{0x078E59A2580355C6ULL, "vxw_bonepitmining_a2_16q_0_0.vlm"},
	{0x078E7E045A2BCF0DULL, "vxw_encounter2_ij_0_0.vlm"},
	{0x078ED83A99975779ULL, "lyriumghost_magicresist.dds"},
	{0x078F673AE4A5C61EULL, "6115072_m.fxe"},
	{0x078FCB3A2A4F3DE7ULL, "war_critimmunity.dds"},
	{0x0790153F81562F40ULL, "vxht_0_hightown_a2_c3k_0_0.vlm"},
	{0x07903CCF058F214AULL, "hlac_ca_caverns1_24422_4y.dds"},
	{0x0790414DFB39F28DULL, "art\\characters\\creatures\\desiredemon\\animation\\inst.ani"},
	{0x079055EA22CEF89FULL, "art\\characters\\weapons\\maces\\w_mce_mc105a.msh"},
	{0x07905BEA22CF0341ULL, "art\\characters\\weapons\\maces\\w_mce_mc105a.mmh"},
	{0x0790A97901D37906ULL, "fx_dazzed_halo.mao"},
	{0x079199C41271A5CCULL, "qun221_petrice6077987.cl"},
	{0x0791A074768F662FULL, "one280_iwan6096927.cl"},
	{0x0791ED384D2EBAC2ULL, "vxht_0_hightown_n_bko_0_0.vlm"},
	{0x0792120B354F4775ULL, "6016297_m.fxe"},
	{0x079262979B1E8315ULL, "vxw_woundedcoast_oa_0_0.vlm"},
	{0x0792704B3357A15EULL, "mcr370_templar16086968.cl"},
	{0x079288EF872B6026ULL, "6147641_f.fxe"},
	{0x07929FA0A478E1E9ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\1ww12_45windownk.phy"},
	{0x07934159153D9DB9ULL, "dae000_tomwise6167323.cl"},
	{0x079354FAAA3C5320ULL, "vxht_0_hightown_c11_0_0.vlm"},
	{0x079358272066AF06ULL, "vxlt_0_lowtown_a2_n_c_gpe_0_0.vlm"},
	{0x0793EFD060BA5E87ULL, "vxlt_0_lowtown_a2_hhf_0_0.vlm"},
	{0x07940EE53DDC2D11ULL, "826508734.ogg"},
	{0x079477AE5B1BB930ULL, "l_mag311ar_warehouse.bnk"},
	{0x07959BB2E95D1DCFULL, "6137508_f.fxe"},
	{0x0795C1A09CD18D0DULL, "art\\levels\\races\\proxy\\tilesets\\setint\\benstestset\\9pp_setprops.msh"},
	{0x0795CFA09CD1A540ULL, "art\\levels\\races\\proxy\\tilesets\\setint\\benstestset\\9pp_setprops.mao"},
	{0x0795D3A09CD1AB93ULL, "art\\levels\\races\\proxy\\tilesets\\setint\\benstestset\\9pp_setprops.mmh"},
	{0x079623CB956077ABULL, "6149704_m.fxe"},
	{0x079624161B63CF5CULL, "vxca_caverns2_95p_0_0.vlm"},
	{0x0796BF8E7CA1C6DFULL, "llac_lt_0_lowtown_docks_-65537_4.dds"},
	{0x0796CB27F69E2A01ULL, "vxlt_0_lowtown_docks_n_b0p_0_0.vlm"},
	{0x0796D2DBA5B734B4ULL, "6051399_f.fxe"},
	{0x0796DCB632C0BFF9ULL, "vxht_0_hightown_n_aua_0_0.vlm"},
	{0x0797BEDDA20FBC0FULL, "vxht_brothel_n_4cc_0_0.vlm"},
	{0x0797E6F5C251AF02ULL, "17813400.ogg"},
	{0x079817387B9A9CB5ULL, "vxht_mansion2b_1af_0_0.vlm"},
	{0x079819512C7794DDULL, "12972369.ogg"},
	{0x07983521B183B8F2ULL, "6110897_m.fxe"},
	{0x07986D21145239A8ULL, "cage_05.dds"},
	{0x0798813AAD1FC871ULL, "hlac_lt_0_lowtown_n_-131071_0y.dds"},
	{0x07988CE964F3F33AULL, "isa111_isabela6068965.cl"},
	{0x0798B20C71F589A5ULL, "6150379_m.fxe"},
	{0x0798E8ECA659A862ULL, "vxlt_0_lowtown_a2_n_ewt_0_0.vlm"},
	{0x0798FF6E04ADA452ULL, "bld_drian_tu3.dds"},
	{0x07990A99AAA6059BULL, "6149058_m.fxe"},
	{0x079910C8282AC02FULL, "296548502.ogg"},
	{0x079941D3A4AE21B0ULL, "285098305.ogg"},
	{0x07998607731E5F3AULL, "17537944.ogg"},
	{0x0799BE966081D64FULL, "6121207_m.fxe"},
	{0x0799E0EDB86F2D66ULL, "follower_banter6136364.cl"},
	{0x079A0B4C2B56BDFBULL, "mrl310_party_comments6142849.cl"},
	{0x079BA109D01A41C0ULL, "746979851.ogg"},
	{0x079BEF6A50F3A48AULL, "vxlt_warehouse_aj_0_0.vlm"},
	{0x079C25A0463BB1DDULL, "vxw_bonepitdestroyed_a3_1p6_0_0.vlm"},
	{0x079C4903423F93C8ULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_nicechair.msh"},
	{0x079C5B03423FB259ULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_nicechair.mao"},
	{0x079C6703423FC6F2ULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_nicechair.mmh"},
	{0x079D596854B51CEDULL, "455142024.ogg"},
	{0x079D5B50F9509E90ULL, "vxw_encounter3_8k_0_0.vlm"},
	{0x079F7C6BB2C03D9CULL, "follower_banter6163535.cl"},
	{0x079F9EAF6E81F1A9ULL, "vxht_0_hightown_a3_avj_0_0.vlm"},
	{0x079FBF9345CA6760ULL, "6115755_m.fxe"},
	{0x079FD839D1194E00ULL, "6147859_m.fxe"},
	{0x079FFD8CDBF605D6ULL, "126757341.ogg"},
	{0x07A04268A503D490ULL, "6108937_m.fxe"},
	{0x07A08AAB9EEA5DEFULL, "vxlt_0_lowtown_a2_n_c_exj_0_0.vlm"},
	{0x07A0B95DB570FE25ULL, "210799245.ogg"},
	{0x07A0E329BACC1713ULL, "vxlt_gallowstemplar_3sc_0_0.vlm"},
	{0x07A154D6ACE182A9ULL, "vxht_0_hightown_ccr_0_0.vlm"},
	{0x07A15C440092CBC1ULL, "586777993.ogg"},
	{0x07A191472A64814CULL, "6070219_f.fxe"},
	{0x07A20620F3790224ULL, "hlac_w_sundermount_-327685_0y.dds"},
	{0x07A208588500619CULL, "vxw_bonepitmining_a2_73_0_0.vlm"},
	{0x07A222C26984998DULL, "w_sundermount_55_0.mmh"},
	{0x07A226C26984A0DEULL, "w_sundermount_55_0.mao"},
	{0x07A234C26984B813ULL, "w_sundermount_55_0.msh"},
	{0x07A28BEEA0BEC17BULL, "gen00fl_isabela6096820.cl"},
	{0x07A2ADC4AF8C58CDULL, "blank_exterior.rml"},
	{0x07A2B4510CB6AAF1ULL, "mag321_comtesse_de_launcet6043255.cl"},
	{0x07A2E7E6890C48DDULL, "mcr370_meredith6085911.cl"},
	{0x07A3111ED1954788ULL, "vxca_caverns2prc_8mm_0_1.vlm"},
	{0x07A355930C07C9E3ULL, "803028275.ogg"},
	{0x07A369EC8950D391ULL, "6092510_m.fxe"},
	{0x07A392C520F29D64ULL, "w_woundedcoast_157_0.msh"},
	{0x07A3A4C520F2BBCDULL, "w_woundedcoast_157_0.mao"},
	{0x07A3A8C520F2C206ULL, "w_woundedcoast_157_0.mmh"},
	{0x07A3AA9D50257441ULL, "hlac_lt_0_lowtown_docks_a3_n_-131074_2x.dds"},
	{0x07A3CA44FE7BE2D7ULL, "vxw_bonepit_8o_0_0.vlm"},
	{0x07A4E3A533F18612ULL, "hlac_dr_dungeon_3207_0x.dds"},
	{0x07A5953ACC3BF506ULL, "175081927.ogg"},
	{0x07A59A7288C49CE6ULL, "vxht_0_hightown_c3a_0_0.vlm"},
	{0x07A59EE13FB0A434ULL, "lt_0_lowtown_n_19_0_lc.mmh"},
	{0x07A5D15842FAB5FEULL, "llac_lt_0_lowtown_docks_a3_n_-65538_4.dds"},
	{0x07A689156D90B535ULL, "vxht_mansion2b_1de_0_0.vlm"},
	{0x07A6E9FAB2ACB9BCULL, "6055434_f.fxe"},
	{0x07A6EAB8ACF5168EULL, "ef_bar_flirt.mor"},
	{0x07A7298FBF9B4CB8ULL, "552606664.ogg"},
	{0x07A747C161E24B38ULL, "leave_chunky_l3.mao"},
	{0x07A77C769C9C1E1BULL, "6142158_m.fxe"},
	{0x07A81B42AB6EB904ULL, "vxht_brothel_n_5dx_0_0.vlm"},
	{0x07A83082117C2D9BULL, "vxlt_hangedman_18q_0_0.vlm"},
	{0x07A8B53FA8A63412ULL, "llac_ht_brothel_n_3374_0.dds"},
	{0x07A8B99A35C7A937ULL, "gencr_desire_demon.utc"},
	{0x07A8E86D0D32D5ABULL, "hm_nobleman_1.mor"},
	{0x07A8FA053BE83271ULL, "vxw_bonepitmining_a2_cv_0_0.vlm"},
	{0x07A90598852A126EULL, "hlac_ca_cavern1_haunted_24426_11x.dds"},
	{0x07A9CF15D1B94718ULL, "llac_dr_primevaltomb_661_1.dds"},
	{0x07A9EAA7C65BF617ULL, "seb321_leliana6197874.cl"},
	{0x07AA2CB96BD5D3AAULL, "storage_n.dds"},
	{0x07AA2FEA6F6ED560ULL, "254787041.ogg"},
	{0x07AA977D3831CA5DULL, "vxw_bonepitdestroyed_a3_1u5_0_0.vlm"},
	{0x07AAE76B5E549F8CULL, "6044875_m.fxe"},
	{0x07AB1827883E8449ULL, "vxlt_hangedman_n_1td_0_0.vlm"},
	{0x07AB3AE75B7662A8ULL, "820697048.ogg"},
	{0x07AB751D546DBA3FULL, "673121504.ogg"},
	{0x07AB804A787A212CULL, "art\\vfx\\combat\\vfx_r_arc_pin_shot_imbz_c.dura.ani"},
	{0x07ABCD2DEB46B710ULL, "vxw_encounter3_5h_0_0.vlm"},
	{0x07AC2B9435B75243ULL, "dre000_sibling_has_blight6036482.cl"},
	{0x07AC953FD096B54CULL, "vxlt_0_lowtown_a3_n_c_ffu_0_0.vlm"},
	{0x07ACAD9F01DB07A0ULL, "6153937_m.fxe"},
	{0x07ACDEC489082DA4ULL, "770637477.ogg"},
	{0x07AD01E573F4A55AULL, "zz_followers_banter6158559.cl"},
	{0x07AD22496EA5F100ULL, "vxlt_0_lowtown_fhg_0_0.vlm"},
	{0x07AD22C2698EA106ULL, "w_sundermount_55_0.rml"},
	{0x07AD48852204ECD7ULL, "vxlt_0_lowtown_a3_n_c_eov_0_0.vlm"},
	{0x07AD4F573B3AE94AULL, "mag360_samson6042088.cl"},
	{0x07AD8AA0A8F86BC3ULL, "llac_dr_0_deeproads_5845_18.dds"},
	{0x07ADA45F82D77C0EULL, "6088420_m.fxe"},
	{0x07ADAF274569187AULL, "134177803.ogg"},
	{0x07ADCB9E55305C07ULL, "w_woundedcoast_15_0_lc.mmh"},
	{0x07ADD2CB84CBAB95ULL, "vxw_sundermount_1yz_0_0.vlm"},
	{0x07ADFA3E9E367B43ULL, "llac_lt_backalley_g_0_1.dds"},
	{0x07AE317037C07FE0ULL, "6110855_m.fxe"},
	{0x07AE4A16C30F6680ULL, "6140959_m.fxe"},
	{0x07AE7B2A964F0C47ULL, "6092400_m.fxe"},
	{0x07AE8EBCD1972F19ULL, "877931651.ogg"},
	{0x07AEAE741F948A36ULL, "hlac_lt_0_lowtown_docks_a3_n_-262144_2x.dds"},
	{0x07AEBE5833C03698ULL, "llac_w_bonepit_0_6.dds"},
	{0x07AEC6C807168794ULL, "art\\characters\\weapons\\greatswords\\textures\\w_gsw_gs104a_0n.dds"},
	{0x07AED024A32023CBULL, "882362404.ogg"},
	{0x07AF05D36D231654ULL, "character_character_p179.psh.fxo.dx11"},
	{0x07AF2A896FFB3848ULL, "vxlt_0_lowtown_docks_b54_0_1.vlm"},
	{0x07AF46517F0AB415ULL, "vxw_woundedcoast_ao_0_0.vlm"},
	{0x07AF4BCEDA73DAC3ULL, "dae211ar_tavern.are"},
	{0x07AFAFA9248D71F7ULL, "art\\cinematics\\_crit path\\prologue\\additional animations\\cs_hurlock_napalm5.gad"},
	{0x07AFD1F6796946EAULL, "601840597.ogg"},
	{0x07B0168BD72BAAE1ULL, "or_db_clo.mao"},
	{0x07B0BBB3338AC02FULL, "vxw_sundermount_31g_0_1.vlm"},
	{0x07B0E18E92DB5245ULL, "22221941.ogg"},
	{0x07B133B8D7ECBAAFULL, "6104945_m.fxe"},
	{0x07B140B50D23AE43ULL, "vxht_mansion2b_32g_0_0.vlm"},
	{0x07B1AC9D747E03D8ULL, "vxlt_0_lowtown_docks_n_c3u_0_0.vlm"},
	{0x07B22C695CDFF021ULL, "6060906_m.fxe"},
	{0x07B26DB8598451CEULL, "6036515_m.fxe"},
	{0x07B28D4A7880566EULL, "art\\vfx\\combat\\vfx_r_arc_pin_shot_imbz_c.dura.gad"},
	{0x07B28DCA3F2B1789ULL, "736528541.ogg"},
	{0x07B2A1EA40C10EB0ULL, "1013623299.ogg"},
	{0x07B300D78E4EBCDCULL, "hlac_w_woundedcoast_-65536_0y.dds"},
	{0x07B31BC26993014AULL, "w_sundermount_55_0.phy"},
	{0x07B34A87F4440C10ULL, "lgt360_mag_orlanna6155391.cl"},
	{0x07B383E04D34C2B6ULL, "589877431.ogg"},
	{0x07B38FBF501341E1ULL, "6077303_f.fxe"},
	{0x07B3E6544945001CULL, "6106971_m.fxe"},
	{0x07B3FCB9051C8214ULL, "combat_smcc_range.gda"},
	{0x07B40C4F7ABAB566ULL, "vxht_0_hightown_c6b_0_0.vlm"},
	{0x07B45363D2204596ULL, "381032126.ogg"},
	{0x07B4F12BC1D5B558ULL, "95636241.ogg"},
	{0x07B53FDC196E4CEFULL, "994898071.ogg"},
	{0x07B54C5C72061B92ULL, "art\\vfx\\combat\\vfx_m_cre_heroaura_mod_c.mmh"},
	{0x07B57C6E1817527CULL, "vxht_0_hightown_a2_5gy_0_0.vlm"},
	{0x07B585200DD5E221ULL, "art\\vfx\\combat\\vfx_r_ibl_loop_modal_c.anb"},
	{0x07B5B694A1A313F9ULL, "6053342_m.fxe"},
	{0x07B5DB4D7CE8B2B7ULL, "vxlt_warehouse_n_199_0_0.vlm"},
	{0x07B5E7C58EA2D9CAULL, "174419138.ogg"},
	{0x07B6281F32FDF995ULL, "hlac_ht_0_hightown_a3_196607_0z.dds"},
	{0x07B639C6379DF431ULL, "zz_followers_banter6158349.cl"},
	{0x07B6BE288456C5E9ULL, "vxht_0_hightown_n_bif_0_0.vlm"},
	{0x07B6E3E20376BFBFULL, "308729787.ogg"},
	{0x07B76BE22DDE4AF1ULL, "vxw_bonepitmining_a2_nu_0_0.vlm"},
	{0x07B7B43422F45761ULL, "vxlt_0_lowtown_a2_fiy_0_0.vlm"},
	{0x07B7CC6F2BBAAA9BULL, "mcr310_argument6075803.cl"},
	{0x07B849EBC9ACD37BULL, "6002799_m.fxe"},
	{0x07B9884A78866C4DULL, "art\\vfx\\combat\\vfx_r_arc_pin_shot_imbz_c.dura.evt"},
	{0x07B98A047A349CC9ULL, "vxlt_hangedman_n_1qk_0_0.vlm"},
	{0x07B9C50993EBB0B4ULL, "llac_ca_caverns1_mines_24424_8.dds"},
	{0x07BA2B1B2AE03BB5ULL, "6170557_m.fxe"},
	{0x07BA59AE5B3C55FAULL, "l_mag311ar_warehouse.wbd"},
	{0x07BAACB93A3E8A6BULL, "163963612.ogg"},
	{0x07BB2EBB5F4185B0ULL, "vxht_chantry_n_1yh_0_0.vlm"},
	{0x07BC163C74CD948EULL, "6083520_m.fxe"},
	{0x07BC414D704117D5ULL, "174587843.ogg"},
	{0x07BC50BC3009F982ULL, "521976231.ogg"},
	{0x07BC5CEA22F43FBEULL, "art\\characters\\weapons\\maces\\w_mce_mc105a.phy"},
	{0x07BC6C6F060CC992ULL, "1024719107.ogg"},
	{0x07BC6FEB1E917977ULL, "232933825.ogg"},
	{0x07BC7D7630B42CD3ULL, "plt_cod_itm_blackfeather.plo"},
	{0x07BC875C762D633CULL, "fxc_spider_web.dds"},
	{0x07BCAAC75DDC7DEFULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\9ff_slabsedge.mao"},
	{0x07BCB6C75DDC918CULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\9ff_slabsedge.mmh"},
	{0x07BCBCC75DDC9C7EULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\9ff_slabsedge.msh"},
	{0x07BD13171E7E16C4ULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\0ff_chantrybasic.msh"},
	{0x07BD25171E7E352DULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\0ff_chantrybasic.mao"},
	{0x07BD27B6165FCEEEULL, "vxht_mansion2b_3g5_0_0.vlm"},
	{0x07BD29171E7E3C66ULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\0ff_chantrybasic.mmh"},
	{0x07BD6E6582D68EEFULL, "vxlt_0_lowtown_a2_n_c_ert_0_0.vlm"},
	{0x07BDC12AFF2F6F52ULL, "181919069.ogg"},
	{0x07BE88A0A49DF50EULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\1ww12_45windownk.mmh"},
	{0x07BE94A0A49E09A5ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\1ww12_45windownk.mao"},
	{0x07BEA2A0A49E21FCULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\1ww12_45windownk.msh"},
	{0x07BEEC1268EC929CULL, "vxw_bonepitmining_a2_91_0_0.vlm"},
	{0x07BF1E690FC0B451ULL, "vxht_chantry_n_339_0_0.vlm"},
	{0x07BF64DFF6FD123CULL, "904428238.ogg"},
	{0x07BFA1B7883D1FAFULL, "vxw_bonepit_bu_0_0.vlm"},
	{0x07BFD7F04E17E561ULL, "845524399.ogg"},
	{0x07C0E99B8BE979C4ULL, "vxdr_dungeon_1m0_0_0.vlm"},
	{0x07C23264A8D08F15ULL, "mcr370_meredith6084398.cl"},
	{0x07C2EB931DD1E0C9ULL, "6090039_m.fxe"},
	{0x07C366200DE1EA9AULL, "art\\vfx\\combat\\vfx_r_ibl_loop_modal_c.mmh"},
	{0x07C369545B364919ULL, "vxdr_primevalentrance_1k4_0_1.vlm"},
	{0x07C3C0238BAC6FCCULL, "344964187.ogg"},
	{0x07C3DCA9249E677DULL, "art\\cinematics\\_crit path\\prologue\\additional animations\\cs_hurlock_napalm5.ani"},
	{0x07C3E0E55A4DBD65ULL, "vxw_woundedcoast_o1_0_0.vlm"},
	{0x07C40F74A30CA4B5ULL, "hlac_lt_0_lowtown_a3_n_c_131071_2y.dds"},
	{0x07C4F9EA32C68A71ULL, "6081349_m.fxe"},
	{0x07C5119A7AF699D4ULL, "vxlt_0_lowtown_h9n_0_0.vlm"},
	{0x07C545B8F27C340EULL, "6216072_m.fxe"},
	{0x07C56FEDD42955DBULL, "lgt110_arl_bann_teagan6115619.cl"},
	{0x07C5A17748BE5696ULL, "lt_0_lowtown_docks_a3_11_0_ll.mmh"},
	{0x07C60A4FC5ECF7F4ULL, "6172543_m.fxe"},
	{0x07C711EF436C8490ULL, "vxlt_0_lowtown_a2_erj_0_0.vlm"},
	{0x07C720171E868131ULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\0ff_chantrybasic.phy"},
	{0x07C7FE7898EE0254ULL, "vxlt_0_lowtown_ev6_0_0.vlm"},
	{0x07C86BE01FAC33D4ULL, "character_character_p252.psh.fxo.dx11.assembly"},
	{0x07C90DF2FEC9FA75ULL, "llac_lt_0_lowtown_docks_a3_n_-65540_2.dds"},
	{0x07C9F571D16F3F08ULL, "llac_w_sundermount_5_0.dds"},
	{0x07C9FC437CF16314ULL, "vxlt_darktown_2v2_0_0.vlm"},
	{0x07CA2C3F05F11DD7ULL, "vxlt_0_lowtown_a3_n_c_eup_0_0.vlm"},
	{0x07CA7ED251DFF8EAULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\basedualwield\\combat\\attackv3.gad"},
	{0x07CAC273B683F907ULL, "6127991_m.fxe"},
	{0x07CB6B6DCC8894C3ULL, "6120846_m.fxe"},
	{0x07CB75E45FBFF9C4ULL, "6155823_m.fxe"},
	{0x07CB9E9BA4087FEDULL, "vxht_mansion2b_3cv_0_0.vlm"},
	{0x07CC3AC30C45AB72ULL, "follower_banter6155631.cl"},
	{0x07CC69ABCF5D4D37ULL, "gen_im_acc_rng_act3_08.uti"},
	{0x07CCC59C904E29D9ULL, "hlac_lt_0_lowtown_a2_1_7z.dds"},
	{0x07CD02A5BD559525ULL, "6208025_m.fxe"},
	{0x07CD2E8BF6D20F48ULL, "qun110_petrice6018527.cl"},
	{0x07CD51A9978277C7ULL, "6115819_m.fxe"},
	{0x07CD84DB701F39A9ULL, "6048135_m.fxe"},
	{0x07CDFF55769A5655ULL, "llac_ht_chantry_2534_46.dds"},
	{0x07CE13947A33382FULL, "vxw_bonepit_mt_0_0.vlm"},
	{0x07CF47373798EA95ULL, "vxw_bonepit_1wg_0_0.vlm"},
	{0x07CFB6632E6AF9C7ULL, "729158139.ogg"},
	{0x07CFC487D6F16D63ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\1ww_oct2window.msh"},
	{0x07CFD287D6F184BDULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\1ww_oct2window.mmh"},
	{0x07CFD687D6F18BCEULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\1ww_oct2window.mao"},
	{0x07CFF08EE58EB8D5ULL, "105379879.ogg"},
	{0x07CFF9369A54D896ULL, "vxlt_gallowscourtyard_n_4r3_0_0.vlm"},
	{0x07D02A95812DF46AULL, "692712829.ogg"},
	{0x07D08387DE683419ULL, "qun221_viscount.bnk"},
	{0x07D11890EFD9ED38ULL, "356147462.ogg"},
	{0x07D11B9884579072ULL, "70752520.ogg"},
	{0x07D137EB96D77FCDULL, "624611782.ogg"},
	{0x07D141F628313D7BULL, "887281049.ogg"},
	{0x07D155AFFBBA550FULL, "6080278_m.fxe"},
	{0x07D161D251E5E581ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\basedualwield\\combat\\attackv3.evt"},
	{0x07D1CC2C0F3756F7ULL, "gen00fl_merrill6162019.cl"},
	{0x07D1DEAC4372FEB5ULL, "vxht_mansion2b_1ub_0_0.vlm"},
	{0x07D226B87A1D0DCFULL, "hlac_w_bonepitdestroyed_a3_458757_0y.dds"},
	{0x07D252C24C43D5E5ULL, "vxw_woundedcoast_d0_0_0.vlm"},
	{0x07D2AFD31EA8630FULL, "art\\vfx\\combat\\vfx_r_ibl_fli_pose1_p.dura.gad"},
	{0x07D2B3E278EC7EB8ULL, "hlac_lt_0_lowtown_docks_a3_-3_0y.dds"},
	{0x07D2BAD7D9F0672DULL, "mag101_aombanter6052830.cl"},
	{0x07D311662E5CD5B6ULL, "6064992_m.fxe"},
	{0x07D326FA453B4DA8ULL, "docks_scene_dockworkers6176473.cl"},
	{0x07D3652E6154B23DULL, "w_bonepitmining_a2_233_0.mao"},
	{0x07D3692E6154B976ULL, "w_bonepitmining_a2_233_0.mmh"},
	{0x07D3732E6154CA74ULL, "w_bonepitmining_a2_233_0.msh"},
	{0x07D3A587DE6AA479ULL, "qun221_viscount.cnv"},
	{0x07D3AF6A30A3CBB1ULL, "vxht_keep_n_55q_0_0.vlm"},
	{0x07D3BAF0E4FB3162ULL, "hlac_lt_backalley_g_65537_7y.dds"},
	{0x07D47026BC7CCD8BULL, "6108079_m.fxe"},
	{0x07D4A65078E14E53ULL, "hlac_w_encounter3_-327681_0z.dds"},
	{0x07D4C955427BD818ULL, "6064859_m.fxe"},
	{0x07D519A017DC7450ULL, "art\\levels\\races\\proxy\\props\\4tools\\inv_ball12.mmh"},
	{0x07D522B9D6078F68ULL, "6109206_m.fxe"},
	{0x07D561ACE1C282B2ULL, "115839401.ogg"},
	{0x07D5902D6D099B0EULL, "gen_im_trs_slv_junk2_02.uti"},
	{0x07D5C8E72988AC19ULL, "rdr111dg_martin6014947.cl"},
	{0x07D6E726D041ECEFULL, "hlac_ca_caverns2prc_5802_26y.dds"},
	{0x07D7668AFC694D34ULL, "84425335.ogg"},
	{0x07D76CAE641B4229ULL, "llac_lt_0_lowtown_docks_a3_n_-131074_6.dds"},
	{0x07D77780C36FB7A8ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\prp_chainhead.mmh"},
	{0x07D78380C36FCC4BULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\prp_chainhead.mao"},
	{0x07D79580C36FEAD2ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\prp_chainhead.msh"},
	{0x07D7FAEFE020517BULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_ferelden_jimmy_da.dds"},
	{0x07D8213AB9E453AAULL, "335318809.ogg"},
	{0x07D833F6FDFEFFB0ULL, "910051035.ogg"},
	{0x07D85CD647C8F9CAULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_buzzardb_d.dds"},
	{0x07D86E206EE77B94ULL, "vxlt_darktown_2y3_0_0.vlm"},
	{0x07D89E1BF7E73657ULL, "vxlt_0_lowtown_a3_n_c_exs_0_0.vlm"},
	{0x07D8A26F67EAAB2FULL, "lt_0_lowtown_a2_n_46_0_ll.mmh"},
	{0x07D8A57D66AE40D8ULL, "isa120cr_letter_thug.utc"},
	{0x07D8C4D31EACF2BCULL, "art\\vfx\\combat\\vfx_r_ibl_fli_pose1_p.dura.evt"},
	{0x07D8D74AAD1D01C7ULL, "vxlt_hangedman_21d_0_1.vlm"},
	{0x07D8D890218061C3ULL, "6021338_m.fxe"},
	{0x07D8FD66EB8E1EF1ULL, "744479932.ogg"},
	{0x07D9889835872B05ULL, "vxlt_0_lowtown_a2_n_c_hvs_0_0.vlm"},
	{0x07DA1EB7C7114AEBULL, "vxlt_0_lowtown_n_gbx_0_0.vlm"},
	{0x07DA52F1584554D2ULL, "959699377.ogg"},
	{0x07DA760481DC04DBULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\pose\\pose_a_female_weightshift_medium.ani"},
	{0x07DA9BE854ECFD95ULL, "vxw_woundedcoast_pl_0_0.vlm"},
	{0x07DB413690946A88ULL, "hlac_lt_0_lowtown_1_8x.dds"},
	{0x07DB54CEDA9924A8ULL, "dae211ar_tavern.lst"},
	{0x07DB8E2C40A908E2ULL, "vxlt_0_lowtown_docks_n_be0_0_0.vlm"},
	{0x07DBBF84405DD60CULL, "art\\characters\\playercharacter\\elfmale\\animation\\exploration\\movement\\mov_rf_leanl.ani"},
	{0x07DBD9C2654AD3F8ULL, "lgt260cr_eau_asst_harbor.utc"},
	{0x07DC3072E359355AULL, "328889444.ogg"},
	{0x07DC48803540D00FULL, "vxlt_0_lowtown_a2_h1u_0_0.vlm"},
	{0x07DC7BB97EDA0D6CULL, "hlac_ht_mansion2_818_6z.dds"},
	{0x07DC85716C2950AFULL, "vxw_bonepit_hw_0_0.vlm"},
	{0x07DD462C7AAA20CDULL, "6108239_m.fxe"},
	{0x07DDB914298F0315ULL, "vxw_bonepit_1xf_0_0.vlm"},
	{0x07DE138E807BFF10ULL, "dae000_porfiria6155264.cl"},
	{0x07DE4B5C72291F39ULL, "art\\vfx\\combat\\vfx_m_cre_heroaura_mod_c.anb"},
	{0x07DE9D338C5B0833ULL, "6094811_m.fxe"},
	{0x07DEECA10E5F3075ULL, "vxw_encounter2_camp_qr_0_0.vlm"},
	{0x07DF59D251F215D0ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\basedualwield\\combat\\attackv3.ani"},
	{0x07DF82C7FCED5DBBULL, "vxlt_0_lowtown_docks_n_bvh_0_0.vlm"},
	{0x07DFA19A8B544504ULL, "340634248.ogg"},
	{0x07DFC299A3EE0525ULL, "vxht_keep_703_0_0.vlm"},
	{0x07E05894B00A22DBULL, "shi_star.mao"},
	{0x07E07A3750CBE7BAULL, "vxw_bonepitmining_a2_fj_0_0.vlm"},
	{0x07E09B7CBE6DF8BDULL, "art\\characters\\playercharacter\\humanmale\\hm_brd_hi04.phy"},
	{0x07E0D7FDCC56A711ULL, "vxht_mansion_n_6jo_0_0.vlm"},
	{0x07E0E28263338CE2ULL, "vxlt_gallowstemplar_f_34x_0_0.vlm"},
	{0x07E13174F29AC538ULL, "gen00fl_fenris6126982.cl"},
	{0x07E1C96D73E0B1F0ULL, "641646339.ogg"},
	{0x07E23CDE7D42F8DDULL, "vxdr_primevaltomb_8m6_0_0.vlm"},
	{0x07E254CEDA9F43F9ULL, "dae211ar_tavern.ncs"},
	{0x07E264263137FA0DULL, "vxht_brothel_n_6ey_0_0.vlm"},
	{0x07E26D9E5D431D15ULL, "art\\characters\\weapons\\dagger\\textures\\w_dgr_dg105a_0t.dds"},
	{0x07E2CC8440640B4EULL, "art\\characters\\playercharacter\\elfmale\\animation\\exploration\\movement\\mov_rf_leanl.gad"},
	{0x07E32562B70EF7C3ULL, "llac_ca_caverns2_5802_22.dds"},
	{0x07E33208635BECAAULL, "vxlt_undercityserialkiller_19p_0_0.vlm"},
	{0x07E3E1591FA46B7CULL, "vxlt_gallowstemplar_4sk_0_0.vlm"},
	{0x07E3F5A92758B590ULL, "vxlt_0_lowtown_a2_exh_0_0.vlm"},
	{0x07E41DC0BDEE8015ULL, "dr_primevalentrance_mmap.dds"},
	{0x07E45EF1000A2C5DULL, "vxw_bonepitdestroyed_a3_1i1_0_0.vlm"},
	{0x07E4D2FAD846C2D3ULL, "art\\characters\\creatures\\shriek\\animation\\combat\\choke.evt"},
	{0x07E57C8E3034968DULL, "vxw_blightlands_7o_0_0.vlm"},
	{0x07E5E4C4ECE2B977ULL, "gen_im_npc_wep_two_qunb.uti"},
	{0x07E603B4B25B4E77ULL, "vxlt_0_lowtown_docks_n_bgc_0_0.vlm"},
	{0x07E693FD5E894D6EULL, "art\\vfx\\cinematic\\vfx_scaredmage_c.mmh"},
	{0x07E69EB749A6C522ULL, "vxht_chantry_2zg_0_0.vlm"},
	{0x07E6A094E5F4C513ULL, "vxht_0_hightown_a2_ary_0_0.vlm"},
	{0x07E6CCD31EB93DD5ULL, "art\\vfx\\combat\\vfx_r_ibl_fli_pose1_p.dura.ani"},
	{0x07E779EF15BACC63ULL, "226769632.ogg"},
	{0x07E7A62D9A702A07ULL, "6129791_m.fxe"},
	{0x07E7CF12E8AA3761ULL, "6041110_m.fxe"},
	{0x07E87E0481E84FD2ULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\pose\\pose_a_female_weightshift_medium.evt"},
	{0x07E881991774C3BEULL, "923356097.ogg"},
	{0x07E9694248A2AFE9ULL, "vxw_encounter3_7w_0_0.vlm"},
	{0x07E9C784406A212DULL, "art\\characters\\playercharacter\\elfmale\\animation\\exploration\\movement\\mov_rf_leanl.evt"},
	{0x07EA0009329F2162ULL, "vxlt_0_lowtown_docks_n_bb1_0_0.vlm"},
	{0x07EA03CCEE93549FULL, "vxht_mansion2_1sj_0_0.vlm"},
	{0x07EAE92A4DA7B9BCULL, "art\\characters\\playercharacter\\elfmale\\animation\\exploration\\movement\\mov_p_r90_p.gad"},
	{0x07EB4922AFFF9559ULL, "gen00fl_isabela6075234.cl"},
	{0x07EBCBFAD84CD514ULL, "art\\characters\\creatures\\shriek\\animation\\combat\\choke.gad"},
	{0x07EC0058576B7C9BULL, "hlac_dr_0_deeproads_5842_9y.dds"},
	{0x07EC3D4E290513E7ULL, "6160819_m.fxe"},
	{0x07EC7C0526A9C441ULL, "follower_banter6171557.cl"},
	{0x07ECE3CCE9C58EB8ULL, "art\\characters\\playercharacter\\humanfemale\\textures\\hf_rob_appc_0n.dds"},
	{0x07ED30F39F45CB24ULL, "6052999_m.fxe"},
	{0x07ED5E7E005548F5ULL, "vxw_encounter2_camp_lq_0_0.vlm"},
	{0x07ED9C31351CC759ULL, "dae200_bran6094781.cl"},
	{0x07EDD51AE276F81EULL, "dae200_bran6097002.cl"},
	{0x07EDE70A36CF65E9ULL, "122676454.ogg"},
	{0x07EDF4A4EEE3763BULL, "vxlt_0_lowtown_docks_n_bsk_0_0.vlm"},
	{0x07EDF5E57743203CULL, "llac_w_encounter2_-131068_0.dds"},
	{0x07EE1BD205F8CE23ULL, "37104844.ogg"},
	{0x07EE4A514B28C2B8ULL, "qcr220_orsino6061367.cl"},
	{0x07EE9403A8EEC7C2ULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\0ww22_galocell.msh"},
	{0x07EE9603A8EECB78ULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\0ww22_galocell.mmh"},
	{0x07EEA203A8EEDF1BULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\0ww22_galocell.mao"},
	{0x07EEE32FDFF0981BULL, "mrl000_marethari6129354.cl"},
	{0x07EF49DABE4CBF91ULL, "vxht_mansion_n_6mn_0_0.vlm"},
	{0x07EF993F519C7F61ULL, "6173699_m.fxe"},
	{0x07EF9D0481EEA365ULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\pose\\pose_a_female_weightshift_medium.gad"},
	{0x07F088A30F120B35ULL, "dae000_hate_enemies6167621.cl"},
	{0x07F0F99C0EB0A979ULL, "6177695_m.fxe"},
	{0x07F13BDDC521EECBULL, "art\\characters\\playercharacter\\humanmale\\mt_bas_hm_facem1.mao"},
	{0x07F171431856CC88ULL, "vxlt_gallowstemplar_n_481_0_0.vlm"},
	{0x07F17DCDECD97B13ULL, "6161632_m.fxe"},
	{0x07F19C7CBE7C6DC2ULL, "art\\characters\\playercharacter\\humanmale\\hm_brd_hi04.mmh"},
	{0x07F19E7CBE7C7178ULL, "art\\characters\\playercharacter\\humanmale\\hm_brd_hi04.msh"},
	{0x07F1E82A4DADD6BAULL, "art\\characters\\playercharacter\\elfmale\\animation\\exploration\\movement\\mov_p_r90_p.ani"},
	{0x07F1EFDFC6A1D2ABULL, "6096331_f.fxe"},
	{0x07F242E2E00B378DULL, "hlac_dr_0_deeproads_5840_1y.dds"},
	{0x07F29B7A2CD89D44ULL, "mrl150_merrill6150737.cl"},
	{0x07F2DAFAD8530DF2ULL, "art\\characters\\creatures\\shriek\\animation\\combat\\choke.ani"},
	{0x07F3519A6FB0F42CULL, "vxw_sundermount_372_0_1.vlm"},
	{0x07F3B2FD5E940C6DULL, "art\\vfx\\cinematic\\vfx_scaredmage_c.anb"},
	{0x07F3EE6B222AAF0DULL, "vxw_blightlands_8n_0_0.vlm"},
	{0x07F4127539BA4520ULL, "hlac_lt_0_lowtown_docks_a3_196607_0z.dds"},
	{0x07F43EEF7A477A97ULL, "6128316_m.fxe"},
	{0x07F46205132A4314ULL, "hlac_ht_0_hightown_n_65533_2x.dds"},
	{0x07F51271D7EADD93ULL, "vxht_0_hightown_a2_amx_0_0.vlm"},
	{0x07F557332BE1BC52ULL, "vxht_chantry_n_27s_0_0.vlm"},
	{0x07F59DF137159CF4ULL, "6149281_f.fxe"},
	{0x07F61D882DE9EFD4ULL, "808050658.ogg"},
	{0x07F62282D3D6B0A9ULL, "842861886.ogg"},
	{0x07F65A924FDE629CULL, "6141581_m.fxe"},
	{0x07F76603428C70EDULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_nicechair.phy"},
	{0x07F87A8C63AFA595ULL, "6059973_m.fxe"},
	{0x07F886528F9CC767ULL, "vxht_brothel_n_61w_0_0.vlm"},
	{0x07F89D03A8F72B03ULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\0ww22_galocell.phy"},
	{0x07F8B0C40AAF68B7ULL, "hlac_ht_0_hightown_a3_-65537_3y.dds"},
	{0x07F8B339F01A4173ULL, "6054734_m.fxe"},
	{0x07F8D7E54E99006EULL, "hlac_ht_chantry_n_2534_0z.dds"},
	{0x07F931CFB75CD138ULL, "vxht_chantry_26y_0_0.vlm"},
	{0x07F94206D0BE3B3FULL, "vxht_bartrand_n_734_0_0.vlm"},
	{0x07F996AFAA410E2BULL, "782189194.ogg"},
	{0x07FA61BD00B15FD6ULL, "vxw_encounter2_camp_4o_0_1.vlm"},
	{0x07FA983386411101ULL, "6148118_m.fxe"},
	{0x07FAFE5BC197CE09ULL, "gen00fl_isabela6101069.cl"},
	{0x07FB4B5E64552F67ULL, "vxw_encounter2_gb_0_0.vlm"},
	{0x07FB7580021D5823ULL, "736649680.ogg"},
	{0x07FBD05AF24B6175ULL, "vxw_encounter2_camp_gp_0_0.vlm"},
	{0x07FC50E970B17472ULL, "vxlt_warehouse_n_154_0_0.vlm"},
	{0x07FC56529079789FULL, "llac_lt_undercitydungeon_-1_28.dds"},
	{0x07FD11233804281EULL, "456708600.ogg"},
	{0x07FD2CD6487981E6ULL, "llac_lt_0_lowtown_a2_n_-65535_7.dds"},
	{0x07FD44A3AF16EFE1ULL, "6156550_f.fxe"},
	{0x07FDACB30F21084EULL, "art\\vfx\\combat\\vfx_c_ogr_wall_carck_p.dura.ani"},
	{0x07FDCB82F62A12B2ULL, "vxht_bartrand_n_5pr_0_0.vlm"},
	{0x07FDD6FF3AADB260ULL, "vxlt_darktown_56u_0_0.vlm"},
	{0x07FDE067C455C221ULL, "677318038.ogg"},
	{0x07FE1C42BF8D8367ULL, "mrl000_marethari6142331.cl"},
	{0x07FE1EA22ABB9579ULL, "follower_banter6026240.cl"},
	{0x07FE20D334896900ULL, "art\\levels\\races\\vfx_models\\_textures\\prp_firecoal_e.dds"},
	{0x07FE8933454F2A6AULL, "gen_im_trs_cot_lgt_hlm_r2.uti"},
	{0x07FEA233BF5409B9ULL, "qun221_petrice6083802.cl"},
	{0x07FEEF1EC22F9A6FULL, "art\\vfx\\combat\\vfx_ha_bd_bspew_aoe_p.impa.gad"},
	{0x07FF6C1156118EF7ULL, "pro000_opening6207400.cl"},
	{0x07FFDFF253161E65ULL, "l_env_wal_lt_qcr_mcr_cityfear_end_game.wbd"},
	{0x08000445E685187AULL, "618493915.ogg"},
	{0x080010307F82E7FBULL, "art\\characters\\weapons\\longswords\\textures\\w_lsw_ls107a_0s.dds"},
	{0x080037E1A45CD917ULL, "art\\levels\\races\\proxy\\tilesets\\setint\\8co_basic\\0wi_black.phy"},
	{0x08009069BDFE12A7ULL, "175557346.ogg"},
	{0x0800B3114F57D5A4ULL, "6154107_m.fxe"},
	{0x0800D466F214FC7FULL, "6102899_m.fxe"},
	{0x0800D4A31417BBEEULL, "fen310_fenris6118195.cl"},
	{0x08012F388329261BULL, "vxw_sundermount_30h_0_1.vlm"},
	{0x08016DD60E066C33ULL, "ss_lowclass_thugs_m_1.bnk"},
	{0x0802470AB86E95DDULL, "6102817_m.fxe"},
	{0x0802716FE03A7B75ULL, "art\\vfx\\combat\\vfx_r_gn2_impact_act_c.mmh"},
	{0x0802BE99893E3AC5ULL, "576591666.ogg"},
	{0x0802F463EB5BA6F6ULL, "vxlt_hangedman_n_sm_0_0.vlm"},
	{0x080305C497A0F344ULL, "dae000_gamlen6102525.cl"},
	{0x08032A1A400EC688ULL, "gen00fl_isabela6085616.cl"},
	{0x08033A37A172C990ULL, "vxht_keep_n_6lp_0_0.vlm"},
	{0x08034C7FF04EEA12ULL, "hlac_dr_slavers_3214_5y.dds"},
	{0x0803A6E9FC3A8CC6ULL, "vxdr_0_deeproads_ec0_0_0.vlm"},
	{0x080465D4F055EA6DULL, "201246150.ogg"},
	{0x0805189B895521A4ULL, "10482083.ogg"},
	{0x0806041EC235DC1CULL, "art\\vfx\\combat\\vfx_ha_bd_bspew_aoe_p.impa.evt"},
	{0x08060F1C698AF2EBULL, "art\\levels\\races\\ferelden\\props\\higherclass\\prp_vnicerchair_0.pwk"},
	{0x0806539E3614E7FAULL, "t3_boy_b00.tnt"},
	{0x080677077DDFCD69ULL, "571674441.ogg"},
	{0x0806A286B61183FAULL, "art\\levels\\races\\proxy\\tilesets\\setint\\4co_basic\\1wo_basic.msh"},
	{0x0806B086B6119B33ULL, "art\\levels\\races\\proxy\\tilesets\\setint\\4co_basic\\1wo_basic.mao"},
	{0x0806B486B611A240ULL, "art\\levels\\races\\proxy\\tilesets\\setint\\4co_basic\\1wo_basic.mmh"},
	{0x08070A8D5C4B0FB9ULL, "246469178.ogg"},
	{0x080711DCC281A008ULL, "art\\vfx\\materials\\vfx_smoke_puffy.mao"},
	{0x08073368A616BDEEULL, "37199919.ogg"},
	{0x080781E508C637F9ULL, "234216133.ogg"},
	{0x0807A3ACA952E9B8ULL, "vxht_chantry_23v_0_0.vlm"},
	{0x0807FEB7FECD899EULL, "461209484.ogg"},
	{0x08089AC7F455077FULL, "hm_rdr_dockworker2.mor"},
	{0x0808A93BD708DF28ULL, "ht_0_hightown_a2_33_0_ll.mmh"},
	{0x08090EAAFF714C95ULL, "vxw_bonepit_1kc_0_0.vlm"},
	{0x0809958B1554B7BBULL, "6214888_m.fxe"},
	{0x0809F6F7BD5E3699ULL, "follower_banter6172839.cl"},
	{0x080A3CB45DD81CB9ULL, "mag291_vincento6023831.cl"},
	{0x080A5F13F019824FULL, "flip_cover_pressed.cur"},
	{0x080A7F491D875940ULL, "vxca_cavern1_haunted_ckj_0_0.vlm"},
	{0x080A8C23313BE160ULL, "vxw_woundedcoast_1d0_0_0.vlm"},
	{0x080AB1FB34B9F696ULL, "art\\vfx\\creatures\\fxc_ohurl_conj_c.impa.ani"},
	{0x080AB8020C7BA420ULL, "gen_im_trs_dal_med_glv_r2.uti"},
	{0x080BA4B30F2D379FULL, "art\\vfx\\combat\\vfx_c_ogr_wall_carck_p.dura.evt"},
	{0x080C0B6CF5C34F6DULL, "zz_followers_banter6157667.cl"},
	{0x080C4EC01729F66CULL, "6085256_m.fxe"},
	{0x080C6272F1CE8073ULL, "392896239.ogg"},
	{0x080CA69E10A48420ULL, "mag101_grace6034078.cl"},
	{0x080CC0656500FB37ULL, "vxlt_backalley_n_1ny_0_0.vlm"},
	{0x080D04C10873914DULL, "vxlt_undercityencounter_1qg_0_0.vlm"},
	{0x080D9C379BCB5911ULL, "vxlt_undercitydungeon_156_0_0.vlm"},
	{0x080DB9BD071A438DULL, "vxht_brothel_n_6tv_0_0.vlm"},
	{0x080E20AFF0E5720CULL, "pro000cr_soldier2.utc"},
	{0x080E2B4047B30D90ULL, "zz_prmpt_debug6200941.cl"},
	{0x080E5DD70C67243DULL, "337179379.ogg"},
	{0x080ECD25165E8821ULL, "6074419_m.fxe"},
	{0x080EE281F9886BABULL, "lla_ht_0_hightown_a2_65538_0.dds"},
	{0x080EF867853145FBULL, "w_woundedcoast_4_0.mmh"},
	{0x080F046785315998ULL, "w_woundedcoast_4_0.mao"},
	{0x080F1667853178E5ULL, "w_woundedcoast_4_0.msh"},
	{0x080F3761E8BE3D65ULL, "vxw_bonepit_1x6_0_0.vlm"},
	{0x080F4B3FFD3AFF10ULL, "vxlt_0_lowtown_a2_eiw_0_0.vlm"},
	{0x080F4E62E09388C8ULL, "958720939.ogg"},
	{0x080F5C93A394FA53ULL, "hlac_lt_backalley_n_65536_1z.dds"},
	{0x080F6E6FE0450002ULL, "art\\vfx\\combat\\vfx_r_gn2_impact_act_c.anb"},
	{0x080FC5CBEF20087AULL, "w_sundermount_159_0.msh"},
	{0x080FC5CEA7F188AFULL, "allan_test.arl"},
	{0x080FD3CBEF201FB3ULL, "w_sundermount_159_0.mao"},
	{0x080FD7CBEF2026C0ULL, "w_sundermount_159_0.mmh"},
	{0x08105487EC171298ULL, "art\\characters\\creatures\\shade\\animation\\blendtree\\shade_movement.anb"},
	{0x0810B8E7AA64AE5DULL, "6107717_m.fxe"},
	{0x0811A0EE1A6D6889ULL, "hm_leonato.mor"},
	{0x0811AC149368E210ULL, "vxht_keep_n_6gq_0_0.vlm"},
	{0x081216427419BFB6ULL, "hlac_ht_bartrand_n_3368_0y.dds"},
	{0x08122F9483DAC72AULL, "qun110_petrice6018536.cl"},
	{0x081285B30F332188ULL, "art\\vfx\\combat\\vfx_c_ogr_wall_carck_p.dura.gad"},
	{0x08130BC257616FD4ULL, "vxca_cavern1_haunted_edy_0_0.vlm"},
	{0x08130C1EC2407435ULL, "art\\vfx\\combat\\vfx_ha_bd_bspew_aoe_p.impa.ani"},
	{0x08131FE092BE95E9ULL, "vxlt_0_lowtown_a2_ewp_0_0.vlm"},
	{0x08134CEC6339CC31ULL, "combat_dw_mv_default.gda"},
	{0x08136A61A537B2A4ULL, "6105490_f.fxe"},
	{0x0813D15BF1A12631ULL, "vxht_0_hightown_n_apz_0_0.vlm"},
	{0x0813E933BA199B1BULL, "vxw_encounter2_camp_kz_0_0.vlm"},
	{0x0814B457BD5F12DAULL, "vxht_0_hightown_a2_ccq_0_0.vlm"},
	{0x0814CD4FDF6B2D51ULL, "qun211cr_braggart_ally_f4.utc"},
	{0x081520A01812646BULL, "art\\levels\\races\\proxy\\props\\4tools\\inv_ball12.phy"},
	{0x081646FB5B6EEEB1ULL, "art\\levels\\races\\ferelden\\props\\mining\\prp_beama.msh"},
	{0x081654FB5B6F06CCULL, "art\\levels\\races\\ferelden\\props\\mining\\prp_beama.mao"},
	{0x081660FB5B6F1B6FULL, "art\\levels\\races\\ferelden\\props\\mining\\prp_beama.mmh"},
	{0x08167791E36A905CULL, "vxlt_gallowstemplar_n_3rx_0_0.vlm"},
	{0x08167A76E1B90DF9ULL, "hlac_lt_0_lowtown_n_-262144_1z.dds"},
	{0x08170F1EB31FD75CULL, "hlac_lt_darktown_-1_48x.dds"},
	{0x08178087F1676515ULL, "vxw_bonepit_1lb_0_0.vlm"},
	{0x08178FA516C3A4A0ULL, "art\\characters\\weapons\\promo\\shields\\textures\\w_shd_011a_0s.dds"},
	{0x0817AECE2D490709ULL, "follower_banter6172729.cl"},
	{0x0817BEAE3825D5CBULL, "fx_ghosta.mao"},
	{0x0817C2F434C6FA67ULL, "394715659.ogg"},
	{0x0817D9D06C5FCDBAULL, "llac_ht_mansion2b_818_44.dds"},
	{0x0817E558645DBD54ULL, "hlac_lt_0_lowtown_-262144_3x.dds"},
	{0x0817E69BC3DDBBCCULL, "712761899.ogg"},
	{0x0818B9FB34C64187ULL, "art\\vfx\\creatures\\fxc_ohurl_conj_c.impa.evt"},
	{0x0818D11327001CDAULL, "vxca_cave_2cq_0_1.vlm"},
	{0x0818E16B4FC8786CULL, "hlac_ht_0_hightown_131071_2x.dds"},
	{0x0818F1260F7D71C0ULL, "vxca_cavern1_haunted_clk_0_0.vlm"},
	{0x081940E73359AA27ULL, "vxht_mansion2_288_0_0.vlm"},
	{0x08195D8681D04B3BULL, "art\\characters\\creatures\\shriek\\animation\\combat\\teleport.ani"},
	{0x0819957B59657988ULL, "default_player.utc"},
	{0x0819B16542C91E5EULL, "hlac_ht_0_hightown_a2_-65536_1x.dds"},
	{0x081AE57A4DAF959AULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\9ff_bridgebusted.msh"},
	{0x081AF37A4DAFADD3ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\9ff_bridgebusted.mao"},
	{0x081AF77A4DAFB4E0ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\9ff_bridgebusted.mmh"},
	{0x081B45B8D6B46ACEULL, "art\\vfx\\combat\\vfx_r_arc_sgrenade_act_c.impa.ani"},
	{0x081B9A325D5C08F7ULL, "hlac_w_encounter1_-131070_0y.dds"},
	{0x081C0452451B5ADDULL, "vxdr_primevaltomb_8y2_0_0.vlm"},
	{0x081C167F59EE0092ULL, "w_encounter2_30_0_ll.mmh"},
	{0x081C3E17831D2407ULL, "vxdr_primevaltomb_3ac_0_0.vlm"},
	{0x081C64040D69D3D4ULL, "6095464_m.fxe"},
	{0x081C94ED978C884FULL, "356182457.ogg"},
	{0x081E560A6D2483A5ULL, "gen00fl_aveline6132934.cl"},
	{0x081ED6734D522E9EULL, "547384812.ogg"},
	{0x081F8D2AB1FF5310ULL, "226344326.ogg"},
	{0x081FDAFB34CC98D0ULL, "art\\vfx\\creatures\\fxc_ohurl_conj_c.impa.gad"},
	{0x0820652CD9CAA556ULL, "dae150_harshal6148872.cl"},
	{0x0820A4414F02B14DULL, "213942103.ogg"},
	{0x0820B130FE56B500ULL, "vxlt_0_lowtown_f0o_0_0.vlm"},
	{0x0820D1AF3F8968F6ULL, "17154077.ogg"},
	{0x08211B6A9EDAA80AULL, "vxlt_blackemporium_13e_0_0.vlm"},
	{0x082127854089F5C7ULL, "gen00fl_fenris6110890.cl"},
	{0x0821339B117BF5DDULL, "418222130.ogg"},
	{0x082135975379B2A8ULL, "zz_qcr_debug6056622.cl"},
	{0x08215EF34F3D238EULL, "dae120cr_hm_guard_2.utc"},
	{0x08217186502AD0B4ULL, "vxlt_0_lowtown_docks_a3_cyc_0_0.vlm"},
	{0x08224338E3973EB1ULL, "vxht_0_hightown_n_auy_0_0.vlm"},
	{0x08225FD9C0CBAFC2ULL, "vxht_0_hightown_n_b9q_0_0.vlm"},
	{0x0822C44B667F8B81ULL, "6097558_f.fxe"},
	{0x0823016298BAA865ULL, "ter_stoney2_d.dds"},
	{0x08234A963A789D69ULL, "hlac_ht_0_hightown_a3_-131072_1y.dds"},
	{0x0824763BABE3C616ULL, "6127477_m.fxe"},
	{0x0824FF28AFA3ABA3ULL, "gen00fl_anders6136005.cl"},
	{0x082519921CE4E508ULL, "vxht_keep_6a4_0_0.vlm"},
	{0x0825437F0F9BD9EBULL, "255456765.ogg"},
	{0x0825847069A41BBAULL, "art\\characters\\playercharacter\\humanmale\\hm_brd_he03.phy"},
	{0x08258DC1DC9E03CDULL, "6149742_m.fxe"},
	{0x082617B54E063100ULL, "follower_banter6149184.cl"},
	{0x0826489BFD0514EBULL, "873052622.ogg"},
	{0x0826658681DAE332ULL, "art\\characters\\creatures\\shriek\\animation\\combat\\teleport.evt"},
	{0x082671651FDCD222ULL, "art\\characters\\playercharacter\\humanmale\\animation\\ambient\\dialogue\\crouch_pray_exit.gad"},
	{0x08268D3CC350AC32ULL, "mrl000_marethari6129278.cl"},
	{0x082725A23F8FFCDCULL, "6107446_m.fxe"},
	{0x08287AFA57D5514DULL, "vxlt_0_lowtown_docks_a3_n_app_0_0.vlm"},
	{0x08292D8D78316145ULL, "vxca_caverns1_mines_e7u_0_0.vlm"},
	{0x08293DB8D6C09A1FULL, "art\\vfx\\combat\\vfx_r_arc_sgrenade_act_c.impa.evt"},
	{0x082A5A9E042F9D1DULL, "6015941_f.fxe"},
	{0x082A9D76EB06748DULL, "vxht_brothel_n_6zt_0_0.vlm"},
	{0x082B00375ED8A7A9ULL, "hlac_w_woundedcoast_-65534_0z.dds"},
	{0x082B795AAF8B7DBEULL, "vxw_woundedcoast_17w_0_0.vlm"},
	{0x082B80AE34D364E1ULL, "vxdr_primevalentrance_2q5_0_1.vlm"},
	{0x082B8FA6E705DEBFULL, "character_character_p273.psh.fxo.dx11.assembly"},
	{0x082B967C0434A616ULL, "one200_gold_followers6101131.cl"},
	{0x082BA25BD303A7D9ULL, "hlac_lt_undercityencounter_-1_19y.dds"},
	{0x082BB2D4D16D3D29ULL, "6140319_f.fxe"},
	{0x082BB30F2DCA486CULL, "vxht_mansion2_af_0_0.vlm"},
	{0x082D7977A316CD9AULL, "follower_banter6136720.cl"},
	{0x082D7E426319B2CFULL, "20967362.ogg"},
	{0x082D848681E136C5ULL, "art\\characters\\creatures\\shriek\\animation\\combat\\teleport.gad"},
	{0x082D94651FE32C19ULL, "art\\characters\\playercharacter\\humanmale\\animation\\ambient\\dialogue\\crouch_pray_exit.evt"},
	{0x082DF40C4F9E5442ULL, "vxdr_primevalentrance_37p_0_0.vlm"},
	{0x082E11584E2E6049ULL, "177772745.ogg"},
	{0x082E5CE1B65128F4ULL, "hlac_w_encounter3_-196603_0y.dds"},
	{0x082E6EDAA6312318ULL, "6135595_m.fxe"},
	{0x082EBE11DA29150FULL, "art\\vfx\\combat\\vfx_w_wns_front_close2_c.impa.gad"},
	{0x082F17AC4CFBDD54ULL, "vxht_chantry_n_2ci_0_0.vlm"},
	{0x082F230DF04CCD80ULL, "vxlt_0_lowtown_f5l_0_0.vlm"},
	{0x08301EB8D6C68408ULL, "art\\vfx\\combat\\vfx_r_arc_sgrenade_act_c.impa.gad"},
	{0x0830B6225455CCBDULL, "art\\levels\\races\\natural\\kirkcoastal\\foliage\\_textures\\tre_highlandplant01_sa.dds"},
	{0x08315829BED140E4ULL, "mcr370_meredith6083626.cl"},
	{0x0831C778207E77A9ULL, "vxht_0_hightown_c5x_0_0.vlm"},
	{0x0831CAF39A154CC0ULL, "mrl310_party_comments6142901.cl"},
	{0x0832E10C9D5C55F7ULL, "6109750_f.fxe"},
	{0x08331A387DFF097DULL, "vxdr_primevalentrance_2st_0_2.vlm"},
	{0x08338B6F0EDAFD88ULL, "vxht_keep_6j5_0_0.vlm"},
	{0x0833E3297713BF94ULL, "lgt270ip_eau_docks_scroll.utp"},
	{0x08341D8CBAAE37A4ULL, "6111323_f.fxe"},
	{0x08344F9AD2C39C6FULL, "6104021_f.fxe"},
	{0x08349E7B597C509CULL, "default_player.mor"},
	{0x0834CFA88946EC3BULL, "6208676_m.fxe"},
	{0x0834D311DA2DA4BCULL, "art\\vfx\\combat\\vfx_w_wns_front_close2_c.impa.evt"},
	{0x0835C567BDED1B54ULL, "6098931_m.fxe"},
	{0x0835E96344E02925ULL, "44354168.ogg"},
	{0x08360B640DCAC701ULL, "636136473.ogg"},
	{0x083663D0C411604BULL, "zz_otl_debug6060672.cl"},
	{0x0836884115650542ULL, "vxlt_0_lowtown_a2_ehz_0_0.vlm"},
	{0x0836E08C837CB8C6ULL, "vxht_chantry_33i_0_0.vlm"},
	{0x08372F911FF6C5C3ULL, "art\\characters\\playercharacter\\elfmale\\animation\\cindes\\custom\\fenris_interrogation_reaction_loop.ani"},
	{0x0837D11740379E64ULL, "6135620_f.fxe"},
	{0x08388D576C3EFD34ULL, "33522670.ogg"},
	{0x08391D01D2F04617ULL, "221582263.ogg"},
	{0x08396D267F14E939ULL, "vxlt_0_lowtown_a3_n_c_fee_0_0.vlm"},
	{0x0839A5A15FAD2243ULL, "w_bonepitdestroyed_a3_33_0.mao"},
	{0x0839A9A15FAD2890ULL, "w_bonepitdestroyed_a3_33_0.mmh"},
	{0x0839B7A15FAD40AAULL, "w_bonepitdestroyed_a3_33_0.msh"},
	{0x0839D729D7447233ULL, "1030072561.ogg"},
	{0x0839EB37A181963EULL, "vxw_woundedcoast_12h_0_0.vlm"},
	{0x083A8A346248EF8BULL, "lightning_f.dds"},
	{0x083A957B121ABB36ULL, "6084430_f.fxe"},
	{0x083A9C651FEDC408ULL, "art\\characters\\playercharacter\\humanmale\\animation\\ambient\\dialogue\\crouch_pray_exit.ani"},
	{0x083AB90C16423FBCULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\grab_action_exit.ani"},
	{0x083AC2A4742BC629ULL, "hlac_lt_0_lowtown_a2_n_-131072_1y.dds"},
	{0x083B5022A3BA0A6BULL, "6175649_f.fxe"},
	{0x083C92DD60182A8CULL, "6067124_m.fxe"},
	{0x083CF961299A5B38ULL, "war_autokillweak.dds"},
	{0x083D706C3E50EA0DULL, "hlac_ht_0_hightown_a3_-65537_2x.dds"},
	{0x083D9B6821B92B6BULL, "6020412_m.fxe"},
	{0x083E259B3C186C0DULL, "mag210_arianni6057338.cl"},
	{0x083F8F4538DE6287ULL, "hlac_lt_0_lowtown_docks_-4_0z.dds"},
	{0x083FB9F2534BC10FULL, "l_env_wal_lt_qcr_mcr_cityfear_end_game.bnk"},
	{0x083FEA7A0A511DF7ULL, "6039304_m.fxe"},
	{0x083FEB21BC7CD387ULL, "hlac_dr_primevaltomb_663_2z.dds"},
	{0x083FEC9D5B5E1638ULL, "668047676.ogg"},
	{0x08405268486BC4FCULL, "926960578.ogg"},
	{0x0840794C4468F539ULL, "hlac_lt_undercityserialkiller_-1_31y.dds"},
	{0x084090E6E3D59767ULL, "172255434.ogg"},
	{0x0841266375D9EE4DULL, "6015692_m.fxe"},
	{0x084152973DF03D40ULL, "hlac_lt_0_lowtown_a2_n_65537_6x.dds"},
	{0x0841634D5FC7CBF0ULL, "w_bonepitmining_a2_218_0.mao"},
	{0x0841674D5FC7D203ULL, "w_bonepitmining_a2_218_0.mmh"},
	{0x0841754D5FC7EA5DULL, "w_bonepitmining_a2_218_0.msh"},
	{0x0841A4A800E06D6AULL, "6098928_m.fxe"},
	{0x0841C60C164874FEULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\grab_action_exit.gad"},
	{0x0841E108F5390ABCULL, "6129305_m.fxe"},
	{0x0842718D97803D99ULL, "vxht_0_hightown_bai_0_0.vlm"},
	{0x0842DB11DA39EFD5ULL, "art\\vfx\\combat\\vfx_w_wns_front_close2_c.impa.ani"},
	{0x0843384682256F42ULL, "hlac_lt_0_lowtown_a3_n_c_-131072_7y.dds"},
	{0x0843DA4089747027ULL, "vxlt_hangedman_sm_0_0.vlm"},
	{0x08441B89D81CEBAEULL, "294877139.ogg"},
	{0x08449DE21175BDADULL, "llac_dr_primevaltomb_658_0.dds"},
	{0x0845479120032BBAULL, "art\\characters\\playercharacter\\elfmale\\animation\\cindes\\custom\\fenris_interrogation_reaction_loop.evt"},
	{0x084552697572D146ULL, "vxht_chantry_36j_0_0.vlm"},
	{0x0846435B3FB6FB2FULL, "mcr310_sibling6076414.cl"},
	{0x08464DCF6C44F784ULL, "vxht_0_hightown_a2_c4n_0_0.vlm"},
	{0x08470CE1A498936FULL, "art\\levels\\races\\proxy\\tilesets\\setint\\8co_basic\\0wi_black.mao"},
	{0x084718E1A498A70CULL, "art\\levels\\races\\proxy\\tilesets\\setint\\8co_basic\\0wi_black.mmh"},
	{0x08471EE1A498B1FEULL, "art\\levels\\races\\proxy\\tilesets\\setint\\8co_basic\\0wi_black.msh"},
	{0x08478D499E3801D1ULL, "hlac_ca_caverns1_24421_7z.dds"},
	{0x0847ACF7DF77B6E9ULL, "6095887_m.fxe"},
	{0x08488BD60E42A191ULL, "ss_lowclass_thugs_m_1.wbd"},
	{0x084895D433D11686ULL, "6178428_m.fxe"},
	{0x0848A10C164E541DULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\grab_action_exit.evt"},
	{0x0848FED5B0969F65ULL, "vxw_bonepit_1l2_0_0.vlm"},
	{0x08495608E2A8486BULL, "6070211_m.fxe"},
	{0x0849629E71815FB3ULL, "art\\characters\\creatures\\gibgolem\\animation\\head\\overwhelmloop.ani"},
	{0x0849BCB185808576ULL, "vxht_mansion_n_5bc_0_0.vlm"},
	{0x0849BD97AB247293ULL, "hlac_lt_gallowscourtyard_-65535_2z.dds"},
	{0x0849C00C2417B543ULL, "6173688_m.fxe"},
	{0x0849C27F6867A919ULL, "ico_gloves_medium_4.dds"},
	{0x0849C3066E2EAEBAULL, "character_character_p205.psh.fxo.dx11"},
	{0x0849F05A54F62CA4ULL, "ser221_gamlen6135081.cl"},
	{0x084A99484AC988D6ULL, "art\\vfx\\combat\\vfx_ha_bd_bfrenzy_act_c.anb"},
	{0x084B0E2E4D2A382AULL, "556695877.ogg"},
	{0x084B4691200795ADULL, "art\\characters\\playercharacter\\elfmale\\animation\\cindes\\custom\\fenris_interrogation_reaction_loop.gad"},
	{0x084B4DF9A7080856ULL, "one100dg_nabil6063822.cl"},
	{0x084B5519E8F5A2D9ULL, "llac_ht_0_hightown_131074_0.dds"},
	{0x084BDF66EA257BAFULL, "chargen_preload.gda"},
	{0x084C3281015852D9ULL, "zz_vrc_debug6200216.cl"},
	{0x084C3C81F794842AULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\_textures\\til_chantryfloor_d.dds"},
	{0x084C61A5B2B8B2D7ULL, "llac_w_bonepit_458758_0.dds"},
	{0x084C7064DCDDBC80ULL, "6135804_f.fxe"},
	{0x084E0234B9E74018ULL, "130874223.ogg"},
	{0x084F677030FF9461ULL, "524477835.ogg"},
	{0x084FE22D737C423CULL, "6113963_f.fxe"},
	{0x08501C23D07BEC84ULL, "1018586397.ogg"},
	{0x08502FFAC622E6E2ULL, "833990235.ogg"},
	{0x08504BB9E7E665DFULL, "405924032.ogg"},
	{0x0850D4778100E0A9ULL, "w_woundedcoast_81_0.rml"},
	{0x0851795538F4F947ULL, "428736614.ogg"},
	{0x0852897DBD97AEDCULL, "6146931_m.fxe"},
	{0x0852ABFB0A12A898ULL, "vxht_bartrand_n_4a3_0_1.vlm"},
	{0x0852BC74900E30F8ULL, "vxw_woundedcoast_1gq_0_0.vlm"},
	{0x0852E0D50E81A09FULL, "6075421_m.fxe"},
	{0x0852E6B734EA97D9ULL, "95733110.ogg"},
	{0x085388944459C49EULL, "hlac_lt_0_lowtown_docks_a3_65533_1x.dds"},
	{0x0853C4234B5AAE86ULL, "codex.gda"},
	{0x08547B78D677ED1CULL, "233853545.ogg"},
	{0x0854BFAC5E3B1004ULL, "vxht_0_hightown_a2_c3o_0_0.vlm"},
	{0x0854C7F299015CE3ULL, "8410723.ogg"},
	{0x0854DAAAE06CB915ULL, "901461952.ogg"},
	{0x085571F46EC4CFCAULL, "vxca_caverns2_78q_0_2.vlm"},
	{0x085663BBB21E15A5ULL, "mag100_cullen6093746.cl"},
	{0x0856834BDA43767DULL, "zz_approval_debug6003608.cl"},
	{0x085683F571D3C590ULL, "vxht_0_hightown_cbi_0_1.vlm"},
	{0x08569E30E50FA21AULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_stackshelf_d.dds"},
	{0x0856CEF1856DC73EULL, "vxw_woundedcoast_18j_0_0.vlm"},
	{0x0856E90C5D7ECB42ULL, "6166896_m.fxe"},
	{0x0857362A988ABDF4ULL, "388643105.ogg"},
	{0x08575A9E718D8FEAULL, "art\\characters\\creatures\\gibgolem\\animation\\head\\overwhelmloop.evt"},
	{0x085781C484482949ULL, "6117773_f.fxe"},
	{0x085781CFBF85B83FULL, "489106010.ogg"},
	{0x085788BFE8DC7D09ULL, "vxlt_gallowscourtyard_77c_0_0.vlm"},
	{0x08578F217B6CBF0CULL, "vxht_0_hightown_a3_2xj_0_0.vlm"},
	{0x08579C484AD41841ULL, "art\\vfx\\combat\\vfx_ha_bd_bfrenzy_act_c.mmh"},
	{0x0857A954EB620782ULL, "6077130_m.fxe"},
	{0x0857BC70CEE7203EULL, "6129032_f.fxe"},
	{0x0857E23DBAC8C4C3ULL, "zz_followers_banter6157809.cl"},
	{0x08589AFF4113CDB6ULL, "art\\levels\\races\\ferelden\\props\\lowerclass\\prp_barrelwht_0.pwk"},
	{0x0858F1B1D35735A4ULL, "one300_zevran6089108.cl"},
	{0x0858FBFF8A1DF994ULL, "gen00fl_sebastian6178362.cl"},
	{0x08590CB967408C59ULL, "gen00fl_merrill6130050.cl"},
	{0x085974004B1F5510ULL, "6118091_m.fxe"},
	{0x0859C32978C314BDULL, "966806491.ogg"},
	{0x0859D2297734724DULL, "lgt270ip_eau_docks_scroll.ncs"},
	{0x085A414F765ABA52ULL, "gen00fl_anders6125154.cl"},
	{0x085AC87781092089ULL, "w_woundedcoast_81_0.mao"},
	{0x085AD47781093522ULL, "w_woundedcoast_81_0.mmh"},
	{0x085AD677810938D8ULL, "w_woundedcoast_81_0.msh"},
	{0x085B5D68400E80F0ULL, "vxlt_0_lowtown_a2_n_c_cml_0_0.vlm"},
	{0x085B8C8E0934A5B1ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\mh_o.eyejitter.gad"},
	{0x085BB34E949556C7ULL, "57643385.ogg"},
	{0x085BC035FC4D0357ULL, "w_blightlands_140_0.phy"},
	{0x085BF2A27FD2ED0AULL, "llac_ca_caverns1_24426_9.dds"},
	{0x085C033344643676ULL, "344996939.ogg"},
	{0x085C8A6CFE9B74E2ULL, "w_blightlands_9_0.msh"},
	{0x085C8C6CFE9B7798ULL, "w_blightlands_9_0.mmh"},
	{0x085C986CFE9B8C3BULL, "w_blightlands_9_0.mao"},
	{0x085CF86785736688ULL, "w_woundedcoast_4_0.rml"},
	{0x085D83D9AF9302F0ULL, "331597471.ogg"},
	{0x085E1C9C18410ABAULL, "ts_beasts_02.gda"},
	{0x085E399E7193765DULL, "art\\characters\\creatures\\gibgolem\\animation\\head\\overwhelmloop.gad"},
	{0x085E4EAE5E2F1799ULL, "hlac_lt_0_lowtown_docks_n_196607_1z.dds"},
	{0x085F5E9D27C83B19ULL, "vxca_caverns2_8xk_0_0.vlm"},
	{0x085F8C6CFE9DADABULL, "w_blightlands_9_0.rml"},
	{0x085FB15CA5B7592EULL, "hlac_lt_gallowscourtyard_-131071_0z.dds"},
	{0x0860582AA51C8954ULL, "vxca_cave_188_0_0.vlm"},
	{0x0860D99CF04E3049ULL, "mrl150_dalish_guard6068673.cl"},
	{0x0860FB5AAF8DC75CULL, "6143631_m.fxe"},
	{0x0861435F0A00F5C9ULL, "art\\characters\\playercharacter\\elffemale\\textures\\ef_glv_hvyo_0n.dds"},
	{0x086190E8A26D1116ULL, "vxht_mansion2_1ay_0_0.vlm"},
	{0x0861A0E9209DE5FFULL, "625500654.ogg"},
	{0x086207F09D42FD2CULL, "zz_t_trailer_isabela.cut"},
	{0x08628D8E093AC65FULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\mh_o.eyejitter.ani"},
	{0x0862EB0A7D6C32F8ULL, "vxlt_darktown_590_0_0.vlm"},
	{0x086300B552A5FA87ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\rogue\\bow\\attk_v4.gad"},
	{0x0863660DC783CE02ULL, "19094807.ogg"},
	{0x08643755776F6FAEULL, "vxlt_warehouse_10l_0_0.vlm"},
	{0x08646ADDE80AC3B3ULL, "static2.psh.fxo"},
	{0x08648F69E20D476FULL, "vxlt_hangedman_n_216_0_0.vlm"},
	{0x0864B2265B479D0CULL, "llac_ht_mansion2_818_44.dds"},
	{0x0864D9EF67B6202AULL, "32253957.ogg"},
	{0x08655C96460378C7ULL, "qcr221_cut_diplomatic_sol6201297.cl"},
	{0x0865C5D6A4297168ULL, "6179003_m.fxe"},
	{0x0865DF36A7F68820ULL, "6121091_m.fxe"},
	{0x0865FA9CDAD29589ULL, "vxlt_gallowscourtyard_78b_0_0.vlm"},
	{0x08661500A4C1025AULL, "vxlt_0_lowtown_docks_a3_ctf_0_0.vlm"},
	{0x0866A06B696CB676ULL, "vxht_mansion_n_5he_0_0.vlm"},
	{0x08670A1736C117D8ULL, "6121925_m.fxe"},
	{0x0867478A01E3757AULL, "art\\levels\\races\\natural\\blightlands\\terrainmats\\ter_blightsand_d.dds"},
	{0x0867F3D87D8F7DE8ULL, "art\\characters\\playercharacter\\elfmale\\cth_coma.msh"},
	{0x086805D87D8F9C79ULL, "art\\characters\\playercharacter\\elfmale\\cth_coma.mao"},
	{0x086811D87D8FB012ULL, "art\\characters\\playercharacter\\elfmale\\cth_coma.mmh"},
	{0x08681C6AFBA1EB1CULL, "gen00fl_sebastian6179875.cl"},
	{0x08685883D3E48ADBULL, "zz_vault_debug6203254.cl"},
	{0x0868800AE7E463E7ULL, "hlac_ca_caverns1_24424_15x.dds"},
	{0x0868C39C9B027CA1ULL, "528381606.ogg"},
	{0x086913941C25A2EEULL, "6070351_m.fxe"},
	{0x08696F27AAE9EE93ULL, "6195820_m.fxe"},
	{0x0869EB1E0BDEA52FULL, "mag311cr_nyssa.utc"},
	{0x0869F5B552AC0694ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\rogue\\bow\\attk_v4.evt"},
	{0x086A40C720A9BCD8ULL, "llac_lt_0_lowtown_a3_n_c_0_1.dds"},
	{0x086A6C929CD0A86BULL, "6169041_m.fxe"},
	{0x086A9FF43BEA114BULL, "vxw_bonepitmining_a2_7p_0_0.vlm"},
	{0x086ADB7FB9291E63ULL, "zz_mag_debug6051238.cl"},
	{0x086B0CD05463C02CULL, "vxlt_0_lowtown_docks_n_cou_0_0.vlm"},
	{0x086B12DC0B2884AEULL, "711015632.ogg"},
	{0x086B2F3CA1A7A94DULL, "hlac_ca_caverns2prc_5802_37y.dds"},
	{0x086BC9C87EF7D8EDULL, "hlac_dr_primevaltomb_657_1z.dds"},
	{0x086BD0C01EF0AF9DULL, "dre000_dougal6129779.cl"},
	{0x086BF54E2DBF4282ULL, "llac_ht_chantry_n_2534_8.dds"},
	{0x086C80DC10F3947AULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\0ff_galo2floor.mmh"},
	{0x086C84DC10F39AB1ULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\0ff_galo2floor.mao"},
	{0x086C92DC10F3B2C0ULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\0ff_galo2floor.msh"},
	{0x086CD0DF4132F106ULL, "6088907_m.fxe"},
	{0x086D58CF568CAEF7ULL, "vxlt_gallowstemplar_f_3e2_0_0.vlm"},
	{0x086D7A249E0099E1ULL, "vxw_bonepitdestroyed_a3_29v_0_0.vlm"},
	{0x086D7CA22B0208D7ULL, "hlac_lt_0_lowtown_docks_-2_4x.dds"},
	{0x086D9DB6A82A53CBULL, "861525396.ogg"},
	{0x086E6708D004F6D7ULL, "512246595.ogg"},
	{0x086EFD932D58B209ULL, "zz_otl_debug6060964.cl"},
	{0x086F4374A1540BEAULL, "ser200_aveline6201545.cl"},
	{0x086FA3654A52F53DULL, "hlac_lt_0_lowtown_a3_n_c_-65538_0x.dds"},
	{0x087019BB3CB5EDF2ULL, "vxlt_backalley_n_1hv_0_0.vlm"},
	{0x087039754A3B170CULL, "art\\characters\\playercharacter\\humanmale\\boo_medb.phy"},
	{0x08704884EF134F4AULL, "vxht_keep_3zn_0_0.vlm"},
	{0x08710A45A2EAE12AULL, "1066186631.ogg"},
	{0x08710A95859F6AC0ULL, "vxlt_0_lowtown_a2_n_c_e89_0_0.vlm"},
	{0x08711AD983BE1D2EULL, "6144096_m.fxe"},
	{0x0871A823180736C1ULL, "zz_dae_debug6131012.cl"},
	{0x0871E0CC4BE9D0FFULL, "vxlt_warehouse_n_12l_0_0.vlm"},
	{0x0872A78CEDA0A6B0ULL, "llac_w_sundermount_0_5.dds"},
	{0x0872AE41DB1CF4DBULL, "465017862.ogg"},
	{0x0872FBBAE5E4A368ULL, "vxlt_undercitydungeon_10h_0_0.vlm"},
	{0x08737E29C1325B09ULL, "6031841_m.fxe"},
	{0x0873A9A15FDEA153ULL, "w_bonepitdestroyed_a3_33_0.rml"},
	{0x0873B72147E9065CULL, "948611953.ogg"},
	{0x0873C1BCC12ED884ULL, "llac_lt_gallowstemplar_f_2383_5.dds"},
	{0x08746A23DB831C87ULL, "fxa_gas_smoke_add.mao"},
	{0x08747437506EC5A3ULL, "vxht_keep_648_0_0.vlm"},
	{0x08748026B9B26552ULL, "6180690_m.fxe"},
	{0x087485E5C30E2CE9ULL, "927798562.ogg"},
	{0x08754BC0056AEA63ULL, "582270531.ogg"},
	{0x0875CF43A5068605ULL, "391285232.ogg"},
	{0x0875EF2FA3D18B8DULL, "vxw_blightlands_ee_0_0.vlm"},
	{0x087634B43F2134D4ULL, "6331090.ogg"},
	{0x087669B464621B75ULL, "vxlt_0_lowtown_a3_n_c_gv5_0_0.vlm"},
	{0x0876F4D1E3C3ECBFULL, "458174934.ogg"},
	{0x087702DA5F5545C8ULL, "vxw_bonepitdestroyed_a3_29s_0_0.vlm"},
	{0x087760F501138267ULL, "wm_ico_sundermount_roll.dds"},
	{0x087785710E1BBB6EULL, "6075051_m.fxe"},
	{0x0877EDB552B836CDULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\rogue\\bow\\attk_v4.ani"},
	{0x08782D1B7AF47448ULL, "dae000cr_grand_cleric6124703.cl"},
	{0x08783BA876172ADBULL, "art\\levels\\races\\proxy\\props\\_textures\\fence_n.dds"},
	{0x0878A2013ACA764BULL, "980530323.ogg"},
	{0x0878B4AE2CA872FCULL, "399496367.ogg"},
	{0x0878C322EBDD8855ULL, "6053272_m.fxe"},
	{0x087956D16C7666F6ULL, "346128455.ogg"},
	{0x0879A912DA3A0288ULL, "905791155.ogg"},
	{0x0879A99CD58E41BDULL, "w_woundedcoast_92_0.mao"},
	{0x0879ACB71508C8D7ULL, "6492903.ogg"},
	{0x0879AD9CD58E48F6ULL, "w_woundedcoast_92_0.mmh"},
	{0x0879B79CD58E59F4ULL, "w_woundedcoast_92_0.msh"},
	{0x087A1755A5BA317CULL, "vxht_0_hightown_c1m_0_0.vlm"},
	{0x087AA27BCD244295ULL, "mag170_thrask6054078.cl"},
	{0x087AAD18700BB7E4ULL, "1061516262.ogg"},
	{0x087AC4A2A95DDE15ULL, "mag291_vincento6027354.cl"},
	{0x087B0136E721D941ULL, "cry_star.mao"},
	{0x087B77D21BDD47AAULL, "default_alpha.mao"},
	{0x087B951B5FDC5DB7ULL, "545884007.ogg"},
	{0x087BE0F8EEF14B2BULL, "7119463.ogg"},
	{0x087BEC018FF6B261ULL, "vxw_bonepitdestroyed_a3_24u_0_0.vlm"},
	{0x087C240E801B3566ULL, "art\\vfx\\combat\\vfx_m_bld_bloodwound_imp_p.anb"},
	{0x087CAFD477483DDBULL, "hlac_lt_0_lowtown_a2_n_131072_2y.dds"},
	{0x087D0325213284C0ULL, "6075202_m.fxe"},
	{0x087D1CECDB121D8FULL, "6076478_f.fxe"},
	{0x087E03867E48B053ULL, "lt_gallowstemplar_n_8165_ll.mmh"},
	{0x087E109A76414B5AULL, "vxlt_0_lowtown_n_bz4_0_0.vlm"},
	{0x087E416B33EFD078ULL, "art\\characters\\weapons\\shields\\textures\\w_shd_001a_0s.dds"},
	{0x087F51079CD51D9EULL, "art\\characters\\weapons\\shields\\textures\\w_shd_srdb_0d.dds"},
	{0x087F53C58CAAB398ULL, "hlac_lt_warehouse_n_65537_1y.dds"},
	{0x087FB5551BA1F769ULL, "art\\characters\\playercharacter\\dwarfmale\\textures\\dm_arm_hvyn_0n.dds"},
	{0x087FD7C3BC548EE0ULL, "vxlt_darktown_5cz_0_0.vlm"},
	{0x088016DD35F9A3D7ULL, "gen_im_trs_chn_junk2_03.uti"},
	{0x08804BF54A935E51ULL, "978512188.ogg"},
	{0x08805AF3F2D91A42ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\1ww_grandcrackl.msh"},
	{0x08805CF3F2D91DF8ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\1ww_grandcrackl.mmh"},
	{0x088068F3F2D9319BULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\1ww_grandcrackl.mao"},
	{0x088114060C75E285ULL, "vxw_woundedcoast_1gh_0_0.vlm"},
	{0x08817D7069F26EE3ULL, "art\\characters\\playercharacter\\humanmale\\hm_brd_he03.msh"},
	{0x0881860BF925395FULL, "1027488832.ogg"},
	{0x0881872B7F452901ULL, "one280_emrys6104302.cl"},
	{0x08818B7069F2863DULL, "art\\characters\\playercharacter\\humanmale\\hm_brd_he03.mmh"},
	{0x0881B76302EBE433ULL, "amb_scene_dishdrop.bnk"},
	{0x0881C2B08253F26BULL, "llac_ht_keep_3743_1.dds"},
	{0x0881DD70832ED287ULL, "hlac_lt_gallowscourtyard_-131073_1z.dds"},
	{0x08820B012392BCCEULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_eye04a.phy"},
	{0x0882681559F532FFULL, "396242984.ogg"},
	{0x08827F2BB4FFDF27ULL, "6036464_m.fxe"},
	{0x0882A7A07356FEF5ULL, "348704072.ogg"},
	{0x0882A9908815A268ULL, "6177203_m.fxe"},
	{0x0882E3296E53C803ULL, "tbe300im_maker_dagger.uti"},
	{0x08830A46E4E82D37ULL, "hlac_ca_cave_2194_0x.dds"},
	{0x0883AB6DC30B4343ULL, "110307944.ogg"},
	{0x0883B92C863F78D2ULL, "6042048_m.fxe"},
	{0x0883C11E5A9C5037ULL, "hlac_ca_caverns1_24423_3z.dds"},
	{0x0883D377812C38C9ULL, "w_woundedcoast_81_0.anb"},
	{0x0883E763992800E6ULL, "darktown_male_lowlife_mob6174406.cl"},
	{0x08844B31045E3761ULL, "217970546.ogg"},
	{0x0884610C95C7A40DULL, "vxw_blightlands_nd_0_0.vlm"},
	{0x08853FEBA2510A02ULL, "hlac_w_bonepit_458753_0x.dds"},
	{0x088574B7514B5E48ULL, "vxw_bonepitdestroyed_a3_24p_0_0.vlm"},
	{0x08858593BF0D697CULL, "mag110_cut_abomination6037101.cl"},
	{0x0885E4F7E54A4491ULL, "vxlt_0_lowtown_a3_n_c_hx0_0_0.vlm"},
	{0x088683BF446DE238ULL, "hlac_ht_store_187_0y.dds"},
	{0x08872462C4414976ULL, "vxlt_gallowsprison_n_5qi_0_0.vlm"},
	{0x088735A4A3CA06F9ULL, "zz_fol_debug6092334.cl"},
	{0x08874CE014EAA976ULL, "art\\levels\\races\\dwarf\\props\\spider\\prp_tileweb.mao"},
	{0x088758E014EABD15ULL, "art\\levels\\races\\dwarf\\props\\spider\\prp_tileweb.mmh"},
	{0x08875AE014EAC08BULL, "art\\levels\\races\\dwarf\\props\\spider\\prp_tileweb.msh"},
	{0x08879F8C4C1D406DULL, "gen_im_trs_krk_hgh_junk1_01.uti"},
	{0x0887C0C13DB90399ULL, "99099205.ogg"},
	{0x0887C43BF2028EFCULL, "329931146.ogg"},
	{0x088809571609D40AULL, "860298652.ogg"},
	{0x0888386CEA88B8C0ULL, "311103735.ogg"},
	{0x088873ECA98BE09BULL, "art\\characters\\playercharacter\\humanmale\\boo_maso.mmh"},
	{0x08887FECA98BF538ULL, "art\\characters\\playercharacter\\humanmale\\boo_maso.mao"},
	{0x08888881A74DC6F4ULL, "6067700_m.fxe"},
	{0x0888893297B049FCULL, "vxht_0_hightown_c4b_0_0.vlm"},
	{0x08888F6A8C294AA2ULL, "art\\vfx\\combat\\vfx_glb_fire_imp_maj_c.impa.gad"},
	{0x088891ECA98C1385ULL, "art\\characters\\playercharacter\\humanmale\\boo_maso.msh"},
	{0x0888B6726957210BULL, "6110221_m.fxe"},
	{0x0888C54023CB71FCULL, "m_mer351ar_cave.wbd"},
	{0x0888FFEEC6E04BEEULL, "6106775_m.fxe"},
	{0x0889270E8025C411ULL, "art\\vfx\\combat\\vfx_m_bld_bloodwound_imp_p.mmh"},
	{0x0889300A4434418AULL, "art\\vfx\\combat\\vfx_w_avl_avaline2_loop_c.cess.gad"},
	{0x0889679119B14426ULL, "one140st_charade.stg"},
	{0x0889757AD3366BBFULL, "follower_barks6173406.cl"},
	{0x088988C04CA34E1AULL, "6059995_f.fxe"},
	{0x088994D93D066B69ULL, "dae300_cullen6135889.cl"},
	{0x0889B499251F2206ULL, "6086707_m.fxe"},
	{0x088A1E5443A4DF90ULL, "llac_lt_0_lowtown_n_1_1.dds"},
	{0x088A2A1DD64E02EEULL, "6169761_m.fxe"},
	{0x088A63F3F2E17D83ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\1ww_grandcrackl.phy"},
	{0x088A8ACC3584D86BULL, "532867125.ogg"},
	{0x088B3E638F9AB2B0ULL, "6164056_m.fxe"},
	{0x088B6D3FF9E637F9ULL, "vxw_bonepit_28i_0_0.vlm"},
	{0x088BE56845989741ULL, "vxca_caverns1_mines_gjx_0_0.vlm"},
	{0x088C115C023658E7ULL, "vxw_blightlands_9c_0_0.vlm"},
	{0x088C4EBB857E73E1ULL, "vxdr_primevaltomb_3oz_0_1.vlm"},
	{0x088C74E3C177A489ULL, "ht_0_hightownexplosion_n_35_0_lc.mmh"},
	{0x088C7C4E426FDA5EULL, "vxlt_gallowscourtyard_n_4wd_0_0.vlm"},
	{0x088C8502744E6312ULL, "art\\vfx\\combat\\vfx_c_profane_s_crust_c.mmh"},
	{0x088C94C70F805C6EULL, "vxlt_gallowstemplar_n_379_0_0.vlm"},
	{0x088C9CF76CA6F400ULL, "hlac_lt_gallowstemplar_f_2391_8z.dds"},
	{0x088CB18BEF297694ULL, "6166875_f.fxe"},
	{0x088CC4E178074868ULL, "72743126.ogg"},
	{0x088CFCAB8E2E3361ULL, "w_encounter1_56_0.phy"},
	{0x088D2E29F1EE43C2ULL, "6163903_m.fxe"},
	{0x088D4926934E3BC3ULL, "vxca_caverns1_mines_i2d_0_0.vlm"},
	{0x088D6A813D43217CULL, "rand_highway.gda"},
	{0x088DE45A139DED34ULL, "6167952_m.fxe"},
	{0x088EB7220357866EULL, "vxlt_gallowstemplar_f_3s5_0_0.vlm"},
	{0x088EF56ED615E966ULL, "vxht_0_hightown_n_c76_0_0.vlm"},
	{0x088F1FEBCC0867E0ULL, "131396371.ogg"},
	{0x088F8084E0309F50ULL, "llac_lt_0_lowtown_docks_-65540_1.dds"},
	{0x088FB26A8C2FA499ULL, "art\\vfx\\combat\\vfx_glb_fire_imp_maj_c.impa.evt"},
	{0x088FD50948715FD9ULL, "clf260_harley6089568.cl"},
	{0x088FD7FDC4CE4097ULL, "hlac_ht_bartrand_n_3365_1x.dds"},
	{0x0890130A443A2F21ULL, "art\\vfx\\combat\\vfx_w_avl_avaline2_loop_c.cess.evt"},
	{0x08901E50A2FE8B41ULL, "art\\vfx\\combat\\vfx_m_sta_rng_atk_v3_c.anb"},
	{0x089038BF000A0937ULL, "art\\levels\\races\\ferelden\\props\\mining\\prp_chainfloora.msh"},
	{0x08904ABF000A279AULL, "art\\levels\\races\\ferelden\\props\\mining\\prp_chainfloora.mao"},
	{0x08904EBF000A2EA9ULL, "art\\levels\\races\\ferelden\\props\\mining\\prp_chainfloora.mmh"},
	{0x089172A6D8E2540FULL, "83012575.ogg"},
	{0x08919FDC1112E485ULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\0ff_galo2floor.phy"},
	{0x0891AB15B61FA76DULL, "83334929.ogg"},
	{0x0891DD67154F4867ULL, "121215777.ogg"},
	{0x08926AFD3A61CB2BULL, "6111183_m.fxe"},
	{0x0892D2E987BDBC8DULL, "vxw_blightlands_sc_0_0.vlm"},
	{0x089340C7D218AF3CULL, "vxht_0_hightown_bj1_0_0.vlm"},
	{0x0893463862018CECULL, "52362697.ogg"},
	{0x0893DAF2DFF6B871ULL, "107773372.ogg"},
	{0x0894234B42C6859BULL, "art\\vfx\\models\\vfx_forearm_opened.msh"},
	{0x0894354B42C6A4E6ULL, "art\\vfx\\models\\vfx_forearm_opened.mao"},
	{0x0894355394992EFEULL, "gen00fl_anders6119122.cl"},
	{0x0894414B42C6B885ULL, "art\\vfx\\models\\vfx_forearm_opened.mmh"},
	{0x089510F64F220EA6ULL, "fen121im_magisters_key.uti"},
	{0x08955425BE3E3DFEULL, "llac_ca_caverns2prc_5802_11.dds"},
	{0x0895E431D912915CULL, "vxlt_0_lowtown_f50_0_0.vlm"},
	{0x08968CF1D311A701ULL, "316360927.ogg"},
	{0x0896B15F95A93E92ULL, "art\\characters\\playercharacter\\elfmale\\animation\\exploration\\movement\\mov_rf_injured_leanr.gad"},
	{0x0896E0BAF88C08B9ULL, "dae000cr_grand_cleric6153122.cl"},
	{0x089708DA7DD4ED07ULL, "vxdr_primevalentrance_1a9_0_0.vlm"},
	{0x0897164AB0CF0CCFULL, "zz_fen_debug6134335.cl"},
	{0x08972CE794F756E5ULL, "436143794.ogg"},
	{0x089739775B9C0F27ULL, "449884873.ogg"},
	{0x08975AB1022A48D8ULL, "w_bonepitdestroyed_a3_24_0_lc.mmh"},
	{0x089761703B2D6C96ULL, "gen00fl_anders6096483.cl"},
	{0x0897DCFFBD5E63D8ULL, "vxlt_gallowstemplar_5cn_0_0.vlm"},
	{0x0897FB9AF47ED25CULL, "631934082.ogg"},
	{0x08985E842C853643ULL, "vxht_mansion2b_3bw_0_0.vlm"},
	{0x08985EF55735B469ULL, "art\\characters\\weapons\\longswords\\textures\\w_lsw_019a_0d.dds"},
	{0x0898D1C455F235BDULL, "vxlt_undercityserialkiller_xs_0_0.vlm"},
	{0x089913FE27A0C144ULL, "vxht_0_hightown_a2_bx3_0_0.vlm"},
	{0x0899764E01540D55ULL, "6043981_m.fxe"},
	{0x089A3C882C12BA6AULL, "gen_im_trs_gem_tigerseye_03.uti"},
	{0x089A82B7A5F98F44ULL, "ave200im_donnic_gift.uti"},
	{0x089AB58279FAB481ULL, "6126709_m.fxe"},
	{0x089ACE985BCCE636ULL, "hlac_w_sundermount_131077_0z.dds"},
	{0x089B25B9AF0B4793ULL, "art\\vfx\\textures\\vfx_fire_tile.dds"},
	{0x089B4B00466C9C3BULL, "vxlt_0_lowtown_docks_n_b7g_0_0.vlm"},
	{0x089B55F81E243679ULL, "6083996_m.fxe"},
	{0x089BA4205B6260DCULL, "hlac_ca_caverns2_5802_70x.dds"},
	{0x089C0F86FA6BF770ULL, "dae000_bodhan6146853.cl"},
	{0x089C10337C9248B1ULL, "6085477_m.fxe"},
	{0x089CBA6A8C3A3C88ULL, "art\\vfx\\combat\\vfx_glb_fire_imp_maj_c.impa.ani"},
	{0x089D0B0A4444AC70ULL, "art\\vfx\\combat\\vfx_w_avl_avaline2_loop_c.cess.ani"},
	{0x089D7FBA12E5DEA2ULL, "xenonreload.psh.fxo"},
	{0x089DA6C84CC831F2ULL, "782658592.ogg"},
	{0x089DB45F95AF62C9ULL, "art\\characters\\playercharacter\\elfmale\\animation\\exploration\\movement\\mov_rf_injured_leanr.evt"},
	{0x089DD51A46F1D329ULL, "6038587_m.fxe"},
	{0x089DF7B12F14B786ULL, "hlac_lt_0_lowtown_a2_-131072_4x.dds"},
	{0x089DFF50A30A93BAULL, "art\\vfx\\combat\\vfx_m_sta_rng_atk_v3_c.mmh"},
	{0x089E3D0817271EB6ULL, "gen00fl_sebastian6178359.cl"},
	{0x089E9D00E0BB9878ULL, "fx_energy01.mao"},
	{0x089FAADB098118CCULL, "vxlt_0_lowtown_h7x_0_0.vlm"},
	{0x089FC80080BDE67AULL, "6104449_m.fxe"},
	{0x08A00290B7449EF2ULL, "zz_vault_debug6203378.cl"},
	{0x08A0177A9D8A1812ULL, "6142310_f.fxe"},
	{0x08A144C679B3D50DULL, "vxw_blightlands_tb_0_0.vlm"},
	{0x08A14A6E57110CDDULL, "sse_ss_revenant_1.gda"},
	{0x08A278657F577A71ULL, "ser221_party_comments6122854.cl"},
	{0x08A27872F2301D99ULL, "lowtown_fem_alien_elf_2_mob6166449.cl"},
	{0x08A28B10817910EEULL, "6170143_m.fxe"},
	{0x08A2C8B1C9367591ULL, "vxlt_0_lowtown_a3_n_c_hb6_0_0.vlm"},
	{0x08A2D1A2E1B67470ULL, "957406270.ogg"},
	{0x08A2D26782F9E0E9ULL, "hlac_ht_0_hightown_a3_-131069_0z.dds"},
	{0x08A2EA0AAE1A8161ULL, "vxlt_0_lowtown_csz_0_0.vlm"},
	{0x08A30C3FCBCD58E7ULL, "443505163.ogg"},
	{0x08A39535FC8A6FAFULL, "w_blightlands_140_0.mao"},
	{0x08A3A135FC8A844CULL, "w_blightlands_140_0.mmh"},
	{0x08A3A735FC8A8E3EULL, "w_blightlands_140_0.msh"},
	{0x08A3A7ADDC4A0AA8ULL, "6156765_m.fxe"},
	{0x08A42299BB3E151EULL, "807062034.ogg"},
	{0x08A44972931293C4ULL, "vxca_caverns2_8pb_0_0.vlm"},
	{0x08A4F7048FEF2D8AULL, "vxht_keep_n_6ai_0_0.vlm"},
	{0x08A5DCADE725ED67ULL, "vxht_brothel_n_6uk_0_0.vlm"},
	{0x08A609E23E87A72CULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\9ff_manhole.phy"},
	{0x08A61C457E95BC11ULL, "6124050_f.fxe"},
	{0x08A6B3F5A092B6C0ULL, "mag100_thrask6031631.cl"},
	{0x08A70DD7BA3A33EEULL, "6163561_m.fxe"},
	{0x08A72434484D3B37ULL, "425370666.ogg"},
	{0x08A75D85758C35E3ULL, "vxht_mansion_n_4c0_0_0.vlm"},
	{0x08A79EB5B40C5C53ULL, "vxht_chantry_n_2ve_0_0.vlm"},
	{0x08A8179C21EA1390ULL, "hlac_dr_0_deeproads_5845_19x.dds"},
	{0x08A830E752FBCDFDULL, "vxca_caverns2prc_62o_0_1.vlm"},
	{0x08A904A47F409087ULL, "gen00fl_aveline6053347.cl"},
	{0x08A92634A500BB86ULL, "gen00fl_dog6146963.cl"},
	{0x08A962C5600C9B60ULL, "art\\levels\\races\\natural\\kirkcave\\foliage\\_textures\\gra_mushroom_e.dds"},
	{0x08A98A9757BE5AE3ULL, "920824244.ogg"},
	{0x08A99EE98378DE96ULL, "twitch_core_settings.gda"},
	{0x08A9B388A6F69F27ULL, "vxht_mansion2_2v2_0_0.vlm"},
	{0x08A9D03B01DFF91BULL, "319427210.ogg"},
	{0x08A9DC648E967A01ULL, "vxlt_0_lowtown_f0b_0_0.vlm"},
	{0x08AA201D296DE018ULL, "hlac_lt_0_lowtown_docks_a3_262142_0z.dds"},
	{0x08AB2D5A9236D860ULL, "vxlt_darktown_5ry_0_0.vlm"},
	{0x08AB766834C69C2EULL, "409749874.ogg"},
	{0x08AB9C5F95BB7738ULL, "art\\characters\\playercharacter\\elfmale\\animation\\exploration\\movement\\mov_rf_injured_leanr.ani"},
	{0x08AB9FE482F64FEDULL, "vxlt_gallowstemplar_f_3ru_0_0.vlm"},
	{0x08ABB77F51DC8CC1ULL, "115300159.ogg"},
	{0x08ABFD07C266CBFFULL, "6180127_m.fxe"},
	{0x08AC2F78AB408165ULL, "6101391_m.fxe"},
	{0x08AC5671861D0990ULL, "follower_banter6155691.cl"},
	{0x08AD55F63AFD6339ULL, "vxca_cavern1_haunted_fn1_0_0.vlm"},
	{0x08ADC5ECF7088665ULL, "vxlt_0_lowtown_docks_a3_cuz_0_0.vlm"},
	{0x08AE25B7CFFDAF43ULL, "vxlt_gallowstemplar_f_37h_0_0.vlm"},
	{0x08AE8BDE3E5F4642ULL, "6039943_m.fxe"},
	{0x08AEDD02165020B6ULL, "cd_mer141_jansen.wbd"},
	{0x08AF2130BD2FA08EULL, "vxlt_undercitydungeon_17k_0_0.vlm"},
	{0x08AF8931055FD4ABULL, "plt_and000pt_end_stage.plo"},
	{0x08AF92A84E5010F9ULL, "120251091.ogg"},
	{0x08AFBF3D6BD1E575ULL, "art\\characters\\weapons\\axes\\textures\\w_axe_ax102a_0s.dds"},
	{0x08B02481B604E03CULL, "vxht_0_hightown_bp7_0_0.vlm"},
	{0x08B0251787F69F39ULL, "vxca_caverns2_62f_0_1.vlm"},
	{0x08B0568F37572622ULL, "lt_gallowstemplar_f.arl"},
	{0x08B05BBDFDFD1293ULL, "art\\levels\\races\\proxy\\props\\rugs_poor\\prp_zzlargegreen.phy"},
	{0x08B0A9F29B54C77CULL, "357781841.ogg"},
	{0x08B0AAEE8FCA3772ULL, "mag170_hugh6080709.cl"},
	{0x08B0B83859A728EBULL, "869624794.ogg"},
	{0x08B0F43E64F8312EULL, "247145366.ogg"},
	{0x08B110754A7268D7ULL, "art\\characters\\playercharacter\\humanmale\\boo_medb.mmh"},
	{0x08B114754A726FE4ULL, "art\\characters\\playercharacter\\humanmale\\boo_medb.mao"},
	{0x08B126754A728E49ULL, "art\\characters\\playercharacter\\humanmale\\boo_medb.msh"},
	{0x08B16D420CD4ABA9ULL, "qun221_petrice6093256.cl"},
	{0x08B185CA5A3A4842ULL, "w_bonepitdestroyed_a3_32_0.rml"},
	{0x08B20C9277D08E1CULL, "147841104.ogg"},
	{0x08B25B63D4E938E2ULL, "568637517.ogg"},
	{0x08B2874A1F9F2A89ULL, "vxht_mansion2_2o7_0_0.vlm"},
	{0x08B2CE37C79A0557ULL, "519455370.ogg"},
	{0x08B371F443A95045ULL, "6172568_m.fxe"},
	{0x08B3C4BB1E913AE4ULL, "417230139.ogg"},
	{0x08B5222D2F09D027ULL, "hlac_ht_bartrand_n_3381_1z.dds"},
	{0x08B561FE945BDB98ULL, "6147190_m.fxe"},
	{0x08B58402747166B9ULL, "art\\vfx\\combat\\vfx_c_profane_s_crust_c.anb"},
	{0x08B5C34023F25C76ULL, "m_mer351ar_cave.bnk"},
	{0x08B5E60B59B01B3BULL, "zz_dae_debug6029904.cl"},
	{0x08B5F7B80B8CF244ULL, "vxht_0_hightown_a2_bf1_0_0.vlm"},
	{0x08B60D1BEB0DBA2FULL, "character_character_p224.psh.fxo.dx11"},
	{0x08B6E4B1E4CED667ULL, "dae000_hate_enemies6167601.cl"},
	{0x08B79CCDFF02BAC5ULL, "6208421_m.fxe"},
	{0x08B80AD2AF2DE9F3ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\twohanded\\tw_buildup.ani"},
	{0x08B84E41808C9281ULL, "vxlt_0_lowtown_f5a_0_0.vlm"},
	{0x08B881890AEE1109ULL, "ave110tr_enter_dpatrol.ncs"},
	{0x08B8C639AE487450ULL, "vxca_caverns2prc_7tu_0_1.vlm"},
	{0x08B8D28A5CEADD7BULL, "6144516_m.fxe"},
	{0x08B8DFFD6D6E53EAULL, "165556938.ogg"},
	{0x08B951300A3DC2DEULL, "gen00fl_sebastian6188657.cl"},
	{0x08B96B385B25CE34ULL, "tbe000im_unknown_gloves.uti"},
	{0x08B99F37842CF0E0ULL, "vxlt_darktown_5wf_0_0.vlm"},
	{0x08B9FE21BD3C107AULL, "vxca_caverns1_mines_ehs_0_0.vlm"},
	{0x08BA299C92CDD5A7ULL, "dae000_maren6148431.cl"},
	{0x08BA8AA935462198ULL, "art\\levels\\races\\proxy\\props\\4tools\\inv_flat2x4.phy"},
	{0x08BB27F5D67E072AULL, "gen00fl_bethany6097824.cl"},
	{0x08BB68F9B4A8995DULL, "819019979.ogg"},
	{0x08BBDBE3ED5F52F0ULL, "vxlt_backalley_g_1kd_0_0.vlm"},
	{0x08BC2FB0771369A9ULL, "667240170.ogg"},
	{0x08BD62100D1F6723ULL, "6020544_m.fxe"},
	{0x08BD77C8B0F8C598ULL, "gen00fl_anders6125215.cl"},
	{0x08BDA461E0176816ULL, "w_blightlands_17_0.rml"},
	{0x08BDAE00E0D665FAULL, "fx_energy01.dds"},
	{0x08BE3E2F611BA09FULL, "6027053_m.fxe"},
	{0x08BE695267949A0DULL, "6084360_m.fxe"},
	{0x08BFD338DEBD4CA5ULL, "lgt310cr_mag_melson_guard1.utc"},
	{0x08C019D16924F2D5ULL, "6180954_m.fxe"},
	{0x08C02D13761DACC4ULL, "594721002.ogg"},
	{0x08C04F23B0368EE9ULL, "6118999_m.fxe"},
	{0x08C05DB50677FA37ULL, "vxlt_0_lowtown_docks_a3_bqq_0_0.vlm"},
	{0x08C07D7A271451A0ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\0ff_quarterd.phy"},
	{0x08C09339EBBB5C7FULL, "vxht_0_hightown_a2_9zm_0_0.vlm"},
	{0x08C0DAB3720C5888ULL, "fex_ruin_circ_b.mmh"},
	{0x08C0F8B3720C8BB2ULL, "fex_ruin_circ_b.msh"},
	{0x08C1043568F61D18ULL, "620373766.ogg"},
	{0x08C1DBF588159BDCULL, "832353037.ogg"},
	{0x08C20A0123C89E91ULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_eye04a.mmh"},
	{0x08C20BC304B3AECDULL, "vxht_mansion2b_1fw_0_0.vlm"},
	{0x08C21A746A58F241ULL, "lt_0_lowtown_docks_a3_n_9_0_lc.mmh"},
	{0x08C2240123C8CB4FULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_eye04a.msh"},
	{0x08C239C15B003694ULL, "mer240_jansen6050409.cl"},
	{0x08C26FCA5A489597ULL, "w_bonepitdestroyed_a3_32_0.msh"},
	{0x08C26FE2B020A69EULL, "mrl000_marethari6129285.cl"},
	{0x08C27BC0DFEDD16BULL, "dae000ip_crowd_mf1_talk11.utp"},
	{0x08C281CA5A48B4FAULL, "w_bonepitdestroyed_a3_32_0.mao"},
	{0x08C285CA5A48BB09ULL, "w_bonepitdestroyed_a3_32_0.mmh"},
	{0x08C2B9DE949CB42EULL, "art\\levels\\races\\natural\\proxy\\props\\prp_zzrocksmall.phy"},
	{0x08C2F64D7BC16CAAULL, "vxht_chantry_36f_0_0.vlm"},
	{0x08C30EA706F03AE8ULL, "mcr370_desten6087319.cl"},
	{0x08C38B9288D8ED2DULL, "qcr221cs_keep_intro.cut"},
	{0x08C3E9007966D3DFULL, "vxlt_0_lowtown_a2_csh_0_0.vlm"},
	{0x08C410D87DDE0D0DULL, "art\\characters\\playercharacter\\elfmale\\cth_coma.phy"},
	{0x08C424885CDA2158ULL, "6024529_m.fxe"},
	{0x08C46994FD830AC4ULL, "vxht_0_hightown_a2_bk6_0_0.vlm"},
	{0x08C46A92FAA82E28ULL, "cs_pro000cs_avaline_intro.bnk"},
	{0x08C4826F97F88D53ULL, "vxht_chantry_n_2lg_0_0.vlm"},
	{0x08C4A5C5686A9EADULL, "vxlt_gallowstemplar_f_6t6_0_0.vlm"},
	{0x08C502D2AF38662AULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\twohanded\\tw_buildup.evt"},
	{0x08C55E4EB694942AULL, "hlac_lt_0_lowtown_docks_n_-65539_6z.dds"},
	{0x08C59A4408D14554ULL, "vxca_caverns2prc_aeu_0_0.vlm"},
	{0x08C5CEEC2265B28DULL, "art\\levels\\races\\dwarf\\props\\_textures\\prp_screen_n.dds"},
	{0x08C659CE4A4F4C19ULL, "6127676_m.fxe"},
	{0x08C6E22300E38C5BULL, "6101227_m.fxe"},
	{0x08C6FEF729BEA01AULL, "6067867_m.fxe"},
	{0x08C744674EE0F5FBULL, "6141616_m.fxe"},
	{0x08C744A84CA2F4BBULL, "hlac_lt_gallowstemplar_n_2391_3x.dds"},
	{0x08C7B8B60BC1137EULL, "ts_templars_acc_props.gda"},
	{0x08C7D71F7C53121EULL, "vxlt_0_lowtown_n_bxz_0_0.vlm"},
	{0x08C8450514AA3820ULL, "6111092_m.fxe"},
	{0x08C89EC795A2886AULL, "ave121_brennan6105926.cl"},
	{0x08C8D56303281991ULL, "amb_scene_dishdrop.wbd"},
	{0x08C93AA8A243A831ULL, "mag101cr_alain.utc"},
	{0x08C93F833A4BDA25ULL, "art\\levels\\races\\proxy\\tilesets\\setext\\8co_basic\\0dd_basic_2.mao"},
	{0x08C96A890AFC5DECULL, "ave110tr_enter_dpatrol.utt"},
	{0x08C9A6ED7FE58C8FULL, "629556143.ogg"},
	{0x08C9D258AA203FC4ULL, "vxlt_gallowstemplar_5ji_0_0.vlm"},
	{0x08CA2D5BE348CF7AULL, "1019429390.ogg"},
	{0x08CA4DC0DF556B70ULL, "vxlt_backalley_g_1pe_0_0.vlm"},
	{0x08CAD14AAF9216E1ULL, "hlac_lt_0_lowtown_docks_196606_0y.dds"},
	{0x08CB3638A6C4D3B0ULL, "6018544_m.fxe"},
	{0x08CB39BEB1E3A184ULL, "494174111.ogg"},
	{0x08CB4F7CB7EF47A5ULL, "6096248_m.fxe"},
	{0x08CB965173FA1183ULL, "vxca_caverns2prc_9le_0_0.vlm"},
	{0x08CBAEB15DFF066AULL, "6178536_m.fxe"},
	{0x08CBC09A49E575F2ULL, "nat380cr_temmerin.utc"},
	{0x08CBE1D2AF3E4C9DULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\twohanded\\tw_buildup.gad"},
	{0x08CC07BF85B0B486ULL, "299148302.ogg"},
	{0x08CC204A40804CDCULL, "6154198_m.fxe"},
	{0x08CC6015E4E57CF9ULL, "al_rogue_talents.gda"},
	{0x08CD121159D0CAD3ULL, "hlac_lt_0_lowtown_docks_a3_-65540_2y.dds"},
	{0x08CD2183D1BCD677ULL, "vxlt_0_lowtown_docks_n_b7s_0_0.vlm"},
	{0x08CD38B1780F4264ULL, "887352302.ogg"},
	{0x08CD555E4D46ABD4ULL, "vxlt_0_lowtown_a2_n_c_hnn_0_0.vlm"},
	{0x08CDDF9236B29C7BULL, "152058437.ogg"},
	{0x08CE14593902D616ULL, "art\\characters\\creatures\\mubari\\animation\\com_cavort.evt"},
	{0x08CE8A7E3DB23225ULL, "vxlt_0_lowtown_a3_n_c_ep0_0_1.vlm"},
	{0x08CEC66B759F2D2BULL, "hlac_w_bonepitmining_a2_5_0x.dds"},
	{0x08CF29B5FA6C3A04ULL, "vxca_cavern1_haunted_ckn_0_0.vlm"},
	{0x08CF94ECA9C81BD0ULL, "art\\characters\\playercharacter\\humanmale\\boo_maso.phy"},
	{0x08D02669DA78D9B4ULL, "43750690.ogg"},
	{0x08D0DCC8C364B6EAULL, "vxht_brothel_n_5cc_0_0.vlm"},
	{0x08D14D5B0B0CA5B3ULL, "6171394_m.fxe"},
	{0x08D1EE95EA8574E3ULL, "hlac_lt_gallowscourtyard_-65537_4z.dds"},
	{0x08D210181C86A14BULL, "family_base.gda"},
	{0x08D24F3C9E9E43F7ULL, "482631263.ogg"},
	{0x08D2AD5806577BDCULL, "gen00fl_fenris6113869.cl"},
	{0x08D2F64548E0D117ULL, "vxlt_0_lowtown_n_eqo_0_0.vlm"},
	{0x08D2FE891E5751ABULL, "6080578_f.fxe"},
	{0x08D30389B0A242A5ULL, "vxlt_0_lowtown_f4z_0_0.vlm"},
	{0x08D3F9C825D194C0ULL, "1045838744.ogg"},
	{0x08D450C10E502355ULL, "6143344_m.fxe"},
	{0x08D4D0823979C1EEULL, "art\\characters\\weapons\\shields\\textures\\w_shd_srdb_0t.dds"},
	{0x08D4D9CA278AC882ULL, "vxht_chantry_n_336_0_0.vlm"},
	{0x08D4E0E3A0D11935ULL, "gen00fl_sebastian6187752.cl"},
	{0x08D5077B25AE268EULL, "follower_barks6153564.cl"},
	{0x08D50B593908E5A1ULL, "art\\characters\\creatures\\mubari\\animation\\com_cavort.gad"},
	{0x08D539BF666062E3ULL, "mcr371_scared_mage_m6086909.cl"},
	{0x08D5CC11AFAD9F66ULL, "6090362_m.fxe"},
	{0x08D5E7CECCE78503ULL, "6128367_m.fxe"},
	{0x08D5F58FC0A1E741ULL, "w_woundedcoast_78_0.rml"},
	{0x08D62E404B808961ULL, "mag131_keran6086265.cl"},
	{0x08D632D3C0BA6773ULL, "lla_ht_0_hightown_1_0.dds"},
	{0x08D64037795A799FULL, "zz_dae_debug6029816.cl"},
	{0x08D64C95E17380B1ULL, "follower_banter6163561.cl"},
	{0x08D793F636B39FDDULL, "6214947_m.fxe"},
	{0x08D7B625A3F68100ULL, "mbv.cut"},
	{0x08D7BA8D41FAEF1CULL, "hlac_lt_0_lowtown_a3_n_c_65536_12y.dds"},
	{0x08D7CD9A9D4B9D81ULL, "6140136_m.fxe"},
	{0x08D83007CBDE1277ULL, "vxca_caverns1_mines_cka_0_0.vlm"},
	{0x08D857FF0764F837ULL, "580427038.ogg"},
	{0x08DA325B808CA097ULL, "820695290.ogg"},
	{0x08DA597E0A84F2EAULL, "509173713.ogg"},
	{0x08DA6F64448F79DDULL, "hlac_lt_0_lowtown_docks_a3_-196610_1y.dds"},
	{0x08DAA289D1745E46ULL, "hm_sib_tobrius.mor"},
	{0x08DAB45FE64A68EBULL, "6068908_m.fxe"},
	{0x08DB030216759C24ULL, "cd_mer141_jansen.bnk"},
	{0x08DBC73B3F3CC454ULL, "vxlt_0_lowtown_a2_n_c_hio_0_0.vlm"},
	{0x08DBDAC990D3A0BDULL, "plt_lgt210pt_qun_blkpwdr.plo"},
	{0x08DC0C59390F064FULL, "art\\characters\\creatures\\mubari\\animation\\com_cavort.ani"},
	{0x08DC5BBCFDEFB1D8ULL, "6080646_m.fxe"},
	{0x08DC67D7F1FAD8EEULL, "art\\characters\\playercharacter\\humanmale\\hm_har_ha16a.phy"},
	{0x08DC811D1A5F12A1ULL, "ser270_cut_aftermath6162278.cl"},
	{0x08DCEFBA5570036BULL, "hlac_lt_0_lowtown_docks_196608_1z.dds"},
	{0x08DCFA84A0B73743ULL, "art\\characters\\creatures\\stonegolem\\animation\\rnf.ani"},
	{0x08DD956AAAC2FCA6ULL, "vxht_keep_73q_0_0.vlm"},
	{0x08DD963E189FE766ULL, "6126809_m.fxe"},
	{0x08DD9FB815286BF9ULL, "gen_im_trs_dal_hvy_cht_r2.uti"},
	{0x08DDA5A5543F141DULL, "vxlt_gallowstemplar_f_5k8_0_0.vlm"},
	{0x08DDCF3ED3C2BE84ULL, "vxht_keep_n_709_0_0.vlm"},
	{0x08DDE99A64065148ULL, "472745481.ogg"},
	{0x08DE6993A64EE285ULL, "384732723.ogg"},
	{0x08DEC1E9EAA2C6ADULL, "gen00fl_aveline6121992.cl"},
	{0x08DEE47DDDAAF8A8ULL, "gen00fl_fenris6113906.cl"},
	{0x08DFADE598969467ULL, "6106873_m.fxe"},
	{0x08DFBF37FD02BE33ULL, "6174094_m.fxe"},
	{0x08DFCB9B661D6C52ULL, "character_character_p50.psh.fxo.dx11.assembly"},
	{0x08E009553EF5F6D0ULL, "95817530.ogg"},
	{0x08E0159EE5DE1EB6ULL, "vxlt_0_lowtown_docks_a3_cyq_0_0.vlm"},
	{0x08E03775D091177FULL, "gen00fl_isabela6090321.cl"},
	{0x08E0C9CEAD0E22ACULL, "art\\characters\\playercharacter\\humanfemale\\arm_nuda.msh"},
	{0x08E0CFCEAD0E2C9EULL, "art\\characters\\playercharacter\\humanfemale\\arm_nuda.mmh"},
	{0x08E0DBCEAD0E4135ULL, "art\\characters\\playercharacter\\humanfemale\\arm_nuda.mao"},
	{0x08E11284F1C7C5C4ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\_textures\\lt_temp_rock_n.dds"},
	{0x08E1DDC990D811EAULL, "plt_lgt210pt_qun_blkpwdr.nss"},
	{0x08E26D08A4663622ULL, "one300_zevran6071479.cl"},
	{0x08E29DE070D5FD64ULL, "vxlt_gallowstemplar_n_2mx_0_0.vlm"},
	{0x08E2F75DF1157FAFULL, "hlac_lt_0_lowtown_a2_65535_4z.dds"},
	{0x08E3092FA63E9920ULL, "art\\vfx\\materials\\vfx_flare01a_add.mao"},
	{0x08E333F44EFCB9F4ULL, "6088839_m.fxe"},
	{0x08E352E7D2EBF215ULL, "vxw_bonepitmining_a2_1oa_0_0.vlm"},
	{0x08E37A61BEE1AE8CULL, "179992175.ogg"},
	{0x08E458F16E8C4D3BULL, "art\\characters\\weapons\\longswords\\w_lsw_ls103a.phy"},
	{0x08E45A5D6B0C9A5CULL, "995673915.ogg"},
	{0x08E481C345E8F87DULL, "vxlt_0_lowtown_docks_a3_n_bsn_0_0.vlm"},
	{0x08E4F2461097A36DULL, "vxca_caverns2_96l_0_0.vlm"},
	{0x08E4F5C5C1B09FC5ULL, "vxca_caverns1_mines_ehx_0_0.vlm"},
	{0x08E507916B69FA48ULL, "849999264.ogg"},
	{0x08E53D38122A0F90ULL, "hlac_ht_0_hightown_131074_0y.dds"},
	{0x08E59A7A27339F35ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\0ff_quarterd.msh"},
	{0x08E59C7A2733A2ABULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\0ff_quarterd.mmh"},
	{0x08E5A87A2733B748ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\0ff_quarterd.mao"},
	{0x08E608C09A437C23ULL, "vxlt_gallowscourtyard_n_7dk_0_0.vlm"},
	{0x08E67FE9952E1D59ULL, "dae021_bureaucrat_1.utc"},
	{0x08E68BB00EE229C2ULL, "hlac_lt_gallowstemplar_f_2383_11y.dds"},
	{0x08E6B6128E0C70C4ULL, "vxlt_gallowstemplar_5po_0_0.vlm"},
	{0x08E6E0E23EBEF8F7ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\9ff_manhole.mmh"},
	{0x08E6E4E23EBEFF04ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\9ff_manhole.mao"},
	{0x08E6F6E23EBF1E69ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\9ff_manhole.msh"},
	{0x08E7317AC3419C70ULL, "vxlt_backalley_g_1zc_0_0.vlm"},
	{0x08E79DC3103202D9ULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\0ww_galow2bwin.msh"},
	{0x08E7A0BE29932393ULL, "gen00fl_varric6088830.cl"},
	{0x08E7A7C3103213C7ULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\0ww_galow2bwin.mmh"},
	{0x08E7ABC310321A14ULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\0ww_galow2bwin.mao"},
	{0x08E7DD56C8FED284ULL, "6068464_m.fxe"},
	{0x08E82C6E169B0534ULL, "6085928_m.fxe"},
	{0x08E884A87370F222ULL, "vxht_0_hightown_a3_b4p_0_0.vlm"},
	{0x08E8B16DEE358294ULL, "l_rdr370ar_finale.bnk"},
	{0x08E99B84180EE28CULL, "6076860_f.fxe"},
	{0x08E9AA1F04E6F861ULL, "ss_circlemages_f_1.bnk"},
	{0x08E9D83F7CCDB45EULL, "18872200.ogg"},
	{0x08EA333A8836E423ULL, "6169193_m.fxe"},
	{0x08EA5C92FAC8E5D2ULL, "cs_pro000cs_avaline_intro.wbd"},
	{0x08EB0EE03363D3BFULL, "718107386.ogg"},
	{0x08EB1284A0C39D3AULL, "art\\characters\\creatures\\stonegolem\\animation\\rnf.evt"},
	{0x08EBD960BD0862FDULL, "mage_forceresistance.dds"},
	{0x08EC441951C23B8AULL, "jagged_rock.mao"},
	{0x08ECCD7E8C2C6ECCULL, "dae100_gamlen_letter6146996.cl"},
	{0x08ED0E8698CEBDD5ULL, "vxlt_warehouse_n_z9_0_0.vlm"},
	{0x08ED8C7DE86BDDC7ULL, "hlac_ht_0_hightown_a3_-65537_2z.dds"},
	{0x08EDD6F4B041C6FFULL, "hf_cgn_006.mop"},
	{0x08EE101BBBD88E04ULL, "hlac_lt_0_lowtown_docks_n_-4_5y.dds"},
	{0x08EE84ADC485AF0BULL, "vxht_0_hightown_a2_akj_0_0.vlm"},
	{0x08EE978950878757ULL, "hlac_lt_0_lowtown_a2_n_-131071_1z.dds"},
	{0x08EEB5E17A94117AULL, "sum_conj_03.mao"},
	{0x08EF3C575E6E24B8ULL, "6173085_m.fxe"},
	{0x08EF71F37BA1D91EULL, "gen_im_trs_krk_g_dagger.uti"},
	{0x08EFBAA8EF39F642ULL, "and111_refugee_barks6111540.cl"},
	{0x08EFC44FEBC982CDULL, "lt_0_lowtown_docks_n_112_0.mmh"},
	{0x08EFC80161DFE5A3ULL, "face_face_p19.psh.fxo.dx11"},
	{0x08EFC84FEBC9891EULL, "lt_0_lowtown_docks_n_112_0.mao"},
	{0x08EFD64FEBC9A153ULL, "lt_0_lowtown_docks_n_112_0.msh"},
	{0x08F001AB8E82B89DULL, "w_encounter1_56_0.mao"},
	{0x08F005AB8E82BFD6ULL, "w_encounter1_56_0.mmh"},
	{0x08F00FAB8E82D0D4ULL, "w_encounter1_56_0.msh"},
	{0x08F04A7940120BE2ULL, "6160959_m.fxe"},
	{0x08F0733AEFCEB895ULL, "follower_banter6169475.cl"},
	{0x08F079CDDC543563ULL, "6074372_m.fxe"},
	{0x08F08581B4637BC2ULL, "759626661.ogg"},
	{0x08F09A4C3FFDAD0EULL, "33704797.ogg"},
	{0x08F0C6FED60678DCULL, "685754883.ogg"},
	{0x08F11184A0C8072DULL, "art\\characters\\creatures\\stonegolem\\animation\\rnf.gad"},
	{0x08F181A87D3E68CAULL, "294247523.ogg"},
	{0x08F1A5D140F2D274ULL, "6083939_m.fxe"},
	{0x08F2F3A037DF10FDULL, "vxlt_0_lowtown_docks_a3_n_bvm_0_0.vlm"},
	{0x08F3CCE06EA9132CULL, "ener_lit.mao"},
	{0x08F3DBC366DAB670ULL, "vxht_chantry_n_353_0_1.vlm"},
	{0x08F3FA6795E0BF4AULL, "art\\characters\\creatures\\mubari\\animation\\rn_p_lf.gad"},
	{0x08F4EBE2EBF282EBULL, "hlac_lt_gallowscourtyard_-65536_3x.dds"},
	{0x08F4F22958F89AEFULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\_textures\\vst_channelwater_n.dds"},
	{0x08F5E9B764FABE87ULL, "38979494.ogg"},
	{0x08F5E9FEAD9CC1ECULL, "242997766.ogg"},
	{0x08F6A87BD823A34EULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\9ss_trampulley_2.mao"},
	{0x08F807D59E87B0BCULL, "37313870.ogg"},
	{0x08F80F99D8491D06ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\isabela_post_sex_action.evt"},
	{0x08F811E6437A9F0FULL, "and340ip_brimstone.utp"},
	{0x08F81464DAF616D3ULL, "and231cr_lyrium_smuggler.utc"},
	{0x08F8AAA62DC1BDD1ULL, "qun221_petrice_temple.stg"},
	{0x08F8F24CF2E1B36EULL, "595816543.ogg"},
	{0x08F93E0C0D8A0A3AULL, "llac_lt_0_lowtown_a2_n_-1_2.dds"},
	{0x08F97E53755DF891ULL, "vxht_brothel_n_6dr_0_0.vlm"},
	{0x08F985C21204A547ULL, "12002886.ogg"},
	{0x08FA3F4353D5F023ULL, "290116499.ogg"},
	{0x08FADD6795E6ACE1ULL, "art\\characters\\creatures\\mubari\\animation\\rn_p_lf.evt"},
	{0x08FB127FE8EF8838ULL, "follower_banter6155769.cl"},
	{0x08FB334C437AD544ULL, "vxlt_0_lowtown_a2_hg4_0_0.vlm"},
	{0x08FB3B9E8BE6CC7FULL, "hlac_lt_gallowscourtyard_n_-1_0x.dds"},
	{0x08FBE4D72924C2FCULL, "hlac_ca_cavern1_haunted_24422_4z.dds"},
	{0x08FBF9CC9A0F7C1EULL, "453353414.ogg"},
	{0x08FC31F0FAD06DD3ULL, "hf_ave1_raider_assassin.mor"},
	{0x08FC749944EFB73EULL, "702908570.ogg"},
	{0x08FD9AD1E2463BD7ULL, "pow_in_clo.mao"},
	{0x08FE55B71DE4BFE9ULL, "vxlt_0_lowtown_byq_0_0.vlm"},
	{0x08FEBC5632082462ULL, "6165659_m.fxe"},
	{0x08FF23155590F382ULL, "6195876_m.fxe"},
	{0x08FF25A09F523589ULL, "6201224_m.fxe"},
	{0x08FF2699D84F6351ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\isabela_post_sex_action.gad"},
	{0x08FF36EC69394A08ULL, "228146478.ogg"},
	{0x08FF92A29E15B289ULL, "6135774_m.fxe"},
	{0x08FFAB7B4AB27CE8ULL, "366330823.ogg"},
	{0x08FFD101D5D9E922ULL, "follower_banter6148736.cl"},
	{0x08FFFC4CBB81C489ULL, "art\\vfx\\combat\\vfx_glb_arw_ice_c.cess.ani"},
	{0x090017AE32E8EAF4ULL, "6086639_m.fxe"},
	{0x090023AE9F31206EULL, "vxlt_gallowstemplar_n_3o1_0_0.vlm"},
	{0x0900C74E22A9541CULL, "vxlt_gallowscourtyard_6xr_0_0.vlm"},
	{0x0900F53F14B031D9ULL, "6064722_m.fxe"},
	{0x09010C4B27621E51ULL, "art\\characters\\weapons\\greatswords\\textures\\w_gsw_gs107a_0d.dds"},
	{0x09016C708CC238D9ULL, "6087228_m.fxe"},
	{0x0902B8DE94D296F1ULL, "art\\levels\\races\\natural\\proxy\\props\\prp_zzrocksmall.mmh"},
	{0x0902BF1030852DA9ULL, "ser221_gamlen6162352.cl"},
	{0x0902C4DE94D2AA92ULL, "art\\levels\\races\\natural\\proxy\\props\\prp_zzrocksmall.mao"},
	{0x0902D2DE94D2C2AFULL, "art\\levels\\races\\natural\\proxy\\props\\prp_zzrocksmall.msh"},
	{0x0902D75D567EB0C1ULL, "vxht_keep_64b_0_0.vlm"},
	{0x0902D84FFFCE0D9EULL, "131265196.ogg"},
	{0x090305A49349490CULL, "hlac_lt_0_lowtown_docks_-131075_0z.dds"},
	{0x09031E7AA4867CE3ULL, "6168042_m.fxe"},
	{0x0903277524E03988ULL, "6123350_m.fxe"},
	{0x09037195AF232B29ULL, "art\\vfx\\combat\\vfx_m_pri_cone_cold_p.impa.ani"},
	{0x09046D75DB1DB18CULL, "mer320_hubert6090238.cl"},
	{0x09048F344068900CULL, "6217099_m.fxe"},
	{0x0904ABD2D0AC6882ULL, "55650100.ogg"},
	{0x0905053BD616D4C2ULL, "vxht_mansion_n_6ma_0_0.vlm"},
	{0x09052799D853D0FFULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\isabela_post_sex_action.ani"},
	{0x09054072D2211B65ULL, "l_qun101ar_talvashoth_cave.wbd"},
	{0x0907174CBB88110BULL, "art\\vfx\\combat\\vfx_glb_arw_ice_c.cess.gad"},
	{0x090733D8EABEAF84ULL, "922868106.ogg"},
	{0x0907F03067541111ULL, "vxht_brothel_n_6aq_0_0.vlm"},
	{0x0907F4B65F43DB42ULL, "gen_im_wep_mag_sta_act0_02.uti"},
	{0x090822556B6B0934ULL, "gen00fl_isabela6075299.cl"},
	{0x09089C555617253FULL, "6078354_m.fxe"},
	{0x0908C5C4EA308FB1ULL, "fonts_pseudo.gfx"},
	{0x0908D56795F2DC30ULL, "art\\characters\\creatures\\mubari\\animation\\rn_p_lf.ani"},
	{0x0908E08B67F3C4B3ULL, "347571273.ogg"},
	{0x09091EE6438928BAULL, "and340ip_brimstone.ncs"},
	{0x09097334E79DCC00ULL, "hlac_ca_caverns2_5802_38x.dds"},
	{0x0909DBB579B2B3DFULL, "da2terrain_terrain_tools_p1.psh.fxo"},
	{0x090A1B393C27EDF5ULL, "6170319_m.fxe"},
	{0x090A8C95AF2977ABULL, "art\\vfx\\combat\\vfx_m_pri_cone_cold_p.impa.gad"},
	{0x090B0494A2454FEDULL, "6180095_m.fxe"},
	{0x090BCC6AFB7FFA56ULL, "gen00fl_isabela6196859.cl"},
	{0x090C1E64EC31A994ULL, "714964153.ogg"},
	{0x090C3D191D8AEC79ULL, "art\\levels\\races\\natural\\kirkcoastal\\foliage\\_textures\\tre_htwood_sa.dds"},
	{0x090C783066127B94ULL, "6128276_m.fxe"},
	{0x090CB26149DE4171ULL, "102010252.ogg"},
	{0x090CDE37553FAD92ULL, "211618669.ogg"},
	{0x090CFA95B58B2DEAULL, "art\\characters\\creatures\\dragon\\animation\\p_wkf.gad"},
	{0x090D58B5AFFC349FULL, "560690973.ogg"},
	{0x090E144CBB8E2B70ULL, "art\\vfx\\combat\\vfx_glb_arw_ice_c.cess.evt"},
	{0x090E3EBC81E83C11ULL, "mrl351cr_faketamlen.utc"},
	{0x090E4FFCD9D06A9CULL, "art\\characters\\playercharacter\\shared\\heads\\textures\\mt_tat_d04_0t.dds"},
	{0x090EF1E3A1CDC6ACULL, "43165186.ogg"},
	{0x090F14B8F51E8434ULL, "vxlt_gamlenshouse_n_2v_0_0.vlm"},
	{0x090F85729E111483ULL, "6175152_m.fxe"},
	{0x090FA1933AE96E41ULL, "264958685.ogg"},
	{0x091038DAD0C27FA6ULL, "vxht_0_hightown_n_56w_0_0.vlm"},
	{0x09103FF16EB15DDAULL, "art\\characters\\weapons\\longswords\\w_lsw_ls103a.msh"},
	{0x09104B5C9792E945ULL, "vxca_caverns1_mines_ege_0_0.vlm"},
	{0x091051F16EB17C20ULL, "art\\characters\\weapons\\longswords\\w_lsw_ls103a.mmh"},
	{0x09107FF8C737BAE9ULL, "art\\characters\\creatures\\crowds\\humanmale\\animation\\exodus\\standing_neutral_gestures_1_sylvesta.ani"},
	{0x09108995AF2DDE10ULL, "art\\vfx\\combat\\vfx_m_pri_cone_cold_p.impa.evt"},
	{0x091120DC55ED98BCULL, "hlac_ca_caverns1_mines_24421_0y.dds"},
	{0x0911493A4874C941ULL, "vxht_keep_61a_0_0.vlm"},
	{0x0911876775619ABCULL, "161326788.ogg"},
	{0x0911CE1C87EBEF8AULL, "6153342_m.fxe"},
	{0x091254BDFE4FD108ULL, "art\\levels\\races\\proxy\\props\\rugs_poor\\prp_zzlargegreen.mmh"},
	{0x091260BDFE4FE5ABULL, "art\\levels\\races\\proxy\\props\\rugs_poor\\prp_zzlargegreen.mao"},
	{0x091272BDFE500432ULL, "art\\levels\\races\\proxy\\props\\rugs_poor\\prp_zzlargegreen.msh"},
	{0x091294076A3A60E2ULL, "gen00fl_varric6076094.cl"},
	{0x0912B4BAEC579FACULL, "zz_dre_debug6025524.cl"},
	{0x0912F646ED821C47ULL, "vxht_mansion2b_3bk_0_0.vlm"},
	{0x0913144F063A2EB0ULL, "318382847.ogg"},
	{0x0913674C743D88FEULL, "71061690.ogg"},
	{0x091381D9B81DF811ULL, "dae100_alistair_drunk6135156.cl"},
	{0x0913CF55BDB3EB2BULL, "dre000_dougal6142217.cl"},
	{0x0913DD95B5911A81ULL, "art\\characters\\creatures\\dragon\\animation\\p_wkf.evt"},
	{0x0913E4B5A03E9FF7ULL, "6118027_m.fxe"},
	{0x09147011C55021F7ULL, "zz_dae_debug6049946.cl"},
	{0x091490C31058DAFCULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\0ww_galow2bwin.phy"},
	{0x0914D135921B2C65ULL, "vxw_bonepitmining_a2_1o1_0_0.vlm"},
	{0x09152E2A3F1341D1ULL, "hlac_dr_0_deeproads_5843_1x.dds"},
	{0x091565CBD10D7B82ULL, "hlac_lt_0_lowtown_docks_a3_196608_1z.dds"},
	{0x09158EAF07152654ULL, "vxlt_0_lowtown_a2_n_c_hus_0_0.vlm"},
	{0x09159CA643BC1DF5ULL, "art\\levels\\races\\natural\\kirkcoastal\\terrainmats\\ter_coastgravel_n.dds"},
	{0x09159EBA607DA9AAULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\turn_right_180.gad"},
	{0x0915A03105B6AF78ULL, "plt_and000pt_end_stage.nss"},
	{0x0915E5D5E91B2DF2ULL, "vxca_cavern1_haunted_gk4_0_0.vlm"},
	{0x0916085350C7A0F0ULL, "462820969.ogg"},
	{0x091624E1248DA001ULL, "hlac_lt_0_lowtown_docks_a3_n_-2_3y.dds"},
	{0x0916620D594A2991ULL, "vxht_brothel_n_6jp_0_0.vlm"},
	{0x091689A935947EB3ULL, "art\\levels\\races\\proxy\\props\\4tools\\inv_flat2x4.mmh"},
	{0x09169A120545D868ULL, "vxlt_0_lowtown_a3_n_c_hli_0_0.vlm"},
	{0x0916A26030A19705ULL, "seb_module.dds"},
	{0x0916B785A8F3BD93ULL, "mer320_hubert6089916.cl"},
	{0x0916E78FC0D96630ULL, "w_woundedcoast_78_0.msh"},
	{0x0916F58FC0D97E4AULL, "w_woundedcoast_78_0.mmh"},
	{0x0916F98FC0D98481ULL, "w_woundedcoast_78_0.mao"},
	{0x09179AF8C73E076BULL, "art\\characters\\creatures\\crowds\\humanmale\\animation\\exodus\\standing_neutral_gestures_1_sylvesta.gad"},
	{0x0918A950422363D5ULL, "vfx_chunkyparticle_p2.vsh.fxo.dx11.assembly"},
	{0x0918F81414B8738EULL, "vxlt_0_lowtown_a3_n_c_hqh_0_0.vlm"},
	{0x0919755BAB640D0BULL, "gen00fl_aveline6140807.cl"},
	{0x091980620E9724DFULL, "692639367.ogg"},
	{0x09198719195A4ED0ULL, "6135075_f.fxe"},
	{0x0919D03F542F7C8AULL, "803181148.ogg"},
	{0x091A2F43FDBAB068ULL, "984050000.ogg"},
	{0x091A633AF78EE314ULL, "63339503.ogg"},
	{0x091A860B059BCD2DULL, "165680980.ogg"},
	{0x091B814006639EB9ULL, "zz_dae_debug6029807.cl"},
	{0x091C81BA60839741ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\turn_right_180.evt"},
	{0x091D66D7F2326DB1ULL, "art\\characters\\playercharacter\\humanmale\\hm_har_ha16a.mmh"},
	{0x091D80D7F2329A6FULL, "art\\characters\\playercharacter\\humanmale\\hm_har_ha16a.msh"},
	{0x091DD9491335088EULL, "mag170_thrask6054013.cl"},
	{0x091DFB20A12B2D2AULL, "6144082_f.fxe"},
	{0x091F05DFC61BCE1FULL, "hlac_lt_gamlenshouse_n_-1_3x.dds"},
	{0x091F33335018ADABULL, "hlac_w_sundermount_-65539_0y.dds"},
	{0x091FBCAB6A8CECE0ULL, "ser191_emeric6104714.cl"},
	{0x091FF81CB459DDF4ULL, "gen00fl_anders6128380.cl"},
	{0x09203FF979E2080AULL, "6156042_m.fxe"},
	{0x0920965050E39758ULL, "6160764_m.fxe"},
	{0x09209661E06BCE63ULL, "w_blightlands_17_0.msh"},
	{0x0920A461E06BE5BDULL, "w_blightlands_17_0.mmh"},
	{0x0920A861E06BECCEULL, "w_blightlands_17_0.mao"},
	{0x0920FB0DC39C3C43ULL, "hlac_lt_0_lowtown_docks_a3_65534_3z.dds"},
	{0x0921A6CEAD457E39ULL, "art\\characters\\playercharacter\\humanfemale\\arm_nuda.phy"},
	{0x0921AA8AD9E1C2E3ULL, "art\\levels\\races\\dwarf\\props\\spider\\prp_4webdetail.mao"},
	{0x0921AE8AD9E1C930ULL, "art\\levels\\races\\dwarf\\props\\spider\\prp_4webdetail.mmh"},
	{0x0921BC8AD9E1E14AULL, "art\\levels\\races\\dwarf\\props\\spider\\prp_4webdetail.msh"},
	{0x0921D595B59D4AD0ULL, "art\\characters\\creatures\\dragon\\animation\\p_wkf.ani"},
	{0x0921DCBE56A67AA2ULL, "6129847_f.fxe"},
	{0x0922F18CA7E9AB39ULL, "gen00fl_bethany6101199.cl"},
	{0x09235798B3956E42ULL, "vxlt_undercitydungeon_159_0_0.vlm"},
	{0x0923AE6CFA1ED6E5ULL, "6133787_m.fxe"},
	{0x0923BE551260B5D2ULL, "gen00fl_aveline6106950.cl"},
	{0x0923C8CF3FC9CA13ULL, "325931453.ogg"},
	{0x09240A7269EAEB64ULL, "dae000cr_grand_cleric6187933.cl"},
	{0x09242578B7FA9FC2ULL, "317868142.ogg"},
	{0x0924358DEA88285FULL, "hlac_lt_0_lowtown_a2_n_1_0x.dds"},
	{0x09245E78556B5EE8ULL, "lt_0_lowtown_docks_a3_209_0.msh"},
	{0x09246F5A32AA3CAFULL, "475785724.ogg"},
	{0x09247078556B7D79ULL, "lt_0_lowtown_docks_a3_209_0.mao"},
	{0x09247C78556B9112ULL, "lt_0_lowtown_docks_a3_209_0.mmh"},
	{0x0925C2496640E865ULL, "art\\levels\\races\\elf\\props\\_textures\\prp_hanginga_n.dds"},
	{0x0925C62023AF5749ULL, "gen00fl_anders6136018.cl"},
	{0x0925F3C5CEE21E6EULL, "594645931.ogg"},
	{0x09264AB000F83BFFULL, "600476901.ogg"},
	{0x0926E03C22253DE1ULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\4so_rightkeep_3.mao"},
	{0x0926F7C911960A0AULL, "vxht_keep_n_64b_0_0.vlm"},
	{0x0927113EBEF251EFULL, "301848913.ogg"},
	{0x09271273FFB00AD2ULL, "6077040_m.fxe"},
	{0x0928C64E931EC094ULL, "hlac_lt_darktown_-1_53x.dds"},
	{0x0928CDADF149A2E2ULL, "179264182.ogg"},
	{0x09290E925C7E9E91ULL, "vxw_sundermount_20b_0_0.vlm"},
	{0x092979BA608E1390ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\turn_right_180.ani"},
	{0x09299DB5C9D5F11AULL, "543998019.ogg"},
	{0x0929CB6DEE6D4646ULL, "l_rdr370ar_finale.wbd"},
	{0x092A5204097BC7BBULL, "art\\vfx\\combat\\vfx_c_mer_grab_c.dura.evt"},
	{0x092AD1426CE72E73ULL, "hlac_lt_0_lowtown_65538_2y.dds"},
	{0x092B542C2CA8987FULL, "326166667.ogg"},
	{0x092B65115441C8A8ULL, "33490386.ogg"},
	{0x092B8C388CBA6C95ULL, "vxw_bonepitmining_a2_1pl_0_0.vlm"},
	{0x092C181A7A0B2667ULL, "vxw_sundermount_2cq_0_0.vlm"},
	{0x092DE4AE9AB73EF1ULL, "character_character_p257.psh.fxo.dx11.assembly"},
	{0x092E06A2DBA0C200ULL, "mcr371_post_harvester6077595.cl"},
	{0x092E1701A69C9589ULL, "6117675_m.fxe"},
	{0x092E2D96EF3BB149ULL, "vxlt_hangedman_n_229_0_0.vlm"},
	{0x092E30203BEB0B51ULL, "589952063.ogg"},
	{0x092E765A4823B229ULL, "162721642.ogg"},
	{0x092EA162D5A6911CULL, "vxlt_darktown_5ko_0_1.vlm"},
	{0x092F6C7ECBDA20C7ULL, "332940664.ogg"},
	{0x092FDEB239CC94E4ULL, "6016946_m.fxe"},
	{0x09304B04098027FCULL, "art\\vfx\\combat\\vfx_c_mer_grab_c.dura.gad"},
	{0x0930A331EDB4037AULL, "6163280_m.fxe"},
	{0x0930B81F05231213ULL, "ss_circlemages_f_1.wbd"},
	{0x093133282E56F44CULL, "134779904.ogg"},
	{0x093146B6AE7FC6ECULL, "vxht_keep_63z_0_0.vlm"},
	{0x0931913F7DC3E3B8ULL, "6092128_m.fxe"},
	{0x093269E4C70B8623ULL, "art\\cinematics\\follower plots\\sebastian\\cs_hench1f_sebastianintro.ani"},
	{0x0932AE148F48B8F5ULL, "492131645.ogg"},
	{0x0932BD01D51C799CULL, "6067810_m.fxe"},
	{0x093327891123DB07ULL, "art\\levels\\races\\mage\\prop\\_textures\\prp_sepulcher_n.dds"},
	{0x09334C9662E492BDULL, "hlac_ht_0_hightown_a3_-65536_0y.dds"},
	{0x09334F2604B1DD84ULL, "hlac_lt_0_lowtown_n_196607_0y.dds"},
	{0x093350034A6013F7ULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_scraplight_dp.dds"},
	{0x0933586A9FFD5511ULL, "hlac_ht_keep_3732_16x.dds"},
	{0x093399ED33AB7F2AULL, "static_static_p18.psh.fxo"},
	{0x0933BB6069D49A0CULL, "vxht_0_hightown_a3_bi9_0_0.vlm"},
	{0x093419526302D12EULL, "277671422.ogg"},
	{0x0934D2AF8333AA0EULL, "fen210_dying_magister6108617.cl"},
	{0x09354C06C7933A43ULL, "gen00fl_aveline6139519.cl"},
	{0x093562222FEB2F00ULL, "6172279_m.fxe"},
	{0x093663E10F24A0DFULL, "87844684.ogg"},
	{0x09374A04098644FAULL, "art\\vfx\\combat\\vfx_c_mer_grab_c.dura.ani"},
	{0x0937784675EC83A6ULL, "6125703_m.fxe"},
	{0x0937806F4E74B711ULL, "vxw_sundermount_25a_0_0.vlm"},
	{0x0937C4F37462E24AULL, "zz_mcr_debug6088425.cl"},
	{0x0937FE69F863063EULL, "vxdr_0_deeproads_a27_0_0.vlm"},
	{0x09382B39962FEDCAULL, "6157193_m.fxe"},
	{0x093890478465264AULL, "277609582.ogg"},
	{0x0938C9DB571203ABULL, "vxht_bartrand_n_68l_0_0.vlm"},
	{0x09392CB8B7FBC49CULL, "881418220.ogg"},
	{0x0939BFE96F29212BULL, "art\\characters\\weapons\\staff\\textures\\w_stf_010a_0t.dds"},
	{0x0939D8FED66C898DULL, "468557509.ogg"},
	{0x093A89F76C013EE7ULL, "vxw_sundermount_2dp_0_0.vlm"},
	{0x093AB078F709FF82ULL, "hlac_ht_chantry_n_2534_42z.dds"},
	{0x093B8E9012C2C1C3ULL, "6149371_m.fxe"},
	{0x093BA856F8679C59ULL, "vxlt_0_lowtown_n_h29_0_0.vlm"},
	{0x093BE53CF33F1F5EULL, "gen00fl_aveline6120819.cl"},
	{0x093C5C498A44BCEEULL, "vxlt_undercityencounter_rr_0_0.vlm"},
	{0x093C7DCC59BE2FC5ULL, "6120857_m.fxe"},
	{0x093CA45158D5725AULL, "vxht_playershouse_n_1qk_0_0.vlm"},
	{0x093D4D7DF6401732ULL, "dae000cr_grand_cleric.utc"},
	{0x093DF691DBB39A1DULL, "vxht_bartrand_n_3i4_0_0.vlm"},
	{0x093DFB7393D174E0ULL, "dae210_gate_guard6084319.cl"},
	{0x093E42175CCFD9A4ULL, "art\\characters\\weapons\\longbows\\w_lbw_lb106.mao"},
	{0x093E64D77A91180CULL, "w_bonepitdestroyed_a3_66_0.msh"},
	{0x093E6AD77A9122FEULL, "w_bonepitdestroyed_a3_66_0.mmh"},
	{0x093E76D77A913695ULL, "w_bonepitdestroyed_a3_66_0.mao"},
	{0x093F68977E890CD2ULL, "6115509_f.fxe"},
	{0x0940031C6FB9FC38ULL, "6097228_m.fxe"},
	{0x09406F5EC0348183ULL, "cd_mage110cs_cut_abomination.wbd"},
	{0x094173AAF0043D70ULL, "hlac_ht_brothel_n_3369_0z.dds"},
	{0x0942CC1FE2635FAAULL, "383236737.ogg"},
	{0x0942E03DD629FBF8ULL, "hlac_ca_caverns1_24425_8z.dds"},
	{0x0942ED5E747F37D0ULL, "vxlt_darktown_2xx_0_0.vlm"},
	{0x09438CBC0A8A1B58ULL, "hlac_ca_caverns2_5802_3z.dds"},
	{0x0943CE49EBEFBB23ULL, "art\\characters\\playercharacter\\elfmale\\textures\\em_rob_appd_0s.dds"},
	{0x09446B3F9CE4A5DAULL, "162602421.ogg"},
	{0x09451A72D256BE0FULL, "l_qun101ar_talvashoth_cave.bnk"},
	{0x09453221EAD8641BULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\dialogue\\lean_back_wall_loop.ani"},
	{0x09455DCEA01A9FA0ULL, "vxht_chantry_n_27q_0_0.vlm"},
	{0x09455E43B3146D07ULL, "6106936_m.fxe"},
	{0x09455F9630667FC2ULL, "zz_followers_banter6158937.cl"},
	{0x09456F3A6C272C51ULL, "vxht_bartrand_n_6ii_0_0.vlm"},
	{0x0945BC7CCD3215B0ULL, "6120015_f.fxe"},
	{0x0945E0C67A986A9EULL, "6072119_m.fxe"},
	{0x09463EA7681E4996ULL, "follower_barks6153821.cl"},
	{0x09464C67B5E005C6ULL, "6162254_m.fxe"},
	{0x09464F8F8CAFDC7CULL, "6167215_m.fxe"},
	{0x0946A2D692E83609ULL, "vxdr_primevaltomb_86g_0_0.vlm"},
	{0x094758961D61F27BULL, "art\\levels\\races\\ferelden\\props\\lowerclass\\prp_cookerpots.msh"},
	{0x09476A961D6210C6ULL, "art\\levels\\races\\ferelden\\props\\lowerclass\\prp_cookerpots.mao"},
	{0x094776961D622565ULL, "art\\levels\\races\\ferelden\\props\\lowerclass\\prp_cookerpots.mmh"},
	{0x094780E4C71E090DULL, "art\\cinematics\\follower plots\\sebastian\\cs_hench1f_sebastianintro.gad"},
	{0x094793DA80105E24ULL, "vxlt_0_lowtown_a2_fip_0_0.vlm"},
	{0x09481FA3A170784BULL, "art\\levels\\races\\design\\_textures\\prp_herbs_dp.dds"},
	{0x09483CF82670E58AULL, "6189370_m.fxe"},
	{0x09485F17FAAA44D9ULL, "605983637.ogg"},
	{0x0948678B045E13BCULL, "hlac_ca_caverns1_24426_11y.dds"},
	{0x0948FBD45DF75767ULL, "vxw_sundermount_2is_0_0.vlm"},
	{0x09494844A792FDFBULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\2ca_basic.phy"},
	{0x0949489BA03A8451ULL, "static_static_p4.psh.fxo.dx11.assembly"},
	{0x09497760E60A722EULL, "vxlt_darktown_5bw_0_0.vlm"},
	{0x0949A4DDC7DC78C2ULL, "vxdr_primevaltomb_vz_0_0.vlm"},
	{0x094A4125FD614E1DULL, "vxht_0_hightown_a3_cbh_0_0.vlm"},
	{0x094A53913EC395F7ULL, "6140615_m.fxe"},
	{0x094B162E4ACB8ADAULL, "vxht_playershouse_n_1td_0_0.vlm"},
	{0x094B46194D377268ULL, "309207643.ogg"},
	{0x094B69AD9268DBE0ULL, "hlac_lt_0_lowtown_a2_-65537_2z.dds"},
	{0x094CA362970ED71EULL, "vxht_playershouse_n_1o5_0_0.vlm"},
	{0x094CBAC91441DA9BULL, "character_character_p105.psh.fxo.dx11.assembly"},
	{0x094CC2FE71F23702ULL, "229209327.ogg"},
	{0x094D186BFA341D89ULL, "vxlt_gallowscourtyard_7hr_0_0.vlm"},
	{0x094D9C4CD6929788ULL, "857409106.ogg"},
	{0x094DD51128DC6C1BULL, "hlac_lt_0_lowtown_docks_196607_2z.dds"},
	{0x094DEDC749CFA753ULL, "gen00fl_aveline6132864.cl"},
	{0x094E8006A5BEB57DULL, "vxht_bartrand_n_5pi_0_0.vlm"},
	{0x094EA6624C9D0CA7ULL, "art\\vfx\\combat\\vfx_w_wns_left_b_close1_c.anb"},
	{0x094F5B0AC3198F21ULL, "421065555.ogg"},
	{0x094F7A078C8BFDF4ULL, "sky_w_blightlands.mao"},
	{0x09513F2D79A09949ULL, "6041543_m.fxe"},
	{0x09514704D4C5B219ULL, "art\\vfx\\combat\\vfx_m_pri_tempest_aoe_p.impa.ani"},
	{0x095195171BE0BCB6ULL, "48012353.ogg"},
	{0x0952042F9F725F1CULL, "llac_lt_0_lowtown_n_0_4.dds"},
	{0x0952B8187C6CA60BULL, "6134825_m.fxe"},
	{0x0952F9532667FC54ULL, "6126977_m.fxe"},
	{0x0952FD1338263FE1ULL, "116379404.ogg"},
	{0x09536009FA21CEE6ULL, "1008723500.ogg"},
	{0x0953D8802E107614ULL, "vxdr_0_deeproads_cxq_0_0.vlm"},
	{0x0953E1175E1D44D1ULL, "vxht_bartrand_n_6rh_0_0.vlm"},
	{0x09542CA9D85704D1ULL, "art\\vfx\\cinematic\\vfx_fire_bunch_p.impa.ani"},
	{0x0954664E6F340611ULL, "art\\characters\\playercharacter\\shared\\textures\\bdy_bnb_0d.dds"},
	{0x0954E3B12931E0E1ULL, "vxht_0_hightown_n_a2l_0_0.vlm"},
	{0x09550731D6D3CFEFULL, "seb321_leliana6180966.cl"},
	{0x095553657AC8C01FULL, "art\\characters\\playercharacter\\humanmale\\animation\\ambient\\dialogue\\hands_on_face_enter.gad"},
	{0x09555D32604B3AEDULL, "vrc321_varric.bnk"},
	{0x0955686FF43933B4ULL, "451044958.ogg"},
	{0x09556B12B776E717ULL, "6121760_m.fxe"},
	{0x0955966942BC451FULL, "dae100_bran6118307.cl"},
	{0x095674A125F8020FULL, "art\\vfx\\combat\\vfx_glb_com_brittle_s2_c.dura.gad"},
	{0x09576DB14FED6FE7ULL, "vxw_sundermount_2rr_0_0.vlm"},
	{0x0957C75C1A3480E5ULL, "qun230ar_cavern.are"},
	{0x09582204D4CB925BULL, "art\\vfx\\combat\\vfx_m_pri_tempest_aoe_p.impa.gad"},
	{0x095843CB301FECEAULL, "zz_lgt_1_debug6140405.cl"},
	{0x0958647AD1FBD057ULL, "6177889_f.fxe"},
	{0x09586DAD61F4E53EULL, "gen00fl_bethany6102771.cl"},
	{0x09587F32604DAA55ULL, "vrc321_varric.cnv"},
	{0x0958D1E7D4857F09ULL, "vxlt_0_lowtown_docks_bat_0_0.vlm"},
	{0x095901F1D96BDA0FULL, "dae000_arishok6067668.cl"},
	{0x09594CE8CC5A2419ULL, "hlac_lt_gallowscourtyard_n_0_1y.dds"},
	{0x09595921EAE94FA5ULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\dialogue\\lean_back_wall_loop.gad"},
	{0x095A34DED322D747ULL, "art\\characters\\playercharacter\\humanmale\\textures\\hn_arm_medj_0e.dds"},
	{0x095A37A9D85B8373ULL, "art\\vfx\\cinematic\\vfx_fire_bunch_p.impa.gad"},
	{0x095AF70759364410ULL, "hlac_w_bonepit_262149_0z.dds"},
	{0x095B06B8836CE88CULL, "vxht_chantry_n_2te_0_1.vlm"},
	{0x095B1C484E91AC65ULL, "vxlt_0_lowtown_docks_a3_c1n_0_0.vlm"},
	{0x095B35F1F28FB7CEULL, "zz_approval_debug6133717.cl"},
	{0x095B792F1CC9722FULL, "follower_banter6172323.cl"},
	{0x095BF092C64AB8DDULL, "6123221_m.fxe"},
	{0x095C120EE88405DCULL, "561135746.ogg"},
	{0x095C3BC7BC7402B9ULL, "llac_ca_caverns1_mines_24425_13.dds"},
	{0x095C48657ACECC0CULL, "art\\characters\\playercharacter\\humanmale\\animation\\ambient\\dialogue\\hands_on_face_enter.evt"},
	{0x095C7C81D3B18979ULL, "pro000_colonel_ewald6040605.cl"},
	{0x095C85B4CE9BAE7CULL, "139084830.ogg"},
	{0x095C89A125FC91BCULL, "art\\vfx\\combat\\vfx_glb_com_brittle_s2_c.dura.evt"},
	{0x095CB5B2F699529FULL, "qun100_mage_party_comment6019114.cl"},
	{0x095CEB923637FB96ULL, "vxw_bonepitdestroyed_a3_2a5_0_0.vlm"},
	{0x095DBF670B215889ULL, "ser100_party_comments6195135.cl"},
	{0x095DCB9ACB1C7361ULL, "dae000_gamlen6121565.cl"},
	{0x095E0C4A177BB403ULL, "mrl000_marethari6129365.cl"},
	{0x095F13C566DFE509ULL, "6147123_f.fxe"},
	{0x095F1752B721CCC9ULL, "700348072.ogg"},
	{0x095F3F04D4D1E240ULL, "art\\vfx\\combat\\vfx_m_pri_tempest_aoe_p.impa.evt"},
	{0x095F8BBC8148C740ULL, "vxlt_gamlenshouse_n_31_0_0.vlm"},
	{0x095F91EF63F668FAULL, "vxlt_0_lowtown_a2_hfe_0_0.vlm"},
	{0x095FD381929B53D9ULL, "6156629_m.fxe"},
	{0x09613164CE86EE0EULL, "vxlt_0_lowtown_a3_n_c_hjm_0_0.vlm"},
	{0x096134A9D8619CD8ULL, "art\\vfx\\cinematic\\vfx_fire_bunch_p.impa.evt"},
	{0x0961A28A69FE3F14ULL, "primitives_primitive_p6.psh.fxo.dx11.assembly"},
	{0x0962557B6CBDA052ULL, "6153535_m.fxe"},
	{0x096255AD908FF959ULL, "genip_sarcophagous.utp"},
	{0x0962B023485BEFDDULL, "hlac_lt_0_lowtown_docks_131070_1x.dds"},
	{0x0962C85C1A3E89CBULL, "qun230ar_cavern.ncs"},
	{0x09635E895A5BFD9CULL, "hlac_w_blightlands_327679_0y.dds"},
	{0x096360E3C4B54BD1ULL, "w_bonepitdestroyed_a3_209_0.msh"},
	{0x09636EE3C4B563ECULL, "w_bonepitdestroyed_a3_209_0.mao"},
	{0x09637AE3C4B5778FULL, "w_bonepitdestroyed_a3_209_0.mmh"},
	{0x0963BA99A0B3FE71ULL, "vxlt_gallowscourtyard_74w_0_1.vlm"},
	{0x0963CACA1C5751A0ULL, "318326508.ogg"},
	{0x0963DEBB268E07D1ULL, "12470751.ogg"},
	{0x096400E4A3A2460EULL, "vxw_encounter2_r8_0_0.vlm"},
	{0x096461B9E2409B55ULL, "302829900.ogg"},
	{0x096545C1D89E646DULL, "vxlt_gallowstemplar_n_3qt_0_0.vlm"},
	{0x0965615EC0538631ULL, "cd_mage110cs_cut_abomination.bnk"},
	{0x0965936D497BD1C0ULL, "hlac_lt_darktown_-1_47y.dds"},
	{0x0965E353319A50C5ULL, "docks_male_human_worker_sta6175994.cl"},
	{0x096641B7926E33A3ULL, "vxht_mansion2_31f_0_0.vlm"},
	{0x0966CBCFF5E767A3ULL, "364560151.ogg"},
	{0x0967B4B747AC5597ULL, "art\\characters\\playercharacter\\elffemale\\textures\\ef_boo_lgtp_0n.dds"},
	{0x0967C8893111742FULL, "hlac_lt_0_lowtown_n_-65537_6x.dds"},
	{0x0967CEC340CB291AULL, "mrl150_merrill6077735.cl"},
	{0x0967FE3FA83980E3ULL, "cod_hst_woundedcoast.utp"},
	{0x0968098F0503A207ULL, "vxlt_backalley_n_1u4_0_0.vlm"},
	{0x096819CD15FEE920ULL, "vxdr_slavers_1t8_0_1.vlm"},
	{0x09682240F2B12C1FULL, "ash_01_l2.mao"},
	{0x0968860F0F379008ULL, "mag291_demon_marethari6027972.cl"},
	{0x09688A7E370EED6EULL, "1021285711.ogg"},
	{0x0968A693007B8579ULL, "6208951_m.fxe"},
	{0x0968C85C1A42F512ULL, "qun230ar_cavern.lst"},
	{0x0968D6A024CF10E3ULL, "art\\vfx\\materials\\level_fx\\vfx_lvl_stars.mao"},
	{0x09692FA96B93A5AAULL, "6135092_m.fxe"},
	{0x0969517DF6655963ULL, "dae000cr_grand_cleric.bnk"},
	{0x096959E5AD1CA1C8ULL, "533095993.ogg"},
	{0x096979732770B526ULL, "hightown_male_elfservnt_mob6170078.cl"},
	{0x0969CDB5BB838245ULL, "hlac_lt_undercityserialkiller_-1_36x.dds"},
	{0x096A10DA2710E905ULL, "vxlt_0_lowtown_docks_a3_n_d56_0_0.vlm"},
	{0x096A30657ADAE165ULL, "art\\characters\\playercharacter\\humanmale\\animation\\ambient\\dialogue\\hands_on_face_enter.ani"},
	{0x096A321C88F0EC9BULL, "921153275.ogg"},
	{0x096A626FB840D15DULL, "6128321_m.fxe"},
	{0x096A91A12608DCD5ULL, "art\\vfx\\combat\\vfx_glb_com_brittle_s2_c.dura.ani"},
	{0x096AE609E425BF51ULL, "ser221cr_gascard.utc"},
	{0x096B333CB42246B9ULL, "vxht_chantry_n_2av_0_0.vlm"},
	{0x096B45EF6657FE31ULL, "ave200_donnic6112157.cl"},
	{0x096BCB608489EB61ULL, "w_encounter2_camp_12_0_lc.mmh"},
	{0x096C2D45DBE20649ULL, "6033762_m.fxe"},
	{0x096C337DF6675B93ULL, "dae000cr_grand_cleric.cnv"},
	{0x096C3EDC7CECFB64ULL, "dae200_cullen6140270.cl"},
	{0x096CD932EE6CF419ULL, "vxdr_primevaltomb_6po_0_0.vlm"},
	{0x096D91DCBB2204A6ULL, "vxca_caverns1_mines_e65_0_0.vlm"},
	{0x096D9B066C091DCBULL, "llac_lt_gallowscourtyard_131071_0.dds"},
	{0x096DC78E4DC8C864ULL, "vxlt_0_lowtown_n_eix_0_0.vlm"},
	{0x096E24C32C2BCFBFULL, "544562243.ogg"},
	{0x096E4F0CE9392C90ULL, "w_bonepitmining_a2_242_0.msh"},
	{0x096E5D0CE93944AAULL, "w_bonepitmining_a2_242_0.mmh"},
	{0x096E610CE9394BE1ULL, "w_bonepitmining_a2_242_0.mao"},
	{0x096EDC05BBEB0BF6ULL, "127586829.ogg"},
	{0x096FD509F40FC0ACULL, "6161598_m.fxe"},
	{0x09704C42032D0569ULL, "dae000ip_crowd_mf5_talk17.utp"},
	{0x0970510266B87667ULL, "6170616_m.fxe"},
	{0x09706BD16B4E8B5EULL, "vxlt_gallowscourtyard_4xb_0_0.vlm"},
	{0x0970A0EAA08A7D99ULL, "hlac_lt_0_lowtown_65538_3x.dds"},
	{0x0970A1BE7336FDAEULL, "1015487318.ogg"},
	{0x0970A8F1AD69047DULL, "609194303.ogg"},
	{0x0970BC3A11FCA714ULL, "vxdr_0_deeproads_crw_0_0.vlm"},
	{0x097104AEB1406F67ULL, "6085906_f.fxe"},
	{0x09716075D339AB9AULL, "6114088_m.fxe"},
	{0x0972B1B75C0B753FULL, "vxlt_gallowstemplar_f_43c_0_0.vlm"},
	{0x0972C25812F60511ULL, "dae000_cerimon6163678.cl"},
	{0x09737047B1BE7015ULL, "gen_im_arm_hlm_lgt_act3_01.uti"},
	{0x09737FEA1FAEFDDFULL, "6117682_m.fxe"},
	{0x097430E00702DD54ULL, "6129550_m.fxe"},
	{0x0974516B33D9A0E7ULL, "vxw_sundermount_2xl_0_0.vlm"},
	{0x09751C378B829985ULL, "6137035_m.fxe"},
	{0x09752F44A7B80D9AULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\2ca_basic.msh"},
	{0x0975340C6AAB5881ULL, "test_2da_str.ncs"},
	{0x09753D44A7B825D3ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\2ca_basic.mao"},
	{0x09754144A7B82CE0ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\2ca_basic.mmh"},
	{0x0975BCDA3005C861ULL, "vxht_0_hightown_a2_be1_0_0.vlm"},
	{0x09762401239BAEA1ULL, "flare01.mao"},
	{0x097663C55242625BULL, "vxlt_gallowsprison_n_6pd_0_0.vlm"},
	{0x0976A00E01C4062AULL, "dae000_sabina6161456.cl"},
	{0x0976F0C2DB8815D4ULL, "309893194.ogg"},
	{0x0977A3624CC00CA8ULL, "art\\vfx\\combat\\vfx_w_wns_left_b_close1_c.mmh"},
	{0x097882B719070185ULL, "vxlt_0_lowtown_docks_a3_n_d05_0_0.vlm"},
	{0x0978E21496E01A3FULL, "vxht_keep_69w_0_0.vlm"},
	{0x097905B1E6219DE3ULL, "818266309.ogg"},
	{0x09795B45F8A4F78EULL, "vxlt_undercitydungeon_1yy_0_0.vlm"},
	{0x0979D59D7BA0FEFDULL, "vxht_bartrand_n_5af_0_0.vlm"},
	{0x097A4C7C8F915C97ULL, "6033598_f.fxe"},
	{0x097A52E4F777C4F9ULL, "303509584.ogg"},
	{0x097B4642751E64C7ULL, "377560973.ogg"},
	{0x097B4B0FE0630C99ULL, "vxdr_primevaltomb_6wn_0_0.vlm"},
	{0x097B60FE7BD37F36ULL, "6075697_m.fxe"},
	{0x097B710D345C1448ULL, "m_mcr371ar_galows_templr_nt.bnk"},
	{0x097B8CA8FE09A08DULL, "lt_0_lowtown_docks_a3_43_0.rml"},
	{0x097C75A947E299FAULL, "vxlt_0_lowtown_a2_hhg_0_0.vlm"},
	{0x097D28D26590CF11ULL, "738668704.ogg"},
	{0x097D435A49B3EBC5ULL, "722036753.ogg"},
	{0x097DBA2A1EC243AEULL, "lowtown_male_drunk_mob6151166.cl"},
	{0x097E0A6E1990784DULL, "911196998.ogg"},
	{0x097E183D084FF928ULL, "vxca_cave_2e4_0_0.vlm"},
	{0x097E449ED0B39FD0ULL, "vxlt_undercityencounter_1fg_0_0.vlm"},
	{0x097F1F1973DD6308ULL, "zz_ave_debug6064627.cl"},
	{0x097F458FF15A7448ULL, "vxht_0_hightown_a2_be4_0_0.vlm"},
	{0x097F58FDF6F0D731ULL, "hlac_dr_slavers_3207_3x.dds"},
	{0x097F67E055961657ULL, "vxht_brothel_n_3wd_0_0.vlm"},
	{0x097F9DEBDA5F9812ULL, "vxw_encounter2_camp_qp_0_1.vlm"},
	{0x098013BAA6FF0CDBULL, "6113985_m.fxe"},
	{0x0980C87B4DC20DA3ULL, "vxlt_warehouse_qw_0_0.vlm"},
	{0x09810B369733FA9FULL, "art\\vfx\\combat\\vfx_r_dwp_combo_loop_c.mmh"},
	{0x09811ACCF4B6574EULL, "hlac_ht_mansion2b_818_49y.dds"},
	{0x09811BA32A3EB613ULL, "6156308_m.fxe"},
	{0x09812320E5FCA69FULL, "187465545.ogg"},
	{0x098138FFF47FD185ULL, "dre000_sibling_has_blight6036457.cl"},
	{0x098244ED834E35CBULL, "6149199_m.fxe"},
	{0x09825C0120929D0CULL, "framebuffereffects_radialblur_p0.psh.fxo.dx11"},
	{0x0982AD43BC45632FULL, "615507067.ogg"},
	{0x0982CCFD23CB6DA3ULL, "6072320_f.fxe"},
	{0x09834FCAB36F934CULL, "vxca_caverns2prc_aaq_0_0.vlm"},
	{0x09839C0A31DBB725ULL, "6013409_m.fxe"},
	{0x0983A3CC18EE0983ULL, "vxht_brothel_n_5l3_0_0.vlm"},
	{0x0984316C3D40C1CDULL, "art\\levels\\races\\proxy\\props\\door\\prp_door_8.pwk"},
	{0x0984B79FADE47A7DULL, "6103560_m.fxe"},
	{0x0984D672AAA76294ULL, "gen00fl_dog6165737.cl"},
	{0x0984FED3399C68E5ULL, "hlac_lt_0_lowtown_a3_n_c_131070_0x.dds"},
	{0x09850B5EDB097B4EULL, "1024737114.ogg"},
	{0x09851A95507DF590ULL, "vxht_0_hightown_a2_av2_0_0.vlm"},
	{0x098520249A69980FULL, "hlac_lt_0_lowtown_a2_n_131072_6y.dds"},
	{0x09854361AB8D456BULL, "mcr310_sibling6076270.cl"},
	{0x0985F1C8E6EAB906ULL, "dae011_drunk2_sta.cnv"},
	{0x09860D568F4B670BULL, "666388170.ogg"},
	{0x09860F75E1B271BBULL, "251747403.ogg"},
	{0x098653324AFC8C6AULL, "w_bonepitdestroyed_a3_243_0.mao"},
	{0x098657324AFC92B9ULL, "w_bonepitdestroyed_a3_243_0.mmh"},
	{0x098661324AFCA3A7ULL, "w_bonepitdestroyed_a3_243_0.msh"},
	{0x0986B50B423A577BULL, "narrative_varric_demo6170807.cl"},
	{0x098816301DACB64DULL, "vxlt_gallowstemplar_f_660_0_0.vlm"},
	{0x0988D9A5B7F1101DULL, "vxlt_gallowstemplar_n_5c8_0_0.vlm"},
	{0x0988F1F73F85772CULL, "blood_conj_08.mao"},
	{0x0988F3C8E6ECF2C6ULL, "dae011_drunk2_sta.bnk"},
	{0x098947F59BFAFDF5ULL, "zz_fol_debug6056998.cl"},
	{0x09895E35E7678733ULL, "art\\levels\\4cinematics\\qunari\\_textures\\prp_qunaricontroller_sa.dds"},
	{0x09896F2DE0BFCE0CULL, "mrl000_marethari6069864.cl"},
	{0x09899907E1C724E3ULL, "118467572.ogg"},
	{0x0989D6EF451F5C2EULL, "65524077.ogg"},
	{0x098A380C078B178DULL, "175363961.ogg"},
	{0x098A4D23DF6023B8ULL, "295840479.ogg"},
	{0x098A8993686AF845ULL, "vxlt_0_lowtown_a2_hfn_0_0.vlm"},
	{0x098AADD9C5EEA4CBULL, "l_ser111ar_foundry.wbd"},
	{0x098ACCA80610BECDULL, "vxlt_0_lowtown_a2_gx3_0_0.vlm"},
	{0x098B3B33BD711376ULL, "6125435_m.fxe"},
	{0x098B42E37D8FCA73ULL, "696277062.ogg"},
	{0x098BD2749E1B4080ULL, "zz_approval_debug6003702.cl"},
	{0x098C3C250706FA4EULL, "541878720.ogg"},
	{0x098E2439CE6945E8ULL, "6167000_m.fxe"},
	{0x098E3F4335CDEC6DULL, "vxlt_0_lowtown_n_g93_0_0.vlm"},
	{0x098E625343D11803ULL, "art\\vfx\\combat\\vfx_glb_stun_c.dura.ani"},
	{0x098E67988DDBA3FDULL, "6125106_m.fxe"},
	{0x098E79109404BAB4ULL, "zz_approval_debug6003698.cl"},
	{0x098E8D1173B9FFE7ULL, "lowtown_fem_alienage1_sta6162565.cl"},
	{0x098EA26BC1D0391AULL, "6092085_m.fxe"},
	{0x098F27E08D129236ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\9uw_tarpouta.mao"},
	{0x098F33E08D12A6D5ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\9uw_tarpouta.mmh"},
	{0x098F35E08D12AA4BULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\9uw_tarpouta.msh"},
	{0x098F4DB66DE62E1EULL, "6118440_m.fxe"},
	{0x098F4F961D9F9812ULL, "art\\levels\\races\\ferelden\\props\\lowerclass\\prp_cookerpots.phy"},
	{0x098FB7F45F4497D3ULL, "6084245_m.fxe"},
	{0x098FE060648F8D85ULL, "art\\vfx\\combat\\vfx_gib_rob_appc.mao"},
	{0x098FEEA50526B50EULL, "6152778_m.fxe"},
	{0x098FF3AE821B6BC1ULL, "6115960_f.fxe"},
	{0x0990029B84665DC8ULL, "mer240_jansen6050328.cl"},
	{0x09900AF74750609DULL, "102546494.ogg"},
	{0x0990377EDB9B288BULL, "766986214.ogg"},
	{0x0990FFBA26E780EBULL, "69222022.ogg"},
	{0x09914689750CCE0AULL, "6104188_m.fxe"},
	{0x09916045D052A9ABULL, "6076868_m.fxe"},
	{0x099171D65F69ECF4ULL, "vxlt_0_lowtown_docks_a3_n_a0m_0_0.vlm"},
	{0x0991F35015B730C7ULL, "hlac_ca_caverns1_mines_24423_10y.dds"},
	{0x09920DE723D1CFA5ULL, "6018509_m.fxe"},
	{0x099267833A5BC509ULL, "vxca_cavern1_haunted_i1k_0_0.vlm"},
	{0x09927DC93B527D38ULL, "vxlt_gallowstemplar_f_6qd_0_0.vlm"},
	{0x0992E4C00D72F67AULL, "thefarcliffskrkwl.dds"},
	{0x099314B561413138ULL, "132738383.ogg"},
	{0x0993297C9FDA92FDULL, "6104460_m.fxe"},
	{0x09933751F7222B77ULL, "186363587.ogg"},
	{0x09938C7242740E10ULL, "vxht_0_hightown_a2_aq3_0_0.vlm"},
	{0x09939B711F452F60ULL, "gen00fl_isabela6079053.cl"},
	{0x099409F67F5B8943ULL, "14733315.ogg"},
	{0x099438C13D6E9E9DULL, "6094512_m.fxe"},
	{0x0994684C8FFE3B61ULL, "737596721.ogg"},
	{0x0994B4A5BC1682F8ULL, "6104087_m.fxe"},
	{0x0995766A893485B5ULL, "509716245.ogg"},
	{0x09957BAF3D47AD9EULL, "6149288_m.fxe"},
	{0x099610E4747EA66AULL, "hlac_w_encounter3_-196608_0z.dds"},
	{0x09961B3A438C03CCULL, "lgt160_ntb_elf_conflict6163374.cl"},
	{0x09962E3DCF1FB65BULL, "396785630.ogg"},
	{0x099687C4B9581CC7ULL, "6075612_m.fxe"},
	{0x0996B9575F8D2FFDULL, "vxht_bartrand_n_5od_0_0.vlm"},
	{0x0997118B9B9C49ECULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\pose\\2arm_crack_knuckles.ani"},
	{0x099718BEC0B9EF0EULL, "art\\vfx\\combat\\vfx_m_arc_modal_loop_c.mmh"},
	{0x099767DDAEC20D26ULL, "vxlt_hangedman_1vf_0_0.vlm"},
	{0x0998AA7A0AD4BDDFULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_nicechair_0.pwk"},
	{0x0998B9D735ACA239ULL, "33386345.ogg"},
	{0x0998D7EEF3505640ULL, "hlac_ht_0_hightown_0_3x.dds"},
	{0x0998FB705A6110C5ULL, "vxlt_0_lowtown_a2_hcm_0_0.vlm"},
	{0x0999781FF93CC49AULL, "vxlt_warehouse_n_12e_0_0.vlm"},
	{0x0999CC8D37EEB813ULL, "6087378_m.fxe"},
	{0x0999D46CD53370F6ULL, "vxw_encounter2_camp_nl_0_0.vlm"},
	{0x0999F987C6CD8F63ULL, "gen_im_acc_blt_act3_05.uti"},
	{0x099A263920C1BDA1ULL, "one300ip_worldmap.utp"},
	{0x099A78209586F6E7ULL, "6042049_m.fxe"},
	{0x099A946381F14806ULL, "410388215.ogg"},
	{0x099AB035CB41C8B4ULL, "393120644.ogg"},
	{0x099B7A5343DBCBFAULL, "art\\vfx\\combat\\vfx_glb_stun_c.dura.evt"},
	{0x099B9165B419413AULL, "zz_mrl_debug6109314.cl"},
	{0x099C171AD1B7E629ULL, "hlac_w_encounter3_-131070_0y.dds"},
	{0x099C4DBEA8066EA8ULL, "gen00fl_dog.utc"},
	{0x099C53F8381CAE28ULL, "mcr370_meredith6083042.cl"},
	{0x099CC0866570C19BULL, "6156379_m.fxe"},
	{0x099CC2161A68FAFAULL, "909532822.ogg"},
	{0x099CCB39CFA799CBULL, "vxlt_0_lowtown_a2_cmn_0_0.vlm"},
	{0x099CF9FF2D96D99FULL, "vxht_mansion_n_5i7_0_0.vlm"},
	{0x099D1E8B9BA0CB2EULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\pose\\2arm_crack_knuckles.gad"},
	{0x099E191EB13C49DBULL, "vxht_mansion_n_4bg_0_0.vlm"},
	{0x099F1B757EC6BC9BULL, "dae000_gen_turn_in_16178392.cl"},
	{0x099F3A011B2C2E7CULL, "lgt120_orz_bln_official6140172.cl"},
	{0x099F5C97D772A5BDULL, "mer140_bonepit_comments6062991.cl"},
	{0x099F86888C8BD623ULL, "hlac_lt_0_lowtown_n_-262144_1x.dds"},
	{0x099F96F036279303ULL, "962727408.ogg"},
	{0x099FC64473207999ULL, "dae000_merchant_armor_hm6164276.cl"},
	{0x099FCB29EFF3C81BULL, "962646565.ogg"},
	{0x099FEBA96C213A6EULL, "835922782.ogg"},
	{0x09A0630D347B18F2ULL, "m_mcr371ar_galows_templr_nt.wbd"},
	{0x09A095777060FBC3ULL, "art\\characters\\weapons\\greataxes\\textures\\w_gax_ga101a_0t.dds"},
	{0x09A0AD6681F1FC33ULL, "qcr220_meredith6185152.cl"},
	{0x09A0AFAD49E05B14ULL, "ceil_clo.mao"},
	{0x09A0C4B3451934BCULL, "vxdr_primevaltomb_6m1_0_0.vlm"},
	{0x09A12120F136231FULL, "6125593_m.fxe"},
	{0x09A128E7F362F690ULL, "llac_lt_0_lowtown_a2_n_-65535_1.dds"},
	{0x09A19BF35C79EB72ULL, "con_ico_romance.dds"},
	{0x09A1B14EFFE1D8A4ULL, "llac_lt_0_lowtown_a2_n_c_131072_4.dds"},
	{0x09A20C90B8352606ULL, "361764089.ogg"},
	{0x09A25133C41F7FC9ULL, "441012331.ogg"},
	{0x09A2795343E1E8EDULL, "art\\vfx\\combat\\vfx_glb_stun_c.dura.gad"},
	{0x09A339A018465CF3ULL, "6109252_m.fxe"},
	{0x09A3456BCD9CB64AULL, "vxht_keep_n_3hk_0_0.vlm"},
	{0x09A36A99A88DF504ULL, "6182009_m.fxe"},
	{0x09A396145CC7E025ULL, "w_woundedcoast_85_0.rml"},
	{0x09A3EA85085E5EBDULL, "vxlt_backalley_n_1bo_0_0.vlm"},
	{0x09A415700AD9A7E5ULL, "follower_banter6163446.cl"},
	{0x09A4198B9BA6E10DULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\pose\\2arm_crack_knuckles.evt"},
	{0x09A52DDDE00E670CULL, "vxca_caverns2prc_9pz_0_0.vlm"},
	{0x09A537BEC0C6610DULL, "art\\vfx\\combat\\vfx_m_arc_modal_loop_c.anb"},
	{0x09A55F5E2CC2E865ULL, "llac_lt_0_lowtown_a2_n_c_65536_2.dds"},
	{0x09A5739B91D7686EULL, "one280_iwan6096894.cl"},
	{0x09A6472E3881EC39ULL, "art\\levels\\races\\dwarf\\props\\paragon\\prp_standb_1.mao"},
	{0x09A666FB80B50A53ULL, "1014972876.ogg"},
	{0x09A70A5728F871B2ULL, "346591727.ogg"},
	{0x09A70CA854ED76A0ULL, "6034375_m.fxe"},
	{0x09A75EF4A8F82072ULL, "6185385_m.fxe"},
	{0x09A792E3027EAFDDULL, "ser131ip_to_darktown.utp"},
	{0x09A7D8DD8EE324FDULL, "hlac_ca_cavern1_haunted_24423_10y.dds"},
	{0x09A84649C7298976ULL, "vxw_encounter2_camp_im_0_0.vlm"},
	{0x09A84E0B11E6E179ULL, "257292565.ogg"},
	{0x09A8CF80159F850CULL, "500226627.ogg"},
	{0x09A8E4B5416D886DULL, "6087857_m.fxe"},
	{0x09A9C1AF27AAA975ULL, "6112889_m.fxe"},
	{0x09AA0C36975701DCULL, "art\\vfx\\combat\\vfx_r_dwp_combo_loop_c.anb"},
	{0x09AAA092E66CF9ACULL, "hlac_w_bonepitdestroyed_a3_262147_0x.dds"},
	{0x09AB1E94F4C5BF27ULL, "893028342.ogg"},
	{0x09AB40EB28D30D75ULL, "vrc221_varric6048575.cl"},
	{0x09AB456EDB505F48ULL, "6167476_f.fxe"},
	{0x09ABF8E8633846ABULL, "hlac_lt_0_lowtown_n_1_0z.dds"},
	{0x09AC49A03FBF4BFDULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\9uw_hngtarpoutcn.mao"},
	{0x09AC4DA03FBF5236ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\9uw_hngtarpoutcn.mmh"},
	{0x09AC57A03FBF6334ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\9uw_hngtarpoutcn.msh"},
	{0x09AC869175FD92E1ULL, "vxlt_0_lowtown_a2_he7_0_0.vlm"},
	{0x09AC8C2F748785D5ULL, "6171882_m.fxe"},
	{0x09ACFE3088B4E826ULL, "art\\vfx\\combat\\vfx_f_mer_ingrasp_tag_lp_c.anb"},
	{0x09AD68451007699DULL, "hlac_lt_0_lowtown_-1_2x.dds"},
	{0x09AE4C44EA40519AULL, "187843324.ogg"},
	{0x09AE606EAB2A4BF1ULL, "vxht_chantry_n_2zk_0_0.vlm"},
	{0x09AE955442E848D4ULL, "6034076_m.fxe"},
	{0x09AEED2E3BCC2701ULL, "6148851_m.fxe"},
	{0x09AF3690370F4D3CULL, "vxdr_primevaltomb_6p6_0_0.vlm"},
	{0x09AFB6542F58D70EULL, "vxht_0_hightown_a2_c4e_0_1.vlm"},
	{0x09B05CC7A410BC2CULL, "mer240_jansen6050412.cl"},
	{0x09B08720F39EA5F5ULL, "6146790_m.fxe"},
	{0x09B08FD9C60F4099ULL, "l_ser111ar_foundry.bnk"},
	{0x09B0C7737FF801ADULL, "art\\levels\\ztools\\default\\generic\\0di_angle.mmh"},
	{0x09B0CB737FF808FEULL, "art\\levels\\ztools\\default\\generic\\0di_angle.mao"},
	{0x09B0D9737FF82033ULL, "art\\levels\\ztools\\default\\generic\\0di_angle.msh"},
	{0x09B11CE17B586470ULL, "w_weapon.wbd"},
	{0x09B1ED0E1D472733ULL, "hlac_dr_primevalentrance_1726_8y.dds"},
	{0x09B1FAC92880E7DEULL, "6164775_m.fxe"},
	{0x09B2343DB6DFFF72ULL, "one110_pryce6088868.cl"},
	{0x09B23B96BF069BDFULL, "hlac_lt_0_lowtown_a2_65538_0y.dds"},
	{0x09B2592D3E5087A0ULL, "6118811_m.fxe"},
	{0x09B2DE60AECADB0FULL, "vxlt_0_lowtown_a2_n_c_ghh_0_0.vlm"},
	{0x09B3034014B9E70CULL, "follower_banter6141931.cl"},
	{0x09B3252A9F1B0183ULL, "436820091.ogg"},
	{0x09B3569C87BFAA07ULL, "ts_kirk_guards_03.gda"},
	{0x09B439B0D201B438ULL, "6040163_m.fxe"},
	{0x09B44B9792AE3E26ULL, "vxlt_hangedman_1xh_0_0.vlm"},
	{0x09B4AD8C823337EFULL, "fen110_anso6094957.cl"},
	{0x09B5251303CFB8FFULL, "hlac_lt_hangedman_n_-1_29z.dds"},
	{0x09B575C16C9FEABEULL, "354328553.ogg"},
	{0x09B5B35B798159B7ULL, "hlac_lt_0_lowtown_a2_n_c_-131072_1y.dds"},
	{0x09B5BD0B0CC1308DULL, "lav_lakemidtone.mao"},
	{0x09B60F4737523EC8ULL, "vxlt_0_lowtown_a2_he2_0_0.vlm"},
	{0x09B6672C844A4570ULL, "dae000_flemeth6000680.cl"},
	{0x09B6BA13D17B6A51ULL, "dae000_bartrand6064088.cl"},
	{0x09B73F289F9C4546ULL, "art\\levels\\races\\elf\\props\\_textures\\prp_gravea_d.dds"},
	{0x09B794238C32865CULL, "ht_0_hightownexplosion_n_20_0_ll.mmh"},
	{0x09B7BE419980AEEEULL, "art\\characters\\playercharacter\\humanmale\\textures\\hn_arm_masf_0s.dds"},
	{0x09B7E7A9A07DB273ULL, "dae000_gamlen6121545.cl"},
	{0x09B83FA2190B3E24ULL, "850986701.ogg"},
	{0x09B9924ED8F3EC34ULL, "vxdr_0_deeproads_eud_0_0.vlm"},
	{0x09B9D44050E8460EULL, "6083549_f.fxe"},
	{0x09B9D909E4697D1FULL, "ser221cr_gascard.ncs"},
	{0x09BA1202ABA6AD14ULL, "680059501.ogg"},
	{0x09BA31BE2298FA09ULL, "vxdr_primevaltomb_8no_0_0.vlm"},
	{0x09BA5A23686ABCFEULL, "ave310_brennan6136885.cl"},
	{0x09BA95E9B1F44CD1ULL, "544583811.ogg"},
	{0x09BAAA9A81E2AA26ULL, "vxlt_gallowsprison_n_5rl_0_0.vlm"},
	{0x09BAF86E67F3AB61ULL, "vxlt_0_lowtown_a2_hb6_0_0.vlm"},
	{0x09BB013088C12AD1ULL, "art\\vfx\\combat\\vfx_f_mer_ingrasp_tag_lp_c.mmh"},
	{0x09BB1AD289AC1BB9ULL, "lt_0_lowtown_docks_a3_n_117_0.msh"},
	{0x09BB24D289AC2CA7ULL, "lt_0_lowtown_docks_a3_n_117_0.mmh"},
	{0x09BB28D289AC33F4ULL, "lt_0_lowtown_docks_a3_n_117_0.mao"},
	{0x09BB315E1597392EULL, "vxca_caverns1_mines_jg1_0_0.vlm"},
	{0x09BB35BEA820F6F9ULL, "gen00fl_dog.bnk"},
	{0x09BBDC9084642B1AULL, "dae300_cullen6140611.cl"},
	{0x09BC2E6863B56FBBULL, "fen110_anso6095015.cl"},
	{0x09BC703994375347ULL, "327751947.ogg"},
	{0x09BD67168A8B1E1AULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\vista_toweratall.msh"},
	{0x09BD75168A8B3653ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\vista_toweratall.mao"},
	{0x09BD79168A8B3D60ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\vista_toweratall.mmh"},
	{0x09BDA86D290565BCULL, "vxdr_primevaltomb_6w7_0_0.vlm"},
	{0x09BE30FF482FEBB5ULL, "6063842_m.fxe"},
	{0x09BE57BEA8236659ULL, "gen00fl_dog.cnv"},
	{0x09BE6BD053032811ULL, "dae000_madam_lusine6151215.cl"},
	{0x09BEF14D6054561EULL, "6078976_f.fxe"},
	{0x09C00100ECB43D0BULL, "295707740.ogg"},
	{0x09C042A962F6E9EAULL, "sound_settings.gda"},
	{0x09C05D397096E636ULL, "w_encounter3_85_0.msh"},
	{0x09C067397096F734ULL, "w_encounter3_85_0.mmh"},
	{0x09C06B397096FE47ULL, "w_encounter3_85_0.mao"},
	{0x09C0CB0A3046A020ULL, "6117911_m.fxe"},
	{0x09C17BD8A430FBE8ULL, "vxht_0_hightown_a2_c6y_0_1.vlm"},
	{0x09C1C0738006696AULL, "art\\levels\\ztools\\default\\generic\\0di_angle.phy"},
	{0x09C1D16F97A3584BULL, "6168136_m.fxe"},
	{0x09C2BD7484A456A6ULL, "vxlt_hangedman_1gi_0_0.vlm"},
	{0x09C3C6749B226C94ULL, "vxht_chantry_2sz_0_0.vlm"},
	{0x09C3CEB604E8E362ULL, "6070997_m.fxe"},
	{0x09C4ADCFC6C084B4ULL, "178286303.ogg"},
	{0x09C534488F394888ULL, "follower_banter6163229.cl"},
	{0x09C585381CDDC5DAULL, "121412595.ogg"},
	{0x09C59B4BA210D0E5ULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_buzzarda_sa.dds"},
	{0x09C6FA1A094B4644ULL, "art\\characters\\playercharacter\\humanfemale\\textures\\hf_arm_nud_0t.dds"},
	{0x09C71C016584F432ULL, "6068466_m.fxe"},
	{0x09C7871E1F0C1FF7ULL, "vxht_0_hightown_avm_0_0.vlm"},
	{0x09C7A131B5F3B425ULL, "hlac_lt_0_lowtown_docks_a3_n_-65539_6x.dds"},
	{0x09C7EE20FB0E0580ULL, "art\\levels\\races\\natural\\proxy\\props\\_textures\\large_d.dds"},
	{0x09C83BACE108CFD2ULL, "vxlt_hangedman_n_24x_0_0.vlm"},
	{0x09C8A39B148F1289ULL, "vxdr_primevaltomb_8in_0_0.vlm"},
	{0x09C8C5C40101665EULL, "vxlt_0_lowtown_a2_fab_0_0.vlm"},
	{0x09C8C64C4E28E23CULL, "mag110_samson6018709.cl"},
	{0x09C976145CE8794EULL, "w_woundedcoast_85_0.mmh"},
	{0x09C982145CE88DE5ULL, "w_woundedcoast_85_0.mao"},
	{0x09C990145CE8A53CULL, "w_woundedcoast_85_0.msh"},
	{0x09C9D80C72902C47ULL, "vis121_saemus6118459.cl"},
	{0x09CB44288F167CF1ULL, "vxht_chantry_n_2pi_0_0.vlm"},
	{0x09CB5D25156CE471ULL, "6070113_f.fxe"},
	{0x09CB935DB779A0CAULL, "vxlt_0_lowtown_gbv_0_0.vlm"},
	{0x09CBA868B290B009ULL, "6096319_f.fxe"},
	{0x09CBF9B3FB053DB8ULL, "6089902_m.fxe"},
	{0x09CC83F0A91FAFAFULL, "hlac_ht_keep_4519_14z.dds"},
	{0x09CC8B6338BBC9A8ULL, "vxw_bonepitdestroyed_a3_7a_0_0.vlm"},
	{0x09CCC89D709AC0F5ULL, "386044264.ogg"},
	{0x09CD071E752178ADULL, "gen00fl_isabela_init6077303.cl"},
	{0x09CD9AC03E3A75D9ULL, "art\\vfx\\combat\\vfx_c_mer_crack_decal_p.dura.ani"},
	{0x09CDCC80D5180997ULL, "6204966_m.fxe"},
	{0x09CDD6EFA79A57B1ULL, "mcr370_meredith6085928.cl"},
	{0x09CDDB266AA20260ULL, "hlac_ht_playershouse_n_855_31y.dds"},
	{0x09CE783E43842CCFULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_daggerc_d.dds"},
	{0x09CE95870A88EFE1ULL, "639377611.ogg"},
	{0x09CE95A2A762D8AFULL, "hlac_lt_0_lowtown_a2_n_65534_4x.dds"},
	{0x09CEA0D57931E8DBULL, "6064597_m.fxe"},
	{0x09CEA7C23DA37DC4ULL, "vxdr_primevaltomb_6rd_0_0.vlm"},
	{0x09CF42A3BDE62F60ULL, "gen00fl_isabela6097013.cl"},
	{0x09CF87D38BFFDA25ULL, "vxw_encounter2_camp_p5_0_0.vlm"},
	{0x09CFA6788448300AULL, "vxca_caverns1_coi_0_0.vlm"},
	{0x09CFD598867BF9CAULL, "vxlt_0_lowtown_a2_n_c_i1d_0_0.vlm"},
	{0x09CFEF10FD3DDC39ULL, "6070424_m.fxe"},
	{0x09D02E4278C4E181ULL, "lt_0_lowtown_docks_a3_n_37_0.rml"},
	{0x09D052550DE4566CULL, "6177913_m.fxe"},
	{0x09D0695CEA7E2649ULL, "vxw_woundedcoast_1de_0_0.vlm"},
	{0x09D0DA46564477BAULL, "vrc000ev_item_acquired.ncs"},
	{0x09D10F4C3C3F5ABAULL, "20615965.ogg"},
	{0x09D11E47C4185844ULL, "dae000_arishok6068501.cl"},
	{0x09D13543AC2CE9E5ULL, "6165182_m.fxe"},
	{0x09D1B0BCD27F7FA3ULL, "693807084.ogg"},
	{0x09D244DF8B7AFADEULL, "687858095.ogg"},
	{0x09D2552CEBB40BFEULL, "494538517.ogg"},
	{0x09D2797EB32570C8ULL, "dae100_bran6118298.cl"},
	{0x09D2A1776D67B06EULL, "gen_im_npc_wep_sta_gena.uti"},
	{0x09D312A3EBE30EE7ULL, "162537775.ogg"},
	{0x09D39A37912D5A3DULL, "hlac_lt_0_lowtown_a2_131070_6z.dds"},
	{0x09D3A560A294C69BULL, "art\\characters\\creatures\\ragedemon\\c_ragedemon.mao"},
	{0x09D3BB5F0F7F4AE5ULL, "llac_w_woundedcoast_0_11.dds"},
	{0x09D3EDA32A9B42D6ULL, "vxlt_0_lowtown_e28_0_0.vlm"},
	{0x09D447D22B22CD5DULL, "ss_lothering.dds"},
	{0x09D475C03E40551BULL, "art\\vfx\\combat\\vfx_c_mer_crack_decal_p.dura.gad"},
	{0x09D4AF09DC923D90ULL, "hlac_ht_keep_3739_1z.dds"},
	{0x09D4C04C5E37B20EULL, "vxlt_0_lowtown_a3_n_c_h2e_0_0.vlm"},
	{0x09D5F196847684B8ULL, "6110442_m.fxe"},
	{0x09D60DED51763EC8ULL, "299841296.ogg"},
	{0x09D60E42CFB49A48ULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_grill_n.dds"},
	{0x09D63F7F95D87C6FULL, "hlac_w_sundermount_262148_0y.dds"},
	{0x09D6819E0B6DF3C3ULL, "zz_vault_debug6203265.cl"},
	{0x09D7157806852B09ULL, "vxdr_primevaltomb_8dm_0_0.vlm"},
	{0x09D72454235ECBB2ULL, "vxca_caverns2_8nz_0_0.vlm"},
	{0x09D738A420313B0EULL, "203133136.ogg"},
	{0x09D746784D0A2750ULL, "6137481_m.fxe"},
	{0x09D84A7F54CEEBF5ULL, "ch2notnt_s2.psh.fxo"},
	{0x09D8F89ABB4C1E3DULL, "montage_subs_demohack6217120.cl"},
	{0x09D936D8A6387E8BULL, "vxlt_0_lowtown_a3_n_c_byw_0_0.vlm"},
	{0x09D9372A27FC6741ULL, "6015955_m.fxe"},
	{0x09D93910A5359722ULL, "6027540_m.fxe"},
	{0x09DA23B1A5463727ULL, "qun221_viscount6084697.cl"},
	{0x09DA2B3E9B175120ULL, "6119460_f.fxe"},
	{0x09DA43AB3951B71BULL, "vxw_sundermount_35f_0_0.vlm"},
	{0x09DA816AA46C84C1ULL, "vxca_caverns1_igx_0_0.vlm"},
	{0x09DA9500FB236456ULL, "pro000_wesley6007723.cl"},
	{0x09DB43B471DF85C3ULL, "695589296.ogg"},
	{0x09DB4BC68518B1F1ULL, "6070378_m.fxe"},
	{0x09DB92C03E46A500ULL, "art\\vfx\\combat\\vfx_c_mer_crack_decal_p.dura.evt"},
	{0x09DBB7DE2ACCF123ULL, "vxw_encounter3_6f_0_0.vlm"},
	{0x09DBD0D30FE9B860ULL, "lt_0_lowtown_a2_n_38_0.anb"},
	{0x09DBD16815DB1F7AULL, "art\\characters\\playercharacter\\humanmale\\textures\\hm_cth_cara_s.dds"},
	{0x09DC2AAE7CC7EAB6ULL, "vxlt_gamlenshouse_n_5e_0_0.vlm"},
	{0x09DC38B696205720ULL, "art\\vfx\\materials\\chunky\\vfx_c_fire_tile_scroll_vt.mao"},
	{0x09DCACD85AE6C44EULL, "744812639.ogg"},
	{0x09DD21F5D895ADD0ULL, "119167267.ogg"},
	{0x09DD49FA9E0E92D1ULL, "art\\levels\\races\\mage\\prop\\_textures\\prp_sepulcher_sa.dds"},
	{0x09DD95AA57289C72ULL, "vxht_brothel_n_53z_0_1.vlm"},
	{0x09DDF9B07DF5F2A5ULL, "vxw_encounter2_camp_k4_0_0.vlm"},
	{0x09DDFFF9DFF6F068ULL, "730295781.ogg"},
	{0x09DE6FD28A639F49ULL, "dae000ip_crowd_mf1_talk13.utp"},
	{0x09DE97A16915FEADULL, "vxht_0_hightown_a3_bcz_0_0.vlm"},
	{0x09DEAF30929EF0B8ULL, "dae021_sister_samea_sta6142178.cl"},
	{0x09DEB6B149525DFDULL, "273715226.ogg"},
	{0x09DED68082FFB3C4ULL, "mag291_demon_marethari6166176.cl"},
	{0x09DEDB39DC743EC9ULL, "vxw_woundedcoast_1od_0_0.vlm"},
	{0x09DEEEA113D02619ULL, "dae000_gamlen6121558.cl"},
	{0x09DF020BB02022A9ULL, "vxht_keep_n_70h_0_0.vlm"},
	{0x09DF696534360763ULL, "gen00fl_dog6117791.cl"},
	{0x09DFBE7D1250BB12ULL, "vxca_caverns1_mines_hw3_0_0.vlm"},
	{0x09DFBF44EB120E93ULL, "vxlt_warehouse_n_eb_0_0.vlm"},
	{0x09DFE5447EC17464ULL, "vxdr_primevaltomb_7zs_0_2.vlm"},
	{0x09E06400B78D190CULL, "6096831_f.fxe"},
	{0x09E07D6516083BB2ULL, "6065933_m.fxe"},
	{0x09E0E5CEB2C5E7ADULL, "zz_qcr_debug6215491.cl"},
	{0x09E14641A00F7F3AULL, "6103787_m.fxe"},
	{0x09E19E80B37A982EULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_candleon.msh"},
	{0x09E1AC80B37AB07FULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_candleon.mao"},
	{0x09E1ACF5F24B0BF0ULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\0pp_post.mao"},
	{0x09E1B0F5F24B1203ULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\0pp_post.mmh"},
	{0x09E1B880B37AC41CULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_candleon.mmh"},
	{0x09E1BEF5F24B2A5DULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\0pp_post.msh"},
	{0x09E209BB8B37142AULL, "gen00fl_dog6115080.cl"},
	{0x09E2C2FB81C96794ULL, "zz_one_debug6074013.cl"},
	{0x09E2DB5A054FE59FULL, "character_character_p138.psh.fxo.dx11.assembly"},
	{0x09E35FAFA13F5006ULL, "lgt311_shw_mer_mettin6156550.cl"},
	{0x09E3A2B062CE813AULL, "484726519.ogg"},
	{0x09E3BA40B00FA747ULL, "qun110_petrice6042305.cl"},
	{0x09E3C17FD7AD2006ULL, "6166294_m.fxe"},
	{0x09E4421BD13735C8ULL, "530771290.ogg"},
	{0x09E45D2BDB7FFEFDULL, "gen_im_trs_chn_light_doc_04.uti"},
	{0x09E47E68EED0FBA4ULL, "hlac_ca_caverns2_5802_58z.dds"},
	{0x09E55CC81D538F7EULL, "hlac_w_encounter2_131072_0x.dds"},
	{0x09E5862C114C8F66ULL, "756587550.ogg"},
	{0x09E5F2C51D9B2E97ULL, "dae000_flemeth6000726.cl"},
	{0x09E5F34042329C49ULL, "6118490_m.fxe"},
	{0x09E5FC57D86F90D3ULL, "ht_0_hightown_a2_36_0_ll.mmh"},
	{0x09E6C26B4DFE2B32ULL, "zz_one_debug6099942.cl"},
	{0x09E6EDCB859B235CULL, "hlac_ht_0_hightown_a3_-196607_0x.dds"},
	{0x09E70DD289D140DCULL, "lt_0_lowtown_docks_a3_n_117_0.phy"},
	{0x09E759AE7BC0BEF6ULL, "hlac_ht_mansion_n_3374_0y.dds"},
	{0x09E7A3BC8F5870DAULL, "6070442_f.fxe"},
	{0x09E7E2D6E5F16B28ULL, "hlac_lt_0_lowtown_docks_a3_-4_3y.dds"},
	{0x09E809F7E79A74EFULL, "prcscr_one.gda"},
	{0x09E82A8D990E501AULL, "vxht_0_hightown_a2_9zd_0_0.vlm"},
	{0x09E858E603A28BB7ULL, "676716628.ogg"},
	{0x09E8733563458572ULL, "671268109.ogg"},
	{0x09E8980278EBC4BCULL, "61233724.ogg"},
	{0x09E8A314749BC253ULL, "vxht_bartrand_n_4hv_0_0.vlm"},
	{0x09E8B00EFC06343DULL, "1054512199.ogg"},
	{0x09E8C4D1C4C0FE6EULL, "6033699_m.fxe"},
	{0x09E8D2999BD92A32ULL, "vxht_0_hightown_a2_bx5_0_0.vlm"},
	{0x09E9057A9581E6B0ULL, "930170233.ogg"},
	{0x09E918DFD062AD69ULL, "vxw_sundermount_36t_0_1.vlm"},
	{0x09E93AB757BFB8D8ULL, "vxdr_primevalentrance_1k8_0_0.vlm"},
	{0x09E9427AD04592B5ULL, "559240996.ogg"},
	{0x09E97834D4A796FDULL, "partypicker.gda"},
	{0x09E97A34D4A79A7AULL, "partypicker.gfx"},
	{0x09E9BDA3770ECA71ULL, "6075078_m.fxe"},
	{0x09E9CC59A74CA043ULL, "vxlt_gallowstemplar_n_3ir_0_0.vlm"},
	{0x09E9E167607879D8ULL, "blurring_additive.mao"},
	{0x09E9E24AFE08324DULL, "art\\levels\\races\\proxy\\props\\4tools\\inv_wall8x3.mmh"},
	{0x09EA29BB1CC309A3ULL, "vxw_encounter3_3i_0_0.vlm"},
	{0x09EA727B08079DDAULL, "llac_lt_0_lowtown_docks_n_65534_6.dds"},
	{0x09EB520D2A31FD4BULL, "vxdr_primevaltomb_4xo_0_0.vlm"},
	{0x09EB82DC43D02AD8ULL, "and111_lirene6084547.cl"},
	{0x09EB98AC1593B672ULL, "173277269.ogg"},
	{0x09EBD43A1503C896ULL, "vxlt_0_lowtown_a2_eny_0_1.vlm"},
	{0x09EC23D5C22CD93DULL, "vxlt_backalley_n_1yj_0_0.vlm"},
	{0x09EC545DC67DC4D1ULL, "hlac_lt_0_lowtown_-65535_0y.dds"},
	{0x09EC6B8D6FEC0B25ULL, "vxw_encounter2_camp_f3_0_0.vlm"},
	{0x09ECE770A823F532ULL, "841637227.ogg"},
	{0x09ED98B27FC178C2ULL, "6101411_m.fxe"},
	{0x09EDFD294D0AB8DCULL, "gen00fl_varric6099451.cl"},
	{0x09EEB7A04A6C8BCEULL, "gen00fl_sebastian6195857.cl"},
	{0x09EF0D12FF383389ULL, "498190422.ogg"},
	{0x09EF2F39210A0D68ULL, "one300ip_worldmap.ncs"},
	{0x09F0080002060598ULL, "pro000_refugee_angry6052170.cl"},
	{0x09F032EE8A447E5EULL, "hlac_lt_0_lowtown_n_-65536_3y.dds"},
	{0x09F04AFBAD871B2FULL, "pro000cr_guard.utc"},
	{0x09F0D534D898C247ULL, "vxlt_gallowstemplar_n_31v_0_0.vlm"},
	{0x09F0ECCC4942CC91ULL, "851042184.ogg"},
	{0x09F1091D310C3B8CULL, "vxdr_primevaltomb_84d_0_0.vlm"},
	{0x09F10BC4DAD32BA9ULL, "follower_banter6169831.cl"},
	{0x09F1734F36BA3580ULL, "llac_lt_0_lowtown_65536_0.dds"},
	{0x09F176B79C312FE4ULL, "333140006.ogg"},
	{0x09F1D6AA1C3A257BULL, "art\\levels\\races\\proxy\\tilesets\\setint\\benstestset\\0ww_slabsnb.mmh"},
	{0x09F1E2AA1C3A3918ULL, "art\\levels\\races\\proxy\\tilesets\\setint\\benstestset\\0ww_slabsnb.mao"},
	{0x09F1F4AA1C3A5865ULL, "art\\levels\\races\\proxy\\tilesets\\setint\\benstestset\\0ww_slabsnb.msh"},
	{0x09F20CFCB5F17D80ULL, "6065957_m.fxe"},
	{0x09F28B1BC8B5EF08ULL, "llac_lt_0_lowtown_a2_n_-131073_2.dds"},
	{0x09F2B1F5F2598728ULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\0pp_post.phy"},
	{0x09F2CDC515DDEFFDULL, "star_yellow.dds"},
	{0x09F2D18E28F1D96EULL, "gen00fl_merrill6121651.cl"},
	{0x09F33E4624E1E1D1ULL, "263945091.ogg"},
	{0x09F34146815DB737ULL, "6088703_f.fxe"},
	{0x09F363B24D956B0BULL, "vxw_blightlands_w9_0_0.vlm"},
	{0x09F37D7450FFD9AAULL, "441521427.ogg"},
	{0x09F38A49102CD854ULL, "vxlt_hangedman_n_2em_0_0.vlm"},
	{0x09F3B6D841D55B0CULL, "6087661_m.fxe"},
	{0x09F3C1A3DA1157BAULL, "439536242.ogg"},
	{0x09F3C5384C16EF97ULL, "zz_ser_debug6078560.cl"},
	{0x09F3C6FC33E1554FULL, "6148350_m.fxe"},
	{0x09F4080E074AFCB2ULL, "vxca_caverns2_8pd_0_0.vlm"},
	{0x09F4A41C36E33D5AULL, "635240222.ogg"},
	{0x09F579C91D3649DCULL, "vxca_caverns1_j28_0_1.vlm"},
	{0x09F5947D6AD99DDDULL, "hlac_lt_darktown_-1_74z.dds"},
	{0x09F5CAF7479EC2AAULL, "ser221_party_comment_found.stg"},
	{0x09F5EB733237B148ULL, "6171847_m.fxe"},
	{0x09F639EE780E51A9ULL, "dae000ip_crowd_mf2_talk3.utp"},
	{0x09F6541A7DF0A96CULL, "122265499.ogg"},
	{0x09F6AAFD9E210AA4ULL, "aip_raiderarcher_base.gda"},
	{0x09F6C58E7C646AD2ULL, "6034367_f.fxe"},
	{0x09F6E96CB587054CULL, "art\\vfx\\combat\\vfx_gib_hm_lgta_rtl.anb"},
	{0x09F7306990AF2673ULL, "6066036_m.fxe"},
	{0x09F7B1A6DED5DF26ULL, "786306303.ogg"},
	{0x09F7BACE9E2CF206ULL, "hlac_ht_playershouse_n_855_36z.dds"},
	{0x09F82837B1433375ULL, "179117506.ogg"},
	{0x09F8339E3DC7D313ULL, "6156119_m.fxe"},
	{0x09F89694D188C108ULL, "6091688_m.fxe"},
	{0x09F8F2E17B95D232ULL, "w_weapon.bnk"},
	{0x09F9DDF42DD8B570ULL, "995976751.ogg"},
	{0x09FA1A3C345C5244ULL, "905875303.ogg"},
	{0x09FA7303AD22DB0DULL, "hlac_lt_0_lowtown_a2_n_c_-131072_0x.dds"},
	{0x09FAA4F26EAE4DE3ULL, "vxlt_warehouse_n_11m_0_0.vlm"},
	{0x09FADB4AFE16990AULL, "art\\levels\\races\\proxy\\props\\4tools\\inv_wall8x3.phy"},
	{0x09FADD6A61E223A5ULL, "vxw_encounter2_camp_a2_0_0.vlm"},
	{0x09FAEB85CE9B841AULL, "909503101.ogg"},
	{0x09FC8BCF33C1CAC1ULL, "zz_followers_banter6158943.cl"},
	{0x09FD1E375E65FCBDULL, "302676281.ogg"},
	{0x09FD33DA0871E99CULL, "art\\characters\\playercharacter\\humanfemale\\textures\\hf_rob_appa_0t.dds"},
	{0x09FE3AFD13EB0C92ULL, "6033542_f.fxe"},
	{0x09FE8F7D2C54700FULL, "656316680.ogg"},
	{0x09FE938DF4CD462CULL, "vxlt_gallowstemplar_5cz_0_0.vlm"},
	{0x09FF0D8ED16DC8E4ULL, "6166726_m.fxe"},
	{0x09FF4C24B8BEEB83ULL, "mer120_hubert6173905.cl"},
	{0x0A00718D10196592ULL, "zz_fol_debug6057008.cl"},
	{0x0A011758E0435A02ULL, "790830175.ogg"},
	{0x0A013772B5A66AD1ULL, "vxht_bartrand_n_66d_0_0.vlm"},
	{0x0A016C96A4BAAEF4ULL, "vxw_bonepitdestroyed_a3_ki_0_0.vlm"},
	{0x0A021E25ED84A07AULL, "hm_danarius.mor"},
	{0x0A026B0EDC677489ULL, "vxdr_primevaltomb_8ur_0_0.vlm"},
	{0x0A03208F575F7AA6ULL, "vxw_encounter2_camp_5a_0_0.vlm"},
	{0x0A0338DCD3D17719ULL, "113623799.ogg"},
	{0x0A03B01ABC6700C9ULL, "165395044.ogg"},
	{0x0A03BB9B6794DBF1ULL, "qcr220_cut_distraction6042317.cl"},
	{0x0A03C3D26A5723EEULL, "hlac_lt_0_lowtown_n_65537_0z.dds"},
	{0x0A04BB3EEB3045CAULL, "hlac_dr_primevaltomb_9621_5y.dds"},
	{0x0A04FD553CE08077ULL, "hlac_ht_0_hightown_a2_-65537_4y.dds"},
	{0x0A0505882DD177EBULL, "6139130_m.fxe"},
	{0x0A05508F35166FCCULL, "vxca_caverns2prc_a4n_0_0.vlm"},
	{0x0A066DE849190A67ULL, "36627740.ogg"},
	{0x0A06D12FDF201BBCULL, "vxlt_0_lowtown_n_f0t_0_0.vlm"},
	{0x0A072405908578F5ULL, "6161278_m.fxe"},
	{0x0A078734D4C0ADDBULL, "partypicker.ncs"},
	{0x0A07873FF7463A99ULL, "6036512_m.fxe"},
	{0x0A078DE87D070FDFULL, "w_encounter2_9_0.msh"},
	{0x0A0793E87D071981ULL, "w_encounter2_9_0.mmh"},
	{0x0A079FE87D072E22ULL, "w_encounter2_9_0.mao"},
	{0x0A0828F00B2771ADULL, "vxca_caverns1_mines_d5z_0_0.vlm"},
	{0x0A0916CF60A46663ULL, "vxlt_warehouse_n_16l_0_0.vlm"},
	{0x0A0918DEB22F1486ULL, "527915924.ogg"},
	{0x0A095DC8AE14953EULL, "zz_and_debug6178602.cl"},
	{0x0A09E68C40905412ULL, "night_probe_3654.mtx"},
	{0x0A09F2A8DBEB366CULL, "vxht_keep_6fu_0_0.vlm"},
	{0x0A0A30D0B2568849ULL, "vxw_woundedcoast_1pi_0_0.vlm"},
	{0x0A0A3A2C5D4EA6A9ULL, "203975519.ogg"},
	{0x0A0A470CEB2A7C0BULL, "vfx_vfx_p4.psh.fxo.dx11"},
	{0x0A0AC838D47C3C56ULL, "vxlt_0_lowtown_hat_0_0.vlm"},
	{0x0A0AE51BEB8E5A22ULL, "155516531.ogg"},
	{0x0A0AFCACAD51D8F5ULL, "hlac_lt_hangedman_n_-1_22y.dds"},
	{0x0A0B1F172C629032ULL, "vxlt_gallowstemplar_5an_0_0.vlm"},
	{0x0A0B59B3E40CA763ULL, "142089744.ogg"},
	{0x0A0BB1DDF85CA94FULL, "mag210_arianni6056861.cl"},
	{0x0A0BE094C9CB4587ULL, "hlac_lt_0_lowtown_a2_n_c_131073_2x.dds"},
	{0x0A0C28DBD6ABACF5ULL, "6035859_f.fxe"},
	{0x0A0C52C89ABDEF6BULL, "vxw_sundermount_370_0_0.vlm"},
	{0x0A0CDCB0DF56CF93ULL, "art\\characters\\playercharacter\\elffemale\\bdy_mrlb.phy"},
	{0x0A0D37740DB1337AULL, "gen00cs_cutscene_end.ncs"},
	{0x0A0DB3FCB5DF6C03ULL, "6114456_m.fxe"},
	{0x0A0F341618CD756CULL, "vxht_0_hightown_n_buc_0_0.vlm"},
	{0x0A0F4F0958BEA1C6ULL, "vxht_mansion2_3c1_0_0.vlm"},
	{0x0A0F86C753F63F04ULL, "llac_lt_0_lowtown_-196608_0.dds"},
	{0x0A1027B161EDC739ULL, "vxca_caverns2prc_aba_0_0.vlm"},
	{0x0A106DB89ED9CAEDULL, "266163076.ogg"},
	{0x0A10E2723EDB6132ULL, "vxht_0_hightown_bub_0_0.vlm"},
	{0x0A10E4884B77D513ULL, "6087129_m.fxe"},
	{0x0A1151422707F173ULL, "hlac_ca_caverns2_5802_47y.dds"},
	{0x0A11573FFB2B6C0EULL, "vxw_encounter2_64_0_0.vlm"},
	{0x0A11B270E83F1BD3ULL, "hlac_lt_0_lowtown_a2_n_131071_2x.dds"},
	{0x0A11E35A6C3B07FBULL, "vxht_keep_57z_0_0.vlm"},
	{0x0A11FE633D575F78ULL, "vxht_mansion2b_32h_0_0.vlm"},
	{0x0A1231853B151D5FULL, "6087600_m.fxe"},
	{0x0A13CC1A3DC33C40ULL, "follower_banter6141870.cl"},
	{0x0A147C1812432CA2ULL, "6163235_m.fxe"},
	{0x0A1538F26FFCD6DBULL, "lt_0_lowtown_docks_a3_117_0.phy"},
	{0x0A15613940E81010ULL, "dae000_mother6099211.cl"},
	{0x0A159F772E6BB036ULL, "vxlt_0_lowtown_docks_a3_9fj_0_0.vlm"},
	{0x0A16A41EDF630031ULL, "195044059.ogg"},
	{0x0A16C456ACB6BD2EULL, "6053697_m.fxe"},
	{0x0A16CBC37739CD31ULL, "sst_ss_sebastian.gda"},
	{0x0A17E4A07B1EB8C3ULL, "one100dg_elren6014256.cl"},
	{0x0A184B55EA35DDDFULL, "121974320.ogg"},
	{0x0A186485CDE14EECULL, "vxht_keep_6cj_0_0.vlm"},
	{0x0A18DE2FB9586C11ULL, "lgt210pt_frame_follow.ncs"},
	{0x0A18EFA4A8820F8AULL, "vxw_encounter2_jd_0_0.vlm"},
	{0x0A19643187FEC48CULL, "var_creature.gda"},
	{0x0A1975E05A6B71E2ULL, "plt_cod_itm_vague_blade.plo"},
	{0x0A1986B8B2EA7093ULL, "vxlt_warehouse_n_yf_0_0.vlm"},
	{0x0A1990F41E58A8B2ULL, "vxlt_gallowstemplar_5jo_0_0.vlm"},
	{0x0A19C05C18ACCE2AULL, "838152494.ogg"},
	{0x0A19E535004B2491ULL, "100095833.ogg"},
	{0x0A1A002E441304E2ULL, "6103888_m.fxe"},
	{0x0A1A3C5E8770AFF2ULL, "250358216.ogg"},
	{0x0A1AB633325885B8ULL, "lgt110_gxa_vgk_silver_order6117038.cl"},
	{0x0A1AC4A58CB407EBULL, "vxw_sundermount_323_0_0.vlm"},
	{0x0A1AD5F3715E25FBULL, "vxca_cavern1_haunted_j2a_0_2.vlm"},
	{0x0A1B7747D8B9772CULL, "vxlt_gallowstemplar_5ix_0_0.vlm"},
	{0x0A1B7BFBB827368FULL, "lt_0_lowtown_docks_n_39_0_lc.mmh"},
	{0x0A1BAFD3101F6388ULL, "lt_0_lowtown_a2_n_38_0.rml"},
	{0x0A1BB994A5EEA3D4ULL, "art\\characters\\playercharacter\\humanfemale\\textures\\hf_arm_nud_0d.dds"},
	{0x0A1BC0A4A4DB0FF9ULL, "vxlt_0_lowtown_a2_n_c_gy9_0_0.vlm"},
	{0x0A1BFDD1D1D67AC5ULL, "6115343_m.fxe"},
	{0x0A1C663970E5542FULL, "w_encounter3_85_0.phy"},
	{0x0A1D1CA3F8D957ECULL, "578594154.ogg"},
	{0x0A1D663EBE4A90AFULL, "8129522.ogg"},
	{0x0A1D726665146954ULL, "vxht_bartrand_n_47h_0_0.vlm"},
	{0x0A1DC0E64AB4BA46ULL, "vxht_mansion2_3f2_0_0.vlm"},
	{0x0A1E346D141DC793ULL, "vxlt_darktown_4un_0_0.vlm"},
	{0x0A1FC91CED21848EULL, "vxw_encounter2_15_0_0.vlm"},
	{0x0A20F42F525E009FULL, "6143218_m.fxe"},
	{0x0A21218D0CB42A2EULL, "gen00fl_anders6103781.cl"},
	{0x0A212EE5FE3DD835ULL, "vxht_keep_3il_0_0.vlm"},
	{0x0A2136AD321ABFE0ULL, "art\\characters\\playercharacter\\humanmale\\mt_bas_ulri.phy"},
	{0x0A217E762C9B4312ULL, "6172282_m.fxe"},
	{0x0A21E04ECA58FC25ULL, "lowtown_guard_chase6169518.cl"},
	{0x0A21F93251F14665ULL, "1028545341.ogg"},
	{0x0A22029640AE15A9ULL, "6204807_m.fxe"},
	{0x0A2269EBF6901DCFULL, "hlac_lt_0_lowtown_docks_n_131071_4y.dds"},
	{0x0A22896A7D97BA82ULL, "vxht_brothel_n_3s9_0_0.vlm"},
	{0x0A229A0D63B18C32ULL, "vxht_0_hightown_a2_bl1_0_0.vlm"},
	{0x0A22E50C9762005EULL, "art\\characters\\creatures\\mubari\\animation\\com_pee.ani"},
	{0x0A22ED182ACCBD9DULL, "vxht_0_hightown_a3_c9y_0_0.vlm"},
	{0x0A233F203801072BULL, "255566682.ogg"},
	{0x0A238F7804364537ULL, "596938124.ogg"},
	{0x0A239119CCC58196ULL, "art\\characters\\weapons\\longbows\\w_lbw_001a.msh"},
	{0x0A239B19CCC59294ULL, "art\\characters\\weapons\\longbows\\w_lbw_001a.mmh"},
	{0x0A239F19CCC599A7ULL, "art\\characters\\weapons\\longbows\\w_lbw_001a.mao"},
	{0x0A23EB04BDE0F833ULL, "llac_lt_0_lowtown_131070_0.dds"},
	{0x0A23EBF8B693FE08ULL, "hlac_ht_0_hightown_n_-65536_4x.dds"},
	{0x0A244B4BAC50E835ULL, "716372601.ogg"},
	{0x0A2477B848097995ULL, "cry_val_b02.mao"},
	{0x0A24CBB6BFEA6616ULL, "147902997.ogg"},
	{0x0A24CECD47950B09ULL, "6053863_m.fxe"},
	{0x0A24D05544439A97ULL, "hlac_ca_cavern1_haunted_24424_18x.dds"},
	{0x0A25CCF542FE52E0ULL, "vxlt_0_lowtown_docks_a3_n_cx3_0_0.vlm"},
	{0x0A261259B529A550ULL, "6188714_f.fxe"},
	{0x0A262F4EB63CEEE2ULL, "6141503_m.fxe"},
	{0x0A267978F672C23DULL, "gen00fl_dog6167668.cl"},
	{0x0A26E6D1DF784379ULL, "glow.mao"},
	{0x0A2727929788CBCAULL, "vxht_chantry_n_31t_0_0.vlm"},
	{0x0A277236A494AD32ULL, "character_rendertodepth_p6.psh.fxo.dx11"},
	{0x0A278E67AE04F63DULL, "261125107.ogg"},
	{0x0A28C3781DC59273ULL, "6102143_m.fxe"},
	{0x0A28CE5F886194A6ULL, "vxca_caverns1_j2c_0_0.vlm"},
	{0x0A290E6CE88D64DDULL, "gen00fl_anders6096600.cl"},
	{0x0A2936827EAA206BULL, "vxw_sundermount_392_0_0.vlm"},
	{0x0A2A3343D6EF2B6EULL, "gen00fl_anders6125622.cl"},
	{0x0A2A6FAA06109A13ULL, "6057357_m.fxe"},
	{0x0A2AB0A41DE72386ULL, "and311ar_andersshop.are"},
	{0x0A2AC41824BDEF6AULL, "dae000_flemeth6000963.cl"},
	{0x0A2B554A6C8A6275ULL, "vxlt_darktown_2uz_0_0.vlm"},
	{0x0A2B9D435967F590ULL, "6027722_m.fxe"},
	{0x0A2BEBF011BC87ACULL, "face0_ds2.psh.fxo"},
	{0x0A2C73E87D25FFAAULL, "w_encounter2_9_0.rml"},
	{0x0A2C9F3D8B178820ULL, "6161086_m.fxe"},
	{0x0A2D1A45DFEC5AD2ULL, "430055694.ogg"},
	{0x0A2D3DB0154DC21AULL, "6156873_m.fxe"},
	{0x0A2D85DBC3F8019EULL, "m_dae310ar_docks.bnk"},
	{0x0A2DFA2CAE7A9B3EULL, "art\\vfx\\combat\\vfx_glb_ele_e_major_c.impa.ani"},
	{0x0A2E043882F9C39CULL, "hlac_ht_0_hightown_a3_-1_2x.dds"},
	{0x0A2E93C5DBDC9994ULL, "fxp_pillar_sym05.dds"},
	{0x0A2F3349E59C7108ULL, "990223814.ogg"},
	{0x0A2FFD0C976CB32FULL, "art\\characters\\creatures\\mubari\\animation\\com_pee.evt"},
	{0x0A304AFD3DAADF7CULL, "751881617.ogg"},
	{0x0A305DA8B7F232C4ULL, "lt_0_lowtown_docks_a3_n_233_0.msh"},
	{0x0A306FA8B7F2512DULL, "lt_0_lowtown_docks_a3_n_233_0.mao"},
	{0x0A3073A8B7F25866ULL, "lt_0_lowtown_docks_a3_n_233_0.mmh"},
	{0x0A315EF51CC2D61DULL, "vxht_0_hightown_a3_c2x_0_0.vlm"},
	{0x0A32198698FE8554ULL, "552911049.ogg"},
	{0x0A323A001F6654B9ULL, "art\\levels\\ztools\\default\\generic\\9cc_beam.msh"},
	{0x0A3244001F6665A7ULL, "art\\levels\\ztools\\default\\generic\\9cc_beam.mmh"},
	{0x0A3248001F666CF4ULL, "art\\levels\\ztools\\default\\generic\\9cc_beam.mao"},
	{0x0A324EE6ABF187D4ULL, "gen_im_arm_cht_nobd_t2.uti"},
	{0x0A326C34336BF838ULL, "324063312.ogg"},
	{0x0A3274603E7F3D35ULL, "vxdr_primevaltomb_50p_0_0.vlm"},
	{0x0A32AAA3F90C4ED3ULL, "6001008_m.fxe"},
	{0x0A32D4B12F3841DCULL, "6114849_f.fxe"},
	{0x0A32E90E00819E8FULL, "docks_male_human_worker_sta6176038.cl"},
	{0x0A3394E87D2C56FEULL, "w_encounter2_9_0.phy"},
	{0x0A339B653636BA7DULL, "40792108.ogg"},
	{0x0A33B19FC803300CULL, "637940195.ogg"},
	{0x0A340AA4C45018E9ULL, "vxht_brothel_n_6bp_0_0.vlm"},
	{0x0A3424FA46CDE976ULL, "lt_gallowscourtyard_n_mmi.gff"},
	{0x0A34817817BFB8D4ULL, "489252282.ogg"},
	{0x0A34AA4CE05F58BFULL, "vxht_keep_6fz_0_0.vlm"},
	{0x0A358B98588F36D7ULL, "189528705.ogg"},
	{0x0A358E5DFDC537B2ULL, "llac_lt_0_lowtown_a3_n_c_131073_0.dds"},
	{0x0A35A4D641BC7E9FULL, "mcr310_argument6127560.cl"},
	{0x0A35D35E8C6E408AULL, "vxw_encounter2_pf_0_0.vlm"},
	{0x0A35E72A8ACB95B4ULL, "vxlt_darktown_514_0_0.vlm"},
	{0x0A362AA201664AC0ULL, "vxlt_undercityserialkiller_1le_0_0.vlm"},
	{0x0A3674AE0244D9B2ULL, "vxlt_gallowstemplar_5pi_0_0.vlm"},
	{0x0A36BBF7B3E4BD08ULL, "hlac_ca_caverns1_24425_9y.dds"},
	{0x0A36E7CF3EE1AFA9ULL, "vxlt_0_lowtown_a2_n_c_e8l_0_0.vlm"},
	{0x0A36FE0C9772D3D8ULL, "art\\characters\\creatures\\mubari\\animation\\com_pee.gad"},
	{0x0A3713F20A16F183ULL, "6121759_m.fxe"},
	{0x0A37958212215624ULL, "979312894.ogg"},
	{0x0A37C60679C90116ULL, "2760946.ogg"},
	{0x0A38328F7F2F07DFULL, "hlac_ht_keep_n_3731_0y.dds"},
	{0x0A3852B3A255D9CAULL, "781781235.ogg"},
	{0x0A38F7AA1C765FB0ULL, "art\\levels\\races\\proxy\\tilesets\\setint\\benstestset\\0ww_slabsnb.phy"},
	{0x0A397F65DA1DFC23ULL, "vxht_0_hightown_n_b8c_0_0.vlm"},
	{0x0A398F5E770F8922ULL, "one300cr_ggt_thug5.utc"},
	{0x0A39DF9F78B52902ULL, "323845168.ogg"},
	{0x0A3A924BFFCE37CDULL, "65532006.ogg"},
	{0x0A3AC971A3A9A23FULL, "vxlt_gallowstemplar_n_45m_0_0.vlm"},
	{0x0A3B122CAE854E0FULL, "art\\vfx\\combat\\vfx_glb_ele_e_major_c.impa.evt"},
	{0x0A3B5860A000F202ULL, "character_character_p94.psh.fxo.dx11.assembly"},
	{0x0A3B72BDAD3FE7F0ULL, "one220_yevhen6102812.cl"},
	{0x0A3C76BE9D7B3EFFULL, "1002404453.ogg"},
	{0x0A3CFC92013E107BULL, "6111764_f.fxe"},
	{0x0A3D38641326EF90ULL, "253786197.ogg"},
	{0x0A3D5303E7C61ED5ULL, "vxht_0_hightown_a3_621_0_0.vlm"},
	{0x0A3D86FC1A00673BULL, "art\\levels\\races\\dwarf\\common\\int\\_textures\\til_cavernlavaedge_e.dds"},
	{0x0A3E2FF2372BB0FBULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\9uw_tarpoutc.mmh"},
	{0x0A3E3BF2372BC498ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\9uw_tarpoutc.mao"},
	{0x0A3E4DF2372BE3E5ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\9uw_tarpoutc.msh"},
	{0x0A3E7B1140834B00ULL, "art\\levels\\races\\proxy\\props\\4tools\\inv_wall8x3_0.pwk"},
	{0x0A3EC9F37448B3C2ULL, "vxw_sundermount_20m_0_0.vlm"},
	{0x0A3ED7940AD9E375ULL, "w_bonepitmining_a2_254_0.msh"},
	{0x0A3ED8A3660A1C19ULL, "art\\vfx\\combat\\vfx_glb_general_imp_c.impa.ani"},
	{0x0A3ED9940AD9E6EBULL, "w_bonepitmining_a2_254_0.mmh"},
	{0x0A3EDB972C45FA8DULL, "673834792.ogg"},
	{0x0A3EE5940AD9FA88ULL, "w_bonepitmining_a2_254_0.mao"},
	{0x0A3EF5D1DF8CEE07ULL, "glow.dds"},
	{0x0A3EFDBDC4226FB9ULL, "w_bonepit_137_0.msh"},
	{0x0A3F07BDC42280A7ULL, "w_bonepit_137_0.mmh"},
	{0x0A3F0BBDC42287F4ULL, "w_bonepit_137_0.mao"},
	{0x0A3F54DFCD9313B2ULL, "6080508_f.fxe"},
	{0x0A3FD0D20EB8EE9DULL, "vxht_0_hightown_a3_c7w_0_0.vlm"},
	{0x0A4055F4B9BF1105ULL, "vxht_mansion2b_32q_0_0.vlm"},
	{0x0A40A119AF3A0A5AULL, "art\\characters\\playercharacter\\humanfemale\\cth_nobe.phy"},
	{0x0A411FF27021E77AULL, "lt_0_lowtown_docks_a3_117_0.msh"},
	{0x0A412DF27021FEB3ULL, "lt_0_lowtown_docks_a3_117_0.mao"},
	{0x0A4131F2702205C0ULL, "lt_0_lowtown_docks_a3_117_0.mmh"},
	{0x0A41FFDD2F3ED7DEULL, "vxw_encounter1_cn_0_0.vlm"},
	{0x0A42089435983EDEULL, "vxlt_hangedman_n_rr_0_0.vlm"},
	{0x0A42132CAE8B6EB8ULL, "art\\vfx\\combat\\vfx_glb_ele_e_major_c.impa.gad"},
	{0x0A427C81B6463169ULL, "vxht_brothel_n_6ow_0_0.vlm"},
	{0x0A42B8D85CA6064CULL, "art\\vfx\\cinematic\\vfx_lightbeam02_p.dura.ani"},
	{0x0A431C29D255713FULL, "vxht_keep_6ce_0_0.vlm"},
	{0x0A433E18CE10643CULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\drunk_gesture_left_1.ani"},
	{0x0A436E2334589C1BULL, "vxlt_darktown_2qp_0_0.vlm"},
	{0x0A43AAD386114316ULL, "w_encounter3_35_0.rml"},
	{0x0A446C38222CDC01ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\reactions\\rct_die_lp_v3.ani"},
	{0x0A459AC5DBEF84C2ULL, "fxp_pillar_sym05.mao"},
	{0x0A45B3A3660FFC5BULL, "art\\vfx\\combat\\vfx_glb_general_imp_c.impa.gad"},
	{0x0A46356802182CACULL, "6020445_f.fxe"},
	{0x0A46729A12E9F458ULL, "vxlt_warehouse_n_12w_0_0.vlm"},
	{0x0A46CED181DA3D18ULL, "mag360_alain6044657.cl"},
	{0x0A46EF650BBEF90EULL, "art\\levels\\races\\vfx_models\\prp_lightbeamsml.mmh"},
	{0x0A46F3EE92E617E6ULL, "6111289_m.fxe"},
	{0x0A46FB650BBF0DA5ULL, "art\\levels\\races\\vfx_models\\prp_lightbeamsml.mao"},
	{0x0A4709650BBF25FCULL, "art\\levels\\races\\vfx_models\\prp_lightbeamsml.msh"},
	{0x0A4753AD323BC075ULL, "art\\characters\\playercharacter\\humanmale\\mt_bas_ulri.msh"},
	{0x0A4755AD323BC3EBULL, "art\\characters\\playercharacter\\humanmale\\mt_bas_ulri.mmh"},
	{0x0A476EDD513666BFULL, "681601438.ogg"},
	{0x0A482AF2D41E6689ULL, "art\\characters\\playercharacter\\elffemale\\ef_boo_lgte.phy"},
	{0x0A484D49D926F237ULL, "art\\vfx\\models\\vfx_arm_shield.mmh"},
	{0x0A486349D92717A9ULL, "art\\vfx\\models\\vfx_arm_shield.msh"},
	{0x0A48920ACE9CA84CULL, "793811975.ogg"},
	{0x0A48D256C83CF171ULL, "hlac_lt_0_lowtown_n_-3_3z.dds"},
	{0x0A4926303C7D2FD0ULL, "llac_lt_gallowscourtyard_n_131071_2.dds"},
	{0x0A4940F969F2C424ULL, "6002796_m.fxe"},
	{0x0A496BEA20117D14ULL, "539622371.ogg"},
	{0x0A49C5D85CAC3A8EULL, "art\\vfx\\cinematic\\vfx_lightbeam02_p.dura.gad"},
	{0x0A4A1F76F26DD3BCULL, "ser270_templar6165681.cl"},
	{0x0A4A4B18CE16997EULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\drunk_gesture_left_1.gad"},
	{0x0A4A773822315AA3ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\reactions\\rct_die_lp_v3.gad"},
	{0x0A4A8412F9D0DAF4ULL, "410329322.ogg"},
	{0x0A4AA3CFB8125073ULL, "vxlt_foundry_n_v5_0_0.vlm"},
	{0x0A4AA3D38617563AULL, "w_encounter3_35_0.phy"},
	{0x0A4ACB19C7EF65F6ULL, "373068092.ogg"},
	{0x0A4AF1E7B7B33E32ULL, "vxlt_0_lowtown_a2_hg2_0_0.vlm"},
	{0x0A4B0AADAC918E16ULL, "dae300_alistair_king6133824.cl"},
	{0x0A4C8DBC1FC7B0F9ULL, "llac_w_encounter2_-65537_0.dds"},
	{0x0A4C94741E5638BBULL, "6071524_m.fxe"},
	{0x0A4CD0A366164C40ULL, "art\\vfx\\combat\\vfx_glb_general_imp_c.impa.evt"},
	{0x0A4D3BD0663ECC42ULL, "vxw_sundermount_25n_0_0.vlm"},
	{0x0A4D402DBE583DECULL, "6094904_m.fxe"},
	{0x0A4DB3392E72E386ULL, "hlac_lt_undercitydungeon_-1_41x.dds"},
	{0x0A4DBA13513FB8FFULL, "hlac_lt_0_lowtown_docks_a3_n_65533_1x.dds"},
	{0x0A4DE9AE6136FC18ULL, "llac_lt_0_lowtown_196607_0.dds"},
	{0x0A4E319E40698C6EULL, "gen00fl_fenris_base_max.stg"},
	{0x0A4E7F61AA4D4853ULL, "fen210_orana6109753.cl"},
	{0x0A4EB4C0A240F8BFULL, "hlac_dr_dungeon_3216_12z.dds"},
	{0x0A4F437D21E06250ULL, "vxlt_0_lowtown_docks_n_cq3_0_0.vlm"},
	{0x0A4F4DB483477C38ULL, "vxlt_gallowscourtyard_75t_0_0.vlm"},
	{0x0A4F797B7CAFC537ULL, "vxlt_gallowscourtyard_n_62d_0_0.vlm"},
	{0x0A4FAD6AEF22B924ULL, "vxlt_0_lowtown_cmq_0_0.vlm"},
	{0x0A4FC0D85CB09E6DULL, "art\\vfx\\cinematic\\vfx_lightbeam02_p.dura.evt"},
	{0x0A5004A472E282CEULL, "zz_prmpt_debug6198217.cl"},
	{0x0A5071BA2134F05EULL, "vxw_encounter1_ho_0_0.vlm"},
	{0x0A50B7A41E07FF71ULL, "and311ar_andersshop.lst"},
	{0x0A50C3BB5FFCF9C0ULL, "llac_w_encounter2_327684_0.dds"},
	{0x0A50ED4887F9AFF8ULL, "6208464_m.fxe"},
	{0x0A50EE5EA83C49E9ULL, "vxht_brothel_n_6hv_0_0.vlm"},
	{0x0A50EEE8980DB240ULL, "6125333_f.fxe"},
	{0x0A51198CBDE47178ULL, "vxlt_0_lowtown_docks_a3_n_cwr_0_0.vlm"},
	{0x0A512618CE1C789DULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\drunk_gesture_left_1.evt"},
	{0x0A516E26FE688220ULL, "6147932_m.fxe"},
	{0x0A517279E753B2CEULL, "w_sundermount_217_0.phy"},
	{0x0A5197437F3D5670ULL, "one100dg_elren6058864.cl"},
	{0x0A51AA3F6E3B8FB9ULL, "vxlt_0_lowtown_a2_n_c_hsx_0_0.vlm"},
	{0x0A51E000264EB49BULL, "vxlt_darktown_2ts_0_0.vlm"},
	{0x0A51E9D2AF28DE6EULL, "995692729.ogg"},
	{0x0A51F6FCD3E39415ULL, "778360758.ogg"},
	{0x0A52BE36B21E191FULL, "vxht_0_hightown_a2_brj_0_0.vlm"},
	{0x0A537C84CDF2258EULL, "111183094.ogg"},
	{0x0A53E544A53CD069ULL, "llac_lt_0_lowtown_n_-1_7.dds"},
	{0x0A54A744F40E452CULL, "6027545_f.fxe"},
	{0x0A54B7630B103EA5ULL, "6015803_m.fxe"},
	{0x0A54C7ADF12E33E4ULL, "1048997341.ogg"},
	{0x0A54C8CB0C6FE5CDULL, "store_dae310_weapons.utm"},
	{0x0A54F3E2B0E02384ULL, "185208169.ogg"},
	{0x0A5617D91826ED10ULL, "tent_n.dds"},
	{0x0A5720EA5AAB4099ULL, "hlac_ca_caverns2_5802_46x.dds"},
	{0x0A574863FFED593FULL, "6077947_f.fxe"},
	{0x0A574C2D35BA2EF3ULL, "nat380_sibling6207560.cl"},
	{0x0A577FEBC70E2041ULL, "6121205_m.fxe"},
	{0x0A5789E674F707EDULL, "vxdr_primevaltomb_75a_0_0.vlm"},
	{0x0A57E8D2A682BBB1ULL, "art\\vfx\\combat\\material__5.mao"},
	{0x0A5803BDE80B223EULL, "vxlt_hangedman_n_2xa_0_1.vlm"},
	{0x0A587492BB3749E7ULL, "461559694.ogg"},
	{0x0A58B7A18D73E859ULL, "984695978.ogg"},
	{0x0A595CD4D28B7758ULL, "one100dg_kelder6060541.cl"},
	{0x0A5A8CADFE63B6B6ULL, "464352406.ogg"},
	{0x0A5AEAFE6693F417ULL, "art\\levels\\races\\ferelden\\props\\lowerclass\\prp_chaircutup_0.pwk"},
	{0x0A5AF3A02DB52FFCULL, "art\\levels\\races\\mage\\prop\\_textures\\prp_vial_sa.dds"},
	{0x0A5B3EFB266F90C3ULL, "lowtown_male_alienage1_sta.bnk"},
	{0x0A5B474AEB24F39AULL, "gen00cs_cs_end_music_int_1.ncs"},
	{0x0A5B6CD7BE750C9CULL, "art\\characters\\playercharacter\\elfmale\\textures\\em_glv_lgtp_0d.dds"},
	{0x0A5B9DA914776806ULL, "hlac_lt_gallowstemplar_n_2383_10z.dds"},
	{0x0A5BBB53143D63FBULL, "214262493.ogg"},
	{0x0A5C21DE99470A09ULL, "481626891.ogg"},
	{0x0A5C24DDECA62497ULL, "zz_mag_debug6053423.cl"},
	{0x0A5C34155984B016ULL, "6043380_m.fxe"},
	{0x0A5C822F26136141ULL, "vxca_caverns1_i2w_0_0.vlm"},
	{0x0A5CAB4792BC7B6AULL, "zz_dre_debug6069193.cl"},
	{0x0A5CB97DA1004011ULL, "416003643.ogg"},
	{0x0A5CFFD39358B41FULL, "lt_0_lowtown_docks_n_104_0.phy"},
	{0x0A5DA0463F02ECF4ULL, "vxht_bartrand_n_5w1_0_0.vlm"},
	{0x0A5DA09C1A00C1CBULL, "fxa_evl_dusts.mao"},
	{0x0A5DF635EA045451ULL, "6180675_m.fxe"},
	{0x0A5E96B8A0772ABBULL, "hlac_lt_0_lowtown_docks_-131073_3z.dds"},
	{0x0A5EA72A7A4E2F8FULL, "vxlt_gallowscourtyard_n_6xr_0_0.vlm"},
	{0x0A5F20FB267346F3ULL, "lowtown_male_alienage1_sta.cnv"},
	{0x0A5F4B70B74626C4ULL, "hlac_lt_0_lowtown_docks_65534_5x.dds"},
	{0x0A5F7B37C9530CFDULL, "784097550.ogg"},
	{0x0A5FD4958189B24DULL, "200528055.ogg"},
	{0x0A5FE372295620FEULL, "6043897_f.fxe"},
	{0x0A6111E2F87F681EULL, "vxlt_warehouse_11o_0_0.vlm"},
	{0x0A6134F33B01BCFDULL, "t1_mul_re2.tnt"},
	{0x0A61D1FA5E310B57ULL, "zz_rdr_debug6066631.cl"},
	{0x0A61D229243F820AULL, "l_dae211ar_gamlens_house_nt.wbd"},
	{0x0A61D5B851F6A49EULL, "6037388_f.fxe"},
	{0x0A621669DA787CE9ULL, "gen00fl_merrill6162123.cl"},
	{0x0A6270EE5B9007EDULL, "6040830_m.fxe"},
	{0x0A62C1843E8A2E26ULL, "art\\vfx\\combat\\vfx_r_gn2_impact_act_c.impa.ani"},
	{0x0A62F630A21E3165ULL, "839946880.ogg"},
	{0x0A63022986448B35ULL, "vxw_bonepit_yr_0_0.vlm"},
	{0x0A63287B7B2E2F02ULL, "343617888.ogg"},
	{0x0A646355CEE93506ULL, "vxht_0_hightown_a3_bwa_0_0.vlm"},
	{0x0A6464FFAE01D77DULL, "hm_nathaniel.mor"},
	{0x0A647F8CFE6607DFULL, "follower_banter6165662.cl"},
	{0x0A64F916A6FA5630ULL, "clf260_harley6069679.cl"},
	{0x0A659B4DA7A1A676ULL, "plt_cod_itm_last_descent.plo"},
	{0x0A65AF4699D178F0ULL, "dae210_gate_guard6085082.cl"},
	{0x0A66A9B7FD6604EBULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\1pp_keepthin_1.mao"},
	{0x0A66AB7205DB5D9AULL, "leave_f.mao"},
	{0x0A672D9E7547E349ULL, "571489586.ogg"},
	{0x0A6766B0D10C2479ULL, "gen_im_trs_qun_epic_sword.uti"},
	{0x0A676ADBE3CF2EFEULL, "vxht_0_hightown_a3_boq_0_0.vlm"},
	{0x0A67A31D9A797705ULL, "vxca_caverns1_mines_cn5_0_1.vlm"},
	{0x0A687F1F76DF0043ULL, "gen00fl_dog6166430.cl"},
	{0x0A687F640C8622FBULL, "art\\characters\\weapons\\staff\\textures\\w_stf_010a_0d.dds"},
	{0x0A68A1AC9C772DF4ULL, "6068496_f.fxe"},
	{0x0A68B80C97616C17ULL, "vxht_0_hightown_9ux_0_0.vlm"},
	{0x0A699C7875E75902ULL, "vxht_bartrand_n_6rg_0_0.vlm"},
	{0x0A69C28CE86E59B5ULL, "6040889_m.fxe"},
	{0x0A69FB874E78F96AULL, "vxlt_0_lowtown_a2_n_c_fga_0_0.vlm"},
	{0x0A6A504DA92F730BULL, "vxlt_darktown_43d_0_0.vlm"},
	{0x0A6A721ED584DEF1ULL, "6005030_f.fxe"},
	{0x0A6A75EB970E6247ULL, "gen00fl_dog6107350.cl"},
	{0x0A6AB3B706CA9361ULL, "art\\characters\\creatures\\crowds\\humanmale\\animation\\custom\\patrolling.gad"},
	{0x0A6AF0BDC44794DCULL, "w_bonepit_137_0.phy"},
	{0x0A6B4069A9F11183ULL, "llac_w_encounter3_-131075_0.dds"},
	{0x0A6C122330F90574ULL, "vxht_bartrand_n_5z6_0_0.vlm"},
	{0x0A6C24A01B7E55E1ULL, "llac_ht_0_hightown_n_-65534_1.dds"},
	{0x0A6C5DC696AC134CULL, "vxw_woundedcoast_1ge_0_0.vlm"},
	{0x0A6D15DFF5A18090ULL, "vxw_blightlands_by_0_0.vlm"},
	{0x0A6D59CEBB160167ULL, "6161531_m.fxe"},
	{0x0A6D6B6C1745D860ULL, "dae200cr_viscount6087233.cl"},
	{0x0A6DED236DC79A22ULL, "6132493_m.fxe"},
	{0x0A6E00D393672924ULL, "lt_0_lowtown_docks_n_104_0.mmh"},
	{0x0A6E04D393673077ULL, "lt_0_lowtown_docks_n_104_0.mao"},
	{0x0A6E16D393674EC6ULL, "lt_0_lowtown_docks_n_104_0.msh"},
	{0x0A6E57DBC42F4AA4ULL, "m_dae310ar_docks.wbd"},
	{0x0A6EA2BD46A3A671ULL, "768846778.ogg"},
	{0x0A6ED5B0DFA98E08ULL, "art\\characters\\playercharacter\\elffemale\\bdy_mrlb.mmh"},
	{0x0A6EE1B0DFA9A2ABULL, "art\\characters\\playercharacter\\elffemale\\bdy_mrlb.mao"},
	{0x0A6EF3B0DFA9C132ULL, "art\\characters\\playercharacter\\elffemale\\bdy_mrlb.msh"},
	{0x0A6F24D413EFF49AULL, "hlac_w_woundedcoast_-131068_0y.dds"},
	{0x0A6F93DBA3EFECDFULL, "6030493_m.fxe"},
	{0x0A6FA9843E948F97ULL, "art\\vfx\\combat\\vfx_r_gn2_impact_act_c.impa.evt"},
	{0x0A6FEE2E7E8AD70AULL, "vxw_blightlands_6o_0_0.vlm"},
	{0x0A70B4B706CF000FULL, "art\\characters\\creatures\\crowds\\humanmale\\animation\\custom\\patrolling.ani"},
	{0x0A70EA7B46DD7E21ULL, "6119710_m.fxe"},
	{0x0A713C4C55B3EB65ULL, "vxdr_0_deeproads_exb_0_0.vlm"},
	{0x0A716FC2EF3037C8ULL, "hlac_lt_gallowstemplar_2391_7x.dds"},
	{0x0A717D9C22D95BCBULL, "594828869.ogg"},
	{0x0A7215C7A57F2219ULL, "llac_ht_mansion2_818_38.dds"},
	{0x0A7261B2792305CFULL, "277415962.ogg"},
	{0x0A72B39E5D1064D6ULL, "fen110_fenris6118126.cl"},
	{0x0A72D5F95A3F9EC7ULL, "vxlt_gallowstemplar_n_3di_0_0.vlm"},
	{0x0A72F84C0FDBE9FFULL, "6071949_f.fxe"},
	{0x0A72F978F3B7E82CULL, "gen00fl_dog6165728.cl"},
	{0x0A73507AF3AEAECFULL, "408993787.ogg"},
	{0x0A7376EF732A5085ULL, "591604512.ogg"},
	{0x0A73D97E3DC9607CULL, "lgt211cr_alistair_drunk.utc"},
	{0x0A7413F2D4437AAEULL, "art\\characters\\playercharacter\\elffemale\\ef_boo_lgte.mmh"},
	{0x0A742DF2D443A69CULL, "art\\characters\\playercharacter\\elffemale\\ef_boo_lgte.msh"},
	{0x0A743E7EFC852943ULL, "409657783.ogg"},
	{0x0A74455C05DE59E1ULL, "hlac_lt_0_lowtown_-65537_6y.dds"},
	{0x0A744B783BDAFD09ULL, "vxw_woundedcoast_nz_0_0.vlm"},
	{0x0A7462F291D75C00ULL, "mag170_recruit016094518.cl"},
	{0x0A746A883ED9167DULL, "hlac_lt_0_lowtown_a2_-196609_1y.dds"},
	{0x0A750D9093032A0CULL, "llac_lt_0_lowtown_docks_a3_-3_1.dds"},
	{0x0A751E67BAB4E2F5ULL, "vxlt_undercityencounter_1ff_0_0.vlm"},
	{0x0A75532FA14C72D9ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\1ww_basic.msh"},
	{0x0A755D2FA14C83C7ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\1ww_basic.mmh"},
	{0x0A75612FA14C8A14ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\1ww_basic.mao"},
	{0x0A75BA9C681480E1ULL, "47635176.ogg"},
	{0x0A75BBC21EB5AC65ULL, "6172861_m.fxe"},
	{0x0A75E94E7787A753ULL, "390526853.ogg"},
	{0x0A760F1F5596C366ULL, "hightown_male_noble2_mob6170930.cl"},
	{0x0A761475137FAC4EULL, "zz_rogue_dragon.cut"},
	{0x0A76820FDD69D222ULL, "vxlt_gallowscourtyard_7en_0_0.vlm"},
	{0x0A769C852147E89FULL, "hlac_w_encounter2_camp_262144_0z.dds"},
	{0x0A76AA843E9AB020ULL, "art\\vfx\\combat\\vfx_r_gn2_impact_act_c.impa.gad"},
	{0x0A76E4AA7FBA3C7DULL, "671824650.ogg"},
	{0x0A773D930436F4C0ULL, "6148681_m.fxe"},
	{0x0A779AF5A8667ABFULL, "zz_qcr_debug6045069.cl"},
	{0x0A77B5680623F377ULL, "6130959_m.fxe"},
	{0x0A7836676EA73A36ULL, "vxdr_0_deeproads_euv_0_0.vlm"},
	{0x0A786B1C2E22E68BULL, "6018815_m.fxe"},
	{0x0A788CE122D2F4C1ULL, "vxw_encounter3_33_0_0.vlm"},
	{0x0A78973DC9ABA01BULL, "hlac_lt_0_lowtown_a2_n_c_131070_6y.dds"},
	{0x0A7934E4D292F836ULL, "6098960_m.fxe"},
	{0x0A7956945D55AB7AULL, "gen00fl_merrill6116150.cl"},
	{0x0A7970FE37C2E26AULL, "vxw_woundedcoast_5h_0_0.vlm"},
	{0x0A797B97C68BE417ULL, "690572462.ogg"},
	{0x0A798249F74183AFULL, "6130101_f.fxe"},
	{0x0A7997E73AAD8240ULL, "dae000cr_grand_cleric6148129.cl"},
	{0x0A79C6EE1421CF13ULL, "v_c_blood_sh.mao"},
	{0x0A7A4504C4E17230ULL, "vxlt_darktown_4zd_0_0.vlm"},
	{0x0A7AB7AA4F0481E7ULL, "405312866.ogg"},
	{0x0A7ABE3047778B26ULL, "265450046.ogg"},
	{0x0A7AC7F1435F65C6ULL, "6138792_m.fxe"},
	{0x0A7ADB7291E44759ULL, "zz_dae_debug6029511.cl"},
	{0x0A7AE04986034E2BULL, "vxht_mansion2_2f6_0_0.vlm"},
	{0x0A7B0C58D813EE7FULL, "vxlt_gallowscourtyard_n_6y1_0_0.vlm"},
	{0x0A7B20184F73DB13ULL, "6129900_m.fxe"},
	{0x0A7B6A10BD0C8374ULL, "mag100_karras6034709.cl"},
	{0x0A7B7DAFE95F339AULL, "6038687_m.fxe"},
	{0x0A7C8B0C387E6A5DULL, "vxdr_0_deeproads_bo0_0_0.vlm"},
	{0x0A7C8DCD5E84BE0FULL, "llac_w_encounter1_131070_0.dds"},
	{0x0A7CAAE3B29FA57DULL, "art\\characters\\weapons\\greataxes\\textures\\w_gax_ga103a_0d.dds"},
	{0x0A7CACF41D1CF2B9ULL, "mrl150_merrill6070991.cl"},
	{0x0A7CAEF6FFC444E9ULL, "6117629_f.fxe"},
	{0x0A7CD02030659545ULL, "6217304_m.fxe"},
	{0x0A7CDE62F2C1EE8BULL, "878723620.ogg"},
	{0x0A7D281822BF09FCULL, "hlac_ht_0_hightown_n_-65536_3y.dds"},
	{0x0A7D94D945F3CD57ULL, "gen_im_trs_dwv_junk2_01.uti"},
	{0x0A7DA1CCB99ED87AULL, "91555014.ogg"},
	{0x0A7DA6638AA8C456ULL, "837047100.ogg"},
	{0x0A7E515B7E3D2EA7ULL, "w_encounter2_42_0_ll.mmh"},
	{0x0A7E562D1BA16B61ULL, "6072259_m.fxe"},
	{0x0A7E9164120D20DCULL, "prty_carver_armor_2_templar.dds"},
	{0x0A7EAC0F50D1A8D3ULL, "128816296.ogg"},
	{0x0A7F3AE964F55D97ULL, "237525966.ogg"},
	{0x0A7F8EB364FD287AULL, "vxdr_primevalentrance_1a8_0_0.vlm"},
	{0x0A7F9B4CD00273B3ULL, "gen00fl_aveline6063555.cl"},
	{0x0A7FCC6D1FA5977DULL, "496463722.ogg"},
	{0x0A805326CE1B994EULL, "vxht_0_hightown_a3_4kg_0_0.vlm"},
	{0x0A80B2B2A13AABF3ULL, "green.dds"},
	{0x0A8118C2F0E6B977ULL, "13993929.ogg"},
	{0x0A81470FB2D56606ULL, "vxht_0_hightown_a3_bic_0_0.vlm"},
	{0x0A8163DD44E589DAULL, "follower_banter6155640.cl"},
	{0x0A817254897B5F1CULL, "qun200_party_comment6090455.cl"},
	{0x0A82280EB1F1EDE6ULL, "vxht_playershouse_n_1q7_0_0.vlm"},
	{0x0A82BD552DD11589ULL, "vxw_woundedcoast_ky_0_0.vlm"},
	{0x0A8366BBD4934466ULL, "vxlt_0_lowtown_a2_n_c_eqd_0_0.vlm"},
	{0x0A8374BADB6C0784ULL, "465924356.ogg"},
	{0x0A83854B52C7B66CULL, "6091891_m.fxe"},
	{0x0A83EFEDC23E2DC4ULL, "gen00fl_aveline6132780.cl"},
	{0x0A8459A463ECF293ULL, "dae111_gamlen_intro_2.cut"},
	{0x0A847508AA0287C0ULL, "6133012_m.fxe"},
	{0x0A84801C748A9ACBULL, "art\\vfx\\creatures\\fxc_arc_shd_c.dura.evt"},
	{0x0A84C52FF11D5F46ULL, "vxw_encounter3_d1_0_0.vlm"},
	{0x0A85100C9FC77119ULL, "hlac_lt_blackemporium_1055_20y.dds"},
	{0x0A85453AD11AA8B1ULL, "619071971.ogg"},
	{0x0A857ED42C35F87FULL, "and121im_alrik_letter.uti"},
	{0x0A85847205F5CC4CULL, "leave_f.dds"},
	{0x0A861F6DC43284CCULL, "126994267.ogg"},
	{0x0A869F0BE24AFB15ULL, "vxht_0_hightown_a3_ds_0_0.vlm"},
	{0x0A86A9A7A9502A30ULL, "zz_dog_debug6204094.cl"},
	{0x0A8713B653C7D15CULL, "6108142_m.fxe"},
	{0x0A872221C898A335ULL, "vxlt_0_lowtown_n_f1v_0_0.vlm"},
	{0x0A87244FC420228FULL, "lgt220ip_shp_mag_suspect.ncs"},
	{0x0A876E381D321309ULL, "460785571.ogg"},
	{0x0A878B12F72AE248ULL, "638230059.ogg"},
	{0x0A87C42784AF5955ULL, "332040683.ogg"},
	{0x0A87CFBCD6784D4DULL, "426668809.ogg"},
	{0x0A87E2DB29B8FAEAULL, "vxw_woundedcoast_2i_0_0.vlm"},
	{0x0A8804C519949BFBULL, "319603203.ogg"},
	{0x0A8818CC7FBB03A2ULL, "330891370.ogg"},
	{0x0A8828C26FD0E02BULL, "6096839_m.fxe"},
	{0x0A888083F0736004ULL, "6004376_m.fxe"},
	{0x0A88D0E1CFBD1D5EULL, "vxht_mansion_n_5cx_0_0.vlm"},
	{0x0A88E78A26918C03ULL, "art\\characters\\creatures\\crowds\\humanmale\\animation\\kirkwall_and_lowtown\\humanmale_ambient_beat_threaten_1_darren.ani"},
	{0x0A8954BA24589951ULL, "40947416.ogg"},
	{0x0A8988AA59685136ULL, "rev_twi.mao"},
	{0x0A8A928069349C4AULL, "1067818744.ogg"},
	{0x0A8B76D6AC505ADCULL, "follower_banter6151332.cl"},
	{0x0A8B7AA3F64275D0ULL, "gen00fl_aveline6106880.cl"},
	{0x0A8B93F32D15F96AULL, "dae000_madam_lusine6152080.cl"},
	{0x0A8B991C7490E38CULL, "art\\vfx\\creatures\\fxc_arc_shd_c.dura.gad"},
	{0x0A8BB190E5590EDBULL, "zz_fen_debug6165492.cl"},
	{0x0A8C1001D45B7AAAULL, "plt_cod_itm_finesse.plo"},
	{0x0A8C7FA3700D896DULL, "hlac_w_encounter1_131070_0z.dds"},
	{0x0A8C9067FD50D39CULL, "vxlt_0_lowtown_env_0_0.vlm"},
	{0x0A8C99B6FE9FB201ULL, "lt_0_lowtown_a2_n_53_0.rml"},
	{0x0A8C9F7F57C8CA17ULL, "art\\vfx\\combat\\vfx_w_2nd_atk_com_c.impa.gad"},
	{0x0A8D30C562564127ULL, "vxca_caverns2_71i_0_0.vlm"},
	{0x0A8D98AB6394FB99ULL, "355521385.ogg"},
	{0x0A8DCE352AC9AF21ULL, "6113110_m.fxe"},
	{0x0A8E6A5768CEC9FCULL, "follower_banter6170724.cl"},
	{0x0A8E736F14AF9D66ULL, "pro000_friedrich6050171.cl"},
	{0x0A8E8A41266AF66FULL, "134959899.ogg"},
	{0x0A8F0016D6358C14ULL, "vxlt_gallowsprison_n_6o6_0_0.vlm"},
	{0x0A8FC32953A30650ULL, "703572349.ogg"},
	{0x0A90998B26EE3C07ULL, "leave_01.mao"},
	{0x0A90AD68A9B14C22ULL, "vxlt_0_lowtown_docks_bcu_0_0.vlm"},
	{0x0A90BDB3A0D4654FULL, "6217135_m.fxe"},
	{0x0A91048965941045ULL, "vxw_sundermount_315_0_1.vlm"},
	{0x0A9114785A601F28ULL, "206116431.ogg"},
	{0x0A914BFC338ABDB6ULL, "vxlt_warehouse_104_0_0.vlm"},
	{0x0A914C56880F8DD7ULL, "6104017_m.fxe"},
	{0x0A917179E7899491ULL, "w_sundermount_217_0.mmh"},
	{0x0A917D79E789A932ULL, "w_sundermount_217_0.mao"},
	{0x0A918B79E789C14FULL, "w_sundermount_217_0.msh"},
	{0x0A91981C74954D8AULL, "art\\vfx\\creatures\\fxc_arc_shd_c.dura.ani"},
	{0x0A923862BCA5F52CULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\ambient\\compressed\\leaning_back_low_gesture_right.gad"},
	{0x0A93B00B691BA0B6ULL, "follower_banter6141850.cl"},
	{0x0A93B47F57CF0CA4ULL, "art\\vfx\\combat\\vfx_w_2nd_atk_com_c.impa.evt"},
	{0x0A93BCC0AF429FDDULL, "6119310_m.fxe"},
	{0x0A93FC7AF4931573ULL, "mag311_nyssa6042233.cl"},
	{0x0A941014CA401546ULL, "vxca_caverns1_inx_0_0.vlm"},
	{0x0A9474834783B3A1ULL, "follower_banter6154048.cl"},
	{0x0A94B03137966846ULL, "fx_rage_flip.mao"},
	{0x0A94D771654394F1ULL, "vxlt_0_lowtown_a2_n_c_hlm_0_0.vlm"},
	{0x0A94EB6328387224ULL, "ser121_jethann6104426.cl"},
	{0x0A9531D6F4B31181ULL, "follower_barks6171983.cl"},
	{0x0A95487C455F49CCULL, "art\\characters\\playercharacter\\humanmale\\textures\\bdy_arm_masf_0d.dds"},
	{0x0A9548AFCD0DA996ULL, "vxca_caverns1_mines_efu_0_0.vlm"},
	{0x0A954D50A30F5E3EULL, "lt_0_lowtown_a2_n_38_0_ll.mmh"},
	{0x0A9593FEBA8EBBB5ULL, "vxlt_0_lowtown_n_f4u_0_0.vlm"},
	{0x0A95DF501285DF7FULL, "hlac_ht_0_hightown_n_-1_1z.dds"},
	{0x0A95F8F0DC308C30ULL, "6115474_m.fxe"},
	{0x0A966C1AA8BA5A8DULL, "739674241.ogg"},
	{0x0A96B54E6EE7D2B8ULL, "mrl000_marethari6129254.cl"},
	{0x0A96BFC93D7EBF71ULL, "dre000_dougal6129740.cl"},
	{0x0A96C48898A17CC1ULL, "6094673_m.fxe"},
	{0x0A972231B7981BDAULL, "lgt160_ntb_elf_conflict6163369.cl"},
	{0x0A979DFBF84E2ECFULL, "art\\characters\\playercharacter\\humanmale\\textures\\hm_hlf_medd_0t.dds"},
	{0x0A9822567BE98B5FULL, "6092512_m.fxe"},
	{0x0A9876D41E8A9D2DULL, "follower_banter6163723.cl"},
	{0x0A9916FD0407F99AULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\r_cwkl.gad"},
	{0x0A993762BCAC122AULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\ambient\\compressed\\leaning_back_low_gesture_right.ani"},
	{0x0A9960E2E8E04760ULL, "gen00fl_anders6125224.cl"},
	{0x0A99A997590224E5ULL, "pro000_athenril6048725.cl"},
	{0x0A99CCED3F63BC44ULL, "hlac_ht_mansion2_818_26z.dds"},
	{0x0A99FAF076818E32ULL, "hlac_ca_cavern1_haunted_24424_28z.dds"},
	{0x0A9AA416D94494AAULL, "vxlt_undercityserialkiller_1iq_0_1.vlm"},
	{0x0A9AB0921F31FD2CULL, "art\\characters\\creatures\\buzzard\\textures\\c_buzzard_0n.dds"},
	{0x0A9B0244EF46EC1CULL, "vxlt_0_lowtown_eiw_0_0.vlm"},
	{0x0A9B27F924931A3FULL, "vxca_cave_23h_0_1.vlm"},
	{0x0A9B71FD1EDD4BF3ULL, "271315077.ogg"},
	{0x0A9B80199ACDC5ECULL, "hla_ht_0_hightown_n_1_0z.dds"},
	{0x0A9BEF72A75A05BEULL, "335256989.ogg"},
	{0x0A9C02E0EFB005CAULL, "testhurlock.dds"},
	{0x0A9C196C981750C3ULL, "zz_isa_debug6133963.cl"},
	{0x0A9C2C2E23E61B19ULL, "art\\characters\\weapons\\longbows\\textures\\w_lbw_lb101a_0d.dds"},
	{0x0A9C9A19AF885C83ULL, "art\\characters\\playercharacter\\humanfemale\\cth_nobe.msh"},
	{0x0A9CA711031E8621ULL, "vxlt_darktown_52t_0_0.vlm"},
	{0x0A9CA819AF8874DDULL, "art\\characters\\playercharacter\\humanfemale\\cth_nobe.mmh"},
	{0x0A9CAC19AF887BEEULL, "art\\characters\\playercharacter\\humanfemale\\cth_nobe.mao"},
	{0x0A9CB17A7260486DULL, "582283085.ogg"},
	{0x0A9CD44809DA9B2BULL, "vxht_0_hightown_a3_anh_0_0.vlm"},
	{0x0A9CE89B21559206ULL, "6210162_m.fxe"},
	{0x0A9CFE8A26A25CEDULL, "art\\characters\\creatures\\crowds\\humanmale\\animation\\kirkwall_and_lowtown\\humanmale_ambient_beat_threaten_1_darren.gad"},
	{0x0A9D57B1FB023CD8ULL, "vxlt_backalley_n_1e7_0_0.vlm"},
	{0x0A9D71F3C82BA494ULL, "vxlt_gallowsprison_n_6h7_0_0.vlm"},
	{0x0A9DD63DF3EBDF1CULL, "vxca_cavern1_haunted_eh5_0_0.vlm"},
	{0x0A9F1F459BA764A2ULL, "vxlt_0_lowtown_docks_bfv_0_0.vlm"},
	{0x0A9FB372A90AB521ULL, "nat380_nathaniel6066493.cl"},
	{0x0A9FE2D3C8E19BC7ULL, "zz_charge_blight2.cut"},
	{0x0AA011B046D0115EULL, "590462237.ogg"},
	{0x0AA019FD040E1DB1ULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\r_cwkl.evt"},
	{0x0AA032C930C3AAF1ULL, "6155286_m.fxe"},
	{0x0AA04E300E9A480FULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\qunari_sit_throne_gestures_right_hand.gad"},
	{0x0AA06E3560FCC329ULL, "hlac_lt_0_lowtown_docks_-65535_0x.dds"},
	{0x0AA07C34C0021021ULL, "var_trigger.gda"},
	{0x0AA0CA3E8304877FULL, "6154207_m.fxe"},
	{0x0AA0CC7F57D9BF9DULL, "art\\vfx\\combat\\vfx_w_2nd_atk_com_c.impa.ani"},
	{0x0AA10588490AD48FULL, "6208558_f.fxe"},
	{0x0AA12AF11B70D6C3ULL, "314456149.ogg"},
	{0x0AA162966099F4F4ULL, "docks_fem_sailor_mob6176421.cl"},
	{0x0AA17CFC558800DBULL, "vxlt_0_lowtown_a3_n_c_e2e_0_0.vlm"},
	{0x0AA1C4E9BA46712FULL, "6179444_m.fxe"},
	{0x0AA1EE5CAA12EFF5ULL, "6201371_f.fxe"},
	{0x0AA2074B572E8871ULL, "gen00fl_fenris6126952.cl"},
	{0x0AA2462FA1734AFCULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\1ww_basic.phy"},
	{0x0AA26865576831ACULL, "pro000_colonel_ewald6047377.cl"},
	{0x0AA2D7CEDA1ECAA6ULL, "gen_im_trs_drk_two_2hs_r2.uti"},
	{0x0AA34A2EE32C2695ULL, "373793276.ogg"},
	{0x0AA3826C0EB00EA2ULL, "6068362_f.fxe"},
	{0x0AA4A454031CBDC1ULL, "vxht_keep_n_6y6_0_0.vlm"},
	{0x0AA51D81E399DD2FULL, "6107236_m.fxe"},
	{0x0AA5B49BB3A94E5EULL, "vxht_mansion_n_5iz_0_0.vlm"},
	{0x0AA5D8C63F15B953ULL, "vxlt_hangedman_n_2dm_0_0.vlm"},
	{0x0AA5E11392489BCFULL, "dre000_vampire6034329.cl"},
	{0x0AA5E2B45954ACDEULL, "vxlt_hangedman_210_0_0.vlm"},
	{0x0AA5E2DBA8A25F17ULL, "vxht_chantry_n_23v_0_0.vlm"},
	{0x0AA663300E9ED7BCULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\qunari_sit_throne_gestures_right_hand.evt"},
	{0x0AA69596B39D20E7ULL, "art\\vfx\\combat\\vfx_sg_quake_act_c.anb"},
	{0x0AA69CD38665A963ULL, "w_encounter3_35_0.msh"},
	{0x0AA6AAD38665C0BDULL, "w_encounter3_35_0.mmh"},
	{0x0AA6AED38665C7CEULL, "w_encounter3_35_0.mao"},
	{0x0AA719D15013A827ULL, "hlac_ca_caverns1_mines_24424_0y.dds"},
	{0x0AA7443D9C1AB408ULL, "6083805_m.fxe"},
	{0x0AA7A08B2701275DULL, "leave_01.dds"},
	{0x0AA7B89F3230B00BULL, "dae021_sister_colinda_sta6150203.cl"},
	{0x0AA8B029247B4B20ULL, "l_dae211ar_gamlens_house_nt.bnk"},
	{0x0AA91BBC24FFDBEAULL, "6051329_m.fxe"},
	{0x0AA95CFC2FCF5829ULL, "dae000_sabina6162342.cl"},
	{0x0AA9BDB1E4E16187ULL, "6071955_m.fxe"},
	{0x0AA9C29F0C956BD3ULL, "6001219_m.fxe"},
	{0x0AAA707BF1EFD4ADULL, "vxht_keep_5xa_0_0.vlm"},
	{0x0AAA86576971B7C5ULL, "vxdr_primevalentrance_1a3_0_0.vlm"},
	{0x0AAA8F15573DCEC7ULL, "hlac_lt_0_lowtown_a2_n_c_-131072_0z.dds"},
	{0x0AAA9BEEE76E50C7ULL, "gen00fl_varric6076062.cl"},
	{0x0AAAA60E5D307F18ULL, "6155733_m.fxe"},
	{0x0AAAD77842A852D7ULL, "6176005_m.fxe"},
	{0x0AAB18EDF5149EA1ULL, "vxlt_darktown_57k_0_0.vlm"},
	{0x0AAB4F9F2AFFF342ULL, "176277273.ogg"},
	{0x0AAB76819D40E9AEULL, "qun220cr_javaris6071521.cl"},
	{0x0AABA83DAD1C561BULL, "art\\characters\\creatures\\gallowsslavestatue\\animation\\slavedropv1.ani"},
	{0x0AABC13334E12D66ULL, "113061345.ogg"},
	{0x0AAC5FDCB00E7200ULL, "6110172_m.fxe"},
	{0x0AAC8983169B837BULL, "zz_dae_debug6047614.cl"},
	{0x0AACB7120A462F09ULL, "6080510_f.fxe"},
	{0x0AAD65F674FFEFF2ULL, "vxdr_0_deeproads_fhz_0_0.vlm"},
	{0x0AAD6798820BA496ULL, "vxht_0_hightown_a2_bzw_0_0.vlm"},
	{0x0AADB82FE7309B14ULL, "lgt311_shw_mer_agatha6156498.cl"},
	{0x0AADEA1C350E9576ULL, "art\\levels\\races\\proxy\\props\\corpses\\prp_zzbonepile.mao"},
	{0x0AADF61C350EA915ULL, "art\\levels\\races\\proxy\\props\\corpses\\prp_zzbonepile.mmh"},
	{0x0AADF81C350EAC8BULL, "art\\levels\\races\\proxy\\props\\corpses\\prp_zzbonepile.msh"},
	{0x0AAE31FD041A8380ULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\r_cwkl.ani"},
	{0x0AAE571E51C1F34AULL, "gen00fl_dog6107335.cl"},
	{0x0AAEDF840DBFA879ULL, "gen00fl_dog6141138.cl"},
	{0x0AAF62B8DFC0AD47ULL, "6101003_m.fxe"},
	{0x0AAFFA6AF19E355EULL, "vxca_caverns1_mines_fms_0_0.vlm"},
	{0x0AB02F654BC79B70ULL, "hlac_lt_hangedman_n_-1_11z.dds"},
	{0x0AB0576F17D2CD13ULL, "hm_cgn_005.mop"},
	{0x0AB0776678BF6F97ULL, "261470896.ogg"},
	{0x0AB0F1BCE44DAC75ULL, "lit_aoe_05.mao"},
	{0x0AB13017D8556489ULL, "vxw_bonepit_1vp_0_0.vlm"},
	{0x0AB165793854CC29ULL, "pro000ar_blightlands_real.are"},
	{0x0AB1DE780BD106A3ULL, "gen00fl_fenris6126521.cl"},
	{0x0AB1E85FDD6394F9ULL, "ht_0_hightown_45_0.anb"},
	{0x0AB29F4687A9DBB4ULL, "6016641_m.fxe"},
	{0x0AB2C6B5969F1BD4ULL, "hlac_ht_0_hightown_n_-65537_3z.dds"},
	{0x0AB3A0376A03471AULL, "143197887.ogg"},
	{0x0AB3F680022D7272ULL, "340531701.ogg"},
	{0x0AB40AF487838F3AULL, "hf_zmother_h2s1.mor"},
	{0x0AB45F4DD6724CFEULL, "523177612.ogg"},
	{0x0AB4637217C1F77AULL, "6078802_f.fxe"},
	{0x0AB46B300EAB22D5ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\qunari_sit_throne_gestures_right_hand.ani"},
	{0x0AB4FC43E239F207ULL, "6100040_m.fxe"},
	{0x0AB5B27EA38A8F40ULL, "63277414.ogg"},
	{0x0AB5C8D86101F0E4ULL, "30078549.ogg"},
	{0x0AB5E5FFCEDF17B6ULL, "445758111.ogg"},
	{0x0AB67170C1B1BB10ULL, "vxw_sundermount_2ja_0_0.vlm"},
	{0x0AB69E3A42A39193ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\arm_grab_reaction.evt"},
	{0x0AB6B570352F7394ULL, "vxht_mansion_n_4w7_0_0.vlm"},
	{0x0AB6F7E9AE1DBC98ULL, "330492809.ogg"},
	{0x0AB7B2DA7864241CULL, "vxlt_0_lowtown_docks_a3_n_cz0_0_1.vlm"},
	{0x0AB7C390CED1944BULL, "lgt250_way_dalish_conflict6147152.cl"},
	{0x0AB7DCB2A16A353AULL, "green.tnt"},
	{0x0AB826A9756E4D0CULL, "6127039_m.fxe"},
	{0x0AB8B03DAD26EE12ULL, "art\\characters\\creatures\\gallowsslavestatue\\animation\\slavedropv1.evt"},
	{0x0AB8B5DE0A54725BULL, "dre000_cut_thewayout.bnk"},
	{0x0AB92EF9149DFDE6ULL, "gen00fl_isabela6075248.cl"},
	{0x0AB9765F26B31B55ULL, "vxht_bartrand_n_5sx_0_0.vlm"},
	{0x0AB99440FC9FED48ULL, "hlac_lt_gallowstemplar_f_7198_6x.dds"},
	{0x0AB9B801EDC6CC2BULL, "vxht_0_hightown_a3_apj_0_0.vlm"},
	{0x0AB9BB81AA8C504DULL, "gen00fl_aveline6152529.cl"},
	{0x0AB9DA87F94FCC09ULL, "498140500.ogg"},
	{0x0ABA20189B4C0451ULL, "6170427_m.fxe"},
	{0x0ABA3B6BDEEE6DD8ULL, "vxlt_backalley_n_1k5_0_0.vlm"},
	{0x0ABA44B60504E1EBULL, "vxw_bonepitdestroyed_a3_by_0_0.vlm"},
	{0x0ABA7A905327D668ULL, "character_character_p254.psh.fxo.dx11"},
	{0x0ABAAE8964087708ULL, "plt_dae010pt_amb_lowtown.plo"},
	{0x0ABB4434A010E602ULL, "hlac_ca_caverns1_24425_0x.dds"},
	{0x0ABB89222EB3F810ULL, "one300_zevran6076551.cl"},
	{0x0ABC647C4AD01E5BULL, "hlac_ht_keep_3732_16z.dds"},
	{0x0ABC973A42A7F1D4ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\arm_grab_reaction.gad"},
	{0x0ABCB7DE0A585E1BULL, "dre000_cut_thewayout.cnv"},
	{0x0ABCFC7541D48E2FULL, "6045982_f.fxe"},
	{0x0ABD079B604FFA56ULL, "vxht_chantry_n_2rx_0_0.vlm"},
	{0x0ABDE29C98DB30E4ULL, "dae000_quintus6154725.cl"},
	{0x0ABE4851B28FAB98ULL, "dae000ip_crowd_mf8_talk15.utp"},
	{0x0ABE60B6397431DBULL, "vxlt_0_lowtown_a3_n_c_e8g_0_0.vlm"},
	{0x0ABE6D3F92E9AD89ULL, "vxca_caverns2_4x7_0_0.vlm"},
	{0x0ABEA77F47D9665BULL, "993190787.ogg"},
	{0x0ABECDEF26D0FD36ULL, "hlac_lt_0_lowtown_docks_65535_1y.dds"},
	{0x0ABEF5F002216603ULL, "follower_banter6170642.cl"},
	{0x0ABF1EF7D7921B93ULL, "mag101_aombanter.bnk"},
	{0x0ABF6AEA00C69698ULL, "gen00fl_dog6109285.cl"},
	{0x0ABF78B6FECAD391ULL, "lt_0_lowtown_a2_n_53_0.anb"},
	{0x0ABFAB8E73ABD806ULL, "vxlt_undercitydungeon_1ip_0_1.vlm"},
	{0x0ABFCF3DAD2D41A5ULL, "art\\characters\\creatures\\gallowsslavestatue\\animation\\slavedropv1.gad"},
	{0x0ABFE0BE885D2335ULL, "hlac_lt_0_lowtown_docks_a3_n_0_1x.dds"},
	{0x0AC02ECFAA3BE83BULL, "lgt310_shw_ors_informant6156340.cl"},
	{0x0AC135EEF63FB660ULL, "hlac_w_woundedcoast_-131065_0x.dds"},
	{0x0AC17EE9639C969AULL, "vxca_cavern1_haunted_gif_0_0.vlm"},
	{0x0AC240F7D7948BE3ULL, "mag101_aombanter.cnv"},
	{0x0AC25CFA98A30F41ULL, "dre000_dougal6130957.cl"},
	{0x0AC265DF122F9C5DULL, "subareaid.gda"},
	{0x0AC267E1E4E98DA1ULL, "mag291_demon_marethari6028848.cl"},
	{0x0AC2ACE4907808EAULL, "836032149.ogg"},
	{0x0AC2FACADAC66287ULL, "qun221_viscount6074078.cl"},
	{0x0AC39AD0AB62FA54ULL, "761268501.ogg"},
	{0x0AC3A63A42AE29B2ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\arm_grab_reaction.ani"},
	{0x0AC424E010157311ULL, "6136230_m.fxe"},
	{0x0AC47C8EF2C9EF82ULL, "6019116_m.fxe"},
	{0x0AC4D584A5E997D4ULL, "hlac_ht_0_hightown_131071_1y.dds"},
	{0x0AC4E34DB3A7D390ULL, "vxw_sundermount_2on_0_0.vlm"},
	{0x0AC5A5DB6AF2A857ULL, "follower_barks6153729.cl"},
	{0x0AC5E1B58A761649ULL, "6119177_f.fxe"},
	{0x0AC61E027EF8C518ULL, "w_woundedcoast_35_0.rml"},
	{0x0AC66397DA707B71ULL, "vxdr_primevaltomb_8mc_0_1.vlm"},
	{0x0AC67E4A15C9D3CEULL, "766476004.ogg"},
	{0x0AC6A2A1630C1241ULL, "llac_lt_warehouse_-65535_1.dds"},
	{0x0AC6B18B56F0BDCBULL, "vxht_chantry_n_20k_0_0.vlm"},
	{0x0AC6C95D7E0397A1ULL, "6099542_m.fxe"},
	{0x0AC6EF9CA4BC1838ULL, "art\\vfx\\materials\\vfx_d_shockwave.mao"},
	{0x0AC7958E395814F4ULL, "929772460.ogg"},
	{0x0AC838245A019377ULL, "6095148_m.fxe"},
	{0x0AC8627C1B1698F9ULL, "w_encounter2_49_0.msh"},
	{0x0AC86C7C1B16A9E7ULL, "w_encounter2_49_0.mmh"},
	{0x0AC86D2F38731C79ULL, "narrative_flemeth6002799.cl"},
	{0x0AC8707C1B16B034ULL, "w_encounter2_49_0.mao"},
	{0x0AC89A26990A1995ULL, "6154506_m.fxe"},
	{0x0AC8AA34CCE18732ULL, "6071257_m.fxe"},
	{0x0AC99AECF32E6627ULL, "6119900_f.fxe"},
	{0x0AC9B472355D6970ULL, "hlac_lt_0_lowtown_n_-65535_5x.dds"},
	{0x0AC9CD23B8634BC6ULL, "206556495.ogg"},
	{0x0AC9D2A33D737E41ULL, "art\\vfx\\combat\\vfx_glb_confusion_crust_c.dura.ani"},
	{0x0ACA05E2688617B7ULL, "6121104_m.fxe"},
	{0x0ACA4F003E265F10ULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_ear05b.mmh"},
	{0x0ACA5D003E26772AULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_ear05b.msh"},
	{0x0ACAC09510E25B9CULL, "6036988_m.fxe"},
	{0x0ACBC222D7FE499AULL, "art\\levels\\races\\proxy\\tilesets\\setint\\4co_basic\\4sd_basic.mmh"},
	{0x0ACBC622D7FE50D1ULL, "art\\levels\\races\\proxy\\tilesets\\setint\\4co_basic\\4sd_basic.mao"},
	{0x0ACBD422D7FE68E0ULL, "art\\levels\\races\\proxy\\tilesets\\setint\\4co_basic\\4sd_basic.msh"},
	{0x0ACC51484097D656ULL, "vxca_cavern1_haunted_edk_0_0.vlm"},
	{0x0ACC6D38280B559BULL, "6137283_f.fxe"},
	{0x0ACC6F41FF647802ULL, "vxca_caverns2prc_7tw_0_1.vlm"},
	{0x0ACDA230390FE16FULL, "vxlt_0_lowtown_a2_n_e84_0_0.vlm"},
	{0x0ACDB33EDA5BE8EBULL, "6136957_m.fxe"},
	{0x0ACE1F1F689F86CAULL, "hlac_ca_caverns2_5802_53y.dds"},
	{0x0ACE5FA25F21D640ULL, "vxlt_warehouse_19h_0_1.vlm"},
	{0x0ACE9296B3BE6DE8ULL, "art\\vfx\\combat\\vfx_sg_quake_act_c.mmh"},
	{0x0ACE9D457254A2ABULL, "6076150_m.fxe"},
	{0x0ACE9EE52D1BF6F1ULL, "vxlt_0_lowtown_a2_n_c_hxa_0_0.vlm"},
	{0x0ACFB12FB5C232A3ULL, "lgt210_eau_symp_templar6154717.cl"},
	{0x0AD01F92D5B7B563ULL, "qun221_petrice6076156.cl"},
	{0x0AD0285959A6E31EULL, "and121_karl6093471.cl"},
	{0x0AD036B803F36F1BULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\9uw_hngtarp16mbn.mmh"},
	{0x0AD042B803F383B8ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\9uw_hngtarp16mbn.mao"},
	{0x0AD049F1A7068D18ULL, "439181655.ogg"},
	{0x0AD054B803F3A205ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\9uw_hngtarp16mbn.msh"},
	{0x0AD0DDA33D79AFE3ULL, "art\\vfx\\combat\\vfx_glb_confusion_crust_c.dura.gad"},
	{0x0AD1231ADF95BBF8ULL, "vxca_caverns1_mines_i8m_0_0.vlm"},
	{0x0AD16AB3AADAC09FULL, "6133806_m.fxe"},
	{0x0AD173459DF8BCFCULL, "6093843_f.fxe"},
	{0x0AD19DC6F12D4358ULL, "llac_lt_undercityserialkiller_-1_33.dds"},
	{0x0AD25C241B9FABBFULL, "543869611.ogg"},
	{0x0AD2A02125811166ULL, "lgt220_shp_mag_guard6175909.cl"},
	{0x0AD3552AA59DEC10ULL, "vxw_sundermount_2po_0_0.vlm"},
	{0x0AD3650FA862AE3DULL, "6153381_m.fxe"},
	{0x0AD373D031345B24ULL, "vxdr_primevalentrance_28f_0_0.vlm"},
	{0x0AD3EA5DDAA7C5A3ULL, "hlac_w_encounter1_0_3y.dds"},
	{0x0AD42D483644803FULL, "889293996.ogg"},
	{0x0AD4EE9CB756B8EFULL, "vxca_caverns1_mines_j5j_0_0.vlm"},
	{0x0AD4FE0F2EBF3B62ULL, "llac_lt_undercitydungeon_-1_43.dds"},
	{0x0AD50B09920CA577ULL, "vxdr_primevaltomb_7km_0_0.vlm"},
	{0x0AD51570D64BB1BCULL, "vxlt_0_lowtown_a2_n_c_i0q_0_0.vlm"},
	{0x0AD5208805A2A830ULL, "491236661.ogg"},
	{0x0AD5CC4DA474BD29ULL, "vxlt_0_lowtown_a2_n_c_eoy_0_0.vlm"},
	{0x0AD61BD0B482383AULL, "6201959_m.fxe"},
	{0x0AD628AD71E43DABULL, "6133312_m.fxe"},
	{0x0AD6D23A68207593ULL, "mrl150_merrill6093770.cl"},
	{0x0AD756793875827FULL, "pro000ar_blightlands_real.ncs"},
	{0x0AD79672A7A5715CULL, "6083953_m.fxe"},
	{0x0AD7A82D1655C36AULL, "vxca_caverns1_e8n_0_0.vlm"},
	{0x0AD7A940F5A37EA1ULL, "6103308_m.fxe"},
	{0x0AD7BAA33D7F92C8ULL, "art\\vfx\\combat\\vfx_glb_confusion_crust_c.dura.evt"},
	{0x0AD7E6FF4B52C6AEULL, "6071089_f.fxe"},
	{0x0AD80EFA059867F6ULL, "gen00fl_aveline6132809.cl"},
	{0x0AD818BE5575C634ULL, "vxca_caverns1_ei6_0_1.vlm"},
	{0x0AD81EF91131790FULL, "6055610_m.fxe"},
	{0x0AD83D1472C2E664ULL, "vxca_caverns2_aid_0_0.vlm"},
	{0x0AD848957F7B5269ULL, "zz_vault_debug6203278.cl"},
	{0x0AD84A1A40261B26ULL, "static_static_p72.psh.fxo.dx11"},
	{0x0AD858FA63D17074ULL, "vxht_mansion2b_1bv_0_0.vlm"},
	{0x0AD87CF9EE1CA143ULL, "6072248_m.fxe"},
	{0x0AD8B3B829609F41ULL, "hlac_w_blightlands_262141_0z.dds"},
	{0x0AD915C22330FE6CULL, "6152075_m.fxe"},
	{0x0AD91DB72855C5BCULL, "face_face_p3.vsh.fxo.dx11"},
	{0x0AD97A1D020A1C4CULL, "6116080_m.fxe"},
	{0x0ADA3A4671749FB6ULL, "hlac_w_encounter1_1_0z.dds"},
	{0x0ADA6F9BBEC36432ULL, "114320933.ogg"},
	{0x0ADAABC38F83955EULL, "6043876_m.fxe"},
	{0x0ADAD69CDF40010AULL, "450163166.ogg"},
	{0x0ADB1D93D89DED7EULL, "hlac_lt_gallowsprison_n_-1_3x.dds"},
	{0x0ADB4A44AC8988BBULL, "6053857_f.fxe"},
	{0x0ADBCC8184CAEA26ULL, "vxw_encounter2_camp_jp_0_0.vlm"},
	{0x0ADC191A19F156F1ULL, "art\\characters\\creatures\\stonegolem\\animation\\kd_exit.ani"},
	{0x0ADC2318101A510CULL, "hlac_w_sundermount_-65542_0x.dds"},
	{0x0ADC33E51A6905ACULL, "llac_w_encounter2_-65536_0.dds"},
	{0x0ADCAB89642558C7ULL, "plt_dae010pt_amb_lowtown.nss"},
	{0x0ADDAE71D9DBC142ULL, "vxw_sundermount_2kp_0_0.vlm"},
	{0x0ADE7679387BD76EULL, "pro000ar_blightlands_real.lst"},
	{0x0ADEE51E817921BAULL, "art\\characters\\weapons\\staff\\w_stf_dssta.msh"},
	{0x0ADEF31E817939F3ULL, "art\\characters\\weapons\\staff\\w_stf_dssta.mao"},
	{0x0ADEF71E81794000ULL, "art\\characters\\weapons\\staff\\w_stf_dssta.mmh"},
	{0x0ADEFB16E3BF4F74ULL, "6148388_m.fxe"},
	{0x0ADF29F3010C1F23ULL, "752287635.ogg"},
	{0x0AE05D7560E8F458ULL, "llac_lt_0_lowtown_docks_a3_-65538_3.dds"},
	{0x0AE060D49BA9A0A6ULL, "art\\characters\\weapons\\shields\\textures\\w_shd_002a_d.dds"},
	{0x0AE07B43388229A3ULL, "vxht_keep_n_63z_0_0.vlm"},
	{0x0AE07D66B4CF500EULL, "dre000_varric6027749.cl"},
	{0x0AE0F3198B4D3C22ULL, "vxlt_0_lowtown_a3_n_c_ev4_0_0.vlm"},
	{0x0AE11E3ABDF1611FULL, "w_bonepitmining_a2_212_0.msh"},
	{0x0AE1243ABDF16BC1ULL, "w_bonepitmining_a2_212_0.mmh"},
	{0x0AE1303ABDF18062ULL, "w_bonepitmining_a2_212_0.mao"},
	{0x0AE17C09C789F287ULL, "6166574_m.fxe"},
	{0x0AE1B3973BEE3A50ULL, "rdr111dg_martin6015059.cl"},
	{0x0AE1D6EC9A58C6BDULL, "6156081_m.fxe"},
	{0x0AE1E89B60F6A441ULL, "vxdr_primevalentrance_319_0_0.vlm"},
	{0x0AE26CAB7A6598B7ULL, "fen110_fenris6100322.cl"},
	{0x0AE2F7C5B5556807ULL, "cd_nat380_sibling.wbd"},
	{0x0AE3241A19F78793ULL, "art\\characters\\creatures\\stonegolem\\animation\\kd_exit.gad"},
	{0x0AE329B9E3F7668AULL, "vxw_encounter2_4z_0_0.vlm"},
	{0x0AE354CF3EEAE7ECULL, "6088325_m.fxe"},
	{0x0AE384AF2B386511ULL, "vxlt_gallowstemplar_f_5pm_0_0.vlm"},
	{0x0AE385F2120756AAULL, "boulder_mac_so.mao"},
	{0x0AE3AD1761EFC8A1ULL, "6093742_m.fxe"},
	{0x0AE51C027F1372F5ULL, "w_woundedcoast_35_0.msh"},
	{0x0AE51E027F13766BULL, "w_woundedcoast_35_0.mmh"},
	{0x0AE52A027F138A08ULL, "w_woundedcoast_35_0.mao"},
	{0x0AE547D96746CBE3ULL, "llac_ca_caverns1_mines_24425_11.dds"},
	{0x0AE59BA82D179B69ULL, "vxw_sundermount_2co_0_0.vlm"},
	{0x0AE5A65F536E96A9ULL, "144074700.ogg"},
	{0x0AE60F8CCD3576B8ULL, "llac_lt_0_lowtown_n_-3_4.dds"},
	{0x0AE61B1DE7999721ULL, "6104208_m.fxe"},
	{0x0AE63794EBBA7F0CULL, "hla_ht_0_hightown_a2_65535_0x.dds"},
	{0x0AE6452716649536ULL, "zz_qun_debug6028227.cl"},
	{0x0AE6AE2E418062FEULL, "967360694.ogg"},
	{0x0AE6E37824A37390ULL, "one130_thug6133091.cl"},
	{0x0AE7679FE8D6CE8FULL, "l_env_wal_lowtowndocks_cough.wbd"},
	{0x0AE958E9299F99E6ULL, "black.psh.fxo"},
	{0x0AE975C6E479ED4DULL, "lt_backalley_g_mmap.dds"},
	{0x0AE98E8BEC22E5FCULL, "w_sundermount_134_0.mao"},
	{0x0AE99A8BEC22F99FULL, "w_sundermount_134_0.mmh"},
	{0x0AE9A08BEC230441ULL, "w_sundermount_134_0.msh"},
	{0x0AEA211A19FDA1F8ULL, "art\\characters\\creatures\\stonegolem\\animation\\kd_exit.evt"},
	{0x0AEA3E5E76C102A6ULL, "vxw_encounter2_camp_eq_0_0.vlm"},
	{0x0AEA602982B5645FULL, "car131ip_chest_gift.utp"},
	{0x0AEA8A0841BFF841ULL, "300651420.ogg"},
	{0x0AEACB221FAC443EULL, "713029642.ogg"},
	{0x0AEAD22AF678839FULL, "91184444.ogg"},
	{0x0AEAE95FDD935BF9ULL, "ht_0_hightown_45_0.rml"},
	{0x0AEAF5AEAA23D342ULL, "gen00fl_anders6119702.cl"},
	{0x0AEB15AE9891A2F0ULL, "6141112_m.fxe"},
	{0x0AEB1F5B6DA1684AULL, "179820971.ogg"},
	{0x0AEB571D9743EAE6ULL, "6129426_m.fxe"},
	{0x0AEB790D780BCE4BULL, "llac_lt_0_lowtown_docks_a3_n_65534_6.dds"},
	{0x0AEB9DE8B1FD5409ULL, "6175147_m.fxe"},
	{0x0AEBBF5D1C29DE04ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\animation\\blendtree\\doorsimple.anb"},
	{0x0AEC204ECBD1D9C2ULL, "vxw_sundermount_2lq_0_0.vlm"},
	{0x0AEC70AACFD6DEACULL, "306938669.ogg"},
	{0x0AED861443E3B232ULL, "vxht_brothel_n_4a8_0_0.vlm"},
	{0x0AEE06D4C381ECF8ULL, "vxca_caverns1_mines_i2k_0_0.vlm"},
	{0x0AEE3E9E27C92F78ULL, "mcr371_hostile_fenris6082231.cl"},
	{0x0AEF545AD2C0999FULL, "6085695_m.fxe"},
	{0x0AEFABB049838BADULL, "6137596_m.fxe"},
	{0x0AF050137EA0194BULL, "healertacticpreset.gda"},
	{0x0AF0ADB6310AA461ULL, "953697932.ogg"},
	{0x0AF0E122D81D9AA5ULL, "art\\levels\\races\\proxy\\tilesets\\setint\\4co_basic\\4sd_basic.phy"},
	{0x0AF0FF021E6A6264ULL, "90634769.ogg"},
	{0x0AF1068AB1EC16C6ULL, "hlac_lt_0_lowtown_a2_n_65533_3y.dds"},
	{0x0AF13B855CAA733FULL, "vxht_mansion_n_60s_0_0.vlm"},
	{0x0AF28A2DE039D8B1ULL, "dre180ip_to_primeval_crypt.utp"},
	{0x0AF28C8BDC5560B8ULL, "vxca_caverns2prc_9pn_0_0.vlm"},
	{0x0AF3353478440024ULL, "vxht_0_hightown_n_a7j_0_0.vlm"},
	{0x0AF351684C0AB7AEULL, "vxlt_warehouse_b7_0_0.vlm"},
	{0x0AF374CAF9E18035ULL, "vxw_bonepit_7x_0_0.vlm"},
	{0x0AF40D851F0DB3E9ULL, "vxw_sundermount_2dn_0_0.vlm"},
	{0x0AF41978DE8DCBAAULL, "dae200_saemus6142455.cl"},
	{0x0AF42C7BE317F453ULL, "mcr310_sibling6076241.cl"},
	{0x0AF42F517A8BFAA2ULL, "vxw_bonepitdestroyed_a3_1no_0_0.vlm"},
	{0x0AF4557C1B3BBD1CULL, "w_encounter2_49_0.phy"},
	{0x0AF53CB447BDA174ULL, "vxht_mansion2b_1xt_0_0.vlm"},
	{0x0AF5C1133DF52CFEULL, "vxca_caverns2prc_9x0_0_0.vlm"},
	{0x0AF5E57D946455F0ULL, "lt_0_lowtown_docks_250_0.mao"},
	{0x0AF5E97D94645C03ULL, "lt_0_lowtown_docks_250_0.mmh"},
	{0x0AF5F77D9464745DULL, "lt_0_lowtown_docks_250_0.msh"},
	{0x0AF6153ABB99110BULL, "vxlt_warehouse_n_qf_0_0.vlm"},
	{0x0AF707D3FE8F54F0ULL, "hlac_lt_darktown_-1_0z.dds"},
	{0x0AF74600226D8177ULL, "dae000ip_crowd_mf2_talk5.utp"},
	{0x0AF77A876C6D2694ULL, "zz_followers_banter6158148.cl"},
	{0x0AF8CF8486C6B368ULL, "vxlt_0_lowtown_docks_a3_n_c14_0_0.vlm"},
	{0x0AF8DD84E9AE9565ULL, "234972057.ogg"},
	{0x0AF91C325CB8E778ULL, "vxht_mansion2b_3bx_0_0.vlm"},
	{0x0AF92817C6154C1FULL, "mer220_hubert6041210.cl"},
	{0x0AFA922BBDC7F242ULL, "vxw_sundermount_2qr_0_0.vlm"},
	{0x0AFAC190409164E6ULL, "6032897_f.fxe"},
	{0x0AFAD9CFB128C165ULL, "6098083_f.fxe"},
	{0x0AFB1A1BE5AD14D8ULL, "gen_im_acc_rng_act3_14.uti"},
	{0x0AFB6C7C1B420344ULL, "w_encounter2_49_0.rml"},
	{0x0AFBC088D2CEEC80ULL, "ht_0_hightown_n_49_0.rml"},
	{0x0AFBF7F135D9CAB2ULL, "vxht_brothel_n_4f9_0_0.vlm"},
	{0x0AFCA6CD4B502352ULL, "vxlt_darktown_5gt_0_0.vlm"},
	{0x0AFCB63CAC47C4F4ULL, "ss_probabilitymodif.gda"},
	{0x0AFCCE0095F27DA7ULL, "mcr370_meredith6083582.cl"},
	{0x0AFCD1C5F63497F4ULL, "vxlt_0_lowtown_a2_n_gvc_0_0.vlm"},
	{0x0AFD1C77B3986D49ULL, "ave100_donnic6058924.cl"},
	{0x0AFD74E575D00812ULL, "gen00fl_sebastian6177817.cl"},
	{0x0AFD85840424CD94ULL, "65534320.ogg"},
	{0x0AFE20C1DB8A83B3ULL, "plt_lgt260pt_way_blood_mage.nss"},
	{0x0AFE7D13D1476D10ULL, "181444053.ogg"},
	{0x0AFEBB47E1307552ULL, "60507516.ogg"},
	{0x0AFEEC91FD56EC59ULL, "art\\characters\\playercharacter\\dwarfmale\\animation\\cindes\\varric\\sitting_back_loop.gad"},
	{0x0AFF72DF3174FBACULL, "6156242_m.fxe"},
	{0x0AFFAB844108F181ULL, "follower_banter6155472.cl"},
	{0x0AFFC4E14DDC0A01ULL, "329491201.ogg"},
	{0x0AFFC913262182ABULL, "hlac_lt_gallowstemplar_n_2395_1z.dds"},
	{0x0B006EEA02C4F71BULL, "art\\levels\\races\\primeval\\props\\plotspecific\\prp_possessionsw.mmh"},
	{0x0B007AEA02C50BB8ULL, "art\\levels\\races\\primeval\\props\\plotspecific\\prp_possessionsw.mao"},
	{0x0B008CDD2F597E79ULL, "dae300_vashothlead6062785.cl"},
	{0x0B008CEA02C52A05ULL, "art\\levels\\races\\primeval\\props\\plotspecific\\prp_possessionsw.msh"},
	{0x0B01E3EB93B346D7ULL, "pro000_cut2d_real_story6001241.cl"},
	{0x0B022B7C442FD6C4ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\rightarm_generic_long_1.ani"},
	{0x0B027F621103CC69ULL, "vxw_sundermount_2im_0_0.vlm"},
	{0x0B02F5A88FB6D4B6ULL, "242270726.ogg"},
	{0x0B034780D4FF1DB2ULL, "39951816.ogg"},
	{0x0B0399CE5FDC662EULL, "hlac_lt_0_lowtown_docks_n_-131071_0y.dds"},
	{0x0B03AE9139B3B9F4ULL, "vxht_mansion2b_1su_0_0.vlm"},
	{0x0B0402A026F4EA19ULL, "art\\characters\\playercharacter\\humanmale\\hlm_tmpb.msh"},
	{0x0B040CA026F4FB07ULL, "art\\characters\\playercharacter\\humanmale\\hlm_tmpb.mmh"},
	{0x0B0432F02FEB457EULL, "vxca_caverns2prc_9g1_0_0.vlm"},
	{0x0B045042A170F839ULL, "vxlt_hangedman_n_2vu_0_0.vlm"},
	{0x0B048717AD8F298BULL, "vxlt_warehouse_n_ty_0_0.vlm"},
	{0x0B04A1D3B0498A1CULL, "cd_qcr221_cut_heros_reward.wbd"},
	{0x0B0548C995F816FBULL, "ttx_window_small_b_1.msh"},
	{0x0B0566C995F849E5ULL, "ttx_window_small_b_1.mmh"},
	{0x0B0599D7955C380DULL, "dae000_male_templar_mob6174369.cl"},
	{0x0B05FD91FD5D27A7ULL, "art\\characters\\playercharacter\\dwarfmale\\animation\\cindes\\varric\\sitting_back_loop.ani"},
	{0x0B0609F1D74B1030ULL, "vxlt_warehouse_n_8b_0_0.vlm"},
	{0x0B0628A174ACA1EEULL, "hlac_lt_undercitydungeon_-1_7z.dds"},
	{0x0B0661DBF270CC63ULL, "dae021_sister_lorena_sta6142647.cl"},
	{0x0B067A1AA10FA696ULL, "6109217_m.fxe"},
	{0x0B070F404705C5F7ULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_ear01b.phy"},
	{0x0B0778FA0CD0B521ULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\r_wkf.ani"},
	{0x0B0817998443C3ADULL, "vxht_brothel_n_5fo_0_0.vlm"},
	{0x0B08357D01CBB530ULL, "llac_lt_0_lowtown_a2_-65536_7.dds"},
	{0x0B08962647ED0CBBULL, "vxlt_0_lowtown_a3_n_c_gnc_0_0.vlm"},
	{0x0B090408AFBE0AC2ULL, "vxw_sundermount_2zs_0_0.vlm"},
	{0x0B09287C4435F026ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\rightarm_generic_long_1.gad"},
	{0x0B094AA3086710A4ULL, "6156545_m.fxe"},
	{0x0B09E76A76764EECULL, "gen_im_trs_crc_sta_nat_r1.uti"},
	{0x0B09F11BA60EDA56ULL, "fx_c_fresnel_add.mao"},
	{0x0B0A61BD8724AB6FULL, "6172988_m.fxe"},
	{0x0B0A73C97584CD47ULL, "773478355.ogg"},
	{0x0B0A9A6F2FE0E188ULL, "mag360_samson6042051.cl"},
	{0x0B0B064D2A503CE0ULL, "vxlt_house_dx_0_0.vlm"},
	{0x0B0B08346A118B90ULL, "hlac_w_woundedcoast_327684_0y.dds"},
	{0x0B0B165D60AFC1F8ULL, "lt_hangedman_3567_ll.mmh"},
	{0x0B0B30365FDD9282ULL, "hlac_w_encounter1_-65535_0z.dds"},
	{0x0B0B56003E5E022BULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_ear05b.phy"},
	{0x0B0BEE0EBC71F239ULL, "6155523_m.fxe"},
	{0x0B0BFC8436061E2BULL, "hlac_ht_brothel_n_3364_1y.dds"},
	{0x0B0BFCD1B25ABFE8ULL, "6066182_f.fxe"},
	{0x0B0C5ABC661A9DC3ULL, "one300_nuncio6073805.cl"},
	{0x0B0CCFD6F133D689ULL, "vxht_0_hightown_a2_ca0_0_0.vlm"},
	{0x0B0D4186FFE7462CULL, "567238867.ogg"},
	{0x0B0DC13E9ADED75AULL, "mer230ip_to_arlan_hideout.utp"},
	{0x0B0DDE05AC6FD153ULL, "513741480.ogg"},
	{0x0B0E6E18B60A5024ULL, "llac_w_bonepitmining_a2_65542_0.dds"},
	{0x0B0E83FA0CD6E6C3ULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\r_wkf.gad"},
	{0x0B0ED2ACE41DE48AULL, "vxw_blightlands_ib_0_0.vlm"},
	{0x0B0F2AA262DA1278ULL, "zz_vault_debug6203354.cl"},
	{0x0B0F7045C04191B8ULL, "vxca_caverns2prc_9zl_0_0.vlm"},
	{0x0B0FAE44386DCC6EULL, "6087209_m.fxe"},
	{0x0B10237C443C0605ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\rightarm_generic_long_1.evt"},
	{0x0B10F097653C1570ULL, "gencr_apostate.utc"},
	{0x0B10F13F02F9E4E9ULL, "vxw_sundermount_2rl_0_0.vlm"},
	{0x0B11898D20AD290AULL, "vxlt_0_lowtown_a3_n_c_by2_0_0.vlm"},
	{0x0B11B3F8D08EB0AAULL, "vxca_caverns1_int_0_0.vlm"},
	{0x0B11E84023AE4D32ULL, "6110365_m.fxe"},
	{0x0B12206E2BA9D274ULL, "vxht_mansion2b_1vz_0_0.vlm"},
	{0x0B13C2AA91EE998CULL, "6035938_m.fxe"},
	{0x0B13E93E1675CB35ULL, "141351815.ogg"},
	{0x0B13F420BE74CA9EULL, "6148702_m.fxe"},
	{0x0B1475A74AA0E022ULL, "art\\characters\\weapons\\staff\\textures\\w_stf_st02a_0s.dds"},
	{0x0B14BCAF47D715A8ULL, "dae000_keep_servant_1_sta6156894.cl"},
	{0x0B14DB9B3460453AULL, "52458453.ogg"},
	{0x0B15AB3C4899EA2EULL, "6154765_m.fxe"},
	{0x0B15EEE969DE4E09ULL, "405398035.ogg"},
	{0x0B15EF5FF4F5D3EFULL, "dae000ip_crowd_ff1_talk1.utp"},
	{0x0B15F103E59D3135ULL, "590412145.ogg"},
	{0x0B1647F62F5F9BE2ULL, "6105706_f.fxe"},
	{0x0B1651577478175FULL, "hlac_lt_shop_-1_4y.dds"},
	{0x0B1684585771D76EULL, "hlac_lt_0_lowtown_a3_n_c_-65537_1y.dds"},
	{0x0B1689767639DC2DULL, "vxht_brothel_n_5in_0_0.vlm"},
	{0x0B16B44326F0EB25ULL, "qun221_petrice6083820.cl"},
	{0x0B16BA94E2C7AB9BULL, "ave331_post_fight.bnk"},
	{0x0B16FF9A9AECF7A1ULL, "292491156.ogg"},
	{0x0B17284D082E9DC9ULL, "313331627.ogg"},
	{0x0B174E1EB27E6D0EULL, "941860372.ogg"},
	{0x0B17653CB78280AEULL, "6098831_f.fxe"},
	{0x0B177102163B25A2ULL, "hlac_lt_0_lowtown_docks_a3_n_-196610_2z.dds"},
	{0x0B17F8FC3F77C48BULL, "hlac_ht_mansion2b_818_37y.dds"},
	{0x0B183918F7494C6FULL, "llac_lt_0_lowtown_a3_n_c_65534_3.dds"},
	{0x0B1842485541B320ULL, "vxlt_hangedman_n_24z_0_0.vlm"},
	{0x0B1852840F4866BBULL, "hlac_lt_0_lowtown_docks_a3_131071_2y.dds"},
	{0x0B19BC94E2C9E55BULL, "ave331_post_fight.cnv"},
	{0x0B1A1DA5374F2F98ULL, "vxdr_primevaltomb_9d9_0_0.vlm"},
	{0x0B1A503DCBA187F5ULL, "6025981_m.fxe"},
	{0x0B1B2ED6605BCD7EULL, "298212969.ogg"},
	{0x0B1B357F4D43E9D9ULL, "vxht_mansion_n_6hm_0_0.vlm"},
	{0x0B1B5FF56A6869FDULL, "llac_ca_caverns2prc_5802_38.dds"},
	{0x0B1B6584A148CF8DULL, "character_character_p129.psh.fxo.dx11.assembly"},
	{0x0B1C10E336AA9890ULL, "hlac_lt_0_lowtown_n_-2_3x.dds"},
	{0x0B1C5E663FE99E85ULL, "vxca_caverns1_mines_i2r_0_0.vlm"},
	{0x0B1C81AB8E34915BULL, "106967439.ogg"},
	{0x0B1CF5EDDA319034ULL, "lgt000cr_epic_merchant_3.utc"},
	{0x0B1D4489D613FD0AULL, "vxw_blightlands_rc_0_0.vlm"},
	{0x0B1D6B0B101BFA1BULL, "lt_0_lowtown_docks_a3_226_0.mmh"},
	{0x0B1D7408A1FD1162ULL, "789051745.ogg"},
	{0x0B1D770B101C0EB8ULL, "lt_0_lowtown_docks_a3_226_0.mao"},
	{0x0B1D890B101C2D05ULL, "lt_0_lowtown_docks_a3_226_0.msh"},
	{0x0B1E1C2B29B076FCULL, "vxlt_0_lowtown_a2_n_c_ekl_0_0.vlm"},
	{0x0B1E5ECF219450C2ULL, "457476779.ogg"},
	{0x0B1E9E3CC0720026ULL, "art\\vfx\\combat\\vfx_r_asn_mdeath_tag_loop_c.impa.evt"},
	{0x0B1EBBE715064C47ULL, "gen00fl_aveline6120768.cl"},
	{0x0B1ECA682A7BFADDULL, "qcr220_grey_wardens6060406.cl"},
	{0x0B1FEFCB93808482ULL, "w_bonepitdestroyed_a3_3_0_ll.mmh"},
	{0x0B200BC91E3900E3ULL, "vxlt_darktown_6lg_0_0.vlm"},
	{0x0B2026750322700AULL, "6155490_m.fxe"},
	{0x0B2055AB109D4EE7ULL, "vxlt_gallowstemplar_f_55p_0_0.vlm"},
	{0x0B2090B19CF67F42ULL, "art\\characters\\playercharacter\\dwarfmale\\textures\\dm_cth_comx_0n.dds"},
	{0x0B210B8DC2B026A8ULL, "hlac_lt_warehouse_0_3x.dds"},
	{0x0B215E9598DAAC2EULL, "lla_ht_0_hightown_a2_-131073_0.dds"},
	{0x0B217DD998C266D6ULL, "vxlt_0_lowtown_a2_n_c_h1u_0_0.vlm"},
	{0x0B21CA5365CCF286ULL, "vxht_keep_5ay_0_0.vlm"},
	{0x0B21CDAAD5898E0AULL, "art\\levels\\races\\proxy\\props\\woodenstuff\\prp_zzsteps_0.pwk"},
	{0x0B21FB8D161712A4ULL, "6087575_m.fxe"},
	{0x0B2264329564E711ULL, "847041324.ogg"},
	{0x0B22887A14CA947BULL, "6076280_m.fxe"},
	{0x0B22A2C4E888ECBBULL, "zz_and_debug6178599.cl"},
	{0x0B22ABBBD90E1FE3ULL, "vxca_cavern1_haunted_ghl_0_0.vlm"},
	{0x0B22C61956085383ULL, "qcr000_arishok6059320.cl"},
	{0x0B2302D125EBDD3AULL, "vxlt_0_lowtown_a2_n_c_hqy_0_0.vlm"},
	{0x0B2320E4FCC7C2A3ULL, "gen00fl_aveline6122766.cl"},
	{0x0B23A698CA2BB11BULL, "hlac_lt_0_lowtown_131071_1x.dds"},
	{0x0B23DA880DADDE2BULL, "pro000_merchant6051630.cl"},
	{0x0B23F09CBCF360DFULL, "ser221_party_comments6166113.cl"},
	{0x0B2401C5B58D103DULL, "cd_nat380_sibling.bnk"},
	{0x0B241467151D4F12ULL, "314295815.ogg"},
	{0x0B241D68B8CD757DULL, "lgt321pt_shw_ors_cache.ncs"},
	{0x0B245F745E845F77ULL, "mag311_nyssa6041944.cl"},
	{0x0B247076AAB240FAULL, "lgt321pt_alistair_king.ncs"},
	{0x0B2494D8691F3602ULL, "ht_0_hightown_a3_28_0_lc.mmh"},
	{0x0B249D6DF2AC57D4ULL, "296437179.ogg"},
	{0x0B24A0797127190EULL, "338715270.ogg"},
	{0x0B24B53CC0769371ULL, "art\\vfx\\combat\\vfx_r_asn_mdeath_tag_loop_c.impa.gad"},
	{0x0B25632F5B047E16ULL, "vxca_caverns2prc_8j7_0_0.vlm"},
	{0x0B2564B0BF669782ULL, "vxht_bartrand_n_63t_0_0.vlm"},
	{0x0B2579E02BD93DBBULL, "vxlt_0_lowtown_a3_n_c_gxe_0_0.vlm"},
	{0x0B2593E7BF312961ULL, "6054295_m.fxe"},
	{0x0B263E95B0A406A6ULL, "llac_ca_cavern1_haunted_24424_20.dds"},
	{0x0B2695FB3E591F98ULL, "mer141_jansen6070135.cl"},
	{0x0B26DEA25F4E380AULL, "art\\characters\\creatures\\gibgolem\\animation\\armless\\walkf.gad"},
	{0x0B273D4FAF51A21CULL, "zz_prmpt_debug6200963.cl"},
	{0x0B2773C3D9209905ULL, "vxht_mansion2b_3ba_0_0.vlm"},
	{0x0B27819FE90CDF65ULL, "l_env_wal_lowtowndocks_cough.bnk"},
	{0x0B27AE6A7FEED617ULL, "471884814.ogg"},
	{0x0B281D2194FA2E57ULL, "w_bonepitmining_a2_249_0.msh"},
	{0x0B282F2194FA4CBAULL, "w_bonepitmining_a2_249_0.mao"},
	{0x0B28332194FA53C9ULL, "w_bonepitmining_a2_249_0.mmh"},
	{0x0B288DF6E0B88ED6ULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_flagchantrya_sa.dds"},
	{0x0B28E2DB3A42793EULL, "269095643.ogg"},
	{0x0B2A66FCBFB71148ULL, "6105901_m.fxe"},
	{0x0B2A87BEF54EA24FULL, "dae000_mother6136944.cl"},
	{0x0B2AF822DFAF05C4ULL, "387598950.ogg"},
	{0x0B2BA5BD19486621ULL, "reticle.vsh.fxo"},
	{0x0B2BB63CC07CB31FULL, "art\\vfx\\combat\\vfx_r_asn_mdeath_tag_loop_c.impa.ani"},
	{0x0B2BC71FE41D9178ULL, "vxlt_gallowstemplar_f_39e_0_0.vlm"},
	{0x0B2C22E6B9247E5AULL, "6149693_m.fxe"},
	{0x0B2C53FFA42DC2B8ULL, "vxca_caverns2prc_9lj_0_0.vlm"},
	{0x0B2C9C1BB4ED861DULL, "mrl000_marethari6129239.cl"},
	{0x0B2D405AB427B40BULL, "122372749.ogg"},
	{0x0B2DA71C1BB71714ULL, "vxlt_shop_dp_0_0.vlm"},
	{0x0B2DC1A25F5425A1ULL, "art\\characters\\creatures\\gibgolem\\animation\\armless\\walkf.evt"},
	{0x0B2DD4F8E6E615E9ULL, "vxw_sundermount_2xr_0_0.vlm"},
	{0x0B2E57A7EC737B2BULL, "follower_banter6160962.cl"},
	{0x0B2EE4EE7D4BD137ULL, "372929401.ogg"},
	{0x0B2F4CD158C318AFULL, "hf_madam_lusine.mor"},
	{0x0B2F8DA82BFAA699ULL, "vxlt_0_lowtown_docks_a3_btp_0_0.vlm"},
	{0x0B2FF5A0271A0F3CULL, "art\\characters\\playercharacter\\humanmale\\hlm_tmpb.phy"},
	{0x0B3046B589CD9E1FULL, "follower_banter6141581.cl"},
	{0x0B307C92BCE021C6ULL, "vxlt_0_lowtown_a2_eq9_0_0.vlm"},
	{0x0B309FD3B06EC196ULL, "cd_qcr221_cut_heros_reward.bnk"},
	{0x0B311D98CB043863ULL, "vxca_cavern1_haunted_geo_0_0.vlm"},
	{0x0B31C5EF53AAB9CDULL, "6043787_m.fxe"},
	{0x0B31F3C2BE953E55ULL, "6105921_f.fxe"},
	{0x0B32A4DFA4F211D9ULL, "w_bonepitdestroyed_a3_155_0.msh"},
	{0x0B32AEDFA4F222C7ULL, "w_bonepitdestroyed_a3_155_0.mmh"},
	{0x0B32B2DFA4F22914ULL, "w_bonepitdestroyed_a3_155_0.mao"},
	{0x0B33809DC7F4E439ULL, "327523708.ogg"},
	{0x0B33F5AA53B72F8EULL, "190461508.ogg"},
	{0x0B342D1239CB748EULL, "art\\characters\\creatures\\dragon\\c_dragonb_eyes.mao"},
	{0x0B3462B4C05EC7EFULL, "vxlt_warehouse_ac_0_0.vlm"},
	{0x0B34B1C33FB14AECULL, "6000974_m.fxe"},
	{0x0B353C084FF8BE89ULL, "610767424.ogg"},
	{0x0B35658E3C0E8BEAULL, "696512283.ogg"},
	{0x0B3580D59D48882CULL, "hlac_lt_backalley_g_65536_5z.dds"},
	{0x0B35C18E22B3EE20ULL, "hlac_lt_0_lowtown_a2_n_-196609_4z.dds"},
	{0x0B35D357C660013CULL, "hlac_ht_keep_n_3735_0z.dds"},
	{0x0B35F2593E4821A0ULL, "vxlt_backalley_n_207_0_0.vlm"},
	{0x0B35F5EC2B9848AAULL, "vxlt_gallowscourtyard_65c_0_0.vlm"},
	{0x0B364456BAD9E157ULL, "hlac_lt_0_lowtown_-1_2z.dds"},
	{0x0B36530B53675A66ULL, "184579480.ogg"},
	{0x0B37AB6919EB8CF5ULL, "vxlt_0_lowtown_docks_n_cv3_0_0.vlm"},
	{0x0B3842F1DF86F0C1ULL, "dae000cr_grand_cleric6124775.cl"},
	{0x0B384449D04E524BULL, "vxlt_0_lowtown_ex3_0_0.vlm"},
	{0x0B385359CBBFC19BULL, "923173248.ogg"},
	{0x0B387F57E5739CB7ULL, "6126952_m.fxe"},
	{0x0B394DE0AC755600ULL, "dae200cr_brothel_gamlen.utc"},
	{0x0B3953A1DA49DFEAULL, "gen00fl_fenris6115371.cl"},
	{0x0B399FAA1C6905C8ULL, "6062112_m.fxe"},
	{0x0B39C48D3A044DFBULL, "vxht_brothel_n_5gz_0_0.vlm"},
	{0x0B3A35BD500854A4ULL, "dae000_katriela6166395.cl"},
	{0x0B3A6CA9D33D46F7ULL, "vxlt_0_lowtown_a3_n_c_gnw_0_0.vlm"},
	{0x0B3A989601ECCE7FULL, "zz_followers_banter6158484.cl"},
	{0x0B3AB9A25F5EA2F0ULL, "art\\characters\\creatures\\gibgolem\\animation\\armless\\walkf.ani"},
	{0x0B3B0FBF25BD5BCAULL, "vxht_brothel_n_3es_0_0.vlm"},
	{0x0B3B4D23BFA4FD68ULL, "6147698_f.fxe"},
	{0x0B3B594F8E8A14A8ULL, "6125257_m.fxe"},
	{0x0B3BCD63F8899F5DULL, "matdef_dca_diffuse.dds"},
	{0x0B3C056E6C45BEF4ULL, "art\\levels\\races\\natural\\kirkcoastal\\props\\prp_01ruinsaltar.phy"},
	{0x0B3CF19487F311A4ULL, "w_bonepitdestroyed_a3_44_0.msh"},
	{0x0B3D039487F3300DULL, "w_bonepitdestroyed_a3_44_0.mao"},
	{0x0B3D079487F33746ULL, "w_bonepitdestroyed_a3_44_0.mmh"},
	{0x0B3ECDA3C55695C3ULL, "6112111_m.fxe"},
	{0x0B3F2ED463CABE0DULL, "vxht_bartrand_n_32u_0_0.vlm"},
	{0x0B3F80BC4ACDA8B1ULL, "qun211_party_comment6072256.cl"},
	{0x0B3FEEC42D6E5A1FULL, "377830166.ogg"},
	{0x0B401C8887F38163ULL, "6054827_f.fxe"},
	{0x0B408DC8DCC8F31CULL, "hlac_lt_0_lowtown_a2_n_-2_5x.dds"},
	{0x0B418C3527FFDC8DULL, "hlac_lt_0_lowtown_docks_n_-131074_1z.dds"},
	{0x0B422A65AA294420ULL, "vxht_bartrand_n_4hq_0_0.vlm"},
	{0x0B422BE42D639BB7ULL, "630793870.ogg"},
	{0x0B42486AA352C882ULL, "vxht_bartrand_n_69v_0_0.vlm"},
	{0x0B424D595DBE35ECULL, "gen00fl_isabela6135667.cl"},
	{0x0B42A86D53EEE903ULL, "933621371.ogg"},
	{0x0B42DBF591566D7EULL, "66959214.ogg"},
	{0x0B42E69D8DED7E80ULL, "962974294.ogg"},
	{0x0B433298E6FCB5C9ULL, "6099577_m.fxe"},
	{0x0B4428E41BD6FD2FULL, "vxht_mansion2b_2uk_0_0.vlm"},
	{0x0B451908D032B5B5ULL, "art\\vfx\\models\\vfx_gui_brittle.msh"},
	{0x0B451B08D032B92BULL, "art\\vfx\\models\\vfx_gui_brittle.mmh"},
	{0x0B45352B7601F157ULL, "vxht_mansion2b_21o_0_0.vlm"},
	{0x0B45A7D8019E411FULL, "vxlt_0_lowtown_docks_cyg_0_0.vlm"},
	{0x0B45C0F495695623ULL, "945065132.ogg"},
	{0x0B45E26E6C4DD7F1ULL, "art\\levels\\races\\natural\\kirkcoastal\\props\\prp_01ruinsaltar.msh"},
	{0x0B45F06E6C4DEF0CULL, "art\\levels\\races\\natural\\kirkcoastal\\props\\prp_01ruinsaltar.mao"},
	{0x0B45FC6E6C4E03AFULL, "art\\levels\\races\\natural\\kirkcoastal\\props\\prp_01ruinsaltar.mmh"},
	{0x0B461D460BE1A575ULL, "vxlt_0_lowtown_docks_n_cq2_0_0.vlm"},
	{0x0B4638C15B5E9200ULL, "43766156.ogg"},
	{0x0B469D4DB758C03AULL, "vxca_caverns2prc_a9o_0_0.vlm"},
	{0x0B46A2A754145800ULL, "ico_gloves_medium_1.dds"},
	{0x0B46B626C2446ACBULL, "vxlt_0_lowtown_es2_0_0.vlm"},
	{0x0B46F18C390317FAULL, "gen00fl_sebastian6174687.cl"},
	{0x0B47670ECD0F3D14ULL, "gen00fl_anders6196824.cl"},
	{0x0B477A22AE08B6F2ULL, "902663523.ogg"},
	{0x0B478B6E9E03856AULL, "art\\levels\\races\\ferelden\\props\\dead\\prp_02bricks.mao"},
	{0x0B478F6E9E038BB9ULL, "art\\levels\\races\\ferelden\\props\\dead\\prp_02bricks.mmh"},
	{0x0B47996E9E039CA7ULL, "art\\levels\\races\\ferelden\\props\\dead\\prp_02bricks.msh"},
	{0x0B48069175D37165ULL, "hlac_ht_brothel_n_3381_0y.dds"},
	{0x0B480F1D2CB66E42ULL, "gen00fl_aveline6197228.cl"},
	{0x0B48A01A189FC165ULL, "6065388_f.fxe"},
	{0x0B4912E9D91F2B5FULL, "ave200_aveline_bar6116376.cl"},
	{0x0B4950D4CED38B6EULL, "art\\vfx\\combat\\vfx_d_qun_aoe_p.cess.evt"},
	{0x0B4988E6F2A86E67ULL, "vxlt_gallowstemplar_n_5by_0_0.vlm"},
	{0x0B49EADA9FF43F91ULL, "vxlt_gallowscourtyard_n_3x1_0_0.vlm"},
	{0x0B4A038DF9E7687DULL, "hlac_lt_0_lowtown_a3_n_c_131071_1x.dds"},
	{0x0B4A315F974835F4ULL, "pro000_opening6178521.cl"},
	{0x0B4B297FBC853B75ULL, "329159013.ogg"},
	{0x0B4B70F478AF562FULL, "vxht_bartrand_n_653_0_0.vlm"},
	{0x0B4B8010E5396BF0ULL, "ser211ip_note_scribbles_3.utp"},
	{0x0B4BBDBE7391A887ULL, "vxdr_0_deeproads_ctz_0_0.vlm"},
	{0x0B4BE142BCADE5D2ULL, "gen00fl_sebastian6198195.cl"},
	{0x0B4C2BA580DC415DULL, "vxht_mansion2b_23g_0_0.vlm"},
	{0x0B4CA38BEC7786E4ULL, "w_sundermount_134_0.phy"},
	{0x0B4CA57E7888DC35ULL, "6108749_m.fxe"},
	{0x0B4CD5DD19B195EEULL, "art\\characters\\creatures\\hurlock\\animation\\sword_shield\\hurlock_sw_combat_runf.evt"},
	{0x0B4E07948801AA0DULL, "w_bonepitdestroyed_a3_44_0.rml"},
	{0x0B4E156D8E574FD1ULL, "art\\levels\\4cinematics\\dwarf\\prp_dwarfchair.msh"},
	{0x0B4E2F6D8E577B8FULL, "art\\levels\\4cinematics\\dwarf\\prp_dwarfchair.mmh"},
	{0x0B4E6989B7DEB8D5ULL, "dae000_arishok6067767.cl"},
	{0x0B4E73EB059B85F0ULL, "art\\vfx\\combat\\vfx_w_2hd_atk_cl2_c.mmh"},
	{0x0B4EC0128E86AA5DULL, "184842275.ogg"},
	{0x0B4EC9C07E679397ULL, "860633416.ogg"},
	{0x0B4ECFC06761EFCAULL, "431782801.ogg"},
	{0x0B4EDECFA99CB53AULL, "6098847_m.fxe"},
	{0x0B4EE88E11F6A079ULL, "hlac_lt_0_lowtown_a2_65533_1y.dds"},
	{0x0B4EF040474346ECULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_ear01b.mmh"},
	{0x0B4EF640474350DEULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_ear01b.msh"},
	{0x0B4F172632A99948ULL, "ser221_party_comments6209855.cl"},
	{0x0B4F47D4CED7E719ULL, "art\\vfx\\combat\\vfx_d_qun_aoe_p.cess.gad"},
	{0x0B4F4B7D0A8F0D8AULL, "6122441_m.fxe"},
	{0x0B4F60DF4575CF85ULL, "177888284.ogg"},
	{0x0B50449F325C1C8CULL, "additivedot01.mao"},
	{0x0B505FE0F9298830ULL, "hlac_lt_0_lowtown_a3_n_c_-196607_0y.dds"},
	{0x0B507E8070D4DD56ULL, "follower_banter6160789.cl"},
	{0x0B508F67EC2BDD7AULL, "vxlt_backalley_n_1u5_0_0.vlm"},
	{0x0B524B6FE975E424ULL, "196488332.ogg"},
	{0x0B529A8BEC7BE36CULL, "w_sundermount_134_0.rml"},
	{0x0B52A8CE392FE21DULL, "llac_ca_caverns1_mines_24422_7.dds"},
	{0x0B534DFCE98CEB8AULL, "gen_im_npc_elf_arm_cht_lgtc.uti"},
	{0x0B53CF82EF632ADEULL, "one200dg_solivitus6060325.cl"},
	{0x0B5462F40E29E0D4ULL, "vxca_caverns1_mines_iib_0_0.vlm"},
	{0x0B549C64BB9AB238ULL, "zz_qun_debug6045390.cl"},
	{0x0B550F2AA94ED8BAULL, "vxca_caverns2prc_a4h_0_0.vlm"},
	{0x0B5572BDF3560C52ULL, "ht_keep_3741.anb"},
	{0x0B558002CE0D356FULL, "680031250.ogg"},
	{0x0B5658D4CEDE2367ULL, "art\\vfx\\combat\\vfx_d_qun_aoe_p.cess.ani"},
	{0x0B56FA4F9A1984B9ULL, "art\\vfx\\cinematic\\vfx_isa_intro_glass_smash.impa.ani"},
	{0x0B5702A8C7B8A824ULL, "6101457_m.fxe"},
	{0x0B575063B72977F7ULL, "vxlt_0_lowtown_a3_n_c_gxq_0_0.vlm"},
	{0x0B57B0E1D5FA9DB3ULL, "vxht_0_hightown_n_7g3_0_0.vlm"},
	{0x0B57D6428FB48F0FULL, "gen_im_arm_cht_nobb_t3.uti"},
	{0x0B5801705A6D6B3FULL, "270179991.ogg"},
	{0x0B5810CAA6C5C83DULL, "hlac_w_encounter2_camp_65538_0z.dds"},
	{0x0B581B7CCF710A46ULL, "6180607_m.fxe"},
	{0x0B5823E7661C5F2EULL, "mrl310_merrill6173877.cl"},
	{0x0B583B74EF26E6A4ULL, "6141619_m.fxe"},
	{0x0B58523F15FEABA4ULL, "vxht_0_hightown_n_ajm_0_0.vlm"},
	{0x0B5879942AEF389BULL, "art\\vfx\\combat\\vfx_w_bsk_regain_sta_imp_c.impa.ani"},
	{0x0B59C1149D0FC0D5ULL, "6141867_m.fxe"},
	{0x0B5A99D419583FBAULL, "vxlt_warehouse_157_0_0.vlm"},
	{0x0B5AB0607F661157ULL, "450056270.ogg"},
	{0x0B5B51F6299737FAULL, "hlac_ca_caverns1_mines_24425_9x.dds"},
	{0x0B5B64C5E0EB8238ULL, "vxht_keep_5u3_0_0.vlm"},
	{0x0B5B87A00ED51C8BULL, "t3_eey_br2.tnt"},
	{0x0B5BD39ADBF732FDULL, "vxlt_gallowstemplar_5au_0_0.vlm"},
	{0x0B5C1F2B2BCC6513ULL, "10225711.ogg"},
	{0x0B5C8646BB530EFDULL, "root_mage_primal2.dds"},
	{0x0B5C96EB05A7FE0FULL, "art\\vfx\\combat\\vfx_w_2hd_atk_cl2_c.anb"},
	{0x0B5D0F144CCC154BULL, "vxca_cave_1b1_0_0.vlm"},
	{0x0B5D8655F798D1B7ULL, "6137068_m.fxe"},
	{0x0B5DB596C0E224FBULL, "art\\characters\\playercharacter\\dwarfmale\\animation\\cindes\\pose\\pose_f_hand_gesture_other.ani"},
	{0x0B5DD54F9A1F64FBULL, "art\\vfx\\cinematic\\vfx_isa_intro_glass_smash.impa.gad"},
	{0x0B5E6E4F0B1D1A21ULL, "dae221_viscount_chantry6154266.cl"},
	{0x0B5EA3520CB43BC5ULL, "vxlt_backalley_n_1o0_0_0.vlm"},
	{0x0B5F8F1984FE6466ULL, "lowtown_male_lowlife2_mob.cnv"},
	{0x0B5F97DFA518E9FCULL, "w_bonepitdestroyed_a3_155_0.phy"},
	{0x0B5FB84B9641116FULL, "vxlt_warehouse_pl_0_0.vlm"},
	{0x0B5FBD1420D4D33FULL, "503869286.ogg"},
	{0x0B601652CAE8E6C9ULL, "6093777_m.fxe"},
	{0x0B6040E6A4714282ULL, "6075532_m.fxe"},
	{0x0B60615092DA8FA8ULL, "ave321cr_gen_guardsman.utc"},
	{0x0B6071C81D94EC72ULL, "qun120cr_javaris6042975.cl"},
	{0x0B6094133DC935BCULL, "6188645_m.fxe"},
	{0x0B60B1A1F6FCECABULL, "6047248_m.fxe"},
	{0x0B60C52304CA23A9ULL, "6105940_m.fxe"},
	{0x0B60ED2256CB64B5ULL, "dae300_vashothlead6065065.cl"},
	{0x0B610843CBCC3E02ULL, "art\\characters\\creatures\\mubari\\animation\\ctn180l_cwk.gad"},
	{0x0B615B11DD310FC6ULL, "plt_cod_bks_black_fox.plo"},
	{0x0B621F3383B17D6CULL, "llac_lt_0_lowtown_a3_n_c_65537_3.dds"},
	{0x0B6237AC0F16FBEAULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\boss\\meredith\\combat\\fightdisengage.gad"},
	{0x0B629DA9CEBD57B5ULL, "vxht_keep_n_3zv_0_0.vlm"},
	{0x0B62B458D50F3526ULL, "6129071_m.fxe"},
	{0x0B62D238AA1EBAA3ULL, "6079667_m.fxe"},
	{0x0B62FD4D67FBA4F5ULL, "57487671.ogg"},
	{0x0B6301297136B22BULL, "vxlt_gallowstemplar_45r_0_0.vlm"},
	{0x0B636A5F6CC4F87FULL, "art\\characters\\playercharacter\\humanfemale\\rob_appb.phy"},
	{0x0B6381079B44F13AULL, "vxca_caverns2prc_a3i_0_0.vlm"},
	{0x0B63911985025026ULL, "lowtown_male_lowlife2_mob.bnk"},
	{0x0B6399E0A6309BCBULL, "vxlt_0_lowtown_ei4_0_0.vlm"},
	{0x0B63B309ECEC98F3ULL, "vxlt_0_lowtown_docks_n_bay_0_0.vlm"},
	{0x0B63CC77A33D324CULL, "vxlt_0_lowtown_docks_a3_n_d08_0_0.vlm"},
	{0x0B63F24F9A2401E0ULL, "art\\vfx\\cinematic\\vfx_isa_intro_glass_smash.impa.evt"},
	{0x0B641BC1DBE12E1CULL, "plt_lgt260pt_way_blood_mage.plo"},
	{0x0B644E4155AFC2D6ULL, "6187945_m.fxe"},
	{0x0B648D7C221546C6ULL, "vxht_0_hightown_n_b95_0_0.vlm"},
	{0x0B64F0E1B5904BECULL, "94414545.ogg"},
	{0x0B64FC21156D7494ULL, "zz_fen_debug6134363.cl"},
	{0x0B6581942AF9D092ULL, "art\\vfx\\combat\\vfx_w_bsk_regain_sta_imp_c.impa.evt"},
	{0x0B659609BC83CB4BULL, "vxht_chantry_n_2yd_0_0.vlm"},
	{0x0B65DF894488BFB3ULL, "vxdr_0_deeproads_fip_0_0.vlm"},
	{0x0B65E9E2825D41FBULL, "hlac_lt_0_lowtown_a2_-131073_2x.dds"},
	{0x0B65FD05A3157AFDULL, "ser210_witness6168177.cl"},
	{0x0B660C3E842F45A9ULL, "gen00fl_varric6103557.cl"},
	{0x0B66875AA9CA40C2ULL, "dae000_bodhan6129078.cl"},
	{0x0B66E1263E93DD21ULL, "vxlt_darktown_5tf_0_0.vlm"},
	{0x0B67357830008C13ULL, "6036925_m.fxe"},
	{0x0B679AF554D69296ULL, "art\\vfx\\combat\\vfx_d_invulner_sustain_c.anb"},
	{0x0B67B37706CFC3A5ULL, "art\\characters\\playercharacter\\shared\\heads\\textures\\mt_hed_infb_0n.dds"},
	{0x0B68207AEF5D4F91ULL, "552102237.ogg"},
	{0x0B682B43CBD298F9ULL, "art\\characters\\creatures\\mubari\\animation\\ctn180l_cwk.evt"},
	{0x0B683B3E22F32113ULL, "vxht_mansion2_1wb_0_0.vlm"},
	{0x0B684F3E8546CBCFULL, "49794167.ogg"},
	{0x0B68CC6DADB5D954ULL, "600069353.ogg"},
	{0x0B691AAC0F1CE881ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\boss\\meredith\\combat\\fightdisengage.evt"},
	{0x0B6974FC1E007CC1ULL, "gen00fl_varric6093616.cl"},
	{0x0B69C1F14154AA71ULL, "6101231_f.fxe"},
	{0x0B6A5BA534C058F6ULL, "vxlt_0_lowtown_a3_n_c_i4f_0_0.vlm"},
	{0x0B6A843B56D57314ULL, "art\\levels\\races\\design\\animation\\beartrap.activetoinactive.ani"},
	{0x0B6B284648137925ULL, "vxw_bonepitdestroyed_a3_4q_0_0.vlm"},
	{0x0B6B3CF8414A76D6ULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\0rw_basic.msh"},
	{0x0B6B46F8414A87D4ULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\0rw_basic.mmh"},
	{0x0B6B4AF8414A8EE7ULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\0rw_basic.mao"},
	{0x0B6B516B6CBE1F9EULL, "art\\vfx\\environment\\vfx_pois_gas_area_med.dura.ani"},
	{0x0B6B5F21D51FB27DULL, "llac_ca_cavern1_haunted_24426_10.dds"},
	{0x0B6B73D74C6C6C02ULL, "art\\vfx\\environment\\vfx_trap_fire_jet_v_p.dura.gad"},
	{0x0B6B88CD88019B38ULL, "55983099.ogg"},
	{0x0B6C86AE9EA2EEEFULL, "vxlt_0_lowtown_a2_n_es9_0_0.vlm"},
	{0x0B6CA0942B002425ULL, "art\\vfx\\combat\\vfx_w_bsk_regain_sta_imp_c.impa.gad"},
	{0x0B6CC57216907426ULL, "6139275_f.fxe"},
	{0x0B6CD9E39FF15D8BULL, "art\\vfx\\combat\\vfx_glb_att_fail_c.cess.ani"},
	{0x0B6CEEFFCD9D1B17ULL, "art\\cinematics\\_crit path\\prologue\\pro000cs_flemeth_real\\cs_hur1_flemeth_real.gad"},
	{0x0B6D56B7FBE9FA01ULL, "vxw_bonepitdestroyed_a3_9n_0_0.vlm"},
	{0x0B6DA5DE96E9C9C9ULL, "gen00fl_aveline6113993.cl"},
	{0x0B6DE69A6C98412EULL, "6043884_f.fxe"},
	{0x0B6E2A28883729EFULL, "vxlt_warehouse_uo_0_0.vlm"},
	{0x0B6E3D62B1CF94EFULL, "llac_lt_undercityencounter_-1_18.dds"},
	{0x0B6EA2D94A21E39DULL, "vxht_mansion_n_5a5_0_0.vlm"},
	{0x0B6EE2AA00ABE327ULL, "character_character_p97.psh.fxo"},
	{0x0B6F7E7AF1B946AFULL, "vxht_mansion2b_2dt_0_0.vlm"},
	{0x0B6F94ED3E0D3165ULL, "6093526_m.fxe"},
	{0x0B6FC7B879484260ULL, "6100259_f.fxe"},
	{0x0B6FEEB0EF1EA55EULL, "vxht_mansion_n_53h_0_0.vlm"},
	{0x0B70FB2B37AA51FBULL, "llac_lt_0_lowtown_n_65534_2.dds"},
	{0x0B7127F57EA27CC3ULL, "vxca_cavern1_haunted_gkp_0_1.vlm"},
	{0x0B7166F9D222F962ULL, "6161431_m.fxe"},
	{0x0B71671C4A279B29ULL, "6055627_m.fxe"},
	{0x0B7173B31EB23372ULL, "gen00flst_isa_init_2.stg"},
	{0x0B71813B56DB8D76ULL, "art\\levels\\races\\design\\animation\\beartrap.activetoinactive.gad"},
	{0x0B71AA5DAC10BEEBULL, "gen_im_wep_war_shd_lore_01.uti"},
	{0x0B71DC96C0F30F85ULL, "art\\characters\\playercharacter\\dwarfmale\\animation\\cindes\\pose\\pose_f_hand_gesture_other.gad"},
	{0x0B72374CC8184223ULL, "art\\vfx\\combat\\vfx_r_shd_tdecay_loop_c.impa.evt"},
	{0x0B7296D74C72C6F9ULL, "art\\vfx\\environment\\vfx_trap_fire_jet_v_p.dura.evt"},
	{0x0B72C51EBBA47A4AULL, "6071508_f.fxe"},
	{0x0B72F77D528239B1ULL, "256887159.ogg"},
	{0x0B734E38ABBA3180ULL, "vxca_caverns1_mines_hu7_0_0.vlm"},
	{0x0B7357E48EA080CDULL, "6087058_f.fxe"},
	{0x0B7407E6AE79E3CBULL, "vxht_chantry_n_2tg_0_0.vlm"},
	{0x0B746B5F6CD36C84ULL, "art\\characters\\playercharacter\\humanfemale\\rob_appb.mmh"},
	{0x0B746F5F6CD373D7ULL, "art\\characters\\playercharacter\\humanfemale\\rob_appb.mao"},
	{0x0B74815F6CD39226ULL, "art\\characters\\playercharacter\\humanfemale\\rob_appb.msh"},
	{0x0B7481F94F197B95ULL, "vxlt_gallowscourtyard_n_774_0_0.vlm"},
	{0x0B74BF432D343F30ULL, "6064654_f.fxe"},
	{0x0B74E41F59890468ULL, "dae000_arishok6080482.cl"},
	{0x0B75015C5A246F82ULL, "6042884_m.fxe"},
	{0x0B757910865AD0A3ULL, "vxca_cavern1_haunted_hz1_0_0.vlm"},
	{0x0B759DF554E2D501ULL, "art\\vfx\\combat\\vfx_d_invulner_sustain_c.mmh"},
	{0x0B75CA7D92861370ULL, "vxca_caverns1_mines_gk6_0_0.vlm"},
	{0x0B75D5F00BAABB98ULL, "art\\levels\\races\\proxy\\props\\4tools\\inv_capsule2x1_tool.phy"},
	{0x0B762E9495B7162AULL, "505876861.ogg"},
	{0x0B763343CBDEE3E8ULL, "art\\characters\\creatures\\mubari\\animation\\ctn180l_cwk.ani"},
	{0x0B76E581D878539EULL, "zz_mrl_debug6185299.cl"},
	{0x0B7712AC0F2918D0ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\boss\\meredith\\combat\\fightdisengage.ani"},
	{0x0B77A725F8FB6DBAULL, "hlac_lt_0_lowtown_-65536_6y.dds"},
	{0x0B7812F70AB90F1BULL, "art\\cinematics\\_crit path\\mage crisis\\rulesflees\\playerrules_extratemplarfemale3_.ani"},
	{0x0B789C3B56E1D9D5ULL, "art\\levels\\races\\design\\animation\\beartrap.activetoinactive.evt"},
	{0x0B78B754BFE363FDULL, "vxlt_gallowstemplar_5ws_0_0.vlm"},
	{0x0B79104CC81E1EA4ULL, "art\\vfx\\combat\\vfx_r_shd_tdecay_loop_c.impa.gad"},
	{0x0B7940BC06DB8559ULL, "art\\characters\\playercharacter\\humanfemale\\hlh_wic.mmh"},
	{0x0B794ABC06DB9647ULL, "art\\characters\\playercharacter\\humanfemale\\hlh_wic.msh"},
	{0x0B79696B6CCA866FULL, "art\\vfx\\environment\\vfx_pois_gas_area_med.dura.evt"},
	{0x0B797C5A20441F56ULL, "art\\characters\\creatures\\mubari\\animation\\drnk_ptn.ani"},
	{0x0B799F011C445D16ULL, "6036329_f.fxe"},
	{0x0B7A55F0C3CBF21DULL, "vxlt_hangedman_1w6_0_0.vlm"},
	{0x0B7A5CA362521A49ULL, "vxlt_0_lowtown_a2_n_e15_0_0.vlm"},
	{0x0B7A6EB414B17F54ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\additive\\weapon_x_pos.gad"},
	{0x0B7AB815B93E3723ULL, "vxca_caverns1_ckp_0_0.vlm"},
	{0x0B7AC1E39FFD7202ULL, "art\\vfx\\combat\\vfx_glb_att_fail_c.cess.evt"},
	{0x0B7ACEFDF7165654ULL, "vrc321st_bartrand_papers.stg"},
	{0x0B7B3E47AD4CD23FULL, "1060156255.ogg"},
	{0x0B7C72F4A8D74150ULL, "mrl000pt_gift_ring.ncs"},
	{0x0B7C7E680F38B242ULL, "ach_comp_level_2.dds"},
	{0x0B7CBFBA5B211C6DULL, "648648962.ogg"},
	{0x0B7CC358D552518FULL, "6110948_m.fxe"},
	{0x0B7D219A93121662ULL, "6153236_m.fxe"},
	{0x0B7D5486F2A1FBA7ULL, "character_character_p40.psh.fxo"},
	{0x0B7DCE1B4066C0CCULL, "971939987.ogg"},
	{0x0B7DE262EF306AD1ULL, "sse_ss_crow_m_1.gda"},
	{0x0B7DE3CDAC0CC509ULL, "293477834.ogg"},
	{0x0B7DF057E3AF5F2FULL, "vxht_mansion2b_2aw_0_0.vlm"},
	{0x0B7E22B38FC66D16ULL, "484098421.ogg"},
	{0x0B7E4103B76D6845ULL, "vxw_blightlands_13f_0_0.vlm"},
	{0x0B7F4A788F5547DCULL, "m_dae240ar_bone_pit.wbd"},
	{0x0B7F6A6B6CCEF318ULL, "art\\vfx\\environment\\vfx_pois_gas_area_med.dura.gad"},
	{0x0B7FA6A944732071ULL, "hlac_lt_backalley_g_0_14z.dds"},
	{0x0B7FBFA7CEA4FD61ULL, "vxw_sundermount_34w_0_0.vlm"},
	{0x0B801F4CC8245682ULL, "art\\vfx\\combat\\vfx_r_shd_tdecay_loop_c.impa.ani"},
	{0x0B8039871CDCEB64ULL, "6156310_m.fxe"},
	{0x0B805699CD5A3596ULL, "866606914.ogg"},
	{0x0B809ED74C7F11E8ULL, "art\\vfx\\environment\\vfx_trap_fire_jet_v_p.dura.ani"},
	{0x0B80B3A0AC70611CULL, "vxca_caverns2prc_6hu_0_0.vlm"},
	{0x0B80B6BEEDD99B77ULL, "gen00fl_anders6111280.cl"},
	{0x0B80BB143F3E7816ULL, "vxlt_0_lowtown_docks_95h_0_0.vlm"},
	{0x0B80EF351FC51C08ULL, "l_env_wal_gallows_callout.wbd"},
	{0x0B8113CFE70137D6ULL, "art\\characters\\creatures\\highdragon\\animation\\talent\\humangrabbed_loop.ani"},
	{0x0B811BFFCDAE109DULL, "art\\cinematics\\_crit path\\prologue\\pro000cs_flemeth_real\\cs_hur1_flemeth_real.ani"},
	{0x0B81342EC4F9D874ULL, "6085551_m.fxe"},
	{0x0B814165DA5EA01DULL, "vxw_bonepitdestroyed_a3_25n_0_0.vlm"},
	{0x0B814172846A347EULL, "6109711_f.fxe"},
	{0x0B816A2EDB1FCA2DULL, "zz_followers_banter6158002.cl"},
	{0x0B817DB414B7B732ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\additive\\weapon_x_pos.ani"},
	{0x0B81C0E3A0038F15ULL, "art\\vfx\\combat\\vfx_glb_att_fail_c.cess.gad"},
	{0x0B81E01BCF366BF4ULL, "zz_mag_debug6039790.cl"},
	{0x0B81E43F18CB3D07ULL, "vxht_chantry_2cu_0_0.vlm"},
	{0x0B82471916E69CF1ULL, "775130811.ogg"},
	{0x0B82A8E19325CBB2ULL, "lgt120_orz_bln_official6176122.cl"},
	{0x0B82F9B7C64B2416ULL, "667564765.ogg"},
	{0x0B83301A61218F9DULL, "6105830_m.fxe"},
	{0x0B8345474B060EDFULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\_textures\\lt_set_hangtarp_new_d.dds"},
	{0x0B8469D87912D72AULL, "vxw_encounter3_96_0_0.vlm"},
	{0x0B847B2D4A4F6EF7ULL, "hlac_ca_caverns2_5802_64z.dds"},
	{0x0B84BEC79739D89AULL, "vxht_mansion2_3gh_0_1.vlm"},
	{0x0B84EF52A5B0AD81ULL, "6103458_m.fxe"},
	{0x0B852F9B9CFA1590ULL, "6121565_m.fxe"},
	{0x0B85AC0633B6BFAEULL, "vxht_0_hightown_n_9pa_0_0.vlm"},
	{0x0B8606780CFE278FULL, "art\\characters\\creatures\\mubari\\animation\\hv_dmg.gad"},
	{0x0B86122565651476ULL, "vxca_caverns1_gkm_0_0.vlm"},
	{0x0B8624FB111DC6A2ULL, "zz_approval_debug6133708.cl"},
	{0x0B864CD752DFD956ULL, "lt_0_lowtown_docks_n_206_0.mao"},
	{0x0B8658D752DFEDF5ULL, "lt_0_lowtown_docks_n_206_0.mmh"},
	{0x0B865AD752DFF16BULL, "lt_0_lowtown_docks_n_206_0.msh"},
	{0x0B86845A204EB747ULL, "art\\characters\\creatures\\mubari\\animation\\drnk_ptn.evt"},
	{0x0B86C63F4BF6E96EULL, "6087018_m.fxe"},
	{0x0B86EC57EDFEA95FULL, "vxlt_hangedman_n_27l_0_0.vlm"},
	{0x0B87893B267890DAULL, "6125835_m.fxe"},
	{0x0B8797214FCEF535ULL, "6163355_m.fxe"},
	{0x0B87BE88C5D15360ULL, "vxlt_gallowstemplar_38u_0_0.vlm"},
	{0x0B87D9C3FDCDA120ULL, "qun221cr_mob_leader.utc"},
	{0x0B8850162080F28CULL, "w_encounter3_173_0.msh"},
	{0x0B8856162080FD7EULL, "w_encounter3_173_0.mmh"},
	{0x0B88621620811115ULL, "w_encounter3_173_0.mao"},
	{0x0B8875BDF3816A8EULL, "ht_keep_3741.rml"},
	{0x0B89449BDAC7D434ULL, "vxdr_primevaltomb_2gr_0_1.vlm"},
	{0x0B89485D8FF9A948ULL, "vxw_sundermount_34r_0_0.vlm"},
	{0x0B8985A87F2D3317ULL, "771886767.ogg"},
	{0x0B8A8B69163592C1ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\idunna_prostitute_action_knife_to_throat_loop.gad"},
	{0x0B8AC5C598E97795ULL, "dae000ip_ambient_liecudlf_s.utp"},
	{0x0B8B38F2EC9D8760ULL, "mag100_thrask6031998.cl"},
	{0x0B8B3EA408AA82ECULL, "6219277_m.fxe"},
	{0x0B8B7401F1B688CFULL, "dae000_hate_follower_react6103218.cl"},
	{0x0B8B95FDE5C1A747ULL, "character_character_p59.psh.fxo.dx11"},
	{0x0B8BC011954BD455ULL, "vxht_keep_493_0_0.vlm"},
	{0x0B8BC663E4981427ULL, "character_character_p31.psh.fxo"},
	{0x0B8C1B780D02B73CULL, "art\\characters\\creatures\\mubari\\animation\\hv_dmg.evt"},
	{0x0B8C2E3E9B71622BULL, "732782075.ogg"},
	{0x0B8C39F70AC9FAA5ULL, "art\\cinematics\\_crit path\\mage crisis\\rulesflees\\playerrules_extratemplarfemale3_.gad"},
	{0x0B8C7F5FAD4FFF3FULL, "6270314.ogg"},
	{0x0B8CD26AD30AD65EULL, "vxht_mansion_n_59j_0_0.vlm"},
	{0x0B8CFCCFFEC621A5ULL, "vfx_chunkyparticle_p4.psh.fxo.dx11"},
	{0x0B8D3BF632551E8DULL, "vxlt_gallowscourtyard_4qj_0_0.vlm"},
	{0x0B8D6E226E7B69C0ULL, "6028411_m.fxe"},
	{0x0B8DA55A20550D90ULL, "art\\characters\\creatures\\mubari\\animation\\drnk_ptn.gad"},
	{0x0B8DABEEB26BF6B9ULL, "vrc221tr_first_corpses.utt"},
	{0x0B8DB3F7043156DDULL, "798892543.ogg"},
	{0x0B8E0B66DA37E4D9ULL, "art\\characters\\weapons\\axes\\textures\\w_axe_ax101a_0t.dds"},
	{0x0B8E3184C09B15E1ULL, "vxw_sundermount_33v_0_0.vlm"},
	{0x0B8F96CB8AE55912ULL, "vxlt_hangedman_28g_0_0.vlm"},
	{0x0B8FD6A6BA197237ULL, "fen110_anso6094988.cl"},
	{0x0B8FFBC4984CEB44ULL, "conversation_ani_variations.gda"},
	{0x0B8FFD03F7AC6495ULL, "vxca_caverns1_mines_gey_0_0.vlm"},
	{0x0B9031FE1CD1F729ULL, "follower_banter6165365.cl"},
	{0x0B90348511C696C2ULL, "llac_lt_0_lowtown_1_4.dds"},
	{0x0B910A6BE6D7F234ULL, "vxw_bonepitmining_a2_1x0_0_1.vlm"},
	{0x0B913119C41BA36BULL, "vxht_mansion2b_1ip_0_0.vlm"},
	{0x0B918C69163BB36FULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\idunna_prostitute_action_knife_to_throat_loop.ani"},
	{0x0B91DCB7909CB64EULL, "182929414.ogg"},
	{0x0B91E525E9EF5717ULL, "hlac_ht_0_hightown_a2_131071_1x.dds"},
	{0x0B91E766A401158FULL, "717188120.ogg"},
	{0x0B9200BCB3EE002DULL, "98479718.ogg"},
	{0x0B9205A0DB73B9FFULL, "637780716.ogg"},
	{0x0B920CA4E708AB1BULL, "vxlt_0_lowtown_a2_n_gz4_0_0.vlm"},
	{0x0B923A62AC1CDA4AULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\_textures\\prp_eaglebody_n.dds"},
	{0x0B92DBB56B08EFAAULL, "vxw_encounter3_27_0_0.vlm"},
	{0x0B93522D99DA9BC4ULL, "art\\characters\\playercharacter\\dwarfmale\\textures\\dm_glv_lgtb_0n.dds"},
	{0x0B93B1FBCA23936FULL, "lt_0_lowtown_10_0.rml"},
	{0x0B9418FA21676EA0ULL, "art\\characters\\playercharacter\\elfmale\\textures\\em_rob_orsa_0d.dds"},
	{0x0B9478C72DAB1FE3ULL, "gencr_dalish_warrior_2w.utc"},
	{0x0B94CD89B16E01F7ULL, "1017516622.ogg"},
	{0x0B95138E8468E4F4ULL, "lgt220_shp_mag_guard6149889.cl"},
	{0x0B953CCFE7122610ULL, "art\\characters\\creatures\\highdragon\\animation\\talent\\humangrabbed_loop.gad"},
	{0x0B955E34DFF4C1DFULL, "vxlt_hangedman_n_22o_0_0.vlm"},
	{0x0B95A3EBE04CA3D4ULL, "art\\vfx\\combat\\vfx_r_brd_psong_act_c.anb"},
	{0x0B95E1EA3681219DULL, "follower_banter6141680.cl"},
	{0x0B96239E77505C1AULL, "llac_lt_0_lowtown_n_-3_6.dds"},
	{0x0B9648817AE3D77AULL, "llac_lt_backalley_n_0_5.dds"},
	{0x0B96C4C6A86806C6ULL, "vxlt_0_lowtown_ewg_0_0.vlm"},
	{0x0B96FC1066075C98ULL, "vxlt_0_lowtown_f4c_0_0.vlm"},
	{0x0B97BA3A81EFC1C8ULL, "vxw_sundermount_33s_0_0.vlm"},
	{0x0B980F7B82EF30C5ULL, "zz_followers_banter6159276.cl"},
	{0x0B987D263D4E0D4DULL, "art\\vfx\\models\\vfx_branch.mmh"},
	{0x0B988A3AF7E0B8ADULL, "vxht_brothel_n_54e_0_0.vlm"},
	{0x0B988F263D4E2BD3ULL, "art\\vfx\\models\\vfx_branch.msh"},
	{0x0B99F3AD610BAC00ULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_brie_d.dds"},
	{0x0B9A23780D0F0255ULL, "art\\characters\\creatures\\mubari\\animation\\hv_dmg.ani"},
	{0x0B9AFC1F5C2E2451ULL, "vxlt_gallowstemplar_4up_0_0.vlm"},
	{0x0B9B7A79036139E6ULL, "dae011_badpoet_sta6145209.cl"},
	{0x0B9BC8FB733DDC79ULL, "vxlt_0_lowtown_a2_gmp_0_0.vlm"},
	{0x0B9BE7E79C41D77DULL, "gen00fl_merrill6145212.cl"},
	{0x0B9C338EEC4A73FCULL, "445473560.ogg"},
	{0x0B9C56AECDE8D3C8ULL, "6060467_f.fxe"},
	{0x0B9C662A8F9AB217ULL, "648664676.ogg"},
	{0x0B9CB892858FD1BEULL, "6107530_m.fxe"},
	{0x0B9D41CB1DF4A584ULL, "fighting.mao"},
	{0x0B9D4BD8EF323003ULL, "hlac_w_bonepitdestroyed_a3_131074_0y.dds"},
	{0x0B9D98AD96EC8F1DULL, "vxlt_gallowsprison_n_5s0_0_0.vlm"},
	{0x0B9E4982C346CA16ULL, "gen00fl_anders6062178.cl"},
	{0x0B9EC05DAA848222ULL, "685383070.ogg"},
	{0x0B9ED9B7595C3DE8ULL, "lgt110cr_gxa_vgk_cnspr_1.utc"},
	{0x0B9EDBCD2B352879ULL, "6028797_m.fxe"},
	{0x0B9FB74AC4FB3AD3ULL, "ave121cr_brennan.utc"},
	{0x0B9FDE0DF66B7B21ULL, "451010169.ogg"},
	{0x0B9FE1AB72444391ULL, "qun111_ketojan6041040.cl"},
	{0x0B9FE2E733DE86B8ULL, "vxca_caverns2prc_94b_0_0.vlm"},
	{0x0BA07E81D8FEC39BULL, "vxlt_0_lowtown_a2_n_gw7_0_0.vlm"},
	{0x0BA16B90CB0BA3DAULL, "345215941.ogg"},
	{0x0BA195D0749A4CEAULL, "vxlt_undercityserialkiller_18c_0_0.vlm"},
	{0x0BA1B49BE9CB2E5AULL, "mag320cr_templar_1.utc"},
	{0x0BA282EBE056F597ULL, "art\\vfx\\combat\\vfx_r_brd_psong_act_c.mmh"},
	{0x0BA28920FEC7382CULL, "423320165.ogg"},
	{0x0BA28D35F3EA01B7ULL, "6031313_m.fxe"},
	{0x0BA2B35E10EB098FULL, "llac_lt_0_lowtown_docks_a3_-65537_3.dds"},
	{0x0BA2DB29B71ADFE8ULL, "40014436.ogg"},
	{0x0BA3047480687CDBULL, "vxlt_0_lowtown_gpi_0_0.vlm"},
	{0x0BA376872F4DC8CCULL, "573927652.ogg"},
	{0x0BA401BAE7A860C8ULL, "hlac_lt_0_lowtown_n_131070_2z.dds"},
	{0x0BA4986B820E8D14ULL, "6135659_f.fxe"},
	{0x0BA49CF24FB089D4ULL, "6050085_m.fxe"},
	{0x0BA4AA5848A1FEEEULL, "6019211_m.fxe"},
	{0x0BA4B208D712AF4DULL, "w_bonepitdestroyed_a3_180_0.mmh"},
	{0x0BA4B608D712B59EULL, "w_bonepitdestroyed_a3_180_0.mao"},
	{0x0BA4C408D712CDD3ULL, "w_bonepitdestroyed_a3_180_0.msh"},
	{0x0BA536A39A5E1F46ULL, "vxlt_0_lowtown_erh_0_0.vlm"},
	{0x0BA5728817324B63ULL, "6164561_m.fxe"},
	{0x0BA5FB90CAC585CDULL, "hlac_lt_0_lowtown_docks_a3_-65538_6x.dds"},
	{0x0BA5FD0F4BEB0FBCULL, "911200841.ogg"},
	{0x0BA61560D0DAEEC6ULL, "209521803.ogg"},
	{0x0BA6A83608FA385AULL, "fen210_tevinter_ambusher6108302.cl"},
	{0x0BA6B52260CBAB10ULL, "mcr371cr_corpse.ncs"},
	{0x0BA6B99D62A28957ULL, "vxlt_hangedman_w1_0_0.vlm"},
	{0x0BA6D7044C55482BULL, "6205648_m.fxe"},
	{0x0BA6DFFF469A534EULL, "vxht_bartrand_n_67t_0_0.vlm"},
	{0x0BA7A4ED653A9445ULL, "114406979.ogg"},
	{0x0BA81D60CCBA6339ULL, "zze_roguevshighdragon_white.cut"},
	{0x0BA83AAA73780E85ULL, "art\\vfx\\combat\\vfx_w_wns_left_close1_c.mmh"},
	{0x0BA8DE504B1D70F5ULL, "llac_lt_backalley_g_0_7.dds"},
	{0x0BA9D063EE213494ULL, "dae000_orsino6154306.cl"},
	{0x0BA9DA6DF349CE4DULL, "vxw_bonepit_la_0_0.vlm"},
	{0x0BAA1BD17049F712ULL, "700866720.ogg"},
	{0x0BAA29497C9F6D9AULL, "27641974.ogg"},
	{0x0BAA551BA79D0A27ULL, "hlac_dr_0_deeproads_5842_17z.dds"},
	{0x0BAADCCD832A173FULL, "595556839.ogg"},
	{0x0BAB27A753318991ULL, "hlac_lt_0_lowtown_131072_0x.dds"},
	{0x0BAB48788F7A7F56ULL, "m_dae240ar_bone_pit.bnk"},
	{0x0BAB7FABA82D421AULL, "228815538.ogg"},
	{0x0BABACEEB284F67CULL, "vrc221tr_first_corpses.ncs"},
	{0x0BAC26F91106CA22ULL, "vxw_woundedcoast_hw_0_0.vlm"},
	{0x0BAC3B1825883FA2ULL, "630835171.ogg"},
	{0x0BAC7A856ED18A12ULL, "vxlt_hangedman_22a_0_0.vlm"},
	{0x0BAC9A7F3633A8F1ULL, "vxlt_hangedman_n_2xc_0_0.vlm"},
	{0x0BACCB9808AAFD4AULL, "354948399.ogg"},
	{0x0BACD614BFEA2772ULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\animation\\blendtree\\doorsimple.anb"},
	{0x0BAD289EDDBDB46AULL, "art\\vfx\\combat\\vfx_m_pri_wintgrsp_imp_p.impa.gad"},
	{0x0BAD29284A58D6ACULL, "vxlt_warehouse_n_12c_0_0.vlm"},
	{0x0BAD3FCC2EA0A53BULL, "6169500_m.fxe"},
	{0x0BAD3FCECA74DC88ULL, "119827350.ogg"},
	{0x0BAD5E011B173EF8ULL, "dre000_dougal6131131.cl"},
	{0x0BADF30424DE119EULL, "vxlt_darktown_588_0_1.vlm"},
	{0x0BAE14D3A807D46BULL, "vxht_mansion2b_1gr_0_0.vlm"},
	{0x0BAEA9C08E320BEBULL, "6144329_m.fxe"},
	{0x0BAEE35DCD57561AULL, "6075770_m.fxe"},
	{0x0BAF3157609D0DD1ULL, "vxht_0_hightown_cds_0_0.vlm"},
	{0x0BAF3F4C792F4E72ULL, "6176299_m.fxe"},
	{0x0BAFCAAF401F3CADULL, "384741376.ogg"},
	{0x0BAFE2D4C4183D54ULL, "hlac_w_encounter2_196610_0y.dds"},
	{0x0BAFF5BB96E4792FULL, "hlac_lt_0_lowtown_a3_n_c_0_4z.dds"},
	{0x0BAFFC63E2FBB9A6ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\ambient\\compressed\\lean_bar_order_1.ani"},
	{0x0BB00B5A329E2D8EULL, "164664799.ogg"},
	{0x0BB00DFA966D4217ULL, "dre000_hireling16038627.cl"},
	{0x0BB0B7C06FDB7CFEULL, "cd_mrl250_pol.bnk"},
	{0x0BB15E2A356CF9D2ULL, "llac_lt_undercitydungeon_-1_50.dds"},
	{0x0BB1ECBDBE67482EULL, "art\\characters\\playercharacter\\humanmale\\textures\\hm_rob_appb_0s.dds"},
	{0x0BB24DC610C9196DULL, "gen00fl_anders6103735.cl"},
	{0x0BB25FA4D73F6F65ULL, "676346168.ogg"},
	{0x0BB28DAA4B63576AULL, "gen00fl_isabela6096764.cl"},
	{0x0BB28E8A47BCF2A1ULL, "vxdr_primevaltomb_8px_0_1.vlm"},
	{0x0BB3505E60790971ULL, "gen00fl_merrill6152667.cl"},
	{0x0BB35DF0F29E60CBULL, "6077078_m.fxe"},
	{0x0BB378A8A2AB7349ULL, "l_env_wal_hightown_int_calm_p2p.bnk"},
	{0x0BB3D27F7E0DFC91ULL, "and121_karl6149823.cl"},
	{0x0BB400F1531DB996ULL, "6177234_m.fxe"},
	{0x0BB40B9EDDC3A101ULL, "art\\vfx\\combat\\vfx_m_pri_wintgrsp_imp_p.impa.evt"},
	{0x0BB42CF436B3A411ULL, "lt_0_lowtown_docks_n_29_0_ll.mmh"},
	{0x0BB4608A328963EBULL, "6127591_m.fxe"},
	{0x0BB4D42F3C501813ULL, "546927435.ogg"},
	{0x0BB51AE531482168ULL, "806540607.ogg"},
	{0x0BB537AA73829452ULL, "art\\vfx\\combat\\vfx_w_wns_left_close1_c.anb"},
	{0x0BB55F8CE2FD8301ULL, "6069646_m.fxe"},
	{0x0BB58F64EB81FE23ULL, "vxw_encounter2_cr_0_0.vlm"},
	{0x0BB5C3EA04FE43B2ULL, "llac_lt_0_lowtown_docks_a3_n_-65537_3.dds"},
	{0x0BB742F1ABE7960FULL, "pro000_merchant6051810.cl"},
	{0x0BB7BB057682B5CAULL, "6109767_m.fxe"},
	{0x0BB84500FDCF94B4ULL, "hightown_market_guard_sta6169631.cl"},
	{0x0BB88F45FFE37BF0ULL, "hlac_ht_mansion2_818_59x.dds"},
	{0x0BB899AC0FE49182ULL, "dae210_gate_guard6085048.cl"},
	{0x0BB8D11A45C8B746ULL, "qcr000_arishok6060150.cl"},
	{0x0BB94A48728BD7F8ULL, "9302965.ogg"},
	{0x0BB954A10610B588ULL, "vrc321_varric6173851.cl"},
	{0x0BB967F1952CA981ULL, "qcr220st_orsino.stg"},
	{0x0BB9B6AEFD23754BULL, "hlac_lt_0_lowtown_docks_a3_n_-131074_2z.dds"},
	{0x0BB9DB18DCA44472ULL, "follower_banter6172795.cl"},
	{0x0BBA0321F69CC39EULL, "vxlt_0_lowtown_docks_a3_n_d4v_0_0.vlm"},
	{0x0BBA686DE1018D41ULL, "vxlt_undercitydungeon_11r_0_0.vlm"},
	{0x0BBA8037AD30CA2BULL, "vxdr_primevaltomb_1cc_0_0.vlm"},
	{0x0BBA91CF1B67C21AULL, "414128224.ogg"},
	{0x0BBAB69A9E52B404ULL, "gencr_dalish_assassin.utc"},
	{0x0BBAF97B75FAFAB0ULL, "183141752.ogg"},
	{0x0BBB6EBA41055C0EULL, "6143713_f.fxe"},
	{0x0BBB838BB537AC91ULL, "character_character_p147.psh.fxo.dx11"},
	{0x0BBB8F9124E5C9C6ULL, "vxca_cave_1ae_0_0.vlm"},
	{0x0BBBC551BBAB2A7BULL, "vxht_brothel_n_52u_0_0.vlm"},
	{0x0BBC138D698DF16CULL, "vxht_mansion_n_6km_0_0.vlm"},
	{0x0BBC1C620AE47C3AULL, "vxw_woundedcoast_1gc_0_0.vlm"},
	{0x0BBC406AD4DF3D7EULL, "vxht_brothel_n_5ko_0_0.vlm"},
	{0x0BBD64282A7E7BD0ULL, "784818604.ogg"},
	{0x0BBD64836CF92DCCULL, "vxlt_0_lowtown_docks_a3_n_c3j_0_0.vlm"},
	{0x0BBDBC3ADC03A268ULL, "vxlt_gallowscourtyard_n_3vj_0_0.vlm"},
	{0x0BBEAE646DA024CEULL, "hlac_lt_0_lowtown_a3_n_c_65536_6x.dds"},
	{0x0BBF0A78047E6893ULL, "art\\characters\\playercharacter\\qunarimale\\animation\\exploration\\idle\\twitchv2.ani"},
	{0x0BBF8D8B4833F990ULL, "art\\levels\\races\\dwarf\\props\\decor\\prp_screenb.mao"},
	{0x0BBF918B483400A3ULL, "art\\levels\\races\\dwarf\\props\\decor\\prp_screenb.mmh"},
	{0x0BBF9F8B483418FDULL, "art\\levels\\races\\dwarf\\props\\decor\\prp_screenb.msh"},
	{0x0BBFE327A5AA5E91ULL, "pro000_athenril6049469.cl"},
	{0x0BBFED87AD3AECFCULL, "vxlt_0_lowtown_a2_n_c_hm3_0_0.vlm"},
	{0x0BC05107AC28AD51ULL, "mag211_marethari6025326.cl"},
	{0x0BC08CF01C0C4516ULL, "vxht_brothel_n_4dc_0_0.vlm"},
	{0x0BC0C989AF5857D4ULL, "hlac_lt_0_lowtown_docks_-196607_0x.dds"},
	{0x0BC16B61FFF275EAULL, "hlac_ca_caverns1_24424_1x.dds"},
	{0x0BC1CFCDE494794BULL, "6078178_m.fxe"},
	{0x0BC2039EDDCFD150ULL, "art\\vfx\\combat\\vfx_m_pri_wintgrsp_imp_p.impa.ani"},
	{0x0BC21E9401F77F79ULL, "dae000_fenarel6111552.cl"},
	{0x0BC2314A89169A42ULL, "gen00fl_isabela6135643.cl"},
	{0x0BC243016F21F564ULL, "261596254.ogg"},
	{0x0BC28C03A23E8157ULL, "art\\levels\\races\\natural\\kirkcoastal\\props\\prp_kcrocksmall.msh"},
	{0x0BC29B75D95BF86FULL, "mrl150_hunter6071003.cl"},
	{0x0BC29E03A23E9FBAULL, "art\\levels\\races\\natural\\kirkcoastal\\props\\prp_kcrocksmall.mao"},
	{0x0BC2A203A23EA6C9ULL, "art\\levels\\races\\natural\\kirkcoastal\\props\\prp_kcrocksmall.mmh"},
	{0x0BC30E97B7885AA6ULL, "gen00fl_sebastian6180158.cl"},
	{0x0BC3E563E30C3BA0ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\ambient\\compressed\\lean_bar_order_1.gad"},
	{0x0BC40141DD7816A3ULL, "vxw_encounter2_nu_0_0.vlm"},
	{0x0BC4B1584A3F06C7ULL, "43993034.ogg"},
	{0x0BC4D4CC6DF1788FULL, "hlac_ht_keep_3732_12z.dds"},
	{0x0BC511E6FA6AE966ULL, "vxht_bartrand_n_73a_0_0.vlm"},
	{0x0BC53BC873A86C09ULL, "vxw_bonepitmining_a2_1nz_0_0.vlm"},
	{0x0BC5404EAD9906A6ULL, "6086259_m.fxe"},
	{0x0BC62CE26878CE4AULL, "6102667_m.fxe"},
	{0x0BC63A9E72D39430ULL, "zz_ave_debug6174720.cl"},
	{0x0BC643033F29C17CULL, "art\\vfx\\combat\\vfx_w_off_accuracy_loop_c.impa.ani"},
	{0x0BC6447C176448EBULL, "qun230tr_contact_ambient.utt"},
	{0x0BC66BF67BB5F8A0ULL, "and111_lirene6084536.cl"},
	{0x0BC68CA265FA6D1BULL, "fx_fl_smoke.dds"},
	{0x0BC68D60360128BCULL, "192539577.ogg"},
	{0x0BC699CF4BC71976ULL, "vxlt_0_lowtown_docks_a3_d0i_0_0.vlm"},
	{0x0BC6AD2B2B064C73ULL, "970669075.ogg"},
	{0x0BC6B7E8B04EFC04ULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\wkf.gad"},
	{0x0BC7450869EC69F8ULL, "525077066.ogg"},
	{0x0BC745F84198E4CFULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\0rw_basic.phy"},
	{0x0BC7554E26DEA3FBULL, "art\\characters\\creatures\\crowds\\humanmale\\textures\\prp_atlas_jimmy_0n.dds"},
	{0x0BC76C4DE51F0FFBULL, "hlac_w_sundermount_0_5y.dds"},
	{0x0BC76C5896004214ULL, "6179528_m.fxe"},
	{0x0BC7AAC1B1EF5C06ULL, "vxht_brothel_n_5fp_0_0.vlm"},
	{0x0BC7B0AC104DCB99ULL, "llac_ht_keep_3732_7.dds"},
	{0x0BC7E7E0D74D4EE2ULL, "vxlt_warehouse_n_dd_0_0.vlm"},
	{0x0BC805669B074D3FULL, "vxht_keep_n_607_0_0.vlm"},
	{0x0BC8153520015FCAULL, "l_env_wal_gallows_callout.bnk"},
	{0x0BC8DF8FEB553B3CULL, "6154257_m.fxe"},
	{0x0BC90AB2F4F2FB22ULL, "vxw_woundedcoast_bq_0_0.vlm"},
	{0x0BC91F6A9E09AB4EULL, "913571004.ogg"},
	{0x0BCA737CCC093FD2ULL, "dae000_gen_player_turn_in6177857.cl"},
	{0x0BCA76C02D79F05EULL, "vxw_encounter2_camp_rp_0_0.vlm"},
	{0x0BCA856A5B8409ECULL, "vxht_mansion_n_6nb_0_0.vlm"},
	{0x0BCAE9C64084AFA3ULL, "vxlt_undercitydungeon_1j5_0_0.vlm"},
	{0x0BCB7F4350F79034ULL, "zz_one_debug6077409.cl"},
	{0x0BCBAEC17981DF1CULL, "vxlt_0_lowtown_a2_n_c_hr2_0_0.vlm"},
	{0x0BCBE00CE35CED8CULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_daggerb_sa.dds"},
	{0x0BCC27873FD8299AULL, "lt_0_lowtown_docks_a3_n_242_0.mmh"},
	{0x0BCC2B873FD830D1ULL, "lt_0_lowtown_docks_a3_n_242_0.mao"},
	{0x0BCC39873FD848E0ULL, "lt_0_lowtown_docks_a3_n_242_0.msh"},
	{0x0BCC43291BF9A6FFULL, "754096776.ogg"},
	{0x0BCC81A27CAE79F4ULL, "575406623.ogg"},
	{0x0BCD0278048A98CAULL, "art\\characters\\playercharacter\\qunarimale\\animation\\exploration\\idle\\twitchv2.evt"},
	{0x0BCD50033F2FF5BEULL, "art\\vfx\\combat\\vfx_w_off_accuracy_loop_c.impa.gad"},
	{0x0BCD5585C8B9E914ULL, "vxw_bonepitdestroyed_a3_19l_0_0.vlm"},
	{0x0BCDC6E8B05534E2ULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\wkf.ani"},
	{0x0BCE644D94E14FFEULL, "qun110_petrice6042287.cl"},
	{0x0BCEA9DFF85C42EEULL, "ave110tr_donnic_bandits.utt"},
	{0x0BCED434F04B8CD6ULL, "vxlt_0_lowtown_a2_n_c_hua_0_0.vlm"},
	{0x0BCF5F4DD58495EDULL, "924997889.ogg"},
	{0x0BCF845537FDAFB4ULL, "12321380.ogg"},
	{0x0BCFCA253C994F37ULL, "6025338_m.fxe"},
	{0x0BCFDE2E7DC996C5ULL, "one300_zevran6076956.cl"},
	{0x0BD0196F96167FF1ULL, "hlac_w_sundermount_65536_0y.dds"},
	{0x0BD0928B484275C8ULL, "art\\levels\\races\\dwarf\\props\\decor\\prp_screenb.phy"},
	{0x0BD09A804BF4DC9CULL, "art\\characters\\creatures\\ashwraith\\animation\\ras_die_v1.ani"},
	{0x0BD0CFDFF896C4ADULL, "jh.cut"},
	{0x0BD117E031FEC54AULL, "w_bonepitmining_a2_186_0.mao"},
	{0x0BD11BE031FECB99ULL, "w_bonepitmining_a2_186_0.mmh"},
	{0x0BD125E031FEDC87ULL, "w_bonepitmining_a2_186_0.msh"},
	{0x0BD165E8534AABC9ULL, "llac_ht_0_hightown_0_0.dds"},
	{0x0BD1C2F00BF8FA2DULL, "art\\levels\\races\\proxy\\props\\4tools\\inv_capsule2x1_tool.msh"},
	{0x0BD1D0F00BF91260ULL, "art\\levels\\races\\proxy\\props\\4tools\\inv_capsule2x1_tool.mao"},
	{0x0BD1D4F00BF918B3ULL, "art\\levels\\races\\proxy\\props\\4tools\\inv_capsule2x1_tool.mmh"},
	{0x0BD266F50C333C91ULL, "6083072_f.fxe"},
	{0x0BD2731ECF6E2F23ULL, "vxw_encounter2_it_0_0.vlm"},
	{0x0BD2BD31BB73279CULL, "6155979_m.fxe"},
	{0x0BD2EDCBF53D2C10ULL, "668140735.ogg"},
	{0x0BD2FDAC1DD54888ULL, "6191248_f.fxe"},
	{0x0BD343128A955E5BULL, "vxht_chantry_n_30r_0_0.vlm"},
	{0x0BD35E3F1DE4B10BULL, "vxht_0_hightown_a3_cf2_0_0.vlm"},
	{0x0BD35E7A64829510ULL, "6150976_m.fxe"},
	{0x0BD3746A1129675EULL, "vxca_caverns2prc_8bg_0_0.vlm"},
	{0x0BD37E950831723CULL, "dae011_gossips6143025.cl"},
	{0x0BD3D23B1393B8CCULL, "6208647_f.fxe"},
	{0x0BD3E17804907F3DULL, "art\\characters\\playercharacter\\qunarimale\\animation\\exploration\\idle\\twitchv2.gad"},
	{0x0BD423DCCBF7CF56ULL, "vxlt_hangedman_n_20p_0_0.vlm"},
	{0x0BD42B033F35D5DDULL, "art\\vfx\\combat\\vfx_w_off_accuracy_loop_c.impa.evt"},
	{0x0BD4D0B99EF25A34ULL, "hlac_lt_0_lowtown_docks_a3_n_-196610_5y.dds"},
	{0x0BD538AFD806A6D7ULL, "497449884.ogg"},
	{0x0BD6182BE59A7C20ULL, "6028514_m.fxe"},
	{0x0BD61C9EA3E57486ULL, "vxht_brothel_n_5iq_0_0.vlm"},
	{0x0BD6AD76FA3C1C8FULL, "177606026.ogg"},
	{0x0BD6F620134294ECULL, "24349950.ogg"},
	{0x0BD7517C1772D19AULL, "qun230tr_contact_ambient.ncs"},
	{0x0BD7A7804BFB11DEULL, "art\\characters\\creatures\\ashwraith\\animation\\ras_die_v1.gad"},
	{0x0BD7C51384C51E62ULL, "hlac_ht_0_hightown_a3_65533_2z.dds"},
	{0x0BD7E6DED50BBC6AULL, "6054511_m.fxe"},
	{0x0BD81E8165DE0AEFULL, "6114856_m.fxe"},
	{0x0BD8365BB13A6C72ULL, "6110062_m.fxe"},
	{0x0BD83A148E14F34FULL, "vxlt_0_lowtown_a2_n_dej_0_0.vlm"},
	{0x0BD86C5A905590D9ULL, "vxlt_undercitydungeon_1ir_0_0.vlm"},
	{0x0BD8774595588B8DULL, "follower_banter6156239.cl"},
	{0x0BD893F264375C12ULL, "hlac_dr_0_deeproads_5845_13x.dds"},
	{0x0BD8BFF46A794936ULL, "art\\levels\\races\\natural\\default\\terrainmaterials\\_textures\\def_cobble_n.dds"},
	{0x0BD8E89D1F7008DEULL, "vxw_encounter2_camp_mq_0_0.vlm"},
	{0x0BD8F84C514BD0D0ULL, "651977225.ogg"},
	{0x0BD90A8EA8B3E6FEULL, "llac_ht_keep_n_3731_2.dds"},
	{0x0BD96A6A7DEA1DEBULL, "vxht_mansion2b_1pw_0_0.vlm"},
	{0x0BDA0F5C947FF55CULL, "vxca_caverns1_mines_huk_0_0.vlm"},
	{0x0BDA47E20E6AACD0ULL, "784278604.ogg"},
	{0x0BDA99BBBB8B1922ULL, "vxw_bonepit_1sh_0_0.vlm"},
	{0x0BDB51BA118E193BULL, "abi_m_glyph_repulsion.bnk"},
	{0x0BDB5628C3293DE3ULL, "758374841.ogg"},
	{0x0BDBA82B834E24F6ULL, "zz_dog_debug6131978.cl"},
	{0x0BDC41BC07300446ULL, "art\\characters\\playercharacter\\humanfemale\\hlh_wic.phy"},
	{0x0BDC6CF54B57A14AULL, "mag110_arianni6016827.cl"},
	{0x0BDD4611E241A556ULL, "vxlt_0_lowtown_a2_n_c_hpb_0_0.vlm"},
	{0x0BDE097463F5ED75ULL, "llac_ca_caverns2_5802_24.dds"},
	{0x0BDE12477BCBCE0AULL, "hlac_w_sundermount_-65536_0y.dds"},
	{0x0BDE82804C00F1FDULL, "art\\characters\\creatures\\ashwraith\\animation\\ras_die_v1.evt"},
	{0x0BDEC374ABB8CA47ULL, "291488537.ogg"},
	{0x0BDEF4F25CC5F063ULL, "mag360_thrask6046979.cl"},
	{0x0BE07B4BA15A6FC2ULL, "479717577.ogg"},
	{0x0BE0CC2D7D50C4C6ULL, "art\\vfx\\combat\\vfx_d_teleport_start_p.anb"},
	{0x0BE0F0763E3504E0ULL, "lt_0_lowtown_n_21_0_ll.mmh"},
	{0x0BE13BBECEBDFFA7ULL, "t1_mue_pu3.tnt"},
	{0x0BE178EAE4B39C4EULL, "art\\characters\\playercharacter\\elfmale\\animation\\trooperassasin\\disengage.ani"},
	{0x0BE1DCA2C2D8A915ULL, "175807580.ogg"},
	{0x0BE25865470019BCULL, "848630544.ogg"},
	{0x0BE2C19B58EB9E45ULL, "vxca_caverns2_62q_0_2.vlm"},
	{0x0BE2CA75D3C7F50AULL, "gen00fl_merrill6142083.cl"},
	{0x0BE2E9C9B2BD1403ULL, "art\\levels\\races\\ferelden\\props\\armory\\prp_sworda.msh"},
	{0x0BE2F7C9B2BD2C5DULL, "art\\levels\\races\\ferelden\\props\\armory\\prp_sworda.mmh"},
	{0x0BE2FBC9B2BD336EULL, "art\\levels\\races\\ferelden\\props\\armory\\prp_sworda.mao"},
	{0x0BE368506C2590E4ULL, "vxlt_gamlenshouse_n_b6_0_0.vlm"},
	{0x0BE3EBA5F469B985ULL, "art\\characters\\weapons\\maces\\textures\\w_mce_mc107a_0n.dds"},
	{0x0BE41D8F2BE9DD28ULL, "117785788.ogg"},
	{0x0BE431604A5884CCULL, "155335925.ogg"},
	{0x0BE4501279EC7314ULL, "6177728_m.fxe"},
	{0x0BE59CDFF86F0B17ULL, "ave110tr_donnic_bandits.ncs"},
	{0x0BE5A3ECDCCA821FULL, "hlac_lt_gallowstemplar_n_2394_2x.dds"},
	{0x0BE5BE706E170F6DULL, "mag100_karras6052953.cl"},
	{0x0BE5EE6CD8DF2C22ULL, "vxw_woundedcoast_ts_0_0.vlm"},
	{0x0BE5FBDE1CABB466ULL, "gen00flst_beth_carv_will.stg"},
	{0x0BE668F6ECD0A4FBULL, "vxht_0_hightown_a2_c6o_0_0.vlm"},
	{0x0BE68C75D7FA4855ULL, "208571404.ogg"},
	{0x0BE6905E57D4236FULL, "6113956_m.fxe"},
	{0x0BE6A0C3CACFC452ULL, "683282465.ogg"},
	{0x0BE6CB316E0213BFULL, "vxht_mansion_n_6kb_0_0.vlm"},
	{0x0BE6DE37824BA959ULL, "vxlt_undercitydungeon_1lq_0_0.vlm"},
	{0x0BE6FE1598BD2F92ULL, "317845451.ogg"},
	{0x0BE75A7A1166215EULL, "vxw_encounter2_camp_hr_0_0.vlm"},
	{0x0BE79FF3C30F2896ULL, "6054865_m.fxe"},
	{0x0BE80F21C91C3EF7ULL, "514691210.ogg"},
	{0x0BE84604D1B28736ULL, "vxca_caverns1_mines_jgi_0_0.vlm"},
	{0x0BE87705E3CA161DULL, "zz_qun_debug6028891.cl"},
	{0x0BE87E644E876FF5ULL, "t3_wdg_gw1.tnt"},
	{0x0BE8D35549AED866ULL, "vxlt_undercitydungeon_xw_0_0.vlm"},
	{0x0BE8F9A301ECBDDFULL, "zz_leaphit.cut"},
	{0x0BE92544A9372EA1ULL, "dae210_gate_guard6085122.cl"},
	{0x0BE978C2B2F5E2ACULL, "6119096_m.fxe"},
	{0x0BE990D5B8CBD8D4ULL, "llac_lt_backalley_g_65537_8.dds"},
	{0x0BE9DF670C6C0A6CULL, "dae000_bartrand6064061.cl"},
	{0x0BEA4101EC796802ULL, "gen00fl_sebastian6189023.cl"},
	{0x0BEB32338782CAC6ULL, "one100it_fang.uti"},
	{0x0BEBA2891783FB14ULL, "vxca_cavern1_haunted_ikh_0_0.vlm"},
	{0x0BEBAA2EFC4097C9ULL, "vxlt_0_lowtown_docks_a3_n_cxf_0_0.vlm"},
	{0x0BEC82985F3C3BA5ULL, "mag370_meredith6046906.cl"},
	{0x0BECCDC15019BB51ULL, "lt_0_lowtown_docks_n_222_0.msh"},
	{0x0BECDBC15019D36CULL, "lt_0_lowtown_docks_n_222_0.mao"},
	{0x0BECE7C15019E70FULL, "lt_0_lowtown_docks_n_222_0.mmh"},
	{0x0BED077AD79CC44DULL, "195736837.ogg"},
	{0x0BED0C28FCFDC373ULL, "qun231_party_comment6077130.cl"},
	{0x0BED181F2C0E8084ULL, "art\\vfx\\combat\\vfx_w_gn1_wcry_act_c.mmh"},
	{0x0BED353F26C9DE61ULL, "1072076198.ogg"},
	{0x0BED4FB216085B6EULL, "art\\vfx\\combat\\vfx_w_2hd_atk_cl_c.impa.evt"},
	{0x0BED5AC7073CE814ULL, "6067115_m.fxe"},
	{0x0BED678B6334034FULL, "ctplt_rock.mao"},
	{0x0BED797DD6AD1400ULL, "6134941_m.fxe"},
	{0x0BEDCF2D7D5B5471ULL, "art\\vfx\\combat\\vfx_d_teleport_start_p.mmh"},
	{0x0BEDE7259910282EULL, "vxht_keep_n_6i5_0_0.vlm"},
	{0x0BEDF48E2BC3CB74ULL, "vxlt_undercityencounter_1c1_0_1.vlm"},
	{0x0BEE1E06BB182D40ULL, "vxht_keep_61l_0_0.vlm"},
	{0x0BEF5BF609CBF6F9ULL, "one280_iwan6096752.cl"},
	{0x0BEF70EAE4BFCB9FULL, "art\\characters\\playercharacter\\elfmale\\animation\\trooperassasin\\disengage.evt"},
	{0x0BEFA005E90B1D0CULL, "llac_lt_gallowscourtyard_65534_1.dds"},
	{0x0BF00256EAFB55BCULL, "6084387_m.fxe"},
	{0x0BF03BA28BB8E2E8ULL, "art\\vfx\\combat\\vfx_v_fir_cast_aoe_c.anb"},
	{0x0BF0746DBDBDA180ULL, "197504679.ogg"},
	{0x0BF0ACA96A9A220CULL, "gen00fl_aveline6128413.cl"},
	{0x0BF0FE9C5CD352DBULL, "vxca_cavern1_haunted_j5f_0_0.vlm"},
	{0x0BF12F86FFDECD71ULL, "6133668_m.fxe"},
	{0x0BF189C07012C504ULL, "cd_mrl250_pol.wbd"},
	{0x0BF1A056FDEF7ECEULL, "dae000_arishok6067708.cl"},
	{0x0BF1A846D21AD047ULL, "hlac_lt_0_lowtown_docks_n_-131074_1x.dds"},
	{0x0BF1BDC882B9766CULL, "gen_im_trs_qun_hvy_hlm_r0.uti"},
	{0x0BF1E658C857376EULL, "hlac_lt_0_lowtown_a2_n_65537_2z.dds"},
	{0x0BF20E36577A3467ULL, "vxht_0_hightown_a3_c39_0_0.vlm"},
	{0x0BF27BE8B800BE88ULL, "6067137_m.fxe"},
	{0x0BF33D77AD2F9862ULL, "vxlt_warehouse_n_ug_0_0.vlm"},
	{0x0BF346B2160CB719ULL, "art\\vfx\\combat\\vfx_w_2hd_atk_cl_c.impa.gad"},
	{0x0BF3953DA0743891ULL, "gen_im_trs_dwv_junk3_04.uti"},
	{0x0BF3E186C9D17146ULL, "dae000_bodhan6129126.cl"},
	{0x0BF40CB6DBCBDC9DULL, "110144642.ogg"},
	{0x0BF4CBE430AC8FC2ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\sitting_drinking_loop.gad"},
	{0x0BF5106616018296ULL, "14498824.ogg"},
	{0x0BF53D0E5FF82C3FULL, "vxht_mansion_n_6nm_0_0.vlm"},
	{0x0BF550147441C1D9ULL, "vxlt_undercitydungeon_1wp_0_0.vlm"},
	{0x0BF57D2AA71DD486ULL, "lgt110_gxa_vgk_silver_order6117025.cl"},
	{0x0BF58BCC93FC3860ULL, "vxht_0_hightown_a88_0_0.vlm"},
	{0x0BF58CC01683894CULL, "6127325_m.fxe"},
	{0x0BF5E9F80BE83BE9ULL, "art\\vfx\\materials\\chunky\\vfx_c_falloff_invert_blend.mao"},
	{0x0BF633AA0DC8F20EULL, "w_blightlands_75_0.phy"},
	{0x0BF651EAE4C5B588ULL, "art\\characters\\playercharacter\\elfmale\\animation\\trooperassasin\\disengage.gad"},
	{0x0BF67EB7306D42A7ULL, "6103631_f.fxe"},
	{0x0BF7DD032B8A9D84ULL, "206073554.ogg"},
	{0x0BF86A5FA65B39EDULL, "7471020.ogg"},
	{0x0BF8FDDA2C9BA7DAULL, "vxlt_gallowstemplar_f_43j_0_0.vlm"},
	{0x0BF9F82DF4D85DAEULL, "vxlt_0_lowtown_a2_en0_0_0.vlm"},
	{0x0BFA11E1C4535F65ULL, "259909773.ogg"},
	{0x0BFA20731D7B6308ULL, "6041538_f.fxe"},
	{0x0BFA265BDD55AC1EULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\threaten_reaction.evt"},
	{0x0BFA56A8A2E73B9BULL, "l_env_wal_hightown_int_calm_p2p.wbd"},
	{0x0BFA57B21612F367ULL, "art\\vfx\\combat\\vfx_w_2hd_atk_cl_c.impa.ani"},
	{0x0BFA8C8B1A423705ULL, "6135770_m.fxe"},
	{0x0BFACEF5FA697EF8ULL, "232094725.ogg"},
	{0x0BFB2697B60C653AULL, "gen_im_trs_krk_hvy_glv_r0.uti"},
	{0x0BFBDB95EA2A622CULL, "vxht_0_hightown_a2_9zb_0_0.vlm"},
	{0x0BFBEEE430B2E9B9ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\sitting_drinking_loop.evt"},
	{0x0BFC998229A3E37CULL, "hlac_lt_darktown_-1_68z.dds"},
	{0x0BFCE965DAA79102ULL, "6036951_m.fxe"},
	{0x0BFCF31357D3414FULL, "gen00fl_aveline6120863.cl"},
	{0x0BFD49AE90CE912AULL, "hlac_lt_undercityserialkiller_-1_41y.dds"},
	{0x0BFD57EA04CC16B4ULL, "gen_im_trs_rdr_hvy_hlm_r0.uti"},
	{0x0BFE585DF94E35D5ULL, "6077090_m.fxe"},
	{0x0BFF01E1FD506334ULL, "vxht_mansion2_3ei_0_0.vlm"},
	{0x0C001E52C9073DA5ULL, "706737645.ogg"},
	{0x0C0036B01D82E130ULL, "vxht_0_hightown_a3_c54_0_0.vlm"},
	{0x0C003D5BDD5A3F49ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\threaten_reaction.gad"},
	{0x0C00801349704CE7ULL, "vxht_0_hightown_a3_c48_0_0.vlm"},
	{0x0C00FE99DE3AC662ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\vista_closebeamb.msh"},
	{0x0C010099DE3AC918ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\vista_closebeamb.mmh"},
	{0x0C010C99DE3ADDBBULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\vista_closebeamb.mao"},
	{0x0C01614C5B496A58ULL, "307117832.ogg"},
	{0x0C01C04A0C7147D2ULL, "vxca_caverns1_mines_imx_0_0.vlm"},
	{0x0C01E239C2DFC51CULL, "6122146_m.fxe"},
	{0x0C01E8CADBA6848FULL, "vxht_0_hightown_a3_b8e_0_0.vlm"},
	{0x0C027E9634F40604ULL, "zz_arshcockforestcu_v2.cut"},
	{0x0C028E73C783D9AEULL, "166053379.ogg"},
	{0x0C02B3915900E807ULL, "character_character_p167.psh.fxo.dx11"},
	{0x0C0306BE6E042274ULL, "follower_banter6149175.cl"},
	{0x0C032C466471DF08ULL, "art\\characters\\weapons\\mauls\\textures\\w_mal_ml103a_0n.dds"},
	{0x0C034CB0D0BCD5FBULL, "vxht_0_hightown_a2_c8a_0_0.vlm"},
	{0x0C036D293D6FCE86ULL, "116951836.ogg"},
	{0x0C039D91CE2725E5ULL, "mag110_vincento6017904.cl"},
	{0x0C03DBA69999ABE1ULL, "gen_im_cns_tom_rog_assassin.uti"},
	{0x0C040FCEF82C813DULL, "art\\characters\\weapons\\dagger\\w_dgr_103a.mao"},
	{0x0C0452E6F2A5D434ULL, "llac_ca_caverns1_mines_24421_5.dds"},
	{0x0C04C00BCB6780EEULL, "dae000ip_crowd_ff6_talk18.utp"},
	{0x0C04C6E5985EE830ULL, "572362461.ogg"},
	{0x0C04F3399356C394ULL, "art\\characters\\creatures\\spider\\animation\\cwkb.ani"},
	{0x0C054D5C2ABA965AULL, "vxlt_gallowstemplar_n_4tw_0_0.vlm"},
	{0x0C05F1E1BB6ADF54ULL, "vxlt_0_lowtown_a2_n_eje_0_0.vlm"},
	{0x0C06A722751405F5ULL, "6061769_m.fxe"},
	{0x0C070E5BDD600D97ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\threaten_reaction.ani"},
	{0x0C071450C642D169ULL, "6164772_m.fxe"},
	{0x0C072A7F38C717AEULL, "vxdr_dungeon_20z_0_0.vlm"},
	{0x0C0826D86709677CULL, "vxlt_0_lowtown_a2_n_c_ht4_0_0.vlm"},
	{0x0C088CC5E060B9BEULL, "6199281_m.fxe"},
	{0x0C08F6E430BD81A8ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\sitting_drinking_loop.ani"},
	{0x0C0981F9BD61CC3DULL, "6123395_m.fxe"},
	{0x0C09BF4D47782FE0ULL, "vxlt_gallowstemplar_3mz_0_0.vlm"},
	{0x0C0A3FB6BA8CB626ULL, "t1_skn_006.tnt"},
	{0x0C0A574A15C5B771ULL, "6064519_m.fxe"},
	{0x0C0A79B8AC9EA829ULL, "vxlt_undercityserialkiller_1a7_0_0.vlm"},
	{0x0C0B5FFA91D76C0EULL, "hlac_w_bonepit_458754_0y.dds"},
	{0x0C0BBC729D6B799DULL, "vxca_caverns1_j59_0_0.vlm"},
	{0x0C0BCD2769460EC2ULL, "571770070.ogg"},
	{0x0C0BD85179FA1F57ULL, "6067758_m.fxe"},
	{0x0C0BEA42306182F0ULL, "vxw_bonepitmining_a2_ow_0_0.vlm"},
	{0x0C0BF039935CDDF6ULL, "art\\characters\\creatures\\spider\\animation\\cwkb.gad"},
	{0x0C0C2FAD1F42E69EULL, "848808727.ogg"},
	{0x0C0C3CAB53F43CA6ULL, "llac_lt_0_lowtown_a2_n_c_1_7.dds"},
	{0x0C0CAC2579371547ULL, "gen_im_npc_arm_hum_cht_slva.uti"},
	{0x0C0D5B3CD3A68453ULL, "6016589_m.fxe"},
	{0x0C0D828A2547DCBFULL, "gen00fl_sebastian6174334.cl"},
	{0x0C0E239FCAE52B67ULL, "458411958.ogg"},
	{0x0C0E55DA79E9DAD1ULL, "vxht_0_hightown_n_c3p_0_1.vlm"},
	{0x0C0E83614022B331ULL, "486835061.ogg"},
	{0x0C0EB4832C6A734CULL, "6072654_m.fxe"},
	{0x0C0EDB87F8BA2D10ULL, "gen00fl_sebastian6180840.cl"},
	{0x0C0F36177E263992ULL, "fer_ter_grass_001_diff.dds"},
	{0x0C0F69F6547E06F2ULL, "524344053.ogg"},
	{0x0C0F74EB6D031880ULL, "622151340.ogg"},
	{0x0C0FE9F8C78F2B9BULL, "w_woundedcoast_220_0.mmh"},
	{0x0C0FEA958AC793A9ULL, "art\\characters\\weapons\\dagger\\w_dgr_002a.phy"},
	{0x0C0FF5F8C78F4038ULL, "w_woundedcoast_220_0.mao"},
	{0x0C1007F8C78F5E85ULL, "w_woundedcoast_220_0.msh"},
	{0x0C103226FE676052ULL, "vxca_caverns1_mines_ijy_0_0.vlm"},
	{0x0C109C75ACB507AEULL, "mcr371cr_harvester.ncs"},
	{0x0C10B0E623232ABFULL, "676169642.ogg"},
	{0x0C10EBEDFAD1FF10ULL, "549455327.ogg"},
	{0x0C115BC16FE3D616ULL, "lt_0_lowtown_n_53_0_lc.mmh"},
	{0x0C1226D2FBC4D0CEULL, "6088226_m.fxe"},
	{0x0C1228512DEDB0BDULL, "fen110_fenris6118039.cl"},
	{0x0C1234D76DEF4511ULL, "gen00fl_anders6119408.cl"},
	{0x0C12450F60C4F294ULL, "vxca_caverns2_5aa_0_1.vlm"},
	{0x0C12C0EB2DBBD8EFULL, "664819166.ogg"},
	{0x0C12D1366E693754ULL, "art\\characters\\playercharacter\\humanfemale\\hlf_tmpb.phy"},
	{0x0C130B3993632A55ULL, "art\\characters\\creatures\\spider\\animation\\cwkb.evt"},
	{0x0C133B4B939BD524ULL, "6207814_m.fxe"},
	{0x0C133DD0F402D28CULL, "vxlt_0_lowtown_a2_n_c_fj4_0_0.vlm"},
	{0x0C137B19175E007BULL, "6175093_m.fxe"},
	{0x0C139B9BA794D0B6ULL, "vxca_caverns1_mines_jtl_0_0.vlm"},
	{0x0C13A80B06B118B5ULL, "dre000_vampire6034334.cl"},
	{0x0C140A2765CAB0D3ULL, "255326345.ogg"},
	{0x0C144E61FE57B951ULL, "vxht_0_hightown_c9t_0_0.vlm"},
	{0x0C1468AAAEB6A22AULL, "vxlt_0_lowtown_ewk_0_0.vlm"},
	{0x0C146E30CAE8A8B8ULL, "w_bonepitdestroyed_a3_27_0.rml"},
	{0x0C149FFE56EBF720ULL, "6090345_f.fxe"},
	{0x0C14B09A351D364FULL, "6093627_m.fxe"},
	{0x0C14D2E274A1F9EFULL, "6072569_m.fxe"},
	{0x0C151B1F2C2FD87BULL, "art\\vfx\\combat\\vfx_w_gn1_wcry_act_c.anb"},
	{0x0C159B752F90A163ULL, "vxlt_darktown_67v_0_0.vlm"},
	{0x0C160DD2650CE779ULL, "vfx_vfx_p8.vsh.fxo.dx11.assembly"},
	{0x0C169180F7773F85ULL, "vfx_chunkyparticle_p4.psh.fxo.dx11.assembly"},
	{0x0C16FA4786A6F01BULL, "vxlt_0_lowtown_a2_n_bxg_0_0.vlm"},
	{0x0C17F3D6AF57E4BDULL, "6128295_m.fxe"},
	{0x0C18BDA1321E42B1ULL, "vxlt_0_lowtown_a2_gvi_0_0.vlm"},
	{0x0C1932B1F6C8E841ULL, "sky_dr_0_deeproads.mao"},
	{0x0C193AA28BDBE583ULL, "art\\vfx\\combat\\vfx_v_fir_cast_aoe_c.mmh"},
	{0x0C19A6A968FA9AA2ULL, "gen00fl_sebastian6195848.cl"},
	{0x0C19C03F6CD98C48ULL, "head_flip_rev.dds"},
	{0x0C1A0A2574C61A97ULL, "art\\vfx\\creatures\\vfx_c_hd_fswipe_c.impa.gad"},
	{0x0C1A111E7503D445ULL, "486172654.ogg"},
	{0x0C1A5363D756EDCBULL, "6108098_f.fxe"},
	{0x0C1A5C1F22579B70ULL, "vxw_bonepitmining_a2_jt_0_0.vlm"},
	{0x0C1A9F0D30D30185ULL, "6118771_m.fxe"},
	{0x0C1AAC6617739885ULL, "hlac_ht_keep_3732_19y.dds"},
	{0x0C1BF64F70869312ULL, "vxlt_foundry_n_12s_0_0.vlm"},
	{0x0C1BF67A1060D3E0ULL, "qun160_vashothlead6053911.cl"},
	{0x0C1C5078500C0F28ULL, "follower_banter6156254.cl"},
	{0x0C1C6120404A39FEULL, "gen00fl_aveline6110489.cl"},
	{0x0C1CC7B76BDFF351ULL, "vxht_0_hightown_n_c8q_0_1.vlm"},
	{0x0C1D98342CB235CEULL, "art\\vfx\\combat\\vfx_itm_myt_favor_aoe_p.mmh"},
	{0x0C1D9B350B110E2AULL, "6155595_m.fxe"},
	{0x0C1DAE366E730351ULL, "art\\characters\\playercharacter\\humanfemale\\hlf_tmpb.msh"},
	{0x0C1DB8C6DD435330ULL, "6134848_m.fxe"},
	{0x0C1DC8366E732F0FULL, "art\\characters\\playercharacter\\humanfemale\\hlf_tmpb.mmh"},
	{0x0C1E131EE69D1E49ULL, "371708524.ogg"},
	{0x0C1E464DF59BD1A4ULL, "art\\vfx\\environment\\level_fx\\lightbeams\\vfx_lightstreaks.mmh"},
	{0x0C1EA403F05D78D2ULL, "vxca_caverns1_mines_igz_0_0.vlm"},
	{0x0C1F04596A749DA4ULL, "6092612_m.fxe"},
	{0x0C1F7A1B407BF725ULL, "6118973_m.fxe"},
	{0x0C2038ABC83FD4D3ULL, "vxht_mansion2b_31n_0_0.vlm"},
	{0x0C20694C62991FBFULL, "hlac_ht_mansion2b_818_33y.dds"},
	{0x0C2098AFEDBAE94EULL, "6083326_m.fxe"},
	{0x0C211F2574CC5D24ULL, "art\\vfx\\creatures\\vfx_c_hd_fswipe_c.impa.evt"},
	{0x0C2158F97F1472A8ULL, "mcr371cr_merrill_cut.utc"},
	{0x0C2172E1828E285EULL, "m_isa121ar_chantry.bnk"},
	{0x0C2194568F930AF6ULL, "vxlt_undercityencounter_1dw_0_0.vlm"},
	{0x0C219F6AE6E939BAULL, "docks_male_elven_worker_sta6175792.cl"},
	{0x0C21C2D4326B74EFULL, "hlac_lt_0_lowtown_docks_a3_131071_6y.dds"},
	{0x0C223FBA11C9FD39ULL, "abi_m_glyph_repulsion.wbd"},
	{0x0C22835C4EBA390AULL, "264106326.ogg"},
	{0x0C22D30C755993A2ULL, "vxw_bonepit_1hu_0_0.vlm"},
	{0x0C2457C74229C23BULL, "6169711_m.fxe"},
	{0x0C246026D5AF1225ULL, "204312777.ogg"},
	{0x0C247D819409EF4BULL, "6152582_m.fxe"},
	{0x0C248994DF335F86ULL, "one100dg_lia6013849.cl"},
	{0x0C251C1469F27D46ULL, "6198708_f.fxe"},
	{0x0C265940A67A5A23ULL, "vxht_brothel_n_6eg_0_0.vlm"},
	{0x0C272238EBB64FF5ULL, "6088672_m.fxe"},
	{0x0C272F7E24145B31ULL, "vxlt_0_lowtown_a2_gmh_0_0.vlm"},
	{0x0C27309193D1A6E7ULL, "hlac_dr_primevaltomb_659_15y.dds"},
	{0x0C275E9CD609A7E3ULL, "342187277.ogg"},
	{0x0C27831E8141C236ULL, "6177861_m.fxe"},
	{0x0C27A6C0ECF06B98ULL, "vxlt_0_lowtown_n_gkx_0_0.vlm"},
	{0x0C27BFF8B4766682ULL, "ht_0_hightown_a2_ll.mmh"},
	{0x0C27ECE415A3A267ULL, "mag311_emile_de_launcet6043701.cl"},
	{0x0C28130DE10081CFULL, "hlac_dr_0_deeproads_5840_2z.dds"},
	{0x0C28873A7A418D74ULL, "vxca_caverns2_8b8_0_0.vlm"},
	{0x0C2890143FFD9FFFULL, "hlac_ht_0_hightown_n_-65534_0y.dds"},
	{0x0C2910EA22C91A05ULL, "6117671_m.fxe"},
	{0x0C2971A8A576F6A3ULL, "vxw_blightlands_7q_0_0.vlm"},
	{0x0C298D8DF8AECF64ULL, "6057325_m.fxe"},
	{0x0C29B96189947352ULL, "vxht_mansion2b_31k_0_0.vlm"},
	{0x0C29CF8B478FAAF3ULL, "gen00fl_fenris6127096.cl"},
	{0x0C2A776354A356E7ULL, "6041573_f.fxe"},
	{0x0C2A7ECABFB464AAULL, "art\\vfx\\combat\\vfx_m_sta_rng_proj_fire_p.anb"},
	{0x0C2AB7342CBCF4CDULL, "art\\vfx\\combat\\vfx_itm_myt_favor_aoe_p.anb"},
	{0x0C2AE56527DF9354ULL, "l_dae421ar_player_home_nt.bnk"},
	{0x0C2AF303276AED17ULL, "art\\characters\\creatures\\gibgolem\\animation\\body\\gibsgolem_attack01.gad"},
	{0x0C2B455E5808EB3CULL, "6106658_m.fxe"},
	{0x0C2B505A2A0A81CDULL, "gen00fl_varric6118759.cl"},
	{0x0C2B5B7D6136D74CULL, "gen00fl_isabela6135595.cl"},
	{0x0C2C6C384BD8018AULL, "6135849_m.fxe"},
	{0x0C2C82DD6C209DE9ULL, "vxlt_gallowstemplar_f_5yb_0_0.vlm"},
	{0x0C2E372574D7101DULL, "art\\vfx\\creatures\\vfx_c_hd_fswipe_c.impa.ani"},
	{0x0C2EF8720D64386FULL, "vxht_keep_n_6fa_0_0.vlm"},
	{0x0C2F180532FC77F1ULL, "gen00fl_aveline6114295.cl"},
	{0x0C2F204A0F56DAC4ULL, "6185387_m.fxe"},
	{0x0C2FBE1C72ACA1D3ULL, "vxlt_gallowstemplar_6sn_0_0.vlm"},
	{0x0C2FCF526490017EULL, "vxht_brothel_n_53g_0_0.vlm"},
	{0x0C31183A6874926EULL, "gen_im_trs_slv_sta_ele_r1.uti"},
	{0x0C31766064C3AADAULL, "548440124.ogg"},
	{0x0C32080327712FA4ULL, "art\\characters\\creatures\\gibgolem\\animation\\body\\gibsgolem_attack01.evt"},
	{0x0C320871E83741EAULL, "vxlt_undercityserialkiller_1ju_0_0.vlm"},
	{0x0C32585F51AED2E8ULL, "vxlt_gallowstemplar_653_0_0.vlm"},
	{0x0C326C30CB01A295ULL, "w_bonepitdestroyed_a3_27_0.msh"},
	{0x0C326E30CB01A60BULL, "w_bonepitdestroyed_a3_27_0.mmh"},
	{0x0C327A30CB01BAA8ULL, "w_bonepitdestroyed_a3_27_0.mao"},
	{0x0C32D6016865724DULL, "lt_0_lowtown_docks_n_158_0.msh"},
	{0x0C32E40168658980ULL, "lt_0_lowtown_docks_n_158_0.mao"},
	{0x0C32E801686590D3ULL, "lt_0_lowtown_docks_n_158_0.mmh"},
	{0x0C32F12C1983E295ULL, "fx_circle_distor.mao"},
	{0x0C331200ECE3E03FULL, "dae000_orsino6081382.cl"},
	{0x0C331CE032519786ULL, "w_bonepitmining_a2_186_0.phy"},
	{0x0C3371D8AAB6E48BULL, "141445750.ogg"},
	{0x0C33B15E3927C82BULL, "303562853.ogg"},
	{0x0C33C9299638B892ULL, "docks_male_human_worker_sta6175960.cl"},
	{0x0C340B8DD5B702D1ULL, "llac_ht_0_hightown_a3_2_1.dds"},
	{0x0C344F8414BF5B91ULL, "art\\characters\\playercharacter\\humanfemale\\hlm_masd.mao"},
	{0x0C34772E882E45C7ULL, "304234957.ogg"},
	{0x0C34A2AF1C4E9C97ULL, "6116003_m.fxe"},
	{0x0C359415DDAC6875ULL, "6083772_m.fxe"},
	{0x0C3604F1A2537451ULL, "llac_lt_gallowsprison_n_-1_15.dds"},
	{0x0C362095CB4D81B9ULL, "gen00fl_varric6105476.cl"},
	{0x0C3624BD1C7DE33AULL, "478888072.ogg"},
	{0x0C36351B03215B0CULL, "881053287.ogg"},
	{0x0C36DD91DB8ADD54ULL, "165834481.ogg"},
	{0x0C3732AA0E0087D1ULL, "w_blightlands_75_0.mmh"},
	{0x0C373EAA0E009C72ULL, "w_blightlands_75_0.mao"},
	{0x0C374CAA0E00B38FULL, "w_blightlands_75_0.msh"},
	{0x0C3782C714BF3285ULL, "6112571_m.fxe"},
	{0x0C3799FADB3736DFULL, "vxw_woundedcoast_1d3_0_0.vlm"},
	{0x0C37B63097CD0E3BULL, "art\\characters\\playercharacter\\humanmale\\animation\\partypicker\\magestaffselectedenter.ani"},
	{0x0C37E385976D0F23ULL, "vxw_blightlands_8p_0_0.vlm"},
	{0x0C38F1D53E1527E2ULL, "rdr111dg_martin6054582.cl"},
	{0x0C3933752B34652AULL, "vxca_cave_1ai_0_0.vlm"},
	{0x0C393ED234014052ULL, "vxlt_gallowstemplar_6sk_0_0.vlm"},
	{0x0C3957EE5282E3A8ULL, "693260994.ogg"},
	{0x0C39F3325768743BULL, "3300113.ogg"},
	{0x0C3A8B697B206688ULL, "6055060_m.fxe"},
	{0x0C3B80C32D8B68D4ULL, "vxca_caverns1_mines_i9r_0_0.vlm"},
	{0x0C3B87BDD449A9D2ULL, "vxca_caverns1_mines_iyd_0_0.vlm"},
	{0x0C3BD3958AECA7CEULL, "art\\characters\\weapons\\dagger\\w_dgr_002a.mmh"},
	{0x0C3BDF958AECBC65ULL, "art\\characters\\weapons\\dagger\\w_dgr_002a.mao"},
	{0x0C3BDFA4DE39C068ULL, "follower_banter6167997.cl"},
	{0x0C3BED958AECD3BCULL, "art\\characters\\weapons\\dagger\\w_dgr_002a.msh"},
	{0x0C3C5833374B9376ULL, "659475424.ogg"},
	{0x0C3C5EFC1A580873ULL, "vxlt_0_lowtown_docks_n_b4f_0_0.vlm"},
	{0x0C3C7C1E9EF3571FULL, "gen00fl_anders6125885.cl"},
	{0x0C3C817675042635ULL, "hlac_lt_backalley_n_0_1y.dds"},
	{0x0C3D234ED237BF0CULL, "llac_ht_0_hightown_n_65537_0.dds"},
	{0x0C3D724C0FF77EFCULL, "zz_followers_banter6159165.cl"},
	{0x0C3D9C40D9A3314CULL, "6171076_m.fxe"},
	{0x0C3DBEDDCBB46F2EULL, "6125581_m.fxe"},
	{0x0C3E4E602087932AULL, "6125426_m.fxe"},
	{0x0C3E9064EA5BD39FULL, "combat_smaoe_r2_a3.gda"},
	{0x0C3EAAD1FF356C98ULL, "slime.mao"},
	{0x0C3EBCFDBDCB305AULL, "hlac_lt_0_lowtown_docks_a3_65533_2y.dds"},
	{0x0C3EC0C2E85CF875ULL, "and111_lirene6044962.cl"},
	{0x0C3EC25AD5A37994ULL, "vxdr_primevaltomb_85s_0_0.vlm"},
	{0x0C3F2003277BE29DULL, "art\\characters\\creatures\\gibgolem\\animation\\body\\gibsgolem_attack01.ani"},
	{0x0C3F367CF6F90C55ULL, "art\\vfx\\environment\\level_fx\\smoke\\vfx_steamloop_10s.mmh"},
	{0x0C3FA612819BB25FULL, "hlac_ht_0_hightown_n_65537_1z.dds"},
	{0x0C404289D967A216ULL, "follower_banter6173129.cl"},
	{0x0C405DD090C2EEFDULL, "hlac_lt_0_lowtown_n_65538_4x.dds"},
	{0x0C41308470023E95ULL, "6062791_m.fxe"},
	{0x0C41994737D612B5ULL, "mcr310_sibling6076346.cl"},
	{0x0C41B55CE34054A2ULL, "hlac_lt_0_lowtown_docks_a3_n_131070_4z.dds"},
	{0x0C424EB280F18D52ULL, "6169332_m.fxe"},
	{0x0C42B8DEB4184807ULL, "lgt000ip_ultra_mage_remains.ncs"},
	{0x0C42CCD30CE011CCULL, "1007726605.ogg"},
	{0x0C42D6472B5BDEB3ULL, "86702611.ogg"},
	{0x0C42E75F84DE3926ULL, "zz_mcr_debug6080861.cl"},
	{0x0C431311013699E7ULL, "art\\characters\\weapons\\greatswords\\textures\\w_gsw_006a_0t.dds"},
	{0x0C433CFA8A668B23ULL, "vxht_brothel_n_6ki_0_0.vlm"},
	{0x0C435450D1CA4DFAULL, "gen00fl_isabela6135658.cl"},
	{0x0C43F7CABE981856ULL, "368979431.ogg"},
	{0x0C441713A7D55B29ULL, "car100pt_gift.ncs"},
	{0x0C4437B3FBE7EB6BULL, "6076507_m.fxe"},
	{0x0C44454B63D912B2ULL, "art\\cinematics\\_crit path\\mage crisis\\mcr310cs_chantry_explosion\\cs_anders_chant_ex2.gad"},
	{0x0C446D8480021750ULL, "168044928.ogg"},
	{0x0C44BE3097D7A632ULL, "art\\characters\\playercharacter\\humanmale\\animation\\partypicker\\magestaffselectedenter.evt"},
	{0x0C45367FA1D43F8FULL, "728832249.ogg"},
	{0x0C459A866FB694ABULL, "gen00fl_fenris6126459.cl"},
	{0x0C45EAF6D82D8703ULL, "vxlt_undercityserialkiller_120_0_0.vlm"},
	{0x0C45F16928EF51F8ULL, "6149303_m.fxe"},
	{0x0C460BD7CD2D4F5FULL, "vxw_woundedcoast_1o2_0_0.vlm"},
	{0x0C4632073866A89DULL, "hla_ht_0_hightown_n_65538_1x.dds"},
	{0x0C46A82DAA2C0DC8ULL, "dre000_varric6208467.cl"},
	{0x0C46C2FB6F449818ULL, "6066772_m.fxe"},
	{0x0C472DEEAD66664EULL, "6108944_m.fxe"},
	{0x0C47B8BD2289E84FULL, "446293436.ogg"},
	{0x0C48134A50DC8C9AULL, "art\\vfx\\combat\\vfx_w_wns_sheild_defen_l_c.anb"},
	{0x0C4827E098380B23ULL, "gen00fl_sebastian6188706.cl"},
	{0x0C4865CA179EFB4BULL, "lgt360_mag_orlanna6155314.cl"},
	{0x0C4869801029D626ULL, "dae000_paivel6111933.cl"},
	{0x0C49BED28402D70EULL, "credits.gfx"},
	{0x0C49C4BED5C66EF2ULL, "dae000ip_crowd_lm2_drunk2.utp"},
	{0x0C49CA67D35BFC5FULL, "fen210_hadriana.bnk"},
	{0x0C4A4FAAF76DD768ULL, "6071916_m.fxe"},
	{0x0C4AFE4380E071A4ULL, "dae000_gamlen6165971.cl"},
	{0x0C4BDD3097DDF9C5ULL, "art\\characters\\playercharacter\\humanmale\\animation\\partypicker\\magestaffselectedenter.gad"},
	{0x0C4BE643D188D9FBULL, "art\\levels\\races\\proxy\\props\\_textures\\woodrope_n.dds"},
	{0x0C4C0D0C769D144BULL, "vxlt_0_lowtown_a2_n_c_hkz_0_0.vlm"},
	{0x0C4C337CF70391E2ULL, "art\\vfx\\environment\\level_fx\\smoke\\vfx_steamloop_10s.anb"},
	{0x0C4CAC67D35DFF67ULL, "fen210_hadriana.cnv"},
	{0x0C4CD3E92ED661CCULL, "gen00fl_merrill6106434.cl"},
	{0x0C4D01DECDD22730ULL, "llac_w_bonepitdestroyed_a3_0_2.dds"},
	{0x0C4D11A1CF54D735ULL, "vxht_brothel_n_33z_0_0.vlm"},
	{0x0C4D2B0AD171719EULL, "6092240_f.fxe"},
	{0x0C4D96205CE1843FULL, "819065408.ogg"},
	{0x0C4DAE255DA3CF24ULL, "art\\vfx\\combat\\vfx_r_shd_decoy_dis_aoe_p.mmh"},
	{0x0C4DB393ADE5741CULL, "vxca_caverns2_937_0_1.vlm"},
	{0x0C4DCE0D1A6DD20BULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\kiss_sweet_reaction.evt"},
	{0x0C4E0C3F680F71E8ULL, "402175030.ogg"},
	{0x0C4E5971350A4582ULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\a_comm_v3.gad"},
	{0x0C4EDC26D924A307ULL, "hlac_lt_0_lowtown_a2_131071_0y.dds"},
	{0x0C4F25FE94FC4587ULL, "art\\characters\\weapons\\longswords\\textures\\w_lsw_014a_0s.dds"},
	{0x0C4F7A0687F32DBFULL, "art\\vfx\\cinematic\\vfx_spark_imp_p.mmh"},
	{0x0C4F7FC3D0A716AFULL, "vxlt_gallowstemplar_f_2mw_0_0.vlm"},
	{0x0C4FF95D7B2D438AULL, "vxlt_darktown_4w5_0_0.vlm"},
	{0x0C5005C006AA3215ULL, "6166932_m.fxe"},
	{0x0C505E25044F528DULL, "vxw_encounter1_eo_0_0.vlm"},
	{0x0C50676CCD7DADFCULL, "9307980.ogg"},
	{0x0C5068A409C3F6D6ULL, "art\\characters\\weapons\\greatswords\\textures\\w_gsw_gs102a_0n.dds"},
	{0x0C508EBF71E2B3BEULL, "vxlt_undercitydungeon_205_0_0.vlm"},
	{0x0C50C3619747197EULL, "lgt210_magu_selby6142015.cl"},
	{0x0C50E3EAE23818D4ULL, "548404038.ogg"},
	{0x0C527588A508EA88ULL, "dae000cr_grand_cleric6124682.cl"},
	{0x0C529778A96825A2ULL, "vxca_caverns1_edr_0_0.vlm"},
	{0x0C52C3647C9A8A38ULL, "6111346_m.fxe"},
	{0x0C52CAC08FFE07CDULL, "vxht_keep_n_3ik_0_0.vlm"},
	{0x0C52DEEDC24FA663ULL, "llac_ca_cavern1_haunted_24425_8.dds"},
	{0x0C5303FE44E8A38AULL, "mag100_cullen6093538.cl"},
	{0x0C530BB323CD59AEULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_paperb.msh"},
	{0x0C53136CEDFE3CF2ULL, "101105837.ogg"},
	{0x0C5319B323CD71FFULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_paperb.mao"},
	{0x0C531E1F3FE99A5DULL, "dae000_orana6149021.cl"},
	{0x0C5325B323CD859CULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_paperb.mmh"},
	{0x0C533BDED02B4EBDULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\_textures\\prp_slave_d.dds"},
	{0x0C53699D647EAD90ULL, "6163277_m.fxe"},
	{0x0C53733F6D43ED00ULL, "vxlt_0_lowtown_gof_0_1.vlm"},
	{0x0C5381CABFD76E9DULL, "art\\vfx\\combat\\vfx_m_sta_rng_proj_fire_p.mmh"},
	{0x0C53E0719EA16DEDULL, "dae000_bartrand6205977.cl"},
	{0x0C545CD3CA239F83ULL, "vxlt_undercityserialkiller_173_0_0.vlm"},
	{0x0C54B5CE8F36C22FULL, "lit_aoe_03.mao"},
	{0x0C54E70D1A741BCCULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\kiss_sweet_reaction.gad"},
	{0x0C55E0997022D7A7ULL, "965624728.ogg"},
	{0x0C564901A7F0BC91ULL, "vxht_bartrand_n_33v_0_0.vlm"},
	{0x0C57B4FE2464AB6DULL, "6078851_m.fxe"},
	{0x0C57C1433D138F83ULL, "ms_clo.mao"},
	{0x0C57D8744202E769ULL, "vxlt_gallowstemplar_f_5jg_0_0.vlm"},
	{0x0C58304B63E99858ULL, "art\\cinematics\\_crit path\\mage crisis\\mcr310cs_chantry_explosion\\cs_anders_chant_ex2.ani"},
	{0x0C58A81E0DFCE0A4ULL, "198980892.ogg"},
	{0x0C590C4BC44F9BF1ULL, "146585284.ogg"},
	{0x0C595C4F9654EDA6ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\shield and weapon\\sw_attk_trn_lft_180.evt"},
	{0x0C596952E17E7B0FULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\boss\\meredith\\combat\\rushingstrike.gad"},
	{0x0C5A3A05C9FEF890ULL, "542586280.ogg"},
	{0x0C5A62E7D27ED54EULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_paintingsmld.msh"},
	{0x0C5A70E7D27EEC9FULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_paintingsmld.mao"},
	{0x0C5A7CE7D27F013CULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_paintingsmld.mmh"},
	{0x0C5A904059F34AF1ULL, "242723182.ogg"},
	{0x0C5ADF0A6BE23C22ULL, "6136963_m.fxe"},
	{0x0C5B1CFC3B672275ULL, "follower_banter6171038.cl"},
	{0x0C5B494DF905B1EEULL, "6160696_m.fxe"},
	{0x0C5BE60D1A7A38CAULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\kiss_sweet_reaction.ani"},
	{0x0C5BFC5D570ECC5CULL, "zz_approval_debug6003931.cl"},
	{0x0C5C25709FDB8C9CULL, "vxca_caverns2_946_0_1.vlm"},
	{0x0C5CA117364A7E89ULL, "follower_banter6160674.cl"},
	{0x0C5D0836AC1D9AE2ULL, "vxlt_darktown_4jv_0_0.vlm"},
	{0x0C5D48D9D482F69CULL, "vxht_0_hightown_a3_9ar_0_0.vlm"},
	{0x0C5E06C50062C241ULL, "vxlt_0_lowtown_a3_n_c_hcq_0_0.vlm"},
	{0x0C5E256B5D854F77ULL, "6167069_m.fxe"},
	{0x0C5ECDE5B4EE24B8ULL, "one200_gold_followers6120150.cl"},
	{0x0C5ECE2DF1B76D2BULL, "ave100_donnic6105996.cl"},
	{0x0C5ED001F6456B0DULL, "vxw_encounter1_jn_0_0.vlm"},
	{0x0C5F0189825073C7ULL, "character_character_p163.psh.fxo.dx11.assembly"},
	{0x0C5F424F7D3A3197ULL, "31616404.ogg"},
	{0x0C5F734F965980F1ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\shield and weapon\\sw_attk_trn_lft_180.gad"},
	{0x0C5F7E52E1830ABCULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\boss\\meredith\\combat\\rushingstrike.evt"},
	{0x0C5F82A0FD43A51CULL, "lgt210_shp_kill_qunari2.cnv"},
	{0x0C5FA4CD763610F0ULL, "hlac_lt_darktown_-1_90y.dds"},
	{0x0C6002271889C4AEULL, "gen00fl_isabela6100074.cl"},
	{0x0C601EE641B34EB7ULL, "prx_upperblank.dds"},
	{0x0C606A7AF33CC2E0ULL, "vxlt_gallowstemplar_3sd_0_0.vlm"},
	{0x0C60A4D2F402A07FULL, "qun100ip_compound_gate.ncs"},
	{0x0C60C301DE15F3FDULL, "6104957_f.fxe"},
	{0x0C6144E182C3BE64ULL, "m_isa121ar_chantry.wbd"},
	{0x0C61A0C92BC6AD48ULL, "gen_im_wep_war_one_act1_02.uti"},
	{0x0C61AD7A0F8DF28BULL, "vxht_mansion2_1ud_0_0.vlm"},
	{0x0C6245DCE86B30ABULL, "pro000_varric6170664.cl"},
	{0x0C627045FA9FFF9FULL, "dae000_meredith.bnk"},
	{0x0C62A4A0FD46156CULL, "lgt210_shp_kill_qunari2.bnk"},
	{0x0C63119EE1C32B80ULL, "zz_followers_banter6157329.cl"},
	{0x0C638471351CEB68ULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\a_comm_v3.ani"},
	{0x0C64B5B78C7C03F8ULL, "art\\characters\\creatures\\desiredemon\\animation\\blendtree\\desiredemon_movement.anb"},
	{0x0C64B98D526519D3ULL, "gen00fl_merrill6108257.cl"},
	{0x0C656AFC564E176CULL, "art\\levels\\races\\design\\animation\\floortrapunactivate.ani"},
	{0x0C6587DD981B30B6ULL, "mag371_elsa6064671.cl"},
	{0x0C65A0E114DDDAB0ULL, "gen00fl_aveline6122068.cl"},
	{0x0C65F2F80BCB6A11ULL, "220209167.ogg"},
	{0x0C662D37C6E7DABFULL, "vxlt_undercityencounter_13v_0_1.vlm"},
	{0x0C665245FAA3B5A7ULL, "dae000_meredith.cnv"},
	{0x0C66744F965FA09FULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\shield and weapon\\sw_attk_trn_lft_180.ani"},
	{0x0C67100C577EC4EEULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\pose_b_reaction_left_shoulder.ani"},
	{0x0C671505A39B7AD8ULL, "vxdr_0_deeproads_f53_0_0.vlm"},
	{0x0C674FFDB8BF19F8ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\1wi_filler.phy"},
	{0x0C67913B24411768ULL, "vxlt_undercityserialkiller_1ic_0_1.vlm"},
	{0x0C67BEC9388AB341ULL, "vxlt_undercitydungeon_1mn_0_0.vlm"},
	{0x0C683A09BEB20702ULL, "hlac_ht_0_hightown_a2_-131073_0y.dds"},
	{0x0C689DA43E033C45ULL, "hlac_lt_0_lowtown_docks_a3_n_-131073_3z.dds"},
	{0x0C68B21507EC1973ULL, "plt_cod_ltr_forgamlen.nss"},
	{0x0C68BD483463D2C6ULL, "dae100_bran6118348.cl"},
	{0x0C690941436620B8ULL, "atl_load_town_dxt5_00.dds"},
	{0x0C69319561E4DFD0ULL, "pro000_merchant6051587.cl"},
	{0x0C694B45C83B4CEDULL, "gen00fl_isabela_init6075998.cl"},
	{0x0C69FD60B79F79D1ULL, "qun120cr_javaris6042831.cl"},
	{0x0C6A55786BD35A17ULL, "vxlt_hangedman_n_25r_0_0.vlm"},
	{0x0C6A6896B5DF4EA0ULL, "hlac_lt_darktown_-1_20z.dds"},
	{0x0C6AA0C61FEE383CULL, "framebuffereffects_hslmatrixfade_p0.psh.fxo.dx11"},
	{0x0C6ABBE3A244A9D9ULL, "gen00fl_aveline6132682.cl"},
	{0x0C6ABFD0D5F5E8A1ULL, "633730133.ogg"},
	{0x0C6ACE544F15FF88ULL, "lt_0_lowtown_docks_17_0_lc.mmh"},
	{0x0C6AD8F0DC6D7D18ULL, "gen_im_trs_rdr_hvy_cht_r2.uti"},
	{0x0C6ADF7FB9AA8DE1ULL, "6072471_m.fxe"},
	{0x0C6AFF652815A306ULL, "l_dae421ar_player_home_nt.wbd"},
	{0x0C6B6408352D1BBAULL, "gen00fl_sebastian6189038.cl"},
	{0x0C6B77FC565298AEULL, "art\\levels\\races\\design\\animation\\floortrapunactivate.gad"},
	{0x0C6BEB06BDF7837EULL, "487551970.ogg"},
	{0x0C6CD3DBD0657E46ULL, "follower_banter6173330.cl"},
	{0x0C6CDD175F19748AULL, "vxlt_darktown_4y7_0_0.vlm"},
	{0x0C6D8652E18F55D5ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\boss\\meredith\\combat\\rushingstrike.ani"},
	{0x0C6DD16E3DA6BFFEULL, "vxw_sundermount_1zh_0_0.vlm"},
	{0x0C6DEE4C45903D33ULL, "hm_templar_captain.mor"},
	{0x0C6E884D06746462ULL, "rot_clo.mao"},
	{0x0C6EA5AC28002C4FULL, "lt_0_lowtown_docks_187_0.mao"},
	{0x0C6EB1AC280040ECULL, "lt_0_lowtown_docks_187_0.mmh"},
	{0x0C6EB7AC28004ADEULL, "lt_0_lowtown_docks_187_0.msh"},
	{0x0C6F10DF609A80ACULL, "6154730_m.fxe"},
	{0x0C6F92852CC1FC08ULL, "6173702_m.fxe"},
	{0x0C6FA1109B31D74AULL, "832724451.ogg"},
	{0x0C6FA9861FB2CF33ULL, "vxht_0_hightown_a3_b09_0_0.vlm"},
	{0x0C706D0ED2BBB58CULL, "art\\characters\\weapons\\greatswords\\w_gsw_gs101a.phy"},
	{0x0C707449499F012FULL, "dae011_gossips6149769.cl"},
	{0x0C70772F96F69E27ULL, "zz_seb_debug6214542.cl"},
	{0x0C70DC043D4814BFULL, "hlac_w_bonepit_131075_0z.dds"},
	{0x0C71164A50FF974DULL, "art\\vfx\\combat\\vfx_w_wns_sheild_defen_l_c.mmh"},
	{0x0C7228B49919F63DULL, "stone.dds"},
	{0x0C723174D8E0342AULL, "wm_ico_doorway_roll.dds"},
	{0x0C7268583DECF515ULL, "vxlt_gallowscourtyard_6yl_0_0.vlm"},
	{0x0C7272FC5658AE8DULL, "art\\levels\\races\\design\\animation\\floortrapunactivate.evt"},
	{0x0C727899C9E9840CULL, "vxca_caverns1_e88_0_0.vlm"},
	{0x0C72B78FE665630CULL, "6125363_m.fxe"},
	{0x0C7329FD4AA60570ULL, "hlac_lt_0_lowtown_a2_2_1x.dds"},
	{0x0C733B10D8C3014BULL, "gen_im_wep_mag_sta_lore_03.uti"},
	{0x0C73A6CAE0BCA500ULL, "1066377583.ogg"},
	{0x0C74080C5789413FULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\pose_b_reaction_left_shoulder.evt"},
	{0x0C740BC44FEF1506ULL, "tbe100_xenon6190980.cl"},
	{0x0C74780895AF20B4ULL, "vxw_encounter3_bi_0_0.vlm"},
	{0x0C747D5D8FD95448ULL, "mag131_keran6044523.cl"},
	{0x0C74D2543A2C1E75ULL, "gen_im_arm_cht_nobb_t1.uti"},
	{0x0C76384067D68100ULL, "art\\levels\\ztools\\helpers\\character\\prp_dwarflean_3.mao"},
	{0x0C76B1255DC6D91BULL, "art\\vfx\\combat\\vfx_r_shd_decoy_dis_aoe_p.anb"},
	{0x0C76D203A6CD2CF6ULL, "vxlt_0_lowtown_docks_a3_n_aor_0_0.vlm"},
	{0x0C76DBE0560FE8DDULL, "844483686.ogg"},
	{0x0C773A65A644EEDAULL, "239454452.ogg"},
	{0x0C777B57DC6ACCD3ULL, "6146781_m.fxe"},
	{0x0C787B06881634FCULL, "art\\vfx\\cinematic\\vfx_spark_imp_p.anb"},
	{0x0C78EF025335D847ULL, "6172953_m.fxe"},
	{0x0C78FF6ED731B49CULL, "565380498.ogg"},
	{0x0C795F25B7B03B14ULL, "w_bonepitdestroyed_a3_11_0_lc.mmh"},
	{0x0C798D25B904346FULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_meat_d.dds"},
	{0x0C79B0FFFCA5B1EDULL, "610192664.ogg"},
	{0x0C7A0C2785A27887ULL, "85544317.ogg"},
	{0x0C7A41C2A205BC6AULL, "vxlt_undercityserialkiller_1sr_0_0.vlm"},
	{0x0C7A4E2398AEB8FDULL, "6176725_f.fxe"},
	{0x0C7AE90C578F2B28ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\pose_b_reaction_left_shoulder.gad"},
	{0x0C7B3A4E0082B6FCULL, "651634276.ogg"},
	{0x0C7B7E3FFC3A1A8EULL, "166715699.ogg"},
	{0x0C7C1C86CC8BFDACULL, "511649613.ogg"},
	{0x0C7C80D669457FE9ULL, "622530748.ogg"},
	{0x0C7D64CBA5B64E8AULL, "6124048_f.fxe"},
	{0x0C7D671A05EF7147ULL, "576938896.ogg"},
	{0x0C7D8CBE5366A1FDULL, "vxdr_primevaltomb_6yx_0_0.vlm"},
	{0x0C7D9F78F2865F8BULL, "6036679_m.fxe"},
	{0x0C7DCAD19D47E7E2ULL, "238227332.ogg"},
	{0x0C7E205765E0514DULL, "vxht_keep_n_3xh_0_0.vlm"},
	{0x0C7E4D3A4903BA5EULL, "146701919.ogg"},
	{0x0C7E522C1AB696C9ULL, "hlac_lt_0_lowtown_a2_n_c_2_1z.dds"},
	{0x0C7E7DA8ABD3672BULL, "vxht_0_hightown_a2_c81_0_0.vlm"},
	{0x0C7E82358D2E654FULL, "lt_0_lowtown_docks_n_34_0.rml"},
	{0x0C7E9133F37A238BULL, "vxht_mansion2_1cf_0_0.vlm"},
	{0x0C7EDC45C4825A4AULL, "vxht_mansion2b_2nc_0_0.vlm"},
	{0x0C7F5BE7D29DE507ULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_paintingsmld.phy"},
	{0x0C7FA799DE89D7DFULL, "one220_yevhen6103018.cl"},
	{0x0C7FEEA76196B0BAULL, "6053180_f.fxe"},
	{0x0C8031F9FD636473ULL, "llac_ht_0_hightown_0_2.dds"},
	{0x0C8053A1E2395DCAULL, "hlac_w_bonepitdestroyed_a3_327681_0y.dds"},
	{0x0C8102A67EF73FB4ULL, "vxht_mansion2_30v_0_0.vlm"},
	{0x0C8132BEFA843FE7ULL, "vxht_keep_73l_0_0.vlm"},
	{0x0C8164ACCE651EF3ULL, "art\\characters\\creatures\\gallowsslavestatue\\animation\\enterv4.evt"},
	{0x0C818038F366F4C3ULL, "sf_textcolormult.psh.fxo"},
	{0x0C81CD0ACDFAB4DFULL, "149033742.ogg"},
	{0x0C81E676A160DB71ULL, "6087220_m.fxe"},
	{0x0C821D8A5E9237F2ULL, "dae000cr_grand_cleric6187948.cl"},
	{0x0C826F2AE2EFDECCULL, "store_dae210_weapons.utm"},
	{0x0C82842A64E837DEULL, "6031631_m.fxe"},
	{0x0C833C742AABFE6BULL, "40033380.ogg"},
	{0x0C8347B80AA8EB11ULL, "hlac_lt_0_lowtown_docks_-196611_1y.dds"},
	{0x0C83BC67629C428CULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\animation\\4co_lowtown_ext_trap.activetoinactive.ani"},
	{0x0C83C44AF699A7E6ULL, "hlac_lt_0_lowtown_a2_n_c_65536_9z.dds"},
	{0x0C8445D133F7F866ULL, "mag210_arianni6021222.cl"},
	{0x0C84720D5D9CEFD4ULL, "6187910_m.fxe"},
	{0x0C850ED045B859DFULL, "824303711.ogg"},
	{0x0C859BDB70EE7A35ULL, "6155664_m.fxe"},
	{0x0C8619EA75C92751ULL, "816561455.ogg"},
	{0x0C869DC0F11EA5FAULL, "lt_0_lowtown_docks_n_102_0.mmh"},
	{0x0C86A1C0F11EAC31ULL, "lt_0_lowtown_docks_n_102_0.mao"},
	{0x0C86AAA30BA2F13AULL, "art\\vfx\\creatures\\fxc_bm_tnt_app_c.cess.gad"},
	{0x0C86AFC0F11EC440ULL, "lt_0_lowtown_docks_n_102_0.msh"},
	{0x0C875B9CB3350853ULL, "dae000_mother6099197.cl"},
	{0x0C87B28F389BB7A3ULL, "6114816_m.fxe"},
	{0x0C885DACCE6B3134ULL, "art\\characters\\creatures\\gallowsslavestatue\\animation\\enterv4.gad"},
	{0x0C88D8C487E4A8A9ULL, "6116324_m.fxe"},
	{0x0C8909B00ED1B162ULL, "art\\levels\\races\\ferelden\\props\\lowerclass\\prp_barrelopen.msh"},
	{0x0C890BB00ED1B418ULL, "art\\levels\\races\\ferelden\\props\\lowerclass\\prp_barrelopen.mmh"},
	{0x0C8917B00ED1C8BBULL, "art\\levels\\races\\ferelden\\props\\lowerclass\\prp_barrelopen.mao"},
	{0x0C89B13E451660C5ULL, "zz_fol_debug6096543.cl"},
	{0x0C89B2639A46AE8CULL, "gen00fl_anders6107655.cl"},
	{0x0C89C729991CA044ULL, "237618843.ogg"},
	{0x0C89CC67A5DFDC18ULL, "300461011.ogg"},
	{0x0C89E2A6F29FC0EDULL, "6111664_m.fxe"},
	{0x0C8A87077D170894ULL, "vxca_cavern1_haunted_i2w_0_0.vlm"},
	{0x0C8AC96762A277CEULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\animation\\4co_lowtown_ext_trap.activetoinactive.gad"},
	{0x0C8AEECCD96FEC1DULL, "6060740_m.fxe"},
	{0x0C8B24281A21B8CBULL, "vxlt_hangedman_n_26o_0_0.vlm"},
	{0x0C8B8DD63BA45389ULL, "6204964_m.fxe"},
	{0x0C8B8E470789AA9CULL, "ser121_jethann6164727.cl"},
	{0x0C8C3D6A4FA5FAE1ULL, "hlac_lt_0_lowtown_0_6x.dds"},
	{0x0C8C801E249B7D62ULL, "6067868_f.fxe"},
	{0x0C8C9E1331C427E5ULL, "568672513.ogg"},
	{0x0C8D09BD081ED774ULL, "518043731.ogg"},
	{0x0C8D2E8B7C508A13ULL, "rune_armor_iron.dds"},
	{0x0C8D3E98DF08D399ULL, "vxht_bartrand_n_6pn_0_0.vlm"},
	{0x0C8D973FBA114848ULL, "art\\characters\\creatures\\strider\\cn_hed_str.msh"},
	{0x0C8DA93FBA1166D9ULL, "art\\characters\\creatures\\strider\\cn_hed_str.mao"},
	{0x0C8DADA30BA91551ULL, "art\\vfx\\creatures\\fxc_bm_tnt_app_c.cess.evt"},
	{0x0C8DB53FBA117B72ULL, "art\\characters\\creatures\\strider\\cn_hed_str.mmh"},
	{0x0C8DBA2CEEE9AEA8ULL, "static_static_p10.psh.fxo.dx11.assembly"},
	{0x0C8DD61570860F28ULL, "llac_w_woundedcoast_-196600_0.dds"},
	{0x0C8E6CACCE6FB612ULL, "art\\characters\\creatures\\gallowsslavestatue\\animation\\enterv4.ani"},
	{0x0C8FA354E4A5F43EULL, "6130976_m.fxe"},
	{0x0C8FBA7B021DB3D6ULL, "6128393_m.fxe"},
	{0x0C90014FAA42E91EULL, "240271662.ogg"},
	{0x0C918C2EFE577FE9ULL, "585289145.ogg"},
	{0x0C9192CB1ADA6BEFULL, "dae000_mother6004098.cl"},
	{0x0C91C46762A88DADULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\animation\\4co_lowtown_ext_trap.activetoinactive.evt"},
	{0x0C927C1A92FFFF1BULL, "lowtown_fem_alienage1_sta6162594.cl"},
	{0x0C92C3A245388D13ULL, "pro000_merchant6051601.cl"},
	{0x0C92D7FCC92CDA00ULL, "vxlt_0_lowtown_docks_n_coy_0_0.vlm"},
	{0x0C9312B00EDA14A3ULL, "art\\levels\\races\\ferelden\\props\\lowerclass\\prp_barrelopen.phy"},
	{0x0C93B43A8BF6F914ULL, "gen00fl_fenris6108076.cl"},
	{0x0C93CCB9BF5732B4ULL, "872871300.ogg"},
	{0x0C93E1D1818801B1ULL, "hlac_lt_0_lowtown_docks_n_65534_6y.dds"},
	{0x0C9439ED966786E1ULL, "6135169_m.fxe"},
	{0x0C943C0A31651B7CULL, "gen_im_wep_rog_dua_act2_07.uti"},
	{0x0C943D95C0BAEB17ULL, "vxht_bartrand_n_4gw_0_0.vlm"},
	{0x0C945EC7B9D5CB25ULL, "gen00fl_carver6104459.cl"},
	{0x0C9521466895BF54ULL, "llac_ht_0_hightown_a3_-131071_0.dds"},
	{0x0C95409D87D44ABCULL, "art\\characters\\playercharacter\\shared\\heads\\textures\\mt_stub_0n.dds"},
	{0x0C95B5C41772B44AULL, "ser200_aveline6201493.cl"},
	{0x0C96246C2A91D023ULL, "6113916_m.fxe"},
	{0x0C9624C832133E54ULL, "vxlt_gallowstemplar_f_38z_0_0.vlm"},
	{0x0C969939D2C915DDULL, "38569578.ogg"},
	{0x0C969A2103D1BD4BULL, "zz_followers_banter6157931.cl"},
	{0x0C969FFD9CD5E90BULL, "vxdr_primevalentrance_15p_0_1.vlm"},
	{0x0C97A5DEB460685AULL, "lgt000ip_ultra_mage_remains.utp"},
	{0x0C97B6092C6F5860ULL, "vxht_0_hightown_a2_awq_0_0.vlm"},
	{0x0C9832AE34FBBE0AULL, "vxlt_darktown_4n2_0_0.vlm"},
	{0x0C985A7535D3E754ULL, "dae200cr_viscount6087228.cl"},
	{0x0C987723DA8980A9ULL, "lt_gamlenshouse_n_probe_6l.dds"},
	{0x0C990D15A03A4617ULL, "ts_apostates_prog_junk2.gda"},
	{0x0C99399CB527E9C6ULL, "250438602.ogg"},
	{0x0C99C82596243165ULL, "vxht_mansion2_3do_0_0.vlm"},
	{0x0C9AC5A30BB3C820ULL, "art\\vfx\\creatures\\fxc_bm_tnt_app_c.cess.ani"},
	{0x0C9AD7BAEF577B7DULL, "hlac_lt_0_lowtown_docks_a3_-65539_4y.dds"},
	{0x0C9B05DF924E363FULL, "follower_banter6147631.cl"},
	{0x0C9B75B1FBC1CC99ULL, "dae021cr_sister_colinda.utc"},
	{0x0C9B91DC362875FFULL, "6037399_m.fxe"},
	{0x0C9BD147CCAEC549ULL, "6177408_m.fxe"},
	{0x0C9C18FA7B6863C9ULL, "hlac_lt_0_lowtown_docks_a3_n_-2_0x.dds"},
	{0x0C9CC4B0BE83E292ULL, "6126367_f.fxe"},
	{0x0C9D047FFC820F5DULL, "6037317_m.fxe"},
	{0x0C9D3847E37A1E48ULL, "6079682_m.fxe"},
	{0x0C9D3DE6BC8067DAULL, "w_encounter2_camp_25_0.rml"},
	{0x0C9DA31F941A4C30ULL, "326138717.ogg"},
	{0x0C9DD1FB24268FEFULL, "vxht_0_hightown_a2_626_0_0.vlm"},
	{0x0C9E38266456764EULL, "hlac_lt_0_lowtown_docks_a3_n_65534_1x.dds"},
	{0x0C9E64446B397B84ULL, "6161046_m.fxe"},
	{0x0C9F259C7B949566ULL, "llac_w_bonepitmining_a2_0_4.dds"},
	{0x0C9F41E4B8202128ULL, "6098971_m.fxe"},
	{0x0C9FA7F46A9599FDULL, "vxlt_0_lowtown_a2_h97_0_0.vlm"},
	{0x0C9FAB3525671077ULL, "hlac_lt_0_lowtown_docks_-4_1y.dds"},
	{0x0C9FD1BE3C746160ULL, "gen00fl_fenris6112022.cl"},
	{0x0C9FF23F5AC56BB0ULL, "hlac_lt_0_lowtown_-65539_0x.dds"},
	{0x0CA00711A9058177ULL, "hlac_dr_primevaltomb_663_3y.dds"},
	{0x0CA19B7B9FAB8615ULL, "6169744_m.fxe"},
	{0x0CA231EE825618FFULL, "hlac_lt_0_lowtown_docks_a3_n_196607_1y.dds"},
	{0x0CA2EC8B719D02ABULL, "6146998_m.fxe"},
	{0x0CA41EE6BC8651CEULL, "w_encounter2_camp_25_0.phy"},
	{0x0CA4651A70E090E4ULL, "6097321_m.fxe"},
	{0x0CA47410A6B47642ULL, "vxlt_gallowstemplar_n_5xb_0_0.vlm"},
	{0x0CA5A6BDD62734B8ULL, "205226737.ogg"},
	{0x0CA5E059C356FDDEULL, "art\\vfx\\creatures\\vfx_c_corpse_c.dura.evt"},
	{0x0CA60D07C1D0FE9BULL, "cod_bks_fifth_blight_2.utp"},
	{0x0CA642F649FDAA20ULL, "6119584_m.fxe"},
	{0x0CA64854204C7B2EULL, "art\\levels\\races\\natural\\proxy\\foliage\\_textures\\pine_dp.dds"},
	{0x0CA6798DA8B58768ULL, "6087180_m.fxe"},
	{0x0CA6E9F96F1DE234ULL, "vxht_0_hightown_n_bca_0_0.vlm"},
	{0x0CA6F7EB9B886989ULL, "204163388.ogg"},
	{0x0CA705B14676DB94ULL, "246493224.ogg"},
	{0x0CA7199DB77E28DCULL, "648564533.ogg"},
	{0x0CA7E473E93E8389ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\twohanded\\tw_cleanse_area.ani"},
	{0x0CA7E4D35B4A20EBULL, "353036889.ogg"},
	{0x0CA8569673871FC0ULL, "6076346_m.fxe"},
	{0x0CA90AAD6C88D65AULL, "vxlt_0_lowtown_a2_e3c_0_0.vlm"},
	{0x0CA9BB1B0C124290ULL, "92256969.ogg"},
	{0x0CA9D2462F90B071ULL, "llac_w_blightlands_196611_0.dds"},
	{0x0CA9D96271548E26ULL, "mcr310_loyalist_templars6076901.cl"},
	{0x0CA9DF9F6D82900AULL, "hlac_lt_warehouse_0_3z.dds"},
	{0x0CAA71C17DDAD668ULL, "seb221_flora_harimann6177688.cl"},
	{0x0CAA8522FD736E33ULL, "6027592_m.fxe"},
	{0x0CAA91606975DB5EULL, "365429249.ogg"},
	{0x0CAB1C6FC44FCF68ULL, "hlac_lt_0_lowtown_131070_3z.dds"},
	{0x0CAB50D732ACA72EULL, "951703720.ogg"},
	{0x0CABA7DA1110730EULL, "art\\characters\\playercharacter\\humanmale\\textures\\bdy_arm_meda_n.dds"},
	{0x0CABAF028DB8D978ULL, "llac_lt_gallowstemplar_n_2383_1.dds"},
	{0x0CABBCC0F13DF605ULL, "lt_0_lowtown_docks_n_102_0.phy"},
	{0x0CAC3C237BC1E9A5ULL, "239022784.ogg"},
	{0x0CAC3E316E3AF86FULL, "6134537_m.fxe"},
	{0x0CAC4B374C83AC7EULL, "6055112_m.fxe"},
	{0x0CACC25CBAE5292AULL, "vxca_cave_19a_0_0.vlm"},
	{0x0CACC6DB7796174EULL, "dae000_arishok6067821.cl"},
	{0x0CACE59B664FE45FULL, "vxlt_darktown_5cy_0_0.vlm"},
	{0x0CACF759C35D4309ULL, "art\\vfx\\creatures\\vfx_c_corpse_c.dura.gad"},
	{0x0CAEFF73E944D00BULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\twohanded\\tw_cleanse_area.gad"},
	{0x0CAFD5CC74C6574DULL, "6040975_m.fxe"},
	{0x0CB02A6F3F58082AULL, "252786503.ogg"},
	{0x0CB03FD624DABA99ULL, "mcr371st_hostile_aveline.stg"},
	{0x0CB0450B2E941018ULL, "21576616.ogg"},
	{0x0CB045736A1E215CULL, "59407539.ogg"},
	{0x0CB07478660A1131ULL, "hlac_w_encounter2_camp_-2_0z.dds"},
	{0x0CB109BDA77B4496ULL, "6162315_m.fxe"},
	{0x0CB1214FA4A71C17ULL, "vxht_bartrand_n_4qq_0_0.vlm"},
	{0x0CB1440ED2F30757ULL, "art\\characters\\weapons\\greatswords\\w_gsw_gs101a.mmh"},
	{0x0CB1480ED2F30E64ULL, "art\\characters\\weapons\\greatswords\\w_gsw_gs101a.mao"},
	{0x0CB15A0ED2F32CC9ULL, "art\\characters\\weapons\\greatswords\\w_gsw_gs101a.msh"},
	{0x0CB15E6863931B2BULL, "6143698_m.fxe"},
	{0x0CB1E8D15DE5DB1BULL, "59411719.ogg"},
	{0x0CB214233EA34087ULL, "one280_merin6120918.cl"},
	{0x0CB23F88CB93BC0BULL, "dae000_worthy6166611.cl"},
	{0x0CB2C87397FD0E04ULL, "vxht_keep_61h_0_0.vlm"},
	{0x0CB34F37B9B24652ULL, "vxlt_0_lowtown_a2_n_efg_0_0.vlm"},
	{0x0CB3683B502D7569ULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_mth02b.phy"},
	{0x0CB3C859C3631257ULL, "art\\vfx\\creatures\\vfx_c_corpse_c.dura.ani"},
	{0x0CB4072307DFB073ULL, "6106947_m.fxe"},
	{0x0CB43365540AA539ULL, "hlac_lt_gallowstemplar_f_2394_10y.dds"},
	{0x0CB437E780DA0230ULL, "gen00fl_anders6125691.cl"},
	{0x0CB441B47187343DULL, "6132937_m.fxe"},
	{0x0CB452A2000FE0B4ULL, "dae160st_saemus_1_alt.stg"},
	{0x0CB4D4FEDA544CCEULL, "414054435.ogg"},
	{0x0CB50254EB4677D7ULL, "6108696_f.fxe"},
	{0x0CB5FC73E94AEA70ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\twohanded\\tw_cleanse_area.evt"},
	{0x0CB648266DB76668ULL, "10513482.ogg"},
	{0x0CB65054F2012306ULL, "zz_one_debug6076181.cl"},
	{0x0CB67ADB86710982ULL, "vxht_0_hightown_cas_0_0.vlm"},
	{0x0CB6C375075948C7ULL, "lt_0_lowtown_a3_n_c_46_0_lc.mmh"},
	{0x0CB72207C1DF95BEULL, "cod_bks_fifth_blight_2.ncs"},
	{0x0CB73C45D4C48582ULL, "6143174_m.fxe"},
	{0x0CB830CC009D8B7BULL, "811996742.ogg"},
	{0x0CB878794CD6AF2DULL, "vxdr_dungeon_11x_0_1.vlm"},
	{0x0CB88F3C7029D5AAULL, "hlac_w_encounter3_0_5x.dds"},
	{0x0CB8A38C22BB88BCULL, "hlac_lt_gallowstemplar_2391_0y.dds"},
	{0x0CB8C60075077BD3ULL, "6120409_f.fxe"},
	{0x0CB9B5048F164B6DULL, "619271542.ogg"},
	{0x0CB9D1119935C04BULL, "171447487.ogg"},
	{0x0CBACDC903D20BCAULL, "lgt110cr_gxa_vgk_cnspr_3.utc"},
	{0x0CBAF66FB0EF8B5AULL, "vxht_keep_6fs_0_0.vlm"},
	{0x0CBB279333E9BE82ULL, "96642979.ogg"},
	{0x0CBC25DF2AE3BAA2ULL, "505974316.ogg"},
	{0x0CBC876CD1A462D1ULL, "883417293.ogg"},
	{0x0CBCDC4CC6C44F45ULL, "hlac_w_sundermount_131073_0y.dds"},
	{0x0CBD034CA25B67CAULL, "337761487.ogg"},
	{0x0CBD93DD363A0790ULL, "l_dae321ar_keep.wbd"},
	{0x0CBDD30CEA3E27C1ULL, "gen_im_flw_dua_isabela_01.uti"},
	{0x0CBEDF2C840E3BDCULL, "6098778_f.fxe"},
	{0x0CBEE4C4D7FF4FD3ULL, "6101465_m.fxe"},
	{0x0CBEEE49BDD0D1F9ULL, "742254234.ogg"},
	{0x0CBEFAA897E094A5ULL, "6085179_f.fxe"},
	{0x0CBF12A3519391F7ULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\dialogue\\kissing_male_female_reaction_exit.gad"},
	{0x0CBF994D2EA9E766ULL, "6000965_m.fxe"},
	{0x0CC1DB20BB7E657AULL, "ripple_sliver2.dds"},
	{0x0CC1F223B67F82DFULL, "aip_apostate_base.gda"},
	{0x0CC23CFDB90BA48DULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\1wi_filler.msh"},
	{0x0CC24AFDB90BBCC0ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\1wi_filler.mao"},
	{0x0CC24EFDB90BC313ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\1wi_filler.mmh"},
	{0x0CC27C7DCB5E0637ULL, "661277529.ogg"},
	{0x0CC34EE2AF720921ULL, "ser221_gamlen6131907.cl"},
	{0x0CC3CDB3530A1334ULL, "vxht_0_hightown_n_big_0_0.vlm"},
	{0x0CC432373B9D90E6ULL, "vxlt_0_lowtown_docks_a3_n_d1u_0_0.vlm"},
	{0x0CC4641AC1F049CCULL, "6051910_m.fxe"},
	{0x0CC4703610821C27ULL, "894305433.ogg"},
	{0x0CC4D20AD6558A0DULL, "hlac_dr_0_deeproads_5843_6y.dds"},
	{0x0CC4EF9AC32D0A73ULL, "zz_ave_debug6056685.cl"},
	{0x0CC53F30B57F77D0ULL, "222902201.ogg"},
	{0x0CC59578B55B5F15ULL, "995805711.ogg"},
	{0x0CC5C37A109496C4ULL, "gen00fl_anders6113100.cl"},
	{0x0CC5C81409060140ULL, "vxca_caverns1_ijr_0_0.vlm"},
	{0x0CC5F568AF698DB3ULL, "blood_conj_08_l2.mao"},
	{0x0CC627A35199D384ULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\dialogue\\kissing_male_female_reaction_exit.evt"},
	{0x0CC6D51D342CDF3CULL, "hlac_ht_mansion2b_818_64z.dds"},
	{0x0CC766AC76A5F668ULL, "72702693.ogg"},
	{0x0CC781B6FC2819DFULL, "hlac_ht_chantry_2534_18y.dds"},
	{0x0CC7D39AFAF6B498ULL, "vxca_caverns2prc_8s2_0_0.vlm"},
	{0x0CC7E93DA0AB1AE3ULL, "qun221_viscount6070092.cl"},
	{0x0CC7EA6624070ED4ULL, "289166105.ogg"},
	{0x0CC8119867727A4FULL, "vxlt_hangedman_286_0_0.vlm"},
	{0x0CC836D67F45CF56ULL, "6032263_m.fxe"},
	{0x0CC8554C71BB7863ULL, "hlac_lt_0_lowtown_docks_a3_n_-131073_0y.dds"},
	{0x0CC8BCA064AB5D99ULL, "plt_lgt110pt_pier_pressure.plo"},
	{0x0CC94191E96ECAB3ULL, "vxht_0_hightown_a3_a7g_0_0.vlm"},
	{0x0CC9684CA2E5A3DAULL, "vxht_keep_6cl_0_0.vlm"},
	{0x0CC9BEEBC3A696FEULL, "llac_lt_0_lowtown_docks_a3_n_-1_0.dds"},
	{0x0CC9F47330D85F08ULL, "269278853.ogg"},
	{0x0CC9F72C306F86CAULL, "489682425.ogg"},
	{0x0CCA2AAEE5041ECAULL, "427560635.ogg"},
	{0x0CCA413CC12C57C0ULL, "6090392_f.fxe"},
	{0x0CCA82781B217E9DULL, "200058222.ogg"},
	{0x0CCAA36FECD1C825ULL, "759648248.ogg"},
	{0x0CCB7B412E6E8778ULL, "6142012_m.fxe"},
	{0x0CCBE58D714F901AULL, "vxht_bartrand_n_64u_0_0.vlm"},
	{0x0CCC2110EEE05F71ULL, "hlac_lt_undercityserialkiller_-1_39y.dds"},
	{0x0CCD12A5B3AA0E36ULL, "mnd_con_custic.mao"},
	{0x0CCD15FB3DDD207EULL, "zz_one_debug6073991.cl"},
	{0x0CCD17C1EC577D42ULL, "316126267.ogg"},
	{0x0CCDC71591D62C7BULL, "vis121_saemus6118420.cl"},
	{0x0CCDCC58CBEB004BULL, "gen00fl_anders6119877.cl"},
	{0x0CCE23EE147BFBBBULL, "art\\characters\\playercharacter\\humanfemale\\mt_ant_hf05.phy"},
	{0x0CCE4922AAC045FCULL, "292747142.ogg"},
	{0x0CCEA9B9DFC7E2DAULL, "vxht_0_hightown_n_bpf_0_0.vlm"},
	{0x0CCEAD150842C3DCULL, "plt_cod_ltr_forgamlen.plo"},
	{0x0CCEDE815DD456FFULL, "art\\characters\\playercharacter\\humanmale\\textures\\hm_hlf_medd_0d.dds"},
	{0x0CCF4E217D5D9C15ULL, "vxw_encounter3_f0_0_0.vlm"},
	{0x0CCF5982D14174D0ULL, "llac_lt_0_lowtown_a2_1_8.dds"},
	{0x0CCF66926CCAC846ULL, "vxht_brothel_n_630_0_0.vlm"},
	{0x0CCF861B1D4B50B2ULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_xenon_sa.dds"},
	{0x0CCF8D6A0B65DDB3ULL, "gen00fl_aveline6137481.cl"},
	{0x0CCFAB64F4244150ULL, "vxht_0_hightown_b94_0_0.vlm"},
	{0x0CD010D76DD3DFBAULL, "gen00fl_isabela6102454.cl"},
	{0x0CD065A2E86DE76FULL, "character_character_p24.psh.fxo.dx11"},
	{0x0CD0D68C2B0D7E7BULL, "vxdr_primevaltomb_26g_0_0.vlm"},
	{0x0CD0F88D4434008FULL, "art\\vfx\\materials\\vfx_cyc_add.mao"},
	{0x0CD159C1273924FCULL, "6090411_f.fxe"},
	{0x0CD179AA0B2182C0ULL, "vxht_mansion2_311_0_0.vlm"},
	{0x0CD1A02090767641ULL, "vxht_brothel_n_6k3_0_0.vlm"},
	{0x0CD1D66B8732B252ULL, "vxlt_warehouse_v7_0_0.vlm"},
	{0x0CD215D985590370ULL, "6094734_f.fxe"},
	{0x0CD25D1340F42DB7ULL, "6152199_m.fxe"},
	{0x0CD29792CE504B38ULL, "character_character_p245.psh.fxo.dx11.assembly"},
	{0x0CD2AD7F5947AEB9ULL, "6123411_f.fxe"},
	{0x0CD2D2ABB4961911ULL, "128801892.ogg"},
	{0x0CD2D76167D8D2CCULL, "27937349.ogg"},
	{0x0CD2D85CE3501970ULL, "vxca_caverns1_ii7_0_0.vlm"},
	{0x0CD2D9CE6D4CF53CULL, "107331289.ogg"},
	{0x0CD32216E9C509C0ULL, "233106016.ogg"},
	{0x0CD33FA351A4877DULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\dialogue\\kissing_male_female_reaction_exit.ani"},
	{0x0CD3CFFCAC0CFE36ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\prp_dooriron1j.mao"},
	{0x0CD3DBFCAC0D12D5ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\prp_dooriron1j.mmh"},
	{0x0CD3DDFCAC0D164BULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\prp_dooriron1j.msh"},
	{0x0CD429057446E2AFULL, "6111257_m.fxe"},
	{0x0CD439F0FAFC19C0ULL, "vxca_caverns1_ims_0_0.vlm"},
	{0x0CD49E28561CBF76ULL, "dae000_merchant_hm_mob6174220.cl"},
	{0x0CD4AEE181742E96ULL, "art\\characters\\weapons\\shortbows\\textures\\w_sbw_sb103a_0s.dds"},
	{0x0CD4D5E11214A02FULL, "150529595.ogg"},
	{0x0CD5205FCE7BBE80ULL, "vxlt_0_lowtown_a2_n_gmj_0_0.vlm"},
	{0x0CD52418CE3FF153ULL, "vxlt_0_lowtown_a2_n_eqg_0_0.vlm"},
	{0x0CD5297A7CDC617FULL, "hlac_lt_warehouse_n_-65536_1x.dds"},
	{0x0CD59248EE21F05BULL, "dae300_vashothlead6062776.cl"},
	{0x0CD641F456FD8A73ULL, "llac_ht_mansion2_818_29.dds"},
	{0x0CD68375596892CFULL, "vxlt_hangedman_251_0_0.vlm"},
	{0x0CD683DF0D1C6F72ULL, "502288018.ogg"},
	{0x0CD78BA3AE28214FULL, "art\\vfx\\cinematic\\vfx_dragon_transform_p.mmh"},
	{0x0CD7B6DDCFD299E9ULL, "vxlt_gallowstemplar_n_5qb_0_0.vlm"},
	{0x0CD7C1C15A9BF27BULL, "885942524.ogg"},
	{0x0CD7D00B876BEB57ULL, "vxlt_gallowstemplar_478_0_0.vlm"},
	{0x0CD7DA2994DBBC5AULL, "vxht_keep_6lm_0_0.vlm"},
	{0x0CD7DEFD1FA59840ULL, "6160636_m.fxe"},
	{0x0CD831D1DD140557ULL, "hlac_ht_chantry_2534_10y.dds"},
	{0x0CD8D21D0CF4CDD8ULL, "176252087.ogg"},
	{0x0CD960611FF1AFD4ULL, "6152358_m.fxe"},
	{0x0CD976A70748A723ULL, "follower_banter6170237.cl"},
	{0x0CD9E4E376385C85ULL, "dae000cr_grand_cleric6148096.cl"},
	{0x0CDADC591A583929ULL, "hlac_lt_0_lowtown_a2_n_c_65536_12y.dds"},
	{0x0CDB2340FA9A4399ULL, "art\\levels\\races\\ferelden\\props\\keep\\prp_flagkeep.msh"},
	{0x0CDB2D40FA9A5487ULL, "art\\levels\\races\\ferelden\\props\\keep\\prp_flagkeep.mmh"},
	{0x0CDB3140FA9A5BD4ULL, "art\\levels\\races\\ferelden\\props\\keep\\prp_flagkeep.mao"},
	{0x0CDB444CBD4F4ECCULL, "377102022.ogg"},
	{0x0CDB84DFE40259D7ULL, "llac_ca_caverns1_mines_24422_5.dds"},
	{0x0CDBDC77F2F2E269ULL, "712909146.ogg"},
	{0x0CDC0A090F1B14C7ULL, "llac_w_bonepitdestroyed_a3_1_0.dds"},
	{0x0CDC4F884048440FULL, "386383903.ogg"},
	{0x0CDC94FA1F2AF5D8ULL, "305657488.ogg"},
	{0x0CDD09072AF8F60FULL, "vxlt_0_lowtown_a2_n_hib_0_0.vlm"},
	{0x0CDE44E9943A94D5ULL, "vxht_0_hightown_a2_bi9_0_0.vlm"},
	{0x0CDE9463DC43D447ULL, "6088245_m.fxe"},
	{0x0CDE990656805215ULL, "vxca_caverns1_ihy_0_0.vlm"},
	{0x0CDEE1729F948CC3ULL, "gen00fl_aveline6132774.cl"},
	{0x0CDF2690E8E2875FULL, "primitives_primitive_p3.vsh.fxo.dx11"},
	{0x0CDF513B5052888EULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_mth02b.mmh"},
	{0x0CDF6B3B5052B57CULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_mth02b.msh"},
	{0x0CDF946421756C91ULL, "vxw_encounter2_camp_p8_0_1.vlm"},
	{0x0CE02ACD04147B13ULL, "vxca_caverns1_mines_e7s_0_0.vlm"},
	{0x0CE09B96257ADBF1ULL, "llac_lt_0_lowtown_docks_a3_n_131072_1.dds"},
	{0x0CE0C7EBF38E25E3ULL, "character_character_p134.psh.fxo.dx11.assembly"},
	{0x0CE11486F785708DULL, "art\\characters\\playercharacter\\humanmale\\boo_medd.mmh"},
	{0x0CE11886F78577DEULL, "art\\characters\\playercharacter\\humanmale\\boo_medd.mao"},
	{0x0CE12686F7858F13ULL, "art\\characters\\playercharacter\\humanmale\\boo_medd.msh"},
	{0x0CE155AAB6D09CE0ULL, "152332884.ogg"},
	{0x0CE160EED677D2A2ULL, "vxlt_backalley_g_1u8_0_0.vlm"},
	{0x0CE1A9CDDBF45DB2ULL, "art\\characters\\weapons\\longbows\\textures\\w_lbw_lb106a_0t.dds"},
	{0x0CE1F9606FB78077ULL, "hlac_lt_gallowscourtyard_-131073_0y.dds"},
	{0x0CE1FB6E195D93F2ULL, "816261673.ogg"},
	{0x0CE25AA675E9FCA3ULL, "1041266337.ogg"},
	{0x0CE29FFD6ED9E69DULL, "201770493.ogg"},
	{0x0CE2AF46C80801FDULL, "6051036_m.fxe"},
	{0x0CE2FDC8B715DB52ULL, "dae000_gen_turn_in_26178484.cl"},
	{0x0CE3291F1B2DA0AAULL, "vxlt_undercitydungeon_15q_0_0.vlm"},
	{0x0CE32B02541EDD79ULL, "vxlt_gallowstemplar_n_6tb_0_0.vlm"},
	{0x0CE3923CC071D700ULL, "vxlt_0_lowtown_a2_n_gjk_0_0.vlm"},
	{0x0CE41DE6BCBC3391ULL, "w_encounter2_camp_25_0.mmh"},
	{0x0CE429E6BCBC4832ULL, "w_encounter2_camp_25_0.mao"},
	{0x0CE430B5D2C3B510ULL, "152598240.ogg"},
	{0x0CE437E6BCBC604FULL, "w_encounter2_camp_25_0.msh"},
	{0x0CE4616BFE9604DCULL, "6095333_f.fxe"},
	{0x0CE463ACA7664E72ULL, "vxht_mansion2_2yf_0_0.vlm"},
	{0x0CE466B6D1443BC3ULL, "lt_0_lowtown_docks_n_1_0.rml"},
	{0x0CE46A2191064658ULL, "6104843_m.fxe"},
	{0x0CE4857AEB24D2E4ULL, "6160795_m.fxe"},
	{0x0CE4F71BA8254595ULL, "w_sundermount_197_0.phy"},
	{0x0CE53FA162E501EAULL, "6152327_m.fxe"},
	{0x0CE59037091392B0ULL, "vxca_caverns1_mines_e89_0_0.vlm"},
	{0x0CE599946BAECEA6ULL, "6088781_m.fxe"},
	{0x0CE5F99BDEA0AA88ULL, "636336917.ogg"},
	{0x0CE61135EA4A0D2AULL, "llac_ca_cavern1_haunted_24427_3.dds"},
	{0x0CE6544F2BD56B09ULL, "vxlt_blackemporium_tf_0_0.vlm"},
	{0x0CE65C210137F523ULL, "vxw_sundermount_2ga_0_0.vlm"},
	{0x0CE6AD0F2E28465FULL, "vxlt_darktown_5we_0_0.vlm"},
	{0x0CE6B70C00662688ULL, "657684970.ogg"},
	{0x0CE7F5CBBEBF3DE7ULL, "mag120_macha6091369.cl"},
	{0x0CE7FFD23DA74788ULL, "dae000_madam_lusine6154912.cl"},
	{0x0CE8B43FBA5E256DULL, "art\\characters\\creatures\\strider\\cn_hed_str.phy"},
	{0x0CE92B074F98D7BDULL, "mrl350ip_at_demoncave.utp"},
	{0x0CE95CF07603FAE4ULL, "777374292.ogg"},
	{0x0CE9AF61C9ADA8A7ULL, "vxlt_warehouse_18a_0_0.vlm"},
	{0x0CE9E98D3CC439B7ULL, "vxht_keep_5hc_0_0.vlm"},
	{0x0CEA087A1006E05BULL, "703839708.ogg"},
	{0x0CEAB7A064C83BAEULL, "plt_lgt110pt_pier_pressure.nss"},
	{0x0CEAC46CE6339E4CULL, "vxca_caverns1_mines_i8y_0_0.vlm"},
	{0x0CEAC8D8521191C7ULL, "6125653_f.fxe"},
	{0x0CEB48C7BA8AD3DAULL, "467616385.ogg"},
	{0x0CEC64EF851DA310ULL, "hlac_w_woundedcoast_-65532_0y.dds"},
	{0x0CECC3FD359E01D3ULL, "451246302.ogg"},
	{0x0CED0640CE39ECC7ULL, "6083345_m.fxe"},
	{0x0CED27F2CD6811E0ULL, "6077983_m.fxe"},
	{0x0CED7AEE70A53674ULL, "zz_vault_debug6177774.cl"},
	{0x0CED8B721635A011ULL, "llac_lt_foundry_n_320_22.dds"},
	{0x0CED902939B6B391ULL, "da2terrain_rendertodepth_p0.psh.fxo"},
	{0x0CEE6371F9A161EBULL, "hlac_lt_0_lowtown_n_2_1x.dds"},
	{0x0CEEFD985E21D820ULL, "hlac_ca_caverns2_5802_18z.dds"},
	{0x0CEF0A1E347832D6ULL, "qun160_party_comment_26061239.cl"},
	{0x0CEF65DD97936FCDULL, "hlac_lt_darktown_-1_85z.dds"},
	{0x0CEFD1039B72417CULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\ambient\\compressed\\leaning_back_low_enter.gad"},
	{0x0CEFD2CBC86DEB22ULL, "vxlt_backalley_g_1r9_0_0.vlm"},
	{0x0CF074425EC909D1ULL, "6071510_f.fxe"},
	{0x0CF0BFFCD839169AULL, "277687265.ogg"},
	{0x0CF108D30D8D1F6AULL, "gen00fl_varric6087601.cl"},
	{0x0CF1147907EB4820ULL, "394453729.ogg"},
	{0x0CF19CDF4614F5F9ULL, "vxlt_gallowstemplar_n_6qa_0_0.vlm"},
	{0x0CF20D86F793D84AULL, "art\\characters\\playercharacter\\humanmale\\boo_medd.phy"},
	{0x0CF23087DBBDE566ULL, "335939910.ogg"},
	{0x0CF2F9ED9875384BULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_junke_n.dds"},
	{0x0CF348BE4765A5A4ULL, "vfx_cone_lightning_f_01.mao"},
	{0x0CF436F84CE57714ULL, "fxc_lactate_add.mao"},
	{0x0CF49493D97D538BULL, "gen00fl_varric6076516.cl"},
	{0x0CF49E851352C9E3ULL, "zz_followers_banter6159111.cl"},
	{0x0CF4B38BB748E0C2ULL, "195685682.ogg"},
	{0x0CF4C4AA13E12A03ULL, "hlac_w_bonepitdestroyed_a3_0_5x.dds"},
	{0x0CF4CDEB7C3996A1ULL, "vxlt_gallowscourtyard_5a3_0_0.vlm"},
	{0x0CF5745525BE494DULL, "209214089.ogg"},
	{0x0CF594CF9E755787ULL, "vxca_caverns1_mines_dpc_0_0.vlm"},
	{0x0CF598DA6D3E76CBULL, "6154988_m.fxe"},
	{0x0CF63469E94048A1ULL, "vxht_mansion2_325_0_0.vlm"},
	{0x0CF678CD1755FD34ULL, "vxw_encounter3_7v_0_0.vlm"},
	{0x0CF6868A3BDC2C7AULL, "1029820033.ogg"},
	{0x0CF68923581898BBULL, "qun211_braggart6075703.cl"},
	{0x0CF6D0039B785E7AULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\ambient\\compressed\\leaning_back_low_enter.ani"},
	{0x0CF6F2F2D65114E7ULL, "vxw_encounter1_kc_0_0.vlm"},
	{0x0CF79ED08AC95705ULL, "llac_ht_keep_n_3734_0.dds"},
	{0x0CF7C7318E756871ULL, "art\\vfx\\combat\\vfx_c_rw_s1_s2_trans_c.impa.ani"},
	{0x0CF8AE64C49FAC30ULL, "298076912.ogg"},
	{0x0CF8FE71B4E9F871ULL, "6087071_m.fxe"},
	{0x0CF987C06E52CFB7ULL, "hlac_w_blightlands_262147_0z.dds"},
	{0x0CF9AF07A2B8222EULL, "mag160_feynriel6016574.cl"},
	{0x0CF9C1E4D3FD9F43ULL, "art\\characters\\creatures\\highdragon\\animation\\exploration\\idletorelax.ani"},
	{0x0CFA0AEE14A10C5AULL, "art\\characters\\playercharacter\\humanfemale\\mt_ant_hf05.msh"},
	{0x0CFA1CEE14A12AA0ULL, "art\\characters\\playercharacter\\humanfemale\\mt_ant_hf05.mmh"},
	{0x0CFA200709687A26ULL, "vxlt_0_lowtown_docks_a3_d3l_0_0.vlm"},
	{0x0CFA469AB55C3925ULL, "hlac_lt_0_lowtown_a2_n_c_65537_7x.dds"},
	{0x0CFA90E826B04B14ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\basestaff\\combat\\defend.ani"},
	{0x0CFAF190C0B933C6ULL, "creature_core.ncs"},
	{0x0CFAFD6F1CCE220EULL, "hlac_lt_0_lowtown_a2_-65537_6x.dds"},
	{0x0CFB3DB04553AE79ULL, "vxht_0_hightown_a3_bu4_0_0.vlm"},
	{0x0CFB7C7694810061ULL, "combat_smaoe_r2_a5.gda"},
	{0x0CFC32B2885BE33BULL, "pou_rock_sm.mao"},
	{0x0CFC411D4E6A6253ULL, "vxca_caverns1_mines_i7o_0_0.vlm"},
	{0x0CFCF3630FF13C9FULL, "vxlt_0_lowtown_docks_a3_d13_0_0.vlm"},
	{0x0CFD0CA05668F046ULL, "qcr000_arishok6042510.cl"},
	{0x0CFD174948D43922ULL, "vxw_bonepitmining_a2_1hw_0_0.vlm"},
	{0x0CFD3AEDDB5386A6ULL, "6216173_m.fxe"},
	{0x0CFE0EE959DE1B53ULL, "318313198.ogg"},
	{0x0CFE45EAF56082A8ULL, "qun110_petrice6042249.cl"},
	{0x0CFED2318E7B9913ULL, "art\\vfx\\combat\\vfx_c_rw_s1_s2_trans_c.impa.gad"},
	{0x0CFF15146BF78934ULL, "hlac_lt_0_lowtown_docks_a3_n_131070_3y.dds"},
	{0x0CFF29D8F6AB3967ULL, "435466421.ogg"},
	{0x0CFF760F0A6DEFC9ULL, "291178791.ogg"},
	{0x0D0008F67266A297ULL, "art\\levels\\races\\vfx_models\\_textures\\prp_lava1_d.dds"},
	{0x0D0032100FA7E164ULL, "420266332.ogg"},
	{0x0D008CA3AE4B278CULL, "art\\vfx\\cinematic\\vfx_dragon_transform_p.anb"},
	{0x0D00BB096EB5FB39ULL, "w_sundermount_0_0.msh"},
	{0x0D00C5096EB60C27ULL, "w_sundermount_0_0.mmh"},
	{0x0D00C80A302B3E3CULL, "hlac_lt_gallowstemplar_f_7198_1y.dds"},
	{0x0D00C9096EB61374ULL, "w_sundermount_0_0.mao"},
	{0x0D00FF6D08CFCA65ULL, "6176065_m.fxe"},
	{0x0D018DE826B66576ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\basestaff\\combat\\defend.gad"},
	{0x0D02116D195C9097ULL, "1011338781.ogg"},
	{0x0D027093390FB3BDULL, "6125502_m.fxe"},
	{0x0D027D4E4F9AFFA6ULL, "6086981_m.fxe"},
	{0x0D028BF2A5BAD286ULL, "gen00fl_merrill6128757.cl"},
	{0x0D02E4C7B31401C5ULL, "205206019.ogg"},
	{0x0D033E56ECE82896ULL, "vxw_blightlands_10t_0_0.vlm"},
	{0x0D033FDAE5242623ULL, "vxw_sundermount_2mc_0_0.vlm"},
	{0x0D0355BA37316685ULL, "503480832.ogg"},
	{0x0D038F48886E95FFULL, "301302406.ogg"},
	{0x0D04051605147F3DULL, "llac_lt_0_lowtown_a2_0_8.dds"},
	{0x0D0427FD9A228D1DULL, "hlac_w_woundedcoast_262144_0y.dds"},
	{0x0D043E11923A8B6CULL, "hm_patron1.mor"},
	{0x0D04620A5C0911E0ULL, "6097872_f.fxe"},
	{0x0D046252846EA635ULL, "one140cr_charade6128483.cl"},
	{0x0D0503C8A2A24E6AULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\additive\\arm_r_side_down.gad"},
	{0x0D0539C590CCD5E0ULL, "zz_vault_debug6178670.cl"},
	{0x0D0569DD36777652ULL, "l_dae321ar_keep.bnk"},
	{0x0D05C1D30FBF00D2ULL, "vxca_caverns1_mines_i7j_0_0.vlm"},
	{0x0D05CF318E81B378ULL, "art\\vfx\\combat\\vfx_c_rw_s1_s2_trans_c.impa.evt"},
	{0x0D05DB66ADC83814ULL, "933479902.ogg"},
	{0x0D065A643E4624A1ULL, "llac_lt_undercitydungeon_-1_26.dds"},
	{0x0D06931BAD99D9A7ULL, "vxlt_warehouse_12c_0_0.vlm"},
	{0x0D069CC3E83C46D5ULL, "vxlt_0_lowtown_a2_hd8_0_0.vlm"},
	{0x0D06D54BACF03146ULL, "and100_anders6095923.cl"},
	{0x0D071640FABF68BCULL, "art\\levels\\races\\ferelden\\props\\keep\\prp_flagkeep.phy"},
	{0x0D07D9E4D40A053AULL, "art\\characters\\creatures\\highdragon\\animation\\exploration\\idletorelax.evt"},
	{0x0D0808C5E269C823ULL, "6162511_f.fxe"},
	{0x0D0826390D3CF427ULL, "gen00fl_merrill6129551.cl"},
	{0x0D084C2A35B99801ULL, "mcr370_meredith6084422.cl"},
	{0x0D0891E3FB5E92A6ULL, "vxlt_0_lowtown_docks_a3_d4m_0_0.vlm"},
	{0x0D08A8E826BCB1D5ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\basestaff\\combat\\defend.evt"},
	{0x0D08B04CF1DFE939ULL, "art\\vfx\\cinematic\\vfx_orsino_swirly.mmh"},
	{0x0D08D59FB5DBA08AULL, "art\\levels\\races\\ferelden\\props\\higherclass\\prp_gong.mao"},
	{0x0D08D99FB5DBA7D9ULL, "art\\levels\\races\\ferelden\\props\\higherclass\\prp_gong.mmh"},
	{0x0D08E39FB5DBB8C7ULL, "art\\levels\\races\\ferelden\\props\\higherclass\\prp_gong.msh"},
	{0x0D091F880A3752EFULL, "zz_varricarrows.cut"},
	{0x0D09377839AE5B13ULL, "vxlt_gallowstemplar_26e_0_0.vlm"},
	{0x0D0951EFE86933A7ULL, "art\\characters\\playercharacter\\qunarimale\\animation\\blendtree\\qunariarishok_overlay.anb"},
	{0x0D099FF1717ECD80ULL, "6034353_m.fxe"},
	{0x0D09D1872FA2E537ULL, "vxht_0_hightown_c78_0_0.vlm"},
	{0x0D09F4A22CFCADA9ULL, "art\\characters\\playercharacter\\qunarimale\\animation\\combat\\arishock\\ari_ability_whirlwind_en.ani"},
	{0x0D0A0001B1068394ULL, "art\\levels\\races\\dwarf\\props\\_textures\\prp_sword_r.dds"},
	{0x0D0A38FA882D9CFEULL, "vxlt_0_lowtown_a2_n_c_cw0_0_0.vlm"},
	{0x0D0A989E90B274E2ULL, "mag360_alain6065045.cl"},
	{0x0D0ABC01BD2EE65EULL, "6111263_m.fxe"},
	{0x0D0ABC511D4C8FEFULL, "122302588.ogg"},
	{0x0D0B6C6FD35D5FE2ULL, "dre000_varric6027752.cl"},
	{0x0D0B9F63BE3FD53BULL, "lowtown_male_beggar_sta6156759.cl"},
	{0x0D0BE6C8A2A83B01ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\additive\\arm_r_side_down.evt"},
	{0x0D0BE79F134B571AULL, "lt_0_lowtown_docks_161_0.msh"},
	{0x0D0BF59F134B6F53ULL, "lt_0_lowtown_docks_161_0.mao"},
	{0x0D0BF99F134B7660ULL, "lt_0_lowtown_docks_161_0.mmh"},
	{0x0D0C0218860B8C94ULL, "mcr371cr_varric_cut.utc"},
	{0x0D0C1DB66CBFBC02ULL, "qun120_arishok_guard6060605.cl"},
	{0x0D0C23D75930C8F9ULL, "vxlt_0_lowtown_a2_n_hd3_0_0.vlm"},
	{0x0D0C5A7A514A8FFDULL, "zz_qun_debug6028288.cl"},
	{0x0D0CA533EE0257EFULL, "282353507.ogg"},
	{0x0D0CCA0A8D74072EULL, "6130006-mrl3_mir_humanswer2.cut"},
	{0x0D0CDC7BA3761144ULL, "vxht_0_hightown_n_7g0_0_0.vlm"},
	{0x0D0D0972A222F02FULL, "286875252.ogg"},
	{0x0D0DAB963E626DB7ULL, "art\\levels\\4cinematics\\ferelden\\prp_moneypoucha.msh"},
	{0x0D0DBD963E628C1AULL, "art\\levels\\4cinematics\\ferelden\\prp_moneypoucha.mao"},
	{0x0D0DC1963E629329ULL, "art\\levels\\4cinematics\\ferelden\\prp_moneypoucha.mmh"},
	{0x0D0DC748811B41FFULL, "6163436_m.fxe"},
	{0x0D0DD8E4D40E6F2DULL, "art\\characters\\creatures\\highdragon\\animation\\exploration\\idletorelax.gad"},
	{0x0D0DE93187696700ULL, "follower_banter6155862.cl"},
	{0x0D0DF57DF1C11F7DULL, "zz_mcr_debug6071547.cl"},
	{0x0D0E22966158C1E4ULL, "gen00fl_anders6201414.cl"},
	{0x0D0E35BB938C8EAFULL, "ave330ip_chest.utp"},
	{0x0D0E7EFEA3DF7DE5ULL, "l_ser211ar_dungeon.wbd"},
	{0x0D0F3E4C2E82EEE5ULL, "vxw_encounter3_co_0_0.vlm"},
	{0x0D0F6C2F7D7AF5CCULL, "vxht_0_hightown_n_9i4_0_0.vlm"},
	{0x0D0F8DF1E1541BBCULL, "804901655.ogg"},
	{0x0D0FB88B2B4DFE82ULL, "lt_shop_3567_lc.mmh"},
	{0x0D0FF4C138041FABULL, "6182015_m.fxe"},
	{0x0D0FF7AE9F8D19E3ULL, "ico_helm_medium_4.dds"},
	{0x0D100892AD2DD95AULL, "470623730.ogg"},
	{0x0D10B673C226477AULL, "15970770.ogg"},
	{0x0D110FA22D02FA2BULL, "art\\characters\\playercharacter\\qunarimale\\animation\\combat\\arishock\\ari_ability_whirlwind_en.gad"},
	{0x0D111938F52D27E5ULL, "vxht_keep_6bt_0_0.vlm"},
	{0x0D113565E40EC037ULL, "llac_lt_0_lowtown_docks_-131074_2.dds"},
	{0x0D114CD6A3800295ULL, "vxlt_gallowscourtyard_66q_0_0.vlm"},
	{0x0D11B1B7D71A3EA3ULL, "vxw_sundermount_2vb_0_0.vlm"},
	{0x0D11C5AFC1233D3FULL, "zz_fen_debug6134199.cl"},
	{0x0D11CA19ED7D3705ULL, "90488000.ogg"},
	{0x0D11DF7B383024E4ULL, "292289617.ogg"},
	{0x0D121C12170F8A20ULL, "457077796.ogg"},
	{0x0D1265CE46EA39B6ULL, "gen_im_trs_slv_hvy_glv_r1.uti"},
	{0x0D12BBFEEBAC016CULL, "vxw_sundermount_24g_0_0.vlm"},
	{0x0D12BD846356C9CCULL, "vxht_mansion2b_3bl_0_0.vlm"},
	{0x0D12E42AA53F7D03ULL, "mag360_thrask6043657.cl"},
	{0x0D132ED1A8F76A9EULL, "vxht_keep_6a2_0_0.vlm"},
	{0x0D13439608AA7B9BULL, "vxht_mansion_n_4xu_0_0.vlm"},
	{0x0D13976CD6CB185FULL, "art\\vfx\\combat\\vfx_glb_phys_imp_maj_c.mmh"},
	{0x0D13BF5D709B5640ULL, "6172644_m.fxe"},
	{0x0D1422B0C37EA0E8ULL, "gen00fl_varric6093592.cl"},
	{0x0D143F1EB4886D08ULL, "674967087.ogg"},
	{0x0D148B4C03E8DD17ULL, "art\\characters\\creatures\\mubari\\animation\\p_tn90l_wkf.gad"},
	{0x0D14A98800D25980ULL, "fx_snow_wisp.mao"},
	{0x0D14EB82ACB75EAFULL, "dae000ip_crowd_mf3_talk13.utp"},
	{0x0D155B159037894DULL, "518721564.ogg"},
	{0x0D15913E403E8802ULL, "gen_im_trs_krk_low_junk1_03.uti"},
	{0x0D15B1886763F9F4ULL, "zz_followers_banter6159396.cl"},
	{0x0D15E88F9BEA9088ULL, "leap_dot.mao"},
	{0x0D16014DBD190D1DULL, "vxht_0_hightown_n_arx_0_0.vlm"},
	{0x0D163F70F5FF772FULL, "hlac_w_bonepitmining_a2_0_9x.dds"},
	{0x0D166F6ACD619ACEULL, "vxht_0_hightown_a3_bg9_0_0.vlm"},
	{0x0D16A3DF392B9C5EULL, "fen110_fenris6100299.cl"},
	{0x0D16D2BAC8915D53ULL, "art\\vfx\\materials\\vfx_fire_whispy.mao"},
	{0x0D16E3D37F92DD9BULL, "vxlt_gallowsprison_n_6mw_0_0.vlm"},
	{0x0D170CA22D076090ULL, "art\\characters\\playercharacter\\qunarimale\\animation\\combat\\arishock\\ari_ability_whirlwind_en.evt"},
	{0x0D170DD9BA4FFD9FULL, "dre000_bethany6037327.cl"},
	{0x0D171A7D6F423107ULL, "vxlt_shop_ex_0_0.vlm"},
	{0x0D177CDA240777E8ULL, "155744020.ogg"},
	{0x0D178E015FD749CDULL, "410680325.ogg"},
	{0x0D17E75379850F74ULL, "pro000_captain_wright6047467.cl"},
	{0x0D17FAABF99D0E57ULL, "6085743_m.fxe"},
	{0x0D1831D4BCC61274ULL, "6164694_m.fxe"},
	{0x0D196474F105F8F9ULL, "hlac_lt_0_lowtown_n_-65537_4x.dds"},
	{0x0D19DEC8A2B46B50ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\additive\\arm_r_side_down.ani"},
	{0x0D1A506C0181558FULL, "500984108.ogg"},
	{0x0D1ADDA898F597A5ULL, "llac_lt_backalley_n_1_7.dds"},
	{0x0D1B0B2312043344ULL, "vxdr_0_deeproads_fis_0_0.vlm"},
	{0x0D1B1D47BE7D22F1ULL, "art\\levels\\races\\natural\\blightlands\\terrainmats\\ter_blightrock_darelief.dds"},
	{0x0D1B22E82778D18FULL, "wm_ico_lowtowndocks_roll.dds"},
	{0x0D1B6A59E5504E97ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\_textures\\til_ceiling_n.dds"},
	{0x0D1BA04C03EF1FA4ULL, "art\\characters\\creatures\\mubari\\animation\\p_tn90l_wkf.evt"},
	{0x0D1BC32FDFF19766ULL, "6050133_m.fxe"},
	{0x0D1C0330596C9F01ULL, "vxlt_0_lowtown_docks_n_cot_0_0.vlm"},
	{0x0D1C6F642F577B4FULL, "29526030.ogg"},
	{0x0D1CBA33243C6C3BULL, "art\\vfx\\combat\\vfx_w_jug_tremor_lar_aoe_p.impa.ani"},
	{0x0D1CBF6F018A528EULL, "6155380_m.fxe"},
	{0x0D1CF17A10B586ADULL, "hlac_ht_chantry_2534_11x.dds"},
	{0x0D1DC6B01E2FCB3CULL, "mer240_jansen6081436.cl"},
	{0x0D1E5887C397EDE4ULL, "vxdr_0_deeproads_es2_0_0.vlm"},
	{0x0D1F6E2E6404535EULL, "6121035_m.fxe"},
	{0x0D202F6B6171E334ULL, "vxlt_0_lowtown_a2_ewq_0_0.vlm"},
	{0x0D2066525FD283A6ULL, "62735463.ogg"},
	{0x0D2072144AB4693BULL, "34952475.ogg"},
	{0x0D20F1E601164537ULL, "vxlt_gallowstemplar_456_0_0.vlm"},
	{0x0D212DDBDDA219ECULL, "vxw_sundermount_29d_0_0.vlm"},
	{0x0D21716EFBC09FECULL, "art\\characters\\playercharacter\\humanfemale\\textures\\hf_cth_nobe_0n.dds"},
	{0x0D21A0AE9AED831EULL, "vxht_keep_6j3_0_0.vlm"},
	{0x0D220B82B85A1822ULL, "78022508.ogg"},
	{0x0D222448FC0D6C3EULL, "art\\vfx\\combat\\vfx_gui_ground_ring_m_p.mmh"},
	{0x0D222C1F3DE2CDD3ULL, "vxlt_blackemporium_152_0_0.vlm"},
	{0x0D22A94469611370ULL, "770749277.ogg"},
	{0x0D22EE8C7EB055A4ULL, "w_bonepitdestroyed_a3_192_0.mmh"},
	{0x0D22F28C7EB05CF7ULL, "w_bonepitdestroyed_a3_192_0.mao"},
	{0x0D23048C7EB07B46ULL, "w_bonepitdestroyed_a3_192_0.msh"},
	{0x0D2324345B4652A4ULL, "vxca_caverns2prc_8kw_0_1.vlm"},
	{0x0D232CA5D0D0D1D5ULL, "isa311_isabela6091720.cl"},
	{0x0D24924BD54B6B7FULL, "166663230.ogg"},
	{0x0D24EBBCD4D33D6BULL, "gen_im_trs_apo_sta_cld_r0.uti"},
	{0x0D25229D6CF5EC54ULL, "vxca_caverns1_mines_i1k_0_1.vlm"},
	{0x0D2549B6D17BA13FULL, "lt_0_lowtown_docks_n_1_0.anb"},
	{0x0D2555D5D88C5E4AULL, "vxlt_0_lowtown_a2_n_f1i_0_0.vlm"},
	{0x0D259C3E6B132040ULL, "571184297.ogg"},
	{0x0D25BAEBE0D7D70FULL, "vxht_0_hightown_a3_au9_0_0.vlm"},
	{0x0D2685E2C9622B3CULL, "dae300_petrice6097866.cl"},
	{0x0D2726104203E8CDULL, "zz_isa_debug6133994.cl"},
	{0x0D272F77CC9A8FAAULL, "and100_anders6095879.cl"},
	{0x0D27770C4FA06059ULL, "zz_qun_debug6028275.cl"},
	{0x0D28B84C03F9D29DULL, "art\\characters\\creatures\\mubari\\animation\\p_tn90l_wkf.ani"},
	{0x0D2922B9350EFD7EULL, "6120357_m.fxe"},
	{0x0D293279268BC2A5ULL, "vxht_mansion2b_1kh_0_0.vlm"},
	{0x0D298C28AD7FF550ULL, "vxw_bonepit_zn_0_0.vlm"},
	{0x0D29C23324470432ULL, "art\\vfx\\combat\\vfx_w_jug_tremor_lar_aoe_p.impa.evt"},
	{0x0D29D40EF94BB649ULL, "follower_banter6147413.cl"},
	{0x0D2A1A9335C72533ULL, "gen00fl_anders6114481.cl"},
	{0x0D2AAEE7E22BCC64ULL, "llac_ht_chantry_n_2534_35.dds"},
	{0x0D2BDDFFD8FE4561ULL, "vxca_caverns1_mines_djf_0_1.vlm"},
	{0x0D2BF5ED9334FA0DULL, "weaponparts.gda"},
	{0x0D2C0858C44D0356ULL, "6132871_m.fxe"},
	{0x0D2C0A1BA86168B0ULL, "w_sundermount_197_0.msh"},
	{0x0D2C181BA86180CAULL, "w_sundermount_197_0.mmh"},
	{0x0D2C1C1BA8618701ULL, "w_sundermount_197_0.mao"},
	{0x0D2D1E3DC4CF4F73ULL, "hlac_lt_0_lowtown_a2_n_c_2_1x.dds"},
	{0x0D2D225195FC3A35ULL, "6207415_m.fxe"},
	{0x0D2D25ABFA91D666ULL, "6096226_f.fxe"},
	{0x0D2DC1D1ADB6DD74ULL, "6088836_f.fxe"},
	{0x0D2DC7788D993FE0ULL, "6111654_m.fxe"},
	{0x0D2DFCF2D91958E5ULL, "vxht_keep_6hr_0_0.vlm"},
	{0x0D2DFFAD23E8AF89ULL, "529531967.ogg"},
	{0x0D2E03F62F95FD69ULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_scroll_sa.dds"},
	{0x0D2E04912080EFACULL, "pro000_opening6207382.cl"},
	{0x0D2E1DE545597CB1ULL, "vxlt_0_lowtown_a2_d66_0_0.vlm"},
	{0x0D2E3AA5FD636A23ULL, "6126810_f.fxe"},
	{0x0D2E53AB123B764AULL, "372827005.ogg"},
	{0x0D2E71769A39B107ULL, "277212910.ogg"},
	{0x0D2E7348A886A49CULL, "dae000_mother6099233.cl"},
	{0x0D2EAE5ABB99EF7DULL, "llac_dr_slavers_3215_6.dds"},
	{0x0D2EC78D570721C2ULL, "vxht_keep_7ac_0_0.vlm"},
	{0x0D2F01C5FE2A0B3FULL, "vxw_sundermount_23i_0_0.vlm"},
	{0x0D300348FC1971FDULL, "art\\vfx\\combat\\vfx_gui_ground_ring_m_p.anb"},
	{0x0D30AD4CF201362EULL, "art\\vfx\\cinematic\\vfx_orsino_swirly.anb"},
	{0x0D30E133244D57C5ULL, "art\\vfx\\combat\\vfx_w_jug_tremor_lar_aoe_p.impa.gad"},
	{0x0D31AA8165994604ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\_1_-_default__3ww_basic_.mao"},
	{0x0D31C07137816CBAULL, "fen110_fenris6100264.cl"},
	{0x0D326F41CE3A76D1ULL, "187159217.ogg"},
	{0x0D327ECF61B36432ULL, "dae000_arishok6075347.cl"},
	{0x0D3330EEEE08479FULL, "mag291_torpor6026991.cl"},
	{0x0D334237087E2207ULL, "qcr220st_meredith.stg"},
	{0x0D33C5096EE16584ULL, "w_sundermount_0_0.rml"},
	{0x0D33E11A26693A74ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\boss\\meredith\\combat\\attackversion3.ani"},
	{0x0D3433B8FE0452DBULL, "hlac_lt_0_lowtown_131072_0z.dds"},
	{0x0D3449ADB67DD0A4ULL, "static_static_p49.psh.fxo"},
	{0x0D34F1E9FEB24ACEULL, "vxht_keep_586_0_0.vlm"},
	{0x0D356C41B153DCF1ULL, "vxht_mansion2b_30g_0_0.vlm"},
	{0x0D366719CAB8D845ULL, "hlac_dr_dungeon_3216_9z.dds"},
	{0x0D3686105484A39CULL, "llac_w_encounter3_0_5.dds"},
	{0x0D36995B4BEDF31BULL, "hlac_ca_caverns1_24421_7x.dds"},
	{0x0D387B6F7C074EFEULL, "vxlt_0_lowtown_a2_n_c_fek_0_0.vlm"},
	{0x0D38882B1B13338DULL, "6064835_f.fxe"},
	{0x0D3889FEDD2D288FULL, "mcr310_argument6127609.cl"},
	{0x0D38A091417956DAULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\4so_leftkeep_3.mao"},
	{0x0D3956FB98B6C540ULL, "vxca_caverns1_i2z_0_0.vlm"},
	{0x0D399AEFCCAC545FULL, "322226114.ogg"},
	{0x0D39DE1A266DA0D6ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\boss\\meredith\\combat\\attackversion3.gad"},
	{0x0D3B986CD6EC6B9CULL, "art\\vfx\\combat\\vfx_glb_phys_imp_maj_c.anb"},
	{0x0D3BC39E0A965C94ULL, "352690038.ogg"},
	{0x0D3BCC759D4A4DA4ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\pose_fa_transition.ani"},
	{0x0D3BDC9024CF4100ULL, "mag291_demon_marethari6029062.cl"},
	{0x0D3D1A7795D98910ULL, "follower_banter6148455.cl"},
	{0x0D3D73A2F02023BFULL, "vxw_sundermount_24h_0_0.vlm"},
	{0x0D3D97C7E8F96220ULL, "635637565.ogg"},
	{0x0D3DA421577D632FULL, "vxlt_0_lowtown_n_exh_0_0.vlm"},
	{0x0D3DB88ABFA18C6DULL, "atl_newcontent_dxt5_dat.xml"},
	{0x0D3E819785B82366ULL, "vxlt_gallowstemplar_f_436_0_0.vlm"},
	{0x0D3F1758F6DF43F5ULL, "165323823.ogg"},
	{0x0D3FA6BF14AA8DB6ULL, "vxht_mansion2_30d_0_0.vlm"},
	{0x0D3FE15FDB5E36ECULL, "llac_ht_chantry_n_2534_6.dds"},
	{0x0D40207A21CF747CULL, "828964641.ogg"},
	{0x0D40B4D642382E0EULL, "one280_merin6104375.cl"},
	{0x0D40E495E8549C96ULL, "zz_mag_debug6039837.cl"},
	{0x0D40F91A2673ED35ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\boss\\meredith\\combat\\attackversion3.evt"},
	{0x0D40FFD93F01CD83ULL, "ser221_gamlen6133082.cl"},
	{0x0D41391A00D69FA2ULL, "zz_isa_debug6133173.cl"},
	{0x0D41C9759D4EB406ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\pose_fa_transition.gad"},
	{0x0D4204503C4805D6ULL, "544623994.ogg"},
	{0x0D420D72229148B9ULL, "vxlt_0_lowtown_a2_n_ehr_0_0.vlm"},
	{0x0D4263BBE4AB4B22ULL, "vxw_bonepitmining_a2_204_0_0.vlm"},
	{0x0D42B5744CAFFC7AULL, "vxdr_0_deeproads_cqt_0_0.vlm"},
	{0x0D42D5E362C63C4BULL, "rock.dds"},
	{0x0D42ECBB9E8888D9ULL, "44934890.ogg"},
	{0x0D436261A89EDE94ULL, "vxca_cavern1_haunted_j28_0_0.vlm"},
	{0x0D43EA34D56EF033ULL, "14909910.ogg"},
	{0x0D43F04C80B0061BULL, "art\\characters\\playercharacter\\elfmale\\textures\\em_rob_enco_0s.dds"},
	{0x0D4439B831A47CD8ULL, "w_encounter2_camp_7_0.phy"},
	{0x0D445281DEA8E23CULL, "fen110_fenris6100356.cl"},
	{0x0D4471DE54D1D6CEULL, "28864596.ogg"},
	{0x0D447CFDCAA3CD46ULL, "art\\vfx\\combat\\vfx_m_sta_atk_close_fin_c.impa.ani"},
	{0x0D4481729AF73B3DULL, "6067671_m.fxe"},
	{0x0D4483CFF62ECA95ULL, "291056774.ogg"},
	{0x0D44B1DFC13B4CCEULL, "vxht_0_hightown_a3_aut_0_0.vlm"},
	{0x0D451CED9E0C1E99ULL, "219253494.ogg"},
	{0x0D455B7D56216867ULL, "gen_im_trs_apo_lgt_cht_r1.uti"},
	{0x0D45A3582F8F44F7ULL, "6016642_m.fxe"},
	{0x0D45A8E888B45943ULL, "6028852_m.fxe"},
	{0x0D45FC574F922B48ULL, "mer320_hubert6090371.cl"},
	{0x0D46462168E8650AULL, "825896411.ogg"},
	{0x0D46EFD0A3804040ULL, "vxw_encounter3_61_0_0.vlm"},
	{0x0D475A4AC096BDB5ULL, "hlac_w_bonepit_393222_0x.dds"},
	{0x0D477124C9FFC710ULL, "vxlt_gallowstemplar_n_53a_0_0.vlm"},
	{0x0D487AA5C0E14B16ULL, "892301746.ogg"},
	{0x0D48860CFE8056A8ULL, "vxlt_gallowstemplar_5mp_0_0.vlm"},
	{0x0D48997CE36045D6ULL, "cn_bdy_abma_0t.dds"},
	{0x0D48C4759D54CAE5ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\pose_fa_transition.evt"},
	{0x0D48E73C1DFC053AULL, "hm_qmq_visguard2.mor"},
	{0x0D498C577498F854ULL, "759558047.ogg"},
	{0x0D49CA84FC782780ULL, "702204230.ogg"},
	{0x0D49D3E90B1B2EECULL, "6106508_m.fxe"},
	{0x0D49F1B0F7B343DDULL, "w_encounter2.arl"},
	{0x0D4A1AEEDBEA24DCULL, "6116754_m.fxe"},
	{0x0D4AE50C25811C73ULL, "hlac_lt_0_lowtown_docks_a3_n_-2_0z.dds"},
	{0x0D4B0DD676FE6EECULL, "vxw_encounter2_53_0_0.vlm"},
	{0x0D4B38B831AA99F0ULL, "w_encounter2_camp_7_0.rml"},
	{0x0D4B8CF4B834A603ULL, "513759809.ogg"},
	{0x0D4BCC6CDA83626EULL, "314276122.ogg"},
	{0x0D4BFD50E38FCD56ULL, "hlac_ht_brothel_n_3370_1y.dds"},
	{0x0D4D000272B673AAULL, "vxlt_0_lowtown_a3_n_c_hsv_0_0.vlm"},
	{0x0D4D097EE01C4593ULL, "hlac_ht_0_hightown_n_65536_4z.dds"},
	{0x0D4D146D23762047ULL, "hlac_dr_primevaltomb_659_4y.dds"},
	{0x0D4D5BB898A017B2ULL, "87059104.ogg"},
	{0x0D4D6453912F1105ULL, "lt_0_lowtown_a2_n_c_mmap.dds"},
	{0x0D4E13E8CBB42738ULL, "art\\characters\\playercharacter\\humanmale\\hlm_wic.mao"},
	{0x0D4E1724AEEDAECDULL, "875590014.ogg"},
	{0x0D4E58FEA415208FULL, "l_ser211ar_dungeon.bnk"},
	{0x0D4EA4842063BD23ULL, "gen00fl_varric6118832.cl"},
	{0x0D4F044A525400C1ULL, "dr_0_deeproads_5842.anb"},
	{0x0D4FB5691632788BULL, "84502893.ogg"},
	{0x0D501576DFBC085EULL, "vxw_bonepit_220_0_0.vlm"},
	{0x0D504723236BA6F2ULL, "vxht_0_hightown_c35_0_0.vlm"},
	{0x0D5063DA0F2E6162ULL, "vxht_mansion2_1vd_0_0.vlm"},
	{0x0D50CF18F55F5D6FULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_containerb_sa.dds"},
	{0x0D510184AFEEAA25ULL, "and100_anders6095957.cl"},
	{0x0D51288E2256F75CULL, "gen00fl_varric6103266.cl"},
	{0x0D51433EB91490D0ULL, "gen00fl_merrill6108436.cl"},
	{0x0D516756EE71A4C6ULL, "vxht_brothel_n_6d9_0_0.vlm"},
	{0x0D51A611B14885D6ULL, "6040984_f.fxe"},
	{0x0D51AD23DF26C553ULL, "367348365.ogg"},
	{0x0D51D4D60B336D69ULL, "art\\levels\\races\\natural\\blightlands\\props\\prp_03ruinsrock.phy"},
	{0x0D521F3BFD8E420AULL, "6017899_m.fxe"},
	{0x0D5264FDCAAFE1B7ULL, "art\\vfx\\combat\\vfx_m_sta_atk_close_fin_c.impa.evt"},
	{0x0D5301A904DBDFBAULL, "152333328.ogg"},
	{0x0D53666E5759E220ULL, "6080206_m.fxe"},
	{0x0D536DE8BFACF1AEULL, "vxw_encounter2_g7_0_0.vlm"},
	{0x0D538C5AAC5D2610ULL, "710937593.ogg"},
	{0x0D53F490D6816BD0ULL, "441054822.ogg"},
	{0x0D5418BFAE78BFDDULL, "vxlt_0_lowtown_n_e3g_0_0.vlm"},
	{0x0D54457951E438B3ULL, "one300_zevran6089732.cl"},
	{0x0D545764E00573FAULL, "vxht_0_hightown_n_7lo_0_0.vlm"},
	{0x0D549F76D3A929A2ULL, "6148799_m.fxe"},
	{0x0D54C42AACA067B9ULL, "gen00fl_anders6111251.cl"},
	{0x0D54E1BF83623ED0ULL, "vxw_bonepit_km_0_0.vlm"},
	{0x0D551A415C715A36ULL, "5547625.ogg"},
	{0x0D55568443FDE5B1ULL, "qun221_viscount6084732.cl"},
	{0x0D555AB8224D350BULL, "art\\characters\\playercharacter\\dwarfmale\\animation\\cindes\\varric\\sitting_gesture_right.ani"},
	{0x0D5568CA2264D281ULL, "679419009.ogg"},
	{0x0D55C90A1F0D8876ULL, "6067979_m.fxe"},
	{0x0D55F5B1BFD1902BULL, "6071249_m.fxe"},
	{0x0D560F4A8E2F51C8ULL, "32106170.ogg"},
	{0x0D563AB57CA2F640ULL, "vxca_caverns1_i8x_0_0.vlm"},
	{0x0D56504650E6A2B7ULL, "hlac_ca_caverns1_24426_6z.dds"},
	{0x0D56B2D4E7D57D9BULL, "134875555.ogg"},
	{0x0D571C3512DB17B6ULL, "vxht_0_hightown_n_bxr_0_0.vlm"},
	{0x0D578B24D37B95F1ULL, "mrl000_marethari6129200.cl"},
	{0x0D578FD7814D017AULL, "204321691.ogg"},
	{0x0D57F44F8326575AULL, "hlac_ca_caverns1_24425_8x.dds"},
	{0x0D5837F1F82B959AULL, "680205625.ogg"},
	{0x0D5888CDAA01BB67ULL, "trap_impact.ncs"},
	{0x0D588EE8A9FEBC3FULL, "w_encounter3_26_0.msh"},
	{0x0D5894E8A9FEC6E1ULL, "w_encounter3_26_0.mmh"},
	{0x0D58A02BB1BC7B97ULL, "gen_im_cns_tom_war_templar.uti"},
	{0x0D58A0E8A9FEDA82ULL, "w_encounter3_26_0.mao"},
	{0x0D58C0650B2B8374ULL, "354240015.ogg"},
	{0x0D5915DD2A5807C1ULL, "lgt210tr_frame_flower.ncs"},
	{0x0D592AD72C9754BFULL, "mcr371_hostile_fenris6082245.cl"},
	{0x0D5965FDCAB60240ULL, "art\\vfx\\combat\\vfx_m_sta_atk_close_fin_c.impa.gad"},
	{0x0D59961A778C3416ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\0wo_wallcap.msh"},
	{0x0D59A01A778C4514ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\0wo_wallcap.mmh"},
	{0x0D59A41A778C4C27ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\0wo_wallcap.mao"},
	{0x0D59CF21EB6DD9CAULL, "gen00fl_merrill6129880.cl"},
	{0x0D5A2C45E7C568C0ULL, "dae000_gamlen6145090.cl"},
	{0x0D5AC5B1351C35A0ULL, "847318740.ogg"},
	{0x0D5ACF3CD223DBBFULL, "12239770.ogg"},
	{0x0D5B1C1A4426BE25ULL, "326540867.ogg"},
	{0x0D5BD23F40F8A523ULL, "6134577_m.fxe"},
	{0x0D5BDC1BD365913FULL, "564612085.ogg"},
	{0x0D5C1DFDE1CA25D0ULL, "vxlt_0_lowtown_n_f5c_0_0.vlm"},
	{0x0D5C4EE2475922B9ULL, "dre000_bethany6037314.cl"},
	{0x0D5C57515C6D167FULL, "lowtown_fem_prostitute_sta6152368.cl"},
	{0x0D5C6C3E298F80BDULL, "vxht_mansion2_22l_0_0.vlm"},
	{0x0D5CF8A20A8DC4ADULL, "6095577_m.fxe"},
	{0x0D5D0A988109389FULL, "art\\cinematics\\_crit path\\mage crisis\\mcr370cs_attack_begins\\max files\\mage4_.gad"},
	{0x0D5D38B38FA8EDA1ULL, "vxw_encounter3_86_0_0.vlm"},
	{0x0D5DC7E1442D30CAULL, "6171287_m.fxe"},
	{0x0D5E47EED1A3ED67ULL, "vxlt_gallowsprison_n_5rq_0_0.vlm"},
	{0x0D5E4B935EBB8381ULL, "vxlt_0_lowtown_a2_n_gmg_0_0.vlm"},
	{0x0D5EAA7967F6E50DULL, "vxlt_hangedman_16y_0_0.vlm"},
	{0x0D5EBCC0B1BD5221ULL, "588099965.ogg"},
	{0x0D5EF71F20736525ULL, "fen210_fenris6109438.cl"},
	{0x0D5F1CEA10089884ULL, "vxca_caverns1_iys_0_0.vlm"},
	{0x0D5FFAB28ED5C5A7ULL, "hlac_w_encounter2_-2_0z.dds"},
	{0x0D611A7F02B0110AULL, "420251112.ogg"},
	{0x0D622A6C0C38FEC0ULL, "200890685.ogg"},
	{0x0D62A20E8830887DULL, "qun231_party_comment6077006.cl"},
	{0x0D62CC5153988A3AULL, "gen_im_npc_arm_hum_glv_rdra.uti"},
	{0x0D6315CBAF16BC33ULL, "vxht_mansion2_2lp_0_0.vlm"},
	{0x0D6336AB56BFDAD0ULL, "mrl210_merrill6102102.cl"},
	{0x0D6342B822594982ULL, "art\\characters\\playercharacter\\dwarfmale\\animation\\cindes\\varric\\sitting_gesture_right.evt"},
	{0x0D6349B6054C2E31ULL, "vxlt_gallowscourtyard_72t_0_0.vlm"},
	{0x0D63535CF33A3066ULL, "mag101_grace6170862.cl"},
	{0x0D63E49506AF1959ULL, "vxw_woundedcoast_109_0_0.vlm"},
	{0x0D6472A3B03A7B64ULL, "51228501.ogg"},
	{0x0D655211F875E4E6ULL, "llac_lt_0_lowtown_n_-131073_2.dds"},
	{0x0D6589F033E19821ULL, "491364466.ogg"},
	{0x0D65902BABC375D8ULL, "w_encounter2_56_0.phy"},
	{0x0D660FB398448300ULL, "553369298.ogg"},
	{0x0D6627059AEFAE89ULL, "mag360_samson6044920.cl"},
	{0x0D66535C229EC164ULL, "6036917_m.fxe"},
	{0x0D6691D87D666864ULL, "vxdr_0_deeproads_eh7_0_0.vlm"},
	{0x0D67C0F9AD200F48ULL, "hlac_w_encounter3_-196607_0y.dds"},
	{0x0D683F56F63B779CULL, "hlac_lt_warehouse_0_4y.dds"},
	{0x0D68811EA86FE388ULL, "6179999_m.fxe"},
	{0x0D691BB9FB1D3F35ULL, "166962717.ogg"},
	{0x0D6975A9FBFF76D7ULL, "32185918.ogg"},
	{0x0D69AD9135462D84ULL, "204559564.ogg"},
	{0x0D69EEC3B46F0E9BULL, "vxht_mansion_n_4fk_0_0.vlm"},
	{0x0D6A28297AB085E5ULL, "vxdr_primevalentrance_3df_0_0.vlm"},
	{0x0D6A41B8225F6695ULL, "art\\characters\\playercharacter\\dwarfmale\\animation\\cindes\\varric\\sitting_gesture_right.gad"},
	{0x0D6A6A17FFF3787AULL, "vxht_chantry_2ct_0_0.vlm"},
	{0x0D6A8DA4FC097891ULL, "llac_ca_caverns1_24424_0.dds"},
	{0x0D6A8EFE9CDB1B25ULL, "vxlt_0_lowtown_a2_n_gwy_0_0.vlm"},
	{0x0D6B39FDD6E8D7BFULL, "mag110_arianni6206382.cl"},
	{0x0D6B7D4DE9507A85ULL, "1055589583.ogg"},
	{0x0D6BB8A2EA8A5E12ULL, "root_rog_general4.dds"},
	{0x0D6BDA9FB63026C6ULL, "art\\levels\\races\\ferelden\\props\\higherclass\\prp_gong.phy"},
	{0x0D6C201B646CF536ULL, "hlac_lt_gallowstemplar_f_2394_3z.dds"},
	{0x0D6C8F2BABC992F0ULL, "w_encounter2_56_0.rml"},
	{0x0D6CA160971FC009ULL, "1021593448.ogg"},
	{0x0D6CF3549F7A7B0EULL, "art\\vfx\\combat\\vfx_sta_rng_atk_fin_c.impa.evt"},
	{0x0D6DC45199122502ULL, "art\\characters\\creatures\\gallowsgateguard\\animation\\custom enter\\second_enter_lp.gad"},
	{0x0D6DFA07CE0EC245ULL, "gen00fl_fenris6126800.cl"},
	{0x0D6DFC38C6F0682EULL, "88363150.ogg"},
	{0x0D6E51ABDAA95229ULL, "hlac_w_bonepitmining_a2_65539_0y.dds"},
	{0x0D6E95969D9BA29BULL, "616687708.ogg"},
	{0x0D6E9A67679A62F7ULL, "6074475_m.fxe"},
	{0x0D6EAA9F64965C20ULL, "vxca_caverns1_mines_igx_0_0.vlm"},
	{0x0D6EFF37B42D414FULL, "hlac_lt_0_lowtown_a2_n_c_131070_2y.dds"},
	{0x0D6F7C084B82E1EBULL, "gen_im_trs_gen_med_hlm_r2.uti"},
	{0x0D6FA53B1DCD39FDULL, "art\\levels\\races\\natural\\proxy\\props\\_textures\\jagged_sa.dds"},
	{0x0D6FC783481B542EULL, "vxht_0_hightown_a2_ass_0_0.vlm"},
	{0x0D6FE2C406011A53ULL, "vxlt_house_ep_0_0.vlm"},
	{0x0D703EF7264D8786ULL, "vxw_woundedcoast_2e_0_0.vlm"},
	{0x0D70A24B36AAEB46ULL, "357755680.ogg"},
	{0x0D70A66EED9C2640ULL, "vxlt_0_lowtown_a2_ep0_0_0.vlm"},
	{0x0D70D6ABE7604D5AULL, "humanfemale_b.fxa"},
	{0x0D71B70677D6E5A8ULL, "eye.mat"},
	{0x0D71E798811B59E5ULL, "art\\cinematics\\_crit path\\mage crisis\\mcr370cs_attack_begins\\max files\\mage4_.ani"},
	{0x0D724A1669FADDDDULL, "vxht_0_hightown_91n_0_0.vlm"},
	{0x0D72E641A4FA7237ULL, "vxht_brothel_n_5gn_0_0.vlm"},
	{0x0D7309884ECA4EC0ULL, "mag311_keran6054391.cl"},
	{0x0D7309E191030AEAULL, "vxca_caverns1_mines_gif_0_0.vlm"},
	{0x0D73D0DFE507ED86ULL, "612909311.ogg"},
	{0x0D73DD784061E9F9ULL, "vxca_cavern1_haunted_imo_0_0.vlm"},
	{0x0D73EA549F808AB9ULL, "art\\vfx\\combat\\vfx_sta_rng_atk_fin_c.impa.gad"},
	{0x0D73FFEEF6C748B6ULL, "vxht_0_hightown_n_bnt_0_0.vlm"},
	{0x0D742A564FB212E2ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\injured_supported_malepc.gad"},
	{0x0D748305D78A4511ULL, "hlac_ca_caverns2prc_5802_30x.dds"},
	{0x0D74B5B0FCB773E0ULL, "6046284_m.fxe"},
	{0x0D74C7A120DB8710ULL, "6054752_f.fxe"},
	{0x0D74CC58548EC279ULL, "vxht_0_hightown_n_4of_0_0.vlm"},
	{0x0D7539BD15F9A7F5ULL, "752986954.ogg"},
	{0x0D75F16074253C67ULL, "6163133_m.fxe"},
	{0x0D75FB0981CC0C04ULL, "6142849_m.fxe"},
	{0x0D762D8502D4484CULL, "638432666.ogg"},
	{0x0D76A78E949755A5ULL, "qcr220_orsino.bnk"},
	{0x0D76EA3D016DBFBAULL, "llac_lt_0_lowtown_a2_-196608_4.dds"},
	{0x0D76FAB0F7DA408BULL, "w_encounter2.rml"},
	{0x0D773BB853BE4F9CULL, "vxw_woundedcoast_19o_0_0.vlm"},
	{0x0D7787A82D3AA5DAULL, "vxht_mansion_n_60z_0_0.vlm"},
	{0x0D7817681C58E765ULL, "6176216_m.fxe"},
	{0x0D7860B1E1571D49ULL, "art\\characters\\creatures\\highdragon\\animation\\combat\\backrightlegattackv1.ani"},
	{0x0D78E0078AFCBFD8ULL, "vxht_bartrand_n_64g_0_0.vlm"},
	{0x0D78E1CE69C4EF65ULL, "gen_im_cns_cft_arn_03.uti"},
	{0x0D790857E554C23FULL, "gen00fl_dog6165769.cl"},
	{0x0D794FF80D7BB1BDULL, "vxht_mansion2_28j_0_0.vlm"},
	{0x0D796379C755B8D2ULL, "vxlt_house_eu_0_0.vlm"},
	{0x0D796FC4509B0451ULL, "427850575.ogg"},
	{0x0D798841E7EEFA6DULL, "547799581.ogg"},
	{0x0D79888E4C98D0BDULL, "6125713_m.fxe"},
	{0x0D79898E949958DDULL, "qcr220_orsino.cnv"},
	{0x0D79BFFCFDCD56A0ULL, "vxht_mansion2b_31i_0_0.vlm"},
	{0x0D7A4D564FB6B9D9ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\injured_supported_malepc.evt"},
	{0x0D7A78260F284901ULL, "dae000_mother6035922.cl"},
	{0x0D7A8FD19E7FD20EULL, "art\\characters\\playercharacter\\qunarimale\\animation\\combat\\commander\\comm_ability_buff.ani"},
	{0x0D7AA861D5CE7D58ULL, "849173354.ogg"},
	{0x0D7AD8D271667083ULL, "docks_male_elven_worker_sta6175809.cl"},
	{0x0D7AFB549F86C607ULL, "art\\vfx\\combat\\vfx_sta_rng_atk_fin_c.impa.ani"},
	{0x0D7B5E50B82DAABAULL, "vxht_mansion2b_32z_0_0.vlm"},
	{0x0D7B726B92D53DCAULL, "6031994_m.fxe"},
	{0x0D7BB0C79CFCA7E4ULL, "6145164_f.fxe"},
	{0x0D7BB970E510AAE2ULL, "vxht_mansion2_1gg_0_0.vlm"},
	{0x0D7C00A3F3F4C984ULL, "vxca_caverns1_igq_0_0.vlm"},
	{0x0D7C0AC5AA2C31B9ULL, "qun111_ketojan6041031.cl"},
	{0x0D7C181D9488A40BULL, "6110851_f.fxe"},
	{0x0D7C56C5E6E1F7AFULL, "472868899.ogg"},
	{0x0D7CE5B79A34D16EULL, "6035203_f.fxe"},
	{0x0D7D74E8AA1DAC0AULL, "w_encounter3_26_0.rml"},
	{0x0D7DBDD60B58808EULL, "art\\levels\\races\\natural\\blightlands\\props\\prp_03ruinsrock.mmh"},
	{0x0D7DC9D60B589525ULL, "art\\levels\\races\\natural\\blightlands\\props\\prp_03ruinsrock.mao"},
	{0x0D7DD7D60B58AD7CULL, "art\\levels\\races\\natural\\blightlands\\props\\prp_03ruinsrock.msh"},
	{0x0D7E1E1B5350F1CFULL, "6103610_m.fxe"},
	{0x0D7E6D42C7EDF9C5ULL, "art\\vfx\\environment\\vfx_tiki_fire.anb"},
	{0x0D7E7BB1E15BB6CBULL, "art\\characters\\creatures\\highdragon\\animation\\combat\\backrightlegattackv1.gad"},
	{0x0D7E9FC26D281E2BULL, "438560726.ogg"},
	{0x0D7ECE7F52D44791ULL, "llac_w_sundermount_-196614_0.dds"},
	{0x0D7F59CECC1D9670ULL, "playersoundsets.gda"},
	{0x0D7F682A2BE12DE1ULL, "6131579_f.fxe"},
	{0x0D7F68F13E143640ULL, "qun110_petrice6042252.cl"},
	{0x0D7F9B8450341DB9ULL, "885229209.ogg"},
	{0x0D80318E67575D79ULL, "hlac_lt_0_lowtown_a2_n_65534_6x.dds"},
	{0x0D8060432EC35693ULL, "6053470_enter_keep.cut"},
	{0x0D812559D76CB68DULL, "157354112.ogg"},
	{0x0D81581E96F08AB7ULL, "vxht_brothel_n_5ji_0_0.vlm"},
	{0x0D81A4D0C4B998FCULL, "339382718.ogg"},
	{0x0D8271CBE8BD6136ULL, "vxht_0_hightown_n_biu_0_0.vlm"},
	{0x0D82BC17DA9DEB6EULL, "vxw_sundermount_30y_0_1.vlm"},
	{0x0D82D80E56261CFBULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\prp_dooriron1h.mmh"},
	{0x0D82E40E56263098ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\prp_dooriron1h.mao"},
	{0x0D82EF519924CAE8ULL, "art\\characters\\creatures\\gallowsgateguard\\animation\\custom enter\\second_enter_lp.ani"},
	{0x0D82F60E56264FE5ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\prp_dooriron1h.msh"},
	{0x0D8307F7A4D222E0ULL, "vxca_caverns1_eib_0_1.vlm"},
	{0x0D83278DEEAD8C60ULL, "6041384_m.fxe"},
	{0x0D832A83DFE1B230ULL, "art\\characters\\playercharacter\\humanmale\\arm_medd.mao"},
	{0x0D832E83DFE1B943ULL, "art\\characters\\playercharacter\\humanmale\\arm_medd.mmh"},
	{0x0D833C83DFE1D09DULL, "art\\characters\\playercharacter\\humanmale\\arm_medd.msh"},
	{0x0D84093FDD011B98ULL, "vxw_encounter2_ox_0_1.vlm"},
	{0x0D8528A6691B056CULL, "vxw_blightlands_m8_0_0.vlm"},
	{0x0D8540DE5061FE50ULL, "199531163.ogg"},
	{0x0D8577240BE701EAULL, "963662978.ogg"},
	{0x0D8578B1E161D030ULL, "art\\characters\\creatures\\highdragon\\animation\\combat\\backrightlegattackv1.evt"},
	{0x0D860E3086A7EAB9ULL, "vxca_caverns1_mines_i0x_0_0.vlm"},
	{0x0D861BB5D442B285ULL, "1063252336.ogg"},
	{0x0D8640DC4A4296F5ULL, "6107182_m.fxe"},
	{0x0D86EF51D9C4D3A1ULL, "vxlt_0_lowtown_a2_er7_0_0.vlm"},
	{0x0D874DD1E3DFA97AULL, "vxht_chantry_2yv_0_0.vlm"},
	{0x0D8787D19E8A4F5FULL, "art\\characters\\playercharacter\\qunarimale\\animation\\combat\\commander\\comm_ability_buff.evt"},
	{0x0D879AE6F9B582C2ULL, "gen00fl_merrill6115872.cl"},
	{0x0D87B5046D31ED48ULL, "6072719_m.fxe"},
	{0x0D87D39E6EE51A40ULL, "vxht_0_hightown_n_b26_0_0.vlm"},
	{0x0D87F270955EC50FULL, "6029060_m.fxe"},
	{0x0D87FB1F288885F4ULL, "nat300ip_lore_item_chest.utp"},
	{0x0D8855564FC304C8ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\injured_supported_malepc.ani"},
	{0x0D886D93CFF7228EULL, "435005434.ogg"},
	{0x0D8889A7E23CAE28ULL, "qun200_party_comment6095254.cl"},
	{0x0D88E54A52854451ULL, "dr_0_deeproads_5842.rml"},
	{0x0D88EFBFAACB93B3ULL, "6071822_f.fxe"},
	{0x0D89456DA83A23A0ULL, "vxlt_gallowstemplar_6si_0_0.vlm"},
	{0x0D898ADCA65FF9B5ULL, "dae000_meredith6085276.cl"},
	{0x0D89AE8DD284A5C7ULL, "277073272.ogg"},
	{0x0D89B0125F4BB894ULL, "193564916.ogg"},
	{0x0D8A7280E5EAE204ULL, "vxca_caverns1_ijv_0_0.vlm"},
	{0x0D8A7ABEB48ED6A5ULL, "6094872_f.fxe"},
	{0x0D8AFCFFB1F73D44ULL, "822259431.ogg"},
	{0x0D8B2EF9EA93524CULL, "344569801.ogg"},
	{0x0D8B8887A1C2AD7EULL, "6116554_m.fxe"},
	{0x0D8CCCF8A1954A6EULL, "vxht_0_hightown_a3_c0y_0_0.vlm"},
	{0x0D8D26F89D783D96ULL, "llac_ca_caverns1_mines_24421_7.dds"},
	{0x0D8D50DDF1B321D9ULL, "vxca_caverns1_fmi_0_0.vlm"},
	{0x0D8E1B424B565988ULL, "vxdr_primevaltomb_531_0_1.vlm"},
	{0x0D8E68D19E903948ULL, "art\\characters\\playercharacter\\qunarimale\\animation\\combat\\commander\\comm_ability_buff.gad"},
	{0x0D8F0C18EB2CCE46ULL, "vxht_mansion2b_1po_0_1.vlm"},
	{0x0D8F25B11D6F71CBULL, "vxlt_hangedman_n_2hq_0_0.vlm"},
	{0x0D8F7CA84515662FULL, "vxht_0_hightown_a2_57f_0_0.vlm"},
	{0x0D8FC9FB88E6A337ULL, "vxht_brothel_n_5mh_0_0.vlm"},
	{0x0D8FE0522ABF29ECULL, "hlac_lt_undercityencounter_-1_35z.dds"},
	{0x0D8FE9690220A5B0ULL, "6077297_m.fxe"},
	{0x0D901ECDE50290F4ULL, "hlac_ht_mansion_n_3364_1y.dds"},
	{0x0D90C132244E1AF9ULL, "vxca_cavern1_haunted_igm_0_0.vlm"},
	{0x0D90F0C6042C9BBDULL, "hlac_lt_hangedman_n_-1_21x.dds"},
	{0x0D915FD517D578D0ULL, "vxlt_warehouse_n_109_0_0.vlm"},
	{0x0D91A300EB341141ULL, "vxw_sundermount_2m9_0_0.vlm"},
	{0x0D92650076E714F0ULL, "6177859_f.fxe"},
	{0x0D927C5F6FDE551FULL, "91257650.ogg"},
	{0x0D92C63192AF9117ULL, "6061851_m.fxe"},
	{0x0D92F97F84D59EE3ULL, "lt_0_lowtown_n_33_0_ll.mmh"},
	{0x0D92FC9089A2F6BFULL, "vxw_blightlands_g5_0_0.vlm"},
	{0x0D936247742AEA1CULL, "framebuffereffects.mat.dx11"},
	{0x0D93698F6B5B9C25ULL, "llac_ca_caverns2_5802_37.dds"},
	{0x0D937003A475AF38ULL, "6201265_m.fxe"},
	{0x0D939A835B111DECULL, "vxw_blightlands_v9_0_0.vlm"},
	{0x0D93BD561C65803BULL, "6034681_f.fxe"},
	{0x0D93E04AC622AD21ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\basebow\\combat\\attackv1.ani"},
	{0x0D942F83DFF02E68ULL, "art\\characters\\playercharacter\\humanmale\\arm_medd.phy"},
	{0x0D947F35340101EFULL, "vxw_encounter2_dc_0_0.vlm"},
	{0x0D95612ECBBAEC21ULL, "vxlt_0_lowtown_a2_eu6_0_0.vlm"},
	{0x0D959B3642369985ULL, "vxlt_0_lowtown_a2_n_c_db3_0_0.vlm"},
	{0x0D95E49572BD64A5ULL, "vxlt_0_lowtown_a2_n_gff_0_0.vlm"},
	{0x0D962416E8066384ULL, "vxht_mansion2_315_0_0.vlm"},
	{0x0D965D7F67160A2CULL, "llac_dr_slavers_3217_5.dds"},
	{0x0D969D79A68FB347ULL, "mcr370_assignments6081788.cl"},
	{0x0D972480E3B505ACULL, "6125554_m.fxe"},
	{0x0D97A37CD2134C12ULL, "ss_matttest.cut"},
	{0x0D9819B39299F213ULL, "fen210st_hadriana.stg"},
	{0x0D985CB9A9A26868ULL, "hlac_lt_0_lowtown_docks_a3_-2_2x.dds"},
	{0x0D989D2AC8FCDBE2ULL, "vxht_mansion2_1ma_0_0.vlm"},
	{0x0D98AF1A5893257CULL, "vxlt_0_lowtown_docks_a3_n_d3q_0_0.vlm"},
	{0x0D98E45DD7E0FA84ULL, "vxca_caverns1_imw_0_0.vlm"},
	{0x0D991E33F1B821EBULL, "vxlt_0_lowtown_n_fip_0_0.vlm"},
	{0x0D99DB5E3179017AULL, "6076497_f.fxe"},
	{0x0D9A31E8BE1DB2C4ULL, "art\\levels\\races\\dwarf\\props\\_textures\\prp_newmetal_d.dds"},
	{0x0D9A69B5970ECF5CULL, "qun110_petrice6018578.cl"},
	{0x0D9AB7CD66BAEA2BULL, "hlac_lt_gallowscourtyard_-131072_2x.dds"},
	{0x0D9ACC8AECB33F4CULL, "391114744.ogg"},
	{0x0D9AEB4AC628DEC3ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\basebow\\combat\\attackv1.gad"},
	{0x0D9B3ED5938B62EEULL, "vxht_0_hightown_a3_c5z_0_0.vlm"},
	{0x0D9B7A21E40EE879ULL, "mag211_marethari6025317.cl"},
	{0x0D9BB373A336AB70ULL, "crt_pride_demon.bnk"},
	{0x0D9C1AD4A3B0B5C9ULL, "hla_ht_0_hightown_a2_1_0z.dds"},
	{0x0D9CA422D417E441ULL, "257050491.ogg"},
	{0x0D9CDD3F76EF1E33ULL, "vxht_mansion2_2xt_0_0.vlm"},
	{0x0D9D14EE35276FEBULL, "qcr210cr_elven_supporter_rg.utc"},
	{0x0D9D23992E4DFF13ULL, "vxlt_0_lowtown_docks_a3_n_d0b_0_1.vlm"},
	{0x0D9D2FC257D2AD91ULL, "dae000_bartrand6206910.cl"},
	{0x0D9DBF88E4EE727DULL, "art\\characters\\weapons\\unique\\aveline\\w_shd_ave.mao"},
	{0x0D9DC388E4EE78B6ULL, "art\\characters\\weapons\\unique\\aveline\\w_shd_ave.mmh"},
	{0x0D9DCD88E4EE89B4ULL, "art\\characters\\weapons\\unique\\aveline\\w_shd_ave.msh"},
	{0x0D9E551384917401ULL, "hlac_lt_0_lowtown_a2_n_131069_0y.dds"},
	{0x0D9E854C69524C49ULL, "vxht_0_hightown_a2o_0_0.vlm"},
	{0x0D9ECF99F38D7173ULL, "art\\vfx\\materials\\vfx_d_impact_add.mao"},
	{0x0D9ED41F9836B429ULL, "dae000_friend_turn_in6179269.cl"},
	{0x0D9ED4B56464D2E2ULL, "terrain_blend6.psh.fxo"},
	{0x0D9EE8BFDAB47736ULL, "pro000cr_refugee5.utc"},
	{0x0D9F680030D62362ULL, "mer221_sabin6040021.cl"},
	{0x0D9F956B11D4FDB9ULL, "vxht_mansion2b_3gn_0_0.vlm"},
	{0x0D9FC2740C0D9D5DULL, "hm_ave3_insurgent1.mor"},
	{0x0DA026B831F2BB6DULL, "w_encounter2_camp_7_0.msh"},
	{0x0DA0285341AF9EE9ULL, "152313039.ogg"},
	{0x0DA02BD810032574ULL, "ht_brothel_n_mmi.gff"},
	{0x0DA034B831F2D2A0ULL, "w_encounter2_camp_7_0.mao"},
	{0x0DA038B831F2D9F3ULL, "w_encounter2_camp_7_0.mmh"},
	{0x0DA11EDD2A95CCD4ULL, "lgt210tr_frame_flower.utt"},
	{0x0DA14A102839F86AULL, "clf260_fell_orden.cnv"},
	{0x0DA14E37695EEC35ULL, "318366800.ogg"},
	{0x0DA16E6D7B990F3FULL, "vxw_blightlands_h4_0_0.vlm"},
	{0x0DA1C84AC62EC1A8ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\basebow\\combat\\attackv1.evt"},
	{0x0DA20783712831D2ULL, "hlac_lt_0_lowtown_docks_n_-262144_0y.dds"},
	{0x0DA20D02EF3E60BBULL, "6133060_m.fxe"},
	{0x0DA2C31971EDB2A4ULL, "908252040.ogg"},
	{0x0DA2F11225F71A6FULL, "vxw_encounter2_ob_0_0.vlm"},
	{0x0DA33FF43349E5ABULL, "gen00fl_anders6183809.cl"},
	{0x0DA3D30BBDB104A1ULL, "vxlt_0_lowtown_a2_ex5_0_0.vlm"},
	{0x0DA42C10283BFA92ULL, "clf260_fell_orden.bnk"},
	{0x0DA445010972E293ULL, "darktown_fem_fereldan_sta6152801.cl"},
	{0x0DA4567264B37D25ULL, "vxlt_0_lowtown_a2_n_gce_0_0.vlm"},
	{0x0DA45FCDDA0CFD88ULL, "665826962.ogg"},
	{0x0DA472719EF9F034ULL, "6103327_m.fxe"},
	{0x0DA5D5D76A4FD46DULL, "art\\levels\\ztools\\default\\tilesets\\setext\\3ci_basic.msh"},
	{0x0DA5E3D76A4FEBA0ULL, "art\\levels\\ztools\\default\\tilesets\\setext\\3ci_basic.mao"},
	{0x0DA5E7D76A4FF2F3ULL, "art\\levels\\ztools\\default\\tilesets\\setext\\3ci_basic.mmh"},
	{0x0DA701A617A930D9ULL, "937334260.ogg"},
	{0x0DA76E42C8110006ULL, "art\\vfx\\environment\\vfx_tiki_fire.mmh"},
	{0x0DA7798C0639C29EULL, "dreev_item_acquired.ncs"},
	{0x0DA82FC915F1C00DULL, "llac_ht_chantry_2534_15.dds"},
	{0x0DA847156F870047ULL, "218805669.ogg"},
	{0x0DA91CE918CE0E1FULL, "gen00fl_anders6119464.cl"},
	{0x0DA91F406765AFE4ULL, "vxlt_darktown_58c_0_0.vlm"},
	{0x0DAA527188A63023ULL, "sse_ss_ambient_f.gda"},
	{0x0DAAC62F5A47F35DULL, "6116056_m.fxe"},
	{0x0DAB78B8F1E6EE76ULL, "seb221_flora_harimann6177695.cl"},
	{0x0DABF2F15C6FD5F2ULL, "mer320_hubert6089685.cl"},
	{0x0DAC2D275180A40DULL, "mag101_aombanter6052746.cl"},
	{0x0DAD00ADC8CA4BD8ULL, "560433250.ogg"},
	{0x0DAD3275C810C617ULL, "vxlt_darktown_5ky_0_0.vlm"},
	{0x0DAD6E59E67B2E39ULL, "leave_07_l3.mao"},
	{0x0DAE525BFEEDC7D4ULL, "vxca_caverns1_mines_cp3_0_0.vlm"},
	{0x0DAE61C8AA334863ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\draw_sword_threaten_enter.ani"},
	{0x0DAE820CB8F72CE7ULL, "gen00fl_sebastian6188618.cl"},
	{0x0DAE8AF806B450FBULL, "gen00fl_varric6076079.cl"},
	{0x0DAEC5C8D9FEF333ULL, "6077812_m.fxe"},
	{0x0DAF0E93133AF811ULL, "llac_w_bonepit_458752_0.dds"},
	{0x0DAF111A89DD6085ULL, "31334373.ogg"},
	{0x0DAF29D63A5FAFD9ULL, "hlac_lt_0_lowtown_n_1_10y.dds"},
	{0x0DAF58268157B5DBULL, "hlac_ht_0_hightown_a3_-196609_0y.dds"},
	{0x0DAF5ED6B2E36518ULL, "vxw_encounter2_pu_0_1.vlm"},
	{0x0DAFE04A6D8F27BFULL, "vxw_blightlands_m7_0_0.vlm"},
	{0x0DB04B8AE1A3EFC0ULL, "vxdr_primevalentrance_347_0_0.vlm"},
	{0x0DB06307F8A4AA97ULL, "art\\levels\\ztools\\default\\tilesets\\setint\\_textures\\xx_plaster.dds"},
	{0x0DB0CB50826089E2ULL, "lgt200cr_pride_abomination.utc"},
	{0x0DB0FA9C2FC6B28AULL, "lgt210cr_qun_blk_qunari.utc"},
	{0x0DB113498A077976ULL, "art\\characters\\creatures\\crowds\\humanmale\\crd_fereldenb7.msh"},
	{0x0DB11D498A078A74ULL, "art\\characters\\creatures\\crowds\\humanmale\\crd_fereldenb7.mmh"},
	{0x0DB14E8AAF35A224ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\qunari_post_escape.ani"},
	{0x0DB162EF17ED32EFULL, "vxw_encounter2_jm_0_0.vlm"},
	{0x0DB1722CBB2D5317ULL, "233758685.ogg"},
	{0x0DB191848A10F969ULL, "vxht_0_hightown_n_5yi_0_0.vlm"},
	{0x0DB1B1806C7B9214ULL, "6107100_f.fxe"},
	{0x0DB1F3F9EF64A4C8ULL, "99495132.ogg"},
	{0x0DB21013B103D2DFULL, "zz_ave_debug6134008.cl"},
	{0x0DB297C2B7D4340BULL, "gen00fl_isabela6090879.cl"},
	{0x0DB2C0ACEFD20DB6ULL, "vxht_0_hightown_n_9p9_0_0.vlm"},
	{0x0DB3F9676DB4D57EULL, "zz_mer_debug6028951.cl"},
	{0x0DB3FB40745A7229ULL, "vxht_keep_619_0_0.vlm"},
	{0x0DB48D82DB961E98ULL, "382057214.ogg"},
	{0x0DB4BD4076E741BAULL, "6105518_m.fxe"},
	{0x0DB4D40B22DA246CULL, "llac_lt_0_lowtown_a3_n_c_-196609_3.dds"},
	{0x0DB50DF10CFAA523ULL, "6206952_f.fxe"},
	{0x0DB51CE5AD094B36ULL, "vxw_encounter3_7d_0_0.vlm"},
	{0x0DB5280F0562323BULL, "vxht_bartrand_n_5vy_0_0.vlm"},
	{0x0DB59940FCD35141ULL, "hlac_ht_0_hightown_a2_-65537_2y.dds"},
	{0x0DB59F1A77DAA20FULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\0wo_wallcap.phy"},
	{0x0DB5A39336EFEEECULL, "hlac_lt_0_lowtown_docks_a3_n_-65538_0x.dds"},
	{0x0DB5ECF9FA84EA8CULL, "6066394_m.fxe"},
	{0x0DB65EB8A97B866FULL, "59345423.ogg"},
	{0x0DB689ACB4B37B35ULL, "lt_0_lowtown_docks_29_0_lc.mmh"},
	{0x0DB6AC01D4381C51ULL, "6061277_m.fxe"},
	{0x0DB6C560ACDE3406ULL, "llac_lt_0_lowtown_a2_n_c_131072_6.dds"},
	{0x0DB74B8AAF3A0886ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\qunari_post_escape.gad"},
	{0x0DB760E92304E9C8ULL, "t3_shd_lst.tnt"},
	{0x0DB793D5EF7CD0BBULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_eye11b.phy"},
	{0x0DB7A412E3274BA4ULL, "6129824_m.fxe"},
	{0x0DB8A866A1F89577ULL, "gen_im_trs_cot_dua_dag_r1.uti"},
	{0x0DB9FC6E629DA511ULL, "vxlt_hangedman_1tg_0_0.vlm"},
	{0x0DBA3F3556FC47C9ULL, "gen00fl_aveline6139265.cl"},
	{0x0DBB79C8AA3DFB5AULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\draw_sword_threaten_enter.evt"},
	{0x0DBBA452BA06DE97ULL, "vxlt_darktown_5lx_0_0.vlm"},
	{0x0DBBFA1331819A43ULL, "6140354_m.fxe"},
	{0x0DBC4C0F04DAF1BCULL, "mcr370_samson6087533.cl"},
	{0x0DBCBB33DAE9AF11ULL, "vxht_mansion_n_5me_0_0.vlm"},
	{0x0DBCDB8FB1D01B8CULL, "gen00fl_fenris6103327.cl"},
	{0x0DBD34B16BA376F8ULL, "6147661_f.fxe"},
	{0x0DBE253C302E1E83ULL, "lt_0_lowtown_a2_n_c_35_0_ll.mmh"},
	{0x0DBE468AAF401F65ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\qunari_post_escape.evt"},
	{0x0DBE52275F85403FULL, "vxw_blightlands_v6_0_0.vlm"},
	{0x0DBEBD67D39A0840ULL, "vxdr_primevalentrance_314_0_0.vlm"},
	{0x0DBEC0E4601278CBULL, "850502517.ogg"},
	{0x0DBEF0BCD32A91BBULL, "6139660_m.fxe"},
	{0x0DBF18B0519422DAULL, "vxca_caverns2_3cf_0_0.vlm"},
	{0x0DBF97083BDBD0E7ULL, "dae000_hate_follower_react6103203.cl"},
	{0x0DBFC3896880C010ULL, "mag360_samson6042060.cl"},
	{0x0DC0044912EA5006ULL, "vxca_caverns1_e8b_0_0.vlm"},
	{0x0DC0099BCC350C3BULL, "zz_vrc_debug6200225.cl"},
	{0x0DC09573A355943AULL, "crt_pride_demon.wbd"},
	{0x0DC0A54354917278ULL, "224198037.ogg"},
	{0x0DC0E95E05ACDD53ULL, "17514919.ogg"},
	{0x0DC1341A1F59F4A2ULL, "6093587_m.fxe"},
	{0x0DC17A111AD67512ULL, "mag101_grace6034201.cl"},
	{0x0DC17D2BAC11B46DULL, "w_encounter2_56_0.msh"},
	{0x0DC18B2BAC11CBA0ULL, "w_encounter2_56_0.mao"},
	{0x0DC18F2BAC11D2F3ULL, "w_encounter2_56_0.mmh"},
	{0x0DC2466860EAE733ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\qunari_sit_throne_sweep_right_hand_2.evt"},
	{0x0DC26325BB15FA47ULL, "380216399.ogg"},
	{0x0DC278C8AA44184DULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\draw_sword_threaten_enter.gad"},
	{0x0DC287D07EE8AF72ULL, "106416055.ogg"},
	{0x0DC2E772ECBD89ABULL, "157220389.ogg"},
	{0x0DC301986EBBE2EFULL, "art\\vfx\\combat\\vfx_q_ari_bstorm_act_c.impa.gad"},
	{0x0DC37F582DFB206FULL, "6013267_m.fxe"},
	{0x0DC399EBF7584ABBULL, "vxht_bartrand_n_5yx_0_0.vlm"},
	{0x0DC3B10189988CE5ULL, "vxht_0_hightown_n_b5n_0_0.vlm"},
	{0x0DC3EBABBDBF20DFULL, "vxca_caverns1_mines_gk4_0_1.vlm"},
	{0x0DC42422DFC896DBULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\blendtree\\pose_urgent_walk.anb"},
	{0x0DC476DC86B217FDULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_quill_sa.dds"},
	{0x0DC47857E5FC72FFULL, "hlac_w_sundermount_0_0z.dds"},
	{0x0DC4CCF0255098BEULL, "6179254_m.fxe"},
	{0x0DC570623796382EULL, "902786864.ogg"},
	{0x0DC59D8B67F140EAULL, "fx_air_twirl.mao"},
	{0x0DC65A69FA818B1FULL, "6088766_m.fxe"},
	{0x0DC678B262638216ULL, "6069861_m.fxe"},
	{0x0DC6852DDE258475ULL, "vxht_0_hightown_b95_0_0.vlm"},
	{0x0DC6A802FDD71C17ULL, "art\\characters\\playercharacter\\humanmale\\textures\\hn_glv_medd_0t.dds"},
	{0x0DC720FC9AF3897FULL, "339850988.ogg"},
	{0x0DC727EE14965169ULL, "22043727.ogg"},
	{0x0DC7799C66AAC14AULL, "sto_fl_light.mao"},
	{0x0DC86E4B5493BD91ULL, "vxlt_hangedman_1sf_0_0.vlm"},
	{0x0DC8B3BD1AD29072ULL, "gr_clu.mao"},
	{0x0DC93F6860F0FA74ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\qunari_sit_throne_sweep_right_hand_2.gad"},
	{0x0DCA16986EC2249CULL, "art\\vfx\\combat\\vfx_q_ari_bstorm_act_c.impa.evt"},
	{0x0DCA5EC44383412DULL, "vxlt_undercityserialkiller_18p_0_0.vlm"},
	{0x0DCA94855C20EC5DULL, "vxlt_0_lowtown_n_h87_0_0.vlm"},
	{0x0DCAF7BD3BC25234ULL, "gallows_avatar.cut"},
	{0x0DCAFFF42C10558FULL, "gen00fl_anders6111381.cl"},
	{0x0DCB4B7250A9DB35ULL, "plt_lgt000pt_ultra_5_01.plo"},
	{0x0DCB8474B2FD4286ULL, "6107439_m.fxe"},
	{0x0DCBCF8260F0228EULL, "hightown_fem_cityguard_mob6168529.cl"},
	{0x0DCBEA7C76CD8245ULL, "850422248.ogg"},
	{0x0DCCCBE55EF895D6ULL, "hf_arthuris.mor"},
	{0x0DCCD0BED83DF42FULL, "vrc321_varric6087124.cl"},
	{0x0DCD9E94A619F3C0ULL, "mag211_marethari6025152.cl"},
	{0x0DCE18F5B4399D7FULL, "6161610_m.fxe"},
	{0x0DCE249A3D22A2A6ULL, "lgt220ip_shp_mag_devrona.utp"},
	{0x0DCE3FF83054CE08ULL, "vxdr_0_deeproads_d3o_0_0.vlm"},
	{0x0DCE46A8FBD963EFULL, "vxw_encounter2_po_0_0.vlm"},
	{0x0DCE9F76A9C3C615ULL, "hlac_lt_gallowscourtyard_n_-2_1x.dds"},
	{0x0DCEE2D20B5D39DFULL, "vxca_caverns1_mines_iko_0_0.vlm"},
	{0x0DCF0B3AF4C0402AULL, "vxht_brothel_n_6d5_0_0.vlm"},
	{0x0DCF0E18E3392057ULL, "hla_ht_0_hightown_n_65538_1z.dds"},
	{0x0DCFF096A3867096ULL, "6074489_m.fxe"},
	{0x0DCFF4822C00C797ULL, "6097898_m.fxe"},
	{0x0DD024F8E84DEE11ULL, "6083057_m.fxe"},
	{0x0DD04E6860F73252ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\qunari_sit_throne_sweep_right_hand_2.ani"},
	{0x0DD12EF71FF5DF93ULL, "pro000_varric6170655.cl"},
	{0x0DD1998066435399ULL, "fadetosepia.mao"},
	{0x0DD1F810A77EB6DEULL, "875149338.ogg"},
	{0x0DD20AB608D4B608ULL, "art\\vfx\\textures\\vfx_vollight_dust.dds"},
	{0x0DD222DE7B8EA565ULL, "vxht_0_hightown_n_b0m_0_0.vlm"},
	{0x0DD26CF5E950597BULL, "6103716_f.fxe"},
	{0x0DD2C9D6C76E3DAEULL, "art\\levels\\races\\proxy\\props\\pressure\\animation\\pressureup.ani"},
	{0x0DD39BD6AA2BDA4BULL, "224711022.ogg"},
	{0x0DD4184BF731F463ULL, "gen00fl_aveline6106948.cl"},
	{0x0DD43BC59651ABB6ULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_lute.msh"},
	{0x0DD445C59651BCB4ULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_lute.mmh"},
	{0x0DD448E2F09A1BFFULL, "vxht_0_hightown_c5n_0_0.vlm"},
	{0x0DD449C59651C3C7ULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_lute.mao"},
	{0x0DD467A6F3C4ACDFULL, "6066335_m.fxe"},
	{0x0DD46ADEFFB7333AULL, "6149427_f.fxe"},
	{0x0DD494BD6CCEB322ULL, "vxdr_0_deeproads_buu_0_0.vlm"},
	{0x0DD49B3196528C9CULL, "one300_zevran6076573.cl"},
	{0x0DD5064ABFC2B5A1ULL, "vxdr_primevalentrance_333_0_0.vlm"},
	{0x0DD55103B64DBA48ULL, "l_mag211ar_ariannis_house.bnk"},
	{0x0DD556D1A8983C54ULL, "hlac_lt_ruinedhouse_n_-1_6z.dds"},
	{0x0DD56B7D8AE824B0ULL, "6089490_m.fxe"},
	{0x0DD62E6E3F2F9AA2ULL, "349863308.ogg"},
	{0x0DD6752277D477B7ULL, "820105412.ogg"},
	{0x0DD6AC1609C4CEB4ULL, "245824188.ogg"},
	{0x0DD6C9AA33C53FE8ULL, "vxlt_0_lowtown_fgn_0_0.vlm"},
	{0x0DD6D0153E39ECEDULL, "153550143.ogg"},
	{0x0DD71E986ECCBCB5ULL, "art\\vfx\\combat\\vfx_q_ari_bstorm_act_c.impa.ani"},
	{0x0DD75853B06F5D37ULL, "dae000ip_crowd_mf5_talk11.utp"},
	{0x0DD8FB6F4E9A2E77ULL, "lt_0_lowtown_docks_173_0.msh"},
	{0x0DD90D6F4E9A4CDAULL, "lt_0_lowtown_docks_173_0.mao"},
	{0x0DD9116F4E9A53E9ULL, "lt_0_lowtown_docks_173_0.mmh"},
	{0x0DD9315607386642ULL, "vxlt_0_lowtown_docks_n_b94_0_0.vlm"},
	{0x0DD93A16780FB2FCULL, "118104706.ogg"},
	{0x0DD95D0DC997360FULL, "gen00fl_varric6093636.cl"},
	{0x0DD9F8790B26DF39ULL, "llac_lt_0_lowtown_docks_n_0_0.dds"},
	{0x0DDA512AC1CEA4C9ULL, "gen_im_arm_glv_med_act3_02.uti"},
	{0x0DDA67B8265E4275ULL, "963778195.ogg"},
	{0x0DDAAAE148ECEADFULL, "707113905.ogg"},
	{0x0DDAC2F98C46B1C6ULL, "llac_ht_chantry_n_2534_37.dds"},
	{0x0DDB03D45785EB4EULL, "art\\characters\\playercharacter\\children\\mt_bas_kf_scalpm1.mao"},
	{0x0DDB4B2232F48C63ULL, "861694627.ogg"},
	{0x0DDB5651AAE7C4E8ULL, "art\\characters\\playercharacter\\humanmale\\textures\\hm_hlf_lgtd_n.dds"},
	{0x0DDB7B0118D73840ULL, "477297282.ogg"},
	{0x0DDB86404B8999CDULL, "vxca_cavern1_haunted_jg6_0_0.vlm"},
	{0x0DDBAC7328C76416ULL, "vxlt_0_lowtown_a2_n_c_gij_0_0.vlm"},
	{0x0DDBFACC0C545A2CULL, "383420846.ogg"},
	{0x0DDBFCB2AB888FB4ULL, "6081689_m.fxe"},
	{0x0DDC8F9E43777771ULL, "art\\characters\\creatures\\darkspawnogre\\animation\\effect\\stun.ani"},
	{0x0DDC9F51C996B857ULL, "lt_0_lowtown_docks_201_0.msh"},
	{0x0DDCB151C996D6BAULL, "lt_0_lowtown_docks_201_0.mao"},
	{0x0DDCB551C996DDC9ULL, "lt_0_lowtown_docks_201_0.mmh"},
	{0x0DDE6273957C8916ULL, "6079789_m.fxe"},
	{0x0DDEC55F30F1E928ULL, "vxca_caverns2_80n_0_0.vlm"},
	{0x0DE0DED2DB4671FBULL, "6104616_f.fxe"},
	{0x0DE121EC1438DDA4ULL, "186334698.ogg"},
	{0x0DE1B9CCBF8602C7ULL, "vxlt_0_lowtown_docks_a3_n_b9b_0_0.vlm"},
	{0x0DE1C5D14D756E6FULL, "cth_cm2.tnt"},
	{0x0DE2391B1405E06EULL, "art\\characters\\playercharacter\\elffemale\\animation\\exploration\\movement\\mov_rf_p_lfoot.ani"},
	{0x0DE2A30744BE9C75ULL, "w_sundermount_140_0.phy"},
	{0x0DE3069A5EC4CBA2ULL, "vxdr_0_deeproads_bpv_0_0.vlm"},
	{0x0DE33A1C7EEB92E1ULL, "6024989_f.fxe"},
	{0x0DE33E23DE6DBC1FULL, "6086966_m.fxe"},
	{0x0DE345E24A005E6CULL, "art\\levels\\races\\natural\\highlands\\props\\prp_01hgertes.msh"},
	{0x0DE34BE24A00685EULL, "art\\levels\\races\\natural\\highlands\\props\\prp_01hgertes.mmh"},
	{0x0DE357E24A007CF5ULL, "art\\levels\\races\\natural\\highlands\\props\\prp_01hgertes.mao"},
	{0x0DE37AD5EFA1E15AULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_eye11b.msh"},
	{0x0DE38CD5EFA1FFA0ULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_eye11b.mmh"},
	{0x0DE39A9E437DA813ULL, "art\\characters\\creatures\\darkspawnogre\\animation\\effect\\stun.gad"},
	{0x0DE3EA960FB22069ULL, "81570327.ogg"},
	{0x0DE40ED3E10B7525ULL, "vxlt_0_lowtown_docks_n_c4y_0_0.vlm"},
	{0x0DE40F113E1220EEULL, "6035297_m.fxe"},
	{0x0DE455EE3E0A4E6BULL, "917328117.ogg"},
	{0x0DE45A5874A30EBEULL, "mag291_torpor6116237.cl"},
	{0x0DE4F90E795CB991ULL, "vxlt_0_lowtown_f24_0_0.vlm"},
	{0x0DE6FE9290B26C77ULL, "vxht_bartrand_n_5vm_0_0.vlm"},
	{0x0DE711FC08FA62DDULL, "6074495_m.fxe"},
	{0x0DE750D13F112C9BULL, "6188612_f.fxe"},
	{0x0DE75401DD15C0F9ULL, "llac_ht_chantry_2534_2.dds"},
	{0x0DE772B334FCBD70ULL, "6095345_m.fxe"},
	{0x0DE7A2D6C78057E8ULL, "art\\levels\\races\\proxy\\props\\pressure\\animation\\pressureup.gad"},
	{0x0DE7A332F92E7EC2ULL, "vxlt_0_lowtown_docks_n_b65_0_0.vlm"},
	{0x0DE7D6D720982FB3ULL, "vxca_caverns1_hm9_0_0.vlm"},
	{0x0DE857E204654C12ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\shield and weapon\\sw_damage_reduction.gad"},
	{0x0DE8C8D411D5D131ULL, "6162565_m.fxe"},
	{0x0DE96789F05B17C4ULL, "35611378.ogg"},
	{0x0DE9C40744C4F2A1ULL, "w_sundermount_140_0.rml"},
	{0x0DEA2043B4C87704ULL, "qcr220cr_noblewoman3.utc"},
	{0x0DEA979E4383C278ULL, "art\\characters\\creatures\\darkspawnogre\\animation\\effect\\stun.evt"},
	{0x0DEAA5073A9C76E1ULL, "hlac_dr_primevaltomb_664_6y.dds"},
	{0x0DEB86A88D6E83EDULL, "529878776.ogg"},
	{0x0DEBADE249C74293ULL, "dr_dungeon_mmi.gff"},
	{0x0DEBDDC610142576ULL, "ave200_donnic6110730.cl"},
	{0x0DEBEEF4D8AC712AULL, "vxht_brothel_n_6j7_0_0.vlm"},
	{0x0DEC053EF2F5457BULL, "6136445_m.fxe"},
	{0x0DEC28EDFC576265ULL, "vxdr_primevalentrance_39m_0_0.vlm"},
	{0x0DECAE9200AAEE64ULL, "hlac_w_encounter2_262142_0y.dds"},
	{0x0DECAF7D60BBA2BCULL, "254329449.ogg"},
	{0x0DED62031B5EE995ULL, "zz_dae_debug6184889.cl"},
	{0x0DED9846A694C4CBULL, "394993736.ogg"},
	{0x0DEDF82ECCDF00D3ULL, "art\\levels\\ztools\\default\\generic\\0wi_angle.phy"},
	{0x0DEE2BD26BB8523BULL, "231059595.ogg"},
	{0x0DEE458A7670E8FFULL, "t3_wdg_bn1.tnt"},
	{0x0DEF311B14105CBFULL, "art\\characters\\playercharacter\\elffemale\\animation\\exploration\\movement\\mov_rf_p_lfoot.evt"},
	{0x0DEF5AE2046B7049ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\shield and weapon\\sw_damage_reduction.evt"},
	{0x0DEFED3B8B7D0BE5ULL, "llac_lt_0_lowtown_a2_n_c_65535_5.dds"},
	{0x0DF06F466BA83FB9ULL, "art\\characters\\creatures\\crowds\\humanfemale\\crd_fereldenb1.msh"},
	{0x0DF079466BA850A7ULL, "art\\characters\\creatures\\crowds\\humanfemale\\crd_fereldenb1.mmh"},
	{0x0DF1A4137638B120ULL, "character_character_p191.psh.fxo.dx11.assembly"},
	{0x0DF1C0BA5FA41247ULL, "6120416_f.fxe"},
	{0x0DF1FA9DA8A82906ULL, "259967446.ogg"},
	{0x0DF220EE0585410AULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\sitting_drinking_standing_cheer.gad"},
	{0x0DF24F376ED455B0ULL, "6087690_m.fxe"},
	{0x0DF285BFCB0DCAEFULL, "dae000_gamlen6166382.cl"},
	{0x0DF2B296138630B0ULL, "6140933_m.fxe"},
	{0x0DF33FB7B5BD34B4ULL, "mer141_jansen6085166.cl"},
	{0x0DF343492994859BULL, "art\\vfx\\materials\\vfx_fire_link_blend.mao"},
	{0x0DF3C09466092FB9ULL, "prm000im_hindsight_belt.uti"},
	{0x0DF3F383B0E97B42ULL, "684785639.ogg"},
	{0x0DF4F6A4B93C90FCULL, "343331210.ogg"},
	{0x0DF5F04BAE9C55A2ULL, "2222095.ogg"},
	{0x0DF6121B141646A8ULL, "art\\characters\\playercharacter\\elffemale\\animation\\exploration\\movement\\mov_rf_p_lfoot.gad"},
	{0x0DF6DF2541A32D0AULL, "art\\characters\\playercharacter\\humanfemale\\bdy_masa.mao"},
	{0x0DF6E32541A33459ULL, "art\\characters\\playercharacter\\humanfemale\\bdy_masa.mmh"},
	{0x0DF6ED2541A34547ULL, "art\\characters\\playercharacter\\humanfemale\\bdy_masa.msh"},
	{0x0DF78A1ABA614262ULL, "llac_ca_caverns1_24423_4.dds"},
	{0x0DF7ADF9AAC9E737ULL, "art\\vfx\\environment\\level_fx\\fire\\vfx_brazfiresml_b.dura.gad"},
	{0x0DF85D05FA80446CULL, "zz_seb_debug6214536.cl"},
	{0x0DF876C4B70A2DA9ULL, "6114872_m.fxe"},
	{0x0DF8A09FC619634AULL, "vxht_0_hightown_a3_am8_0_0.vlm"},
	{0x0DF8AA7DB074C9F5ULL, "946067711.ogg"},
	{0x0DF9002939586643ULL, "gen00fl_bethany6101218.cl"},
	{0x0DF903EE058B2EA1ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\sitting_drinking_standing_cheer.evt"},
	{0x0DF91EFD6E2AB603ULL, "gen00fl_sebastian6180492.cl"},
	{0x0DF990C8D98CD80CULL, "786607815.ogg"},
	{0x0DF9927A9BCD5706ULL, "hlac_ca_caverns2_5802_58x.dds"},
	{0x0DFA39755CB91CA9ULL, "hlac_lt_0_lowtown_-65537_5x.dds"},
	{0x0DFA4303B66CBEF2ULL, "l_mag211ar_ariannis_house.wbd"},
	{0x0DFA838725482577ULL, "gen00fl_fenris6110784.cl"},
	{0x0DFA9ACAEE4D7AE5ULL, "vxdr_primevalentrance_32l_0_0.vlm"},
	{0x0DFAE4B3ED8B843EULL, "749601575.ogg"},
	{0x0DFB1CA2FC658A96ULL, "837006204.ogg"},
	{0x0DFB9DEF8223700BULL, "hlac_lt_0_lowtown_a3_n_c_1_8y.dds"},
	{0x0DFB9FB2152C0DCAULL, "vxlt_gallowstemplar_n_2jc_0_0.vlm"},
	{0x0DFB9FC368824DA8ULL, "vxht_0_hightown_b5h_0_0.vlm"},
	{0x0DFBAE7F24B69382ULL, "6139971_m.fxe"},
	{0x0DFBC358550CD91FULL, "6164781_m.fxe"},
	{0x0DFC96B0E5D9DB6EULL, "gencr_qunari_captain.utc"},
	{0x0DFCD1D931C63411ULL, "character_character_p280.psh.fxo.dx11.assembly"},
	{0x0DFCEFDD10BB1C0CULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\pose_f_weightshift_small.ani"},
	{0x0DFD402EDAE0BACBULL, "288343672.ogg"},
	{0x0DFD42E2047784B8ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\shield and weapon\\sw_damage_reduction.ani"},
	{0x0DFDC2F9AACE76C4ULL, "art\\vfx\\environment\\level_fx\\fire\\vfx_brazfiresml_b.dura.evt"},
	{0x0DFDC51A687D0BEAULL, "gen00fl_merrill6162034.cl"},
	{0x0DFE20E629220292ULL, "mcr371_hostile_aveline6082179.cl"},
	{0x0DFE5BA9DEE7AE3CULL, "6167245_f.fxe"},
	{0x0DFEB6FEEA84DD0DULL, "hlac_dr_dungeon_3216_1z.dds"},
	{0x0DFF1BD1EA32E9F7ULL, "722667049.ogg"},
	{0x0DFF43FAF13868B1ULL, "vxca_cavern1_haunted_imw_0_0.vlm"},
	{0x0DFF4EE27BAE3EAAULL, "6087533_m.fxe"},
	{0x0DFFC173EC401BDEULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\punched_from_front_stumble.evt"},
	{0x0E00008AD61AC1B7ULL, "6060500_m.fxe"},
	{0x0E002AEC84D8CB10ULL, "mrl150_merrill6078329.cl"},
	{0x0E00665D227E23DFULL, "ht_0_hightownexplosion_n_20_0_lc.mmh"},
	{0x0E0204D669D4F62AULL, "vxht_0_hightown_a3_9bi_0_0.vlm"},
	{0x0E0220EA786D6F97ULL, "vxw_woundedcoast_161_0_0.vlm"},
	{0x0E02C506F7E19241ULL, "qun111_ketojan6120459.cl"},
	{0x0E02C855139B149EULL, "930359926.ogg"},
	{0x0E03632FC2526565ULL, "gen00fl_isabela6135619.cl"},
	{0x0E03ED5EB55FA3E5ULL, "hlac_w_bonepitdestroyed_a3_3_0x.dds"},
	{0x0E03FCDD10C1514EULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\pose_f_weightshift_small.gad"},
	{0x0E042C8E50051330ULL, "vxlt_gallowstemplar_2pd_0_0.vlm"},
	{0x0E043A3BECC9331BULL, "hlac_dr_0_deeproads_5843_1z.dds"},
	{0x0E046D0FFDCA99B5ULL, "vxht_mansion2b_2pr_0_0.vlm"},
	{0x0E04714820087AD3ULL, "w_blightlands_166_0.phy"},
	{0x0E0484201DAFD89AULL, "6124775_m.fxe"},
	{0x0E04EDBC4B4C032AULL, "ts_raiders_wpn_table.gda"},
	{0x0E05BB9A46EE45E6ULL, "vxht_0_hightown_n_bu9_0_0.vlm"},
	{0x0E05C6F8027A05C2ULL, "character_character_p171.psh.fxo.dx11.assembly"},
	{0x0E05FBEE0595ABF0ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\sitting_drinking_standing_cheer.ani"},
	{0x0E061882B1260695ULL, "vxca_cave_1cl_0_0.vlm"},
	{0x0E062DD016783933ULL, "564513357.ogg"},
	{0x0E0638D9802FDF75ULL, "vxlt_warehouse_n_1c1_0_0.vlm"},
	{0x0E06701E3EDFB9CBULL, "lt_gallowsprison_n_mmap_ph.dds"},
	{0x0E0689FBA7E52D92ULL, "ico_staff_1.dds"},
	{0x0E069951CD565066ULL, "vxlt_gallowstemplar_n_458_0_0.vlm"},
	{0x0E06C7C3ED594002ULL, "6160847_m.fxe"},
	{0x0E06D873EC466109ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\punched_from_front_stumble.gad"},
	{0x0E072651C1382361ULL, "vxlt_0_lowtown_a2_gpo_0_0.vlm"},
	{0x0E083E7250DD7312ULL, "plt_lgt000pt_ultra_5_01.nss"},
	{0x0E08570619BB6C36ULL, "6042295_m.fxe"},
	{0x0E086178EDF5FA6FULL, "vxw_blightlands_6f_0_0.vlm"},
	{0x0E0A8124317B6A12ULL, "97213169.ogg"},
	{0x0E0ABD1BAE564793ULL, "vxht_0_hightown_a2_bjf_0_0.vlm"},
	{0x0E0AF7DD10C7672DULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\pose_f_weightshift_small.evt"},
	{0x0E0B63419522EB50ULL, "6190424_m.fxe"},
	{0x0E0B6541EC980D34ULL, "vxlt_0_lowtown_byp_0_0.vlm"},
	{0x0E0B70E5CAB55361ULL, "mag360_thrask6043893.cl"},
	{0x0E0B9A3D80652104ULL, "vxw_encounter3_65_0_0.vlm"},
	{0x0E0BC508C413EBCDULL, "370560824.ogg"},
	{0x0E0BDAF9AADADCBDULL, "art\\vfx\\environment\\level_fx\\fire\\vfx_brazfiresml_b.dura.ani"},
	{0x0E0C184A0E15DBD9ULL, "vxlt_foundry_n_wd_0_0.vlm"},
	{0x0E0C3249DE332F3BULL, "hlac_w_sundermount_131066_0y.dds"},
	{0x0E0C488672F7C258ULL, "6078128_m.fxe"},
	{0x0E0C67C7F47D5A1BULL, "face2_s1.psh.fxo"},
	{0x0E0C6E57AF9AAB0CULL, "6053329_m.fxe"},
	{0x0E0CCDF8E9C44B77ULL, "367748839.ogg"},
	{0x0E0CDFC43A5C4BCEULL, "one110_pryce6088370.cl"},
	{0x0E0D171D3A6D2AB8ULL, "147729235.ogg"},
	{0x0E0DA879F7604756ULL, "vxca_caverns1_mines_iip_0_0.vlm"},
	{0x0E0DA973EC4C3057ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\punched_from_front_stumble.ani"},
	{0x0E0DB5D7E32E8131ULL, "vxca_cavern1_haunted_ijv_0_0.vlm"},
	{0x0E0DC475B170111BULL, "art\\vfx\\environment\\level_fx\\fire\\vfx_brazfiremed.dura.evt"},
	{0x0E0E054AC4EE3C5FULL, "6059148_m.fxe"},
	{0x0E0E33B3DEDF902EULL, "llac_dr_dungeon_3214_2.dds"},
	{0x0E0E4F356D8D5A34ULL, "773858585.ogg"},
	{0x0E0E528E1163E051ULL, "475887682.ogg"},
	{0x0E0E7E211020EBA1ULL, "6181926_m.fxe"},
	{0x0E0EBA57730B4C22ULL, "315887770.ogg"},
	{0x0E0EEC09FC021F66ULL, "vxht_keep_6c0_0_0.vlm"},
	{0x0E0F20F822E3D5FBULL, "887837806.ogg"},
	{0x0E0F7FCD4A14D75EULL, "426507163.ogg"},
	{0x0E0F9B6D0256B5AAULL, "mag360_samson6044886.cl"},
	{0x0E0FD551424FA560ULL, "vxca_caverns2_62p_0_2.vlm"},
	{0x0E0FE7304E1279CAULL, "cd_mag291_torpor.wbd"},
	{0x0E1092C76A638817ULL, "vxw_woundedcoast_110_0_0.vlm"},
	{0x0E10A79C0E623811ULL, "vxlt_hangedman_1jy_0_0.vlm"},
	{0x0E10AF07828CCF48ULL, "vxlt_0_lowtown_a2_gpj_0_0.vlm"},
	{0x0E112E94CF8DEB5DULL, "vxht_0_hightown_9zc_0_0.vlm"},
	{0x0E116A37492F3FC6ULL, "vxw_sundermount_2f3_0_0.vlm"},
	{0x0E118DA5B0FD0CB2ULL, "vxlt_undercityencounter_142_0_1.vlm"},
	{0x0E11C797AC8E4A07ULL, "vxlt_gallowscourtyard_7br_0_0.vlm"},
	{0x0E11DCC8B8BFA761ULL, "6202067_m.fxe"},
	{0x0E11E07CAFC59595ULL, "hlac_lt_0_lowtown_docks_a3_n_-1_0y.dds"},
	{0x0E12554F7D6DAB58ULL, "follower_banter6141913.cl"},
	{0x0E12CFE6CF264F0DULL, "llac_w_bonepitmining_a2_196609_0.dds"},
	{0x0E13BDE5870D8D20ULL, "gen00fl_isabela6096858.cl"},
	{0x0E144A1CA7185830ULL, "art\\vfx\\environment\\level_fx\\bugs\\vfx_flys.mmh"},
	{0x0E1465C78A41F11DULL, "mrl310_merrill6119663.cl"},
	{0x0E147FA1D67683AAULL, "llac_ca_cave_2194_12.dds"},
	{0x0E148102F0571F2FULL, "hlac_lt_hangedman_n_-1_28y.dds"},
	{0x0E1499CC12AD164EULL, "vxht_brothel_n_5q1_0_0.vlm"},
	{0x0E14BD75B176245CULL, "art\\vfx\\environment\\level_fx\\fire\\vfx_brazfiremed.dura.gad"},
	{0x0E151CF8FA298FCEULL, "136166894.ogg"},
	{0x0E1523A3051FEA7EULL, "6100114_f.fxe"},
	{0x0E16DF698BA5225EULL, "vxlt_darktown_2vh_0_0.vlm"},
	{0x0E173A7BDD300353ULL, "vfx_chunkyparticle_p0.psh.fxo"},
	{0x0E173A80E865254AULL, "vxht_0_hightown_cb2_0_0.vlm"},
	{0x0E176F0198D3CC9DULL, "553780562.ogg"},
	{0x0E17D4131577DA09ULL, "snow_t_f.mao"},
	{0x0E1801C19CDB7BB6ULL, "284856955.ogg"},
	{0x0E184A3A1FE5643FULL, "dre000_dougal6131145.cl"},
	{0x0E186C8E4A41D111ULL, "6042120_m.fxe"},
	{0x0E189AF7BEFD9336ULL, "vxlt_0_lowtown_a2_ekg_0_0.vlm"},
	{0x0E1967DB9D893755ULL, "vxlt_0_lowtown_docks_c51_0_0.vlm"},
	{0x0E197394EB51D419ULL, "6115609_f.fxe"},
	{0x0E1996A0BEA81753ULL, "vxht_keep_n_676_0_0.vlm"},
	{0x0E1A371401CBEAA3ULL, "dre000_varric6042938.cl"},
	{0x0E1A989964F7A175ULL, "317704018.ogg"},
	{0x0E1ABC75B17A8E5AULL, "art\\vfx\\environment\\level_fx\\fire\\vfx_brazfiremed.dura.ani"},
	{0x0E1B89F032D93994ULL, "vxw_bonepitmining_a2_2a1_0_0.vlm"},
	{0x0E1CE2379846A7AEULL, "llac_lt_0_lowtown_a3_n_c_-131072_2.dds"},
	{0x0E1D252BB56F61FEULL, "llac_ht_0_hightown_a3_65533_2.dds"},
	{0x0E1D3C77976D98A3ULL, "6103236_m.fxe"},
	{0x0E1D3EC7314EDF47ULL, "ht_0_hightown_n_19_0.rml"},
	{0x0E1D5DE6EDF837E6ULL, "vxht_keep_6l1_0_0.vlm"},
	{0x0E1D72D865AC0143ULL, "6102908_f.fxe"},
	{0x0E1E9AFA68D9BA18ULL, "6165147_m.fxe"},
	{0x0E1F0220653EA738ULL, "6101941_m.fxe"},
	{0x0E1F20E47482E7C8ULL, "vxlt_0_lowtown_a2_gwk_0_0.vlm"},
	{0x0E1F22B222B2EBD3ULL, "hlac_ht_mansion2b_818_18z.dds"},
	{0x0E1F52B9F9A3427DULL, "hlac_ht_mansion2_818_10z.dds"},
	{0x0E1F5E1656A1A1E9ULL, "gen00fl_varric6093627.cl"},
	{0x0E1FD27D49470305ULL, "art\\characters\\weapons\\longbows\\textures\\w_lbw_dslb_0n.dds"},
	{0x0E1FDC143B255846ULL, "vxw_sundermount_2k4_0_0.vlm"},
	{0x0E2039749E846287ULL, "vxlt_gallowscourtyard_7gm_0_0.vlm"},
	{0x0E2065F197813875ULL, "vxw_bonepit_zo_0_0.vlm"},
	{0x0E20909F9357AED9ULL, "6079338_f.fxe"},
	{0x0E20CFEB514DCB7CULL, "295444963.ogg"},
	{0x0E20EC54691E075FULL, "sky_blank_exterior.mao"},
	{0x0E213A93CE54E729ULL, "6029057_m.fxe"},
	{0x0E216D1CA7231E4FULL, "art\\vfx\\environment\\level_fx\\bugs\\vfx_flys.anb"},
	{0x0E21B1D91CD7D062ULL, "845972467.ogg"},
	{0x0E21CC0D5F114A76ULL, "w_blightlands_49_0.msh"},
	{0x0E21D60D5F115B74ULL, "w_blightlands_49_0.mmh"},
	{0x0E21DA0D5F116187ULL, "w_blightlands_49_0.mao"},
	{0x0E21DD06FB6B95A0ULL, "vxlt_warehouse_v5_0_0.vlm"},
	{0x0E2265CF1A49A113ULL, "ser270_templar6127488.cl"},
	{0x0E22AAF0344C4C7BULL, "gen00fl_aveline6120777.cl"},
	{0x0E22AB28162DCF42ULL, "art\\characters\\creatures\\ragedemon\\textures\\c_ragedmon_0s.dds"},
	{0x0E22E57F0EC624A4ULL, "gencr_dalish_warrior_ws.utc"},
	{0x0E2317567FFCB5D2ULL, "vxht_keep_n_673_0_0.vlm"},
	{0x0E2322E24A3606F9ULL, "art\\levels\\races\\natural\\highlands\\props\\prp_01hgertes.phy"},
	{0x0E23A6EE167C7C9AULL, "308533721.ogg"},
	{0x0E23CF8C62CA3795ULL, "dae000_mother6002007.cl"},
	{0x0E240A0BA5245461ULL, "vxlt_0_lowtown_a2_gjm_0_0.vlm"},
	{0x0E240A410BC3AC9CULL, "vxht_mansion2b_333_0_0.vlm"},
	{0x0E240D2670F34BCCULL, "gen00fl_aveline6140907.cl"},
	{0x0E2551467D9B3ADEULL, "vxlt_darktown_2yi_0_0.vlm"},
	{0x0E2552F46678402FULL, "zz_isabellakiss.cut"},
	{0x0E25931FBA400DA8ULL, "6102764_m.fxe"},
	{0x0E25B5200B3B1291ULL, "vxw_encounter2_camp_7l_0_0.vlm"},
	{0x0E25D2781BF9E3F7ULL, "ser270_templar.bnk"},
	{0x0E2647B4AC7F07C9ULL, "vxlt_gallowstemplar_3s1_0_0.vlm"},
	{0x0E265EBA69966000ULL, "vxlt_gallowstemplar_5cv_0_0.vlm"},
	{0x0E26B5D8A94F7966ULL, "293216412.ogg"},
	{0x0E26CBCFD96E6FDEULL, "215262589.ogg"},
	{0x0E26D7166FEF7C27ULL, "framebuffereffects_distortion_p0.psh.fxo.dx11.assembly"},
	{0x0E27A5AC9FBE48CDULL, "253518188.ogg"},
	{0x0E28087DB09E2FD3ULL, "vxht_keep_n_629_0_0.vlm"},
	{0x0E28A82E2B78B7ACULL, "136340555.ogg"},
	{0x0E28B4781BFBE60FULL, "ser270_templar.cnv"},
	{0x0E2927A8F5F8C1D0ULL, "vxht_keep_n_3yb_0_0.vlm"},
	{0x0E294AAA36BF413FULL, "art\\characters\\creatures\\darkspawnogre\\animation\\combat\\walkbackleft.gad"},
	{0x0E29AD13EB91F9ABULL, "vxlt_0_lowtown_a2_n_c_euq_0_0.vlm"},
	{0x0E29B60744FABE90ULL, "w_sundermount_140_0.msh"},
	{0x0E29C40744FAD6AAULL, "w_sundermount_140_0.mmh"},
	{0x0E29C80744FADDE1ULL, "w_sundermount_140_0.mao"},
	{0x0E2A8C33DB4C7856ULL, "vxca_caverns1_mines_icr_0_0.vlm"},
	{0x0E2AE7EEA0F3E066ULL, "93924218.ogg"},
	{0x0E2AEDED470A729EULL, "182923908.ogg"},
	{0x0E2AFBEDC2AB7D71ULL, "vxht_mansion2b_3gv_0_0.vlm"},
	{0x0E2B0D4D77C6224AULL, "art\\characters\\playercharacter\\shared\\textures\\uv_color_guide.dds"},
	{0x0E2B107959A9CD19ULL, "vxht_0_hightown_a3_a2n_0_0.vlm"},
	{0x0E2BAE548963B123ULL, "6104336_m.fxe"},
	{0x0E2BCAF446D32822ULL, "lgt310cr_mag_melson.utc"},
	{0x0E2D2FBC05EBD3CAULL, "vxw_encounter2_p7_0_1.vlm"},
	{0x0E2DB319625795BBULL, "lgt220_mag_apost_apost16156739.cl"},
	{0x0E2E124EB37A1C5DULL, "vxht_0_hightown_9pa_0_0.vlm"},
	{0x0E2E4DF12D1B70C6ULL, "vxw_sundermount_2l5_0_0.vlm"},
	{0x0E2F37E34A80E6F7ULL, "vxht_bartrand_n_5mh_0_0.vlm"},
	{0x0E2F6153CC24FC2CULL, "88256946.ogg"},
	{0x0E2F6F78729C8354ULL, "art\\vfx\\combat\\vfx_r_arc_burst_sh_aoe_p.mmh"},
	{0x0E303FAA36C54D2CULL, "art\\characters\\creatures\\darkspawnogre\\animation\\combat\\walkbackleft.evt"},
	{0x0E3044C596A019AFULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_lute.phy"},
	{0x0E306562F5054D60ULL, "vxlt_darktown_41e_0_0.vlm"},
	{0x0E307ED68204E1E7ULL, "art\\vfx\\combat\\vfx_m_cre_conj_c.impa.gad"},
	{0x0E308E5BC0806476ULL, "947021027.ogg"},
	{0x0E309B28D8C3D90DULL, "gen00fl_anders6111080.cl"},
	{0x0E315696EBE25726ULL, "211630191.ogg"},
	{0x0E31C1C165DBF026ULL, "art\\vfx\\combat\\vfx_c_mer_burst_c.impa.evt"},
	{0x0E31DE24343C5C31ULL, "vxca_caverns2prc_9xc_0_0.vlm"},
	{0x0E32170028FB4DF3ULL, "6105379_m.fxe"},
	{0x0E3224285BD4F270ULL, "isa211_isabela6202182.cl"},
	{0x0E325144742DDDF1ULL, "vxlt_gallowstemplar_f_35d_0_0.vlm"},
	{0x0E32BDA11773C0A6ULL, "art\\vfx\\cinematic\\vfx_blood_smear_p.dura.ani"},
	{0x0E32C2D0E2FAB12EULL, "zz_one000_barber6179405.cl"},
	{0x0E33F0816F2238CAULL, "hlac_lt_0_lowtown_131070_3x.dds"},
	{0x0E340F6D5816724FULL, "6094966_f.fxe"},
	{0x0E3426FCFD312B11ULL, "vxw_encounter2_camp_2c_0_0.vlm"},
	{0x0E3455FE2D4FD42AULL, "6101491_m.fxe"},
	{0x0E34794FF2A79F6DULL, "vrc321ip_spooky5_obj_3.utp"},
	{0x0E34B4770DCA7AC6ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\reactions\\rct_fire_extreme_v1.evt"},
	{0x0E35007780383FA1ULL, "vxlt_gallowstemplar_39k_0_0.vlm"},
	{0x0E351F584BB4BB35ULL, "gen00fl_anders6096455.cl"},
	{0x0E3550DBCA810704ULL, "vxlt_0_lowtown_a2_ep4_0_0.vlm"},
	{0x0E35714776DF35BCULL, "6016902_m.fxe"},
	{0x0E35A191748372D6ULL, "vxht_mansion2b_35c_0_0.vlm"},
	{0x0E35A692284CEC72ULL, "zz_slate_3_baddy.cut"},
	{0x0E35C747E3A44B4DULL, "vxlt_backalley_g_1e3_0_0.vlm"},
	{0x0E35DC99E96BF5BCULL, "6163484_m.fxe"},
	{0x0E3646A253ADAB3AULL, "464368739.ogg"},
	{0x0E3671A2AE20CC4FULL, "art\\characters\\playercharacter\\shared\\heads\\textures\\mt_hed_fema_0d.dds"},
	{0x0E3673D682093AF4ULL, "art\\vfx\\combat\\vfx_m_cre_conj_c.impa.evt"},
	{0x0E36A7991B24AC07ULL, "vxdr_primevaltomb_81x_0_0.vlm"},
	{0x0E377B5850CA6C0EULL, "w_sundermount_78_0_lc.mmh"},
	{0x0E37AC1885DB3DD2ULL, "235794163.ogg"},
	{0x0E37D8C165E08371ULL, "art\\vfx\\combat\\vfx_c_mer_burst_c.impa.gad"},
	{0x0E3969A2A985055BULL, "vxht_mansion2b_2jz_0_0.vlm"},
	{0x0E3991E1DB02A4E4ULL, "vxlt_darktown_5ni_0_0.vlm"},
	{0x0E39A21F70BCF372ULL, "vxht_0_hightown_n_c7b_0_0.vlm"},
	{0x0E3A8351AC9EE3D5ULL, "341358766.ogg"},
	{0x0E3AB64CAD90B4E9ULL, "vxht_keep_73r_0_0.vlm"},
	{0x0E3ABE9FE98D0D78ULL, "6117263_m.fxe"},
	{0x0E3B036839691351ULL, "53954294.ogg"},
	{0x0E3B365A2A144DBCULL, "6050412_m.fxe"},
	{0x0E3B5C9C34FBB3F1ULL, "520490748.ogg"},
	{0x0E3BAD847A87227AULL, "554188506.ogg"},
	{0x0E3BCB770DD0C011ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\reactions\\rct_fire_extreme_v1.gad"},
	{0x0E3BD0ED1F163AD8ULL, "6166458_m.fxe"},
	{0x0E3BD5A0CD432C1EULL, "6096452_m.fxe"},
	{0x0E3BF9A536344EE0ULL, "vxw_bonepitmining_a2_1xd_0_1.vlm"},
	{0x0E3C4D8B42E5469FULL, "w_bonepit_42_0.rml"},
	{0x0E3C6E653BB2FA37ULL, "vxht_0_hightown_a2_c8u_0_0.vlm"},
	{0x0E3C9A158510C2F4ULL, "art\\levels\\races\\natural\\highlands\\props\\prp_02hgertes.phy"},
	{0x0E3CA4092125AD0FULL, "hlac_lt_0_lowtown_131072_4z.dds"},
	{0x0E3CBFCE1F118946ULL, "vxw_sundermount_2q6_0_0.vlm"},
	{0x0E3D8E1781D71CF5ULL, "logpile_d.dds"},
	{0x0E3DF7FF2173F981ULL, "869504405.ogg"},
	{0x0E3E27AA36D16185ULL, "art\\characters\\creatures\\darkspawnogre\\animation\\combat\\walkbackleft.ani"},
	{0x0E3EA83E6EAD1C7EULL, "art\\characters\\playercharacter\\qunarimale\\animation\\combat\\arishock\\hu_reaction_impale_en.evt"},
	{0x0E3ED9C165E6A31FULL, "art\\vfx\\combat\\vfx_c_mer_burst_c.impa.ani"},
	{0x0E3EDB0778D606ADULL, "zz_approval_debug6003680.cl"},
	{0x0E3F6CDA83A06F26ULL, "6038618_m.fxe"},
	{0x0E3FA5A1177E2217ULL, "art\\vfx\\cinematic\\vfx_blood_smear_p.dura.evt"},
	{0x0E3FB553A8BCB1D1ULL, "6150685_m.fxe"},
	{0x0E3FC1037E30AE8BULL, "zz_rdr_debug6066648.cl"},
	{0x0E3FC6AF309586BEULL, "rdr111dg_martin6015062.cl"},
	{0x0E4039C684D031FEULL, "dre000_varric6208061.cl"},
	{0x0E40895BD1856683ULL, "698310132.ogg"},
	{0x0E40F9F9595C9BF3ULL, "zz_followers_banter6157881.cl"},
	{0x0E41488C647915BEULL, "art\\levels\\races\\ferelden\\kirklow\\int\\_textures\\lt_paintings_backtoback_da.dds"},
	{0x0E4164438F97480EULL, "hlac_ca_caverns1_24422_0y.dds"},
	{0x0E4186D2F44E1213ULL, "vxlt_0_lowtown_a2_hfh_0_0.vlm"},
	{0x0E418CC43A01ECFFULL, "gen00fl_aveline6139762.cl"},
	{0x0E41B30C2A5F1D73ULL, "6161445_m.fxe"},
	{0x0E42018EA826921CULL, "vxlt_gallowstemplar_6t0_0_0.vlm"},
	{0x0E42877E85297468ULL, "vxlt_gallowstemplar_n_4sq_0_1.vlm"},
	{0x0E42CC770DD6E0BFULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\reactions\\rct_fire_extreme_v1.ani"},
	{0x0E42D9D11522B7E4ULL, "one350_variel6085544.cl"},
	{0x0E4308D6498846E5ULL, "art\\characters\\weapons\\quivers\\textures\\w_qui_qu101a_0d.dds"},
	{0x0E4321AE267E0A10ULL, "llac_w_bonepitmining_a2_0_2.dds"},
	{0x0E4342744D829100ULL, "vxlt_gallowstemplar_5it_0_0.vlm"},
	{0x0E4442B2B6636CA6ULL, "6113648_f.fxe"},
	{0x0E44503F0B0BD434ULL, "zz_qun_debug6028218.cl"},
	{0x0E446BD682156A2DULL, "art\\vfx\\combat\\vfx_m_cre_conj_c.impa.ani"},
	{0x0E44F1620195F0D5ULL, "6051875_f.fxe"},
	{0x0E44F3BA54804C27ULL, "dae000_madam_lusine6151389.cl"},
	{0x0E464D798E4BC253ULL, "zz_hawke_hightownnt.cut"},
	{0x0E4677158518DBF1ULL, "art\\levels\\races\\natural\\highlands\\props\\prp_02hgertes.msh"},
	{0x0E4685158518F30CULL, "art\\levels\\races\\natural\\highlands\\props\\prp_02hgertes.mao"},
	{0x0E469115851907AFULL, "art\\levels\\races\\natural\\highlands\\props\\prp_02hgertes.mmh"},
	{0x0E46A6A1178442A0ULL, "art\\vfx\\cinematic\\vfx_blood_smear_p.dura.gad"},
	{0x0E46BC2A78EFBADDULL, "6080176_f.fxe"},
	{0x0E4756C294439C00ULL, "706631450.ogg"},
	{0x0E4808A4F3489B56ULL, "6163284_m.fxe"},
	{0x0E4869A2F91BAADAULL, "art\\characters\\creatures\\arh_abm_rev\\c_abomination.phy"},
	{0x0E48B55F80AEB2D4ULL, "6082272_f.fxe"},
	{0x0E48D6AA5CC1F507ULL, "vxlt_0_lowtown_docks_bde_0_0.vlm"},
	{0x0E49043CB9A4CDA6ULL, "hlac_lt_warehouse_n_0_4z.dds"},
	{0x0E49312D453BBA64ULL, "6060737_f.fxe"},
	{0x0E496B78CFAEB919ULL, "hightown_male_noble_gossip6171386.cl"},
	{0x0E49C9702293292CULL, "hlac_ca_caverns1_24427_3y.dds"},
	{0x0E49E7B8390729C9ULL, "mcr370_guardsman6087667.cl"},
	{0x0E4A11F80108DCFAULL, "zz_vault_debug6177615.cl"},
	{0x0E4A742419A91F87ULL, "vxht_0_hightown_n_cb0_0_0.vlm"},
	{0x0E4AAEEDF4E7F39EULL, "hlac_w_encounter2_327679_0x.dds"},
	{0x0E4AE6E64DC96E73ULL, "hla_ht_0_hightown_a2_1_0x.dds"},
	{0x0E4AF644B23A1F09ULL, "rdr110dg_isabela6098928.cl"},
	{0x0E4B31AB1107A1C6ULL, "vxw_sundermount_2z7_0_0.vlm"},
	{0x0E4B3D3F17CC9638ULL, "432694206.ogg"},
	{0x0E4B62F03C3D2C91ULL, "vxca_caverns1_j4b_0_0.vlm"},
	{0x0E4B68DCCCD75EFCULL, "zz_mcr_debug6071698.cl"},
	{0x0E4B9D35F7386BCAULL, "6100104_m.fxe"},
	{0x0E4BEF2327E44C7BULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\9cc_rock1a.mmh"},
	{0x0E4BFB2327E46018ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\9cc_rock1a.mao"},
	{0x0E4C0D2327E47F65ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\9cc_rock1a.msh"},
	{0x0E4C68F24E4FEEF3ULL, "6037852_m.fxe"},
	{0x0E4C7E0B4BC9FB04ULL, "vxht_0_hightown_n_b22_0_0.vlm"},
	{0x0E4CB2876DCB9717ULL, "spit_spit.mao"},
	{0x0E4D180C18ECB126ULL, "hlac_ca_caverns2prc_5802_46z.dds"},
	{0x0E4D6DACEB4E3A73ULL, "6153260_m.fxe"},
	{0x0E4DA6F4D84FCDD4ULL, "mag360_alain6044617.cl"},
	{0x0E4E6C7859C9B467ULL, "vxca_caverns1_jgq_0_0.vlm"},
	{0x0E4E9C811CE19827ULL, "hlac_lt_gallowstemplar_2394_0y.dds"},
	{0x0E4F688B8C37BFF8ULL, "hlac_lt_0_lowtown_n_-65535_6y.dds"},
	{0x0E4F9EA4F482D707ULL, "vxht_keep_n_4an_0_0.vlm"},
	{0x0E4FF8AFE6442A93ULL, "vxlt_0_lowtown_a2_hck_0_0.vlm"},
	{0x0E50356307A58E24ULL, "420180243.ogg"},
	{0x0E504B0DCE7A0792ULL, "corpses_n.dds"},
	{0x0E50697AE4F44843ULL, "345473490.ogg"},
	{0x0E50F12ECD337248ULL, "art\\levels\\ztools\\default\\generic\\0wi_angle.mmh"},
	{0x0E50FD2ECD3386EBULL, "art\\levels\\ztools\\default\\generic\\0wi_angle.mao"},
	{0x0E510F2ECD33A572ULL, "art\\levels\\ztools\\default\\generic\\0wi_angle.msh"},
	{0x0E518C34F6118F97ULL, "hlac_lt_0_lowtown_docks_131070_1z.dds"},
	{0x0E51B4513F78A980ULL, "vxlt_gallowstemplar_5ru_0_0.vlm"},
	{0x0E51B6489D8E1253ULL, "follower_barks6153215.cl"},
	{0x0E527E054BF4EC0DULL, "gen00fl_sebastian6174703.cl"},
	{0x0E52A176F573FFAFULL, "6116559_m.fxe"},
	{0x0E52DACCECF9A85FULL, "vxlt_0_lowtown_docks_a3_n_cx0_0_0.vlm"},
	{0x0E52EC12181D612FULL, "166000882.ogg"},
	{0x0E52F7C6CBCB68F5ULL, "vxlt_0_lowtown_n_f5b_0_0.vlm"},
	{0x0E532C6D09D1115AULL, "ht_0_hightown_n_50_0.rml"},
	{0x0E5346666C4E3892ULL, "1066076473.ogg"},
	{0x0E53510B8C72340CULL, "11568203.ogg"},
	{0x0E53614A625AF76FULL, "llac_lt_0_lowtown_n_-2_6.dds"},
	{0x0E5380519272F955ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\3wo_short.phy"},
	{0x0E53D420B76F4FACULL, "hlac_lt_gallowscourtyard_n_-65538_1x.dds"},
	{0x0E543DEE473DD650ULL, "6110477_m.fxe"},
	{0x0E55333E3041F2F7ULL, "769325577.ogg"},
	{0x0E553756E2E81DB9ULL, "6076263_f.fxe"},
	{0x0E55854A5B14BE7DULL, "dre000_bodahn6031357.cl"},
	{0x0E559C445C7EE0A3ULL, "crate_03.dds"},
	{0x0E55B9D488581D54ULL, "787879008.ogg"},
	{0x0E55F6B248156010ULL, "vxw_encounter1_cp_0_0.vlm"},
	{0x0E55FF694E6EC710ULL, "vxlt_hangedman_n_rl_0_0.vlm"},
	{0x0E562162DB43DDEDULL, "6101225_m.fxe"},
	{0x0E563BAFD66A9BB5ULL, "967752300.ogg"},
	{0x0E56AD47F538E592ULL, "vxw_bonepit_21a_0_0.vlm"},
	{0x0E56D4C6729D0A7BULL, "art\\vfx\\combat\\vfx_q_ari_heal_act_c.mmh"},
	{0x0E57410674E7BFC3ULL, "658160999.ogg"},
	{0x0E5766C23E58383AULL, "vxca_caverns1_mines_i2y_0_0.vlm"},
	{0x0E5786F05AAD25B4ULL, "dae000cr_grand_cleric6148136.cl"},
	{0x0E579A0663965DE2ULL, "hlac_lt_0_lowtown_docks_-131075_4x.dds"},
	{0x0E57BC8B7BA9FA41ULL, "art\\levels\\races\\proxy\\props\\4tools\\vis_spike_wall.phy"},
	{0x0E57C5304E4FF5E0ULL, "cd_mag291_torpor.bnk"},
	{0x0E584BDAC00CB3C7ULL, "llac_ht_chantry_2534_17.dds"},
	{0x0E58727872BF8D0BULL, "art\\vfx\\combat\\vfx_r_arc_burst_sh_aoe_p.anb"},
	{0x0E58880E0AC0CC6EULL, "ico_crossbow_5.dds"},
	{0x0E58C3D5481115D3ULL, "hlac_lt_0_lowtown_a2_n_c_0_8y.dds"},
	{0x0E58CD4624CDD4E7ULL, "jagged_d.dds"},
	{0x0E590FE0D9323A70ULL, "llac_ht_0_hightown_a2_-65536_1.dds"},
	{0x0E591BBB4033DA87ULL, "163082636.ogg"},
	{0x0E592DDF7AFA778AULL, "mag291cr_torpor.utc"},
	{0x0E592E76D8C45746ULL, "6134590_m.fxe"},
	{0x0E59AE7277EA89EBULL, "6205367_m.fxe"},
	{0x0E59D0092610DCDFULL, "mag111pt_act_of_mercy.ncs"},
	{0x0E59E0986E4199CDULL, "ca_caverns1_24425_lc.mmh"},
	{0x0E59E42541F7B346ULL, "art\\characters\\playercharacter\\humanfemale\\bdy_masa.phy"},
	{0x0E5A0B3D43D69210ULL, "llac_lt_0_lowtown_docks_n_131070_1.dds"},
	{0x0E5A4D84B2D18AC9ULL, "vxht_0_hightown_aqz_0_0.vlm"},
	{0x0E5AD7465CAF17F1ULL, "565526286.ogg"},
	{0x0E5AF0F06B65B934ULL, "hlac_lt_0_lowtown_docks_-1_2x.dds"},
	{0x0E5B80C71590170AULL, "106538625.ogg"},
	{0x0E5BCD5383F9AD28ULL, "vxdr_primevaltomb_6vt_0_0.vlm"},
	{0x0E5BCEDC3F431052ULL, "lt_gallowscourtyard_n_50_0_lc.mmh"},
	{0x0E5BFD2204BA1613ULL, "water_water_p2.vsh.fxo.dx11.assembly"},
	{0x0E5C9B1717AE7A32ULL, "vxht_0_hightown_n_7g6_0_0.vlm"},
	{0x0E5CD31CCC7B90CEULL, "vxht_brothel_n_5h6_0_0.vlm"},
	{0x0E5D66C6022ED171ULL, "mag160_feynriel6037376.cl"},
	{0x0E5DC66B1725E852ULL, "300635177.ogg"},
	{0x0E5E684FA10C4FC0ULL, "w_sundermount_88_0.phy"},
	{0x0E5E6A8CD83A4313ULL, "vxlt_0_lowtown_a2_hhj_0_0.vlm"},
	{0x0E5EE1179DAD85A0ULL, "6096007_m.fxe"},
	{0x0E5F4B2914EE94EEULL, "24470962.ogg"},
	{0x0E5FC326D62CE19EULL, "hlac_lt_hangedman_n_-1_15x.dds"},
	{0x0E5FF616947F6FABULL, "fen210_fenris6109403.cl"},
	{0x0E604287D2A78828ULL, "art\\characters\\creatures\\gallowsslavestatue\\animation\\blendtree\\gallowsslavestatue_pose4.anb"},
	{0x0E608257F94B34FCULL, "vxht_mansion2b_315_0_0.vlm"},
	{0x0E60910C6A1BA776ULL, "vxca_caverns2_8o1_0_0.vlm"},
	{0x0E6109EAC5C8A2A1ULL, "llac_ht_keep_n_4519_0.dds"},
	{0x0E613C8DF19BD235ULL, "matdef_dc_normal.dds"},
	{0x0E613D8C854C5E30ULL, "390386713.ogg"},
	{0x0E61765E6BD809B8ULL, "6140951_m.fxe"},
	{0x0E6196C185ED055BULL, "art\\levels\\races\\elf\\props\\dalish\\prp_dream.msh"},
	{0x0E61A8C185ED23A6ULL, "art\\levels\\races\\elf\\props\\dalish\\prp_dream.mao"},
	{0x0E61B4C185ED3845ULL, "art\\levels\\races\\elf\\props\\dalish\\prp_dream.mmh"},
	{0x0E627C1FD78F32BAULL, "vxht_mansion2b_3bj_0_0.vlm"},
	{0x0E62C0397806EEE4ULL, "lt_warehouse_ll.mmh"},
	{0x0E62CC195C8278A8ULL, "hlac_ht_mansion_n_3373_0z.dds"},
	{0x0E62D5495A6D55C4ULL, "hlac_dr_primevaltomb_661_3x.dds"},
	{0x0E632A1FC169E459ULL, "art\\characters\\playercharacter\\humanmale\\animation\\ambient\\dialogue\\sitting_injured_pose_loop.gad"},
	{0x0E63B9ECECF2EE23ULL, "gen00fl_merrill6119611.cl"},
	{0x0E63DC1808356C1AULL, "6165545_m.fxe"},
	{0x0E640463FE7A5CD6ULL, "vxca_caverns2_801_0_1.vlm"},
	{0x0E642A28EC723CE8ULL, "zz_followers_banter6159459.cl"},
	{0x0E64688F3A0B7890ULL, "vxw_encounter1_hq_0_0.vlm"},
	{0x0E64874FA110F078ULL, "w_sundermount_88_0.rml"},
	{0x0E64DA919C4A5FC4ULL, "6013358_m.fxe"},
	{0x0E64EA6B2AB69FD8ULL, "zz_qun_debug6028302.cl"},
	{0x0E64F5C672A97FE0ULL, "art\\vfx\\combat\\vfx_q_ari_heal_act_c.anb"},
	{0x0E64F827D0883115ULL, "hlac_w_encounter3_-262148_0z.dds"},
	{0x0E662A46D3447296ULL, "dae200cr_viscount6087374.cl"},
	{0x0E662D91F06391DCULL, "6127659_m.fxe"},
	{0x0E664C4145B4DA7AULL, "6056618_m.fxe"},
	{0x0E673C4BA76234E8ULL, "vxlt_0_lowtown_f5d_0_0.vlm"},
	{0x0E6767EB99C58208ULL, "vxht_0_hightown_a3_c7f_0_0.vlm"},
	{0x0E676A48205CEC48ULL, "w_blightlands_166_0.mmh"},
	{0x0E677648205D00EBULL, "w_blightlands_166_0.mao"},
	{0x0E678848205D1F72ULL, "w_blightlands_166_0.msh"},
	{0x0E6789E929A07857ULL, "285747765.ogg"},
	{0x0E67A17B4A1137BEULL, "6162346_m.fxe"},
	{0x0E67D2176C736101ULL, "583368163.ogg"},
	{0x0E67F73F8F8F04CAULL, "161735566.ogg"},
	{0x0E68204F69E0A26BULL, "6208467_m.fxe"},
	{0x0E683D35BC23DFDCULL, "gen00fl_anders6111304.cl"},
	{0x0E684CAEB2932898ULL, "vxw_sundermount_351_0_0.vlm"},
	{0x0E69277B28288B90ULL, "vrc321st_spooky_scene_1.stg"},
	{0x0E698074B150E226ULL, "358607580.ogg"},
	{0x0E69CB55D278832BULL, "ser270_emeric6113588.cl"},
	{0x0E69E5A60E32A7CAULL, "vxlt_0_lowtown_n_exa_0_0.vlm"},
	{0x0E69F8121CA89037ULL, "6187939_m.fxe"},
	{0x0E6A3B1FC1701FA7ULL, "art\\characters\\playercharacter\\humanmale\\animation\\ambient\\dialogue\\sitting_injured_pose_loop.ani"},
	{0x0E6A6306A409F2E8ULL, "597288201.ogg"},
	{0x0E6A64B928DBAE6BULL, "6013322_m.fxe"},
	{0x0E6AC9BE863C9C32ULL, "vxdr_0_deeproads_fiu_0_0.vlm"},
	{0x0E6ADF739AAF3EA2ULL, "w_encounter1_47_0.msh"},
	{0x0E6AE1739AAF4258ULL, "w_encounter1_47_0.mmh"},
	{0x0E6AE2FF66BC8DC7ULL, "888904479.ogg"},
	{0x0E6AED739AAF56FBULL, "w_encounter1_47_0.mao"},
	{0x0E6B2344B4E58B43ULL, "gen00fl_aveline6132899.cl"},
	{0x0E6B6D952A501A9AULL, "vxht_0_hightown_ccm_0_0.vlm"},
	{0x0E6B9E23606EBFE6ULL, "6122368_m.fxe"},
	{0x0E6BEC33AE62180FULL, "vxlt_undercitydungeon_1iu_0_1.vlm"},
	{0x0E6C4FF003C5BB8DULL, "lit_up_05_l3.mao"},
	{0x0E6C54EC9D4AFB12ULL, "zz_one000_merchant6179348.cl"},
	{0x0E6C5EBDA077598EULL, "6005764_m.fxe"},
	{0x0E6CEACE26374C0BULL, "578937536.ogg"},
	{0x0E6D00BBCF41BAF7ULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_staffc_sa.dds"},
	{0x0E6D17543E92465AULL, "vxht_mansion_n_6kk_0_0.vlm"},
	{0x0E6D380F95724AA8ULL, "450525049.ogg"},
	{0x0E6D6F248A46F2E9ULL, "lt_0_lowtown_docks_n_25_0.rml"},
	{0x0E6E00F2EC37A529ULL, "zz_neel_test.cut"},
	{0x0E6E0B6EF7D92661ULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\a_strt.ani"},
	{0x0E6E1A641D23358BULL, "539043476.ogg"},
	{0x0E6E5F7B56154852ULL, "hlac_ca_caverns1_24424_6y.dds"},
	{0x0E6EF993AFB5F376ULL, "vxht_0_hightown_be6_0_0.vlm"},
	{0x0E6F027C0AF58B39ULL, "vxlt_gallowstemplar_n_37b_0_0.vlm"},
	{0x0E6F02E95C11BFF6ULL, "vxca_caverns2_8j2_0_0.vlm"},
	{0x0E6F247BEBD5897FULL, "hlac_lt_undercitydungeon_-1_39z.dds"},
	{0x0E6F5247C9E670DDULL, "rdr111dg_martin6015184.cl"},
	{0x0E6FE0FA3F88CE66ULL, "art\\characters\\playercharacter\\elffemale\\textures\\ef_glv_lgtp_0n.dds"},
	{0x0E7007C8A3B801FCULL, "vxlt_gallowstemplar_6s5_0_0.vlm"},
	{0x0E70DEB98C8AB30EULL, "vxw_bonepitdestroyed_a3_233_0_0.vlm"},
	{0x0E70F7F8E29614E7ULL, "vxw_encounter2_camp_or_0_0.vlm"},
	{0x0E7124D6FECCB539ULL, "vxlt_gallowstemplar_f_3sn_0_0.vlm"},
	{0x0E71793B54AD31F0ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\3ww_basic.mao"},
	{0x0E717D3B54AD3803ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\3ww_basic.mmh"},
	{0x0E718414DD514E84ULL, "6132926_f.fxe"},
	{0x0E718B3B54AD505DULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\3ww_basic.msh"},
	{0x0E72252864EAFDA1ULL, "vxlt_0_lowtown_bz0_0_0.vlm"},
	{0x0E7391DE0AD4B40AULL, "hlac_ht_keep_3732_1y.dds"},
	{0x0E74447851DCD28DULL, "6080654_m.fxe"},
	{0x0E7472243D85A372ULL, "vxht_0_hightown_a2_c1w_0_0.vlm"},
	{0x0E75166EF7DF5703ULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\a_strt.gad"},
	{0x0E756E9A2AC5AC46ULL, "533343636.ogg"},
	{0x0E7613583C07503EULL, "6167046_m.fxe"},
	{0x0E769783C108695BULL, "llac_lt_foundry_n_320_20.dds"},
	{0x0E76D60D5F599B77ULL, "w_blightlands_49_0.rml"},
	{0x0E77160689F9CA25ULL, "6101966_m.fxe"},
	{0x0E771C5CCA684FCFULL, "art\\levels\\atmosphere\\sky\\cloudmaterial.mao"},
	{0x0E7742441DD61F03ULL, "art\\characters\\creatures\\desiredemon\\animation\\kiss.ani"},
	{0x0E776140F98AB94CULL, "463900396.ogg"},
	{0x0E78294ADDE6A21AULL, "vxht_0_hightown_a3_asv_0_0.vlm"},
	{0x0E78403599678125ULL, "6168912_m.fxe"},
	{0x0E7857830028C04AULL, "vxlt_0_lowtown_n_eub_0_0.vlm"},
	{0x0E78605F53D10F17ULL, "6134292_m.fxe"},
	{0x0E794833BFC2AF9EULL, "hlac_lt_0_lowtown_n_-65535_1z.dds"},
	{0x0E797CE7FA250673ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\baseswordshield\\combat\\walkforwardleft.ani"},
	{0x0E79DD9B45ADA00FULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\reactions\\additive_damage_fr.gad"},
	{0x0E7B06229D1D3956ULL, "item_variations_col.gda"},
	{0x0E7B893130885EDAULL, "vxht_mansion_n_6nd_0_0.vlm"},
	{0x0E7C5B084EFD0FC9ULL, "t3_arm_klr.tnt"},
	{0x0E7CA9FCCC446560ULL, "6104277_f.fxe"},
	{0x0E7D8A2C844F8184ULL, "vxlt_0_lowtown_a2_eo9_0_0.vlm"},
	{0x0E7DD5B317FF1B7CULL, "vxca_caverns1_mines_j7c_0_0.vlm"},
	{0x0E7E6C8142D0A306ULL, "gen00tr_talk.ncs"},
	{0x0E7E9FB8A078226FULL, "art\\toolset\\tools_lightprobe.mao"},
	{0x0E7EABB8A078360CULL, "art\\toolset\\tools_lightprobe.mmh"},
	{0x0E7EB1B8A07840FEULL, "art\\toolset\\tools_lightprobe.msh"},
	{0x0E7F69D5D48C2D67ULL, "vxw_encounter2_camp_jm_0_0.vlm"},
	{0x0E7FF29B45B22FBCULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\reactions\\additive_damage_fr.evt"},
	{0x0E80985C7F1C146EULL, "vxca_caverns1_mines_e8w_0_0.vlm"},
	{0x0E8162D047D6C91AULL, "vxw_encounter2_qp_0_1.vlm"},
	{0x0E81651D98A4C571ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\9ww_litwindows.msh"},
	{0x0E81731D98A4DC8CULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\9ww_litwindows.mao"},
	{0x0E817F1D98A4F12FULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\9ww_litwindows.mmh"},
	{0x0E818435E223B82EULL, "gen00fl_isabela_init6076477.cl"},
	{0x0E81E2D63A5E9C95ULL, "che_wispy.mao"},
	{0x0E827E3B54BBAD28ULL, "art\\levels\\races\\ferelden\\kirklow\\ext\\4co_lowtown\\3ww_basic.phy"},
	{0x0E829EEABFBF42DFULL, "art\\characters\\creatures\\stonegolem\\animation\\kd_lp.gad"},
	{0x0E82A98713AAF4CCULL, "mer110_jansen6032610.cl"},
	{0x0E8367D4B07EE904ULL, "vxdr_primevalentrance_310_0_0.vlm"},
	{0x0E83854FA12B9D55ULL, "w_sundermount_88_0.msh"},
	{0x0E83874FA12BA0CBULL, "w_sundermount_88_0.mmh"},
	{0x0E83934FA12BB568ULL, "w_sundermount_88_0.mao"},
	{0x0E841189C8C14B5AULL, "6127309_m.fxe"},
	{0x0E845A441DE0D2FAULL, "art\\characters\\creatures\\desiredemon\\animation\\kiss.evt"},
	{0x0E85005EDD1FDD26ULL, "676193633.ogg"},
	{0x0E856DF1BB6D68D3ULL, "w_woundedcoast_29_0.anb"},
	{0x0E8601EFD271D3B0ULL, "hlac_lt_0_lowtown_a3_n_c_-65539_0z.dds"},
	{0x0E8617BB672D0C60ULL, "dae000_mother6103123.cl"},
	{0x0E8626FC8D027FADULL, "vxlt_undercityserialkiller_1yc_0_0.vlm"},
	{0x0E8636E34AB9E033ULL, "gen00fl_merrill6147025.cl"},
	{0x0E864C41695CB017ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\9ff_sandpile4a_2.mao"},
	{0x0E8674E7FA2F82AAULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\baseswordshield\\combat\\walkforwardleft.evt"},
	{0x0E8682B697A7D779ULL, "vxlt_0_lowtown_docks_b61_0_0.vlm"},
	{0x0E86AF7160CF3A8AULL, "6054371_m.fxe"},
	{0x0E874AD369D15D99ULL, "dae100_bran6072449.cl"},
	{0x0E88399E01D6BBF5ULL, "vxlt_warehouse_n_108_0_0.vlm"},
	{0x0E8845533FA0CF7AULL, "gen00fl_anders6125640.cl"},
	{0x0E88BF7B28574210ULL, "143308887.ogg"},
	{0x0E88CAE2B9DD4E75ULL, "hlac_lt_0_lowtown_n_131071_2x.dds"},
	{0x0E89F1EC712ED986ULL, "vxlt_undercityserialkiller_18o_0_0.vlm"},
	{0x0E8A418B4327538FULL, "w_bonepit_42_0.mao"},
	{0x0E8A4D8B4327682CULL, "w_bonepit_42_0.mmh"},
	{0x0E8A538B4327721EULL, "w_bonepit_42_0.msh"},
	{0x0E8A623D6F9CD6EFULL, "vxw_blightlands_ci_0_0.vlm"},
	{0x0E8A7593D619113AULL, "6031987_m.fxe"},
	{0x0E8AA85F97B816DBULL, "art\\levels\\races\\ferelden\\props\\lowerclass\\prp_barrelbasic_0.pwk"},
	{0x0E8B45E7929B6579ULL, "dae150_chandan6148637.cl"},
	{0x0E8B55EB7D217ECCULL, "227185722.ogg"},
	{0x0E8B59441DE6EFEDULL, "art\\characters\\creatures\\desiredemon\\animation\\kiss.gad"},
	{0x0E8B6B3D5751B532ULL, "hlac_lt_0_lowtown_65534_5y.dds"},
	{0x0E8B96B5B0A073F7ULL, "hlac_w_woundedcoast_-131071_0z.dds"},
	{0x0E8BB5850FA55583ULL, "mag200_feynriel6217339.cl"},
	{0x0E8C7E596CE0C07BULL, "vxlt_undercityserialkiller_1iu_0_0.vlm"},
	{0x0E8CD655C5C8F655ULL, "qun211ar_poison_encounter.are"},
	{0x0E8D53E7FA35691DULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\baseswordshield\\combat\\walkforwardleft.gad"},
	{0x0E8D872C601D63D7ULL, "vxca_cavern1_haunted_fm6_0_0.vlm"},
	{0x0E8DDBB2C68245E7ULL, "vxw_encounter2_camp_el_0_0.vlm"},
	{0x0E8DF45720129001ULL, "vxlt_0_lowtown_docks_n_8pz_0_0.vlm"},
	{0x0E8DFA9B45BE7AD5ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\reactions\\additive_damage_fr.ani"},
	{0x0E8E88B7278E75A3ULL, "vxw_woundedcoast_25_0_1.vlm"},
	{0x0E8ED4479D7A4B06ULL, "one300_murder_followers6089903.cl"},
	{0x0E8F1604937CBFF3ULL, "dae000_arishok6080476.cl"},
	{0x0E8F3F0DC9D52A50ULL, "142464900.ogg"},
	{0x0E8F74D949B07287ULL, "vxw_bonepitdestroyed_a3_1gm_0_0.vlm"},
	{0x0E8FA0FDC8B672A8ULL, "6051147_m.fxe"},
	{0x0E901E1044081E99ULL, "follower_banter6166966.cl"},
	{0x0E90274D153A79C9ULL, "6120822_m.fxe"},
	{0x0E90484D47F7C1CDULL, "dae000ip_lean_back_d.utp"},
	{0x0E9064DF5509EED3ULL, "blood_clu.mao"},
	{0x0E912B5750751E5CULL, "lowtown_male_lowlife1_sta6165030.cl"},
	{0x0E91684FA137A930ULL, "w_sundermount_88_0.anb"},
	{0x0E9188B07AFE1653ULL, "17732377.ogg"},
	{0x0E91AD010AF0A491ULL, "6045966_f.fxe"},
	{0x0E91D0DDBF2D01F1ULL, "696701112.ogg"},
	{0x0E92E1258EDE7361ULL, "6102570_m.fxe"},
	{0x0E92F36BDF4788E8ULL, "hlac_lt_0_lowtown_docks_65532_0y.dds"},
	{0x0E93218EC80BD5FDULL, "vxlt_0_lowtown_a2_n_c_gv8_0_0.vlm"},
	{0x0E9370F1BB79AB4EULL, "w_woundedcoast_29_0.msh"},
	{0x0E937EF1BB79C29FULL, "w_woundedcoast_29_0.mao"},
	{0x0E9386E90B03039DULL, "vxw_sundermount_30o_0_0.vlm"},
	{0x0E938AF1BB79D73CULL, "w_woundedcoast_29_0.mmh"},
	{0x0E93B9A686108ED7ULL, "lgt311_shw_mer_mettin6156548.cl"},
	{0x0E93BA9372606F39ULL, "hlac_lt_0_lowtown_docks_n_-3_0z.dds"},
	{0x0E93CE735FD3F0C1ULL, "6048419_f.fxe"},
	{0x0E9459A0A3AE15A3ULL, "dae011cr_swindler.utc"},
	{0x0E947C7916F44C46ULL, "gen00fl_anders6125121.cl"},
	{0x0E949D2BB3A3FDE3ULL, "6015971_m.fxe"},
	{0x0E94AE4F3A1FF78AULL, "357835676.ogg"},
	{0x0E950033C869DCD2ULL, "vxlt_gallowstemplar_3fu_0_0.vlm"},
	{0x0E95590BA7995B7FULL, "6097151_m.fxe"},
	{0x0E955BEFEDBC6A9DULL, "220712573.ogg"},
	{0x0E959A5DE74AF6B3ULL, "6170708_m.fxe"},
	{0x0E959B3F3920EE44ULL, "art\\characters\\playercharacter\\dwarfmale\\textures\\dm_boo_lgta_0n.dds"},
	{0x0E95A9F1CDBBFF87ULL, "art\\vfx\\combat\\vfx_sg_hurl_act_c.anb"},
	{0x0E964DAA5670ED7BULL, "650403436.ogg"},
	{0x0E967BEABFCFB025ULL, "art\\characters\\creatures\\stonegolem\\animation\\kd_lp.ani"},
	{0x0E9690F5813AD66AULL, "llac_ht_mansion_n_3370_1.dds"},
	{0x0E96F755C5D182BBULL, "qun211ar_poison_encounter.ncs"},
	{0x0E9749DF6FC4A5B0ULL, "6076122_m.fxe"},
	{0x0E974B4C4EB07CE5ULL, "1050182349.ogg"},
	{0x0E9756E3C463C7A9ULL, "vxht_mansion2_31d_0_0.vlm"},
	{0x0E97A19C8F4D1EA1ULL, "vxht_0_hightown_a2_a2s_0_0.vlm"},
	{0x0E980109B6B1C09FULL, "vxdr_0_deeproads_cxf_0_0.vlm"},
	{0x0E989169D9FE2836ULL, "186338304.ogg"},
	{0x0E98A27075E15EA4ULL, "hlac_lt_gallowstemplar_2383_1y.dds"},
	{0x0E98A769E876D780ULL, "hlac_lt_0_lowtown_a2_131072_6y.dds"},
	{0x0E98D41A6192EF6FULL, "vxw_blightlands_dh_0_0.vlm"},
	{0x0E98F93CD6F17981ULL, "100564434.ogg"},
	{0x0E99133A83EB8DD9ULL, "6016947_m.fxe"},
	{0x0E9995C52F6DDA2FULL, "art\\levels\\races\\elf\\props\\_textures\\prp_evilchya_sa.dds"},
	{0x0E9A184D2B4F6226ULL, "6053161_m.fxe"},
	{0x0E9A935192AF1C70ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\3wo_short.msh"},
	{0x0E9AA15192AF338AULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\3wo_short.mmh"},
	{0x0E9AA55192AF3AC1ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\3wo_short.mao"},
	{0x0E9AF0365ED6D8FBULL, "vxlt_undercityserialkiller_1nt_0_0.vlm"},
	{0x0E9AF908724DC975ULL, "vxlt_0_lowtown_g9l_0_0.vlm"},
	{0x0E9BC4A3B8DF5787ULL, "vxdr_primevaltomb_8nq_0_0.vlm"},
	{0x0E9C47764B284F19ULL, "6108943_m.fxe"},
	{0x0E9C6107889589D5ULL, "vxht_0_hightown_a3_c3r_0_1.vlm"},
	{0x0E9C768C54F31FFFULL, "art\\characters\\playercharacter\\dwarfmale\\glv_lgtb.phy"},
	{0x0E9D1930BF825EF6ULL, "6024568_f.fxe"},
	{0x0E9D7FF84173F3AAULL, "vxw_sundermount_2k8_0_0.vlm"},
	{0x0E9D8A7918CDD0F4ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\hugging_kill_action.ani"},
	{0x0E9D94FB60268235ULL, "858244772.ogg"},
	{0x0E9DD755C5D76B62ULL, "qun211ar_poison_encounter.lst"},
	{0x0E9DE6B63BA68B07ULL, "vxw_bonepitdestroyed_a3_1hl_0_0.vlm"},
	{0x0E9E2FA29D7598BFULL, "6139817_m.fxe"},
	{0x0E9E6EE3F4E0D694ULL, "94625899.ogg"},
	{0x0E9EA35826A3937EULL, "llac_lt_0_lowtown_n_-65535_5.dds"},
	{0x0E9F169931483CEAULL, "746608687.ogg"},
	{0x0E9F1DB552E85975ULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_gear_n.dds"},
	{0x0E9F273806D399ADULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\3wo12_galofliplo.mmh"},
	{0x0E9F2B3806D3A0FEULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\3wo12_galofliplo.mao"},
	{0x0E9F393806D3B833ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\3wo12_galofliplo.msh"},
	{0x0E9FBA2021E3F8CEULL, "209383272.ogg"},
	{0x0EA02C70A7A1EB4DULL, "417638619.ogg"},
	{0x0EA06E248A724A09ULL, "lt_0_lowtown_docks_n_25_0.anb"},
	{0x0EA09B4B4329469BULL, "6005400_m.fxe"},
	{0x0EA0CE5CABA0CEC3ULL, "6128958_m.fxe"},
	{0x0EA0E744E5EBA484ULL, "6089573_m.fxe"},
	{0x0EA168C84A71E8BCULL, "vxlt_hangedman_tc_0_0.vlm"},
	{0x0EA183600E54645BULL, "art\\characters\\playercharacter\\humanmale\\textures\\bdy_arm_masb_0s.dds"},
	{0x0EA18D225F7471AEULL, "art\\vfx\\combat\\vfx_gib_hlk_bld_p_v2.mmh"},
	{0x0EA1C194FFEE82A3ULL, "6042979_m.fxe"},
	{0x0EA270869BD8C123ULL, "hlac_lt_0_lowtown_n_-65537_4z.dds"},
	{0x0EA2A910DA7F9DA9ULL, "157399160.ogg"},
	{0x0EA2DBDAC065741EULL, "vfx_distortionmask_p0.psh.fxo"},
	{0x0EA2F88122A12E53ULL, "hlac_dr_slavers_3216_9z.dds"},
	{0x0EA30AB670EEB0ADULL, "vxlt_undercityserialkiller_1sa_0_0.vlm"},
	{0x0EA3766A8FAD23DFULL, "738537602.ogg"},
	{0x0EA37B6119973993ULL, "zz_dae_debug6029849.cl"},
	{0x0EA380B41BB02DC6ULL, "6103062_m.fxe"},
	{0x0EA3877918D23756ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\hugging_kill_action.gad"},
	{0x0EA3D5AC8DFB8CB7ULL, "plt_vrc300pt_n_rival_max.nss"},
	{0x0EA3E1F3E71DB0D6ULL, "118740193.ogg"},
	{0x0EA441D52572A615ULL, "m_dae321ar_chantry.bnk"},
	{0x0EA462A2F969FD03ULL, "art\\characters\\creatures\\arh_abm_rev\\c_abomination.msh"},
	{0x0EA470A2F96A155DULL, "art\\characters\\creatures\\arh_abm_rev\\c_abomination.mmh"},
	{0x0EA49C36904A70E8ULL, "hightown_market_guard_sta6174611.cl"},
	{0x0EA5B3641A1028A5ULL, "zz_followers_banter6158103.cl"},
	{0x0EA5C8C0B659E029ULL, "vxht_mansion2_34k_0_0.vlm"},
	{0x0EA60D6F0B4ED177ULL, "dae000_bodhan6146809.cl"},
	{0x0EA66FB5FA102227ULL, "392129461.ogg"},
	{0x0EA672E6A8A7D91FULL, "vxdr_0_deeproads_cua_0_0.vlm"},
	{0x0EA675D08BA3B6F6ULL, "6123389_m.fxe"},
	{0x0EA69B9588F49B1EULL, "hlac_lt_darktown_-1_4x.dds"},
	{0x0EA6C74A59139171ULL, "qun221_viscount6095594.cl"},
	{0x0EA6E0E8D9B614C9ULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_gibbet_dp.dds"},
	{0x0EA70D262EF7A0F0ULL, "bet131ip_chest_gift.utp"},
	{0x0EA7A3574F18D9F8ULL, "art\\characters\\creatures\\textures\\cn_gut_0n.dds"},
	{0x0EA800F93B658D99ULL, "vxht_mansion2_2eu_0_0.vlm"},
	{0x0EA80E15C44699D4ULL, "vxht_mansion_n_288_0_0.vlm"},
	{0x0EA88A3844B9D85FULL, "art\\vfx\\combat\\vfx_m_sta_phys_rng_proj_p.mmh"},
	{0x0EA90CF6E518FDC5ULL, "ht_0_hightown_a2_26_0_lc.mmh"},
	{0x0EA92572E4430E46ULL, "hlac_w_bonepitdestroyed_a3_393220_0y.dds"},
	{0x0EA92FAC062E173BULL, "llac_lt_0_lowtown_docks_a3_0_4.dds"},
	{0x0EA953BD885E09C4ULL, "dae000_hate_enemies6167576.cl"},
	{0x0EA98DC1862AABF2ULL, "art\\levels\\races\\elf\\props\\dalish\\prp_dream.phy"},
	{0x0EA9C9080EF6E7B6ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\shield and weapon\\sw_dmg_bonus_stagger.evt"},
	{0x0EAA3680AAD57007ULL, "vxdr_primevaltomb_8ip_0_0.vlm"},
	{0x0EAAA27918D883B5ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\custom\\hugging_kill_action.evt"},
	{0x0EAB24EDACEEC43DULL, "792711765.ogg"},
	{0x0EAB4570275B05AEULL, "6133146_m.fxe"},
	{0x0EABF1D5336A0C2AULL, "vxw_sundermount_2l9_0_0.vlm"},
	{0x0EAC19B37B09FE55ULL, "llac_dr_primevaltomb_9621_7.dds"},
	{0x0EAC8F6FF09B12D9ULL, "gen00fl_anders6113002.cl"},
	{0x0EACC71118159221ULL, "art\\vfx\\combat\\vfx_gib_hm_meda_rtl.anb"},
	{0x0EACF6582A5FE36CULL, "vxlt_foundry_n_zj_0_0.vlm"},
	{0x0EACFA0E39E50B81ULL, "gen00fl_fenris6126624.cl"},
	{0x0EAD44B0207AB78AULL, "6179917_m.fxe"},
	{0x0EAD633CCDF63621ULL, "fxe_caustics_dif.dds"},
	{0x0EAD778C55019404ULL, "art\\characters\\playercharacter\\dwarfmale\\glv_lgtb.mmh"},
	{0x0EAD7B8C55019B57ULL, "art\\characters\\playercharacter\\dwarfmale\\glv_lgtb.mao"},
	{0x0EAD8D8C5501B9A6ULL, "art\\characters\\playercharacter\\dwarfmale\\glv_lgtb.msh"},
	{0x0EADF3B4FA7CA087ULL, "vxlt_0_lowtown_docks_b9b_0_0.vlm"},
	{0x0EAE33E4551865D2ULL, "6133821_m.fxe"},
	{0x0EAE612FBFB5386BULL, "vxht_keep_n_36t_0_0.vlm"},
	{0x0EAEA271749E2D22ULL, "vxlt_blackemporium_8m_0_0.vlm"},
	{0x0EAEC453BB9B0E1BULL, "vxlt_warehouse_n_115_0_0.vlm"},
	{0x0EAED310949FD0C7ULL, "hlac_dr_dungeon_3216_1x.dds"},
	{0x0EAFAC225F80E3ADULL, "art\\vfx\\combat\\vfx_gib_hlk_bld_p_v2.anb"},
	{0x0EAFB4BCE6EA6A7FULL, "lt_0_lowtown_a2_n_25_0_lc.mmh"},
	{0x0EB0000F7DD2528BULL, "vxlt_gallowstemplar_2xs_0_0.vlm"},
	{0x0EB00C352F8AE41EULL, "6041139_m.fxe"},
	{0x0EB017BA23E8FAAEULL, "dae300_cullen6140646.cl"},
	{0x0EB0203806E2016AULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\3wo12_galofliplo.phy"},
	{0x0EB042CAA96505E7ULL, "gen_im_npc_arm_hum_glv_krkb.uti"},
	{0x0EB0777C4F0CBAB8ULL, "6149450_f.fxe"},
	{0x0EB0837ED810C8D7ULL, "93258327.ogg"},
	{0x0EB0C0080EFCF741ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\shield and weapon\\sw_dmg_bonus_stagger.gad"},
	{0x0EB0E1F044CE82D7ULL, "aip_spidercorrupted_base.gda"},
	{0x0EB17340C4E3C1CCULL, "hlac_w_encounter3_-131076_0z.dds"},
	{0x0EB18AF1BB92D48FULL, "w_woundedcoast_29_0.rml"},
	{0x0EB1CB83F4EC5723ULL, "zz_fol_debug6057010.cl"},
	{0x0EB1D69AFD162A78ULL, "6117432_m.fxe"},
	{0x0EB1DDA77FB7A1A9ULL, "vxlt_warehouse_qu_0_0.vlm"},
	{0x0EB1EDF376F3AD3BULL, "403058786.ogg"},
	{0x0EB248FB1E1ABC0BULL, "vxca_caverns1_mines_edb_0_0.vlm"},
	{0x0EB3ABB635EC7D4BULL, "22027622.ogg"},
	{0x0EB42753CFFA32E6ULL, "259643081.ogg"},
	{0x0EB430BD493BABE3ULL, "27529987.ogg"},
	{0x0EB45407CDC323EDULL, "dae000ip_crowd_mf4_talk5.utp"},
	{0x0EB522ED5B17740BULL, "vxdr_primevalentrance_230_0_0.vlm"},
	{0x0EB54F045BF8B3F3ULL, "188562319.ogg"},
	{0x0EB5B7D4457F206FULL, "vxw_blightlands_rj_0_0.vlm"},
	{0x0EB66D15AFD1A91EULL, "6122764_m.fxe"},
	{0x0EB672D62D5BA619ULL, "vxht_mansion2_2nt_0_0.vlm"},
	{0x0EB67B8FE87F0CB7ULL, "hlac_lt_undercityserialkiller_-1_9z.dds"},
	{0x0EB6822E52D86F72ULL, "vxca_caverns2_4zz_0_0.vlm"},
	{0x0EB6BB4715E9F445ULL, "6028843_m.fxe"},
	{0x0EB6CF05F066D376ULL, "6135180_m.fxe"},
	{0x0EB6FDA4057321FAULL, "vxlt_0_lowtown_a2_n_h2j_0_0.vlm"},
	{0x0EB76AEF35F5C3B8ULL, "wm_ico_keep_roll.dds"},
	{0x0EB7B1D97083B2B2ULL, "6118192_m.fxe"},
	{0x0EB7C1080F0317EFULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\warrior\\shield and weapon\\sw_dmg_bonus_stagger.ani"},
	{0x0EB819CB17D6116EULL, "273448024.ogg"},
	{0x0EB8652F9B52BA63ULL, "vxlt_gallowstemplar_4qf_0_0.vlm"},
	{0x0EB86C9050771CD4ULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\a_atk_v1.ani"},
	{0x0EB96541B25B6098ULL, "hlac_ht_0_hightown_-65534_0y.dds"},
	{0x0EB9861A10AFB5B1ULL, "mag121_viveka6083365.cl"},
	{0x0EB9C58B7BFCD3B6ULL, "art\\levels\\races\\proxy\\props\\4tools\\vis_spike_wall.mmh"},
	{0x0EBAACD51C3E1B63ULL, "6043868_m.fxe"},
	{0x0EBB5AADC9EC89A6ULL, "6113059_f.fxe"},
	{0x0EBB68351C55FBECULL, "vxlt_foundry_n_uk_0_0.vlm"},
	{0x0EBB8C6374B15BA6ULL, "lgt210_qun_blk_qunari6142328.cl"},
	{0x0EBBB18FB93C66E3ULL, "6107276_m.fxe"},
	{0x0EBBD032DDC432A6ULL, "6107440_m.fxe"},
	{0x0EBC2A50076EEB60ULL, "vxlt_warehouse_n_oc_0_0.vlm"},
	{0x0EBCF6BCD4E9DEBFULL, "809287636.ogg"},
	{0x0EBD13A8B99B5C1DULL, "one300_zevran6089110.cl"},
	{0x0EBD3630AD91269BULL, "vxlt_warehouse_n_164_0_0.vlm"},
	{0x0EBD78E10ADE8530ULL, "vxht_0_hightown_a2_b7f_0_0.vlm"},
	{0x0EBDCAFEC9D7D584ULL, "6087773_m.fxe"},
	{0x0EBE02189D3B4AAEULL, "vxw_sundermount_2w8_0_0.vlm"},
	{0x0EBEA6F1CDDEFF88ULL, "art\\vfx\\combat\\vfx_sg_hurl_act_c.mmh"},
	{0x0EBEC03EC86FAF6CULL, "vxlt_gallowstemplar_f_5qc_0_0.vlm"},
	{0x0EBF4E5B1AA03B1AULL, "6028163_m.fxe"},
	{0x0EBF6990507D3636ULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\a_atk_v1.gad"},
	{0x0EC035A4C0891F4EULL, "vxht_0_hightown_n_c0k_0_0.vlm"},
	{0x0EC051D3EB1D97AAULL, "6077670_f.fxe"},
	{0x0EC069FB049C2953ULL, "hlac_ht_0_hightown_-65536_4x.dds"},
	{0x0EC0D047FACE033FULL, "art\\vfx\\combat\\vfx_v_arc_single_proj_c.cess.gad"},
	{0x0EC178EAE705F01BULL, "zz_and_debug6134047.cl"},
	{0x0EC1DF3EDEED5315ULL, "dae300_bran6143453.cl"},
	{0x0EC2C3E5E7CAD2F1ULL, "vxlt_gallowstemplar_f_3gr_0_0.vlm"},
	{0x0EC2DBB395D028D7ULL, "6103980_m.fxe"},
	{0x0EC2EFA4B8C9B92FULL, "6032828_m.fxe"},
	{0x0EC32178DB627C98ULL, "6172824_m.fxe"},
	{0x0EC33C9A8FF50A2FULL, "523705500.ogg"},
	{0x0EC346E6AD8B445DULL, "359531418.ogg"},
	{0x0EC3CE78078D0A33ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\pushed_from_behind.ani"},
	{0x0EC3DD97304217B7ULL, "300919988.ogg"},
	{0x0EC3F1400A74BD90ULL, "vxca_caverns1_mines_fmm_0_0.vlm"},
	{0x0EC490D3FBFA7A80ULL, "llac_ht_keep_3732_2.dds"},
	{0x0EC4B89CC027E22FULL, "dae000_loyal_mage.bnk"},
	{0x0EC4E035DB691348ULL, "vxht_chantry_n_2ta_0_1.vlm"},
	{0x0EC4FC159557A975ULL, "hlac_lt_undercitydungeon_-1_32y.dds"},
	{0x0EC546F1CEEDA053ULL, "vxlt_gallowstemplar_3rt_0_0.vlm"},
	{0x0EC584905081CF95ULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\a_atk_v1.evt"},
	{0x0EC623B66279CB32ULL, "6117092_m.fxe"},
	{0x0EC641705A1B7D60ULL, "llac_lt_gallowstemplar_2391_9.dds"},
	{0x0EC66589CF31E49DULL, "ave121_brennan6105817.cl"},
	{0x0EC6C51474441751ULL, "vxlt_0_lowtown_a2_n_c_eht_0_0.vlm"},
	{0x0EC7012A96354403ULL, "hlac_w_sundermount_-196610_0z.dds"},
	{0x0EC702821B96EE2EULL, "w_bonepit_51_0.msh"},
	{0x0EC70AFFB1D681E1ULL, "llac_ca_caverns1_24424_13.dds"},
	{0x0EC710821B97067FULL, "w_bonepit_51_0.mao"},
	{0x0EC71C821B971A1CULL, "w_bonepit_51_0.mmh"},
	{0x0EC75BA76AFF4FD2ULL, "6064390_m.fxe"},
	{0x0EC76E139D59A32CULL, "6133849_m.fxe"},
	{0x0EC7C547FAD40F2CULL, "art\\vfx\\combat\\vfx_v_arc_single_proj_c.cess.evt"},
	{0x0EC7DA9CC02A5197ULL, "dae000_loyal_mage.cnv"},
	{0x0EC827349CFE8854ULL, "hlac_lt_0_lowtown_docks_131071_7x.dds"},
	{0x0EC8BDB45D190B71ULL, "hlac_lt_0_lowtown_a2_1_9x.dds"},
	{0x0ECB04118FA3C904ULL, "and321tr_distract.utt"},
	{0x0ECBEABDFCD49DB0ULL, "vxht_0_hightown_a2_b8g_0_0.vlm"},
	{0x0ECC44A59A8F71D5ULL, "vxlt_backalley_n_1fk_0_0.vlm"},
	{0x0ECC6DA7DDC25601ULL, "vxlt_gallowstemplar_5iy_0_0.vlm"},
	{0x0ECC73F58F31632EULL, "vxw_sundermount_2x9_0_0.vlm"},
	{0x0ECD242B1FB19FC6ULL, "136513656.ogg"},
	{0x0ECD477E21538F75ULL, "dae321ar_player_home_nt.are"},
	{0x0ECD5C4D737A500DULL, "llac_lt_blackemporium_1055_16.dds"},
	{0x0ECD88236DC2189AULL, "6054069_m.fxe"},
	{0x0ECD8BE9636F2207ULL, "215779781.ogg"},
	{0x0ECDA2305D7CAE8CULL, "6162528_m.fxe"},
	{0x0ECDA94496975E56ULL, "vxlt_0_lowtown_a2_n_eox_0_0.vlm"},
	{0x0ECDFCFB81FD9F07ULL, "art\\vfx\\combat\\vfx_glb_com_stagger_s2_c.impa.gad"},
	{0x0ECE25A17968CB80ULL, "ch2_dy.psh.fxo"},
	{0x0ECEB74A0DCCF636ULL, "w_encounter2_camp_19_0.msh"},
	{0x0ECEC14A0DCD0734ULL, "w_encounter2_camp_19_0.mmh"},
	{0x0ECEC54A0DCD0E47ULL, "w_encounter2_camp_19_0.mao"},
	{0x0ECEC7A790423ED2ULL, "vxlt_gallowstemplar_3rq_0_0.vlm"},
	{0x0ECFB7CF7A67BA42ULL, "vxlt_hangedman_1th_0_0.vlm"},
	{0x0ED00963B4E1ED07ULL, "gen_im_cns_tom_attribute_01.uti"},
	{0x0ED017C0E1CD10EAULL, "vxca_caverns1_iog_0_0.vlm"},
	{0x0ED037B50A9B9CD7ULL, "vxlt_0_lowtown_n_by8_0_0.vlm"},
	{0x0ED067D52598219FULL, "m_dae321ar_chantry.wbd"},
	{0x0ED08B3844DB2B9CULL, "art\\vfx\\combat\\vfx_m_sta_phys_rng_proj_p.anb"},
	{0x0ED08EF07A209E88ULL, "w_encounter3_250_0.mmh"},
	{0x0ED099FD3DBC148BULL, "vxlt_hangedman_2hj_0_0.vlm"},
	{0x0ED09AF07A20B32BULL, "w_encounter3_250_0.mao"},
	{0x0ED0ACF07A20D1B2ULL, "w_encounter3_250_0.msh"},
	{0x0ED1C469A41CDCCDULL, "llac_lt_0_lowtown_a3_n_c_1_10.dds"},
	{0x0ED1C67807993A6AULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\pushed_from_behind.evt"},
	{0x0ED2107FE66DDCB0ULL, "376412146.ogg"},
	{0x0ED22C7CB32BA036ULL, "art\\vfx\\environment\\level_fx\\ambient\\vfx_cenotaphfire_day.mmh"},
	{0x0ED235ECE66D8551ULL, "496269977.ogg"},
	{0x0ED2503E5E1F9ABDULL, "546050065.ogg"},
	{0x0ED25356B4274F12ULL, "qun110_petrice6042298.cl"},
	{0x0ED27694F2B3C442ULL, "vxht_mansion_n_5mj_0_0.vlm"},
	{0x0ED29515FA75F2FAULL, "684513353.ogg"},
	{0x0ED328DA315B4DF0ULL, "vxca_caverns2prc_aie_0_0.vlm"},
	{0x0ED33E2D488875AFULL, "956066678.ogg"},
	{0x0ED3BD84A9F971C2ULL, "ss_templars_f_1.bnk"},
	{0x0ED419489155ED19ULL, "6152162_m.fxe"},
	{0x0ED4218E667A74E0ULL, "w_sundermount_152_0.phy"},
	{0x0ED429AC86C17F47ULL, "vxht_0_hightown_a3_c84_0_0.vlm"},
	{0x0ED43D92D884FFFAULL, "gen00fl_aveline6053326.cl"},
	{0x0ED4AC8EEFB0780EULL, "pro000st_colonel_ewald_1.stg"},
	{0x0ED4B7AA26AF3AFBULL, "vxlt_undercityserialkiller_1rh_0_0.vlm"},
	{0x0ED4BB72FA13CB66ULL, "6106511_m.fxe"},
	{0x0ED4F1FB8203AB14ULL, "art\\vfx\\combat\\vfx_glb_com_stagger_s2_c.impa.evt"},
	{0x0ED51F7B47AC496FULL, "vxw_sundermount_36r_0_1.vlm"},
	{0x0ED5693C3DA6292BULL, "819379552.ogg"},
	{0x0ED58C1780B7B987ULL, "vxdr_primevaltomb_8zm_0_0.vlm"},
	{0x0ED5AD47FAE02385ULL, "art\\vfx\\combat\\vfx_v_arc_single_proj_c.cess.ani"},
	{0x0ED6672BCBE2FFB0ULL, "vxht_0_hightown_n_ap7_0_0.vlm"},
	{0x0ED6B17D5DB9741FULL, "sib111_mother6121433.cl"},
	{0x0ED735B501AAE5E9ULL, "6121325_f.fxe"},
	{0x0ED73B4606BDDB6DULL, "llac_lt_0_lowtown_a2_n_c_-3_5.dds"},
	{0x0ED7A9800D7E175FULL, "llac_ca_caverns1_mines_24426_14.dds"},
	{0x0ED7ADFC2ED405BFULL, "vxlt_foundry_n_ze_0_0.vlm"},
	{0x0ED84E210F1812DBULL, "vxht_mansion2b_25w_0_0.vlm"},
	{0x0ED8A578079F20DDULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\pushed_from_behind.gad"},
	{0x0ED8B0D26695CEEEULL, "vxlt_hangedman_20s_0_0.vlm"},
	{0x0ED8E608CDD4C872ULL, "779426117.ogg"},
	{0x0ED8F948EE67CE9EULL, "llac_dr_primevaltomb_663_1.dds"},
	{0x0ED90E09EB5B1C60ULL, "vxlt_warehouse_n_ya_0_0.vlm"},
	{0x0ED924A44FA61717ULL, "follower_banter6161024.cl"},
	{0x0ED93898010CAF54ULL, "6119333_m.fxe"},
	{0x0ED9BE4C7B8B4AB2ULL, "hlac_ht_bartrand_n_3371_2y.dds"},
	{0x0ED9EC31CB066A12ULL, "qun131_party_comment6117263.cl"},
	{0x0EDA67ED312D94F9ULL, "6171791_f.fxe"},
	{0x0EDABFB1619BFC8AULL, "docks_male_elven_worker_sta6175846.cl"},
	{0x0EDACFC552A5930AULL, "vxw_encounter1_do_0_0.vlm"},
	{0x0EDAE6AC73934CA2ULL, "6108954_f.fxe"},
	{0x0EDAFA847C890389ULL, "gen00fl_isabela_init6076616.cl"},
	{0x0EDBD8E7BD31BD84ULL, "6022567_m.fxe"},
	{0x0EDBDA2F220D1AC9ULL, "808085176.ogg"},
	{0x0EDBE8C9B7B1EECFULL, "vxlt_0_lowtown_a3_n_c_csh_0_0.vlm"},
	{0x0EDC60FF46124E13ULL, "190449068.ogg"},
	{0x0EDCD71A8C8559ACULL, "hlac_w_woundedcoast_0_3z.dds"},
	{0x0EDD0E1A1E725F1DULL, "vxlt_0_lowtown_docks_a3_d20_0_0.vlm"},
	{0x0EDD664EB4A03EA1ULL, "follower_banter6156200.cl"},
	{0x0EDDBD9B827C0DFCULL, "art\\vfx\\materials\\vfx_bub_burst.mao"},
	{0x0EDE1AA8859D92FAULL, "vxca_caverns1_mines_dpb_0_0.vlm"},
	{0x0EDE468DB47A44AEULL, "801688610.ogg"},
	{0x0EDE487E21620382ULL, "dae321ar_player_home_nt.lst"},
	{0x0EDE99DE98699538ULL, "939745582.ogg"},
	{0x0EDECFB08CC2B523ULL, "vxlt_gallowscourtyard_n_4uw_0_0.vlm"},
	{0x0EDEFFCC044FC88EULL, "hlac_w_encounter3_-327680_0x.dds"},
	{0x0EDF0BDA2FB22D0BULL, "vxlt_hangedman_2em_0_0.vlm"},
	{0x0EDF7E05E3548D2CULL, "and111_lirene6084518.cl"},
	{0x0EDF9695C26BA22CULL, "vxht_bartrand_n_64s_0_0.vlm"},
	{0x0EDFD2F4B3B2EA0BULL, "pro000_cut2d_real_story6001238.cl"},
	{0x0EDFFAA7CC719C60ULL, "755812824.ogg"},
	{0x0EE08A8BBD3FDF48ULL, "6103060_m.fxe"},
	{0x0EE0D3711D07466BULL, "vxlt_warehouse_n_13o_0_0.vlm"},
	{0x0EE128969EE48433ULL, "940652375.ogg"},
	{0x0EE15C2FF2E312A2ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\prp_cavedraw3sm.msh"},
	{0x0EE15E2FF2E31658ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\prp_cavedraw3sm.mmh"},
	{0x0EE15E61073D03D7ULL, "vxlt_0_lowtown_h8r_0_0.vlm"},
	{0x0EE16A2FF2E32AFBULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\prp_cavedraw3sm.mao"},
	{0x0EE19AB723516670ULL, "vxca_caverns2prc_adb_0_0.vlm"},
	{0x0EE1D7C528BAB8F2ULL, "vxca_caverns2_4ke_0_0.vlm"},
	{0x0EE1F54809E7B145ULL, "vxlt_0_lowtown_a2_n_h2a_0_0.vlm"},
	{0x0EE26ED55D288D8FULL, "vxw_woundedcoast_fd_0_0.vlm"},
	{0x0EE26F3C0F8FA79DULL, "pro_mother_lookatvillage.cut"},
	{0x0EE2A1C368259161ULL, "vxw_bonepitdestroyed_a3_1wj_0_1.vlm"},
	{0x0EE2CECB78D9AAD6ULL, "gen00fl_isabela6101455.cl"},
	{0x0EE2E9FB820FDB4DULL, "art\\vfx\\combat\\vfx_glb_com_stagger_s2_c.impa.ani"},
	{0x0EE385C5D52E3354ULL, "8930040.ogg"},
	{0x0EE406995831CAA8ULL, "hlac_lt_0_lowtown_a2_n_c_65535_8z.dds"},
	{0x0EE409E5F00095F2ULL, "6101235_m.fxe"},
	{0x0EE4632A6EE4FEB4ULL, "art\\characters\\creatures\\spider\\animation\\p_tn90l_wkf.ani"},
	{0x0EE464E53AA199A8ULL, "814672642.ogg"},
	{0x0EE4708F188F3D97ULL, "hlac_lt_darktown_-1_74x.dds"},
	{0x0EE4A013BFD8D3BFULL, "vxht_chantry_147_0_0.vlm"},
	{0x0EE5226BE9C683E7ULL, "vxht_0_hightown_n_at8_0_0.vlm"},
	{0x0EE52E9ACFD745FDULL, "6040666_f.fxe"},
	{0x0EE5B822AC312859ULL, "6149070_m.fxe"},
	{0x0EE61C821BB1CB6FULL, "w_bonepit_51_0.rml"},
	{0x0EE6DBC54E1B5EC4ULL, "art\\levels\\races\\vfx_models\\prp_camplogmed.msh"},
	{0x0EE6E4CD0ECF707CULL, "6129355_f.fxe"},
	{0x0EE6EDC54E1B7D2DULL, "art\\levels\\races\\vfx_models\\prp_camplogmed.mao"},
	{0x0EE6F1C54E1B8466ULL, "art\\levels\\races\\vfx_models\\prp_camplogmed.mmh"},
	{0x0EE7455830BFF7FDULL, "900741820.ogg"},
	{0x0EE74C8E030A576BULL, "vxht_0_hightown_a2_9bi_0_0.vlm"},
	{0x0EE7D424744E376FULL, "llac_lt_0_lowtown_a3_n_c_-65535_3.dds"},
	{0x0EE7FC7C8D8D1B88ULL, "hlac_ht_chantry_n_2534_19x.dds"},
	{0x0EE820D03D5305C6ULL, "481458918.ogg"},
	{0x0EE837786A2C45ECULL, "807079634.ogg"},
	{0x0EE865B2A72264C1ULL, "6209919_f.fxe"},
	{0x0EE8C371C542521CULL, "6179829_m.fxe"},
	{0x0EE8D96BF3236BB6ULL, "vxlt_0_lowtown_docks_bdy_0_1.vlm"},
	{0x0EE9285F7E7BA2D5ULL, "vxlt_backalley_n_1li_0_0.vlm"},
	{0x0EE93A3C3A8F076DULL, "llac_lt_0_lowtown_n_-131072_4.dds"},
	{0x0EE941A2449BAB8AULL, "vxw_encounter1_ih_0_0.vlm"},
	{0x0EE977E2CCE3D1BFULL, "vxlt_gallowstemplar_f_5ql_0_0.vlm"},
	{0x0EEA8CFE7A838F56ULL, "vxlt_0_lowtown_a2_n_euz_0_0.vlm"},
	{0x0EEA92AF713BF3F3ULL, "88084391.ogg"},
	{0x0EEB06515EA11D5FULL, "gen_im_wep_war_two_act2_02.uti"},
	{0x0EEB1996EDB4A325ULL, "character_character_p187.psh.fxo.dx11.assembly"},
	{0x0EEB2875E679FA15ULL, "hightown_male_elfservnt_mob.bnk"},
	{0x0EEB602A6EEB1816ULL, "art\\characters\\creatures\\spider\\animation\\p_tn90l_wkf.gad"},
	{0x0EEB926F6B621E37ULL, "follower_banter6149291.cl"},
	{0x0EEBD6575C8B3D9AULL, "art\\characters\\weapons\\longswords\\w_lsw_ls101.mao"},
	{0x0EEC2D672970CE7AULL, "vxht_mansion2_216_0_0.vlm"},
	{0x0EED7DB721A8458BULL, "vxlt_hangedman_2bl_0_0.vlm"},
	{0x0EED942AFE3BE37CULL, "art\\characters\\creatures\\qunari\\textures\\qm_bdy_stn_0n.dds"},
	{0x0EEE6DE2B42902E7ULL, "365332191.ogg"},
	{0x0EEEB8AC8E3B4698ULL, "plt_vrc300pt_n_rival_max.plo"},
	{0x0EEEBFC9D65EA04BULL, "hlac_w_blightlands_262141_0x.dds"},
	{0x0EEF454E0EFD5EEBULL, "vxlt_warehouse_n_18n_0_0.vlm"},
	{0x0EEF4A75E67E1D6DULL, "hightown_male_elfservnt_mob.cnv"},
	{0x0EEF65BE28F7F6E2ULL, "vxdr_dungeon_1mf_0_0.vlm"},
	{0x0EEFA34EC6CAB3BFULL, "911924409.ogg"},
	{0x0EEFA88F16558856ULL, "6125817_m.fxe"},
	{0x0EF00C9415477EF0ULL, "vxca_caverns2prc_acc_0_0.vlm"},
	{0x0EF01B118FC30CB1ULL, "and321tr_distract.ncs"},
	{0x0EF0E0B24F1EA60FULL, "vxw_woundedcoast_cg_0_0.vlm"},
	{0x0EF0E8C54E23C931ULL, "art\\levels\\races\\vfx_models\\prp_camplogmed.phy"},
	{0x0EF0EB40467DDDA7ULL, "6059984_m.fxe"},
	{0x0EF2009F3EDFE228ULL, "91551225.ogg"},
	{0x0EF2064EC448623AULL, "vxlt_0_lowtown_docks_a3_n_czf_0_1.vlm"},
	{0x0EF21BB24045C333ULL, "vxht_0_hightown_n_c6x_0_0.vlm"},
	{0x0EF2476AA77663A3ULL, "art\\characters\\playercharacter\\humanfemale\\textures\\hf_rob_maga_0s.dds"},
	{0x0EF27B2A6EF16575ULL, "art\\characters\\creatures\\spider\\animation\\p_tn90l_wkf.evt"},
	{0x0EF2C3AF3CDEBC03ULL, "6054688_m.fxe"},
	{0x0EF30397DAFEDD6FULL, "42246380.ogg"},
	{0x0EF3403AFE7B134EULL, "gen_im_wep_war_shd_act1_03.uti"},
	{0x0EF399B6A6DC41DBULL, "llac_ca_caverns1_24424_2.dds"},
	{0x0EF3F89DE7023EBBULL, "mag120cr_macha.utc"},
	{0x0EF488BFB18FC569ULL, "vxlt_hangedman_1ve_0_0.vlm"},
	{0x0EF53B3FFF0D45AFULL, "hlac_ht_0_hightown_n_-1_0y.dds"},
	{0x0EF579EA28525CB4ULL, "188453346.ogg"},
	{0x0EF58DBD284CF049ULL, "art\\characters\\creatures\\gallowsslavestatue\\animation\\blendtree\\gallowsslavestatue_pose2left.anb"},
	{0x0EF5A5EB4DC1FD26ULL, "6143637_m.fxe"},
	{0x0EF5B4A967224338ULL, "hlac_lt_0_lowtown_docks_n_0_3x.dds"},
	{0x0EF60B144D849746ULL, "vxlt_0_lowtown_a2_n_c_bxp_0_0.vlm"},
	{0x0EF630F70E8506CCULL, "vxw_sundermount_318_0_1.vlm"},
	{0x0EF67423DEEAFAE1ULL, "49671202.ogg"},
	{0x0EF6892EAE1424D5ULL, "6104180_f.fxe"},
	{0x0EF6D787174AA5DFULL, "202839850.ogg"},
	{0x0EF6F98C0F170943ULL, "6046830_m.fxe"},
	{0x0EF79A3C7071BB55ULL, "vxlt_backalley_n_1wh_0_0.vlm"},
	{0x0EF7C44A0DF011EBULL, "w_encounter2_camp_19_0.anb"},
	{0x0EF7FAABB3B1AFA2ULL, "vxlt_0_lowtown_docks_a3_n_d07_0_1.vlm"},
	{0x0EF81F1AEA1A38A6ULL, "383956914.ogg"},
	{0x0EF81FEDF9B80611ULL, "262962501.ogg"},
	{0x0EF922F7CA660432ULL, "zz_and_debug6134145.cl"},
	{0x0EF980BA170333B0ULL, "6118019_m.fxe"},
	{0x0EF9A784AA1A1BE0ULL, "ss_templars_f_1.wbd"},
	{0x0EFA32137BCB08E3ULL, "art\\characters\\playercharacter\\humanmale\\mt_aet_hm01.mmh"},
	{0x0EFA3E8E669B7575ULL, "w_sundermount_152_0.msh"},
	{0x0EFA40137BCB203DULL, "art\\characters\\playercharacter\\humanmale\\mt_aet_hm01.msh"},
	{0x0EFA408E669B78EBULL, "w_sundermount_152_0.mmh"},
	{0x0EFA4C8E669B8C88ULL, "w_sundermount_152_0.mao"},
	{0x0EFACAAB06766C6BULL, "662825978.ogg"},
	{0x0EFADBEE2A921341ULL, "6119890_f.fxe"},
	{0x0EFB2B7C00CAA7AFULL, "zz_isa_debug6133945.cl"},
	{0x0EFB464CB8539426ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\kissing_male_female_action_exit.evt"},
	{0x0EFB686A53AE2302ULL, "6052089_m.fxe"},
	{0x0EFB7B6721718667ULL, "gen00fl_anders6096475.cl"},
	{0x0EFBDEBA12AB26D4ULL, "418233689.ogg"},
	{0x0EFBE0369DED0F9AULL, "vxht_0_hightown_c5g_0_0.vlm"},
	{0x0EFBF57B42CC3CE1ULL, "follower_banter6167961.cl"},
	{0x0EFC2AB576F814FCULL, "6163981_m.fxe"},
	{0x0EFC41C3730770D2ULL, "222087197.ogg"},
	{0x0EFC47FADC6C08C5ULL, "hlac_w_encounter1_5_0y.dds"},
	{0x0EFC8B59AFA334F1ULL, "vxlt_gallowstemplar_f_3sv_0_0.vlm"},
	{0x0EFCE00B2A26ACCAULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\0rw_keep.mao"},
	{0x0EFCE40B2A26B319ULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\0rw_keep.mmh"},
	{0x0EFCEE0B2A26C407ULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\0rw_keep.msh"},
	{0x0EFD1C9F639E0453ULL, "vxw_encounter2_b1_0_0.vlm"},
	{0x0EFD8F17825D0E5BULL, "hlac_ca_caverns2prc_5802_30z.dds"},
	{0x0EFE4F72F6C6E808ULL, "6072342_m.fxe"},
	{0x0EFE5F3D7A4ED152ULL, "art\\vfx\\textures\\vfx_gaping_wound.dds"},
	{0x0EFF528F4114BE8FULL, "vxw_woundedcoast_xf_0_0.vlm"},
	{0x0EFFA056566A6C7AULL, "vxlt_shop_ey_0_0.vlm"},
	{0x0F0050E8A54B421EULL, "hlac_ht_0_hightown_a3_1_1z.dds"},
	{0x0F00D2E94569CAE7ULL, "6128777_f.fxe"},
	{0x0F015D4CB8582771ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\kissing_male_female_action_exit.gad"},
	{0x0F01697609EF456CULL, "vxca_cavern1_haunted_j2b_0_2.vlm"},
	{0x0F0175CABE121E3CULL, "hlac_ht_brothel_n_3372_0z.dds"},
	{0x0F01F852CC1712B7ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\1ww_2windows.phy"},
	{0x0F02389C82862C6BULL, "6197880_m.fxe"},
	{0x0F026F80F062EF81ULL, "art\\vfx\\materials\\weapontrail\\weapontrail_streaks.mao"},
	{0x0F0304DD9326D7EEULL, "t3_com_c00.tnt"},
	{0x0F03413E5E2166D4ULL, "w_encounter2_41_0.phy"},
	{0x0F036620F682373BULL, "art\\levels\\races\\dwarf\\props\\decor\\prp_paintinga.phy"},
	{0x0F03FEED668C40A5ULL, "79713091.ogg"},
	{0x0F042F579AF1190BULL, "vxht_0_hightown_a2_bhx_0_0.vlm"},
	{0x0F044DA803BD3A7DULL, "6102941_m.fxe"},
	{0x0F04EDBD6F11D99EULL, "237036029.ogg"},
	{0x0F054F32C07A61C0ULL, "889945597.ogg"},
	{0x0F05823884CECB8AULL, "zz_spider_stomp_v2.cut"},
	{0x0F05B8C1F9823809ULL, "6016679_m.fxe"},
	{0x0F05F0E5FE0798F3ULL, "art\\characters\\creatures\\ashwraith\\animation\\ras_p_wkf.ani"},
	{0x0F060ABF52BE1F78ULL, "165074452.ogg"},
	{0x0F060DA7C8167EB8ULL, "6076100_m.fxe"},
	{0x0F068AFD640D8670ULL, "zz_jp_spike_head.cut"},
	{0x0F069D5524F2A2D2ULL, "vxw_encounter2_b4_0_0.vlm"},
	{0x0F07303233683C85ULL, "vxht_mansion2_1ge_0_1.vlm"},
	{0x0F0786A0DF4A910FULL, "ser121_jethann6104618.cl"},
	{0x0F078B707081D1CAULL, "739763591.ogg"},
	{0x0F0820689C6D2C27ULL, "94398666.ogg"},
	{0x0F08400B13A7A0D8ULL, "vxht_0_hightown_a3_aif_0_0.vlm"},
	{0x0F085513C65DF57DULL, "art\\levels\\races\\natural\\blightlands\\props\\prp_01ruinsstair.mao"},
	{0x0F085913C65DFBB6ULL, "art\\levels\\races\\natural\\blightlands\\props\\prp_01ruinsstair.mmh"},
	{0x0F085E4CB85E471FULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\kissing_male_female_action_exit.ani"},
	{0x0F086313C65E0CB4ULL, "art\\levels\\races\\natural\\blightlands\\props\\prp_01ruinsstair.msh"},
	{0x0F089F61F6E973FCULL, "vxlt_warehouse_vi_0_0.vlm"},
	{0x0F08B11C375929B3ULL, "art\\characters\\heraldry\\hld_alienage.dds"},
	{0x0F08CAF0338A9BF0ULL, "character_character_p10.psh.fxo.dx11"},
	{0x0F090FFC8F6DA8E2ULL, "m_rai301ar_caverns.wbd"},
	{0x0F09124C8A122245ULL, "vxca_caverns1_mines_dpi_0_0.vlm"},
	{0x0F093DA0122A25A3ULL, "hlac_lt_0_lowtown_a2_n_65534_6z.dds"},
	{0x0F094093A5A72321ULL, "character_character_p190.psh.fxo.dx11"},
	{0x0F096599F44D1912ULL, "879917676.ogg"},
	{0x0F0988FF1B719806ULL, "zz_dae_debug6029077.cl"},
	{0x0F09C7672EA3F79EULL, "ser270_emeric6111484.cl"},
	{0x0F0A383E5E27767CULL, "w_encounter2_41_0.rml"},
	{0x0F0A69A20A8D39B7ULL, "6119411_m.fxe"},
	{0x0F0A9AA597CF6BB9ULL, "art\\characters\\creatures\\shriek\\animation\\combat\\walkright.ani"},
	{0x0F0B33137BD97D08ULL, "art\\characters\\playercharacter\\humanmale\\mt_aet_hm01.phy"},
	{0x0F0B733306B1FC95ULL, "vxlt_0_lowtown_a3_n_c_htx_0_0.vlm"},
	{0x0F0B80D91B33F20BULL, "38807922.ogg"},
	{0x0F0B8E7C55941CD3ULL, "vxw_encounter2_m0_0_0.vlm"},
	{0x0F0CCD0A5CC28529ULL, "vxw_encounter3_6d_0_0.vlm"},
	{0x0F0D516123B0A5D6ULL, "vxlt_0_lowtown_n_f9h_0_1.vlm"},
	{0x0F0D62A145C42167ULL, "6169438_m.fxe"},
	{0x0F0D7F5043F514A0ULL, "948066913.ogg"},
	{0x0F0D8F2B36787F6FULL, "6128012_f.fxe"},
	{0x0F0E1E3E5E2B32D1ULL, "w_encounter2_41_0.msh"},
	{0x0F0E24668577DC70ULL, "vxlt_0_lowtown_a3_n_c_hp5_0_0.vlm"},
	{0x0F0E2C3E5E2B4AECULL, "w_encounter2_41_0.mao"},
	{0x0F0E383E5E2B5E8FULL, "w_encounter2_41_0.mmh"},
	{0x0F0E98C99C2F70A7ULL, "dae000_pc_home_ambients6208687.cl"},
	{0x0F0EFF6C2431F433ULL, "vxht_0_hightown_n_c8z_0_0.vlm"},
	{0x0F0F43BD1757467FULL, "one200_gold_followers6117306.cl"},
	{0x0F0F52FA815F5E6DULL, "6113987_m.fxe"},
	{0x0F0F7CCEB3A919ABULL, "character_character_p95.psh.fxo.dx11.assembly"},
	{0x0F10087E90E33A4EULL, "6110619_m.fxe"},
	{0x0F10E8E6B181CC3BULL, "55454190.ogg"},
	{0x0F1147B26BD0A96CULL, "vxdr_0_deeproads_a6y_0_0.vlm"},
	{0x0F116C79957BF669ULL, "vxlt_hangedman_1xk_0_0.vlm"},
	{0x0F1175A597D54BFBULL, "art\\characters\\creatures\\shriek\\animation\\combat\\walkright.gad"},
	{0x0F117C14C45FBE22ULL, "578816827.ogg"},
	{0x0F1181265EAD4417ULL, "dae000_tomwise6167343.cl"},
	{0x0F11980315E420DEULL, "dae340st_cowan.stg"},
	{0x0F12E8E5FE12152AULL, "art\\characters\\creatures\\ashwraith\\animation\\ras_p_wkf.evt"},
	{0x0F1325E06FE591E5ULL, "888840423.ogg"},
	{0x0F13D041AB8B9D63ULL, "zz_vault_debug6177793.cl"},
	{0x0F13FD2A67039936ULL, "vxca_cavern1_haunted_i1i_0_1.vlm"},
	{0x0F1430431B15BC07ULL, "1058424968.ogg"},
	{0x0F145C3B32342759ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\reactions\\rct_quick_rct_die_right_down_loop.gad"},
	{0x0F14BA5144340910ULL, "6175906_m.fxe"},
	{0x0F150F3216E8BB52ULL, "vxw_encounter2_m5_0_0.vlm"},
	{0x0F15D66EF23D4D64ULL, "6207397_m.fxe"},
	{0x0F163C0450E41ABCULL, "def_dirt_n.dds"},
	{0x0F163E9C1CDA1B18ULL, "vxw_bonepitdestroyed_a3_dx_0_0.vlm"},
	{0x0F16B1E8059DB958ULL, "vxht_0_hightown_a3_avg_0_0.vlm"},
	{0x0F16E8B20252F1A9ULL, "6103021_m.fxe"},
	{0x0F1795E5F36CDEE7ULL, "docks_male_laborer_mob6175405.cl"},
	{0x0F17D124DA825866ULL, "hlac_lt_0_lowtown_a2_n_131070_6z.dds"},
	{0x0F18B8D531213B3FULL, "vxca_caverns2_8l0_0_1.vlm"},
	{0x0F18F9F4DA041221ULL, "mag360_thrask6053254.cl"},
	{0x0F197F617F54AF63ULL, "776926097.ogg"},
	{0x0F19C7E5FE17FB9DULL, "art\\characters\\creatures\\ashwraith\\animation\\ras_p_wkf.gad"},
	{0x0F1A0059478A3553ULL, "vxw_encounter2_h3_0_0.vlm"},
	{0x0F1A378C9811FC38ULL, "vxlt_undercityserialkiller_1j7_0_0.vlm"},
	{0x0F1AA8FCB043EB9BULL, "vxlt_0_lowtown_a3_n_c_ere_0_0.vlm"},
	{0x0F1AC095E6F7BA4BULL, "hlac_w_sundermount_-196611_0y.dds"},
	{0x0F1AD0D47ED6330CULL, "hlac_lt_0_lowtown_a2_n_c_1_10x.dds"},
	{0x0F1B14865D35C030ULL, "dae000_mother6156396.cl"},
	{0x0F1B3EE74EB89DA9ULL, "vxw_encounter3_3k_0_0.vlm"},
	{0x0F1B44CC1C92B311ULL, "vxw_encounter2_camp_bs_0_0.vlm"},
	{0x0F1B6D3B323A62A7ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\reactions\\rct_quick_rct_die_right_down_loop.ani"},
	{0x0F1B94ED0629E814ULL, "mag320cr_circle_mage.utc"},
	{0x0F1C623C21696B46ULL, "root_rog_duelist.dds"},
	{0x0F1D1050BFFD4B12ULL, "lgt261cr_archer.utc"},
	{0x0F1DB93E80101EBFULL, "987619100.ogg"},
	{0x0F1E07FF6AD4BDEFULL, "follower_banter6170624.cl"},
	{0x0F1E17EA0FF58674ULL, "362391790.ogg"},
	{0x0F1E3C5EEEE74E2BULL, "lgt000im_ultra_shawl.uti"},
	{0x0F1E4E076C9DAC69ULL, "dae000ip_crowd_ff3_talk9.utp"},
	{0x0F1ECBD319F67CEEULL, "707707055.ogg"},
	{0x0F1F7220E935F0C3ULL, "hlac_w_blightlands_65536_1z.dds"},
	{0x0F1FDE5687720EE9ULL, "vxlt_hangedman_1gj_0_0.vlm"},
	{0x0F1FE76A464CFA41ULL, "795165451.ogg"},
	{0x0F2041781733336BULL, "75531064.ogg"},
	{0x0F20952EDC4F982EULL, "617091763.ogg"},
	{0x0F210A8D7C0A4B98ULL, "hlac_lt_0_lowtown_a2_n_c_-1_6z.dds"},
	{0x0F2130CB5474D1CAULL, "hlac_lt_0_lowtown_docks_a3_-2_2z.dds"},
	{0x0F21527179838293ULL, "vxht_keep_n_5zv_0_0.vlm"},
	{0x0F2185CC2829C2E6ULL, "594547529.ogg"},
	{0x0F219E6B71A5623FULL, "gen00flst_varric_posthaunt.stg"},
	{0x0F21A7DB10667E59ULL, "671612925.ogg"},
	{0x0F21F1189FA4119CULL, "llac_lt_0_lowtown_docks_n_131071_4.dds"},
	{0x0F22390558A39F44ULL, "art\\vfx\\combat\\vfx_m_cre_glyph_rep_tar_c.mmh"},
	{0x0F22CA0584F979E0ULL, "hlac_w_bonepit_196612_0y.dds"},
	{0x0F22EFD3465404D5ULL, "vxlt_backalley_n_1xu_0_0.vlm"},
	{0x0F234C0C5A4AB8FAULL, "dog221ip_table_leg.utp"},
	{0x0F23680CD7EB165DULL, "gen00fl_anders6112966.cl"},
	{0x0F23810F08DED3D2ULL, "vxw_encounter2_h6_0_0.vlm"},
	{0x0F23AB36F27D526EULL, "w_blightlands_67_0.msh"},
	{0x0F23B936F27D69BFULL, "w_blightlands_67_0.mao"},
	{0x0F23C14A0E154737ULL, "w_encounter2_camp_19_0.rml"},
	{0x0F23C536F27D7E5CULL, "w_blightlands_67_0.mmh"},
	{0x0F23EF64C5A758BCULL, "mag130_danzig6018820.cl"},
	{0x0F243DCBAEC23396ULL, "art\\levels\\ztools\\default\\generic\\0co_basic.phy"},
	{0x0F24706D541D6B27ULL, "579770097.ogg"},
	{0x0F247916BB1A1174ULL, "6128926_m.fxe"},
	{0x0F24B0790ED03398ULL, "vxw_bonepitdestroyed_a3_oy_0_0.vlm"},
	{0x0F24D9F9CB528EB3ULL, "vxw_woundedcoast_o7_0_0.vlm"},
	{0x0F24E2BDF05FC0ECULL, "6075084_m.fxe"},
	{0x0F2503E480D4B975ULL, "6110267_m.fxe"},
	{0x0F2537ECF06A6789ULL, "6069848_m.fxe"},
	{0x0F2549A43FA6A6B5ULL, "mag160_varian_ilithis6016511.cl"},
	{0x0F259F8209900DA0ULL, "6018746_m.fxe"},
	{0x0F25D6B24C0F13DAULL, "hlac_w_bonepitmining_a2_458754_0y.dds"},
	{0x0F25D95189AF7A21ULL, "vxht_brothel_n_628_0_0.vlm"},
	{0x0F2621984253DB98ULL, "6033099_m.fxe"},
	{0x0F264E47AF95E5F0ULL, "vxlt_undercitydungeon_1c9_0_0.vlm"},
	{0x0F2662FD262C4D42ULL, "vxlt_hangedman_1jv_0_0.vlm"},
	{0x0F26A87089B94895ULL, "6064726_m.fxe"},
	{0x0F272460A817EE1AULL, "follower_banter6163421.cl"},
	{0x0F272AF6493404E5ULL, "follower_banter6136358.cl"},
	{0x0F27422C36C19050ULL, "9244160.ogg"},
	{0x0F27AFDBC741E640ULL, "vxlt_undercitydungeon_1ge_0_0.vlm"},
	{0x0F28723639804DD3ULL, "vxw_encounter2_s2_0_0.vlm"},
	{0x0F2875C748CC0B3DULL, "follower_banter6151353.cl"},
	{0x0F28A9698A0814B8ULL, "vxlt_undercityserialkiller_1o4_0_0.vlm"},
	{0x0F291AD9A23A041BULL, "vxlt_0_lowtown_a3_n_c_eud_0_0.vlm"},
	{0x0F2A97FA5ADEFBC5ULL, "vxlt_shop_er_0_0.vlm"},
	{0x0F2AC04A0E1B642FULL, "w_encounter2_camp_19_0.phy"},
	{0x0F2AC6ED9AE95396ULL, "follower_banter6160608.cl"},
	{0x0F2AE64AFDA851C6ULL, "singleroofa_n.dds"},
	{0x0F2AFF24B92D1E2BULL, "vxca_caverns1_mines_im4_0_1.vlm"},
	{0x0F2B1914961D90A2ULL, "zz_lgt_2_debug6206577.cl"},
	{0x0F2B2F2A7EFEAA9EULL, "685525186.ogg"},
	{0x0F2B6BA9E5EABE75ULL, "vxlt_0_lowtown_gkr_0_0.vlm"},
	{0x0F2B92DB81F9BC4EULL, "6084712_m.fxe"},
	{0x0F2C1668AD642812ULL, "vxlt_0_lowtown_docks_a3_aoi_0_0.vlm"},
	{0x0F2C795F9E01ECA7ULL, "6082235_f.fxe"},
	{0x0F2C9E3A44E2424BULL, "6110427_m.fxe"},
	{0x0F2CD8CCC9A554F3ULL, "lgt120cr_orz_bln_servant.utc"},
	{0x0F2CF0B46CC3B64CULL, "6059015_m.fxe"},
	{0x0F2D497FA76A74F4ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\sitting_table_loop.ani"},
	{0x0F2D87086E73542BULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\reactions\\rct_kdwn_v1_loop.ani"},
	{0x0F2DC34E1060AE61ULL, "dae100cr_mother.utc"},
	{0x0F2DD81692946F6CULL, "172714367.ogg"},
	{0x0F2E246E3C8B3CF0ULL, "lt_0_lowtown_docks_a3_150_0.mao"},
	{0x0F2E24F991721C5DULL, "gen00fl_varric6119002.cl"},
	{0x0F2E26CBAECA6137ULL, "art\\levels\\ztools\\default\\generic\\0co_basic.msh"},
	{0x0F2E286E3C8B4303ULL, "lt_0_lowtown_docks_a3_150_0.mmh"},
	{0x0F2E366E3C8B5B5DULL, "lt_0_lowtown_docks_a3_150_0.msh"},
	{0x0F2E38CBAECA7F9AULL, "art\\levels\\ztools\\default\\generic\\0co_basic.mao"},
	{0x0F2E3CCBAECA86A9ULL, "art\\levels\\ztools\\default\\generic\\0co_basic.mmh"},
	{0x0F2E503379682769ULL, "vxlt_hangedman_1bi_0_0.vlm"},
	{0x0F2E81BC9310E13AULL, "313276275.ogg"},
	{0x0F2EED8B257A7064ULL, "6031699_m.fxe"},
	{0x0F2F349F598BB152ULL, "gen00fl_isabela6101547.cl"},
	{0x0F2F4D20F6A747DAULL, "art\\levels\\races\\dwarf\\props\\decor\\prp_paintinga.msh"},
	{0x0F2F4F2E3E8363C5ULL, "6107344_m.fxe"},
	{0x0F2F5B20F6A75F13ULL, "art\\levels\\races\\dwarf\\props\\decor\\prp_paintinga.mao"},
	{0x0F2F5F20F6A76620ULL, "art\\levels\\races\\dwarf\\props\\decor\\prp_paintinga.mmh"},
	{0x0F2F6DD1CD4DA4FFULL, "139513733.ogg"},
	{0x0F2FF38302D87393ULL, "556935112.ogg"},
	{0x0F30430889D1635CULL, "6036742_m.fxe"},
	{0x0F3074A9016CB793ULL, "prp_testladder.mao"},
	{0x0F30ECEC442831B6ULL, "hlac_lt_0_lowtown_a2_65536_11z.dds"},
	{0x0F3154E51C419ED1ULL, "6149154_f.fxe"},
	{0x0F317BF00CFA15C4ULL, "6189912_m.fxe"},
	{0x0F31BA3F00B6686CULL, "llac_lt_0_lowtown_a3_n_c_-65536_3.dds"},
	{0x0F31E04186EA2CA5ULL, "6043758_m.fxe"},
	{0x0F32768392F8BD91ULL, "dre180ip_deep_roads_chest_2.utp"},
	{0x0F32B3EF78438DA4ULL, "801723807.ogg"},
	{0x0F32D3B2B1941D02ULL, "art\\levels\\races\\natural\\kirkcoastal\\props\\prp_kcrockxxl.msh"},
	{0x0F32D5B2B19420B8ULL, "art\\levels\\races\\natural\\kirkcoastal\\props\\prp_kcrockxxl.mmh"},
	{0x0F32E1B2B194355BULL, "art\\levels\\races\\natural\\kirkcoastal\\props\\prp_kcrockxxl.mao"},
	{0x0F331E614330B474ULL, "vxca_cavern1_haunted_gkb_0_0.vlm"},
	{0x0F33467FA76EDB56ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\sitting_table_loop.gad"},
	{0x0F334BD6BD48A733ULL, "vxw_woundedcoast_d6_0_0.vlm"},
	{0x0F3367485B083F24ULL, "one280_iwan6096874.cl"},
	{0x0F338D30ECA1595FULL, "lgt000ip_toe_crate.utp"},
	{0x0F341FBB49A7FECAULL, "vxlt_0_lowtown_n_e2w_0_0.vlm"},
	{0x0F34372ACBFBCE42ULL, "6142634_m.fxe"},
	{0x0F343F568360950AULL, "vxw_bonepitdestroyed_a3_26x_0_0.vlm"},
	{0x0F346E384CA28DF5ULL, "art\\characters\\weapons\\unique\\arishok\\textures\\w_gax_002a_n.dds"},
	{0x0F34B10EF5C04AE4ULL, "193064054.ogg"},
	{0x0F34CC200BFF0266ULL, "gen00fl_aveline6114845.cl"},
	{0x0F357C32F7867FEEULL, "hlac_lt_0_lowtown_docks_n_-131072_0x.dds"},
	{0x0F35E7891A47282CULL, "6136693_m.fxe"},
	{0x0F3621B8B937FEC0ULL, "vxlt_undercitydungeon_1jb_0_0.vlm"},
	{0x0F362E001F728813ULL, "vxdr_primevalentrance_1r0_0_0.vlm"},
	{0x0F365D3BA924516CULL, "6106720_m.fxe"},
	{0x0F366F27C1ADB199ULL, "art\\characters\\creatures\\desiredemon\\animation\\kd_lp.gad"},
	{0x0F36825D89FFCAB3ULL, "follower_banter6156260.cl"},
	{0x0F371B467BFE2D38ULL, "vxlt_undercityserialkiller_1d5_0_0.vlm"},
	{0x0F374BAAA1F8E053ULL, "ach_seb_retribution.dds"},
	{0x0F378CB694301C9BULL, "vxlt_0_lowtown_a3_n_c_exg_0_0.vlm"},
	{0x0F37B77F0D84EECBULL, "6149515_m.fxe"},
	{0x0F38384D866A9D28ULL, "vxht_keep_n_3c8_0_0.vlm"},
	{0x0F383BBD0D04DFE7ULL, "lt_0_lowtown_a2_n_mmi.gff"},
	{0x0F3852C9B466C187ULL, "dre000_varric6033307.cl"},
	{0x0F38DC9F7EE0F3DFULL, "art\\characters\\playercharacter\\humanmale\\hm_brd_hb05.phy"},
	{0x0F38F51F9D040DE1ULL, "gen00fl_dog6176631.cl"},
	{0x0F3953A95EAA5D9DULL, "6103731_m.fxe"},
	{0x0F39B162ECD187D2ULL, "vxca_caverns2_62p_0_0.vlm"},
	{0x0F3A747D8443EFA5ULL, "416150262.ogg"},
	{0x0F3B41495AEF802CULL, "6089601_m.fxe"},
	{0x0F3B7081230527C2ULL, "vxw_encounter2_camp_7c_0_0.vlm"},
	{0x0F3B8B79DC67D27AULL, "w_bonepitdestroyed_a3_194_0.mmh"},
	{0x0F3B8F79DC67D8B1ULL, "w_bonepitdestroyed_a3_194_0.mao"},
	{0x0F3B9D79DC67F0C0ULL, "w_bonepitdestroyed_a3_194_0.msh"},
	{0x0F3BAC95C1EAD9FFULL, "305417067.ogg"},
	{0x0F3BB96D9784D205ULL, "llac_ht_0_hightown_a3_65536_0.dds"},
	{0x0F3BFF567044CBBFULL, "vxdr_0_deeproads_a6v_0_0.vlm"},
	{0x0F3C1B6BFE8E8035ULL, "zz_dre_debug6168233.cl"},
	{0x0F3C5CD67EA88466ULL, "6049452_m.fxe"},
	{0x0F3C6D6D37AD0355ULL, "vxca_caverns1_dq7_0_0.vlm"},
	{0x0F3CF295E74C2ED7ULL, "vxht_chantry_n_2ip_0_0.vlm"},
	{0x0F3D10FA18928B60ULL, "327284048.ogg"},
	{0x0F3D197007235403ULL, "385805222.ogg"},
	{0x0F3D8027C1B3EDE7ULL, "art\\characters\\creatures\\desiredemon\\animation\\kd_lp.ani"},
	{0x0F3EC335661D0C55ULL, "321886379.ogg"},
	{0x0F3F296E3C99B828ULL, "lt_0_lowtown_docks_a3_150_0.phy"},
	{0x0F3F690DD9FD7E70ULL, "625297623.ogg"},
	{0x0F3FD483E97BD406ULL, "6066385_m.fxe"},
	{0x0F3FE8FB8A621142ULL, "641855487.ogg"},
	{0x0F40294F2F7267E3ULL, "vxht_0_hightown_c8m_0_0.vlm"},
	{0x0F40C410ED2EB1E5ULL, "351914352.ogg"},
	{0x0F40C70109295611ULL, "689196802.ogg"},
	{0x0F410649957D41D7ULL, "6135655_m.fxe"},
	{0x0F411A2F93036634ULL, "vxht_brothel_n_6bq_0_0.vlm"},
	{0x0F414E690F0A4071ULL, "117155377.ogg"},
	{0x0F41BDB3AF3EBFB3ULL, "vxw_woundedcoast_a9_0_0.vlm"},
	{0x0F42612CDF72634BULL, "vxlt_0_lowtown_docks_n_aoh_0_0.vlm"},
	{0x0F426E086E8585B5ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\reactions\\rct_kdwn_v1_loop.gad"},
	{0x0F427E4EF78C598EULL, "vxlt_0_lowtown_n_gby_0_0.vlm"},
	{0x0F429397D740967FULL, "gen00fl_aveline6043100.cl"},
	{0x0F4303A23EA1871DULL, "6125317_m.fxe"},
	{0x0F439092290B2B60ULL, "fx_arrow.dds"},
	{0x0F43EF8912575495ULL, "6165267_m.fxe"},
	{0x0F448FEE8820CFD4ULL, "lgt000im_azurejamos.uti"},
	{0x0F44C7D14E3C62E0ULL, "441208645.ogg"},
	{0x0F44CCDCE1266A36ULL, "6185235_m.fxe"},
	{0x0F44F7F7E6D644A1ULL, "vxht_0_hightown_a2_avo_0_0.vlm"},
	{0x0F451B666245A157ULL, "zz_fol_debug6098073.cl"},
	{0x0F45897A6131DA0BULL, "6111536_m.fxe"},
	{0x0F458CF75524DFDAULL, "one120_athenril6088904.cl"},
	{0x0F458D236DF445B8ULL, "vxlt_undercityserialkiller_1y2_0_0.vlm"},
	{0x0F465E65B980B253ULL, "character_character_p28.psh.fxo.dx11"},
	{0x0F46B51C39763F64ULL, "6127090_f.fxe"},
	{0x0F46E650E3367245ULL, "ab_cyc.mao"},
	{0x0F480D2B4A1F99A9ULL, "dae011_conspiracy_mob6141231.cl"},
	{0x0F4875C079802525ULL, "360945303.ogg"},
	{0x0F48B71AFCE5B095ULL, "163526990.ogg"},
	{0x0F48CD52CC52CC0FULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\1ww_2windows.mao"},
	{0x0F48D6ED6DDD760CULL, "85178065.ogg"},
	{0x0F48D952CC52E0ACULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\1ww_2windows.mmh"},
	{0x0F48DF52CC52EA9EULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\1ww_2windows.msh"},
	{0x0F49771E981FC28DULL, "839778586.ogg"},
	{0x0F49DD9F7EEF68E4ULL, "art\\characters\\playercharacter\\humanmale\\hm_brd_hb05.mmh"},
	{0x0F49F39F7EEF8D86ULL, "art\\characters\\playercharacter\\humanmale\\hm_brd_hb05.msh"},
	{0x0F4A6621C5580B47ULL, "dae000_flemeth6013380.cl"},
	{0x0F4A9C1A65F77FF9ULL, "face_face_p2.psh.fxo.dx11.assembly"},
	{0x0F4B3C0558C6A93BULL, "art\\vfx\\combat\\vfx_m_cre_glyph_rep_tar_c.anb"},
	{0x0F4B57CC37676092ULL, "art\\characters\\playercharacter\\humanmale\\textures\\hm_hlf_medg_0d.dds"},
	{0x0F4B6472D9424757ULL, "vxht_chantry_n_2ds_0_0.vlm"},
	{0x0F4BECD7C4D20344ULL, "llac_dr_0_deeproads_17038_1.dds"},
	{0x0F4C99226ABFA545ULL, "vxlt_gallowscourtyard_77w_0_0.vlm"},
	{0x0F4CE99A1035508AULL, "6096863_m.fxe"},
	{0x0F4CF670286F770BULL, "dae000_flemeth6013418.cl"},
	{0x0F4D51455390C6CBULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_hightlighta_sa.dds"},
	{0x0F4DD22CF5A2C0FAULL, "hlac_lt_gallowstemplar_n_2383_7z.dds"},
	{0x0F4E872EEA0302EDULL, "gen00fl_bethany6097595.cl"},
	{0x0F4E9266C5A3F4B4ULL, "269454286.ogg"},
	{0x0F4F8C0C84F97EB4ULL, "vxht_brothel_n_6ov_0_0.vlm"},
	{0x0F4FC4E574FAED59ULL, "703482465.ogg"},
	{0x0F4FEC677C3FAA17ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\cower_enter.gad"},
	{0x0F501B4AA595D38EULL, "l_dae110ar_lowtown_nt.wbd"},
	{0x0F501DFC8FA9C298ULL, "m_rai301ar_caverns.bnk"},
	{0x0F5067265A74442EULL, "hlac_w_blightlands_-131070_0x.dds"},
	{0x0F50BC043DE70647ULL, "lit_aoe_03_l3.mao"},
	{0x0F51E8B82C83D362ULL, "gen_im_cns_tom_rog_shadow.uti"},
	{0x0F52DF60669AE5F3ULL, "dm_dwarven_scout.mor"},
	{0x0F52F3903D7B1400ULL, "vxlt_warehouse_n_1ef_0_0.vlm"},
	{0x0F53065B23D2942EULL, "6076854_m.fxe"},
	{0x0F535020A6C0D94BULL, "6015845_m.fxe"},
	{0x0F53FE71D8B737AAULL, "6072289_m.fxe"},
	{0x0F5431F6F9743852ULL, "269498730.ogg"},
	{0x0F548BCA03B402EAULL, "follower_banter6156054.cl"},
	{0x0F54A2ABE68A6890ULL, "character_character_p67.psh.fxo.dx11.assembly"},
	{0x0F54C6C5BCCDBDD4ULL, "vxlt_0_lowtown_a3_n_c_g60_0_0.vlm"},
	{0x0F5570C575AAB334ULL, "art\\levels\\races\\proxy\\props\\corpses\\prp_zzbodyhang.phy"},
	{0x0F55DE5DD2215A87ULL, "vxw_woundedcoast_nd_0_0.vlm"},
	{0x0F55FDDDA7B4FF71ULL, "vxht_0_hightown_a2_a81_0_0.vlm"},
	{0x0F56197E74027544ULL, "art\\vfx\\combat\\vfx_m_sta_rng_proj_ice_p.mmh"},
	{0x0F562ACAC4F354C9ULL, "narrative_varric6172515.cl"},
	{0x0F56456E988487F2ULL, "vxlt_0_lowtown_docks_a3_ctn_0_0.vlm"},
	{0x0F5679B7B20F32E5ULL, "544989242.ogg"},
	{0x0F56C8D71A1B7D97ULL, "plt_cod_ltr_otl_emile.plo"},
	{0x0F56F1E822443984ULL, "ch1armtnt_s2y.psh.fxo"},
	{0x0F5701677C45ECA4ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\cower_enter.evt"},
	{0x0F57093893F3E004ULL, "isa211_isabela6202285.cl"},
	{0x0F578D453D422D51ULL, "457946509.ogg"},
	{0x0F58E1C410DBC71AULL, "mrl000_marethari6068818.cl"},
	{0x0F59202CB845C14FULL, "ser221_gascard6133878.cl"},
	{0x0F592D8F5AADB45AULL, "t1_mub_re3.tnt"},
	{0x0F594F750D420CA8ULL, "893009599.ogg"},
	{0x0F59B19BAFBEB287ULL, "6166880_m.fxe"},
	{0x0F5A4CAB963D6A9CULL, "hf_mage_1.mor"},
	{0x0F5A790719C928C0ULL, "ser120_ghyslain6104994.cl"},
	{0x0F5A84E0A44A0384ULL, "vxlt_0_lowtown_docks_a3_n_czh_0_1.vlm"},
	{0x0F5B0AFF5CB5BDC5ULL, "vxlt_gallowscourtyard_78v_0_0.vlm"},
	{0x0F5B219E3EA02AF2ULL, "dae000ip_crowd_mf2_merchant.utp"},
	{0x0F5B3F9C18DB5A10ULL, "hlac_lt_0_lowtown_docks_a3_262143_0y.dds"},
	{0x0F5B9557CEE48002ULL, "93682556.ogg"},
	{0x0F5B9C837BB56512ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\basestaff\\combat\\exit.gad"},
	{0x0F5BB9461C15E027ULL, "1052649284.ogg"},
	{0x0F5C5540FC08CDD3ULL, "32214188.ogg"},
	{0x0F5CD62DF02F596AULL, "619188711.ogg"},
	{0x0F5D0D09135E98E3ULL, "vxht_0_hightown_c2o_0_0.vlm"},
	{0x0F5D75C155304B40ULL, "llac_ca_caverns2_5802_69.dds"},
	{0x0F5DF01B13DD3F07ULL, "vxht_0_hightown_a2_c74_0_0.vlm"},
	{0x0F5DFDE976EF9734ULL, "vxht_brothel_n_6hw_0_0.vlm"},
	{0x0F5EE50B2A797F06ULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\0rw_keep.phy"},
	{0x0F5F2B8603D2F335ULL, "658655956.ogg"},
	{0x0F5F44E6C35E944BULL, "vxlt_0_lowtown_docks_n_aqj_0_0.vlm"},
	{0x0F5F4DC575B2CC31ULL, "art\\levels\\races\\proxy\\props\\corpses\\prp_zzbodyhang.msh"},
	{0x0F5F5BC575B2E44CULL, "art\\levels\\races\\proxy\\props\\corpses\\prp_zzbodyhang.mao"},
	{0x0F5F67C575B2F8EFULL, "art\\levels\\races\\proxy\\props\\corpses\\prp_zzbodyhang.mmh"},
	{0x0F5FA6CD2CE86814ULL, "665474121.ogg"},
	{0x0F5FD5895AF0CFE9ULL, "6111264_m.fxe"},
	{0x0F602EF9B9094DBDULL, "zz_lgt_1_debug6136993.cl"},
	{0x0F6054F36FF9F6A8ULL, "docks_male_overseer_mob6176575.cl"},
	{0x0F6087AEFEF25906ULL, "6098836_m.fxe"},
	{0x0F610AE5780B76DFULL, "zz_followers_banter6158556.cl"},
	{0x0F611E51540741C2ULL, "vxca_caverns1_j4m_0_0.vlm"},
	{0x0F618396F554D193ULL, "vxdr_primevalentrance_1a5_0_0.vlm"},
	{0x0F618864FE78724BULL, "shelf_sa.dds"},
	{0x0F61DBB1CAC275A1ULL, "vxht_0_hightown_a2_alm_0_0.vlm"},
	{0x0F629F837BBB8949ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\basestaff\\combat\\exit.evt"},
	{0x0F62AB859BE7D738ULL, "lla_ht_0_hightown_a3_65536_0.dds"},
	{0x0F63048AB5F9A863ULL, "llac_lt_0_lowtown_docks_n_0_2.dds"},
	{0x0F630A46A576B9FCULL, "blood_conj_06a_l2.mao"},
	{0x0F635119057C4C62ULL, "6101132_m.fxe"},
	{0x0F6389E99DB18BF6ULL, "6068376_m.fxe"},
	{0x0F638E98257998B0ULL, "6045858_m.fxe"},
	{0x0F63B55697B84112ULL, "mcr371_hostile_merrill6082145.cl"},
	{0x0F63C81AE7F464E4ULL, "vxht_mansion2_3g7_0_0.vlm"},
	{0x0F6419677C509F9DULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\cower_enter.ani"},
	{0x0F642DF0DA970F8BULL, "hlac_lt_0_lowtown_a2_n_131074_0y.dds"},
	{0x0F648355C64AF512ULL, "hlac_w_encounter3_0_2y.dds"},
	{0x0F64EDCF882C675EULL, "146336760.ogg"},
	{0x0F65C170A2D36A74ULL, "vxdr_primevaltomb_6up_0_1.vlm"},
	{0x0F65C29C8DFF3390ULL, "vxca_caverns1_mines_ec8_0_0.vlm"},
	{0x0F65FE1073AD526AULL, "72752030.ogg"},
	{0x0F66310669962F53ULL, "gen00fl_sebastian6180298.cl"},
	{0x0F66EBAA2D47D2B9ULL, "6146857_m.fxe"},
	{0x0F66FA9974C85DCBULL, "174184466.ogg"},
	{0x0F66FAB79FCFF5BBULL, "425195658.ogg"},
	{0x0F6736CE39E2EE5CULL, "bkl_star.mao"},
	{0x0F675E6B092446D7ULL, "art\\levels\\races\\proxy\\props\\4tools\\inv_ball4_tool.phy"},
	{0x0F676CA1FA902D36ULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\a_atk_v3.ani"},
	{0x0F687EBDC64A6957ULL, "60034220.ogg"},
	{0x0F688A9276676A03ULL, "gen00fl_fenris6115354.cl"},
	{0x0F68D06538A66893ULL, "560566663.ogg"},
	{0x0F69F5429AE3BCB4ULL, "1045647137.ogg"},
	{0x0F6A1C24E20DB875ULL, "vxht_chantry_n_36v_0_0.vlm"},
	{0x0F6A4FEB166A92CCULL, "331867574.ogg"},
	{0x0F6A94920C139029ULL, "vxht_0_hightown_n_b8a_0_0.vlm"},
	{0x0F6AD1704BB02FC6ULL, "694873346.ogg"},
	{0x0F6AD7D71A2C400CULL, "plt_cod_ltr_otl_emile.nss"},
	{0x0F6B1060832E02EBULL, "lgt260_eau_tmp_roderick6143718.cl"},
	{0x0F6BE0929B45978CULL, "art\\vfx\\combat\\material__28.mao"},
	{0x0F6C4802220C5770ULL, "6020420_m.fxe"},
	{0x0F6C6FC668E5AFB4ULL, "vxht_brothel_n_6ut_0_0.vlm"},
	{0x0F6CCCCA911F3D99ULL, "vxht_0_hightown_n_cas_0_0.vlm"},
	{0x0F6CF4C3CDFFD5C1ULL, "224700743.ogg"},
	{0x0F6D6EF36ACEF39DULL, "follower_banter6172981.cl"},
	{0x0F6DF7E64B7CCEF1ULL, "vxlt_gallowstemplar_n_3or_0_0.vlm"},
	{0x0F6E24E3006C7C76ULL, "vxht_keep_3zr_0_0.vlm"},
	{0x0F6E639824B8F2B0ULL, "vxlt_0_lowtown_a3_n_c_cmq_0_0.vlm"},
	{0x0F6E7F11F2A9666DULL, "prcscr_col.gda"},
	{0x0F6F73692293A9E3ULL, "fen110_fenris6118060.cl"},
	{0x0F6F9EFD16F7D2EEULL, "art\\levels\\races\\ferelden\\props\\plotspecifc\\prp_urn.msh"},
	{0x0F6FACFD16F7EA3FULL, "art\\levels\\races\\ferelden\\props\\plotspecifc\\prp_urn.mao"},
	{0x0F6FB8FD16F7FEDCULL, "art\\levels\\races\\ferelden\\props\\plotspecifc\\prp_urn.mmh"},
	{0x0F70307659C30EB3ULL, "vxw_bonepit_1x0_0_0.vlm"},
	{0x0F70601387E88923ULL, "llac_ht_chantry_2534_0.dds"},
	{0x0F7087837BC79DB8ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\basestaff\\combat\\exit.ani"},
	{0x0F70AA6E7A240C34ULL, "vxw_sundermount_26a_0_0.vlm"},
	{0x0F716F7DAD70D976ULL, "958788315.ogg"},
	{0x0F724719232B2DDEULL, "hlac_ca_cavern1_haunted_24424_21y.dds"},
	{0x0F724FA0BECCB420ULL, "500851506.ogg"},
	{0x0F72C3C6CB5A0082ULL, "dae320ip_at_chantry.utp"},
	{0x0F731DF1F4359920ULL, "vxht_keep_n_671_0_0.vlm"},
	{0x0F733A57313FF928ULL, "vxlt_0_lowtown_a2_n_c_i16_0_0.vlm"},
	{0x0F7395102B6063ECULL, "vxht_0_hightown_a3_ca4_0_0.vlm"},
	{0x0F7487B601C63CE9ULL, "6121032_m.fxe"},
	{0x0F75581C7D349C3EULL, "art\\characters\\playercharacter\\humanmale\\animation\\exploration\\idle\\am_idle_tw_v5.ani"},
	{0x0F7574A1FA9C7827ULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\a_atk_v3.evt"},
	{0x0F75A6784ACA60E3ULL, "gen00fl_merrill6127294.cl"},
	{0x0F761C1032E2A019ULL, "618145372.ogg"},
	{0x0F7628C49AE634A9ULL, "vxdr_primevalentrance_34b_0_0.vlm"},
	{0x0F766027E380459DULL, "dae000_sandal6147803.cl"},
	{0x0F76A09423DE47A4ULL, "6082122_f.fxe"},
	{0x0F76D9229EE4C5A3ULL, "art\\vfx\\combat\\vfx_w_2nd_mblow_c.impa.evt"},
	{0x0F77654D5ABF4B1FULL, "909190760.ogg"},
	{0x0F779726320EF7E2ULL, "563947505.ogg"},
	{0x0F78010B5F448C5AULL, "6083703_f.fxe"},
	{0x0F78044CC728ADE4ULL, "6068877_m.fxe"},
	{0x0F783627BA9911B6ULL, "art\\vfx\\combat\\vfx_ha_hd_bdrain_lp_c.dura.evt"},
	{0x0F783F2942842440ULL, "tal_thre_clu.mao"},
	{0x0F78681895B45E1AULL, "vxw_sundermount_2ak_0_1.vlm"},
	{0x0F7896D087992208ULL, "art\\vfx\\combat\\vfx_m_pri_wintgrsp_imp_p.anb"},
	{0x0F78A30CC98F8233ULL, "359300357.ogg"},
	{0x0F7939AB821B5BCDULL, "114078512.ogg"},
	{0x0F7A799932B72A0BULL, "812927460.ogg"},
	{0x0F7AB9DF14026FF0ULL, "6027520_m.fxe"},
	{0x0F7AD3D4F7C97007ULL, "vxht_0_hightown_a2_c96_0_0.vlm"},
	{0x0F7AD922BA221E9FULL, "6053186_m.fxe"},
	{0x0F7B272DDC30466DULL, "6165036_m.fxe"},
	{0x0F7B281B50E409A7ULL, "827421384.ogg"},
	{0x0F7B2FCBFA88A3F8ULL, "755393826.ogg"},
	{0x0F7B8855F886D573ULL, "6084515_m.fxe"},
	{0x0F7B95A1FAA11C70ULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\a_atk_v3.gad"},
	{0x0F7BA70BC275EE96ULL, "6071907_m.fxe"},
	{0x0F7BB4022CE4CE33ULL, "lt_0_lowtown_a2_n_c_45_0_lc.mmh"},
	{0x0F7BE31326B406A7ULL, "mag160_feynriel6020055.cl"},
	{0x0F7C114AA5BAFDBCULL, "l_dae110ar_lowtown_nt.bnk"},
	{0x0F7C13314C7FC223ULL, "6151028_m.fxe"},
	{0x0F7C1C3AC362953CULL, "mag291_ntbanter6116213.cl"},
	{0x0F7D07E085A90F35ULL, "811332124.ogg"},
	{0x0F7D37DAF4568244ULL, "gen00fl_fenris6106683.cl"},
	{0x0F7D3922E33FB6E0ULL, "gen00fl_isabela6090250.cl"},
	{0x0F7D785F1D9543C7ULL, "llac_lt_blackemporium_1055_14.dds"},
	{0x0F7DAAB2B689FF3BULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\r_inst.ani"},
	{0x0F7DB2229EEAA224ULL, "art\\vfx\\combat\\vfx_w_2nd_mblow_c.impa.gad"},
	{0x0F7DBD83A70FD7B6ULL, "vxht_brothel_n_61z_0_0.vlm"},
	{0x0F7E34BEF3D19F0AULL, "6065270_m.fxe"},
	{0x0F7EBC4D907BE798ULL, "art\\levels\\races\\natural\\kirkcoastal\\props\\prp_03ruinswall.phy"},
	{0x0F7F1C7E74257F3BULL, "art\\vfx\\combat\\vfx_m_sta_rng_proj_ice_p.anb"},
	{0x0F7F2D27BA9F2141ULL, "art\\vfx\\combat\\vfx_ha_hd_bdrain_lp_c.dura.gad"},
	{0x0F7F6D9CD4A3CF69ULL, "leave_03.mao"},
	{0x0F7F875E47C44D01ULL, "vxht_0_hightown_b1q_0_0.vlm"},
	{0x0F7FAE0393ADC7A5ULL, "6107783_f.fxe"},
	{0x0F8025EF8271C632ULL, "6089571_m.fxe"},
	{0x0F80586F548E2B43ULL, "mag121_eau_banter6107384.cl"},
	{0x0F80828F7BC83FB6ULL, "vxw_encounter2_go_0_0.vlm"},
	{0x0F80B637DFCFEE4FULL, "vxw_bonepit_261_0_0.vlm"},
	{0x0F80C59F7D2F3E2EULL, "981778884.ogg"},
	{0x0F80DF326B72C590ULL, "vxht_0_hightown_n_bns_0_1.vlm"},
	{0x0F813FE07517560CULL, "196580357.ogg"},
	{0x0F81AFFC34A2D919ULL, "vxlt_gallowstemplar_432_0_0.vlm"},
	{0x0F81EA0BDDD50CF2ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\crouch_squat_enter.gad"},
	{0x0F826EEEB86157BAULL, "275211974.ogg"},
	{0x0F82701C7D3F4F0FULL, "art\\characters\\playercharacter\\humanmale\\animation\\exploration\\idle\\am_idle_tw_v5.evt"},
	{0x0F8293FD3ECB761AULL, "6127147_m.fxe"},
	{0x0F82AFEDE30AEE13ULL, "art\\vfx\\textures\\vfx_crater.dds"},
	{0x0F82EF32CD65DCC9ULL, "615382288.ogg"},
	{0x0F83119B3DE5503EULL, "vxw_bonepitdestroyed_a3_e1_0_0.vlm"},
	{0x0F83C299F72B1D42ULL, "mage_sleep.dds"},
	{0x0F83E58607F52D90ULL, "vxht_mansion_n_53v_0_0.vlm"},
	{0x0F84139EADA08C10ULL, "vxlt_hangedman_2o0_0_0.vlm"},
	{0x0F84B52A09BE4149ULL, "vxdr_primevaltomb_7bx_0_0.vlm"},
	{0x0F84C1229EF0DA02ULL, "art\\vfx\\combat\\vfx_w_2nd_mblow_c.impa.ani"},
	{0x0F850E8DD1E80B02ULL, "isa211_isabela6202148.cl"},
	{0x0F8534876F0CEF0EULL, "gen00fl_merrill6141240.cl"},
	{0x0F858B4A4C641D42ULL, "art\\characters\\playercharacter\\qunarimale\\animation\\combat\\arishock\\ari_charge.gad"},
	{0x0F85A5AB1D8ABC89ULL, "vxlt_0_lowtown_n_h7j_0_0.vlm"},
	{0x0F85D9C323D15352ULL, "52710470.ogg"},
	{0x0F862E27BAA541EFULL, "art\\vfx\\combat\\vfx_ha_hd_bdrain_lp_c.dura.ani"},
	{0x0F86808C74353095ULL, "876925927.ogg"},
	{0x0F8687BEB7D0D805ULL, "6204960_m.fxe"},
	{0x0F86917113B8907FULL, "fen210_orana6109775.cl"},
	{0x0F8699E5505B7AE1ULL, "vxlt_gallowsprison_n_64r_0_0.vlm"},
	{0x0F86CF7689AA68F7ULL, "w_bonepitmining_a2_70_0.msh"},
	{0x0F86E17689AA875AULL, "w_bonepitmining_a2_70_0.mao"},
	{0x0F86E57689AA8E69ULL, "w_bonepitmining_a2_70_0.mmh"},
	{0x0F871658FE860A0CULL, "vxw_bonepitmining_a2_gs_0_0.vlm"},
	{0x0F873DE271DEE066ULL, "6178360_f.fxe"},
	{0x0F8743844DFE3A86ULL, "275712774.ogg"},
	{0x0F876638E8979937ULL, "261532022.ogg"},
	{0x0F876C489CB35491ULL, "vxlt_0_lowtown_a3_n_c_gv9_0_0.vlm"},
	{0x0F884C7BC0519D76ULL, "6143057_m.fxe"},
	{0x0F88ED0BDDDB3029ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\crouch_squat_enter.evt"},
	{0x0F88FD8A8EF7B905ULL, "character_character_p262.psh.fxo.dx11.assembly"},
	{0x0F89711C7D456FB8ULL, "art\\characters\\playercharacter\\humanmale\\animation\\exploration\\idle\\am_idle_tw_v5.gad"},
	{0x0F89ABD09A4EE570ULL, "llac_lt_0_lowtown_docks_a3_-65539_2.dds"},
	{0x0F8A012FAFAA88CFULL, "cd_mer221_sabin.wbd"},
	{0x0F8A04A88D2CA05FULL, "365404519.ogg"},
	{0x0F8A4D795DA1613FULL, "544871257.ogg"},
	{0x0F8A56A32051953DULL, "vxht_0_hightown_n_c7y_0_0.vlm"},
	{0x0F8A6E0A95A90534ULL, "170829027.ogg"},
	{0x0F8AB2B2B6949732ULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\r_inst.evt"},
	{0x0F8AFE2A16C39A26ULL, "mrl310_merrill6144516.cl"},
	{0x0F8C5FA243271863ULL, "748056564.ogg"},
	{0x0F8CAE4A4C6A7739ULL, "art\\characters\\playercharacter\\qunarimale\\animation\\combat\\arishock\\ari_charge.evt"},
	{0x0F8CAEA295799480ULL, "420918080.ogg"},
	{0x0F8D07AAC87E0081ULL, "gen_im_wep_rog_dua_act3_01.uti"},
	{0x0F8D7E9A9A735454ULL, "hlac_lt_gallowstemplar_2384_1x.dds"},
	{0x0F8E00AD15DC0EA9ULL, "6103252_m.fxe"},
	{0x0F8E01AD59C79258ULL, "6130080_m.fxe"},
	{0x0F8E2143C1E096E0ULL, "vxlt_0_lowtown_docks_a3_n_b11_0_0.vlm"},
	{0x0F8E6AB77932A703ULL, "211370078.ogg"},
	{0x0F8E84A115C373EBULL, "gen00fl_aveline6139296.cl"},
	{0x0F8EAD54646C5031ULL, "203133711.ogg"},
	{0x0F8EC36201256796ULL, "236539776.ogg"},
	{0x0F8EF0BCC5A4463BULL, "45555694.ogg"},
	{0x0F8F2814D1C606CFULL, "vxw_bonepit_210_0_0.vlm"},
	{0x0F8FA3883BB11F64ULL, "hlac_ca_cavern1_haunted_24421_2y.dds"},
	{0x0F8FE31493ABF436ULL, "llac_ht_chantry_n_2534_24.dds"},
	{0x0F90250EA572B695ULL, "gen_im_trs_cot_arc_lbw_r2.uti"},
	{0x0F90343B4AFDA533ULL, "art\\vfx\\combat\\vfx_m_sta_crust_c.dura.ani"},
	{0x0F90E3237075C04AULL, "212092150.ogg"},
	{0x0F90FB83E3281C36ULL, "521503042.ogg"},
	{0x0F91D1B2B69AEAC5ULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\r_inst.gad"},
	{0x0F91E9E24A3A2CAFULL, "vxca_caverns2_7jl_0_0.vlm"},
	{0x0F920FD5457942A2ULL, "12758294.ogg"},
	{0x0F92857B9F96A490ULL, "vxlt_hangedman_2d1_0_0.vlm"},
	{0x0F92C2FD6E9BC207ULL, "vxw_bonepit_1vn_0_0.vlm"},
	{0x0F931B6CD6B7604BULL, "vxlt_0_lowtown_a2_n_erf_0_0.vlm"},
	{0x0F93277FE404AC73ULL, "art\\characters\\weapons\\staff\\textures\\w_stf_018a_0d.dds"},
	{0x0F93CCA9FA26F075ULL, "m_dae211ar_gamlens_house_nt.bnk"},
	{0x0F9417880F80D509ULL, "vxlt_0_lowtown_n_h2i_0_0.vlm"},
	{0x0F94554879AF0CD7ULL, "6067985_m.fxe"},
	{0x0F94604EAAC98792ULL, "zz_mrl_debug6086558.cl"},
	{0x0F9510EFA2431C5DULL, "dae011_corff_sta6142475.cl"},
	{0x0F9549D60203F4ECULL, "vxlt_gallowstemplar_n_5jf_0_0.vlm"},
	{0x0F958835F07C228CULL, "vxw_bonepitmining_a2_bp_0_0.vlm"},
	{0x0F95A1E4592849E4ULL, "mag311_nyssa6042401.cl"},
	{0x0F9658645268FE1CULL, "gen00fl_isabela6101057.cl"},
	{0x0F967C9CD4B6C717ULL, "leave_03.dds"},
	{0x0F96A168DE31D3B0ULL, "711211728.ogg"},
	{0x0F96D50BDDE74498ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\crouch_squat_enter.ani"},
	{0x0F9705F81E542A24ULL, "vxlt_0_lowtown_a2_gpf_0_0.vlm"},
	{0x0F97241E8EDC54A0ULL, "6066497_m.fxe"},
	{0x0F9727381064508EULL, "hlac_ht_0_hightown_a2_-131074_0x.dds"},
	{0x0F975F1DFB371987ULL, "zz_qun_debug6028259.cl"},
	{0x0F98539C80423BDBULL, "358227305.ogg"},
	{0x0F991C897F464BF3ULL, "6156858_m.fxe"},
	{0x0F99B64A4C750F28ULL, "art\\characters\\playercharacter\\qunarimale\\animation\\combat\\arishock\\ari_charge.ani"},
	{0x0F99C27D07BD096AULL, "one280_iwan6096915.cl"},
	{0x0F99DBA10F339A49ULL, "6095933_m.fxe"},
	{0x0F9A7A73B6BA9BCAULL, "lt_0_lowtown_docks_n_109_0.mao"},
	{0x0F9A7E73B6BAA219ULL, "lt_0_lowtown_docks_n_109_0.mmh"},
	{0x0F9A8873B6BAB307ULL, "lt_0_lowtown_docks_n_109_0.msh"},
	{0x0F9B860D2FA55833ULL, "vxw_bonepit_1k5_0_0.vlm"},
	{0x0F9BCC507A25AB27ULL, "vxht_keep_6f2_0_0.vlm"},
	{0x0F9BD29F12721258ULL, "395625042.ogg"},
	{0x0F9BF0487AB1E195ULL, "vxht_brothel_n_6f8_0_0.vlm"},
	{0x0F9D19A97665D410ULL, "hlac_ht_mansion2_818_39z.dds"},
	{0x0F9D50C9B7A40931ULL, "1222583.ogg"},
	{0x0F9DA8028F15A934ULL, "art\\levels\\4cinematics\\ferelden\\_textures\\prp_deadmeredith_n.dds"},
	{0x0F9E17AE8BEFD507ULL, "vxw_woundedcoast_wc_0_0.vlm"},
	{0x0F9E1E6F2151DA8DULL, "6136106_m.fxe"},
	{0x0F9E2C3B4B09D56AULL, "art\\vfx\\combat\\vfx_m_sta_crust_c.dura.evt"},
	{0x0F9E35BB7BDCF19EULL, "art\\vfx\\combat\\vfx_glb_fire_die_c.cess.ani"},
	{0x0F9E41745EE29CB1ULL, "llac_lt_undercityencounter_-1_16.dds"},
	{0x0F9F5CE2862AF07BULL, "vxlt_gallowstemplar_f_4qe_0_0.vlm"},
	{0x0F9FF55521D1813EULL, "vxw_bonepitdestroyed_a3_k3_0_0.vlm"},
	{0x0FA095D087BA72A3ULL, "art\\vfx\\combat\\vfx_m_pri_wintgrsp_imp_p.mmh"},
	{0x0FA0C93FEBE15E90ULL, "vxht_mansion_n_59t_0_0.vlm"},
	{0x0FA18D49C8AD78CBULL, "vxlt_0_lowtown_a2_n_ewy_0_0.vlm"},
	{0x0FA1B463A5CC22E0ULL, "and111_lirene6084499.cl"},
	{0x0FA3425417F03C7AULL, "140754036.ogg"},
	{0x0FA36B1AB8844051ULL, "llac_ca_caverns1_24424_26.dds"},
	{0x0FA38522A66CE1D2ULL, "llac_w_woundedcoast_131080_0.dds"},
	{0x0FA3BBB2F3FA0D6CULL, "vxlt_gallowstemplar_n_5mg_0_0.vlm"},
	{0x0FA47BB4905F8320ULL, "w_encounter2_camp_41_0.phy"},
	{0x0FA4927B2CFE2468ULL, "vxlt_0_lowtown_a3_n_c_gjx_0_0.vlm"},
	{0x0FA50B3B4B0FBBDDULL, "art\\vfx\\combat\\vfx_m_sta_crust_c.dura.gad"},
	{0x0FA537842D203112ULL, "603705575.ogg"},
	{0x0FA575C38AC51DC3ULL, "vxlt_hangedman_282_0_0.vlm"},
	{0x0FA57D454EEBC55CULL, "hlac_ht_chantry_2534_45x.dds"},
	{0x0FA5C08FDD5C1A7AULL, "632524987.ogg"},
	{0x0FA5C202D4221E52ULL, "246769183.ogg"},
	{0x0FA5C8BB113D289AULL, "hlac_lt_0_lowtown_docks_n_0_3z.dds"},
	{0x0FA5D50F35241395ULL, "lgt260_eau_tmp_conrad6143789.cl"},
	{0x0FA64BEE05149BF9ULL, "zz_followers_banter6158547.cl"},
	{0x0FA66B7B30F2A79CULL, "zz_ave_debug6059191.cl"},
	{0x0FA675A8840D5264ULL, "6164598_f.fxe"},
	{0x0FA6DB2654D4457CULL, "758325967.ogg"},
	{0x0FA73A5D043DC63DULL, "vxht_0_hightown_n_c9w_0_0.vlm"},
	{0x0FA75B039D5A805CULL, "6036535_m.fxe"},
	{0x0FA76E3B7CD5F063ULL, "6113128_f.fxe"},
	{0x0FA7D315F898C276ULL, "vxdr_primevaltomb_6s9_0_0.vlm"},
	{0x0FA7DA160B5656A1ULL, "vxht_brothel_n_6g7_0_0.vlm"},
	{0x0FA7EA5E99FD3F57ULL, "1025809327.ogg"},
	{0x0FA84F61BD37AD6AULL, "isa321_velasco6091861.cl"},
	{0x0FA879E4402CC610ULL, "730765068.ogg"},
	{0x0FA8DAB0B79A3F58ULL, "vxht_0_hightown_c5u_0_0.vlm"},
	{0x0FA9532C48E36A89ULL, "gen00fl_isabela6106415.cl"},
	{0x0FA958659DC61CCEULL, "6189666_m.fxe"},
	{0x0FA989C75C318F84ULL, "db_dot.mao"},
	{0x0FAA43BDD080AD69ULL, "llac_w_sundermount_65535_0.dds"},
	{0x0FAAA0BA6EA000D2ULL, "6200879_m.fxe"},
	{0x0FAB5D73FA08282CULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\ambient\\compressed\\sitting_table_right_hand_gesture_2.ani"},
	{0x0FAB8942B8F2F5C3ULL, "vxht_0_hightown_c3q_0_1.vlm"},
	{0x0FAB9AB49065D6D8ULL, "w_encounter2_camp_41_0.rml"},
	{0x0FAC4DBB7BE9586FULL, "art\\vfx\\combat\\vfx_glb_fire_die_c.cess.evt"},
	{0x0FADADC21EE7F31BULL, "hlac_lt_0_lowtown_0_7y.dds"},
	{0x0FADBD73A23AAFB5ULL, "hlac_ca_caverns1_mines_24424_22y.dds"},
	{0x0FADCEBF782108FBULL, "vxlt_gallowstemplar_f_4vd_0_0.vlm"},
	{0x0FADE4AAEAADB06FULL, "hlac_dr_primevaltomb_659_12x.dds"},
	{0x0FAE50554A7AB44AULL, "17196619.ogg"},
	{0x0FAE7EC00E10B289ULL, "isa211ar_foundry.are"},
	{0x0FAEF31ACB6E111EULL, "6120407_m.fxe"},
	{0x0FAF336B0961B32FULL, "art\\levels\\races\\proxy\\props\\4tools\\inv_ball4_tool.mao"},
	{0x0FAF3F6B0961C7CCULL, "art\\levels\\races\\proxy\\props\\4tools\\inv_ball4_tool.mmh"},
	{0x0FAF456B0961D1BEULL, "art\\levels\\races\\proxy\\props\\4tools\\inv_ball4_tool.msh"},
	{0x0FAFA2E55FE9B645ULL, "406457087.ogg"},
	{0x0FAFADBB68F5860FULL, "208170311.ogg"},
	{0x0FAFFF26BAA3914BULL, "vxlt_0_lowtown_a2_n_exx_0_0.vlm"},
	{0x0FB091C6CD40F156ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\animation\\dwarven_trap.inactivetoactive.evt"},
	{0x0FB10CD7FE0B4E3DULL, "282013755.ogg"},
	{0x0FB10FE1498393EBULL, "214847128.ogg"},
	{0x0FB17D3BF397B117ULL, "rdr110dg_guard_warehouse6016696.cl"},
	{0x0FB18061578FD2AFULL, "955539371.ogg"},
	{0x0FB1C6C6ED4B6D07ULL, "dae000cr_shopper_f.utc"},
	{0x0FB1E586CBF10CF7ULL, "vxca_caverns2_8mj_0_0.vlm"},
	{0x0FB208A4CC1AA457ULL, "vxlt_darktown_52s_0_1.vlm"},
	{0x0FB21785248EE076ULL, "mer141cr_jansen.utc"},
	{0x0FB24EBB7BEDC518ULL, "art\\vfx\\combat\\vfx_glb_fire_die_c.cess.gad"},
	{0x0FB26A73FA0E5D6EULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\ambient\\compressed\\sitting_table_right_hand_gesture_2.gad"},
	{0x0FB284531DBE77BAULL, "zz_party.ncs"},
	{0x0FB29B1197AEC1E6ULL, "vxht_mansion_n_6k7_0_0.vlm"},
	{0x0FB31E3C98AE6514ULL, "vxlt_blackemporium_zm_0_0.vlm"},
	{0x0FB3E7A07CBB3643ULL, "vxlt_hangedman_255_0_0.vlm"},
	{0x0FB3FEAD40A5EEDEULL, "llac_lt_0_lowtown_a2_n_-3_0.dds"},
	{0x0FB4771A59EF86D2ULL, "54522659.ogg"},
	{0x0FB4CC606FCCA8DFULL, "vxlt_gallowstemplar_38v_0_0.vlm"},
	{0x0FB4CD268266E081ULL, "hlac_w_sundermount_131068_0y.dds"},
	{0x0FB5359EBC051C69ULL, "6041734_m.fxe"},
	{0x0FB53691F4B93E89ULL, "w_bonepitmining_a2_149_0.mao"},
	{0x0FB54291F4B95322ULL, "w_bonepitmining_a2_149_0.mmh"},
	{0x0FB54491F4B956D8ULL, "w_bonepitmining_a2_149_0.msh"},
	{0x0FB5890A1D59CD88ULL, "t3_wdg_w01.tnt"},
	{0x0FB6344CE68DDE72ULL, "6118358_m.fxe"},
	{0x0FB63C5F56FC8BBAULL, "6056856_m.fxe"},
	{0x0FB6AE3105AF17E4ULL, "6136741_m.fxe"},
	{0x0FB6D78363A924FAULL, "follower_barks6173416.cl"},
	{0x0FB732DDD9409B41ULL, "6020992_f.fxe"},
	{0x0FB77A808371BC14ULL, "one220_yevhen6103110.cl"},
	{0x0FB788C6CD4700E1ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\animation\\dwarven_trap.inactivetoactive.gad"},
	{0x0FB8353D7E6288BBULL, "ip_restrict_style_dualwield.ncs"},
	{0x0FB86158813ED3F6ULL, "mag210_arianni6030493.cl"},
	{0x0FB8841C88C7FDABULL, "t3_grl_g03.tnt"},
	{0x0FB91E43D2B925E2ULL, "vxht_0_hightown_a2_bwd_0_0.vlm"},
	{0x0FB9C8C2744F545AULL, "dae000_female_tranquil6165136.cl"},
	{0x0FBB17A277C1EE87ULL, "hlac_lt_0_lowtown_docks_a3_-65538_6z.dds"},
	{0x0FBB7E4235D13EFDULL, "vxdr_0_deeproads_fin_0_0.vlm"},
	{0x0FBBA40A82D43E33ULL, "6101546_m.fxe"},
	{0x0FBBEC8F6EFBA2F9ULL, "6111668_m.fxe"},
	{0x0FBC1C53122B14D6ULL, "art\\vfx\\combat\\vfx_w_gn1_wcry_loop_pen_c.dura.ani"},
	{0x0FBC33A5C1A9C3B3ULL, "263571850.ogg"},
	{0x0FBC4839A58F4337ULL, "vxlt_0_lowtown_a2_n_c_ef8_0_0.vlm"},
	{0x0FBD0EFCE97AFC45ULL, "hlac_lt_0_lowtown_docks_n_-3_1y.dds"},
	{0x0FBD35446E590A29ULL, "mcr370_meredith6084413.cl"},
	{0x0FBD89C6CD4B6D8FULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\animation\\dwarven_trap.inactivetoactive.ani"},
	{0x0FBD91A88EA4D13BULL, "6120603_f.fxe"},
	{0x0FBDD51CEA1AD20CULL, "6144527_f.fxe"},
	{0x0FBDE0C8D22E1EEDULL, "vxlt_0_lowtown_n_cmk_0_0.vlm"},
	{0x0FBE3DF26DEC860EULL, "65274583.ogg"},
	{0x0FBEA26C0A9DC437ULL, "dae200_saemus6142812.cl"},
	{0x0FBF6C2B0027F58AULL, "877543222.ogg"},
	{0x0FBF7A9B86B76CEAULL, "vxlt_hangedman_n_1r9_0_0.vlm"},
	{0x0FBFF2A9FA4C6BFFULL, "m_dae211ar_gamlens_house_nt.wbd"},
	{0x0FC0017A0678173FULL, "vxlt_gallowstemplar_n_5ji_0_0.vlm"},
	{0x0FC030C1660DF265ULL, "lt_0_lowtown_docks_n_21_0.rml"},
	{0x0FC0BAD706D365FAULL, "492974546.ogg"},
	{0x0FC10CEE89A4DA66ULL, "vxht_mansion_n_6n8_0_0.vlm"},
	{0x0FC13E77D4392952ULL, "llac_ht_playershouse_n_855_14.dds"},
	{0x0FC1B4EDB7E12929ULL, "708673468.ogg"},
	{0x0FC223D580365918ULL, "lt_0_lowtown_docks_12_0_ll.mmh"},
	{0x0FC2597D6EB14EC3ULL, "vxlt_hangedman_224_0_0.vlm"},
	{0x0FC25AA9C29128A8ULL, "llac_lt_0_lowtown_docks_131070_4.dds"},
	{0x0FC2758FCBDEE9FDULL, "hlac_lt_backalley_n_0_2x.dds"},
	{0x0FC2CB4E16CCDB52ULL, "mrl351_phantomelves6170344.cl"},
	{0x0FC321AB432653CCULL, "6094489_m.fxe"},
	{0x0FC3BFC5C0B0565AULL, "vxw_sundermount_24a_0_0.vlm"},
	{0x0FC3DBBF3DBF3C28ULL, "zz_prmpt_debug6198224.cl"},
	{0x0FC40D597D45323CULL, "vxlt_gallowstemplar_43k_0_0.vlm"},
	{0x0FC418F5EA625B6EULL, "vxht_0_hightown_n_a4m_0_0.vlm"},
	{0x0FC42489FB828C0EULL, "593822890.ogg"},
	{0x0FC43163B257A029ULL, "art\\levels\\races\\ferelden\\props\\docks\\barge\\prp_02brgdebris.mao"},
	{0x0FC43D63B257B4C2ULL, "art\\levels\\races\\ferelden\\props\\docks\\barge\\prp_02brgdebris.mmh"},
	{0x0FC43F63B257B878ULL, "art\\levels\\races\\ferelden\\props\\docks\\barge\\prp_02brgdebris.msh"},
	{0x0FC4C10839629A85ULL, "6086365_f.fxe"},
	{0x0FC4EBBD6988D598ULL, "41336708.ogg"},
	{0x0FC50282DAE705BDULL, "da_defaultcc.dds"},
	{0x0FC50A7AEECDCAD2ULL, "art\\characters\\playercharacter\\humanmale\\textures\\hm_hlh_medg_0n.dds"},
	{0x0FC526A7ED1A453DULL, "vxht_0_hightown_a2_c1l_0_0.vlm"},
	{0x0FC53F6F3068939FULL, "384000354.ogg"},
	{0x0FC5BE8A8812A504ULL, "vxlt_0_lowtown_a3_n_c_hwt_0_0.vlm"},
	{0x0FC5F1C4D4027770ULL, "871939010.ogg"},
	{0x0FC676884622B9D2ULL, "qun110_petrice6041358.cl"},
	{0x0FC6B00A923841BCULL, "dae000cr_grand_cleric6148161.cl"},
	{0x0FC70466459CF3ADULL, "follower_banter6162104.cl"},
	{0x0FC714EAF28B6741ULL, "dae000_bartrand6206859.cl"},
	{0x0FC79020C4AF3E62ULL, "vxht_0_hightown_a2_bxe_0_0.vlm"},
	{0x0FC7CAE233AB3707ULL, "752234.ogg"},
	{0x0FC84248CFA3A02CULL, "hlac_lt_0_lowtown_docks_a3_n_-3_0z.dds"},
	{0x0FC8AA4E48D7E9CBULL, "vxca_caverns1_mines_i1q_0_0.vlm"},
	{0x0FC924531235ACC7ULL, "art\\vfx\\combat\\vfx_w_gn1_wcry_loop_pen_c.dura.evt"},
	{0x0FC93197221A1EC1ULL, "ch2emnotnt.psh.fxo"},
	{0x0FC93815C054EEBAULL, "6117947_f.fxe"},
	{0x0FC998B4907ED0B5ULL, "w_encounter2_camp_41_0.msh"},
	{0x0FC99AB4907ED42BULL, "w_encounter2_camp_41_0.mmh"},
	{0x0FC9A6B4907EE8C8ULL, "w_encounter2_camp_41_0.mao"},
	{0x0FC9C7DEE435A8BEULL, "6166301_f.fxe"},
	{0x0FCA0F242FDC915AULL, "6019152_m.fxe"},
	{0x0FCA1CF2BC418C92ULL, "vxlt_0_lowtown_a2_n_c_gz7_0_0.vlm"},
	{0x0FCA466407E1CDC5ULL, "vxw_bonepitdestroyed_a3_18v_0_0.vlm"},
	{0x0FCACC648E5D09E4ULL, "vxw_encounter3_g6_0_0.vlm"},
	{0x0FCB1B2FAFE24BA5ULL, "cd_mer221_sabin.bnk"},
	{0x0FCB1E6AC6A623FEULL, "mag100_cullen6148681.cl"},
	{0x0FCB2B480096F5B5ULL, "nat380_sibling6207557.cl"},
	{0x0FCC185D426FA6CDULL, "gen_im_wep_mag_sta_act3_08.uti"},
	{0x0FCC6E7942455C28ULL, "zz_ave_debug6056756.cl"},
	{0x0FCC8A7136742407ULL, "vxw_bonepit_1jj_0_0.vlm"},
	{0x0FCCFDE0236E9CC4ULL, "vxlt_0_lowtown_docks_bg7_0_0.vlm"},
	{0x0FCD533BAEB1EA10ULL, "109137364.ogg"},
	{0x0FCD6E9639E11BE5ULL, "vxht_brothel_n_6fh_0_0.vlm"},
	{0x0FCE09E9DCFEE8A3ULL, "art\\characters\\weapons\\maces\\textures\\w_mce_mc103a_0t.dds"},
	{0x0FCE5C82D65225E2ULL, "art\\characters\\playercharacter\\humanmale\\textures\\hm_arm_ric_0s.dds"},
	{0x0FCEC27D1FA88A5BULL, "546733285.ogg"},
	{0x0FCEE2B257A151C9ULL, "253801431.ogg"},
	{0x0FCF99059F273CEBULL, "gen_im_trs_dem_sta_fir_r0.uti"},
	{0x0FD01993100A4013ULL, "hlac_lt_0_lowtown_n_0_0y.dds"},
	{0x0FD04553123C0310ULL, "art\\vfx\\combat\\vfx_w_gn1_wcry_loop_pen_c.dura.gad"},
	{0x0FD0D308EFD000E2ULL, "zz_followers_banter6157738.cl"},
	{0x0FD131874E759A78ULL, "tanner_deep_roads_v1.cut"},
	{0x0FD1ADC2A11F4F56ULL, "657757005.ogg"},
	{0x0FD1C1A8CA9B1FCEULL, "hlac_ht_keep_3732_5y.dds"},
	{0x0FD27F366F3B4ABCULL, "vxlt_gallowstemplar_46h_0_0.vlm"},
	{0x0FD2ACE27038D7AEULL, "9150258.ogg"},
	{0x0FD2C595302110E2ULL, "vxca_caverns2_8p4_0_0.vlm"},
	{0x0FD36FC00E2FB5DFULL, "isa211ar_foundry.ncs"},
	{0x0FD3B25F0C5E8856ULL, "fx_dust05_a.dds"},
	{0x0FD3CD141FB5B9D3ULL, "llac_lt_0_lowtown_a2_n_c_-1_5.dds"},
	{0x0FD3E5787C199601ULL, "428692525.ogg"},
	{0x0FD48875629CABB9ULL, "lgt210_eau_keran_father6144906.cl"},
	{0x0FD490402FFEF98FULL, "hlac_lt_0_lowtown_a2_131071_7x.dds"},
	{0x0FD4D8C029B35056ULL, "6122095_f.fxe"},
	{0x0FD4F454BF21F080ULL, "vxlt_warehouse_n_12o_0_0.vlm"},
	{0x0FD5620E96E61908ULL, "gen_im_npc_elf_arm_cht_lgta.uti"},
	{0x0FD5BC0160295849ULL, "6085741_m.fxe"},
	{0x0FD5F4D44CBBCB66ULL, "gen00fl_aveline6112704.cl"},
	{0x0FD6C9C4B581D74CULL, "and111_lirene6085100.cl"},
	{0x0FD6F58A938BDB03ULL, "428179558.ogg"},
	{0x0FD771483DFE1AFFULL, "6118855_f.fxe"},
	{0x0FD7D0EE73BD9E96ULL, "6087195_m.fxe"},
	{0x0FD7FE816E802E0FULL, "vxw_woundedcoast_3w_0_0.vlm"},
	{0x0FD83BC7339D8705ULL, "6034679_m.fxe"},
	{0x0FD8419A576B1464ULL, "art\\vfx\\combat\\vfx_w_2nd_whirl_c.mmh"},
	{0x0FD87060C71FA740ULL, "zz_followers_banter6158382.cl"},
	{0x0FD90E2173A3A24EULL, "vxlt_0_lowtown_a2_n_c_ff2_0_0.vlm"},
	{0x0FD9F26453C8A015ULL, "vxw_sundermount_277_0_0.vlm"},
	{0x0FDA6C0BAA8B2DF4ULL, "hlac_lt_0_lowtown_docks_n_-4_4z.dds"},
	{0x0FDA8FC00E360ACEULL, "isa211ar_foundry.lst"},
	{0x0FDA94E13A90DE48ULL, "192513964.ogg"},
	{0x0FDAA94D90CA262DULL, "art\\levels\\races\\natural\\kirkcoastal\\props\\prp_03ruinswall.msh"},
	{0x0FDAB74D90CA3E60ULL, "art\\levels\\races\\natural\\kirkcoastal\\props\\prp_03ruinswall.mao"},
	{0x0FDABB4D90CA44B3ULL, "art\\levels\\races\\natural\\kirkcoastal\\props\\prp_03ruinswall.mmh"},
	{0x0FDB06A016BC0559ULL, "hlac_ht_keep_4519_2y.dds"},
	{0x0FDBA8DE8D400D7AULL, "art\\characters\\creatures\\qunari\\textures\\qm_hed_mag_0d.dds"},
	{0x0FDBD84C1C6DA2C4ULL, "6075215_m.fxe"},
	{0x0FDC69E9504710BBULL, "hlac_lt_0_lowtown_a2_n_c_-65537_2x.dds"},
	{0x0FDC6F61398511F3ULL, "gen00fl_merrill6151105.cl"},
	{0x0FDC77E64CA6CAFEULL, "ave200_donnic6112145.cl"},
	{0x0FDCB8AAAA3E60D5ULL, "amulet_of_ashes.dds"},
	{0x0FDD061091A1A787ULL, "vxlt_undercityserialkiller_1j5_0_1.vlm"},
	{0x0FDD265B034BD6AEULL, "1009000110.ogg"},
	{0x0FDD81F27D69AEB5ULL, "vxlt_hangedman_4n_0_0.vlm"},
	{0x0FDD8944D79D08B7ULL, "433271960.ogg"},
	{0x0FDDB7755DC3C144ULL, "6034817_f.fxe"},
	{0x0FDDC30CD21BFF51ULL, "vxdr_0_deeproads_esk_0_0.vlm"},
	{0x0FDE4F2B981A4BB4ULL, "follower_banter6149214.cl"},
	{0x0FDE6D9546507890ULL, "vxw_encounter2_camp_rn_0_0.vlm"},
	{0x0FDE73D36E90AE94ULL, "vxlt_blackemporium_up_0_0.vlm"},
	{0x0FDE74E1E1C330DFULL, "314517347.ogg"},
	{0x0FDE858F89B5854CULL, "276485947.ogg"},
	{0x0FDE9082DD2C065CULL, "hlac_lt_0_lowtown_docks_a3_-2_5y.dds"},
	{0x0FDED4DC89BE48E0ULL, "hlac_lt_0_lowtown_docks_n_65533_1x.dds"},
	{0x0FDF5F5D1F86E0BBULL, "vxlt_gallowstemplar_5uo_0_0.vlm"},
	{0x0FDF85710E4964D4ULL, "llac_ca_caverns2_5802_54.dds"},
	{0x0FE02B0FE737DA0BULL, "vxdr_primevaltomb_3gq_0_0.vlm"},
	{0x0FE0FD0D1A0DC094ULL, "zz_dae_debug6142967.cl"},
	{0x0FE118F0B35BCBAEULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\3ca_gallowtopoc.mmh"},
	{0x0FE124F0B35BE045ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\3ca_gallowtopoc.mao"},
	{0x0FE132F0B35BF79CULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\3ca_gallowtopoc.msh"},
	{0x0FE17187D076861BULL, "522738497.ogg"},
	{0x0FE1E13054EC0891ULL, "gen00fl_anders6201084.cl"},
	{0x0FE1E5CC717A1AB4ULL, "art\\vfx\\combat\\vfx_c_pride_cast_f_c.mmh"},
	{0x0FE24EEC27CFE77CULL, "gen00fl_varric6076229.cl"},
	{0x0FE3B1AB83E94622ULL, "pro000ip_corpse_weapons3.utp"},
	{0x0FE3C50216382296ULL, "hlac_lt_0_lowtown_docks_-1_2z.dds"},
	{0x0FE419F451E0FA14ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\charge_2h_sw.ani"},
	{0x0FE4448CC60300D4ULL, "399293560.ogg"},
	{0x0FE4471FAFE75568ULL, "6102409_m.fxe"},
	{0x0FE504C2419869B5ULL, "6042267_m.fxe"},
	{0x0FE528693990B824ULL, "6070528_m.fxe"},
	{0x0FE563143A08D865ULL, "character_character_p28.psh.fxo"},
	{0x0FE5817C1B5F3E50ULL, "hlac_lt_gallowsprison_n_-1_7z.dds"},
	{0x0FE5CFD4A19997F3ULL, "6045798_m.fxe"},
	{0x0FE5EA1A9DA987DAULL, "151553231.ogg"},
	{0x0FE6949DD3EF5B76ULL, "vrc221_varric6043508.cl"},
	{0x0FE73C4FDBB561F2ULL, "hlac_w_encounter2_131070_0z.dds"},
	{0x0FE740B717C63315ULL, "648229699.ogg"},
	{0x0FE7EA2848777285ULL, "57915477.ogg"},
	{0x0FE824D2A2E21467ULL, "vxw_bonepit_243_0_0.vlm"},
	{0x0FE84E5F9EE0EB3FULL, "gen00fl_merrill6107454.cl"},
	{0x0FE859445ECF2B6AULL, "564301498.ogg"},
	{0x0FE8644145BEB895ULL, "vxw_sundermount_286_0_0.vlm"},
	{0x0FE888D69181F3AAULL, "llac_lt_blackemporium_1055_0.dds"},
	{0x0FE9A756E764E4D2ULL, "160353541.ogg"},
	{0x0FE9FC4A616A3BDDULL, "llac_ht_0_hightown_-1_0.dds"},
	{0x0FEAC7CD1FED372DULL, "hlac_ca_cavern1_haunted_24425_4z.dds"},
	{0x0FEADDA85673BE0FULL, "6035905_f.fxe"},
	{0x0FEAECED1D593A37ULL, "18362339.ogg"},
	{0x0FEB16F451E71476ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\charge_2h_sw.gad"},
	{0x0FEB1EB533C726CBULL, "1008875558.ogg"},
	{0x0FEB9E3824564761ULL, "mage_crushingprison.dds"},
	{0x0FEBA02B0754E20AULL, "hlac_ht_mansion_n_3373_0x.dds"},
	{0x0FEBD228231C8B12ULL, "6018902_m.fxe"},
	{0x0FEBFF185F0F700BULL, "vxht_brothel_n_4gs_0_0.vlm"},
	{0x0FEC2D0018B81034ULL, "6041512_f.fxe"},
	{0x0FEC3F224CF15098ULL, "vxlt_0_lowtown_docks_n_cuo_0_0.vlm"},
	{0x0FEC5A48A426C704ULL, "vxlt_undercitydungeon_1ga_0_0.vlm"},
	{0x0FECDF7238469110ULL, "vxw_encounter2_camp_mo_0_0.vlm"},
	{0x0FED13DF19C4080AULL, "444961707.ogg"},
	{0x0FED2AFFDBF0DC98ULL, "vxht_mansion2b_1kq_0_0.vlm"},
	{0x0FEE4058A208C7A7ULL, "vxht_0_hightown_n_9bb_0_0.vlm"},
	{0x0FEED48E901AECEFULL, "930376762.ogg"},
	{0x0FEF9FF967B17562ULL, "vxht_0_hightown_bu2_0_0.vlm"},
	{0x0FEFF52E13487370ULL, "6174875_m.fxe"},
	{0x0FF03396774763C2ULL, "938662822.ogg"},
	{0x0FF18D3376F16541ULL, "6134764_m.fxe"},
	{0x0FF1C4C2475711F4ULL, "894926718.ogg"},
	{0x0FF1CAECF73E0CC5ULL, "782868473.ogg"},
	{0x0FF1F2E99966FC61ULL, "211654374.ogg"},
	{0x0FF231F451ED60D5ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\charge_2h_sw.evt"},
	{0x0FF2F723BA0770E5ULL, "gencr_circle_mage_f.utc"},
	{0x0FF2FFA79F39D953ULL, "vxw_blightlands_a7_0_0.vlm"},
	{0x0FF325A0B907CE5DULL, "969669369.ogg"},
	{0x0FF32FC16639498DULL, "lt_0_lowtown_docks_n_21_0.anb"},
	{0x0FF42258C7F38E92ULL, "78170993.ogg"},
	{0x0FF4421DB82F9D30ULL, "art\\vfx\\models\\vfx_slash.mmh"},
	{0x0FF446EE3E324B7BULL, "isa321cr_velasco.utc"},
	{0x0FF4501DB82FB54AULL, "art\\vfx\\models\\vfx_slash.msh"},
	{0x0FF45226C6EAE9F7ULL, "vxlt_gallowstemplar_5oy_0_0.vlm"},
	{0x0FF49FCCD6679B0DULL, "6143151_m.fxe"},
	{0x0FF4AB8C5AAB66B6ULL, "al_warrior_t06.gda"},
	{0x0FF4E4B7483B5F95ULL, "189472556.ogg"},
	{0x0FF4EE831D0448EAULL, "art\\vfx\\combat\\vfx_m_frc_forcecrush_c.anb"},
	{0x0FF50193C5D80F39ULL, "742246714.ogg"},
	{0x0FF523040DDF5BC1ULL, "6098259_m.fxe"},
	{0x0FF58D8FDE0B580BULL, "201251057.ogg"},
	{0x0FF5F2C96022C146ULL, "55696102.ogg"},
	{0x0FF6FAC85435ABAAULL, "hlac_dr_primevaltomb_658_7z.dds"},
	{0x0FF70EEE3C9E6257ULL, "mcr310_argument6074431.cl"},
	{0x0FF757341D322920ULL, "237833891.ogg"},
	{0x0FF7787A54635B8AULL, "dae000_arishok6067740.cl"},
	{0x0FF7E70228892320ULL, "narrative_varric_demo6170819.cl"},
	{0x0FF8646AAB02D796ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\pose_a_kick_feet_shrug.ani"},
	{0x0FF8B221F9EA182BULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\rogue\\bow\\bw_ent.ani"},
	{0x0FF929141188F2D3ULL, "gen00fl_aveline6197230.cl"},
	{0x0FF9672B33D6011DULL, "hlac_w_bonepit_0_0y.dds"},
	{0x0FFA49DA167C9DE0ULL, "204915911.ogg"},
	{0x0FFA4D7093B6857AULL, "vxlt_gallowscourtyard_7bs_0_0.vlm"},
	{0x0FFA6DE9F7298E65ULL, "435845770.ogg"},
	{0x0FFABFBC235E00AAULL, "764311949.ogg"},
	{0x0FFADCCB55EA6007ULL, "spt_fronds.psh.fxo"},
	{0x0FFB4BF06B6C5CC3ULL, "art\\characters\\creatures\\dragon\\animation\\cwkb.evt"},
	{0x0FFB514F2A3CA990ULL, "vxw_encounter2_camp_hl_0_0.vlm"},
	{0x0FFB9DB579C8B2DDULL, "433171578.ogg"},
	{0x0FFBA4BE72465D2DULL, "622642780.ogg"},
	{0x0FFBE77EC782370AULL, "vxdr_primevalentrance_26v_0_0.vlm"},
	{0x0FFC119D4C02C3DAULL, "vxw_encounter2_55_0_0.vlm"},
	{0x0FFC1F91137FB2A9ULL, "396437647.ogg"},
	{0x0FFC805D608E77D2ULL, "vxw_blightlands_a2_0_0.vlm"},
	{0x0FFC841FB5AA4118ULL, "ico_crossbow_3.dds"},
	{0x0FFCEE1CC507754BULL, "art\\characters\\creatures\\desiredemon\\animation\\com_v3.evt"},
	{0x0FFD5A258A588E99ULL, "6219135_m.fxe"},
	{0x0FFDE84D3970FFB5ULL, "vxlt_0_lowtown_n_ews_0_0.vlm"},
	{0x0FFEF001D12CD308ULL, "fen000im_gft_book.uti"},
	{0x0FFF85FE4FEB0704ULL, "vxlt_0_lowtown_a3_n_c_hcx_0_0.vlm"},
	{0x0FFF98578D686C2FULL, "602272725.ogg"},
	{0x0FFF9EC41FAA6928ULL, "vxlt_darktown_6oa_0_0.vlm"},
	{0x0FFFBE4828B6B436ULL, "vxht_brothel_n_6bc_0_0.vlm"},
	{0x1000D9D2D29C7F6CULL, "zz_vis_debug6050354.cl"},
	{0x1000F9054B7EDEFBULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\9ff_1rubblepile.msh"},
	{0x10010B054B7EFD46ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\9ff_1rubblepile.mao"},
	{0x100117054B7F11E5ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\9ff_1rubblepile.mmh"},
	{0x10011D0FEDC10134ULL, "vxw_sundermount_2do_0_0.vlm"},
	{0x1001449A578E1E5BULL, "art\\vfx\\combat\\vfx_w_2nd_whirl_c.anb"},
	{0x10017184912FF1D3ULL, "vxw_blightlands_j6_0_0.vlm"},
	{0x1001DC436B08AEFEULL, "abi_m_fireball.bnk"},
	{0x100202B034119D04ULL, "445069167.ogg"},
	{0x100224F06B723944ULL, "art\\characters\\creatures\\dragon\\animation\\cwkb.gad"},
	{0x1002A9BF53D4D2C0ULL, "897744690.ogg"},
	{0x1002C41B5BD6F522ULL, "llac_ht_0_hightown_n_-65536_0.dds"},
	{0x100310B3A1F92070ULL, "hlac_lt_0_lowtown_a2_n_65535_8x.dds"},
	{0x1004071CC50DBE0CULL, "art\\characters\\creatures\\desiredemon\\animation\\com_v3.gad"},
	{0x1005448B23390B2CULL, "ts_kirk_hightown_prog_junk1.gda"},
	{0x1005AA965E7A7EC2ULL, "vxw_encounter2_camp_qq_0_0.vlm"},
	{0x1005FF67A57E688FULL, "hlac_ca_caverns2prc_5802_34z.dds"},
	{0x1006205F0D69AC2BULL, "art\\characters\\playercharacter\\qunarimale\\animation\\combat\\trooperbase\\troop_attk_v1.evt"},
	{0x10066C6AAB0F2287ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\pose_a_kick_feet_shrug.evt"},
	{0x10069A21F9F62CA2ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\rogue\\bow\\bw_ent.evt"},
	{0x10071C4A7704599EULL, "665595950.ogg"},
	{0x1007D47A7BCA4C42ULL, "hlac_ht_keep_3731_3z.dds"},
	{0x1007F0060CD62969ULL, "vxca_caverns1_jgo_0_0.vlm"},
	{0x100833F06B76BE22ULL, "art\\characters\\creatures\\dragon\\animation\\cwkb.ani"},
	{0x100840A62127AC5FULL, "lgt110_gxa_coa_meeting6117497.cl"},
	{0x100860E2509D10DFULL, "hlac_lt_0_lowtown_n_1_4z.dds"},
	{0x1008B484BD493886ULL, "hlac_lt_0_lowtown_docks_n_65533_0y.dds"},
	{0x1008BF4D85AC9DFAULL, "vxlt_gallowscourtyard_7gl_0_0.vlm"},
	{0x10090FB0F0ECFE72ULL, "6165004_m.fxe"},
	{0x100918A3A7FE48D1ULL, "vxdr_0_deeproads_ebh_0_0.vlm"},
	{0x1009AAA5DE3AC6D9ULL, "nosynccombatanims.gda"},
	{0x100A061CC512280AULL, "art\\characters\\creatures\\desiredemon\\animation\\com_v3.ani"},
	{0x100A17162103CC30ULL, "mag211_marethari6021347.cl"},
	{0x100AE8CC719D256BULL, "art\\vfx\\combat\\vfx_c_pride_cast_f_c.anb"},
	{0x100AF23A52849052ULL, "vxw_blightlands_j3_0_0.vlm"},
	{0x100AF9643E348873ULL, "llac_w_bonepit_327681_0.dds"},
	{0x100B04E4A6FB7A57ULL, "ave200_aveline_bar6116444.cl"},
	{0x100B60B2C88C1B44ULL, "6163765_m.fxe"},
	{0x100B63C90F830CFFULL, "6019126_m.fxe"},
	{0x100B70B212F7DA65ULL, "vxw_sundermount_27g_0_0.vlm"},
	{0x100B9AD0C8B18382ULL, "dae000_bartrand6207004.cl"},
	{0x100BE45F2CD8DD1BULL, "mcr310_sibling6076379.cl"},
	{0x100D395F0D6FF5ECULL, "art\\characters\\playercharacter\\qunarimale\\animation\\combat\\trooperbase\\troop_attk_v1.gad"},
	{0x100D8D6AAB1579D0ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\pose_a_kick_feet_shrug.gad"},
	{0x100D98C82146DB95ULL, "hlac_w_encounter2_camp_-131072_0z.dds"},
	{0x100D9921F9FC49B5ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\rogue\\bow\\bw_ent.gad"},
	{0x100DCD5AE5BF0161ULL, "hlac_lt_0_lowtown_a2_n_c_131072_2y.dds"},
	{0x100E83361A8A0FAAULL, "llac_lt_0_lowtown_docks_-1_2.dds"},
	{0x100EAF4407BB94B2ULL, "art\\characters\\playercharacter\\humanmale\\textures\\hm_glv_wrba_0d.dds"},
	{0x100EC55890AF22FBULL, "6177891_m.fxe"},
	{0x100F7BF70F7FB09DULL, "art\\levels\\races\\primeval\\props\\_texture\\prp_onyxstatuette_sa.dds"},
	{0x100F8EECDFB719B4ULL, "vxw_sundermount_2il_0_0.vlm"},
	{0x100FE36183260A53ULL, "vxw_blightlands_o9_0_0.vlm"},
	{0x100FF566249DD24FULL, "252866714.ogg"},
	{0x1010267C93C37818ULL, "mag110_vincento6017675.cl"},
	{0x1010BFC672B76D04ULL, "122308552.ogg"},
	{0x1011488D19A96936ULL, "vxw_blightlands_9n_0_0.vlm"},
	{0x1011757BD6DFDBFAULL, "858950708.ogg"},
	{0x1011906F6A77BDA9ULL, "vxht_keep_n_63x_0_0.vlm"},
	{0x1011E0BB38D669C5ULL, "6128945_m.fxe"},
	{0x1011FE22D090EDFAULL, "96274457.ogg"},
	{0x101224E9AA89298CULL, "dae000_mother6004110.cl"},
	{0x101282EEFE9B8CD9ULL, "zz_followers_banter6159711.cl"},
	{0x1012D8681C4D8E33ULL, "6127511_m.fxe"},
	{0x101344139796156BULL, "462500170.ogg"},
	{0x1013828663EE41DBULL, "465245935.ogg"},
	{0x10141C7350709742ULL, "vxw_encounter2_camp_lr_0_0.vlm"},
	{0x10142957D7A59A05ULL, "6016055_m.fxe"},
	{0x1014385F0D7612EAULL, "art\\characters\\playercharacter\\qunarimale\\animation\\combat\\trooperbase\\troop_attk_v1.ani"},
	{0x1014E93B43ABDD39ULL, "hlac_lt_0_lowtown_n_0_1x.dds"},
	{0x1015044A2460FEC4ULL, "388691627.ogg"},
	{0x1016669B372D81DAULL, "6129912_m.fxe"},
	{0x1016A067B02915CEULL, "vxlt_0_lowtown_docks_a3_coc_0_0.vlm"},
	{0x1016CBD46EBF848DULL, "vxlt_darktown_2qv_0_0.vlm"},
	{0x1017256D61D1764CULL, "6108519_m.fxe"},
	{0x1017312A77A2B67AULL, "vxlt_gallowscourtyard_7hm_0_0.vlm"},
	{0x101754AF34F1B98BULL, "vxht_brothel_n_4vt_0_0.vlm"},
	{0x1017C549BABA13D4ULL, "fen210_orana6174969.cl"},
	{0x1017DE6C3A4504E0ULL, "873881610.ogg"},
	{0x10180FAA444DA6EDULL, "6053569_f.fxe"},
	{0x101847BEE321E1EDULL, "6024563_f.fxe"},
	{0x101862F4442DF2C1ULL, "gen_im_arm_glv_med_start.uti"},
	{0x1019129EE9AADC7BULL, "hlac_lt_0_lowtown_docks_a3_n_-131074_3y.dds"},
	{0x1019E28F04EDF2E5ULL, "vxw_sundermount_28f_0_0.vlm"},
	{0x101A63DADFB91D56ULL, "696004131.ogg"},
	{0x101A8592450855D7ULL, "134964688.ogg"},
	{0x101A9641F85C52B8ULL, "221381274.ogg"},
	{0x101AD2FB0F334B6AULL, "449371631.ogg"},
	{0x101AF6819E8B5D1AULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_woodburn_d.dds"},
	{0x101B12913943FC8FULL, "vxlt_gamlenshouse_n_n5_0_0.vlm"},
	{0x101B46F82C2ED164ULL, "cod_bks_first_children.utp"},
	{0x101B7D4552204317ULL, "vxlt_house_fy_0_0.vlm"},
	{0x101BFF7979003090ULL, "w_woundedcoast_228_0.mao"},
	{0x101C0379790037A3ULL, "w_woundedcoast_228_0.mmh"},
	{0x101C117979004FFDULL, "w_woundedcoast_228_0.msh"},
	{0x101C391DB78C061CULL, "vxca_caverns1_mines_cny_0_0.vlm"},
	{0x101C8B126D13A893ULL, "llac_lt_0_lowtown_docks_n_-3_8.dds"},
	{0x101CC6A51D333863ULL, "hlac_lt_0_lowtown_docks_n_-3_0x.dds"},
	{0x101D08539B62B71EULL, "350141820.ogg"},
	{0x101DF1831D2752DDULL, "art\\vfx\\combat\\vfx_m_frc_forcecrush_c.mmh"},
	{0x101E00C9D1AD3234ULL, "vxw_sundermount_2rm_0_0.vlm"},
	{0x101E021345D9E010ULL, "gen_im_trs_rdr_arc_lbw_r1.uti"},
	{0x101E09B5189DB2D6ULL, "6070240_f.fxe"},
	{0x101ECE43047B2220ULL, "vxlt_gallowstemplar_3rs_0_0.vlm"},
	{0x101EDC425D711647ULL, "405573371.ogg"},
	{0x101EDC830877EBF9ULL, "lt_0_lowtown_docks_n_110_0.msh"},
	{0x101EE6830877FCE7ULL, "lt_0_lowtown_docks_n_110_0.mmh"},
	{0x101EEA8308780334ULL, "lt_0_lowtown_docks_n_110_0.mao"},
	{0x101F2D72024CE77AULL, "vxdr_primevaltomb_81y_0_0.vlm"},
	{0x101F6A18B500EC44ULL, "6050183_f.fxe"},
	{0x101F81656363788FULL, "6072324_f.fxe"},
	{0x101FAB15F3647D74ULL, "hlac_lt_0_lowtown_n_196607_1z.dds"},
	{0x102015A8604A1C98ULL, "vxlt_0_lowtown_a2_n_gzc_0_0.vlm"},
	{0x10202E4B7106C506ULL, "426564933.ogg"},
	{0x10209EC2AB5D4381ULL, "612201241.ogg"},
	{0x1021476E862E4A02ULL, "ser221ip_note_shipment.utp"},
	{0x10216DEE34D6266AULL, "396991481.ogg"},
	{0x10217B1137170BCCULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\r_rdy.gad"},
	{0x1021E6E6B1826E96ULL, "64280875.ogg"},
	{0x102204B716D32D83ULL, "vxw_encounter2_y_0_0.vlm"},
	{0x102281D62B98BDACULL, "hlac_w_woundedcoast_6_0y.dds"},
	{0x10228E504266AFC2ULL, "vxw_encounter2_camp_gs_0_0.vlm"},
	{0x1022EAA651F417FAULL, "vxlt_0_lowtown_docks_bak_0_0.vlm"},
	{0x1022F8B038DE1FE1ULL, "6086573_m.fxe"},
	{0x102310500D3E2E83ULL, "vxca_caverns2prc_8h1_0_1.vlm"},
	{0x10237F798D8A2094ULL, "6125330_m.fxe"},
	{0x102380BD0BD1562DULL, "vxw_bonepitmining_a2_b5_0_0.vlm"},
	{0x1023A6C25EF37C00ULL, "vxht_0_hightown_a2_9zn_0_0.vlm"},
	{0x1023E7BEF0C81E10ULL, "vxdr_slavers_2hf_0_0.vlm"},
	{0x1024521A6FD1B009ULL, "vxlt_0_lowtown_e1v_0_0.vlm"},
	{0x10247302F33552DBULL, "vxlt_0_lowtown_n_etx_0_0.vlm"},
	{0x102488A0A65C237DULL, "vxlt_0_lowtown_a2_db6_0_0.vlm"},
	{0x10253DB160B59D0DULL, "vxlt_darktown_2tu_0_0.vlm"},
	{0x10254514982B14C5ULL, "vxlt_gallowscourtyard_7bx_0_0.vlm"},
	{0x10256FC487D7544DULL, "796202674.ogg"},
	{0x102583DEDE4CF9F2ULL, "438428311.ogg"},
	{0x10270E233F071519ULL, "vxlt_0_lowtown_a2_n_c_eox_0_1.vlm"},
	{0x102750AD91362984ULL, "mcr371_hostile_fenris6082213.cl"},
	{0x10278E225329B47CULL, "6087531_m.fxe"},
	{0x1027C28F5B63CCE0ULL, "vxht_mansion2_1we_0_0.vlm"},
	{0x102829EDBD400F6BULL, "follower_banter6136363.cl"},
	{0x10286B1E1298EFD0ULL, "6127685_f.fxe"},
	{0x10286C1E4EC22319ULL, "w_encounter3_204_0.msh"},
	{0x1028761E4EC23407ULL, "w_encounter3_204_0.mmh"},
	{0x10287A11371D28CAULL, "art\\characters\\creatures\\arh_abm_rev\\animation\\r_rdy.ani"},
	{0x10287A1E4EC23B54ULL, "w_encounter3_204_0.mao"},
	{0x1028D388543E5F63ULL, "6081692_f.fxe"},
	{0x1028F1412AA104A5ULL, "vxlt_0_lowtown_a3_n_c_e38_0_0.vlm"},
	{0x10293600881D06EEULL, "vxht_0_hightown_n_akd_0_0.vlm"},
	{0x1029AF6EE0895262ULL, "vxlt_0_lowtown_a2_hgb_0_0.vlm"},
	{0x1029BCE6E0554F88ULL, "dae000ip_crowd_ff5_talk17.utp"},
	{0x102A084E22724503ULL, "vxlt_hangedman_n_1xr_0_0.vlm"},
	{0x102A2F6D733F206BULL, "art\\levels\\4cinematics\\ferelden\\_textures\\prp_crown_n.dds"},
	{0x102A7C246A92D5E1ULL, "6171510_m.fxe"},
	{0x102B13DEFE98FDB6ULL, "vxht_brothel_n_6uf_0_0.vlm"},
	{0x102B15196149E8E1ULL, "zz_isa_debug6133987.cl"},
	{0x102B1844357CC1F4ULL, "6179511_m.fxe"},
	{0x102B4506055C650BULL, "dae000ip_crowd_ff4_talk4.utp"},
	{0x102BFCE9CDBA6A8EULL, "106942336.ogg"},
	{0x102CD4BF4C98F7F2ULL, "vxht_playershouse_n_1qc_0_0.vlm"},
	{0x102D0479790EACC8ULL, "w_woundedcoast_228_0.phy"},
	{0x102D04E88FBAF623ULL, "vxht_keep_70e_0_0.vlm"},
	{0x102D35D640AE3BDCULL, "vxlt_shop_cl_0_1.vlm"},
	{0x102D8D9606EC7EE3ULL, "334576850.ogg"},
	{0x102E124C12F77EBCULL, "40744779.ogg"},
	{0x102E622A664EA710ULL, "llac_ht_0_hightown_n_2_0.dds"},
	{0x102E72EBEC5F01EEULL, "llac_dr_0_deeproads_5845_11.dds"},
	{0x102EC88C2AFF00B4ULL, "754198095.ogg"},
	{0x102EE289E19D116EULL, "llac_lt_backalley_g_0_8.dds"},
	{0x102F133E4F01B633ULL, "6067674_f.fxe"},
	{0x102F2A21D3BE2D2FULL, "art\\characters\\weapons\\shortbows\\textures\\w_sbw_sb102a_0s.dds"},
	{0x102F4675E2C3DB48ULL, "hlac_ht_mansion2_818_64x.dds"},
	{0x102F9C2D89EC1F88ULL, "zz_mer_debug6039335.cl"},
	{0x10303AA75E70A550ULL, "llac_ca_cavern1_haunted_24424_26.dds"},
	{0x10304B3BCBBAAC85ULL, "mag320st_orsino.stg"},
	{0x1030654D435B692CULL, "art\\characters\\playercharacter\\humanfemale\\hf_arm_flma.mao"},
	{0x10307AAF451BD6E9ULL, "445873987.ogg"},
	{0x103092552617F184ULL, "221181258.ogg"},
	{0x103103BD7464F99AULL, "llac_w_blightlands_393213_0.dds"},
	{0x10314D5F80EE26EEULL, "vxht_keep_n_55r_0_0.vlm"},
	{0x10315757B5D72F3BULL, "6172676_m.fxe"},
	{0x10315C8343EA307AULL, "vxlt_0_lowtown_docks_bdd_0_0.vlm"},
	{0x1031DA87A7D9A1ECULL, "390229168.ogg"},
	{0x1031F299FDC76EADULL, "vxw_bonepitmining_a2_i4_0_0.vlm"},
	{0x10324BAE8E445220ULL, "gen00fl_varric6104708.cl"},
	{0x1032763202DBD39BULL, "art\\vfx\\cinematic\\vfx_chantry_int_exp_p.impa.ani"},
	{0x1032844B7C838693ULL, "vxlt_backalley_n_1u8_0_0.vlm"},
	{0x1032B27CE82ADA27ULL, "343238617.ogg"},
	{0x1032F9FD00D15AFAULL, "vxht_0_hightown_n_cb1_0_0.vlm"},
	{0x10334212B4096AEDULL, "zz_mag_debug6039788.cl"},
	{0x1033B6F18A212D45ULL, "vxlt_gallowscourtyard_7gg_0_0.vlm"},
	{0x1034B4C7C996CD56ULL, "w_bonepitmining_a2_195_0.mao"},
	{0x1034C0C7C996E1F5ULL, "w_bonepitmining_a2_195_0.mmh"},
	{0x1034C2C7C996E56BULL, "w_bonepitmining_a2_195_0.msh"},
	{0x1035934C4E1E154FULL, "km_pryce.mor"},
	{0x10362C4C30339894ULL, "mrl310_party_comments6142848.cl"},
	{0x10366D21C6A7CC19ULL, "6093253_f.fxe"},
	{0x1036E850EBD61C9BULL, "149281381.ogg"},
	{0x10371A95EB80A01AULL, "141009153.ogg"},
	{0x1037B6846693F6A9ULL, "art\\levels\\races\\ferelden\\props\\lowerclass\\prp_cot.phy"},
	{0x1037B741F192535AULL, "art\\levels\\races\\design\\animation\\beartrap.inactivetoactive.gad"},
	{0x1038247DDBAB127AULL, "vxht_keep_n_4ao_0_0.vlm"},
	{0x103851BE5EDB5079ULL, "vxlt_backalley_g_1kq_0_0.vlm"},
	{0x10387A2B14685D83ULL, "vxlt_hangedman_n_1uu_0_0.vlm"},
	{0x1038C60E21F5C0E7ULL, "6089668_m.fxe"},
	{0x1038E72CA0265652ULL, "584914368.ogg"},
	{0x1038ED638798A269ULL, "vxw_encounter2_camp_js_0_0.vlm"},
	{0x103921084CD1FE7EULL, "951668140.ogg"},
	{0x1039541E8D8FC4E2ULL, "91101451.ogg"},
	{0x103AE483B5996334ULL, "vxw_sundermount_2xs_0_0.vlm"},
	{0x103B01701FF06CFAULL, "367044552.ogg"},
	{0x103B467E26A997D0ULL, "art\\vfx\\environment\\level_fx\\fire\\vfx_brazfireout.mmh"},
	{0x103B469C3E8F1072ULL, "vxht_playershouse_n_1tl_0_0.vlm"},
	{0x103B544C602B6984ULL, "dae000cr_grand_cleric6185504.cl"},
	{0x103BB578A7E7B45CULL, "vxlt_warehouse_n_123_0_0.vlm"},
	{0x103BDFDE32AD2A12ULL, "gen00fl_fenris6126874.cl"},
	{0x103BF0D5E8B2D8B4ULL, "genip_exploding_crate_lg.utp"},
	{0x103C8E0F57E7BC36ULL, "art\\levels\\ztools\\helpers\\texels\\_textures\\128m.dds"},
	{0x103D072817BFB4ACULL, "vxht_mansion2b_78_0_0.vlm"},
	{0x103D7F05AF725117ULL, "6130137_m.fxe"},
	{0x103DB3A51B7B2227ULL, "plt_cod_prc_brotherjoffrey.plo"},
	{0x103E219779E222DDULL, "vxlt_hangedman_n_27v_0_0.vlm"},
	{0x103E34A878E977F2ULL, "isa211_isabela6070110.cl"},
	{0x103E4F068E5C4A97ULL, "349068568.ogg"},
	{0x103E68E0EA1B38D7ULL, "vrc221_varric6044050.cl"},
	{0x103EACEEA59B9C14ULL, "gen00fl_isabela6096821.cl"},
	{0x103EBA41F1987771ULL, "art\\levels\\races\\design\\animation\\beartrap.inactivetoactive.evt"},
	{0x103F7E3202E66B92ULL, "art\\vfx\\cinematic\\vfx_chantry_int_exp_p.impa.evt"},
	{0x104002E6BF9181B6ULL, "vxca_caverns2_4x5_0_1.vlm"},
	{0x1040A24FFD6EF525ULL, "6172256_m.fxe"},
	{0x1040AAED9957D5D8ULL, "6015184_f.fxe"},
	{0x104189C9BE91607EULL, "vxca_caverns1_mines_gkh_0_0.vlm"},
	{0x1041DA697FB11564ULL, "mag160_varian_ilithis6171471.cl"},
	{0x1042221DA0F9330EULL, "6097468_m.fxe"},
	{0x10423C8A4CEDDB98ULL, "gen00fl_aveline6053337.cl"},
	{0x10428C60A8CC1994ULL, "ser121_jethann6104549.cl"},
	{0x10428D620657C06DULL, "vxca_caverns1_mines_icl_0_1.vlm"},
	{0x1042A31893371E11ULL, "26299951.ogg"},
	{0x1042AE436B3FF704ULL, "abi_m_fireball.wbd"},
	{0x1042B3063B168DC6ULL, "vxlt_0_lowtown_docks_n_bfa_0_0.vlm"},
	{0x1042FB75EF8FEA4EULL, "lgt260_eau_tmp_flowergirl6143867.cl"},
	{0x1043042D30E92EAAULL, "gen00fl_anders6111319.cl"},
	{0x1043A6A6E3F19E5DULL, "458395588.ogg"},
	{0x1043AD23400D6643ULL, "6105876_m.fxe"},
	{0x1043ADEA777511D0ULL, "1026023958.ogg"},
	{0x1043C5D09AE83270ULL, "355552108.ogg"},
	{0x10442F80E00332D7ULL, "vxht_bartrand_n_4aa_0_0.vlm"},
	{0x104434F9C31E236DULL, "gen_im_fla_carver_02.uti"},
	{0x1044B3C5FBE95C01ULL, "6029062_m.fxe"},
	{0x1044E2E9222FA4C8ULL, "fx_lightning_2.mao"},
	{0x10452ED0AF39F432ULL, "243222676.ogg"},
	{0x10453F01E3699DDFULL, "vxlt_gallowstemplar_3jx_0_0.vlm"},
	{0x1045758FBC9B0D10ULL, "vxlt_0_lowtown_docks_cs6_0_0.vlm"},
	{0x1045DFB992F37023ULL, "15464333.ogg"},
	{0x10466BEB35E30976ULL, "dre000_dougal6130978.cl"},
	{0x10467A0ECE76181FULL, "vxlt_shop_kz_0_0.vlm"},
	{0x10469D3202ECBF25ULL, "art\\vfx\\cinematic\\vfx_chantry_int_exp_p.impa.gad"},
	{0x1046C39B50D168F9ULL, "vxlt_backalley_g_1pp_0_0.vlm"},
	{0x10475F40798EBAE9ULL, "vxw_encounter2_camp_er_0_0.vlm"},
	{0x10482D7EB4F29780ULL, "457197641.ogg"},
	{0x10484E34AB1AECB4ULL, "m_mag201ar_caverns.wbd"},
	{0x1048697E26B45DEFULL, "art\\vfx\\environment\\level_fx\\fire\\vfx_brazfireout.anb"},
	{0x10487B620B9D8ACEULL, "vxlt_gamlenshouse_n_ay_0_0.vlm"},
	{0x10488F17FA6B3DFCULL, "483427831.ogg"},
	{0x1048F0054BBC8492ULL, "art\\levels\\races\\dwarf\\common\\int\\deeproads\\9ff_1rubblepile.phy"},
	{0x10497E5398B21FA3ULL, "gen00fl_anders6119125.cl"},
	{0x10498CC345F06FCDULL, "art\\characters\\creatures\\arh_abm_rev\\c_rev_head.mao"},
	{0x1049CED7C350CAAEULL, "6096823_m.fxe"},
	{0x104A514DABD35441ULL, "6104712_m.fxe"},
	{0x104A7823DEF7C377ULL, "hlac_lt_0_lowtown_a2_n_c_0_3x.dds"},
	{0x104B89C795691580ULL, "str_creater_half.mao"},
	{0x104BB4ED09A97387ULL, "mag370_meredith6064603.cl"},
	{0x104BD241F1A32A40ULL, "art\\levels\\races\\design\\animation\\beartrap.inactivetoactive.ani"},
	{0x104C6E1CE8B92769ULL, "gen00fl_merrill6111165.cl"},
	{0x104C93746BD83B5DULL, "vxlt_hangedman_n_22u_0_0.vlm"},
	{0x104C98F565561967ULL, "gen_im_trs_dal_amu_r2.uti"},
	{0x104CED3DD17A59E8ULL, "6109318_m.fxe"},
	{0x104DC4269635D40BULL, "vxlt_0_lowtown_docks_c1c_0_0.vlm"},
	{0x104DC85E253E7094ULL, "zz_followers_banter6157642.cl"},
	{0x104E0473B174A08CULL, "vxca_caverns1_ia3_0_0.vlm"},
	{0x104E3F15DB3A3421ULL, "hlac_ht_keep_3733_1y.dds"},
	{0x104F77806D33C432ULL, "217548164.ogg"},
	{0x104FCC020EC6F8A8ULL, "vrc221cr_guard_fake.utc"},
	{0x104FD88EC392A0C1ULL, "hlac_ht_keep_3732_10z.dds"},
	{0x104FF00DA4BE9486ULL, "6041148_m.fxe"},
	{0x104FF10E34191567ULL, "vxlt_0_lowtown_a3_n_c_i0w_0_0.vlm"},
	{0x104FF2D9B32DFFEAULL, "336466569.ogg"},
	{0x10500BC19EE0E978ULL, "6028806_m.fxe"},
	{0x105016383D1B85E5ULL, "hlac_ht_mansion2b_818_5x.dds"},
	{0x105042DF78E22D8FULL, "vxw_bonepitmining_a2_1xf_0_0.vlm"},
	{0x10504720E559E141ULL, "hlac_lt_0_lowtown_docks_-4_7y.dds"},
	{0x1050B6913F34C6C2ULL, "952998857.ogg"},
	{0x105124E32D0CA646ULL, "vxlt_0_lowtown_docks_n_bcb_0_0.vlm"},
	{0x105133DBACFA0FADULL, "150529302.ogg"},
	{0x10515ABBC08E6BD7ULL, "isa311_castillon6092116.cl"},
	{0x10522E9C9566F901ULL, "gen00fl_anders6125308.cl"},
	{0x1052EBCAFF812817ULL, "622740422.ogg"},
	{0x1052F7C41B4FD75AULL, "w_woundedcoast_160_0.msh"},
	{0x105305C41B4FEE93ULL, "w_woundedcoast_160_0.mao"},
	{0x105309C41B4FF5A0ULL, "w_woundedcoast_160_0.mmh"},
	{0x10536DEF800F88AFULL, "art\\vfx\\creatures\\fxc_shade_c.dura.gad"},
	{0x10537C531714B3FBULL, "6175804_m.fxe"},
	{0x105399E4EA123244ULL, "genip_trap_door.utp"},
	{0x1053EA99B080C599ULL, "vxht_0_hightown_n_c1c_0_0.vlm"},
	{0x1053ED6F30978634ULL, "mcr310_argument6075802.cl"},
	{0x1054511B077F3DB2ULL, "808917618.ogg"},
	{0x10549158CEB4D24BULL, "store_and211cr_lirene.utm"},
	{0x10555DE4F8548E83ULL, "vxlt_hangedman_n_1ow_0_0.vlm"},
	{0x10558AA1468EF1A8ULL, "w_bonepitmining_a2_176_0.mmh"},
	{0x105596A1468F064BULL, "w_bonepitmining_a2_176_0.mao"},
	{0x1055994960B73A3DULL, "vxht_0_hightown_a2_ccb_0_0.vlm"},
	{0x1055A8A1468F24D2ULL, "w_bonepitmining_a2_176_0.msh"},
	{0x1055B3EF97B3C7E1ULL, "vxlt_gallowscourtyard_7f8_0_0.vlm"},
	{0x1055DFF98DC9438CULL, "6108887_m.fxe"},
	{0x1056202067DE1EDDULL, "art\\levels\\races\\primeval\\props\\_texture\\prp_bigtreasure_dp.dds"},
	{0x1056A3F0992B8620ULL, "vxw_encounter2_b6_0_0.vlm"},
	{0x1056C1CBDCAE85B1ULL, "6170504_m.fxe"},
	{0x1056F8857B60F9B6ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\abilities\\tw_tal_crtstrk.ani"},
	{0x10572B5ECE995CADULL, "6170514_f.fxe"},
	{0x105789BA4C739806ULL, "6162531_m.fxe"},
	{0x1057F1EE5BAE0640ULL, "243928189.ogg"},
	{0x105A0E488B2E9EE1ULL, "vxca_caverns2prc_9ne_0_0.vlm"},
	{0x105A0F7C00B09BABULL, "vxca_caverns1_mines_jfo_0_0.vlm"},
	{0x105A47EC0B99C9B3ULL, "dae000_gen_turn_in_46198681.cl"},
	{0x105A6BA591FE34B7ULL, "104386566.ogg"},
	{0x105A82EF8015CB5CULL, "art\\vfx\\creatures\\fxc_shade_c.dura.evt"},
	{0x105A8775D9642AEFULL, "qun231tr_varnell_amb.ncs"},
	{0x105AA46437D427C2ULL, "dae000_gen_turn_in_56178423.cl"},
	{0x105AB6477960C121ULL, "37218068.ogg"},
	{0x105AF5F9E7C50D5AULL, "125269205.ogg"},
	{0x105B793375D3CA92ULL, "vxht_chantry_23l_0_0.vlm"},
	{0x105B8089D8E23A4DULL, "995277998.ogg"},
	{0x105B8CF1C6CB4BFCULL, "2522359.ogg"},
	{0x105BB79901A53C13ULL, "mag121_viveka6082603.cl"},
	{0x105C5427485EBFC5ULL, "vxlt_0_lowtown_docks_bdo_0_0.vlm"},
	{0x105C953E04CB49FCULL, "6101493_m.fxe"},
	{0x105D24700AD6446BULL, "6149300_m.fxe"},
	{0x105D6C67CE3D4387ULL, "hlac_ht_mansion2_818_44y.dds"},
	{0x105E1F884F61B581ULL, "vxlt_warehouse_n_12b_0_0.vlm"},
	{0x105E9CC752D0CE90ULL, "art\\levels\\ztools\\helpers\\character\\prp_dwarfrun_2.mao"},
	{0x105F1606D24C3490ULL, "vfx_vfx_p10.vsh.fxo.dx11.assembly"},
	{0x105F4BC37835F038ULL, "6103583_m.fxe"},
	{0x105F57FDF6F62BDCULL, "hlac_lt_0_lowtown_a2_n_c_65536_1x.dds"},
	{0x105FABA3726110B3ULL, "art\\characters\\weapons\\longswords\\textures\\w_lsw_008a_0d.dds"},
	{0x105FF269B0591660ULL, "fake_im_arm_hlm_hvy_champ.uti"},
	{0x10602EEE38E4E41DULL, "zz_lgt_3_debug6140424.cl"},
	{0x1060482B1657E641ULL, "353764939.ogg"},
	{0x10608F271310C303ULL, "10781822.ogg"},
	{0x1060AFB50FB9AF4AULL, "6057078_f.fxe"},
	{0x1060F115BAC6B7E7ULL, "hlac_lt_darktown_-1_11y.dds"},
	{0x10612F0B355DF3CDULL, "647651750.ogg"},
	{0x1061754E5B3D47FFULL, "art\\characters\\playercharacter\\humanfemale\\hf_boo_lgte.mmh"},
	{0x10617B4E5B3D51A1ULL, "art\\characters\\playercharacter\\humanfemale\\hf_boo_lgte.msh"},
	{0x106197AC8D110E54ULL, "hlac_lt_0_lowtown_docks_a3_n_-65538_3y.dds"},
	{0x106296991484760EULL, "follower_banter6165409.cl"},
	{0x10629E0A9C0A555BULL, "lt_gallowscourtyard_44_0.rml"},
	{0x1062DA5E9359492FULL, "256050399.ogg"},
	{0x10631C21E01FA1C5ULL, "vxht_keep_n_4ad_0_0.vlm"},
	{0x10635ACD6818904DULL, "genip_pickaxe.utp"},
	{0x10635DC8B262491FULL, "vxlt_shop_ad_0_0.vlm"},
	{0x1063641815E914A1ULL, "art\\cinematics\\_crit path\\mage crisis\\mcr370cs_attack_begins\\max files\\extraanims_mage1_.ani"},
	{0x106377641F6DC8E0ULL, "art\\toolset\\characters\\playercharacter\\humanmale\\mt_art_hm00.phy"},
	{0x10639F8466B90ACEULL, "art\\levels\\races\\ferelden\\props\\lowerclass\\prp_cot.mmh"},
	{0x1063A75534BD99F9ULL, "vxlt_backalley_g_1zv_0_0.vlm"},
	{0x1063AB8466B91F65ULL, "art\\levels\\races\\ferelden\\props\\lowerclass\\prp_cot.mao"},
	{0x1063B98466B936BCULL, "art\\levels\\races\\ferelden\\props\\lowerclass\\prp_cot.msh"},
	{0x1063F0C01D4B0B73ULL, "mcr371_db_harvester_warr.cut"},
	{0x10641499DB495591ULL, "llac_dr_0_deeproads_5849_3.dds"},
	{0x1064604E5583EB43ULL, "6111749_m.fxe"},
	{0x106493EC563259E1ULL, "123525061.ogg"},
	{0x106500857B6D44A7ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\abilities\\tw_tal_crtstrk.evt"},
	{0x106515CD8B219EA0ULL, "vxw_encounter2_m7_0_0.vlm"},
	{0x106523CFDCC4FF49ULL, "vxlt_warehouse_n_e0_0_0.vlm"},
	{0x10653FE75609B71EULL, "vxw_encounter2_qt_0_1.vlm"},
	{0x106571728F41AC39ULL, "combat_smcf_range.gda"},
	{0x1066477803ABC7EEULL, "821351248.ogg"},
	{0x106662ED30F81AC7ULL, "hlac_lt_0_lowtown_a2_n_c_-65536_0y.dds"},
	{0x106684736164046AULL, "llac_ht_0_hightown_a3_-65536_4.dds"},
	{0x1067792747258FD0ULL, "6015679_m.fxe"},
	{0x1067B30DA80D7FD9ULL, "498695169.ogg"},
	{0x1067E3CCA9C5A7F8ULL, "285269219.ogg"},
	{0x1067FD71F598F247ULL, "80568563.ogg"},
	{0x10681F8563B8D1D2ULL, "hlac_lt_backalley_g_1_1z.dds"},
	{0x10684096E2B748BBULL, "qun221_viscount6070481.cl"},
	{0x10685229E1A98FFCULL, "vxht_keep_n_62b_0_0.vlm"},
	{0x10685C81BCF3F198ULL, "6190969_m.fxe"},
	{0x106877554B04EAD5ULL, "380178987.ogg"},
	{0x10688AEF80221675ULL, "art\\vfx\\creatures\\fxc_shade_c.dura.ani"},
	{0x1068AE776D2D599FULL, "6084205_m.fxe"},
	{0x10693AB6EACC3896ULL, "577221824.ogg"},
	{0x106988D53EE2957AULL, "vxw_bonepitdestroyed_a3_1bs_0_0.vlm"},
	{0x10699C31FD13E837ULL, "679104789.ogg"},
	{0x1069C847866F0BFDULL, "309037319.ogg"},
	{0x106A6F1815EF4643ULL, "art\\cinematics\\_crit path\\mage crisis\\mcr370cs_attack_begins\\max files\\extraanims_mage1_.gad"},
	{0x106A9879A03DA025ULL, "vxlt_0_lowtown_a2_bz0_0_0.vlm"},
	{0x106AE42C269DBF86ULL, "6107058_f.fxe"},
	{0x106B21857B71E8F0ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\trooper\\abilities\\tw_tal_crtstrk.gad"},
	{0x106B72E3163B3722ULL, "1000599855.ogg"},
	{0x106BCD338B726B32ULL, "6084693_m.fxe"},
	{0x106C0FBF0D9973BFULL, "6165122_m.fxe"},
	{0x106C2305FBCA6114ULL, "vxdr_0_deeproads_cvr_0_1.vlm"},
	{0x106CBAB00BF138BCULL, "6097835_m.fxe"},
	{0x106CCE22CF620DE7ULL, "swashbklr_expertise.dds"},
	{0x106CDAAB0304340AULL, "hlac_lt_0_lowtown_a2_n_c_65535_8x.dds"},
	{0x106CE5B96E768D5CULL, "art\\characters\\creatures\\hurlock\\animation\\sword_shield\\hurlock_sw_die_v2.ani"},
	{0x106CED2C14127E90ULL, "gen00fl_merrill6162018.cl"},
	{0x106D03EC33E0CF77ULL, "vxdr_primevalentrance_146_0_0.vlm"},
	{0x106D38ECC600D721ULL, "qun160_vashothbandits6069277.cl"},
	{0x106D588D3DCA124DULL, "274586344.ogg"},
	{0x106D7DD2A45686E3ULL, "vxdr_primevaltomb_4h7_0_0.vlm"},
	{0x106DC1C45ED3F4A1ULL, "gen00fl_fenris6126390.cl"},
	{0x106DD29AFC0402A6ULL, "darktown_male_sickly_sta6175325.cl"},
	{0x106DF990311F3EBCULL, "hlac_lt_0_lowtown_a2_0_1z.dds"},
	{0x106E089D10F8D746ULL, "vxlt_0_lowtown_docks_n_bud_0_0.vlm"},
	{0x106E2072E737BA11ULL, "813578535.ogg"},
	{0x106E77CD5DD77540ULL, "vxw_sundermount_2m4_0_0.vlm"},
	{0x106EFC499DCF9F26ULL, "vxlt_shop_f8_0_0.vlm"},
	{0x106F3A6E867055DFULL, "ser221ip_note_shipment.ncs"},
	{0x106F3BF9E9BA8CEDULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\9ff_sidewlkendb.mmh"},
	{0x106F3FF9E9BA933EULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\9ff_sidewlkendb.mao"},
	{0x106F4DF9E9BAAB73ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\9ff_sidewlkendb.msh"},
	{0x10716F5141EB816EULL, "vxht_0_hightown_n_apy_0_0.vlm"},
	{0x1071CFA5A458619FULL, "vxlt_shop_dg_0_0.vlm"},
	{0x1071DDA1A1468138ULL, "art\\levels\\races\\ferelden\\props\\chantry\\prp_flagchantryb.phy"},
	{0x10720730AD7205FFULL, "w_bonepitdestroyed_a3_0_0.rml"},
	{0x107288C8B5115865ULL, "gen00fl_anders6125216.cl"},
	{0x1072F2B96E7B0E9EULL, "art\\characters\\creatures\\hurlock\\animation\\sword_shield\\hurlock_sw_die_v2.gad"},
	{0x1073501D582364EFULL, "292896713.ogg"},
	{0x107369AE783C60F1ULL, "ts_generic_acc_props.gda"},
	{0x1073B004649DB95FULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\9ff_stala2a.phy"},
	{0x1073DD7A37721C50ULL, "w_bonepit_23_0.mao"},
	{0x1073E17A37722363ULL, "w_bonepit_23_0.mmh"},
	{0x1073EF7A37723ABDULL, "w_bonepit_23_0.msh"},
	{0x1073F9F4655BA3DBULL, "vxw_bonepitmining_a2_p9_0_0.vlm"},
	{0x10743C34AB40094EULL, "m_mag201ar_caverns.bnk"},
	{0x10749880724E7E31ULL, "vxlt_0_lowtown_a2_n_c_fe8_0_0.vlm"},
	{0x1074A8FD259D442BULL, "909335204.ogg"},
	{0x107581991EA4CD4FULL, "follower_banter6143095.cl"},
	{0x10766EC7F156ADB2ULL, "6110784_f.fxe"},
	{0x1076BE85093E2D55ULL, "vxca_caverns1_j25_0_1.vlm"},
	{0x1076E0478DEF4AC5ULL, "707820498.ogg"},
	{0x107704B3A4C8141FULL, "781239137.ogg"},
	{0x1077A51503398113ULL, "dae100_alistair_drunk6135129.cl"},
	{0x1077B0FBA193BD69ULL, "dre000_sibling_has_blight6036567.cl"},
	{0x1077FAB230D8ADFAULL, "vxw_bonepitdestroyed_a3_1gl_0_0.vlm"},
	{0x107813AE64D246AAULL, "6127662_m.fxe"},
	{0x10782F4EA9DA3BCAULL, "87738496.ogg"},
	{0x10784DB87490617FULL, "vxlt_0_lowtown_a2_n_c_eoo_0_0.vlm"},
	{0x10787475D97D0796ULL, "qun231tr_varnell_amb.utt"},
	{0x1078891DA4092F08ULL, "218777929.ogg"},
	{0x1078D1F652B7D4CCULL, "pro000_cut2d_thefirstyear6015285.cl"},
	{0x107905B1E8F20DC0ULL, "dae000_hate_enemies6167600.cl"},
	{0x10792E6C6AEC634DULL, "hlac_lt_0_lowtown_n_-196608_1y.dds"},
	{0x107965EC4C7987AFULL, "645210087.ogg"},
	{0x10799CA9CBB1C102ULL, "109165710.ogg"},
	{0x1079A0D0D48CC1CEULL, "vxht_playershouse_n_1lh_0_0.vlm"},
	{0x1079CDB96E80EEBDULL, "art\\characters\\creatures\\hurlock\\animation\\sword_shield\\hurlock_sw_die_v2.evt"},
	{0x107A1C116E4FC542ULL, "art\\toolset\\characters\\playercharacter\\elfmale\\mt_art_em00.msh"},
	{0x107A1E116E4FC8F8ULL, "art\\toolset\\characters\\playercharacter\\elfmale\\mt_art_em00.mmh"},
	{0x107A251D346E4849ULL, "llac_ca_caverns1_24421_5.dds"},
	{0x107A2E52370AA1F0ULL, "ca_caverns1_24425.rml"},
	{0x107A6849B5FE3C72ULL, "vxlt_0_lowtown_ekf_0_1.vlm"},
	{0x107A7369F0729D40ULL, "343792752.ogg"},
	{0x107AEF7EE9F06328ULL, "zz_area_signage1.utp"},
	{0x107B19FCA6ED192EULL, "6065230_m.fxe"},
	{0x107B8C03B48DAD1DULL, "ser270_templar6175481.cl"},
	{0x107BB25AB5E8FF3FULL, "ico_mace_2.dds"},
	{0x107C77DD69E9B7AFULL, "gen00fl_varric6152698.cl"},
	{0x107C7A7A02EEEFC6ULL, "vxlt_0_lowtown_docks_n_bre_0_0.vlm"},
	{0x107D0383D0BE25EEULL, "vxdr_primevalentrance_2dx_0_0.vlm"},
	{0x107D3AA9636592B8ULL, "gen_im_trs_krk_hvy_glv_r2.uti"},
	{0x107D68763D0C9BACULL, "vxht_0_hightown_a3_avs_0_0.vlm"},
	{0x107DD8721A7B9B34ULL, "6060203_f.fxe"},
	{0x107DD9D80F0E69D9ULL, "character_character_p257.psh.fxo.dx11"},
	{0x107DF16616644B05ULL, "gen00fl_fenris6103359.cl"},
	{0x107E2AEC35391394ULL, "6108609_m.fxe"},
	{0x107E43646EFCE44CULL, "44168757.ogg"},
	{0x107E512555F15A61ULL, "vxlt_0_lowtown_docks_be0_0_0.vlm"},
	{0x107F868E85EC2AE0ULL, "hlac_lt_0_lowtown_a2_n_c_-196609_2z.dds"},
	{0x107FE12E33E199EEULL, "vxht_0_hightown_n_auz_0_0.vlm"},
	{0x107FEA1A4072FBB2ULL, "hlac_ht_0_hightown_a3_65535_2x.dds"},
	{0x108010069FA204BCULL, "6032011_f.fxe"},
	{0x10801255A2183DAAULL, "cod_hst_kirk_arch_1a_5.utp"},
	{0x1080330EF9A926A8ULL, "6119275_f.fxe"},
	{0x108034F9E9C8F3AAULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\9ff_sidewlkendb.phy"},
	{0x10803A2AC807EE00ULL, "hla_ht_0_hightown_n_-131074_0x.dds"},
	{0x10805B06A1B8EF61ULL, "6031342_m.fxe"},
	{0x10808BBA9F7090C7ULL, "darktown_male_elf_sta6174545.cl"},
	{0x1080AC7A31A91F91ULL, "mrl150_merrill6150734.cl"},
	{0x10810D6BCAA3F398ULL, "6045919_m.fxe"},
	{0x1081DA9DA2D544B8ULL, "dae150_pol.cnv"},
	{0x1081F34E9420FAD5ULL, "446293522.ogg"},
	{0x1081F9876F0DCFA0ULL, "vxw_encounter2_s5_0_0.vlm"},
	{0x10820789C0B13049ULL, "vxlt_warehouse_n_o6_0_0.vlm"},
	{0x10821EAB3A8A2D08ULL, "mag291_vincento6027348.cl"},
	{0x1082CA8ADA66435EULL, "art\\characters\\creatures\\corpse\\animation\\combat\\meleeattackv4.ani"},
	{0x1082D4B83AAD1757ULL, "723864490.ogg"},
	{0x1082EF2A296AF94BULL, "6169423_m.fxe"},
	{0x108325A838756C26ULL, "llac_lt_0_lowtown_a2_n_c_-262144_0.dds"},
	{0x1084230929C7AA93ULL, "gen00fl_merrill6126316.cl"},
	{0x108425116E582883ULL, "art\\toolset\\characters\\playercharacter\\elfmale\\mt_art_em00.phy"},
	{0x10844A7CA00792FAULL, "vxdr_primevaltomb_8np_0_0.vlm"},
	{0x10847BF7D58CFD3CULL, "rdr111dg_martin6014874.cl"},
	{0x10847CA242ADCD16ULL, "hla_ht_0_hightown_a2_131073_0y.dds"},
	{0x1084B10464AC2E64ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\9ff_stala2a.mmh"},
	{0x1084B50464AC34B7ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\9ff_stala2a.mao"},
	{0x1084BC9DA2D747C0ULL, "dae150_pol.bnk"},
	{0x1084C0B04A0022A1ULL, "vxw_sundermount_2o3_0_0.vlm"},
	{0x1084C70464AC5306ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\9ff_stala2a.msh"},
	{0x108503C87A067D96ULL, "vxlt_hangedman_1vg_0_1.vlm"},
	{0x10854E403DDDD4E3ULL, "287886985.ogg"},
	{0x10855EF0C242F29FULL, "vxlt_gallowscourtyard_73y_0_0.vlm"},
	{0x10857E2643129884ULL, "llac_lt_hangedman_n_-1_11.dds"},
	{0x1085A539F8497DF4ULL, "6121154_m.fxe"},
	{0x1085C34F74DBB006ULL, "follower_banter6143113.cl"},
	{0x1085E16577D9F287ULL, "572985416.ogg"},
	{0x1085FF3FB307690BULL, "279238501.ogg"},
	{0x10863F17BBF7D6E3ULL, "lgt100cr_gang_3_lt.utc"},
	{0x10866A22B4DABDFAULL, "gen00fl_isabela6075235.cl"},
	{0x10867DA437D34F0EULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\weapon_draw_ready_exit.ani"},
	{0x1086B2B0721A9B8FULL, "lgt160_ntb_elf_conflict6172100.cl"},
	{0x108796AD0348F363ULL, "940104672.ogg"},
	{0x1087CF3720193ADDULL, "1006468383.ogg"},
	{0x1087D9DB17460648ULL, "vxlt_0_lowtown_docks_be5_0_0.vlm"},
	{0x10892CB756DC9708ULL, "w_bonepitdestroyed_a3_9_0_ll.mmh"},
	{0x10897038A54DD4AFULL, "art\\characters\\weapons\\mauls\\textures\\w_mal_ml101a_0s.dds"},
	{0x10898798059DEAFEULL, "hlac_lt_0_lowtown_a3_n_c_-65536_8y.dds"},
	{0x108994641F8EC975ULL, "art\\toolset\\characters\\playercharacter\\humanmale\\mt_art_hm00.msh"},
	{0x108996641F8ECCEBULL, "art\\toolset\\characters\\playercharacter\\humanmale\\mt_art_hm00.mmh"},
	{0x1089AABE5AADCB16ULL, "hlac_w_woundedcoast_196608_0y.dds"},
	{0x108A3CD560F4DDCCULL, "gen00fl_sebastian6178335.cl"},
	{0x108A9B9ADAA8A40CULL, "260990103.ogg"},
	{0x108B042FE4CD73B4ULL, "mrl000_marethari6129355.cl"},
	{0x108B0AB572BD1808ULL, "hlac_lt_0_lowtown_docks_a3_-131074_6x.dds"},
	{0x108B195FFC9552D5ULL, "333330169.ogg"},
	{0x108B35DBCAD87387ULL, "weapon_weapon_p2.psh.fxo.dx11"},
	{0x108B3CDA938CC227ULL, "vxlt_warehouse_5w_0_0.vlm"},
	{0x108B41AA601F88A1ULL, "6148419_m.fxe"},
	{0x108BDA532F02B42CULL, "vxht_0_hightown_a3_asp_0_0.vlm"},
	{0x108BE17A3786B480ULL, "w_bonepit_23_0.rml"},
	{0x108BF14340960C34ULL, "gen_im_trs_chn_light_doc_01.uti"},
	{0x108C56B1C78DE829ULL, "69958103.ogg"},
	{0x108D78529D38185FULL, "vxlt_gallowstemplar_3sg_0_0.vlm"},
	{0x108DB0F6DB68C064ULL, "vxht_0_hightown_n_b44_0_0.vlm"},
	{0x108E0BC0BA3A3DBAULL, "gen00fl_sebastian6191246.cl"},
	{0x108E0F9FBF8CE296ULL, "vxca_caverns1_mines_dqa_0_0.vlm"},
	{0x108E3E32B0B1F1B6ULL, "6004411_m.fxe"},
	{0x108E8DCD79A04F62ULL, "vxca_caverns2_839_0_0.vlm"},
	{0x108EDDA318E830D3ULL, "fen310_fenris6118192.cl"},
	{0x108F354D4FBD7726ULL, "290392438.ogg"},
	{0x108F4D392916C314ULL, "gen_im_trs_slv_lgt_hlm_r1.uti"},
	{0x108FE28ADA70F62FULL, "art\\characters\\creatures\\corpse\\animation\\combat\\meleeattackv4.evt"},
	{0x1090755C09C4E027ULL, "6174044_f.fxe"},
	{0x10907966B2A748C9ULL, "vxlt_warehouse_n_r5_0_0.vlm"},
	{0x109093A69530D253ULL, "follower_banter6173016.cl"},
	{0x109121311F9FE7C9ULL, "13072554.ogg"},
	{0x10922DB7138783EFULL, "6174818_m.fxe"},
	{0x1092BC5991FDAB7AULL, "vxdr_primevaltomb_8iq_0_0.vlm"},
	{0x1092C4D36192522DULL, "219334717.ogg"},
	{0x1093328D3BF63B21ULL, "vxw_sundermount_2p2_0_0.vlm"},
	{0x109375A437DDCC5FULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\weapon_draw_ready_exit.evt"},
	{0x1093D0CDB4390B1FULL, "vxlt_gallowscourtyard_74x_0_0.vlm"},
	{0x10941252EB350B9FULL, "6045372_m.fxe"},
	{0x10943330ABFA8018ULL, "750152552.ogg"},
	{0x10943FE78E745DC3ULL, "vxht_0_hightown_a2_7vs_0_0.vlm"},
	{0x1094533F7991B2A3ULL, "vxw_encounter1_eq_0_0.vlm"},
	{0x1095222AAD1607F2ULL, "hlac_lt_0_lowtown_a2_n_-2_1z.dds"},
	{0x1095F49A39B93954ULL, "hlac_lt_0_lowtown_a2_1_11z.dds"},
	{0x109601E70AC57414ULL, "232900200.ogg"},
	{0x109611A95D06AC68ULL, "llac_ca_cave_2193_0.dds"},
	{0x109618571AAD95A5ULL, "946422068.ogg"},
	{0x1096E38ADA7716D8ULL, "art\\characters\\creatures\\corpse\\animation\\combat\\meleeattackv4.gad"},
	{0x1097382C76D1F2E0ULL, "6019095_m.fxe"},
	{0x109771D3373D6D9CULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\rogue_below_belt_action_idle.ani"},
	{0x109776B1AD8A7DC9ULL, "464670217.ogg"},
	{0x1097FA28CFF13C2AULL, "91504283.ogg"},
	{0x109804DFCB95F335ULL, "136837153.ogg"},
	{0x1098F82BF593BAF6ULL, "w_blightlands_162_0.msh"},
	{0x1099022BF593CBF4ULL, "w_blightlands_162_0.mmh"},
	{0x1099062BF593D207ULL, "w_blightlands_162_0.mao"},
	{0x109A56A437E3B648ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\weapon_draw_ready_exit.gad"},
	{0x109AC67C51624231ULL, "176577738.ogg"},
	{0x109AD12656C9C919ULL, "hlac_lt_blackemporium_1055_0y.dds"},
	{0x109B5A84734B6A3EULL, "vxlt_0_lowtown_a2_n_gv9_0_0.vlm"},
	{0x109D2073A2D9C561ULL, "6206863_m.fxe"},
	{0x109D27A9B47D69C2ULL, "vxlt_0_lowtown_a3_n_c_gv6_0_0.vlm"},
	{0x109D8470E2EE8C1DULL, "981780945.ogg"},
	{0x109DBD5B5929BD67ULL, "vxlt_0_lowtown_docks_a3_d3q_0_0.vlm"},
	{0x109DEC4E4D4BDB60ULL, "285572359.ogg"},
	{0x109E7ED33743A2DEULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\rogue_below_belt_action_idle.gad"},
	{0x109EEB43A49D6149ULL, "vxlt_warehouse_n_y4_0_0.vlm"},
	{0x109F5C11F3660E62ULL, "vxca_caverns2prc_9pu_0_1.vlm"},
	{0x109F80539885BE00ULL, "gen00fl_isabela6079282.cl"},
	{0x10A0476D8509E4A7ULL, "985261941.ogg"},
	{0x10A0720174DC228FULL, "893574924.ogg"},
	{0x10A1586AED6244E6ULL, "6070395_m.fxe"},
	{0x10A1A46A2DEC53A1ULL, "vxw_sundermount_2u1_0_0.vlm"},
	{0x10A1B9461AC92349ULL, "hlac_w_woundedcoast_-196606_0y.dds"},
	{0x10A21E232882C38DULL, "ts_circleofmagi_02.gda"},
	{0x10A27A4139DCA2A7ULL, "hlac_lt_gallowstemplar_f_2383_4z.dds"},
	{0x10A2A10A9C403E27ULL, "lt_gallowscourtyard_44_0.anb"},
	{0x10A2A83DEC930F0DULL, "vxdr_0_deeproads_6qw_0_1.vlm"},
	{0x10A2BB8D1C3069B3ULL, "qun200_party_comment6095260.cl"},
	{0x10A2BFB43BD78760ULL, "art\\characters\\playercharacter\\dwarfmale\\dm_brd_be01a.phy"},
	{0x10A2C51C6B87CB23ULL, "vxw_encounter1_jp_0_0.vlm"},
	{0x10A2F256354D3D45ULL, "vxw_bonepitdestroyed_a3_1gg_0_0.vlm"},
	{0x10A3676D6DB706A3ULL, "vxdr_primevaltomb_9dv_0_0.vlm"},
	{0x10A3F53A438A1E6EULL, "gen00fl_aveline6114834.cl"},
	{0x10A42A6579091446ULL, "6179269_m.fxe"},
	{0x10A42C798F3DF715ULL, "vxw_sundermount_2y9_0_0.vlm"},
	{0x10A45E138C5788E7ULL, "art\\vfx\\combat\\vfx_gib_hlk_p.justfeet.ces.gad"},
	{0x10A57E3B0DCAC005ULL, "190789917.ogg"},
	{0x10A5845ECA4DA58EULL, "mag210_arianni6030488.cl"},
	{0x10A597FBB6AE65E2ULL, "hlac_ht_bartrand_n_3373_3x.dds"},
	{0x10A5A72D06B7CA5FULL, "291578766.ogg"},
	{0x10A5DC743C549CF3ULL, "zz_dragon_debug.ncs"},
	{0x10A6220788A1405AULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\2arm_questioning.gad"},
	{0x10A6650B2D55181CULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_vnicerchair_sa.dds"},
	{0x10A69797A2DD8AC4ULL, "mag100_cullen6094605.cl"},
	{0x10A6CEAE09EEC987ULL, "vxw_bonepitmining_a2_1nd_0_0.vlm"},
	{0x10A7529533B5A085ULL, "6069924_f.fxe"},
	{0x10A7AB0EF4545138ULL, "742461215.ogg"},
	{0x10A7AC560EEC7D36ULL, "69155451.ogg"},
	{0x10A7B61C1DD003BAULL, "hlac_lt_gallowscourtyard_n_65535_0y.dds"},
	{0x10A8885EB72965CFULL, "ci_smoke05.dds"},
	{0x10A8AD7743EA45BBULL, "gen_im_trs_rdr_junk1_01.uti"},
	{0x10A8DE20E1074CABULL, "vxlt_gallowstemplar_n_4o3_0_0.vlm"},
	{0x10A95C993961DEB9ULL, "llac_ca_caverns1_24425_10.dds"},
	{0x10AA5530788AE7E5ULL, "vxw_sundermount_2nl_0_0.vlm"},
	{0x10AB30172A17212AULL, "follower_banter6141450.cl"},
	{0x10AB37BEB3BDB515ULL, "zz_dre_debug6056958.cl"},
	{0x10ABB1D701CE9409ULL, "897966864.ogg"},
	{0x10ABCD41EB33CB29ULL, "vxca_caverns1_ckr_0_0.vlm"},
	{0x10AC2F384B1FD5E7ULL, "vxlt_0_lowtown_docks_a3_d4p_0_0.vlm"},
	{0x10AC5ADE4229E281ULL, "art\\vfx\\combat\\vfx_w_off_stun_chc_a_c.impa.ani"},
	{0x10AC859C7A3D2343ULL, "art\\levels\\races\\natural\\kirkcoastal\\props\\prp_3craggyrock.mao"},
	{0x10AC899C7A3D2990ULL, "art\\levels\\races\\natural\\kirkcoastal\\props\\prp_3craggyrock.mmh"},
	{0x10AC979C7A3D41AAULL, "art\\levels\\races\\natural\\kirkcoastal\\props\\prp_3craggyrock.msh"},
	{0x10ACCE5BBB9C1641ULL, "mcr310_sibling6103598.cl"},
	{0x10ACD1F5EF334101ULL, "vxht_0_hightown_a2_9zc_0_0.vlm"},
	{0x10AD250788A76471ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\2arm_questioning.evt"},
	{0x10ADEFF0E96D12F6ULL, "vrc321tr_spooky_scenes_3.ncs"},
	{0x10AE240DA5FAD7F8ULL, "476874127.ogg"},
	{0x10AEA08A34A46CC5ULL, "hlac_ht_keep_n_4519_8x.dds"},
	{0x10AF4697584A35F5ULL, "zz_qcr_debug6056621.cl"},
	{0x10AF7E4A73EC2DDCULL, "vxlt_backalley_n_1kq_0_0.vlm"},
	{0x10AF9A2D7A73091DULL, "one220_yevhen6116245.cl"},
	{0x10AFB336C85D567BULL, "dae111st_dog_family.stg"},
	{0x10AFB98A0EF958CAULL, "vxca_caverns2_7ud_0_0.vlm"},
	{0x10B010D94E1AE080ULL, "dae000ip_crowd_em1_talk8.utp"},
	{0x10B0AB8D7D530BD7ULL, "739366632.ogg"},
	{0x10B11C3070C85721ULL, "mag101_aombanter6052759.cl"},
	{0x10B16433274355C5ULL, "vxw_bonepitdestroyed_a3_1hf_0_0.vlm"},
	{0x10B1C516E6AB7EB5ULL, "art\\characters\\playercharacter\\elffemale\\glv_hvyn.phy"},
	{0x10B1FDB602EC17E2ULL, "vxdr_primevaltomb_3hm_0_0.vlm"},
	{0x10B22989547CF472ULL, "art\\vfx\\combat\\vfx_m_sta_ice_range_f_p.impa.gad"},
	{0x10B265DE422E6123ULL, "art\\vfx\\combat\\vfx_w_off_stun_chc_a_c.impa.gad"},
	{0x10B32A9AB4DA4CF0ULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\3ww_ornate.mao"},
	{0x10B32E9AB4DA5303ULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\3ww_ornate.mmh"},
	{0x10B33C9AB4DA6B5DULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\3ww_ornate.msh"},
	{0x10B4247D2B92FDC5ULL, "6172421_m.fxe"},
	{0x10B44EDBB0985DA4ULL, "334587398.ogg"},
	{0x10B468B552D7216DULL, "follower_banter6149187.cl"},
	{0x10B48E2243966455ULL, "6122449_f.fxe"},
	{0x10B50B87A5775228ULL, "gen_im_cutscene_dagger.uti"},
	{0x10B537F1784ED70CULL, "6141497_m.fxe"},
	{0x10B5408AFBE4E207ULL, "vxw_bonepitmining_a2_1kg_0_0.vlm"},
	{0x10B5432010D1D751ULL, "lgt160_ntb_elf_conflict6163346.cl"},
	{0x10B580C06F6473C0ULL, "ser221_gascard6134540.cl"},
	{0x10B5A30129AF4870ULL, "6191294_m.fxe"},
	{0x10B5A9992EDEB557ULL, "dae000cr_grand_cleric6185387.cl"},
	{0x10B5B928FFB74F44ULL, "vxht_0_hightown_n_c3x_0_0.vlm"},
	{0x10B5E2B375386C8EULL, "hlac_dr_primevaltomb_9621_1y.dds"},
	{0x10B666B73239A977ULL, "zz_vault_debug6214944.cl"},
	{0x10B7E787769D26A4ULL, "542504273.ogg"},
	{0x10B84796481AE141ULL, "441464200.ogg"},
	{0x10B84B138C68112DULL, "art\\vfx\\combat\\vfx_gib_hlk_p.justfeet.ces.ani"},
	{0x10B8722AD71F31A6ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\rogue\\dual weapons\\dw_right_up_lhand.evt"},
	{0x10B8C70D6A810065ULL, "vxw_sundermount_2sk_0_0.vlm"},
	{0x10B92C89548317A9ULL, "art\\vfx\\combat\\vfx_m_sta_ice_range_f_p.impa.evt"},
	{0x10B942DE42344408ULL, "art\\vfx\\combat\\vfx_w_off_stun_chc_a_c.impa.evt"},
	{0x10B9766F5E92FF19ULL, "llac_ca_cavern1_haunted_24424_16.dds"},
	{0x10B99121BE65B07FULL, "art\\characters\\playercharacter\\humanmale\\arm_bda.phy"},
	{0x10B9E54A096A2FE5ULL, "lt_gallowscourtyard_59_0_ll.mmh"},
	{0x10BA26D046900E49ULL, "follower_banter6172295.cl"},
	{0x10BA29898865D801ULL, "197412335.ogg"},
	{0x10BA3D0788B21740ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\2arm_questioning.ani"},
	{0x10BA9EFE5AED48C8ULL, "6054849_m.fxe"},
	{0x10BAC5D478809EFEULL, "vxw_bonepitmining_a2_e7_0_0.vlm"},
	{0x10BAE5C9AD3336F0ULL, "art\\vfx\\combat\\vfx_m_sph_healaura_crust_c.mmh"},
	{0x10BB3BD214F5C548ULL, "vxlt_0_lowtown_a3_n_c_env_0_0.vlm"},
	{0x10BC155CBF967DB3ULL, "vxlt_0_lowtown_docks_a3_c1h_0_0.vlm"},
	{0x10BC250A221B3977ULL, "fen110_fenris6096216.cl"},
	{0x10BCB70D58412F3BULL, "6096765_m.fxe"},
	{0x10BDF02765E2465CULL, "vxlt_backalley_n_1vv_0_0.vlm"},
	{0x10BE892AD723C4F1ULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\rogue\\dual weapons\\dw_right_up_lhand.gad"},
	{0x10BE93D4DFBE2358ULL, "vxht_0_hightown_a2_c42_0_0.vlm"},
	{0x10BEF7BF924FBC61ULL, "vxca_caverns1_k2u_0_0.vlm"},
	{0x10BF121BFFD3A9CEULL, "6027557_m.fxe"},
	{0x10BF41A62A1A2D1FULL, "mag291_ntbanter6056619.cl"},
	{0x10BF6929C3A1C331ULL, "mcr370_meredith6083625.cl"},
	{0x10BF961F02840170ULL, "119650611.ogg"},
	{0x10BFD61019396E45ULL, "vxw_bonepitdestroyed_a3_1me_0_0.vlm"},
	{0x10BFF0580A3CE315ULL, "hlac_lt_0_lowtown_65535_2x.dds"},
	{0x10BFFB30ADB412EFULL, "w_bonepitdestroyed_a3_0_0.mao"},
	{0x10C00730ADB4268CULL, "w_bonepitdestroyed_a3_0_0.mmh"},
	{0x10C00D30ADB4317EULL, "w_bonepitdestroyed_a3_0_0.msh"},
	{0x10C024F9459731E1ULL, "vxlt_0_lowtown_a3_n_c_eqr_0_0.vlm"},
	{0x10C0252734F403FFULL, "hlac_lt_0_lowtown_a2_65537_4x.dds"},
	{0x10C0746C370FCCA4ULL, "hlac_ca_cavern1_haunted_24421_12x.dds"},
	{0x10C0C3143AB534DAULL, "llac_lt_gallowstemplar_n_2383_3.dds"},
	{0x10C12C34C496A18AULL, "art\\vfx\\environment\\vfx_trap_fire_jet_s_p.anb"},
	{0x10C17131E6196B45ULL, "vxlt_0_lowtown_docks_b9h_0_0.vlm"},
	{0x10C21434A02FD99FULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\1wi_basic.msh"},
	{0x10C21A34A02FE441ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\1wi_basic.mmh"},
	{0x10C22634A02FF8E2ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\1wi_basic.mao"},
	{0x10C2736BAD2039D6ULL, "402752921.ogg"},
	{0x10C3064B3B582AB9ULL, "193990240.ogg"},
	{0x10C33A1E0B1AF2C7ULL, "vxca_cavern1_haunted_j26_0_1.vlm"},
	{0x10C3B8410FB1CF31ULL, "vxlt_backalley_g_1ki_0_0.vlm"},
	{0x10C3DD26CFBFF418ULL, "art\\characters\\creatures\\gallowsslavestatue\\animation\\blendtree\\gallowsslavestatue.anb"},
	{0x10C42B05F1AD67C4ULL, "vxht_0_hightown_n_c8y_0_0.vlm"},
	{0x10C42DCF2B9BB1E4ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\follower_hm_enter.ani"},
	{0x10C42F9AB4E8C828ULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\3ww_ornate.phy"},
	{0x10C47E4E5B91D444ULL, "art\\characters\\playercharacter\\humanfemale\\hf_boo_lgte.phy"},
	{0x10C4C3E0D66AADDFULL, "vxht_0_hightown_a2_awr_0_0.vlm"},
	{0x10C51EF0FCD00668ULL, "349728597.ogg"},
	{0x10C58A2AD729E49FULL, "art\\characters\\playercharacter\\humanmale\\animation\\combat\\rogue\\dual weapons\\dw_right_up_lhand.ani"},
	{0x10C5DAFE2B9E5B6FULL, "6017810_m.fxe"},
	{0x10C60380D01F9542ULL, "art\\characters\\playercharacter\\elffemale\\textures\\ef_arm_nud_0s.dds"},
	{0x10C645C4B98C6787ULL, "s_ballista_04_l3.mao"},
	{0x10C6FA60DAD514A8ULL, "art\\characters\\weapons\\staff\\textures\\w_stf_dssta_0t.dds"},
	{0x10C71489548F2C18ULL, "art\\vfx\\combat\\vfx_m_sta_ice_range_f_p.impa.ani"},
	{0x10C738EA5C7718E5ULL, "vxw_sundermount_2tj_0_0.vlm"},
	{0x10C75AD1ADE50632ULL, "6110769_m.fxe"},
	{0x10C7EA610AA6B07FULL, "vxlt_gamlenshouse_n_3r_0_0.vlm"},
	{0x10C854A80B6455BAULL, "6148194_f.fxe"},
	{0x10C875FEFE05C5BBULL, "754900286.ogg"},
	{0x10C8807553A46848ULL, "vxca_caverns1_k2p_0_0.vlm"},
	{0x10C8DCB43BF887F5ULL, "art\\characters\\playercharacter\\dwarfmale\\dm_brd_be01a.msh"},
	{0x10C8DEB43BF88B6BULL, "art\\characters\\playercharacter\\dwarfmale\\dm_brd_be01a.mmh"},
	{0x10C8DF7D2F93E439ULL, "hlac_lt_0_lowtown_docks_a3_n_-196609_0y.dds"},
	{0x10C908C9AD3FAF0FULL, "art\\vfx\\combat\\vfx_m_sph_healaura_crust_c.anb"},
	{0x10C910DB4CE36148ULL, "6051749_m.fxe"},
	{0x10C9124CB06175A0ULL, "hlac_lt_0_lowtown_a3_n_c_65536_2z.dds"},
	{0x10C9ADAF06EBDDC8ULL, "vxlt_0_lowtown_a3_n_c_eqw_0_0.vlm"},
	{0x10C9B76E70637B37ULL, "plt_cod_crt_shade.plo"},
	{0x10CA3F9C5E2AFEB1ULL, "follower_banter6163280.cl"},
	{0x10CA9221BE742484ULL, "art\\characters\\playercharacter\\humanmale\\arm_bda.mmh"},
	{0x10CA9621BE742BD7ULL, "art\\characters\\playercharacter\\humanmale\\arm_bda.mao"},
	{0x10CAA821BE744A26ULL, "art\\characters\\playercharacter\\humanmale\\arm_bda.msh"},
	{0x10CAB572B432154FULL, "170224692.ogg"},
	{0x10CACC2DB57AFF1FULL, "tm_gwp_h01.tnt"},
	{0x10CACCBCC0C2C6CEULL, "6075154_m.fxe"},
	{0x10CAEA556085FD4CULL, "125469942.ogg"},
	{0x10CB2ACF2BA1CB46ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\follower_hm_enter.gad"},
	{0x10CB2F2740BA2E1EULL, "224008536.ogg"},
	{0x10CB866232E222AEULL, "6150219_m.fxe"},
	{0x10CBA76FA36040FDULL, "follower_banter6167877.cl"},
	{0x10CBBE1E03074AA3ULL, "art\\characters\\playercharacter\\elffemale\\animation\\exploration\\idle\\twitch_v5.ani"},
	{0x10CC02C7ACCDEAA9ULL, "art\\characters\\weapons\\shields\\w_shd_eln.mao"},
	{0x10CC439B6CB9C662ULL, "vxht_brothel_n_4ah_0_0.vlm"},
	{0x10CC83CD59D60D7AULL, "vxdr_primevaltomb_8um_0_0.vlm"},
	{0x10CC94BB5E294B8EULL, "6175386_m.fxe"},
	{0x10CCC95C5EA120A5ULL, "vxw_bonepit_85_0_0.vlm"},
	{0x10CD05B1D1B43BD8ULL, "vxht_0_hightown_a2_c33_0_0.vlm"},
	{0x10CDCAA1A194BFCDULL, "art\\levels\\races\\ferelden\\props\\chantry\\prp_flagchantryb.msh"},
	{0x10CDCE41CF9FA839ULL, "hlac_w_sundermount_-65538_0x.dds"},
	{0x10CDD8A1A194D700ULL, "art\\levels\\races\\ferelden\\props\\chantry\\prp_flagchantryb.mao"},
	{0x10CDDCA1A194DE53ULL, "art\\levels\\races\\ferelden\\props\\chantry\\prp_flagchantryb.mmh"},
	{0x10CDDDA0B95ECDD7ULL, "art\\levels\\races\\natural\\kirkcoastal\\foliage\\tre_01deadtree.phy"},
	{0x10CEBA866507A589ULL, "hlac_w_bonepit_0_9z.dds"},
	{0x10CEC8C7335B41AAULL, "vxlt_0_lowtown_docks_n_bcn_0_0.vlm"},
	{0x10CEDC2550CAFB89ULL, "6054691_m.fxe"},
	{0x10CF399FC372A517ULL, "23301679.ogg"},
	{0x10CF43988AB23AABULL, "hlac_lt_0_lowtown_a2_n_c_-65535_3y.dds"},
	{0x10CF555A5842A11FULL, "6149421_m.fxe"},
	{0x10CFBD26D42E9502ULL, "102077245.ogg"},
	{0x10D148A9D7DAC789ULL, "hlac_lt_gallowscourtyard_n_131071_0y.dds"},
	{0x10D225CF2BA7E125ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\follower_hm_enter.evt"},
	{0x10D2C8867A5076CFULL, "hlac_dr_primevaltomb_659_3x.dds"},
	{0x10D2DBAE69F8C190ULL, "gen00fl_aveline6139165.cl"},
	{0x10D2FAF0E98C4107ULL, "vrc321tr_spooky_scenes_3.utt"},
	{0x10D315B296AA9094ULL, "art\\characters\\playercharacter\\humanfemale\\arm_mede.phy"},
	{0x10D4313AC802042AULL, "art\\characters\\weapons\\staff\\textures\\w_stf_016a_s.dds"},
	{0x10D43C6B1E5F616BULL, "6149111_m.fxe"},
	{0x10D46DF153F73527ULL, "hlac_lt_0_lowtown_docks_a3_196606_0z.dds"},
	{0x10D47506FACD03B8ULL, "vxw_bonepitmining_a2_gg_0_0.vlm"},
	{0x10D4ABA4C66CCF30ULL, "6167304_f.fxe"},
	{0x10D58355F431DFCCULL, "115476244.ogg"},
	{0x10D5AAC74E6D3165ULL, "vxw_sundermount_2yi_0_0.vlm"},
	{0x10D64A1FC29EAB9AULL, "dae000_gamlen6005774.cl"},
	{0x10D727B967EC1E4CULL, "75657167.ogg"},
	{0x10D7575370F801E1ULL, "1027612044.ogg"},
	{0x10D7E198BAC8E676ULL, "vxlt_0_lowtown_gik_0_0.vlm"},
	{0x10D81F8BF8E1F648ULL, "vxlt_0_lowtown_a3_n_c_ett_0_0.vlm"},
	{0x10D84C5794272A4CULL, "hlac_ht_playershouse_n_855_5z.dds"},
	{0x10D87012CCFC49A9ULL, "960859523.ogg"},
	{0x10D9A7701B2C7DF4ULL, "6152800_m.fxe"},
	{0x10D9AE41D5FD73D0ULL, "6112930_m.fxe"},
	{0x10D9CF58708C7ED4ULL, "544175002.ogg"},
	{0x10D9D61E0313B09AULL, "art\\characters\\playercharacter\\elffemale\\animation\\exploration\\idle\\twitch_v5.evt"},
	{0x10D9F3C6F4B7A1A5ULL, "projdecal_projdecal_p0.vsh.fxo.dx11.assembly"},
	{0x10DA059398CE0244ULL, "vxw_woundedcoast_o4_0_0.vlm"},
	{0x10DA94D93F19DCBBULL, "624105322.ogg"},
	{0x10DAB5785EAFDEE2ULL, "vxht_brothel_n_4fi_0_0.vlm"},
	{0x10DAD7C08AA6964FULL, "6174915_m.fxe"},
	{0x10DB1D371B7AD2B0ULL, "llac_lt_0_lowtown_-262144_1.dds"},
	{0x10DB4ED49C78B65DULL, "6151630_f.fxe"},
	{0x10DB6345251CFE2EULL, "nat320_delilah6065924.cl"},
	{0x10DB6C8A138E01EBULL, "191217051.ogg"},
	{0x10DBC40DF23EA9DCULL, "370393524.ogg"},
	{0x10DC235F833CABD5ULL, "6173581_m.fxe"},
	{0x10DC2D63B50C761CULL, "vxht_mansion_n_2br_0_0.vlm"},
	{0x10DC7D75B622E8B6ULL, "qcr000_varric6208237.cl"},
	{0x10DCB48F83A0CFF4ULL, "art\\characters\\creatures\\dragon\\animation\\hv_dmg.ani"},
	{0x10DCBC5549972224ULL, "123514954.ogg"},
	{0x10DCF2B296B2A991ULL, "art\\characters\\playercharacter\\humanfemale\\arm_mede.msh"},
	{0x10DD00B296B2C1ACULL, "art\\characters\\playercharacter\\humanfemale\\arm_mede.mao"},
	{0x10DD0CB296B2D64FULL, "art\\characters\\playercharacter\\humanfemale\\arm_mede.mmh"},
	{0x10DD139251F2CD86ULL, "6167460_m.fxe"},
	{0x10DDC7C25B04F480ULL, "6147608_m.fxe"},
	{0x10DDDB887104FE5CULL, "hlac_ht_0_hightown_n_131070_0z.dds"},
	{0x10DE190EC96EC867ULL, "gen00fl_fenris6105845.cl"},
	{0x10DE201E1241F5FDULL, "471135860.ogg"},
	{0x10DE567AF1DEFC9BULL, "vxht_brothel_n_3wp_0_0.vlm"},
	{0x10DF39F77ECBDBC0ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\0wi_edgebase.phy"},
	{0x10DF54D3EA357234ULL, "hlac_ht_0_hightown_a3_131072_0x.dds"},
	{0x10DFFFC0D6AD0806ULL, "106766541.ogg"},
	{0x10E033E94316BBD6ULL, "60194282.ogg"},
	{0x10E03B3B3CB4915BULL, "vxlt_0_lowtown_n_e5w_0_0.vlm"},
	{0x10E080F1D812862DULL, "gen_im_delete_me_03.uti"},
	{0x10E0CFAC3A4F8059ULL, "418332583.ogg"},
	{0x10E0D51E0319CD8DULL, "art\\characters\\playercharacter\\elffemale\\animation\\exploration\\idle\\twitch_v5.gad"},
	{0x10E17672154BBCF9ULL, "6118202_f.fxe"},
	{0x10E22C5882674582ULL, "6083591_m.fxe"},
	{0x10E280574E4C1ED4ULL, "6037330_m.fxe"},
	{0x10E284433443849BULL, "6126736_m.fxe"},
	{0x10E2B18F83A53656ULL, "art\\characters\\creatures\\dragon\\animation\\hv_dmg.gad"},
	{0x10E2E6E3ECC31C38ULL, "vxw_bonepitmining_a2_bd_0_0.vlm"},
	{0x10E2F507CFCEA3B2ULL, "vxlt_0_lowtown_docks_bs5_0_0.vlm"},
	{0x10E3527341CF1A55ULL, "llac_lt_0_lowtown_docks_-131075_1.dds"},
	{0x10E36EA637531E67ULL, "gen00fl_merrill6129472.cl"},
	{0x10E446D4DEE823FCULL, "hlac_lt_0_lowtown_docks_a3_-131074_1y.dds"},
	{0x10E48178C690F8D7ULL, "vxlt_hangedman_n_2ur_0_0.vlm"},
	{0x10E523207B0BAA23ULL, "docks_male_human_worker_sta6175978.cl"},
	{0x10E541C0190EF528ULL, "6148960_m.fxe"},
	{0x10E5B405B4E27CCFULL, "vxlt_0_lowtown_a2_n_c_gdd_0_0.vlm"},
	{0x10E5E3DFD75DAEA9ULL, "6085936_m.fxe"},
	{0x10E6A4911D779229ULL, "vxw_encounter2_cp_0_0.vlm"},
	{0x10E6D519841AE4A9ULL, "216359917.ogg"},
	{0x10E6F4E5A51B02D4ULL, "vxlt_darktown_4ii_0_0.vlm"},
	{0x10E73F8E65AB5C19ULL, "hlac_lt_0_lowtown_docks_-65537_5y.dds"},
	{0x10E76F3C0EB5111FULL, "90748317.ogg"},
	{0x10E7D4A9205029FCULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\0ww_2windows.mao"},
	{0x10E7E0A920503D9FULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\0ww_2windows.mmh"},
	{0x10E7E6A920504841ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\0ww_2windows.msh"},
	{0x10E7F3FDFF369213ULL, "mcr370_cut_player_survives6098269.cl"},
	{0x10E7F953F1358318ULL, "plt_cod_hst_dalish_culture.plo"},
	{0x10E8496193E90ED2ULL, "mag291_torpor6116228.cl"},
	{0x10E85D326F6E27C4ULL, "350498975.ogg"},
	{0x10E85E67EC167A73ULL, "framebuffereffects_cocdownsample_p0.psh.fxo.dx11"},
	{0x10E8E54801E4EAA4ULL, "llac_lt_0_lowtown_a2_n_131072_6.dds"},
	{0x10E909946C548445ULL, "vxdr_primevaltomb_8zg_0_0.vlm"},
	{0x10E92E6798A95BCFULL, "19824812.ogg"},
	{0x10E92F34C4B7F97DULL, "art\\vfx\\environment\\vfx_trap_fire_jet_s_p.mmh"},
	{0x10E99A3A7843A096ULL, "vxlt_gallowsprison_n_649_0_0.vlm"},
	{0x10E9B36F5DD0F02AULL, "art\\vfx\\combat\\vfx_c_2nd_buff_loop_c.cess.gad"},
	{0x10E9CC8F83AB82B5ULL, "art\\characters\\creatures\\dragon\\animation\\hv_dmg.evt"},
	{0x10EA615D4FD9722FULL, "6125214_m.fxe"},
	{0x10EAA4A5DB0B8FDCULL, "qun221_petrice6082796.cl"},
	{0x10EABD1349E96C63ULL, "6111217_m.fxe"},
	{0x10EBF65DAB5F5154ULL, "art\\vfx\\combat\\vfx_m_spi_walkbomb_c.impa.ani"},
	{0x10EBFDF386C2A67CULL, "6127109_m.fxe"},
	{0x10EC30D28AFD4E58ULL, "gen00fl_bethany6005722.cl"},
	{0x10ECBA6A1C2874E4ULL, "6002778_m.fxe"},
	{0x10ECD766CAEE3829ULL, "6151080_m.fxe"},
	{0x10ED92F2255698E6ULL, "6051384_f.fxe"},
	{0x10EDBEB15E6CB505ULL, "hlac_w_sundermount_131076_0z.dds"},
	{0x10EE1B34A05520BEULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\1wi_basic.phy"},
	{0x10EE1F6CDF19769AULL, "w_bonepit_255_0.msh"},
	{0x10EE2D6CDF198ED3ULL, "w_bonepit_255_0.mao"},
	{0x10EE316CDF1995E0ULL, "w_bonepit_255_0.mmh"},
	{0x10EF137601C41354ULL, "art\\characters\\playercharacter\\humanmale\\textures\\hn_glv_masa_0s.dds"},
	{0x10EF1B0E6DBA378BULL, "86074426.ogg"},
	{0x10EF703A949BBCE5ULL, "gen_im_cns_pot_stamina_01.uti"},
	{0x10EF764FE226BE27ULL, "566244196.ogg"},
	{0x10F0623A3EFBC620ULL, "6115636_m.fxe"},
	{0x10F094417C917465ULL, "6043453_m.fxe"},
	{0x10F0966F5DD6DDC1ULL, "art\\vfx\\combat\\vfx_c_2nd_buff_loop_c.cess.evt"},
	{0x10F0AB36A57A3AF9ULL, "oil_loop_a.dds"},
	{0x10F1002820AD5E47ULL, "dre000_sandal6033756.cl"},
	{0x10F1F56DDA00D338ULL, "watertoolset.psh.fxo"},
	{0x10F26B4F2504FFACULL, "mag131_keran6086244.cl"},
	{0x10F2F35DAB656AB6ULL, "art\\vfx\\combat\\vfx_m_spi_walkbomb_c.impa.gad"},
	{0x10F373EBD83B53D7ULL, "6128275_m.fxe"},
	{0x10F3C02F0E17E033ULL, "vxht_playershouse_n_1uz_0_0.vlm"},
	{0x10F3F6BC43202078ULL, "301765833.ogg"},
	{0x10F3F9F657DCBBEEULL, "6149051_m.fxe"},
	{0x10F46F4A9DAC73A5ULL, "gen00fl_aveline6121897.cl"},
	{0x10F4F14225780E88ULL, "vxdr_slavers_14a_0_0.vlm"},
	{0x10F5166E0F6DAAA9ULL, "vxw_encounter2_nw_0_0.vlm"},
	{0x10F54149A5FC838FULL, "zz_followers_banter6158031.cl"},
	{0x10F54BAD7B701436ULL, "6155898_m.fxe"},
	{0x10F57ECB6D457A37ULL, "956844289.ogg"},
	{0x10F64CA5FAE84D50ULL, "vxht_mansion2b_2bm_0_0.vlm"},
	{0x10F6C842D3644FBEULL, "follower_banter6164730.cl"},
	{0x10F77508610C2590ULL, "art\\characters\\playercharacter\\humanmale\\animation\\blendtree\\humanoid_local_ogre_frozen.anb"},
	{0x10F7B3E08B96CFF0ULL, "6033705_m.fxe"},
	{0x10F7DD52C333C900ULL, "322572922.ogg"},
	{0x10F7E32BD27A98EEULL, "gen00fl_isabela6015840.cl"},
	{0x10F8760E6B514B29ULL, "256466217.ogg"},
	{0x10F89E8BE9F1903AULL, "lt_0_lowtown_docks_n.rml"},
	{0x10F8E9A3144A7236ULL, "dae000_hate_enemies6167620.cl"},
	{0x10F90E5DAB6A0415ULL, "art\\vfx\\combat\\vfx_m_spi_walkbomb_c.impa.evt"},
	{0x10F927905A6E27F7ULL, "vxdr_primevaltomb_6qb_0_0.vlm"},
	{0x10F92D5ED414A1AEULL, "gen00fl_aveline6140127.cl"},
	{0x10F930327AEB958DULL, "488251618.ogg"},
	{0x10F977AA6AF38540ULL, "6149202_m.fxe"},
	{0x10F9D816E6E954D0ULL, "art\\characters\\playercharacter\\elffemale\\glv_hvyn.msh"},
	{0x10F9E616E6E96CEAULL, "art\\characters\\playercharacter\\elffemale\\glv_hvyn.mmh"},
	{0x10F9EA16E6E97321ULL, "art\\characters\\playercharacter\\elffemale\\glv_hvyn.mao"},
	{0x10FA021F98ADF355ULL, "hlac_lt_gallowstemplar_7198_7y.dds"},
	{0x10FA21D93E8082E5ULL, "gen00fl_aveline6115476.cl"},
	{0x10FA37EE9FFCD324ULL, "character_character_p9.psh.fxo"},
	{0x10FA55236CE13F4EULL, "6163204_m.fxe"},
	{0x10FAA235D74AD44FULL, "hlac_dr_0_deeproads_5843_5z.dds"},
	{0x10FB228B1A3C6F99ULL, "6081204_m.fxe"},
	{0x10FB800B8DFA4826ULL, "273331502.ogg"},
	{0x10FBB6F80D52C46CULL, "lt_0_lowtown_docks_n_186_0.msh"},
	{0x10FBBCF80D52CE5EULL, "lt_0_lowtown_docks_n_186_0.mmh"},
	{0x10FBC8F80D52E2F5ULL, "lt_0_lowtown_docks_n_186_0.mao"},
	{0x10FBFEF2727A43A9ULL, "vxlt_undercitydungeon_1j7_0_0.vlm"},
	{0x10FC1EC6D6AD68CAULL, "framebuffereffects_twopassblurcochorizontl_p0.psh.fxo.dx11.assembly"},
	{0x10FCAA041B34F396ULL, "pro000_soldier16040818.cl"},
	{0x10FCD50EE03E66DBULL, "6048425_m.fxe"},
	{0x10FCFB1E2981C280ULL, "vxca_cavern1_haunted_gil_0_0.vlm"},
	{0x10FD1EF920CC5B14ULL, "art\\vfx\\combat\\vfx_glb_lit_imp_c.impa.ani"},
	{0x10FD427E97D52D47ULL, "gen_im_trs_krk_shd_met_r0.uti"},
	{0x10FD8E6F5DE15A10ULL, "art\\vfx\\combat\\vfx_c_2nd_buff_loop_c.cess.ani"},
	{0x10FDA11BDB2F3707ULL, "gen_im_wep_rog_dua_act1_05.uti"},
	{0x10FDADE7FD98BF8AULL, "6059336_m.fxe"},
	{0x10FDCF05B2249AB6ULL, "vxht_chantry_1yi_0_0.vlm"},
	{0x10FE94C049E81656ULL, "art\\levels\\races\\ferelden\\props\\plotspecifc\\prp_codexwild.msh"},
	{0x10FE9A3080725BF0ULL, "hlac_ca_caverns1_mines_24425_0y.dds"},
	{0x10FE9EC049E82754ULL, "art\\levels\\races\\ferelden\\props\\plotspecifc\\prp_codexwild.mmh"},
	{0x10FEA2C049E82E67ULL, "art\\levels\\races\\ferelden\\props\\plotspecifc\\prp_codexwild.mao"},
	{0x10FEEC99B280CDBDULL, "vxca_caverns2_8xf_0_1.vlm"},
	{0x10FF3CD903321D43ULL, "523387906.ogg"},
	{0x10FF6F3FF0BED963ULL, "6121009_m.fxe"},
	{0x10FFB012FF0A7618ULL, "gen00fl_aveline6132928.cl"},
	{0x10FFB7DF760C3541ULL, "ht_bartrand_n_3364.rml"},
	{0x10FFBF60AFADA866ULL, "978304317.ogg"},
	{0x110038D658DC58A6ULL, "vxw_bonepitdestroyed_a3_163_0_0.vlm"},
	{0x1100C83A0D922214ULL, "hlac_lt_gallowscourtyard_n_-65538_2y.dds"},
	{0x1100D77ADA42F47EULL, "6108752_f.fxe"},
	{0x1100D8A8AC22B592ULL, "art\\characters\\playercharacter\\humanmale\\animation\\blendtree\\shielddefence_override.anb"},
	{0x11010C322B7E7A23ULL, "411382895.ogg"},
	{0x11017EC6D3E3BFC4ULL, "64580335.ogg"},
	{0x1101E0B7D3D4EE8AULL, "6188566_m.fxe"},
	{0x1102210304B89EF7ULL, "6208687_m.fxe"},
	{0x1102328DDF519B8EULL, "plt_cod_mgc_andraste.plo"},
	{0x110253062BBA0843ULL, "169528174.ogg"},
	{0x11027E5D91551470ULL, "llac_lt_0_lowtown_-131071_1.dds"},
	{0x1102937B0F1489FAULL, "pet_so.mao"},
	{0x110297BF98CEADCFULL, "vxlt_0_lowtown_a2_n_c_gzf_0_0.vlm"},
	{0x1102B20AB6DA86D2ULL, "6014029_m.fxe"},
	{0x1103884B0163C329ULL, "vxw_encounter2_iv_0_0.vlm"},
	{0x1103BA627A1E3DEFULL, "54209894.ogg"},
	{0x1103CD682975889CULL, "hlac_lt_0_lowtown_docks_n_-262143_0z.dds"},
	{0x1103DB34B2C6859CULL, "129680488.ogg"},
	{0x11041BF920D27576ULL, "art\\vfx\\combat\\vfx_glb_lit_imp_c.impa.gad"},
	{0x110456F77EEB2955ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\0wi_edgebase.msh"},
	{0x110458F77EEB2CCBULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\0wi_edgebase.mmh"},
	{0x11045DAA03C85851ULL, "art\\levels\\races\\proxy\\props\\rugs_poor\\prp_zzlargegreen_0.pwk"},
	{0x110464F77EEB4168ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\0wi_edgebase.mao"},
	{0x110481EF448FD887ULL, "hlac_lt_darktown_-1_85x.dds"},
	{0x1104C245953288BFULL, "406155329.ogg"},
	{0x1104D9C581695362ULL, "107559896.ogg"},
	{0x1104FD13AC0A0F92ULL, "6048187_m.fxe"},
	{0x1105691FEAD72DECULL, "qun111_ketojan6039017.cl"},
	{0x11063E1B7FC4F695ULL, "dae000cr_grand_cleric6124700.cl"},
	{0x1106765C56B518C6ULL, "dre000_varric6207965.cl"},
	{0x11068ECB7A869FFEULL, "gen00fl_fenris6115326.cl"},
	{0x1106FC7FBE05F91CULL, "zz_mag_debug6051239.cl"},
	{0x1107CB2B95CE0B7AULL, "218850730.ogg"},
	{0x1107CBCD9E233881ULL, "hlac_ht_0_hightown_-131074_0y.dds"},
	{0x11081D4A6398B274ULL, "146025936.ogg"},
	{0x1108522EEF4F444CULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\9ff_rock3a.phy"},
	{0x1108E489FC8C7423ULL, "285320014.ogg"},
	{0x11092369EAFC7B02ULL, "6164652_m.fxe"},
	{0x1109CA06287BD90BULL, "dae000_bodhan_letter6144309.cl"},
	{0x110A44424B9AB58CULL, "hlac_ht_playershouse_n_855_14x.dds"},
	{0x110A484F6E5D6676ULL, "art\\vfx\\combat\\vfx_r_gn4_stealth_loop_c.impa.evt"},
	{0x110A5B391F0D6FB8ULL, "hlac_w_encounter2_262146_0y.dds"},
	{0x110B36F920D8C1D5ULL, "art\\vfx\\combat\\vfx_glb_lit_imp_c.impa.evt"},
	{0x110B6DA91B65CB78ULL, "122509621.ogg"},
	{0x110B9989B715E01CULL, "vxca_caverns2_9fi_0_0.vlm"},
	{0x110BF337972305CCULL, "dae000ip_crowd_lm5_talk1.utp"},
	{0x110C75C86607EC49ULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\additive\\arm_r_tochest.ani"},
	{0x110CB6AD660C23E3ULL, "gen00fl_bethany6102776.cl"},
	{0x110D24F83CDA88DFULL, "6042122_m.fxe"},
	{0x110D905FA18C69E4ULL, "392628389.ogg"},
	{0x110E5D58548E9804ULL, "597023176.ogg"},
	{0x110E90405B30CD39ULL, "lla_ht_0_hightown_a2_-65535_0.dds"},
	{0x110EB5A78183C156ULL, "vxht_0_hightown_av0_0_0.vlm"},
	{0x110F3C12A87483F6ULL, "482833819.ogg"},
	{0x110F6A7FC605321AULL, "6081208_f.fxe"},
	{0x110F776E6D0E642BULL, "llac_w_blightlands_-65538_0.dds"},
	{0x110F913EEF0921ACULL, "vxht_0_hightown_c5a_0_0.vlm"},
	{0x110FB29A9EA0C1C4ULL, "gen00fl_aveline6139716.cl"},
	{0x11103F4F6E61C201ULL, "art\\vfx\\combat\\vfx_r_gn4_stealth_loop_c.impa.gad"},
	{0x111049061158C603ULL, "6201514_m.fxe"},
	{0x111064EE7233EF6BULL, "s_ballista_03_l2.mao"},
	{0x1110935A91F530AFULL, "cli600cr_hurlock_ranged.utc"},
	{0x1110B608101C4303ULL, "6135224_m.fxe"},
	{0x1110EC766864C0EAULL, "dae000_gamlen6088135.cl"},
	{0x11113473048894E9ULL, "hlac_lt_backalley_g_0_9z.dds"},
	{0x111143F6EA74D488ULL, "mcr371_orsino.cnv"},
	{0x111158C45E43A558ULL, "gen00fl_aveline6122110.cl"},
	{0x11119644FC061E4BULL, "6124679_m.fxe"},
	{0x1111AACD932F5ECFULL, "ach_kill_rock.dds"},
	{0x1111C8F0F121CC3BULL, "vxca_caverns1_gk5_0_1.vlm"},
	{0x111201DFD592F6BFULL, "art\\characters\\creatures\\crowds\\humanmale\\animation\\kirkwall_and_lowtown\\humanmale_ambient_angrydebate_1_darren.gad"},
	{0x111236E93C77CB8CULL, "6031985_m.fxe"},
	{0x111290C8660C85CBULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\additive\\arm_r_tochest.gad"},
	{0x1112EB24C4C7E8A6ULL, "792789914.ogg"},
	{0x1112F737ED7F8601ULL, "6107559_m.fxe"},
	{0x1113272A9B3AAB34ULL, "56470116.ogg"},
	{0x1113B08FE316BE14ULL, "mcr371_hostile_anders6082105.cl"},
	{0x1113B7D15B102606ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\assassin\\templarseeker\\ass_atk_v2.ani"},
	{0x11140D26899693D1ULL, "art\\vfx\\combat\\vfx_c_2nd_buff_frd_loop_c.cess.ani"},
	{0x1114AA7D71730A90ULL, "nat380_sibling6103577.cl"},
	{0x1114DA59B77BC43EULL, "vxlt_0_lowtown_docks_n_c59_0_0.vlm"},
	{0x1114F4DCEC8A8B91ULL, "6040834_m.fxe"},
	{0x111515DFA6D823D5ULL, "vxdr_0_deeproads_cf6_0_0.vlm"},
	{0x111565F6EA78F7F0ULL, "mcr371_orsino.bnk"},
	{0x11157670168FF6ACULL, "919856686.ogg"},
	{0x1115B2A0B99C3A2FULL, "art\\levels\\races\\natural\\kirkcoastal\\foliage\\tre_01deadtree.mao"},
	{0x1115BEA0B99C4ECCULL, "art\\levels\\races\\natural\\kirkcoastal\\foliage\\tre_01deadtree.mmh"},
	{0x1115C4A0B99C58BEULL, "art\\levels\\races\\natural\\kirkcoastal\\foliage\\tre_01deadtree.msh"},
	{0x1115E020CD72E751ULL, "vxlt_darktown_5t6_0_0.vlm"},
	{0x11162C1D3C48B3DDULL, "65163093.ogg"},
	{0x111638BD01C0DB50ULL, "llac_lt_0_lowtown_a2_n_c_1_1.dds"},
	{0x11163E8056D02504ULL, "vxw_sundermount_2g2_0_0.vlm"},
	{0x11166A4A53A60516ULL, "vxca_caverns1_j2b_0_1.vlm"},
	{0x1116B7F125031AE9ULL, "art\\characters\\weapons\\longswords\\textures\\w_lsw_ls106a_0d.dds"},
	{0x11172831DAF70748ULL, "seb321_leliana6180967.cl"},
	{0x1117404F6E67E2AFULL, "art\\vfx\\combat\\vfx_r_gn4_stealth_loop_c.impa.ani"},
	{0x11176A8F58552035ULL, "650696229.ogg"},
	{0x111822AD7F477208ULL, "739854136.ogg"},
	{0x111869FFB726BBDCULL, "w_woundedcoast_39_0.mao"},
	{0x111875FFB726D07FULL, "w_woundedcoast_39_0.mmh"},
	{0x1118792331963DE6ULL, "6120762_m.fxe"},
	{0x11187BFFB726DA21ULL, "w_woundedcoast_39_0.msh"},
	{0x1118FA929EF96E8BULL, "mag170_thrask6020545.cl"},
	{0x11192855F8D96D8AULL, "6068422_m.fxe"},
	{0x11193B1C75E75CB2ULL, "421903520.ogg"},
	{0x11198DC866129F30ULL, "art\\characters\\playercharacter\\humanfemale\\animation\\cindes\\additive\\arm_r_tochest.evt"},
	{0x1119D1CF7D632C70ULL, "hlac_lt_0_lowtown_a2_-1_3x.dds"},
	{0x1119E57252C835D4ULL, "hlac_lt_0_lowtown_docks_a3_-131075_1z.dds"},
	{0x111A1826899B1273ULL, "art\\vfx\\combat\\vfx_c_2nd_buff_frd_loop_c.cess.gad"},
	{0x111A30D054903264ULL, "llac_ht_0_hightown_-65534_1.dds"},
	{0x111ABA61AFF44B2AULL, "ave331_jeven6152174.cl"},
	{0x111AE2875786293CULL, "mrl351cr_fakemaleelf.utc"},
	{0x111B42D37F9A5854ULL, "hlac_lt_undercityencounter_-1_6z.dds"},
	{0x111B5EF86881FD46ULL, "vxlt_0_lowtown_docks_n_b9p_0_0.vlm"},
	{0x111B92A201E4BB2AULL, "ser270_emeric6149324.cl"},
	{0x111BBBC9AD2C41ADULL, "w_blightlands_48_0.mmh"},
	{0x111BBFC9AD2C48FEULL, "w_blightlands_48_0.mao"},
	{0x111BCDC9AD2C6033ULL, "w_blightlands_48_0.msh"},
	{0x111BE54DD89F46CDULL, "6079280_f.fxe"},
	{0x111C02B9DD4191E8ULL, "art\\levels\\races\\proxy\\props\\4tools\\vis_blades_wall_tool.msh"},
	{0x111C06DC638B6825ULL, "vxlt_0_lowtown_gj5_0_0.vlm"},
	{0x111C14B9DD41B079ULL, "art\\levels\\races\\proxy\\props\\4tools\\vis_blades_wall_tool.mao"},
	{0x111C20B9DD41C412ULL, "art\\levels\\races\\proxy\\props\\4tools\\vis_blades_wall_tool.mmh"},
	{0x111C2BE204BCEB71ULL, "6079019_f.fxe"},
	{0x111C44C222BF9C68ULL, "vxw_bonepitdestroyed_a3_9k_0_0.vlm"},
	{0x111C6B7F286BCC67ULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_sidetable_n.dds"},
	{0x111C888BF357C94BULL, "art\\characters\\playercharacter\\humanfemale\\animation\\ambient\\dialogue\\lean_back_wall_loop.ani"},
	{0x111D0163A49316DFULL, "684258576.ogg"},
	{0x111D1A7D11D5B1E2ULL, "rdr110dg_guard_warehouse6016707.cl"},
	{0x111D568D7B718959ULL, "28056367.ogg"},
	{0x111D79DF0CC414B5ULL, "zz_vault_debug6177759.cl"},
	{0x111DD9994CDF7D61ULL, "6118080_f.fxe"},
	{0x111DE447A03C789AULL, "vxlt_0_lowtown_a2_n_c_e2m_0_0.vlm"},
	{0x111E30F3B0CE150CULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_junkb_n.dds"},
	{0x111E365D2EE229C1ULL, "vxlt_0_lowtown_ffx_0_0.vlm"},
	{0x111E4E338A468930ULL, "179973343.ogg"},
	{0x111E83D280148695ULL, "292595847.ogg"},
	{0x111EE04ADAB749FDULL, "vxht_0_hightown_a3_a7y_0_0.vlm"},
	{0x111FD8A1BCEC7C00ULL, "6045999_m.fxe"},
	{0x11200035B64692D0ULL, "6169868_m.fxe"},
	{0x1120179451A32D93ULL, "art\\characters\\playercharacter\\elfmale\\mt_art_em02.phy"},
	{0x1120468966FAAE16ULL, "vxht_0_hightown_a3_bfn_0_0.vlm"},
	{0x112093403224F6D3ULL, "dae100_viscount6118510.cl"},
	{0x1121049497218FF7ULL, "02_fire.dds"},
	{0x1121152689A12BD8ULL, "art\\vfx\\combat\\vfx_c_2nd_buff_frd_loop_c.cess.evt"},
	{0x11212966DB36A06AULL, "one100_solivitus6048107.cl"},
	{0x11213160A2AEB4DEULL, "ip_restrict_style_staff.ncs"},
	{0x11219FD15B1C3B77ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\assassin\\templarseeker\\ass_atk_v2.evt"},
	{0x11223EE4529C7CC4ULL, "vxw_woundedcoast_p9_0_0.vlm"},
	{0x1122AD2D456B72A2ULL, "character_character_p285.psh.fxo.dx11"},
	{0x1122C5625987D22BULL, "vxlt_gallowscourtyard_n_7gj_0_0.vlm"},
	{0x112389AEE3D2A1C8ULL, "178410830.ogg"},
	{0x112465BB34C053D7ULL, "tbe100_xenon6190998.cl"},
	{0x112496107329BE2BULL, "6042144_m.fxe"},
	{0x1124D67AD8120098ULL, "follower_barks6173407.cl"},
	{0x1124E1161A0786BCULL, "6189031_m.fxe"},
	{0x1124EB85D14C651FULL, "6076153_f.fxe"},
	{0x1124EF4CBC56877EULL, "322260179.ogg"},
	{0x11253120369B4CE7ULL, "355318965.ogg"},
	{0x11255C10273E8244ULL, "vxw_bonepit_1x3_0_0.vlm"},
	{0x1125BBC9AD349626ULL, "w_blightlands_48_0.rml"},
	{0x1125DA1BACB047C7ULL, "art\\vfx\\combat\\vfx_itm_tar_bomb_proj_p.cess.gad"},
	{0x1126088972A72F66ULL, "686449346.ogg"},
	{0x11261905EE10A48BULL, "tbe000im_ancient_chest.uti"},
	{0x112673004762447EULL, "376905316.ogg"},
	{0x1126DEDFD5A51705ULL, "art\\characters\\creatures\\crowds\\humanmale\\animation\\kirkwall_and_lowtown\\humanmale_ambient_angrydebate_1_darren.ani"},
	{0x11274773F1CC7D9BULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\9uw_pennantsml.mmh"},
	{0x11275373F1CC9238ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\9uw_pennantsml.mao"},
	{0x11276573F1CCB085ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\9uw_pennantsml.msh"},
	{0x11276D5AE9EC4508ULL, "395378432.ogg"},
	{0x1127A0D15B20A800ULL, "art\\characters\\playercharacter\\humanmale\\animation\\archetypes\\assassin\\templarseeker\\ass_atk_v2.gad"},
	{0x1127C79627830872ULL, "6127486_m.fxe"},
	{0x1127E676D49B97ACULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\9ww_shield.msh"},
	{0x1127EC76D49BA19EULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\9ww_shield.mmh"},
	{0x1127F876D49BB635ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\9ww_shield.mao"},
	{0x11283E6E36DD224BULL, "zz_amb_beggar.utc"},
	{0x1128478467E104EAULL, "vxlt_0_lowtown_a3_n_c_hdc_0_0.vlm"},
	{0x1128E047CBA99779ULL, "dae000_tranquil_1.bnk"},
	{0x1128F12EDE8700F3ULL, "llac_ca_caverns1_24421_7.dds"},
	{0x1129233AB8F89956ULL, "t3_shl_grn.tnt"},
	{0x1129499092AEC055ULL, "984315696.ogg"},
	{0x1129A5715FCA5746ULL, "dre000_varric6029149.cl"},
	{0x1129D0D55A7815C6ULL, "vxlt_0_lowtown_docks_n_b6q_0_0.vlm"},
	{0x112A27DBCBC6AE91ULL, "hlac_dr_0_deeproads_5845_5z.dds"},
	{0x112A5DFB8A78D987ULL, "vis121_saemus6089667.cl"},
	{0x112B141EC9A6008DULL, "575656980.ogg"},
	{0x112BB7AEFFE30AA6ULL, "vxw_bonepit_nd_0_0.vlm"},
	{0x112BC282435A2C27ULL, "6135762_f.fxe"},
	{0x112C0247CBAC06D9ULL, "dae000_tranquil_1.cnv"},
	{0x112C9B0ED9DEE39BULL, "w_bonepitdestroyed_a3_52_0.mmh"},
	{0x112CA70ED9DEF838ULL, "w_bonepitdestroyed_a3_52_0.mao"},
	{0x112CB90ED9DF1685ULL, "w_bonepitdestroyed_a3_52_0.msh"},
	{0x112CCF1BACB653D4ULL, "art\\vfx\\combat\\vfx_itm_tar_bomb_proj_p.cess.evt"},
	{0x112CF7B6999C8326ULL, "vxca_cavern1_haunted_ii0_0_0.vlm"},
	{0x112D62473EB8D5C1ULL, "506594307.ogg"},
	{0x112DE601B2E58311ULL, "vxlt_darktown_2rw_0_0.vlm"},
	{0x112E21DE58465596ULL, "6036991_m.fxe"},
	{0x112E53892044E93DULL, "lgt220_shp_mag_guard6149521.cl"},
	{0x112F1E17E9C3172FULL, "135874654.ogg"},
	{0x112F2945A87D3997ULL, "dae000_epic_merchant_36179506.cl"},
	{0x112F5A32AD5A2DCBULL, "364452715.ogg"},
	{0x112F6AC16E9CFAD1ULL, "vxlt_gallowscourtyard_n_76k_0_0.vlm"},
	{0x112F78A3D515E153ULL, "vxlt_warehouse_n_sx_0_0.vlm"},
	{0x112FE27C9B593116ULL, "sp_module_item_lost.ncs"},
	{0x11306F8BF36847D5ULL, "art\\characters\\playercharacter\\humanfemale\\animation\\ambient\\dialogue\\lean_back_wall_loop.gad"},
	{0x1130AF66F1DC9F2CULL, "ser121_jethann6104552.cl"},
	{0x1130CF389B005B2FULL, "50291781.ogg"},
	{0x1131738CEA5563A9ULL, "6018670_f.fxe"},
	{0x11318C9627C33D7EULL, "lt_0_lowtown_53_0_ll.mmh"},
	{0x1131CF639FC46A9CULL, "plt_fen100pt_recruited.nss"},
	{0x1131D1F5CCE7DA8FULL, "llac_lt_0_lowtown_docks_a3_-4_5.dds"},
	{0x1131ECF0505374F5ULL, "gen00fl_aveline6105437.cl"},
	{0x1132253C586DBDF8ULL, "fighting70a.mao"},
	{0x11327F8BEA22D39CULL, "lt_0_lowtown_docks_n.arl"},
	{0x113296DF763756D1ULL, "ht_bartrand_n_3364.anb"},
	{0x1132A046681255AFULL, "vxlt_warehouse_10a_0_0.vlm"},
	{0x1132F6F069E1B332ULL, "6175070_m.fxe"},
	{0x1133223A3ABC5604ULL, "vxw_sundermount_2m0_0_0.vlm"},
	{0x11332A852B4965A6ULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\animation\\gallowint_basicdoor.closedtoopen2.evt"},
	{0x113349CA84E05871ULL, "hlac_w_blightlands_65539_1y.dds"},
	{0x113622CA4B99FCFEULL, "vxca_caverns1_mines_icx_0_2.vlm"},
	{0x11364EB4B4802A4DULL, "vxlt_0_lowtown_n_eoc_0_0.vlm"},
	{0x113672D9C724708EULL, "468377537.ogg"},
	{0x1136A4B75C324E67ULL, "dae150_chandan6148796.cl"},
	{0x1136B9A3D1E9CFE5ULL, "6209853_m.fxe"},
	{0x1136C5DA357316BEULL, "6116231_m.fxe"},
	{0x1137193258BC7B76ULL, "w_bonepitmining_a2_16_0.rml"},
	{0x11372D2F437D790CULL, "430181720.ogg"},
	{0x1137510DCC5B75CAULL, "6108746_m.fxe"},
	{0x113770309E092283ULL, "vxlt_undercitydungeon_190_0_0.vlm"},
	{0x1137AAF332266ADFULL, "dre000_vampire6167443.cl"},
	{0x1137D14CE1825A9DULL, "vxlt_0_lowtown_a2_gne_0_0.vlm"},
	{0x1137F249C30C006CULL, "1034369215.ogg"},
	{0x11385665AF7989E3ULL, "vxca_cavern1_haunted_fmz_0_0.vlm"},
	{0x1138AB54CD865F18ULL, "dae000_bartrand6027550.cl"},
	{0x1138E18E3D2D7080ULL, "46845371.ogg"},
	{0x11392CA9A2605A1CULL, "6059428_m.fxe"},
	{0x113941852B4DF8F1ULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\animation\\gallowint_basicdoor.closedtoopen2.gad"},
	{0x113973E894BAFA4EULL, "vxw_bonepitmining_a2_169_0_0.vlm"},
	{0x1139BCCE9E8DDA66ULL, "art\\characters\\weapons\\dagger\\textures\\w_dgr_dg102a_0d.dds"},
	{0x1139C71BACC0D00DULL, "art\\vfx\\combat\\vfx_itm_tar_bomb_proj_p.cess.ani"},
	{0x113A1561545817E2ULL, "173653223.ogg"},
	{0x113B0A7AB3E9609CULL, "follower_banter6161090.cl"},
	{0x113BA605E407BA65ULL, "475161832.ogg"},
	{0x113BB7BEA3B48E02ULL, "vxdr_0_deeproads_eha_0_0.vlm"},
	{0x113C2D3D6DBAF377ULL, "plt_cod_ltr_orsino_thank.nss"},
	{0x113C9CF2D269E743ULL, "hlac_lt_0_lowtown_131073_0y.dds"},
	{0x113CFCD74A775445ULL, "6042488_f.fxe"},
	{0x113D664B94A9647CULL, "vxw_encounter2_bj_0_0.vlm"},
	{0x113D674E565EFC8CULL, "vxlt_hangedman_n_1xi_0_0.vlm"},
	{0x113E16A7EBF38E21ULL, "vxlt_warehouse_n_ad_0_0.vlm"},
	{0x113E6436B94995FAULL, "vxw_woundedcoast_ne_0_0.vlm"},
	{0x113FD645CAD5B6FCULL, "vxht_mansion2_1bt_0_2.vlm"},
	{0x113FFAD2730387AEULL, "81754052.ogg"},
	{0x11401D28AAFDB8E3ULL, "6041747_m.fxe"},
	{0x114042852B54189FULL, "art\\levels\\races\\ferelden\\kirkgallows\\int\\4co_gallowsint\\animation\\gallowint_basicdoor.closedtoopen2.ani"},
	{0x114056FFBEB58CCFULL, "dae000ip_crowd_ff8_talk4.utp"},
	{0x1140F125CA703005ULL, "llac_lt_0_lowtown_a2_n_c_-1_3.dds"},
	{0x1141745F2995B37FULL, "follower_banter6172972.cl"},
	{0x114176FFB749D6BCULL, "w_woundedcoast_39_0.anb"},
	{0x1141955AA51F3F66ULL, "vxw_encounter2_5i_0_0.vlm"},
	{0x1142542531EFD27BULL, "vxlt_warehouse_r6_0_0.vlm"},
	{0x11428AEABE8887ABULL, "gen00fl_sebastian6177919.cl"},
	{0x1143B64F0AD5F0EFULL, "follower_barks6153439.cl"},
	{0x11442B4AEEFC558DULL, "6160866_m.fxe"},
	{0x11445D2DE106C433ULL, "756817694.ogg"},
	{0x114504D19025E253ULL, "732050374.ogg"},
	{0x114552913D81B948ULL, "em_orsino.mor"},
	{0x114574215A77568BULL, "vxca_caverns1_dnh_0_0.vlm"},
	{0x1145B9A75923B76CULL, "mer341_earl6054259.cl"},
	{0x1145DBF333727849ULL, "158126299.ogg"},
	{0x11466E91AC443A94ULL, "hlac_lt_0_lowtown_docks_a3_131070_3z.dds"},
	{0x114675F3FB057A7AULL, "vxht_0_hightown_a2_c75_0_0.vlm"},
	{0x11476FBB6DE565D8ULL, "vxca_cavern1_haunted_ck2_0_0.vlm"},
	{0x1148292EEF84E217ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\9ff_rock3a.mmh"},
	{0x11482D2EEF84E924ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\9ff_rock3a.mao"},
	{0x11483F2EEF850789ULL, "art\\levels\\races\\natural\\kirkcave\\setint\\caveinterior\\9ff_rock3a.msh"},
	{0x11484DB5F24D34CDULL, "363188930.ogg"},
	{0x114855DF4955AAD4ULL, "496896455.ogg"},
	{0x114877035AB95349ULL, "pro000_friedrich6050185.cl"},
	{0x1148BEB188BE9A6CULL, "6018558_f.fxe"},
	{0x1148E8DE04B9B174ULL, "ave200_donnic6110769.cl"},
	{0x114A30C0B2D1F6AEULL, "vxht_0_hightown_a3_c3l_0_0.vlm"},
	{0x114AA90CEDF7C260ULL, "6119698_m.fxe"},
	{0x114AB27B7FBE5031ULL, "6059991_m.fxe"},
	{0x114ACFB000A416CDULL, "467915841.ogg"},
	{0x114AE9A920A4CAE4ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\0ww_2windows.phy"},
	{0x114BA874CFAAAC62ULL, "6147909_m.fxe"},
	{0x114BB983FD3F99F3ULL, "6050179_f.fxe"},
	{0x114BD828869F7CFCULL, "vxw_encounter2_mk_0_0.vlm"},
	{0x114C05BA21CEA2CAULL, "hlac_ht_0_hightown_a2_65535_0y.dds"},
	{0x114C52C88060B840ULL, "hlac_w_encounter3_-131074_0x.dds"},
	{0x114CD613AB3FAE7AULL, "vxw_woundedcoast_kf_0_0.vlm"},
	{0x114CD93D4C073730ULL, "vxlt_0_lowtown_a2_n_c_gcn_0_0.vlm"},
	{0x114D5C0704E8EAC0ULL, "107933667.ogg"},
	{0x114D7845CFA567BCULL, "follower_banter6149247.cl"},
	{0x114E2CBABD6550FFULL, "art\\levels\\ztools\\default\\tilesets\\setext\\_textures\\xx_extfloor.dds"},
	{0x114F84BA6E5325C6ULL, "6129721_m.fxe"},
	{0x114FD5C894234B77ULL, "gen00fl_merrill6130135.cl"},
	{0x1150124C72978CF1ULL, "396806395.ogg"},
	{0x1150434F07ACF2AAULL, "follower_banter6155514.cl"},
	{0x1150A31AB93F946CULL, "405690248.ogg"},
	{0x1150AE1177DD3740ULL, "lowtown_fem_alienage1_sta6162564.cl"},
	{0x1150B1A6FD20CBC4ULL, "vxw_bonepit_1k6_0_0.vlm"},
	{0x1150CC7BFA62EF63ULL, "472364353.ogg"},
	{0x1150EBE119B98DACULL, "867814080.ogg"},
	{0x1151B9768C3F5E0FULL, "6041219_m.fxe"},
	{0x1152070B5E8CF6D4ULL, "zz_dae_debug6029905.cl"},
	{0x11520B97B2FAD128ULL, "hlac_lt_foundry_n_320_14x.dds"},
	{0x11521172C8ADCCD7ULL, "5952820.ogg"},
	{0x11522AFC04FA89E2ULL, "952238458.ogg"},
	{0x1152509AD783D13BULL, "vxlt_0_lowtown_docks_a3_d01_0_1.vlm"},
	{0x11528EF3B56C7B1EULL, "art\\characters\\playercharacter\\humanmale\\textures\\hn_arm_lgtb_0s.dds"},
	{0x1153185E37EDA7C6ULL, "1071436600.ogg"},
	{0x1153D5F49B62C41BULL, "dae000_meredith6085249.cl"},
	{0x1154A654F5723863ULL, "fen200ar_ambush_exterior.are"},
	{0x1154C02C6F327917ULL, "6017667_m.fxe"},
	{0x115565D151E3B4D3ULL, "859068641.ogg"},
	{0x1155B022498CCB33ULL, "hm_gen_high_mage.mor"},
	{0x1155D91ADE7B1B05ULL, "6217700_m.fxe"},
	{0x11561544473BEA58ULL, "vxht_keep_n_70e_0_0.vlm"},
	{0x1156904A5B445419ULL, "llac_ht_bartrand_n_3366_0.dds"},
	{0x1157356D6CCA8260ULL, "zz_ser_debug6085774.cl"},
	{0x1157565D24864BC0ULL, "402958725.ogg"},
	{0x11576E6CD86FEE29ULL, "vxht_brothel_n_6ee_0_0.vlm"},
	{0x1157A7461598EFF7ULL, "6139674_f.fxe"},
	{0x1157DF4640BDE11DULL, "ser191_emeric6120416.cl"},
	{0x115818808932747FULL, "752111606.ogg"},
	{0x1159245871B468B1ULL, "6056891_m.fxe"},
	{0x1159C4AB887952CDULL, "hlac_lt_gallowstemplar_2388_4z.dds"},
	{0x1159D6373A2FD22EULL, "dae000cr_grand_cleric6182015.cl"},
	{0x1159DD78867FAB1AULL, "hlac_ht_0_hightown_a2_-65536_2y.dds"},
	{0x115A0F83D9E742D4ULL, "6050409_m.fxe"},
	{0x115A4A057895957CULL, "vxw_encounter2_hh_0_0.vlm"},
	{0x115A4B083A4B2D8CULL, "vxlt_hangedman_n_1ro_0_0.vlm"},
	{0x115A74CCBF4BCE1DULL, "325821336.ogg"},
	{0x115A86D4D76C8AA9ULL, "vxw_blightlands_7s_0_0.vlm"},
	{0x115A8AC7DB8A07D2ULL, "vxlt_darktown_4qg_0_0.vlm"},
	{0x115A8D5DFA39A3C0ULL, "885573967.ogg"},
	{0x115A9DC04A36844FULL, "art\\levels\\races\\ferelden\\props\\plotspecifc\\prp_codexwild.phy"},
	{0x115AF8D5C6BD73A0ULL, "vxw_blightlands_j1_0_0.vlm"},
	{0x115B15BF399BF77EULL, "art\\vfx\\combat\\vfx_m_blood_charge_c.dura.ani"},
	{0x115B2C8C80D31FCBULL, "674832085.ogg"},
	{0x115B34CEDE86CF5CULL, "vxdr_primevaltomb_82c_0_1.vlm"},
	{0x115B8913513063A2ULL, "art\\characters\\playercharacter\\elfmale\\har_fls.phy"},
	{0x115C8780235A2836ULL, "pro000_colonel_ewald6047380.cl"},
	{0x115D2460E7590DD7ULL, "vxlt_darktown_5ak_0_0.vlm"},
	{0x115D314B2BE3FBF0ULL, "623505397.ogg"},
	{0x115D547885B179D0ULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_paper1b.mao"},
	{0x115D587885B180E3ULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_paper1b.mmh"},
	{0x115D667885B1983DULL, "art\\levels\\races\\ferelden\\props\\middleclass\\prp_paper1b.msh"},
	{0x115DAFB6034CB820ULL, "vxlt_0_lowtown_a3_n_c_bza_0_0.vlm"},
	{0x115DCF2653E38EA6ULL, "6088855_m.fxe"},
	{0x115DFA1EFF64385AULL, "vxlt_foundry_n_zl_0_0.vlm"},
	{0x115E9E3DF665ED60ULL, "vxlt_0_lowtown_a2_n_d06_0_0.vlm"},
	{0x115EBD5A88BE407BULL, "hlac_lt_0_lowtown_docks_a3_n_-131072_0z.dds"},
	{0x115EDBE0AD26EDB4ULL, "6173618_m.fxe"},
	{0x115F2383EF16E444ULL, "vxw_bonepit_1l7_0_0.vlm"},
	{0x115F54115A753DE2ULL, "vxdr_primevaltomb_34q_0_0.vlm"},
	{0x115FC82323B9C7C7ULL, "186872759.ogg"},
	{0x115FEFCACA3FCCD8ULL, "gen00fl_anders6125769.cl"},
	{0x1160202C6CAB5980ULL, "art\\vfx\\materials\\vfx_d_shockwave_blend.mao"},
	{0x1160803B6C251935ULL, "character_character_p150.psh.fxo.dx11.assembly"},
	{0x11610CA368FEE632ULL, "778434797.ogg"},
	{0x11611345D82499DAULL, "framebuffereffects_downsample_p0.psh.fxo.dx11.assembly"},
	{0x11611705AE6081D8ULL, "genip_cin_arrow.utp"},
	{0x1161362CB8C790F5ULL, "6077230_m.fxe"},
	{0x116178794FEB3430ULL, "zz_prmpt_debug6200934.cl"},
	{0x1161B99DE545F664ULL, "vxlt_0_lowtown_bz9_0_0.vlm"},
	{0x116257DB3E63878BULL, "vxca_caverns1_dpj_0_0.vlm"},
	{0x116274EFFBBBB5A3ULL, "6142763_m.fxe"},
	{0x1163108E63B4D344ULL, "226865289.ogg"},
	{0x1163401D555D9756ULL, "hlac_lt_0_lowtown_docks_n_-4_4x.dds"},
	{0x116359ADDEF1AB7AULL, "vxht_0_hightown_a2_c97_0_0.vlm"},
	{0x1163AC35A119D333ULL, "22830499.ogg"},
	{0x1163F10C5DEF197DULL, "art\\characters\\heraldry\\hld_seekers.dds"},
	{0x1164107B2A804F73ULL, "follower_barks6153563.cl"},
	{0x11641ADAF2335DDDULL, "vxlt_darktown_5cc_0_0.vlm"},
	{0x11646ABF258FCFEAULL, "153050626.ogg"},
	{0x116491D12480BEB2ULL, "115604014.ogg"},
	{0x1164E7FF10B634B4ULL, "vxlt_gallowstemplar_f_5jf_0_0.vlm"},
	{0x1164F24F96A2FA0DULL, "hairalpha_hair_p0.psh.fxo"},
	{0x11655A11B75572C3ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\pose_eb_transition.evt"},
	{0x11656A56216F938DULL, "vxlt_0_lowtown_a2_n_gz2_0_0.vlm"},
	{0x116699017FE6E9B1ULL, "1044349752.ogg"},
	{0x11677224A81F5B83ULL, "604170302.ogg"},
	{0x1167A399639A3935ULL, "dae000_unhappy_mage.bnk"},
	{0x1167E2C120FB5316ULL, "ico_helm_heavy_5.dds"},
	{0x1167E765795EF22AULL, "art\\characters\\playercharacter\\humanmale\\_hm_hlf_masb.mao"},
	{0x11683E759C0133B5ULL, "115176259.ogg"},
	{0x1168574D4E93D281ULL, "6053489_m.fxe"},
	{0x11687D199FD532AEULL, "6205674_m.fxe"},
	{0x1168C376D4D2F339ULL, "art\\levels\\races\\ferelden\\kirkhigh\\ext\\4co_hightown\\9ww_shield.phy"},
	{0x1168F8B1C962A329ULL, "vxw_blightlands_8r_0_0.vlm"},
	{0x116919FC1B5E5204ULL, "art\\characters\\weapons\\longswords\\textures\\w_lsw_ls103a_0d.dds"},
	{0x11692DBF39A85D4FULL, "art\\vfx\\combat\\vfx_m_blood_charge_c.dura.evt"},
	{0x11695BDABDBE2545ULL, "vxw_woundedcoast_nn_0_0.vlm"},
	{0x1169823339335241ULL, "6208471_f.fxe"},
	{0x1169830A24647C18ULL, "ave200_donnic6110633.cl"},
	{0x1169AFB4DDAB1E52ULL, "6068695_m.fxe"},
	{0x1169BCF72FF36830ULL, "vxlt_0_lowtown_a2_n_c_gyl_0_0.vlm"},
	{0x116AC599639CA88DULL, "dae000_unhappy_mage.cnv"},
	{0x116B963DD94F2657ULL, "vxlt_darktown_5jj_0_0.vlm"},
	{0x116C009F5D93139AULL, "lav_river03a.dds"},
	{0x116C3311B75B4F44ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\pose_eb_transition.gad"},
	{0x116C410345D9A726ULL, "6083955_m.fxe"},
	{0x116CEE53261535E3ULL, "6148372_m.fxe"},
	{0x116CF91159325A2BULL, "vxht_keep_6v6_0_0.vlm"},
	{0x116CFFDC877D25ABULL, "6013843_m.fxe"},
	{0x116D392CDF99076DULL, "785948541.ogg"},
	{0x116D3E4DA545BD66ULL, "vxw_blightlands_ha_0_0.vlm"},
	{0x116D71C3ECF8C547ULL, "6084340_m.fxe"},
	{0x116DC0DC829F7428ULL, "vxht_bartrand_n_6qt_0_0.vlm"},
	{0x116EF2E15263355AULL, "character_character_p166.psh.fxo.dx11.assembly"},
	{0x116F17660A37813DULL, "vxht_playershouse_n_1zy_0_0.vlm"},
	{0x116F853E3984D003ULL, "vxca_cavern1_haunted_e8n_0_0.vlm"},
	{0x116F8DCEF53FA6D1ULL, "vxw_encounter3_fm_0_1.vlm"},
	{0x116FA3236C83ADA3ULL, "ave310_brennan6136882.cl"},
	{0x116FA809AABDA975ULL, "6078330_m.fxe"},
	{0x116FBE979511F372ULL, "vxht_bartrand_n_735_0_0.vlm"},
	{0x116FBFE16A243ABBULL, "mer240cr_miner3.utc"},
	{0x116FC4059D74045AULL, "vxlt_gallowstemplar_f_5qe_0_0.vlm"},
	{0x116FE3ED793BF0F5ULL, "57148215.ogg"},
	{0x11702EBF39AE7DF8ULL, "art\\vfx\\combat\\vfx_m_blood_charge_c.dura.gad"},
	{0x11708DE8D8485A19ULL, "vxlt_0_lowtown_docks_a3_czg_0_0.vlm"},
	{0x11709920917739A9ULL, "hlac_dr_primevaltomb_664_5x.dds"},
	{0x1170E1731B0CC917ULL, "vxlt_gallowstemplar_3qu_0_0.vlm"},
	{0x1171243312DC1A39ULL, "hlac_lt_undercitydungeon_-1_24z.dds"},
	{0x117154813E042AAFULL, "6173019_m.fxe"},
	{0x1171583EED56AFA4ULL, "zz_eric_soundtst.cut"},
	{0x11715ABF6B3D3D9CULL, "mcr371_scared_mage_m6086908.cl"},
	{0x1171C1C65AD80B22ULL, "hlac_dr_0_deeproads_5838_0z.dds"},
	{0x117210FD7BB546CEULL, "gen00fl_isabela6096801.cl"},
	{0x11723A862DC58C99ULL, "6081415_m.fxe"},
	{0x11724211B75FD422ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\pose_eb_transition.ani"},
	{0x1172F5CEAB8CA4DFULL, "art\\characters\\playercharacter\\humanfemale\\boo_medd.phy"},
	{0x1172FE9AA90D6DCEULL, "vxht_0_hightown_a2_cbc_0_1.vlm"},
	{0x117359DC02AC4D34ULL, "vxlt_gallowstemplar_f_5mg_0_0.vlm"},
	{0x11745226BC5C1F29ULL, "vxht_brothel_n_6kk_0_0.vlm"},
	{0x117457CFB91EC029ULL, "llac_lt_0_lowtown_-1_2.dds"},
	{0x11745FC895CFFB27ULL, "dae000_arishok6072820.cl"},
	{0x1174B25FCB38FE0EULL, "vxw_sundermount_371_0_0.vlm"},
	{0x1174E4325AF49712ULL, "498147362.ogg"},
	{0x1174F9BC7119CF4EULL, "vxdr_0_deeproads_ewd_0_0.vlm"},
	{0x11755ED1653C9883ULL, "gen00fl_anders6125478.cl"},
	{0x11756489553EBB49ULL, "hlac_lt_0_lowtown_docks_a3_65532_1x.dds"},
	{0x1175834FFC2E8049ULL, "266667253.ogg"},
	{0x11761F18FAEA3C0AULL, "61332704.ogg"},
	{0x117676AFFDDB148DULL, "vxca_caverns2_8g7_0_1.vlm"},
	{0x1176B3442B0CE6ACULL, "art\\cinematics\\additional animation\\cs_femnoblei_cinematicanim.gad"},
	{0x1176E7BBE914F4EFULL, "188291091.ogg"},
	{0x1177A21D1F3A3B42ULL, "ser270_cut_aftermath6162279.cl"},
	{0x1177CDB7AFB43DC5ULL, "vxw_woundedcoast_km_0_0.vlm"},
	{0x1177D44299569A11ULL, "designertags.gda"},
	{0x1177F40A2C2EBF61ULL, "gen_im_npc_wep_shd_gena.uti"},
	{0x117800FD81BF4990ULL, "6104098_m.fxe"},
	{0x11784D6A7C8F29D0ULL, "vxht_mansion2b_27r_0_0.vlm"},
	{0x117888BC91BC1D98ULL, "hlac_ca_caverns1_24424_27z.dds"},
	{0x1178AE6AB3194421ULL, "6117284_m.fxe"},
	{0x1178C4351DB1FEE4ULL, "vxht_0_hightown_a2_a7u_0_0.vlm"},
	{0x1178E52D646F5F66ULL, "ser221_gascard6149263.cl"},
	{0x1178F6522D4F76A8ULL, "lgt150_nat_were_supporter6111009.cl"},
	{0x117969FADA6DE59AULL, "6110612_f.fxe"},
	{0x117A081ACB453ED7ULL, "vxlt_darktown_5oe_0_0.vlm"},
	{0x117A9B0EDA210528ULL, "w_bonepitdestroyed_a3_52_0.rml"},
	{0x117B06EE59F7C2E3ULL, "vxlt_0_lowtown_n_byt_0_0.vlm"},
	{0x117B3B0C2DDAA9BDULL, "hlac_lt_0_lowtown_docks_n_-2_5z.dds"},
	{0x117B47B34AC56F7CULL, "dae011_badpoet_sta6145184.cl"},
	{0x117B48D655C3FD7AULL, "vxw_bonepit_1vo_0_0.vlm"},
	{0x117B58380C9113B7ULL, "zz_rain_o_arrows.cut"},
	{0x117BB02A973BD5E6ULL, "vxw_blightlands_mb_0_0.vlm"},
	{0x117C9875D56E5F58ULL, "gen00fl_isabela6090320.cl"},
	{0x117C99D43288845BULL, "6005822_m.fxe"},
	{0x117D8942FC2D99BDULL, "vxht_playershouse_n_1qx_0_0.vlm"},
	{0x117DB2442B1303AAULL, "art\\cinematics\\additional animation\\cs_femnoblei_cinematicanim.ani"},
	{0x117DB280147FECE0ULL, "mage_petrify.dds"},
	{0x117E45268DB3D582ULL, "vxlt_darktown_585_0_0.vlm"},
	{0x117E97616450CA89ULL, "lgt160cr_ntb_elf_dalish.utc"},
	{0x117EFE512994736FULL, "6041549_m.fxe"},
	{0x117F2796D519BFD8ULL, "6031743_m.fxe"},
	{0x117F4693F2A7AD8EULL, "hlac_ca_caverns2_5802_57y.dds"},
	{0x117F591C179DF60FULL, "607565516.ogg"},
	{0x117F6A6808638372ULL, "6076353_m.fxe"},
	{0x117F73A2379DFE06ULL, "art\\vfx\\combat\\vfx_m_sta_fire_range_f_c.anb"},
	{0x1180178857BC01A6ULL, "vxdr_primevaltomb_877_0_0.vlm"},
	{0x118019064F19FF22ULL, "ser221_gamlen6135300.cl"},
	{0x11801BBA7D2291B0ULL, "one000_ilen6205660.cl"},
	{0x11809693C43ABDA9ULL, "mag110_cut_abomination6037102.cl"},
	{0x1180AF54F5978248ULL, "fen200ar_ambush_exterior.lst"},
	{0x1181207EC3A94B9FULL, "6155529_m.fxe"},
	{0x1181280892DE1730ULL, "qun221_viscount6069183.cl"},
	{0x11817013515108F5ULL, "art\\characters\\playercharacter\\elfmale\\har_fls.mmh"},
	{0x1181721351510C6BULL, "art\\characters\\playercharacter\\elfmale\\har_fls.msh"},
	{0x1182109451F5EC08ULL, "art\\characters\\playercharacter\\elfmale\\mt_art_em02.mmh"},
	{0x11822E9451F61F32ULL, "art\\characters\\playercharacter\\elfmale\\mt_art_em02.msh"},
	{0x11825E6393106DA1ULL, "6135662_m.fxe"},
	{0x118275FFB7816C4CULL, "w_woundedcoast_39_0.rml"},
	{0x11829064A10A390CULL, "6087763_f.fxe"},
	{0x1182CAFDC2238708ULL, "art\\levels\\4cinematics\\qunari\\prp_qunariamulet.mmh"},
	{0x1182D6FDC2239BABULL, "art\\levels\\4cinematics\\qunari\\prp_qunariamulet.mao"},
	{0x1182D818A8B7F14CULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\walk_forward_1_step.ani"},
	{0x1182E22304E7E300ULL, "148742565.ogg"},
	{0x1182E8FDC223BA32ULL, "art\\levels\\4cinematics\\qunari\\prp_qunariamulet.msh"},
	{0x1183037BD25CA996ULL, "mag170_thrask6054079.cl"},
	{0x11830731F3B2F5EBULL, "hlac_lt_0_lowtown_a2_65535_6x.dds"},
	{0x1183243CBD2F168EULL, "vxw_sundermount_322_0_0.vlm"},
	{0x11836B99630FE7CEULL, "vxdr_0_deeproads_epe_0_0.vlm"},
	{0x1183E279AAB82F37ULL, "hlac_lt_0_lowtown_a2_65536_5x.dds"},
	{0x1183F6CEAB9B19E4ULL, "art\\characters\\playercharacter\\humanfemale\\boo_medd.mmh"},
	{0x1183FACEAB9B2037ULL, "art\\characters\\playercharacter\\humanfemale\\boo_medd.mao"},
	{0x118408C83F5964C8ULL, "6162339_m.fxe"},
	{0x11840CCEAB9B3E86ULL, "art\\characters\\playercharacter\\humanfemale\\boo_medd.msh"},
	{0x11843CF35063A5DFULL, "6042373_m.fxe"},
	{0x11845638D6FEB7E4ULL, "6179854_m.fxe"},
	{0x118538FDBB6AEBF4ULL, "6177849_m.fxe"},
	{0x11858223606AD34FULL, "vxht_chantry_262_0_0.vlm"},
	{0x11858F9ED4D0F681ULL, "dae000_orsino6085515.cl"},
	{0x118620FBF3061C11ULL, "vxlt_undercityencounter_1fj_0_0.vlm"},
	{0x118642B53964CED7ULL, "357787736.ogg"},
	{0x1186580069F805A5ULL, "dae000_mother6101038.cl"},
	{0x11866BAE02EA53ACULL, "hlac_w_bonepit_393221_0x.dds"},
	{0x11869D877318107AULL, "vxw_woundedcoast_wb_0_0.vlm"},
	{0x1186BF476E854250ULL, "vxht_mansion2b_20s_0_0.vlm"},
	{0x1187103D6DFAAD58ULL, "plt_cod_ltr_orsino_thank.plo"},
	{0x1187310325BE2C7EULL, "6098937_m.fxe"},
	{0x1187AF54F59DA099ULL, "fen200ar_ambush_exterior.ncs"},
	{0x118819424D777F6BULL, "1024576961.ogg"},
	{0x11882B4B579FBFF9ULL, "6101229_m.fxe"},
	{0x11886E2CDF54BE5BULL, "dae150_harshal6148875.cl"},
	{0x1188E302C9743436ULL, "hlac_lt_undercityencounter_-1_50y.dds"},
	{0x1189E063A00EFDC7ULL, "plt_fen100pt_recruited.plo"},
	{0x1189E518A8BE258EULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\walk_forward_1_step.gad"},
	{0x118A22078931EE66ULL, "vxw_blightlands_vc_0_0.vlm"},
	{0x118A28DF34D82250ULL, "184899696.ogg"},
	{0x118AC96E8F993AFBULL, "mcr310_argument6075090.cl"},
	{0x118B37A1F6033003ULL, "hlac_w_bonepitmining_a2_196612_0z.dds"},
	{0x118B493ABF037C6CULL, "6097905_m.fxe"},
	{0x118B7FACE052AD47ULL, "6027574_m.fxe"},
	{0x118BD111761A29C7ULL, "6108497_m.fxe"},
	{0x118D2A125A2B9444ULL, "587919227.ogg"},
	{0x118D2EFB0E970640ULL, "47340202.ogg"},
	{0x118D76A237AA40B1ULL, "art\\vfx\\combat\\vfx_m_sta_fire_range_f_c.mmh"},
	{0x118D8D9B71F1F7D7ULL, "533844651.ogg"},
	{0x118D8E051C81BD36ULL, "6106612_m.fxe"},
	{0x118DCFCE32DF86EAULL, "follower_banter6172728.cl"},
	{0x118E896549B21A26ULL, "vxdr_primevaltomb_828_0_0.vlm"},
	{0x118EF77676240EB8ULL, "332993883.ogg"},
	{0x118F0EE5EE528B49ULL, "6104882_m.fxe"},
	{0x118F526A9E7ACDFFULL, "art\\characters\\playercharacter\\humanmale\\animation\\blendtree\\pp_male_mage.anb"},
	{0x118F59B7D2A07661ULL, "lgt320_shw_ors_marlein6156376.cl"},
	{0x118FC9BD4C8D628DULL, "art\\levels\\4cinematics\\ferelden\\_textures\\prp_moneypouchb_d.dds"},
	{0x118FE018A8C2896DULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\dialogue\\walk_forward_1_step.evt"},
	{0x119025E5180CED91ULL, "119174024.ogg"},
	{0x11902A1B3F2B800FULL, "llac_ht_brothel_n_2995_3.dds"},
	{0x11904E05C2D0AFA0ULL, "zz_vault_debug6203178.cl"},
	{0x11911B4041533FD2ULL, "art\\vfx\\combat\\vfx_r_dlt_duelling_loop_c.mmh"},
	{0x1191732F2FED3C9AULL, "vxlt_0_lowtown_a2_n_c_eje_0_0.vlm"},
	{0x1191C7CE9F9B108EULL, "6030491_f.fxe"},
	{0x11922F8637241A2BULL, "vxlt_warehouse_ce_0_0.vlm"},
	{0x11929D22246190B5ULL, "645747309.ogg"},
	{0x1192E7CAC14BE785ULL, "one280_emrys6099750.cl"},
	{0x11935A789038A54DULL, "llac_dr_slavers_3216_10.dds"},
	{0x11939F7C09BEBF1BULL, "one200_gold_followers6101136.cl"},
	{0x11944B8358AB8FD7ULL, "plt_cod_ltr_sk_starkhaven.plo"},
	{0x119566B68EA04A60ULL, "6076319_m.fxe"},
	{0x119637CEC76C5D41ULL, "vxht_0_hightown_4n2_0_0.vlm"},
	{0x11972E3E5BA4B965ULL, "6135635_m.fxe"},
	{0x11977ABAF7B7E622ULL, "vxht_0_hightown_6pm_0_0.vlm"},
	{0x1197B079A80FA10BULL, "vxw_bonepit_2al_0_0.vlm"},
	{0x1197C46DA1708599ULL, "hlac_ca_caverns1_mines_24426_10z.dds"},
	{0x1197D49606A0595DULL, "hlac_lt_0_lowtown_docks_a3_n_-1_3x.dds"},
	{0x1198B255E19D616EULL, "328205121.ogg"},
	{0x11990B32590F2DC3ULL, "w_bonepitmining_a2_16_0.msh"},
	{0x119915BC8B297F16ULL, "dae000ip_crowd_lm5_talk18.utp"},
	{0x11991932590F451DULL, "w_bonepitmining_a2_16_0.mmh"},
	{0x11991D32590F4C2EULL, "w_bonepitmining_a2_16_0.mao"},
	{0x11994362D403E3D3ULL, "vxlt_darktown_4hi_0_0.vlm"},
	{0x1199456063597D84ULL, "lt_0_lowtown_docks_n_210_0.mmh"},
	{0x11994960635984D7ULL, "lt_0_lowtown_docks_n_210_0.mao"},
	{0x11995B606359A326ULL, "lt_0_lowtown_docks_n_210_0.msh"},
	{0x11997A8B2E5889D0ULL, "rdr111dg_martin6053592.cl"},
	{0x1199BC19C24FEEBCULL, "hlac_lt_undercityserialkiller_-1_17x.dds"},
	{0x1199EC698BD5E06FULL, "6033768_m.fxe"},
	{0x119A4F39E20F589CULL, "follower_banter6170197.cl"},
	{0x119A61A766277867ULL, "100455776.ogg"},
	{0x119AB43A26E60E25ULL, "533247878.ogg"},
	{0x119AB6771F770680ULL, "fxa_rocka.dds"},
	{0x119ADAD50F671983ULL, "vxca_cavern1_haunted_e7s_0_0.vlm"},
	{0x119AF9C948D29364ULL, "395336632.ogg"},
	{0x119B19607F23A7B1ULL, "follower_banter6141738.cl"},
	{0x119B9022FA756C86ULL, "vxca_caverns1_iai_0_0.vlm"},
	{0x119BA62635EAB525ULL, "isa311_isabela6092512.cl"},
	{0x119BE80FD2D6EEB4ULL, "nat380_nathaniel_banter6090043.cl"},
	{0x119C8B6A3A4B7FDEULL, "vxlt_gallowscourtyard_745_0_0.vlm"},
	{0x119C8CDA56AB329CULL, "6083760_m.fxe"},
	{0x119C9C9405D1179DULL, "hlac_w_blightlands_327682_0z.dds"},
	{0x119CA491ED43A978ULL, "718719559.ogg"},
	{0x119CDC5F1090653BULL, "llac_lt_gallowstemplar_n_2388_1.dds"},
	{0x119D463AA3D98568ULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_barge03_sa.dds"},
	{0x119D9022CEAC8C9FULL, "w_blightlands_171_0.phy"},
	{0x119DD79E2E76AF4BULL, "6155312_m.fxe"},
	{0x119DF54CEE7EA663ULL, "hlac_lt_0_lowtown_n_0_1z.dds"},
	{0x119EB7C783958695ULL, "234260086.ogg"},
	{0x119EB88EA80DA2E7ULL, "199565837.ogg"},
	{0x119EDCE55578356DULL, "zz_vault_debug6177764.cl"},
	{0x119F00EECBF90480ULL, "gen00fl_merrill6106993.cl"},
	{0x119F155C92ED0A58ULL, "lt_0_lowtown_docks_a3_n_16_0_ll.mmh"},
	{0x119F1D8AA5E4C608ULL, "zz_otl_debug6060956.cl"},
	{0x119F307056DC21F5ULL, "6166600_m.fxe"},
	{0x119FB21E3C8526E0ULL, "vxca_caverns1_mines_e7t_0_0.vlm"},
	{0x119FEB405ACA72AEULL, "art\\characters\\creatures\\ashwraith\\animation\\ras_kd_lp.ani"},
	{0x11A03D00A705EE87ULL, "art\\vfx\\cinematic\\vfx_isa_intro_glass_smash.anb"},
	{0x11A05F263BB7B1C4ULL, "441791271.ogg"},
	{0x11A07A2524EAA019ULL, "179972604.ogg"},
	{0x11A08D85CD654152ULL, "lgt310_mis_burn_murder26152766.cl"},
	{0x11A09FF192F0B759ULL, "vxdr_primevaltomb_706_0_0.vlm"},
	{0x11A0A163291A32ABULL, "vxlt_warehouse_dd_0_0.vlm"},
	{0x11A1AF03FE2069EFULL, "vxlt_gallowscourtyard_4wb_0_0.vlm"},
	{0x11A1D8CE2895CA36ULL, "one110cr_harimann_guard_2.utc"},
	{0x11A1F50D8FEE2C4EULL, "6125897_m.fxe"},
	{0x11A25E1BED7F5049ULL, "6174243_m.fxe"},
	{0x11A29F93DD394523ULL, "vxlt_hangedman_16g_0_0.vlm"},
	{0x11A31A17C79B803DULL, "349671030.ogg"},
	{0x11A3347ADB51612CULL, "6096872_m.fxe"},
	{0x11A335F63719335EULL, "mag360_samson6044903.cl"},
	{0x11A36C972DE189A8ULL, "441868887.ogg"},
	{0x11A4E743215F4C17ULL, "vxht_0_hightown_as2_0_0.vlm"},
	{0x11A50B2DD643404FULL, "mag101_grace6034376.cl"},
	{0x11A517BE6AE2AEB0ULL, "6206088_m.fxe"},
	{0x11A51900061E9865ULL, "pro000_refugee_angry6052173.cl"},
	{0x11A5254332DA4434ULL, "6113130_f.fxe"},
	{0x11A6407A5A388CC5ULL, "vxw_bonepit_1vd_0_0.vlm"},
	{0x11A695B1823A14E4ULL, "vxlt_gallowstemplar_3nk_0_0.vlm"},
	{0x11A7031CA611496EULL, "6070980_m.fxe"},
	{0x11A731EC174E97EBULL, "6106882_m.fxe"},
	{0x11A78A06B6011E16ULL, "vxlt_0_lowtown_docks_bqw_0_0.vlm"},
	{0x11A79888B90AA2D7ULL, "557596028.ogg"},
	{0x11A7C0A40575FF1EULL, "vxlt_foundry_n_x2_0_0.vlm"},
	{0x11A8288659E17E4CULL, "w_bonepitmining_a2_2_0_lc.mmh"},
	{0x11A83C92AF3CF0E1ULL, "vxw_woundedcoast_g5_0_0.vlm"},
	{0x11A855884AF8E1EEULL, "art\\vfx\\cinematic\\vfx_flem_drag_breath_imp_p.cess.ani"},
	{0x11A85A8358BC524CULL, "plt_cod_ltr_sk_starkhaven.nss"},
	{0x11A87F279E36E6D6ULL, "hlac_lt_0_lowtown_n_196607_1x.dds"},
	{0x11A89C0EA717C720ULL, "hlac_lt_0_lowtown_docks_a3_n_65534_5z.dds"},
	{0x11A8B4CB5A065AC7ULL, "6106697_m.fxe"},
	{0x11A9670F4FA10534ULL, "524036172.ogg"},
	{0x11A9B40BF8E62325ULL, "hlac_w_sundermount_-196613_0y.dds"},
	{0x11A9C07D6804C74AULL, "vxlt_gamlenshouse_n_59_0_0.vlm"},
	{0x11AA555CA020BC19ULL, "vxlt_0_lowtown_docks_a3_cnk_0_0.vlm"},
	{0x11AA5D1E85D9F68FULL, "character_character_p246.psh.fxo.dx11.assembly"},
	{0x11AA85CD2D891C2EULL, "vxlt_0_lowtown_a2_ff7_0_0.vlm"},
	{0x11AAD080001342C3ULL, "hf_flora_harimann.mor"},
	{0x11AAD2D24DF10169ULL, "seb201cr_lady_harimann.utc"},
	{0x11AAFD472C41985EULL, "vxlt_gallowscourtyard_796_0_0.vlm"},
	{0x11AAFF182DB9E683ULL, "vxlt_undercitydungeon_1a8_0_0.vlm"},
	{0x11AB327580E0F1EDULL, "zz_mrl_debug6184213.cl"},
	{0x11AB43D429A0D028ULL, "6113278_m.fxe"},
	{0x11ABB56500F0CE7BULL, "vxw_encounter3_48_0_0.vlm"},
	{0x11ABFA55361C28B6ULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_chk01b.msh"},
	{0x11AC0455361C39B4ULL, "art\\characters\\playercharacter\\shared\\heads\\mt_amt_chk01b.mmh"},
	{0x11AC8E7037D2283CULL, "dae000cr_grand_cleric6173686.cl"},
	{0x11AC9FE321F0B411ULL, "zz_lgt_1_debug6114618.cl"},
	{0x11ACAD8420F83125ULL, "vxlt_darktown_6gp_0_0.vlm"},
	{0x11ACE24918073173ULL, "mag170_thrask6054014.cl"},
	{0x11ACF7D2DEF94982ULL, "rock_aa.dds"},
	{0x11AD66973B3A2E17ULL, "qcr000_arishok6042508.cl"},
	{0x11ADFB07D5FD7758ULL, "vxlt_0_lowtown_a2_n_c_e8a_0_0.vlm"},
	{0x11AE4C7FD6D1A019ULL, "6104306_f.fxe"},
	{0x11AE5A5DF93CFF7CULL, "gen_im_trs_qun_two_2ha_r1.uti"},
	{0x11AE9122CEBB01A4ULL, "w_blightlands_171_0.mmh"},
	{0x11AE9522CEBB08F7ULL, "w_blightlands_171_0.mao"},
	{0x11AEA722CEBB2746ULL, "w_blightlands_171_0.msh"},
	{0x11AEA8FEA27F1EC1ULL, "vxlt_0_lowtown_f4v_0_0.vlm"},
	{0x11AF29C2394C38FCULL, "zz_followers_banter6157361.cl"},
	{0x11AF4D76DBCD5025ULL, "vxca_cavern1_haunted_gey_0_0.vlm"},
	{0x11AF9111CA2B4990ULL, "zz_dae_debug6049947.cl"},
	{0x11B002B0A3732D3FULL, "hlac_w_bonepitmining_a2_1_0x.dds"},
	{0x11B0628DB6EE411FULL, "vxw_bonepitdestroyed_a3_1pl_0_0.vlm"},
	{0x11B0DC96F197A714ULL, "art\\levels\\races\\ferelden\\kirkhigh\\int\\4co_hightownint\\dr2a_simple_9.pwk"},
	{0x11B1C54870919CC8ULL, "vxw_woundedcoast_g0_0_0.vlm"},
	{0x11B207FE2AFD08D1ULL, "lgt260_way_blood_mage6148375.cl"},
	{0x11B20C2F6E77A0D7ULL, "lt_0_lowtown_docks_a3_n_186_0.msh"},
	{0x11B21B726EBB6DB1ULL, "dae000cr_grand_cleric6187930.cl"},
	{0x11B21E2F6E77BF3AULL, "lt_0_lowtown_docks_a3_n_186_0.mao"},
	{0x11B2222F6E77C649ULL, "lt_0_lowtown_docks_a3_n_186_0.mmh"},
	{0x11B2E5E98A8C1157ULL, "gen00fl_merrill6128729.cl"},
	{0x11B3D036477B67A3ULL, "vxht_mansion_n_54h_0_0.vlm"},
	{0x11B40006BA674A4DULL, "6042102_m.fxe"},
	{0x11B41CE843A1D4E5ULL, "vxlt_foundry_n_vs_0_0.vlm"},
	{0x11B427774BAA58E1ULL, "484021365.ogg"},
	{0x11B4708A574E754AULL, "hlac_lt_0_lowtown_docks_n_-196609_0y.dds"},
	{0x11B4C4405ADC8CE8ULL, "art\\characters\\creatures\\ashwraith\\animation\\ras_kd_lp.gad"},
	{0x11B54D884B035E3FULL, "art\\vfx\\cinematic\\vfx_flem_drag_breath_imp_p.cess.evt"},
	{0x11B55740CE633321ULL, "hlac_lt_darktown_-1_39y.dds"},
	{0x11B56D2E8219D293ULL, "6071890_m.fxe"},
	{0x11B694E058884392ULL, "6115832_m.fxe"},
	{0x11B6FE87863EF688ULL, "hlac_ht_bartrand_n_3369_0x.dds"},
	{0x11B7A2400B407A5AULL, "zz_dae_debug6029806.cl"},
	{0x11B873DCDE619D86ULL, "vxca_caverns1_iok_0_0.vlm"},
	{0x11B8B444BF9BED96ULL, "vxlt_gallowscourtyard_7fs_0_0.vlm"},
	{0x11B9756591DEAE6FULL, "6086287_m.fxe"},
	{0x11B9C4CB069D2B48ULL, "694234596.ogg"},
	{0x11B9DAC84522F8F6ULL, "hlac_lt_gallowstemplar_2383_0x.dds"},
	{0x11BA1A4041764379ULL, "art\\vfx\\combat\\vfx_r_dlt_duelling_loop_c.anb"},
	{0x11BA56497794AF0EULL, "6092510_f.fxe"},
	{0x11BA93EE426E0E70ULL, "vxlt_0_lowtown_docks_a3_n_cor_0_0.vlm"},
	{0x11BB333D1A684DF7ULL, "w_bonepit_247_0.msh"},
	{0x11BB453D1A686C5AULL, "w_bonepit_247_0.mao"},
	{0x11BB493D1A687369ULL, "w_bonepit_247_0.mmh"},
	{0x11BB701376A4A410ULL, "711099403.ogg"},
	{0x11BB757E1F01785AULL, "6059039_m.fxe"},
	{0x11BC122ADEACB2AFULL, "6018598_m.fxe"},
	{0x11BC2E884B094828ULL, "art\\vfx\\cinematic\\vfx_flem_drag_breath_imp_p.cess.gad"},
	{0x11BC58593C1E1FF1ULL, "141054431.ogg"},
	{0x11BC5DC9ED006DACULL, "6088729_m.fxe"},
	{0x11BCD7539122FACCULL, "dae000_worthy6166571.cl"},
	{0x11BD2E28F2B01A94ULL, "204816395.ogg"},
	{0x11BD53B2A906DEFAULL, "hlac_ca_caverns2_5802_23z.dds"},
	{0x11BDC0EFD07388DFULL, "dae000_quintus6165250.cl"},
	{0x11BDF51300574D02ULL, "764821954.ogg"},
	{0x11BE985812685C24ULL, "vxca_caverns2prc_9yi_0_0.vlm"},
	{0x11BED976B8036294ULL, "art\\characters\\weapons\\longswords\\textures\\w_lsw_ls103a_0t.dds"},
	{0x11BFF4104B5F1A35ULL, "vxlt_0_lowtown_a3_n_c_fi5_0_0.vlm"},
	{0x11C06AB32D5D3C52ULL, "6141240_m.fxe"},
	{0x11C0D6D47A746690ULL, "845713336.ogg"},
	{0x11C1619579098D63ULL, "weapon_weapon_p0.vsh.fxo.dx11"},
	{0x11C1FF9C6C8DEAE1ULL, "918031519.ogg"},
	{0x11C24B73FF69D43CULL, "339475308.ogg"},
	{0x11C2C56EA94598E4ULL, "ser120_ghyslain6118443.cl"},
	{0x11C3114702F5D549ULL, "6142992_m.fxe"},
	{0x11C382270F9277FAULL, "vxw_bonepit_1od_0_0.vlm"},
	{0x11C39BFB257B40F5ULL, "6087633_m.fxe"},
	{0x11C406CA63FCA6B3ULL, "llac_lt_0_lowtown_docks_a3_n_196608_1.dds"},
	{0x11C4185195535F44ULL, "fx_explosion02_flipb.dds"},
	{0x11C448884DEA995FULL, "170373646.ogg"},
	{0x11C49CBCC9102E66ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\pose_e_weightshift_medium.evt"},
	{0x11C4D08E98537AEAULL, "81389740.ogg"},
	{0x11C510BB382B8F97ULL, "vxht_0_hightown_a2_aob_0_0.vlm"},
	{0x11C5A203411FF3A9ULL, "vxlt_0_lowtown_docks_a3_n_cz9_0_1.vlm"},
	{0x11C5CEE221C5A0E6ULL, "615852746.ogg"},
	{0x11C5F6A6A119E0D3ULL, "618268947.ogg"},
	{0x11C6AC46E004732EULL, "vxht_0_hightown_n_8rr_0_0.vlm"},
	{0x11C6DB171ADBBFBFULL, "vxw_bonepitdestroyed_a3_272_0_0.vlm"},
	{0x11C70FF8F7585FBEULL, "vxca_caverns2_8bb_0_0.vlm"},
	{0x11C71C0623F2490CULL, "vxca_cavern1_haunted_egx_0_0.vlm"},
	{0x11C72661925553FDULL, "vxht_mansion2_2im_0_0.vlm"},
	{0x11C72878A2FAC86FULL, "weapon_rendertodepth_p0.psh.fxo.dx11"},
	{0x11C73A0A703A3047ULL, "353087041.ogg"},
	{0x11C7E4C34C70652BULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_bookstand_d.dds"},
	{0x11C85E17FBB0550DULL, "zz_fen_debug6134359.cl"},
	{0x11C90D3BBFB6B099ULL, "121017869.ogg"},
	{0x11C93A00A728EE88ULL, "art\\vfx\\cinematic\\vfx_isa_intro_glass_smash.mmh"},
	{0x11CA149359497A2BULL, "573796998.ogg"},
	{0x11CA8EE63FC3479CULL, "tm_hlk_b01.tnt"},
	{0x11CAD3A3B619C479ULL, "6088247_m.fxe"},
	{0x11CADE1C11368000ULL, "976821850.ogg"},
	{0x11CB3650E988C9CBULL, "6087473_m.fxe"},
	{0x11CB37F3B1590662ULL, "569593425.ogg"},
	{0x11CBB3BCC91673B1ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\pose_e_weightshift_medium.gad"},
	{0x11CBF6F9FEFC7C2BULL, "vxlt_warehouse_wy_0_0.vlm"},
	{0x11CC60904E0F2944ULL, "616798476.ogg"},
	{0x11CD364037953911ULL, "llac_dr_primevaltomb_659_14.dds"},
	{0x11CD469A2A18EE0CULL, "6153458_m.fxe"},
	{0x11CDBE7A099F1899ULL, "vxlt_gamlenshouse_n_mn_0_0.vlm"},
	{0x11CDD04A81DD522EULL, "vxht_keep_4mp_0_0.vlm"},
	{0x11CE20F4E779AA87ULL, "643091602.ogg"},
	{0x11CE5E69029147C4ULL, "llac_ca_cavern1_haunted_24423_9.dds"},
	{0x11CE7B48E23197FEULL, "art\\vfx\\combat\\vfx_w_bsk_regain_sta_imp_c.mmh"},
	{0x11CE84F63C89B274ULL, "art\\vfx\\materials\\chunky\\vfx_c_moving_cloud.mao"},
	{0x11CFF5B7417CCE9DULL, "430607087.ogg"},
	{0x11D008E16F1C95F3ULL, "vxlt_0_lowtown_docks_n_cnm_0_0.vlm"},
	{0x11D025600E629F1CULL, "hlac_ca_caverns1_24427_2z.dds"},
	{0x11D045EEA5852479ULL, "944003737.ogg"},
	{0x11D09202A3359BB6ULL, "lowtown_fem_alienage1_sta6162544.cl"},
	{0x11D0BDCF9E69D076ULL, "dae200_cullen6140315.cl"},
	{0x11D175BBD1C7000FULL, "vxht_chantry_36o_0_0.vlm"},
	{0x11D19D2763E38657ULL, "lt_0_lowtown_docks_a3_n_247_0.mmh"},
	{0x11D1A12763E38D64ULL, "lt_0_lowtown_docks_a3_n_247_0.mao"},
	{0x11D1B32763E3ABC9ULL, "lt_0_lowtown_docks_a3_n_247_0.msh"},
	{0x11D1C62E291FD348ULL, "vxw_bonepit_1q8_0_0.vlm"},
	{0x11D214BF90535D63ULL, "378635256.ogg"},
	{0x11D26EFD97487698ULL, "llac_dr_0_deeproads_5845_17.dds"},
	{0x11D2B4BCC91C945FULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\pose_e_weightshift_medium.ani"},
	{0x11D34244EDDFB715ULL, "plt_lgt250pt_way_dlsh_cnflc.nss"},
	{0x11D360FA2CB4DB60ULL, "6165969_m.fxe"},
	{0x11D3D73DA47085C9ULL, "gen00fl_isabela_init6110277.cl"},
	{0x11D50AEE6E9D0761ULL, "6041411_m.fxe"},
	{0x11D52080D40F1781ULL, "fxa_evl_aura2.mao"},
	{0x11D56C034B0CB47BULL, "vxlt_0_lowtown_a2_ev9_0_0.vlm"},
	{0x11D581D5E94E783EULL, "vxca_caverns2_8ic_0_0.vlm"},
	{0x11D58F0BF2ADE8F1ULL, "6115295_m.fxe"},
	{0x11D5AFBF0B03E747ULL, "vxht_0_hightown_n_av6_0_0.vlm"},
	{0x11D5ED09867C385DULL, "vxw_woundedcoast_149_0_0.vlm"},
	{0x11D6BD446A901750ULL, "228598428.ogg"},
	{0x11D777A8265A3F70ULL, "vxlt_0_lowtown_docks_a3_n_cup_0_0.vlm"},
	{0x11D7F8445C28EB4AULL, "hlac_ht_chantry_2534_26y.dds"},
	{0x11D87E96F12114B7ULL, "vxlt_gallowstemplar_f_4qq_0_0.vlm"},
	{0x11D8C60CF39E6517ULL, "6171866_m.fxe"},
	{0x11D8CD9F1514A52EULL, "mcr371cr_anders_cut.utc"},
	{0x11D90E3E63665787ULL, "clf260_harley6173865.cl"},
	{0x11D914B63D57AC55ULL, "dae300_alistair_king6133818.cl"},
	{0x11D91D2AC8783490ULL, "vxca_caverns1_gjl_0_0.vlm"},
	{0x11D93F7FF84F75F1ULL, "418657927.ogg"},
	{0x11D9554955DC022EULL, "vxht_keep_739_0_0.vlm"},
	{0x11D9C9E806693BBCULL, "art\\levels\\races\\proxy\\props\\armoire\\animation\\armoire.closing.ani"},
	{0x11D9DB782B3DD6ECULL, "6165279_m.fxe"},
	{0x11DA68D6F0F294ABULL, "vxlt_warehouse_xx_0_0.vlm"},
	{0x11DA747F9EF1F970ULL, "304208144.ogg"},
	{0x11DA765B6C49C9DBULL, "762856043.ogg"},
	{0x11DB39ED33507020ULL, "171137149.ogg"},
	{0x11DBE8350B2D2309ULL, "ht_0_hightown_a3_33_0_ll.mmh"},
	{0x11DC5A48E23D9CBDULL, "art\\vfx\\combat\\vfx_w_bsk_regain_sta_imp_c.anb"},
	{0x11DD1E985D8CA7DCULL, "llac_lt_0_lowtown_a2_n_c_0_4.dds"},
	{0x11DD8A9B51D49B24ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\9uw_tarpoutd.msh"},
	{0x11DD9C9B51D4B98DULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\9uw_tarpoutd.mao"},
	{0x11DDA09B51D4C0C6ULL, "art\\levels\\races\\ferelden\\kirklow\\int\\4co_lowtown\\9uw_tarpoutd.mmh"},
	{0x11DDAE0FC9D62B88ULL, "6001289_m.fxe"},
	{0x11DDEFF9A778478AULL, "art\\characters\\creatures\\stonegolem\\textures\\cn_bdy_snga_0n.dds"},
	{0x11DE530BB8B407CAULL, "pro000_cut2d_thefirstyear6005295.cl"},
	{0x11DEB491F2F377F5ULL, "vxht_mansion2b_2im_0_0.vlm"},
	{0x11DEF70D46C5A64BULL, "hlac_w_bonepitmining_a2_196613_0y.dds"},
	{0x11DFBE56DDE2B983ULL, "789231335.ogg"},
	{0x11DFCBB85DE2ECCFULL, "plt_zz_apppt_debug.plo"},
	{0x11E0380B1B15EBC8ULL, "vxw_bonepit_1z9_0_0.vlm"},
	{0x11E073194B36EA44ULL, "gen00fl_fenris6126739.cl"},
	{0x11E08BF1AB5C32C5ULL, "6148240_m.fxe"},
	{0x11E0D6E8066F70FEULL, "art\\levels\\races\\proxy\\props\\armoire\\animation\\armoire.closing.gad"},
	{0x11E119DC2F8452CDULL, "vxht_mansion_n_6zz_0_1.vlm"},
	{0x11E22A209FCFCB10ULL, "vxw_blightlands_128_0_0.vlm"},
	{0x11E2B2202E628566ULL, "one280_emrys6120059.cl"},
	{0x11E2CA88C59DD1E5ULL, "vxlt_gallowstemplar_f_5cx_0_0.vlm"},
	{0x11E300F23E1741B9ULL, "711839004.ogg"},
	{0x11E30985E0B93208ULL, "mcr370_messenger6068983.cl"},
	{0x11E346774E3B2CD9ULL, "vxlt_undercityencounter_1bw_0_1.vlm"},
	{0x11E4C6689E85E734ULL, "vxlt_gallowstemplar_n_5qc_0_0.vlm"},
	{0x11E4CBD0E8F70EEAULL, "vxw_woundedcoast_1n8_0_0.vlm"},
	{0x11E4E06300F1B7B8ULL, "gen_im_npc_arm_hum_glv_rdrc.uti"},
	{0x11E5021955810479ULL, "875458192.ogg"},
	{0x11E58A611798375BULL, "vxht_0_hightown_9cn_0_0.vlm"},
	{0x11E59395F4CFCEA4ULL, "vxlt_gallowscourtyard_7f1_0_0.vlm"},
	{0x11E5A4EB6CC9A346ULL, "zz_fen_debug6165671.cl"},
	{0x11E5F2F1BA85284EULL, "art\\levels\\races\\ferelden\\props\\_textures\\prp_junkd_sa.dds"},
	{0x11E5F8EFCCE573CEULL, "hlac_w_woundedcoast_0_12z.dds"},
	{0x11E65CA391941A75ULL, "546726120.ogg"},
	{0x11E66ED56792D617ULL, "gen_im_trs_qun_lgt_boo_r2.uti"},
	{0x11E67547901A8EF2ULL, "mcr370_samson.cnv"},
	{0x11E687447F7CA773ULL, "fen210_fenris6108721.cl"},
	{0x11E6E8C3925D1134ULL, "vxlt_gallowstemplar_f_55o_0_0.vlm"},
	{0x11E71E7E91587D79ULL, "dae100_gamlen_letter6146995.cl"},
	{0x11E75C534AF490B6ULL, "art\\characters\\playercharacter\\humanfemale\\arm_wrba.msh"},
	{0x11E766534AF4A1B4ULL, "art\\characters\\playercharacter\\humanfemale\\arm_wrba.mmh"},
	{0x11E76A534AF4A8C7ULL, "art\\characters\\playercharacter\\humanfemale\\arm_wrba.mao"},
	{0x11E7B09C0F75B3A0ULL, "54157363.ogg"},
	{0x11E7F479D1863F8BULL, "517550101.ogg"},
	{0x11E804ABAA6AAAE5ULL, "hlac_w_bonepitdestroyed_a3_0_4y.dds"},
	{0x11E84C4C231C4383ULL, "vxca_caverns2prc_a5f_0_0.vlm"},
	{0x11E8A5AA769B59BEULL, "6119753_m.fxe"},
	{0x11E8D45731CEF46BULL, "6133019_m.fxe"},
	{0x11E93EF1F7602CB3ULL, "zz_approval_debug6133710.cl"},
	{0x11E99747901CFE4AULL, "mcr370_samson.bnk"},
	{0x11E9C317C4D7CD28ULL, "art\\levels\\races\\qunari\\props\\_textures\\prp_quanarithrone_d.dds"},
	{0x11E9F75642AD6FF8ULL, "764350333.ogg"},
	{0x11EA335823487A48ULL, "870339646.ogg"},
	{0x11EA3AE187DE5E0BULL, "vxlt_0_lowtown_a2_n_c_gnv_0_0.vlm"},
	{0x11EB9719A1DD2C6FULL, "vxlt_0_lowtown_a2_fec_0_0.vlm"},
	{0x11EC7B66C84AECC2ULL, "6039392_m.fxe"},
	{0x11ECEDDA2B8909C3ULL, "art\\characters\\playercharacter\\humanmale\\animation\\cindes\\pose\\rightarm_rub_neck.ani"}