## TFSA-2022-007: Type confusion in shape inference for `ConcatV2`

### CVE Number
CVE-2022-21731

### Impact
The [implementation of shape inference for `ConcatV2`](https://github.com/tensorflow/tensorflow/blob/5100e359aef5c8021f2e71c7b986420b85ce7b3d/tensorflow/core/framework/common_shape_fns.cc#L1961-L2059) can be used to trigger a denial of service attack via a segfault caused by a type confusion:

```python
import tensorflow as tf

@tf.function
def test():
  y = tf.raw_ops.ConcatV2(
    values=[[1,2,3],[4,5,6]],
    axis = 0xb500005b)
  return y

test()
```

The `axis` argument is translated into `concat_dim` in the `ConcatShapeHelper` helper function. Then, a value for `min_rank` is computed based on `concat_dim`. This is then used to validate that the `values` tensor has at least the required rank:

```cc
  int64_t concat_dim;
  if (concat_dim_t->dtype() == DT_INT32) {
    concat_dim = static_cast<int64_t>(concat_dim_t->flat<int32>()(0));
  } else {
    concat_dim = concat_dim_t->flat<int64_t>()(0);
  }

  // Minimum required number of dimensions.
  const int min_rank = concat_dim < 0 ? -concat_dim : concat_dim + 1;

  // ...
  ShapeHandle input = c->input(end_value_index - 1);
  TF_RETURN_IF_ERROR(c->WithRankAtLeast(input, min_rank, &input));
```

However, [`WithRankAtLeast`](https://github.com/tensorflow/tensorflow/blob/5100e359aef5c8021f2e71c7b986420b85ce7b3d/tensorflow/core/framework/shape_inference.cc#L345-L358) receives the lower bound as a 64-bits value and then compares it against the maximum 32-bits integer value that could be represented:

```cc
Status InferenceContext::WithRankAtLeast(ShapeHandle shape, int64_t rank,
                                         ShapeHandle* out) {
  if (rank > kint32max) {
    return errors::InvalidArgument("Rank cannot exceed kint32max");
  }
  // ...
}
```

Due to the fact that `min_rank` is a 32-bits value and the value of `axis`, the `rank` argument is a [negative value](https://godbolt.org/z/Gcr5haMob), so the error check is bypassed.

### Patches
We have patched the issue in GitHub commit [08d7b00c0a5a20926363849f611729f53f3ec022](https://github.com/tensorflow/tensorflow/commit/08d7b00c0a5a20926363849f611729f53f3ec022).

The fix will be included in TensorFlow 2.8.0. We will also cherrypick this commit on TensorFlow 2.7.1, TensorFlow 2.6.3, and TensorFlow 2.5.3, as these are also affected and still in supported range.

### For more information
Please consult [our security guide](https://github.com/tensorflow/tensorflow/blob/master/SECURITY.md) for more information regarding the security model and how to contact us with issues and questions.

### Attribution
This vulnerability has been reported by Yu Tian of Qihoo 360 AIVul Team.
